package Business::CPI::Role::Exception;
# ABSTRACT: Exceptions from the gateway
use Moo::Role;
use Types::Standard qw/Int Str HashRef/;
with 'Throwable';

our $VERSION = '0.917'; # VERSION

has code => (
    coerce   => Int->coercion,
    isa      => Int,
    is       => 'ro',
    required => 1,
);

has message => (
    coerce   => Str->coercion,
    isa      => Str,
    is       => 'ro',
    required => 1,
);

has gateway_data => (
    coerce   => HashRef->coercion,
    isa      => HashRef,
    is       => 'ro',
    required => 1,
);

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Business::CPI::Role::Exception - Exceptions from the gateway

=head1 VERSION

version 0.917

=head1 DESCRIPTION

This role is meant to be used by the drivers to encapsulate gateway exceptions,
put them in the required format, and rethrowing them as real Perl exceptions
(using L<Throwable>).

=head1 ATTRIBUTES

=head2 code

Numeric code to identify the exception. B<TODO:> This has not been standardized
yet. A list shall be documented here with all the possible error codes,
possibly similar to HTTP status codes.

=head2 message

A human readable message of the error, in English, either generated by
L<Business::CPI>, the driver, or the gateway. This serves only for debuging
purposes, and not for your code to parse this and handle the exception.

=head2 gateway_data

Plain HashRef (i.e., not blessed) containing any extra data regarding the
exception that might be useful. For example:

    {
        raw_lwp_request    => $req, # $res->isa('HTTP::Request')
        raw_lwp_response   => $res, # $res->isa('HTTP::Response')
        http_status_code   => 400,
        error_code         => 'XM-231',
        message            => 'That currency is currently not supported',
        context            => 'currency',
        exception_id       => 'e171eadad51791966aad6ac10bb6d16354d1952',
    }

That HashRef is supposed to be non-standard, as a way of keeping information
that is only relevant to certain gateways. Although some things might be
standardized, such as the LWP request and response, for instance. Again, this
is still being designed.

=head1 DISCLAIMER

This is very B<EXPERIMENTAL>. We're still designing the interface, and the
error codes are not defined yet. This role will not be usable before the
documentation of all error codes.

=head1 AUTHOR

André Walker <andre@andrewalker.net>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by André Walker.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
