/* Callback.hpp
 * Copyright (C) 2018 - 2021  Sven Jähnichen
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef BWIDGETS_CALLBACK_HPP_
#define BWIDGETS_CALLBACK_HPP_

#include <cstdint>
#include <map>
#include <functional>
#include "../../BEvents/Event.hpp"

namespace BWidgets
{

/**
 *  @brief  Callback functionality
 *
 *  The %Callback class provides callback functionality for EventTypes.
 */
class Callback : protected std::map<uint8_t, std::function<void (BEvents::Event*)>>
{
public:

    /**
     *  @brief  Copy assignment operator.
     *  @return  Reference to @c *this . 
     *
     *  Forcing a copy assignment operator to be generated by the compiler.
     *  Prevents multiple move assignments calls in virtual inheritance.
     */
    Callback& operator= (const Callback&) = default;

    /**
	 *  @brief  (Re-)Defines the callback function for an event type. 
     *  @param eventType  EventType.
     *  @param callbackFunction  Function of the type 
     *  void func (BEvents::Event*).
     *
     *  It is on the widgets onXXX methods whether a callback function will be 
     *  called or not. By default, the callback is set to the empty 
     *  defaultCallback.
     *  
     *  Callback functions fascilitate to handle events without changing the 
     *  widget onXXX method and thus changing the class definition.
	 */
	void setCallbackFunction (const uint32_t eventType, const std::function<void (BEvents::Event*)>& callbackFunction)
    {
        for (uint32_t i = 0; i < 32; ++i)
        {
            if ((1 << i) & eventType) operator[] (i) = callbackFunction;
        }
    }

    /**
	 *  @brief  Removes the callback function for an event type. 
     *  @param eventType  EventType.
	 */
    void removeCallbackFunction (const uint32_t eventType)
    {
        for (uint32_t i = 0; i < 32; ++i)
        {
            if ((1 << i) & eventType)
            {
                const_iterator cit = find (i);
                if (cit != end()) erase (cit);
            } 
        }
    }

    /**
	 *  @brief  Access the callback function for an event type. 
     *  @param eventType  EventType.
     *
     *  If multiple event types are passed, then only the callback function
     *  for the first match is returned.
	 */
    std::function<void (BEvents::Event*)> callback (const BEvents::Event::EventType eventType) 
    {
        for (uint32_t i = 0; i < 32; ++i)
        {
            if ((1 << i) & eventType)
            {
                if (find(i) != end()) return operator[] (i);
            } 
        }
        
        return defaultCallback;
    }

    /**
     *  @brief  Empty default callback function.
     *  @param event  Event.
     */
    static void defaultCallback (BEvents::Event* event) {}
};

}
#endif /* BWIDGETS_CALLBACK_HPP_ */