{-
Copyright (C) 2009 John MacFarlane <jgm@berkeley.edu>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

{- | Function for replacing strings of characters with their respective mathvariant and vice versa.
-}

module Text.TeXMath.Unicode.ToUnicode (fromUnicodeChar,
                                      toUnicodeChar,
                                      fromUnicode,
                                      toUnicode)
where

import Text.TeXMath.Types
import qualified Data.Map as M
import qualified Data.Text as T
import Data.Maybe (fromMaybe)

-- | Replace characters with their corresponding mathvariant unicode character.
--  MathML has a mathvariant attribute which is unimplemented in Firefox
--  (see <https://bugzilla.mozilla.org/show_bug.cgi?id=114365 here>)
--  Therefore, we may want to translate mathscr, etc to unicode symbols directly.
toUnicode :: TextType -> T.Text -> T.Text
toUnicode TextNormal = id
toUnicode tt = T.map (\c -> fromMaybe c (toUnicodeChar (tt, c)))

toUnicodeChar :: (TextType, Char) -> Maybe Char
toUnicodeChar x = M.lookup x unicodeMap

-- | The inverse of 'toUnicodeChar': returns the corresponding
--  unstyled character and 'TextType' of a unicode character.
fromUnicodeChar :: Char -> Maybe (TextType, Char)
fromUnicodeChar c = M.lookup c reverseUnicodeMap

-- | Inverse of 'toUnicode'.
fromUnicode :: TextType -> T.Text -> T.Text
fromUnicode tt cs =
  T.map (\c -> case fromUnicodeChar c of
                 Just (tt', c') | tt' == tt -> c'
                 _ -> c) cs

reverseUnicodeMap :: M.Map Char (TextType, Char)
reverseUnicodeMap = M.fromList $ map (\(a,b) -> (b,a)) unicodeTable

unicodeMap :: M.Map (TextType, Char) Char
unicodeMap = M.fromList unicodeTable

-- DO NOT EDIT BY HAND BELOW THIS POINT!
-- unicodeTable is automatically generated by mkUnicodeTable.hs in
-- lib/tounicode.
-- Note:  when the table contains duplicates (as is the case here
-- in a few cases, e.g. with DoubleStruck 'C'), M.fromList
-- will take the one that occurs LAST in the list.  So the
-- 'MATHEMATICAL' entries will be preferred.
unicodeTable :: [((TextType, Char), Char)]
unicodeTable = [ ((TextDoubleStruck,'C'),'\8450')  -- C -> ℂ
               , ((TextScript,'g'),'\8458')  -- g -> ℊ
               , ((TextScript,'H'),'\8459')  -- H -> ℋ
               , ((TextFraktur,'H'),'\8460')  -- H -> ℌ
               , ((TextDoubleStruck,'H'),'\8461')  -- H -> ℍ
               , ((TextScript,'I'),'\8464')  -- I -> ℐ
               , ((TextFraktur,'I'),'\8465')  -- I -> ℑ
               , ((TextScript,'L'),'\8466')  -- L -> ℒ
               , ((TextScript,'l'),'\8467')  -- l -> ℓ
               , ((TextDoubleStruck,'N'),'\8469')  -- N -> ℕ
               , ((TextDoubleStruck,'P'),'\8473')  -- P -> ℙ
               , ((TextDoubleStruck,'Q'),'\8474')  -- Q -> ℚ
               , ((TextScript,'R'),'\8475')  -- R -> ℛ
               , ((TextFraktur,'R'),'\8476')  -- R -> ℜ
               , ((TextDoubleStruck,'R'),'\8477')  -- R -> ℝ
               , ((TextDoubleStruck,'Z'),'\8484')  -- Z -> ℤ
               , ((TextFraktur,'Z'),'\8488')  -- Z -> ℨ
               , ((TextScript,'B'),'\8492')  -- B -> ℬ
               , ((TextFraktur,'C'),'\8493')  -- C -> ℭ
               , ((TextScript,'e'),'\8495')  -- e -> ℯ
               , ((TextScript,'E'),'\8496')  -- E -> ℰ
               , ((TextScript,'F'),'\8497')  -- F -> ℱ
               , ((TextScript,'M'),'\8499')  -- M -> ℳ
               , ((TextScript,'o'),'\8500')  -- o -> ℴ
               , ((TextDoubleStruck,'\960'),'\8508')  -- π -> ℼ
               , ((TextDoubleStruck,'\947'),'\8509')  -- γ -> ℽ
               , ((TextDoubleStruck,'\915'),'\8510')  -- Γ -> ℾ
               , ((TextDoubleStruck,'\928'),'\8511')  -- Π -> ℿ
               , ((TextDoubleStruck,'\8721'),'\8512')  -- ∑ -> ⅀
               , ((TextDoubleStruck,'D'),'\8517')  -- D -> ⅅ
               , ((TextDoubleStruck,'d'),'\8518')  -- d -> ⅆ
               , ((TextDoubleStruck,'e'),'\8519')  -- e -> ⅇ
               , ((TextDoubleStruck,'i'),'\8520')  -- i -> ⅈ
               , ((TextDoubleStruck,'j'),'\8521')  -- j -> ⅉ
               , ((TextBold,'A'),'\119808')  -- A -> 𝐀
               , ((TextBold,'B'),'\119809')  -- B -> 𝐁
               , ((TextBold,'C'),'\119810')  -- C -> 𝐂
               , ((TextBold,'D'),'\119811')  -- D -> 𝐃
               , ((TextBold,'E'),'\119812')  -- E -> 𝐄
               , ((TextBold,'F'),'\119813')  -- F -> 𝐅
               , ((TextBold,'G'),'\119814')  -- G -> 𝐆
               , ((TextBold,'H'),'\119815')  -- H -> 𝐇
               , ((TextBold,'I'),'\119816')  -- I -> 𝐈
               , ((TextBold,'J'),'\119817')  -- J -> 𝐉
               , ((TextBold,'K'),'\119818')  -- K -> 𝐊
               , ((TextBold,'L'),'\119819')  -- L -> 𝐋
               , ((TextBold,'M'),'\119820')  -- M -> 𝐌
               , ((TextBold,'N'),'\119821')  -- N -> 𝐍
               , ((TextBold,'O'),'\119822')  -- O -> 𝐎
               , ((TextBold,'P'),'\119823')  -- P -> 𝐏
               , ((TextBold,'Q'),'\119824')  -- Q -> 𝐐
               , ((TextBold,'R'),'\119825')  -- R -> 𝐑
               , ((TextBold,'S'),'\119826')  -- S -> 𝐒
               , ((TextBold,'T'),'\119827')  -- T -> 𝐓
               , ((TextBold,'U'),'\119828')  -- U -> 𝐔
               , ((TextBold,'V'),'\119829')  -- V -> 𝐕
               , ((TextBold,'W'),'\119830')  -- W -> 𝐖
               , ((TextBold,'X'),'\119831')  -- X -> 𝐗
               , ((TextBold,'Y'),'\119832')  -- Y -> 𝐘
               , ((TextBold,'Z'),'\119833')  -- Z -> 𝐙
               , ((TextBold,'a'),'\119834')  -- a -> 𝐚
               , ((TextBold,'b'),'\119835')  -- b -> 𝐛
               , ((TextBold,'c'),'\119836')  -- c -> 𝐜
               , ((TextBold,'d'),'\119837')  -- d -> 𝐝
               , ((TextBold,'e'),'\119838')  -- e -> 𝐞
               , ((TextBold,'f'),'\119839')  -- f -> 𝐟
               , ((TextBold,'g'),'\119840')  -- g -> 𝐠
               , ((TextBold,'h'),'\119841')  -- h -> 𝐡
               , ((TextBold,'i'),'\119842')  -- i -> 𝐢
               , ((TextBold,'j'),'\119843')  -- j -> 𝐣
               , ((TextBold,'k'),'\119844')  -- k -> 𝐤
               , ((TextBold,'l'),'\119845')  -- l -> 𝐥
               , ((TextBold,'m'),'\119846')  -- m -> 𝐦
               , ((TextBold,'n'),'\119847')  -- n -> 𝐧
               , ((TextBold,'o'),'\119848')  -- o -> 𝐨
               , ((TextBold,'p'),'\119849')  -- p -> 𝐩
               , ((TextBold,'q'),'\119850')  -- q -> 𝐪
               , ((TextBold,'r'),'\119851')  -- r -> 𝐫
               , ((TextBold,'s'),'\119852')  -- s -> 𝐬
               , ((TextBold,'t'),'\119853')  -- t -> 𝐭
               , ((TextBold,'u'),'\119854')  -- u -> 𝐮
               , ((TextBold,'v'),'\119855')  -- v -> 𝐯
               , ((TextBold,'w'),'\119856')  -- w -> 𝐰
               , ((TextBold,'x'),'\119857')  -- x -> 𝐱
               , ((TextBold,'y'),'\119858')  -- y -> 𝐲
               , ((TextBold,'z'),'\119859')  -- z -> 𝐳
               , ((TextItalic,'A'),'\119860')  -- A -> 𝐴
               , ((TextItalic,'B'),'\119861')  -- B -> 𝐵
               , ((TextItalic,'C'),'\119862')  -- C -> 𝐶
               , ((TextItalic,'D'),'\119863')  -- D -> 𝐷
               , ((TextItalic,'E'),'\119864')  -- E -> 𝐸
               , ((TextItalic,'F'),'\119865')  -- F -> 𝐹
               , ((TextItalic,'G'),'\119866')  -- G -> 𝐺
               , ((TextItalic,'H'),'\119867')  -- H -> 𝐻
               , ((TextItalic,'I'),'\119868')  -- I -> 𝐼
               , ((TextItalic,'J'),'\119869')  -- J -> 𝐽
               , ((TextItalic,'K'),'\119870')  -- K -> 𝐾
               , ((TextItalic,'L'),'\119871')  -- L -> 𝐿
               , ((TextItalic,'M'),'\119872')  -- M -> 𝑀
               , ((TextItalic,'N'),'\119873')  -- N -> 𝑁
               , ((TextItalic,'O'),'\119874')  -- O -> 𝑂
               , ((TextItalic,'P'),'\119875')  -- P -> 𝑃
               , ((TextItalic,'Q'),'\119876')  -- Q -> 𝑄
               , ((TextItalic,'R'),'\119877')  -- R -> 𝑅
               , ((TextItalic,'S'),'\119878')  -- S -> 𝑆
               , ((TextItalic,'T'),'\119879')  -- T -> 𝑇
               , ((TextItalic,'U'),'\119880')  -- U -> 𝑈
               , ((TextItalic,'V'),'\119881')  -- V -> 𝑉
               , ((TextItalic,'W'),'\119882')  -- W -> 𝑊
               , ((TextItalic,'X'),'\119883')  -- X -> 𝑋
               , ((TextItalic,'Y'),'\119884')  -- Y -> 𝑌
               , ((TextItalic,'Z'),'\119885')  -- Z -> 𝑍
               , ((TextItalic,'a'),'\119886')  -- a -> 𝑎
               , ((TextItalic,'b'),'\119887')  -- b -> 𝑏
               , ((TextItalic,'c'),'\119888')  -- c -> 𝑐
               , ((TextItalic,'d'),'\119889')  -- d -> 𝑑
               , ((TextItalic,'e'),'\119890')  -- e -> 𝑒
               , ((TextItalic,'f'),'\119891')  -- f -> 𝑓
               , ((TextItalic,'g'),'\119892')  -- g -> 𝑔
               , ((TextItalic,'i'),'\119894')  -- i -> 𝑖
               , ((TextItalic,'j'),'\119895')  -- j -> 𝑗
               , ((TextItalic,'k'),'\119896')  -- k -> 𝑘
               , ((TextItalic,'l'),'\119897')  -- l -> 𝑙
               , ((TextItalic,'m'),'\119898')  -- m -> 𝑚
               , ((TextItalic,'n'),'\119899')  -- n -> 𝑛
               , ((TextItalic,'o'),'\119900')  -- o -> 𝑜
               , ((TextItalic,'p'),'\119901')  -- p -> 𝑝
               , ((TextItalic,'q'),'\119902')  -- q -> 𝑞
               , ((TextItalic,'r'),'\119903')  -- r -> 𝑟
               , ((TextItalic,'s'),'\119904')  -- s -> 𝑠
               , ((TextItalic,'t'),'\119905')  -- t -> 𝑡
               , ((TextItalic,'u'),'\119906')  -- u -> 𝑢
               , ((TextItalic,'v'),'\119907')  -- v -> 𝑣
               , ((TextItalic,'w'),'\119908')  -- w -> 𝑤
               , ((TextItalic,'x'),'\119909')  -- x -> 𝑥
               , ((TextItalic,'y'),'\119910')  -- y -> 𝑦
               , ((TextItalic,'z'),'\119911')  -- z -> 𝑧
               , ((TextBoldItalic,'A'),'\119912')  -- A -> 𝑨
               , ((TextBoldItalic,'B'),'\119913')  -- B -> 𝑩
               , ((TextBoldItalic,'C'),'\119914')  -- C -> 𝑪
               , ((TextBoldItalic,'D'),'\119915')  -- D -> 𝑫
               , ((TextBoldItalic,'E'),'\119916')  -- E -> 𝑬
               , ((TextBoldItalic,'F'),'\119917')  -- F -> 𝑭
               , ((TextBoldItalic,'G'),'\119918')  -- G -> 𝑮
               , ((TextBoldItalic,'H'),'\119919')  -- H -> 𝑯
               , ((TextBoldItalic,'I'),'\119920')  -- I -> 𝑰
               , ((TextBoldItalic,'J'),'\119921')  -- J -> 𝑱
               , ((TextBoldItalic,'K'),'\119922')  -- K -> 𝑲
               , ((TextBoldItalic,'L'),'\119923')  -- L -> 𝑳
               , ((TextBoldItalic,'M'),'\119924')  -- M -> 𝑴
               , ((TextBoldItalic,'N'),'\119925')  -- N -> 𝑵
               , ((TextBoldItalic,'O'),'\119926')  -- O -> 𝑶
               , ((TextBoldItalic,'P'),'\119927')  -- P -> 𝑷
               , ((TextBoldItalic,'Q'),'\119928')  -- Q -> 𝑸
               , ((TextBoldItalic,'R'),'\119929')  -- R -> 𝑹
               , ((TextBoldItalic,'S'),'\119930')  -- S -> 𝑺
               , ((TextBoldItalic,'T'),'\119931')  -- T -> 𝑻
               , ((TextBoldItalic,'U'),'\119932')  -- U -> 𝑼
               , ((TextBoldItalic,'V'),'\119933')  -- V -> 𝑽
               , ((TextBoldItalic,'W'),'\119934')  -- W -> 𝑾
               , ((TextBoldItalic,'X'),'\119935')  -- X -> 𝑿
               , ((TextBoldItalic,'Y'),'\119936')  -- Y -> 𝒀
               , ((TextBoldItalic,'Z'),'\119937')  -- Z -> 𝒁
               , ((TextBoldItalic,'a'),'\119938')  -- a -> 𝒂
               , ((TextBoldItalic,'b'),'\119939')  -- b -> 𝒃
               , ((TextBoldItalic,'c'),'\119940')  -- c -> 𝒄
               , ((TextBoldItalic,'d'),'\119941')  -- d -> 𝒅
               , ((TextBoldItalic,'e'),'\119942')  -- e -> 𝒆
               , ((TextBoldItalic,'f'),'\119943')  -- f -> 𝒇
               , ((TextBoldItalic,'g'),'\119944')  -- g -> 𝒈
               , ((TextBoldItalic,'h'),'\119945')  -- h -> 𝒉
               , ((TextBoldItalic,'i'),'\119946')  -- i -> 𝒊
               , ((TextBoldItalic,'j'),'\119947')  -- j -> 𝒋
               , ((TextBoldItalic,'k'),'\119948')  -- k -> 𝒌
               , ((TextBoldItalic,'l'),'\119949')  -- l -> 𝒍
               , ((TextBoldItalic,'m'),'\119950')  -- m -> 𝒎
               , ((TextBoldItalic,'n'),'\119951')  -- n -> 𝒏
               , ((TextBoldItalic,'o'),'\119952')  -- o -> 𝒐
               , ((TextBoldItalic,'p'),'\119953')  -- p -> 𝒑
               , ((TextBoldItalic,'q'),'\119954')  -- q -> 𝒒
               , ((TextBoldItalic,'r'),'\119955')  -- r -> 𝒓
               , ((TextBoldItalic,'s'),'\119956')  -- s -> 𝒔
               , ((TextBoldItalic,'t'),'\119957')  -- t -> 𝒕
               , ((TextBoldItalic,'u'),'\119958')  -- u -> 𝒖
               , ((TextBoldItalic,'v'),'\119959')  -- v -> 𝒗
               , ((TextBoldItalic,'w'),'\119960')  -- w -> 𝒘
               , ((TextBoldItalic,'x'),'\119961')  -- x -> 𝒙
               , ((TextBoldItalic,'y'),'\119962')  -- y -> 𝒚
               , ((TextBoldItalic,'z'),'\119963')  -- z -> 𝒛
               , ((TextScript,'A'),'\119964')  -- A -> 𝒜
               , ((TextScript,'C'),'\119966')  -- C -> 𝒞
               , ((TextScript,'D'),'\119967')  -- D -> 𝒟
               , ((TextScript,'G'),'\119970')  -- G -> 𝒢
               , ((TextScript,'J'),'\119973')  -- J -> 𝒥
               , ((TextScript,'K'),'\119974')  -- K -> 𝒦
               , ((TextScript,'N'),'\119977')  -- N -> 𝒩
               , ((TextScript,'O'),'\119978')  -- O -> 𝒪
               , ((TextScript,'P'),'\119979')  -- P -> 𝒫
               , ((TextScript,'Q'),'\119980')  -- Q -> 𝒬
               , ((TextScript,'S'),'\119982')  -- S -> 𝒮
               , ((TextScript,'T'),'\119983')  -- T -> 𝒯
               , ((TextScript,'U'),'\119984')  -- U -> 𝒰
               , ((TextScript,'V'),'\119985')  -- V -> 𝒱
               , ((TextScript,'W'),'\119986')  -- W -> 𝒲
               , ((TextScript,'X'),'\119987')  -- X -> 𝒳
               , ((TextScript,'Y'),'\119988')  -- Y -> 𝒴
               , ((TextScript,'Z'),'\119989')  -- Z -> 𝒵
               , ((TextScript,'a'),'\119990')  -- a -> 𝒶
               , ((TextScript,'b'),'\119991')  -- b -> 𝒷
               , ((TextScript,'c'),'\119992')  -- c -> 𝒸
               , ((TextScript,'d'),'\119993')  -- d -> 𝒹
               , ((TextScript,'f'),'\119995')  -- f -> 𝒻
               , ((TextScript,'h'),'\119997')  -- h -> 𝒽
               , ((TextScript,'i'),'\119998')  -- i -> 𝒾
               , ((TextScript,'j'),'\119999')  -- j -> 𝒿
               , ((TextScript,'k'),'\120000')  -- k -> 𝓀
               , ((TextScript,'l'),'\120001')  -- l -> 𝓁
               , ((TextScript,'m'),'\120002')  -- m -> 𝓂
               , ((TextScript,'n'),'\120003')  -- n -> 𝓃
               , ((TextScript,'p'),'\120005')  -- p -> 𝓅
               , ((TextScript,'q'),'\120006')  -- q -> 𝓆
               , ((TextScript,'r'),'\120007')  -- r -> 𝓇
               , ((TextScript,'s'),'\120008')  -- s -> 𝓈
               , ((TextScript,'t'),'\120009')  -- t -> 𝓉
               , ((TextScript,'u'),'\120010')  -- u -> 𝓊
               , ((TextScript,'v'),'\120011')  -- v -> 𝓋
               , ((TextScript,'w'),'\120012')  -- w -> 𝓌
               , ((TextScript,'x'),'\120013')  -- x -> 𝓍
               , ((TextScript,'y'),'\120014')  -- y -> 𝓎
               , ((TextScript,'z'),'\120015')  -- z -> 𝓏
               , ((TextBoldScript,'A'),'\120016')  -- A -> 𝓐
               , ((TextBoldScript,'B'),'\120017')  -- B -> 𝓑
               , ((TextBoldScript,'C'),'\120018')  -- C -> 𝓒
               , ((TextBoldScript,'D'),'\120019')  -- D -> 𝓓
               , ((TextBoldScript,'E'),'\120020')  -- E -> 𝓔
               , ((TextBoldScript,'F'),'\120021')  -- F -> 𝓕
               , ((TextBoldScript,'G'),'\120022')  -- G -> 𝓖
               , ((TextBoldScript,'H'),'\120023')  -- H -> 𝓗
               , ((TextBoldScript,'I'),'\120024')  -- I -> 𝓘
               , ((TextBoldScript,'J'),'\120025')  -- J -> 𝓙
               , ((TextBoldScript,'K'),'\120026')  -- K -> 𝓚
               , ((TextBoldScript,'L'),'\120027')  -- L -> 𝓛
               , ((TextBoldScript,'M'),'\120028')  -- M -> 𝓜
               , ((TextBoldScript,'N'),'\120029')  -- N -> 𝓝
               , ((TextBoldScript,'O'),'\120030')  -- O -> 𝓞
               , ((TextBoldScript,'P'),'\120031')  -- P -> 𝓟
               , ((TextBoldScript,'Q'),'\120032')  -- Q -> 𝓠
               , ((TextBoldScript,'R'),'\120033')  -- R -> 𝓡
               , ((TextBoldScript,'S'),'\120034')  -- S -> 𝓢
               , ((TextBoldScript,'T'),'\120035')  -- T -> 𝓣
               , ((TextBoldScript,'U'),'\120036')  -- U -> 𝓤
               , ((TextBoldScript,'V'),'\120037')  -- V -> 𝓥
               , ((TextBoldScript,'W'),'\120038')  -- W -> 𝓦
               , ((TextBoldScript,'X'),'\120039')  -- X -> 𝓧
               , ((TextBoldScript,'Y'),'\120040')  -- Y -> 𝓨
               , ((TextBoldScript,'Z'),'\120041')  -- Z -> 𝓩
               , ((TextBoldScript,'a'),'\120042')  -- a -> 𝓪
               , ((TextBoldScript,'b'),'\120043')  -- b -> 𝓫
               , ((TextBoldScript,'c'),'\120044')  -- c -> 𝓬
               , ((TextBoldScript,'d'),'\120045')  -- d -> 𝓭
               , ((TextBoldScript,'e'),'\120046')  -- e -> 𝓮
               , ((TextBoldScript,'f'),'\120047')  -- f -> 𝓯
               , ((TextBoldScript,'g'),'\120048')  -- g -> 𝓰
               , ((TextBoldScript,'h'),'\120049')  -- h -> 𝓱
               , ((TextBoldScript,'i'),'\120050')  -- i -> 𝓲
               , ((TextBoldScript,'j'),'\120051')  -- j -> 𝓳
               , ((TextBoldScript,'k'),'\120052')  -- k -> 𝓴
               , ((TextBoldScript,'l'),'\120053')  -- l -> 𝓵
               , ((TextBoldScript,'m'),'\120054')  -- m -> 𝓶
               , ((TextBoldScript,'n'),'\120055')  -- n -> 𝓷
               , ((TextBoldScript,'o'),'\120056')  -- o -> 𝓸
               , ((TextBoldScript,'p'),'\120057')  -- p -> 𝓹
               , ((TextBoldScript,'q'),'\120058')  -- q -> 𝓺
               , ((TextBoldScript,'r'),'\120059')  -- r -> 𝓻
               , ((TextBoldScript,'s'),'\120060')  -- s -> 𝓼
               , ((TextBoldScript,'t'),'\120061')  -- t -> 𝓽
               , ((TextBoldScript,'u'),'\120062')  -- u -> 𝓾
               , ((TextBoldScript,'v'),'\120063')  -- v -> 𝓿
               , ((TextBoldScript,'w'),'\120064')  -- w -> 𝔀
               , ((TextBoldScript,'x'),'\120065')  -- x -> 𝔁
               , ((TextBoldScript,'y'),'\120066')  -- y -> 𝔂
               , ((TextBoldScript,'z'),'\120067')  -- z -> 𝔃
               , ((TextFraktur,'A'),'\120068')  -- A -> 𝔄
               , ((TextFraktur,'B'),'\120069')  -- B -> 𝔅
               , ((TextFraktur,'D'),'\120071')  -- D -> 𝔇
               , ((TextFraktur,'E'),'\120072')  -- E -> 𝔈
               , ((TextFraktur,'F'),'\120073')  -- F -> 𝔉
               , ((TextFraktur,'G'),'\120074')  -- G -> 𝔊
               , ((TextFraktur,'J'),'\120077')  -- J -> 𝔍
               , ((TextFraktur,'K'),'\120078')  -- K -> 𝔎
               , ((TextFraktur,'L'),'\120079')  -- L -> 𝔏
               , ((TextFraktur,'M'),'\120080')  -- M -> 𝔐
               , ((TextFraktur,'N'),'\120081')  -- N -> 𝔑
               , ((TextFraktur,'O'),'\120082')  -- O -> 𝔒
               , ((TextFraktur,'P'),'\120083')  -- P -> 𝔓
               , ((TextFraktur,'Q'),'\120084')  -- Q -> 𝔔
               , ((TextFraktur,'S'),'\120086')  -- S -> 𝔖
               , ((TextFraktur,'T'),'\120087')  -- T -> 𝔗
               , ((TextFraktur,'U'),'\120088')  -- U -> 𝔘
               , ((TextFraktur,'V'),'\120089')  -- V -> 𝔙
               , ((TextFraktur,'W'),'\120090')  -- W -> 𝔚
               , ((TextFraktur,'X'),'\120091')  -- X -> 𝔛
               , ((TextFraktur,'Y'),'\120092')  -- Y -> 𝔜
               , ((TextFraktur,'a'),'\120094')  -- a -> 𝔞
               , ((TextFraktur,'b'),'\120095')  -- b -> 𝔟
               , ((TextFraktur,'c'),'\120096')  -- c -> 𝔠
               , ((TextFraktur,'d'),'\120097')  -- d -> 𝔡
               , ((TextFraktur,'e'),'\120098')  -- e -> 𝔢
               , ((TextFraktur,'f'),'\120099')  -- f -> 𝔣
               , ((TextFraktur,'g'),'\120100')  -- g -> 𝔤
               , ((TextFraktur,'h'),'\120101')  -- h -> 𝔥
               , ((TextFraktur,'i'),'\120102')  -- i -> 𝔦
               , ((TextFraktur,'j'),'\120103')  -- j -> 𝔧
               , ((TextFraktur,'k'),'\120104')  -- k -> 𝔨
               , ((TextFraktur,'l'),'\120105')  -- l -> 𝔩
               , ((TextFraktur,'m'),'\120106')  -- m -> 𝔪
               , ((TextFraktur,'n'),'\120107')  -- n -> 𝔫
               , ((TextFraktur,'o'),'\120108')  -- o -> 𝔬
               , ((TextFraktur,'p'),'\120109')  -- p -> 𝔭
               , ((TextFraktur,'q'),'\120110')  -- q -> 𝔮
               , ((TextFraktur,'r'),'\120111')  -- r -> 𝔯
               , ((TextFraktur,'s'),'\120112')  -- s -> 𝔰
               , ((TextFraktur,'t'),'\120113')  -- t -> 𝔱
               , ((TextFraktur,'u'),'\120114')  -- u -> 𝔲
               , ((TextFraktur,'v'),'\120115')  -- v -> 𝔳
               , ((TextFraktur,'w'),'\120116')  -- w -> 𝔴
               , ((TextFraktur,'x'),'\120117')  -- x -> 𝔵
               , ((TextFraktur,'y'),'\120118')  -- y -> 𝔶
               , ((TextFraktur,'z'),'\120119')  -- z -> 𝔷
               , ((TextDoubleStruck,'A'),'\120120')  -- A -> 𝔸
               , ((TextDoubleStruck,'B'),'\120121')  -- B -> 𝔹
               , ((TextDoubleStruck,'D'),'\120123')  -- D -> 𝔻
               , ((TextDoubleStruck,'E'),'\120124')  -- E -> 𝔼
               , ((TextDoubleStruck,'F'),'\120125')  -- F -> 𝔽
               , ((TextDoubleStruck,'G'),'\120126')  -- G -> 𝔾
               , ((TextDoubleStruck,'I'),'\120128')  -- I -> 𝕀
               , ((TextDoubleStruck,'J'),'\120129')  -- J -> 𝕁
               , ((TextDoubleStruck,'K'),'\120130')  -- K -> 𝕂
               , ((TextDoubleStruck,'L'),'\120131')  -- L -> 𝕃
               , ((TextDoubleStruck,'M'),'\120132')  -- M -> 𝕄
               , ((TextDoubleStruck,'O'),'\120134')  -- O -> 𝕆
               , ((TextDoubleStruck,'S'),'\120138')  -- S -> 𝕊
               , ((TextDoubleStruck,'T'),'\120139')  -- T -> 𝕋
               , ((TextDoubleStruck,'U'),'\120140')  -- U -> 𝕌
               , ((TextDoubleStruck,'V'),'\120141')  -- V -> 𝕍
               , ((TextDoubleStruck,'W'),'\120142')  -- W -> 𝕎
               , ((TextDoubleStruck,'X'),'\120143')  -- X -> 𝕏
               , ((TextDoubleStruck,'Y'),'\120144')  -- Y -> 𝕐
               , ((TextDoubleStruck,'a'),'\120146')  -- a -> 𝕒
               , ((TextDoubleStruck,'b'),'\120147')  -- b -> 𝕓
               , ((TextDoubleStruck,'c'),'\120148')  -- c -> 𝕔
               , ((TextDoubleStruck,'d'),'\120149')  -- d -> 𝕕
               , ((TextDoubleStruck,'e'),'\120150')  -- e -> 𝕖
               , ((TextDoubleStruck,'f'),'\120151')  -- f -> 𝕗
               , ((TextDoubleStruck,'g'),'\120152')  -- g -> 𝕘
               , ((TextDoubleStruck,'h'),'\120153')  -- h -> 𝕙
               , ((TextDoubleStruck,'i'),'\120154')  -- i -> 𝕚
               , ((TextDoubleStruck,'j'),'\120155')  -- j -> 𝕛
               , ((TextDoubleStruck,'k'),'\120156')  -- k -> 𝕜
               , ((TextDoubleStruck,'l'),'\120157')  -- l -> 𝕝
               , ((TextDoubleStruck,'m'),'\120158')  -- m -> 𝕞
               , ((TextDoubleStruck,'n'),'\120159')  -- n -> 𝕟
               , ((TextDoubleStruck,'o'),'\120160')  -- o -> 𝕠
               , ((TextDoubleStruck,'p'),'\120161')  -- p -> 𝕡
               , ((TextDoubleStruck,'q'),'\120162')  -- q -> 𝕢
               , ((TextDoubleStruck,'r'),'\120163')  -- r -> 𝕣
               , ((TextDoubleStruck,'s'),'\120164')  -- s -> 𝕤
               , ((TextDoubleStruck,'t'),'\120165')  -- t -> 𝕥
               , ((TextDoubleStruck,'u'),'\120166')  -- u -> 𝕦
               , ((TextDoubleStruck,'v'),'\120167')  -- v -> 𝕧
               , ((TextDoubleStruck,'w'),'\120168')  -- w -> 𝕨
               , ((TextDoubleStruck,'x'),'\120169')  -- x -> 𝕩
               , ((TextDoubleStruck,'y'),'\120170')  -- y -> 𝕪
               , ((TextDoubleStruck,'z'),'\120171')  -- z -> 𝕫
               , ((TextBoldFraktur,'A'),'\120172')  -- A -> 𝕬
               , ((TextBoldFraktur,'B'),'\120173')  -- B -> 𝕭
               , ((TextBoldFraktur,'C'),'\120174')  -- C -> 𝕮
               , ((TextBoldFraktur,'D'),'\120175')  -- D -> 𝕯
               , ((TextBoldFraktur,'E'),'\120176')  -- E -> 𝕰
               , ((TextBoldFraktur,'F'),'\120177')  -- F -> 𝕱
               , ((TextBoldFraktur,'G'),'\120178')  -- G -> 𝕲
               , ((TextBoldFraktur,'H'),'\120179')  -- H -> 𝕳
               , ((TextBoldFraktur,'I'),'\120180')  -- I -> 𝕴
               , ((TextBoldFraktur,'J'),'\120181')  -- J -> 𝕵
               , ((TextBoldFraktur,'K'),'\120182')  -- K -> 𝕶
               , ((TextBoldFraktur,'L'),'\120183')  -- L -> 𝕷
               , ((TextBoldFraktur,'M'),'\120184')  -- M -> 𝕸
               , ((TextBoldFraktur,'N'),'\120185')  -- N -> 𝕹
               , ((TextBoldFraktur,'O'),'\120186')  -- O -> 𝕺
               , ((TextBoldFraktur,'P'),'\120187')  -- P -> 𝕻
               , ((TextBoldFraktur,'Q'),'\120188')  -- Q -> 𝕼
               , ((TextBoldFraktur,'R'),'\120189')  -- R -> 𝕽
               , ((TextBoldFraktur,'S'),'\120190')  -- S -> 𝕾
               , ((TextBoldFraktur,'T'),'\120191')  -- T -> 𝕿
               , ((TextBoldFraktur,'U'),'\120192')  -- U -> 𝖀
               , ((TextBoldFraktur,'V'),'\120193')  -- V -> 𝖁
               , ((TextBoldFraktur,'W'),'\120194')  -- W -> 𝖂
               , ((TextBoldFraktur,'X'),'\120195')  -- X -> 𝖃
               , ((TextBoldFraktur,'Y'),'\120196')  -- Y -> 𝖄
               , ((TextBoldFraktur,'Z'),'\120197')  -- Z -> 𝖅
               , ((TextBoldFraktur,'a'),'\120198')  -- a -> 𝖆
               , ((TextBoldFraktur,'b'),'\120199')  -- b -> 𝖇
               , ((TextBoldFraktur,'c'),'\120200')  -- c -> 𝖈
               , ((TextBoldFraktur,'d'),'\120201')  -- d -> 𝖉
               , ((TextBoldFraktur,'e'),'\120202')  -- e -> 𝖊
               , ((TextBoldFraktur,'f'),'\120203')  -- f -> 𝖋
               , ((TextBoldFraktur,'g'),'\120204')  -- g -> 𝖌
               , ((TextBoldFraktur,'h'),'\120205')  -- h -> 𝖍
               , ((TextBoldFraktur,'i'),'\120206')  -- i -> 𝖎
               , ((TextBoldFraktur,'j'),'\120207')  -- j -> 𝖏
               , ((TextBoldFraktur,'k'),'\120208')  -- k -> 𝖐
               , ((TextBoldFraktur,'l'),'\120209')  -- l -> 𝖑
               , ((TextBoldFraktur,'m'),'\120210')  -- m -> 𝖒
               , ((TextBoldFraktur,'n'),'\120211')  -- n -> 𝖓
               , ((TextBoldFraktur,'o'),'\120212')  -- o -> 𝖔
               , ((TextBoldFraktur,'p'),'\120213')  -- p -> 𝖕
               , ((TextBoldFraktur,'q'),'\120214')  -- q -> 𝖖
               , ((TextBoldFraktur,'r'),'\120215')  -- r -> 𝖗
               , ((TextBoldFraktur,'s'),'\120216')  -- s -> 𝖘
               , ((TextBoldFraktur,'t'),'\120217')  -- t -> 𝖙
               , ((TextBoldFraktur,'u'),'\120218')  -- u -> 𝖚
               , ((TextBoldFraktur,'v'),'\120219')  -- v -> 𝖛
               , ((TextBoldFraktur,'w'),'\120220')  -- w -> 𝖜
               , ((TextBoldFraktur,'x'),'\120221')  -- x -> 𝖝
               , ((TextBoldFraktur,'y'),'\120222')  -- y -> 𝖞
               , ((TextBoldFraktur,'z'),'\120223')  -- z -> 𝖟
               , ((TextSansSerif,'A'),'\120224')  -- A -> 𝖠
               , ((TextSansSerif,'B'),'\120225')  -- B -> 𝖡
               , ((TextSansSerif,'C'),'\120226')  -- C -> 𝖢
               , ((TextSansSerif,'D'),'\120227')  -- D -> 𝖣
               , ((TextSansSerif,'E'),'\120228')  -- E -> 𝖤
               , ((TextSansSerif,'F'),'\120229')  -- F -> 𝖥
               , ((TextSansSerif,'G'),'\120230')  -- G -> 𝖦
               , ((TextSansSerif,'H'),'\120231')  -- H -> 𝖧
               , ((TextSansSerif,'I'),'\120232')  -- I -> 𝖨
               , ((TextSansSerif,'J'),'\120233')  -- J -> 𝖩
               , ((TextSansSerif,'K'),'\120234')  -- K -> 𝖪
               , ((TextSansSerif,'L'),'\120235')  -- L -> 𝖫
               , ((TextSansSerif,'M'),'\120236')  -- M -> 𝖬
               , ((TextSansSerif,'N'),'\120237')  -- N -> 𝖭
               , ((TextSansSerif,'O'),'\120238')  -- O -> 𝖮
               , ((TextSansSerif,'P'),'\120239')  -- P -> 𝖯
               , ((TextSansSerif,'Q'),'\120240')  -- Q -> 𝖰
               , ((TextSansSerif,'R'),'\120241')  -- R -> 𝖱
               , ((TextSansSerif,'S'),'\120242')  -- S -> 𝖲
               , ((TextSansSerif,'T'),'\120243')  -- T -> 𝖳
               , ((TextSansSerif,'U'),'\120244')  -- U -> 𝖴
               , ((TextSansSerif,'V'),'\120245')  -- V -> 𝖵
               , ((TextSansSerif,'W'),'\120246')  -- W -> 𝖶
               , ((TextSansSerif,'X'),'\120247')  -- X -> 𝖷
               , ((TextSansSerif,'Y'),'\120248')  -- Y -> 𝖸
               , ((TextSansSerif,'Z'),'\120249')  -- Z -> 𝖹
               , ((TextSansSerif,'a'),'\120250')  -- a -> 𝖺
               , ((TextSansSerif,'b'),'\120251')  -- b -> 𝖻
               , ((TextSansSerif,'c'),'\120252')  -- c -> 𝖼
               , ((TextSansSerif,'d'),'\120253')  -- d -> 𝖽
               , ((TextSansSerif,'e'),'\120254')  -- e -> 𝖾
               , ((TextSansSerif,'f'),'\120255')  -- f -> 𝖿
               , ((TextSansSerif,'g'),'\120256')  -- g -> 𝗀
               , ((TextSansSerif,'h'),'\120257')  -- h -> 𝗁
               , ((TextSansSerif,'i'),'\120258')  -- i -> 𝗂
               , ((TextSansSerif,'j'),'\120259')  -- j -> 𝗃
               , ((TextSansSerif,'k'),'\120260')  -- k -> 𝗄
               , ((TextSansSerif,'l'),'\120261')  -- l -> 𝗅
               , ((TextSansSerif,'m'),'\120262')  -- m -> 𝗆
               , ((TextSansSerif,'n'),'\120263')  -- n -> 𝗇
               , ((TextSansSerif,'o'),'\120264')  -- o -> 𝗈
               , ((TextSansSerif,'p'),'\120265')  -- p -> 𝗉
               , ((TextSansSerif,'q'),'\120266')  -- q -> 𝗊
               , ((TextSansSerif,'r'),'\120267')  -- r -> 𝗋
               , ((TextSansSerif,'s'),'\120268')  -- s -> 𝗌
               , ((TextSansSerif,'t'),'\120269')  -- t -> 𝗍
               , ((TextSansSerif,'u'),'\120270')  -- u -> 𝗎
               , ((TextSansSerif,'v'),'\120271')  -- v -> 𝗏
               , ((TextSansSerif,'w'),'\120272')  -- w -> 𝗐
               , ((TextSansSerif,'x'),'\120273')  -- x -> 𝗑
               , ((TextSansSerif,'y'),'\120274')  -- y -> 𝗒
               , ((TextSansSerif,'z'),'\120275')  -- z -> 𝗓
               , ((TextSansSerifBold,'A'),'\120276')  -- A -> 𝗔
               , ((TextSansSerifBold,'B'),'\120277')  -- B -> 𝗕
               , ((TextSansSerifBold,'C'),'\120278')  -- C -> 𝗖
               , ((TextSansSerifBold,'D'),'\120279')  -- D -> 𝗗
               , ((TextSansSerifBold,'E'),'\120280')  -- E -> 𝗘
               , ((TextSansSerifBold,'F'),'\120281')  -- F -> 𝗙
               , ((TextSansSerifBold,'G'),'\120282')  -- G -> 𝗚
               , ((TextSansSerifBold,'H'),'\120283')  -- H -> 𝗛
               , ((TextSansSerifBold,'I'),'\120284')  -- I -> 𝗜
               , ((TextSansSerifBold,'J'),'\120285')  -- J -> 𝗝
               , ((TextSansSerifBold,'K'),'\120286')  -- K -> 𝗞
               , ((TextSansSerifBold,'L'),'\120287')  -- L -> 𝗟
               , ((TextSansSerifBold,'M'),'\120288')  -- M -> 𝗠
               , ((TextSansSerifBold,'N'),'\120289')  -- N -> 𝗡
               , ((TextSansSerifBold,'O'),'\120290')  -- O -> 𝗢
               , ((TextSansSerifBold,'P'),'\120291')  -- P -> 𝗣
               , ((TextSansSerifBold,'Q'),'\120292')  -- Q -> 𝗤
               , ((TextSansSerifBold,'R'),'\120293')  -- R -> 𝗥
               , ((TextSansSerifBold,'S'),'\120294')  -- S -> 𝗦
               , ((TextSansSerifBold,'T'),'\120295')  -- T -> 𝗧
               , ((TextSansSerifBold,'U'),'\120296')  -- U -> 𝗨
               , ((TextSansSerifBold,'V'),'\120297')  -- V -> 𝗩
               , ((TextSansSerifBold,'W'),'\120298')  -- W -> 𝗪
               , ((TextSansSerifBold,'X'),'\120299')  -- X -> 𝗫
               , ((TextSansSerifBold,'Y'),'\120300')  -- Y -> 𝗬
               , ((TextSansSerifBold,'Z'),'\120301')  -- Z -> 𝗭
               , ((TextSansSerifBold,'a'),'\120302')  -- a -> 𝗮
               , ((TextSansSerifBold,'b'),'\120303')  -- b -> 𝗯
               , ((TextSansSerifBold,'c'),'\120304')  -- c -> 𝗰
               , ((TextSansSerifBold,'d'),'\120305')  -- d -> 𝗱
               , ((TextSansSerifBold,'e'),'\120306')  -- e -> 𝗲
               , ((TextSansSerifBold,'f'),'\120307')  -- f -> 𝗳
               , ((TextSansSerifBold,'g'),'\120308')  -- g -> 𝗴
               , ((TextSansSerifBold,'h'),'\120309')  -- h -> 𝗵
               , ((TextSansSerifBold,'i'),'\120310')  -- i -> 𝗶
               , ((TextSansSerifBold,'j'),'\120311')  -- j -> 𝗷
               , ((TextSansSerifBold,'k'),'\120312')  -- k -> 𝗸
               , ((TextSansSerifBold,'l'),'\120313')  -- l -> 𝗹
               , ((TextSansSerifBold,'m'),'\120314')  -- m -> 𝗺
               , ((TextSansSerifBold,'n'),'\120315')  -- n -> 𝗻
               , ((TextSansSerifBold,'o'),'\120316')  -- o -> 𝗼
               , ((TextSansSerifBold,'p'),'\120317')  -- p -> 𝗽
               , ((TextSansSerifBold,'q'),'\120318')  -- q -> 𝗾
               , ((TextSansSerifBold,'r'),'\120319')  -- r -> 𝗿
               , ((TextSansSerifBold,'s'),'\120320')  -- s -> 𝘀
               , ((TextSansSerifBold,'t'),'\120321')  -- t -> 𝘁
               , ((TextSansSerifBold,'u'),'\120322')  -- u -> 𝘂
               , ((TextSansSerifBold,'v'),'\120323')  -- v -> 𝘃
               , ((TextSansSerifBold,'w'),'\120324')  -- w -> 𝘄
               , ((TextSansSerifBold,'x'),'\120325')  -- x -> 𝘅
               , ((TextSansSerifBold,'y'),'\120326')  -- y -> 𝘆
               , ((TextSansSerifBold,'z'),'\120327')  -- z -> 𝘇
               , ((TextSansSerifItalic,'A'),'\120328')  -- A -> 𝘈
               , ((TextSansSerifItalic,'B'),'\120329')  -- B -> 𝘉
               , ((TextSansSerifItalic,'C'),'\120330')  -- C -> 𝘊
               , ((TextSansSerifItalic,'D'),'\120331')  -- D -> 𝘋
               , ((TextSansSerifItalic,'E'),'\120332')  -- E -> 𝘌
               , ((TextSansSerifItalic,'F'),'\120333')  -- F -> 𝘍
               , ((TextSansSerifItalic,'G'),'\120334')  -- G -> 𝘎
               , ((TextSansSerifItalic,'H'),'\120335')  -- H -> 𝘏
               , ((TextSansSerifItalic,'I'),'\120336')  -- I -> 𝘐
               , ((TextSansSerifItalic,'J'),'\120337')  -- J -> 𝘑
               , ((TextSansSerifItalic,'K'),'\120338')  -- K -> 𝘒
               , ((TextSansSerifItalic,'L'),'\120339')  -- L -> 𝘓
               , ((TextSansSerifItalic,'M'),'\120340')  -- M -> 𝘔
               , ((TextSansSerifItalic,'N'),'\120341')  -- N -> 𝘕
               , ((TextSansSerifItalic,'O'),'\120342')  -- O -> 𝘖
               , ((TextSansSerifItalic,'P'),'\120343')  -- P -> 𝘗
               , ((TextSansSerifItalic,'Q'),'\120344')  -- Q -> 𝘘
               , ((TextSansSerifItalic,'R'),'\120345')  -- R -> 𝘙
               , ((TextSansSerifItalic,'S'),'\120346')  -- S -> 𝘚
               , ((TextSansSerifItalic,'T'),'\120347')  -- T -> 𝘛
               , ((TextSansSerifItalic,'U'),'\120348')  -- U -> 𝘜
               , ((TextSansSerifItalic,'V'),'\120349')  -- V -> 𝘝
               , ((TextSansSerifItalic,'W'),'\120350')  -- W -> 𝘞
               , ((TextSansSerifItalic,'X'),'\120351')  -- X -> 𝘟
               , ((TextSansSerifItalic,'Y'),'\120352')  -- Y -> 𝘠
               , ((TextSansSerifItalic,'Z'),'\120353')  -- Z -> 𝘡
               , ((TextSansSerifItalic,'a'),'\120354')  -- a -> 𝘢
               , ((TextSansSerifItalic,'b'),'\120355')  -- b -> 𝘣
               , ((TextSansSerifItalic,'c'),'\120356')  -- c -> 𝘤
               , ((TextSansSerifItalic,'d'),'\120357')  -- d -> 𝘥
               , ((TextSansSerifItalic,'e'),'\120358')  -- e -> 𝘦
               , ((TextSansSerifItalic,'f'),'\120359')  -- f -> 𝘧
               , ((TextSansSerifItalic,'g'),'\120360')  -- g -> 𝘨
               , ((TextSansSerifItalic,'h'),'\120361')  -- h -> 𝘩
               , ((TextSansSerifItalic,'i'),'\120362')  -- i -> 𝘪
               , ((TextSansSerifItalic,'j'),'\120363')  -- j -> 𝘫
               , ((TextSansSerifItalic,'k'),'\120364')  -- k -> 𝘬
               , ((TextSansSerifItalic,'l'),'\120365')  -- l -> 𝘭
               , ((TextSansSerifItalic,'m'),'\120366')  -- m -> 𝘮
               , ((TextSansSerifItalic,'n'),'\120367')  -- n -> 𝘯
               , ((TextSansSerifItalic,'o'),'\120368')  -- o -> 𝘰
               , ((TextSansSerifItalic,'p'),'\120369')  -- p -> 𝘱
               , ((TextSansSerifItalic,'q'),'\120370')  -- q -> 𝘲
               , ((TextSansSerifItalic,'r'),'\120371')  -- r -> 𝘳
               , ((TextSansSerifItalic,'s'),'\120372')  -- s -> 𝘴
               , ((TextSansSerifItalic,'t'),'\120373')  -- t -> 𝘵
               , ((TextSansSerifItalic,'u'),'\120374')  -- u -> 𝘶
               , ((TextSansSerifItalic,'v'),'\120375')  -- v -> 𝘷
               , ((TextSansSerifItalic,'w'),'\120376')  -- w -> 𝘸
               , ((TextSansSerifItalic,'x'),'\120377')  -- x -> 𝘹
               , ((TextSansSerifItalic,'y'),'\120378')  -- y -> 𝘺
               , ((TextSansSerifItalic,'z'),'\120379')  -- z -> 𝘻
               , ((TextSansSerifBoldItalic,'A'),'\120380')  -- A -> 𝘼
               , ((TextSansSerifBoldItalic,'B'),'\120381')  -- B -> 𝘽
               , ((TextSansSerifBoldItalic,'C'),'\120382')  -- C -> 𝘾
               , ((TextSansSerifBoldItalic,'D'),'\120383')  -- D -> 𝘿
               , ((TextSansSerifBoldItalic,'E'),'\120384')  -- E -> 𝙀
               , ((TextSansSerifBoldItalic,'F'),'\120385')  -- F -> 𝙁
               , ((TextSansSerifBoldItalic,'G'),'\120386')  -- G -> 𝙂
               , ((TextSansSerifBoldItalic,'H'),'\120387')  -- H -> 𝙃
               , ((TextSansSerifBoldItalic,'I'),'\120388')  -- I -> 𝙄
               , ((TextSansSerifBoldItalic,'J'),'\120389')  -- J -> 𝙅
               , ((TextSansSerifBoldItalic,'K'),'\120390')  -- K -> 𝙆
               , ((TextSansSerifBoldItalic,'L'),'\120391')  -- L -> 𝙇
               , ((TextSansSerifBoldItalic,'M'),'\120392')  -- M -> 𝙈
               , ((TextSansSerifBoldItalic,'N'),'\120393')  -- N -> 𝙉
               , ((TextSansSerifBoldItalic,'O'),'\120394')  -- O -> 𝙊
               , ((TextSansSerifBoldItalic,'P'),'\120395')  -- P -> 𝙋
               , ((TextSansSerifBoldItalic,'Q'),'\120396')  -- Q -> 𝙌
               , ((TextSansSerifBoldItalic,'R'),'\120397')  -- R -> 𝙍
               , ((TextSansSerifBoldItalic,'S'),'\120398')  -- S -> 𝙎
               , ((TextSansSerifBoldItalic,'T'),'\120399')  -- T -> 𝙏
               , ((TextSansSerifBoldItalic,'U'),'\120400')  -- U -> 𝙐
               , ((TextSansSerifBoldItalic,'V'),'\120401')  -- V -> 𝙑
               , ((TextSansSerifBoldItalic,'W'),'\120402')  -- W -> 𝙒
               , ((TextSansSerifBoldItalic,'X'),'\120403')  -- X -> 𝙓
               , ((TextSansSerifBoldItalic,'Y'),'\120404')  -- Y -> 𝙔
               , ((TextSansSerifBoldItalic,'Z'),'\120405')  -- Z -> 𝙕
               , ((TextSansSerifBoldItalic,'a'),'\120406')  -- a -> 𝙖
               , ((TextSansSerifBoldItalic,'b'),'\120407')  -- b -> 𝙗
               , ((TextSansSerifBoldItalic,'c'),'\120408')  -- c -> 𝙘
               , ((TextSansSerifBoldItalic,'d'),'\120409')  -- d -> 𝙙
               , ((TextSansSerifBoldItalic,'e'),'\120410')  -- e -> 𝙚
               , ((TextSansSerifBoldItalic,'f'),'\120411')  -- f -> 𝙛
               , ((TextSansSerifBoldItalic,'g'),'\120412')  -- g -> 𝙜
               , ((TextSansSerifBoldItalic,'h'),'\120413')  -- h -> 𝙝
               , ((TextSansSerifBoldItalic,'i'),'\120414')  -- i -> 𝙞
               , ((TextSansSerifBoldItalic,'j'),'\120415')  -- j -> 𝙟
               , ((TextSansSerifBoldItalic,'k'),'\120416')  -- k -> 𝙠
               , ((TextSansSerifBoldItalic,'l'),'\120417')  -- l -> 𝙡
               , ((TextSansSerifBoldItalic,'m'),'\120418')  -- m -> 𝙢
               , ((TextSansSerifBoldItalic,'n'),'\120419')  -- n -> 𝙣
               , ((TextSansSerifBoldItalic,'o'),'\120420')  -- o -> 𝙤
               , ((TextSansSerifBoldItalic,'p'),'\120421')  -- p -> 𝙥
               , ((TextSansSerifBoldItalic,'q'),'\120422')  -- q -> 𝙦
               , ((TextSansSerifBoldItalic,'r'),'\120423')  -- r -> 𝙧
               , ((TextSansSerifBoldItalic,'s'),'\120424')  -- s -> 𝙨
               , ((TextSansSerifBoldItalic,'t'),'\120425')  -- t -> 𝙩
               , ((TextSansSerifBoldItalic,'u'),'\120426')  -- u -> 𝙪
               , ((TextSansSerifBoldItalic,'v'),'\120427')  -- v -> 𝙫
               , ((TextSansSerifBoldItalic,'w'),'\120428')  -- w -> 𝙬
               , ((TextSansSerifBoldItalic,'x'),'\120429')  -- x -> 𝙭
               , ((TextSansSerifBoldItalic,'y'),'\120430')  -- y -> 𝙮
               , ((TextSansSerifBoldItalic,'z'),'\120431')  -- z -> 𝙯
               , ((TextMonospace,'A'),'\120432')  -- A -> 𝙰
               , ((TextMonospace,'B'),'\120433')  -- B -> 𝙱
               , ((TextMonospace,'C'),'\120434')  -- C -> 𝙲
               , ((TextMonospace,'D'),'\120435')  -- D -> 𝙳
               , ((TextMonospace,'E'),'\120436')  -- E -> 𝙴
               , ((TextMonospace,'F'),'\120437')  -- F -> 𝙵
               , ((TextMonospace,'G'),'\120438')  -- G -> 𝙶
               , ((TextMonospace,'H'),'\120439')  -- H -> 𝙷
               , ((TextMonospace,'I'),'\120440')  -- I -> 𝙸
               , ((TextMonospace,'J'),'\120441')  -- J -> 𝙹
               , ((TextMonospace,'K'),'\120442')  -- K -> 𝙺
               , ((TextMonospace,'L'),'\120443')  -- L -> 𝙻
               , ((TextMonospace,'M'),'\120444')  -- M -> 𝙼
               , ((TextMonospace,'N'),'\120445')  -- N -> 𝙽
               , ((TextMonospace,'O'),'\120446')  -- O -> 𝙾
               , ((TextMonospace,'P'),'\120447')  -- P -> 𝙿
               , ((TextMonospace,'Q'),'\120448')  -- Q -> 𝚀
               , ((TextMonospace,'R'),'\120449')  -- R -> 𝚁
               , ((TextMonospace,'S'),'\120450')  -- S -> 𝚂
               , ((TextMonospace,'T'),'\120451')  -- T -> 𝚃
               , ((TextMonospace,'U'),'\120452')  -- U -> 𝚄
               , ((TextMonospace,'V'),'\120453')  -- V -> 𝚅
               , ((TextMonospace,'W'),'\120454')  -- W -> 𝚆
               , ((TextMonospace,'X'),'\120455')  -- X -> 𝚇
               , ((TextMonospace,'Y'),'\120456')  -- Y -> 𝚈
               , ((TextMonospace,'Z'),'\120457')  -- Z -> 𝚉
               , ((TextMonospace,'a'),'\120458')  -- a -> 𝚊
               , ((TextMonospace,'b'),'\120459')  -- b -> 𝚋
               , ((TextMonospace,'c'),'\120460')  -- c -> 𝚌
               , ((TextMonospace,'d'),'\120461')  -- d -> 𝚍
               , ((TextMonospace,'e'),'\120462')  -- e -> 𝚎
               , ((TextMonospace,'f'),'\120463')  -- f -> 𝚏
               , ((TextMonospace,'g'),'\120464')  -- g -> 𝚐
               , ((TextMonospace,'h'),'\120465')  -- h -> 𝚑
               , ((TextMonospace,'i'),'\120466')  -- i -> 𝚒
               , ((TextMonospace,'j'),'\120467')  -- j -> 𝚓
               , ((TextMonospace,'k'),'\120468')  -- k -> 𝚔
               , ((TextMonospace,'l'),'\120469')  -- l -> 𝚕
               , ((TextMonospace,'m'),'\120470')  -- m -> 𝚖
               , ((TextMonospace,'n'),'\120471')  -- n -> 𝚗
               , ((TextMonospace,'o'),'\120472')  -- o -> 𝚘
               , ((TextMonospace,'p'),'\120473')  -- p -> 𝚙
               , ((TextMonospace,'q'),'\120474')  -- q -> 𝚚
               , ((TextMonospace,'r'),'\120475')  -- r -> 𝚛
               , ((TextMonospace,'s'),'\120476')  -- s -> 𝚜
               , ((TextMonospace,'t'),'\120477')  -- t -> 𝚝
               , ((TextMonospace,'u'),'\120478')  -- u -> 𝚞
               , ((TextMonospace,'v'),'\120479')  -- v -> 𝚟
               , ((TextMonospace,'w'),'\120480')  -- w -> 𝚠
               , ((TextMonospace,'x'),'\120481')  -- x -> 𝚡
               , ((TextMonospace,'y'),'\120482')  -- y -> 𝚢
               , ((TextMonospace,'z'),'\120483')  -- z -> 𝚣
               , ((TextItalic,'\305'),'\120484')  -- ı -> 𝚤
               , ((TextItalic,'\567'),'\120485')  -- ȷ -> 𝚥
               , ((TextBold,'\913'),'\120488')  -- Α -> 𝚨
               , ((TextBold,'\914'),'\120489')  -- Β -> 𝚩
               , ((TextBold,'\915'),'\120490')  -- Γ -> 𝚪
               , ((TextBold,'\916'),'\120491')  -- Δ -> 𝚫
               , ((TextBold,'\917'),'\120492')  -- Ε -> 𝚬
               , ((TextBold,'\918'),'\120493')  -- Ζ -> 𝚭
               , ((TextBold,'\919'),'\120494')  -- Η -> 𝚮
               , ((TextBold,'\920'),'\120495')  -- Θ -> 𝚯
               , ((TextBold,'\921'),'\120496')  -- Ι -> 𝚰
               , ((TextBold,'\922'),'\120497')  -- Κ -> 𝚱
               , ((TextBold,'\923'),'\120498')  -- Λ -> 𝚲
               , ((TextBold,'\924'),'\120499')  -- Μ -> 𝚳
               , ((TextBold,'\925'),'\120500')  -- Ν -> 𝚴
               , ((TextBold,'\926'),'\120501')  -- Ξ -> 𝚵
               , ((TextBold,'\927'),'\120502')  -- Ο -> 𝚶
               , ((TextBold,'\928'),'\120503')  -- Π -> 𝚷
               , ((TextBold,'\929'),'\120504')  -- Ρ -> 𝚸
               , ((TextBold,'\1012'),'\120505')  -- ϴ -> 𝚹
               , ((TextBold,'\931'),'\120506')  -- Σ -> 𝚺
               , ((TextBold,'\932'),'\120507')  -- Τ -> 𝚻
               , ((TextBold,'\933'),'\120508')  -- Υ -> 𝚼
               , ((TextBold,'\934'),'\120509')  -- Φ -> 𝚽
               , ((TextBold,'\935'),'\120510')  -- Χ -> 𝚾
               , ((TextBold,'\936'),'\120511')  -- Ψ -> 𝚿
               , ((TextBold,'\937'),'\120512')  -- Ω -> 𝛀
               , ((TextBold,'\8711'),'\120513')  -- ∇ -> 𝛁
               , ((TextBold,'\945'),'\120514')  -- α -> 𝛂
               , ((TextBold,'\946'),'\120515')  -- β -> 𝛃
               , ((TextBold,'\947'),'\120516')  -- γ -> 𝛄
               , ((TextBold,'\948'),'\120517')  -- δ -> 𝛅
               , ((TextBold,'\949'),'\120518')  -- ε -> 𝛆
               , ((TextBold,'\950'),'\120519')  -- ζ -> 𝛇
               , ((TextBold,'\951'),'\120520')  -- η -> 𝛈
               , ((TextBold,'\952'),'\120521')  -- θ -> 𝛉
               , ((TextBold,'\953'),'\120522')  -- ι -> 𝛊
               , ((TextBold,'\954'),'\120523')  -- κ -> 𝛋
               , ((TextBold,'\955'),'\120524')  -- λ -> 𝛌
               , ((TextBold,'\956'),'\120525')  -- μ -> 𝛍
               , ((TextBold,'\957'),'\120526')  -- ν -> 𝛎
               , ((TextBold,'\958'),'\120527')  -- ξ -> 𝛏
               , ((TextBold,'\959'),'\120528')  -- ο -> 𝛐
               , ((TextBold,'\960'),'\120529')  -- π -> 𝛑
               , ((TextBold,'\961'),'\120530')  -- ρ -> 𝛒
               , ((TextBold,'\962'),'\120531')  -- ς -> 𝛓
               , ((TextBold,'\963'),'\120532')  -- σ -> 𝛔
               , ((TextBold,'\964'),'\120533')  -- τ -> 𝛕
               , ((TextBold,'\965'),'\120534')  -- υ -> 𝛖
               , ((TextBold,'\966'),'\120535')  -- φ -> 𝛗
               , ((TextBold,'\967'),'\120536')  -- χ -> 𝛘
               , ((TextBold,'\968'),'\120537')  -- ψ -> 𝛙
               , ((TextBold,'\969'),'\120538')  -- ω -> 𝛚
               , ((TextBold,'\8706'),'\120539')  -- ∂ -> 𝛛
               , ((TextBold,'\1013'),'\120540')  -- ϵ -> 𝛜
               , ((TextBold,'\977'),'\120541')  -- ϑ -> 𝛝
               , ((TextBold,'\1008'),'\120542')  -- ϰ -> 𝛞
               , ((TextBold,'\981'),'\120543')  -- ϕ -> 𝛟
               , ((TextBold,'\1009'),'\120544')  -- ϱ -> 𝛠
               , ((TextBold,'\982'),'\120545')  -- ϖ -> 𝛡
               , ((TextItalic,'\913'),'\120546')  -- Α -> 𝛢
               , ((TextItalic,'\914'),'\120547')  -- Β -> 𝛣
               , ((TextItalic,'\915'),'\120548')  -- Γ -> 𝛤
               , ((TextItalic,'\916'),'\120549')  -- Δ -> 𝛥
               , ((TextItalic,'\917'),'\120550')  -- Ε -> 𝛦
               , ((TextItalic,'\918'),'\120551')  -- Ζ -> 𝛧
               , ((TextItalic,'\919'),'\120552')  -- Η -> 𝛨
               , ((TextItalic,'\920'),'\120553')  -- Θ -> 𝛩
               , ((TextItalic,'\921'),'\120554')  -- Ι -> 𝛪
               , ((TextItalic,'\922'),'\120555')  -- Κ -> 𝛫
               , ((TextItalic,'\923'),'\120556')  -- Λ -> 𝛬
               , ((TextItalic,'\924'),'\120557')  -- Μ -> 𝛭
               , ((TextItalic,'\925'),'\120558')  -- Ν -> 𝛮
               , ((TextItalic,'\926'),'\120559')  -- Ξ -> 𝛯
               , ((TextItalic,'\927'),'\120560')  -- Ο -> 𝛰
               , ((TextItalic,'\928'),'\120561')  -- Π -> 𝛱
               , ((TextItalic,'\929'),'\120562')  -- Ρ -> 𝛲
               , ((TextItalic,'\1012'),'\120563')  -- ϴ -> 𝛳
               , ((TextItalic,'\931'),'\120564')  -- Σ -> 𝛴
               , ((TextItalic,'\932'),'\120565')  -- Τ -> 𝛵
               , ((TextItalic,'\933'),'\120566')  -- Υ -> 𝛶
               , ((TextItalic,'\934'),'\120567')  -- Φ -> 𝛷
               , ((TextItalic,'\935'),'\120568')  -- Χ -> 𝛸
               , ((TextItalic,'\936'),'\120569')  -- Ψ -> 𝛹
               , ((TextItalic,'\937'),'\120570')  -- Ω -> 𝛺
               , ((TextItalic,'\8711'),'\120571')  -- ∇ -> 𝛻
               , ((TextItalic,'\945'),'\120572')  -- α -> 𝛼
               , ((TextItalic,'\946'),'\120573')  -- β -> 𝛽
               , ((TextItalic,'\947'),'\120574')  -- γ -> 𝛾
               , ((TextItalic,'\948'),'\120575')  -- δ -> 𝛿
               , ((TextItalic,'\949'),'\120576')  -- ε -> 𝜀
               , ((TextItalic,'\950'),'\120577')  -- ζ -> 𝜁
               , ((TextItalic,'\951'),'\120578')  -- η -> 𝜂
               , ((TextItalic,'\952'),'\120579')  -- θ -> 𝜃
               , ((TextItalic,'\953'),'\120580')  -- ι -> 𝜄
               , ((TextItalic,'\954'),'\120581')  -- κ -> 𝜅
               , ((TextItalic,'\955'),'\120582')  -- λ -> 𝜆
               , ((TextItalic,'\956'),'\120583')  -- μ -> 𝜇
               , ((TextItalic,'\957'),'\120584')  -- ν -> 𝜈
               , ((TextItalic,'\958'),'\120585')  -- ξ -> 𝜉
               , ((TextItalic,'\959'),'\120586')  -- ο -> 𝜊
               , ((TextItalic,'\960'),'\120587')  -- π -> 𝜋
               , ((TextItalic,'\961'),'\120588')  -- ρ -> 𝜌
               , ((TextItalic,'\962'),'\120589')  -- ς -> 𝜍
               , ((TextItalic,'\963'),'\120590')  -- σ -> 𝜎
               , ((TextItalic,'\964'),'\120591')  -- τ -> 𝜏
               , ((TextItalic,'\965'),'\120592')  -- υ -> 𝜐
               , ((TextItalic,'\966'),'\120593')  -- φ -> 𝜑
               , ((TextItalic,'\967'),'\120594')  -- χ -> 𝜒
               , ((TextItalic,'\968'),'\120595')  -- ψ -> 𝜓
               , ((TextItalic,'\969'),'\120596')  -- ω -> 𝜔
               , ((TextItalic,'\8706'),'\120597')  -- ∂ -> 𝜕
               , ((TextItalic,'\1013'),'\120598')  -- ϵ -> 𝜖
               , ((TextItalic,'\977'),'\120599')  -- ϑ -> 𝜗
               , ((TextItalic,'\1008'),'\120600')  -- ϰ -> 𝜘
               , ((TextItalic,'\981'),'\120601')  -- ϕ -> 𝜙
               , ((TextItalic,'\1009'),'\120602')  -- ϱ -> 𝜚
               , ((TextItalic,'\982'),'\120603')  -- ϖ -> 𝜛
               , ((TextBoldItalic,'\913'),'\120604')  -- Α -> 𝜜
               , ((TextBoldItalic,'\914'),'\120605')  -- Β -> 𝜝
               , ((TextBoldItalic,'\915'),'\120606')  -- Γ -> 𝜞
               , ((TextBoldItalic,'\916'),'\120607')  -- Δ -> 𝜟
               , ((TextBoldItalic,'\917'),'\120608')  -- Ε -> 𝜠
               , ((TextBoldItalic,'\918'),'\120609')  -- Ζ -> 𝜡
               , ((TextBoldItalic,'\919'),'\120610')  -- Η -> 𝜢
               , ((TextBoldItalic,'\920'),'\120611')  -- Θ -> 𝜣
               , ((TextBoldItalic,'\921'),'\120612')  -- Ι -> 𝜤
               , ((TextBoldItalic,'\922'),'\120613')  -- Κ -> 𝜥
               , ((TextBoldItalic,'\923'),'\120614')  -- Λ -> 𝜦
               , ((TextBoldItalic,'\924'),'\120615')  -- Μ -> 𝜧
               , ((TextBoldItalic,'\925'),'\120616')  -- Ν -> 𝜨
               , ((TextBoldItalic,'\926'),'\120617')  -- Ξ -> 𝜩
               , ((TextBoldItalic,'\927'),'\120618')  -- Ο -> 𝜪
               , ((TextBoldItalic,'\928'),'\120619')  -- Π -> 𝜫
               , ((TextBoldItalic,'\929'),'\120620')  -- Ρ -> 𝜬
               , ((TextBoldItalic,'\1012'),'\120621')  -- ϴ -> 𝜭
               , ((TextBoldItalic,'\931'),'\120622')  -- Σ -> 𝜮
               , ((TextBoldItalic,'\932'),'\120623')  -- Τ -> 𝜯
               , ((TextBoldItalic,'\933'),'\120624')  -- Υ -> 𝜰
               , ((TextBoldItalic,'\934'),'\120625')  -- Φ -> 𝜱
               , ((TextBoldItalic,'\935'),'\120626')  -- Χ -> 𝜲
               , ((TextBoldItalic,'\936'),'\120627')  -- Ψ -> 𝜳
               , ((TextBoldItalic,'\937'),'\120628')  -- Ω -> 𝜴
               , ((TextBoldItalic,'\8711'),'\120629')  -- ∇ -> 𝜵
               , ((TextBoldItalic,'\945'),'\120630')  -- α -> 𝜶
               , ((TextBoldItalic,'\946'),'\120631')  -- β -> 𝜷
               , ((TextBoldItalic,'\947'),'\120632')  -- γ -> 𝜸
               , ((TextBoldItalic,'\948'),'\120633')  -- δ -> 𝜹
               , ((TextBoldItalic,'\949'),'\120634')  -- ε -> 𝜺
               , ((TextBoldItalic,'\950'),'\120635')  -- ζ -> 𝜻
               , ((TextBoldItalic,'\951'),'\120636')  -- η -> 𝜼
               , ((TextBoldItalic,'\952'),'\120637')  -- θ -> 𝜽
               , ((TextBoldItalic,'\953'),'\120638')  -- ι -> 𝜾
               , ((TextBoldItalic,'\954'),'\120639')  -- κ -> 𝜿
               , ((TextBoldItalic,'\955'),'\120640')  -- λ -> 𝝀
               , ((TextBoldItalic,'\956'),'\120641')  -- μ -> 𝝁
               , ((TextBoldItalic,'\957'),'\120642')  -- ν -> 𝝂
               , ((TextBoldItalic,'\958'),'\120643')  -- ξ -> 𝝃
               , ((TextBoldItalic,'\959'),'\120644')  -- ο -> 𝝄
               , ((TextBoldItalic,'\960'),'\120645')  -- π -> 𝝅
               , ((TextBoldItalic,'\961'),'\120646')  -- ρ -> 𝝆
               , ((TextBoldItalic,'\962'),'\120647')  -- ς -> 𝝇
               , ((TextBoldItalic,'\963'),'\120648')  -- σ -> 𝝈
               , ((TextBoldItalic,'\964'),'\120649')  -- τ -> 𝝉
               , ((TextBoldItalic,'\965'),'\120650')  -- υ -> 𝝊
               , ((TextBoldItalic,'\966'),'\120651')  -- φ -> 𝝋
               , ((TextBoldItalic,'\967'),'\120652')  -- χ -> 𝝌
               , ((TextBoldItalic,'\968'),'\120653')  -- ψ -> 𝝍
               , ((TextBoldItalic,'\969'),'\120654')  -- ω -> 𝝎
               , ((TextBoldItalic,'\8706'),'\120655')  -- ∂ -> 𝝏
               , ((TextBoldItalic,'\1013'),'\120656')  -- ϵ -> 𝝐
               , ((TextBoldItalic,'\977'),'\120657')  -- ϑ -> 𝝑
               , ((TextBoldItalic,'\1008'),'\120658')  -- ϰ -> 𝝒
               , ((TextBoldItalic,'\981'),'\120659')  -- ϕ -> 𝝓
               , ((TextBoldItalic,'\1009'),'\120660')  -- ϱ -> 𝝔
               , ((TextBoldItalic,'\982'),'\120661')  -- ϖ -> 𝝕
               , ((TextSansSerifBold,'\913'),'\120662')  -- Α -> 𝝖
               , ((TextSansSerifBold,'\914'),'\120663')  -- Β -> 𝝗
               , ((TextSansSerifBold,'\915'),'\120664')  -- Γ -> 𝝘
               , ((TextSansSerifBold,'\916'),'\120665')  -- Δ -> 𝝙
               , ((TextSansSerifBold,'\917'),'\120666')  -- Ε -> 𝝚
               , ((TextSansSerifBold,'\918'),'\120667')  -- Ζ -> 𝝛
               , ((TextSansSerifBold,'\919'),'\120668')  -- Η -> 𝝜
               , ((TextSansSerifBold,'\920'),'\120669')  -- Θ -> 𝝝
               , ((TextSansSerifBold,'\921'),'\120670')  -- Ι -> 𝝞
               , ((TextSansSerifBold,'\922'),'\120671')  -- Κ -> 𝝟
               , ((TextSansSerifBold,'\923'),'\120672')  -- Λ -> 𝝠
               , ((TextSansSerifBold,'\924'),'\120673')  -- Μ -> 𝝡
               , ((TextSansSerifBold,'\925'),'\120674')  -- Ν -> 𝝢
               , ((TextSansSerifBold,'\926'),'\120675')  -- Ξ -> 𝝣
               , ((TextSansSerifBold,'\927'),'\120676')  -- Ο -> 𝝤
               , ((TextSansSerifBold,'\928'),'\120677')  -- Π -> 𝝥
               , ((TextSansSerifBold,'\929'),'\120678')  -- Ρ -> 𝝦
               , ((TextSansSerifBold,'\1012'),'\120679')  -- ϴ -> 𝝧
               , ((TextSansSerifBold,'\931'),'\120680')  -- Σ -> 𝝨
               , ((TextSansSerifBold,'\932'),'\120681')  -- Τ -> 𝝩
               , ((TextSansSerifBold,'\933'),'\120682')  -- Υ -> 𝝪
               , ((TextSansSerifBold,'\934'),'\120683')  -- Φ -> 𝝫
               , ((TextSansSerifBold,'\935'),'\120684')  -- Χ -> 𝝬
               , ((TextSansSerifBold,'\936'),'\120685')  -- Ψ -> 𝝭
               , ((TextSansSerifBold,'\937'),'\120686')  -- Ω -> 𝝮
               , ((TextSansSerifBold,'\8711'),'\120687')  -- ∇ -> 𝝯
               , ((TextSansSerifBold,'\945'),'\120688')  -- α -> 𝝰
               , ((TextSansSerifBold,'\946'),'\120689')  -- β -> 𝝱
               , ((TextSansSerifBold,'\947'),'\120690')  -- γ -> 𝝲
               , ((TextSansSerifBold,'\948'),'\120691')  -- δ -> 𝝳
               , ((TextSansSerifBold,'\949'),'\120692')  -- ε -> 𝝴
               , ((TextSansSerifBold,'\950'),'\120693')  -- ζ -> 𝝵
               , ((TextSansSerifBold,'\951'),'\120694')  -- η -> 𝝶
               , ((TextSansSerifBold,'\952'),'\120695')  -- θ -> 𝝷
               , ((TextSansSerifBold,'\953'),'\120696')  -- ι -> 𝝸
               , ((TextSansSerifBold,'\954'),'\120697')  -- κ -> 𝝹
               , ((TextSansSerifBold,'\955'),'\120698')  -- λ -> 𝝺
               , ((TextSansSerifBold,'\956'),'\120699')  -- μ -> 𝝻
               , ((TextSansSerifBold,'\957'),'\120700')  -- ν -> 𝝼
               , ((TextSansSerifBold,'\958'),'\120701')  -- ξ -> 𝝽
               , ((TextSansSerifBold,'\959'),'\120702')  -- ο -> 𝝾
               , ((TextSansSerifBold,'\960'),'\120703')  -- π -> 𝝿
               , ((TextSansSerifBold,'\961'),'\120704')  -- ρ -> 𝞀
               , ((TextSansSerifBold,'\962'),'\120705')  -- ς -> 𝞁
               , ((TextSansSerifBold,'\963'),'\120706')  -- σ -> 𝞂
               , ((TextSansSerifBold,'\964'),'\120707')  -- τ -> 𝞃
               , ((TextSansSerifBold,'\965'),'\120708')  -- υ -> 𝞄
               , ((TextSansSerifBold,'\966'),'\120709')  -- φ -> 𝞅
               , ((TextSansSerifBold,'\967'),'\120710')  -- χ -> 𝞆
               , ((TextSansSerifBold,'\968'),'\120711')  -- ψ -> 𝞇
               , ((TextSansSerifBold,'\969'),'\120712')  -- ω -> 𝞈
               , ((TextSansSerifBold,'\8706'),'\120713')  -- ∂ -> 𝞉
               , ((TextSansSerifBold,'\1013'),'\120714')  -- ϵ -> 𝞊
               , ((TextSansSerifBold,'\977'),'\120715')  -- ϑ -> 𝞋
               , ((TextSansSerifBold,'\1008'),'\120716')  -- ϰ -> 𝞌
               , ((TextSansSerifBold,'\981'),'\120717')  -- ϕ -> 𝞍
               , ((TextSansSerifBold,'\1009'),'\120718')  -- ϱ -> 𝞎
               , ((TextSansSerifBold,'\982'),'\120719')  -- ϖ -> 𝞏
               , ((TextSansSerifBoldItalic,'\913'),'\120720')  -- Α -> 𝞐
               , ((TextSansSerifBoldItalic,'\914'),'\120721')  -- Β -> 𝞑
               , ((TextSansSerifBoldItalic,'\915'),'\120722')  -- Γ -> 𝞒
               , ((TextSansSerifBoldItalic,'\916'),'\120723')  -- Δ -> 𝞓
               , ((TextSansSerifBoldItalic,'\917'),'\120724')  -- Ε -> 𝞔
               , ((TextSansSerifBoldItalic,'\918'),'\120725')  -- Ζ -> 𝞕
               , ((TextSansSerifBoldItalic,'\919'),'\120726')  -- Η -> 𝞖
               , ((TextSansSerifBoldItalic,'\920'),'\120727')  -- Θ -> 𝞗
               , ((TextSansSerifBoldItalic,'\921'),'\120728')  -- Ι -> 𝞘
               , ((TextSansSerifBoldItalic,'\922'),'\120729')  -- Κ -> 𝞙
               , ((TextSansSerifBoldItalic,'\923'),'\120730')  -- Λ -> 𝞚
               , ((TextSansSerifBoldItalic,'\924'),'\120731')  -- Μ -> 𝞛
               , ((TextSansSerifBoldItalic,'\925'),'\120732')  -- Ν -> 𝞜
               , ((TextSansSerifBoldItalic,'\926'),'\120733')  -- Ξ -> 𝞝
               , ((TextSansSerifBoldItalic,'\927'),'\120734')  -- Ο -> 𝞞
               , ((TextSansSerifBoldItalic,'\928'),'\120735')  -- Π -> 𝞟
               , ((TextSansSerifBoldItalic,'\929'),'\120736')  -- Ρ -> 𝞠
               , ((TextSansSerifBoldItalic,'\1012'),'\120737')  -- ϴ -> 𝞡
               , ((TextSansSerifBoldItalic,'\931'),'\120738')  -- Σ -> 𝞢
               , ((TextSansSerifBoldItalic,'\932'),'\120739')  -- Τ -> 𝞣
               , ((TextSansSerifBoldItalic,'\933'),'\120740')  -- Υ -> 𝞤
               , ((TextSansSerifBoldItalic,'\934'),'\120741')  -- Φ -> 𝞥
               , ((TextSansSerifBoldItalic,'\935'),'\120742')  -- Χ -> 𝞦
               , ((TextSansSerifBoldItalic,'\936'),'\120743')  -- Ψ -> 𝞧
               , ((TextSansSerifBoldItalic,'\937'),'\120744')  -- Ω -> 𝞨
               , ((TextSansSerifBoldItalic,'\8711'),'\120745')  -- ∇ -> 𝞩
               , ((TextSansSerifBoldItalic,'\945'),'\120746')  -- α -> 𝞪
               , ((TextSansSerifBoldItalic,'\946'),'\120747')  -- β -> 𝞫
               , ((TextSansSerifBoldItalic,'\947'),'\120748')  -- γ -> 𝞬
               , ((TextSansSerifBoldItalic,'\948'),'\120749')  -- δ -> 𝞭
               , ((TextSansSerifBoldItalic,'\949'),'\120750')  -- ε -> 𝞮
               , ((TextSansSerifBoldItalic,'\950'),'\120751')  -- ζ -> 𝞯
               , ((TextSansSerifBoldItalic,'\951'),'\120752')  -- η -> 𝞰
               , ((TextSansSerifBoldItalic,'\952'),'\120753')  -- θ -> 𝞱
               , ((TextSansSerifBoldItalic,'\953'),'\120754')  -- ι -> 𝞲
               , ((TextSansSerifBoldItalic,'\954'),'\120755')  -- κ -> 𝞳
               , ((TextSansSerifBoldItalic,'\955'),'\120756')  -- λ -> 𝞴
               , ((TextSansSerifBoldItalic,'\956'),'\120757')  -- μ -> 𝞵
               , ((TextSansSerifBoldItalic,'\957'),'\120758')  -- ν -> 𝞶
               , ((TextSansSerifBoldItalic,'\958'),'\120759')  -- ξ -> 𝞷
               , ((TextSansSerifBoldItalic,'\959'),'\120760')  -- ο -> 𝞸
               , ((TextSansSerifBoldItalic,'\960'),'\120761')  -- π -> 𝞹
               , ((TextSansSerifBoldItalic,'\961'),'\120762')  -- ρ -> 𝞺
               , ((TextSansSerifBoldItalic,'\962'),'\120763')  -- ς -> 𝞻
               , ((TextSansSerifBoldItalic,'\963'),'\120764')  -- σ -> 𝞼
               , ((TextSansSerifBoldItalic,'\964'),'\120765')  -- τ -> 𝞽
               , ((TextSansSerifBoldItalic,'\965'),'\120766')  -- υ -> 𝞾
               , ((TextSansSerifBoldItalic,'\966'),'\120767')  -- φ -> 𝞿
               , ((TextSansSerifBoldItalic,'\967'),'\120768')  -- χ -> 𝟀
               , ((TextSansSerifBoldItalic,'\968'),'\120769')  -- ψ -> 𝟁
               , ((TextSansSerifBoldItalic,'\969'),'\120770')  -- ω -> 𝟂
               , ((TextSansSerifBoldItalic,'\8706'),'\120771')  -- ∂ -> 𝟃
               , ((TextSansSerifBoldItalic,'\1013'),'\120772')  -- ϵ -> 𝟄
               , ((TextSansSerifBoldItalic,'\977'),'\120773')  -- ϑ -> 𝟅
               , ((TextSansSerifBoldItalic,'\1008'),'\120774')  -- ϰ -> 𝟆
               , ((TextSansSerifBoldItalic,'\981'),'\120775')  -- ϕ -> 𝟇
               , ((TextSansSerifBoldItalic,'\1009'),'\120776')  -- ϱ -> 𝟈
               , ((TextSansSerifBoldItalic,'\982'),'\120777')  -- ϖ -> 𝟉
               , ((TextBold,'\988'),'\120778')  -- Ϝ -> 𝟊
               , ((TextBold,'\989'),'\120779')  -- ϝ -> 𝟋
               , ((TextBold,'0'),'\120782')  -- 0 -> 𝟎
               , ((TextBold,'1'),'\120783')  -- 1 -> 𝟏
               , ((TextBold,'2'),'\120784')  -- 2 -> 𝟐
               , ((TextBold,'3'),'\120785')  -- 3 -> 𝟑
               , ((TextBold,'4'),'\120786')  -- 4 -> 𝟒
               , ((TextBold,'5'),'\120787')  -- 5 -> 𝟓
               , ((TextBold,'6'),'\120788')  -- 6 -> 𝟔
               , ((TextBold,'7'),'\120789')  -- 7 -> 𝟕
               , ((TextBold,'8'),'\120790')  -- 8 -> 𝟖
               , ((TextBold,'9'),'\120791')  -- 9 -> 𝟗
               , ((TextDoubleStruck,'0'),'\120792')  -- 0 -> 𝟘
               , ((TextDoubleStruck,'1'),'\120793')  -- 1 -> 𝟙
               , ((TextDoubleStruck,'2'),'\120794')  -- 2 -> 𝟚
               , ((TextDoubleStruck,'3'),'\120795')  -- 3 -> 𝟛
               , ((TextDoubleStruck,'4'),'\120796')  -- 4 -> 𝟜
               , ((TextDoubleStruck,'5'),'\120797')  -- 5 -> 𝟝
               , ((TextDoubleStruck,'6'),'\120798')  -- 6 -> 𝟞
               , ((TextDoubleStruck,'7'),'\120799')  -- 7 -> 𝟟
               , ((TextDoubleStruck,'8'),'\120800')  -- 8 -> 𝟠
               , ((TextDoubleStruck,'9'),'\120801')  -- 9 -> 𝟡
               , ((TextSansSerif,'0'),'\120802')  -- 0 -> 𝟢
               , ((TextSansSerif,'1'),'\120803')  -- 1 -> 𝟣
               , ((TextSansSerif,'2'),'\120804')  -- 2 -> 𝟤
               , ((TextSansSerif,'3'),'\120805')  -- 3 -> 𝟥
               , ((TextSansSerif,'4'),'\120806')  -- 4 -> 𝟦
               , ((TextSansSerif,'5'),'\120807')  -- 5 -> 𝟧
               , ((TextSansSerif,'6'),'\120808')  -- 6 -> 𝟨
               , ((TextSansSerif,'7'),'\120809')  -- 7 -> 𝟩
               , ((TextSansSerif,'8'),'\120810')  -- 8 -> 𝟪
               , ((TextSansSerif,'9'),'\120811')  -- 9 -> 𝟫
               , ((TextSansSerifBold,'0'),'\120812')  -- 0 -> 𝟬
               , ((TextSansSerifBold,'1'),'\120813')  -- 1 -> 𝟭
               , ((TextSansSerifBold,'2'),'\120814')  -- 2 -> 𝟮
               , ((TextSansSerifBold,'3'),'\120815')  -- 3 -> 𝟯
               , ((TextSansSerifBold,'4'),'\120816')  -- 4 -> 𝟰
               , ((TextSansSerifBold,'5'),'\120817')  -- 5 -> 𝟱
               , ((TextSansSerifBold,'6'),'\120818')  -- 6 -> 𝟲
               , ((TextSansSerifBold,'7'),'\120819')  -- 7 -> 𝟳
               , ((TextSansSerifBold,'8'),'\120820')  -- 8 -> 𝟴
               , ((TextSansSerifBold,'9'),'\120821')  -- 9 -> 𝟵
               , ((TextMonospace,'0'),'\120822')  -- 0 -> 𝟶
               , ((TextMonospace,'1'),'\120823')  -- 1 -> 𝟷
               , ((TextMonospace,'2'),'\120824')  -- 2 -> 𝟸
               , ((TextMonospace,'3'),'\120825')  -- 3 -> 𝟹
               , ((TextMonospace,'4'),'\120826')  -- 4 -> 𝟺
               , ((TextMonospace,'5'),'\120827')  -- 5 -> 𝟻
               , ((TextMonospace,'6'),'\120828')  -- 6 -> 𝟼
               , ((TextMonospace,'7'),'\120829')  -- 7 -> 𝟽
               , ((TextMonospace,'8'),'\120830')  -- 8 -> 𝟾
               , ((TextMonospace,'9'),'\120831')  -- 9 -> 𝟿
               ]
