use crate::models::ArticleID;
use crate::models::Url;
use crate::schema::enclosures;

#[derive(Identifiable, Queryable, Debug, Insertable, Clone)]
#[diesel(primary_key(article_id, url))]
#[diesel(table_name = enclosures)]
pub struct Enclosure {
    pub article_id: ArticleID,
    pub url: Url,
    pub mime_type: Option<String>,
    pub title: Option<String>,
    pub summary: Option<String>,
    pub thumbnail_url: Option<String>,
    pub filesize: Option<i32>,
    pub width: Option<i32>,
    pub height: Option<i32>,
    pub duration: Option<i32>,
    pub framerate: Option<f64>,
    pub position: Option<f64>,
    pub alternative: Option<Url>,
    pub is_default: bool,
}

impl PartialEq for Enclosure {
    fn eq(&self, other: &Self) -> bool {
        self.article_id == other.article_id && self.url == other.url
    }
}

impl Eq for Enclosure {}

impl Enclosure {
    pub fn is_video(&self) -> bool {
        self.mime_type
            .as_deref()
            .map(|mime| mime.starts_with("video") || mime == "application/x-shockwave-flash")
            .unwrap_or(false)
    }

    pub fn is_image(&self) -> bool {
        self.mime_type.as_deref().map(|mime| mime.starts_with("image")).unwrap_or(false)
    }

    pub fn is_audio(&self) -> bool {
        self.mime_type.as_deref().map(|mime| mime.starts_with("audio")).unwrap_or(false)
    }
}
