/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.root;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Factory;

import java.time.Instant;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * Factory for elements of the events namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class RootFactory {

	private RootFactory() {
	}

	/**
	 * Create a factory for {@link Started} elements.
	 *
	 * @param id   test id (unique within the test run)
	 * @param time time of the event
	 * @param name test name
	 * @return Started factory
	 */
	public static Factory<Started> started(String id, Instant time, String name) {
		return context -> new Started(context).withId(id).withTime(time).withName(name);
	}

	/**
	 * Create a factory for {@link Reported} elements.
	 *
	 * @param id   test id (must match id of previous {@link Started} element)
	 * @param time time of the event
	 * @return Reported factory
	 */
	public static Factory<Reported> reported(String id, Instant time) {
		return context -> new Reported(context).withId(id).withTime(time);
	}

	/**
	 * Create a factory for {@link Finished} elements.
	 *
	 * @param id   test id (must match id of previous {@link Started} element)
	 * @param time time of the event
	 * @return Finished factory
	 */
	public static Factory<Finished> finished(String id, Instant time) {
		return context -> new Finished(context).withId(id).withTime(time);
	}
}
