/* gkrellm-bluez - A BlueZ monitor plug-in for GKrellM2
 *
 * Copyright (C) 2006 Ludovic Cintrat <lcintrat@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gkrellm-bluez.h"

#include "gkrellm-bluez-chart.h"
#include "gkrellm-bluez-linux.h"
#include "gkrellm-bluez-preferences.h"


#define PLUGIN_PLACEMENT MON_NET | MON_INSERT_AFTER

/*  prototypes  */

static void gkrellm_bluez_create (GtkWidget *vbox,
                                  gint       first_create);
static void gkrellm_bluez_update (void);


/*  global variables  */

GkrellmMonitor *gkrellm_bluez;
gint            gkrellm_bluez_style_id;
GtkWidget      *gkrellm_bluez_vbox;
GkrellmTicks   *gkrellm_bluez_ticks;

gchar          *gkrellm_bluez_format_string;

GList          *gkrellm_bluez_monitor_list = NULL;



/*  public function  */

GkrellmMonitor *
gkrellm_init_plugin (void)
{
    static GkrellmMonitor plugin_mon = {
                                           N_("BlueZ Monitor"),             /* name, for config tab                     */
                                           0,                               /* id, 0 if a plugin                        */
                                           gkrellm_bluez_create,            /* the create_plugin() function             */
                                           gkrellm_bluez_update,            /* the update_plugin() function             */
                                           gkrellm_bluez_preferences_show,  /* the create_plugin_tab() config function  */
                                           gkrellm_bluez_preferences_apply, /* the apply_plugin_config() function       */

                                           gkrellm_bluez_preferences_save,  /* the save_plugin_config() function        */
                                           gkrellm_bluez_preferences_load,  /* the load_plugin_config() function        */
                                           GKRELLM_BLUEZ_PLUGIN_KEYWORD,    /* config keyword                           */

                                           NULL,                            /* undefined 2                              */
                                           NULL,                            /* undefined 1                              */
                                           NULL,                            /* undefined 0                              */

                                           PLUGIN_PLACEMENT,                /* insert plugin before this monitor        */

                                           NULL,                            /* handle if a plugin, filled in by GKrellM */
                                           NULL                             /* path if a plugin, filled in by GKrellM   */
                                       };

    gkrellm_bluez_format_string = g_strdup (GKRELLM_BLUEZ_DEFAULT_FORMAT_STRING);
    gkrellm_bluez_ticks         = gkrellm_ticks ();
    gkrellm_bluez_style_id      = gkrellm_lookup_chart_style_id( NET_STYLE_NAME );
    gkrellm_bluez               = &plugin_mon;

    return gkrellm_bluez;
}


/*  private functions  */

gboolean
is_monitor_visible(GkrellmBlueZMonitor *bluezmon)
{
  return ((bluezmon->forced || 
	  (bluezmon->present && bluezmon->updated)) &&
	  bluezmon->enabled);
}

static void
gkrellm_bluez_create (GtkWidget *vbox,
                      gint       first_create)
{
    GkrellmBlueZMonitor *bluezmon;
    GList               *list;

    g_assert (vbox != NULL);

    if (first_create)
    {
      gkrellm_bluez_vbox  = vbox;

      gkrellm_bluez_init();
      gkrellm_bluez_scan_devices();
      gkrellm_bluez_info_read ();
    }

    gkrellm_bluez_load_extra_piximages();

    for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
      bluezmon = list->data;

      gkrellm_bluez_chart_create (bluezmon, gkrellm_bluez_vbox, first_create);

      if (is_monitor_visible(bluezmon))
      {
	gkrellm_bluez_chart_refresh(bluezmon);
      }
      else
      {
	gkrellm_chart_hide(bluezmon->chart, TRUE);
      }
    }
}

static void
gkrellm_bluez_update (void)
{
    GkrellmBlueZMonitor *bluezmon;
    GList               *list;

    static int last_tick = 0;
    static int tick_per_second = 0;


    gkrellm_bluez_info_read ();

    for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
        bluezmon = list->data;

        if (is_monitor_visible(bluezmon))
        {
	  gkrellm_chart_show(bluezmon->chart, TRUE);

	  /* Every second */
	  if (gkrellm_bluez_ticks->second_tick)
          {
	    /* adjust Krell scale */
	    tick_per_second = gkrellm_bluez_ticks->timer_ticks - last_tick;
	    last_tick = gkrellm_bluez_ticks->timer_ticks;
	    
	    if (tick_per_second != 0)
	      gkrellm_set_krell_full_scale (bluezmon->krell, MAX_BITRATE/tick_per_second, 1);

	    /* Chart */
	    gkrellm_bluez_chart_refresh(bluezmon);
	  }

	  /* Panel, LEDs, Krell */
	  gkrellm_bluez_panel_refresh(bluezmon);
        }
        else
        {
	  gkrellm_chart_hide(bluezmon->chart, TRUE);
        }
    }
}
