/*
 * captInitSquare.c --- special treatment when a piece captures as his last move
 *                      a piece that has not moved.
 *
 * Copyright (c) 2002, 2003, 2004 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2004-07-01 12:45:43 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"
#include	"captInitSquare.h"
#include	"pcpj.h"
#include	"distance.h"
#include	"trajet.h"

/*--------------------------------------------------------------------------*/

static piece_t*	is_piece_concerned( piece_t* piece );
static piece_t*	piece_for_index( piece_index_t index );

/*--------------------------------------------------------------------------*/

/*
 * When a piece is captured on its initial square without having the possibility
 * to have moved, and the capture happened on the last move, then the capturer
 * may have to make a switchback to allow other pieces to go through the square.
 */

void
capturer_on_initial_square_must_switchback( int indice )
{
    piece_t*	capturer      = &ListeSures[ indice ];
    piece_t*	capturedPiece = NULL ;
    piece_t*	p ;
    distance_t	distance_table[ 6 * SIZE_PIECE ];
    int		i ;

    while ( indice < NbSures + NbRestantes )
    {
	if ( ( capturedPiece = is_piece_concerned( capturer ) ) != NULL )
	{
	    break ;
	}
	indice++;
	capturer++;
    }

    if ( indice == NbSures + NbRestantes )
    {
	trajet();
	return ;
    }

    CasesInterdites[ capturer->caseArrivee ] = TRUE ;

    distance_create_table( distance_table );
    distance_push( distance_table );

    for ( i = 0, p = ListeSures ; i < NbSures + NbRestantes ; i++, p++ )
    {
	int		indice_destination ;
	piece_type_t	type_piece = p->typePiece ;
	square_t	position_actuelle = p->caseInitiale ;

	if ( p == capturer || p == capturedPiece )
	{
	    continue ;
	}

	if ( capturedPiece->castling != NO_CASTLING
	     && p->camp == capturedPiece->camp
	     && ( ( p->typePiece == ROI )
		  || ( p->typePiece == FOU
		       && capturedPiece->castling == KING_SIDE )
		  || ( p->typePiece == DAME
		       && capturedPiece->castling == QUEEN_SIDE ) ) )
	{
	    indice_destination = 1 ;
	    position_actuelle = p->destination[ 0 ];
	}
	else
	{
	    indice_destination = 0 ;
	}

	while ( indice_destination < p->nbDestination )
	{
	    if ( p->casePromotion != CASE_PAS_DE_PROMOTION
		 && position_actuelle == p->casePromotion )
	    {
		/* If the piece returns on it's promotion square, this code is
		 * still correct. */
		type_piece = p->typePromotion ;
	    }

	    if ( capturer->destination[ indice_destination ]
		 != CASE_ARRIVEE_QUELCONQUE
		 && position_actuelle != CASE_ARRIVEE_QUELCONQUE )
	    {
		distance_t delta = distance_delta(
		    type_piece,
		    p->camp,
		    position_actuelle,
		    p->destination[ indice_destination ]);
		if ( delta > 0
		     && ((p->camp == BLANC && delta > NbCoupsBlancsRestants)||
			 (p->camp == NOIR  && delta > NbCoupsNoirsRestants )))
		{
		    distance_pop();
		    CasesInterdites[ capturer->caseArrivee ] = FALSE ;

		    /* capturer OR capturedPiece must make a switchback */
		    if ( NbCoupsBlancsRestants < 2 && NbCoupsNoirsRestants < 2 )
		    {
			/* Nobody can make the switchback */
		    }
		    else if ( NbCoupsBlancsRestants >= 2
			      && NbCoupsNoirsRestants >= 2 )
		    {
			/* Both pieces can make the switchback */
			capturer_on_initial_square_must_switchback(
			    indice + 1 );
		    }
		    else if ( ( NbCoupsBlancsRestants >= 2
				&& capturer->camp == BLANC )
			      || ( NbCoupsNoirsRestants >= 2
				   && capturer->camp == NOIR ) )
		    {
			/* capturer makes switchback */
			int j = capturer->nbDestination ;
			capturer->destination[ j ]   = CASE_ARRIVEE_QUELCONQUE ;
			capturer->distance[ j ]        = 1 ;
			capturer->pieceCapturee[ j ]   = PAS_DE_CAPTURE ;
			capturer->destination[ j + 1 ] = capturer->caseArrivee ;
			capturer->distance[ j + 1 ]    = 1 ;
			capturer->pieceCapturee[ j + 1 ] = PAS_DE_CAPTURE ;
			capturer->nbDestination        += 2 ;

			if ( capturer->camp == BLANC )
			{
			    NbCoupsBlancsRestants -= 2 ;
			}
			else
			{
			    NbCoupsNoirsRestants  -= 2 ;
			}

			capturer_on_initial_square_must_switchback(
			    indice + 1 );

			if ( capturer->camp == BLANC )
			{
			    NbCoupsBlancsRestants += 2 ;
			}
			else
			{
			    NbCoupsNoirsRestants  += 2 ;
			}

			capturer->nbDestination  -= 2;
		    }
		    else
		    {
			/* capturedPiece makes switchback */
			capturedPiece->destination[ 0 ]=CASE_ARRIVEE_QUELCONQUE;
			capturedPiece->distance[ 0 ]   = 1 ;
			capturedPiece->destination[ 1 ]= capturer->caseArrivee ;
			capturedPiece->distance[ 1 ]      = 1 ;
			capturedPiece->pieceCapturee[ 1 ] = PAS_DE_CAPTURE ;
			capturedPiece->nbDestination      = 2 ;

			if ( capturedPiece->camp == BLANC )
			{
			    NbCoupsBlancsRestants -= 2 ;
			}
			else
			{
			    NbCoupsNoirsRestants  -= 2 ;
			}

			capturer_on_initial_square_must_switchback(
			    indice + 1 );

			if ( capturedPiece->camp == BLANC )
			{
			    NbCoupsBlancsRestants += 2 ;
			}
			else
			{
			    NbCoupsNoirsRestants  += 2 ;
			}

			capturedPiece->destination[ 0 ] =
			    capturedPiece->caseInitiale ;
			capturedPiece->distance[ 0 ]    = 0 ;
			capturedPiece->nbDestination  	= 1;
		    }
		    return ;
		}
	    }
	    position_actuelle = p->destination[ indice_destination ];
	    indice_destination++;
	}
    }

    distance_pop();
    CasesInterdites[ capturer->caseArrivee ] = FALSE ;

    capturer_on_initial_square_must_switchback( indice + 1 );
}

/*--------------------------------------------------------------------------*/

static piece_t*
is_piece_concerned( piece_t* piece )
{
    piece_index_t index = piece->pieceCapturee[ piece->nbDestination - 1 ];

    if ( piece->pieceCapturante == PIECE_PAS_CAPTUREE
	 && index != PAS_DE_CAPTURE )
    {
	piece_t* p = piece_for_index( index );
	/* The capture of a Rook having only made a castling is not
	 * considered. */
	if ( p->distance[ 0 ] == 0
	     && ( p->typePiece != TOUR || p->castling == NO_CASTLING ) )
	{
	    return p;
	}
    }
    return NULL ;
}

static piece_t*
piece_for_index( piece_index_t index )
{
    int		i = 0 ;
    piece_t*	p = ListeSures ;

    for (; i < NbSures + NbRestantes ; i++, p++ )
    {
	if ( p->index == index )
	{
	    return p ;
	}
    }
    return NULL ;
}
