/*
 * pcpjcoup.c --- search legal moves and other tools.
 *
 * Copyright (c) 1997-2002, 2005 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2005-02-18 14:08:08 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#include	"pcpjcoup.h"
#include	"pcpj.h"
#include	"pcpjtool.h"

#include	<string.h>	/* memcpy */

static	UCHAR*		CoupsListePtr = CoupsListe;

static	bool_t		Prise_En_Passant_Prochain_Coup = FALSE;
static	square_t	Prise_En_Passant_Case_Depart;
static	square_t	Prise_En_Passant_Case_Capture;

void
coupsListePrint()
{
    unsigned int	i ;
    unsigned int	num_coup = 0 ;

    for (i=0; i<CoupsNb; i++)
    {
	if ((i & 1) == 0)
	{
	    if (num_coup && (num_coup % 3 == 0))	fputc('\n', MainFD);
	    num_coup++;
	    fprintf(MainFD, "%d.", num_coup);
	}
	printCoup(&CoupsListe[i*6]);
    }
    fputc('\n', MainFD);
}

void
joueCoup(
    exploration_t*	datas,
    const UCHAR*	coup)
{
    piece_index_t	c;
    piece_index_t	piece_capturee;

    memcpy(CoupsListePtr, coup, SIZE_COUP);
    CoupsListePtr += SIZE_COUP;
    CoupsNb++;

    c = datas->board[ coup[ CASE_DEPART ] ];
    datas->board[ coup[ CASE_DEPART ] ] = INDEX_CASE_VIDE ;
    if ( ( piece_capturee = coup[ PIECE_CAPTUREE ] ) != PAS_DE_CAPTURE )
    {
	datas->board[ coup[ CASE_CAPTURE ] ] = INDEX_CASE_VIDE ;
	datas->pieces_squares[ piece_capturee ] = CASE_D_UNE_PIECE_CAPTUREE;
	if ( datas->pieces[ piece_capturee ].camp == BLANC )
	{
	    datas->white_pieces_to_capture-- ;
	}
	else
	{
	    datas->black_pieces_to_capture-- ;
	}
    }
    else if (coup[CASE_CAPTURE])
    {
	/* Si aucune piece n'a ete capturee mais la case
	 * de capture n'est pas vide, alors c'est le prelude
	 * a une prise en passant. La case de capture donne
	 * alors la case de depart du pion qui prend en passant.
	 */
	Prise_En_Passant_Prochain_Coup = TRUE;
	Prise_En_Passant_Case_Capture  = coup[CASE_ARRIVEE];
	Prise_En_Passant_Case_Depart   = coup[CASE_CAPTURE];
    }

    datas->board[ coup[ CASE_ARRIVEE ] ] = c ;

    datas->pieces_squares[ c ] = coup[ CASE_ARRIVEE ];
    datas->pieces_types  [ c ] = coup[ TYPE_ARRIVEE ];

    /* Cas particulier des roques */
    if (coup[TYPE_DEPART] == ROI)
    {
	if (coup[CASE_DEPART] == e1)
	{
	    if (coup[CASE_ARRIVEE] == g1)
	    {
		c = datas->board[ h1 ];
		datas->board[ h1 ] = INDEX_CASE_VIDE ;
		datas->board[ f1 ] = c ;
		datas->pieces_squares[ c ] = f1 ;
		datas->pieces[ c ].castling = NO_CASTLING ;
		White_king->castling = NO_CASTLING ;
	    }
	    else if (coup[CASE_ARRIVEE] == c1)
	    {
		c = datas->board[ a1 ];
		datas->board[ a1 ] = INDEX_CASE_VIDE ;
		datas->board[ d1 ] = c ;
		datas->pieces_squares[c] = d1;
		datas->pieces[ c ].castling = NO_CASTLING ;
		White_king->castling = NO_CASTLING ;
	    }
	}
	else if (coup[CASE_DEPART] == e8)
	{
	    if (coup[CASE_ARRIVEE] == g8)
	    {
		c = datas->board[ h8 ];
		datas->board[ h8 ] = INDEX_CASE_VIDE ;
		datas->board[ f8 ] = c ;
		datas->pieces_squares[ c ] = f8 ;
		datas->pieces[ c ].castling = NO_CASTLING ;
		Black_king->castling = NO_CASTLING ;
	    }
	    else if (coup[CASE_ARRIVEE] == c8)
	    {
		c = datas->board[ a8 ];
		datas->board[ a8 ] = INDEX_CASE_VIDE ;
		datas->board[ d8 ] = c ;
		datas->pieces_squares[ c ] = d8 ;
		datas->pieces[ c ].castling = NO_CASTLING ;
		Black_king->castling = NO_CASTLING ;
	    }
	}
    }
}

void
dejoueCoup(
    exploration_t*	datas,
    const UCHAR*	coup)
{
    piece_index_t	c;

    CoupsListePtr -= SIZE_COUP;
    CoupsNb--;

    c = datas->board[ coup[ CASE_ARRIVEE ] ];
    datas->board[ coup[ CASE_ARRIVEE ] ] = INDEX_CASE_VIDE ;
    datas->board[ coup[ CASE_DEPART  ] ] = c ;

    datas->pieces_squares[ c ] = coup[ CASE_DEPART ];
    datas->pieces_types  [ c ] = coup[ TYPE_DEPART ];

    if ( ( c = coup[ PIECE_CAPTUREE ] ) != PAS_DE_CAPTURE )
    {
	datas->pieces_squares[ c ] = coup[ CASE_CAPTURE ];
	datas->board[ coup[ CASE_CAPTURE ] ] = c ;
	if ( datas->pieces[ c ].camp == BLANC )
	{
	    datas->white_pieces_to_capture++ ;
	}
	else
	{
	    datas->black_pieces_to_capture++ ;
	}
    }

    /* Cas particulier des roques */
    if (coup[TYPE_DEPART] == ROI)
    {
	if (coup[CASE_DEPART] == e1)
	{
	    if (coup[CASE_ARRIVEE] == g1)
	    {
		c = datas->board[ f1 ];
		datas->board[ f1 ] = INDEX_CASE_VIDE ;
		datas->board[ h1 ] = c ;
		datas->pieces_squares[ c ] = h1 ;
		datas->pieces[ c ].castling = KING_SIDE ;
		White_king->castling = KING_SIDE ;
	    }
	    else if (coup[CASE_ARRIVEE] == c1)
	    {
		c = datas->board[ d1 ];
		datas->board[ d1 ] = INDEX_CASE_VIDE ;
		datas->board[ a1 ] = c ;
		datas->pieces_squares[ c ] = a1 ;
		datas->pieces[ c ].castling = QUEEN_SIDE ;
		White_king->castling = QUEEN_SIDE ;
	    }
	}
	else if (coup[CASE_DEPART] == e8)
	{
	    if (coup[CASE_ARRIVEE] == g8)
	    {
		c = datas->board[ f8 ];
		datas->board[ f8 ] = INDEX_CASE_VIDE ;
		datas->board[ h8 ] = c ;
		datas->pieces_squares[ c ] = h8 ;
		datas->pieces[ c ].castling = KING_SIDE ;
		Black_king->castling = KING_SIDE ;
	    }
	    else if (coup[CASE_ARRIVEE] == c8)
	    {
		c = datas->board[ d8 ];
		datas->board[ d8 ] = INDEX_CASE_VIDE ;
		datas->board[ a8 ] = c ;
		datas->pieces_squares[ c ] = a8 ;
		datas->pieces[ c ].castling = QUEEN_SIDE ;
		Black_king->castling = QUEEN_SIDE ;
	    }
	}
    }
}

static	char	listeRoi[ 8 ]      = { -17, +1,  +1, +14,  +2, +14,  +1, +1 };
static	char	listeCavalier[ 8 ] = { -33, +2, +13,  +4, +28,  +4, +13, +2 };
static	char	listeFou[ 4 ]      = { -17, -15, +15, +17 };
static	char	listeTour[ 4 ]     = {  -1,  +1, -16, +16 };

int
listeCoups( const exploration_t*	datas,
	    colour_t			camp,
	    UCHAR*			liste)
{
    piece_index_t	index, index2;
    unsigned int	nbCoupsRestants;
    char*		ptr_liste;
    unsigned int	index_liste ;
    square_delta_t	delta;
    char		deltaPion;	/* Doit reste signe car sinon
					 * field+deltaPion devient trop grand.*/
    row_t		rangee_depart;
    row_t		rangee_promotion;
    square_t		posRoiEnnemi;
    int			nbCoups = 0;	/* Must be signed 'cause of ILLEGAL */
    UCHAR*		liste_debut = liste;

    if (camp == BLANC)
    {
	nbCoupsRestants       = datas->white_spare_moves ;
	rangee_depart         = 0x10;
	rangee_promotion      = 0x60;
	deltaPion             = 16;
	posRoiEnnemi          = datas->pieces_squares[ Black_king->index ];
    }
    else
    {
	nbCoupsRestants       = datas->black_spare_moves ;
	rangee_depart         = 0x60;
	rangee_promotion      = 0x10;
	deltaPion             = -16;
	posRoiEnnemi          = datas->pieces_squares[ White_king->index ];
    }

    if ( case_est_controlee_par_camp( datas, posRoiEnnemi, camp ) )
    {
	Prise_En_Passant_Prochain_Coup = FALSE ;
	return ILLEGAL ;
    }

    if ( Prise_En_Passant_Prochain_Coup )
    {
	liste_debut[CASE_DEPART]    = Prise_En_Passant_Case_Depart;
	liste_debut[CASE_ARRIVEE]   =
	    ( Prise_En_Passant_Case_Depart >= 0x3F ) /* Noir ou Blanche ?   */
	    ? Prise_En_Passant_Case_Capture + 0x10   /* Pb prend en passant */
	    : Prise_En_Passant_Case_Capture - 0x10 ; /* Pn prend en passant */
	liste_debut[TYPE_DEPART]    = PION;
	liste_debut[PIECE_CAPTUREE] = datas->board[Prise_En_Passant_Case_Capture];
	liste_debut[CASE_CAPTURE]   = Prise_En_Passant_Case_Capture;
	liste_debut[TYPE_ARRIVEE]   = PION;

	Prise_En_Passant_Prochain_Coup = FALSE ;
	return 1 ;
    }

    for ( index = INDEX_ROI_BLANC; index <= INDEX_PION_NOIR_H; index++ )
    {
	const piece_t*	p = &( datas->pieces[ index ] );
	square_t	depart = datas->pieces_squares[ index ];
	square_t	field ;
	piece_type_t	type ;
	piece_index_t	indexPieceCapturee ;
	const piece_t*	pieceCapturee = NULL;
	square_t	nextDestination ;

	if ( p->camp != camp || depart == CASE_D_UNE_PIECE_CAPTUREE )
	{
	    /* La piece a ete capturee ... */
	    continue;
	}

	field = depart ;
	type = datas->pieces_types[ index ];

	indexPieceCapturee = p->pieceCapturee[ p->indDestAct ];
	nextDestination    = p->destination[ p->indDestAct ];
	if ( indexPieceCapturee != PAS_DE_CAPTURE )
	{
	    pieceCapturee = &( datas->pieces[ indexPieceCapturee ] );
	}

	switch(type)
	{
	case ROI:
	    if ( p->castling == KING_SIDE )
	    {
		/* Teste si la case d'arrivee est vide */
		if ( camp == BLANC )
		{
		    if ( datas->board[ f1 ] == INDEX_CASE_VIDE &&
			 datas->board[ g1 ] == INDEX_CASE_VIDE &&
			 (! case_est_controlee_par_camp( datas, e1, NOIR ) ) &&
			 (! case_est_controlee_par_camp( datas, f1, NOIR ) ) )
		    {
			liste[CASE_DEPART]    = e1;
			liste[CASE_ARRIVEE]   = g1;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = ROI;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
			liste += 6;
			nbCoups++;
		    }
		}
		else
		{
		    if ( datas->board[ f8 ] == INDEX_CASE_VIDE &&
			 datas->board[ g8 ] == INDEX_CASE_VIDE &&
			 (! case_est_controlee_par_camp( datas, e8, BLANC ) ) &&
			 (! case_est_controlee_par_camp( datas, f8, BLANC ) ) )
		    {
			liste[CASE_DEPART]    = e8;
			liste[CASE_ARRIVEE]   = g8;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = ROI;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
			liste += 6;
			nbCoups++;
		    }
		}
		continue;
	    }
	    else if ( p->castling == QUEEN_SIDE )
	    {
		/* Teste si la case d'arrivee est vide */
		if (camp == BLANC)
		{
		    if ( datas->board[ d1 ] == INDEX_CASE_VIDE &&
			 datas->board[ c1 ] == INDEX_CASE_VIDE &&
			 datas->board[ b1 ] == INDEX_CASE_VIDE &&
			 (! case_est_controlee_par_camp( datas, e1, NOIR ) ) &&
			 (! case_est_controlee_par_camp( datas, d1, NOIR ) ) )
		    {
			liste[CASE_DEPART]    = e1;
			liste[CASE_ARRIVEE]   = c1;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = ROI;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
			liste += 6;
			nbCoups++;
		    }
		}
		else
		{
		    if ( datas->board[ d8 ] == INDEX_CASE_VIDE &&
			 datas->board[ c8 ] == INDEX_CASE_VIDE &&
			 datas->board[ b8 ] == INDEX_CASE_VIDE &&
			 (! case_est_controlee_par_camp( datas, e8, BLANC ) ) &&
			 (! case_est_controlee_par_camp( datas, d8, BLANC ) ) )
		    {
			liste[CASE_DEPART]    = e8;
			liste[CASE_ARRIVEE]   = c8;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = ROI;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
			liste += 6;
			nbCoups++;
		    }
		}
		continue;
	    }
	    else	ptr_liste = listeRoi;
	    goto CavalierRoi;	/* Pour optimisation */

	case CAVALIER:
	    ptr_liste = listeCavalier;
	CavalierRoi:
	    /* Si la piece est sur sa case d'arrivee et qu'il reste
	     * moins de 2 coups de rab.
	     */
	    if (nbCoupsRestants < 2 && depart == p->caseArrivee) continue;

	    for ( index_liste = 0; index_liste < 8; index_liste++ )
	    {
		field += ptr_liste[ index_liste ];
		if (! CasesInterdites[field])
		{
		    index2 = datas->board[field];

		    if ( index2 != INDEX_CASE_VIDE )
		    {
			/* Si on capture une piece qui doit rester sur
			 * l'echiquier ou etre capturee par qq d'autre. */
			if (index2 != indexPieceCapturee)	continue;

			/* Si on capture une tour qui doit d'abord roquer,
			 * c'est interdit pour le moment. */
			if ( datas->pieces[ index2 ].typePiece == TOUR &&
			     datas->pieces[ index2 ].castling  != NO_CASTLING )
			    continue;

			liste[PIECE_CAPTUREE] = index2;
			liste[CASE_CAPTURE]   = field;
		    }
		    else
		    {
			/* If the piece must capture on this square but doesn't
			 * and there are not enough spare moves to make a
			 * switchback, and the capture is not yet done.
			 */
			if ( nextDestination == field
			     && indexPieceCapturee != PAS_DE_CAPTURE
			     && nbCoupsRestants < 2
			     && pieceCapturee->indDestAct
			     != pieceCapturee->nbDestination - 1 )
			{
			    continue ;
			}

			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
		    }
		    liste[CASE_DEPART]  = depart;
		    liste[CASE_ARRIVEE] = field;
		    liste[TYPE_DEPART]  = liste[TYPE_ARRIVEE] = type;
		    liste += 6;
		    nbCoups++;
		}
	    }
	    break;

	case TOUR:
	    /* Teste si la tour doit d'abord roquer. */
	    if ( p->castling != NO_CASTLING )
	    {
		continue ;
	    }

	case DAME:
	    /* Si la piece est sur sa case d'arrivee et qu'il reste moins
	     * de 2 coups de rab.
	     */
	    if ( nbCoupsRestants < 2 && depart == p->caseArrivee )
	    {
		continue ;
	    }

	    for ( index_liste = 0; index_liste < 4; index_liste++ )
	    {
		delta = listeTour[ index_liste ];
		for (field=depart+delta; CASE_OK(field); field+=delta)
		{
		    index2 = datas->board[field];
		    if ( index2 != INDEX_CASE_VIDE )
		    {
			/* Si on capture une piece qui doit rester sur
			 * l'echiquier ou etre capturee par qq d'autre. */
			if (index2 != indexPieceCapturee)	break;

			/* Si on capture une tour qui doit d'abord roquer */
			if ( datas->pieces[ index2 ].typePiece == TOUR &&
			     datas->pieces[ index2 ].castling  != NO_CASTLING )
			    break ;

			liste[CASE_DEPART]    = depart;
			liste[CASE_ARRIVEE]   = field;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = type;
			liste[PIECE_CAPTUREE] = index2;
			liste[CASE_CAPTURE]   = field;
			nbCoups++;
			liste += 6;
			break;
		    }
		    else
		    {
			/* If the piece must capture on this square but doesn't
			 * and there are not enough spare moves to make a
			 * switchback, and the capture is not yet done.
			 */
			if ( nextDestination == field
			     && indexPieceCapturee != PAS_DE_CAPTURE
			     && nbCoupsRestants < 2
			     && pieceCapturee->indDestAct
			     != pieceCapturee->nbDestination - 1 )
			{
			    continue ;
			}

			liste[CASE_DEPART]    = depart;
			liste[CASE_ARRIVEE]   = field;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = type;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
			nbCoups++;
			liste += 6;
		    }
		} /* Boucle tant qu'on reste sur l'echiquier */
	    } /* Boucle sur les 4 directions */

	    if (type == DAME)	goto FouDame;

	    break;

	case FOU:
	    /* Si la piece est sur sa case d'arrivee et qu'il reste moins
	     * de 2 coups de rab.
	     */
	    if (nbCoupsRestants < 2 && depart == p->caseArrivee )
	    {
		continue;
	    }

	FouDame:
	    for ( index_liste = 0; index_liste < 4; index_liste++ )
	    {
		delta = listeFou[ index_liste ];
		for (field=depart+delta; CASE_OK(field); field+=delta)
		{
		    index2 = datas->board[field];
		    if ( index2 != INDEX_CASE_VIDE )
		    {
			/* Si on capture une piece qui doit rester sur
			 * l'datas->board ou etre capturee par qq d'autre. */
			if (index2 != indexPieceCapturee)	break;

			/* Si on capture une tour qui doit d'abord roquer */
			if ( datas->pieces[ index2 ].typePiece == TOUR &&
			     datas->pieces[ index2 ].castling  != NO_CASTLING )
			    break ;

			liste[CASE_DEPART]    = depart;
			liste[CASE_ARRIVEE]   = field;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = type;
			liste[PIECE_CAPTUREE] = index2;
			liste[CASE_CAPTURE]   = field;
			nbCoups++;
			liste += 6;
			break;
		    }
		    else
		    {
			/* If the piece must capture on this square but doesn't
			 * and there are not enough spare moves to make a
			 * switchback, and the capture is not yet done.
			 */
			if ( nextDestination == field
			     && indexPieceCapturee != PAS_DE_CAPTURE
			     && nbCoupsRestants < 2
			     && pieceCapturee->indDestAct
			     != pieceCapturee->nbDestination - 1 )
			{
			    continue ;
			}

			liste[CASE_DEPART]    = depart;
			liste[CASE_ARRIVEE]   = field;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = type;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;
			nbCoups++;
			liste += 6;
		    }
		}
	    }
	    break;

	case PION:
	    /* Si le pion ne va pas a promotion, il lui est interdit de
	     * depasser sa case d'arrivee.
	     */
	    if ( p->casePromotion == CASE_PAS_DE_PROMOTION
		 && depart == p->caseArrivee)
	    {
		continue;
	    }

	    field += deltaPion;

	    if ( datas->board[ field ] == INDEX_CASE_VIDE )
	    {
		liste[CASE_DEPART]    = depart;
		liste[CASE_ARRIVEE]   = field;
		liste[TYPE_DEPART]    = PION;
		liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
		liste[CASE_CAPTURE]   = 0;
		if (row(depart) == rangee_promotion)
		{
		    if (p->casePromotion == field)
		    {
			liste[TYPE_ARRIVEE] = p->typePromotion;
			liste += 6;
			nbCoups++;
		    }
		}
		else
		{
		    liste[TYPE_ARRIVEE] = type;
		    nbCoups++;
		    liste += 6;

		    /* Eventuel double pas */
		    if ( row( depart ) == rangee_depart &&
			 datas->board[ field + deltaPion ] ==  INDEX_CASE_VIDE )
		    {
			liste[CASE_DEPART]    = depart;
			liste[CASE_ARRIVEE]   = field + deltaPion;
			liste[TYPE_DEPART]    = liste[TYPE_ARRIVEE] = PION;
			liste[PIECE_CAPTUREE] = PAS_DE_CAPTURE;
			liste[CASE_CAPTURE]   = 0;

			/* Test pour la prise en passant :
			 * - si la case d'arrive est celle apres un simple pas
			 * - si c'est sans promotion
			 * - si la piece est capturee
			 * - si c'est par un pion non promu qui fait la capture
			 * - si le pion adverse est pret  capturer
			 */
			if (p->caseArrivee == field                   &&
			    p->casePromotion == CASE_PAS_DE_PROMOTION &&
			    p->pieceCapturante != PIECE_PAS_CAPTUREE  &&
			    datas->pieces_types[ p->pieceCapturante ] == PION &&
			    (datas->board[field + deltaPion - 1]
			     == p->pieceCapturante ||
			     datas->board[field + deltaPion + 1]
			     == p->pieceCapturante ) )
			{
			    liste[ CASE_CAPTURE ] =
				( datas->board[field + deltaPion - 1]
				 == p->pieceCapturante )
				? field + deltaPion - 1
				: field + deltaPion + 1 ;
			}

			nbCoups++;
			liste += 6;
		    } /* Eventuel double pas */
		} /* Pas de promotion */
	    } /* Simple pas possible */

	    /* Le pion capture ... */
	    field++;
	    if (CASE_OK(field))
	    {
		index2 = datas->board[field];
		/* Si on capture une tour qui doit d'abord roquer : on passe */
		if ( ( index2 == indexPieceCapturee ) &&
		     ( datas->pieces[ index2 ].typePiece != TOUR ||
		       datas->pieces[ index2 ].castling  == NO_CASTLING ) )
		{
		    liste[CASE_DEPART]    = depart;
		    liste[CASE_ARRIVEE]   = field;
		    liste[TYPE_DEPART]    = PION;
		    liste[PIECE_CAPTUREE] = index2;
		    liste[CASE_CAPTURE]   = field;
		    if (row(depart) == rangee_promotion)
		    {
			if (p->casePromotion == field)
			{
			    liste[TYPE_ARRIVEE] = p->typePromotion;
			    liste += 6;
			    nbCoups++;
			}
		    }
		    else
		    {
			liste[TYPE_ARRIVEE] = PION;
			liste += 6;
			nbCoups++;
		    }
		}
	    }

	    field -= 2;
	    if (CASE_OK(field))
	    {
		index2 = datas->board[field];
		/* Si on capture une tour qui doit d'abord roquer : on passe */
		if ( ( index2 == indexPieceCapturee ) &&
		     ( datas->pieces[ index2 ].typePiece != TOUR ||
		       datas->pieces[ index2 ].castling  == NO_CASTLING ) )
		{
		    liste[CASE_DEPART]    = depart;
		    liste[CASE_ARRIVEE]   = field;
		    liste[TYPE_DEPART]    = PION;
		    liste[PIECE_CAPTUREE] = index2;
		    liste[CASE_CAPTURE]   = field;
		    if (row(depart) == rangee_promotion)
		    {
			if (p->casePromotion == field)
			{
			    liste[TYPE_ARRIVEE] = p->typePromotion;
			    liste += 6;
			    nbCoups++;
			}
		    }
		    else
		    {
			liste[TYPE_ARRIVEE] = PION;
			liste += 6;
			nbCoups++;
		    }
		}
	    }
	    break;
	}
    }
    return nbCoups ;
}

/* Cette fonction rend TRUE si la case 'la_case' est controle par
 * le camp 'camp'. Elle peut servir  tester si un Roi est en chec
 * mais aussi pour savoir si un roque est lgal en testant la ou les
 * case(s) intermdiaire(s).
 */
bool_t
case_est_controlee_par_camp(
    const exploration_t*	datas   ,
    square_t			la_case ,
    colour_t			camp    )
{
    int			i 	    ;
    piece_index_t	index_piece ;
    square_t		une_case    ;

    /* Pion */
    if (camp == BLANC)
    {
	if (CASE_OK(la_case - 15))
	{
	    index_piece = datas->board[la_case - 15];
	    if (index_piece != 0)
	    {
		if ( datas->pieces_types[ index_piece ] == PION &&
		     datas->pieces[ index_piece ].camp == BLANC )
		{
		    return TRUE;
		}
	    }
	}
	if (CASE_OK(la_case - 17))
	{
	    index_piece = datas->board[la_case - 17];
	    if (index_piece != 0)
	    {
		if ( datas->pieces_types[ index_piece ] == PION &&
		     datas->pieces[ index_piece ].camp == BLANC )
		{
		    return TRUE;
		}
	    }
	}
    }
    else
    {
	if (CASE_OK(la_case + 15))
	{
	    index_piece = datas->board[la_case + 15];
	    if (index_piece != 0)
	    {
		if ( datas->pieces_types[ index_piece ] == PION &&
		     datas->pieces[ index_piece ].camp  == NOIR )
		{
		    return TRUE;
		}
	    }
	}
	if (CASE_OK(la_case + 17))
	{
	    index_piece = datas->board[la_case + 17];
	    if (index_piece != 0)
	    {
		if ( datas->pieces_types[ index_piece ] == PION &&
		     datas->pieces[ index_piece ].camp  == NOIR )
		{
		    return TRUE;
		}
	    }
	}
    }

    /* Vrification orthogonale */
    for (i=0; i<4; i++)
    {
	int	delta    = listeTour[ i ];
	une_case = la_case + delta ;
	while (CASE_OK(une_case))
	{
	    index_piece = datas->board[une_case];
	    if (index_piece != 0)
	    {
		if ( datas->pieces[ index_piece ].camp == camp &&
		     ( datas->pieces_types[ index_piece ] == TOUR ||
		       datas->pieces_types[ index_piece ] == DAME ) )
		{
		    return TRUE;
		}
		break;
	    }
	    une_case += delta;
	}
    }

    /* Vrification diagonale */
    for (i=0; i<4; i++)
    {
	int	delta    = listeFou[ i ];
	une_case = la_case + delta ;
	while (CASE_OK(une_case))
	{
	    index_piece = datas->board[une_case];
	    if (index_piece != 0)
	    {
		if ( datas->pieces[ index_piece ].camp == camp &&
		     ( datas->pieces_types[ index_piece ] == FOU ||
		       datas->pieces_types[ index_piece ] == DAME ) )
		{
		    return TRUE;
		}
		break;
	    }
	    une_case += delta;
	}
    }

    /* Vrification cavalier */
    une_case = la_case ;
    for (i=0; i<8; i++)
    {
	une_case += listeCavalier[ i ];
	if ( CASE_OK( une_case ) )
	{
	    index_piece = datas->board[ une_case ];
	    if (index_piece != 0)
	    {
		if ( datas->pieces_types[ index_piece ] == CAVALIER &&
		     datas->pieces[ index_piece ].camp  == camp )
		{
		    return TRUE;
		}
	    }
	}
    }

    /* Roi */
    une_case = la_case ;
    for( i = 0; i < 8; i++ )
    {
	une_case += listeRoi[ i ];
	if ( CASE_OK( la_case ) )
	{
	    index_piece = datas->board[ une_case ];
	    if ( index_piece != 0 )
	    {
		if ( datas->pieces_types[ index_piece ] == ROI &&
		     datas->pieces[ index_piece ].camp  == camp )
		{
		    return TRUE;
		}
	    }
	}
    }

    return FALSE;
}
