/* Mutagenesis dataset from
Srinivasan A, Muggleton S, Sternberg MJE, King RD (1996) 
Theories for mutagenicity: A study in first-order and feature-based induction. 
AI 85(1-2):277-299

Downloaded from
http://www.doc.ic.ac.uk/~shm/mutagenesis.html

Used in
Nicola Di Mauro, Elena Bellodi, and Fabrizio Riguzzi. Bandit-based Monte-Carlo 
structure learning of probabilistic logic programs. 
Machine Learning, 100(1):127-156, July 2015
Elena Bellodi and Fabrizio Riguzzi. Structure learning of probabilistic logic 
programs by searching the clause space. 
Theory and Practice of Logic Programming, 15(2):169-212, 2015

*/

/** <examples>
?- induce_par([1,2,3,4,5,6,7,8,9],P),test(P,[10],LL,AUCROC,ROC,AUCPR,PR). % learn the parameteters and test the result
?- induce([1,2,3,4,5,6,7,8,9],P),test(P,[10],LL,AUCROC,ROC,AUCPR,PR).  % learn the structure and the parameters and test the result
?- in(P),test(P,[test],LL,AUCROC,ROC,AUCPR,PR). % test the input theory
?- induce_par([1,2,3,4,5,6,7,8,9,10],P).  % learn the parameteters
?- induce([1,2,3,4,5,6,7,8,9,10],P). % learn the structure and the parameters 
*/
:- use_module(library(slipcover)).

:- if(current_predicate(use_rendering/1)).
:- use_rendering(c3).
:- use_rendering(lpad).
:- endif.


:-sc.

:-set_sc(megaex_bottom,4).
:-set_sc(neg_ex,given).
bg([]).


in([
(active:0.5 :-
  lumo(A),
  bond(B,C,2),
  atm(C,n,32,D)
),
(active:0.5 :-
  lumo(A),
  atm(B,o,40,C),
  atm(D,n,32,C)
),
(active:0.5 :-
  atm(A,o,40,B),
  atm(C,n,32,B)
),
(active:0.5 :-
  atm(A,c,28,B)
),
(active:0.5 :-
  lumo(A),
  bond(B,C,2),
  bond(C,D,1)
),
(active:0.5 :-
  lumo(A),
  bond(B,C,2),
  bond(D,C,1)
),
(active:0.5 :-
  benzene(A),
  ring_size_5(B)
),
(active:0.5 :-
  atm(A,c,29,B)
),
(active:0.5 :-
  lumo(A),
  bond(B,C,2),
  ring_size_5(D)
),
(active:0.5 :-
  lumo(A),
  atm(B,h,3,C),
  ring_size_5(D)
)]).

fold(1,[d18,d26,d28,d51,d63,d67,d107,d127,d137,d151,d174,d178,d85,d92,
  d38,d84,d100,d116,d160]).
fold(2,[d10,d61,d86,d94,d105,d128,d173,d183,d29,
  d42,d147,d186,d3,d5,d39,d78,d142,d182,d185]).
fold(3,[d46,d52,d57,d80,d82,d125,d152,d169,d21,d22,d108,d180,
  d119,d34,d65,d66,d143,d154,d181]).
fold(4,[d1,d12,d20,d31,d83,d115,d164,d165,d166,d184,d16,d99,d102,d145,d161,d170,
 d114,d19,d133]).
fold(5,[d6,d15,d69,d71,d87,d95,d104,d109,d177,d187,d44,d93,d97,d106,
 d132,d141,d73,d76,d113]).
fold(6,[d25,d45,d54,d58,d75,d101,d140,d149,d159,d171,d8,d47,d121,d134,
  d70,d77,d111,d62,d179]).
fold(7,[d13,d24,d43,d64,d72,d79,d90,d122,d126,d163,d172,d30,d37,
  d2,d17,d120,d130,d131,d129]).
fold(8,[d11,d27,d53,d56,d68,d146,d33,d74,
  d188,d7,d9,d14,d36,d40,d123,d135,d150,d155,d175]).
fold(9,[d4,d41,d49,d50,d59,d96,d117,d136,d153,d23,d32,d158,d167,d176,
  d98,d88,d124,d139,d168]).
fold(10,[d48,d60,d112,d148,d157,d35,d81,d91,d103,d118,d162,
  d89,d138,d144,d55,d110,d156]).

output(active/0).

input_cw(lumo/1).
input_cw(logp/2).
input_cw(bond/3).
input_cw(atm/4).
input_cw(benzene/1).
input_cw(carbon_5_aromatic_ring/1).
input_cw(carbon_6_ring/1).
input_cw(hetero_aromatic_6_ring/1).
input_cw(hetero_aromatic_5_ring/1).
input_cw(ring_size_6/1).
input_cw(ring_size_5/1).
input_cw(nitro/1).
input_cw(methyl/1).
input_cw(anthracene/1).
input_cw(phenanthrene/1).
input_cw(ball3/1).


modeh(1,active).

modeb(1,lumo(-energy)).
modeb(1,logp(-hydrophob)).
modeb(*,atm(-atomid,-#element,-#int,-charge)).
modeb(*,bond(-atomid,-atomid,-#int)).
modeb(1,(+charge) >= (#charge)).
modeb(1,(+charge) =< (#charge)).
modeb(1,(+charge)= #charge).
modeb(1,(+hydrophob) >= (#hydrophob)).
modeb(1,(+hydrophob) =< (#hydrophob)).
modeb(1,(+hydrophob)= #hydrophob).
modeb(1,(+energy) >= (#energy)).
modeb(1,(+energy) =< (#energy)).
modeb(1,(+energy)= #energy).

modeb(*,benzene(-ring)).
modeb(*,carbon_5_aromatic_ring(-ring)).
modeb(*,carbon_6_ring(-ring)).
modeb(*,hetero_aromatic_6_ring(-ring)).
modeb(*,hetero_aromatic_5_ring(-ring)).
modeb(*,ring_size_6(-ring)).
modeb(*,ring_size_5(-ring)).
modeb(*,nitro(-ring)).
modeb(*,methyl(-ring)).
modeb(*,anthracene(-ringlist)).
modeb(*,phenanthrene(-ringlist)).
modeb(*,ball3(-ringlist)).

modeb(*,member(-ring,+ringlist)).
modeb(1,member(+ring,+ringlist)).


lookahead(logp(B),[(B=_C)]).
lookahead(logp(B),[>=(B,_C)]).
lookahead(logp(B),[=<(B,_C)]).
lookahead(lumo(B),[(B=_C)]).
lookahead(lumo(B),[>=(B,_C)]).
lookahead(lumo(B),[=<(B,_C)]).
%lookahead(atm(_,_,_,_,C),[>=(C,_)]).
%lookahead(atm(_,_,_,_,C),[=<(C,_)]).
%lookahead(atm(_,_,_,_,C),[(C=_)]).

determination(active/0,lumo/1).
determination(active/0,logp/2).
determination(active/0,bond/3).
determination(active/0,atm/4).
determination(active/0,benzene/1).
determination(active/0,carbon_5_aromatic_ring/1).
determination(active/0,carbon_6_ring/1).
determination(active/0,hetero_aromatic_6_ring/1).
determination(active/0,hetero_aromatic_5_ring/1).
determination(active/0,ring_size_6/1).
determination(active/0,ring_size_5/1).
determination(active/0,nitro/1).
determination(active/0,methyl/1).
determination(active/0,anthracene/1).
determination(active/0,phenanthrene/1).
determination(active/0,ball3/1).



%begin(model(s1)).
% fold 1
active(d18).
active(d26).
active(d28).
active(d51).
active(d63).
active(d67).
active(d107).
active(d127).
active(d137).
active(d151).
active(d174).
active(d178).
active(d85).
active(d92).

neg(active(d38)).
neg(active(d84)).
neg(active(d100)).
neg(active(d116)).
neg(active(d160)).

% fold 2
active(d10).
active(d61).
active(d86).
active(d94).
active(d105).
active(d128).
active(d173).
active(d183).
active(d29).

neg(active(d42)).
neg(active(d147)).
neg(active(d186)).
neg(active(d3)).
neg(active(d5)).
neg(active(d39)).
neg(active(d78)).
neg(active(d142)).
neg(active(d182)).
neg(active(d185)).

% fold 3

active(d46).
active(d52).
active(d57).
active(d80).
active(d82).
active(d125).
active(d152).
active(d169).
active(d21).
active(d22).
active(d108).
active(d180).

neg(active(d119)).
neg(active(d34)).
neg(active(d65)).
neg(active(d66)).
neg(active(d143)).
neg(active(d154)).
neg(active(d181)).

% fold 4
active(d1).
active(d12).
active(d20).
active(d31).
active(d83).
active(d115).
active(d164).
active(d165).
active(d166).
active(d184).
active(d16).
active(d99).
active(d102).
active(d145).
active(d161).
active(d170).

neg(active(d114)).
neg(active(d19)).
neg(active(d133)).

% fold 5
active(d6).
active(d15).
active(d69).
active(d71).
active(d87).
active(d95).
active(d104).
active(d109).
active(d177).
active(d187).
active(d44).
active(d93).
active(d97).
active(d106).

neg(active(d132)).
neg(active(d141)).
neg(active(d73)).
neg(active(d76)).
neg(active(d113)).

% fold 6
active(d25).
active(d45).
active(d54).
active(d58).
active(d75).
active(d101).
active(d140).
active(d149).
active(d159).
active(d171).
active(d8).
active(d47).
active(d121).
active(d134).

neg(active(d70)).
neg(active(d77)).
neg(active(d111)).
neg(active(d62)).
neg(active(d179)).

% fold 7
active(d13).
active(d24).
active(d43).
active(d64).
active(d72).
active(d79).
active(d90).
active(d122).
active(d126).
active(d163).
active(d172).
active(d30).
active(d37).

neg(active(d2)).
neg(active(d17)).
neg(active(d120)).
neg(active(d130)).
neg(active(d131)).
neg(active(d129)).

% fold 8
active(d11).
active(d27).
active(d53).
active(d56).
active(d68).
active(d146).
active(d33).
active(d74).

neg(active(d188)).
neg(active(d7)).
neg(active(d9)).
neg(active(d14)).
neg(active(d36)).
neg(active(d40)).
neg(active(d123)).
neg(active(d135)).
neg(active(d150)).
neg(active(d155)).
neg(active(d175)).

% fold 9
active(d4).
active(d41).
active(d49).
active(d50).
active(d59).
active(d96).
active(d117).
active(d136).
active(d153).
active(d23).
active(d32).
active(d158).
active(d167).
active(d176).

neg(active(d98)).
neg(active(d88)).
neg(active(d124)).
neg(active(d139)).
neg(active(d168)).

% fold 10
active(d48).
active(d60).
active(d112).
active(d148).
active(d157).
active(d35).
active(d81).
active(d91).
active(d103).
active(d118).
active(d162).

neg(active(d89)).
neg(active(d138)).
neg(active(d144)).
neg(active(d55)).
neg(active(d110)).
neg(active(d156)).


atm(d1,d1_1,c,22,-0.117).
atm(d1,d1_2,c,22,-0.117).
atm(d1,d1_3,c,22,-0.117).
atm(d1,d1_4,c,195,-0.087).
atm(d1,d1_5,c,195,0.013).
atm(d1,d1_6,c,22,-0.117).
atm(d1,d1_7,h,3,0.142).
atm(d1,d1_8,h,3,0.143).
atm(d1,d1_9,h,3,0.142).
atm(d1,d1_10,h,3,0.142).
atm(d1,d1_11,c,27,-0.087).
atm(d1,d1_12,c,27,0.013).
atm(d1,d1_13,c,22,-0.117).
atm(d1,d1_14,c,22,-0.117).
atm(d1,d1_15,h,3,0.143).
atm(d1,d1_16,h,3,0.143).
atm(d1,d1_17,c,22,-0.117).
atm(d1,d1_18,c,22,-0.117).
atm(d1,d1_19,c,22,-0.117).
atm(d1,d1_20,c,22,-0.117).
atm(d1,d1_21,h,3,0.142).
atm(d1,d1_22,h,3,0.143).
atm(d1,d1_23,h,3,0.142).
atm(d1,d1_24,n,38,0.812).
atm(d1,d1_25,o,40,-0.388).
atm(d1,d1_26,o,40,-0.388).

bond(d1,d1_1,d1_2,7).
bond(d1,d1_2,d1_3,7).
bond(d1,d1_3,d1_4,7).
bond(d1,d1_4,d1_5,7).
bond(d1,d1_5,d1_6,7).
bond(d1,d1_6,d1_1,7).
bond(d1,d1_1,d1_7,1).
bond(d1,d1_2,d1_8,1).
bond(d1,d1_3,d1_9,1).
bond(d1,d1_6,d1_10,1).
bond(d1,d1_4,d1_11,7).
bond(d1,d1_11,d1_12,7).
bond(d1,d1_12,d1_13,7).
bond(d1,d1_13,d1_14,7).
bond(d1,d1_14,d1_5,7).
bond(d1,d1_13,d1_15,1).
bond(d1,d1_14,d1_16,1).
bond(d1,d1_11,d1_17,7).
bond(d1,d1_17,d1_18,7).
bond(d1,d1_18,d1_19,7).
bond(d1,d1_19,d1_20,7).
bond(d1,d1_20,d1_12,7).
bond(d1,d1_17,d1_21,1).
bond(d1,d1_18,d1_22,1).
bond(d1,d1_20,d1_23,1).
bond(d1,d1_24,d1_19,1).
bond(d1,d1_24,d1_25,2).
bond(d1,d1_24,d1_26,2).
atm(d10,d10_1,c,22,-0.119).
atm(d10,d10_2,c,22,-0.119).
atm(d10,d10_3,c,22,-0.119).
atm(d10,d10_4,c,29,0.011).
atm(d10,d10_5,c,22,-0.119).
atm(d10,d10_6,c,22,-0.118).
atm(d10,d10_7,c,22,-0.119).
atm(d10,d10_8,c,22,-0.119).
atm(d10,d10_9,c,22,-0.168).
atm(d10,d10_10,c,22,-0.119).
atm(d10,d10_11,c,22,-0.118).
atm(d10,d10_12,c,29,0.011).
atm(d10,d10_13,c,10,0.011).
atm(d10,d10_14,n,38,0.812).
atm(d10,d10_15,o,40,-0.388).
atm(d10,d10_16,o,40,-0.388).
atm(d10,d10_17,c,10,0.011).
atm(d10,d10_18,h,3,0.142).
atm(d10,d10_19,h,3,0.142).
atm(d10,d10_20,h,3,0.142).
atm(d10,d10_21,h,3,0.142).
atm(d10,d10_22,h,3,0.142).
atm(d10,d10_23,h,3,0.142).
atm(d10,d10_24,h,3,0.061).
atm(d10,d10_25,h,3,0.061).
atm(d10,d10_26,h,3,0.061).
atm(d10,d10_27,h,3,0.061).
atm(d10,d10_28,h,3,0.061).

bond(d10,d10_1,d10_2,7).
bond(d10,d10_2,d10_3,7).
bond(d10,d10_3,d10_4,7).
bond(d10,d10_4,d10_5,7).
bond(d10,d10_5,d10_6,7).
bond(d10,d10_6,d10_1,7).
bond(d10,d10_7,d10_8,7).
bond(d10,d10_8,d10_9,7).
bond(d10,d10_9,d10_10,7).
bond(d10,d10_10,d10_11,7).
bond(d10,d10_11,d10_12,7).
bond(d10,d10_12,d10_7,7).
bond(d10,d10_4,d10_12,1).
bond(d10,d10_3,d10_13,1).
bond(d10,d10_13,d10_7,1).
bond(d10,d10_1,d10_14,1).
bond(d10,d10_14,d10_15,2).
bond(d10,d10_14,d10_16,2).
bond(d10,d10_9,d10_17,1).
bond(d10,d10_2,d10_18,1).
bond(d10,d10_5,d10_19,1).
bond(d10,d10_6,d10_20,1).
bond(d10,d10_8,d10_21,1).
bond(d10,d10_10,d10_22,1).
bond(d10,d10_11,d10_23,1).
bond(d10,d10_17,d10_24,1).
bond(d10,d10_17,d10_25,1).
bond(d10,d10_17,d10_26,1).
bond(d10,d10_13,d10_27,1).
bond(d10,d10_13,d10_28,1).
atm(d100,d100_1,c,22,-0.128).
atm(d100,d100_2,c,22,-0.128).
atm(d100,d100_3,c,22,-0.128).
atm(d100,d100_4,c,29,0.002).
atm(d100,d100_5,c,22,-0.128).
atm(d100,d100_6,c,22,-0.128).
atm(d100,d100_7,h,3,0.132).
atm(d100,d100_8,h,3,0.132).
atm(d100,d100_9,h,3,0.132).
atm(d100,d100_10,h,3,0.132).
atm(d100,d100_11,c,29,0.002).
atm(d100,d100_12,c,22,-0.128).
atm(d100,d100_13,c,22,-0.128).
atm(d100,d100_14,c,22,-0.128).
atm(d100,d100_15,c,22,0.202).
atm(d100,d100_16,c,22,-0.128).
atm(d100,d100_17,h,3,0.132).
atm(d100,d100_18,h,3,0.132).
atm(d100,d100_19,h,3,0.132).
atm(d100,d100_20,h,3,0.132).
atm(d100,d100_21,n,32,-0.779).
atm(d100,d100_22,n,38,0.801).
atm(d100,d100_23,o,40,-0.398).
atm(d100,d100_24,o,40,-0.398).
atm(d100,d100_25,h,1,0.332).
atm(d100,d100_26,h,1,0.332).

bond(d100,d100_1,d100_2,7).
bond(d100,d100_2,d100_3,7).
bond(d100,d100_3,d100_4,7).
bond(d100,d100_4,d100_5,7).
bond(d100,d100_5,d100_6,7).
bond(d100,d100_6,d100_1,7).
bond(d100,d100_1,d100_7,1).
bond(d100,d100_2,d100_8,1).
bond(d100,d100_3,d100_9,1).
bond(d100,d100_5,d100_10,1).
bond(d100,d100_11,d100_12,7).
bond(d100,d100_12,d100_13,7).
bond(d100,d100_13,d100_14,7).
bond(d100,d100_14,d100_15,7).
bond(d100,d100_15,d100_16,7).
bond(d100,d100_16,d100_11,7).
bond(d100,d100_12,d100_17,1).
bond(d100,d100_13,d100_18,1).
bond(d100,d100_14,d100_19,1).
bond(d100,d100_16,d100_20,1).
bond(d100,d100_4,d100_11,1).
bond(d100,d100_15,d100_21,1).
bond(d100,d100_6,d100_22,1).
bond(d100,d100_22,d100_23,2).
bond(d100,d100_22,d100_24,2).
bond(d100,d100_21,d100_25,1).
bond(d100,d100_21,d100_26,1).
atm(d101,d101_1,c,22,-0.121).
atm(d101,d101_2,c,22,-0.121).
atm(d101,d101_3,c,27,0.009).
atm(d101,d101_4,c,27,-0.091).
atm(d101,d101_5,c,27,0.009).
atm(d101,d101_6,c,22,-0.122).
atm(d101,d101_7,c,22,-0.121).
atm(d101,d101_8,c,22,-0.121).
atm(d101,d101_9,c,27,0.009).
atm(d101,d101_10,c,27,-0.091).
atm(d101,d101_11,c,27,0.009).
atm(d101,d101_12,c,22,0.009).
atm(d101,d101_13,c,22,0.009).
atm(d101,d101_14,c,22,-0.121).
atm(d101,d101_15,c,22,-0.121).
atm(d101,d101_16,c,22,-0.122).
atm(d101,d101_17,c,10,-0.091).
atm(d101,d101_18,c,10,-0.091).
atm(d101,d101_19,c,10,-0.091).
atm(d101,d101_20,c,10,-0.091).
atm(d101,d101_21,n,38,0.808).
atm(d101,d101_22,o,40,-0.392).
atm(d101,d101_23,o,40,-0.392).
atm(d101,d101_24,h,3,0.138).
atm(d101,d101_25,h,3,0.138).
atm(d101,d101_26,h,3,0.138).
atm(d101,d101_27,h,3,0.138).
atm(d101,d101_28,h,3,0.138).
atm(d101,d101_29,h,3,0.138).
atm(d101,d101_30,h,3,0.138).
atm(d101,d101_31,h,3,0.059).
atm(d101,d101_32,h,3,0.059).
atm(d101,d101_33,h,3,0.059).
atm(d101,d101_34,h,3,0.059).
atm(d101,d101_35,h,3,0.059).
atm(d101,d101_36,h,3,0.059).
atm(d101,d101_37,h,3,0.059).
atm(d101,d101_38,h,3,0.059).

bond(d101,d101_1,d101_2,7).
bond(d101,d101_2,d101_3,7).
bond(d101,d101_3,d101_4,7).
bond(d101,d101_4,d101_5,7).
bond(d101,d101_5,d101_6,7).
bond(d101,d101_6,d101_1,7).
bond(d101,d101_3,d101_7,7).
bond(d101,d101_7,d101_8,7).
bond(d101,d101_8,d101_9,7).
bond(d101,d101_9,d101_10,7).
bond(d101,d101_10,d101_4,7).
bond(d101,d101_10,d101_11,7).
bond(d101,d101_11,d101_12,7).
bond(d101,d101_12,d101_13,7).
bond(d101,d101_13,d101_5,7).
bond(d101,d101_9,d101_14,7).
bond(d101,d101_14,d101_15,7).
bond(d101,d101_15,d101_16,7).
bond(d101,d101_16,d101_11,7).
bond(d101,d101_12,d101_17,1).
bond(d101,d101_17,d101_18,1).
bond(d101,d101_18,d101_19,1).
bond(d101,d101_19,d101_20,1).
bond(d101,d101_20,d101_13,1).
bond(d101,d101_14,d101_21,1).
bond(d101,d101_21,d101_22,2).
bond(d101,d101_21,d101_23,2).
bond(d101,d101_1,d101_24,1).
bond(d101,d101_2,d101_25,1).
bond(d101,d101_6,d101_26,1).
bond(d101,d101_7,d101_27,1).
bond(d101,d101_8,d101_28,1).
bond(d101,d101_15,d101_29,1).
bond(d101,d101_16,d101_30,1).
bond(d101,d101_20,d101_31,1).
bond(d101,d101_20,d101_32,1).
bond(d101,d101_19,d101_33,1).
bond(d101,d101_19,d101_34,1).
bond(d101,d101_17,d101_35,1).
bond(d101,d101_17,d101_36,1).
bond(d101,d101_18,d101_37,1).
bond(d101,d101_18,d101_38,1).
atm(d102,d102_1,c,22,-0.119).
atm(d102,d102_2,c,22,-0.119).
atm(d102,d102_3,c,22,0.111).
atm(d102,d102_4,c,29,0.011).
atm(d102,d102_5,c,22,-0.119).
atm(d102,d102_6,c,22,-0.119).
atm(d102,d102_7,h,3,0.141).
atm(d102,d102_8,h,3,0.141).
atm(d102,d102_9,c,22,0.111).
atm(d102,d102_10,c,22,-0.119).
atm(d102,d102_11,c,22,-0.119).
atm(d102,d102_12,c,22,-0.119).
atm(d102,d102_13,c,22,-0.119).
atm(d102,d102_14,c,29,0.011).
atm(d102,d102_15,h,3,0.141).
atm(d102,d102_16,h,3,0.141).
atm(d102,d102_17,c,14,0.56).
atm(d102,d102_18,o,40,-0.54).
atm(d102,d102_19,n,38,0.81).
atm(d102,d102_20,n,38,0.81).
atm(d102,d102_21,n,38,0.81).
atm(d102,d102_22,n,38,0.81).
atm(d102,d102_23,o,40,-0.39).
atm(d102,d102_24,o,40,-0.389).
atm(d102,d102_25,o,40,-0.389).
atm(d102,d102_26,o,40,-0.389).
atm(d102,d102_27,o,40,-0.389).
atm(d102,d102_28,o,40,-0.39).
atm(d102,d102_29,o,40,-0.39).
atm(d102,d102_30,o,40,-0.39).

bond(d102,d102_1,d102_2,7).
bond(d102,d102_2,d102_3,7).
bond(d102,d102_3,d102_4,7).
bond(d102,d102_4,d102_5,7).
bond(d102,d102_5,d102_6,7).
bond(d102,d102_6,d102_1,7).
bond(d102,d102_2,d102_7,1).
bond(d102,d102_6,d102_8,1).
bond(d102,d102_9,d102_10,7).
bond(d102,d102_10,d102_11,7).
bond(d102,d102_11,d102_12,7).
bond(d102,d102_12,d102_13,7).
bond(d102,d102_13,d102_14,7).
bond(d102,d102_14,d102_9,7).
bond(d102,d102_10,d102_15,1).
bond(d102,d102_12,d102_16,1).
bond(d102,d102_4,d102_14,1).
bond(d102,d102_3,d102_17,1).
bond(d102,d102_17,d102_9,1).
bond(d102,d102_17,d102_18,2).
bond(d102,d102_11,d102_19,1).
bond(d102,d102_13,d102_20,1).
bond(d102,d102_5,d102_21,1).
bond(d102,d102_1,d102_22,1).
bond(d102,d102_22,d102_23,2).
bond(d102,d102_22,d102_24,2).
bond(d102,d102_21,d102_25,2).
bond(d102,d102_21,d102_26,2).
bond(d102,d102_20,d102_27,2).
bond(d102,d102_20,d102_28,2).
bond(d102,d102_19,d102_29,2).
bond(d102,d102_19,d102_30,2).
atm(d103,d103_1,c,22,-0.118).
atm(d103,d103_2,c,22,-0.118).
atm(d103,d103_3,c,27,0.012).
atm(d103,d103_4,c,27,-0.088).
atm(d103,d103_5,c,27,0.012).
atm(d103,d103_6,c,22,-0.118).
atm(d103,d103_7,h,3,0.141).
atm(d103,d103_8,h,3,0.142).
atm(d103,d103_9,h,3,0.142).
atm(d103,d103_10,c,22,-0.118).
atm(d103,d103_11,c,22,-0.118).
atm(d103,d103_12,c,27,0.012).
atm(d103,d103_13,c,27,-0.088).
atm(d103,d103_14,h,3,0.141).
atm(d103,d103_15,h,3,0.142).
atm(d103,d103_16,c,27,0.012).
atm(d103,d103_17,c,22,-0.118).
atm(d103,d103_18,c,22,-0.118).
atm(d103,d103_19,h,3,0.141).
atm(d103,d103_20,h,3,0.142).
atm(d103,d103_21,c,22,-0.118).
atm(d103,d103_22,c,22,-0.118).
atm(d103,d103_23,c,22,-0.118).
atm(d103,d103_24,h,3,0.142).
atm(d103,d103_25,h,3,0.142).
atm(d103,d103_26,n,38,0.811).
atm(d103,d103_27,o,40,-0.389).
atm(d103,d103_28,o,40,-0.389).

bond(d103,d103_1,d103_2,7).
bond(d103,d103_2,d103_3,7).
bond(d103,d103_3,d103_4,7).
bond(d103,d103_4,d103_5,7).
bond(d103,d103_5,d103_6,7).
bond(d103,d103_6,d103_1,7).
bond(d103,d103_1,d103_7,1).
bond(d103,d103_2,d103_8,1).
bond(d103,d103_6,d103_9,1).
bond(d103,d103_3,d103_10,7).
bond(d103,d103_10,d103_11,7).
bond(d103,d103_11,d103_12,7).
bond(d103,d103_12,d103_13,7).
bond(d103,d103_13,d103_4,7).
bond(d103,d103_10,d103_14,1).
bond(d103,d103_11,d103_15,1).
bond(d103,d103_13,d103_16,7).
bond(d103,d103_16,d103_17,7).
bond(d103,d103_17,d103_18,7).
bond(d103,d103_18,d103_5,7).
bond(d103,d103_17,d103_19,1).
bond(d103,d103_18,d103_20,1).
bond(d103,d103_12,d103_21,7).
bond(d103,d103_21,d103_22,7).
bond(d103,d103_22,d103_23,7).
bond(d103,d103_23,d103_16,7).
bond(d103,d103_21,d103_24,1).
bond(d103,d103_23,d103_25,1).
bond(d103,d103_22,d103_26,1).
bond(d103,d103_26,d103_27,2).
bond(d103,d103_26,d103_28,2).
atm(d104,d104_1,c,22,-0.121).
atm(d104,d104_2,c,22,-0.121).
atm(d104,d104_3,c,27,0.009).
atm(d104,d104_4,c,27,-0.091).
atm(d104,d104_5,c,27,0.009).
atm(d104,d104_6,c,22,-0.122).
atm(d104,d104_7,c,22,-0.121).
atm(d104,d104_8,c,22,-0.121).
atm(d104,d104_9,c,27,0.009).
atm(d104,d104_10,c,27,-0.091).
atm(d104,d104_11,c,27,0.009).
atm(d104,d104_12,c,22,0.009).
atm(d104,d104_13,c,22,0.009).
atm(d104,d104_14,c,22,-0.121).
atm(d104,d104_15,c,22,-0.121).
atm(d104,d104_16,c,22,-0.122).
atm(d104,d104_17,c,10,-0.091).
atm(d104,d104_18,c,10,-0.091).
atm(d104,d104_19,c,10,-0.091).
atm(d104,d104_20,c,10,-0.091).
atm(d104,d104_21,n,38,0.808).
atm(d104,d104_22,o,40,-0.392).
atm(d104,d104_23,o,40,-0.392).
atm(d104,d104_24,h,3,0.138).
atm(d104,d104_25,h,3,0.138).
atm(d104,d104_26,h,3,0.138).
atm(d104,d104_27,h,3,0.138).
atm(d104,d104_28,h,3,0.138).
atm(d104,d104_29,h,3,0.138).
atm(d104,d104_30,h,3,0.138).
atm(d104,d104_31,h,3,0.059).
atm(d104,d104_32,h,3,0.059).
atm(d104,d104_33,h,3,0.059).
atm(d104,d104_34,h,3,0.059).
atm(d104,d104_35,h,3,0.059).
atm(d104,d104_36,h,3,0.059).
atm(d104,d104_37,h,3,0.059).
atm(d104,d104_38,h,3,0.059).

bond(d104,d104_1,d104_2,7).
bond(d104,d104_2,d104_3,7).
bond(d104,d104_3,d104_4,7).
bond(d104,d104_4,d104_5,7).
bond(d104,d104_5,d104_6,7).
bond(d104,d104_6,d104_1,7).
bond(d104,d104_3,d104_7,7).
bond(d104,d104_7,d104_8,7).
bond(d104,d104_8,d104_9,7).
bond(d104,d104_9,d104_10,7).
bond(d104,d104_10,d104_4,7).
bond(d104,d104_10,d104_11,7).
bond(d104,d104_11,d104_12,7).
bond(d104,d104_12,d104_13,7).
bond(d104,d104_13,d104_5,7).
bond(d104,d104_9,d104_14,7).
bond(d104,d104_14,d104_15,7).
bond(d104,d104_15,d104_16,7).
bond(d104,d104_16,d104_11,7).
bond(d104,d104_12,d104_17,1).
bond(d104,d104_17,d104_18,1).
bond(d104,d104_18,d104_19,1).
bond(d104,d104_19,d104_20,1).
bond(d104,d104_20,d104_13,1).
bond(d104,d104_16,d104_21,1).
bond(d104,d104_21,d104_22,2).
bond(d104,d104_21,d104_23,2).
bond(d104,d104_1,d104_24,1).
bond(d104,d104_2,d104_25,1).
bond(d104,d104_6,d104_26,1).
bond(d104,d104_7,d104_27,1).
bond(d104,d104_8,d104_28,1).
bond(d104,d104_14,d104_29,1).
bond(d104,d104_15,d104_30,1).
bond(d104,d104_20,d104_31,1).
bond(d104,d104_20,d104_32,1).
bond(d104,d104_19,d104_33,1).
bond(d104,d104_19,d104_34,1).
bond(d104,d104_17,d104_35,1).
bond(d104,d104_17,d104_36,1).
bond(d104,d104_18,d104_37,1).
bond(d104,d104_18,d104_38,1).
atm(d105,d105_1,c,22,-0.084).
atm(d105,d105_2,c,22,-0.084).
atm(d105,d105_3,c,22,-0.084).
atm(d105,d105_4,c,27,-0.054).
atm(d105,d105_5,c,27,0.046).
atm(d105,d105_6,c,22,-0.084).
atm(d105,d105_7,h,3,0.176).
atm(d105,d105_8,h,3,0.176).
atm(d105,d105_9,h,3,0.176).
atm(d105,d105_10,c,27,-0.054).
atm(d105,d105_11,c,27,0.435).
atm(d105,d105_12,n,35,-0.725).
atm(d105,d105_13,c,22,-0.084).
atm(d105,d105_14,h,3,0.176).
atm(d105,d105_15,c,22,-0.084).
atm(d105,d105_16,c,22,-0.084).
atm(d105,d105_17,c,22,0.296).
atm(d105,d105_18,n,35,-0.725).
atm(d105,d105_19,h,3,0.176).
atm(d105,d105_20,h,3,0.176).
atm(d105,d105_21,h,3,0.176).
atm(d105,d105_22,n,38,0.845).
atm(d105,d105_23,o,40,-0.354).
atm(d105,d105_24,o,40,-0.354).

bond(d105,d105_1,d105_2,7).
bond(d105,d105_2,d105_3,7).
bond(d105,d105_3,d105_4,7).
bond(d105,d105_4,d105_5,7).
bond(d105,d105_5,d105_6,7).
bond(d105,d105_6,d105_1,7).
bond(d105,d105_1,d105_7,1).
bond(d105,d105_2,d105_8,1).
bond(d105,d105_6,d105_9,1).
bond(d105,d105_4,d105_10,7).
bond(d105,d105_10,d105_11,7).
bond(d105,d105_11,d105_12,7).
bond(d105,d105_12,d105_13,7).
bond(d105,d105_13,d105_5,7).
bond(d105,d105_13,d105_14,1).
bond(d105,d105_10,d105_15,7).
bond(d105,d105_15,d105_16,7).
bond(d105,d105_16,d105_17,7).
bond(d105,d105_17,d105_18,7).
bond(d105,d105_18,d105_11,7).
bond(d105,d105_15,d105_19,1).
bond(d105,d105_16,d105_20,1).
bond(d105,d105_17,d105_21,1).
bond(d105,d105_3,d105_22,1).
bond(d105,d105_22,d105_23,2).
bond(d105,d105_22,d105_24,2).
atm(d106,d106_1,c,22,-0.119).
atm(d106,d106_2,c,22,-0.119).
atm(d106,d106_3,c,27,0.011).
atm(d106,d106_4,c,27,-0.089).
atm(d106,d106_5,c,27,0.011).
atm(d106,d106_6,c,22,-0.119).
atm(d106,d106_7,h,3,0.141).
atm(d106,d106_8,h,3,0.142).
atm(d106,d106_9,h,3,0.142).
atm(d106,d106_10,c,22,-0.119).
atm(d106,d106_11,c,22,-0.119).
atm(d106,d106_12,c,27,0.011).
atm(d106,d106_13,c,27,-0.089).
atm(d106,d106_14,h,3,0.141).
atm(d106,d106_15,h,3,0.142).
atm(d106,d106_16,c,27,0.011).
atm(d106,d106_17,c,22,0.262).
atm(d106,d106_18,c,22,-0.119).
atm(d106,d106_19,h,3,0.141).
atm(d106,d106_20,c,22,-0.119).
atm(d106,d106_21,c,22,-0.119).
atm(d106,d106_22,c,22,-0.119).
atm(d106,d106_23,h,3,0.142).
atm(d106,d106_24,h,3,0.142).
atm(d106,d106_25,n,38,0.812).
atm(d106,d106_26,o,45,-0.638).
atm(d106,d106_27,o,40,-0.388).
atm(d106,d106_28,o,40,-0.388).
atm(d106,d106_29,h,8,0.412).

bond(d106,d106_1,d106_2,7).
bond(d106,d106_2,d106_3,7).
bond(d106,d106_3,d106_4,7).
bond(d106,d106_4,d106_5,7).
bond(d106,d106_5,d106_6,7).
bond(d106,d106_6,d106_1,7).
bond(d106,d106_1,d106_7,1).
bond(d106,d106_2,d106_8,1).
bond(d106,d106_6,d106_9,1).
bond(d106,d106_3,d106_10,7).
bond(d106,d106_10,d106_11,7).
bond(d106,d106_11,d106_12,7).
bond(d106,d106_12,d106_13,7).
bond(d106,d106_13,d106_4,7).
bond(d106,d106_10,d106_14,1).
bond(d106,d106_11,d106_15,1).
bond(d106,d106_13,d106_16,7).
bond(d106,d106_16,d106_17,7).
bond(d106,d106_17,d106_18,7).
bond(d106,d106_18,d106_5,7).
bond(d106,d106_18,d106_19,1).
bond(d106,d106_12,d106_20,7).
bond(d106,d106_20,d106_21,7).
bond(d106,d106_21,d106_22,7).
bond(d106,d106_22,d106_16,7).
bond(d106,d106_20,d106_23,1).
bond(d106,d106_21,d106_24,1).
bond(d106,d106_22,d106_25,1).
bond(d106,d106_17,d106_26,1).
bond(d106,d106_25,d106_27,2).
bond(d106,d106_25,d106_28,2).
bond(d106,d106_26,d106_29,1).
atm(d107,d107_1,c,22,-0.115).
atm(d107,d107_2,c,22,-0.115).
atm(d107,d107_3,c,27,0.015).
atm(d107,d107_4,c,27,-0.085).
atm(d107,d107_5,c,28,0.015).
atm(d107,d107_6,c,22,-0.115).
atm(d107,d107_7,h,3,0.145).
atm(d107,d107_8,h,3,0.145).
atm(d107,d107_9,c,22,-0.115).
atm(d107,d107_10,c,22,-0.115).
atm(d107,d107_11,c,22,-0.115).
atm(d107,d107_12,c,29,0.015).
atm(d107,d107_13,h,3,0.145).
atm(d107,d107_14,c,28,0.015).
atm(d107,d107_15,c,28,0.015).
atm(d107,d107_16,c,22,-0.115).
atm(d107,d107_17,c,22,-0.115).
atm(d107,d107_18,c,22,-0.115).
atm(d107,d107_19,c,22,-0.115).
atm(d107,d107_20,h,3,0.146).
atm(d107,d107_21,h,3,0.145).
atm(d107,d107_22,h,3,0.145).
atm(d107,d107_23,h,3,0.145).
atm(d107,d107_24,n,38,0.816).
atm(d107,d107_25,n,38,0.816).
atm(d107,d107_26,n,38,0.816).
atm(d107,d107_27,o,40,-0.384).
atm(d107,d107_28,o,40,-0.384).
atm(d107,d107_29,o,40,-0.384).
atm(d107,d107_30,o,40,-0.384).
atm(d107,d107_31,o,40,-0.384).
atm(d107,d107_32,o,40,-0.384).

bond(d107,d107_1,d107_2,7).
bond(d107,d107_2,d107_3,7).
bond(d107,d107_3,d107_4,7).
bond(d107,d107_4,d107_5,7).
bond(d107,d107_5,d107_6,7).
bond(d107,d107_6,d107_1,7).
bond(d107,d107_2,d107_7,1).
bond(d107,d107_6,d107_8,1).
bond(d107,d107_3,d107_9,7).
bond(d107,d107_9,d107_10,7).
bond(d107,d107_10,d107_11,7).
bond(d107,d107_11,d107_12,7).
bond(d107,d107_12,d107_4,7).
bond(d107,d107_11,d107_13,1).
bond(d107,d107_14,d107_15,7).
bond(d107,d107_15,d107_16,7).
bond(d107,d107_16,d107_17,7).
bond(d107,d107_17,d107_18,7).
bond(d107,d107_18,d107_19,7).
bond(d107,d107_19,d107_14,7).
bond(d107,d107_16,d107_20,1).
bond(d107,d107_17,d107_21,1).
bond(d107,d107_18,d107_22,1).
bond(d107,d107_19,d107_23,1).
bond(d107,d107_14,d107_5,1).
bond(d107,d107_15,d107_12,1).
bond(d107,d107_10,d107_24,1).
bond(d107,d107_9,d107_25,1).
bond(d107,d107_1,d107_26,1).
bond(d107,d107_26,d107_27,2).
bond(d107,d107_26,d107_28,2).
bond(d107,d107_25,d107_29,2).
bond(d107,d107_25,d107_30,2).
bond(d107,d107_24,d107_31,2).
bond(d107,d107_24,d107_32,2).
atm(d108,d108_1,c,22,-0.122).
atm(d108,d108_2,c,22,-0.122).
atm(d108,d108_3,c,27,0.008).
atm(d108,d108_4,c,27,-0.092).
atm(d108,d108_5,c,28,0.008).
atm(d108,d108_6,c,22,-0.122).
atm(d108,d108_7,h,3,0.139).
atm(d108,d108_8,h,3,0.139).
atm(d108,d108_9,c,22,-0.122).
atm(d108,d108_10,c,22,-0.122).
atm(d108,d108_11,c,22,-0.122).
atm(d108,d108_12,c,29,0.008).
atm(d108,d108_13,h,3,0.138).
atm(d108,d108_14,c,28,0.008).
atm(d108,d108_15,c,28,0.008).
atm(d108,d108_16,c,22,-0.122).
atm(d108,d108_17,c,22,-0.122).
atm(d108,d108_18,c,22,-0.122).
atm(d108,d108_19,c,22,-0.122).
atm(d108,d108_20,h,3,0.138).
atm(d108,d108_21,h,3,0.138).
atm(d108,d108_22,h,3,0.138).
atm(d108,d108_23,h,3,0.138).
atm(d108,d108_24,n,38,0.809).
atm(d108,d108_25,o,40,-0.391).
atm(d108,d108_26,o,40,-0.391).
atm(d108,d108_27,h,3,0.138).
atm(d108,d108_28,h,3,0.139).

bond(d108,d108_1,d108_2,7).
bond(d108,d108_2,d108_3,7).
bond(d108,d108_3,d108_4,7).
bond(d108,d108_4,d108_5,7).
bond(d108,d108_5,d108_6,7).
bond(d108,d108_6,d108_1,7).
bond(d108,d108_2,d108_7,1).
bond(d108,d108_6,d108_8,1).
bond(d108,d108_3,d108_9,7).
bond(d108,d108_9,d108_10,7).
bond(d108,d108_10,d108_11,7).
bond(d108,d108_11,d108_12,7).
bond(d108,d108_12,d108_4,7).
bond(d108,d108_11,d108_13,1).
bond(d108,d108_14,d108_15,7).
bond(d108,d108_15,d108_16,7).
bond(d108,d108_16,d108_17,7).
bond(d108,d108_17,d108_18,7).
bond(d108,d108_18,d108_19,7).
bond(d108,d108_19,d108_14,7).
bond(d108,d108_16,d108_20,1).
bond(d108,d108_17,d108_21,1).
bond(d108,d108_18,d108_22,1).
bond(d108,d108_19,d108_23,1).
bond(d108,d108_14,d108_5,1).
bond(d108,d108_15,d108_12,1).
bond(d108,d108_9,d108_24,1).
bond(d108,d108_24,d108_25,2).
bond(d108,d108_24,d108_26,2).
bond(d108,d108_1,d108_27,1).
bond(d108,d108_10,d108_28,1).
atm(d109,d109_1,c,22,-0.113).
atm(d109,d109_2,c,22,-0.113).
atm(d109,d109_3,c,22,-0.113).
atm(d109,d109_4,c,29,0.017).
atm(d109,d109_5,c,27,-0.083).
atm(d109,d109_6,c,27,0.017).
atm(d109,d109_7,c,29,0.017).
atm(d109,d109_8,c,27,-0.083).
atm(d109,d109_9,c,27,-0.083).
atm(d109,d109_10,c,27,-0.083).
atm(d109,d109_11,c,27,0.017).
atm(d109,d109_12,c,22,-0.113).
atm(d109,d109_13,c,22,-0.113).
atm(d109,d109_14,c,27,0.017).
atm(d109,d109_15,c,27,0.017).
atm(d109,d109_16,c,22,-0.113).
atm(d109,d109_17,c,22,-0.113).
atm(d109,d109_18,c,22,-0.113).
atm(d109,d109_19,c,22,-0.113).
atm(d109,d109_20,c,22,-0.113).
atm(d109,d109_21,c,22,-0.113).
atm(d109,d109_22,c,22,-0.113).
atm(d109,d109_23,n,38,0.818).
atm(d109,d109_24,n,38,0.818).
atm(d109,d109_25,o,40,-0.382).
atm(d109,d109_26,o,40,-0.382).
atm(d109,d109_27,o,40,-0.382).
atm(d109,d109_28,o,40,-0.382).
atm(d109,d109_29,h,3,0.148).
atm(d109,d109_30,h,3,0.148).
atm(d109,d109_31,h,3,0.148).
atm(d109,d109_32,h,3,0.147).
atm(d109,d109_33,h,3,0.147).
atm(d109,d109_34,h,3,0.148).
atm(d109,d109_35,h,3,0.148).
atm(d109,d109_36,h,3,0.148).
atm(d109,d109_37,h,3,0.148).
atm(d109,d109_38,h,3,0.148).

bond(d109,d109_1,d109_2,7).
bond(d109,d109_2,d109_3,7).
bond(d109,d109_3,d109_4,7).
bond(d109,d109_4,d109_5,7).
bond(d109,d109_5,d109_6,7).
bond(d109,d109_6,d109_1,7).
bond(d109,d109_4,d109_7,1).
bond(d109,d109_7,d109_8,7).
bond(d109,d109_8,d109_9,7).
bond(d109,d109_9,d109_10,7).
bond(d109,d109_10,d109_5,7).
bond(d109,d109_8,d109_11,7).
bond(d109,d109_11,d109_12,7).
bond(d109,d109_12,d109_13,7).
bond(d109,d109_13,d109_14,7).
bond(d109,d109_14,d109_9,7).
bond(d109,d109_10,d109_15,7).
bond(d109,d109_15,d109_16,7).
bond(d109,d109_16,d109_17,7).
bond(d109,d109_17,d109_6,7).
bond(d109,d109_14,d109_18,7).
bond(d109,d109_18,d109_19,7).
bond(d109,d109_19,d109_15,7).
bond(d109,d109_7,d109_20,7).
bond(d109,d109_20,d109_21,7).
bond(d109,d109_21,d109_22,7).
bond(d109,d109_22,d109_11,7).
bond(d109,d109_22,d109_23,1).
bond(d109,d109_1,d109_24,1).
bond(d109,d109_23,d109_25,2).
bond(d109,d109_23,d109_26,2).
bond(d109,d109_24,d109_27,2).
bond(d109,d109_24,d109_28,2).
bond(d109,d109_2,d109_29,1).
bond(d109,d109_3,d109_30,1).
bond(d109,d109_12,d109_31,1).
bond(d109,d109_13,d109_32,1).
bond(d109,d109_16,d109_33,1).
bond(d109,d109_17,d109_34,1).
bond(d109,d109_18,d109_35,1).
bond(d109,d109_19,d109_36,1).
bond(d109,d109_20,d109_37,1).
bond(d109,d109_21,d109_38,1).
atm(d11,d11_1,c,22,-0.117).
atm(d11,d11_2,c,22,-0.117).
atm(d11,d11_3,c,22,-0.117).
atm(d11,d11_4,c,195,-0.087).
atm(d11,d11_5,c,195,0.013).
atm(d11,d11_6,c,22,-0.117).
atm(d11,d11_7,h,3,0.142).
atm(d11,d11_8,h,3,0.143).
atm(d11,d11_9,h,3,0.142).
atm(d11,d11_10,h,3,0.142).
atm(d11,d11_11,c,27,-0.087).
atm(d11,d11_12,c,27,0.013).
atm(d11,d11_13,c,22,-0.117).
atm(d11,d11_14,c,22,-0.117).
atm(d11,d11_15,h,3,0.143).
atm(d11,d11_16,c,22,-0.117).
atm(d11,d11_17,c,22,-0.117).
atm(d11,d11_18,c,22,-0.117).
atm(d11,d11_19,c,22,-0.117).
atm(d11,d11_20,h,3,0.143).
atm(d11,d11_21,h,3,0.142).
atm(d11,d11_22,h,3,0.143).
atm(d11,d11_23,h,3,0.142).
atm(d11,d11_24,n,38,0.812).
atm(d11,d11_25,o,40,-0.388).
atm(d11,d11_26,o,40,-0.388).

bond(d11,d11_1,d11_2,7).
bond(d11,d11_2,d11_3,7).
bond(d11,d11_3,d11_4,7).
bond(d11,d11_4,d11_5,7).
bond(d11,d11_5,d11_6,7).
bond(d11,d11_6,d11_1,7).
bond(d11,d11_1,d11_7,1).
bond(d11,d11_2,d11_8,1).
bond(d11,d11_3,d11_9,1).
bond(d11,d11_6,d11_10,1).
bond(d11,d11_4,d11_11,7).
bond(d11,d11_11,d11_12,7).
bond(d11,d11_12,d11_13,7).
bond(d11,d11_13,d11_14,7).
bond(d11,d11_14,d11_5,7).
bond(d11,d11_13,d11_15,1).
bond(d11,d11_11,d11_16,7).
bond(d11,d11_16,d11_17,7).
bond(d11,d11_17,d11_18,7).
bond(d11,d11_18,d11_19,7).
bond(d11,d11_19,d11_12,7).
bond(d11,d11_16,d11_20,1).
bond(d11,d11_17,d11_21,1).
bond(d11,d11_19,d11_22,1).
bond(d11,d11_18,d11_23,1).
bond(d11,d11_24,d11_14,1).
bond(d11,d11_25,d11_24,2).
bond(d11,d11_26,d11_24,2).
atm(d110,d110_1,c,22,-0.113).
atm(d110,d110_2,c,22,-0.113).
atm(d110,d110_3,c,22,-0.113).
atm(d110,d110_4,c,29,0.017).
atm(d110,d110_5,c,22,-0.113).
atm(d110,d110_6,c,22,-0.113).
atm(d110,d110_7,h,3,0.147).
atm(d110,d110_8,h,3,0.147).
atm(d110,d110_9,h,3,0.147).
atm(d110,d110_10,c,29,0.017).
atm(d110,d110_11,c,22,-0.113).
atm(d110,d110_12,c,22,-0.113).
atm(d110,d110_13,c,22,-0.113).
atm(d110,d110_14,c,22,-0.113).
atm(d110,d110_15,c,22,-0.113).
atm(d110,d110_16,h,3,0.147).
atm(d110,d110_17,h,3,0.147).
atm(d110,d110_18,h,3,0.147).
atm(d110,d110_19,n,38,0.818).
atm(d110,d110_20,n,38,0.818).
atm(d110,d110_21,n,38,0.818).
atm(d110,d110_22,n,38,0.818).
atm(d110,d110_23,o,40,-0.382).
atm(d110,d110_24,o,40,-0.382).
atm(d110,d110_25,o,40,-0.382).
atm(d110,d110_26,o,40,-0.383).
atm(d110,d110_27,o,40,-0.383).
atm(d110,d110_28,o,40,-0.382).
atm(d110,d110_29,o,40,-0.382).
atm(d110,d110_30,o,40,-0.382).

bond(d110,d110_1,d110_2,7).
bond(d110,d110_2,d110_3,7).
bond(d110,d110_3,d110_4,7).
bond(d110,d110_4,d110_5,7).
bond(d110,d110_5,d110_6,7).
bond(d110,d110_6,d110_1,7).
bond(d110,d110_1,d110_7,1).
bond(d110,d110_2,d110_8,1).
bond(d110,d110_6,d110_9,1).
bond(d110,d110_10,d110_11,7).
bond(d110,d110_11,d110_12,7).
bond(d110,d110_12,d110_13,7).
bond(d110,d110_13,d110_14,7).
bond(d110,d110_14,d110_15,7).
bond(d110,d110_15,d110_10,7).
bond(d110,d110_11,d110_16,1).
bond(d110,d110_12,d110_17,1).
bond(d110,d110_14,d110_18,1).
bond(d110,d110_4,d110_10,1).
bond(d110,d110_15,d110_19,1).
bond(d110,d110_13,d110_20,1).
bond(d110,d110_5,d110_21,1).
bond(d110,d110_3,d110_22,1).
bond(d110,d110_21,d110_23,2).
bond(d110,d110_21,d110_24,2).
bond(d110,d110_19,d110_25,2).
bond(d110,d110_19,d110_26,2).
bond(d110,d110_20,d110_27,2).
bond(d110,d110_20,d110_28,2).
bond(d110,d110_22,d110_29,2).
bond(d110,d110_22,d110_30,2).
atm(d111,d111_1,c,22,-0.122).
atm(d111,d111_2,c,22,-0.122).
atm(d111,d111_3,c,22,-0.122).
atm(d111,d111_4,c,22,-0.122).
atm(d111,d111_5,c,22,0.008).
atm(d111,d111_6,c,22,-0.122).
atm(d111,d111_7,h,3,0.138).
atm(d111,d111_8,h,3,0.138).
atm(d111,d111_9,h,3,0.138).
atm(d111,d111_10,c,14,0.238).
atm(d111,d111_11,o,41,-0.421).
atm(d111,d111_12,h,3,0.208).
atm(d111,d111_13,h,3,0.138).
atm(d111,d111_14,n,38,0.809).
atm(d111,d111_15,o,40,-0.392).
atm(d111,d111_16,o,40,-0.392).

bond(d111,d111_1,d111_2,7).
bond(d111,d111_2,d111_3,7).
bond(d111,d111_3,d111_4,7).
bond(d111,d111_4,d111_5,7).
bond(d111,d111_5,d111_6,7).
bond(d111,d111_6,d111_1,7).
bond(d111,d111_1,d111_7,1).
bond(d111,d111_3,d111_8,1).
bond(d111,d111_6,d111_9,1).
bond(d111,d111_5,d111_10,1).
bond(d111,d111_10,d111_11,2).
bond(d111,d111_10,d111_12,1).
bond(d111,d111_4,d111_13,1).
bond(d111,d111_2,d111_14,1).
bond(d111,d111_14,d111_15,2).
bond(d111,d111_14,d111_16,2).
atm(d112,d112_1,c,22,-0.125).
atm(d112,d112_2,c,22,-0.125).
atm(d112,d112_3,c,27,0.005).
atm(d112,d112_4,c,27,0.005).
atm(d112,d112_5,c,22,0.005).
atm(d112,d112_6,c,22,-0.124).
atm(d112,d112_7,h,3,0.136).
atm(d112,d112_8,h,3,0.136).
atm(d112,d112_9,h,3,0.136).
atm(d112,d112_10,c,22,-0.125).
atm(d112,d112_11,c,22,-0.124).
atm(d112,d112_12,c,22,-0.125).
atm(d112,d112_13,c,22,0.005).
atm(d112,d112_14,h,3,0.136).
atm(d112,d112_15,h,3,0.136).
atm(d112,d112_16,c,10,-0.095).
atm(d112,d112_17,c,10,-0.095).
atm(d112,d112_18,h,3,0.055).
atm(d112,d112_19,h,3,0.055).
atm(d112,d112_20,h,3,0.055).
atm(d112,d112_21,h,3,0.055).
atm(d112,d112_22,n,38,0.806).
atm(d112,d112_23,o,40,-0.394).
atm(d112,d112_24,o,40,-0.394).

bond(d112,d112_1,d112_2,7).
bond(d112,d112_2,d112_3,7).
bond(d112,d112_3,d112_4,7).
bond(d112,d112_4,d112_5,7).
bond(d112,d112_5,d112_6,7).
bond(d112,d112_6,d112_1,7).
bond(d112,d112_1,d112_7,1).
bond(d112,d112_2,d112_8,1).
bond(d112,d112_6,d112_9,1).
bond(d112,d112_3,d112_10,7).
bond(d112,d112_10,d112_11,7).
bond(d112,d112_11,d112_12,7).
bond(d112,d112_12,d112_13,7).
bond(d112,d112_13,d112_4,7).
bond(d112,d112_10,d112_14,1).
bond(d112,d112_11,d112_15,1).
bond(d112,d112_5,d112_16,1).
bond(d112,d112_16,d112_17,1).
bond(d112,d112_16,d112_18,1).
bond(d112,d112_16,d112_19,1).
bond(d112,d112_17,d112_13,1).
bond(d112,d112_17,d112_20,1).
bond(d112,d112_17,d112_21,1).
bond(d112,d112_12,d112_22,1).
bond(d112,d112_22,d112_23,2).
bond(d112,d112_22,d112_24,2).
atm(d113,d113_1,c,22,-0.12).
atm(d113,d113_2,c,22,-0.12).
atm(d113,d113_3,c,22,-0.12).
atm(d113,d113_4,c,22,-0.12).
atm(d113,d113_5,c,22,0.21).
atm(d113,d113_6,c,22,-0.12).
atm(d113,d113_7,h,3,0.14).
atm(d113,d113_8,h,3,0.14).
atm(d113,d113_9,h,3,0.14).
atm(d113,d113_10,n,32,-0.77).
atm(d113,d113_11,n,38,0.81).
atm(d113,d113_12,n,38,0.81).
atm(d113,d113_13,o,40,-0.39).
atm(d113,d113_14,o,40,-0.39).
atm(d113,d113_15,o,40,-0.39).
atm(d113,d113_16,o,40,-0.39).
atm(d113,d113_17,h,1,0.34).
atm(d113,d113_18,h,1,0.34).

bond(d113,d113_1,d113_2,7).
bond(d113,d113_2,d113_3,7).
bond(d113,d113_3,d113_4,7).
bond(d113,d113_4,d113_5,7).
bond(d113,d113_5,d113_6,7).
bond(d113,d113_6,d113_1,7).
bond(d113,d113_1,d113_7,1).
bond(d113,d113_3,d113_8,1).
bond(d113,d113_6,d113_9,1).
bond(d113,d113_5,d113_10,1).
bond(d113,d113_4,d113_11,1).
bond(d113,d113_2,d113_12,1).
bond(d113,d113_11,d113_13,2).
bond(d113,d113_11,d113_14,2).
bond(d113,d113_12,d113_15,2).
bond(d113,d113_12,d113_16,2).
bond(d113,d113_10,d113_17,1).
bond(d113,d113_10,d113_18,1).
atm(d114,d114_1,c,22,-0.128).
atm(d114,d114_2,c,22,-0.128).
atm(d114,d114_3,c,22,-0.128).
atm(d114,d114_4,c,29,0.002).
atm(d114,d114_5,c,22,-0.128).
atm(d114,d114_6,c,22,-0.128).
atm(d114,d114_7,h,3,0.132).
atm(d114,d114_8,h,3,0.132).
atm(d114,d114_9,h,3,0.132).
atm(d114,d114_10,h,3,0.132).
atm(d114,d114_11,c,29,0.002).
atm(d114,d114_12,c,22,-0.128).
atm(d114,d114_13,c,22,-0.128).
atm(d114,d114_14,c,22,-0.128).
atm(d114,d114_15,c,22,-0.128).
atm(d114,d114_16,c,22,0.202).
atm(d114,d114_17,h,3,0.132).
atm(d114,d114_18,h,3,0.132).
atm(d114,d114_19,h,3,0.132).
atm(d114,d114_20,h,3,0.132).
atm(d114,d114_21,n,32,-0.779).
atm(d114,d114_22,n,38,0.801).
atm(d114,d114_23,o,40,-0.398).
atm(d114,d114_24,o,40,-0.398).
atm(d114,d114_25,h,1,0.332).
atm(d114,d114_26,h,1,0.332).

bond(d114,d114_1,d114_2,7).
bond(d114,d114_2,d114_3,7).
bond(d114,d114_3,d114_4,7).
bond(d114,d114_4,d114_5,7).
bond(d114,d114_5,d114_6,7).
bond(d114,d114_6,d114_1,7).
bond(d114,d114_2,d114_7,1).
bond(d114,d114_3,d114_8,1).
bond(d114,d114_5,d114_9,1).
bond(d114,d114_6,d114_10,1).
bond(d114,d114_11,d114_12,7).
bond(d114,d114_12,d114_13,7).
bond(d114,d114_13,d114_14,7).
bond(d114,d114_14,d114_15,7).
bond(d114,d114_15,d114_16,7).
bond(d114,d114_16,d114_11,7).
bond(d114,d114_12,d114_17,1).
bond(d114,d114_13,d114_18,1).
bond(d114,d114_14,d114_19,1).
bond(d114,d114_15,d114_20,1).
bond(d114,d114_4,d114_11,1).
bond(d114,d114_16,d114_21,1).
bond(d114,d114_1,d114_22,1).
bond(d114,d114_22,d114_23,2).
bond(d114,d114_22,d114_24,2).
bond(d114,d114_21,d114_25,1).
bond(d114,d114_21,d114_26,1).
atm(d115,d115_1,c,22,-0.122).
atm(d115,d115_2,c,22,-0.122).
atm(d115,d115_3,c,22,-0.122).
atm(d115,d115_4,c,22,0.108).
atm(d115,d115_5,c,22,-0.122).
atm(d115,d115_6,c,22,-0.122).
atm(d115,d115_7,h,3,0.139).
atm(d115,d115_8,h,3,0.139).
atm(d115,d115_9,h,3,0.139).
atm(d115,d115_10,c,10,0.108).
atm(d115,d115_11,c,16,-0.191).
atm(d115,d115_12,h,3,0.058).
atm(d115,d115_13,h,3,0.058).
atm(d115,d115_14,c,16,-0.191).
atm(d115,d115_15,h,3,0.108).
atm(d115,d115_16,h,3,0.108).
atm(d115,d115_17,n,38,0.809).
atm(d115,d115_18,o,40,-0.391).
atm(d115,d115_19,o,40,-0.391).

bond(d115,d115_1,d115_2,7).
bond(d115,d115_2,d115_3,7).
bond(d115,d115_3,d115_4,7).
bond(d115,d115_4,d115_5,7).
bond(d115,d115_5,d115_6,7).
bond(d115,d115_6,d115_1,7).
bond(d115,d115_1,d115_7,1).
bond(d115,d115_2,d115_8,1).
bond(d115,d115_5,d115_9,1).
bond(d115,d115_3,d115_10,1).
bond(d115,d115_10,d115_11,1).
bond(d115,d115_10,d115_12,1).
bond(d115,d115_10,d115_13,1).
bond(d115,d115_4,d115_14,1).
bond(d115,d115_14,d115_11,2).
bond(d115,d115_14,d115_15,1).
bond(d115,d115_11,d115_16,1).
bond(d115,d115_6,d115_17,1).
bond(d115,d115_17,d115_18,2).
bond(d115,d115_17,d115_19,2).
atm(d116,d116_1,c,22,-0.121).
atm(d116,d116_2,c,22,0.31).
atm(d116,d116_3,c,22,-0.121).
atm(d116,d116_4,c,22,-0.121).
atm(d116,d116_5,c,22,-0.121).
atm(d116,d116_6,c,22,-0.121).
atm(d116,d116_7,h,3,0.139).
atm(d116,d116_8,h,3,0.139).
atm(d116,d116_9,h,3,0.139).
atm(d116,d116_10,h,3,0.139).
atm(d116,d116_11,n,38,0.81).
atm(d116,d116_12,f,92,-0.291).
atm(d116,d116_13,o,40,-0.39).
atm(d116,d116_14,o,40,-0.39).

bond(d116,d116_1,d116_2,7).
bond(d116,d116_2,d116_3,7).
bond(d116,d116_3,d116_4,7).
bond(d116,d116_4,d116_5,7).
bond(d116,d116_5,d116_6,7).
bond(d116,d116_6,d116_1,7).
bond(d116,d116_1,d116_7,1).
bond(d116,d116_3,d116_8,1).
bond(d116,d116_4,d116_9,1).
bond(d116,d116_6,d116_10,1).
bond(d116,d116_5,d116_11,1).
bond(d116,d116_2,d116_12,1).
bond(d116,d116_11,d116_13,2).
bond(d116,d116_11,d116_14,2).
atm(d117,d117_1,c,22,-0.116).
atm(d117,d117_2,c,22,-0.116).
atm(d117,d117_3,c,22,-0.116).
atm(d117,d117_4,c,29,0.014).
atm(d117,d117_5,c,22,-0.116).
atm(d117,d117_6,c,22,-0.116).
atm(d117,d117_7,h,3,0.144).
atm(d117,d117_8,h,3,0.144).
atm(d117,d117_9,h,3,0.144).
atm(d117,d117_10,h,3,0.144).
atm(d117,d117_11,c,29,0.014).
atm(d117,d117_12,c,22,-0.116).
atm(d117,d117_13,c,22,-0.116).
atm(d117,d117_14,c,22,-0.116).
atm(d117,d117_15,c,22,-0.116).
atm(d117,d117_16,c,22,-0.116).
atm(d117,d117_17,h,3,0.144).
atm(d117,d117_18,h,3,0.144).
atm(d117,d117_19,h,3,0.144).
atm(d117,d117_20,n,38,0.813).
atm(d117,d117_21,n,38,0.813).
atm(d117,d117_22,n,38,0.814).
atm(d117,d117_23,o,40,-0.386).
atm(d117,d117_24,o,40,-0.386).
atm(d117,d117_25,o,40,-0.386).
atm(d117,d117_26,o,40,-0.386).
atm(d117,d117_27,o,40,-0.386).
atm(d117,d117_28,o,40,-0.386).

bond(d117,d117_1,d117_2,7).
bond(d117,d117_2,d117_3,7).
bond(d117,d117_3,d117_4,7).
bond(d117,d117_4,d117_5,7).
bond(d117,d117_5,d117_6,7).
bond(d117,d117_6,d117_1,7).
bond(d117,d117_1,d117_7,1).
bond(d117,d117_2,d117_8,1).
bond(d117,d117_3,d117_9,1).
bond(d117,d117_5,d117_10,1).
bond(d117,d117_11,d117_12,7).
bond(d117,d117_12,d117_13,7).
bond(d117,d117_13,d117_14,7).
bond(d117,d117_14,d117_15,7).
bond(d117,d117_15,d117_16,7).
bond(d117,d117_16,d117_11,7).
bond(d117,d117_12,d117_17,1).
bond(d117,d117_13,d117_18,1).
bond(d117,d117_15,d117_19,1).
bond(d117,d117_4,d117_11,1).
bond(d117,d117_16,d117_20,1).
bond(d117,d117_14,d117_21,1).
bond(d117,d117_6,d117_22,1).
bond(d117,d117_22,d117_23,2).
bond(d117,d117_22,d117_24,2).
bond(d117,d117_20,d117_25,2).
bond(d117,d117_20,d117_26,2).
bond(d117,d117_21,d117_27,2).
bond(d117,d117_21,d117_28,2).
atm(d118,d118_1,c,10,-0.091).
atm(d118,d118_2,c,10,-0.091).
atm(d118,d118_3,c,22,0.009).
atm(d118,d118_4,c,22,0.009).
atm(d118,d118_5,c,10,-0.091).
atm(d118,d118_6,c,10,-0.091).
atm(d118,d118_7,c,22,-0.122).
atm(d118,d118_8,c,27,0.009).
atm(d118,d118_9,c,27,-0.091).
atm(d118,d118_10,c,27,0.009).
atm(d118,d118_11,c,22,-0.121).
atm(d118,d118_12,c,22,-0.122).
atm(d118,d118_13,c,27,0.009).
atm(d118,d118_14,c,27,-0.091).
atm(d118,d118_15,c,27,0.009).
atm(d118,d118_16,c,22,-0.121).
atm(d118,d118_17,c,22,-0.121).
atm(d118,d118_18,c,22,-0.121).
atm(d118,d118_19,c,22,-0.121).
atm(d118,d118_20,c,22,-0.121).
atm(d118,d118_21,n,38,0.808).
atm(d118,d118_22,o,40,-0.392).
atm(d118,d118_23,o,40,-0.392).
atm(d118,d118_24,h,3,0.138).
atm(d118,d118_25,h,3,0.138).
atm(d118,d118_26,h,3,0.138).
atm(d118,d118_27,h,3,0.138).
atm(d118,d118_28,h,3,0.138).
atm(d118,d118_29,h,3,0.138).
atm(d118,d118_30,h,3,0.138).
atm(d118,d118_31,h,3,0.059).
atm(d118,d118_32,h,3,0.059).
atm(d118,d118_33,h,3,0.059).
atm(d118,d118_34,h,3,0.059).
atm(d118,d118_35,h,3,0.059).
atm(d118,d118_36,h,3,0.059).
atm(d118,d118_37,h,3,0.059).
atm(d118,d118_38,h,3,0.059).

bond(d118,d118_1,d118_2,1).
bond(d118,d118_2,d118_3,1).
bond(d118,d118_3,d118_4,7).
bond(d118,d118_4,d118_5,1).
bond(d118,d118_5,d118_6,1).
bond(d118,d118_6,d118_1,1).
bond(d118,d118_3,d118_7,7).
bond(d118,d118_7,d118_8,7).
bond(d118,d118_8,d118_9,7).
bond(d118,d118_9,d118_10,7).
bond(d118,d118_10,d118_4,7).
bond(d118,d118_8,d118_11,7).
bond(d118,d118_11,d118_12,7).
bond(d118,d118_12,d118_13,7).
bond(d118,d118_13,d118_14,7).
bond(d118,d118_14,d118_9,7).
bond(d118,d118_14,d118_15,7).
bond(d118,d118_15,d118_16,7).
bond(d118,d118_16,d118_17,7).
bond(d118,d118_17,d118_10,7).
bond(d118,d118_13,d118_18,7).
bond(d118,d118_18,d118_19,7).
bond(d118,d118_19,d118_20,7).
bond(d118,d118_20,d118_15,7).
bond(d118,d118_18,d118_21,1).
bond(d118,d118_21,d118_22,2).
bond(d118,d118_21,d118_23,2).
bond(d118,d118_7,d118_24,1).
bond(d118,d118_11,d118_25,1).
bond(d118,d118_12,d118_26,1).
bond(d118,d118_16,d118_27,1).
bond(d118,d118_17,d118_28,1).
bond(d118,d118_19,d118_29,1).
bond(d118,d118_20,d118_30,1).
bond(d118,d118_2,d118_31,1).
bond(d118,d118_2,d118_32,1).
bond(d118,d118_1,d118_33,1).
bond(d118,d118_1,d118_34,1).
bond(d118,d118_6,d118_35,1).
bond(d118,d118_6,d118_36,1).
bond(d118,d118_5,d118_37,1).
bond(d118,d118_5,d118_38,1).
atm(d119,d119_1,c,22,-0.116).
atm(d119,d119_2,c,22,-0.116).
atm(d119,d119_3,c,22,-0.116).
atm(d119,d119_4,c,29,0.014).
atm(d119,d119_5,c,22,-0.116).
atm(d119,d119_6,c,22,-0.116).
atm(d119,d119_7,h,3,0.144).
atm(d119,d119_8,h,3,0.144).
atm(d119,d119_9,h,3,0.144).
atm(d119,d119_10,h,3,0.144).
atm(d119,d119_11,c,29,0.014).
atm(d119,d119_12,c,22,-0.116).
atm(d119,d119_13,c,22,-0.116).
atm(d119,d119_14,c,22,-0.116).
atm(d119,d119_15,c,22,-0.116).
atm(d119,d119_16,c,22,-0.116).
atm(d119,d119_17,h,3,0.144).
atm(d119,d119_18,h,3,0.144).
atm(d119,d119_19,h,3,0.144).
atm(d119,d119_20,n,38,0.813).
atm(d119,d119_21,n,38,0.813).
atm(d119,d119_22,n,38,0.814).
atm(d119,d119_23,o,40,-0.386).
atm(d119,d119_24,o,40,-0.386).
atm(d119,d119_25,o,40,-0.386).
atm(d119,d119_26,o,40,-0.386).
atm(d119,d119_27,o,40,-0.386).
atm(d119,d119_28,o,40,-0.386).

bond(d119,d119_1,d119_2,7).
bond(d119,d119_2,d119_3,7).
bond(d119,d119_3,d119_4,7).
bond(d119,d119_4,d119_5,7).
bond(d119,d119_5,d119_6,7).
bond(d119,d119_6,d119_1,7).
bond(d119,d119_1,d119_7,1).
bond(d119,d119_2,d119_8,1).
bond(d119,d119_3,d119_9,1).
bond(d119,d119_6,d119_10,1).
bond(d119,d119_11,d119_12,7).
bond(d119,d119_12,d119_13,7).
bond(d119,d119_13,d119_14,7).
bond(d119,d119_14,d119_15,7).
bond(d119,d119_15,d119_16,7).
bond(d119,d119_16,d119_11,7).
bond(d119,d119_12,d119_17,1).
bond(d119,d119_13,d119_18,1).
bond(d119,d119_15,d119_19,1).
bond(d119,d119_4,d119_11,1).
bond(d119,d119_16,d119_20,1).
bond(d119,d119_14,d119_21,1).
bond(d119,d119_5,d119_22,1).
bond(d119,d119_22,d119_23,2).
bond(d119,d119_22,d119_24,2).
bond(d119,d119_20,d119_25,2).
bond(d119,d119_20,d119_26,2).
bond(d119,d119_21,d119_27,2).
bond(d119,d119_21,d119_28,2).
atm(d12,d12_1,c,22,-0.111).
atm(d12,d12_2,c,22,-0.111).
atm(d12,d12_3,c,27,0.019).
atm(d12,d12_4,c,27,-0.081).
atm(d12,d12_5,c,27,0.019).
atm(d12,d12_6,c,22,-0.111).
atm(d12,d12_7,h,3,0.149).
atm(d12,d12_8,h,3,0.15).
atm(d12,d12_9,c,27,-0.081).
atm(d12,d12_10,c,27,0.019).
atm(d12,d12_11,c,22,-0.111).
atm(d12,d12_12,c,22,-0.111).
atm(d12,d12_13,h,3,0.15).
atm(d12,d12_14,h,3,0.15).
atm(d12,d12_15,c,22,-0.111).
atm(d12,d12_16,c,22,-0.111).
atm(d12,d12_17,c,27,0.019).
atm(d12,d12_18,h,3,0.149).
atm(d12,d12_19,h,3,0.149).
atm(d12,d12_20,c,22,-0.111).
atm(d12,d12_21,c,22,-0.111).
atm(d12,d12_22,c,22,-0.111).
atm(d12,d12_23,h,3,0.15).
atm(d12,d12_24,h,3,0.149).
atm(d12,d12_25,n,38,0.82).
atm(d12,d12_26,o,40,-0.38).
atm(d12,d12_27,o,40,-0.38).
atm(d12,d12_28,n,32,-0.76).
atm(d12,d12_29,h,1,0.35).
atm(d12,d12_30,h,1,0.35).

bond(d12,d12_1,d12_2,7).
bond(d12,d12_2,d12_3,7).
bond(d12,d12_3,d12_4,7).
bond(d12,d12_4,d12_5,7).
bond(d12,d12_5,d12_6,7).
bond(d12,d12_6,d12_1,7).
bond(d12,d12_1,d12_7,1).
bond(d12,d12_2,d12_8,1).
bond(d12,d12_4,d12_9,7).
bond(d12,d12_9,d12_10,7).
bond(d12,d12_10,d12_11,7).
bond(d12,d12_11,d12_12,7).
bond(d12,d12_12,d12_5,7).
bond(d12,d12_11,d12_13,1).
bond(d12,d12_12,d12_14,1).
bond(d12,d12_3,d12_15,7).
bond(d12,d12_15,d12_16,7).
bond(d12,d12_16,d12_17,7).
bond(d12,d12_17,d12_9,7).
bond(d12,d12_15,d12_18,1).
bond(d12,d12_16,d12_19,1).
bond(d12,d12_17,d12_20,7).
bond(d12,d12_20,d12_21,7).
bond(d12,d12_21,d12_22,7).
bond(d12,d12_22,d12_10,7).
bond(d12,d12_20,d12_23,1).
bond(d12,d12_21,d12_24,1).
bond(d12,d12_25,d12_6,1).
bond(d12,d12_26,d12_25,2).
bond(d12,d12_27,d12_25,2).
bond(d12,d12_22,d12_28,1).
bond(d12,d12_28,d12_29,1).
bond(d12,d12_28,d12_30,1).
atm(d120,d120_1,c,22,-0.107).
atm(d120,d120_2,c,22,-0.107).
atm(d120,d120_3,c,26,0.323).
atm(d120,d120_4,c,26,0.123).
atm(d120,d120_5,c,22,-0.107).
atm(d120,d120_6,c,22,-0.107).
atm(d120,d120_7,n,34,-0.517).
atm(d120,d120_8,n,34,-0.377).
atm(d120,d120_9,c,21,-0.107).
atm(d120,d120_10,c,10,0.113).
atm(d120,d120_11,n,38,0.823).
atm(d120,d120_12,o,40,-0.377).
atm(d120,d120_13,o,40,-0.377).
atm(d120,d120_14,h,3,0.153).
atm(d120,d120_15,h,3,0.153).
atm(d120,d120_16,h,3,0.123).
atm(d120,d120_17,h,3,0.153).
atm(d120,d120_18,h,3,0.073).
atm(d120,d120_19,h,3,0.073).
atm(d120,d120_20,h,3,0.073).

bond(d120,d120_1,d120_2,7).
bond(d120,d120_2,d120_3,7).
bond(d120,d120_3,d120_4,7).
bond(d120,d120_4,d120_5,7).
bond(d120,d120_5,d120_6,7).
bond(d120,d120_6,d120_1,7).
bond(d120,d120_3,d120_7,7).
bond(d120,d120_7,d120_8,7).
bond(d120,d120_4,d120_9,7).
bond(d120,d120_9,d120_8,7).
bond(d120,d120_7,d120_10,1).
bond(d120,d120_6,d120_11,1).
bond(d120,d120_11,d120_12,2).
bond(d120,d120_11,d120_13,2).
bond(d120,d120_1,d120_14,1).
bond(d120,d120_2,d120_15,1).
bond(d120,d120_5,d120_16,1).
bond(d120,d120_9,d120_17,1).
bond(d120,d120_10,d120_18,1).
bond(d120,d120_10,d120_19,1).
bond(d120,d120_10,d120_20,1).
atm(d121,d121_1,c,22,-0.115).
atm(d121,d121_2,c,22,-0.115).
atm(d121,d121_3,c,27,0.015).
atm(d121,d121_4,c,27,-0.085).
atm(d121,d121_5,c,28,0.015).
atm(d121,d121_6,c,22,-0.115).
atm(d121,d121_7,h,3,0.145).
atm(d121,d121_8,h,3,0.145).
atm(d121,d121_9,c,22,-0.115).
atm(d121,d121_10,c,22,-0.115).
atm(d121,d121_11,c,22,-0.115).
atm(d121,d121_12,c,29,0.015).
atm(d121,d121_13,h,3,0.145).
atm(d121,d121_14,c,28,0.015).
atm(d121,d121_15,c,28,0.015).
atm(d121,d121_16,c,22,-0.115).
atm(d121,d121_17,c,22,-0.115).
atm(d121,d121_18,c,22,-0.115).
atm(d121,d121_19,c,22,-0.115).
atm(d121,d121_20,h,3,0.146).
atm(d121,d121_21,h,3,0.145).
atm(d121,d121_22,h,3,0.145).
atm(d121,d121_23,h,3,0.145).
atm(d121,d121_24,n,38,0.816).
atm(d121,d121_25,n,38,0.816).
atm(d121,d121_26,n,38,0.816).
atm(d121,d121_27,o,40,-0.384).
atm(d121,d121_28,o,40,-0.384).
atm(d121,d121_29,o,40,-0.384).
atm(d121,d121_30,o,40,-0.384).
atm(d121,d121_31,o,40,-0.384).
atm(d121,d121_32,o,40,-0.384).

bond(d121,d121_1,d121_2,7).
bond(d121,d121_2,d121_3,7).
bond(d121,d121_3,d121_4,7).
bond(d121,d121_4,d121_5,7).
bond(d121,d121_5,d121_6,7).
bond(d121,d121_6,d121_1,7).
bond(d121,d121_2,d121_7,1).
bond(d121,d121_6,d121_8,1).
bond(d121,d121_3,d121_9,7).
bond(d121,d121_9,d121_10,7).
bond(d121,d121_10,d121_11,7).
bond(d121,d121_11,d121_12,7).
bond(d121,d121_12,d121_4,7).
bond(d121,d121_9,d121_13,1).
bond(d121,d121_14,d121_15,7).
bond(d121,d121_15,d121_16,7).
bond(d121,d121_16,d121_17,7).
bond(d121,d121_17,d121_18,7).
bond(d121,d121_18,d121_19,7).
bond(d121,d121_19,d121_14,7).
bond(d121,d121_16,d121_20,1).
bond(d121,d121_17,d121_21,1).
bond(d121,d121_18,d121_22,1).
bond(d121,d121_19,d121_23,1).
bond(d121,d121_14,d121_5,1).
bond(d121,d121_15,d121_12,1).
bond(d121,d121_11,d121_24,1).
bond(d121,d121_10,d121_25,1).
bond(d121,d121_1,d121_26,1).
bond(d121,d121_24,d121_27,2).
bond(d121,d121_24,d121_28,2).
bond(d121,d121_25,d121_29,2).
bond(d121,d121_25,d121_30,2).
bond(d121,d121_26,d121_31,2).
bond(d121,d121_26,d121_32,2).
atm(d122,d122_1,c,22,-0.113).
atm(d122,d122_2,c,22,-0.113).
atm(d122,d122_3,c,27,0.017).
atm(d122,d122_4,c,27,-0.083).
atm(d122,d122_5,c,22,-0.113).
atm(d122,d122_6,c,22,-0.113).
atm(d122,d122_7,h,3,0.146).
atm(d122,d122_8,h,3,0.146).
atm(d122,d122_9,h,3,0.146).
atm(d122,d122_10,h,3,0.146).
atm(d122,d122_11,c,22,-0.113).
atm(d122,d122_12,c,22,-0.113).
atm(d122,d122_13,c,195,-0.083).
atm(d122,d122_14,c,195,-0.083).
atm(d122,d122_15,h,3,0.146).
atm(d122,d122_16,c,27,-0.083).
atm(d122,d122_17,c,27,0.017).
atm(d122,d122_18,c,22,-0.113).
atm(d122,d122_19,c,22,-0.113).
atm(d122,d122_20,h,3,0.146).
atm(d122,d122_21,h,3,0.146).
atm(d122,d122_22,c,22,-0.113).
atm(d122,d122_23,c,22,-0.113).
atm(d122,d122_24,c,22,-0.113).
atm(d122,d122_25,c,22,-0.113).
atm(d122,d122_26,h,3,0.146).
atm(d122,d122_27,h,3,0.146).
atm(d122,d122_28,h,3,0.146).
atm(d122,d122_29,h,3,0.146).
atm(d122,d122_30,n,38,0.816).
atm(d122,d122_31,o,40,-0.384).
atm(d122,d122_32,o,40,-0.384).

bond(d122,d122_1,d122_2,7).
bond(d122,d122_2,d122_3,7).
bond(d122,d122_3,d122_4,7).
bond(d122,d122_4,d122_5,7).
bond(d122,d122_5,d122_6,7).
bond(d122,d122_6,d122_1,7).
bond(d122,d122_1,d122_7,1).
bond(d122,d122_2,d122_8,1).
bond(d122,d122_5,d122_9,1).
bond(d122,d122_6,d122_10,1).
bond(d122,d122_3,d122_11,7).
bond(d122,d122_11,d122_12,7).
bond(d122,d122_12,d122_13,7).
bond(d122,d122_13,d122_14,7).
bond(d122,d122_14,d122_4,7).
bond(d122,d122_12,d122_15,1).
bond(d122,d122_13,d122_16,7).
bond(d122,d122_16,d122_17,7).
bond(d122,d122_17,d122_18,7).
bond(d122,d122_18,d122_19,7).
bond(d122,d122_19,d122_14,7).
bond(d122,d122_18,d122_20,1).
bond(d122,d122_19,d122_21,1).
bond(d122,d122_16,d122_22,7).
bond(d122,d122_22,d122_23,7).
bond(d122,d122_23,d122_24,7).
bond(d122,d122_24,d122_25,7).
bond(d122,d122_25,d122_17,7).
bond(d122,d122_22,d122_26,1).
bond(d122,d122_23,d122_27,1).
bond(d122,d122_24,d122_28,1).
bond(d122,d122_25,d122_29,1).
bond(d122,d122_11,d122_30,1).
bond(d122,d122_30,d122_31,2).
bond(d122,d122_30,d122_32,2).
atm(d123,d123_1,c,22,-0.124).
atm(d123,d123_2,c,22,-0.124).
atm(d123,d123_3,c,22,-0.124).
atm(d123,d123_4,c,29,0.006).
atm(d123,d123_5,c,22,-0.173).
atm(d123,d123_6,c,22,-0.124).
atm(d123,d123_7,h,3,0.137).
atm(d123,d123_8,h,3,0.137).
atm(d123,d123_9,h,3,0.137).
atm(d123,d123_10,h,3,0.137).
atm(d123,d123_11,c,29,0.006).
atm(d123,d123_12,c,22,-0.124).
atm(d123,d123_13,c,22,-0.124).
atm(d123,d123_14,c,22,-0.124).
atm(d123,d123_15,c,22,-0.173).
atm(d123,d123_16,c,22,-0.124).
atm(d123,d123_17,h,3,0.136).
atm(d123,d123_18,h,3,0.136).
atm(d123,d123_19,c,10,0.006).
atm(d123,d123_20,h,3,0.056).
atm(d123,d123_21,h,3,0.056).
atm(d123,d123_22,h,3,0.056).
atm(d123,d123_23,h,3,0.137).
atm(d123,d123_24,c,10,0.006).
atm(d123,d123_25,h,3,0.056).
atm(d123,d123_26,h,3,0.056).
atm(d123,d123_27,h,3,0.056).
atm(d123,d123_28,n,38,0.807).
atm(d123,d123_29,o,40,-0.393).
atm(d123,d123_30,o,40,-0.393).

bond(d123,d123_1,d123_2,7).
bond(d123,d123_2,d123_3,7).
bond(d123,d123_3,d123_4,7).
bond(d123,d123_4,d123_5,7).
bond(d123,d123_5,d123_6,7).
bond(d123,d123_6,d123_1,7).
bond(d123,d123_1,d123_7,1).
bond(d123,d123_2,d123_8,1).
bond(d123,d123_3,d123_9,1).
bond(d123,d123_6,d123_10,1).
bond(d123,d123_11,d123_12,7).
bond(d123,d123_12,d123_13,7).
bond(d123,d123_13,d123_14,7).
bond(d123,d123_14,d123_15,7).
bond(d123,d123_15,d123_16,7).
bond(d123,d123_16,d123_11,7).
bond(d123,d123_12,d123_17,1).
bond(d123,d123_13,d123_18,1).
bond(d123,d123_4,d123_11,1).
bond(d123,d123_15,d123_19,1).
bond(d123,d123_19,d123_20,1).
bond(d123,d123_19,d123_21,1).
bond(d123,d123_19,d123_22,1).
bond(d123,d123_16,d123_23,1).
bond(d123,d123_5,d123_24,1).
bond(d123,d123_24,d123_25,1).
bond(d123,d123_24,d123_26,1).
bond(d123,d123_24,d123_27,1).
bond(d123,d123_14,d123_28,1).
bond(d123,d123_28,d123_29,2).
bond(d123,d123_28,d123_30,2).
atm(d124,d124_1,c,22,-0.121).
atm(d124,d124_2,c,22,0.309).
atm(d124,d124_3,c,22,-0.121).
atm(d124,d124_4,c,22,0.31).
atm(d124,d124_5,c,22,-0.121).
atm(d124,d124_6,c,22,-0.121).
atm(d124,d124_7,h,3,0.139).
atm(d124,d124_8,h,3,0.139).
atm(d124,d124_9,h,3,0.139).
atm(d124,d124_10,n,38,0.81).
atm(d124,d124_11,f,92,-0.291).
atm(d124,d124_12,f,92,-0.291).
atm(d124,d124_13,o,40,-0.39).
atm(d124,d124_14,o,40,-0.39).

bond(d124,d124_1,d124_2,7).
bond(d124,d124_2,d124_3,7).
bond(d124,d124_3,d124_4,7).
bond(d124,d124_4,d124_5,7).
bond(d124,d124_5,d124_6,7).
bond(d124,d124_6,d124_1,7).
bond(d124,d124_1,d124_7,1).
bond(d124,d124_3,d124_8,1).
bond(d124,d124_6,d124_9,1).
bond(d124,d124_5,d124_10,1).
bond(d124,d124_4,d124_11,1).
bond(d124,d124_2,d124_12,1).
bond(d124,d124_10,d124_13,2).
bond(d124,d124_10,d124_14,2).
atm(d125,d125_1,c,22,-0.119).
atm(d125,d125_2,c,22,-0.119).
atm(d125,d125_3,c,22,0.011).
atm(d125,d125_4,c,195,-0.089).
atm(d125,d125_5,c,195,0.011).
atm(d125,d125_6,c,22,-0.119).
atm(d125,d125_7,c,10,-0.089).
atm(d125,d125_8,c,10,-0.089).
atm(d125,d125_9,c,22,0.011).
atm(d125,d125_10,c,27,-0.089).
atm(d125,d125_11,c,27,0.011).
atm(d125,d125_12,c,22,-0.119).
atm(d125,d125_13,c,22,-0.119).
atm(d125,d125_14,c,22,-0.119).
atm(d125,d125_15,c,22,-0.119).
atm(d125,d125_16,c,22,-0.119).
atm(d125,d125_17,n,38,0.811).
atm(d125,d125_18,h,3,0.141).
atm(d125,d125_19,h,3,0.141).
atm(d125,d125_20,h,3,0.141).
atm(d125,d125_21,h,3,0.141).
atm(d125,d125_22,h,3,0.141).
atm(d125,d125_23,h,3,0.141).
atm(d125,d125_24,h,3,0.141).
atm(d125,d125_25,h,3,0.061).
atm(d125,d125_26,h,3,0.061).
atm(d125,d125_27,h,3,0.061).
atm(d125,d125_28,h,3,0.061).
atm(d125,d125_29,o,40,-0.389).
atm(d125,d125_30,o,40,-0.389).

bond(d125,d125_1,d125_2,7).
bond(d125,d125_2,d125_3,7).
bond(d125,d125_3,d125_4,7).
bond(d125,d125_4,d125_5,7).
bond(d125,d125_5,d125_6,7).
bond(d125,d125_6,d125_1,7).
bond(d125,d125_3,d125_7,1).
bond(d125,d125_7,d125_8,1).
bond(d125,d125_8,d125_9,1).
bond(d125,d125_9,d125_10,7).
bond(d125,d125_10,d125_4,7).
bond(d125,d125_10,d125_11,7).
bond(d125,d125_11,d125_12,7).
bond(d125,d125_12,d125_13,7).
bond(d125,d125_13,d125_5,7).
bond(d125,d125_9,d125_14,7).
bond(d125,d125_14,d125_15,7).
bond(d125,d125_15,d125_16,7).
bond(d125,d125_16,d125_11,7).
bond(d125,d125_15,d125_17,1).
bond(d125,d125_1,d125_18,1).
bond(d125,d125_2,d125_19,1).
bond(d125,d125_6,d125_20,1).
bond(d125,d125_12,d125_21,1).
bond(d125,d125_13,d125_22,1).
bond(d125,d125_14,d125_23,1).
bond(d125,d125_16,d125_24,1).
bond(d125,d125_7,d125_25,1).
bond(d125,d125_7,d125_26,1).
bond(d125,d125_8,d125_27,1).
bond(d125,d125_8,d125_28,1).
bond(d125,d125_17,d125_29,2).
bond(d125,d125_17,d125_30,2).
atm(d126,d126_1,c,22,-0.142).
atm(d126,d126_2,c,22,-0.142).
atm(d126,d126_3,c,22,0.188).
atm(d126,d126_4,c,22,0.188).
atm(d126,d126_5,c,22,-0.142).
atm(d126,d126_6,c,22,-0.142).
atm(d126,d126_7,h,3,0.118).
atm(d126,d126_8,h,3,0.118).
atm(d126,d126_9,h,3,0.118).
atm(d126,d126_10,n,32,-0.412).
atm(d126,d126_11,c,22,0.188).
atm(d126,d126_12,c,22,0.188).
atm(d126,d126_13,n,32,-0.412).
atm(d126,d126_14,c,22,-0.142).
atm(d126,d126_15,c,22,-0.142).
atm(d126,d126_16,c,22,-0.142).
atm(d126,d126_17,c,22,-0.142).
atm(d126,d126_18,h,3,0.118).
atm(d126,d126_19,h,3,0.118).
atm(d126,d126_20,h,3,0.118).
atm(d126,d126_21,h,1,0.288).
atm(d126,d126_22,h,1,0.288).
atm(d126,d126_23,n,38,0.787).
atm(d126,d126_24,n,38,0.787).
atm(d126,d126_25,o,40,-0.413).
atm(d126,d126_26,o,40,-0.413).
atm(d126,d126_27,o,40,-0.412).
atm(d126,d126_28,o,40,-0.412).

bond(d126,d126_1,d126_2,7).
bond(d126,d126_2,d126_3,7).
bond(d126,d126_3,d126_4,7).
bond(d126,d126_4,d126_5,7).
bond(d126,d126_5,d126_6,7).
bond(d126,d126_6,d126_1,7).
bond(d126,d126_2,d126_7,1).
bond(d126,d126_5,d126_8,1).
bond(d126,d126_6,d126_9,1).
bond(d126,d126_3,d126_10,1).
bond(d126,d126_10,d126_11,1).
bond(d126,d126_11,d126_12,7).
bond(d126,d126_12,d126_13,1).
bond(d126,d126_13,d126_4,1).
bond(d126,d126_11,d126_14,7).
bond(d126,d126_14,d126_15,7).
bond(d126,d126_15,d126_16,7).
bond(d126,d126_16,d126_17,7).
bond(d126,d126_17,d126_12,7).
bond(d126,d126_14,d126_18,1).
bond(d126,d126_15,d126_19,1).
bond(d126,d126_16,d126_20,1).
bond(d126,d126_13,d126_21,1).
bond(d126,d126_10,d126_22,1).
bond(d126,d126_17,d126_23,1).
bond(d126,d126_1,d126_24,1).
bond(d126,d126_23,d126_25,2).
bond(d126,d126_23,d126_26,2).
bond(d126,d126_24,d126_27,2).
bond(d126,d126_24,d126_28,2).
atm(d127,d127_1,c,22,-0.116).
atm(d127,d127_2,c,22,-0.116).
atm(d127,d127_3,c,22,-0.116).
atm(d127,d127_4,c,29,0.014).
atm(d127,d127_5,c,22,-0.116).
atm(d127,d127_6,c,22,-0.116).
atm(d127,d127_7,h,3,0.144).
atm(d127,d127_8,h,3,0.144).
atm(d127,d127_9,h,3,0.144).
atm(d127,d127_10,h,3,0.144).
atm(d127,d127_11,c,29,0.014).
atm(d127,d127_12,c,22,-0.116).
atm(d127,d127_13,c,22,-0.116).
atm(d127,d127_14,c,22,-0.116).
atm(d127,d127_15,c,22,-0.116).
atm(d127,d127_16,c,22,-0.116).
atm(d127,d127_17,h,3,0.144).
atm(d127,d127_18,h,3,0.144).
atm(d127,d127_19,h,3,0.144).
atm(d127,d127_20,n,38,0.813).
atm(d127,d127_21,n,38,0.813).
atm(d127,d127_22,n,38,0.814).
atm(d127,d127_23,o,40,-0.386).
atm(d127,d127_24,o,40,-0.386).
atm(d127,d127_25,o,40,-0.386).
atm(d127,d127_26,o,40,-0.386).
atm(d127,d127_27,o,40,-0.386).
atm(d127,d127_28,o,40,-0.386).

bond(d127,d127_1,d127_2,7).
bond(d127,d127_2,d127_3,7).
bond(d127,d127_3,d127_4,7).
bond(d127,d127_4,d127_5,7).
bond(d127,d127_5,d127_6,7).
bond(d127,d127_6,d127_1,7).
bond(d127,d127_1,d127_7,1).
bond(d127,d127_2,d127_8,1).
bond(d127,d127_3,d127_9,1).
bond(d127,d127_5,d127_10,1).
bond(d127,d127_11,d127_12,7).
bond(d127,d127_12,d127_13,7).
bond(d127,d127_13,d127_14,7).
bond(d127,d127_14,d127_15,7).
bond(d127,d127_15,d127_16,7).
bond(d127,d127_16,d127_11,7).
bond(d127,d127_12,d127_17,1).
bond(d127,d127_13,d127_18,1).
bond(d127,d127_16,d127_19,1).
bond(d127,d127_4,d127_11,1).
bond(d127,d127_15,d127_20,1).
bond(d127,d127_6,d127_21,1).
bond(d127,d127_14,d127_22,1).
bond(d127,d127_20,d127_23,2).
bond(d127,d127_20,d127_24,2).
bond(d127,d127_22,d127_25,2).
bond(d127,d127_22,d127_26,2).
bond(d127,d127_21,d127_27,2).
bond(d127,d127_21,d127_28,2).
atm(d128,d128_1,c,22,-0.11).
atm(d128,d128_2,c,22,-0.11).
atm(d128,d128_3,c,27,0.02).
atm(d128,d128_4,c,27,-0.08).
atm(d128,d128_5,c,26,-0.08).
atm(d128,d128_6,c,22,-0.11).
atm(d128,d128_7,c,22,-0.11).
atm(d128,d128_8,c,22,-0.11).
atm(d128,d128_9,c,22,-0.11).
atm(d128,d128_10,c,26,-0.08).
atm(d128,d128_11,c,21,-0.08).
atm(d128,d128_12,c,21,-0.08).
atm(d128,d128_13,h,3,0.149).
atm(d128,d128_14,h,3,0.149).
atm(d128,d128_15,h,3,0.149).
atm(d128,d128_16,h,3,0.149).
atm(d128,d128_17,h,3,0.149).
atm(d128,d128_18,h,3,0.119).
atm(d128,d128_19,h,3,0.119).
atm(d128,d128_20,n,38,0.819).
atm(d128,d128_21,o,40,-0.381).
atm(d128,d128_22,o,40,-0.381).

bond(d128,d128_1,d128_2,7).
bond(d128,d128_2,d128_3,7).
bond(d128,d128_3,d128_4,7).
bond(d128,d128_4,d128_5,7).
bond(d128,d128_5,d128_6,7).
bond(d128,d128_6,d128_1,7).
bond(d128,d128_3,d128_7,7).
bond(d128,d128_7,d128_8,7).
bond(d128,d128_8,d128_9,7).
bond(d128,d128_9,d128_10,7).
bond(d128,d128_10,d128_4,7).
bond(d128,d128_5,d128_11,7).
bond(d128,d128_10,d128_12,7).
bond(d128,d128_11,d128_12,7).
bond(d128,d128_1,d128_13,1).
bond(d128,d128_2,d128_14,1).
bond(d128,d128_6,d128_15,1).
bond(d128,d128_8,d128_16,1).
bond(d128,d128_9,d128_17,1).
bond(d128,d128_11,d128_18,1).
bond(d128,d128_12,d128_19,1).
bond(d128,d128_7,d128_20,1).
bond(d128,d128_20,d128_21,2).
bond(d128,d128_20,d128_22,2).
atm(d129,d129_1,c,22,-0.122).
atm(d129,d129_2,c,22,-0.122).
atm(d129,d129_3,c,22,-0.122).
atm(d129,d129_4,c,22,-0.122).
atm(d129,d129_5,c,22,0.208).
atm(d129,d129_6,c,22,-0.122).
atm(d129,d129_7,h,3,0.138).
atm(d129,d129_8,h,3,0.138).
atm(d129,d129_9,h,3,0.138).
atm(d129,d129_10,n,32,-0.392).
atm(d129,d129_11,n,36,-0.292).
atm(d129,d129_12,h,1,0.308).
atm(d129,d129_13,h,1,0.158).
atm(d129,d129_14,h,1,0.158).
atm(d129,d129_15,n,38,0.808).
atm(d129,d129_16,n,38,0.808).
atm(d129,d129_17,o,40,-0.392).
atm(d129,d129_18,o,40,-0.392).
atm(d129,d129_19,o,40,-0.392).
atm(d129,d129_20,o,40,-0.392).

bond(d129,d129_1,d129_2,7).
bond(d129,d129_2,d129_3,7).
bond(d129,d129_3,d129_4,7).
bond(d129,d129_4,d129_5,7).
bond(d129,d129_5,d129_6,7).
bond(d129,d129_6,d129_1,7).
bond(d129,d129_1,d129_7,1).
bond(d129,d129_3,d129_8,1).
bond(d129,d129_6,d129_9,1).
bond(d129,d129_5,d129_10,1).
bond(d129,d129_10,d129_11,1).
bond(d129,d129_10,d129_12,1).
bond(d129,d129_11,d129_13,1).
bond(d129,d129_11,d129_14,1).
bond(d129,d129_4,d129_15,1).
bond(d129,d129_2,d129_16,1).
bond(d129,d129_15,d129_17,2).
bond(d129,d129_15,d129_18,2).
bond(d129,d129_16,d129_19,2).
bond(d129,d129_16,d129_20,2).
atm(d13,d13_1,c,22,-0.118).
atm(d13,d13_2,c,22,-0.118).
atm(d13,d13_3,c,27,0.012).
atm(d13,d13_4,c,27,-0.088).
atm(d13,d13_5,c,28,0.012).
atm(d13,d13_6,c,22,-0.118).
atm(d13,d13_7,c,22,-0.118).
atm(d13,d13_8,c,22,-0.118).
atm(d13,d13_9,c,22,-0.118).
atm(d13,d13_10,c,29,0.012).
atm(d13,d13_11,c,22,-0.118).
atm(d13,d13_12,c,29,0.012).
atm(d13,d13_13,c,28,0.012).
atm(d13,d13_14,c,22,-0.118).
atm(d13,d13_15,c,22,-0.118).
atm(d13,d13_16,c,22,-0.118).
atm(d13,d13_17,n,38,0.812).
atm(d13,d13_18,n,38,0.812).
atm(d13,d13_19,o,40,-0.388).
atm(d13,d13_20,o,40,-0.388).
atm(d13,d13_21,o,40,-0.388).
atm(d13,d13_22,o,40,-0.388).
atm(d13,d13_23,h,3,0.142).
atm(d13,d13_24,h,3,0.142).
atm(d13,d13_25,h,3,0.142).
atm(d13,d13_26,h,3,0.142).
atm(d13,d13_27,h,3,0.142).
atm(d13,d13_28,h,3,0.142).
atm(d13,d13_29,h,3,0.142).
atm(d13,d13_30,h,3,0.142).

bond(d13,d13_1,d13_2,7).
bond(d13,d13_2,d13_3,7).
bond(d13,d13_3,d13_4,7).
bond(d13,d13_4,d13_5,7).
bond(d13,d13_5,d13_6,7).
bond(d13,d13_6,d13_1,7).
bond(d13,d13_3,d13_7,7).
bond(d13,d13_7,d13_8,7).
bond(d13,d13_8,d13_9,7).
bond(d13,d13_9,d13_10,7).
bond(d13,d13_10,d13_4,7).
bond(d13,d13_11,d13_12,7).
bond(d13,d13_12,d13_13,7).
bond(d13,d13_13,d13_14,7).
bond(d13,d13_14,d13_15,7).
bond(d13,d13_15,d13_16,7).
bond(d13,d13_16,d13_11,7).
bond(d13,d13_12,d13_5,1).
bond(d13,d13_13,d13_10,1).
bond(d13,d13_8,d13_17,1).
bond(d13,d13_2,d13_18,1).
bond(d13,d13_17,d13_19,2).
bond(d13,d13_17,d13_20,2).
bond(d13,d13_18,d13_21,2).
bond(d13,d13_18,d13_22,2).
bond(d13,d13_1,d13_23,1).
bond(d13,d13_6,d13_24,1).
bond(d13,d13_7,d13_25,1).
bond(d13,d13_9,d13_26,1).
bond(d13,d13_11,d13_27,1).
bond(d13,d13_14,d13_28,1).
bond(d13,d13_15,d13_29,1).
bond(d13,d13_16,d13_30,1).
atm(d130,d130_1,c,22,-0.133).
atm(d130,d130_2,c,22,-0.133).
atm(d130,d130_3,c,22,-0.133).
atm(d130,d130_4,c,22,-0.133).
atm(d130,d130_5,c,22,0.176).
atm(d130,d130_6,c,22,-0.134).
atm(d130,d130_7,h,3,0.127).
atm(d130,d130_8,h,3,0.127).
atm(d130,d130_9,h,3,0.127).
atm(d130,d130_10,h,3,0.127).
atm(d130,d130_11,o,50,-0.234).
atm(d130,d130_12,c,10,0.087).
atm(d130,d130_13,h,3,0.047).
atm(d130,d130_14,h,3,0.047).
atm(d130,d130_15,h,3,0.047).
atm(d130,d130_16,n,38,0.796).
atm(d130,d130_17,o,40,-0.404).
atm(d130,d130_18,o,40,-0.404).

bond(d130,d130_1,d130_2,7).
bond(d130,d130_2,d130_3,7).
bond(d130,d130_3,d130_4,7).
bond(d130,d130_4,d130_5,7).
bond(d130,d130_5,d130_6,7).
bond(d130,d130_6,d130_1,7).
bond(d130,d130_1,d130_7,1).
bond(d130,d130_2,d130_8,1).
bond(d130,d130_3,d130_9,1).
bond(d130,d130_6,d130_10,1).
bond(d130,d130_5,d130_11,1).
bond(d130,d130_11,d130_12,1).
bond(d130,d130_12,d130_13,1).
bond(d130,d130_12,d130_14,1).
bond(d130,d130_12,d130_15,1).
bond(d130,d130_4,d130_16,1).
bond(d130,d130_16,d130_17,2).
bond(d130,d130_16,d130_18,2).
atm(d131,d131_1,c,22,-0.13).
atm(d131,d131_2,c,22,-0.131).
atm(d131,d131_3,c,22,-0.131).
atm(d131,d131_4,c,22,-0.131).
atm(d131,d131_5,c,22,0.099).
atm(d131,d131_6,c,22,-0.131).
atm(d131,d131_7,h,3,0.129).
atm(d131,d131_8,h,3,0.129).
atm(d131,d131_9,h,3,0.129).
atm(d131,d131_10,h,3,0.129).
atm(d131,d131_11,c,16,-0.2).
atm(d131,d131_12,c,16,0.099).
atm(d131,d131_13,h,3,0.099).
atm(d131,d131_14,o,50,-0.23).
atm(d131,d131_15,h,3,0.099).
atm(d131,d131_16,c,22,0.18).
atm(d131,d131_17,c,22,-0.131).
atm(d131,d131_18,c,22,-0.131).
atm(d131,d131_19,c,22,-0.131).
atm(d131,d131_20,c,22,-0.13).
atm(d131,d131_21,c,22,-0.13).
atm(d131,d131_22,h,3,0.129).
atm(d131,d131_23,h,3,0.129).
atm(d131,d131_24,h,3,0.129).
atm(d131,d131_25,h,3,0.129).
atm(d131,d131_26,h,3,0.129).
atm(d131,d131_27,n,38,0.8).
atm(d131,d131_28,o,40,-0.4).
atm(d131,d131_29,o,40,-0.4).

bond(d131,d131_1,d131_2,7).
bond(d131,d131_2,d131_3,7).
bond(d131,d131_3,d131_4,7).
bond(d131,d131_4,d131_5,7).
bond(d131,d131_5,d131_6,7).
bond(d131,d131_6,d131_1,7).
bond(d131,d131_1,d131_7,1).
bond(d131,d131_3,d131_8,1).
bond(d131,d131_4,d131_9,1).
bond(d131,d131_6,d131_10,1).
bond(d131,d131_5,d131_11,1).
bond(d131,d131_11,d131_12,2).
bond(d131,d131_11,d131_13,1).
bond(d131,d131_12,d131_14,1).
bond(d131,d131_12,d131_15,1).
bond(d131,d131_14,d131_16,1).
bond(d131,d131_16,d131_17,7).
bond(d131,d131_17,d131_18,7).
bond(d131,d131_18,d131_19,7).
bond(d131,d131_19,d131_20,7).
bond(d131,d131_20,d131_21,7).
bond(d131,d131_21,d131_16,7).
bond(d131,d131_17,d131_22,1).
bond(d131,d131_18,d131_23,1).
bond(d131,d131_19,d131_24,1).
bond(d131,d131_20,d131_25,1).
bond(d131,d131_21,d131_26,1).
bond(d131,d131_2,d131_27,1).
bond(d131,d131_27,d131_28,2).
bond(d131,d131_27,d131_29,2).
atm(d132,d132_1,c,22,-0.122).
atm(d132,d132_2,c,22,-0.122).
atm(d132,d132_3,c,22,-0.122).
atm(d132,d132_4,c,22,-0.122).
atm(d132,d132_5,c,22,0.008).
atm(d132,d132_6,c,22,-0.122).
atm(d132,d132_7,h,3,0.138).
atm(d132,d132_8,h,3,0.138).
atm(d132,d132_9,h,3,0.138).
atm(d132,d132_10,h,3,0.138).
atm(d132,d132_11,c,14,0.238).
atm(d132,d132_12,o,41,-0.421).
atm(d132,d132_13,h,3,0.208).
atm(d132,d132_14,n,38,0.809).
atm(d132,d132_15,o,40,-0.392).
atm(d132,d132_16,o,40,-0.392).

bond(d132,d132_1,d132_2,7).
bond(d132,d132_2,d132_3,7).
bond(d132,d132_3,d132_4,7).
bond(d132,d132_4,d132_5,7).
bond(d132,d132_5,d132_6,7).
bond(d132,d132_6,d132_1,7).
bond(d132,d132_1,d132_7,1).
bond(d132,d132_2,d132_8,1).
bond(d132,d132_3,d132_9,1).
bond(d132,d132_6,d132_10,1).
bond(d132,d132_5,d132_11,1).
bond(d132,d132_11,d132_12,2).
bond(d132,d132_11,d132_13,1).
bond(d132,d132_4,d132_14,1).
bond(d132,d132_14,d132_15,2).
bond(d132,d132_14,d132_16,2).
atm(d133,d133_1,c,22,-0.08).
atm(d133,d133_2,c,22,-0.08).
atm(d133,d133_3,c,22,0.229).
atm(d133,d133_4,c,22,0.229).
atm(d133,d133_5,c,22,-0.08).
atm(d133,d133_6,c,22,-0.08).
atm(d133,d133_7,h,3,0.179).
atm(d133,d133_8,h,3,0.179).
atm(d133,d133_9,o,50,-0.181).
atm(d133,d133_10,c,22,0.229).
atm(d133,d133_11,c,22,0.229).
atm(d133,d133_12,o,50,-0.18).
atm(d133,d133_13,c,22,-0.08).
atm(d133,d133_14,c,22,-0.08).
atm(d133,d133_15,c,22,-0.08).
atm(d133,d133_16,c,22,-0.08).
atm(d133,d133_17,h,3,0.18).
atm(d133,d133_18,n,38,0.849).
atm(d133,d133_19,cl,93,-0.15).
atm(d133,d133_20,cl,93,-0.15).
atm(d133,d133_21,cl,93,-0.15).
atm(d133,d133_22,cl,93,-0.15).
atm(d133,d133_23,o,40,-0.351).
atm(d133,d133_24,o,40,-0.351).

bond(d133,d133_1,d133_2,7).
bond(d133,d133_2,d133_3,7).
bond(d133,d133_3,d133_4,7).
bond(d133,d133_4,d133_5,7).
bond(d133,d133_5,d133_6,7).
bond(d133,d133_6,d133_1,7).
bond(d133,d133_2,d133_7,1).
bond(d133,d133_5,d133_8,1).
bond(d133,d133_3,d133_9,1).
bond(d133,d133_9,d133_10,1).
bond(d133,d133_10,d133_11,7).
bond(d133,d133_11,d133_12,1).
bond(d133,d133_12,d133_4,1).
bond(d133,d133_10,d133_13,7).
bond(d133,d133_13,d133_14,7).
bond(d133,d133_14,d133_15,7).
bond(d133,d133_15,d133_16,7).
bond(d133,d133_16,d133_11,7).
bond(d133,d133_13,d133_17,1).
bond(d133,d133_15,d133_18,1).
bond(d133,d133_16,d133_19,1).
bond(d133,d133_14,d133_20,1).
bond(d133,d133_1,d133_21,1).
bond(d133,d133_6,d133_22,1).
bond(d133,d133_18,d133_23,2).
bond(d133,d133_18,d133_24,2).
atm(d134,d134_1,c,22,-0.116).
atm(d134,d134_2,c,22,-0.116).
atm(d134,d134_3,c,22,0.014).
atm(d134,d134_4,c,195,-0.086).
atm(d134,d134_5,c,195,0.014).
atm(d134,d134_6,c,22,-0.116).
atm(d134,d134_7,c,10,-0.086).
atm(d134,d134_8,c,10,-0.086).
atm(d134,d134_9,c,22,0.014).
atm(d134,d134_10,c,27,-0.086).
atm(d134,d134_11,c,27,0.014).
atm(d134,d134_12,c,22,-0.116).
atm(d134,d134_13,c,22,-0.116).
atm(d134,d134_14,c,22,-0.116).
atm(d134,d134_15,c,22,-0.116).
atm(d134,d134_16,c,22,-0.116).
atm(d134,d134_17,h,3,0.145).
atm(d134,d134_18,h,3,0.145).
atm(d134,d134_19,h,3,0.145).
atm(d134,d134_20,h,3,0.145).
atm(d134,d134_21,h,3,0.145).
atm(d134,d134_22,h,3,0.145).
atm(d134,d134_23,h,3,0.064).
atm(d134,d134_24,h,3,0.064).
atm(d134,d134_25,h,3,0.064).
atm(d134,d134_26,h,3,0.064).
atm(d134,d134_27,n,38,0.815).
atm(d134,d134_28,n,38,0.815).
atm(d134,d134_29,o,40,-0.385).
atm(d134,d134_30,o,40,-0.385).
atm(d134,d134_31,o,40,-0.385).
atm(d134,d134_32,o,40,-0.385).

bond(d134,d134_1,d134_2,7).
bond(d134,d134_2,d134_3,7).
bond(d134,d134_3,d134_4,7).
bond(d134,d134_4,d134_5,7).
bond(d134,d134_5,d134_6,7).
bond(d134,d134_6,d134_1,7).
bond(d134,d134_3,d134_7,1).
bond(d134,d134_7,d134_8,1).
bond(d134,d134_8,d134_9,1).
bond(d134,d134_9,d134_10,7).
bond(d134,d134_10,d134_4,7).
bond(d134,d134_10,d134_11,7).
bond(d134,d134_11,d134_12,7).
bond(d134,d134_12,d134_13,7).
bond(d134,d134_13,d134_5,7).
bond(d134,d134_9,d134_14,7).
bond(d134,d134_14,d134_15,7).
bond(d134,d134_15,d134_16,7).
bond(d134,d134_16,d134_11,7).
bond(d134,d134_2,d134_17,1).
bond(d134,d134_6,d134_18,1).
bond(d134,d134_12,d134_19,1).
bond(d134,d134_13,d134_20,1).
bond(d134,d134_14,d134_21,1).
bond(d134,d134_16,d134_22,1).
bond(d134,d134_7,d134_23,1).
bond(d134,d134_7,d134_24,1).
bond(d134,d134_8,d134_25,1).
bond(d134,d134_8,d134_26,1).
bond(d134,d134_27,d134_15,1).
bond(d134,d134_1,d134_28,1).
bond(d134,d134_27,d134_29,2).
bond(d134,d134_27,d134_30,2).
bond(d134,d134_28,d134_31,2).
bond(d134,d134_28,d134_32,2).
atm(d135,d135_1,c,22,-0.097).
atm(d135,d135_2,c,22,0.333).
atm(d135,d135_3,c,22,-0.097).
atm(d135,d135_4,c,22,-0.097).
atm(d135,d135_5,c,22,-0.097).
atm(d135,d135_6,c,22,-0.097).
atm(d135,d135_7,h,3,0.163).
atm(d135,d135_8,h,3,0.163).
atm(d135,d135_9,h,3,0.163).
atm(d135,d135_10,n,38,0.832).
atm(d135,d135_11,o,40,-0.367).
atm(d135,d135_12,o,40,-0.368).
atm(d135,d135_13,cl,93,-0.167).
atm(d135,d135_14,f,92,-0.267).

bond(d135,d135_1,d135_2,7).
bond(d135,d135_2,d135_3,7).
bond(d135,d135_3,d135_4,7).
bond(d135,d135_4,d135_5,7).
bond(d135,d135_5,d135_6,7).
bond(d135,d135_6,d135_1,7).
bond(d135,d135_1,d135_7,1).
bond(d135,d135_4,d135_8,1).
bond(d135,d135_6,d135_9,1).
bond(d135,d135_5,d135_10,1).
bond(d135,d135_10,d135_11,2).
bond(d135,d135_10,d135_12,2).
bond(d135,d135_3,d135_13,1).
bond(d135,d135_2,d135_14,1).
atm(d136,d136_1,c,22,-0.119).
atm(d136,d136_2,c,22,-0.119).
atm(d136,d136_3,c,27,0.011).
atm(d136,d136_4,c,27,-0.089).
atm(d136,d136_5,c,27,0.011).
atm(d136,d136_6,c,22,-0.119).
atm(d136,d136_7,h,3,0.141).
atm(d136,d136_8,h,3,0.142).
atm(d136,d136_9,h,3,0.142).
atm(d136,d136_10,c,22,-0.119).
atm(d136,d136_11,c,22,-0.119).
atm(d136,d136_12,c,27,0.011).
atm(d136,d136_13,c,27,-0.089).
atm(d136,d136_14,h,3,0.141).
atm(d136,d136_15,h,3,0.142).
atm(d136,d136_16,c,27,0.011).
atm(d136,d136_17,c,22,-0.119).
atm(d136,d136_18,c,22,-0.119).
atm(d136,d136_19,h,3,0.141).
atm(d136,d136_20,h,3,0.142).
atm(d136,d136_21,c,22,-0.119).
atm(d136,d136_22,c,22,-0.119).
atm(d136,d136_23,c,22,0.262).
atm(d136,d136_24,h,3,0.142).
atm(d136,d136_25,o,45,-0.638).
atm(d136,d136_26,n,38,0.812).
atm(d136,d136_27,o,40,-0.388).
atm(d136,d136_28,o,40,-0.388).
atm(d136,d136_29,h,8,0.412).

bond(d136,d136_1,d136_2,7).
bond(d136,d136_2,d136_3,7).
bond(d136,d136_3,d136_4,7).
bond(d136,d136_4,d136_5,7).
bond(d136,d136_5,d136_6,7).
bond(d136,d136_6,d136_1,7).
bond(d136,d136_1,d136_7,1).
bond(d136,d136_2,d136_8,1).
bond(d136,d136_6,d136_9,1).
bond(d136,d136_3,d136_10,7).
bond(d136,d136_10,d136_11,7).
bond(d136,d136_11,d136_12,7).
bond(d136,d136_12,d136_13,7).
bond(d136,d136_13,d136_4,7).
bond(d136,d136_10,d136_14,1).
bond(d136,d136_11,d136_15,1).
bond(d136,d136_13,d136_16,7).
bond(d136,d136_16,d136_17,7).
bond(d136,d136_17,d136_18,7).
bond(d136,d136_18,d136_5,7).
bond(d136,d136_17,d136_19,1).
bond(d136,d136_18,d136_20,1).
bond(d136,d136_12,d136_21,7).
bond(d136,d136_21,d136_22,7).
bond(d136,d136_22,d136_23,7).
bond(d136,d136_23,d136_16,7).
bond(d136,d136_22,d136_24,1).
bond(d136,d136_23,d136_25,1).
bond(d136,d136_21,d136_26,1).
bond(d136,d136_26,d136_27,2).
bond(d136,d136_26,d136_28,2).
bond(d136,d136_25,d136_29,1).
atm(d137,d137_1,c,22,-0.118).
atm(d137,d137_2,c,22,-0.118).
atm(d137,d137_3,c,27,0.012).
atm(d137,d137_4,c,27,-0.088).
atm(d137,d137_5,c,28,0.012).
atm(d137,d137_6,c,22,-0.118).
atm(d137,d137_7,h,3,0.142).
atm(d137,d137_8,h,3,0.142).
atm(d137,d137_9,h,3,0.142).
atm(d137,d137_10,c,22,-0.118).
atm(d137,d137_11,c,22,-0.118).
atm(d137,d137_12,c,22,-0.118).
atm(d137,d137_13,c,29,0.012).
atm(d137,d137_14,h,3,0.142).
atm(d137,d137_15,c,28,0.012).
atm(d137,d137_16,c,28,0.012).
atm(d137,d137_17,c,22,-0.118).
atm(d137,d137_18,c,22,-0.118).
atm(d137,d137_19,c,22,-0.118).
atm(d137,d137_20,c,22,-0.118).
atm(d137,d137_21,h,3,0.142).
atm(d137,d137_22,h,3,0.142).
atm(d137,d137_23,h,3,0.142).
atm(d137,d137_24,h,3,0.142).
atm(d137,d137_25,n,38,0.812).
atm(d137,d137_26,n,38,0.812).
atm(d137,d137_27,o,40,-0.388).
atm(d137,d137_28,o,40,-0.388).
atm(d137,d137_29,o,40,-0.388).
atm(d137,d137_30,o,40,-0.388).

bond(d137,d137_1,d137_2,7).
bond(d137,d137_2,d137_3,7).
bond(d137,d137_3,d137_4,7).
bond(d137,d137_4,d137_5,7).
bond(d137,d137_5,d137_6,7).
bond(d137,d137_6,d137_1,7).
bond(d137,d137_1,d137_7,1).
bond(d137,d137_2,d137_8,1).
bond(d137,d137_6,d137_9,1).
bond(d137,d137_3,d137_10,7).
bond(d137,d137_10,d137_11,7).
bond(d137,d137_11,d137_12,7).
bond(d137,d137_12,d137_13,7).
bond(d137,d137_13,d137_4,7).
bond(d137,d137_12,d137_14,1).
bond(d137,d137_15,d137_16,7).
bond(d137,d137_16,d137_17,7).
bond(d137,d137_17,d137_18,7).
bond(d137,d137_18,d137_19,7).
bond(d137,d137_19,d137_20,7).
bond(d137,d137_20,d137_15,7).
bond(d137,d137_17,d137_21,1).
bond(d137,d137_18,d137_22,1).
bond(d137,d137_19,d137_23,1).
bond(d137,d137_20,d137_24,1).
bond(d137,d137_15,d137_5,1).
bond(d137,d137_16,d137_13,1).
bond(d137,d137_11,d137_25,1).
bond(d137,d137_10,d137_26,1).
bond(d137,d137_25,d137_27,2).
bond(d137,d137_25,d137_28,2).
bond(d137,d137_26,d137_29,2).
bond(d137,d137_26,d137_30,2).
atm(d138,d138_1,c,22,-0.122).
atm(d138,d138_2,c,22,-0.122).
atm(d138,d138_3,c,26,0.307).
atm(d138,d138_4,c,26,0.108).
atm(d138,d138_5,c,22,-0.122).
atm(d138,d138_6,c,22,-0.122).
atm(d138,d138_7,n,34,-0.393).
atm(d138,d138_8,n,34,-0.533).
atm(d138,d138_9,c,21,-0.122).
atm(d138,d138_10,c,10,0.098).
atm(d138,d138_11,n,38,0.807).
atm(d138,d138_12,o,40,-0.393).
atm(d138,d138_13,o,40,-0.393).
atm(d138,d138_14,h,3,0.137).
atm(d138,d138_15,h,3,0.108).
atm(d138,d138_16,h,3,0.138).
atm(d138,d138_17,h,3,0.138).
atm(d138,d138_18,h,3,0.058).
atm(d138,d138_19,h,3,0.058).
atm(d138,d138_20,h,3,0.058).
atm(d138,d138_21,h,1,0.307).

bond(d138,d138_1,d138_2,7).
bond(d138,d138_2,d138_3,7).
bond(d138,d138_3,d138_4,7).
bond(d138,d138_4,d138_5,7).
bond(d138,d138_5,d138_6,7).
bond(d138,d138_6,d138_1,7).
bond(d138,d138_3,d138_7,7).
bond(d138,d138_7,d138_8,7).
bond(d138,d138_4,d138_9,7).
bond(d138,d138_9,d138_8,7).
bond(d138,d138_8,d138_10,1).
bond(d138,d138_1,d138_11,1).
bond(d138,d138_11,d138_12,2).
bond(d138,d138_11,d138_13,2).
bond(d138,d138_2,d138_14,1).
bond(d138,d138_5,d138_15,1).
bond(d138,d138_6,d138_16,1).
bond(d138,d138_9,d138_17,1).
bond(d138,d138_10,d138_18,1).
bond(d138,d138_10,d138_19,1).
bond(d138,d138_10,d138_20,1).
bond(d138,d138_7,d138_21,1).
atm(d139,d139_1,c,22,-0.106).
atm(d139,d139_2,c,22,-0.106).
atm(d139,d139_3,c,22,-0.106).
atm(d139,d139_4,c,22,-0.106).
atm(d139,d139_5,c,22,0.224).
atm(d139,d139_6,c,22,-0.106).
atm(d139,d139_7,h,3,0.154).
atm(d139,d139_8,h,3,0.154).
atm(d139,d139_9,h,3,0.154).
atm(d139,d139_10,n,32,-0.757).
atm(d139,d139_11,h,1,0.354).
atm(d139,d139_12,h,1,0.354).
atm(d139,d139_13,n,38,0.823).
atm(d139,d139_14,cl,93,-0.176).
atm(d139,d139_15,o,40,-0.377).
atm(d139,d139_16,o,40,-0.377).

bond(d139,d139_1,d139_2,7).
bond(d139,d139_2,d139_3,7).
bond(d139,d139_3,d139_4,7).
bond(d139,d139_4,d139_5,7).
bond(d139,d139_5,d139_6,7).
bond(d139,d139_6,d139_1,7).
bond(d139,d139_1,d139_7,1).
bond(d139,d139_3,d139_8,1).
bond(d139,d139_6,d139_9,1).
bond(d139,d139_5,d139_10,1).
bond(d139,d139_10,d139_11,1).
bond(d139,d139_10,d139_12,1).
bond(d139,d139_4,d139_13,1).
bond(d139,d139_2,d139_14,1).
bond(d139,d139_13,d139_15,2).
bond(d139,d139_13,d139_16,2).
atm(d14,d14_1,c,22,-0.107).
atm(d14,d14_2,c,22,-0.107).
atm(d14,d14_3,c,26,0.323).
atm(d14,d14_4,c,26,0.123).
atm(d14,d14_5,c,22,-0.107).
atm(d14,d14_6,c,22,-0.107).
atm(d14,d14_7,c,21,-0.107).
atm(d14,d14_8,n,34,-0.377).
atm(d14,d14_9,n,34,-0.517).
atm(d14,d14_10,c,10,0.113).
atm(d14,d14_11,n,38,0.823).
atm(d14,d14_12,o,40,-0.377).
atm(d14,d14_13,o,40,-0.377).
atm(d14,d14_14,h,3,0.153).
atm(d14,d14_15,h,3,0.123).
atm(d14,d14_16,h,3,0.153).
atm(d14,d14_17,h,3,0.153).
atm(d14,d14_18,h,3,0.073).
atm(d14,d14_19,h,3,0.073).
atm(d14,d14_20,h,3,0.073).

bond(d14,d14_1,d14_2,7).
bond(d14,d14_2,d14_3,7).
bond(d14,d14_3,d14_4,7).
bond(d14,d14_4,d14_5,7).
bond(d14,d14_5,d14_6,7).
bond(d14,d14_6,d14_1,7).
bond(d14,d14_4,d14_7,7).
bond(d14,d14_7,d14_8,7).
bond(d14,d14_8,d14_9,7).
bond(d14,d14_9,d14_3,7).
bond(d14,d14_9,d14_10,1).
bond(d14,d14_1,d14_11,1).
bond(d14,d14_12,d14_11,2).
bond(d14,d14_11,d14_13,2).
bond(d14,d14_2,d14_14,1).
bond(d14,d14_5,d14_15,1).
bond(d14,d14_6,d14_16,1).
bond(d14,d14_7,d14_17,1).
bond(d14,d14_10,d14_18,1).
bond(d14,d14_10,d14_19,1).
bond(d14,d14_10,d14_20,1).
atm(d140,d140_1,c,22,-0.118).
atm(d140,d140_2,c,22,-0.118).
atm(d140,d140_3,c,27,0.012).
atm(d140,d140_4,c,27,0.012).
atm(d140,d140_5,c,22,-0.118).
atm(d140,d140_6,c,22,-0.118).
atm(d140,d140_7,h,3,0.142).
atm(d140,d140_8,h,3,0.142).
atm(d140,d140_9,h,3,0.142).
atm(d140,d140_10,c,22,-0.118).
atm(d140,d140_11,c,22,-0.118).
atm(d140,d140_12,c,22,-0.118).
atm(d140,d140_13,c,22,-0.118).
atm(d140,d140_14,h,3,0.142).
atm(d140,d140_15,h,3,0.142).
atm(d140,d140_16,h,3,0.142).
atm(d140,d140_17,n,38,0.811).
atm(d140,d140_18,n,38,0.811).
atm(d140,d140_19,o,40,-0.389).
atm(d140,d140_20,o,40,-0.389).
atm(d140,d140_21,o,40,-0.388).
atm(d140,d140_22,o,40,-0.388).

bond(d140,d140_1,d140_2,7).
bond(d140,d140_2,d140_3,7).
bond(d140,d140_3,d140_4,7).
bond(d140,d140_4,d140_5,7).
bond(d140,d140_5,d140_6,7).
bond(d140,d140_6,d140_1,7).
bond(d140,d140_1,d140_7,1).
bond(d140,d140_2,d140_8,1).
bond(d140,d140_6,d140_9,1).
bond(d140,d140_3,d140_10,7).
bond(d140,d140_10,d140_11,7).
bond(d140,d140_11,d140_12,7).
bond(d140,d140_12,d140_13,7).
bond(d140,d140_13,d140_4,7).
bond(d140,d140_10,d140_14,1).
bond(d140,d140_11,d140_15,1).
bond(d140,d140_12,d140_16,1).
bond(d140,d140_13,d140_17,1).
bond(d140,d140_5,d140_18,1).
bond(d140,d140_18,d140_19,2).
bond(d140,d140_18,d140_20,2).
bond(d140,d140_17,d140_21,2).
bond(d140,d140_17,d140_22,2).
atm(d141,d141_1,c,22,-0.074).
atm(d141,d141_2,c,22,-0.074).
atm(d141,d141_3,c,22,-0.074).
atm(d141,d141_4,c,22,-0.074).
atm(d141,d141_5,c,22,-0.074).
atm(d141,d141_6,c,22,-0.074).
atm(d141,d141_7,h,3,0.187).
atm(d141,d141_8,h,3,0.187).
atm(d141,d141_9,h,3,0.187).
atm(d141,d141_10,n,38,0.857).
atm(d141,d141_11,cl,93,-0.144).
atm(d141,d141_12,cl,93,-0.144).
atm(d141,d141_13,o,40,-0.343).
atm(d141,d141_14,o,40,-0.343).

bond(d141,d141_1,d141_2,7).
bond(d141,d141_2,d141_3,7).
bond(d141,d141_3,d141_4,7).
bond(d141,d141_4,d141_5,7).
bond(d141,d141_5,d141_6,7).
bond(d141,d141_6,d141_1,7).
bond(d141,d141_1,d141_7,1).
bond(d141,d141_2,d141_8,1).
bond(d141,d141_6,d141_9,1).
bond(d141,d141_5,d141_10,1).
bond(d141,d141_4,d141_11,1).
bond(d141,d141_3,d141_12,1).
bond(d141,d141_10,d141_13,2).
bond(d141,d141_10,d141_14,2).
atm(d142,d142_1,c,22,-0.094).
atm(d142,d142_2,c,22,-0.094).
atm(d142,d142_3,c,22,0.216).
atm(d142,d142_4,c,22,0.216).
atm(d142,d142_5,c,22,-0.094).
atm(d142,d142_6,c,22,-0.094).
atm(d142,d142_7,h,3,0.166).
atm(d142,d142_8,h,3,0.166).
atm(d142,d142_9,o,50,-0.194).
atm(d142,d142_10,c,22,0.216).
atm(d142,d142_11,c,22,0.215).
atm(d142,d142_12,o,50,-0.194).
atm(d142,d142_13,c,22,-0.094).
atm(d142,d142_14,c,22,-0.094).
atm(d142,d142_15,c,22,-0.094).
atm(d142,d142_16,c,22,-0.094).
atm(d142,d142_17,h,3,0.166).
atm(d142,d142_18,h,3,0.166).
atm(d142,d142_19,n,38,0.835).
atm(d142,d142_20,o,40,-0.365).
atm(d142,d142_21,o,40,-0.365).
atm(d142,d142_22,cl,93,-0.164).
atm(d142,d142_23,cl,93,-0.164).
atm(d142,d142_24,cl,93,-0.164).

bond(d142,d142_1,d142_2,7).
bond(d142,d142_2,d142_3,7).
bond(d142,d142_3,d142_4,7).
bond(d142,d142_4,d142_5,7).
bond(d142,d142_5,d142_6,7).
bond(d142,d142_6,d142_1,7).
bond(d142,d142_2,d142_7,1).
bond(d142,d142_5,d142_8,1).
bond(d142,d142_3,d142_9,1).
bond(d142,d142_9,d142_10,1).
bond(d142,d142_10,d142_11,7).
bond(d142,d142_11,d142_12,1).
bond(d142,d142_12,d142_4,1).
bond(d142,d142_10,d142_13,7).
bond(d142,d142_13,d142_14,7).
bond(d142,d142_14,d142_15,7).
bond(d142,d142_15,d142_16,7).
bond(d142,d142_16,d142_11,7).
bond(d142,d142_13,d142_17,1).
bond(d142,d142_16,d142_18,1).
bond(d142,d142_6,d142_19,1).
bond(d142,d142_19,d142_20,2).
bond(d142,d142_19,d142_21,2).
bond(d142,d142_15,d142_22,1).
bond(d142,d142_14,d142_23,1).
bond(d142,d142_1,d142_24,1).
atm(d143,d143_1,c,22,-0.13).
atm(d143,d143_2,c,22,-0.13).
atm(d143,d143_3,c,22,-0.13).
atm(d143,d143_4,c,22,-0.13).
atm(d143,d143_5,c,22,0.18).
atm(d143,d143_6,c,22,-0.13).
atm(d143,d143_7,h,3,0.13).
atm(d143,d143_8,h,3,0.13).
atm(d143,d143_9,h,3,0.13).
atm(d143,d143_10,h,3,0.13).
atm(d143,d143_11,o,50,-0.23).
atm(d143,d143_12,c,22,0.18).
atm(d143,d143_13,c,22,-0.13).
atm(d143,d143_14,c,22,-0.13).
atm(d143,d143_15,c,22,-0.13).
atm(d143,d143_16,c,22,-0.13).
atm(d143,d143_17,c,22,-0.13).
atm(d143,d143_18,h,3,0.13).
atm(d143,d143_19,h,3,0.13).
atm(d143,d143_20,h,3,0.13).
atm(d143,d143_21,h,3,0.13).
atm(d143,d143_22,h,3,0.13).
atm(d143,d143_23,n,38,0.8).
atm(d143,d143_24,o,40,-0.4).
atm(d143,d143_25,o,40,-0.4).

bond(d143,d143_1,d143_2,7).
bond(d143,d143_2,d143_3,7).
bond(d143,d143_3,d143_4,7).
bond(d143,d143_4,d143_5,7).
bond(d143,d143_5,d143_6,7).
bond(d143,d143_6,d143_1,7).
bond(d143,d143_1,d143_7,1).
bond(d143,d143_2,d143_8,1).
bond(d143,d143_3,d143_9,1).
bond(d143,d143_6,d143_10,1).
bond(d143,d143_5,d143_11,1).
bond(d143,d143_11,d143_12,1).
bond(d143,d143_12,d143_13,7).
bond(d143,d143_13,d143_14,7).
bond(d143,d143_14,d143_15,7).
bond(d143,d143_15,d143_16,7).
bond(d143,d143_16,d143_17,7).
bond(d143,d143_17,d143_12,7).
bond(d143,d143_13,d143_18,1).
bond(d143,d143_14,d143_19,1).
bond(d143,d143_15,d143_20,1).
bond(d143,d143_16,d143_21,1).
bond(d143,d143_17,d143_22,1).
bond(d143,d143_4,d143_23,1).
bond(d143,d143_23,d143_24,2).
bond(d143,d143_23,d143_25,2).
atm(d144,d144_1,c,22,-0.074).
atm(d144,d144_2,c,22,-0.074).
atm(d144,d144_3,c,22,-0.074).
atm(d144,d144_4,c,22,-0.074).
atm(d144,d144_5,c,22,-0.074).
atm(d144,d144_6,c,22,-0.074).
atm(d144,d144_7,h,3,0.187).
atm(d144,d144_8,h,3,0.187).
atm(d144,d144_9,h,3,0.187).
atm(d144,d144_10,n,38,0.857).
atm(d144,d144_11,cl,93,-0.144).
atm(d144,d144_12,cl,93,-0.144).
atm(d144,d144_13,o,40,-0.343).
atm(d144,d144_14,o,40,-0.343).

bond(d144,d144_1,d144_2,7).
bond(d144,d144_2,d144_3,7).
bond(d144,d144_3,d144_4,7).
bond(d144,d144_4,d144_5,7).
bond(d144,d144_5,d144_6,7).
bond(d144,d144_6,d144_1,7).
bond(d144,d144_2,d144_7,1).
bond(d144,d144_3,d144_8,1).
bond(d144,d144_6,d144_9,1).
bond(d144,d144_5,d144_10,1).
bond(d144,d144_4,d144_11,1).
bond(d144,d144_1,d144_12,1).
bond(d144,d144_10,d144_13,2).
bond(d144,d144_10,d144_14,2).
atm(d145,d145_1,c,22,-0.128).
atm(d145,d145_2,c,22,-0.128).
atm(d145,d145_3,c,22,-0.128).
atm(d145,d145_4,c,29,0.002).
atm(d145,d145_5,c,22,-0.128).
atm(d145,d145_6,c,22,-0.128).
atm(d145,d145_7,h,3,0.132).
atm(d145,d145_8,h,3,0.132).
atm(d145,d145_9,h,3,0.132).
atm(d145,d145_10,h,3,0.132).
atm(d145,d145_11,c,29,0.002).
atm(d145,d145_12,c,22,-0.128).
atm(d145,d145_13,c,22,-0.128).
atm(d145,d145_14,c,22,-0.128).
atm(d145,d145_15,c,22,0.202).
atm(d145,d145_16,c,22,-0.128).
atm(d145,d145_17,h,3,0.132).
atm(d145,d145_18,h,3,0.132).
atm(d145,d145_19,h,3,0.132).
atm(d145,d145_20,h,3,0.132).
atm(d145,d145_21,n,32,-0.779).
atm(d145,d145_22,n,38,0.801).
atm(d145,d145_23,o,40,-0.398).
atm(d145,d145_24,o,40,-0.398).
atm(d145,d145_25,h,1,0.332).
atm(d145,d145_26,h,1,0.332).

bond(d145,d145_1,d145_2,7).
bond(d145,d145_2,d145_3,7).
bond(d145,d145_3,d145_4,7).
bond(d145,d145_4,d145_5,7).
bond(d145,d145_5,d145_6,7).
bond(d145,d145_6,d145_1,7).
bond(d145,d145_2,d145_7,1).
bond(d145,d145_3,d145_8,1).
bond(d145,d145_5,d145_9,1).
bond(d145,d145_6,d145_10,1).
bond(d145,d145_11,d145_12,7).
bond(d145,d145_12,d145_13,7).
bond(d145,d145_13,d145_14,7).
bond(d145,d145_14,d145_15,7).
bond(d145,d145_15,d145_16,7).
bond(d145,d145_16,d145_11,7).
bond(d145,d145_12,d145_17,1).
bond(d145,d145_13,d145_18,1).
bond(d145,d145_14,d145_19,1).
bond(d145,d145_16,d145_20,1).
bond(d145,d145_4,d145_11,1).
bond(d145,d145_15,d145_21,1).
bond(d145,d145_1,d145_22,1).
bond(d145,d145_22,d145_23,2).
bond(d145,d145_22,d145_24,2).
bond(d145,d145_21,d145_25,1).
bond(d145,d145_21,d145_26,1).
atm(d146,d146_1,c,22,-0.128).
atm(d146,d146_2,c,22,-0.128).
atm(d146,d146_3,c,22,-0.128).
atm(d146,d146_4,c,29,0.002).
atm(d146,d146_5,c,22,-0.128).
atm(d146,d146_6,c,22,-0.128).
atm(d146,d146_7,h,3,0.132).
atm(d146,d146_8,h,3,0.132).
atm(d146,d146_9,h,3,0.132).
atm(d146,d146_10,h,3,0.132).
atm(d146,d146_11,c,29,0.002).
atm(d146,d146_12,c,22,-0.128).
atm(d146,d146_13,c,22,-0.128).
atm(d146,d146_14,c,22,0.202).
atm(d146,d146_15,c,22,-0.128).
atm(d146,d146_16,c,22,-0.128).
atm(d146,d146_17,h,3,0.132).
atm(d146,d146_18,h,3,0.132).
atm(d146,d146_19,h,3,0.132).
atm(d146,d146_20,n,38,0.801).
atm(d146,d146_21,o,40,-0.398).
atm(d146,d146_22,o,40,-0.398).
atm(d146,d146_23,h,3,0.132).
atm(d146,d146_24,n,32,-0.779).
atm(d146,d146_25,h,1,0.332).
atm(d146,d146_26,h,1,0.332).

bond(d146,d146_1,d146_2,7).
bond(d146,d146_2,d146_3,7).
bond(d146,d146_3,d146_4,7).
bond(d146,d146_4,d146_5,7).
bond(d146,d146_5,d146_6,7).
bond(d146,d146_6,d146_1,7).
bond(d146,d146_2,d146_7,1).
bond(d146,d146_3,d146_8,1).
bond(d146,d146_5,d146_9,1).
bond(d146,d146_6,d146_10,1).
bond(d146,d146_11,d146_12,7).
bond(d146,d146_12,d146_13,7).
bond(d146,d146_13,d146_14,7).
bond(d146,d146_14,d146_15,7).
bond(d146,d146_15,d146_16,7).
bond(d146,d146_16,d146_11,7).
bond(d146,d146_12,d146_17,1).
bond(d146,d146_13,d146_18,1).
bond(d146,d146_16,d146_19,1).
bond(d146,d146_4,d146_11,1).
bond(d146,d146_1,d146_20,1).
bond(d146,d146_20,d146_21,2).
bond(d146,d146_20,d146_22,2).
bond(d146,d146_15,d146_23,1).
bond(d146,d146_14,d146_24,1).
bond(d146,d146_24,d146_25,1).
bond(d146,d146_24,d146_26,1).
atm(d147,d147_1,c,22,-0.068).
atm(d147,d147_2,c,22,-0.068).
atm(d147,d147_3,c,26,-0.038).
atm(d147,d147_4,c,26,-0.038).
atm(d147,d147_5,c,22,-0.068).
atm(d147,d147_6,c,22,-0.068).
atm(d147,d147_7,n,34,-0.479).
atm(d147,d147_8,c,21,0.062).
atm(d147,d147_9,n,34,-0.479).
atm(d147,d147_10,c,10,0.152).
atm(d147,d147_11,n,38,0.861).
atm(d147,d147_12,o,40,-0.339).
atm(d147,d147_13,o,40,-0.339).
atm(d147,d147_14,h,3,0.191).
atm(d147,d147_15,h,3,0.191).
atm(d147,d147_16,h,3,0.191).
atm(d147,d147_17,h,3,0.112).
atm(d147,d147_18,h,3,0.112).
atm(d147,d147_19,h,3,0.112).

bond(d147,d147_1,d147_2,7).
bond(d147,d147_2,d147_3,7).
bond(d147,d147_3,d147_4,7).
bond(d147,d147_4,d147_5,7).
bond(d147,d147_5,d147_6,7).
bond(d147,d147_6,d147_1,7).
bond(d147,d147_3,d147_7,7).
bond(d147,d147_7,d147_8,7).
bond(d147,d147_4,d147_9,7).
bond(d147,d147_9,d147_8,7).
bond(d147,d147_8,d147_10,1).
bond(d147,d147_6,d147_11,1).
bond(d147,d147_11,d147_12,2).
bond(d147,d147_11,d147_13,2).
bond(d147,d147_1,d147_14,1).
bond(d147,d147_2,d147_15,1).
bond(d147,d147_5,d147_16,1).
bond(d147,d147_10,d147_17,1).
bond(d147,d147_10,d147_18,1).
bond(d147,d147_10,d147_19,1).
atm(d148,d148_1,c,22,-0.11).
atm(d148,d148_2,c,22,-0.11).
atm(d148,d148_3,c,27,0.02).
atm(d148,d148_4,c,27,-0.08).
atm(d148,d148_5,c,26,-0.08).
atm(d148,d148_6,c,22,-0.11).
atm(d148,d148_7,c,22,-0.11).
atm(d148,d148_8,c,22,-0.11).
atm(d148,d148_9,c,22,-0.11).
atm(d148,d148_10,c,26,-0.08).
atm(d148,d148_11,c,21,-0.08).
atm(d148,d148_12,c,21,-0.08).
atm(d148,d148_13,h,3,0.149).
atm(d148,d148_14,h,3,0.149).
atm(d148,d148_15,h,3,0.149).
atm(d148,d148_16,h,3,0.149).
atm(d148,d148_17,h,3,0.149).
atm(d148,d148_18,h,3,0.119).
atm(d148,d148_19,h,3,0.119).
atm(d148,d148_20,n,38,0.819).
atm(d148,d148_21,o,40,-0.381).
atm(d148,d148_22,o,40,-0.381).

bond(d148,d148_1,d148_2,7).
bond(d148,d148_2,d148_3,7).
bond(d148,d148_3,d148_4,7).
bond(d148,d148_4,d148_5,7).
bond(d148,d148_5,d148_6,7).
bond(d148,d148_6,d148_1,7).
bond(d148,d148_3,d148_7,7).
bond(d148,d148_7,d148_8,7).
bond(d148,d148_8,d148_9,7).
bond(d148,d148_9,d148_10,7).
bond(d148,d148_10,d148_4,7).
bond(d148,d148_5,d148_11,7).
bond(d148,d148_10,d148_12,7).
bond(d148,d148_11,d148_12,7).
bond(d148,d148_1,d148_13,1).
bond(d148,d148_2,d148_14,1).
bond(d148,d148_6,d148_15,1).
bond(d148,d148_7,d148_16,1).
bond(d148,d148_8,d148_17,1).
bond(d148,d148_11,d148_18,1).
bond(d148,d148_12,d148_19,1).
bond(d148,d148_9,d148_20,1).
bond(d148,d148_20,d148_21,2).
bond(d148,d148_20,d148_22,2).
atm(d149,d149_1,c,22,-0.142).
atm(d149,d149_2,c,22,-0.142).
atm(d149,d149_3,c,22,0.188).
atm(d149,d149_4,c,22,0.188).
atm(d149,d149_5,c,22,-0.142).
atm(d149,d149_6,c,22,-0.142).
atm(d149,d149_7,h,3,0.118).
atm(d149,d149_8,h,3,0.118).
atm(d149,d149_9,h,3,0.118).
atm(d149,d149_10,n,32,-0.412).
atm(d149,d149_11,c,22,0.188).
atm(d149,d149_12,c,22,0.188).
atm(d149,d149_13,n,32,-0.412).
atm(d149,d149_14,c,22,-0.142).
atm(d149,d149_15,c,22,-0.142).
atm(d149,d149_16,c,22,-0.142).
atm(d149,d149_17,c,22,-0.142).
atm(d149,d149_18,h,3,0.118).
atm(d149,d149_19,h,3,0.118).
atm(d149,d149_20,h,3,0.118).
atm(d149,d149_21,h,1,0.288).
atm(d149,d149_22,h,1,0.288).
atm(d149,d149_23,n,38,0.787).
atm(d149,d149_24,n,38,0.787).
atm(d149,d149_25,o,40,-0.413).
atm(d149,d149_26,o,40,-0.413).
atm(d149,d149_27,o,40,-0.412).
atm(d149,d149_28,o,40,-0.412).

bond(d149,d149_1,d149_2,7).
bond(d149,d149_2,d149_3,7).
bond(d149,d149_3,d149_4,7).
bond(d149,d149_4,d149_5,7).
bond(d149,d149_5,d149_6,7).
bond(d149,d149_6,d149_1,7).
bond(d149,d149_2,d149_7,1).
bond(d149,d149_5,d149_8,1).
bond(d149,d149_6,d149_9,1).
bond(d149,d149_3,d149_10,1).
bond(d149,d149_10,d149_11,1).
bond(d149,d149_11,d149_12,7).
bond(d149,d149_12,d149_13,1).
bond(d149,d149_13,d149_4,1).
bond(d149,d149_11,d149_14,7).
bond(d149,d149_14,d149_15,7).
bond(d149,d149_15,d149_16,7).
bond(d149,d149_16,d149_17,7).
bond(d149,d149_17,d149_12,7).
bond(d149,d149_14,d149_18,1).
bond(d149,d149_15,d149_19,1).
bond(d149,d149_17,d149_20,1).
bond(d149,d149_13,d149_21,1).
bond(d149,d149_10,d149_22,1).
bond(d149,d149_16,d149_23,1).
bond(d149,d149_1,d149_24,1).
bond(d149,d149_23,d149_25,2).
bond(d149,d149_23,d149_26,2).
bond(d149,d149_24,d149_27,2).
bond(d149,d149_24,d149_28,2).
atm(d15,d15_1,c,22,-0.118).
atm(d15,d15_2,c,22,-0.118).
atm(d15,d15_3,c,27,0.012).
atm(d15,d15_4,c,27,-0.088).
atm(d15,d15_5,c,27,0.012).
atm(d15,d15_6,c,22,-0.118).
atm(d15,d15_7,h,3,0.141).
atm(d15,d15_8,h,3,0.142).
atm(d15,d15_9,h,3,0.142).
atm(d15,d15_10,c,27,-0.088).
atm(d15,d15_11,c,27,0.012).
atm(d15,d15_12,c,22,-0.118).
atm(d15,d15_13,c,22,-0.118).
atm(d15,d15_14,h,3,0.141).
atm(d15,d15_15,h,3,0.142).
atm(d15,d15_16,c,22,-0.118).
atm(d15,d15_17,c,22,-0.118).
atm(d15,d15_18,c,27,0.012).
atm(d15,d15_19,h,3,0.141).
atm(d15,d15_20,h,3,0.142).
atm(d15,d15_21,c,22,-0.118).
atm(d15,d15_22,c,22,-0.118).
atm(d15,d15_23,c,22,-0.118).
atm(d15,d15_24,h,3,0.142).
atm(d15,d15_25,h,3,0.142).
atm(d15,d15_26,n,38,0.811).
atm(d15,d15_27,o,40,-0.389).
atm(d15,d15_28,o,40,-0.389).

bond(d15,d15_1,d15_2,7).
bond(d15,d15_2,d15_3,7).
bond(d15,d15_3,d15_4,7).
bond(d15,d15_4,d15_5,7).
bond(d15,d15_5,d15_6,7).
bond(d15,d15_6,d15_1,7).
bond(d15,d15_1,d15_7,1).
bond(d15,d15_2,d15_8,1).
bond(d15,d15_6,d15_9,1).
bond(d15,d15_4,d15_10,7).
bond(d15,d15_10,d15_11,7).
bond(d15,d15_11,d15_12,7).
bond(d15,d15_12,d15_13,7).
bond(d15,d15_13,d15_5,7).
bond(d15,d15_12,d15_14,1).
bond(d15,d15_13,d15_15,1).
bond(d15,d15_3,d15_16,7).
bond(d15,d15_16,d15_17,7).
bond(d15,d15_17,d15_18,7).
bond(d15,d15_18,d15_10,7).
bond(d15,d15_16,d15_19,1).
bond(d15,d15_17,d15_20,1).
bond(d15,d15_18,d15_21,7).
bond(d15,d15_21,d15_22,7).
bond(d15,d15_22,d15_23,7).
bond(d15,d15_23,d15_11,7).
bond(d15,d15_21,d15_24,1).
bond(d15,d15_22,d15_25,1).
bond(d15,d15_26,d15_23,1).
bond(d15,d15_26,d15_27,2).
bond(d15,d15_26,d15_28,2).
atm(d150,d150_1,c,21,0.03).
atm(d150,d150_2,c,21,0.03).
atm(d150,d150_3,n,34,-0.509).
atm(d150,d150_4,c,21,0.03).
atm(d150,d150_5,n,34,-0.509).
atm(d150,d150_6,n,38,0.831).
atm(d150,d150_7,o,40,-0.369).
atm(d150,d150_8,o,40,-0.369).
atm(d150,d150_9,c,10,0.03).
atm(d150,d150_10,c,10,0.201).
atm(d150,d150_11,o,45,-0.619).
atm(d150,d150_12,c,10,0.121).
atm(d150,d150_13,h,3,0.131).
atm(d150,d150_14,h,3,0.08).
atm(d150,d150_15,h,3,0.08).
atm(d150,d150_16,h,3,0.07).
atm(d150,d150_17,h,3,0.07).
atm(d150,d150_18,h,8,0.431).
atm(d150,d150_19,h,3,0.08).
atm(d150,d150_20,h,3,0.08).
atm(d150,d150_21,h,3,0.08).

bond(d150,d150_1,d150_2,7).
bond(d150,d150_2,d150_3,7).
bond(d150,d150_3,d150_4,7).
bond(d150,d150_4,d150_5,7).
bond(d150,d150_5,d150_1,7).
bond(d150,d150_1,d150_6,1).
bond(d150,d150_6,d150_7,2).
bond(d150,d150_6,d150_8,2).
bond(d150,d150_5,d150_9,1).
bond(d150,d150_9,d150_10,1).
bond(d150,d150_10,d150_11,1).
bond(d150,d150_4,d150_12,1).
bond(d150,d150_2,d150_13,1).
bond(d150,d150_9,d150_14,1).
bond(d150,d150_9,d150_15,1).
bond(d150,d150_10,d150_16,1).
bond(d150,d150_10,d150_17,1).
bond(d150,d150_11,d150_18,1).
bond(d150,d150_12,d150_19,1).
bond(d150,d150_12,d150_20,1).
bond(d150,d150_12,d150_21,1).
atm(d151,d151_1,c,22,-0.115).
atm(d151,d151_2,c,22,-0.115).
atm(d151,d151_3,c,26,0.315).
atm(d151,d151_4,c,26,0.115).
atm(d151,d151_5,c,22,-0.115).
atm(d151,d151_6,c,22,-0.115).
atm(d151,d151_7,n,34,-0.384).
atm(d151,d151_8,n,34,-0.384).
atm(d151,d151_9,c,21,-0.115).
atm(d151,d151_10,n,38,0.816).
atm(d151,d151_11,o,40,-0.384).
atm(d151,d151_12,o,40,-0.384).
atm(d151,d151_13,h,3,0.145).
atm(d151,d151_14,h,3,0.115).
atm(d151,d151_15,h,3,0.145).
atm(d151,d151_16,h,3,0.145).
atm(d151,d151_17,h,1,0.315).

bond(d151,d151_1,d151_2,7).
bond(d151,d151_2,d151_3,7).
bond(d151,d151_3,d151_4,7).
bond(d151,d151_4,d151_5,7).
bond(d151,d151_5,d151_6,7).
bond(d151,d151_6,d151_1,7).
bond(d151,d151_3,d151_7,7).
bond(d151,d151_7,d151_8,7).
bond(d151,d151_4,d151_9,7).
bond(d151,d151_9,d151_8,7).
bond(d151,d151_2,d151_10,1).
bond(d151,d151_10,d151_11,2).
bond(d151,d151_10,d151_12,2).
bond(d151,d151_1,d151_13,1).
bond(d151,d151_5,d151_14,1).
bond(d151,d151_6,d151_15,1).
bond(d151,d151_9,d151_16,1).
bond(d151,d151_7,d151_17,1).
atm(d152,d152_1,c,22,-0.118).
atm(d152,d152_2,c,22,-0.118).
atm(d152,d152_3,c,27,0.012).
atm(d152,d152_4,c,27,-0.088).
atm(d152,d152_5,c,28,0.012).
atm(d152,d152_6,c,22,-0.118).
atm(d152,d152_7,h,3,0.142).
atm(d152,d152_8,h,3,0.142).
atm(d152,d152_9,c,22,-0.118).
atm(d152,d152_10,c,22,-0.118).
atm(d152,d152_11,c,22,-0.118).
atm(d152,d152_12,c,29,0.012).
atm(d152,d152_13,h,3,0.142).
atm(d152,d152_14,h,3,0.142).
atm(d152,d152_15,c,28,0.012).
atm(d152,d152_16,c,28,0.012).
atm(d152,d152_17,c,22,-0.118).
atm(d152,d152_18,c,22,-0.118).
atm(d152,d152_19,c,22,-0.118).
atm(d152,d152_20,c,22,-0.118).
atm(d152,d152_21,h,3,0.142).
atm(d152,d152_22,h,3,0.142).
atm(d152,d152_23,h,3,0.142).
atm(d152,d152_24,h,3,0.142).
atm(d152,d152_25,n,38,0.812).
atm(d152,d152_26,n,38,0.812).
atm(d152,d152_27,o,40,-0.388).
atm(d152,d152_28,o,40,-0.388).
atm(d152,d152_29,o,40,-0.388).
atm(d152,d152_30,o,40,-0.388).

bond(d152,d152_1,d152_2,7).
bond(d152,d152_2,d152_3,7).
bond(d152,d152_3,d152_4,7).
bond(d152,d152_4,d152_5,7).
bond(d152,d152_5,d152_6,7).
bond(d152,d152_6,d152_1,7).
bond(d152,d152_2,d152_7,1).
bond(d152,d152_6,d152_8,1).
bond(d152,d152_3,d152_9,7).
bond(d152,d152_9,d152_10,7).
bond(d152,d152_10,d152_11,7).
bond(d152,d152_11,d152_12,7).
bond(d152,d152_12,d152_4,7).
bond(d152,d152_9,d152_13,1).
bond(d152,d152_11,d152_14,1).
bond(d152,d152_15,d152_16,7).
bond(d152,d152_16,d152_17,7).
bond(d152,d152_17,d152_18,7).
bond(d152,d152_18,d152_19,7).
bond(d152,d152_19,d152_20,7).
bond(d152,d152_20,d152_15,7).
bond(d152,d152_17,d152_21,1).
bond(d152,d152_18,d152_22,1).
bond(d152,d152_19,d152_23,1).
bond(d152,d152_20,d152_24,1).
bond(d152,d152_15,d152_5,1).
bond(d152,d152_16,d152_12,1).
bond(d152,d152_10,d152_25,1).
bond(d152,d152_1,d152_26,1).
bond(d152,d152_26,d152_27,2).
bond(d152,d152_26,d152_28,2).
bond(d152,d152_25,d152_29,2).
bond(d152,d152_25,d152_30,2).
atm(d153,d153_1,c,22,-0.125).
atm(d153,d153_2,c,22,-0.125).
atm(d153,d153_3,c,27,0.005).
atm(d153,d153_4,c,27,0.005).
atm(d153,d153_5,c,22,0.005).
atm(d153,d153_6,c,22,-0.124).
atm(d153,d153_7,h,3,0.136).
atm(d153,d153_8,h,3,0.136).
atm(d153,d153_9,h,3,0.136).
atm(d153,d153_10,c,22,-0.125).
atm(d153,d153_11,c,22,-0.124).
atm(d153,d153_12,c,22,-0.125).
atm(d153,d153_13,c,22,0.005).
atm(d153,d153_14,h,3,0.136).
atm(d153,d153_15,h,3,0.136).
atm(d153,d153_16,c,10,-0.095).
atm(d153,d153_17,c,10,-0.095).
atm(d153,d153_18,h,3,0.055).
atm(d153,d153_19,h,3,0.055).
atm(d153,d153_20,h,3,0.055).
atm(d153,d153_21,h,3,0.055).
atm(d153,d153_22,n,38,0.806).
atm(d153,d153_23,o,40,-0.394).
atm(d153,d153_24,o,40,-0.394).

bond(d153,d153_1,d153_2,7).
bond(d153,d153_2,d153_3,7).
bond(d153,d153_3,d153_4,7).
bond(d153,d153_4,d153_5,7).
bond(d153,d153_5,d153_6,7).
bond(d153,d153_6,d153_1,7).
bond(d153,d153_1,d153_7,1).
bond(d153,d153_2,d153_8,1).
bond(d153,d153_6,d153_9,1).
bond(d153,d153_3,d153_10,7).
bond(d153,d153_10,d153_11,7).
bond(d153,d153_11,d153_12,7).
bond(d153,d153_12,d153_13,7).
bond(d153,d153_13,d153_4,7).
bond(d153,d153_11,d153_14,1).
bond(d153,d153_12,d153_15,1).
bond(d153,d153_5,d153_16,1).
bond(d153,d153_13,d153_17,1).
bond(d153,d153_16,d153_17,1).
bond(d153,d153_16,d153_18,1).
bond(d153,d153_16,d153_19,1).
bond(d153,d153_17,d153_20,1).
bond(d153,d153_17,d153_21,1).
bond(d153,d153_10,d153_22,1).
bond(d153,d153_22,d153_23,2).
bond(d153,d153_22,d153_24,2).
atm(d154,d154_1,c,22,-0.088).
atm(d154,d154_2,c,22,-0.088).
atm(d154,d154_3,c,26,0.002).
atm(d154,d154_4,c,26,-0.058).
atm(d154,d154_5,c,22,-0.088).
atm(d154,d154_6,c,22,-0.088).
atm(d154,d154_7,n,34,-0.497).
atm(d154,d154_8,c,21,0.042).
atm(d154,d154_9,n,34,-0.497).
atm(d154,d154_10,n,38,0.843).
atm(d154,d154_11,o,40,-0.357).
atm(d154,d154_12,o,40,-0.357).
atm(d154,d154_13,h,3,0.172).
atm(d154,d154_14,h,3,0.172).
atm(d154,d154_15,h,3,0.172).
atm(d154,d154_16,h,3,0.172).
atm(d154,d154_17,h,1,0.543).

bond(d154,d154_1,d154_2,7).
bond(d154,d154_2,d154_3,7).
bond(d154,d154_3,d154_4,7).
bond(d154,d154_4,d154_5,7).
bond(d154,d154_5,d154_6,7).
bond(d154,d154_6,d154_1,7).
bond(d154,d154_3,d154_7,7).
bond(d154,d154_7,d154_8,7).
bond(d154,d154_4,d154_9,7).
bond(d154,d154_9,d154_8,7).
bond(d154,d154_8,d154_10,1).
bond(d154,d154_10,d154_11,2).
bond(d154,d154_10,d154_12,2).
bond(d154,d154_1,d154_13,1).
bond(d154,d154_2,d154_14,1).
bond(d154,d154_5,d154_15,1).
bond(d154,d154_6,d154_16,1).
bond(d154,d154_7,d154_17,1).
atm(d155,d155_1,c,22,-0.126).
atm(d155,d155_2,c,22,-0.126).
atm(d155,d155_3,c,22,-0.126).
atm(d155,d155_4,c,22,-0.126).
atm(d155,d155_5,c,22,0.184).
atm(d155,d155_6,c,22,-0.126).
atm(d155,d155_7,h,3,0.133).
atm(d155,d155_8,h,3,0.133).
atm(d155,d155_9,h,3,0.133).
atm(d155,d155_10,o,50,-0.227).
atm(d155,d155_11,c,10,0.094).
atm(d155,d155_12,h,3,0.054).
atm(d155,d155_13,h,3,0.054).
atm(d155,d155_14,h,3,0.054).
atm(d155,d155_15,n,38,0.802).
atm(d155,d155_16,n,38,0.802).
atm(d155,d155_17,o,40,-0.397).
atm(d155,d155_18,o,40,-0.397).
atm(d155,d155_19,o,40,-0.396).
atm(d155,d155_20,o,40,-0.396).

bond(d155,d155_1,d155_2,7).
bond(d155,d155_2,d155_3,7).
bond(d155,d155_3,d155_4,7).
bond(d155,d155_4,d155_5,7).
bond(d155,d155_5,d155_6,7).
bond(d155,d155_6,d155_1,7).
bond(d155,d155_1,d155_7,1).
bond(d155,d155_3,d155_8,1).
bond(d155,d155_6,d155_9,1).
bond(d155,d155_5,d155_10,1).
bond(d155,d155_10,d155_11,1).
bond(d155,d155_11,d155_12,1).
bond(d155,d155_11,d155_13,1).
bond(d155,d155_11,d155_14,1).
bond(d155,d155_4,d155_15,1).
bond(d155,d155_2,d155_16,1).
bond(d155,d155_15,d155_17,2).
bond(d155,d155_15,d155_18,2).
bond(d155,d155_16,d155_19,2).
bond(d155,d155_16,d155_20,2).
atm(d156,d156_1,c,22,-0.133).
atm(d156,d156_2,c,22,-0.133).
atm(d156,d156_3,c,22,0.197).
atm(d156,d156_4,c,22,-0.003).
atm(d156,d156_5,c,22,-0.133).
atm(d156,d156_6,c,22,-0.133).
atm(d156,d156_7,h,3,0.126).
atm(d156,d156_8,h,3,0.126).
atm(d156,d156_9,h,3,0.126).
atm(d156,d156_10,n,32,-0.405).
atm(d156,d156_11,c,10,-0.003).
atm(d156,d156_12,c,10,-0.104).
atm(d156,d156_13,h,1,0.297).
atm(d156,d156_14,h,3,0.047).
atm(d156,d156_15,h,3,0.047).
atm(d156,d156_16,h,3,0.047).
atm(d156,d156_17,h,3,0.047).
atm(d156,d156_18,n,38,0.796).
atm(d156,d156_19,o,40,-0.405).
atm(d156,d156_20,o,40,-0.404).

bond(d156,d156_1,d156_2,7).
bond(d156,d156_2,d156_3,7).
bond(d156,d156_3,d156_4,7).
bond(d156,d156_4,d156_5,7).
bond(d156,d156_5,d156_6,7).
bond(d156,d156_6,d156_1,7).
bond(d156,d156_2,d156_7,1).
bond(d156,d156_5,d156_8,1).
bond(d156,d156_6,d156_9,1).
bond(d156,d156_3,d156_10,1).
bond(d156,d156_10,d156_11,1).
bond(d156,d156_4,d156_12,1).
bond(d156,d156_12,d156_11,1).
bond(d156,d156_10,d156_13,1).
bond(d156,d156_12,d156_14,1).
bond(d156,d156_12,d156_15,1).
bond(d156,d156_11,d156_16,1).
bond(d156,d156_11,d156_17,1).
bond(d156,d156_1,d156_18,1).
bond(d156,d156_18,d156_19,2).
bond(d156,d156_18,d156_20,2).
atm(d157,d157_1,c,16,-0.191).
atm(d157,d157_2,c,10,0.109).
atm(d157,d157_3,c,22,-0.121).
atm(d157,d157_4,c,27,0.009).
atm(d157,d157_5,c,22,0.109).
atm(d157,d157_6,c,16,-0.191).
atm(d157,d157_7,c,22,-0.121).
atm(d157,d157_8,c,22,-0.121).
atm(d157,d157_9,c,27,0.009).
atm(d157,d157_10,c,27,-0.091).
atm(d157,d157_11,c,27,0.009).
atm(d157,d157_12,c,22,-0.121).
atm(d157,d157_13,c,22,-0.121).
atm(d157,d157_14,c,22,-0.121).
atm(d157,d157_15,c,22,-0.121).
atm(d157,d157_16,c,22,-0.121).
atm(d157,d157_17,h,3,0.139).
atm(d157,d157_18,h,3,0.139).
atm(d157,d157_19,h,3,0.139).
atm(d157,d157_20,h,3,0.139).
atm(d157,d157_21,h,3,0.139).
atm(d157,d157_22,h,3,0.139).
atm(d157,d157_23,h,3,0.059).
atm(d157,d157_24,h,3,0.059).
atm(d157,d157_25,h,3,0.109).
atm(d157,d157_26,h,3,0.109).
atm(d157,d157_27,n,38,0.808).
atm(d157,d157_28,o,40,-0.391).
atm(d157,d157_29,o,40,-0.391).

bond(d157,d157_1,d157_2,1).
bond(d157,d157_2,d157_3,1).
bond(d157,d157_3,d157_4,7).
bond(d157,d157_4,d157_5,7).
bond(d157,d157_5,d157_6,1).
bond(d157,d157_6,d157_1,2).
bond(d157,d157_3,d157_7,7).
bond(d157,d157_7,d157_8,7).
bond(d157,d157_8,d157_9,7).
bond(d157,d157_9,d157_10,7).
bond(d157,d157_10,d157_4,7).
bond(d157,d157_10,d157_11,7).
bond(d157,d157_11,d157_12,7).
bond(d157,d157_12,d157_13,7).
bond(d157,d157_13,d157_5,7).
bond(d157,d157_9,d157_14,7).
bond(d157,d157_14,d157_15,7).
bond(d157,d157_15,d157_16,7).
bond(d157,d157_16,d157_11,7).
bond(d157,d157_7,d157_17,1).
bond(d157,d157_8,d157_18,1).
bond(d157,d157_12,d157_19,1).
bond(d157,d157_13,d157_20,1).
bond(d157,d157_14,d157_21,1).
bond(d157,d157_15,d157_22,1).
bond(d157,d157_2,d157_23,1).
bond(d157,d157_2,d157_24,1).
bond(d157,d157_6,d157_25,1).
bond(d157,d157_1,d157_26,1).
bond(d157,d157_16,d157_27,1).
bond(d157,d157_27,d157_28,2).
bond(d157,d157_27,d157_29,2).
atm(d158,d158_1,c,22,-0.124).
atm(d158,d158_2,c,22,-0.124).
atm(d158,d158_3,c,22,-0.124).
atm(d158,d158_4,c,29,0.006).
atm(d158,d158_5,c,22,-0.124).
atm(d158,d158_6,c,22,-0.124).
atm(d158,d158_7,h,3,0.136).
atm(d158,d158_8,h,3,0.136).
atm(d158,d158_9,h,3,0.136).
atm(d158,d158_10,c,22,-0.124).
atm(d158,d158_11,c,22,-0.124).
atm(d158,d158_12,c,22,0.206).
atm(d158,d158_13,c,22,-0.124).
atm(d158,d158_14,c,22,-0.124).
atm(d158,d158_15,c,29,0.006).
atm(d158,d158_16,h,3,0.136).
atm(d158,d158_17,h,3,0.136).
atm(d158,d158_18,h,3,0.136).
atm(d158,d158_19,c,10,0.006).
atm(d158,d158_20,h,3,0.056).
atm(d158,d158_21,h,3,0.056).
atm(d158,d158_22,n,38,0.805).
atm(d158,d158_23,o,40,-0.394).
atm(d158,d158_24,o,40,-0.394).
atm(d158,d158_25,n,32,-0.394).
atm(d158,d158_26,c,14,0.605).
atm(d158,d158_27,c,10,-0.144).
atm(d158,d158_28,h,1,0.256).
atm(d158,d158_29,o,40,-0.544).
atm(d158,d158_30,h,3,0.056).
atm(d158,d158_31,h,3,0.056).
atm(d158,d158_32,h,3,0.056).

bond(d158,d158_1,d158_2,7).
bond(d158,d158_2,d158_3,7).
bond(d158,d158_3,d158_4,7).
bond(d158,d158_4,d158_5,7).
bond(d158,d158_5,d158_6,7).
bond(d158,d158_6,d158_1,7).
bond(d158,d158_2,d158_7,1).
bond(d158,d158_5,d158_8,1).
bond(d158,d158_6,d158_9,1).
bond(d158,d158_10,d158_11,7).
bond(d158,d158_11,d158_12,7).
bond(d158,d158_12,d158_13,7).
bond(d158,d158_13,d158_14,7).
bond(d158,d158_14,d158_15,7).
bond(d158,d158_15,d158_10,7).
bond(d158,d158_11,d158_16,1).
bond(d158,d158_13,d158_17,1).
bond(d158,d158_14,d158_18,1).
bond(d158,d158_4,d158_15,1).
bond(d158,d158_3,d158_19,1).
bond(d158,d158_19,d158_10,1).
bond(d158,d158_19,d158_20,1).
bond(d158,d158_19,d158_21,1).
bond(d158,d158_1,d158_22,1).
bond(d158,d158_23,d158_22,2).
bond(d158,d158_22,d158_24,2).
bond(d158,d158_12,d158_25,1).
bond(d158,d158_25,d158_26,1).
bond(d158,d158_26,d158_27,1).
bond(d158,d158_25,d158_28,1).
bond(d158,d158_26,d158_29,2).
bond(d158,d158_27,d158_30,1).
bond(d158,d158_27,d158_31,1).
bond(d158,d158_27,d158_32,1).
atm(d159,d159_1,c,16,-0.191).
atm(d159,d159_2,c,16,-0.191).
atm(d159,d159_3,c,22,0.109).
atm(d159,d159_4,c,27,0.009).
atm(d159,d159_5,c,22,-0.121).
atm(d159,d159_6,c,10,0.109).
atm(d159,d159_7,c,22,-0.121).
atm(d159,d159_8,c,22,-0.121).
atm(d159,d159_9,c,27,0.009).
atm(d159,d159_10,c,27,-0.091).
atm(d159,d159_11,c,27,0.009).
atm(d159,d159_12,c,22,-0.121).
atm(d159,d159_13,c,22,-0.121).
atm(d159,d159_14,c,22,-0.121).
atm(d159,d159_15,c,22,-0.121).
atm(d159,d159_16,c,22,-0.121).
atm(d159,d159_17,h,3,0.139).
atm(d159,d159_18,h,3,0.139).
atm(d159,d159_19,h,3,0.139).
atm(d159,d159_20,h,3,0.139).
atm(d159,d159_21,h,3,0.139).
atm(d159,d159_22,h,3,0.139).
atm(d159,d159_23,h,3,0.059).
atm(d159,d159_24,h,3,0.059).
atm(d159,d159_25,h,3,0.109).
atm(d159,d159_26,h,3,0.109).
atm(d159,d159_27,n,38,0.808).
atm(d159,d159_28,o,40,-0.391).
atm(d159,d159_29,o,40,-0.391).

bond(d159,d159_1,d159_2,2).
bond(d159,d159_2,d159_3,1).
bond(d159,d159_3,d159_4,7).
bond(d159,d159_4,d159_5,7).
bond(d159,d159_5,d159_6,1).
bond(d159,d159_6,d159_1,1).
bond(d159,d159_3,d159_7,7).
bond(d159,d159_7,d159_8,7).
bond(d159,d159_8,d159_9,7).
bond(d159,d159_9,d159_10,7).
bond(d159,d159_10,d159_4,7).
bond(d159,d159_10,d159_11,7).
bond(d159,d159_11,d159_12,7).
bond(d159,d159_12,d159_13,7).
bond(d159,d159_13,d159_5,7).
bond(d159,d159_9,d159_14,7).
bond(d159,d159_14,d159_15,7).
bond(d159,d159_15,d159_16,7).
bond(d159,d159_16,d159_11,7).
bond(d159,d159_7,d159_17,1).
bond(d159,d159_8,d159_18,1).
bond(d159,d159_12,d159_19,1).
bond(d159,d159_13,d159_20,1).
bond(d159,d159_14,d159_21,1).
bond(d159,d159_15,d159_22,1).
bond(d159,d159_6,d159_23,1).
bond(d159,d159_6,d159_24,1).
bond(d159,d159_1,d159_25,1).
bond(d159,d159_2,d159_26,1).
bond(d159,d159_16,d159_27,1).
bond(d159,d159_27,d159_28,2).
bond(d159,d159_27,d159_29,2).
atm(d16,d16_1,c,22,-0.118).
atm(d16,d16_2,c,22,-0.118).
atm(d16,d16_3,c,27,0.012).
atm(d16,d16_4,c,27,-0.088).
atm(d16,d16_5,c,28,0.012).
atm(d16,d16_6,c,22,-0.118).
atm(d16,d16_7,c,22,-0.118).
atm(d16,d16_8,c,22,-0.118).
atm(d16,d16_9,c,22,-0.118).
atm(d16,d16_10,c,29,0.012).
atm(d16,d16_11,c,28,0.012).
atm(d16,d16_12,c,28,0.012).
atm(d16,d16_13,c,22,-0.118).
atm(d16,d16_14,c,22,-0.118).
atm(d16,d16_15,c,22,-0.118).
atm(d16,d16_16,c,22,-0.118).
atm(d16,d16_17,n,38,0.812).
atm(d16,d16_18,n,38,0.812).
atm(d16,d16_19,o,40,-0.388).
atm(d16,d16_20,o,40,-0.388).
atm(d16,d16_21,o,40,-0.388).
atm(d16,d16_22,o,40,-0.388).
atm(d16,d16_23,h,3,0.142).
atm(d16,d16_24,h,3,0.142).
atm(d16,d16_25,h,3,0.142).
atm(d16,d16_26,h,3,0.142).
atm(d16,d16_27,h,3,0.142).
atm(d16,d16_28,h,3,0.142).
atm(d16,d16_29,h,3,0.142).
atm(d16,d16_30,h,3,0.142).

bond(d16,d16_1,d16_2,7).
bond(d16,d16_2,d16_3,7).
bond(d16,d16_3,d16_4,7).
bond(d16,d16_4,d16_5,7).
bond(d16,d16_5,d16_6,7).
bond(d16,d16_6,d16_1,7).
bond(d16,d16_3,d16_7,7).
bond(d16,d16_7,d16_8,7).
bond(d16,d16_8,d16_9,7).
bond(d16,d16_9,d16_10,7).
bond(d16,d16_10,d16_4,7).
bond(d16,d16_11,d16_12,7).
bond(d16,d16_12,d16_13,7).
bond(d16,d16_13,d16_14,7).
bond(d16,d16_14,d16_15,7).
bond(d16,d16_15,d16_16,7).
bond(d16,d16_16,d16_11,7).
bond(d16,d16_11,d16_5,1).
bond(d16,d16_7,d16_17,1).
bond(d16,d16_2,d16_18,1).
bond(d16,d16_17,d16_19,2).
bond(d16,d16_17,d16_20,2).
bond(d16,d16_18,d16_21,2).
bond(d16,d16_18,d16_22,2).
bond(d16,d16_1,d16_23,1).
bond(d16,d16_6,d16_24,1).
bond(d16,d16_8,d16_25,1).
bond(d16,d16_9,d16_26,1).
bond(d16,d16_14,d16_27,1).
bond(d16,d16_15,d16_28,1).
bond(d16,d16_16,d16_29,1).
bond(d16,d16_13,d16_30,1).
bond(d16,d16_12,d16_10,1).
atm(d160,d160_1,c,22,-0.104).
atm(d160,d160_2,c,22,-0.104).
atm(d160,d160_3,c,22,-0.104).
atm(d160,d160_4,c,22,-0.104).
atm(d160,d160_5,c,22,0.226).
atm(d160,d160_6,c,22,-0.104).
atm(d160,d160_7,h,3,0.156).
atm(d160,d160_8,h,3,0.156).
atm(d160,d160_9,n,32,-0.755).
atm(d160,d160_10,h,1,0.355).
atm(d160,d160_11,h,1,0.356).
atm(d160,d160_12,br,94,-0.124).
atm(d160,d160_13,n,38,0.825).
atm(d160,d160_14,n,38,0.825).
atm(d160,d160_15,o,40,-0.375).
atm(d160,d160_16,o,40,-0.375).
atm(d160,d160_17,o,40,-0.375).
atm(d160,d160_18,o,40,-0.375).

bond(d160,d160_1,d160_2,7).
bond(d160,d160_2,d160_3,7).
bond(d160,d160_3,d160_4,7).
bond(d160,d160_4,d160_5,7).
bond(d160,d160_5,d160_6,7).
bond(d160,d160_6,d160_1,7).
bond(d160,d160_1,d160_7,1).
bond(d160,d160_3,d160_8,1).
bond(d160,d160_5,d160_9,1).
bond(d160,d160_9,d160_10,1).
bond(d160,d160_9,d160_11,1).
bond(d160,d160_4,d160_12,1).
bond(d160,d160_2,d160_13,1).
bond(d160,d160_6,d160_14,1).
bond(d160,d160_14,d160_15,2).
bond(d160,d160_14,d160_16,2).
bond(d160,d160_13,d160_17,2).
bond(d160,d160_13,d160_18,2).
atm(d161,d161_1,c,22,-0.106).
atm(d161,d161_2,c,22,-0.106).
atm(d161,d161_3,c,27,0.024).
atm(d161,d161_4,c,195,-0.076).
atm(d161,d161_5,c,195,-0.076).
atm(d161,d161_6,c,22,-0.106).
atm(d161,d161_7,h,3,0.154).
atm(d161,d161_8,h,3,0.154).
atm(d161,d161_9,c,22,-0.106).
atm(d161,d161_10,c,22,-0.106).
atm(d161,d161_11,c,27,0.024).
atm(d161,d161_12,c,195,-0.076).
atm(d161,d161_13,h,3,0.154).
atm(d161,d161_14,h,3,0.154).
atm(d161,d161_15,c,195,-0.076).
atm(d161,d161_16,c,27,-0.076).
atm(d161,d161_17,c,27,-0.076).
atm(d161,d161_18,c,22,-0.106).
atm(d161,d161_19,c,22,-0.106).
atm(d161,d161_20,c,22,-0.106).
atm(d161,d161_21,h,3,0.154).
atm(d161,d161_22,h,3,0.154).
atm(d161,d161_23,c,22,-0.106).
atm(d161,d161_24,c,22,-0.106).
atm(d161,d161_25,c,22,-0.106).
atm(d161,d161_26,c,22,-0.106).
atm(d161,d161_27,h,3,0.154).
atm(d161,d161_28,h,3,0.154).
atm(d161,d161_29,h,3,0.154).
atm(d161,d161_30,h,3,0.154).
atm(d161,d161_31,n,38,0.823).
atm(d161,d161_32,n,38,0.823).
atm(d161,d161_33,o,40,-0.377).
atm(d161,d161_34,o,40,-0.376).
atm(d161,d161_35,o,40,-0.376).
atm(d161,d161_36,o,40,-0.377).

bond(d161,d161_1,d161_2,7).
bond(d161,d161_2,d161_3,7).
bond(d161,d161_3,d161_4,7).
bond(d161,d161_4,d161_5,7).
bond(d161,d161_5,d161_6,7).
bond(d161,d161_6,d161_1,7).
bond(d161,d161_1,d161_7,1).
bond(d161,d161_6,d161_8,1).
bond(d161,d161_3,d161_9,7).
bond(d161,d161_9,d161_10,7).
bond(d161,d161_10,d161_11,7).
bond(d161,d161_11,d161_12,7).
bond(d161,d161_12,d161_4,7).
bond(d161,d161_9,d161_13,1).
bond(d161,d161_10,d161_14,1).
bond(d161,d161_12,d161_15,7).
bond(d161,d161_15,d161_16,7).
bond(d161,d161_16,d161_17,7).
bond(d161,d161_17,d161_5,7).
bond(d161,d161_11,d161_18,7).
bond(d161,d161_18,d161_19,7).
bond(d161,d161_19,d161_20,7).
bond(d161,d161_20,d161_15,7).
bond(d161,d161_18,d161_21,1).
bond(d161,d161_19,d161_22,1).
bond(d161,d161_16,d161_23,7).
bond(d161,d161_23,d161_24,7).
bond(d161,d161_24,d161_25,7).
bond(d161,d161_25,d161_26,7).
bond(d161,d161_26,d161_17,7).
bond(d161,d161_23,d161_27,1).
bond(d161,d161_24,d161_28,1).
bond(d161,d161_25,d161_29,1).
bond(d161,d161_26,d161_30,1).
bond(d161,d161_20,d161_31,1).
bond(d161,d161_2,d161_32,1).
bond(d161,d161_31,d161_33,2).
bond(d161,d161_31,d161_34,2).
bond(d161,d161_32,d161_35,2).
bond(d161,d161_32,d161_36,2).
atm(d162,d162_1,c,22,-0.113).
atm(d162,d162_2,c,22,-0.113).
atm(d162,d162_3,c,22,-0.113).
atm(d162,d162_4,c,29,0.017).
atm(d162,d162_5,c,22,-0.113).
atm(d162,d162_6,c,22,-0.113).
atm(d162,d162_7,h,3,0.147).
atm(d162,d162_8,h,3,0.147).
atm(d162,d162_9,h,3,0.147).
atm(d162,d162_10,c,29,0.017).
atm(d162,d162_11,c,22,-0.113).
atm(d162,d162_12,c,22,-0.113).
atm(d162,d162_13,c,22,-0.113).
atm(d162,d162_14,c,22,-0.113).
atm(d162,d162_15,c,22,-0.113).
atm(d162,d162_16,h,3,0.147).
atm(d162,d162_17,h,3,0.147).
atm(d162,d162_18,h,3,0.147).
atm(d162,d162_19,n,38,0.818).
atm(d162,d162_20,n,38,0.818).
atm(d162,d162_21,n,38,0.818).
atm(d162,d162_22,n,38,0.818).
atm(d162,d162_23,o,40,-0.382).
atm(d162,d162_24,o,40,-0.382).
atm(d162,d162_25,o,40,-0.382).
atm(d162,d162_26,o,40,-0.383).
atm(d162,d162_27,o,40,-0.383).
atm(d162,d162_28,o,40,-0.382).
atm(d162,d162_29,o,40,-0.382).
atm(d162,d162_30,o,40,-0.382).

bond(d162,d162_1,d162_2,7).
bond(d162,d162_2,d162_3,7).
bond(d162,d162_3,d162_4,7).
bond(d162,d162_4,d162_5,7).
bond(d162,d162_5,d162_6,7).
bond(d162,d162_6,d162_1,7).
bond(d162,d162_2,d162_7,1).
bond(d162,d162_3,d162_8,1).
bond(d162,d162_5,d162_9,1).
bond(d162,d162_10,d162_11,7).
bond(d162,d162_11,d162_12,7).
bond(d162,d162_12,d162_13,7).
bond(d162,d162_13,d162_14,7).
bond(d162,d162_14,d162_15,7).
bond(d162,d162_15,d162_10,7).
bond(d162,d162_11,d162_16,1).
bond(d162,d162_12,d162_17,1).
bond(d162,d162_15,d162_18,1).
bond(d162,d162_4,d162_10,1).
bond(d162,d162_14,d162_19,1).
bond(d162,d162_13,d162_20,1).
bond(d162,d162_6,d162_21,1).
bond(d162,d162_1,d162_22,1).
bond(d162,d162_22,d162_23,2).
bond(d162,d162_22,d162_24,2).
bond(d162,d162_21,d162_25,2).
bond(d162,d162_21,d162_26,2).
bond(d162,d162_19,d162_27,2).
bond(d162,d162_19,d162_28,2).
bond(d162,d162_20,d162_29,2).
bond(d162,d162_20,d162_30,2).
atm(d163,d163_1,c,22,-0.115).
atm(d163,d163_2,c,22,-0.115).
atm(d163,d163_3,c,27,0.015).
atm(d163,d163_4,c,27,-0.085).
atm(d163,d163_5,c,27,0.015).
atm(d163,d163_6,c,22,-0.115).
atm(d163,d163_7,h,3,0.145).
atm(d163,d163_8,h,3,0.146).
atm(d163,d163_9,c,22,-0.115).
atm(d163,d163_10,c,22,-0.115).
atm(d163,d163_11,c,27,0.015).
atm(d163,d163_12,c,27,-0.085).
atm(d163,d163_13,h,3,0.146).
atm(d163,d163_14,h,3,0.146).
atm(d163,d163_15,c,27,0.015).
atm(d163,d163_16,c,22,-0.115).
atm(d163,d163_17,c,22,-0.115).
atm(d163,d163_18,h,3,0.145).
atm(d163,d163_19,h,3,0.145).
atm(d163,d163_20,c,22,-0.115).
atm(d163,d163_21,c,22,-0.115).
atm(d163,d163_22,c,22,-0.115).
atm(d163,d163_23,h,3,0.146).
atm(d163,d163_24,h,3,0.145).
atm(d163,d163_25,n,38,0.816).
atm(d163,d163_26,n,38,0.816).
atm(d163,d163_27,o,40,-0.384).
atm(d163,d163_28,o,40,-0.384).
atm(d163,d163_29,o,40,-0.384).
atm(d163,d163_30,o,40,-0.384).

bond(d163,d163_1,d163_2,7).
bond(d163,d163_2,d163_3,7).
bond(d163,d163_3,d163_4,7).
bond(d163,d163_4,d163_5,7).
bond(d163,d163_5,d163_6,7).
bond(d163,d163_6,d163_1,7).
bond(d163,d163_2,d163_7,1).
bond(d163,d163_6,d163_8,1).
bond(d163,d163_3,d163_9,7).
bond(d163,d163_9,d163_10,7).
bond(d163,d163_10,d163_11,7).
bond(d163,d163_11,d163_12,7).
bond(d163,d163_12,d163_4,7).
bond(d163,d163_9,d163_13,1).
bond(d163,d163_10,d163_14,1).
bond(d163,d163_12,d163_15,7).
bond(d163,d163_15,d163_16,7).
bond(d163,d163_16,d163_17,7).
bond(d163,d163_17,d163_5,7).
bond(d163,d163_16,d163_18,1).
bond(d163,d163_17,d163_19,1).
bond(d163,d163_11,d163_20,7).
bond(d163,d163_20,d163_21,7).
bond(d163,d163_21,d163_22,7).
bond(d163,d163_22,d163_15,7).
bond(d163,d163_20,d163_23,1).
bond(d163,d163_22,d163_24,1).
bond(d163,d163_21,d163_25,1).
bond(d163,d163_1,d163_26,1).
bond(d163,d163_25,d163_27,2).
bond(d163,d163_25,d163_28,2).
bond(d163,d163_26,d163_29,2).
bond(d163,d163_26,d163_30,2).
atm(d164,d164_1,c,22,-0.115).
atm(d164,d164_2,c,22,-0.115).
atm(d164,d164_3,c,27,0.015).
atm(d164,d164_4,c,27,-0.085).
atm(d164,d164_5,c,27,0.015).
atm(d164,d164_6,c,22,-0.115).
atm(d164,d164_7,h,3,0.145).
atm(d164,d164_8,h,3,0.146).
atm(d164,d164_9,c,22,-0.115).
atm(d164,d164_10,c,22,-0.115).
atm(d164,d164_11,c,27,0.015).
atm(d164,d164_12,c,27,-0.085).
atm(d164,d164_13,h,3,0.146).
atm(d164,d164_14,h,3,0.146).
atm(d164,d164_15,c,27,0.015).
atm(d164,d164_16,c,22,-0.115).
atm(d164,d164_17,c,22,-0.115).
atm(d164,d164_18,h,3,0.145).
atm(d164,d164_19,h,3,0.145).
atm(d164,d164_20,c,22,-0.115).
atm(d164,d164_21,c,22,-0.115).
atm(d164,d164_22,c,22,-0.115).
atm(d164,d164_23,h,3,0.146).
atm(d164,d164_24,h,3,0.145).
atm(d164,d164_25,n,38,0.816).
atm(d164,d164_26,n,38,0.816).
atm(d164,d164_27,o,40,-0.384).
atm(d164,d164_28,o,40,-0.384).
atm(d164,d164_29,o,40,-0.384).
atm(d164,d164_30,o,40,-0.384).

bond(d164,d164_1,d164_2,7).
bond(d164,d164_2,d164_3,7).
bond(d164,d164_3,d164_4,7).
bond(d164,d164_4,d164_5,7).
bond(d164,d164_5,d164_6,7).
bond(d164,d164_6,d164_1,7).
bond(d164,d164_2,d164_7,1).
bond(d164,d164_6,d164_8,1).
bond(d164,d164_3,d164_9,7).
bond(d164,d164_9,d164_10,7).
bond(d164,d164_10,d164_11,7).
bond(d164,d164_11,d164_12,7).
bond(d164,d164_12,d164_4,7).
bond(d164,d164_9,d164_13,1).
bond(d164,d164_10,d164_14,1).
bond(d164,d164_12,d164_15,7).
bond(d164,d164_15,d164_16,7).
bond(d164,d164_16,d164_17,7).
bond(d164,d164_17,d164_5,7).
bond(d164,d164_16,d164_18,1).
bond(d164,d164_17,d164_19,1).
bond(d164,d164_11,d164_20,7).
bond(d164,d164_20,d164_21,7).
bond(d164,d164_21,d164_22,7).
bond(d164,d164_22,d164_15,7).
bond(d164,d164_21,d164_23,1).
bond(d164,d164_1,d164_24,1).
bond(d164,d164_22,d164_25,1).
bond(d164,d164_20,d164_26,1).
bond(d164,d164_25,d164_27,2).
bond(d164,d164_25,d164_28,2).
bond(d164,d164_26,d164_29,2).
bond(d164,d164_26,d164_30,2).
atm(d165,d165_1,c,22,-0.122).
atm(d165,d165_2,c,22,-0.122).
atm(d165,d165_3,c,27,0.008).
atm(d165,d165_4,c,27,-0.092).
atm(d165,d165_5,c,28,0.008).
atm(d165,d165_6,c,22,-0.122).
atm(d165,d165_7,h,3,0.139).
atm(d165,d165_8,h,3,0.138).
atm(d165,d165_9,h,3,0.138).
atm(d165,d165_10,c,22,-0.122).
atm(d165,d165_11,c,22,-0.122).
atm(d165,d165_12,c,22,-0.122).
atm(d165,d165_13,c,29,0.008).
atm(d165,d165_14,h,3,0.139).
atm(d165,d165_15,h,3,0.138).
atm(d165,d165_16,h,3,0.138).
atm(d165,d165_17,c,28,0.008).
atm(d165,d165_18,c,28,0.008).
atm(d165,d165_19,c,22,-0.122).
atm(d165,d165_20,c,22,-0.122).
atm(d165,d165_21,c,22,-0.122).
atm(d165,d165_22,c,22,-0.122).
atm(d165,d165_23,h,3,0.139).
atm(d165,d165_24,h,3,0.138).
atm(d165,d165_25,h,3,0.138).
atm(d165,d165_26,n,38,0.809).
atm(d165,d165_27,o,40,-0.391).
atm(d165,d165_28,o,40,-0.391).

bond(d165,d165_1,d165_2,7).
bond(d165,d165_2,d165_3,7).
bond(d165,d165_3,d165_4,7).
bond(d165,d165_4,d165_5,7).
bond(d165,d165_5,d165_6,7).
bond(d165,d165_6,d165_1,7).
bond(d165,d165_1,d165_7,1).
bond(d165,d165_2,d165_8,1).
bond(d165,d165_6,d165_9,1).
bond(d165,d165_3,d165_10,7).
bond(d165,d165_10,d165_11,7).
bond(d165,d165_11,d165_12,7).
bond(d165,d165_12,d165_13,7).
bond(d165,d165_13,d165_4,7).
bond(d165,d165_10,d165_14,1).
bond(d165,d165_11,d165_15,1).
bond(d165,d165_12,d165_16,1).
bond(d165,d165_17,d165_18,7).
bond(d165,d165_18,d165_19,7).
bond(d165,d165_19,d165_20,7).
bond(d165,d165_20,d165_21,7).
bond(d165,d165_21,d165_22,7).
bond(d165,d165_22,d165_17,7).
bond(d165,d165_19,d165_23,1).
bond(d165,d165_20,d165_24,1).
bond(d165,d165_22,d165_25,1).
bond(d165,d165_17,d165_5,1).
bond(d165,d165_18,d165_13,1).
bond(d165,d165_21,d165_26,1).
bond(d165,d165_26,d165_27,2).
bond(d165,d165_26,d165_28,2).
atm(d166,d166_1,c,22,-0.125).
atm(d166,d166_2,c,22,-0.125).
atm(d166,d166_3,c,27,0.005).
atm(d166,d166_4,c,27,-0.095).
atm(d166,d166_5,c,27,0.005).
atm(d166,d166_6,c,22,-0.125).
atm(d166,d166_7,h,3,0.135).
atm(d166,d166_8,h,3,0.136).
atm(d166,d166_9,h,3,0.136).
atm(d166,d166_10,c,22,-0.125).
atm(d166,d166_11,c,22,-0.125).
atm(d166,d166_12,c,27,0.005).
atm(d166,d166_13,c,27,-0.095).
atm(d166,d166_14,h,3,0.136).
atm(d166,d166_15,h,3,0.135).
atm(d166,d166_16,c,27,0.005).
atm(d166,d166_17,c,22,-0.125).
atm(d166,d166_18,c,22,-0.125).
atm(d166,d166_19,h,3,0.135).
atm(d166,d166_20,h,3,0.135).
atm(d166,d166_21,c,22,0.256).
atm(d166,d166_22,c,22,-0.125).
atm(d166,d166_23,c,22,-0.125).
atm(d166,d166_24,h,3,0.135).
atm(d166,d166_25,n,38,0.806).
atm(d166,d166_26,o,40,-0.394).
atm(d166,d166_27,o,40,-0.394).
atm(d166,d166_28,o,49,-0.394).
atm(d166,d166_29,c,14,0.706).
atm(d166,d166_30,o,51,-0.544).
atm(d166,d166_31,c,10,0.005).
atm(d166,d166_32,h,3,0.055).
atm(d166,d166_33,h,3,0.055).
atm(d166,d166_34,h,3,0.055).

bond(d166,d166_1,d166_2,7).
bond(d166,d166_2,d166_3,7).
bond(d166,d166_3,d166_4,7).
bond(d166,d166_4,d166_5,7).
bond(d166,d166_5,d166_6,7).
bond(d166,d166_6,d166_1,7).
bond(d166,d166_1,d166_7,1).
bond(d166,d166_2,d166_8,1).
bond(d166,d166_6,d166_9,1).
bond(d166,d166_3,d166_10,7).
bond(d166,d166_10,d166_11,7).
bond(d166,d166_11,d166_12,7).
bond(d166,d166_12,d166_13,7).
bond(d166,d166_13,d166_4,7).
bond(d166,d166_10,d166_14,1).
bond(d166,d166_11,d166_15,1).
bond(d166,d166_13,d166_16,7).
bond(d166,d166_16,d166_17,7).
bond(d166,d166_17,d166_18,7).
bond(d166,d166_18,d166_5,7).
bond(d166,d166_17,d166_19,1).
bond(d166,d166_18,d166_20,1).
bond(d166,d166_12,d166_21,7).
bond(d166,d166_21,d166_22,7).
bond(d166,d166_22,d166_23,7).
bond(d166,d166_23,d166_16,7).
bond(d166,d166_22,d166_24,1).
bond(d166,d166_23,d166_25,1).
bond(d166,d166_25,d166_26,2).
bond(d166,d166_25,d166_27,2).
bond(d166,d166_21,d166_28,1).
bond(d166,d166_28,d166_29,1).
bond(d166,d166_29,d166_30,2).
bond(d166,d166_29,d166_31,1).
bond(d166,d166_31,d166_32,1).
bond(d166,d166_31,d166_33,1).
bond(d166,d166_31,d166_34,1).
atm(d167,d167_1,c,22,-0.115).
atm(d167,d167_2,c,22,-0.115).
atm(d167,d167_3,c,27,0.015).
atm(d167,d167_4,c,27,-0.085).
atm(d167,d167_5,c,27,0.015).
atm(d167,d167_6,c,22,-0.115).
atm(d167,d167_7,h,3,0.145).
atm(d167,d167_8,h,3,0.146).
atm(d167,d167_9,c,22,-0.115).
atm(d167,d167_10,c,22,-0.115).
atm(d167,d167_11,c,27,0.015).
atm(d167,d167_12,c,27,-0.085).
atm(d167,d167_13,h,3,0.146).
atm(d167,d167_14,h,3,0.146).
atm(d167,d167_15,c,27,0.015).
atm(d167,d167_16,c,22,-0.115).
atm(d167,d167_17,c,22,-0.115).
atm(d167,d167_18,h,3,0.145).
atm(d167,d167_19,h,3,0.145).
atm(d167,d167_20,c,22,-0.115).
atm(d167,d167_21,c,22,-0.115).
atm(d167,d167_22,c,22,-0.115).
atm(d167,d167_23,h,3,0.146).
atm(d167,d167_24,h,3,0.145).
atm(d167,d167_25,n,38,0.816).
atm(d167,d167_26,n,38,0.816).
atm(d167,d167_27,o,40,-0.384).
atm(d167,d167_28,o,40,-0.384).
atm(d167,d167_29,o,40,-0.384).
atm(d167,d167_30,o,40,-0.384).

bond(d167,d167_1,d167_2,7).
bond(d167,d167_2,d167_3,7).
bond(d167,d167_3,d167_4,7).
bond(d167,d167_4,d167_5,7).
bond(d167,d167_5,d167_6,7).
bond(d167,d167_6,d167_1,7).
bond(d167,d167_1,d167_7,1).
bond(d167,d167_6,d167_8,1).
bond(d167,d167_3,d167_9,7).
bond(d167,d167_9,d167_10,7).
bond(d167,d167_10,d167_11,7).
bond(d167,d167_11,d167_12,7).
bond(d167,d167_12,d167_4,7).
bond(d167,d167_9,d167_13,1).
bond(d167,d167_10,d167_14,1).
bond(d167,d167_12,d167_15,7).
bond(d167,d167_15,d167_16,7).
bond(d167,d167_16,d167_17,7).
bond(d167,d167_17,d167_5,7).
bond(d167,d167_16,d167_18,1).
bond(d167,d167_17,d167_19,1).
bond(d167,d167_11,d167_20,7).
bond(d167,d167_20,d167_21,7).
bond(d167,d167_21,d167_22,7).
bond(d167,d167_22,d167_15,7).
bond(d167,d167_20,d167_23,1).
bond(d167,d167_21,d167_24,1).
bond(d167,d167_22,d167_25,1).
bond(d167,d167_2,d167_26,1).
bond(d167,d167_25,d167_27,2).
bond(d167,d167_25,d167_28,2).
bond(d167,d167_26,d167_29,2).
bond(d167,d167_26,d167_30,2).
atm(d168,d168_1,c,22,-0.133).
atm(d168,d168_2,c,22,-0.133).
atm(d168,d168_3,c,22,-0.133).
atm(d168,d168_4,c,22,-0.133).
atm(d168,d168_5,c,22,0.176).
atm(d168,d168_6,c,22,-0.134).
atm(d168,d168_7,h,3,0.127).
atm(d168,d168_8,h,3,0.127).
atm(d168,d168_9,h,3,0.127).
atm(d168,d168_10,h,3,0.127).
atm(d168,d168_11,o,50,-0.234).
atm(d168,d168_12,c,10,0.087).
atm(d168,d168_13,h,3,0.047).
atm(d168,d168_14,h,3,0.047).
atm(d168,d168_15,h,3,0.047).
atm(d168,d168_16,n,38,0.796).
atm(d168,d168_17,o,40,-0.404).
atm(d168,d168_18,o,40,-0.404).

bond(d168,d168_1,d168_2,7).
bond(d168,d168_2,d168_3,7).
bond(d168,d168_3,d168_4,7).
bond(d168,d168_4,d168_5,7).
bond(d168,d168_5,d168_6,7).
bond(d168,d168_6,d168_1,7).
bond(d168,d168_1,d168_7,1).
bond(d168,d168_3,d168_8,1).
bond(d168,d168_4,d168_9,1).
bond(d168,d168_6,d168_10,1).
bond(d168,d168_5,d168_11,1).
bond(d168,d168_11,d168_12,1).
bond(d168,d168_12,d168_13,1).
bond(d168,d168_12,d168_14,1).
bond(d168,d168_12,d168_15,1).
bond(d168,d168_2,d168_16,1).
bond(d168,d168_16,d168_17,2).
bond(d168,d168_16,d168_18,2).
atm(d169,d169_1,c,22,-0.114).
atm(d169,d169_2,c,22,-0.114).
atm(d169,d169_3,c,22,-0.114).
atm(d169,d169_4,c,22,-0.114).
atm(d169,d169_5,c,22,0.316).
atm(d169,d169_6,c,22,-0.114).
atm(d169,d169_7,h,3,0.146).
atm(d169,d169_8,h,3,0.146).
atm(d169,d169_9,h,3,0.146).
atm(d169,d169_10,f,92,-0.284).
atm(d169,d169_11,n,38,0.817).
atm(d169,d169_12,n,38,0.817).
atm(d169,d169_13,o,40,-0.383).
atm(d169,d169_14,o,40,-0.384).
atm(d169,d169_15,o,40,-0.384).
atm(d169,d169_16,o,40,-0.383).

bond(d169,d169_1,d169_2,7).
bond(d169,d169_2,d169_3,7).
bond(d169,d169_3,d169_4,7).
bond(d169,d169_4,d169_5,7).
bond(d169,d169_5,d169_6,7).
bond(d169,d169_6,d169_1,7).
bond(d169,d169_1,d169_7,1).
bond(d169,d169_3,d169_8,1).
bond(d169,d169_6,d169_9,1).
bond(d169,d169_5,d169_10,1).
bond(d169,d169_4,d169_11,1).
bond(d169,d169_2,d169_12,1).
bond(d169,d169_11,d169_13,2).
bond(d169,d169_11,d169_14,2).
bond(d169,d169_12,d169_15,2).
bond(d169,d169_12,d169_16,2).
atm(d17,d17_1,c,22,-0.132).
atm(d17,d17_2,c,22,-0.132).
atm(d17,d17_3,c,22,0.198).
atm(d17,d17_4,c,22,-0.132).
atm(d17,d17_5,c,22,0.198).
atm(d17,d17_6,c,22,-0.132).
atm(d17,d17_7,h,3,0.128).
atm(d17,d17_8,h,3,0.128).
atm(d17,d17_9,h,3,0.128).
atm(d17,d17_10,n,38,0.799).
atm(d17,d17_11,o,40,-0.401).
atm(d17,d17_12,o,40,-0.401).
atm(d17,d17_13,n,32,-0.781).
atm(d17,d17_14,n,32,-0.781).
atm(d17,d17_15,h,1,0.328).
atm(d17,d17_16,h,1,0.328).
atm(d17,d17_17,h,1,0.328).
atm(d17,d17_18,h,1,0.329).

bond(d17,d17_1,d17_2,7).
bond(d17,d17_2,d17_3,7).
bond(d17,d17_3,d17_4,7).
bond(d17,d17_4,d17_5,7).
bond(d17,d17_5,d17_6,7).
bond(d17,d17_6,d17_1,7).
bond(d17,d17_1,d17_7,1).
bond(d17,d17_2,d17_8,1).
bond(d17,d17_6,d17_9,1).
bond(d17,d17_4,d17_10,1).
bond(d17,d17_11,d17_10,2).
bond(d17,d17_10,d17_12,2).
bond(d17,d17_13,d17_5,1).
bond(d17,d17_3,d17_14,1).
bond(d17,d17_13,d17_15,1).
bond(d17,d17_13,d17_16,1).
bond(d17,d17_14,d17_17,1).
bond(d17,d17_14,d17_18,1).
atm(d170,d170_1,c,22,-0.148).
atm(d170,d170_2,c,22,-0.148).
atm(d170,d170_3,c,22,0.182).
atm(d170,d170_4,c,22,0.182).
atm(d170,d170_5,c,22,-0.148).
atm(d170,d170_6,c,22,-0.148).
atm(d170,d170_7,h,3,0.112).
atm(d170,d170_8,h,3,0.112).
atm(d170,d170_9,h,3,0.112).
atm(d170,d170_10,h,3,0.112).
atm(d170,d170_11,n,32,-0.419).
atm(d170,d170_12,c,22,0.182).
atm(d170,d170_13,c,22,0.182).
atm(d170,d170_14,n,32,-0.418).
atm(d170,d170_15,c,22,-0.148).
atm(d170,d170_16,c,22,-0.148).
atm(d170,d170_17,c,22,-0.148).
atm(d170,d170_18,c,22,-0.148).
atm(d170,d170_19,h,3,0.112).
atm(d170,d170_20,h,3,0.112).
atm(d170,d170_21,h,3,0.112).
atm(d170,d170_22,h,1,0.282).
atm(d170,d170_23,h,1,0.282).
atm(d170,d170_24,n,38,0.781).
atm(d170,d170_25,o,40,-0.418).
atm(d170,d170_26,o,40,-0.418).

bond(d170,d170_1,d170_2,7).
bond(d170,d170_2,d170_3,7).
bond(d170,d170_3,d170_4,7).
bond(d170,d170_4,d170_5,7).
bond(d170,d170_5,d170_6,7).
bond(d170,d170_6,d170_1,7).
bond(d170,d170_1,d170_7,1).
bond(d170,d170_2,d170_8,1).
bond(d170,d170_5,d170_9,1).
bond(d170,d170_6,d170_10,1).
bond(d170,d170_3,d170_11,1).
bond(d170,d170_11,d170_12,1).
bond(d170,d170_12,d170_13,7).
bond(d170,d170_13,d170_14,1).
bond(d170,d170_14,d170_4,1).
bond(d170,d170_12,d170_15,7).
bond(d170,d170_15,d170_16,7).
bond(d170,d170_16,d170_17,7).
bond(d170,d170_17,d170_18,7).
bond(d170,d170_18,d170_13,7).
bond(d170,d170_15,d170_19,1).
bond(d170,d170_16,d170_20,1).
bond(d170,d170_17,d170_21,1).
bond(d170,d170_14,d170_22,1).
bond(d170,d170_11,d170_23,1).
bond(d170,d170_18,d170_24,1).
bond(d170,d170_24,d170_25,2).
bond(d170,d170_24,d170_26,2).
atm(d171,d171_1,c,22,-0.113).
atm(d171,d171_2,c,22,-0.113).
atm(d171,d171_3,c,22,-0.113).
atm(d171,d171_4,c,29,0.017).
atm(d171,d171_5,c,22,-0.113).
atm(d171,d171_6,c,22,-0.113).
atm(d171,d171_7,h,3,0.147).
atm(d171,d171_8,h,3,0.147).
atm(d171,d171_9,h,3,0.147).
atm(d171,d171_10,c,29,0.017).
atm(d171,d171_11,c,22,-0.113).
atm(d171,d171_12,c,22,-0.113).
atm(d171,d171_13,c,22,-0.113).
atm(d171,d171_14,c,22,-0.113).
atm(d171,d171_15,c,22,-0.113).
atm(d171,d171_16,h,3,0.147).
atm(d171,d171_17,h,3,0.147).
atm(d171,d171_18,h,3,0.147).
atm(d171,d171_19,n,38,0.818).
atm(d171,d171_20,n,38,0.818).
atm(d171,d171_21,n,38,0.818).
atm(d171,d171_22,n,38,0.818).
atm(d171,d171_23,o,40,-0.382).
atm(d171,d171_24,o,40,-0.382).
atm(d171,d171_25,o,40,-0.382).
atm(d171,d171_26,o,40,-0.383).
atm(d171,d171_27,o,40,-0.383).
atm(d171,d171_28,o,40,-0.382).
atm(d171,d171_29,o,40,-0.382).
atm(d171,d171_30,o,40,-0.382).

bond(d171,d171_1,d171_2,7).
bond(d171,d171_2,d171_3,7).
bond(d171,d171_3,d171_4,7).
bond(d171,d171_4,d171_5,7).
bond(d171,d171_5,d171_6,7).
bond(d171,d171_6,d171_1,7).
bond(d171,d171_2,d171_7,1).
bond(d171,d171_3,d171_8,1).
bond(d171,d171_6,d171_9,1).
bond(d171,d171_10,d171_11,7).
bond(d171,d171_11,d171_12,7).
bond(d171,d171_12,d171_13,7).
bond(d171,d171_13,d171_14,7).
bond(d171,d171_14,d171_15,7).
bond(d171,d171_15,d171_10,7).
bond(d171,d171_11,d171_16,1).
bond(d171,d171_12,d171_17,1).
bond(d171,d171_14,d171_18,1).
bond(d171,d171_4,d171_10,1).
bond(d171,d171_15,d171_19,1).
bond(d171,d171_13,d171_20,1).
bond(d171,d171_5,d171_21,1).
bond(d171,d171_1,d171_22,1).
bond(d171,d171_22,d171_23,2).
bond(d171,d171_22,d171_24,2).
bond(d171,d171_21,d171_25,2).
bond(d171,d171_21,d171_26,2).
bond(d171,d171_19,d171_27,2).
bond(d171,d171_19,d171_28,2).
bond(d171,d171_20,d171_29,2).
bond(d171,d171_20,d171_30,2).
atm(d172,d172_1,c,22,-0.115).
atm(d172,d172_2,c,22,-0.115).
atm(d172,d172_3,c,26,0.315).
atm(d172,d172_4,c,26,0.115).
atm(d172,d172_5,c,22,-0.115).
atm(d172,d172_6,c,22,-0.115).
atm(d172,d172_7,n,34,-0.384).
atm(d172,d172_8,n,34,-0.384).
atm(d172,d172_9,c,21,-0.115).
atm(d172,d172_10,n,38,0.816).
atm(d172,d172_11,o,40,-0.384).
atm(d172,d172_12,o,40,-0.384).
atm(d172,d172_13,h,1,0.315).
atm(d172,d172_14,h,3,0.145).
atm(d172,d172_15,h,3,0.115).
atm(d172,d172_16,h,3,0.145).
atm(d172,d172_17,h,3,0.145).

bond(d172,d172_1,d172_2,7).
bond(d172,d172_2,d172_3,7).
bond(d172,d172_3,d172_4,7).
bond(d172,d172_4,d172_5,7).
bond(d172,d172_5,d172_6,7).
bond(d172,d172_6,d172_1,7).
bond(d172,d172_3,d172_7,7).
bond(d172,d172_7,d172_8,7).
bond(d172,d172_4,d172_9,7).
bond(d172,d172_9,d172_8,7).
bond(d172,d172_1,d172_10,1).
bond(d172,d172_10,d172_11,2).
bond(d172,d172_10,d172_12,2).
bond(d172,d172_7,d172_13,1).
bond(d172,d172_2,d172_14,1).
bond(d172,d172_5,d172_15,1).
bond(d172,d172_6,d172_16,1).
bond(d172,d172_9,d172_17,1).
atm(d173,d173_1,c,29,0.014).
atm(d173,d173_2,c,22,-0.116).
atm(d173,d173_3,c,22,-0.116).
atm(d173,d173_4,c,22,-0.116).
atm(d173,d173_5,c,22,-0.116).
atm(d173,d173_6,c,22,-0.116).
atm(d173,d173_7,h,3,0.144).
atm(d173,d173_8,h,3,0.144).
atm(d173,d173_9,h,3,0.144).
atm(d173,d173_10,c,22,-0.116).
atm(d173,d173_11,c,22,-0.116).
atm(d173,d173_12,c,22,-0.116).
atm(d173,d173_13,c,29,0.014).
atm(d173,d173_14,c,22,-0.116).
atm(d173,d173_15,c,22,-0.116).
atm(d173,d173_16,h,3,0.144).
atm(d173,d173_17,h,3,0.144).
atm(d173,d173_18,h,3,0.144).
atm(d173,d173_19,h,3,0.144).
atm(d173,d173_20,n,38,0.813).
atm(d173,d173_21,n,38,0.813).
atm(d173,d173_22,n,38,0.814).
atm(d173,d173_23,o,40,-0.386).
atm(d173,d173_24,o,40,-0.386).
atm(d173,d173_25,o,40,-0.386).
atm(d173,d173_26,o,40,-0.386).
atm(d173,d173_27,o,40,-0.386).
atm(d173,d173_28,o,40,-0.386).

bond(d173,d173_1,d173_2,7).
bond(d173,d173_2,d173_3,7).
bond(d173,d173_3,d173_4,7).
bond(d173,d173_4,d173_5,7).
bond(d173,d173_5,d173_6,7).
bond(d173,d173_6,d173_1,7).
bond(d173,d173_2,d173_7,1).
bond(d173,d173_3,d173_8,1).
bond(d173,d173_6,d173_9,1).
bond(d173,d173_10,d173_11,7).
bond(d173,d173_11,d173_12,7).
bond(d173,d173_12,d173_13,7).
bond(d173,d173_13,d173_14,7).
bond(d173,d173_14,d173_15,7).
bond(d173,d173_15,d173_10,7).
bond(d173,d173_11,d173_16,1).
bond(d173,d173_12,d173_17,1).
bond(d173,d173_14,d173_18,1).
bond(d173,d173_15,d173_19,1).
bond(d173,d173_13,d173_1,1).
bond(d173,d173_5,d173_20,1).
bond(d173,d173_4,d173_21,1).
bond(d173,d173_10,d173_22,1).
bond(d173,d173_22,d173_23,2).
bond(d173,d173_22,d173_24,2).
bond(d173,d173_20,d173_25,2).
bond(d173,d173_20,d173_26,2).
bond(d173,d173_21,d173_27,2).
bond(d173,d173_21,d173_28,2).
atm(d174,d174_1,c,22,-0.118).
atm(d174,d174_2,c,22,-0.118).
atm(d174,d174_3,c,27,0.012).
atm(d174,d174_4,c,27,-0.088).
atm(d174,d174_5,c,28,0.012).
atm(d174,d174_6,c,22,-0.118).
atm(d174,d174_7,h,3,0.142).
atm(d174,d174_8,h,3,0.142).
atm(d174,d174_9,h,3,0.142).
atm(d174,d174_10,c,22,-0.118).
atm(d174,d174_11,c,22,-0.118).
atm(d174,d174_12,c,22,-0.118).
atm(d174,d174_13,c,29,0.012).
atm(d174,d174_14,h,3,0.142).
atm(d174,d174_15,h,3,0.142).
atm(d174,d174_16,c,28,0.012).
atm(d174,d174_17,c,28,0.012).
atm(d174,d174_18,c,22,-0.118).
atm(d174,d174_19,c,22,-0.118).
atm(d174,d174_20,c,22,-0.118).
atm(d174,d174_21,c,22,-0.118).
atm(d174,d174_22,h,3,0.142).
atm(d174,d174_23,h,3,0.142).
atm(d174,d174_24,h,3,0.142).
atm(d174,d174_25,n,38,0.812).
atm(d174,d174_26,n,38,0.812).
atm(d174,d174_27,o,40,-0.388).
atm(d174,d174_28,o,40,-0.388).
atm(d174,d174_29,o,40,-0.388).
atm(d174,d174_30,o,40,-0.388).

bond(d174,d174_1,d174_2,7).
bond(d174,d174_2,d174_3,7).
bond(d174,d174_3,d174_4,7).
bond(d174,d174_4,d174_5,7).
bond(d174,d174_5,d174_6,7).
bond(d174,d174_6,d174_1,7).
bond(d174,d174_1,d174_7,1).
bond(d174,d174_2,d174_8,1).
bond(d174,d174_6,d174_9,1).
bond(d174,d174_3,d174_10,7).
bond(d174,d174_10,d174_11,7).
bond(d174,d174_11,d174_12,7).
bond(d174,d174_12,d174_13,7).
bond(d174,d174_13,d174_4,7).
bond(d174,d174_11,d174_14,1).
bond(d174,d174_12,d174_15,1).
bond(d174,d174_16,d174_17,7).
bond(d174,d174_17,d174_18,7).
bond(d174,d174_18,d174_19,7).
bond(d174,d174_19,d174_20,7).
bond(d174,d174_20,d174_21,7).
bond(d174,d174_21,d174_16,7).
bond(d174,d174_18,d174_22,1).
bond(d174,d174_20,d174_23,1).
bond(d174,d174_21,d174_24,1).
bond(d174,d174_16,d174_5,1).
bond(d174,d174_17,d174_13,1).
bond(d174,d174_10,d174_25,1).
bond(d174,d174_19,d174_26,1).
bond(d174,d174_25,d174_27,2).
bond(d174,d174_25,d174_28,2).
bond(d174,d174_26,d174_29,2).
bond(d174,d174_26,d174_30,2).
atm(d175,d175_1,c,22,-0.067).
atm(d175,d175_2,c,22,-0.067).
atm(d175,d175_3,c,22,0.244).
atm(d175,d175_4,c,22,0.244).
atm(d175,d175_5,c,22,-0.067).
atm(d175,d175_6,c,22,-0.067).
atm(d175,d175_7,h,3,0.194).
atm(d175,d175_8,h,3,0.193).
atm(d175,d175_9,o,50,-0.167).
atm(d175,d175_10,c,22,0.244).
atm(d175,d175_11,c,22,0.244).
atm(d175,d175_12,o,50,-0.167).
atm(d175,d175_13,c,22,-0.067).
atm(d175,d175_14,c,22,-0.067).
atm(d175,d175_15,c,22,-0.067).
atm(d175,d175_16,c,22,-0.067).
atm(d175,d175_17,n,38,0.864).
atm(d175,d175_18,o,40,-0.336).
atm(d175,d175_19,o,40,-0.336).
atm(d175,d175_20,cl,93,-0.137).
atm(d175,d175_21,cl,93,-0.137).
atm(d175,d175_22,cl,93,-0.137).
atm(d175,d175_23,cl,93,-0.137).
atm(d175,d175_24,cl,93,-0.137).

bond(d175,d175_1,d175_2,7).
bond(d175,d175_2,d175_3,7).
bond(d175,d175_3,d175_4,7).
bond(d175,d175_4,d175_5,7).
bond(d175,d175_5,d175_6,7).
bond(d175,d175_6,d175_1,7).
bond(d175,d175_2,d175_7,1).
bond(d175,d175_5,d175_8,1).
bond(d175,d175_3,d175_9,1).
bond(d175,d175_9,d175_10,1).
bond(d175,d175_10,d175_11,7).
bond(d175,d175_11,d175_12,1).
bond(d175,d175_12,d175_4,1).
bond(d175,d175_10,d175_13,7).
bond(d175,d175_13,d175_14,7).
bond(d175,d175_14,d175_15,7).
bond(d175,d175_15,d175_16,7).
bond(d175,d175_16,d175_11,7).
bond(d175,d175_14,d175_17,1).
bond(d175,d175_17,d175_18,2).
bond(d175,d175_17,d175_19,2).
bond(d175,d175_16,d175_20,1).
bond(d175,d175_15,d175_21,1).
bond(d175,d175_13,d175_22,1).
bond(d175,d175_1,d175_23,1).
bond(d175,d175_6,d175_24,1).
atm(d176,d176_1,c,22,-0.122).
atm(d176,d176_2,c,22,-0.122).
atm(d176,d176_3,c,22,-0.122).
atm(d176,d176_4,c,22,0.108).
atm(d176,d176_5,c,22,-0.122).
atm(d176,d176_6,c,22,-0.122).
atm(d176,d176_7,h,3,0.139).
atm(d176,d176_8,h,3,0.139).
atm(d176,d176_9,h,3,0.139).
atm(d176,d176_10,c,10,0.108).
atm(d176,d176_11,c,16,-0.191).
atm(d176,d176_12,h,3,0.058).
atm(d176,d176_13,h,3,0.058).
atm(d176,d176_14,c,16,-0.191).
atm(d176,d176_15,h,3,0.108).
atm(d176,d176_16,h,3,0.108).
atm(d176,d176_17,n,38,0.809).
atm(d176,d176_18,o,40,-0.391).
atm(d176,d176_19,o,40,-0.391).

bond(d176,d176_1,d176_2,7).
bond(d176,d176_2,d176_3,7).
bond(d176,d176_3,d176_4,7).
bond(d176,d176_4,d176_5,7).
bond(d176,d176_5,d176_6,7).
bond(d176,d176_6,d176_1,7).
bond(d176,d176_2,d176_7,1).
bond(d176,d176_5,d176_8,1).
bond(d176,d176_6,d176_9,1).
bond(d176,d176_3,d176_10,1).
bond(d176,d176_10,d176_11,1).
bond(d176,d176_10,d176_12,1).
bond(d176,d176_10,d176_13,1).
bond(d176,d176_4,d176_14,1).
bond(d176,d176_14,d176_11,2).
bond(d176,d176_14,d176_15,1).
bond(d176,d176_11,d176_16,1).
bond(d176,d176_1,d176_17,1).
bond(d176,d176_17,d176_18,2).
bond(d176,d176_17,d176_19,2).
atm(d177,d177_1,c,22,-0.142).
atm(d177,d177_2,c,22,-0.142).
atm(d177,d177_3,c,22,0.188).
atm(d177,d177_4,c,22,0.188).
atm(d177,d177_5,c,22,-0.142).
atm(d177,d177_6,c,22,-0.142).
atm(d177,d177_7,h,3,0.118).
atm(d177,d177_8,h,3,0.118).
atm(d177,d177_9,h,3,0.118).
atm(d177,d177_10,n,32,-0.412).
atm(d177,d177_11,c,22,0.188).
atm(d177,d177_12,c,22,0.188).
atm(d177,d177_13,n,32,-0.412).
atm(d177,d177_14,c,22,-0.142).
atm(d177,d177_15,c,22,-0.142).
atm(d177,d177_16,c,22,-0.142).
atm(d177,d177_17,c,22,-0.142).
atm(d177,d177_18,h,3,0.118).
atm(d177,d177_19,h,3,0.118).
atm(d177,d177_20,h,3,0.118).
atm(d177,d177_21,h,1,0.288).
atm(d177,d177_22,h,1,0.288).
atm(d177,d177_23,n,38,0.787).
atm(d177,d177_24,n,38,0.787).
atm(d177,d177_25,o,40,-0.413).
atm(d177,d177_26,o,40,-0.413).
atm(d177,d177_27,o,40,-0.412).
atm(d177,d177_28,o,40,-0.412).

bond(d177,d177_1,d177_2,7).
bond(d177,d177_2,d177_3,7).
bond(d177,d177_3,d177_4,7).
bond(d177,d177_4,d177_5,7).
bond(d177,d177_5,d177_6,7).
bond(d177,d177_6,d177_1,7).
bond(d177,d177_1,d177_7,1).
bond(d177,d177_2,d177_8,1).
bond(d177,d177_6,d177_9,1).
bond(d177,d177_3,d177_10,1).
bond(d177,d177_10,d177_11,1).
bond(d177,d177_11,d177_12,7).
bond(d177,d177_12,d177_13,1).
bond(d177,d177_13,d177_4,1).
bond(d177,d177_11,d177_14,7).
bond(d177,d177_14,d177_15,7).
bond(d177,d177_15,d177_16,7).
bond(d177,d177_16,d177_17,7).
bond(d177,d177_17,d177_12,7).
bond(d177,d177_14,d177_18,1).
bond(d177,d177_15,d177_19,1).
bond(d177,d177_16,d177_20,1).
bond(d177,d177_13,d177_21,1).
bond(d177,d177_10,d177_22,1).
bond(d177,d177_17,d177_23,1).
bond(d177,d177_5,d177_24,1).
bond(d177,d177_24,d177_25,2).
bond(d177,d177_24,d177_26,2).
bond(d177,d177_23,d177_27,2).
bond(d177,d177_23,d177_28,2).
atm(d178,d178_1,c,22,-0.152).
atm(d178,d178_2,c,22,-0.152).
atm(d178,d178_3,c,22,0.177).
atm(d178,d178_4,c,22,0.078).
atm(d178,d178_5,c,22,-0.152).
atm(d178,d178_6,c,22,-0.152).
atm(d178,d178_7,h,3,0.108).
atm(d178,d178_8,h,3,0.108).
atm(d178,d178_9,h,3,0.108).
atm(d178,d178_10,n,32,-0.423).
atm(d178,d178_11,n,32,-0.423).
atm(d178,d178_12,c,14,0.577).
atm(d178,d178_13,h,3,0.028).
atm(d178,d178_14,h,1,0.277).
atm(d178,d178_15,c,10,-0.022).
atm(d178,d178_16,h,3,0.028).
atm(d178,d178_17,h,3,0.028).
atm(d178,d178_18,h,3,0.028).
atm(d178,d178_19,n,38,0.777).
atm(d178,d178_20,o,40,-0.423).
atm(d178,d178_21,o,40,-0.423).

bond(d178,d178_1,d178_2,7).
bond(d178,d178_2,d178_3,7).
bond(d178,d178_3,d178_4,7).
bond(d178,d178_4,d178_5,7).
bond(d178,d178_5,d178_6,7).
bond(d178,d178_6,d178_1,7).
bond(d178,d178_1,d178_7,1).
bond(d178,d178_5,d178_8,1).
bond(d178,d178_6,d178_9,1).
bond(d178,d178_3,d178_10,1).
bond(d178,d178_10,d178_11,1).
bond(d178,d178_4,d178_12,1).
bond(d178,d178_12,d178_11,2).
bond(d178,d178_12,d178_13,1).
bond(d178,d178_10,d178_14,1).
bond(d178,d178_11,d178_15,1).
bond(d178,d178_15,d178_16,1).
bond(d178,d178_15,d178_17,1).
bond(d178,d178_15,d178_18,1).
bond(d178,d178_2,d178_19,1).
bond(d178,d178_19,d178_20,2).
bond(d178,d178_19,d178_21,2).
atm(d179,d179_1,c,22,-0.126).
atm(d179,d179_2,c,22,-0.126).
atm(d179,d179_3,c,22,-0.126).
atm(d179,d179_4,c,22,-0.126).
atm(d179,d179_5,c,22,0.104).
atm(d179,d179_6,c,22,-0.126).
atm(d179,d179_7,h,3,0.134).
atm(d179,d179_8,h,3,0.133).
atm(d179,d179_9,h,3,0.133).
atm(d179,d179_10,h,3,0.133).
atm(d179,d179_11,c,16,-0.197).
atm(d179,d179_12,c,16,0.104).
atm(d179,d179_13,h,3,0.104).
atm(d179,d179_14,o,50,-0.227).
atm(d179,d179_15,h,3,0.104).
atm(d179,d179_16,c,22,0.183).
atm(d179,d179_17,c,22,-0.126).
atm(d179,d179_18,c,22,-0.126).
atm(d179,d179_19,c,22,-0.126).
atm(d179,d179_20,c,22,-0.126).
atm(d179,d179_21,c,22,-0.126).
atm(d179,d179_22,h,3,0.133).
atm(d179,d179_23,h,3,0.133).
atm(d179,d179_24,h,3,0.134).
atm(d179,d179_25,h,3,0.134).
atm(d179,d179_26,n,38,0.803).
atm(d179,d179_27,o,40,-0.397).
atm(d179,d179_28,o,40,-0.397).
atm(d179,d179_29,n,38,0.803).
atm(d179,d179_30,o,40,-0.397).
atm(d179,d179_31,o,40,-0.397).

bond(d179,d179_1,d179_2,7).
bond(d179,d179_2,d179_3,7).
bond(d179,d179_3,d179_4,7).
bond(d179,d179_4,d179_5,7).
bond(d179,d179_5,d179_6,7).
bond(d179,d179_6,d179_1,7).
bond(d179,d179_1,d179_7,1).
bond(d179,d179_3,d179_8,1).
bond(d179,d179_4,d179_9,1).
bond(d179,d179_6,d179_10,1).
bond(d179,d179_5,d179_11,1).
bond(d179,d179_11,d179_12,2).
bond(d179,d179_11,d179_13,1).
bond(d179,d179_12,d179_14,1).
bond(d179,d179_12,d179_15,1).
bond(d179,d179_14,d179_16,1).
bond(d179,d179_16,d179_17,7).
bond(d179,d179_17,d179_18,7).
bond(d179,d179_18,d179_19,7).
bond(d179,d179_19,d179_20,7).
bond(d179,d179_20,d179_21,7).
bond(d179,d179_21,d179_16,7).
bond(d179,d179_17,d179_22,1).
bond(d179,d179_18,d179_23,1).
bond(d179,d179_20,d179_24,1).
bond(d179,d179_21,d179_25,1).
bond(d179,d179_2,d179_26,1).
bond(d179,d179_26,d179_27,2).
bond(d179,d179_26,d179_28,2).
bond(d179,d179_19,d179_29,1).
bond(d179,d179_29,d179_30,2).
bond(d179,d179_29,d179_31,2).
atm(d18,d18_1,c,22,-0.133).
atm(d18,d18_2,c,22,-0.133).
atm(d18,d18_3,c,22,0.097).
atm(d18,d18_4,c,29,-0.003).
atm(d18,d18_5,c,22,-0.133).
atm(d18,d18_6,c,22,-0.133).
atm(d18,d18_7,c,22,0.097).
atm(d18,d18_8,c,22,-0.133).
atm(d18,d18_9,c,22,-0.133).
atm(d18,d18_10,c,22,-0.133).
atm(d18,d18_11,c,22,-0.133).
atm(d18,d18_12,c,29,-0.003).
atm(d18,d18_13,c,14,0.547).
atm(d18,d18_14,o,40,-0.552).
atm(d18,d18_15,n,38,0.798).
atm(d18,d18_16,o,40,-0.403).
atm(d18,d18_17,o,40,-0.403).
atm(d18,d18_18,h,3,0.127).
atm(d18,d18_19,h,3,0.127).
atm(d18,d18_20,h,3,0.127).
atm(d18,d18_21,h,3,0.127).
atm(d18,d18_22,h,3,0.127).
atm(d18,d18_23,h,3,0.127).
atm(d18,d18_24,h,3,0.127).

bond(d18,d18_1,d18_2,7).
bond(d18,d18_2,d18_3,7).
bond(d18,d18_3,d18_4,7).
bond(d18,d18_4,d18_5,7).
bond(d18,d18_5,d18_6,7).
bond(d18,d18_6,d18_1,7).
bond(d18,d18_7,d18_8,7).
bond(d18,d18_8,d18_9,7).
bond(d18,d18_9,d18_10,7).
bond(d18,d18_10,d18_11,7).
bond(d18,d18_11,d18_12,7).
bond(d18,d18_12,d18_7,7).
bond(d18,d18_4,d18_12,1).
bond(d18,d18_3,d18_13,1).
bond(d18,d18_13,d18_7,1).
bond(d18,d18_13,d18_14,2).
bond(d18,d18_10,d18_15,1).
bond(d18,d18_16,d18_15,2).
bond(d18,d18_15,d18_17,2).
bond(d18,d18_1,d18_18,1).
bond(d18,d18_2,d18_19,1).
bond(d18,d18_5,d18_20,1).
bond(d18,d18_6,d18_21,1).
bond(d18,d18_8,d18_22,1).
bond(d18,d18_9,d18_23,1).
bond(d18,d18_11,d18_24,1).
atm(d180,d180_1,c,22,-0.113).
atm(d180,d180_2,c,22,-0.113).
atm(d180,d180_3,c,22,-0.113).
atm(d180,d180_4,c,29,0.017).
atm(d180,d180_5,c,27,-0.083).
atm(d180,d180_6,c,27,0.017).
atm(d180,d180_7,c,27,-0.083).
atm(d180,d180_8,c,27,0.017).
atm(d180,d180_9,c,22,-0.113).
atm(d180,d180_10,c,22,-0.113).
atm(d180,d180_11,c,27,-0.083).
atm(d180,d180_12,c,27,0.017).
atm(d180,d180_13,c,22,-0.113).
atm(d180,d180_14,c,22,-0.113).
atm(d180,d180_15,c,27,-0.083).
atm(d180,d180_16,c,27,0.017).
atm(d180,d180_17,c,22,-0.113).
atm(d180,d180_18,c,22,-0.113).
atm(d180,d180_19,c,29,0.017).
atm(d180,d180_20,c,22,-0.113).
atm(d180,d180_21,c,22,-0.113).
atm(d180,d180_22,c,22,-0.113).
atm(d180,d180_23,n,38,0.818).
atm(d180,d180_24,n,38,0.818).
atm(d180,d180_25,o,40,-0.382).
atm(d180,d180_26,o,40,-0.382).
atm(d180,d180_27,o,40,-0.382).
atm(d180,d180_28,o,40,-0.382).
atm(d180,d180_29,h,3,0.148).
atm(d180,d180_30,h,3,0.148).
atm(d180,d180_31,h,3,0.148).
atm(d180,d180_32,h,3,0.147).
atm(d180,d180_33,h,3,0.147).
atm(d180,d180_34,h,3,0.148).
atm(d180,d180_35,h,3,0.148).
atm(d180,d180_36,h,3,0.148).
atm(d180,d180_37,h,3,0.148).
atm(d180,d180_38,h,3,0.148).

bond(d180,d180_1,d180_2,7).
bond(d180,d180_2,d180_3,7).
bond(d180,d180_3,d180_4,7).
bond(d180,d180_4,d180_5,7).
bond(d180,d180_5,d180_6,7).
bond(d180,d180_6,d180_1,7).
bond(d180,d180_5,d180_7,7).
bond(d180,d180_7,d180_8,7).
bond(d180,d180_8,d180_9,7).
bond(d180,d180_9,d180_10,7).
bond(d180,d180_10,d180_6,7).
bond(d180,d180_7,d180_11,7).
bond(d180,d180_11,d180_12,7).
bond(d180,d180_12,d180_13,7).
bond(d180,d180_13,d180_14,7).
bond(d180,d180_14,d180_8,7).
bond(d180,d180_11,d180_15,7).
bond(d180,d180_15,d180_16,7).
bond(d180,d180_16,d180_17,7).
bond(d180,d180_17,d180_18,7).
bond(d180,d180_18,d180_12,7).
bond(d180,d180_15,d180_19,7).
bond(d180,d180_19,d180_20,7).
bond(d180,d180_20,d180_21,7).
bond(d180,d180_21,d180_22,7).
bond(d180,d180_22,d180_16,7).
bond(d180,d180_4,d180_19,1).
bond(d180,d180_22,d180_23,1).
bond(d180,d180_3,d180_24,1).
bond(d180,d180_23,d180_25,2).
bond(d180,d180_23,d180_26,2).
bond(d180,d180_24,d180_27,2).
bond(d180,d180_24,d180_28,2).
bond(d180,d180_1,d180_29,1).
bond(d180,d180_2,d180_30,1).
bond(d180,d180_9,d180_31,1).
bond(d180,d180_10,d180_32,1).
bond(d180,d180_13,d180_33,1).
bond(d180,d180_14,d180_34,1).
bond(d180,d180_17,d180_35,1).
bond(d180,d180_18,d180_36,1).
bond(d180,d180_20,d180_37,1).
bond(d180,d180_21,d180_38,1).
atm(d181,d181_1,c,22,-0.132).
atm(d181,d181_2,c,22,0.198).
atm(d181,d181_3,c,22,-0.132).
atm(d181,d181_4,c,22,-0.132).
atm(d181,d181_5,c,22,0.198).
atm(d181,d181_6,c,22,-0.132).
atm(d181,d181_7,h,3,0.128).
atm(d181,d181_8,h,3,0.128).
atm(d181,d181_9,h,3,0.128).
atm(d181,d181_10,n,32,-0.781).
atm(d181,d181_11,n,32,-0.781).
atm(d181,d181_12,n,38,0.799).
atm(d181,d181_13,o,40,-0.401).
atm(d181,d181_14,o,40,-0.401).
atm(d181,d181_15,h,1,0.328).
atm(d181,d181_16,h,1,0.328).
atm(d181,d181_17,h,1,0.328).
atm(d181,d181_18,h,1,0.329).

bond(d181,d181_1,d181_2,7).
bond(d181,d181_2,d181_3,7).
bond(d181,d181_3,d181_4,7).
bond(d181,d181_4,d181_5,7).
bond(d181,d181_5,d181_6,7).
bond(d181,d181_6,d181_1,7).
bond(d181,d181_1,d181_7,1).
bond(d181,d181_3,d181_8,1).
bond(d181,d181_6,d181_9,1).
bond(d181,d181_5,d181_10,1).
bond(d181,d181_2,d181_11,1).
bond(d181,d181_4,d181_12,1).
bond(d181,d181_12,d181_13,2).
bond(d181,d181_12,d181_14,2).
bond(d181,d181_10,d181_15,1).
bond(d181,d181_10,d181_16,1).
bond(d181,d181_11,d181_17,1).
bond(d181,d181_11,d181_18,1).
atm(d182,d182_1,c,22,-0.114).
atm(d182,d182_2,c,22,-0.114).
atm(d182,d182_3,c,27,-0.084).
atm(d182,d182_4,c,27,0.016).
atm(d182,d182_5,c,22,-0.114).
atm(d182,d182_6,c,22,-0.114).
atm(d182,d182_7,h,3,0.146).
atm(d182,d182_8,h,3,0.146).
atm(d182,d182_9,h,3,0.146).
atm(d182,d182_10,n,35,-0.755).
atm(d182,d182_11,c,22,0.516).
atm(d182,d182_12,c,22,-0.114).
atm(d182,d182_13,c,22,-0.114).
atm(d182,d182_14,h,3,0.146).
atm(d182,d182_15,h,3,0.146).
atm(d182,d182_16,c,10,0.016).
atm(d182,d182_17,h,3,0.066).
atm(d182,d182_18,h,3,0.066).
atm(d182,d182_19,h,3,0.066).
atm(d182,d182_20,n,38,0.815).
atm(d182,d182_21,o,40,-0.384).
atm(d182,d182_22,o,40,-0.384).

bond(d182,d182_1,d182_2,7).
bond(d182,d182_2,d182_3,7).
bond(d182,d182_3,d182_4,7).
bond(d182,d182_4,d182_5,7).
bond(d182,d182_5,d182_6,7).
bond(d182,d182_6,d182_1,7).
bond(d182,d182_1,d182_7,1).
bond(d182,d182_5,d182_8,1).
bond(d182,d182_6,d182_9,1).
bond(d182,d182_3,d182_10,7).
bond(d182,d182_10,d182_11,7).
bond(d182,d182_11,d182_12,7).
bond(d182,d182_12,d182_13,7).
bond(d182,d182_13,d182_4,7).
bond(d182,d182_12,d182_14,1).
bond(d182,d182_13,d182_15,1).
bond(d182,d182_11,d182_16,1).
bond(d182,d182_16,d182_17,1).
bond(d182,d182_16,d182_18,1).
bond(d182,d182_16,d182_19,1).
bond(d182,d182_2,d182_20,1).
bond(d182,d182_20,d182_21,2).
bond(d182,d182_20,d182_22,2).
atm(d183,d183_1,c,22,-0.115).
atm(d183,d183_2,c,22,-0.115).
atm(d183,d183_3,c,27,0.015).
atm(d183,d183_4,c,27,-0.085).
atm(d183,d183_5,c,27,0.015).
atm(d183,d183_6,c,22,-0.115).
atm(d183,d183_7,h,3,0.145).
atm(d183,d183_8,h,3,0.146).
atm(d183,d183_9,c,22,-0.115).
atm(d183,d183_10,c,22,-0.115).
atm(d183,d183_11,c,27,0.015).
atm(d183,d183_12,c,27,-0.085).
atm(d183,d183_13,h,3,0.146).
atm(d183,d183_14,h,3,0.146).
atm(d183,d183_15,c,27,0.015).
atm(d183,d183_16,c,22,-0.115).
atm(d183,d183_17,c,22,-0.115).
atm(d183,d183_18,h,3,0.145).
atm(d183,d183_19,h,3,0.145).
atm(d183,d183_20,c,22,-0.115).
atm(d183,d183_21,c,22,-0.115).
atm(d183,d183_22,c,22,-0.115).
atm(d183,d183_23,h,3,0.146).
atm(d183,d183_24,h,3,0.145).
atm(d183,d183_25,n,38,0.816).
atm(d183,d183_26,n,38,0.816).
atm(d183,d183_27,o,40,-0.384).
atm(d183,d183_28,o,40,-0.384).
atm(d183,d183_29,o,40,-0.384).
atm(d183,d183_30,o,40,-0.384).

bond(d183,d183_1,d183_2,7).
bond(d183,d183_2,d183_3,7).
bond(d183,d183_3,d183_4,7).
bond(d183,d183_4,d183_5,7).
bond(d183,d183_5,d183_6,7).
bond(d183,d183_6,d183_1,7).
bond(d183,d183_1,d183_7,1).
bond(d183,d183_2,d183_8,1).
bond(d183,d183_3,d183_9,7).
bond(d183,d183_9,d183_10,7).
bond(d183,d183_10,d183_11,7).
bond(d183,d183_11,d183_12,7).
bond(d183,d183_12,d183_4,7).
bond(d183,d183_9,d183_13,1).
bond(d183,d183_10,d183_14,1).
bond(d183,d183_12,d183_15,7).
bond(d183,d183_15,d183_16,7).
bond(d183,d183_16,d183_17,7).
bond(d183,d183_17,d183_5,7).
bond(d183,d183_16,d183_18,1).
bond(d183,d183_17,d183_19,1).
bond(d183,d183_11,d183_20,7).
bond(d183,d183_20,d183_21,7).
bond(d183,d183_21,d183_22,7).
bond(d183,d183_22,d183_15,7).
bond(d183,d183_20,d183_23,1).
bond(d183,d183_21,d183_24,1).
bond(d183,d183_22,d183_25,1).
bond(d183,d183_6,d183_26,1).
bond(d183,d183_26,d183_27,2).
bond(d183,d183_26,d183_28,2).
bond(d183,d183_25,d183_29,2).
bond(d183,d183_25,d183_30,2).
atm(d184,d184_1,c,22,-0.118).
atm(d184,d184_2,c,22,-0.118).
atm(d184,d184_3,c,27,0.012).
atm(d184,d184_4,c,27,-0.088).
atm(d184,d184_5,c,28,0.012).
atm(d184,d184_6,c,22,-0.118).
atm(d184,d184_7,h,3,0.142).
atm(d184,d184_8,h,3,0.142).
atm(d184,d184_9,h,3,0.142).
atm(d184,d184_10,c,22,-0.118).
atm(d184,d184_11,c,22,-0.118).
atm(d184,d184_12,c,22,-0.118).
atm(d184,d184_13,c,29,0.012).
atm(d184,d184_14,h,3,0.142).
atm(d184,d184_15,h,3,0.142).
atm(d184,d184_16,c,28,0.012).
atm(d184,d184_17,c,28,0.012).
atm(d184,d184_18,c,22,-0.118).
atm(d184,d184_19,c,22,-0.118).
atm(d184,d184_20,c,22,-0.118).
atm(d184,d184_21,c,22,-0.118).
atm(d184,d184_22,h,3,0.142).
atm(d184,d184_23,h,3,0.142).
atm(d184,d184_24,h,3,0.142).
atm(d184,d184_25,n,38,0.812).
atm(d184,d184_26,n,38,0.812).
atm(d184,d184_27,o,40,-0.388).
atm(d184,d184_28,o,40,-0.388).
atm(d184,d184_29,o,40,-0.388).
atm(d184,d184_30,o,40,-0.388).

bond(d184,d184_1,d184_2,7).
bond(d184,d184_2,d184_3,7).
bond(d184,d184_3,d184_4,7).
bond(d184,d184_4,d184_5,7).
bond(d184,d184_5,d184_6,7).
bond(d184,d184_6,d184_1,7).
bond(d184,d184_1,d184_7,1).
bond(d184,d184_2,d184_8,1).
bond(d184,d184_6,d184_9,1).
bond(d184,d184_3,d184_10,7).
bond(d184,d184_10,d184_11,7).
bond(d184,d184_11,d184_12,7).
bond(d184,d184_12,d184_13,7).
bond(d184,d184_13,d184_4,7).
bond(d184,d184_11,d184_14,1).
bond(d184,d184_12,d184_15,1).
bond(d184,d184_16,d184_17,7).
bond(d184,d184_17,d184_18,7).
bond(d184,d184_18,d184_19,7).
bond(d184,d184_19,d184_20,7).
bond(d184,d184_20,d184_21,7).
bond(d184,d184_21,d184_16,7).
bond(d184,d184_18,d184_22,1).
bond(d184,d184_19,d184_23,1).
bond(d184,d184_20,d184_24,1).
bond(d184,d184_16,d184_5,1).
bond(d184,d184_17,d184_13,1).
bond(d184,d184_10,d184_25,1).
bond(d184,d184_21,d184_26,1).
bond(d184,d184_26,d184_27,2).
bond(d184,d184_26,d184_28,2).
bond(d184,d184_25,d184_29,2).
bond(d184,d184_25,d184_30,2).
atm(d185,d185_1,c,22,-0.114).
atm(d185,d185_2,c,22,-0.114).
atm(d185,d185_3,c,22,-0.114).
atm(d185,d185_4,c,22,-0.114).
atm(d185,d185_5,c,22,0.316).
atm(d185,d185_6,c,22,-0.114).
atm(d185,d185_7,h,3,0.146).
atm(d185,d185_8,h,3,0.146).
atm(d185,d185_9,h,3,0.146).
atm(d185,d185_10,f,92,-0.284).
atm(d185,d185_11,n,38,0.817).
atm(d185,d185_12,n,38,0.817).
atm(d185,d185_13,o,40,-0.383).
atm(d185,d185_14,o,40,-0.384).
atm(d185,d185_15,o,40,-0.384).
atm(d185,d185_16,o,40,-0.383).

bond(d185,d185_1,d185_2,7).
bond(d185,d185_2,d185_3,7).
bond(d185,d185_3,d185_4,7).
bond(d185,d185_4,d185_5,7).
bond(d185,d185_5,d185_6,7).
bond(d185,d185_6,d185_1,7).
bond(d185,d185_1,d185_7,1).
bond(d185,d185_4,d185_8,1).
bond(d185,d185_6,d185_9,1).
bond(d185,d185_5,d185_10,1).
bond(d185,d185_3,d185_11,1).
bond(d185,d185_2,d185_12,1).
bond(d185,d185_12,d185_13,2).
bond(d185,d185_12,d185_14,2).
bond(d185,d185_11,d185_15,2).
bond(d185,d185_11,d185_16,2).
atm(d186,d186_1,c,22,-0.133).
atm(d186,d186_2,c,22,-0.133).
atm(d186,d186_3,c,22,0.197).
atm(d186,d186_4,c,22,-0.004).
atm(d186,d186_5,c,22,-0.133).
atm(d186,d186_6,c,22,-0.133).
atm(d186,d186_7,h,3,0.126).
atm(d186,d186_8,h,3,0.126).
atm(d186,d186_9,h,3,0.126).
atm(d186,d186_10,n,32,-0.403).
atm(d186,d186_11,c,10,-0.004).
atm(d186,d186_12,c,10,-0.104).
atm(d186,d186_13,h,3,0.046).
atm(d186,d186_14,h,3,0.046).
atm(d186,d186_15,h,3,0.046).
atm(d186,d186_16,h,3,0.046).
atm(d186,d186_17,h,1,0.296).
atm(d186,d186_18,n,38,0.798).
atm(d186,d186_19,o,40,-0.403).
atm(d186,d186_20,o,40,-0.403).

bond(d186,d186_1,d186_2,7).
bond(d186,d186_2,d186_3,7).
bond(d186,d186_3,d186_4,7).
bond(d186,d186_4,d186_5,7).
bond(d186,d186_5,d186_6,7).
bond(d186,d186_6,d186_1,7).
bond(d186,d186_1,d186_7,1).
bond(d186,d186_2,d186_8,1).
bond(d186,d186_5,d186_9,1).
bond(d186,d186_3,d186_10,1).
bond(d186,d186_10,d186_11,1).
bond(d186,d186_4,d186_12,1).
bond(d186,d186_12,d186_11,1).
bond(d186,d186_12,d186_13,1).
bond(d186,d186_12,d186_14,1).
bond(d186,d186_11,d186_15,1).
bond(d186,d186_11,d186_16,1).
bond(d186,d186_10,d186_17,1).
bond(d186,d186_6,d186_18,1).
bond(d186,d186_18,d186_19,2).
bond(d186,d186_18,d186_20,2).
atm(d187,d187_1,c,194,0.004).
atm(d187,d187_2,c,194,0.004).
atm(d187,d187_3,c,29,0.004).
atm(d187,d187_4,c,194,0.004).
atm(d187,d187_5,c,29,0.004).
atm(d187,d187_6,c,29,0.004).
atm(d187,d187_7,c,22,-0.126).
atm(d187,d187_8,c,22,-0.126).
atm(d187,d187_9,c,22,-0.126).
atm(d187,d187_10,c,22,-0.126).
atm(d187,d187_11,h,3,0.134).
atm(d187,d187_12,h,3,0.134).
atm(d187,d187_13,h,3,0.134).
atm(d187,d187_14,c,22,-0.126).
atm(d187,d187_15,c,22,-0.126).
atm(d187,d187_16,c,22,-0.126).
atm(d187,d187_17,c,22,-0.126).
atm(d187,d187_18,h,3,0.134).
atm(d187,d187_19,h,3,0.134).
atm(d187,d187_20,h,3,0.134).
atm(d187,d187_21,h,3,0.134).
atm(d187,d187_22,c,22,-0.126).
atm(d187,d187_23,c,22,-0.126).
atm(d187,d187_24,c,22,-0.126).
atm(d187,d187_25,c,22,-0.126).
atm(d187,d187_26,h,3,0.134).
atm(d187,d187_27,h,3,0.134).
atm(d187,d187_28,h,3,0.134).
atm(d187,d187_29,h,3,0.134).
atm(d187,d187_30,n,38,0.805).
atm(d187,d187_31,o,40,-0.395).
atm(d187,d187_32,o,40,-0.396).

bond(d187,d187_1,d187_2,1).
bond(d187,d187_2,d187_3,7).
bond(d187,d187_3,d187_4,1).
bond(d187,d187_4,d187_5,7).
bond(d187,d187_5,d187_6,1).
bond(d187,d187_6,d187_1,7).
bond(d187,d187_4,d187_7,7).
bond(d187,d187_7,d187_8,7).
bond(d187,d187_8,d187_9,7).
bond(d187,d187_9,d187_10,7).
bond(d187,d187_10,d187_5,7).
bond(d187,d187_7,d187_11,1).
bond(d187,d187_8,d187_12,1).
bond(d187,d187_10,d187_13,1).
bond(d187,d187_2,d187_14,7).
bond(d187,d187_14,d187_15,7).
bond(d187,d187_15,d187_16,7).
bond(d187,d187_16,d187_17,7).
bond(d187,d187_17,d187_3,7).
bond(d187,d187_14,d187_18,1).
bond(d187,d187_15,d187_19,1).
bond(d187,d187_16,d187_20,1).
bond(d187,d187_17,d187_21,1).
bond(d187,d187_6,d187_22,7).
bond(d187,d187_22,d187_23,7).
bond(d187,d187_23,d187_24,7).
bond(d187,d187_24,d187_25,7).
bond(d187,d187_25,d187_1,7).
bond(d187,d187_22,d187_26,1).
bond(d187,d187_23,d187_27,1).
bond(d187,d187_24,d187_28,1).
bond(d187,d187_25,d187_29,1).
bond(d187,d187_9,d187_30,1).
bond(d187,d187_30,d187_31,2).
bond(d187,d187_30,d187_32,2).
atm(d188,d188_1,c,22,-0.137).
atm(d188,d188_2,c,22,-0.137).
atm(d188,d188_3,c,22,0.193).
atm(d188,d188_4,c,29,-0.007).
atm(d188,d188_5,c,22,-0.137).
atm(d188,d188_6,c,22,-0.137).
atm(d188,d188_7,h,3,0.123).
atm(d188,d188_8,h,3,0.123).
atm(d188,d188_9,h,3,0.123).
atm(d188,d188_10,h,3,0.123).
atm(d188,d188_11,c,22,0.193).
atm(d188,d188_12,c,22,-0.137).
atm(d188,d188_13,c,22,-0.137).
atm(d188,d188_14,c,22,-0.137).
atm(d188,d188_15,c,22,-0.137).
atm(d188,d188_16,c,29,-0.007).
atm(d188,d188_17,h,3,0.123).
atm(d188,d188_18,h,3,0.123).
atm(d188,d188_19,h,3,0.123).
atm(d188,d188_20,n,32,-0.408).
atm(d188,d188_21,h,1,0.293).
atm(d188,d188_22,n,38,0.792).
atm(d188,d188_23,o,40,-0.407).
atm(d188,d188_24,o,40,-0.407).

bond(d188,d188_1,d188_2,7).
bond(d188,d188_2,d188_3,7).
bond(d188,d188_3,d188_4,7).
bond(d188,d188_4,d188_5,7).
bond(d188,d188_5,d188_6,7).
bond(d188,d188_6,d188_1,7).
bond(d188,d188_1,d188_7,1).
bond(d188,d188_2,d188_8,1).
bond(d188,d188_5,d188_9,1).
bond(d188,d188_6,d188_10,1).
bond(d188,d188_11,d188_12,7).
bond(d188,d188_12,d188_13,7).
bond(d188,d188_13,d188_14,7).
bond(d188,d188_14,d188_15,7).
bond(d188,d188_15,d188_16,7).
bond(d188,d188_16,d188_11,7).
bond(d188,d188_12,d188_17,1).
bond(d188,d188_13,d188_18,1).
bond(d188,d188_15,d188_19,1).
bond(d188,d188_4,d188_16,1).
bond(d188,d188_3,d188_20,1).
bond(d188,d188_20,d188_11,1).
bond(d188,d188_20,d188_21,1).
bond(d188,d188_14,d188_22,1).
bond(d188,d188_22,d188_23,2).
bond(d188,d188_22,d188_24,2).
atm(d189,d189_1,c,22,-0.11).
atm(d189,d189_2,c,22,-0.11).
atm(d189,d189_3,c,27,0.02).
atm(d189,d189_4,c,27,0.02).
atm(d189,d189_5,c,22,-0.11).
atm(d189,d189_6,c,22,-0.11).
atm(d189,d189_7,h,3,0.15).
atm(d189,d189_8,h,3,0.15).
atm(d189,d189_9,c,22,-0.11).
atm(d189,d189_10,c,22,-0.11).
atm(d189,d189_11,c,22,-0.11).
atm(d189,d189_12,c,22,-0.11).
atm(d189,d189_13,h,3,0.15).
atm(d189,d189_14,h,3,0.15).
atm(d189,d189_15,n,38,0.82).
atm(d189,d189_16,n,38,0.82).
atm(d189,d189_17,n,38,0.82).
atm(d189,d189_18,n,38,0.82).
atm(d189,d189_19,o,40,-0.38).
atm(d189,d189_20,o,40,-0.38).
atm(d189,d189_21,o,40,-0.38).
atm(d189,d189_22,o,40,-0.38).
atm(d189,d189_23,o,40,-0.38).
atm(d189,d189_24,o,40,-0.38).
atm(d189,d189_25,o,40,-0.38).
atm(d189,d189_26,o,40,-0.38).

bond(d189,d189_1,d189_2,7).
bond(d189,d189_2,d189_3,7).
bond(d189,d189_3,d189_4,7).
bond(d189,d189_4,d189_5,7).
bond(d189,d189_5,d189_6,7).
bond(d189,d189_6,d189_1,7).
bond(d189,d189_2,d189_7,1).
bond(d189,d189_6,d189_8,1).
bond(d189,d189_3,d189_9,7).
bond(d189,d189_9,d189_10,7).
bond(d189,d189_10,d189_11,7).
bond(d189,d189_11,d189_12,7).
bond(d189,d189_12,d189_4,7).
bond(d189,d189_9,d189_13,1).
bond(d189,d189_11,d189_14,1).
bond(d189,d189_12,d189_15,1).
bond(d189,d189_10,d189_16,1).
bond(d189,d189_5,d189_17,1).
bond(d189,d189_1,d189_18,1).
bond(d189,d189_17,d189_19,2).
bond(d189,d189_17,d189_20,2).
bond(d189,d189_15,d189_21,2).
bond(d189,d189_15,d189_22,2).
bond(d189,d189_16,d189_23,2).
bond(d189,d189_16,d189_24,2).
bond(d189,d189_18,d189_25,2).
bond(d189,d189_18,d189_26,2).
atm(d19,d19_1,c,22,-0.097).
atm(d19,d19_2,c,22,-0.097).
atm(d19,d19_3,c,27,-0.067).
atm(d19,d19_4,c,27,0.033).
atm(d19,d19_5,c,22,-0.097).
atm(d19,d19_6,c,22,-0.097).
atm(d19,d19_7,h,3,0.162).
atm(d19,d19_8,h,3,0.162).
atm(d19,d19_9,h,3,0.163).
atm(d19,d19_10,n,35,-0.738).
atm(d19,d19_11,c,22,0.282).
atm(d19,d19_12,c,22,-0.097).
atm(d19,d19_13,c,22,-0.097).
atm(d19,d19_14,h,3,0.163).
atm(d19,d19_15,h,3,0.163).
atm(d19,d19_16,h,3,0.163).
atm(d19,d19_17,n,38,0.832).
atm(d19,d19_18,o,40,-0.368).
atm(d19,d19_19,o,40,-0.368).

bond(d19,d19_1,d19_2,7).
bond(d19,d19_2,d19_3,7).
bond(d19,d19_3,d19_4,7).
bond(d19,d19_4,d19_5,7).
bond(d19,d19_5,d19_6,7).
bond(d19,d19_6,d19_1,7).
bond(d19,d19_1,d19_7,1).
bond(d19,d19_2,d19_8,1).
bond(d19,d19_5,d19_9,1).
bond(d19,d19_3,d19_10,7).
bond(d19,d19_10,d19_11,7).
bond(d19,d19_11,d19_12,7).
bond(d19,d19_12,d19_13,7).
bond(d19,d19_13,d19_4,7).
bond(d19,d19_11,d19_14,1).
bond(d19,d19_12,d19_15,1).
bond(d19,d19_13,d19_16,1).
bond(d19,d19_17,d19_6,1).
bond(d19,d19_18,d19_17,2).
bond(d19,d19_19,d19_17,2).
atm(d190,d190_1,c,22,-0.138).
atm(d190,d190_2,c,22,-0.138).
atm(d190,d190_3,c,26,-0.048).
atm(d190,d190_4,c,26,0.092).
atm(d190,d190_5,c,22,-0.138).
atm(d190,d190_6,c,22,-0.138).
atm(d190,d190_7,n,34,-0.367).
atm(d190,d190_8,c,21,0.162).
atm(d190,d190_9,c,21,-0.108).
atm(d190,d190_10,n,38,0.793).
atm(d190,d190_11,o,40,-0.407).
atm(d190,d190_12,o,40,-0.407).
atm(d190,d190_13,h,1,0.292).
atm(d190,d190_14,h,3,0.122).
atm(d190,d190_15,h,3,0.122).
atm(d190,d190_16,h,3,0.092).
atm(d190,d190_17,h,3,0.122).
atm(d190,d190_18,h,3,0.092).

bond(d190,d190_1,d190_2,7).
bond(d190,d190_2,d190_3,7).
bond(d190,d190_3,d190_4,7).
bond(d190,d190_4,d190_5,7).
bond(d190,d190_5,d190_6,7).
bond(d190,d190_6,d190_1,7).
bond(d190,d190_3,d190_7,7).
bond(d190,d190_7,d190_8,7).
bond(d190,d190_4,d190_9,7).
bond(d190,d190_9,d190_8,7).
bond(d190,d190_6,d190_10,1).
bond(d190,d190_10,d190_11,2).
bond(d190,d190_10,d190_12,2).
bond(d190,d190_7,d190_13,1).
bond(d190,d190_1,d190_14,1).
bond(d190,d190_2,d190_15,1).
bond(d190,d190_5,d190_16,1).
bond(d190,d190_8,d190_17,1).
bond(d190,d190_9,d190_18,1).
atm(d191,d191_1,c,22,-0.133).
atm(d191,d191_2,c,22,-0.133).
atm(d191,d191_3,c,27,-0.002).
atm(d191,d191_4,c,27,-0.102).
atm(d191,d191_5,c,27,-0.002).
atm(d191,d191_6,c,22,-0.133).
atm(d191,d191_7,c,27,-0.102).
atm(d191,d191_8,c,27,-0.002).
atm(d191,d191_9,c,22,-0.133).
atm(d191,d191_10,c,22,-0.133).
atm(d191,d191_11,c,26,0.098).
atm(d191,d191_12,c,26,0.098).
atm(d191,d191_13,c,22,-0.133).
atm(d191,d191_14,c,22,-0.133).
atm(d191,d191_15,c,22,-0.132).
atm(d191,d191_16,c,26,0.098).
atm(d191,d191_17,c,21,-0.102).
atm(d191,d191_18,c,21,-0.002).
atm(d191,d191_19,h,3,0.128).
atm(d191,d191_20,h,3,0.128).
atm(d191,d191_21,h,3,0.128).
atm(d191,d191_22,h,3,0.128).
atm(d191,d191_23,h,3,0.128).
atm(d191,d191_24,h,3,0.098).
atm(d191,d191_25,h,3,0.128).
atm(d191,d191_26,h,3,0.128).
atm(d191,d191_27,h,3,0.098).
atm(d191,d191_28,n,38,0.797).
atm(d191,d191_29,o,40,-0.403).
atm(d191,d191_30,o,40,-0.403).

bond(d191,d191_1,d191_2,7).
bond(d191,d191_2,d191_3,7).
bond(d191,d191_3,d191_4,7).
bond(d191,d191_4,d191_5,7).
bond(d191,d191_5,d191_6,7).
bond(d191,d191_6,d191_1,7).
bond(d191,d191_4,d191_7,7).
bond(d191,d191_7,d191_8,7).
bond(d191,d191_8,d191_9,7).
bond(d191,d191_9,d191_10,7).
bond(d191,d191_10,d191_5,7).
bond(d191,d191_7,d191_11,7).
bond(d191,d191_11,d191_12,7).
bond(d191,d191_12,d191_13,7).
bond(d191,d191_13,d191_14,7).
bond(d191,d191_14,d191_8,7).
bond(d191,d191_3,d191_15,7).
bond(d191,d191_15,d191_16,7).
bond(d191,d191_16,d191_11,7).
bond(d191,d191_12,d191_17,7).
bond(d191,d191_16,d191_18,7).
bond(d191,d191_18,d191_17,7).
bond(d191,d191_1,d191_19,1).
bond(d191,d191_2,d191_20,1).
bond(d191,d191_6,d191_21,1).
bond(d191,d191_9,d191_22,1).
bond(d191,d191_10,d191_23,1).
bond(d191,d191_13,d191_24,1).
bond(d191,d191_14,d191_25,1).
bond(d191,d191_15,d191_26,1).
bond(d191,d191_17,d191_27,1).
bond(d191,d191_18,d191_28,1).
bond(d191,d191_28,d191_29,2).
bond(d191,d191_28,d191_30,2).
atm(d192,d192_1,c,22,-0.113).
atm(d192,d192_2,c,22,-0.113).
atm(d192,d192_3,c,27,0.017).
atm(d192,d192_4,c,27,-0.083).
atm(d192,d192_5,c,22,-0.113).
atm(d192,d192_6,c,22,-0.113).
atm(d192,d192_7,h,3,0.146).
atm(d192,d192_8,h,3,0.146).
atm(d192,d192_9,h,3,0.146).
atm(d192,d192_10,h,3,0.146).
atm(d192,d192_11,c,22,-0.113).
atm(d192,d192_12,c,22,-0.113).
atm(d192,d192_13,c,195,-0.083).
atm(d192,d192_14,c,195,-0.083).
atm(d192,d192_15,h,3,0.146).
atm(d192,d192_16,c,27,-0.083).
atm(d192,d192_17,c,27,0.017).
atm(d192,d192_18,c,22,-0.113).
atm(d192,d192_19,c,22,-0.113).
atm(d192,d192_20,h,3,0.146).
atm(d192,d192_21,h,3,0.146).
atm(d192,d192_22,c,22,-0.113).
atm(d192,d192_23,c,22,-0.113).
atm(d192,d192_24,c,22,-0.113).
atm(d192,d192_25,c,22,-0.113).
atm(d192,d192_26,h,3,0.146).
atm(d192,d192_27,h,3,0.146).
atm(d192,d192_28,h,3,0.146).
atm(d192,d192_29,h,3,0.146).
atm(d192,d192_30,n,38,0.816).
atm(d192,d192_31,o,40,-0.384).
atm(d192,d192_32,o,40,-0.384).

bond(d192,d192_1,d192_2,7).
bond(d192,d192_2,d192_3,7).
bond(d192,d192_3,d192_4,7).
bond(d192,d192_4,d192_5,7).
bond(d192,d192_5,d192_6,7).
bond(d192,d192_6,d192_1,7).
bond(d192,d192_1,d192_7,1).
bond(d192,d192_2,d192_8,1).
bond(d192,d192_5,d192_9,1).
bond(d192,d192_6,d192_10,1).
bond(d192,d192_3,d192_11,7).
bond(d192,d192_11,d192_12,7).
bond(d192,d192_12,d192_13,7).
bond(d192,d192_13,d192_14,7).
bond(d192,d192_14,d192_4,7).
bond(d192,d192_11,d192_15,1).
bond(d192,d192_13,d192_16,7).
bond(d192,d192_16,d192_17,7).
bond(d192,d192_17,d192_18,7).
bond(d192,d192_18,d192_19,7).
bond(d192,d192_19,d192_14,7).
bond(d192,d192_18,d192_20,1).
bond(d192,d192_19,d192_21,1).
bond(d192,d192_16,d192_22,7).
bond(d192,d192_22,d192_23,7).
bond(d192,d192_23,d192_24,7).
bond(d192,d192_24,d192_25,7).
bond(d192,d192_25,d192_17,7).
bond(d192,d192_22,d192_26,1).
bond(d192,d192_23,d192_27,1).
bond(d192,d192_24,d192_28,1).
bond(d192,d192_25,d192_29,1).
bond(d192,d192_12,d192_30,1).
bond(d192,d192_30,d192_31,2).
bond(d192,d192_30,d192_32,2).
atm(d193,d193_1,c,22,-0.113).
atm(d193,d193_2,c,22,-0.113).
atm(d193,d193_3,c,27,0.017).
atm(d193,d193_4,c,27,-0.083).
atm(d193,d193_5,c,22,-0.113).
atm(d193,d193_6,c,22,-0.113).
atm(d193,d193_7,h,3,0.146).
atm(d193,d193_8,h,3,0.146).
atm(d193,d193_9,h,3,0.146).
atm(d193,d193_10,h,3,0.146).
atm(d193,d193_11,c,22,-0.113).
atm(d193,d193_12,c,22,-0.113).
atm(d193,d193_13,c,195,-0.083).
atm(d193,d193_14,c,195,-0.083).
atm(d193,d193_15,h,3,0.146).
atm(d193,d193_16,c,27,-0.083).
atm(d193,d193_17,c,27,0.017).
atm(d193,d193_18,c,22,-0.113).
atm(d193,d193_19,c,22,-0.113).
atm(d193,d193_20,h,3,0.146).
atm(d193,d193_21,h,3,0.146).
atm(d193,d193_22,c,22,-0.113).
atm(d193,d193_23,c,22,-0.113).
atm(d193,d193_24,c,22,-0.113).
atm(d193,d193_25,c,22,-0.113).
atm(d193,d193_26,h,3,0.146).
atm(d193,d193_27,h,3,0.146).
atm(d193,d193_28,h,3,0.146).
atm(d193,d193_29,h,3,0.146).
atm(d193,d193_30,n,38,0.816).
atm(d193,d193_31,o,40,-0.384).
atm(d193,d193_32,o,40,-0.384).

bond(d193,d193_1,d193_2,7).
bond(d193,d193_2,d193_3,7).
bond(d193,d193_3,d193_4,7).
bond(d193,d193_4,d193_5,7).
bond(d193,d193_5,d193_6,7).
bond(d193,d193_6,d193_1,7).
bond(d193,d193_1,d193_7,1).
bond(d193,d193_2,d193_8,1).
bond(d193,d193_5,d193_9,1).
bond(d193,d193_6,d193_10,1).
bond(d193,d193_3,d193_11,7).
bond(d193,d193_11,d193_12,7).
bond(d193,d193_12,d193_13,7).
bond(d193,d193_13,d193_14,7).
bond(d193,d193_14,d193_4,7).
bond(d193,d193_11,d193_15,1).
bond(d193,d193_13,d193_16,7).
bond(d193,d193_16,d193_17,7).
bond(d193,d193_17,d193_18,7).
bond(d193,d193_18,d193_19,7).
bond(d193,d193_19,d193_14,7).
bond(d193,d193_18,d193_20,1).
bond(d193,d193_19,d193_21,1).
bond(d193,d193_16,d193_22,7).
bond(d193,d193_22,d193_23,7).
bond(d193,d193_23,d193_24,7).
bond(d193,d193_24,d193_25,7).
bond(d193,d193_25,d193_17,7).
bond(d193,d193_22,d193_26,1).
bond(d193,d193_23,d193_27,1).
bond(d193,d193_25,d193_28,1).
bond(d193,d193_12,d193_29,1).
bond(d193,d193_24,d193_30,1).
bond(d193,d193_30,d193_31,2).
bond(d193,d193_30,d193_32,2).
atm(d194,d194_1,c,22,-0.132).
atm(d194,d194_2,c,22,-0.132).
atm(d194,d194_3,c,22,-0.132).
atm(d194,d194_4,c,22,0.198).
atm(d194,d194_5,c,22,0.198).
atm(d194,d194_6,c,22,-0.132).
atm(d194,d194_7,h,3,0.128).
atm(d194,d194_8,h,3,0.128).
atm(d194,d194_9,h,3,0.128).
atm(d194,d194_10,n,32,-0.781).
atm(d194,d194_11,n,38,0.799).
atm(d194,d194_12,h,1,0.328).
atm(d194,d194_13,h,1,0.328).
atm(d194,d194_14,o,40,-0.401).
atm(d194,d194_15,o,40,-0.401).
atm(d194,d194_16,n,32,-0.781).
atm(d194,d194_17,h,1,0.329).
atm(d194,d194_18,h,1,0.328).

bond(d194,d194_1,d194_2,7).
bond(d194,d194_2,d194_3,7).
bond(d194,d194_3,d194_4,7).
bond(d194,d194_4,d194_5,7).
bond(d194,d194_5,d194_6,7).
bond(d194,d194_6,d194_1,7).
bond(d194,d194_1,d194_7,1).
bond(d194,d194_3,d194_8,1).
bond(d194,d194_6,d194_9,1).
bond(d194,d194_5,d194_10,1).
bond(d194,d194_2,d194_11,1).
bond(d194,d194_10,d194_12,1).
bond(d194,d194_10,d194_13,1).
bond(d194,d194_11,d194_14,2).
bond(d194,d194_11,d194_15,2).
bond(d194,d194_4,d194_16,1).
bond(d194,d194_16,d194_17,1).
bond(d194,d194_16,d194_18,1).
atm(d195,d195_1,c,22,-0.05).
atm(d195,d195_2,c,22,-0.05).
atm(d195,d195_3,c,22,-0.05).
atm(d195,d195_4,c,22,-0.05).
atm(d195,d195_5,c,22,-0.05).
atm(d195,d195_6,c,22,-0.05).
atm(d195,d195_7,h,3,0.21).
atm(d195,d195_8,h,3,0.21).
atm(d195,d195_9,cl,93,-0.12).
atm(d195,d195_10,cl,93,-0.12).
atm(d195,d195_11,cl,93,-0.12).
atm(d195,d195_12,n,38,0.88).
atm(d195,d195_13,o,40,-0.32).
atm(d195,d195_14,o,40,-0.32).

bond(d195,d195_1,d195_2,7).
bond(d195,d195_2,d195_3,7).
bond(d195,d195_3,d195_4,7).
bond(d195,d195_4,d195_5,7).
bond(d195,d195_5,d195_6,7).
bond(d195,d195_6,d195_1,7).
bond(d195,d195_1,d195_7,1).
bond(d195,d195_6,d195_8,1).
bond(d195,d195_5,d195_9,1).
bond(d195,d195_4,d195_10,1).
bond(d195,d195_3,d195_11,1).
bond(d195,d195_2,d195_12,1).
bond(d195,d195_12,d195_13,2).
bond(d195,d195_12,d195_14,2).
atm(d196,d196_1,c,22,-0.137).
atm(d196,d196_2,c,22,-0.137).
atm(d196,d196_3,c,22,0.193).
atm(d196,d196_4,c,29,-0.007).
atm(d196,d196_5,c,22,-0.137).
atm(d196,d196_6,c,22,-0.137).
atm(d196,d196_7,h,3,0.123).
atm(d196,d196_8,h,3,0.123).
atm(d196,d196_9,h,3,0.123).
atm(d196,d196_10,h,3,0.123).
atm(d196,d196_11,c,22,0.193).
atm(d196,d196_12,c,22,-0.137).
atm(d196,d196_13,c,22,-0.137).
atm(d196,d196_14,c,22,-0.137).
atm(d196,d196_15,c,22,-0.137).
atm(d196,d196_16,c,29,-0.007).
atm(d196,d196_17,h,3,0.123).
atm(d196,d196_18,h,3,0.123).
atm(d196,d196_19,n,32,-0.408).
atm(d196,d196_20,h,1,0.293).
atm(d196,d196_21,h,3,0.123).
atm(d196,d196_22,n,38,0.792).
atm(d196,d196_23,o,40,-0.407).
atm(d196,d196_24,o,40,-0.407).

bond(d196,d196_1,d196_2,7).
bond(d196,d196_2,d196_3,7).
bond(d196,d196_3,d196_4,7).
bond(d196,d196_4,d196_5,7).
bond(d196,d196_5,d196_6,7).
bond(d196,d196_6,d196_1,7).
bond(d196,d196_1,d196_7,1).
bond(d196,d196_2,d196_8,1).
bond(d196,d196_5,d196_9,1).
bond(d196,d196_6,d196_10,1).
bond(d196,d196_11,d196_12,7).
bond(d196,d196_12,d196_13,7).
bond(d196,d196_13,d196_14,7).
bond(d196,d196_14,d196_15,7).
bond(d196,d196_15,d196_16,7).
bond(d196,d196_16,d196_11,7).
bond(d196,d196_12,d196_17,1).
bond(d196,d196_13,d196_18,1).
bond(d196,d196_4,d196_16,1).
bond(d196,d196_3,d196_19,1).
bond(d196,d196_19,d196_11,1).
bond(d196,d196_19,d196_20,1).
bond(d196,d196_14,d196_21,1).
bond(d196,d196_15,d196_22,1).
bond(d196,d196_22,d196_23,2).
bond(d196,d196_22,d196_24,2).
atm(d197,d197_1,c,22,-0.086).
atm(d197,d197_2,c,22,-0.086).
atm(d197,d197_3,c,26,-0.056).
atm(d197,d197_4,c,26,-0.056).
atm(d197,d197_5,c,22,-0.086).
atm(d197,d197_6,c,22,-0.086).
atm(d197,d197_7,n,34,-0.496).
atm(d197,d197_8,c,21,0.044).
atm(d197,d197_9,n,34,-0.496).
atm(d197,d197_10,c,10,0.294).
atm(d197,d197_11,n,38,0.844).
atm(d197,d197_12,o,40,-0.356).
atm(d197,d197_13,o,40,-0.356).
atm(d197,d197_14,h,3,0.174).
atm(d197,d197_15,h,3,0.174).
atm(d197,d197_16,h,3,0.174).
atm(d197,d197_17,h,3,0.174).
atm(d197,d197_18,h,3,0.094).
atm(d197,d197_19,h,3,0.094).
atm(d197,d197_20,h,3,0.094).

bond(d197,d197_1,d197_2,7).
bond(d197,d197_2,d197_3,7).
bond(d197,d197_3,d197_4,7).
bond(d197,d197_4,d197_5,7).
bond(d197,d197_5,d197_6,7).
bond(d197,d197_6,d197_1,7).
bond(d197,d197_3,d197_7,7).
bond(d197,d197_7,d197_8,7).
bond(d197,d197_4,d197_9,7).
bond(d197,d197_9,d197_8,7).
bond(d197,d197_7,d197_10,1).
bond(d197,d197_8,d197_11,1).
bond(d197,d197_11,d197_12,2).
bond(d197,d197_11,d197_13,2).
bond(d197,d197_1,d197_14,1).
bond(d197,d197_2,d197_15,1).
bond(d197,d197_5,d197_16,1).
bond(d197,d197_6,d197_17,1).
bond(d197,d197_10,d197_18,1).
bond(d197,d197_10,d197_19,1).
bond(d197,d197_10,d197_20,1).
atm(d2,d2_1,c,22,-0.097).
atm(d2,d2_2,c,22,-0.097).
atm(d2,d2_3,c,27,-0.067).
atm(d2,d2_4,c,27,0.033).
atm(d2,d2_5,c,22,-0.097).
atm(d2,d2_6,c,22,-0.097).
atm(d2,d2_7,h,3,0.162).
atm(d2,d2_8,h,3,0.162).
atm(d2,d2_9,h,3,0.163).
atm(d2,d2_10,n,35,-0.738).
atm(d2,d2_11,c,22,0.282).
atm(d2,d2_12,c,22,-0.097).
atm(d2,d2_13,c,22,-0.097).
atm(d2,d2_14,h,3,0.163).
atm(d2,d2_15,h,3,0.163).
atm(d2,d2_16,h,3,0.163).
atm(d2,d2_17,n,38,0.832).
atm(d2,d2_18,o,40,-0.368).
atm(d2,d2_19,o,40,-0.368).

bond(d2,d2_1,d2_2,7).
bond(d2,d2_2,d2_3,7).
bond(d2,d2_3,d2_4,7).
bond(d2,d2_4,d2_5,7).
bond(d2,d2_5,d2_6,7).
bond(d2,d2_6,d2_1,7).
bond(d2,d2_1,d2_7,1).
bond(d2,d2_5,d2_8,1).
bond(d2,d2_6,d2_9,1).
bond(d2,d2_3,d2_10,7).
bond(d2,d2_10,d2_11,7).
bond(d2,d2_11,d2_12,7).
bond(d2,d2_12,d2_13,7).
bond(d2,d2_13,d2_4,7).
bond(d2,d2_11,d2_14,1).
bond(d2,d2_12,d2_15,1).
bond(d2,d2_13,d2_16,1).
bond(d2,d2_17,d2_2,1).
bond(d2,d2_17,d2_18,2).
bond(d2,d2_17,d2_19,2).
atm(d20,d20_1,c,22,-0.111).
atm(d20,d20_2,c,22,-0.111).
atm(d20,d20_3,c,22,-0.111).
atm(d20,d20_4,c,29,0.019).
atm(d20,d20_5,c,22,-0.111).
atm(d20,d20_6,c,22,-0.111).
atm(d20,d20_7,c,22,-0.111).
atm(d20,d20_8,c,22,-0.111).
atm(d20,d20_9,c,22,-0.111).
atm(d20,d20_10,c,22,-0.111).
atm(d20,d20_11,c,22,-0.111).
atm(d20,d20_12,c,29,0.019).
atm(d20,d20_13,c,10,0.019).
atm(d20,d20_14,n,38,0.818).
atm(d20,d20_15,o,40,-0.381).
atm(d20,d20_16,o,40,-0.382).
atm(d20,d20_17,h,3,0.149).
atm(d20,d20_18,h,3,0.149).
atm(d20,d20_19,h,3,0.149).
atm(d20,d20_20,h,3,0.149).
atm(d20,d20_21,h,3,0.149).
atm(d20,d20_22,h,3,0.149).
atm(d20,d20_23,h,3,0.069).
atm(d20,d20_24,h,3,0.069).
atm(d20,d20_25,c,19,0.448).
atm(d20,d20_26,n,31,-0.482).

bond(d20,d20_1,d20_2,7).
bond(d20,d20_2,d20_3,7).
bond(d20,d20_3,d20_4,7).
bond(d20,d20_4,d20_5,7).
bond(d20,d20_5,d20_6,7).
bond(d20,d20_6,d20_1,7).
bond(d20,d20_7,d20_8,7).
bond(d20,d20_8,d20_9,7).
bond(d20,d20_9,d20_10,7).
bond(d20,d20_10,d20_11,7).
bond(d20,d20_11,d20_12,7).
bond(d20,d20_12,d20_7,7).
bond(d20,d20_4,d20_12,1).
bond(d20,d20_3,d20_13,1).
bond(d20,d20_13,d20_7,1).
bond(d20,d20_1,d20_14,1).
bond(d20,d20_14,d20_15,2).
bond(d20,d20_14,d20_16,2).
bond(d20,d20_2,d20_17,1).
bond(d20,d20_5,d20_18,1).
bond(d20,d20_6,d20_19,1).
bond(d20,d20_8,d20_20,1).
bond(d20,d20_10,d20_21,1).
bond(d20,d20_11,d20_22,1).
bond(d20,d20_13,d20_23,1).
bond(d20,d20_13,d20_24,1).
bond(d20,d20_9,d20_25,1).
bond(d20,d20_25,d20_26,3).
atm(d21,d21_1,c,22,-0.12).
atm(d21,d21_2,c,22,-0.12).
atm(d21,d21_3,c,22,-0.12).
atm(d21,d21_4,c,29,0.01).
atm(d21,d21_5,c,22,-0.12).
atm(d21,d21_6,c,22,-0.12).
atm(d21,d21_7,h,3,0.14).
atm(d21,d21_8,h,3,0.14).
atm(d21,d21_9,h,3,0.14).
atm(d21,d21_10,h,3,0.14).
atm(d21,d21_11,c,29,0.01).
atm(d21,d21_12,c,22,-0.12).
atm(d21,d21_13,c,22,-0.12).
atm(d21,d21_14,c,22,-0.12).
atm(d21,d21_15,c,22,-0.12).
atm(d21,d21_16,c,22,-0.12).
atm(d21,d21_17,h,3,0.14).
atm(d21,d21_18,h,3,0.14).
atm(d21,d21_19,h,3,0.14).
atm(d21,d21_20,h,3,0.14).
atm(d21,d21_21,n,38,0.81).
atm(d21,d21_22,n,38,0.81).
atm(d21,d21_23,o,40,-0.39).
atm(d21,d21_24,o,40,-0.39).
atm(d21,d21_25,o,40,-0.39).
atm(d21,d21_26,o,40,-0.39).

bond(d21,d21_1,d21_2,7).
bond(d21,d21_2,d21_3,7).
bond(d21,d21_3,d21_4,7).
bond(d21,d21_4,d21_5,7).
bond(d21,d21_5,d21_6,7).
bond(d21,d21_6,d21_1,7).
bond(d21,d21_1,d21_7,1).
bond(d21,d21_2,d21_8,1).
bond(d21,d21_3,d21_9,1).
bond(d21,d21_5,d21_10,1).
bond(d21,d21_11,d21_12,7).
bond(d21,d21_12,d21_13,7).
bond(d21,d21_13,d21_14,7).
bond(d21,d21_14,d21_15,7).
bond(d21,d21_15,d21_16,7).
bond(d21,d21_16,d21_11,7).
bond(d21,d21_12,d21_17,1).
bond(d21,d21_13,d21_18,1).
bond(d21,d21_15,d21_19,1).
bond(d21,d21_16,d21_20,1).
bond(d21,d21_4,d21_11,1).
bond(d21,d21_21,d21_14,1).
bond(d21,d21_22,d21_6,1).
bond(d21,d21_23,d21_21,2).
bond(d21,d21_24,d21_21,2).
bond(d21,d21_25,d21_22,2).
bond(d21,d21_22,d21_26,2).
atm(d22,d22_1,c,22,-0.111).
atm(d22,d22_2,c,22,-0.111).
atm(d22,d22_3,c,22,-0.111).
atm(d22,d22_4,c,29,0.019).
atm(d22,d22_5,c,22,-0.111).
atm(d22,d22_6,c,22,-0.111).
atm(d22,d22_7,c,22,-0.111).
atm(d22,d22_8,c,22,-0.111).
atm(d22,d22_9,c,22,-0.111).
atm(d22,d22_10,c,22,-0.111).
atm(d22,d22_11,c,22,-0.111).
atm(d22,d22_12,c,29,0.019).
atm(d22,d22_13,c,10,0.019).
atm(d22,d22_14,i,95,-0.031).
atm(d22,d22_15,n,38,0.818).
atm(d22,d22_16,o,40,-0.382).
atm(d22,d22_17,o,40,-0.382).
atm(d22,d22_18,h,3,0.148).
atm(d22,d22_19,h,3,0.149).
atm(d22,d22_20,h,3,0.149).
atm(d22,d22_21,h,3,0.149).
atm(d22,d22_22,h,3,0.148).
atm(d22,d22_23,h,3,0.149).
atm(d22,d22_24,h,3,0.069).
atm(d22,d22_25,h,3,0.069).

bond(d22,d22_1,d22_2,7).
bond(d22,d22_2,d22_3,7).
bond(d22,d22_3,d22_4,7).
bond(d22,d22_4,d22_5,7).
bond(d22,d22_5,d22_6,7).
bond(d22,d22_6,d22_1,7).
bond(d22,d22_7,d22_8,7).
bond(d22,d22_8,d22_9,7).
bond(d22,d22_9,d22_10,7).
bond(d22,d22_10,d22_11,7).
bond(d22,d22_11,d22_12,7).
bond(d22,d22_12,d22_7,7).
bond(d22,d22_4,d22_12,1).
bond(d22,d22_3,d22_13,1).
bond(d22,d22_13,d22_7,1).
bond(d22,d22_9,d22_14,1).
bond(d22,d22_1,d22_15,1).
bond(d22,d22_15,d22_16,2).
bond(d22,d22_15,d22_17,2).
bond(d22,d22_2,d22_18,1).
bond(d22,d22_5,d22_19,1).
bond(d22,d22_6,d22_20,1).
bond(d22,d22_8,d22_21,1).
bond(d22,d22_10,d22_22,1).
bond(d22,d22_11,d22_23,1).
bond(d22,d22_13,d22_24,1).
bond(d22,d22_13,d22_25,1).
atm(d23,d23_1,c,22,-0.115).
atm(d23,d23_2,c,22,-0.115).
atm(d23,d23_3,c,22,-0.115).
atm(d23,d23_4,c,27,-0.085).
atm(d23,d23_5,c,27,0.015).
atm(d23,d23_6,c,22,-0.115).
atm(d23,d23_7,c,195,0.015).
atm(d23,d23_8,c,27,-0.085).
atm(d23,d23_9,c,22,-0.115).
atm(d23,d23_10,c,22,-0.115).
atm(d23,d23_11,c,22,-0.115).
atm(d23,d23_12,c,27,0.015).
atm(d23,d23_13,c,27,-0.085).
atm(d23,d23_14,c,25,-0.085).
atm(d23,d23_15,c,22,-0.115).
atm(d23,d23_16,c,22,-0.115).
atm(d23,d23_17,c,22,-0.115).
atm(d23,d23_18,c,26,-0.085).
atm(d23,d23_19,c,21,-0.115).
atm(d23,d23_20,c,21,0.015).
atm(d23,d23_21,h,3,0.144).
atm(d23,d23_22,h,3,0.144).
atm(d23,d23_23,h,3,0.144).
atm(d23,d23_24,h,3,0.145).
atm(d23,d23_25,h,3,0.145).
atm(d23,d23_26,h,3,0.145).
atm(d23,d23_27,h,3,0.145).
atm(d23,d23_28,h,3,0.144).
atm(d23,d23_29,h,3,0.144).
atm(d23,d23_30,h,3,0.144).
atm(d23,d23_31,h,3,0.144).
atm(d23,d23_32,n,38,0.814).
atm(d23,d23_33,o,40,-0.386).
atm(d23,d23_34,o,40,-0.386).

bond(d23,d23_1,d23_2,7).
bond(d23,d23_2,d23_3,7).
bond(d23,d23_3,d23_4,7).
bond(d23,d23_4,d23_5,7).
bond(d23,d23_5,d23_6,7).
bond(d23,d23_6,d23_1,7).
bond(d23,d23_4,d23_7,7).
bond(d23,d23_7,d23_8,7).
bond(d23,d23_8,d23_9,7).
bond(d23,d23_9,d23_10,7).
bond(d23,d23_10,d23_5,7).
bond(d23,d23_7,d23_11,7).
bond(d23,d23_11,d23_12,7).
bond(d23,d23_12,d23_13,7).
bond(d23,d23_13,d23_14,7).
bond(d23,d23_14,d23_8,7).
bond(d23,d23_12,d23_15,7).
bond(d23,d23_15,d23_16,7).
bond(d23,d23_16,d23_17,7).
bond(d23,d23_17,d23_18,7).
bond(d23,d23_18,d23_13,7).
bond(d23,d23_19,d23_14,7).
bond(d23,d23_19,d23_20,7).
bond(d23,d23_20,d23_18,7).
bond(d23,d23_1,d23_21,1).
bond(d23,d23_2,d23_22,1).
bond(d23,d23_3,d23_23,1).
bond(d23,d23_6,d23_24,1).
bond(d23,d23_9,d23_25,1).
bond(d23,d23_10,d23_26,1).
bond(d23,d23_11,d23_27,1).
bond(d23,d23_15,d23_28,1).
bond(d23,d23_16,d23_29,1).
bond(d23,d23_17,d23_30,1).
bond(d23,d23_19,d23_31,1).
bond(d23,d23_32,d23_20,1).
bond(d23,d23_33,d23_32,2).
bond(d23,d23_32,d23_34,2).
atm(d24,d24_1,c,22,-0.111).
atm(d24,d24_2,c,22,-0.111).
atm(d24,d24_3,c,27,0.019).
atm(d24,d24_4,c,27,-0.081).
atm(d24,d24_5,c,27,-0.081).
atm(d24,d24_6,c,27,0.019).
atm(d24,d24_7,h,3,0.15).
atm(d24,d24_8,h,3,0.15).
atm(d24,d24_9,c,22,-0.111).
atm(d24,d24_10,c,22,-0.111).
atm(d24,d24_11,c,27,0.019).
atm(d24,d24_12,c,27,-0.081).
atm(d24,d24_13,h,3,0.149).
atm(d24,d24_14,h,3,0.149).
atm(d24,d24_15,c,22,-0.111).
atm(d24,d24_16,c,22,-0.111).
atm(d24,d24_17,c,27,0.019).
atm(d24,d24_18,c,27,-0.081).
atm(d24,d24_19,h,3,0.149).
atm(d24,d24_20,h,3,0.149).
atm(d24,d24_21,c,22,-0.111).
atm(d24,d24_22,c,22,-0.111).
atm(d24,d24_23,c,27,0.019).
atm(d24,d24_24,c,27,-0.081).
atm(d24,d24_25,h,3,0.149).
atm(d24,d24_26,c,22,-0.111).
atm(d24,d24_27,c,22,-0.111).
atm(d24,d24_28,c,27,0.019).
atm(d24,d24_29,c,27,-0.081).
atm(d24,d24_30,h,3,0.15).
atm(d24,d24_31,h,3,0.15).
atm(d24,d24_32,c,22,-0.111).
atm(d24,d24_33,c,22,-0.111).
atm(d24,d24_34,h,3,0.15).
atm(d24,d24_35,h,3,0.149).
atm(d24,d24_36,n,38,0.82).
atm(d24,d24_37,o,40,-0.38).
atm(d24,d24_38,o,40,-0.38).

bond(d24,d24_1,d24_2,7).
bond(d24,d24_2,d24_3,7).
bond(d24,d24_3,d24_4,7).
bond(d24,d24_4,d24_5,7).
bond(d24,d24_5,d24_6,7).
bond(d24,d24_6,d24_1,7).
bond(d24,d24_1,d24_7,1).
bond(d24,d24_2,d24_8,1).
bond(d24,d24_3,d24_9,7).
bond(d24,d24_9,d24_10,7).
bond(d24,d24_10,d24_11,7).
bond(d24,d24_11,d24_12,7).
bond(d24,d24_12,d24_4,7).
bond(d24,d24_9,d24_13,1).
bond(d24,d24_10,d24_14,1).
bond(d24,d24_11,d24_15,7).
bond(d24,d24_15,d24_16,7).
bond(d24,d24_16,d24_17,7).
bond(d24,d24_17,d24_18,7).
bond(d24,d24_18,d24_12,7).
bond(d24,d24_15,d24_19,1).
bond(d24,d24_16,d24_20,1).
bond(d24,d24_17,d24_21,7).
bond(d24,d24_21,d24_22,7).
bond(d24,d24_22,d24_23,7).
bond(d24,d24_23,d24_24,7).
bond(d24,d24_24,d24_18,7).
bond(d24,d24_21,d24_25,1).
bond(d24,d24_23,d24_26,7).
bond(d24,d24_26,d24_27,7).
bond(d24,d24_27,d24_28,7).
bond(d24,d24_28,d24_29,7).
bond(d24,d24_29,d24_24,7).
bond(d24,d24_26,d24_30,1).
bond(d24,d24_27,d24_31,1).
bond(d24,d24_5,d24_29,7).
bond(d24,d24_28,d24_32,7).
bond(d24,d24_32,d24_33,7).
bond(d24,d24_33,d24_6,7).
bond(d24,d24_32,d24_34,1).
bond(d24,d24_33,d24_35,1).
bond(d24,d24_36,d24_22,1).
bond(d24,d24_36,d24_37,2).
bond(d24,d24_36,d24_38,2).
atm(d25,d25_1,c,22,-0.119).
atm(d25,d25_2,c,22,-0.119).
atm(d25,d25_3,c,22,-0.119).
atm(d25,d25_4,c,29,0.011).
atm(d25,d25_5,c,22,-0.119).
atm(d25,d25_6,c,22,-0.119).
atm(d25,d25_7,c,22,-0.119).
atm(d25,d25_8,c,22,-0.119).
atm(d25,d25_9,c,22,0.261).
atm(d25,d25_10,c,22,-0.119).
atm(d25,d25_11,c,22,-0.119).
atm(d25,d25_12,c,29,0.011).
atm(d25,d25_13,c,10,0.011).
atm(d25,d25_14,o,45,-0.638).
atm(d25,d25_15,n,38,0.812).
atm(d25,d25_16,o,40,-0.388).
atm(d25,d25_17,o,40,-0.389).
atm(d25,d25_18,h,3,0.141).
atm(d25,d25_19,h,3,0.141).
atm(d25,d25_20,h,3,0.141).
atm(d25,d25_21,h,3,0.141).
atm(d25,d25_22,h,3,0.141).
atm(d25,d25_23,h,3,0.141).
atm(d25,d25_24,h,8,0.412).
atm(d25,d25_25,h,3,0.061).
atm(d25,d25_26,h,3,0.061).

bond(d25,d25_1,d25_2,7).
bond(d25,d25_2,d25_3,7).
bond(d25,d25_3,d25_4,7).
bond(d25,d25_4,d25_5,7).
bond(d25,d25_5,d25_6,7).
bond(d25,d25_6,d25_1,7).
bond(d25,d25_7,d25_8,7).
bond(d25,d25_8,d25_9,7).
bond(d25,d25_9,d25_10,7).
bond(d25,d25_10,d25_11,7).
bond(d25,d25_11,d25_12,7).
bond(d25,d25_12,d25_7,7).
bond(d25,d25_4,d25_12,1).
bond(d25,d25_3,d25_13,1).
bond(d25,d25_13,d25_7,1).
bond(d25,d25_9,d25_14,1).
bond(d25,d25_1,d25_15,1).
bond(d25,d25_15,d25_16,2).
bond(d25,d25_15,d25_17,2).
bond(d25,d25_2,d25_18,1).
bond(d25,d25_5,d25_19,1).
bond(d25,d25_6,d25_20,1).
bond(d25,d25_8,d25_21,1).
bond(d25,d25_10,d25_22,1).
bond(d25,d25_11,d25_23,1).
bond(d25,d25_14,d25_24,1).
bond(d25,d25_13,d25_25,1).
bond(d25,d25_13,d25_26,1).
atm(d26,d26_1,c,22,-0.093).
atm(d26,d26_2,c,22,-0.093).
atm(d26,d26_3,c,22,-0.093).
atm(d26,d26_4,c,22,-0.093).
atm(d26,d26_5,c,22,-0.093).
atm(d26,d26_6,c,22,-0.093).
atm(d26,d26_7,h,3,0.167).
atm(d26,d26_8,h,3,0.167).
atm(d26,d26_9,h,3,0.167).
atm(d26,d26_10,cl,93,-0.163).
atm(d26,d26_11,n,38,0.836).
atm(d26,d26_12,n,38,0.836).
atm(d26,d26_13,o,40,-0.363).
atm(d26,d26_14,o,40,-0.363).
atm(d26,d26_15,o,40,-0.363).
atm(d26,d26_16,o,40,-0.363).

bond(d26,d26_1,d26_2,7).
bond(d26,d26_2,d26_3,7).
bond(d26,d26_3,d26_4,7).
bond(d26,d26_4,d26_5,7).
bond(d26,d26_5,d26_6,7).
bond(d26,d26_6,d26_1,7).
bond(d26,d26_1,d26_7,1).
bond(d26,d26_3,d26_8,1).
bond(d26,d26_6,d26_9,1).
bond(d26,d26_10,d26_5,1).
bond(d26,d26_4,d26_11,1).
bond(d26,d26_2,d26_12,1).
bond(d26,d26_13,d26_11,2).
bond(d26,d26_11,d26_14,2).
bond(d26,d26_15,d26_12,2).
bond(d26,d26_12,d26_16,2).
atm(d27,d27_1,c,22,-0.114).
atm(d27,d27_2,c,22,-0.114).
atm(d27,d27_3,c,27,0.016).
atm(d27,d27_4,c,27,-0.084).
atm(d27,d27_5,c,22,-0.114).
atm(d27,d27_6,c,22,-0.114).
atm(d27,d27_7,h,3,0.145).
atm(d27,d27_8,h,3,0.145).
atm(d27,d27_9,h,3,0.145).
atm(d27,d27_10,h,3,0.145).
atm(d27,d27_11,c,22,-0.114).
atm(d27,d27_12,c,27,0.016).
atm(d27,d27_13,c,195,-0.084).
atm(d27,d27_14,c,195,-0.084).
atm(d27,d27_15,h,3,0.145).
atm(d27,d27_16,c,22,-0.114).
atm(d27,d27_17,c,22,-0.114).
atm(d27,d27_18,c,27,0.016).
atm(d27,d27_19,c,27,-0.084).
atm(d27,d27_20,h,3,0.145).
atm(d27,d27_21,h,3,0.145).
atm(d27,d27_22,c,27,0.016).
atm(d27,d27_23,c,22,-0.114).
atm(d27,d27_24,c,22,-0.114).
atm(d27,d27_25,h,3,0.145).
atm(d27,d27_26,h,3,0.145).
atm(d27,d27_27,c,22,-0.114).
atm(d27,d27_28,c,22,-0.114).
atm(d27,d27_29,c,22,-0.114).
atm(d27,d27_30,h,3,0.145).
atm(d27,d27_31,h,3,0.145).
atm(d27,d27_32,n,38,0.815).
atm(d27,d27_33,o,40,-0.385).
atm(d27,d27_34,o,40,-0.385).

bond(d27,d27_1,d27_2,7).
bond(d27,d27_2,d27_3,7).
bond(d27,d27_3,d27_4,7).
bond(d27,d27_4,d27_5,7).
bond(d27,d27_5,d27_6,7).
bond(d27,d27_6,d27_1,7).
bond(d27,d27_1,d27_7,1).
bond(d27,d27_2,d27_8,1).
bond(d27,d27_5,d27_9,1).
bond(d27,d27_6,d27_10,1).
bond(d27,d27_3,d27_11,7).
bond(d27,d27_11,d27_12,7).
bond(d27,d27_12,d27_13,7).
bond(d27,d27_13,d27_14,7).
bond(d27,d27_14,d27_4,7).
bond(d27,d27_11,d27_15,1).
bond(d27,d27_12,d27_16,7).
bond(d27,d27_16,d27_17,7).
bond(d27,d27_17,d27_18,7).
bond(d27,d27_18,d27_19,7).
bond(d27,d27_19,d27_13,7).
bond(d27,d27_16,d27_20,1).
bond(d27,d27_17,d27_21,1).
bond(d27,d27_19,d27_22,7).
bond(d27,d27_22,d27_23,7).
bond(d27,d27_23,d27_24,7).
bond(d27,d27_24,d27_14,7).
bond(d27,d27_23,d27_25,1).
bond(d27,d27_24,d27_26,1).
bond(d27,d27_18,d27_27,7).
bond(d27,d27_27,d27_28,7).
bond(d27,d27_28,d27_29,7).
bond(d27,d27_29,d27_22,7).
bond(d27,d27_27,d27_30,1).
bond(d27,d27_28,d27_31,1).
bond(d27,d27_32,d27_29,1).
bond(d27,d27_32,d27_33,2).
bond(d27,d27_32,d27_34,2).
atm(d28,d28_1,c,22,-0.118).
atm(d28,d28_2,c,22,-0.118).
atm(d28,d28_3,c,22,-0.118).
atm(d28,d28_4,c,29,0.012).
atm(d28,d28_5,c,22,-0.118).
atm(d28,d28_6,c,22,-0.118).
atm(d28,d28_7,c,22,-0.118).
atm(d28,d28_8,c,22,-0.118).
atm(d28,d28_9,c,22,0.311).
atm(d28,d28_10,c,22,-0.118).
atm(d28,d28_11,c,22,-0.118).
atm(d28,d28_12,c,29,0.012).
atm(d28,d28_13,c,10,0.012).
atm(d28,d28_14,n,38,0.811).
atm(d28,d28_15,o,40,-0.389).
atm(d28,d28_16,o,40,-0.389).
atm(d28,d28_17,f,92,-0.289).
atm(d28,d28_18,h,3,0.141).
atm(d28,d28_19,h,3,0.141).
atm(d28,d28_20,h,3,0.142).
atm(d28,d28_21,h,3,0.141).
atm(d28,d28_22,h,3,0.141).
atm(d28,d28_23,h,3,0.141).
atm(d28,d28_24,h,3,0.062).
atm(d28,d28_25,h,3,0.062).

bond(d28,d28_1,d28_2,7).
bond(d28,d28_2,d28_3,7).
bond(d28,d28_3,d28_4,7).
bond(d28,d28_4,d28_5,7).
bond(d28,d28_5,d28_6,7).
bond(d28,d28_6,d28_1,7).
bond(d28,d28_7,d28_8,7).
bond(d28,d28_8,d28_9,7).
bond(d28,d28_9,d28_10,7).
bond(d28,d28_10,d28_11,7).
bond(d28,d28_11,d28_12,7).
bond(d28,d28_12,d28_7,7).
bond(d28,d28_4,d28_12,1).
bond(d28,d28_3,d28_13,1).
bond(d28,d28_13,d28_7,1).
bond(d28,d28_1,d28_14,1).
bond(d28,d28_14,d28_15,2).
bond(d28,d28_14,d28_16,2).
bond(d28,d28_9,d28_17,1).
bond(d28,d28_2,d28_18,1).
bond(d28,d28_5,d28_19,1).
bond(d28,d28_6,d28_20,1).
bond(d28,d28_8,d28_21,1).
bond(d28,d28_10,d28_22,1).
bond(d28,d28_11,d28_23,1).
bond(d28,d28_13,d28_24,1).
bond(d28,d28_13,d28_25,1).
atm(d29,d29_1,c,22,-0.106).
atm(d29,d29_2,c,22,-0.106).
atm(d29,d29_3,c,22,-0.106).
atm(d29,d29_4,c,29,0.024).
atm(d29,d29_5,c,22,-0.106).
atm(d29,d29_6,c,22,-0.106).
atm(d29,d29_7,c,22,-0.106).
atm(d29,d29_8,c,22,-0.106).
atm(d29,d29_9,c,22,-0.106).
atm(d29,d29_10,c,22,-0.106).
atm(d29,d29_11,c,22,-0.106).
atm(d29,d29_12,c,29,0.024).
atm(d29,d29_13,c,230,0.194).
atm(d29,d29_14,n,38,0.823).
atm(d29,d29_15,o,40,-0.376).
atm(d29,d29_16,o,40,-0.377).
atm(d29,d29_17,n,38,0.823).
atm(d29,d29_18,o,40,-0.376).
atm(d29,d29_19,o,40,-0.376).
atm(d29,d29_20,n,38,0.823).
atm(d29,d29_21,o,40,-0.377).
atm(d29,d29_22,o,40,-0.377).
atm(d29,d29_23,f,92,-0.276).
atm(d29,d29_24,h,3,0.154).
atm(d29,d29_25,h,3,0.154).
atm(d29,d29_26,h,3,0.154).
atm(d29,d29_27,h,3,0.154).
atm(d29,d29_28,h,3,0.154).
atm(d29,d29_29,h,3,0.114).

bond(d29,d29_1,d29_2,7).
bond(d29,d29_2,d29_3,7).
bond(d29,d29_3,d29_4,7).
bond(d29,d29_4,d29_5,7).
bond(d29,d29_5,d29_6,7).
bond(d29,d29_6,d29_1,7).
bond(d29,d29_7,d29_8,7).
bond(d29,d29_8,d29_9,7).
bond(d29,d29_9,d29_10,7).
bond(d29,d29_10,d29_11,7).
bond(d29,d29_11,d29_12,7).
bond(d29,d29_12,d29_7,7).
bond(d29,d29_4,d29_12,1).
bond(d29,d29_3,d29_13,1).
bond(d29,d29_13,d29_7,1).
bond(d29,d29_1,d29_14,1).
bond(d29,d29_14,d29_15,2).
bond(d29,d29_14,d29_16,2).
bond(d29,d29_9,d29_17,1).
bond(d29,d29_18,d29_17,2).
bond(d29,d29_19,d29_17,2).
bond(d29,d29_20,d29_11,1).
bond(d29,d29_21,d29_20,2).
bond(d29,d29_22,d29_20,2).
bond(d29,d29_13,d29_23,1).
bond(d29,d29_2,d29_24,1).
bond(d29,d29_5,d29_25,1).
bond(d29,d29_6,d29_26,1).
bond(d29,d29_8,d29_27,1).
bond(d29,d29_10,d29_28,1).
bond(d29,d29_13,d29_29,1).
atm(d3,d3_1,c,22,-0.097).
atm(d3,d3_2,c,22,-0.097).
atm(d3,d3_3,c,27,-0.067).
atm(d3,d3_4,c,27,0.033).
atm(d3,d3_5,c,22,-0.097).
atm(d3,d3_6,c,22,-0.097).
atm(d3,d3_7,h,3,0.162).
atm(d3,d3_8,h,3,0.162).
atm(d3,d3_9,h,3,0.163).
atm(d3,d3_10,n,35,-0.738).
atm(d3,d3_11,c,22,0.282).
atm(d3,d3_12,c,22,-0.097).
atm(d3,d3_13,c,22,-0.097).
atm(d3,d3_14,h,3,0.163).
atm(d3,d3_15,h,3,0.163).
atm(d3,d3_16,h,3,0.163).
atm(d3,d3_17,n,38,0.832).
atm(d3,d3_18,o,40,-0.368).
atm(d3,d3_19,o,40,-0.368).

bond(d3,d3_1,d3_2,7).
bond(d3,d3_2,d3_3,7).
bond(d3,d3_3,d3_4,7).
bond(d3,d3_4,d3_5,7).
bond(d3,d3_5,d3_6,7).
bond(d3,d3_6,d3_1,7).
bond(d3,d3_1,d3_7,1).
bond(d3,d3_2,d3_8,1).
bond(d3,d3_6,d3_9,1).
bond(d3,d3_3,d3_10,7).
bond(d3,d3_10,d3_11,7).
bond(d3,d3_11,d3_12,7).
bond(d3,d3_12,d3_13,7).
bond(d3,d3_13,d3_4,7).
bond(d3,d3_11,d3_14,1).
bond(d3,d3_12,d3_15,1).
bond(d3,d3_13,d3_16,1).
bond(d3,d3_17,d3_5,1).
bond(d3,d3_17,d3_18,2).
bond(d3,d3_17,d3_19,2).
atm(d30,d30_1,c,22,-0.115).
atm(d30,d30_2,c,22,-0.115).
atm(d30,d30_3,c,27,-0.085).
atm(d30,d30_4,c,27,-0.085).
atm(d30,d30_5,c,26,-0.085).
atm(d30,d30_6,c,22,-0.115).
atm(d30,d30_7,c,27,-0.085).
atm(d30,d30_8,c,27,0.015).
atm(d30,d30_9,c,22,-0.115).
atm(d30,d30_10,c,26,-0.085).
atm(d30,d30_11,c,22,-0.115).
atm(d30,d30_12,c,27,0.015).
atm(d30,d30_13,c,27,0.015).
atm(d30,d30_14,c,22,-0.115).
atm(d30,d30_15,c,22,-0.115).
atm(d30,d30_16,c,22,-0.115).
atm(d30,d30_17,c,22,-0.115).
atm(d30,d30_18,c,22,-0.115).
atm(d30,d30_19,c,21,0.015).
atm(d30,d30_20,c,21,-0.085).
atm(d30,d30_21,h,3,0.145).
atm(d30,d30_22,h,3,0.145).
atm(d30,d30_23,h,3,0.144).
atm(d30,d30_24,h,3,0.145).
atm(d30,d30_25,h,3,0.144).
atm(d30,d30_26,h,3,0.144).
atm(d30,d30_27,h,3,0.144).
atm(d30,d30_28,h,3,0.144).
atm(d30,d30_29,h,3,0.144).
atm(d30,d30_30,h,3,0.144).
atm(d30,d30_31,h,3,0.115).
atm(d30,d30_32,n,38,0.814).
atm(d30,d30_33,o,40,-0.386).
atm(d30,d30_34,o,40,-0.386).

bond(d30,d30_1,d30_2,7).
bond(d30,d30_2,d30_3,7).
bond(d30,d30_3,d30_4,7).
bond(d30,d30_4,d30_5,7).
bond(d30,d30_5,d30_6,7).
bond(d30,d30_6,d30_1,7).
bond(d30,d30_3,d30_7,7).
bond(d30,d30_7,d30_8,7).
bond(d30,d30_8,d30_9,7).
bond(d30,d30_9,d30_10,7).
bond(d30,d30_10,d30_4,7).
bond(d30,d30_7,d30_11,7).
bond(d30,d30_11,d30_12,7).
bond(d30,d30_12,d30_13,7).
bond(d30,d30_13,d30_14,7).
bond(d30,d30_14,d30_8,7).
bond(d30,d30_12,d30_15,7).
bond(d30,d30_15,d30_16,7).
bond(d30,d30_16,d30_17,7).
bond(d30,d30_17,d30_18,7).
bond(d30,d30_18,d30_13,7).
bond(d30,d30_19,d30_5,7).
bond(d30,d30_19,d30_20,7).
bond(d30,d30_20,d30_10,7).
bond(d30,d30_1,d30_21,1).
bond(d30,d30_2,d30_22,1).
bond(d30,d30_6,d30_23,1).
bond(d30,d30_9,d30_24,1).
bond(d30,d30_11,d30_25,1).
bond(d30,d30_14,d30_26,1).
bond(d30,d30_15,d30_27,1).
bond(d30,d30_16,d30_28,1).
bond(d30,d30_17,d30_29,1).
bond(d30,d30_18,d30_30,1).
bond(d30,d30_20,d30_31,1).
bond(d30,d30_32,d30_19,1).
bond(d30,d30_33,d30_32,2).
bond(d30,d30_34,d30_32,2).
atm(d31,d31_1,c,22,-0.118).
atm(d31,d31_2,c,22,-0.118).
atm(d31,d31_3,c,27,0.012).
atm(d31,d31_4,c,27,-0.088).
atm(d31,d31_5,c,28,0.012).
atm(d31,d31_6,c,22,-0.118).
atm(d31,d31_7,h,3,0.142).
atm(d31,d31_8,h,3,0.142).
atm(d31,d31_9,h,3,0.142).
atm(d31,d31_10,c,22,-0.118).
atm(d31,d31_11,c,22,-0.118).
atm(d31,d31_12,c,22,-0.118).
atm(d31,d31_13,c,29,0.012).
atm(d31,d31_14,h,3,0.142).
atm(d31,d31_15,c,28,0.012).
atm(d31,d31_16,c,28,0.012).
atm(d31,d31_17,c,22,-0.118).
atm(d31,d31_18,c,22,-0.118).
atm(d31,d31_19,c,22,-0.118).
atm(d31,d31_20,c,22,-0.118).
atm(d31,d31_21,h,3,0.142).
atm(d31,d31_22,h,3,0.142).
atm(d31,d31_23,h,3,0.142).
atm(d31,d31_24,n,38,0.812).
atm(d31,d31_25,n,38,0.812).
atm(d31,d31_26,o,40,-0.388).
atm(d31,d31_27,o,40,-0.388).
atm(d31,d31_28,o,40,-0.388).
atm(d31,d31_29,o,40,-0.388).
atm(d31,d31_30,h,3,0.142).

bond(d31,d31_1,d31_2,7).
bond(d31,d31_2,d31_3,7).
bond(d31,d31_3,d31_4,7).
bond(d31,d31_4,d31_5,7).
bond(d31,d31_5,d31_6,7).
bond(d31,d31_6,d31_1,7).
bond(d31,d31_1,d31_7,1).
bond(d31,d31_2,d31_8,1).
bond(d31,d31_6,d31_9,1).
bond(d31,d31_3,d31_10,7).
bond(d31,d31_10,d31_11,7).
bond(d31,d31_11,d31_12,7).
bond(d31,d31_12,d31_13,7).
bond(d31,d31_13,d31_4,7).
bond(d31,d31_10,d31_14,1).
bond(d31,d31_15,d31_16,7).
bond(d31,d31_16,d31_17,7).
bond(d31,d31_17,d31_18,7).
bond(d31,d31_18,d31_19,7).
bond(d31,d31_19,d31_20,7).
bond(d31,d31_20,d31_15,7).
bond(d31,d31_18,d31_21,1).
bond(d31,d31_19,d31_22,1).
bond(d31,d31_20,d31_23,1).
bond(d31,d31_15,d31_5,1).
bond(d31,d31_12,d31_24,1).
bond(d31,d31_11,d31_25,1).
bond(d31,d31_26,d31_24,2).
bond(d31,d31_24,d31_27,2).
bond(d31,d31_25,d31_28,2).
bond(d31,d31_25,d31_29,2).
bond(d31,d31_17,d31_30,1).
bond(d31,d31_16,d31_13,1).
atm(d32,d32_1,c,22,-0.113).
atm(d32,d32_2,c,22,-0.113).
atm(d32,d32_3,c,22,-0.113).
atm(d32,d32_4,c,29,0.017).
atm(d32,d32_5,c,22,-0.113).
atm(d32,d32_6,c,22,-0.113).
atm(d32,d32_7,h,3,0.147).
atm(d32,d32_8,h,3,0.147).
atm(d32,d32_9,h,3,0.147).
atm(d32,d32_10,c,29,0.017).
atm(d32,d32_11,c,22,-0.113).
atm(d32,d32_12,c,22,-0.113).
atm(d32,d32_13,c,22,-0.113).
atm(d32,d32_14,c,22,-0.113).
atm(d32,d32_15,c,22,-0.113).
atm(d32,d32_16,h,3,0.147).
atm(d32,d32_17,h,3,0.147).
atm(d32,d32_18,h,3,0.147).
atm(d32,d32_19,n,38,0.818).
atm(d32,d32_20,n,38,0.818).
atm(d32,d32_21,n,38,0.818).
atm(d32,d32_22,n,38,0.818).
atm(d32,d32_23,o,40,-0.382).
atm(d32,d32_24,o,40,-0.382).
atm(d32,d32_25,o,40,-0.382).
atm(d32,d32_26,o,40,-0.383).
atm(d32,d32_27,o,40,-0.383).
atm(d32,d32_28,o,40,-0.382).
atm(d32,d32_29,o,40,-0.382).
atm(d32,d32_30,o,40,-0.382).

bond(d32,d32_1,d32_2,7).
bond(d32,d32_2,d32_3,7).
bond(d32,d32_3,d32_4,7).
bond(d32,d32_4,d32_5,7).
bond(d32,d32_5,d32_6,7).
bond(d32,d32_6,d32_1,7).
bond(d32,d32_2,d32_7,1).
bond(d32,d32_3,d32_8,1).
bond(d32,d32_5,d32_9,1).
bond(d32,d32_10,d32_11,7).
bond(d32,d32_11,d32_12,7).
bond(d32,d32_12,d32_13,7).
bond(d32,d32_13,d32_14,7).
bond(d32,d32_14,d32_15,7).
bond(d32,d32_15,d32_10,7).
bond(d32,d32_11,d32_16,1).
bond(d32,d32_12,d32_17,1).
bond(d32,d32_14,d32_18,1).
bond(d32,d32_10,d32_4,1).
bond(d32,d32_19,d32_15,1).
bond(d32,d32_13,d32_20,1).
bond(d32,d32_6,d32_21,1).
bond(d32,d32_1,d32_22,1).
bond(d32,d32_23,d32_19,2).
bond(d32,d32_19,d32_24,2).
bond(d32,d32_25,d32_20,2).
bond(d32,d32_20,d32_26,2).
bond(d32,d32_27,d32_22,2).
bond(d32,d32_22,d32_28,2).
bond(d32,d32_29,d32_21,2).
bond(d32,d32_21,d32_30,2).
atm(d33,d33_1,c,22,-0.127).
atm(d33,d33_2,c,22,-0.127).
atm(d33,d33_3,c,22,-0.127).
atm(d33,d33_4,c,29,0.003).
atm(d33,d33_5,c,22,-0.127).
atm(d33,d33_6,c,22,-0.127).
atm(d33,d33_7,h,3,0.133).
atm(d33,d33_8,h,3,0.133).
atm(d33,d33_9,h,3,0.133).
atm(d33,d33_10,c,22,-0.127).
atm(d33,d33_11,c,22,-0.127).
atm(d33,d33_12,c,22,0.203).
atm(d33,d33_13,c,22,-0.127).
atm(d33,d33_14,c,22,-0.127).
atm(d33,d33_15,c,29,0.003).
atm(d33,d33_16,h,3,0.133).
atm(d33,d33_17,h,3,0.133).
atm(d33,d33_18,h,3,0.133).
atm(d33,d33_19,c,10,0.003).
atm(d33,d33_20,h,3,0.053).
atm(d33,d33_21,h,3,0.053).
atm(d33,d33_22,n,38,0.802).
atm(d33,d33_23,o,40,-0.398).
atm(d33,d33_24,o,40,-0.397).
atm(d33,d33_25,n,32,-0.398).
atm(d33,d33_26,c,14,0.602).
atm(d33,d33_27,c,232,1.002).
atm(d33,d33_28,f,92,-0.297).
atm(d33,d33_29,f,92,-0.297).
atm(d33,d33_30,f,92,-0.297).
atm(d33,d33_31,h,1,0.253).
atm(d33,d33_32,o,40,-0.548).

bond(d33,d33_1,d33_2,7).
bond(d33,d33_2,d33_3,7).
bond(d33,d33_3,d33_4,7).
bond(d33,d33_4,d33_5,7).
bond(d33,d33_5,d33_6,7).
bond(d33,d33_6,d33_1,7).
bond(d33,d33_2,d33_7,1).
bond(d33,d33_5,d33_8,1).
bond(d33,d33_6,d33_9,1).
bond(d33,d33_10,d33_11,7).
bond(d33,d33_11,d33_12,7).
bond(d33,d33_12,d33_13,7).
bond(d33,d33_13,d33_14,7).
bond(d33,d33_14,d33_15,7).
bond(d33,d33_15,d33_10,7).
bond(d33,d33_11,d33_16,1).
bond(d33,d33_13,d33_17,1).
bond(d33,d33_14,d33_18,1).
bond(d33,d33_4,d33_15,1).
bond(d33,d33_3,d33_19,1).
bond(d33,d33_19,d33_10,1).
bond(d33,d33_19,d33_20,1).
bond(d33,d33_19,d33_21,1).
bond(d33,d33_1,d33_22,1).
bond(d33,d33_23,d33_22,2).
bond(d33,d33_22,d33_24,2).
bond(d33,d33_12,d33_25,1).
bond(d33,d33_25,d33_26,1).
bond(d33,d33_26,d33_27,1).
bond(d33,d33_27,d33_28,1).
bond(d33,d33_27,d33_29,1).
bond(d33,d33_27,d33_30,1).
bond(d33,d33_25,d33_31,1).
bond(d33,d33_26,d33_32,2).
atm(d34,d34_1,c,22,-0.123).
atm(d34,d34_2,c,22,-0.123).
atm(d34,d34_3,c,27,0.006).
atm(d34,d34_4,c,27,0.006).
atm(d34,d34_5,c,22,-0.123).
atm(d34,d34_6,c,22,-0.123).
atm(d34,d34_7,h,3,0.136).
atm(d34,d34_8,h,3,0.136).
atm(d34,d34_9,h,3,0.137).
atm(d34,d34_10,h,3,0.136).
atm(d34,d34_11,c,22,-0.123).
atm(d34,d34_12,c,22,-0.123).
atm(d34,d34_13,c,22,-0.123).
atm(d34,d34_14,c,22,-0.123).
atm(d34,d34_15,h,3,0.137).
atm(d34,d34_16,h,3,0.137).
atm(d34,d34_17,h,3,0.137).
atm(d34,d34_18,n,38,0.806).
atm(d34,d34_19,o,40,-0.395).
atm(d34,d34_20,o,40,-0.395).

bond(d34,d34_1,d34_2,7).
bond(d34,d34_2,d34_3,7).
bond(d34,d34_3,d34_4,7).
bond(d34,d34_4,d34_5,7).
bond(d34,d34_5,d34_6,7).
bond(d34,d34_6,d34_1,7).
bond(d34,d34_1,d34_7,1).
bond(d34,d34_2,d34_8,1).
bond(d34,d34_5,d34_9,1).
bond(d34,d34_6,d34_10,1).
bond(d34,d34_3,d34_11,7).
bond(d34,d34_11,d34_12,7).
bond(d34,d34_12,d34_13,7).
bond(d34,d34_13,d34_14,7).
bond(d34,d34_14,d34_4,7).
bond(d34,d34_11,d34_15,1).
bond(d34,d34_12,d34_16,1).
bond(d34,d34_14,d34_17,1).
bond(d34,d34_18,d34_13,1).
bond(d34,d34_19,d34_18,2).
bond(d34,d34_18,d34_20,2).
atm(d35,d35_1,c,22,-0.122).
atm(d35,d35_2,c,22,-0.122).
atm(d35,d35_3,c,22,-0.122).
atm(d35,d35_4,c,29,0.008).
atm(d35,d35_5,c,22,-0.122).
atm(d35,d35_6,c,22,-0.122).
atm(d35,d35_7,h,3,0.138).
atm(d35,d35_8,h,3,0.138).
atm(d35,d35_9,h,3,0.138).
atm(d35,d35_10,c,22,-0.122).
atm(d35,d35_11,c,22,-0.122).
atm(d35,d35_12,c,22,0.208).
atm(d35,d35_13,c,22,-0.122).
atm(d35,d35_14,c,22,-0.122).
atm(d35,d35_15,c,29,0.008).
atm(d35,d35_16,h,3,0.138).
atm(d35,d35_17,h,3,0.138).
atm(d35,d35_18,h,3,0.138).
atm(d35,d35_19,c,10,0.008).
atm(d35,d35_20,h,3,0.058).
atm(d35,d35_21,h,3,0.058).
atm(d35,d35_22,n,38,0.807).
atm(d35,d35_23,n,32,-0.773).
atm(d35,d35_24,o,40,-0.393).
atm(d35,d35_25,o,40,-0.393).
atm(d35,d35_26,h,1,0.337).
atm(d35,d35_27,h,1,0.337).

bond(d35,d35_1,d35_2,7).
bond(d35,d35_2,d35_3,7).
bond(d35,d35_3,d35_4,7).
bond(d35,d35_4,d35_5,7).
bond(d35,d35_5,d35_6,7).
bond(d35,d35_6,d35_1,7).
bond(d35,d35_2,d35_7,1).
bond(d35,d35_5,d35_8,1).
bond(d35,d35_6,d35_9,1).
bond(d35,d35_10,d35_11,7).
bond(d35,d35_11,d35_12,7).
bond(d35,d35_12,d35_13,7).
bond(d35,d35_13,d35_14,7).
bond(d35,d35_14,d35_15,7).
bond(d35,d35_15,d35_10,7).
bond(d35,d35_11,d35_16,1).
bond(d35,d35_13,d35_17,1).
bond(d35,d35_14,d35_18,1).
bond(d35,d35_4,d35_15,1).
bond(d35,d35_3,d35_19,1).
bond(d35,d35_19,d35_10,1).
bond(d35,d35_19,d35_20,1).
bond(d35,d35_19,d35_21,1).
bond(d35,d35_1,d35_22,1).
bond(d35,d35_12,d35_23,1).
bond(d35,d35_24,d35_22,2).
bond(d35,d35_22,d35_25,2).
bond(d35,d35_23,d35_26,1).
bond(d35,d35_23,d35_27,1).
atm(d36,d36_1,c,22,-0.123).
atm(d36,d36_2,c,22,-0.123).
atm(d36,d36_3,c,27,0.007).
atm(d36,d36_4,c,27,0.007).
atm(d36,d36_5,c,22,-0.123).
atm(d36,d36_6,c,22,-0.123).
atm(d36,d36_7,h,3,0.137).
atm(d36,d36_8,h,3,0.137).
atm(d36,d36_9,h,3,0.137).
atm(d36,d36_10,h,3,0.137).
atm(d36,d36_11,c,22,-0.123).
atm(d36,d36_12,c,22,-0.173).
atm(d36,d36_13,c,22,-0.123).
atm(d36,d36_14,c,22,-0.123).
atm(d36,d36_15,h,3,0.137).
atm(d36,d36_16,h,3,0.137).
atm(d36,d36_17,c,10,0.007).
atm(d36,d36_18,h,3,0.057).
atm(d36,d36_19,h,3,0.057).
atm(d36,d36_20,h,3,0.057).
atm(d36,d36_21,n,38,0.806).
atm(d36,d36_22,o,40,-0.393).
atm(d36,d36_23,o,40,-0.393).

bond(d36,d36_1,d36_2,7).
bond(d36,d36_2,d36_3,7).
bond(d36,d36_3,d36_4,7).
bond(d36,d36_4,d36_5,7).
bond(d36,d36_5,d36_6,7).
bond(d36,d36_6,d36_1,7).
bond(d36,d36_1,d36_7,1).
bond(d36,d36_2,d36_8,1).
bond(d36,d36_5,d36_9,1).
bond(d36,d36_6,d36_10,1).
bond(d36,d36_3,d36_11,7).
bond(d36,d36_11,d36_12,7).
bond(d36,d36_12,d36_13,7).
bond(d36,d36_13,d36_14,7).
bond(d36,d36_14,d36_4,7).
bond(d36,d36_11,d36_15,1).
bond(d36,d36_14,d36_16,1).
bond(d36,d36_12,d36_17,1).
bond(d36,d36_17,d36_18,1).
bond(d36,d36_17,d36_19,1).
bond(d36,d36_17,d36_20,1).
bond(d36,d36_13,d36_21,1).
bond(d36,d36_22,d36_21,2).
bond(d36,d36_21,d36_23,2).
atm(d37,d37_1,c,22,-0.126).
atm(d37,d37_2,c,22,-0.126).
atm(d37,d37_3,c,22,-0.126).
atm(d37,d37_4,c,22,0.004).
atm(d37,d37_5,c,22,-0.126).
atm(d37,d37_6,c,22,-0.126).
atm(d37,d37_7,h,3,0.135).
atm(d37,d37_8,h,3,0.135).
atm(d37,d37_9,h,3,0.135).
atm(d37,d37_10,h,3,0.135).
atm(d37,d37_11,c,10,-0.096).
atm(d37,d37_12,c,10,-0.096).
atm(d37,d37_13,h,3,0.054).
atm(d37,d37_14,h,3,0.054).
atm(d37,d37_15,c,22,0.004).
atm(d37,d37_16,h,3,0.054).
atm(d37,d37_17,h,3,0.054).
atm(d37,d37_18,c,22,-0.126).
atm(d37,d37_19,c,22,-0.126).
atm(d37,d37_20,c,22,-0.126).
atm(d37,d37_21,c,22,-0.126).
atm(d37,d37_22,c,22,-0.126).
atm(d37,d37_23,h,3,0.134).
atm(d37,d37_24,h,3,0.135).
atm(d37,d37_25,h,3,0.134).
atm(d37,d37_26,h,3,0.135).
atm(d37,d37_27,h,3,0.135).
atm(d37,d37_28,n,38,0.805).
atm(d37,d37_29,o,40,-0.395).
atm(d37,d37_30,o,40,-0.395).

bond(d37,d37_1,d37_2,7).
bond(d37,d37_2,d37_3,7).
bond(d37,d37_3,d37_4,7).
bond(d37,d37_4,d37_5,7).
bond(d37,d37_5,d37_6,7).
bond(d37,d37_6,d37_1,7).
bond(d37,d37_2,d37_7,1).
bond(d37,d37_3,d37_8,1).
bond(d37,d37_5,d37_9,1).
bond(d37,d37_6,d37_10,1).
bond(d37,d37_4,d37_11,1).
bond(d37,d37_11,d37_12,1).
bond(d37,d37_11,d37_13,1).
bond(d37,d37_11,d37_14,1).
bond(d37,d37_12,d37_15,1).
bond(d37,d37_12,d37_16,1).
bond(d37,d37_12,d37_17,1).
bond(d37,d37_15,d37_18,7).
bond(d37,d37_18,d37_19,7).
bond(d37,d37_19,d37_20,7).
bond(d37,d37_20,d37_21,7).
bond(d37,d37_21,d37_22,7).
bond(d37,d37_22,d37_15,7).
bond(d37,d37_18,d37_23,1).
bond(d37,d37_19,d37_24,1).
bond(d37,d37_20,d37_25,1).
bond(d37,d37_21,d37_26,1).
bond(d37,d37_22,d37_27,1).
bond(d37,d37_1,d37_28,1).
bond(d37,d37_29,d37_28,2).
bond(d37,d37_28,d37_30,2).
atm(d38,d38_1,c,22,-0.125).
atm(d38,d38_2,c,22,-0.125).
atm(d38,d38_3,c,22,-0.125).
atm(d38,d38_4,c,29,0.005).
atm(d38,d38_5,c,22,-0.125).
atm(d38,d38_6,c,22,-0.125).
atm(d38,d38_7,h,3,0.136).
atm(d38,d38_8,h,3,0.136).
atm(d38,d38_9,h,3,0.136).
atm(d38,d38_10,h,3,0.136).
atm(d38,d38_11,h,3,0.136).
atm(d38,d38_12,c,29,0.005).
atm(d38,d38_13,c,22,-0.125).
atm(d38,d38_14,c,22,-0.125).
atm(d38,d38_15,c,22,-0.125).
atm(d38,d38_16,c,22,-0.125).
atm(d38,d38_17,c,22,-0.125).
atm(d38,d38_18,h,3,0.136).
atm(d38,d38_19,h,3,0.135).
atm(d38,d38_20,h,3,0.135).
atm(d38,d38_21,h,3,0.136).
atm(d38,d38_22,n,38,0.806).
atm(d38,d38_23,o,40,-0.394).
atm(d38,d38_24,o,40,-0.394).

bond(d38,d38_1,d38_2,7).
bond(d38,d38_2,d38_3,7).
bond(d38,d38_3,d38_4,7).
bond(d38,d38_4,d38_5,7).
bond(d38,d38_5,d38_6,7).
bond(d38,d38_6,d38_1,7).
bond(d38,d38_1,d38_7,1).
bond(d38,d38_2,d38_8,1).
bond(d38,d38_3,d38_9,1).
bond(d38,d38_5,d38_10,1).
bond(d38,d38_6,d38_11,1).
bond(d38,d38_12,d38_13,7).
bond(d38,d38_13,d38_14,7).
bond(d38,d38_14,d38_15,7).
bond(d38,d38_15,d38_16,7).
bond(d38,d38_16,d38_17,7).
bond(d38,d38_17,d38_12,7).
bond(d38,d38_13,d38_18,1).
bond(d38,d38_14,d38_19,1).
bond(d38,d38_16,d38_20,1).
bond(d38,d38_17,d38_21,1).
bond(d38,d38_12,d38_4,1).
bond(d38,d38_22,d38_15,1).
bond(d38,d38_23,d38_22,2).
bond(d38,d38_22,d38_24,2).
atm(d39,d39_1,c,22,-0.111).
atm(d39,d39_2,c,22,-0.111).
atm(d39,d39_3,c,27,-0.081).
atm(d39,d39_4,c,27,0.019).
atm(d39,d39_5,c,22,-0.111).
atm(d39,d39_6,c,22,0.198).
atm(d39,d39_7,h,3,0.149).
atm(d39,d39_8,h,3,0.149).
atm(d39,d39_9,n,35,-0.752).
atm(d39,d39_10,c,22,0.268).
atm(d39,d39_11,c,22,-0.111).
atm(d39,d39_12,c,22,-0.111).
atm(d39,d39_13,h,3,0.149).
atm(d39,d39_14,h,3,0.149).
atm(d39,d39_15,h,3,0.149).
atm(d39,d39_16,n,38,0.818).
atm(d39,d39_17,o,40,-0.382).
atm(d39,d39_18,o,40,-0.382).
atm(d39,d39_19,o,50,-0.212).
atm(d39,d39_20,c,10,0.109).
atm(d39,d39_21,h,3,0.069).
atm(d39,d39_22,h,3,0.069).
atm(d39,d39_23,h,3,0.069).

bond(d39,d39_1,d39_2,7).
bond(d39,d39_2,d39_3,7).
bond(d39,d39_3,d39_4,7).
bond(d39,d39_4,d39_5,7).
bond(d39,d39_5,d39_6,7).
bond(d39,d39_6,d39_1,7).
bond(d39,d39_1,d39_7,1).
bond(d39,d39_5,d39_8,1).
bond(d39,d39_3,d39_9,7).
bond(d39,d39_9,d39_10,7).
bond(d39,d39_10,d39_11,7).
bond(d39,d39_11,d39_12,7).
bond(d39,d39_12,d39_4,7).
bond(d39,d39_10,d39_13,1).
bond(d39,d39_11,d39_14,1).
bond(d39,d39_12,d39_15,1).
bond(d39,d39_2,d39_16,1).
bond(d39,d39_16,d39_17,2).
bond(d39,d39_16,d39_18,2).
bond(d39,d39_6,d39_19,1).
bond(d39,d39_19,d39_20,1).
bond(d39,d39_20,d39_21,1).
bond(d39,d39_20,d39_22,1).
bond(d39,d39_20,d39_23,1).
atm(d4,d4_1,c,22,-0.122).
atm(d4,d4_2,c,22,-0.122).
atm(d4,d4_3,c,27,0.008).
atm(d4,d4_4,c,27,-0.092).
atm(d4,d4_5,c,28,0.008).
atm(d4,d4_6,c,22,-0.122).
atm(d4,d4_7,c,22,-0.122).
atm(d4,d4_8,c,22,-0.122).
atm(d4,d4_9,c,22,-0.122).
atm(d4,d4_10,c,29,0.008).
atm(d4,d4_11,c,22,-0.122).
atm(d4,d4_12,c,29,0.008).
atm(d4,d4_13,c,28,0.008).
atm(d4,d4_14,c,22,-0.122).
atm(d4,d4_15,c,22,-0.122).
atm(d4,d4_16,c,22,-0.122).
atm(d4,d4_17,n,38,0.809).
atm(d4,d4_18,o,40,-0.391).
atm(d4,d4_19,o,40,-0.391).
atm(d4,d4_20,h,3,0.138).
atm(d4,d4_21,h,3,0.138).
atm(d4,d4_22,h,3,0.138).
atm(d4,d4_23,h,3,0.139).
atm(d4,d4_24,h,3,0.139).
atm(d4,d4_25,h,3,0.138).
atm(d4,d4_26,h,3,0.138).
atm(d4,d4_27,h,3,0.138).
atm(d4,d4_28,h,3,0.139).

bond(d4,d4_1,d4_2,7).
bond(d4,d4_2,d4_3,7).
bond(d4,d4_3,d4_4,7).
bond(d4,d4_4,d4_5,7).
bond(d4,d4_5,d4_6,7).
bond(d4,d4_6,d4_1,7).
bond(d4,d4_3,d4_7,7).
bond(d4,d4_7,d4_8,7).
bond(d4,d4_8,d4_9,7).
bond(d4,d4_9,d4_10,7).
bond(d4,d4_10,d4_4,7).
bond(d4,d4_11,d4_12,7).
bond(d4,d4_12,d4_13,7).
bond(d4,d4_13,d4_14,7).
bond(d4,d4_14,d4_15,7).
bond(d4,d4_15,d4_16,7).
bond(d4,d4_16,d4_11,7).
bond(d4,d4_12,d4_5,1).
bond(d4,d4_13,d4_10,1).
bond(d4,d4_17,d4_9,1).
bond(d4,d4_17,d4_18,2).
bond(d4,d4_17,d4_19,2).
bond(d4,d4_1,d4_20,1).
bond(d4,d4_2,d4_21,1).
bond(d4,d4_6,d4_22,1).
bond(d4,d4_7,d4_23,1).
bond(d4,d4_8,d4_24,1).
bond(d4,d4_11,d4_25,1).
bond(d4,d4_14,d4_26,1).
bond(d4,d4_15,d4_27,1).
bond(d4,d4_16,d4_28,1).
atm(d40,d40_1,c,22,-0.123).
atm(d40,d40_2,c,22,-0.123).
atm(d40,d40_3,c,27,0.006).
atm(d40,d40_4,c,27,0.006).
atm(d40,d40_5,c,22,-0.123).
atm(d40,d40_6,c,22,-0.123).
atm(d40,d40_7,h,3,0.136).
atm(d40,d40_8,h,3,0.136).
atm(d40,d40_9,h,3,0.137).
atm(d40,d40_10,h,3,0.136).
atm(d40,d40_11,c,22,-0.123).
atm(d40,d40_12,c,22,-0.123).
atm(d40,d40_13,c,22,-0.123).
atm(d40,d40_14,c,22,-0.123).
atm(d40,d40_15,h,3,0.137).
atm(d40,d40_16,h,3,0.137).
atm(d40,d40_17,h,3,0.137).
atm(d40,d40_18,n,38,0.806).
atm(d40,d40_19,o,40,-0.395).
atm(d40,d40_20,o,40,-0.395).

bond(d40,d40_1,d40_2,7).
bond(d40,d40_2,d40_3,7).
bond(d40,d40_3,d40_4,7).
bond(d40,d40_4,d40_5,7).
bond(d40,d40_5,d40_6,7).
bond(d40,d40_6,d40_1,7).
bond(d40,d40_1,d40_7,1).
bond(d40,d40_2,d40_8,1).
bond(d40,d40_5,d40_9,1).
bond(d40,d40_6,d40_10,1).
bond(d40,d40_3,d40_11,7).
bond(d40,d40_11,d40_12,7).
bond(d40,d40_12,d40_13,7).
bond(d40,d40_13,d40_14,7).
bond(d40,d40_14,d40_4,7).
bond(d40,d40_11,d40_15,1).
bond(d40,d40_12,d40_16,1).
bond(d40,d40_13,d40_17,1).
bond(d40,d40_18,d40_14,1).
bond(d40,d40_19,d40_18,2).
bond(d40,d40_18,d40_20,2).
atm(d41,d41_1,c,22,-0.107).
atm(d41,d41_2,c,22,-0.107).
atm(d41,d41_3,c,22,-0.107).
atm(d41,d41_4,c,29,0.023).
atm(d41,d41_5,c,22,-0.107).
atm(d41,d41_6,c,22,-0.107).
atm(d41,d41_7,h,3,0.152).
atm(d41,d41_8,h,3,0.152).
atm(d41,d41_9,h,3,0.153).
atm(d41,d41_10,c,22,-0.107).
atm(d41,d41_11,c,22,-0.107).
atm(d41,d41_12,c,22,-0.107).
atm(d41,d41_13,c,22,-0.107).
atm(d41,d41_14,c,22,-0.107).
atm(d41,d41_15,c,29,0.023).
atm(d41,d41_16,h,3,0.153).
atm(d41,d41_17,h,3,0.153).
atm(d41,d41_18,h,3,0.153).
atm(d41,d41_19,c,10,0.023).
atm(d41,d41_20,h,3,0.073).
atm(d41,d41_21,h,3,0.073).
atm(d41,d41_22,br,94,-0.127).
atm(d41,d41_23,n,38,0.822).
atm(d41,d41_24,o,40,-0.378).
atm(d41,d41_25,o,40,-0.378).

bond(d41,d41_1,d41_2,7).
bond(d41,d41_2,d41_3,7).
bond(d41,d41_3,d41_4,7).
bond(d41,d41_4,d41_5,7).
bond(d41,d41_5,d41_6,7).
bond(d41,d41_6,d41_1,7).
bond(d41,d41_2,d41_7,1).
bond(d41,d41_5,d41_8,1).
bond(d41,d41_6,d41_9,1).
bond(d41,d41_10,d41_11,7).
bond(d41,d41_11,d41_12,7).
bond(d41,d41_12,d41_13,7).
bond(d41,d41_13,d41_14,7).
bond(d41,d41_14,d41_15,7).
bond(d41,d41_15,d41_10,7).
bond(d41,d41_11,d41_16,1).
bond(d41,d41_13,d41_17,1).
bond(d41,d41_14,d41_18,1).
bond(d41,d41_4,d41_15,1).
bond(d41,d41_3,d41_19,1).
bond(d41,d41_10,d41_19,1).
bond(d41,d41_19,d41_20,1).
bond(d41,d41_19,d41_21,1).
bond(d41,d41_12,d41_22,1).
bond(d41,d41_1,d41_23,1).
bond(d41,d41_24,d41_23,2).
bond(d41,d41_23,d41_25,2).
atm(d42,d42_1,c,22,-0.122).
atm(d42,d42_2,c,22,-0.122).
atm(d42,d42_3,c,26,0.307).
atm(d42,d42_4,c,26,0.108).
atm(d42,d42_5,c,22,-0.122).
atm(d42,d42_6,c,22,-0.122).
atm(d42,d42_7,n,34,-0.393).
atm(d42,d42_8,n,34,-0.533).
atm(d42,d42_9,c,21,-0.122).
atm(d42,d42_10,c,10,0.098).
atm(d42,d42_11,n,38,0.807).
atm(d42,d42_12,o,40,-0.393).
atm(d42,d42_13,o,40,-0.393).
atm(d42,d42_14,h,3,0.137).
atm(d42,d42_15,h,3,0.138).
atm(d42,d42_16,h,3,0.108).
atm(d42,d42_17,h,3,0.138).
atm(d42,d42_18,h,3,0.058).
atm(d42,d42_19,h,3,0.058).
atm(d42,d42_20,h,3,0.058).
atm(d42,d42_21,h,1,0.307).

bond(d42,d42_1,d42_2,7).
bond(d42,d42_2,d42_3,7).
bond(d42,d42_3,d42_4,7).
bond(d42,d42_4,d42_5,7).
bond(d42,d42_5,d42_6,7).
bond(d42,d42_6,d42_1,7).
bond(d42,d42_3,d42_7,7).
bond(d42,d42_7,d42_8,7).
bond(d42,d42_4,d42_9,7).
bond(d42,d42_9,d42_8,7).
bond(d42,d42_8,d42_10,1).
bond(d42,d42_6,d42_11,1).
bond(d42,d42_12,d42_11,2).
bond(d42,d42_11,d42_13,2).
bond(d42,d42_1,d42_14,1).
bond(d42,d42_2,d42_15,1).
bond(d42,d42_5,d42_16,1).
bond(d42,d42_9,d42_17,1).
bond(d42,d42_10,d42_18,1).
bond(d42,d42_10,d42_19,1).
bond(d42,d42_10,d42_20,1).
bond(d42,d42_7,d42_21,1).
atm(d43,d43_1,c,22,-0.122).
atm(d43,d43_2,c,22,-0.122).
atm(d43,d43_3,c,27,0.008).
atm(d43,d43_4,c,27,-0.092).
atm(d43,d43_5,c,28,0.008).
atm(d43,d43_6,c,22,-0.122).
atm(d43,d43_7,h,3,0.139).
atm(d43,d43_8,h,3,0.138).
atm(d43,d43_9,h,3,0.138).
atm(d43,d43_10,c,22,-0.122).
atm(d43,d43_11,c,22,-0.122).
atm(d43,d43_12,c,22,-0.122).
atm(d43,d43_13,c,29,0.008).
atm(d43,d43_14,h,3,0.139).
atm(d43,d43_15,h,3,0.138).
atm(d43,d43_16,c,28,0.008).
atm(d43,d43_17,c,28,0.008).
atm(d43,d43_18,c,22,-0.122).
atm(d43,d43_19,c,22,-0.122).
atm(d43,d43_20,c,22,-0.122).
atm(d43,d43_21,c,22,-0.122).
atm(d43,d43_22,h,3,0.139).
atm(d43,d43_23,h,3,0.138).
atm(d43,d43_24,h,3,0.138).
atm(d43,d43_25,n,38,0.809).
atm(d43,d43_26,o,40,-0.391).
atm(d43,d43_27,o,40,-0.391).
atm(d43,d43_28,h,3,0.138).

bond(d43,d43_1,d43_2,7).
bond(d43,d43_2,d43_3,7).
bond(d43,d43_3,d43_4,7).
bond(d43,d43_4,d43_5,7).
bond(d43,d43_5,d43_6,7).
bond(d43,d43_6,d43_1,7).
bond(d43,d43_1,d43_7,1).
bond(d43,d43_2,d43_8,1).
bond(d43,d43_6,d43_9,1).
bond(d43,d43_3,d43_10,7).
bond(d43,d43_10,d43_11,7).
bond(d43,d43_11,d43_12,7).
bond(d43,d43_12,d43_13,7).
bond(d43,d43_13,d43_4,7).
bond(d43,d43_10,d43_14,1).
bond(d43,d43_12,d43_15,1).
bond(d43,d43_16,d43_17,7).
bond(d43,d43_17,d43_18,7).
bond(d43,d43_18,d43_19,7).
bond(d43,d43_19,d43_20,7).
bond(d43,d43_20,d43_21,7).
bond(d43,d43_21,d43_16,7).
bond(d43,d43_19,d43_22,1).
bond(d43,d43_20,d43_23,1).
bond(d43,d43_21,d43_24,1).
bond(d43,d43_16,d43_5,1).
bond(d43,d43_11,d43_25,1).
bond(d43,d43_26,d43_25,2).
bond(d43,d43_25,d43_27,2).
bond(d43,d43_18,d43_28,1).
bond(d43,d43_17,d43_13,1).
atm(d44,d44_1,c,22,-0.125).
atm(d44,d44_2,c,22,-0.125).
atm(d44,d44_3,c,22,-0.125).
atm(d44,d44_4,c,29,0.005).
atm(d44,d44_5,c,22,0.005).
atm(d44,d44_6,c,22,-0.125).
atm(d44,d44_7,c,29,0.005).
atm(d44,d44_8,c,22,0.005).
atm(d44,d44_9,c,10,-0.095).
atm(d44,d44_10,c,10,-0.095).
atm(d44,d44_11,c,22,-0.125).
atm(d44,d44_12,c,22,-0.125).
atm(d44,d44_13,c,22,-0.125).
atm(d44,d44_14,c,22,-0.125).
atm(d44,d44_15,n,38,0.804).
atm(d44,d44_16,o,40,-0.396).
atm(d44,d44_17,o,40,-0.396).
atm(d44,d44_18,h,3,0.134).
atm(d44,d44_19,h,3,0.134).
atm(d44,d44_20,h,3,0.134).
atm(d44,d44_21,h,3,0.134).
atm(d44,d44_22,h,3,0.134).
atm(d44,d44_23,h,3,0.134).
atm(d44,d44_24,h,3,0.134).
atm(d44,d44_25,h,3,0.055).
atm(d44,d44_26,h,3,0.055).
atm(d44,d44_27,h,3,0.055).
atm(d44,d44_28,h,3,0.055).

bond(d44,d44_1,d44_2,7).
bond(d44,d44_2,d44_3,7).
bond(d44,d44_3,d44_4,7).
bond(d44,d44_4,d44_5,7).
bond(d44,d44_5,d44_6,7).
bond(d44,d44_6,d44_1,7).
bond(d44,d44_4,d44_7,1).
bond(d44,d44_7,d44_8,7).
bond(d44,d44_8,d44_9,1).
bond(d44,d44_9,d44_10,1).
bond(d44,d44_10,d44_5,1).
bond(d44,d44_7,d44_11,7).
bond(d44,d44_11,d44_12,7).
bond(d44,d44_12,d44_13,7).
bond(d44,d44_13,d44_14,7).
bond(d44,d44_14,d44_8,7).
bond(d44,d44_13,d44_15,1).
bond(d44,d44_16,d44_15,2).
bond(d44,d44_15,d44_17,2).
bond(d44,d44_1,d44_18,1).
bond(d44,d44_2,d44_19,1).
bond(d44,d44_3,d44_20,1).
bond(d44,d44_6,d44_21,1).
bond(d44,d44_11,d44_22,1).
bond(d44,d44_12,d44_23,1).
bond(d44,d44_14,d44_24,1).
bond(d44,d44_9,d44_25,1).
bond(d44,d44_9,d44_26,1).
bond(d44,d44_10,d44_27,1).
bond(d44,d44_10,d44_28,1).
atm(d45,d45_1,c,22,-0.114).
atm(d45,d45_2,c,22,-0.114).
atm(d45,d45_3,c,22,-0.114).
atm(d45,d45_4,c,22,-0.114).
atm(d45,d45_5,c,22,-0.114).
atm(d45,d45_6,c,22,-0.114).
atm(d45,d45_7,h,3,0.146).
atm(d45,d45_8,h,3,0.146).
atm(d45,d45_9,h,3,0.146).
atm(d45,d45_10,h,3,0.146).
atm(d45,d45_11,n,38,0.817).
atm(d45,d45_12,n,38,0.817).
atm(d45,d45_13,o,40,-0.383).
atm(d45,d45_14,o,40,-0.384).
atm(d45,d45_15,o,40,-0.384).
atm(d45,d45_16,o,40,-0.383).

bond(d45,d45_1,d45_2,7).
bond(d45,d45_2,d45_3,7).
bond(d45,d45_3,d45_4,7).
bond(d45,d45_4,d45_5,7).
bond(d45,d45_5,d45_6,7).
bond(d45,d45_6,d45_1,7).
bond(d45,d45_1,d45_7,1).
bond(d45,d45_3,d45_8,1).
bond(d45,d45_4,d45_9,1).
bond(d45,d45_6,d45_10,1).
bond(d45,d45_11,d45_5,1).
bond(d45,d45_2,d45_12,1).
bond(d45,d45_13,d45_11,2).
bond(d45,d45_11,d45_14,2).
bond(d45,d45_12,d45_15,2).
bond(d45,d45_12,d45_16,2).
atm(d46,d46_1,c,10,-0.091).
atm(d46,d46_2,c,10,-0.091).
atm(d46,d46_3,c,22,0.009).
atm(d46,d46_4,c,22,0.009).
atm(d46,d46_5,c,10,-0.091).
atm(d46,d46_6,c,10,-0.091).
atm(d46,d46_7,c,22,-0.122).
atm(d46,d46_8,c,27,0.009).
atm(d46,d46_9,c,27,-0.091).
atm(d46,d46_10,c,27,0.009).
atm(d46,d46_11,c,22,-0.121).
atm(d46,d46_12,c,22,-0.122).
atm(d46,d46_13,c,27,0.009).
atm(d46,d46_14,c,27,-0.091).
atm(d46,d46_15,c,27,0.009).
atm(d46,d46_16,c,22,-0.121).
atm(d46,d46_17,c,22,-0.121).
atm(d46,d46_18,c,22,-0.121).
atm(d46,d46_19,c,22,-0.121).
atm(d46,d46_20,c,22,-0.121).
atm(d46,d46_21,n,38,0.808).
atm(d46,d46_22,o,40,-0.392).
atm(d46,d46_23,o,40,-0.392).
atm(d46,d46_24,h,3,0.138).
atm(d46,d46_25,h,3,0.138).
atm(d46,d46_26,h,3,0.138).
atm(d46,d46_27,h,3,0.138).
atm(d46,d46_28,h,3,0.138).
atm(d46,d46_29,h,3,0.138).
atm(d46,d46_30,h,3,0.138).
atm(d46,d46_31,h,3,0.059).
atm(d46,d46_32,h,3,0.059).
atm(d46,d46_33,h,3,0.059).
atm(d46,d46_34,h,3,0.059).
atm(d46,d46_35,h,3,0.059).
atm(d46,d46_36,h,3,0.059).
atm(d46,d46_37,h,3,0.059).
atm(d46,d46_38,h,3,0.059).

bond(d46,d46_1,d46_2,1).
bond(d46,d46_2,d46_3,1).
bond(d46,d46_3,d46_4,7).
bond(d46,d46_4,d46_5,1).
bond(d46,d46_5,d46_6,1).
bond(d46,d46_6,d46_1,1).
bond(d46,d46_3,d46_7,7).
bond(d46,d46_7,d46_8,7).
bond(d46,d46_8,d46_9,7).
bond(d46,d46_9,d46_10,7).
bond(d46,d46_10,d46_4,7).
bond(d46,d46_8,d46_11,7).
bond(d46,d46_11,d46_12,7).
bond(d46,d46_12,d46_13,7).
bond(d46,d46_13,d46_14,7).
bond(d46,d46_14,d46_9,7).
bond(d46,d46_14,d46_15,7).
bond(d46,d46_15,d46_16,7).
bond(d46,d46_16,d46_17,7).
bond(d46,d46_17,d46_10,7).
bond(d46,d46_13,d46_18,7).
bond(d46,d46_18,d46_19,7).
bond(d46,d46_19,d46_20,7).
bond(d46,d46_20,d46_15,7).
bond(d46,d46_21,d46_20,1).
bond(d46,d46_21,d46_22,2).
bond(d46,d46_23,d46_21,2).
bond(d46,d46_7,d46_24,1).
bond(d46,d46_11,d46_25,1).
bond(d46,d46_12,d46_26,1).
bond(d46,d46_16,d46_27,1).
bond(d46,d46_17,d46_28,1).
bond(d46,d46_18,d46_29,1).
bond(d46,d46_19,d46_30,1).
bond(d46,d46_5,d46_31,1).
bond(d46,d46_5,d46_32,1).
bond(d46,d46_1,d46_33,1).
bond(d46,d46_1,d46_34,1).
bond(d46,d46_6,d46_35,1).
bond(d46,d46_6,d46_36,1).
bond(d46,d46_2,d46_37,1).
bond(d46,d46_2,d46_38,1).
atm(d47,d47_1,c,22,-0.122).
atm(d47,d47_2,c,22,-0.122).
atm(d47,d47_3,c,22,0.008).
atm(d47,d47_4,c,29,0.008).
atm(d47,d47_5,c,22,0.008).
atm(d47,d47_6,c,22,-0.122).
atm(d47,d47_7,c,10,-0.092).
atm(d47,d47_8,c,10,-0.092).
atm(d47,d47_9,c,22,0.008).
atm(d47,d47_10,c,29,0.008).
atm(d47,d47_11,c,22,0.008).
atm(d47,d47_12,c,10,-0.092).
atm(d47,d47_13,c,10,-0.092).
atm(d47,d47_14,c,22,-0.123).
atm(d47,d47_15,c,22,-0.123).
atm(d47,d47_16,c,22,-0.122).
atm(d47,d47_17,n,38,0.807).
atm(d47,d47_18,n,38,0.807).
atm(d47,d47_19,o,40,-0.393).
atm(d47,d47_20,o,40,-0.393).
atm(d47,d47_21,o,40,-0.393).
atm(d47,d47_22,o,40,-0.393).
atm(d47,d47_23,h,3,0.137).
atm(d47,d47_24,h,3,0.137).
atm(d47,d47_25,h,3,0.058).
atm(d47,d47_26,h,3,0.058).
atm(d47,d47_27,h,3,0.058).
atm(d47,d47_28,h,3,0.058).
atm(d47,d47_29,h,3,0.137).
atm(d47,d47_30,h,3,0.137).
atm(d47,d47_31,h,3,0.058).
atm(d47,d47_32,h,3,0.058).
atm(d47,d47_33,h,3,0.058).
atm(d47,d47_34,h,3,0.058).

bond(d47,d47_1,d47_2,7).
bond(d47,d47_2,d47_3,7).
bond(d47,d47_3,d47_4,7).
bond(d47,d47_4,d47_5,7).
bond(d47,d47_5,d47_6,7).
bond(d47,d47_6,d47_1,7).
bond(d47,d47_3,d47_7,1).
bond(d47,d47_7,d47_8,1).
bond(d47,d47_8,d47_9,1).
bond(d47,d47_9,d47_10,7).
bond(d47,d47_10,d47_4,1).
bond(d47,d47_10,d47_11,7).
bond(d47,d47_11,d47_12,1).
bond(d47,d47_12,d47_13,1).
bond(d47,d47_13,d47_5,1).
bond(d47,d47_9,d47_14,7).
bond(d47,d47_14,d47_15,7).
bond(d47,d47_15,d47_16,7).
bond(d47,d47_16,d47_11,7).
bond(d47,d47_15,d47_17,1).
bond(d47,d47_1,d47_18,1).
bond(d47,d47_19,d47_17,2).
bond(d47,d47_17,d47_20,2).
bond(d47,d47_21,d47_18,2).
bond(d47,d47_18,d47_22,2).
bond(d47,d47_2,d47_23,1).
bond(d47,d47_6,d47_24,1).
bond(d47,d47_7,d47_25,1).
bond(d47,d47_7,d47_26,1).
bond(d47,d47_8,d47_27,1).
bond(d47,d47_8,d47_28,1).
bond(d47,d47_14,d47_29,1).
bond(d47,d47_16,d47_30,1).
bond(d47,d47_12,d47_31,1).
bond(d47,d47_12,d47_32,1).
bond(d47,d47_13,d47_33,1).
bond(d47,d47_13,d47_34,1).
atm(d48,d48_1,c,22,-0.125).
atm(d48,d48_2,c,22,-0.125).
atm(d48,d48_3,c,27,0.005).
atm(d48,d48_4,c,27,0.005).
atm(d48,d48_5,c,22,-0.125).
atm(d48,d48_6,c,22,-0.125).
atm(d48,d48_7,h,3,0.135).
atm(d48,d48_8,h,3,0.135).
atm(d48,d48_9,h,3,0.135).
atm(d48,d48_10,h,3,0.135).
atm(d48,d48_11,c,22,-0.125).
atm(d48,d48_12,c,27,0.005).
atm(d48,d48_13,c,27,0.005).
atm(d48,d48_14,c,22,-0.125).
atm(d48,d48_15,h,3,0.135).
atm(d48,d48_16,h,3,0.135).
atm(d48,d48_17,c,22,-0.125).
atm(d48,d48_18,c,22,-0.125).
atm(d48,d48_19,c,22,-0.125).
atm(d48,d48_20,c,22,-0.125).
atm(d48,d48_21,h,3,0.135).
atm(d48,d48_22,h,3,0.135).
atm(d48,d48_23,h,3,0.135).
atm(d48,d48_24,n,38,0.805).
atm(d48,d48_25,o,40,-0.395).
atm(d48,d48_26,o,40,-0.395).

bond(d48,d48_1,d48_2,7).
bond(d48,d48_2,d48_3,7).
bond(d48,d48_3,d48_4,7).
bond(d48,d48_4,d48_5,7).
bond(d48,d48_5,d48_6,7).
bond(d48,d48_6,d48_1,7).
bond(d48,d48_1,d48_7,1).
bond(d48,d48_2,d48_8,1).
bond(d48,d48_5,d48_9,1).
bond(d48,d48_6,d48_10,1).
bond(d48,d48_3,d48_11,7).
bond(d48,d48_11,d48_12,7).
bond(d48,d48_12,d48_13,7).
bond(d48,d48_13,d48_14,7).
bond(d48,d48_14,d48_4,7).
bond(d48,d48_11,d48_15,1).
bond(d48,d48_14,d48_16,1).
bond(d48,d48_12,d48_17,7).
bond(d48,d48_17,d48_18,7).
bond(d48,d48_18,d48_19,7).
bond(d48,d48_19,d48_20,7).
bond(d48,d48_20,d48_13,7).
bond(d48,d48_17,d48_21,1).
bond(d48,d48_18,d48_22,1).
bond(d48,d48_20,d48_23,1).
bond(d48,d48_19,d48_24,1).
bond(d48,d48_25,d48_24,2).
bond(d48,d48_24,d48_26,2).
atm(d49,d49_1,c,22,-0.142).
atm(d49,d49_2,c,22,-0.142).
atm(d49,d49_3,c,22,0.188).
atm(d49,d49_4,c,22,0.188).
atm(d49,d49_5,c,22,-0.142).
atm(d49,d49_6,c,22,-0.142).
atm(d49,d49_7,h,3,0.118).
atm(d49,d49_8,h,3,0.118).
atm(d49,d49_9,h,3,0.118).
atm(d49,d49_10,c,22,0.188).
atm(d49,d49_11,c,22,-0.142).
atm(d49,d49_12,c,22,-0.142).
atm(d49,d49_13,c,22,-0.142).
atm(d49,d49_14,c,22,-0.142).
atm(d49,d49_15,c,22,0.188).
atm(d49,d49_16,h,3,0.118).
atm(d49,d49_17,h,3,0.118).
atm(d49,d49_18,h,3,0.118).
atm(d49,d49_19,n,32,-0.412).
atm(d49,d49_20,n,32,-0.412).
atm(d49,d49_21,h,1,0.288).
atm(d49,d49_22,h,1,0.288).
atm(d49,d49_23,n,38,0.787).
atm(d49,d49_24,n,38,0.787).
atm(d49,d49_25,o,40,-0.413).
atm(d49,d49_26,o,40,-0.413).
atm(d49,d49_27,o,40,-0.412).
atm(d49,d49_28,o,40,-0.412).

bond(d49,d49_1,d49_2,7).
bond(d49,d49_2,d49_3,7).
bond(d49,d49_3,d49_4,7).
bond(d49,d49_4,d49_5,7).
bond(d49,d49_5,d49_6,7).
bond(d49,d49_6,d49_1,7).
bond(d49,d49_1,d49_7,1).
bond(d49,d49_2,d49_8,1).
bond(d49,d49_5,d49_9,1).
bond(d49,d49_10,d49_11,7).
bond(d49,d49_11,d49_12,7).
bond(d49,d49_12,d49_13,7).
bond(d49,d49_13,d49_14,7).
bond(d49,d49_14,d49_15,7).
bond(d49,d49_15,d49_10,7).
bond(d49,d49_11,d49_16,1).
bond(d49,d49_12,d49_17,1).
bond(d49,d49_14,d49_18,1).
bond(d49,d49_3,d49_19,1).
bond(d49,d49_19,d49_10,1).
bond(d49,d49_15,d49_20,1).
bond(d49,d49_20,d49_4,1).
bond(d49,d49_19,d49_21,1).
bond(d49,d49_20,d49_22,1).
bond(d49,d49_23,d49_13,1).
bond(d49,d49_24,d49_6,1).
bond(d49,d49_25,d49_23,2).
bond(d49,d49_23,d49_26,2).
bond(d49,d49_27,d49_24,2).
bond(d49,d49_24,d49_28,2).
atm(d5,d5_1,c,22,0.309).
atm(d5,d5_2,c,22,-0.121).
atm(d5,d5_3,c,22,-0.121).
atm(d5,d5_4,c,22,0.31).
atm(d5,d5_5,c,22,-0.121).
atm(d5,d5_6,c,22,-0.121).
atm(d5,d5_7,h,3,0.139).
atm(d5,d5_8,h,3,0.139).
atm(d5,d5_9,h,3,0.139).
atm(d5,d5_10,n,38,0.81).
atm(d5,d5_11,f,92,-0.291).
atm(d5,d5_12,f,92,-0.291).
atm(d5,d5_13,o,40,-0.39).
atm(d5,d5_14,o,40,-0.39).

bond(d5,d5_1,d5_2,7).
bond(d5,d5_2,d5_3,7).
bond(d5,d5_3,d5_4,7).
bond(d5,d5_4,d5_5,7).
bond(d5,d5_5,d5_6,7).
bond(d5,d5_6,d5_1,7).
bond(d5,d5_2,d5_7,1).
bond(d5,d5_3,d5_8,1).
bond(d5,d5_6,d5_9,1).
bond(d5,d5_10,d5_5,1).
bond(d5,d5_11,d5_4,1).
bond(d5,d5_12,d5_1,1).
bond(d5,d5_10,d5_13,2).
bond(d5,d5_10,d5_14,2).
atm(d50,d50_1,c,22,-0.118).
atm(d50,d50_2,c,22,-0.118).
atm(d50,d50_3,c,27,0.012).
atm(d50,d50_4,c,27,0.012).
atm(d50,d50_5,c,22,-0.118).
atm(d50,d50_6,c,22,-0.118).
atm(d50,d50_7,h,3,0.142).
atm(d50,d50_8,h,3,0.142).
atm(d50,d50_9,h,3,0.142).
atm(d50,d50_10,c,22,-0.118).
atm(d50,d50_11,c,22,-0.118).
atm(d50,d50_12,c,22,-0.118).
atm(d50,d50_13,c,22,-0.118).
atm(d50,d50_14,h,3,0.142).
atm(d50,d50_15,h,3,0.142).
atm(d50,d50_16,h,3,0.142).
atm(d50,d50_17,n,38,0.811).
atm(d50,d50_18,n,38,0.811).
atm(d50,d50_19,o,40,-0.389).
atm(d50,d50_20,o,40,-0.389).
atm(d50,d50_21,o,40,-0.388).
atm(d50,d50_22,o,40,-0.388).

bond(d50,d50_1,d50_2,7).
bond(d50,d50_2,d50_3,7).
bond(d50,d50_3,d50_4,7).
bond(d50,d50_4,d50_5,7).
bond(d50,d50_5,d50_6,7).
bond(d50,d50_6,d50_1,7).
bond(d50,d50_1,d50_7,1).
bond(d50,d50_5,d50_8,1).
bond(d50,d50_6,d50_9,1).
bond(d50,d50_3,d50_10,7).
bond(d50,d50_10,d50_11,7).
bond(d50,d50_11,d50_12,7).
bond(d50,d50_12,d50_13,7).
bond(d50,d50_13,d50_4,7).
bond(d50,d50_10,d50_14,1).
bond(d50,d50_11,d50_15,1).
bond(d50,d50_13,d50_16,1).
bond(d50,d50_12,d50_17,1).
bond(d50,d50_2,d50_18,1).
bond(d50,d50_19,d50_17,2).
bond(d50,d50_17,d50_20,2).
bond(d50,d50_18,d50_21,2).
bond(d50,d50_18,d50_22,2).
atm(d51,d51_1,c,22,-0.118).
atm(d51,d51_2,c,22,-0.118).
atm(d51,d51_3,c,27,0.012).
atm(d51,d51_4,c,27,-0.089).
atm(d51,d51_5,c,27,0.012).
atm(d51,d51_6,c,22,-0.118).
atm(d51,d51_7,c,22,-0.118).
atm(d51,d51_8,c,22,-0.118).
atm(d51,d51_9,c,27,0.012).
atm(d51,d51_10,c,27,-0.089).
atm(d51,d51_11,c,27,0.012).
atm(d51,d51_12,c,22,0.012).
atm(d51,d51_13,c,22,0.012).
atm(d51,d51_14,c,22,-0.118).
atm(d51,d51_15,c,22,-0.118).
atm(d51,d51_16,c,22,-0.118).
atm(d51,d51_17,c,10,-0.089).
atm(d51,d51_18,c,10,-0.089).
atm(d51,d51_19,c,10,-0.089).
atm(d51,d51_20,c,10,-0.089).
atm(d51,d51_21,n,38,0.811).
atm(d51,d51_22,n,38,0.811).
atm(d51,d51_23,o,40,-0.39).
atm(d51,d51_24,o,40,-0.389).
atm(d51,d51_25,o,40,-0.389).
atm(d51,d51_26,o,40,-0.39).
atm(d51,d51_27,h,3,0.141).
atm(d51,d51_28,h,3,0.141).
atm(d51,d51_29,h,3,0.141).
atm(d51,d51_30,h,3,0.141).
atm(d51,d51_31,h,3,0.141).
atm(d51,d51_32,h,3,0.141).
atm(d51,d51_33,h,3,0.062).
atm(d51,d51_34,h,3,0.062).
atm(d51,d51_35,h,3,0.062).
atm(d51,d51_36,h,3,0.062).
atm(d51,d51_37,h,3,0.062).
atm(d51,d51_38,h,3,0.062).
atm(d51,d51_39,h,3,0.062).
atm(d51,d51_40,h,3,0.062).

bond(d51,d51_1,d51_2,7).
bond(d51,d51_2,d51_3,7).
bond(d51,d51_3,d51_4,7).
bond(d51,d51_4,d51_5,7).
bond(d51,d51_5,d51_6,7).
bond(d51,d51_6,d51_1,7).
bond(d51,d51_3,d51_7,7).
bond(d51,d51_7,d51_8,7).
bond(d51,d51_8,d51_9,7).
bond(d51,d51_9,d51_10,7).
bond(d51,d51_10,d51_4,7).
bond(d51,d51_10,d51_11,7).
bond(d51,d51_11,d51_12,7).
bond(d51,d51_12,d51_13,7).
bond(d51,d51_13,d51_5,7).
bond(d51,d51_9,d51_14,7).
bond(d51,d51_14,d51_15,7).
bond(d51,d51_15,d51_16,7).
bond(d51,d51_16,d51_11,7).
bond(d51,d51_12,d51_17,1).
bond(d51,d51_17,d51_18,1).
bond(d51,d51_18,d51_19,1).
bond(d51,d51_19,d51_20,1).
bond(d51,d51_20,d51_13,1).
bond(d51,d51_21,d51_16,1).
bond(d51,d51_2,d51_22,1).
bond(d51,d51_23,d51_21,2).
bond(d51,d51_21,d51_24,2).
bond(d51,d51_22,d51_25,2).
bond(d51,d51_22,d51_26,2).
bond(d51,d51_1,d51_27,1).
bond(d51,d51_6,d51_28,1).
bond(d51,d51_7,d51_29,1).
bond(d51,d51_8,d51_30,1).
bond(d51,d51_14,d51_31,1).
bond(d51,d51_15,d51_32,1).
bond(d51,d51_17,d51_33,1).
bond(d51,d51_17,d51_34,1).
bond(d51,d51_18,d51_35,1).
bond(d51,d51_18,d51_36,1).
bond(d51,d51_19,d51_37,1).
bond(d51,d51_19,d51_38,1).
bond(d51,d51_20,d51_39,1).
bond(d51,d51_20,d51_40,1).
atm(d52,d52_1,c,22,-0.118).
atm(d52,d52_2,c,22,-0.118).
atm(d52,d52_3,c,27,0.012).
atm(d52,d52_4,c,27,-0.089).
atm(d52,d52_5,c,27,0.012).
atm(d52,d52_6,c,22,-0.118).
atm(d52,d52_7,c,22,-0.118).
atm(d52,d52_8,c,22,-0.118).
atm(d52,d52_9,c,27,0.012).
atm(d52,d52_10,c,27,-0.089).
atm(d52,d52_11,c,27,0.012).
atm(d52,d52_12,c,22,0.012).
atm(d52,d52_13,c,22,0.012).
atm(d52,d52_14,c,22,-0.118).
atm(d52,d52_15,c,22,-0.118).
atm(d52,d52_16,c,22,-0.118).
atm(d52,d52_17,c,10,-0.089).
atm(d52,d52_18,c,10,-0.089).
atm(d52,d52_19,c,10,-0.089).
atm(d52,d52_20,c,10,-0.089).
atm(d52,d52_21,n,38,0.811).
atm(d52,d52_22,o,40,-0.389).
atm(d52,d52_23,o,40,-0.39).
atm(d52,d52_24,n,38,0.811).
atm(d52,d52_25,o,40,-0.389).
atm(d52,d52_26,o,40,-0.39).
atm(d52,d52_27,h,3,0.141).
atm(d52,d52_28,h,3,0.141).
atm(d52,d52_29,h,3,0.141).
atm(d52,d52_30,h,3,0.141).
atm(d52,d52_31,h,3,0.141).
atm(d52,d52_32,h,3,0.141).
atm(d52,d52_33,h,3,0.062).
atm(d52,d52_34,h,3,0.062).
atm(d52,d52_35,h,3,0.062).
atm(d52,d52_36,h,3,0.062).
atm(d52,d52_37,h,3,0.062).
atm(d52,d52_38,h,3,0.062).
atm(d52,d52_39,h,3,0.062).
atm(d52,d52_40,h,3,0.062).

bond(d52,d52_1,d52_2,7).
bond(d52,d52_2,d52_3,7).
bond(d52,d52_3,d52_4,7).
bond(d52,d52_4,d52_5,7).
bond(d52,d52_5,d52_6,7).
bond(d52,d52_6,d52_1,7).
bond(d52,d52_3,d52_7,7).
bond(d52,d52_7,d52_8,7).
bond(d52,d52_8,d52_9,7).
bond(d52,d52_9,d52_10,7).
bond(d52,d52_10,d52_4,7).
bond(d52,d52_10,d52_11,7).
bond(d52,d52_11,d52_12,7).
bond(d52,d52_12,d52_13,7).
bond(d52,d52_13,d52_5,7).
bond(d52,d52_9,d52_14,7).
bond(d52,d52_14,d52_15,7).
bond(d52,d52_15,d52_16,7).
bond(d52,d52_16,d52_11,7).
bond(d52,d52_12,d52_17,1).
bond(d52,d52_17,d52_18,1).
bond(d52,d52_18,d52_19,1).
bond(d52,d52_19,d52_20,1).
bond(d52,d52_20,d52_13,1).
bond(d52,d52_21,d52_16,1).
bond(d52,d52_22,d52_21,2).
bond(d52,d52_21,d52_23,2).
bond(d52,d52_14,d52_24,1).
bond(d52,d52_25,d52_24,2).
bond(d52,d52_24,d52_26,2).
bond(d52,d52_1,d52_27,1).
bond(d52,d52_2,d52_28,1).
bond(d52,d52_6,d52_29,1).
bond(d52,d52_7,d52_30,1).
bond(d52,d52_8,d52_31,1).
bond(d52,d52_15,d52_32,1).
bond(d52,d52_17,d52_33,1).
bond(d52,d52_17,d52_34,1).
bond(d52,d52_18,d52_35,1).
bond(d52,d52_18,d52_36,1).
bond(d52,d52_19,d52_37,1).
bond(d52,d52_19,d52_38,1).
bond(d52,d52_20,d52_39,1).
bond(d52,d52_20,d52_40,1).
atm(d53,d53_1,c,22,-0.114).
atm(d53,d53_2,c,22,-0.114).
atm(d53,d53_3,c,22,-0.114).
atm(d53,d53_4,c,29,0.016).
atm(d53,d53_5,c,22,-0.114).
atm(d53,d53_6,c,22,-0.114).
atm(d53,d53_7,h,3,0.145).
atm(d53,d53_8,h,3,0.145).
atm(d53,d53_9,h,3,0.145).
atm(d53,d53_10,c,22,-0.114).
atm(d53,d53_11,c,22,-0.114).
atm(d53,d53_12,c,22,-0.114).
atm(d53,d53_13,c,22,-0.114).
atm(d53,d53_14,c,22,-0.114).
atm(d53,d53_15,c,29,0.016).
atm(d53,d53_16,h,3,0.145).
atm(d53,d53_17,h,3,0.145).
atm(d53,d53_18,h,3,0.145).
atm(d53,d53_19,c,10,0.016).
atm(d53,d53_20,h,3,0.066).
atm(d53,d53_21,h,3,0.066).
atm(d53,d53_22,n,38,0.815).
atm(d53,d53_23,n,38,0.815).
atm(d53,d53_24,o,40,-0.385).
atm(d53,d53_25,o,40,-0.385).
atm(d53,d53_26,o,40,-0.385).
atm(d53,d53_27,o,40,-0.385).

bond(d53,d53_1,d53_2,7).
bond(d53,d53_2,d53_3,7).
bond(d53,d53_3,d53_4,7).
bond(d53,d53_4,d53_5,7).
bond(d53,d53_5,d53_6,7).
bond(d53,d53_6,d53_1,7).
bond(d53,d53_2,d53_7,1).
bond(d53,d53_5,d53_8,1).
bond(d53,d53_6,d53_9,1).
bond(d53,d53_10,d53_11,7).
bond(d53,d53_11,d53_12,7).
bond(d53,d53_12,d53_13,7).
bond(d53,d53_13,d53_14,7).
bond(d53,d53_14,d53_15,7).
bond(d53,d53_15,d53_10,7).
bond(d53,d53_11,d53_16,1).
bond(d53,d53_13,d53_17,1).
bond(d53,d53_14,d53_18,1).
bond(d53,d53_4,d53_15,1).
bond(d53,d53_3,d53_19,1).
bond(d53,d53_19,d53_10,1).
bond(d53,d53_19,d53_20,1).
bond(d53,d53_19,d53_21,1).
bond(d53,d53_12,d53_22,1).
bond(d53,d53_1,d53_23,1).
bond(d53,d53_22,d53_24,2).
bond(d53,d53_22,d53_25,2).
bond(d53,d53_23,d53_26,2).
bond(d53,d53_23,d53_27,2).
atm(d54,d54_1,c,22,-0.108).
atm(d54,d54_2,c,22,-0.108).
atm(d54,d54_3,c,22,0.202).
atm(d54,d54_4,c,22,0.202).
atm(d54,d54_5,c,22,-0.108).
atm(d54,d54_6,c,22,-0.108).
atm(d54,d54_7,h,3,0.152).
atm(d54,d54_8,h,3,0.152).
atm(d54,d54_9,h,3,0.152).
atm(d54,d54_10,c,22,0.202).
atm(d54,d54_11,c,22,-0.108).
atm(d54,d54_12,c,22,-0.108).
atm(d54,d54_13,c,22,-0.108).
atm(d54,d54_14,c,22,-0.108).
atm(d54,d54_15,c,22,0.202).
atm(d54,d54_16,h,3,0.152).
atm(d54,d54_17,h,3,0.152).
atm(d54,d54_18,o,50,-0.208).
atm(d54,d54_19,o,50,-0.208).
atm(d54,d54_20,n,38,0.823).
atm(d54,d54_21,o,40,-0.377).
atm(d54,d54_22,o,40,-0.378).
atm(d54,d54_23,cl,93,-0.178).
atm(d54,d54_24,cl,93,-0.178).

bond(d54,d54_1,d54_2,7).
bond(d54,d54_2,d54_3,7).
bond(d54,d54_3,d54_4,7).
bond(d54,d54_4,d54_5,7).
bond(d54,d54_5,d54_6,7).
bond(d54,d54_6,d54_1,7).
bond(d54,d54_2,d54_7,1).
bond(d54,d54_5,d54_8,1).
bond(d54,d54_6,d54_9,1).
bond(d54,d54_10,d54_11,7).
bond(d54,d54_11,d54_12,7).
bond(d54,d54_12,d54_13,7).
bond(d54,d54_13,d54_14,7).
bond(d54,d54_14,d54_15,7).
bond(d54,d54_15,d54_10,7).
bond(d54,d54_11,d54_16,1).
bond(d54,d54_14,d54_17,1).
bond(d54,d54_3,d54_18,1).
bond(d54,d54_18,d54_10,1).
bond(d54,d54_15,d54_19,1).
bond(d54,d54_19,d54_4,1).
bond(d54,d54_1,d54_20,1).
bond(d54,d54_21,d54_20,2).
bond(d54,d54_20,d54_22,2).
bond(d54,d54_13,d54_23,1).
bond(d54,d54_12,d54_24,1).
atm(d55,d55_1,c,22,-0.123).
atm(d55,d55_2,c,22,-0.123).
atm(d55,d55_3,c,27,0.007).
atm(d55,d55_4,c,27,0.007).
atm(d55,d55_5,c,22,-0.123).
atm(d55,d55_6,c,22,-0.123).
atm(d55,d55_7,h,3,0.137).
atm(d55,d55_8,h,3,0.137).
atm(d55,d55_9,h,3,0.137).
atm(d55,d55_10,h,3,0.137).
atm(d55,d55_11,c,22,-0.123).
atm(d55,d55_12,c,22,-0.123).
atm(d55,d55_13,c,22,-0.123).
atm(d55,d55_14,c,22,-0.173).
atm(d55,d55_15,h,3,0.137).
atm(d55,d55_16,h,3,0.137).
atm(d55,d55_17,c,10,0.007).
atm(d55,d55_18,h,3,0.057).
atm(d55,d55_19,h,3,0.057).
atm(d55,d55_20,h,3,0.057).
atm(d55,d55_21,n,38,0.806).
atm(d55,d55_22,o,40,-0.393).
atm(d55,d55_23,o,40,-0.393).

bond(d55,d55_1,d55_2,7).
bond(d55,d55_2,d55_3,7).
bond(d55,d55_3,d55_4,7).
bond(d55,d55_4,d55_5,7).
bond(d55,d55_5,d55_6,7).
bond(d55,d55_6,d55_1,7).
bond(d55,d55_1,d55_7,1).
bond(d55,d55_2,d55_8,1).
bond(d55,d55_5,d55_9,1).
bond(d55,d55_6,d55_10,1).
bond(d55,d55_3,d55_11,7).
bond(d55,d55_11,d55_12,7).
bond(d55,d55_12,d55_13,7).
bond(d55,d55_13,d55_14,7).
bond(d55,d55_14,d55_4,7).
bond(d55,d55_11,d55_15,1).
bond(d55,d55_12,d55_16,1).
bond(d55,d55_17,d55_14,1).
bond(d55,d55_17,d55_18,1).
bond(d55,d55_17,d55_19,1).
bond(d55,d55_17,d55_20,1).
bond(d55,d55_13,d55_21,1).
bond(d55,d55_22,d55_21,2).
bond(d55,d55_21,d55_23,2).
atm(d56,d56_1,c,22,-0.148).
atm(d56,d56_2,c,22,-0.148).
atm(d56,d56_3,c,22,0.182).
atm(d56,d56_4,c,22,0.182).
atm(d56,d56_5,c,22,-0.148).
atm(d56,d56_6,c,22,-0.148).
atm(d56,d56_7,h,3,0.112).
atm(d56,d56_8,h,3,0.112).
atm(d56,d56_9,h,3,0.112).
atm(d56,d56_10,h,3,0.112).
atm(d56,d56_11,c,22,0.182).
atm(d56,d56_12,c,22,-0.148).
atm(d56,d56_13,c,22,-0.148).
atm(d56,d56_14,c,22,-0.148).
atm(d56,d56_15,c,22,-0.148).
atm(d56,d56_16,c,22,0.182).
atm(d56,d56_17,h,3,0.112).
atm(d56,d56_18,h,3,0.112).
atm(d56,d56_19,h,3,0.112).
atm(d56,d56_20,n,32,-0.419).
atm(d56,d56_21,n,32,-0.418).
atm(d56,d56_22,h,1,0.282).
atm(d56,d56_23,h,1,0.282).
atm(d56,d56_24,n,38,0.781).
atm(d56,d56_25,o,40,-0.418).
atm(d56,d56_26,o,40,-0.418).

bond(d56,d56_1,d56_2,7).
bond(d56,d56_2,d56_3,7).
bond(d56,d56_3,d56_4,7).
bond(d56,d56_4,d56_5,7).
bond(d56,d56_5,d56_6,7).
bond(d56,d56_6,d56_1,7).
bond(d56,d56_1,d56_7,1).
bond(d56,d56_2,d56_8,1).
bond(d56,d56_5,d56_9,1).
bond(d56,d56_6,d56_10,1).
bond(d56,d56_11,d56_12,7).
bond(d56,d56_12,d56_13,7).
bond(d56,d56_13,d56_14,7).
bond(d56,d56_14,d56_15,7).
bond(d56,d56_15,d56_16,7).
bond(d56,d56_16,d56_11,7).
bond(d56,d56_12,d56_17,1).
bond(d56,d56_13,d56_18,1).
bond(d56,d56_15,d56_19,1).
bond(d56,d56_3,d56_20,1).
bond(d56,d56_20,d56_11,1).
bond(d56,d56_16,d56_21,1).
bond(d56,d56_21,d56_4,1).
bond(d56,d56_20,d56_22,1).
bond(d56,d56_21,d56_23,1).
bond(d56,d56_14,d56_24,1).
bond(d56,d56_25,d56_24,2).
bond(d56,d56_24,d56_26,2).
atm(d57,d57_1,c,22,-0.116).
atm(d57,d57_2,c,22,-0.116).
atm(d57,d57_3,c,22,-0.116).
atm(d57,d57_4,c,29,0.014).
atm(d57,d57_5,c,22,-0.116).
atm(d57,d57_6,c,22,-0.116).
atm(d57,d57_7,h,3,0.144).
atm(d57,d57_8,h,3,0.144).
atm(d57,d57_9,h,3,0.144).
atm(d57,d57_10,h,3,0.144).
atm(d57,d57_11,c,29,0.014).
atm(d57,d57_12,c,22,-0.116).
atm(d57,d57_13,c,22,-0.116).
atm(d57,d57_14,c,22,-0.116).
atm(d57,d57_15,c,22,-0.116).
atm(d57,d57_16,c,22,-0.116).
atm(d57,d57_17,h,3,0.144).
atm(d57,d57_18,h,3,0.144).
atm(d57,d57_19,h,3,0.144).
atm(d57,d57_20,n,38,0.813).
atm(d57,d57_21,n,38,0.813).
atm(d57,d57_22,n,38,0.814).
atm(d57,d57_23,o,40,-0.386).
atm(d57,d57_24,o,40,-0.386).
atm(d57,d57_25,o,40,-0.386).
atm(d57,d57_26,o,40,-0.386).
atm(d57,d57_27,o,40,-0.386).
atm(d57,d57_28,o,40,-0.386).

bond(d57,d57_1,d57_2,7).
bond(d57,d57_2,d57_3,7).
bond(d57,d57_3,d57_4,7).
bond(d57,d57_4,d57_5,7).
bond(d57,d57_5,d57_6,7).
bond(d57,d57_6,d57_1,7).
bond(d57,d57_2,d57_7,1).
bond(d57,d57_3,d57_8,1).
bond(d57,d57_5,d57_9,1).
bond(d57,d57_6,d57_10,1).
bond(d57,d57_11,d57_12,7).
bond(d57,d57_12,d57_13,7).
bond(d57,d57_13,d57_14,7).
bond(d57,d57_14,d57_15,7).
bond(d57,d57_15,d57_16,7).
bond(d57,d57_16,d57_11,7).
bond(d57,d57_12,d57_17,1).
bond(d57,d57_13,d57_18,1).
bond(d57,d57_15,d57_19,1).
bond(d57,d57_11,d57_4,1).
bond(d57,d57_20,d57_16,1).
bond(d57,d57_14,d57_21,1).
bond(d57,d57_1,d57_22,1).
bond(d57,d57_23,d57_20,2).
bond(d57,d57_20,d57_24,2).
bond(d57,d57_25,d57_21,2).
bond(d57,d57_21,d57_26,2).
bond(d57,d57_27,d57_22,2).
bond(d57,d57_22,d57_28,2).
atm(d58,d58_1,c,22,-0.099).
atm(d58,d58_2,c,22,-0.099).
atm(d58,d58_3,c,27,0.031).
atm(d58,d58_4,c,195,-0.069).
atm(d58,d58_5,c,22,-0.099).
atm(d58,d58_6,c,22,-0.099).
atm(d58,d58_7,c,22,-0.099).
atm(d58,d58_8,c,22,-0.099).
atm(d58,d58_9,c,27,0.031).
atm(d58,d58_10,c,195,-0.069).
atm(d58,d58_11,c,22,-0.099).
atm(d58,d58_12,c,10,0.031).
atm(d58,d58_13,c,10,0.031).
atm(d58,d58_14,c,22,-0.099).
atm(d58,d58_15,c,22,-0.099).
atm(d58,d58_16,c,22,-0.099).
atm(d58,d58_17,n,38,0.83).
atm(d58,d58_18,o,40,-0.369).
atm(d58,d58_19,o,40,-0.369).
atm(d58,d58_20,h,3,0.161).
atm(d58,d58_21,h,3,0.161).
atm(d58,d58_22,h,3,0.161).
atm(d58,d58_23,h,3,0.161).
atm(d58,d58_24,h,3,0.161).
atm(d58,d58_25,h,3,0.161).
atm(d58,d58_26,h,3,0.161).
atm(d58,d58_27,o,45,-0.62).
atm(d58,d58_28,o,45,-0.619).
atm(d58,d58_29,h,8,0.431).
atm(d58,d58_30,h,8,0.431).
atm(d58,d58_31,h,3,0.081).
atm(d58,d58_32,h,3,0.081).

bond(d58,d58_1,d58_2,7).
bond(d58,d58_2,d58_3,7).
bond(d58,d58_3,d58_4,7).
bond(d58,d58_4,d58_5,7).
bond(d58,d58_5,d58_6,7).
bond(d58,d58_6,d58_1,7).
bond(d58,d58_3,d58_7,7).
bond(d58,d58_7,d58_8,7).
bond(d58,d58_8,d58_9,7).
bond(d58,d58_9,d58_10,7).
bond(d58,d58_10,d58_4,7).
bond(d58,d58_10,d58_11,7).
bond(d58,d58_11,d58_12,1).
bond(d58,d58_12,d58_13,1).
bond(d58,d58_13,d58_5,1).
bond(d58,d58_9,d58_14,7).
bond(d58,d58_14,d58_15,7).
bond(d58,d58_15,d58_16,7).
bond(d58,d58_16,d58_11,7).
bond(d58,d58_14,d58_17,1).
bond(d58,d58_17,d58_18,2).
bond(d58,d58_17,d58_19,2).
bond(d58,d58_1,d58_20,1).
bond(d58,d58_2,d58_21,1).
bond(d58,d58_6,d58_22,1).
bond(d58,d58_7,d58_23,1).
bond(d58,d58_8,d58_24,1).
bond(d58,d58_15,d58_25,1).
bond(d58,d58_16,d58_26,1).
bond(d58,d58_12,d58_27,4).
bond(d58,d58_13,d58_28,5).
bond(d58,d58_27,d58_29,1).
bond(d58,d58_28,d58_30,1).
bond(d58,d58_12,d58_31,1).
bond(d58,d58_13,d58_32,1).
atm(d59,d59_1,c,22,-0.115).
atm(d59,d59_2,c,27,0.015).
atm(d59,d59_3,c,27,-0.085).
atm(d59,d59_4,c,22,-0.115).
atm(d59,d59_5,c,22,-0.115).
atm(d59,d59_6,c,22,-0.115).
atm(d59,d59_7,c,22,-0.115).
atm(d59,d59_8,c,22,-0.115).
atm(d59,d59_9,c,27,0.015).
atm(d59,d59_10,c,27,-0.085).
atm(d59,d59_11,c,22,-0.115).
atm(d59,d59_12,c,27,0.015).
atm(d59,d59_13,c,27,-0.085).
atm(d59,d59_14,c,25,-0.085).
atm(d59,d59_15,c,22,-0.115).
atm(d59,d59_16,c,22,-0.115).
atm(d59,d59_17,c,22,-0.115).
atm(d59,d59_18,c,26,-0.085).
atm(d59,d59_19,c,21,-0.115).
atm(d59,d59_20,c,21,0.015).
atm(d59,d59_21,h,3,0.144).
atm(d59,d59_22,h,3,0.144).
atm(d59,d59_23,h,3,0.144).
atm(d59,d59_24,h,3,0.145).
atm(d59,d59_25,h,3,0.145).
atm(d59,d59_26,h,3,0.145).
atm(d59,d59_27,h,3,0.145).
atm(d59,d59_28,h,3,0.144).
atm(d59,d59_29,h,3,0.144).
atm(d59,d59_30,h,3,0.144).
atm(d59,d59_31,h,3,0.144).
atm(d59,d59_32,n,38,0.814).
atm(d59,d59_33,o,40,-0.386).
atm(d59,d59_34,o,40,-0.386).

bond(d59,d59_1,d59_2,7).
bond(d59,d59_2,d59_3,7).
bond(d59,d59_3,d59_4,7).
bond(d59,d59_4,d59_5,7).
bond(d59,d59_5,d59_6,7).
bond(d59,d59_6,d59_1,7).
bond(d59,d59_2,d59_7,7).
bond(d59,d59_7,d59_8,7).
bond(d59,d59_8,d59_9,7).
bond(d59,d59_9,d59_10,7).
bond(d59,d59_10,d59_3,7).
bond(d59,d59_9,d59_11,7).
bond(d59,d59_11,d59_12,7).
bond(d59,d59_12,d59_13,7).
bond(d59,d59_13,d59_14,7).
bond(d59,d59_14,d59_10,7).
bond(d59,d59_12,d59_15,7).
bond(d59,d59_15,d59_16,7).
bond(d59,d59_16,d59_17,7).
bond(d59,d59_17,d59_18,7).
bond(d59,d59_18,d59_13,7).
bond(d59,d59_19,d59_14,7).
bond(d59,d59_19,d59_20,7).
bond(d59,d59_20,d59_18,7).
bond(d59,d59_1,d59_21,1).
bond(d59,d59_4,d59_22,1).
bond(d59,d59_5,d59_23,1).
bond(d59,d59_6,d59_24,1).
bond(d59,d59_7,d59_25,1).
bond(d59,d59_8,d59_26,1).
bond(d59,d59_11,d59_27,1).
bond(d59,d59_15,d59_28,1).
bond(d59,d59_16,d59_29,1).
bond(d59,d59_17,d59_30,1).
bond(d59,d59_19,d59_31,1).
bond(d59,d59_32,d59_20,1).
bond(d59,d59_33,d59_32,2).
bond(d59,d59_32,d59_34,2).
atm(d6,d6_1,c,22,-0.109).
atm(d6,d6_2,c,22,-0.109).
atm(d6,d6_3,c,27,0.021).
atm(d6,d6_4,c,27,-0.079).
atm(d6,d6_5,c,27,0.021).
atm(d6,d6_6,c,22,-0.109).
atm(d6,d6_7,c,27,-0.079).
atm(d6,d6_8,c,27,0.021).
atm(d6,d6_9,c,22,-0.109).
atm(d6,d6_10,c,22,-0.109).
atm(d6,d6_11,c,22,-0.109).
atm(d6,d6_12,c,22,-0.109).
atm(d6,d6_13,c,27,0.021).
atm(d6,d6_14,c,22,-0.109).
atm(d6,d6_15,c,22,-0.109).
atm(d6,d6_16,c,22,-0.109).
atm(d6,d6_17,h,3,0.151).
atm(d6,d6_18,h,3,0.151).
atm(d6,d6_19,h,3,0.151).
atm(d6,d6_20,h,3,0.151).
atm(d6,d6_21,h,3,0.151).
atm(d6,d6_22,h,3,0.151).
atm(d6,d6_23,n,38,0.822).
atm(d6,d6_24,n,38,0.822).
atm(d6,d6_25,o,40,-0.379).
atm(d6,d6_26,o,40,-0.379).
atm(d6,d6_27,o,40,-0.379).
atm(d6,d6_28,o,40,-0.379).
atm(d6,d6_29,o,40,-0.378).
atm(d6,d6_30,o,40,-0.379).
atm(d6,d6_31,o,40,-0.379).
atm(d6,d6_32,o,40,-0.378).
atm(d6,d6_33,n,38,0.822).
atm(d6,d6_34,n,38,0.822).

bond(d6,d6_1,d6_2,7).
bond(d6,d6_2,d6_3,7).
bond(d6,d6_3,d6_4,7).
bond(d6,d6_4,d6_5,7).
bond(d6,d6_5,d6_6,7).
bond(d6,d6_6,d6_1,7).
bond(d6,d6_4,d6_7,7).
bond(d6,d6_7,d6_8,7).
bond(d6,d6_8,d6_9,7).
bond(d6,d6_9,d6_10,7).
bond(d6,d6_10,d6_5,7).
bond(d6,d6_3,d6_11,7).
bond(d6,d6_11,d6_12,7).
bond(d6,d6_12,d6_13,7).
bond(d6,d6_13,d6_7,7).
bond(d6,d6_14,d6_15,7).
bond(d6,d6_15,d6_8,7).
bond(d6,d6_16,d6_13,7).
bond(d6,d6_14,d6_16,7).
bond(d6,d6_1,d6_17,1).
bond(d6,d6_9,d6_18,1).
bond(d6,d6_10,d6_19,1).
bond(d6,d6_11,d6_20,1).
bond(d6,d6_12,d6_21,1).
bond(d6,d6_14,d6_22,1).
bond(d6,d6_23,d6_16,1).
bond(d6,d6_24,d6_2,1).
bond(d6,d6_27,d6_23,2).
bond(d6,d6_23,d6_28,2).
bond(d6,d6_24,d6_29,2).
bond(d6,d6_24,d6_30,2).
bond(d6,d6_6,d6_33,1).
bond(d6,d6_32,d6_33,2).
bond(d6,d6_31,d6_33,2).
bond(d6,d6_15,d6_34,1).
bond(d6,d6_26,d6_34,2).
bond(d6,d6_25,d6_34,2).
atm(d60,d60_1,c,22,-0.114).
atm(d60,d60_2,c,22,-0.114).
atm(d60,d60_3,c,27,0.016).
atm(d60,d60_4,c,27,0.016).
atm(d60,d60_5,c,22,-0.114).
atm(d60,d60_6,c,22,-0.114).
atm(d60,d60_7,h,3,0.146).
atm(d60,d60_8,h,3,0.146).
atm(d60,d60_9,h,3,0.146).
atm(d60,d60_10,c,22,-0.114).
atm(d60,d60_11,c,22,-0.114).
atm(d60,d60_12,c,22,-0.114).
atm(d60,d60_13,c,22,-0.114).
atm(d60,d60_14,h,3,0.146).
atm(d60,d60_15,h,3,0.146).
atm(d60,d60_16,n,38,0.817).
atm(d60,d60_17,n,38,0.817).
atm(d60,d60_18,n,38,0.817).
atm(d60,d60_19,o,40,-0.384).
atm(d60,d60_20,o,40,-0.384).
atm(d60,d60_21,o,40,-0.383).
atm(d60,d60_22,o,40,-0.384).
atm(d60,d60_23,o,40,-0.383).
atm(d60,d60_24,o,40,-0.383).

bond(d60,d60_1,d60_2,7).
bond(d60,d60_2,d60_3,7).
bond(d60,d60_3,d60_4,7).
bond(d60,d60_4,d60_5,7).
bond(d60,d60_5,d60_6,7).
bond(d60,d60_6,d60_1,7).
bond(d60,d60_1,d60_7,1).
bond(d60,d60_2,d60_8,1).
bond(d60,d60_6,d60_9,1).
bond(d60,d60_3,d60_10,7).
bond(d60,d60_10,d60_11,7).
bond(d60,d60_11,d60_12,7).
bond(d60,d60_12,d60_13,7).
bond(d60,d60_13,d60_4,7).
bond(d60,d60_10,d60_14,1).
bond(d60,d60_12,d60_15,1).
bond(d60,d60_16,d60_13,1).
bond(d60,d60_11,d60_17,1).
bond(d60,d60_18,d60_5,1).
bond(d60,d60_19,d60_18,2).
bond(d60,d60_18,d60_20,2).
bond(d60,d60_21,d60_16,2).
bond(d60,d60_16,d60_22,2).
bond(d60,d60_23,d60_17,2).
bond(d60,d60_17,d60_24,2).
atm(d61,d61_1,c,22,-0.105).
atm(d61,d61_2,c,22,-0.105).
atm(d61,d61_3,c,22,-0.105).
atm(d61,d61_4,c,29,0.025).
atm(d61,d61_5,c,22,-0.105).
atm(d61,d61_6,c,22,-0.105).
atm(d61,d61_7,h,3,0.155).
atm(d61,d61_8,h,3,0.154).
atm(d61,d61_9,h,3,0.155).
atm(d61,d61_10,c,22,-0.105).
atm(d61,d61_11,c,22,-0.105).
atm(d61,d61_12,c,22,-0.105).
atm(d61,d61_13,c,22,-0.105).
atm(d61,d61_14,c,22,-0.105).
atm(d61,d61_15,c,29,0.025).
atm(d61,d61_16,h,3,0.155).
atm(d61,d61_17,h,3,0.155).
atm(d61,d61_18,h,3,0.155).
atm(d61,d61_19,c,10,0.025).
atm(d61,d61_20,h,3,0.075).
atm(d61,d61_21,h,3,0.075).
atm(d61,d61_22,cl,93,-0.176).
atm(d61,d61_23,n,38,0.824).
atm(d61,d61_24,o,40,-0.376).
atm(d61,d61_25,o,40,-0.376).

bond(d61,d61_1,d61_2,7).
bond(d61,d61_2,d61_3,7).
bond(d61,d61_3,d61_4,7).
bond(d61,d61_4,d61_5,7).
bond(d61,d61_5,d61_6,7).
bond(d61,d61_6,d61_1,7).
bond(d61,d61_2,d61_7,1).
bond(d61,d61_5,d61_8,1).
bond(d61,d61_6,d61_9,1).
bond(d61,d61_10,d61_11,7).
bond(d61,d61_11,d61_12,7).
bond(d61,d61_12,d61_13,7).
bond(d61,d61_13,d61_14,7).
bond(d61,d61_14,d61_15,7).
bond(d61,d61_15,d61_10,7).
bond(d61,d61_11,d61_16,1).
bond(d61,d61_13,d61_17,1).
bond(d61,d61_14,d61_18,1).
bond(d61,d61_4,d61_15,1).
bond(d61,d61_3,d61_19,1).
bond(d61,d61_19,d61_10,1).
bond(d61,d61_19,d61_20,1).
bond(d61,d61_19,d61_21,1).
bond(d61,d61_12,d61_22,1).
bond(d61,d61_1,d61_23,1).
bond(d61,d61_24,d61_23,2).
bond(d61,d61_23,d61_25,2).
atm(d62,d62_1,c,22,-0.127).
atm(d62,d62_2,c,22,-0.127).
atm(d62,d62_3,c,22,-0.127).
atm(d62,d62_4,c,22,0.203).
atm(d62,d62_5,c,22,0.253).
atm(d62,d62_6,c,22,-0.127).
atm(d62,d62_7,h,3,0.133).
atm(d62,d62_8,h,3,0.133).
atm(d62,d62_9,h,3,0.133).
atm(d62,d62_10,o,45,-0.647).
atm(d62,d62_11,h,8,0.403).
atm(d62,d62_12,n,32,-0.777).
atm(d62,d62_13,n,38,0.802).
atm(d62,d62_14,o,40,-0.397).
atm(d62,d62_15,o,40,-0.397).
atm(d62,d62_16,h,1,0.333).
atm(d62,d62_17,h,1,0.333).

bond(d62,d62_1,d62_2,7).
bond(d62,d62_2,d62_3,7).
bond(d62,d62_3,d62_4,7).
bond(d62,d62_4,d62_5,7).
bond(d62,d62_5,d62_6,7).
bond(d62,d62_6,d62_1,7).
bond(d62,d62_2,d62_7,1).
bond(d62,d62_3,d62_8,1).
bond(d62,d62_6,d62_9,1).
bond(d62,d62_10,d62_5,1).
bond(d62,d62_10,d62_11,1).
bond(d62,d62_4,d62_12,1).
bond(d62,d62_1,d62_13,1).
bond(d62,d62_14,d62_13,2).
bond(d62,d62_13,d62_15,2).
bond(d62,d62_12,d62_16,1).
bond(d62,d62_12,d62_17,1).
atm(d63,d63_1,c,21,-0.004).
atm(d63,d63_2,c,21,-0.134).
atm(d63,d63_3,c,21,-0.134).
atm(d63,d63_4,c,21,0.096).
atm(d63,d63_5,o,52,-0.034).
atm(d63,d63_6,n,38,0.797).
atm(d63,d63_7,o,40,-0.403).
atm(d63,d63_8,o,40,-0.403).
atm(d63,d63_9,c,14,0.597).
atm(d63,d63_10,n,32,-0.404).
atm(d63,d63_11,n,32,-0.404).
atm(d63,d63_12,c,14,0.597).
atm(d63,d63_13,c,16,-0.204).
atm(d63,d63_14,o,40,-0.553).
atm(d63,d63_15,c,16,-0.204).
atm(d63,d63_16,c,21,0.096).
atm(d63,d63_17,c,21,-0.134).
atm(d63,d63_18,c,21,-0.134).
atm(d63,d63_19,c,21,-0.004).
atm(d63,d63_20,c,21,-0.134).
atm(d63,d63_21,n,38,0.797).
atm(d63,d63_22,o,40,-0.404).
atm(d63,d63_23,o,40,-0.403).
atm(d63,d63_24,h,3,0.126).
atm(d63,d63_25,h,3,0.126).
atm(d63,d63_26,h,3,0.046).
atm(d63,d63_27,h,1,0.246).
atm(d63,d63_28,h,3,0.096).
atm(d63,d63_29,h,3,0.126).
atm(d63,d63_30,h,3,0.126).
atm(d63,d63_31,h,3,0.126).
atm(d63,d63_32,h,3,0.096).

bond(d63,d63_1,d63_2,7).
bond(d63,d63_2,d63_3,7).
bond(d63,d63_3,d63_4,7).
bond(d63,d63_4,d63_5,7).
bond(d63,d63_5,d63_1,7).
bond(d63,d63_1,d63_6,1).
bond(d63,d63_6,d63_7,2).
bond(d63,d63_6,d63_8,2).
bond(d63,d63_4,d63_9,1).
bond(d63,d63_9,d63_10,2).
bond(d63,d63_10,d63_11,1).
bond(d63,d63_11,d63_12,1).
bond(d63,d63_12,d63_13,1).
bond(d63,d63_12,d63_14,2).
bond(d63,d63_13,d63_15,2).
bond(d63,d63_16,d63_17,7).
bond(d63,d63_17,d63_18,7).
bond(d63,d63_18,d63_19,7).
bond(d63,d63_19,d63_20,7).
bond(d63,d63_20,d63_16,7).
bond(d63,d63_19,d63_21,1).
bond(d63,d63_21,d63_22,2).
bond(d63,d63_21,d63_23,2).
bond(d63,d63_2,d63_24,1).
bond(d63,d63_3,d63_25,1).
bond(d63,d63_9,d63_26,1).
bond(d63,d63_11,d63_27,1).
bond(d63,d63_13,d63_28,1).
bond(d63,d63_17,d63_29,1).
bond(d63,d63_18,d63_30,1).
bond(d63,d63_20,d63_31,1).
bond(d63,d63_15,d63_16,1).
bond(d63,d63_15,d63_32,1).
atm(d64,d64_1,c,22,-0.128).
atm(d64,d64_2,c,22,-0.128).
atm(d64,d64_3,c,22,0.102).
atm(d64,d64_4,c,29,0.002).
atm(d64,d64_5,c,22,-0.128).
atm(d64,d64_6,c,22,-0.128).
atm(d64,d64_7,h,3,0.132).
atm(d64,d64_8,h,3,0.132).
atm(d64,d64_9,h,3,0.132).
atm(d64,d64_10,c,22,0.102).
atm(d64,d64_11,c,22,-0.128).
atm(d64,d64_12,c,22,-0.128).
atm(d64,d64_13,c,22,-0.128).
atm(d64,d64_14,c,22,-0.128).
atm(d64,d64_15,c,29,0.002).
atm(d64,d64_16,h,3,0.132).
atm(d64,d64_17,h,3,0.132).
atm(d64,d64_18,h,3,0.132).
atm(d64,d64_19,c,14,0.553).
atm(d64,d64_20,n,38,0.803).
atm(d64,d64_21,n,38,0.803).
atm(d64,d64_22,o,40,-0.397).
atm(d64,d64_23,o,40,-0.397).
atm(d64,d64_24,o,40,-0.397).
atm(d64,d64_25,o,40,-0.397).
atm(d64,d64_26,o,40,-0.547).

bond(d64,d64_1,d64_2,7).
bond(d64,d64_2,d64_3,7).
bond(d64,d64_3,d64_4,7).
bond(d64,d64_4,d64_5,7).
bond(d64,d64_5,d64_6,7).
bond(d64,d64_6,d64_1,7).
bond(d64,d64_2,d64_7,1).
bond(d64,d64_5,d64_8,1).
bond(d64,d64_6,d64_9,1).
bond(d64,d64_10,d64_11,7).
bond(d64,d64_11,d64_12,7).
bond(d64,d64_12,d64_13,7).
bond(d64,d64_13,d64_14,7).
bond(d64,d64_14,d64_15,7).
bond(d64,d64_15,d64_10,7).
bond(d64,d64_11,d64_16,1).
bond(d64,d64_13,d64_17,1).
bond(d64,d64_14,d64_18,1).
bond(d64,d64_4,d64_15,1).
bond(d64,d64_3,d64_19,1).
bond(d64,d64_19,d64_10,1).
bond(d64,d64_12,d64_20,1).
bond(d64,d64_1,d64_21,1).
bond(d64,d64_22,d64_21,2).
bond(d64,d64_21,d64_23,2).
bond(d64,d64_20,d64_24,2).
bond(d64,d64_20,d64_25,2).
bond(d64,d64_19,d64_26,2).
atm(d65,d65_1,c,22,-0.118).
atm(d65,d65_2,c,22,-0.118).
atm(d65,d65_3,c,27,0.012).
atm(d65,d65_4,c,27,0.012).
atm(d65,d65_5,c,22,-0.118).
atm(d65,d65_6,c,22,-0.118).
atm(d65,d65_7,h,3,0.142).
atm(d65,d65_8,h,3,0.142).
atm(d65,d65_9,h,3,0.142).
atm(d65,d65_10,h,3,0.142).
atm(d65,d65_11,c,22,-0.118).
atm(d65,d65_12,c,22,-0.118).
atm(d65,d65_13,c,22,-0.118).
atm(d65,d65_14,c,22,-0.118).
atm(d65,d65_15,h,3,0.142).
atm(d65,d65_16,h,3,0.142).
atm(d65,d65_17,n,38,0.811).
atm(d65,d65_18,n,38,0.811).
atm(d65,d65_19,o,40,-0.388).
atm(d65,d65_20,o,40,-0.389).
atm(d65,d65_21,o,40,-0.389).
atm(d65,d65_22,o,40,-0.388).

bond(d65,d65_1,d65_2,7).
bond(d65,d65_2,d65_3,7).
bond(d65,d65_3,d65_4,7).
bond(d65,d65_4,d65_5,7).
bond(d65,d65_5,d65_6,7).
bond(d65,d65_6,d65_1,7).
bond(d65,d65_1,d65_7,1).
bond(d65,d65_2,d65_8,1).
bond(d65,d65_5,d65_9,1).
bond(d65,d65_6,d65_10,1).
bond(d65,d65_3,d65_11,7).
bond(d65,d65_11,d65_12,7).
bond(d65,d65_12,d65_13,7).
bond(d65,d65_13,d65_14,7).
bond(d65,d65_14,d65_4,7).
bond(d65,d65_11,d65_15,1).
bond(d65,d65_13,d65_16,1).
bond(d65,d65_17,d65_14,1).
bond(d65,d65_12,d65_18,1).
bond(d65,d65_19,d65_17,2).
bond(d65,d65_17,d65_20,2).
bond(d65,d65_21,d65_18,2).
bond(d65,d65_18,d65_22,2).
atm(d66,d66_1,c,22,-0.128).
atm(d66,d66_2,c,22,-0.128).
atm(d66,d66_3,c,22,-0.128).
atm(d66,d66_4,c,29,0.002).
atm(d66,d66_5,c,22,-0.128).
atm(d66,d66_6,c,22,-0.128).
atm(d66,d66_7,h,3,0.132).
atm(d66,d66_8,h,3,0.132).
atm(d66,d66_9,h,3,0.132).
atm(d66,d66_10,h,3,0.132).
atm(d66,d66_11,c,29,0.002).
atm(d66,d66_12,c,22,-0.128).
atm(d66,d66_13,c,22,-0.128).
atm(d66,d66_14,c,22,0.202).
atm(d66,d66_15,c,22,-0.128).
atm(d66,d66_16,c,22,-0.128).
atm(d66,d66_17,h,3,0.132).
atm(d66,d66_18,h,3,0.132).
atm(d66,d66_19,h,3,0.132).
atm(d66,d66_20,h,3,0.132).
atm(d66,d66_21,n,32,-0.779).
atm(d66,d66_22,n,38,0.801).
atm(d66,d66_23,o,40,-0.398).
atm(d66,d66_24,o,40,-0.398).
atm(d66,d66_25,h,1,0.332).
atm(d66,d66_26,h,1,0.332).

bond(d66,d66_1,d66_2,7).
bond(d66,d66_2,d66_3,7).
bond(d66,d66_3,d66_4,7).
bond(d66,d66_4,d66_5,7).
bond(d66,d66_5,d66_6,7).
bond(d66,d66_6,d66_1,7).
bond(d66,d66_1,d66_7,1).
bond(d66,d66_2,d66_8,1).
bond(d66,d66_3,d66_9,1).
bond(d66,d66_5,d66_10,1).
bond(d66,d66_11,d66_12,7).
bond(d66,d66_12,d66_13,7).
bond(d66,d66_13,d66_14,7).
bond(d66,d66_14,d66_15,7).
bond(d66,d66_15,d66_16,7).
bond(d66,d66_16,d66_11,7).
bond(d66,d66_12,d66_17,1).
bond(d66,d66_13,d66_18,1).
bond(d66,d66_15,d66_19,1).
bond(d66,d66_16,d66_20,1).
bond(d66,d66_4,d66_11,1).
bond(d66,d66_14,d66_21,1).
bond(d66,d66_6,d66_22,1).
bond(d66,d66_23,d66_22,2).
bond(d66,d66_22,d66_24,2).
bond(d66,d66_21,d66_25,1).
bond(d66,d66_21,d66_26,1).
atm(d67,d67_1,c,22,-0.122).
atm(d67,d67_2,c,22,-0.122).
atm(d67,d67_3,c,27,0.008).
atm(d67,d67_4,c,27,-0.092).
atm(d67,d67_5,c,28,0.008).
atm(d67,d67_6,c,22,-0.122).
atm(d67,d67_7,h,3,0.138).
atm(d67,d67_8,h,3,0.138).
atm(d67,d67_9,h,3,0.138).
atm(d67,d67_10,c,22,-0.122).
atm(d67,d67_11,c,22,0.258).
atm(d67,d67_12,c,22,-0.122).
atm(d67,d67_13,c,29,0.008).
atm(d67,d67_14,h,3,0.138).
atm(d67,d67_15,c,28,0.008).
atm(d67,d67_16,c,28,0.008).
atm(d67,d67_17,c,22,-0.122).
atm(d67,d67_18,c,22,-0.122).
atm(d67,d67_19,c,22,-0.122).
atm(d67,d67_20,c,22,-0.122).
atm(d67,d67_21,h,3,0.138).
atm(d67,d67_22,h,3,0.138).
atm(d67,d67_23,h,3,0.138).
atm(d67,d67_24,n,38,0.807).
atm(d67,d67_25,o,45,-0.643).
atm(d67,d67_26,o,40,-0.392).
atm(d67,d67_27,o,40,-0.392).
atm(d67,d67_28,h,8,0.408).
atm(d67,d67_29,h,3,0.138).

bond(d67,d67_1,d67_2,7).
bond(d67,d67_2,d67_3,7).
bond(d67,d67_3,d67_4,7).
bond(d67,d67_4,d67_5,7).
bond(d67,d67_5,d67_6,7).
bond(d67,d67_6,d67_1,7).
bond(d67,d67_1,d67_7,1).
bond(d67,d67_2,d67_8,1).
bond(d67,d67_6,d67_9,1).
bond(d67,d67_3,d67_10,7).
bond(d67,d67_10,d67_11,7).
bond(d67,d67_11,d67_12,7).
bond(d67,d67_12,d67_13,7).
bond(d67,d67_13,d67_4,7).
bond(d67,d67_10,d67_14,1).
bond(d67,d67_15,d67_16,7).
bond(d67,d67_16,d67_17,7).
bond(d67,d67_17,d67_18,7).
bond(d67,d67_18,d67_19,7).
bond(d67,d67_19,d67_20,7).
bond(d67,d67_20,d67_15,7).
bond(d67,d67_18,d67_21,1).
bond(d67,d67_19,d67_22,1).
bond(d67,d67_20,d67_23,1).
bond(d67,d67_15,d67_5,1).
bond(d67,d67_12,d67_24,1).
bond(d67,d67_11,d67_25,1).
bond(d67,d67_26,d67_24,2).
bond(d67,d67_24,d67_27,2).
bond(d67,d67_25,d67_28,1).
bond(d67,d67_17,d67_29,1).
bond(d67,d67_13,d67_16,1).
atm(d68,d68_1,c,22,-0.114).
atm(d68,d68_2,c,22,-0.114).
atm(d68,d68_3,c,27,0.016).
atm(d68,d68_4,c,27,-0.084).
atm(d68,d68_5,c,22,-0.114).
atm(d68,d68_6,c,22,-0.114).
atm(d68,d68_7,h,3,0.145).
atm(d68,d68_8,h,3,0.145).
atm(d68,d68_9,h,3,0.145).
atm(d68,d68_10,h,3,0.145).
atm(d68,d68_11,c,22,-0.114).
atm(d68,d68_12,c,27,0.016).
atm(d68,d68_13,c,195,-0.084).
atm(d68,d68_14,c,195,-0.084).
atm(d68,d68_15,h,3,0.145).
atm(d68,d68_16,c,22,-0.114).
atm(d68,d68_17,c,22,-0.114).
atm(d68,d68_18,c,27,0.016).
atm(d68,d68_19,c,27,-0.084).
atm(d68,d68_20,h,3,0.145).
atm(d68,d68_21,h,3,0.145).
atm(d68,d68_22,c,27,0.016).
atm(d68,d68_23,c,22,-0.114).
atm(d68,d68_24,c,22,-0.114).
atm(d68,d68_25,h,3,0.145).
atm(d68,d68_26,h,3,0.145).
atm(d68,d68_27,c,22,-0.114).
atm(d68,d68_28,c,22,-0.114).
atm(d68,d68_29,c,22,-0.114).
atm(d68,d68_30,h,3,0.145).
atm(d68,d68_31,h,3,0.145).
atm(d68,d68_32,n,38,0.815).
atm(d68,d68_33,o,40,-0.385).
atm(d68,d68_34,o,40,-0.385).

bond(d68,d68_1,d68_2,7).
bond(d68,d68_2,d68_3,7).
bond(d68,d68_3,d68_4,7).
bond(d68,d68_4,d68_5,7).
bond(d68,d68_5,d68_6,7).
bond(d68,d68_6,d68_1,7).
bond(d68,d68_1,d68_7,1).
bond(d68,d68_2,d68_8,1).
bond(d68,d68_5,d68_9,1).
bond(d68,d68_6,d68_10,1).
bond(d68,d68_3,d68_11,7).
bond(d68,d68_11,d68_12,7).
bond(d68,d68_12,d68_13,7).
bond(d68,d68_13,d68_14,7).
bond(d68,d68_14,d68_4,7).
bond(d68,d68_11,d68_15,1).
bond(d68,d68_12,d68_16,7).
bond(d68,d68_16,d68_17,7).
bond(d68,d68_17,d68_18,7).
bond(d68,d68_18,d68_19,7).
bond(d68,d68_19,d68_13,7).
bond(d68,d68_16,d68_20,1).
bond(d68,d68_17,d68_21,1).
bond(d68,d68_19,d68_22,7).
bond(d68,d68_22,d68_23,7).
bond(d68,d68_23,d68_24,7).
bond(d68,d68_24,d68_14,7).
bond(d68,d68_23,d68_25,1).
bond(d68,d68_24,d68_26,1).
bond(d68,d68_18,d68_27,7).
bond(d68,d68_27,d68_28,7).
bond(d68,d68_28,d68_29,7).
bond(d68,d68_29,d68_22,7).
bond(d68,d68_28,d68_30,1).
bond(d68,d68_29,d68_31,1).
bond(d68,d68_27,d68_32,1).
bond(d68,d68_32,d68_33,2).
bond(d68,d68_32,d68_34,2).
atm(d69,d69_1,c,22,-0.114).
atm(d69,d69_2,c,22,-0.114).
atm(d69,d69_3,c,22,-0.114).
atm(d69,d69_4,c,29,0.016).
atm(d69,d69_5,c,22,-0.114).
atm(d69,d69_6,c,22,-0.114).
atm(d69,d69_7,h,3,0.145).
atm(d69,d69_8,h,3,0.145).
atm(d69,d69_9,h,3,0.145).
atm(d69,d69_10,c,22,-0.114).
atm(d69,d69_11,c,22,-0.114).
atm(d69,d69_12,c,22,-0.114).
atm(d69,d69_13,c,22,-0.114).
atm(d69,d69_14,c,22,-0.114).
atm(d69,d69_15,c,29,0.016).
atm(d69,d69_16,h,3,0.145).
atm(d69,d69_17,h,3,0.145).
atm(d69,d69_18,h,3,0.145).
atm(d69,d69_19,c,10,0.016).
atm(d69,d69_20,h,3,0.066).
atm(d69,d69_21,h,3,0.066).
atm(d69,d69_22,n,38,0.815).
atm(d69,d69_23,n,38,0.815).
atm(d69,d69_24,o,40,-0.385).
atm(d69,d69_25,o,40,-0.385).
atm(d69,d69_26,o,40,-0.385).
atm(d69,d69_27,o,40,-0.385).

bond(d69,d69_1,d69_2,7).
bond(d69,d69_2,d69_3,7).
bond(d69,d69_3,d69_4,7).
bond(d69,d69_4,d69_5,7).
bond(d69,d69_5,d69_6,7).
bond(d69,d69_6,d69_1,7).
bond(d69,d69_1,d69_7,1).
bond(d69,d69_2,d69_8,1).
bond(d69,d69_6,d69_9,1).
bond(d69,d69_10,d69_11,7).
bond(d69,d69_11,d69_12,7).
bond(d69,d69_12,d69_13,7).
bond(d69,d69_13,d69_14,7).
bond(d69,d69_14,d69_15,7).
bond(d69,d69_15,d69_10,7).
bond(d69,d69_11,d69_16,1).
bond(d69,d69_13,d69_17,1).
bond(d69,d69_14,d69_18,1).
bond(d69,d69_4,d69_15,1).
bond(d69,d69_3,d69_19,1).
bond(d69,d69_19,d69_10,1).
bond(d69,d69_19,d69_20,1).
bond(d69,d69_19,d69_21,1).
bond(d69,d69_12,d69_22,1).
bond(d69,d69_5,d69_23,1).
bond(d69,d69_24,d69_23,2).
bond(d69,d69_23,d69_25,2).
bond(d69,d69_26,d69_22,2).
bond(d69,d69_22,d69_27,2).
atm(d7,d7_1,c,22,-0.124).
atm(d7,d7_2,c,22,-0.124).
atm(d7,d7_3,c,22,-0.124).
atm(d7,d7_4,c,29,0.006).
atm(d7,d7_5,c,22,-0.124).
atm(d7,d7_6,c,22,-0.124).
atm(d7,d7_7,h,3,0.136).
atm(d7,d7_8,h,3,0.136).
atm(d7,d7_9,h,3,0.136).
atm(d7,d7_10,h,3,0.136).
atm(d7,d7_11,h,3,0.136).
atm(d7,d7_12,c,29,0.006).
atm(d7,d7_13,c,22,-0.124).
atm(d7,d7_14,c,22,-0.124).
atm(d7,d7_15,c,22,-0.124).
atm(d7,d7_16,c,22,-0.174).
atm(d7,d7_17,c,22,-0.124).
atm(d7,d7_18,h,3,0.136).
atm(d7,d7_19,h,3,0.136).
atm(d7,d7_20,h,3,0.136).
atm(d7,d7_21,c,10,0.006).
atm(d7,d7_22,h,3,0.056).
atm(d7,d7_23,h,3,0.056).
atm(d7,d7_24,h,3,0.056).
atm(d7,d7_25,n,38,0.805).
atm(d7,d7_26,o,40,-0.395).
atm(d7,d7_27,o,40,-0.394).

bond(d7,d7_1,d7_2,7).
bond(d7,d7_2,d7_3,7).
bond(d7,d7_3,d7_4,7).
bond(d7,d7_4,d7_5,7).
bond(d7,d7_5,d7_6,7).
bond(d7,d7_6,d7_1,7).
bond(d7,d7_1,d7_7,1).
bond(d7,d7_2,d7_8,1).
bond(d7,d7_3,d7_9,1).
bond(d7,d7_5,d7_10,1).
bond(d7,d7_6,d7_11,1).
bond(d7,d7_12,d7_13,7).
bond(d7,d7_13,d7_14,7).
bond(d7,d7_14,d7_15,7).
bond(d7,d7_15,d7_16,7).
bond(d7,d7_16,d7_17,7).
bond(d7,d7_17,d7_12,7).
bond(d7,d7_13,d7_18,1).
bond(d7,d7_14,d7_19,1).
bond(d7,d7_17,d7_20,1).
bond(d7,d7_4,d7_12,1).
bond(d7,d7_21,d7_16,1).
bond(d7,d7_21,d7_22,1).
bond(d7,d7_21,d7_23,1).
bond(d7,d7_21,d7_24,1).
bond(d7,d7_25,d7_15,1).
bond(d7,d7_25,d7_26,2).
bond(d7,d7_25,d7_27,2).
atm(d70,d70_1,c,22,-0.128).
atm(d70,d70_2,c,22,-0.128).
atm(d70,d70_3,c,22,0.202).
atm(d70,d70_4,c,22,0.002).
atm(d70,d70_5,c,22,-0.128).
atm(d70,d70_6,c,22,-0.128).
atm(d70,d70_7,h,3,0.132).
atm(d70,d70_8,h,3,0.132).
atm(d70,d70_9,h,3,0.132).
atm(d70,d70_10,n,32,-0.398).
atm(d70,d70_11,c,14,0.601).
atm(d70,d70_12,c,14,0.551).
atm(d70,d70_13,o,40,-0.549).
atm(d70,d70_14,o,40,-0.549).
atm(d70,d70_15,h,1,0.252).
atm(d70,d70_16,n,38,0.801).
atm(d70,d70_17,o,40,-0.399).
atm(d70,d70_18,o,40,-0.398).

bond(d70,d70_1,d70_2,7).
bond(d70,d70_2,d70_3,7).
bond(d70,d70_3,d70_4,7).
bond(d70,d70_4,d70_5,7).
bond(d70,d70_5,d70_6,7).
bond(d70,d70_6,d70_1,7).
bond(d70,d70_1,d70_7,1).
bond(d70,d70_2,d70_8,1).
bond(d70,d70_5,d70_9,1).
bond(d70,d70_3,d70_10,1).
bond(d70,d70_10,d70_11,1).
bond(d70,d70_11,d70_12,1).
bond(d70,d70_12,d70_4,1).
bond(d70,d70_11,d70_13,2).
bond(d70,d70_12,d70_14,2).
bond(d70,d70_10,d70_15,1).
bond(d70,d70_6,d70_16,1).
bond(d70,d70_17,d70_16,2).
bond(d70,d70_16,d70_18,2).
atm(d71,d71_1,c,22,-0.118).
atm(d71,d71_2,c,22,-0.118).
atm(d71,d71_3,c,27,0.012).
atm(d71,d71_4,c,27,-0.089).
atm(d71,d71_5,c,27,0.012).
atm(d71,d71_6,c,22,-0.118).
atm(d71,d71_7,c,22,-0.118).
atm(d71,d71_8,c,22,-0.118).
atm(d71,d71_9,c,27,0.012).
atm(d71,d71_10,c,27,-0.089).
atm(d71,d71_11,c,27,0.012).
atm(d71,d71_12,c,22,0.012).
atm(d71,d71_13,c,22,0.012).
atm(d71,d71_14,c,22,-0.118).
atm(d71,d71_15,c,22,-0.118).
atm(d71,d71_16,c,22,-0.118).
atm(d71,d71_17,c,10,-0.089).
atm(d71,d71_18,c,10,-0.089).
atm(d71,d71_19,c,10,-0.089).
atm(d71,d71_20,c,10,-0.089).
atm(d71,d71_21,n,38,0.811).
atm(d71,d71_22,n,38,0.811).
atm(d71,d71_23,o,40,-0.39).
atm(d71,d71_24,o,40,-0.389).
atm(d71,d71_25,o,40,-0.389).
atm(d71,d71_26,o,40,-0.39).
atm(d71,d71_27,h,3,0.141).
atm(d71,d71_28,h,3,0.141).
atm(d71,d71_29,h,3,0.141).
atm(d71,d71_30,h,3,0.141).
atm(d71,d71_31,h,3,0.141).
atm(d71,d71_32,h,3,0.141).
atm(d71,d71_33,h,3,0.062).
atm(d71,d71_34,h,3,0.062).
atm(d71,d71_35,h,3,0.062).
atm(d71,d71_36,h,3,0.062).
atm(d71,d71_37,h,3,0.062).
atm(d71,d71_38,h,3,0.062).
atm(d71,d71_39,h,3,0.062).
atm(d71,d71_40,h,3,0.062).

bond(d71,d71_1,d71_2,7).
bond(d71,d71_2,d71_3,7).
bond(d71,d71_3,d71_4,7).
bond(d71,d71_4,d71_5,7).
bond(d71,d71_5,d71_6,7).
bond(d71,d71_6,d71_1,7).
bond(d71,d71_3,d71_7,7).
bond(d71,d71_7,d71_8,7).
bond(d71,d71_8,d71_9,7).
bond(d71,d71_9,d71_10,7).
bond(d71,d71_10,d71_4,7).
bond(d71,d71_10,d71_11,7).
bond(d71,d71_11,d71_12,7).
bond(d71,d71_12,d71_13,7).
bond(d71,d71_13,d71_5,7).
bond(d71,d71_9,d71_14,7).
bond(d71,d71_14,d71_15,7).
bond(d71,d71_15,d71_16,7).
bond(d71,d71_16,d71_11,7).
bond(d71,d71_12,d71_17,1).
bond(d71,d71_17,d71_18,1).
bond(d71,d71_18,d71_19,1).
bond(d71,d71_19,d71_20,1).
bond(d71,d71_20,d71_13,1).
bond(d71,d71_21,d71_16,1).
bond(d71,d71_6,d71_22,1).
bond(d71,d71_23,d71_21,2).
bond(d71,d71_21,d71_24,2).
bond(d71,d71_22,d71_25,2).
bond(d71,d71_26,d71_22,2).
bond(d71,d71_1,d71_27,1).
bond(d71,d71_2,d71_28,1).
bond(d71,d71_7,d71_29,1).
bond(d71,d71_8,d71_30,1).
bond(d71,d71_14,d71_31,1).
bond(d71,d71_15,d71_32,1).
bond(d71,d71_18,d71_33,1).
bond(d71,d71_18,d71_34,1).
bond(d71,d71_17,d71_35,1).
bond(d71,d71_17,d71_36,1).
bond(d71,d71_20,d71_37,1).
bond(d71,d71_20,d71_38,1).
bond(d71,d71_19,d71_39,1).
bond(d71,d71_19,d71_40,1).
atm(d72,d72_1,c,22,-0.118).
atm(d72,d72_2,c,22,-0.118).
atm(d72,d72_3,c,22,-0.118).
atm(d72,d72_4,c,29,0.012).
atm(d72,d72_5,c,22,-0.118).
atm(d72,d72_6,c,22,-0.118).
atm(d72,d72_7,h,3,0.141).
atm(d72,d72_8,h,3,0.141).
atm(d72,d72_9,h,3,0.141).
atm(d72,d72_10,h,3,0.141).
atm(d72,d72_11,c,22,-0.118).
atm(d72,d72_12,c,22,-0.118).
atm(d72,d72_13,c,22,-0.118).
atm(d72,d72_14,c,22,-0.118).
atm(d72,d72_15,c,22,-0.118).
atm(d72,d72_16,c,29,0.012).
atm(d72,d72_17,h,3,0.141).
atm(d72,d72_18,h,3,0.141).
atm(d72,d72_19,h,3,0.141).
atm(d72,d72_20,c,10,0.012).
atm(d72,d72_21,h,3,0.062).
atm(d72,d72_22,h,3,0.062).
atm(d72,d72_23,n,38,0.811).
atm(d72,d72_24,o,40,-0.389).
atm(d72,d72_25,o,40,-0.389).

bond(d72,d72_1,d72_2,7).
bond(d72,d72_2,d72_3,7).
bond(d72,d72_3,d72_4,7).
bond(d72,d72_4,d72_5,7).
bond(d72,d72_5,d72_6,7).
bond(d72,d72_6,d72_1,7).
bond(d72,d72_1,d72_7,1).
bond(d72,d72_2,d72_8,1).
bond(d72,d72_5,d72_9,1).
bond(d72,d72_6,d72_10,1).
bond(d72,d72_11,d72_12,7).
bond(d72,d72_12,d72_13,7).
bond(d72,d72_13,d72_14,7).
bond(d72,d72_14,d72_15,7).
bond(d72,d72_15,d72_16,7).
bond(d72,d72_16,d72_11,7).
bond(d72,d72_12,d72_17,1).
bond(d72,d72_14,d72_18,1).
bond(d72,d72_15,d72_19,1).
bond(d72,d72_4,d72_16,1).
bond(d72,d72_3,d72_20,1).
bond(d72,d72_20,d72_11,1).
bond(d72,d72_20,d72_21,1).
bond(d72,d72_20,d72_22,1).
bond(d72,d72_13,d72_23,1).
bond(d72,d72_24,d72_23,2).
bond(d72,d72_23,d72_25,2).
atm(d73,d73_1,c,22,-0.128).
atm(d73,d73_2,c,22,-0.128).
atm(d73,d73_3,c,22,-0.128).
atm(d73,d73_4,c,29,0.002).
atm(d73,d73_5,c,22,-0.128).
atm(d73,d73_6,c,22,-0.128).
atm(d73,d73_7,h,3,0.132).
atm(d73,d73_8,h,3,0.132).
atm(d73,d73_9,h,3,0.132).
atm(d73,d73_10,h,3,0.132).
atm(d73,d73_11,c,29,0.002).
atm(d73,d73_12,c,22,-0.128).
atm(d73,d73_13,c,22,-0.128).
atm(d73,d73_14,c,22,-0.128).
atm(d73,d73_15,c,22,0.202).
atm(d73,d73_16,c,22,-0.128).
atm(d73,d73_17,h,3,0.132).
atm(d73,d73_18,h,3,0.132).
atm(d73,d73_19,h,3,0.132).
atm(d73,d73_20,h,3,0.132).
atm(d73,d73_21,n,32,-0.779).
atm(d73,d73_22,n,38,0.801).
atm(d73,d73_23,o,40,-0.398).
atm(d73,d73_24,o,40,-0.398).
atm(d73,d73_25,h,1,0.332).
atm(d73,d73_26,h,1,0.332).

bond(d73,d73_1,d73_2,7).
bond(d73,d73_2,d73_3,7).
bond(d73,d73_3,d73_4,7).
bond(d73,d73_4,d73_5,7).
bond(d73,d73_5,d73_6,7).
bond(d73,d73_6,d73_1,7).
bond(d73,d73_1,d73_7,1).
bond(d73,d73_2,d73_8,1).
bond(d73,d73_3,d73_9,1).
bond(d73,d73_6,d73_10,1).
bond(d73,d73_11,d73_12,7).
bond(d73,d73_12,d73_13,7).
bond(d73,d73_13,d73_14,7).
bond(d73,d73_14,d73_15,7).
bond(d73,d73_15,d73_16,7).
bond(d73,d73_16,d73_11,7).
bond(d73,d73_12,d73_17,1).
bond(d73,d73_13,d73_18,1).
bond(d73,d73_14,d73_19,1).
bond(d73,d73_16,d73_20,1).
bond(d73,d73_4,d73_11,1).
bond(d73,d73_21,d73_15,1).
bond(d73,d73_22,d73_5,1).
bond(d73,d73_23,d73_22,2).
bond(d73,d73_22,d73_24,2).
bond(d73,d73_21,d73_25,1).
bond(d73,d73_21,d73_26,1).
atm(d74,d74_1,c,22,-0.123).
atm(d74,d74_2,c,22,-0.123).
atm(d74,d74_3,c,27,0.007).
atm(d74,d74_4,c,27,-0.093).
atm(d74,d74_5,c,28,0.007).
atm(d74,d74_6,c,22,-0.123).
atm(d74,d74_7,h,3,0.136).
atm(d74,d74_8,h,3,0.137).
atm(d74,d74_9,h,3,0.137).
atm(d74,d74_10,c,22,-0.123).
atm(d74,d74_11,c,22,-0.123).
atm(d74,d74_12,c,22,-0.123).
atm(d74,d74_13,c,29,0.007).
atm(d74,d74_14,h,3,0.136).
atm(d74,d74_15,h,3,0.136).
atm(d74,d74_16,h,3,0.136).
atm(d74,d74_17,c,28,0.007).
atm(d74,d74_18,c,28,0.007).
atm(d74,d74_19,c,22,-0.123).
atm(d74,d74_20,c,27,0.007).
atm(d74,d74_21,c,27,0.007).
atm(d74,d74_22,c,22,-0.123).
atm(d74,d74_23,h,3,0.136).
atm(d74,d74_24,h,3,0.137).
atm(d74,d74_25,c,22,-0.123).
atm(d74,d74_26,c,22,-0.123).
atm(d74,d74_27,c,22,-0.123).
atm(d74,d74_28,c,22,-0.123).
atm(d74,d74_29,h,3,0.137).
atm(d74,d74_30,h,3,0.137).
atm(d74,d74_31,h,3,0.137).
atm(d74,d74_32,n,38,0.806).
atm(d74,d74_33,o,40,-0.394).
atm(d74,d74_34,o,40,-0.394).

bond(d74,d74_1,d74_2,7).
bond(d74,d74_2,d74_3,7).
bond(d74,d74_3,d74_4,7).
bond(d74,d74_4,d74_5,7).
bond(d74,d74_5,d74_6,7).
bond(d74,d74_6,d74_1,7).
bond(d74,d74_1,d74_7,1).
bond(d74,d74_2,d74_8,1).
bond(d74,d74_6,d74_9,1).
bond(d74,d74_3,d74_10,7).
bond(d74,d74_10,d74_11,7).
bond(d74,d74_11,d74_12,7).
bond(d74,d74_12,d74_13,7).
bond(d74,d74_13,d74_4,7).
bond(d74,d74_10,d74_14,1).
bond(d74,d74_11,d74_15,1).
bond(d74,d74_12,d74_16,1).
bond(d74,d74_17,d74_18,7).
bond(d74,d74_18,d74_19,7).
bond(d74,d74_19,d74_20,7).
bond(d74,d74_20,d74_21,7).
bond(d74,d74_21,d74_22,7).
bond(d74,d74_22,d74_17,7).
bond(d74,d74_19,d74_23,1).
bond(d74,d74_22,d74_24,1).
bond(d74,d74_20,d74_25,7).
bond(d74,d74_25,d74_26,7).
bond(d74,d74_26,d74_27,7).
bond(d74,d74_27,d74_28,7).
bond(d74,d74_28,d74_21,7).
bond(d74,d74_25,d74_29,1).
bond(d74,d74_27,d74_30,1).
bond(d74,d74_28,d74_31,1).
bond(d74,d74_17,d74_5,1).
bond(d74,d74_18,d74_13,1).
bond(d74,d74_26,d74_32,1).
bond(d74,d74_33,d74_32,2).
bond(d74,d74_32,d74_34,2).
atm(d75,d75_1,c,22,-0.12).
atm(d75,d75_2,c,22,-0.12).
atm(d75,d75_3,c,22,-0.12).
atm(d75,d75_4,c,29,0.01).
atm(d75,d75_5,c,22,-0.12).
atm(d75,d75_6,c,22,-0.12).
atm(d75,d75_7,h,3,0.14).
atm(d75,d75_8,h,3,0.14).
atm(d75,d75_9,h,3,0.14).
atm(d75,d75_10,h,3,0.14).
atm(d75,d75_11,c,29,0.01).
atm(d75,d75_12,c,22,-0.12).
atm(d75,d75_13,c,22,-0.12).
atm(d75,d75_14,c,22,-0.12).
atm(d75,d75_15,c,22,-0.12).
atm(d75,d75_16,c,22,-0.12).
atm(d75,d75_17,h,3,0.14).
atm(d75,d75_18,h,3,0.14).
atm(d75,d75_19,h,3,0.14).
atm(d75,d75_20,h,3,0.14).
atm(d75,d75_21,n,38,0.81).
atm(d75,d75_22,n,38,0.81).
atm(d75,d75_23,o,40,-0.39).
atm(d75,d75_24,o,40,-0.39).
atm(d75,d75_25,o,40,-0.39).
atm(d75,d75_26,o,40,-0.39).

bond(d75,d75_1,d75_2,7).
bond(d75,d75_2,d75_3,7).
bond(d75,d75_3,d75_4,7).
bond(d75,d75_4,d75_5,7).
bond(d75,d75_5,d75_6,7).
bond(d75,d75_6,d75_1,7).
bond(d75,d75_2,d75_7,1).
bond(d75,d75_3,d75_8,1).
bond(d75,d75_5,d75_9,1).
bond(d75,d75_6,d75_10,1).
bond(d75,d75_11,d75_12,7).
bond(d75,d75_12,d75_13,7).
bond(d75,d75_13,d75_14,7).
bond(d75,d75_14,d75_15,7).
bond(d75,d75_15,d75_16,7).
bond(d75,d75_16,d75_11,7).
bond(d75,d75_12,d75_17,1).
bond(d75,d75_13,d75_18,1).
bond(d75,d75_15,d75_19,1).
bond(d75,d75_16,d75_20,1).
bond(d75,d75_4,d75_11,1).
bond(d75,d75_21,d75_14,1).
bond(d75,d75_1,d75_22,1).
bond(d75,d75_23,d75_21,2).
bond(d75,d75_21,d75_24,2).
bond(d75,d75_22,d75_25,2).
bond(d75,d75_22,d75_26,2).
atm(d76,d76_1,c,22,-0.097).
atm(d76,d76_2,c,22,-0.097).
atm(d76,d76_3,c,22,-0.097).
atm(d76,d76_4,c,22,-0.097).
atm(d76,d76_5,c,22,-0.097).
atm(d76,d76_6,c,22,-0.097).
atm(d76,d76_7,h,3,0.163).
atm(d76,d76_8,h,3,0.163).
atm(d76,d76_9,h,3,0.163).
atm(d76,d76_10,h,3,0.163).
atm(d76,d76_11,n,38,0.832).
atm(d76,d76_12,cl,93,-0.167).
atm(d76,d76_13,o,40,-0.368).
atm(d76,d76_14,o,40,-0.367).

bond(d76,d76_1,d76_2,7).
bond(d76,d76_2,d76_3,7).
bond(d76,d76_3,d76_4,7).
bond(d76,d76_4,d76_5,7).
bond(d76,d76_5,d76_6,7).
bond(d76,d76_6,d76_1,7).
bond(d76,d76_1,d76_7,1).
bond(d76,d76_2,d76_8,1).
bond(d76,d76_3,d76_9,1).
bond(d76,d76_6,d76_10,1).
bond(d76,d76_11,d76_5,1).
bond(d76,d76_4,d76_12,1).
bond(d76,d76_13,d76_11,2).
bond(d76,d76_11,d76_14,2).
atm(d77,d77_1,c,22,-0.128).
atm(d77,d77_2,c,22,-0.128).
atm(d77,d77_3,c,22,-0.128).
atm(d77,d77_4,c,29,0.002).
atm(d77,d77_5,c,22,-0.128).
atm(d77,d77_6,c,22,-0.128).
atm(d77,d77_7,h,3,0.132).
atm(d77,d77_8,h,3,0.132).
atm(d77,d77_9,h,3,0.132).
atm(d77,d77_10,h,3,0.132).
atm(d77,d77_11,c,29,0.002).
atm(d77,d77_12,c,22,-0.128).
atm(d77,d77_13,c,22,-0.128).
atm(d77,d77_14,c,22,-0.128).
atm(d77,d77_15,c,22,-0.128).
atm(d77,d77_16,c,22,0.202).
atm(d77,d77_17,h,3,0.132).
atm(d77,d77_18,h,3,0.132).
atm(d77,d77_19,h,3,0.132).
atm(d77,d77_20,h,3,0.132).
atm(d77,d77_21,n,32,-0.779).
atm(d77,d77_22,n,38,0.801).
atm(d77,d77_23,o,40,-0.398).
atm(d77,d77_24,o,40,-0.398).
atm(d77,d77_25,h,1,0.332).
atm(d77,d77_26,h,1,0.332).

bond(d77,d77_1,d77_2,7).
bond(d77,d77_2,d77_3,7).
bond(d77,d77_3,d77_4,7).
bond(d77,d77_4,d77_5,7).
bond(d77,d77_5,d77_6,7).
bond(d77,d77_6,d77_1,7).
bond(d77,d77_1,d77_7,1).
bond(d77,d77_2,d77_8,1).
bond(d77,d77_3,d77_9,1).
bond(d77,d77_5,d77_10,1).
bond(d77,d77_11,d77_12,7).
bond(d77,d77_12,d77_13,7).
bond(d77,d77_13,d77_14,7).
bond(d77,d77_14,d77_15,7).
bond(d77,d77_15,d77_16,7).
bond(d77,d77_16,d77_11,7).
bond(d77,d77_12,d77_17,1).
bond(d77,d77_13,d77_18,1).
bond(d77,d77_14,d77_19,1).
bond(d77,d77_15,d77_20,1).
bond(d77,d77_4,d77_11,1).
bond(d77,d77_16,d77_21,1).
bond(d77,d77_6,d77_22,1).
bond(d77,d77_23,d77_22,2).
bond(d77,d77_22,d77_24,2).
bond(d77,d77_21,d77_25,1).
bond(d77,d77_21,d77_26,1).
atm(d78,d78_1,c,22,-0.124).
atm(d78,d78_2,c,22,-0.124).
atm(d78,d78_3,c,22,-0.124).
atm(d78,d78_4,c,29,0.006).
atm(d78,d78_5,c,22,-0.174).
atm(d78,d78_6,c,22,-0.124).
atm(d78,d78_7,h,3,0.136).
atm(d78,d78_8,h,3,0.136).
atm(d78,d78_9,h,3,0.136).
atm(d78,d78_10,h,3,0.136).
atm(d78,d78_11,c,29,0.006).
atm(d78,d78_12,c,22,-0.124).
atm(d78,d78_13,c,22,-0.124).
atm(d78,d78_14,c,22,-0.124).
atm(d78,d78_15,c,22,-0.124).
atm(d78,d78_16,c,22,-0.124).
atm(d78,d78_17,h,3,0.136).
atm(d78,d78_18,h,3,0.136).
atm(d78,d78_19,h,3,0.136).
atm(d78,d78_20,h,3,0.136).
atm(d78,d78_21,c,10,0.006).
atm(d78,d78_22,h,3,0.056).
atm(d78,d78_23,h,3,0.056).
atm(d78,d78_24,h,3,0.056).
atm(d78,d78_25,n,38,0.805).
atm(d78,d78_26,o,40,-0.394).
atm(d78,d78_27,o,40,-0.395).

bond(d78,d78_1,d78_2,7).
bond(d78,d78_2,d78_3,7).
bond(d78,d78_3,d78_4,7).
bond(d78,d78_4,d78_5,7).
bond(d78,d78_5,d78_6,7).
bond(d78,d78_6,d78_1,7).
bond(d78,d78_1,d78_7,1).
bond(d78,d78_2,d78_8,1).
bond(d78,d78_3,d78_9,1).
bond(d78,d78_6,d78_10,1).
bond(d78,d78_11,d78_12,7).
bond(d78,d78_12,d78_13,7).
bond(d78,d78_13,d78_14,7).
bond(d78,d78_14,d78_15,7).
bond(d78,d78_15,d78_16,7).
bond(d78,d78_16,d78_11,7).
bond(d78,d78_12,d78_17,1).
bond(d78,d78_13,d78_18,1).
bond(d78,d78_15,d78_19,1).
bond(d78,d78_16,d78_20,1).
bond(d78,d78_4,d78_11,1).
bond(d78,d78_21,d78_5,1).
bond(d78,d78_21,d78_22,1).
bond(d78,d78_21,d78_23,1).
bond(d78,d78_21,d78_24,1).
bond(d78,d78_14,d78_25,1).
bond(d78,d78_26,d78_25,2).
bond(d78,d78_25,d78_27,2).
atm(d79,d79_1,c,22,-0.135).
atm(d79,d79_2,c,22,-0.136).
atm(d79,d79_3,c,22,0.174).
atm(d79,d79_4,c,22,0.174).
atm(d79,d79_5,c,22,-0.135).
atm(d79,d79_6,c,22,-0.135).
atm(d79,d79_7,h,3,0.125).
atm(d79,d79_8,h,3,0.125).
atm(d79,d79_9,h,3,0.125).
atm(d79,d79_10,h,3,0.125).
atm(d79,d79_11,c,22,0.174).
atm(d79,d79_12,c,22,-0.135).
atm(d79,d79_13,c,22,-0.135).
atm(d79,d79_14,c,22,-0.135).
atm(d79,d79_15,c,22,-0.135).
atm(d79,d79_16,c,22,0.174).
atm(d79,d79_17,h,3,0.125).
atm(d79,d79_18,h,3,0.125).
atm(d79,d79_19,h,3,0.125).
atm(d79,d79_20,o,50,-0.236).
atm(d79,d79_21,o,50,-0.236).
atm(d79,d79_22,n,38,0.794).
atm(d79,d79_23,o,40,-0.406).
atm(d79,d79_24,o,40,-0.406).

bond(d79,d79_1,d79_2,7).
bond(d79,d79_2,d79_3,7).
bond(d79,d79_3,d79_4,7).
bond(d79,d79_4,d79_5,7).
bond(d79,d79_5,d79_6,7).
bond(d79,d79_6,d79_1,7).
bond(d79,d79_1,d79_7,1).
bond(d79,d79_2,d79_8,1).
bond(d79,d79_5,d79_9,1).
bond(d79,d79_6,d79_10,1).
bond(d79,d79_11,d79_12,7).
bond(d79,d79_12,d79_13,7).
bond(d79,d79_13,d79_14,7).
bond(d79,d79_14,d79_15,7).
bond(d79,d79_15,d79_16,7).
bond(d79,d79_16,d79_11,7).
bond(d79,d79_12,d79_17,1).
bond(d79,d79_13,d79_18,1).
bond(d79,d79_15,d79_19,1).
bond(d79,d79_3,d79_20,1).
bond(d79,d79_20,d79_11,1).
bond(d79,d79_16,d79_21,1).
bond(d79,d79_21,d79_4,1).
bond(d79,d79_22,d79_14,1).
bond(d79,d79_23,d79_22,2).
bond(d79,d79_22,d79_24,2).
atm(d8,d8_1,c,22,-0.125).
atm(d8,d8_2,c,22,-0.125).
atm(d8,d8_3,c,22,-0.125).
atm(d8,d8_4,c,29,0.005).
atm(d8,d8_5,c,22,-0.125).
atm(d8,d8_6,c,22,-0.126).
atm(d8,d8_7,c,22,-0.125).
atm(d8,d8_8,c,22,-0.126).
atm(d8,d8_9,c,22,0.254).
atm(d8,d8_10,c,22,-0.125).
atm(d8,d8_11,c,22,-0.125).
atm(d8,d8_12,c,29,0.005).
atm(d8,d8_13,c,10,0.005).
atm(d8,d8_14,n,38,0.804).
atm(d8,d8_15,o,40,-0.396).
atm(d8,d8_16,o,40,-0.396).
atm(d8,d8_17,h,3,0.134).
atm(d8,d8_18,h,3,0.134).
atm(d8,d8_19,h,3,0.134).
atm(d8,d8_20,h,3,0.134).
atm(d8,d8_21,h,3,0.134).
atm(d8,d8_22,h,3,0.134).
atm(d8,d8_23,h,3,0.055).
atm(d8,d8_24,h,3,0.055).
atm(d8,d8_25,o,49,-0.396).
atm(d8,d8_26,c,14,0.704).
atm(d8,d8_27,c,10,0.005).
atm(d8,d8_28,h,3,0.055).
atm(d8,d8_29,h,3,0.055).
atm(d8,d8_30,h,3,0.055).
atm(d8,d8_31,o,51,-0.546).

bond(d8,d8_1,d8_2,7).
bond(d8,d8_2,d8_3,7).
bond(d8,d8_3,d8_4,7).
bond(d8,d8_4,d8_5,7).
bond(d8,d8_5,d8_6,7).
bond(d8,d8_6,d8_1,7).
bond(d8,d8_7,d8_8,7).
bond(d8,d8_8,d8_9,7).
bond(d8,d8_9,d8_10,7).
bond(d8,d8_10,d8_11,7).
bond(d8,d8_11,d8_12,7).
bond(d8,d8_12,d8_7,7).
bond(d8,d8_4,d8_12,1).
bond(d8,d8_3,d8_13,1).
bond(d8,d8_13,d8_7,1).
bond(d8,d8_1,d8_14,1).
bond(d8,d8_14,d8_15,2).
bond(d8,d8_14,d8_16,2).
bond(d8,d8_2,d8_17,1).
bond(d8,d8_5,d8_18,1).
bond(d8,d8_6,d8_19,1).
bond(d8,d8_8,d8_20,1).
bond(d8,d8_10,d8_21,1).
bond(d8,d8_11,d8_22,1).
bond(d8,d8_13,d8_23,1).
bond(d8,d8_13,d8_24,1).
bond(d8,d8_9,d8_25,1).
bond(d8,d8_25,d8_26,1).
bond(d8,d8_26,d8_27,1).
bond(d8,d8_27,d8_28,1).
bond(d8,d8_27,d8_29,1).
bond(d8,d8_27,d8_30,1).
bond(d8,d8_26,d8_31,2).
atm(d80,d80_1,c,22,-0.122).
atm(d80,d80_2,c,22,-0.122).
atm(d80,d80_3,c,27,0.008).
atm(d80,d80_4,c,27,-0.092).
atm(d80,d80_5,c,28,0.008).
atm(d80,d80_6,c,22,-0.122).
atm(d80,d80_7,h,3,0.139).
atm(d80,d80_8,h,3,0.138).
atm(d80,d80_9,h,3,0.138).
atm(d80,d80_10,c,22,-0.122).
atm(d80,d80_11,c,22,-0.122).
atm(d80,d80_12,c,22,-0.122).
atm(d80,d80_13,c,29,0.008).
atm(d80,d80_14,h,3,0.139).
atm(d80,d80_15,h,3,0.138).
atm(d80,d80_16,h,3,0.138).
atm(d80,d80_17,c,28,0.008).
atm(d80,d80_18,c,28,0.008).
atm(d80,d80_19,c,22,-0.122).
atm(d80,d80_20,c,22,-0.122).
atm(d80,d80_21,c,22,-0.122).
atm(d80,d80_22,c,22,-0.122).
atm(d80,d80_23,h,3,0.139).
atm(d80,d80_24,h,3,0.138).
atm(d80,d80_25,h,3,0.138).
atm(d80,d80_26,n,38,0.809).
atm(d80,d80_27,o,40,-0.391).
atm(d80,d80_28,o,40,-0.391).

bond(d80,d80_1,d80_2,7).
bond(d80,d80_2,d80_3,7).
bond(d80,d80_3,d80_4,7).
bond(d80,d80_4,d80_5,7).
bond(d80,d80_5,d80_6,7).
bond(d80,d80_6,d80_1,7).
bond(d80,d80_1,d80_7,1).
bond(d80,d80_2,d80_8,1).
bond(d80,d80_6,d80_9,1).
bond(d80,d80_3,d80_10,7).
bond(d80,d80_10,d80_11,7).
bond(d80,d80_11,d80_12,7).
bond(d80,d80_12,d80_13,7).
bond(d80,d80_13,d80_4,7).
bond(d80,d80_10,d80_14,1).
bond(d80,d80_11,d80_15,1).
bond(d80,d80_12,d80_16,1).
bond(d80,d80_17,d80_18,7).
bond(d80,d80_18,d80_19,7).
bond(d80,d80_19,d80_20,7).
bond(d80,d80_20,d80_21,7).
bond(d80,d80_21,d80_22,7).
bond(d80,d80_22,d80_17,7).
bond(d80,d80_19,d80_23,1).
bond(d80,d80_20,d80_24,1).
bond(d80,d80_21,d80_25,1).
bond(d80,d80_17,d80_5,1).
bond(d80,d80_18,d80_13,1).
bond(d80,d80_22,d80_26,1).
bond(d80,d80_27,d80_26,2).
bond(d80,d80_26,d80_28,2).
atm(d81,d81_1,c,22,-0.114).
atm(d81,d81_2,c,22,-0.114).
atm(d81,d81_3,c,22,-0.114).
atm(d81,d81_4,c,22,-0.114).
atm(d81,d81_5,c,22,-0.114).
atm(d81,d81_6,c,22,-0.114).
atm(d81,d81_7,h,3,0.146).
atm(d81,d81_8,h,3,0.146).
atm(d81,d81_9,h,3,0.146).
atm(d81,d81_10,h,3,0.146).
atm(d81,d81_11,n,38,0.817).
atm(d81,d81_12,n,38,0.817).
atm(d81,d81_13,o,40,-0.383).
atm(d81,d81_14,o,40,-0.384).
atm(d81,d81_15,o,40,-0.384).
atm(d81,d81_16,o,40,-0.383).

bond(d81,d81_1,d81_2,7).
bond(d81,d81_2,d81_3,7).
bond(d81,d81_3,d81_4,7).
bond(d81,d81_4,d81_5,7).
bond(d81,d81_5,d81_6,7).
bond(d81,d81_6,d81_1,7).
bond(d81,d81_1,d81_7,1).
bond(d81,d81_2,d81_8,1).
bond(d81,d81_4,d81_9,1).
bond(d81,d81_6,d81_10,1).
bond(d81,d81_11,d81_5,1).
bond(d81,d81_3,d81_12,1).
bond(d81,d81_13,d81_11,2).
bond(d81,d81_11,d81_14,2).
bond(d81,d81_15,d81_12,2).
bond(d81,d81_12,d81_16,2).
atm(d82,d82_1,c,22,-0.115).
atm(d82,d82_2,c,22,-0.115).
atm(d82,d82_3,c,27,0.015).
atm(d82,d82_4,c,27,-0.085).
atm(d82,d82_5,c,28,0.015).
atm(d82,d82_6,c,22,-0.115).
atm(d82,d82_7,h,3,0.145).
atm(d82,d82_8,h,3,0.145).
atm(d82,d82_9,c,22,-0.115).
atm(d82,d82_10,c,22,-0.115).
atm(d82,d82_11,c,22,-0.115).
atm(d82,d82_12,c,29,0.015).
atm(d82,d82_13,h,3,0.145).
atm(d82,d82_14,c,28,0.015).
atm(d82,d82_15,c,28,0.015).
atm(d82,d82_16,c,22,-0.115).
atm(d82,d82_17,c,22,-0.115).
atm(d82,d82_18,c,22,-0.115).
atm(d82,d82_19,c,22,-0.115).
atm(d82,d82_20,h,3,0.146).
atm(d82,d82_21,h,3,0.145).
atm(d82,d82_22,h,3,0.145).
atm(d82,d82_23,h,3,0.145).
atm(d82,d82_24,n,38,0.816).
atm(d82,d82_25,n,38,0.816).
atm(d82,d82_26,n,38,0.816).
atm(d82,d82_27,o,40,-0.384).
atm(d82,d82_28,o,40,-0.384).
atm(d82,d82_29,o,40,-0.384).
atm(d82,d82_30,o,40,-0.384).
atm(d82,d82_31,o,40,-0.384).
atm(d82,d82_32,o,40,-0.384).

bond(d82,d82_1,d82_2,7).
bond(d82,d82_2,d82_3,7).
bond(d82,d82_3,d82_4,7).
bond(d82,d82_4,d82_5,7).
bond(d82,d82_5,d82_6,7).
bond(d82,d82_6,d82_1,7).
bond(d82,d82_1,d82_7,1).
bond(d82,d82_6,d82_8,1).
bond(d82,d82_3,d82_9,7).
bond(d82,d82_9,d82_10,7).
bond(d82,d82_10,d82_11,7).
bond(d82,d82_11,d82_12,7).
bond(d82,d82_12,d82_4,7).
bond(d82,d82_9,d82_13,1).
bond(d82,d82_14,d82_15,7).
bond(d82,d82_15,d82_16,7).
bond(d82,d82_16,d82_17,7).
bond(d82,d82_17,d82_18,7).
bond(d82,d82_18,d82_19,7).
bond(d82,d82_19,d82_14,7).
bond(d82,d82_16,d82_20,1).
bond(d82,d82_17,d82_21,1).
bond(d82,d82_18,d82_22,1).
bond(d82,d82_19,d82_23,1).
bond(d82,d82_14,d82_5,1).
bond(d82,d82_15,d82_12,1).
bond(d82,d82_11,d82_24,1).
bond(d82,d82_10,d82_25,1).
bond(d82,d82_2,d82_26,1).
bond(d82,d82_24,d82_27,2).
bond(d82,d82_24,d82_28,2).
bond(d82,d82_29,d82_25,2).
bond(d82,d82_25,d82_30,2).
bond(d82,d82_26,d82_31,2).
bond(d82,d82_26,d82_32,2).
atm(d83,d83_1,c,22,-0.148).
atm(d83,d83_2,c,22,-0.148).
atm(d83,d83_3,c,26,0.282).
atm(d83,d83_4,c,26,0.082).
atm(d83,d83_5,c,22,-0.148).
atm(d83,d83_6,c,22,-0.148).
atm(d83,d83_7,c,26,0.282).
atm(d83,d83_8,c,22,-0.148).
atm(d83,d83_9,c,22,-0.148).
atm(d83,d83_10,c,22,-0.148).
atm(d83,d83_11,c,22,-0.148).
atm(d83,d83_12,c,26,0.082).
atm(d83,d83_13,n,34,-0.557).
atm(d83,d83_14,n,38,0.783).
atm(d83,d83_15,o,40,-0.418).
atm(d83,d83_16,o,40,-0.418).
atm(d83,d83_17,h,3,0.112).
atm(d83,d83_18,h,3,0.112).
atm(d83,d83_19,h,3,0.112).
atm(d83,d83_20,h,3,0.112).
atm(d83,d83_21,h,3,0.112).
atm(d83,d83_22,h,3,0.112).
atm(d83,d83_23,h,3,0.112).
atm(d83,d83_24,h,1,0.282).

bond(d83,d83_1,d83_2,7).
bond(d83,d83_2,d83_3,7).
bond(d83,d83_3,d83_4,7).
bond(d83,d83_4,d83_5,7).
bond(d83,d83_5,d83_6,7).
bond(d83,d83_6,d83_1,7).
bond(d83,d83_7,d83_8,7).
bond(d83,d83_8,d83_9,7).
bond(d83,d83_9,d83_10,7).
bond(d83,d83_10,d83_11,7).
bond(d83,d83_11,d83_12,7).
bond(d83,d83_12,d83_7,7).
bond(d83,d83_4,d83_12,7).
bond(d83,d83_3,d83_13,7).
bond(d83,d83_13,d83_7,7).
bond(d83,d83_9,d83_14,1).
bond(d83,d83_15,d83_14,2).
bond(d83,d83_14,d83_16,2).
bond(d83,d83_1,d83_17,1).
bond(d83,d83_2,d83_18,1).
bond(d83,d83_5,d83_19,1).
bond(d83,d83_6,d83_20,1).
bond(d83,d83_8,d83_21,1).
bond(d83,d83_10,d83_22,1).
bond(d83,d83_11,d83_23,1).
bond(d83,d83_13,d83_24,1).
atm(d84,d84_1,c,22,-0.123).
atm(d84,d84_2,c,22,-0.123).
atm(d84,d84_3,c,22,-0.123).
atm(d84,d84_4,c,22,-0.123).
atm(d84,d84_5,c,22,0.107).
atm(d84,d84_6,c,22,-0.123).
atm(d84,d84_7,h,3,0.137).
atm(d84,d84_8,h,3,0.137).
atm(d84,d84_9,h,3,0.137).
atm(d84,d84_10,h,3,0.137).
atm(d84,d84_11,c,16,-0.193).
atm(d84,d84_12,c,16,-0.192).
atm(d84,d84_13,h,3,0.107).
atm(d84,d84_14,h,3,0.107).
atm(d84,d84_15,h,3,0.107).
atm(d84,d84_16,n,38,0.808).
atm(d84,d84_17,o,40,-0.392).
atm(d84,d84_18,o,40,-0.392).

bond(d84,d84_1,d84_2,7).
bond(d84,d84_2,d84_3,7).
bond(d84,d84_3,d84_4,7).
bond(d84,d84_4,d84_5,7).
bond(d84,d84_5,d84_6,7).
bond(d84,d84_6,d84_1,7).
bond(d84,d84_1,d84_7,1).
bond(d84,d84_3,d84_8,1).
bond(d84,d84_4,d84_9,1).
bond(d84,d84_6,d84_10,1).
bond(d84,d84_11,d84_5,1).
bond(d84,d84_11,d84_12,2).
bond(d84,d84_11,d84_13,1).
bond(d84,d84_12,d84_14,1).
bond(d84,d84_12,d84_15,1).
bond(d84,d84_2,d84_16,1).
bond(d84,d84_16,d84_17,2).
bond(d84,d84_16,d84_18,2).
atm(d85,d85_1,c,22,-0.112).
atm(d85,d85_2,c,22,-0.112).
atm(d85,d85_3,c,22,-0.112).
atm(d85,d85_4,c,27,-0.082).
atm(d85,d85_5,c,27,0.018).
atm(d85,d85_6,c,22,-0.112).
atm(d85,d85_7,c,27,-0.082).
atm(d85,d85_8,c,27,-0.082).
atm(d85,d85_9,c,26,-0.082).
atm(d85,d85_10,c,22,-0.112).
atm(d85,d85_11,c,22,-0.112).
atm(d85,d85_12,c,27,0.018).
atm(d85,d85_13,c,27,-0.082).
atm(d85,d85_14,c,27,-0.082).
atm(d85,d85_15,c,22,-0.112).
atm(d85,d85_16,c,22,-0.112).
atm(d85,d85_17,c,22,-0.112).
atm(d85,d85_18,c,22,-0.112).
atm(d85,d85_19,c,21,0.018).
atm(d85,d85_20,c,21,-0.112).
atm(d85,d85_21,h,3,0.147).
atm(d85,d85_22,h,3,0.148).
atm(d85,d85_23,h,3,0.148).
atm(d85,d85_24,h,3,0.147).
atm(d85,d85_25,h,3,0.147).
atm(d85,d85_26,h,3,0.147).
atm(d85,d85_27,h,3,0.147).
atm(d85,d85_28,h,3,0.147).
atm(d85,d85_29,h,3,0.147).
atm(d85,d85_30,h,3,0.147).
atm(d85,d85_31,h,3,0.147).
atm(d85,d85_32,n,38,0.817).
atm(d85,d85_33,o,40,-0.383).
atm(d85,d85_34,o,40,-0.383).

bond(d85,d85_1,d85_2,7).
bond(d85,d85_2,d85_3,7).
bond(d85,d85_3,d85_4,7).
bond(d85,d85_4,d85_5,7).
bond(d85,d85_5,d85_6,7).
bond(d85,d85_6,d85_1,7).
bond(d85,d85_4,d85_7,7).
bond(d85,d85_7,d85_8,7).
bond(d85,d85_8,d85_9,7).
bond(d85,d85_9,d85_10,7).
bond(d85,d85_10,d85_5,7).
bond(d85,d85_7,d85_11,7).
bond(d85,d85_11,d85_12,7).
bond(d85,d85_12,d85_13,7).
bond(d85,d85_13,d85_14,7).
bond(d85,d85_14,d85_8,7).
bond(d85,d85_12,d85_15,7).
bond(d85,d85_15,d85_16,7).
bond(d85,d85_16,d85_17,7).
bond(d85,d85_17,d85_18,7).
bond(d85,d85_18,d85_13,7).
bond(d85,d85_19,d85_9,7).
bond(d85,d85_19,d85_20,7).
bond(d85,d85_20,d85_14,7).
bond(d85,d85_1,d85_21,1).
bond(d85,d85_2,d85_22,1).
bond(d85,d85_3,d85_23,1).
bond(d85,d85_6,d85_24,1).
bond(d85,d85_10,d85_25,1).
bond(d85,d85_11,d85_26,1).
bond(d85,d85_15,d85_27,1).
bond(d85,d85_16,d85_28,1).
bond(d85,d85_17,d85_29,1).
bond(d85,d85_18,d85_30,1).
bond(d85,d85_20,d85_31,1).
bond(d85,d85_32,d85_19,1).
bond(d85,d85_33,d85_32,2).
bond(d85,d85_32,d85_34,2).
atm(d86,d86_1,c,22,-0.109).
atm(d86,d86_2,c,22,-0.109).
atm(d86,d86_3,c,22,-0.109).
atm(d86,d86_4,c,27,-0.079).
atm(d86,d86_5,c,27,0.021).
atm(d86,d86_6,c,22,-0.109).
atm(d86,d86_7,c,27,-0.079).
atm(d86,d86_8,c,27,-0.079).
atm(d86,d86_9,c,26,-0.079).
atm(d86,d86_10,c,22,-0.109).
atm(d86,d86_11,c,22,-0.109).
atm(d86,d86_12,c,27,0.021).
atm(d86,d86_13,c,27,-0.079).
atm(d86,d86_14,c,27,-0.079).
atm(d86,d86_15,c,22,-0.109).
atm(d86,d86_16,c,22,-0.108).
atm(d86,d86_17,c,22,-0.108).
atm(d86,d86_18,c,22,-0.109).
atm(d86,d86_19,c,21,-0.079).
atm(d86,d86_20,c,21,-0.109).
atm(d86,d86_21,h,3,0.152).
atm(d86,d86_22,h,3,0.152).
atm(d86,d86_23,h,3,0.152).
atm(d86,d86_24,h,3,0.152).
atm(d86,d86_25,h,3,0.152).
atm(d86,d86_26,h,3,0.152).
atm(d86,d86_27,h,3,0.152).
atm(d86,d86_28,h,3,0.152).
atm(d86,d86_29,h,3,0.152).
atm(d86,d86_30,h,3,0.152).
atm(d86,d86_31,h,3,0.122).
atm(d86,d86_32,n,38,0.822).
atm(d86,d86_33,o,40,-0.378).
atm(d86,d86_34,o,40,-0.378).

bond(d86,d86_1,d86_2,7).
bond(d86,d86_2,d86_3,7).
bond(d86,d86_3,d86_4,7).
bond(d86,d86_4,d86_5,7).
bond(d86,d86_5,d86_6,7).
bond(d86,d86_6,d86_1,7).
bond(d86,d86_4,d86_7,7).
bond(d86,d86_7,d86_8,7).
bond(d86,d86_8,d86_9,7).
bond(d86,d86_9,d86_10,7).
bond(d86,d86_10,d86_5,7).
bond(d86,d86_7,d86_11,7).
bond(d86,d86_11,d86_12,7).
bond(d86,d86_12,d86_13,7).
bond(d86,d86_13,d86_14,7).
bond(d86,d86_14,d86_8,7).
bond(d86,d86_12,d86_15,7).
bond(d86,d86_15,d86_16,7).
bond(d86,d86_16,d86_17,7).
bond(d86,d86_17,d86_18,7).
bond(d86,d86_18,d86_13,7).
bond(d86,d86_19,d86_9,7).
bond(d86,d86_19,d86_20,7).
bond(d86,d86_20,d86_14,7).
bond(d86,d86_1,d86_21,1).
bond(d86,d86_2,d86_22,1).
bond(d86,d86_3,d86_23,1).
bond(d86,d86_6,d86_24,1).
bond(d86,d86_11,d86_25,1).
bond(d86,d86_15,d86_26,1).
bond(d86,d86_16,d86_27,1).
bond(d86,d86_17,d86_28,1).
bond(d86,d86_18,d86_29,1).
bond(d86,d86_20,d86_30,1).
bond(d86,d86_19,d86_31,1).
bond(d86,d86_32,d86_10,1).
bond(d86,d86_33,d86_32,2).
bond(d86,d86_32,d86_34,2).
atm(d87,d87_1,c,22,-0.141).
atm(d87,d87_2,c,22,-0.141).
atm(d87,d87_3,c,26,0.09).
atm(d87,d87_4,c,26,0.09).
atm(d87,d87_5,c,22,-0.141).
atm(d87,d87_6,c,22,-0.141).
atm(d87,d87_7,c,26,0.09).
atm(d87,d87_8,c,22,-0.141).
atm(d87,d87_9,c,22,-0.141).
atm(d87,d87_10,c,22,-0.14).
atm(d87,d87_11,c,22,-0.141).
atm(d87,d87_12,c,26,0.09).
atm(d87,d87_13,o,52,-0.04).
atm(d87,d87_14,n,38,0.789).
atm(d87,d87_15,o,40,-0.411).
atm(d87,d87_16,o,40,-0.411).
atm(d87,d87_17,h,3,0.12).
atm(d87,d87_18,h,3,0.12).
atm(d87,d87_19,h,3,0.12).
atm(d87,d87_20,h,3,0.12).
atm(d87,d87_21,h,3,0.12).
atm(d87,d87_22,h,3,0.12).
atm(d87,d87_23,h,3,0.12).

bond(d87,d87_1,d87_2,7).
bond(d87,d87_2,d87_3,7).
bond(d87,d87_3,d87_4,7).
bond(d87,d87_4,d87_5,7).
bond(d87,d87_5,d87_6,7).
bond(d87,d87_6,d87_1,7).
bond(d87,d87_7,d87_8,7).
bond(d87,d87_8,d87_9,7).
bond(d87,d87_9,d87_10,7).
bond(d87,d87_10,d87_11,7).
bond(d87,d87_11,d87_12,7).
bond(d87,d87_12,d87_7,7).
bond(d87,d87_4,d87_12,7).
bond(d87,d87_3,d87_13,7).
bond(d87,d87_13,d87_7,7).
bond(d87,d87_9,d87_14,1).
bond(d87,d87_14,d87_15,2).
bond(d87,d87_14,d87_16,2).
bond(d87,d87_1,d87_17,1).
bond(d87,d87_2,d87_18,1).
bond(d87,d87_5,d87_19,1).
bond(d87,d87_6,d87_20,1).
bond(d87,d87_8,d87_21,1).
bond(d87,d87_10,d87_22,1).
bond(d87,d87_11,d87_23,1).
atm(d88,d88_1,c,22,-0.129).
atm(d88,d88_2,c,22,-0.129).
atm(d88,d88_3,c,22,-0.129).
atm(d88,d88_4,c,22,-0.129).
atm(d88,d88_5,c,22,0.001).
atm(d88,d88_6,c,22,-0.129).
atm(d88,d88_7,h,3,0.13).
atm(d88,d88_8,h,3,0.13).
atm(d88,d88_9,h,3,0.13).
atm(d88,d88_10,h,3,0.13).
atm(d88,d88_11,c,14,0.55).
atm(d88,d88_12,c,10,0.001).
atm(d88,d88_13,h,3,0.051).
atm(d88,d88_14,h,3,0.051).
atm(d88,d88_15,h,3,0.051).
atm(d88,d88_16,o,42,-0.58).
atm(d88,d88_17,n,38,0.8).
atm(d88,d88_18,o,40,-0.4).
atm(d88,d88_19,o,40,-0.4).

bond(d88,d88_1,d88_2,7).
bond(d88,d88_2,d88_3,7).
bond(d88,d88_3,d88_4,7).
bond(d88,d88_4,d88_5,7).
bond(d88,d88_5,d88_6,7).
bond(d88,d88_6,d88_1,7).
bond(d88,d88_1,d88_7,1).
bond(d88,d88_3,d88_8,1).
bond(d88,d88_4,d88_9,1).
bond(d88,d88_6,d88_10,1).
bond(d88,d88_11,d88_5,1).
bond(d88,d88_11,d88_12,1).
bond(d88,d88_12,d88_13,1).
bond(d88,d88_12,d88_14,1).
bond(d88,d88_12,d88_15,1).
bond(d88,d88_11,d88_16,2).
bond(d88,d88_2,d88_17,1).
bond(d88,d88_17,d88_18,2).
bond(d88,d88_17,d88_19,2).
atm(d89,d89_1,c,22,-0.107).
atm(d89,d89_2,c,22,-0.107).
atm(d89,d89_3,c,26,0.323).
atm(d89,d89_4,c,26,0.123).
atm(d89,d89_5,c,22,-0.107).
atm(d89,d89_6,c,22,-0.107).
atm(d89,d89_7,n,34,-0.517).
atm(d89,d89_8,n,34,-0.377).
atm(d89,d89_9,c,21,-0.107).
atm(d89,d89_10,c,10,0.113).
atm(d89,d89_11,n,38,0.823).
atm(d89,d89_12,o,40,-0.377).
atm(d89,d89_13,o,40,-0.377).
atm(d89,d89_14,h,3,0.153).
atm(d89,d89_15,h,3,0.123).
atm(d89,d89_16,h,3,0.153).
atm(d89,d89_17,h,3,0.153).
atm(d89,d89_18,h,3,0.073).
atm(d89,d89_19,h,3,0.073).
atm(d89,d89_20,h,3,0.073).

bond(d89,d89_1,d89_2,7).
bond(d89,d89_2,d89_3,7).
bond(d89,d89_3,d89_4,7).
bond(d89,d89_4,d89_5,7).
bond(d89,d89_5,d89_6,7).
bond(d89,d89_6,d89_1,7).
bond(d89,d89_3,d89_7,7).
bond(d89,d89_7,d89_8,7).
bond(d89,d89_4,d89_9,7).
bond(d89,d89_9,d89_8,7).
bond(d89,d89_7,d89_10,1).
bond(d89,d89_2,d89_11,1).
bond(d89,d89_11,d89_12,2).
bond(d89,d89_11,d89_13,2).
bond(d89,d89_1,d89_14,1).
bond(d89,d89_5,d89_15,1).
bond(d89,d89_6,d89_16,1).
bond(d89,d89_9,d89_17,1).
bond(d89,d89_10,d89_18,1).
bond(d89,d89_10,d89_19,1).
bond(d89,d89_10,d89_20,1).
atm(d9,d9_1,c,22,-0.102).
atm(d9,d9_2,c,22,-0.102).
atm(d9,d9_3,c,26,-0.012).
atm(d9,d9_4,c,26,-0.072).
atm(d9,d9_5,c,22,-0.102).
atm(d9,d9_6,c,22,-0.102).
atm(d9,d9_7,n,34,-0.511).
atm(d9,d9_8,c,21,0.298).
atm(d9,d9_9,n,34,-0.511).
atm(d9,d9_10,n,38,0.829).
atm(d9,d9_11,o,40,-0.372).
atm(d9,d9_12,o,40,-0.372).
atm(d9,d9_13,h,3,0.158).
atm(d9,d9_14,h,3,0.158).
atm(d9,d9_15,h,3,0.158).
atm(d9,d9_16,h,3,0.128).
atm(d9,d9_17,h,1,0.529).

bond(d9,d9_1,d9_2,7).
bond(d9,d9_2,d9_3,7).
bond(d9,d9_3,d9_4,7).
bond(d9,d9_4,d9_5,7).
bond(d9,d9_5,d9_6,7).
bond(d9,d9_6,d9_1,7).
bond(d9,d9_4,d9_7,7).
bond(d9,d9_7,d9_8,7).
bond(d9,d9_8,d9_9,7).
bond(d9,d9_9,d9_3,7).
bond(d9,d9_6,d9_10,1).
bond(d9,d9_10,d9_11,2).
bond(d9,d9_12,d9_10,2).
bond(d9,d9_1,d9_13,1).
bond(d9,d9_2,d9_14,1).
bond(d9,d9_5,d9_15,1).
bond(d9,d9_8,d9_16,1).
bond(d9,d9_9,d9_17,1).
atm(d90,d90_1,c,22,-0.109).
atm(d90,d90_2,c,22,-0.109).
atm(d90,d90_3,c,27,0.021).
atm(d90,d90_4,c,195,-0.079).
atm(d90,d90_5,c,195,-0.079).
atm(d90,d90_6,c,22,-0.109).
atm(d90,d90_7,h,3,0.152).
atm(d90,d90_8,h,3,0.152).
atm(d90,d90_9,h,3,0.152).
atm(d90,d90_10,c,22,-0.108).
atm(d90,d90_11,c,22,-0.109).
atm(d90,d90_12,c,27,0.021).
atm(d90,d90_13,c,195,-0.079).
atm(d90,d90_14,h,3,0.152).
atm(d90,d90_15,h,3,0.152).
atm(d90,d90_16,c,195,-0.079).
atm(d90,d90_17,c,27,-0.079).
atm(d90,d90_18,c,27,-0.079).
atm(d90,d90_19,c,22,-0.109).
atm(d90,d90_20,c,22,-0.108).
atm(d90,d90_21,c,22,-0.109).
atm(d90,d90_22,h,3,0.152).
atm(d90,d90_23,h,3,0.152).
atm(d90,d90_24,c,22,-0.109).
atm(d90,d90_25,c,22,-0.109).
atm(d90,d90_26,c,22,-0.109).
atm(d90,d90_27,c,22,-0.109).
atm(d90,d90_28,h,3,0.152).
atm(d90,d90_29,h,3,0.152).
atm(d90,d90_30,h,3,0.152).
atm(d90,d90_31,h,3,0.152).
atm(d90,d90_32,n,38,0.822).
atm(d90,d90_33,o,40,-0.378).
atm(d90,d90_34,o,40,-0.378).

bond(d90,d90_1,d90_2,7).
bond(d90,d90_2,d90_3,7).
bond(d90,d90_3,d90_4,7).
bond(d90,d90_4,d90_5,7).
bond(d90,d90_5,d90_6,7).
bond(d90,d90_6,d90_1,7).
bond(d90,d90_1,d90_7,1).
bond(d90,d90_2,d90_8,1).
bond(d90,d90_6,d90_9,1).
bond(d90,d90_3,d90_10,7).
bond(d90,d90_10,d90_11,7).
bond(d90,d90_11,d90_12,7).
bond(d90,d90_12,d90_13,7).
bond(d90,d90_13,d90_4,7).
bond(d90,d90_10,d90_14,1).
bond(d90,d90_11,d90_15,1).
bond(d90,d90_13,d90_16,7).
bond(d90,d90_16,d90_17,7).
bond(d90,d90_17,d90_18,7).
bond(d90,d90_18,d90_5,7).
bond(d90,d90_12,d90_19,7).
bond(d90,d90_19,d90_20,7).
bond(d90,d90_20,d90_21,7).
bond(d90,d90_21,d90_16,7).
bond(d90,d90_19,d90_22,1).
bond(d90,d90_20,d90_23,1).
bond(d90,d90_17,d90_24,7).
bond(d90,d90_24,d90_25,7).
bond(d90,d90_25,d90_26,7).
bond(d90,d90_26,d90_27,7).
bond(d90,d90_27,d90_18,7).
bond(d90,d90_24,d90_28,1).
bond(d90,d90_25,d90_29,1).
bond(d90,d90_26,d90_30,1).
bond(d90,d90_27,d90_31,1).
bond(d90,d90_32,d90_21,1).
bond(d90,d90_33,d90_32,2).
bond(d90,d90_32,d90_34,2).
atm(d91,d91_1,c,10,0.206).
atm(d91,d91_2,c,10,0.017).
atm(d91,d91_3,c,22,-0.114).
atm(d91,d91_4,c,22,0.017).
atm(d91,d91_5,c,10,-0.083).
atm(d91,d91_6,c,10,-0.083).
atm(d91,d91_7,c,22,-0.113).
atm(d91,d91_8,c,27,0.017).
atm(d91,d91_9,c,27,-0.083).
atm(d91,d91_10,c,27,0.017).
atm(d91,d91_11,c,22,-0.113).
atm(d91,d91_12,c,22,-0.113).
atm(d91,d91_13,c,27,0.017).
atm(d91,d91_14,c,27,-0.083).
atm(d91,d91_15,c,27,0.017).
atm(d91,d91_16,c,22,-0.113).
atm(d91,d91_17,c,22,-0.114).
atm(d91,d91_18,c,22,-0.114).
atm(d91,d91_19,c,22,-0.113).
atm(d91,d91_20,c,22,-0.113).
atm(d91,d91_21,n,38,0.816).
atm(d91,d91_22,o,40,-0.385).
atm(d91,d91_23,o,40,-0.385).
atm(d91,d91_24,h,3,0.067).
atm(d91,d91_25,h,3,0.067).
atm(d91,d91_26,h,3,0.067).
atm(d91,d91_27,h,3,0.067).
atm(d91,d91_28,h,3,0.146).
atm(d91,d91_29,h,3,0.146).
atm(d91,d91_30,h,3,0.146).
atm(d91,d91_31,h,3,0.146).
atm(d91,d91_32,h,3,0.146).
atm(d91,d91_33,h,3,0.146).
atm(d91,d91_34,h,3,0.146).
atm(d91,d91_35,o,45,-0.634).
atm(d91,d91_36,h,3,0.077).
atm(d91,d91_37,o,45,-0.634).
atm(d91,d91_38,h,3,0.067).
atm(d91,d91_39,h,8,0.416).
atm(d91,d91_40,h,8,0.416).

bond(d91,d91_1,d91_2,1).
bond(d91,d91_2,d91_3,1).
bond(d91,d91_3,d91_4,7).
bond(d91,d91_4,d91_5,1).
bond(d91,d91_5,d91_6,1).
bond(d91,d91_6,d91_1,1).
bond(d91,d91_3,d91_7,7).
bond(d91,d91_7,d91_8,7).
bond(d91,d91_8,d91_9,7).
bond(d91,d91_9,d91_10,7).
bond(d91,d91_10,d91_4,7).
bond(d91,d91_8,d91_11,7).
bond(d91,d91_11,d91_12,7).
bond(d91,d91_12,d91_13,7).
bond(d91,d91_13,d91_14,7).
bond(d91,d91_14,d91_9,7).
bond(d91,d91_14,d91_15,7).
bond(d91,d91_15,d91_16,7).
bond(d91,d91_16,d91_17,7).
bond(d91,d91_17,d91_10,7).
bond(d91,d91_13,d91_18,7).
bond(d91,d91_18,d91_19,7).
bond(d91,d91_19,d91_20,7).
bond(d91,d91_20,d91_15,7).
bond(d91,d91_18,d91_21,1).
bond(d91,d91_21,d91_22,2).
bond(d91,d91_21,d91_23,2).
bond(d91,d91_5,d91_24,1).
bond(d91,d91_5,d91_25,1).
bond(d91,d91_6,d91_26,1).
bond(d91,d91_6,d91_27,1).
bond(d91,d91_7,d91_28,1).
bond(d91,d91_11,d91_29,1).
bond(d91,d91_12,d91_30,1).
bond(d91,d91_16,d91_31,1).
bond(d91,d91_17,d91_32,1).
bond(d91,d91_19,d91_33,1).
bond(d91,d91_20,d91_34,1).
bond(d91,d91_1,d91_35,4).
bond(d91,d91_1,d91_36,1).
bond(d91,d91_2,d91_37,5).
bond(d91,d91_2,d91_38,1).
bond(d91,d91_35,d91_39,1).
bond(d91,d91_37,d91_40,1).
atm(d92,d92_1,c,22,-0.126).
atm(d92,d92_2,c,22,-0.126).
atm(d92,d92_3,c,22,0.004).
atm(d92,d92_4,c,29,0.004).
atm(d92,d92_5,c,22,0.004).
atm(d92,d92_6,c,22,-0.126).
atm(d92,d92_7,c,10,-0.096).
atm(d92,d92_8,c,10,-0.096).
atm(d92,d92_9,c,22,0.004).
atm(d92,d92_10,c,29,0.004).
atm(d92,d92_11,c,22,0.004).
atm(d92,d92_12,c,10,-0.096).
atm(d92,d92_13,c,10,-0.096).
atm(d92,d92_14,c,22,-0.126).
atm(d92,d92_15,c,22,-0.126).
atm(d92,d92_16,c,22,-0.126).
atm(d92,d92_17,n,38,0.805).
atm(d92,d92_18,o,40,-0.395).
atm(d92,d92_19,o,40,-0.396).
atm(d92,d92_20,h,3,0.134).
atm(d92,d92_21,h,3,0.134).
atm(d92,d92_22,h,3,0.134).
atm(d92,d92_23,h,3,0.134).
atm(d92,d92_24,h,3,0.134).
atm(d92,d92_25,h,3,0.054).
atm(d92,d92_26,h,3,0.054).
atm(d92,d92_27,h,3,0.054).
atm(d92,d92_28,h,3,0.054).
atm(d92,d92_29,h,3,0.054).
atm(d92,d92_30,h,3,0.054).
atm(d92,d92_31,h,3,0.054).
atm(d92,d92_32,h,3,0.054).

bond(d92,d92_1,d92_2,7).
bond(d92,d92_2,d92_3,7).
bond(d92,d92_3,d92_4,7).
bond(d92,d92_4,d92_5,7).
bond(d92,d92_5,d92_6,7).
bond(d92,d92_6,d92_1,7).
bond(d92,d92_3,d92_7,1).
bond(d92,d92_7,d92_8,1).
bond(d92,d92_8,d92_9,1).
bond(d92,d92_9,d92_10,7).
bond(d92,d92_10,d92_4,1).
bond(d92,d92_10,d92_11,7).
bond(d92,d92_11,d92_12,1).
bond(d92,d92_12,d92_13,1).
bond(d92,d92_13,d92_5,1).
bond(d92,d92_9,d92_14,7).
bond(d92,d92_14,d92_15,7).
bond(d92,d92_15,d92_16,7).
bond(d92,d92_16,d92_11,7).
bond(d92,d92_15,d92_17,1).
bond(d92,d92_17,d92_18,2).
bond(d92,d92_17,d92_19,2).
bond(d92,d92_1,d92_20,1).
bond(d92,d92_2,d92_21,1).
bond(d92,d92_6,d92_22,1).
bond(d92,d92_14,d92_23,1).
bond(d92,d92_16,d92_24,1).
bond(d92,d92_13,d92_25,1).
bond(d92,d92_13,d92_26,1).
bond(d92,d92_12,d92_27,1).
bond(d92,d92_12,d92_28,1).
bond(d92,d92_8,d92_29,1).
bond(d92,d92_8,d92_30,1).
bond(d92,d92_7,d92_31,1).
bond(d92,d92_7,d92_32,1).
atm(d93,d93_1,c,22,-0.109).
atm(d93,d93_2,c,22,-0.109).
atm(d93,d93_3,c,27,0.021).
atm(d93,d93_4,c,195,-0.079).
atm(d93,d93_5,c,195,-0.079).
atm(d93,d93_6,c,22,-0.109).
atm(d93,d93_7,h,3,0.152).
atm(d93,d93_8,h,3,0.152).
atm(d93,d93_9,h,3,0.152).
atm(d93,d93_10,c,22,-0.108).
atm(d93,d93_11,c,22,-0.109).
atm(d93,d93_12,c,27,0.021).
atm(d93,d93_13,c,195,-0.079).
atm(d93,d93_14,h,3,0.152).
atm(d93,d93_15,h,3,0.152).
atm(d93,d93_16,c,195,-0.079).
atm(d93,d93_17,c,27,-0.079).
atm(d93,d93_18,c,27,-0.079).
atm(d93,d93_19,c,22,-0.109).
atm(d93,d93_20,c,22,-0.108).
atm(d93,d93_21,c,22,-0.109).
atm(d93,d93_22,h,3,0.152).
atm(d93,d93_23,h,3,0.152).
atm(d93,d93_24,c,22,-0.109).
atm(d93,d93_25,c,22,-0.109).
atm(d93,d93_26,c,22,-0.109).
atm(d93,d93_27,c,22,-0.109).
atm(d93,d93_28,h,3,0.152).
atm(d93,d93_29,h,3,0.152).
atm(d93,d93_30,h,3,0.152).
atm(d93,d93_31,h,3,0.152).
atm(d93,d93_32,n,38,0.822).
atm(d93,d93_33,o,40,-0.378).
atm(d93,d93_34,o,40,-0.378).

bond(d93,d93_1,d93_2,7).
bond(d93,d93_2,d93_3,7).
bond(d93,d93_3,d93_4,7).
bond(d93,d93_4,d93_5,7).
bond(d93,d93_5,d93_6,7).
bond(d93,d93_6,d93_1,7).
bond(d93,d93_1,d93_7,1).
bond(d93,d93_2,d93_8,1).
bond(d93,d93_6,d93_9,1).
bond(d93,d93_3,d93_10,7).
bond(d93,d93_10,d93_11,7).
bond(d93,d93_11,d93_12,7).
bond(d93,d93_12,d93_13,7).
bond(d93,d93_13,d93_4,7).
bond(d93,d93_10,d93_14,1).
bond(d93,d93_11,d93_15,1).
bond(d93,d93_13,d93_16,7).
bond(d93,d93_16,d93_17,7).
bond(d93,d93_17,d93_18,7).
bond(d93,d93_18,d93_5,7).
bond(d93,d93_12,d93_19,7).
bond(d93,d93_19,d93_20,7).
bond(d93,d93_20,d93_21,7).
bond(d93,d93_21,d93_16,7).
bond(d93,d93_20,d93_22,1).
bond(d93,d93_21,d93_23,1).
bond(d93,d93_17,d93_24,7).
bond(d93,d93_24,d93_25,7).
bond(d93,d93_25,d93_26,7).
bond(d93,d93_26,d93_27,7).
bond(d93,d93_27,d93_18,7).
bond(d93,d93_24,d93_28,1).
bond(d93,d93_25,d93_29,1).
bond(d93,d93_26,d93_30,1).
bond(d93,d93_27,d93_31,1).
bond(d93,d93_19,d93_32,1).
bond(d93,d93_32,d93_33,2).
bond(d93,d93_32,d93_34,2).
atm(d94,d94_1,c,22,-0.118).
atm(d94,d94_2,c,22,-0.118).
atm(d94,d94_3,c,27,0.012).
atm(d94,d94_4,c,27,-0.088).
atm(d94,d94_5,c,27,0.012).
atm(d94,d94_6,c,22,-0.118).
atm(d94,d94_7,h,3,0.142).
atm(d94,d94_8,h,3,0.141).
atm(d94,d94_9,h,3,0.141).
atm(d94,d94_10,c,22,-0.118).
atm(d94,d94_11,c,22,-0.118).
atm(d94,d94_12,c,27,0.012).
atm(d94,d94_13,c,27,-0.088).
atm(d94,d94_14,h,3,0.141).
atm(d94,d94_15,c,27,0.012).
atm(d94,d94_16,c,22,-0.118).
atm(d94,d94_17,c,22,-0.118).
atm(d94,d94_18,h,3,0.142).
atm(d94,d94_19,h,3,0.142).
atm(d94,d94_20,c,22,-0.118).
atm(d94,d94_21,c,22,-0.118).
atm(d94,d94_22,c,22,-0.118).
atm(d94,d94_23,h,3,0.142).
atm(d94,d94_24,h,3,0.142).
atm(d94,d94_25,h,3,0.142).
atm(d94,d94_26,n,38,0.811).
atm(d94,d94_27,o,40,-0.389).
atm(d94,d94_28,o,40,-0.389).

bond(d94,d94_1,d94_2,7).
bond(d94,d94_2,d94_3,7).
bond(d94,d94_3,d94_4,7).
bond(d94,d94_4,d94_5,7).
bond(d94,d94_5,d94_6,7).
bond(d94,d94_6,d94_1,7).
bond(d94,d94_1,d94_7,1).
bond(d94,d94_2,d94_8,1).
bond(d94,d94_6,d94_9,1).
bond(d94,d94_3,d94_10,7).
bond(d94,d94_10,d94_11,7).
bond(d94,d94_11,d94_12,7).
bond(d94,d94_12,d94_13,7).
bond(d94,d94_13,d94_4,7).
bond(d94,d94_10,d94_14,1).
bond(d94,d94_13,d94_15,7).
bond(d94,d94_15,d94_16,7).
bond(d94,d94_16,d94_17,7).
bond(d94,d94_17,d94_5,7).
bond(d94,d94_16,d94_18,1).
bond(d94,d94_17,d94_19,1).
bond(d94,d94_12,d94_20,7).
bond(d94,d94_20,d94_21,7).
bond(d94,d94_21,d94_22,7).
bond(d94,d94_22,d94_15,7).
bond(d94,d94_20,d94_23,1).
bond(d94,d94_21,d94_24,1).
bond(d94,d94_22,d94_25,1).
bond(d94,d94_11,d94_26,1).
bond(d94,d94_26,d94_27,2).
bond(d94,d94_26,d94_28,2).
atm(d95,d95_1,c,22,-0.114).
atm(d95,d95_2,c,22,-0.114).
atm(d95,d95_3,c,27,0.016).
atm(d95,d95_4,c,27,0.016).
atm(d95,d95_5,c,22,-0.114).
atm(d95,d95_6,c,22,-0.114).
atm(d95,d95_7,h,3,0.146).
atm(d95,d95_8,h,3,0.146).
atm(d95,d95_9,h,3,0.146).
atm(d95,d95_10,c,22,-0.114).
atm(d95,d95_11,c,22,-0.114).
atm(d95,d95_12,c,22,-0.114).
atm(d95,d95_13,c,22,-0.114).
atm(d95,d95_14,h,3,0.146).
atm(d95,d95_15,h,3,0.146).
atm(d95,d95_16,n,38,0.817).
atm(d95,d95_17,n,38,0.817).
atm(d95,d95_18,n,38,0.817).
atm(d95,d95_19,o,40,-0.384).
atm(d95,d95_20,o,40,-0.384).
atm(d95,d95_21,o,40,-0.383).
atm(d95,d95_22,o,40,-0.384).
atm(d95,d95_23,o,40,-0.383).
atm(d95,d95_24,o,40,-0.383).

bond(d95,d95_1,d95_2,7).
bond(d95,d95_2,d95_3,7).
bond(d95,d95_3,d95_4,7).
bond(d95,d95_4,d95_5,7).
bond(d95,d95_5,d95_6,7).
bond(d95,d95_6,d95_1,7).
bond(d95,d95_1,d95_7,1).
bond(d95,d95_5,d95_8,1).
bond(d95,d95_6,d95_9,1).
bond(d95,d95_3,d95_10,7).
bond(d95,d95_10,d95_11,7).
bond(d95,d95_11,d95_12,7).
bond(d95,d95_12,d95_13,7).
bond(d95,d95_13,d95_4,7).
bond(d95,d95_10,d95_14,1).
bond(d95,d95_13,d95_15,1).
bond(d95,d95_12,d95_16,1).
bond(d95,d95_11,d95_17,1).
bond(d95,d95_2,d95_18,1).
bond(d95,d95_16,d95_19,2).
bond(d95,d95_16,d95_20,2).
bond(d95,d95_17,d95_21,2).
bond(d95,d95_17,d95_22,2).
bond(d95,d95_18,d95_23,2).
bond(d95,d95_18,d95_24,2).
atm(d96,d96_1,c,22,-0.112).
atm(d96,d96_2,c,22,-0.112).
atm(d96,d96_3,c,27,0.018).
atm(d96,d96_4,c,27,-0.082).
atm(d96,d96_5,c,27,0.018).
atm(d96,d96_6,c,22,-0.112).
atm(d96,d96_7,h,3,0.149).
atm(d96,d96_8,h,3,0.149).
atm(d96,d96_9,c,22,-0.112).
atm(d96,d96_10,c,22,-0.112).
atm(d96,d96_11,c,27,0.018).
atm(d96,d96_12,c,27,-0.082).
atm(d96,d96_13,h,3,0.149).
atm(d96,d96_14,h,3,0.148).
atm(d96,d96_15,c,27,0.018).
atm(d96,d96_16,c,22,-0.112).
atm(d96,d96_17,c,22,-0.112).
atm(d96,d96_18,h,3,0.149).
atm(d96,d96_19,h,3,0.149).
atm(d96,d96_20,c,22,-0.112).
atm(d96,d96_21,c,22,-0.112).
atm(d96,d96_22,c,22,-0.112).
atm(d96,d96_23,h,3,0.148).
atm(d96,d96_24,n,38,0.819).
atm(d96,d96_25,n,38,0.819).
atm(d96,d96_26,n,38,0.819).
atm(d96,d96_27,o,40,-0.381).
atm(d96,d96_28,o,40,-0.381).
atm(d96,d96_29,o,40,-0.381).
atm(d96,d96_30,o,40,-0.381).
atm(d96,d96_31,o,40,-0.381).
atm(d96,d96_32,o,40,-0.381).

bond(d96,d96_1,d96_2,7).
bond(d96,d96_2,d96_3,7).
bond(d96,d96_3,d96_4,7).
bond(d96,d96_4,d96_5,7).
bond(d96,d96_5,d96_6,7).
bond(d96,d96_6,d96_1,7).
bond(d96,d96_1,d96_7,1).
bond(d96,d96_6,d96_8,1).
bond(d96,d96_3,d96_9,7).
bond(d96,d96_9,d96_10,7).
bond(d96,d96_10,d96_11,7).
bond(d96,d96_11,d96_12,7).
bond(d96,d96_12,d96_4,7).
bond(d96,d96_9,d96_13,1).
bond(d96,d96_10,d96_14,1).
bond(d96,d96_12,d96_15,7).
bond(d96,d96_15,d96_16,7).
bond(d96,d96_16,d96_17,7).
bond(d96,d96_17,d96_5,7).
bond(d96,d96_16,d96_18,1).
bond(d96,d96_17,d96_19,1).
bond(d96,d96_11,d96_20,7).
bond(d96,d96_20,d96_21,7).
bond(d96,d96_21,d96_22,7).
bond(d96,d96_22,d96_15,7).
bond(d96,d96_21,d96_23,1).
bond(d96,d96_22,d96_24,1).
bond(d96,d96_20,d96_25,1).
bond(d96,d96_2,d96_26,1).
bond(d96,d96_24,d96_27,2).
bond(d96,d96_24,d96_28,2).
bond(d96,d96_25,d96_29,2).
bond(d96,d96_25,d96_30,2).
bond(d96,d96_26,d96_31,2).
bond(d96,d96_26,d96_32,2).
atm(d97,d97_1,c,22,-0.118).
atm(d97,d97_2,c,22,-0.118).
atm(d97,d97_3,c,22,-0.119).
atm(d97,d97_4,c,29,0.011).
atm(d97,d97_5,c,22,-0.119).
atm(d97,d97_6,c,22,-0.119).
atm(d97,d97_7,h,3,0.142).
atm(d97,d97_8,h,3,0.142).
atm(d97,d97_9,h,3,0.142).
atm(d97,d97_10,c,22,-0.119).
atm(d97,d97_11,c,22,-0.119).
atm(d97,d97_12,c,22,-0.168).
atm(d97,d97_13,c,22,-0.119).
atm(d97,d97_14,c,22,-0.119).
atm(d97,d97_15,c,29,0.011).
atm(d97,d97_16,h,3,0.142).
atm(d97,d97_17,h,3,0.142).
atm(d97,d97_18,h,3,0.142).
atm(d97,d97_19,c,10,0.011).
atm(d97,d97_20,h,3,0.061).
atm(d97,d97_21,h,3,0.061).
atm(d97,d97_22,c,10,0.011).
atm(d97,d97_23,h,3,0.061).
atm(d97,d97_24,h,3,0.061).
atm(d97,d97_25,h,3,0.061).
atm(d97,d97_26,n,38,0.812).
atm(d97,d97_27,o,40,-0.388).
atm(d97,d97_28,o,40,-0.388).

bond(d97,d97_1,d97_2,7).
bond(d97,d97_2,d97_3,7).
bond(d97,d97_3,d97_4,7).
bond(d97,d97_4,d97_5,7).
bond(d97,d97_5,d97_6,7).
bond(d97,d97_6,d97_1,7).
bond(d97,d97_2,d97_7,1).
bond(d97,d97_5,d97_8,1).
bond(d97,d97_6,d97_9,1).
bond(d97,d97_10,d97_11,7).
bond(d97,d97_11,d97_12,7).
bond(d97,d97_12,d97_13,7).
bond(d97,d97_13,d97_14,7).
bond(d97,d97_14,d97_15,7).
bond(d97,d97_15,d97_10,7).
bond(d97,d97_11,d97_16,1).
bond(d97,d97_13,d97_17,1).
bond(d97,d97_14,d97_18,1).
bond(d97,d97_4,d97_15,1).
bond(d97,d97_3,d97_19,1).
bond(d97,d97_19,d97_10,1).
bond(d97,d97_19,d97_20,1).
bond(d97,d97_19,d97_21,1).
bond(d97,d97_12,d97_22,1).
bond(d97,d97_22,d97_23,1).
bond(d97,d97_22,d97_24,1).
bond(d97,d97_22,d97_25,1).
bond(d97,d97_1,d97_26,1).
bond(d97,d97_26,d97_27,2).
bond(d97,d97_26,d97_28,2).
atm(d98,d98_1,c,22,-0.115).
atm(d98,d98_2,c,22,-0.115).
atm(d98,d98_3,c,27,0.015).
atm(d98,d98_4,c,27,0.015).
atm(d98,d98_5,c,22,-0.115).
atm(d98,d98_6,c,22,-0.115).
atm(d98,d98_7,h,3,0.145).
atm(d98,d98_8,h,3,0.145).
atm(d98,d98_9,h,3,0.145).
atm(d98,d98_10,c,22,-0.115).
atm(d98,d98_11,n,35,-0.524).
atm(d98,d98_12,c,22,0.266).
atm(d98,d98_13,c,22,-0.115).
atm(d98,d98_14,h,3,0.145).
atm(d98,d98_15,h,3,0.145).
atm(d98,d98_16,h,3,0.145).
atm(d98,d98_17,n,38,0.816).
atm(d98,d98_18,o,40,-0.384).
atm(d98,d98_19,o,40,-0.384).

bond(d98,d98_1,d98_2,7).
bond(d98,d98_2,d98_3,7).
bond(d98,d98_3,d98_4,7).
bond(d98,d98_4,d98_5,7).
bond(d98,d98_5,d98_6,7).
bond(d98,d98_6,d98_1,7).
bond(d98,d98_1,d98_7,1).
bond(d98,d98_2,d98_8,1).
bond(d98,d98_6,d98_9,1).
bond(d98,d98_3,d98_10,7).
bond(d98,d98_10,d98_11,7).
bond(d98,d98_11,d98_12,7).
bond(d98,d98_12,d98_13,7).
bond(d98,d98_13,d98_4,7).
bond(d98,d98_10,d98_14,1).
bond(d98,d98_12,d98_15,1).
bond(d98,d98_13,d98_16,1).
bond(d98,d98_5,d98_17,1).
bond(d98,d98_18,d98_17,2).
bond(d98,d98_17,d98_19,2).
atm(d99,d99_1,c,22,-0.116).
atm(d99,d99_2,c,22,-0.116).
atm(d99,d99_3,c,27,0.014).
atm(d99,d99_4,c,27,-0.086).
atm(d99,d99_5,c,26,-0.086).
atm(d99,d99_6,c,22,-0.116).
atm(d99,d99_7,c,22,-0.116).
atm(d99,d99_8,c,22,-0.116).
atm(d99,d99_9,c,22,-0.116).
atm(d99,d99_10,c,26,-0.086).
atm(d99,d99_11,c,21,0.014).
atm(d99,d99_12,c,21,-0.086).
atm(d99,d99_13,n,38,0.813).
atm(d99,d99_14,o,40,-0.387).
atm(d99,d99_15,o,40,-0.387).
atm(d99,d99_16,h,3,0.143).
atm(d99,d99_17,h,3,0.143).
atm(d99,d99_18,h,3,0.143).
atm(d99,d99_19,h,3,0.143).
atm(d99,d99_20,h,3,0.144).
atm(d99,d99_21,h,3,0.143).
atm(d99,d99_22,h,3,0.114).

bond(d99,d99_1,d99_2,7).
bond(d99,d99_2,d99_3,7).
bond(d99,d99_3,d99_4,7).
bond(d99,d99_4,d99_5,7).
bond(d99,d99_5,d99_6,7).
bond(d99,d99_6,d99_1,7).
bond(d99,d99_3,d99_7,7).
bond(d99,d99_7,d99_8,7).
bond(d99,d99_8,d99_9,7).
bond(d99,d99_9,d99_10,7).
bond(d99,d99_10,d99_4,7).
bond(d99,d99_5,d99_11,7).
bond(d99,d99_11,d99_12,7).
bond(d99,d99_12,d99_10,7).
bond(d99,d99_11,d99_13,1).
bond(d99,d99_13,d99_14,2).
bond(d99,d99_13,d99_15,2).
bond(d99,d99_1,d99_16,1).
bond(d99,d99_2,d99_17,1).
bond(d99,d99_6,d99_18,1).
bond(d99,d99_7,d99_19,1).
bond(d99,d99_8,d99_20,1).
bond(d99,d99_9,d99_21,1).
bond(d99,d99_12,d99_22,1).
atm(e1,e1_1,c,22,-0.114).
atm(e1,e1_2,c,22,-0.114).
atm(e1,e1_3,c,27,0.016).
atm(e1,e1_4,c,27,-0.084).
atm(e1,e1_5,c,22,-0.114).
atm(e1,e1_6,c,22,-0.114).
atm(e1,e1_7,h,3,0.145).
atm(e1,e1_8,h,3,0.145).
atm(e1,e1_9,h,3,0.145).
atm(e1,e1_10,h,3,0.145).
atm(e1,e1_11,c,22,-0.114).
atm(e1,e1_12,c,27,0.016).
atm(e1,e1_13,c,195,-0.084).
atm(e1,e1_14,c,195,-0.084).
atm(e1,e1_15,c,22,-0.114).
atm(e1,e1_16,c,22,-0.114).
atm(e1,e1_17,c,27,0.016).
atm(e1,e1_18,c,27,-0.084).
atm(e1,e1_19,h,3,0.145).
atm(e1,e1_20,h,3,0.145).
atm(e1,e1_21,c,27,0.016).
atm(e1,e1_22,c,22,-0.114).
atm(e1,e1_23,c,22,-0.114).
atm(e1,e1_24,h,3,0.145).
atm(e1,e1_25,h,3,0.145).
atm(e1,e1_26,c,22,-0.114).
atm(e1,e1_27,c,22,-0.114).
atm(e1,e1_28,c,22,-0.114).
atm(e1,e1_29,h,3,0.145).
atm(e1,e1_30,h,3,0.145).
atm(e1,e1_31,h,3,0.145).
atm(e1,e1_32,n,38,0.815).
atm(e1,e1_33,o,40,-0.385).
atm(e1,e1_34,o,40,-0.385).

bond(e1,e1_1,e1_2,7).
bond(e1,e1_2,e1_3,7).
bond(e1,e1_3,e1_4,7).
bond(e1,e1_4,e1_5,7).
bond(e1,e1_5,e1_6,7).
bond(e1,e1_6,e1_1,7).
bond(e1,e1_1,e1_7,1).
bond(e1,e1_2,e1_8,1).
bond(e1,e1_5,e1_9,1).
bond(e1,e1_6,e1_10,1).
bond(e1,e1_3,e1_11,7).
bond(e1,e1_11,e1_12,7).
bond(e1,e1_12,e1_13,7).
bond(e1,e1_13,e1_14,7).
bond(e1,e1_14,e1_4,7).
bond(e1,e1_12,e1_15,7).
bond(e1,e1_15,e1_16,7).
bond(e1,e1_16,e1_17,7).
bond(e1,e1_17,e1_18,7).
bond(e1,e1_18,e1_13,7).
bond(e1,e1_15,e1_19,1).
bond(e1,e1_16,e1_20,1).
bond(e1,e1_18,e1_21,7).
bond(e1,e1_21,e1_22,7).
bond(e1,e1_22,e1_23,7).
bond(e1,e1_23,e1_14,7).
bond(e1,e1_22,e1_24,1).
bond(e1,e1_23,e1_25,1).
bond(e1,e1_17,e1_26,7).
bond(e1,e1_26,e1_27,7).
bond(e1,e1_27,e1_28,7).
bond(e1,e1_28,e1_21,7).
bond(e1,e1_26,e1_29,1).
bond(e1,e1_27,e1_30,1).
bond(e1,e1_28,e1_31,1).
bond(e1,e1_11,e1_32,1).
bond(e1,e1_32,e1_33,2).
bond(e1,e1_32,e1_34,2).
atm(e10,e10_1,c,22,-0.138).
atm(e10,e10_2,c,22,-0.138).
atm(e10,e10_3,c,26,-0.048).
atm(e10,e10_4,c,26,0.092).
atm(e10,e10_5,c,22,-0.138).
atm(e10,e10_6,c,22,-0.138).
atm(e10,e10_7,n,34,-0.367).
atm(e10,e10_8,c,21,0.162).
atm(e10,e10_9,c,21,-0.108).
atm(e10,e10_10,n,38,0.793).
atm(e10,e10_11,o,40,-0.407).
atm(e10,e10_12,o,40,-0.407).
atm(e10,e10_13,h,3,0.122).
atm(e10,e10_14,h,3,0.092).
atm(e10,e10_15,h,3,0.122).
atm(e10,e10_16,h,3,0.122).
atm(e10,e10_17,h,3,0.092).
atm(e10,e10_18,h,1,0.292).

bond(e10,e10_1,e10_2,7).
bond(e10,e10_2,e10_3,7).
bond(e10,e10_3,e10_4,7).
bond(e10,e10_4,e10_5,7).
bond(e10,e10_5,e10_6,7).
bond(e10,e10_6,e10_1,7).
bond(e10,e10_3,e10_7,7).
bond(e10,e10_7,e10_8,7).
bond(e10,e10_4,e10_9,7).
bond(e10,e10_9,e10_8,7).
bond(e10,e10_2,e10_10,1).
bond(e10,e10_10,e10_11,2).
bond(e10,e10_10,e10_12,2).
bond(e10,e10_1,e10_13,1).
bond(e10,e10_5,e10_14,1).
bond(e10,e10_6,e10_15,1).
bond(e10,e10_8,e10_16,1).
bond(e10,e10_9,e10_17,1).
bond(e10,e10_7,e10_18,1).
atm(e11,e11_1,c,22,-0.121).
atm(e11,e11_2,c,22,-0.121).
atm(e11,e11_3,c,22,0.31).
atm(e11,e11_4,c,22,-0.121).
atm(e11,e11_5,c,22,-0.121).
atm(e11,e11_6,c,22,-0.121).
atm(e11,e11_7,h,3,0.139).
atm(e11,e11_8,h,3,0.139).
atm(e11,e11_9,h,3,0.139).
atm(e11,e11_10,n,38,0.81).
atm(e11,e11_11,o,40,-0.39).
atm(e11,e11_12,o,40,-0.39).
atm(e11,e11_13,h,3,0.139).
atm(e11,e11_14,f,92,-0.291).

bond(e11,e11_1,e11_2,7).
bond(e11,e11_2,e11_3,7).
bond(e11,e11_3,e11_4,7).
bond(e11,e11_4,e11_5,7).
bond(e11,e11_5,e11_6,7).
bond(e11,e11_6,e11_1,7).
bond(e11,e11_1,e11_7,1).
bond(e11,e11_2,e11_8,1).
bond(e11,e11_6,e11_9,1).
bond(e11,e11_5,e11_10,1).
bond(e11,e11_10,e11_11,2).
bond(e11,e11_10,e11_12,2).
bond(e11,e11_4,e11_13,1).
bond(e11,e11_3,e11_14,1).
atm(e12,e12_1,c,22,-0.114).
atm(e12,e12_2,c,22,-0.114).
atm(e12,e12_3,c,22,-0.114).
atm(e12,e12_4,c,22,-0.114).
atm(e12,e12_5,c,22,-0.114).
atm(e12,e12_6,c,22,-0.114).
atm(e12,e12_7,h,3,0.146).
atm(e12,e12_8,h,3,0.146).
atm(e12,e12_9,h,3,0.146).
atm(e12,e12_10,n,38,0.817).
atm(e12,e12_11,o,40,-0.384).
atm(e12,e12_12,o,40,-0.384).
atm(e12,e12_13,h,3,0.146).
atm(e12,e12_14,n,38,0.817).
atm(e12,e12_15,o,40,-0.383).
atm(e12,e12_16,o,40,-0.383).

bond(e12,e12_1,e12_2,7).
bond(e12,e12_2,e12_3,7).
bond(e12,e12_3,e12_4,7).
bond(e12,e12_4,e12_5,7).
bond(e12,e12_5,e12_6,7).
bond(e12,e12_6,e12_1,7).
bond(e12,e12_1,e12_7,1).
bond(e12,e12_2,e12_8,1).
bond(e12,e12_6,e12_9,1).
bond(e12,e12_5,e12_10,1).
bond(e12,e12_10,e12_11,2).
bond(e12,e12_10,e12_12,2).
bond(e12,e12_3,e12_13,1).
bond(e12,e12_4,e12_14,1).
bond(e12,e12_14,e12_15,2).
bond(e12,e12_14,e12_16,2).
atm(e13,e13_1,c,22,-0.097).
atm(e13,e13_2,c,22,-0.097).
atm(e13,e13_3,c,22,-0.097).
atm(e13,e13_4,c,22,-0.097).
atm(e13,e13_5,c,22,-0.097).
atm(e13,e13_6,c,22,-0.097).
atm(e13,e13_7,h,3,0.163).
atm(e13,e13_8,h,3,0.163).
atm(e13,e13_9,h,3,0.163).
atm(e13,e13_10,n,38,0.832).
atm(e13,e13_11,o,40,-0.368).
atm(e13,e13_12,o,40,-0.367).
atm(e13,e13_13,h,3,0.163).
atm(e13,e13_14,cl,93,-0.167).

bond(e13,e13_1,e13_2,7).
bond(e13,e13_2,e13_3,7).
bond(e13,e13_3,e13_4,7).
bond(e13,e13_4,e13_5,7).
bond(e13,e13_5,e13_6,7).
bond(e13,e13_6,e13_1,7).
bond(e13,e13_1,e13_7,1).
bond(e13,e13_2,e13_8,1).
bond(e13,e13_6,e13_9,1).
bond(e13,e13_5,e13_10,1).
bond(e13,e13_10,e13_11,2).
bond(e13,e13_10,e13_12,2).
bond(e13,e13_4,e13_13,1).
bond(e13,e13_14,e13_3,1).
atm(e14,e14_1,c,22,-0.097).
atm(e14,e14_2,c,22,-0.097).
atm(e14,e14_3,c,22,-0.097).
atm(e14,e14_4,c,22,-0.097).
atm(e14,e14_5,c,22,-0.097).
atm(e14,e14_6,c,22,-0.097).
atm(e14,e14_7,h,3,0.163).
atm(e14,e14_8,h,3,0.163).
atm(e14,e14_9,n,38,0.832).
atm(e14,e14_10,o,40,-0.367).
atm(e14,e14_11,o,40,-0.368).
atm(e14,e14_12,h,3,0.163).
atm(e14,e14_13,cl,93,-0.167).
atm(e14,e14_14,h,3,0.163).

bond(e14,e14_1,e14_2,7).
bond(e14,e14_2,e14_3,7).
bond(e14,e14_3,e14_4,7).
bond(e14,e14_4,e14_5,7).
bond(e14,e14_5,e14_6,7).
bond(e14,e14_6,e14_1,7).
bond(e14,e14_1,e14_7,1).
bond(e14,e14_6,e14_8,1).
bond(e14,e14_5,e14_9,1).
bond(e14,e14_9,e14_10,2).
bond(e14,e14_9,e14_11,2).
bond(e14,e14_4,e14_12,1).
bond(e14,e14_2,e14_13,1).
bond(e14,e14_3,e14_14,1).
atm(e15,e15_1,c,22,-0.12).
atm(e15,e15_2,c,22,-0.12).
atm(e15,e15_3,c,27,0.01).
atm(e15,e15_4,c,27,0.01).
atm(e15,e15_5,c,22,-0.12).
atm(e15,e15_6,c,22,-0.12).
atm(e15,e15_7,h,3,0.141).
atm(e15,e15_8,h,3,0.141).
atm(e15,e15_9,h,3,0.141).
atm(e15,e15_10,h,3,0.141).
atm(e15,e15_11,c,22,-0.12).
atm(e15,e15_12,c,27,0.01).
atm(e15,e15_13,c,195,-0.09).
atm(e15,e15_14,c,22,-0.12).
atm(e15,e15_15,h,3,0.141).
atm(e15,e15_16,c,22,-0.12).
atm(e15,e15_17,c,22,-0.12).
atm(e15,e15_18,c,27,0.01).
atm(e15,e15_19,c,195,-0.09).
atm(e15,e15_20,h,3,0.14).
atm(e15,e15_21,h,3,0.141).
atm(e15,e15_22,c,22,-0.12).
atm(e15,e15_23,c,22,-0.12).
atm(e15,e15_24,c,22,-0.12).
atm(e15,e15_25,c,22,-0.12).
atm(e15,e15_26,h,3,0.14).
atm(e15,e15_27,h,3,0.14).
atm(e15,e15_28,h,3,0.14).
atm(e15,e15_29,h,3,0.141).
atm(e15,e15_30,n,38,0.811).
atm(e15,e15_31,o,40,-0.389).
atm(e15,e15_32,o,40,-0.389).

bond(e15,e15_1,e15_2,7).
bond(e15,e15_2,e15_3,7).
bond(e15,e15_3,e15_4,7).
bond(e15,e15_4,e15_5,7).
bond(e15,e15_5,e15_6,7).
bond(e15,e15_6,e15_1,7).
bond(e15,e15_1,e15_7,1).
bond(e15,e15_2,e15_8,1).
bond(e15,e15_5,e15_9,1).
bond(e15,e15_6,e15_10,1).
bond(e15,e15_3,e15_11,7).
bond(e15,e15_11,e15_12,7).
bond(e15,e15_12,e15_13,7).
bond(e15,e15_13,e15_14,7).
bond(e15,e15_14,e15_4,7).
bond(e15,e15_14,e15_15,1).
bond(e15,e15_12,e15_16,7).
bond(e15,e15_16,e15_17,7).
bond(e15,e15_17,e15_18,7).
bond(e15,e15_18,e15_19,7).
bond(e15,e15_19,e15_13,7).
bond(e15,e15_16,e15_20,1).
bond(e15,e15_17,e15_21,1).
bond(e15,e15_18,e15_22,7).
bond(e15,e15_22,e15_23,7).
bond(e15,e15_23,e15_24,7).
bond(e15,e15_24,e15_25,7).
bond(e15,e15_25,e15_19,7).
bond(e15,e15_22,e15_26,1).
bond(e15,e15_23,e15_27,1).
bond(e15,e15_24,e15_28,1).
bond(e15,e15_25,e15_29,1).
bond(e15,e15_11,e15_30,1).
bond(e15,e15_30,e15_31,2).
bond(e15,e15_30,e15_32,2).
atm(e16,e16_1,c,22,-0.074).
atm(e16,e16_2,c,22,-0.074).
atm(e16,e16_3,c,22,-0.074).
atm(e16,e16_4,c,22,-0.074).
atm(e16,e16_5,c,22,-0.074).
atm(e16,e16_6,c,22,-0.074).
atm(e16,e16_7,h,3,0.187).
atm(e16,e16_8,h,3,0.187).
atm(e16,e16_9,n,38,0.857).
atm(e16,e16_10,o,40,-0.343).
atm(e16,e16_11,o,40,-0.343).
atm(e16,e16_12,cl,93,-0.144).
atm(e16,e16_13,h,3,0.187).
atm(e16,e16_14,cl,93,-0.144).

bond(e16,e16_1,e16_2,7).
bond(e16,e16_2,e16_3,7).
bond(e16,e16_3,e16_4,7).
bond(e16,e16_4,e16_5,7).
bond(e16,e16_5,e16_6,7).
bond(e16,e16_6,e16_1,7).
bond(e16,e16_1,e16_7,1).
bond(e16,e16_6,e16_8,1).
bond(e16,e16_5,e16_9,1).
bond(e16,e16_9,e16_10,2).
bond(e16,e16_9,e16_11,2).
bond(e16,e16_2,e16_12,1).
bond(e16,e16_3,e16_13,1).
bond(e16,e16_4,e16_14,1).
atm(e17,e17_1,c,22,-0.125).
atm(e17,e17_2,c,22,-0.125).
atm(e17,e17_3,c,27,0.005).
atm(e17,e17_4,c,27,0.005).
atm(e17,e17_5,c,22,-0.125).
atm(e17,e17_6,c,22,-0.125).
atm(e17,e17_7,h,3,0.135).
atm(e17,e17_8,h,3,0.135).
atm(e17,e17_9,h,3,0.135).
atm(e17,e17_10,h,3,0.135).
atm(e17,e17_11,c,22,-0.125).
atm(e17,e17_12,c,27,0.005).
atm(e17,e17_13,c,27,0.005).
atm(e17,e17_14,c,22,-0.125).
atm(e17,e17_15,h,3,0.135).
atm(e17,e17_16,c,22,-0.125).
atm(e17,e17_17,c,22,-0.125).
atm(e17,e17_18,c,22,-0.125).
atm(e17,e17_19,c,22,-0.125).
atm(e17,e17_20,h,3,0.135).
atm(e17,e17_21,h,3,0.135).
atm(e17,e17_22,h,3,0.135).
atm(e17,e17_23,h,3,0.135).
atm(e17,e17_24,n,38,0.805).
atm(e17,e17_25,o,40,-0.395).
atm(e17,e17_26,o,40,-0.395).

bond(e17,e17_1,e17_2,7).
bond(e17,e17_2,e17_3,7).
bond(e17,e17_3,e17_4,7).
bond(e17,e17_4,e17_5,7).
bond(e17,e17_5,e17_6,7).
bond(e17,e17_6,e17_1,7).
bond(e17,e17_1,e17_7,1).
bond(e17,e17_2,e17_8,1).
bond(e17,e17_5,e17_9,1).
bond(e17,e17_6,e17_10,1).
bond(e17,e17_3,e17_11,7).
bond(e17,e17_11,e17_12,7).
bond(e17,e17_12,e17_13,7).
bond(e17,e17_13,e17_14,7).
bond(e17,e17_14,e17_4,7).
bond(e17,e17_11,e17_15,1).
bond(e17,e17_12,e17_16,7).
bond(e17,e17_16,e17_17,7).
bond(e17,e17_17,e17_18,7).
bond(e17,e17_18,e17_19,7).
bond(e17,e17_19,e17_13,7).
bond(e17,e17_16,e17_20,1).
bond(e17,e17_17,e17_21,1).
bond(e17,e17_18,e17_22,1).
bond(e17,e17_19,e17_23,1).
bond(e17,e17_14,e17_24,1).
bond(e17,e17_24,e17_25,2).
bond(e17,e17_24,e17_26,2).
atm(e18,e18_1,c,22,-0.074).
atm(e18,e18_2,c,22,-0.074).
atm(e18,e18_3,c,22,-0.074).
atm(e18,e18_4,c,22,-0.074).
atm(e18,e18_5,c,22,-0.074).
atm(e18,e18_6,c,22,-0.074).
atm(e18,e18_7,h,3,0.187).
atm(e18,e18_8,h,3,0.187).
atm(e18,e18_9,n,38,0.857).
atm(e18,e18_10,o,40,-0.343).
atm(e18,e18_11,o,40,-0.343).
atm(e18,e18_12,cl,93,-0.144).
atm(e18,e18_13,h,3,0.187).
atm(e18,e18_14,cl,93,-0.144).

bond(e18,e18_1,e18_2,7).
bond(e18,e18_2,e18_3,7).
bond(e18,e18_3,e18_4,7).
bond(e18,e18_4,e18_5,7).
bond(e18,e18_5,e18_6,7).
bond(e18,e18_6,e18_1,7).
bond(e18,e18_1,e18_7,1).
bond(e18,e18_6,e18_8,1).
bond(e18,e18_5,e18_9,1).
bond(e18,e18_9,e18_10,2).
bond(e18,e18_9,e18_11,2).
bond(e18,e18_2,e18_12,1).
bond(e18,e18_4,e18_13,1).
bond(e18,e18_3,e18_14,1).
atm(e19,e19_1,c,22,-0.108).
atm(e19,e19_2,c,22,-0.108).
atm(e19,e19_3,c,27,0.022).
atm(e19,e19_4,c,27,0.022).
atm(e19,e19_5,c,22,-0.108).
atm(e19,e19_6,c,22,-0.108).
atm(e19,e19_7,c,22,-0.108).
atm(e19,e19_8,c,22,0.122).
atm(e19,e19_9,c,22,-0.108).
atm(e19,e19_10,c,22,-0.108).
atm(e19,e19_11,c,16,-0.178).
atm(e19,e19_12,c,16,-0.178).
atm(e19,e19_13,c,10,0.122).
atm(e19,e19_14,c,10,0.022).
atm(e19,e19_15,n,38,0.822).
atm(e19,e19_16,o,40,-0.378).
atm(e19,e19_17,o,40,-0.378).
atm(e19,e19_18,h,3,0.152).
atm(e19,e19_19,h,3,0.152).
atm(e19,e19_20,h,3,0.152).
atm(e19,e19_21,h,3,0.152).
atm(e19,e19_22,h,3,0.152).
atm(e19,e19_23,h,3,0.122).
atm(e19,e19_24,h,3,0.122).
atm(e19,e19_25,o,45,-0.628).
atm(e19,e19_26,h,3,0.072).
atm(e19,e19_27,o,45,-0.628).
atm(e19,e19_28,h,3,0.072).
atm(e19,e19_29,h,8,0.422).
atm(e19,e19_30,h,8,0.422).

bond(e19,e19_1,e19_2,7).
bond(e19,e19_2,e19_3,7).
bond(e19,e19_3,e19_4,7).
bond(e19,e19_4,e19_5,7).
bond(e19,e19_5,e19_6,7).
bond(e19,e19_6,e19_1,7).
bond(e19,e19_3,e19_7,7).
bond(e19,e19_7,e19_8,7).
bond(e19,e19_8,e19_9,7).
bond(e19,e19_9,e19_10,7).
bond(e19,e19_10,e19_4,7).
bond(e19,e19_8,e19_11,1).
bond(e19,e19_11,e19_12,2).
bond(e19,e19_12,e19_13,1).
bond(e19,e19_13,e19_14,1).
bond(e19,e19_14,e19_9,1).
bond(e19,e19_10,e19_15,1).
bond(e19,e19_15,e19_16,2).
bond(e19,e19_15,e19_17,2).
bond(e19,e19_1,e19_18,1).
bond(e19,e19_2,e19_19,1).
bond(e19,e19_5,e19_20,1).
bond(e19,e19_6,e19_21,1).
bond(e19,e19_7,e19_22,1).
bond(e19,e19_11,e19_23,1).
bond(e19,e19_12,e19_24,1).
bond(e19,e19_14,e19_25,5).
bond(e19,e19_14,e19_26,1).
bond(e19,e19_13,e19_27,4).
bond(e19,e19_13,e19_28,1).
bond(e19,e19_25,e19_29,1).
bond(e19,e19_27,e19_30,1).
atm(e2,e2_1,c,22,-0.115).
atm(e2,e2_2,c,22,-0.115).
atm(e2,e2_3,c,27,0.015).
atm(e2,e2_4,c,27,-0.085).
atm(e2,e2_5,c,22,-0.115).
atm(e2,e2_6,c,22,-0.115).
atm(e2,e2_7,c,22,-0.115).
atm(e2,e2_8,c,22,0.015).
atm(e2,e2_9,c,195,-0.085).
atm(e2,e2_10,c,195,-0.085).
atm(e2,e2_11,c,10,-0.085).
atm(e2,e2_12,c,10,-0.085).
atm(e2,e2_13,c,22,0.015).
atm(e2,e2_14,c,27,-0.085).
atm(e2,e2_15,c,27,0.015).
atm(e2,e2_16,c,22,-0.115).
atm(e2,e2_17,c,22,-0.115).
atm(e2,e2_18,c,22,-0.115).
atm(e2,e2_19,c,22,-0.115).
atm(e2,e2_20,c,22,-0.115).
atm(e2,e2_21,n,38,0.814).
atm(e2,e2_22,o,40,-0.386).
atm(e2,e2_23,o,40,-0.386).
atm(e2,e2_24,h,3,0.144).
atm(e2,e2_25,h,3,0.144).
atm(e2,e2_26,h,3,0.144).
atm(e2,e2_27,h,3,0.144).
atm(e2,e2_28,h,3,0.144).
atm(e2,e2_29,h,3,0.145).
atm(e2,e2_30,h,3,0.145).
atm(e2,e2_31,h,3,0.144).
atm(e2,e2_32,h,3,0.144).
atm(e2,e2_33,h,3,0.065).
atm(e2,e2_34,h,3,0.065).
atm(e2,e2_35,h,3,0.065).
atm(e2,e2_36,h,3,0.065).

bond(e2,e2_1,e2_2,7).
bond(e2,e2_2,e2_3,7).
bond(e2,e2_3,e2_4,7).
bond(e2,e2_4,e2_5,7).
bond(e2,e2_5,e2_6,7).
bond(e2,e2_6,e2_1,7).
bond(e2,e2_3,e2_7,7).
bond(e2,e2_7,e2_8,7).
bond(e2,e2_8,e2_9,7).
bond(e2,e2_9,e2_10,7).
bond(e2,e2_10,e2_4,7).
bond(e2,e2_8,e2_11,1).
bond(e2,e2_11,e2_12,1).
bond(e2,e2_12,e2_13,1).
bond(e2,e2_13,e2_14,7).
bond(e2,e2_14,e2_9,7).
bond(e2,e2_14,e2_15,7).
bond(e2,e2_15,e2_16,7).
bond(e2,e2_16,e2_17,7).
bond(e2,e2_17,e2_10,7).
bond(e2,e2_13,e2_18,7).
bond(e2,e2_18,e2_19,7).
bond(e2,e2_19,e2_20,7).
bond(e2,e2_20,e2_15,7).
bond(e2,e2_7,e2_21,1).
bond(e2,e2_21,e2_22,2).
bond(e2,e2_21,e2_23,2).
bond(e2,e2_1,e2_24,1).
bond(e2,e2_2,e2_25,1).
bond(e2,e2_5,e2_26,1).
bond(e2,e2_6,e2_27,1).
bond(e2,e2_16,e2_28,1).
bond(e2,e2_17,e2_29,1).
bond(e2,e2_18,e2_30,1).
bond(e2,e2_19,e2_31,1).
bond(e2,e2_20,e2_32,1).
bond(e2,e2_11,e2_33,1).
bond(e2,e2_11,e2_34,1).
bond(e2,e2_12,e2_35,1).
bond(e2,e2_12,e2_36,1).
atm(e20,e20_1,c,22,-0.108).
atm(e20,e20_2,c,22,-0.108).
atm(e20,e20_3,c,27,0.022).
atm(e20,e20_4,c,27,0.022).
atm(e20,e20_5,c,22,-0.108).
atm(e20,e20_6,c,22,-0.108).
atm(e20,e20_7,c,22,-0.108).
atm(e20,e20_8,c,22,-0.108).
atm(e20,e20_9,c,22,0.122).
atm(e20,e20_10,c,22,-0.108).
atm(e20,e20_11,c,10,0.022).
atm(e20,e20_12,c,10,0.122).
atm(e20,e20_13,c,16,-0.178).
atm(e20,e20_14,c,16,-0.178).
atm(e20,e20_15,n,38,0.822).
atm(e20,e20_16,o,40,-0.378).
atm(e20,e20_17,o,40,-0.378).
atm(e20,e20_18,h,3,0.152).
atm(e20,e20_19,h,3,0.152).
atm(e20,e20_20,h,3,0.152).
atm(e20,e20_21,h,3,0.152).
atm(e20,e20_22,h,3,0.152).
atm(e20,e20_23,h,3,0.122).
atm(e20,e20_24,h,3,0.122).
atm(e20,e20_25,o,45,-0.628).
atm(e20,e20_26,h,3,0.072).
atm(e20,e20_27,o,45,-0.628).
atm(e20,e20_28,h,3,0.072).
atm(e20,e20_29,h,8,0.422).
atm(e20,e20_30,h,8,0.422).

bond(e20,e20_1,e20_2,7).
bond(e20,e20_2,e20_3,7).
bond(e20,e20_3,e20_4,7).
bond(e20,e20_4,e20_5,7).
bond(e20,e20_5,e20_6,7).
bond(e20,e20_6,e20_1,7).
bond(e20,e20_3,e20_7,7).
bond(e20,e20_7,e20_8,7).
bond(e20,e20_8,e20_9,7).
bond(e20,e20_9,e20_10,7).
bond(e20,e20_10,e20_4,7).
bond(e20,e20_8,e20_11,1).
bond(e20,e20_11,e20_12,1).
bond(e20,e20_12,e20_13,1).
bond(e20,e20_13,e20_14,2).
bond(e20,e20_14,e20_9,1).
bond(e20,e20_10,e20_15,1).
bond(e20,e20_15,e20_16,2).
bond(e20,e20_15,e20_17,2).
bond(e20,e20_1,e20_18,1).
bond(e20,e20_2,e20_19,1).
bond(e20,e20_5,e20_20,1).
bond(e20,e20_6,e20_21,1).
bond(e20,e20_7,e20_22,1).
bond(e20,e20_14,e20_23,1).
bond(e20,e20_13,e20_24,1).
bond(e20,e20_12,e20_25,5).
bond(e20,e20_12,e20_26,1).
bond(e20,e20_11,e20_27,4).
bond(e20,e20_11,e20_28,1).
bond(e20,e20_27,e20_29,1).
bond(e20,e20_25,e20_30,1).
atm(e21,e21_1,c,10,-0.091).
atm(e21,e21_2,c,10,-0.091).
atm(e21,e21_3,c,22,0.009).
atm(e21,e21_4,c,22,0.009).
atm(e21,e21_5,c,10,-0.091).
atm(e21,e21_6,c,10,-0.091).
atm(e21,e21_7,c,22,-0.122).
atm(e21,e21_8,c,27,0.009).
atm(e21,e21_9,c,27,-0.091).
atm(e21,e21_10,c,27,0.009).
atm(e21,e21_11,c,22,-0.121).
atm(e21,e21_12,c,22,-0.122).
atm(e21,e21_13,c,27,0.009).
atm(e21,e21_14,c,27,-0.091).
atm(e21,e21_15,c,27,0.009).
atm(e21,e21_16,c,22,-0.121).
atm(e21,e21_17,c,22,-0.121).
atm(e21,e21_18,c,22,-0.121).
atm(e21,e21_19,c,22,-0.121).
atm(e21,e21_20,c,22,-0.121).
atm(e21,e21_21,n,38,0.808).
atm(e21,e21_22,o,40,-0.392).
atm(e21,e21_23,o,40,-0.392).
atm(e21,e21_24,h,3,0.059).
atm(e21,e21_25,h,3,0.059).
atm(e21,e21_26,h,3,0.059).
atm(e21,e21_27,h,3,0.059).
atm(e21,e21_28,h,3,0.138).
atm(e21,e21_29,h,3,0.138).
atm(e21,e21_30,h,3,0.138).
atm(e21,e21_31,h,3,0.138).
atm(e21,e21_32,h,3,0.138).
atm(e21,e21_33,h,3,0.138).
atm(e21,e21_34,h,3,0.138).
atm(e21,e21_35,h,3,0.059).
atm(e21,e21_36,h,3,0.059).
atm(e21,e21_37,h,3,0.059).
atm(e21,e21_38,h,3,0.059).

bond(e21,e21_1,e21_2,1).
bond(e21,e21_2,e21_3,1).
bond(e21,e21_3,e21_4,7).
bond(e21,e21_4,e21_5,1).
bond(e21,e21_5,e21_6,1).
bond(e21,e21_6,e21_1,1).
bond(e21,e21_3,e21_7,7).
bond(e21,e21_7,e21_8,7).
bond(e21,e21_8,e21_9,7).
bond(e21,e21_9,e21_10,7).
bond(e21,e21_10,e21_4,7).
bond(e21,e21_8,e21_11,7).
bond(e21,e21_11,e21_12,7).
bond(e21,e21_12,e21_13,7).
bond(e21,e21_13,e21_14,7).
bond(e21,e21_14,e21_9,7).
bond(e21,e21_14,e21_15,7).
bond(e21,e21_15,e21_16,7).
bond(e21,e21_16,e21_17,7).
bond(e21,e21_17,e21_10,7).
bond(e21,e21_13,e21_18,7).
bond(e21,e21_18,e21_19,7).
bond(e21,e21_19,e21_20,7).
bond(e21,e21_20,e21_15,7).
bond(e21,e21_7,e21_21,1).
bond(e21,e21_21,e21_22,2).
bond(e21,e21_21,e21_23,2).
bond(e21,e21_5,e21_24,1).
bond(e21,e21_5,e21_25,1).
bond(e21,e21_6,e21_26,1).
bond(e21,e21_6,e21_27,1).
bond(e21,e21_11,e21_28,1).
bond(e21,e21_12,e21_29,1).
bond(e21,e21_16,e21_30,1).
bond(e21,e21_17,e21_31,1).
bond(e21,e21_18,e21_32,1).
bond(e21,e21_19,e21_33,1).
bond(e21,e21_20,e21_34,1).
bond(e21,e21_1,e21_35,1).
bond(e21,e21_1,e21_36,1).
bond(e21,e21_2,e21_37,1).
bond(e21,e21_2,e21_38,1).
atm(e22,e22_1,c,10,0.124).
atm(e22,e22_2,c,10,0.024).
atm(e22,e22_3,c,22,-0.106).
atm(e22,e22_4,c,22,0.124).
atm(e22,e22_5,c,16,-0.176).
atm(e22,e22_6,c,16,-0.176).
atm(e22,e22_7,c,22,-0.106).
atm(e22,e22_8,c,27,0.024).
atm(e22,e22_9,c,195,-0.076).
atm(e22,e22_10,c,22,-0.106).
atm(e22,e22_11,c,22,-0.106).
atm(e22,e22_12,c,22,-0.106).
atm(e22,e22_13,c,27,0.024).
atm(e22,e22_14,c,195,-0.076).
atm(e22,e22_15,c,22,-0.106).
atm(e22,e22_16,c,22,-0.106).
atm(e22,e22_17,c,22,-0.106).
atm(e22,e22_18,c,22,-0.106).
atm(e22,e22_19,n,38,0.823).
atm(e22,e22_20,o,40,-0.376).
atm(e22,e22_21,o,40,-0.376).
atm(e22,e22_22,h,3,0.154).
atm(e22,e22_23,h,3,0.154).
atm(e22,e22_24,h,3,0.154).
atm(e22,e22_25,h,3,0.154).
atm(e22,e22_26,h,3,0.154).
atm(e22,e22_27,h,3,0.154).
atm(e22,e22_28,h,3,0.154).
atm(e22,e22_29,h,3,0.124).
atm(e22,e22_30,h,3,0.124).
atm(e22,e22_31,o,45,-0.627).
atm(e22,e22_32,h,3,0.074).
atm(e22,e22_33,o,45,-0.627).
atm(e22,e22_34,h,3,0.074).
atm(e22,e22_35,h,8,0.423).
atm(e22,e22_36,h,8,0.424).

bond(e22,e22_1,e22_2,1).
bond(e22,e22_2,e22_3,1).
bond(e22,e22_3,e22_4,7).
bond(e22,e22_4,e22_5,1).
bond(e22,e22_5,e22_6,2).
bond(e22,e22_6,e22_1,1).
bond(e22,e22_3,e22_7,7).
bond(e22,e22_7,e22_8,7).
bond(e22,e22_8,e22_9,7).
bond(e22,e22_9,e22_10,7).
bond(e22,e22_10,e22_4,7).
bond(e22,e22_8,e22_11,7).
bond(e22,e22_11,e22_12,7).
bond(e22,e22_12,e22_13,7).
bond(e22,e22_13,e22_14,7).
bond(e22,e22_14,e22_9,7).
bond(e22,e22_13,e22_15,7).
bond(e22,e22_15,e22_16,7).
bond(e22,e22_16,e22_17,7).
bond(e22,e22_17,e22_18,7).
bond(e22,e22_18,e22_14,7).
bond(e22,e22_7,e22_19,1).
bond(e22,e22_19,e22_20,2).
bond(e22,e22_19,e22_21,2).
bond(e22,e22_10,e22_22,1).
bond(e22,e22_11,e22_23,1).
bond(e22,e22_12,e22_24,1).
bond(e22,e22_15,e22_25,1).
bond(e22,e22_16,e22_26,1).
bond(e22,e22_17,e22_27,1).
bond(e22,e22_18,e22_28,1).
bond(e22,e22_5,e22_29,1).
bond(e22,e22_6,e22_30,1).
bond(e22,e22_1,e22_31,5).
bond(e22,e22_1,e22_32,1).
bond(e22,e22_2,e22_33,4).
bond(e22,e22_2,e22_34,1).
bond(e22,e22_31,e22_35,1).
bond(e22,e22_33,e22_36,1).
atm(e23,e23_1,c,22,-0.112).
atm(e23,e23_2,c,22,-0.112).
atm(e23,e23_3,c,27,0.018).
atm(e23,e23_4,c,27,0.018).
atm(e23,e23_5,c,22,-0.112).
atm(e23,e23_6,c,22,-0.112).
atm(e23,e23_7,c,22,-0.112).
atm(e23,e23_8,c,27,0.018).
atm(e23,e23_9,c,27,0.018).
atm(e23,e23_10,c,22,-0.112).
atm(e23,e23_11,c,22,-0.112).
atm(e23,e23_12,c,22,-0.112).
atm(e23,e23_13,c,22,-0.112).
atm(e23,e23_14,c,22,0.118).
atm(e23,e23_15,c,10,0.018).
atm(e23,e23_16,c,10,0.118).
atm(e23,e23_17,c,16,-0.181).
atm(e23,e23_18,c,16,-0.181).
atm(e23,e23_19,n,38,0.819).
atm(e23,e23_20,o,40,-0.381).
atm(e23,e23_21,o,40,-0.381).
atm(e23,e23_22,h,3,0.148).
atm(e23,e23_23,h,3,0.148).
atm(e23,e23_24,h,3,0.149).
atm(e23,e23_25,h,3,0.149).
atm(e23,e23_26,h,3,0.149).
atm(e23,e23_27,h,3,0.148).
atm(e23,e23_28,h,3,0.148).
atm(e23,e23_29,o,45,-0.631).
atm(e23,e23_30,h,3,0.068).
atm(e23,e23_31,o,45,-0.631).
atm(e23,e23_32,h,3,0.068).
atm(e23,e23_33,h,8,0.419).
atm(e23,e23_34,h,8,0.419).
atm(e23,e23_35,h,3,0.118).
atm(e23,e23_36,h,3,0.118).

bond(e23,e23_1,e23_2,7).
bond(e23,e23_2,e23_3,7).
bond(e23,e23_3,e23_4,7).
bond(e23,e23_4,e23_5,7).
bond(e23,e23_5,e23_6,7).
bond(e23,e23_6,e23_1,7).
bond(e23,e23_3,e23_7,7).
bond(e23,e23_7,e23_8,7).
bond(e23,e23_8,e23_9,7).
bond(e23,e23_9,e23_10,7).
bond(e23,e23_10,e23_4,7).
bond(e23,e23_8,e23_11,7).
bond(e23,e23_11,e23_12,7).
bond(e23,e23_12,e23_13,7).
bond(e23,e23_13,e23_14,7).
bond(e23,e23_14,e23_9,7).
bond(e23,e23_13,e23_15,1).
bond(e23,e23_15,e23_16,1).
bond(e23,e23_16,e23_17,1).
bond(e23,e23_17,e23_18,2).
bond(e23,e23_18,e23_14,1).
bond(e23,e23_7,e23_19,1).
bond(e23,e23_19,e23_20,2).
bond(e23,e23_19,e23_21,2).
bond(e23,e23_1,e23_22,1).
bond(e23,e23_2,e23_23,1).
bond(e23,e23_5,e23_24,1).
bond(e23,e23_6,e23_25,1).
bond(e23,e23_10,e23_26,1).
bond(e23,e23_11,e23_27,1).
bond(e23,e23_12,e23_28,1).
bond(e23,e23_16,e23_29,5).
bond(e23,e23_16,e23_30,1).
bond(e23,e23_15,e23_31,4).
bond(e23,e23_15,e23_32,1).
bond(e23,e23_29,e23_33,1).
bond(e23,e23_31,e23_34,1).
bond(e23,e23_17,e23_35,1).
bond(e23,e23_18,e23_36,1).
atm(e24,e24_1,c,194,0.004).
atm(e24,e24_2,c,194,0.004).
atm(e24,e24_3,c,29,0.004).
atm(e24,e24_4,c,194,0.004).
atm(e24,e24_5,c,29,0.004).
atm(e24,e24_6,c,29,0.004).
atm(e24,e24_7,c,22,-0.126).
atm(e24,e24_8,c,22,-0.126).
atm(e24,e24_9,c,22,-0.126).
atm(e24,e24_10,c,22,-0.126).
atm(e24,e24_11,h,3,0.134).
atm(e24,e24_12,h,3,0.134).
atm(e24,e24_13,h,3,0.134).
atm(e24,e24_14,c,22,-0.126).
atm(e24,e24_15,c,22,-0.126).
atm(e24,e24_16,c,22,-0.126).
atm(e24,e24_17,c,22,-0.126).
atm(e24,e24_18,h,3,0.134).
atm(e24,e24_19,h,3,0.134).
atm(e24,e24_20,h,3,0.134).
atm(e24,e24_21,h,3,0.134).
atm(e24,e24_22,c,22,-0.126).
atm(e24,e24_23,c,22,-0.126).
atm(e24,e24_24,c,22,-0.126).
atm(e24,e24_25,c,22,-0.126).
atm(e24,e24_26,h,3,0.134).
atm(e24,e24_27,h,3,0.134).
atm(e24,e24_28,h,3,0.134).
atm(e24,e24_29,h,3,0.134).
atm(e24,e24_30,n,38,0.805).
atm(e24,e24_31,o,40,-0.395).
atm(e24,e24_32,o,40,-0.396).

bond(e24,e24_1,e24_2,1).
bond(e24,e24_2,e24_3,7).
bond(e24,e24_3,e24_4,1).
bond(e24,e24_4,e24_5,7).
bond(e24,e24_5,e24_6,1).
bond(e24,e24_6,e24_1,7).
bond(e24,e24_4,e24_7,7).
bond(e24,e24_7,e24_8,7).
bond(e24,e24_8,e24_9,7).
bond(e24,e24_9,e24_10,7).
bond(e24,e24_10,e24_5,7).
bond(e24,e24_7,e24_11,1).
bond(e24,e24_8,e24_12,1).
bond(e24,e24_9,e24_13,1).
bond(e24,e24_6,e24_14,7).
bond(e24,e24_14,e24_15,7).
bond(e24,e24_15,e24_16,7).
bond(e24,e24_16,e24_17,7).
bond(e24,e24_17,e24_1,7).
bond(e24,e24_14,e24_18,1).
bond(e24,e24_15,e24_19,1).
bond(e24,e24_16,e24_20,1).
bond(e24,e24_17,e24_21,1).
bond(e24,e24_2,e24_22,7).
bond(e24,e24_22,e24_23,7).
bond(e24,e24_23,e24_24,7).
bond(e24,e24_24,e24_25,7).
bond(e24,e24_25,e24_3,7).
bond(e24,e24_22,e24_26,1).
bond(e24,e24_23,e24_27,1).
bond(e24,e24_24,e24_28,1).
bond(e24,e24_25,e24_29,1).
bond(e24,e24_10,e24_30,1).
bond(e24,e24_30,e24_31,2).
bond(e24,e24_30,e24_32,2).
atm(e25,e25_1,c,10,-0.083).
atm(e25,e25_2,c,10,0.017).
atm(e25,e25_3,c,22,-0.113).
atm(e25,e25_4,c,22,0.017).
atm(e25,e25_5,c,10,-0.083).
atm(e25,e25_6,c,10,-0.083).
atm(e25,e25_7,c,22,-0.113).
atm(e25,e25_8,c,27,0.017).
atm(e25,e25_9,c,27,-0.083).
atm(e25,e25_10,c,27,0.017).
atm(e25,e25_11,c,22,-0.113).
atm(e25,e25_12,c,22,-0.113).
atm(e25,e25_13,c,27,0.017).
atm(e25,e25_14,c,27,-0.083).
atm(e25,e25_15,c,27,0.017).
atm(e25,e25_16,c,22,-0.113).
atm(e25,e25_17,c,22,-0.113).
atm(e25,e25_18,c,22,-0.113).
atm(e25,e25_19,c,22,-0.113).
atm(e25,e25_20,c,22,-0.113).
atm(e25,e25_21,n,38,0.816).
atm(e25,e25_22,o,40,-0.383).
atm(e25,e25_23,o,40,-0.383).
atm(e25,e25_24,h,3,0.067).
atm(e25,e25_25,h,3,0.067).
atm(e25,e25_26,h,3,0.067).
atm(e25,e25_27,h,3,0.067).
atm(e25,e25_28,h,3,0.147).
atm(e25,e25_29,h,3,0.147).
atm(e25,e25_30,h,3,0.147).
atm(e25,e25_31,h,3,0.147).
atm(e25,e25_32,h,3,0.147).
atm(e25,e25_33,h,3,0.147).
atm(e25,e25_34,h,3,0.147).
atm(e25,e25_35,h,3,0.067).
atm(e25,e25_36,h,3,0.067).
atm(e25,e25_37,o,45,-0.634).
atm(e25,e25_38,h,3,0.067).
atm(e25,e25_39,h,8,0.416).

bond(e25,e25_1,e25_2,1).
bond(e25,e25_2,e25_3,1).
bond(e25,e25_3,e25_4,7).
bond(e25,e25_4,e25_5,1).
bond(e25,e25_5,e25_6,1).
bond(e25,e25_6,e25_1,1).
bond(e25,e25_3,e25_7,7).
bond(e25,e25_7,e25_8,7).
bond(e25,e25_8,e25_9,7).
bond(e25,e25_9,e25_10,7).
bond(e25,e25_10,e25_4,7).
bond(e25,e25_8,e25_11,7).
bond(e25,e25_11,e25_12,7).
bond(e25,e25_12,e25_13,7).
bond(e25,e25_13,e25_14,7).
bond(e25,e25_14,e25_9,7).
bond(e25,e25_14,e25_15,7).
bond(e25,e25_15,e25_16,7).
bond(e25,e25_16,e25_17,7).
bond(e25,e25_17,e25_10,7).
bond(e25,e25_13,e25_18,7).
bond(e25,e25_18,e25_19,7).
bond(e25,e25_19,e25_20,7).
bond(e25,e25_20,e25_15,7).
bond(e25,e25_7,e25_21,1).
bond(e25,e25_21,e25_22,2).
bond(e25,e25_21,e25_23,2).
bond(e25,e25_5,e25_24,1).
bond(e25,e25_5,e25_25,1).
bond(e25,e25_6,e25_26,1).
bond(e25,e25_6,e25_27,1).
bond(e25,e25_11,e25_28,1).
bond(e25,e25_12,e25_29,1).
bond(e25,e25_16,e25_30,1).
bond(e25,e25_17,e25_31,1).
bond(e25,e25_18,e25_32,1).
bond(e25,e25_19,e25_33,1).
bond(e25,e25_20,e25_34,1).
bond(e25,e25_1,e25_35,1).
bond(e25,e25_1,e25_36,1).
bond(e25,e25_2,e25_37,1).
bond(e25,e25_2,e25_38,1).
bond(e25,e25_37,e25_39,1).
atm(e26,e26_1,c,22,-0.12).
atm(e26,e26_2,c,22,-0.12).
atm(e26,e26_3,c,27,0.01).
atm(e26,e26_4,c,27,0.01).
atm(e26,e26_5,c,22,0.26).
atm(e26,e26_6,c,22,-0.12).
atm(e26,e26_7,c,22,-0.12).
atm(e26,e26_8,c,27,0.01).
atm(e26,e26_9,c,195,-0.09).
atm(e26,e26_10,c,22,-0.12).
atm(e26,e26_11,c,22,-0.12).
atm(e26,e26_12,c,22,-0.12).
atm(e26,e26_13,c,27,0.01).
atm(e26,e26_14,c,195,-0.09).
atm(e26,e26_15,c,22,-0.12).
atm(e26,e26_16,c,22,-0.12).
atm(e26,e26_17,c,22,-0.12).
atm(e26,e26_18,c,22,-0.12).
atm(e26,e26_19,o,45,-0.64).
atm(e26,e26_20,h,3,0.14).
atm(e26,e26_21,h,3,0.14).
atm(e26,e26_22,h,3,0.14).
atm(e26,e26_23,h,3,0.14).
atm(e26,e26_24,h,3,0.14).
atm(e26,e26_25,h,3,0.14).
atm(e26,e26_26,h,3,0.14).
atm(e26,e26_27,h,3,0.14).
atm(e26,e26_28,h,3,0.14).
atm(e26,e26_29,h,3,0.14).
atm(e26,e26_30,h,8,0.41).
atm(e26,e26_31,n,38,0.81).
atm(e26,e26_32,o,40,-0.39).
atm(e26,e26_33,o,40,-0.39).

bond(e26,e26_1,e26_2,7).
bond(e26,e26_2,e26_3,7).
bond(e26,e26_3,e26_4,7).
bond(e26,e26_4,e26_5,7).
bond(e26,e26_5,e26_6,7).
bond(e26,e26_6,e26_1,7).
bond(e26,e26_3,e26_7,7).
bond(e26,e26_7,e26_8,7).
bond(e26,e26_8,e26_9,7).
bond(e26,e26_9,e26_10,7).
bond(e26,e26_10,e26_4,7).
bond(e26,e26_8,e26_11,7).
bond(e26,e26_11,e26_12,7).
bond(e26,e26_12,e26_13,7).
bond(e26,e26_13,e26_14,7).
bond(e26,e26_14,e26_9,7).
bond(e26,e26_13,e26_15,7).
bond(e26,e26_15,e26_16,7).
bond(e26,e26_16,e26_17,7).
bond(e26,e26_17,e26_18,7).
bond(e26,e26_18,e26_14,7).
bond(e26,e26_5,e26_19,1).
bond(e26,e26_1,e26_20,1).
bond(e26,e26_2,e26_21,1).
bond(e26,e26_6,e26_22,1).
bond(e26,e26_10,e26_23,1).
bond(e26,e26_11,e26_24,1).
bond(e26,e26_12,e26_25,1).
bond(e26,e26_15,e26_26,1).
bond(e26,e26_16,e26_27,1).
bond(e26,e26_17,e26_28,1).
bond(e26,e26_18,e26_29,1).
bond(e26,e26_19,e26_30,1).
bond(e26,e26_7,e26_31,1).
bond(e26,e26_31,e26_32,2).
bond(e26,e26_31,e26_33,2).
atm(e27,e27_1,c,22,-0.12).
atm(e27,e27_2,c,22,-0.12).
atm(e27,e27_3,c,29,0.01).
atm(e27,e27_4,c,27,-0.09).
atm(e27,e27_5,c,27,0.01).
atm(e27,e27_6,c,22,-0.12).
atm(e27,e27_7,h,3,0.139).
atm(e27,e27_8,h,3,0.139).
atm(e27,e27_9,h,3,0.139).
atm(e27,e27_10,c,29,0.01).
atm(e27,e27_11,c,22,-0.12).
atm(e27,e27_12,c,22,-0.12).
atm(e27,e27_13,c,22,-0.12).
atm(e27,e27_14,h,3,0.14).
atm(e27,e27_15,h,3,0.139).
atm(e27,e27_16,h,3,0.139).
atm(e27,e27_17,c,29,0.01).
atm(e27,e27_18,c,27,-0.09).
atm(e27,e27_19,c,29,0.01).
atm(e27,e27_20,c,27,0.01).
atm(e27,e27_21,c,22,-0.12).
atm(e27,e27_22,c,22,-0.12).
atm(e27,e27_23,c,22,-0.12).
atm(e27,e27_24,h,3,0.14).
atm(e27,e27_25,h,3,0.14).
atm(e27,e27_26,h,3,0.139).
atm(e27,e27_27,c,22,-0.12).
atm(e27,e27_28,c,22,-0.12).
atm(e27,e27_29,c,22,-0.12).
atm(e27,e27_30,h,3,0.14).
atm(e27,e27_31,h,3,0.139).
atm(e27,e27_32,n,38,0.809).
atm(e27,e27_33,o,40,-0.391).
atm(e27,e27_34,o,40,-0.391).

bond(e27,e27_1,e27_2,7).
bond(e27,e27_2,e27_3,7).
bond(e27,e27_3,e27_4,7).
bond(e27,e27_4,e27_5,7).
bond(e27,e27_5,e27_6,7).
bond(e27,e27_6,e27_1,7).
bond(e27,e27_1,e27_7,1).
bond(e27,e27_2,e27_8,1).
bond(e27,e27_6,e27_9,1).
bond(e27,e27_4,e27_10,7).
bond(e27,e27_10,e27_11,7).
bond(e27,e27_11,e27_12,7).
bond(e27,e27_12,e27_13,7).
bond(e27,e27_13,e27_5,7).
bond(e27,e27_11,e27_14,1).
bond(e27,e27_12,e27_15,1).
bond(e27,e27_13,e27_16,1).
bond(e27,e27_3,e27_17,1).
bond(e27,e27_17,e27_18,7).
bond(e27,e27_18,e27_19,7).
bond(e27,e27_19,e27_10,1).
bond(e27,e27_18,e27_20,7).
bond(e27,e27_20,e27_21,7).
bond(e27,e27_21,e27_22,7).
bond(e27,e27_22,e27_23,7).
bond(e27,e27_23,e27_19,7).
bond(e27,e27_21,e27_24,1).
bond(e27,e27_22,e27_25,1).
bond(e27,e27_23,e27_26,1).
bond(e27,e27_17,e27_27,7).
bond(e27,e27_27,e27_28,7).
bond(e27,e27_28,e27_29,7).
bond(e27,e27_29,e27_20,7).
bond(e27,e27_27,e27_30,1).
bond(e27,e27_28,e27_31,1).
bond(e27,e27_29,e27_32,1).
bond(e27,e27_32,e27_33,2).
bond(e27,e27_32,e27_34,2).
atm(e3,e3_1,c,22,-0.127).
atm(e3,e3_2,c,22,-0.126).
atm(e3,e3_3,c,22,0.204).
atm(e3,e3_4,c,22,0.003).
atm(e3,e3_5,c,22,-0.126).
atm(e3,e3_6,c,22,-0.126).
atm(e3,e3_7,h,3,0.134).
atm(e3,e3_8,h,3,0.134).
atm(e3,e3_9,h,3,0.134).
atm(e3,e3_10,n,32,-0.396).
atm(e3,e3_11,c,10,0.003).
atm(e3,e3_12,c,10,-0.097).
atm(e3,e3_13,c,10,0.003).
atm(e3,e3_14,h,3,0.053).
atm(e3,e3_15,h,3,0.053).
atm(e3,e3_16,h,3,0.053).
atm(e3,e3_17,n,38,0.804).
atm(e3,e3_18,o,40,-0.396).
atm(e3,e3_19,o,40,-0.396).
atm(e3,e3_20,h,3,0.053).
atm(e3,e3_21,h,3,0.053).
atm(e3,e3_22,h,3,0.053).
atm(e3,e3_23,h,3,0.053).

bond(e3,e3_1,e3_2,7).
bond(e3,e3_2,e3_3,7).
bond(e3,e3_3,e3_4,7).
bond(e3,e3_4,e3_5,7).
bond(e3,e3_5,e3_6,7).
bond(e3,e3_6,e3_1,7).
bond(e3,e3_1,e3_7,1).
bond(e3,e3_2,e3_8,1).
bond(e3,e3_5,e3_9,1).
bond(e3,e3_3,e3_10,1).
bond(e3,e3_10,e3_11,1).
bond(e3,e3_4,e3_12,1).
bond(e3,e3_12,e3_11,1).
bond(e3,e3_10,e3_13,1).
bond(e3,e3_13,e3_14,1).
bond(e3,e3_13,e3_15,1).
bond(e3,e3_13,e3_16,1).
bond(e3,e3_6,e3_17,1).
bond(e3,e3_17,e3_18,2).
bond(e3,e3_17,e3_19,2).
bond(e3,e3_12,e3_20,1).
bond(e3,e3_12,e3_21,1).
bond(e3,e3_11,e3_22,1).
bond(e3,e3_11,e3_23,1).
atm(e4,e4_1,c,22,-0.149).
atm(e4,e4_2,c,22,-0.149).
atm(e4,e4_3,c,26,0.281).
atm(e4,e4_4,c,26,0.081).
atm(e4,e4_5,c,22,-0.149).
atm(e4,e4_6,c,22,-0.149).
atm(e4,e4_7,n,34,-0.559).
atm(e4,e4_8,c,21,0.151).
atm(e4,e4_9,c,21,-0.119).
atm(e4,e4_10,c,10,0.231).
atm(e4,e4_11,n,38,0.78).
atm(e4,e4_12,o,40,-0.419).
atm(e4,e4_13,o,40,-0.419).
atm(e4,e4_14,h,3,0.111).
atm(e4,e4_15,h,3,0.111).
atm(e4,e4_16,h,3,0.081).
atm(e4,e4_17,h,3,0.111).
atm(e4,e4_18,h,3,0.081).
atm(e4,e4_19,h,3,0.031).
atm(e4,e4_20,h,3,0.031).
atm(e4,e4_21,h,3,0.031).

bond(e4,e4_1,e4_2,7).
bond(e4,e4_2,e4_3,7).
bond(e4,e4_3,e4_4,7).
bond(e4,e4_4,e4_5,7).
bond(e4,e4_5,e4_6,7).
bond(e4,e4_6,e4_1,7).
bond(e4,e4_3,e4_7,7).
bond(e4,e4_7,e4_8,7).
bond(e4,e4_4,e4_9,7).
bond(e4,e4_9,e4_8,7).
bond(e4,e4_7,e4_10,1).
bond(e4,e4_6,e4_11,1).
bond(e4,e4_11,e4_12,2).
bond(e4,e4_11,e4_13,2).
bond(e4,e4_1,e4_14,1).
bond(e4,e4_2,e4_15,1).
bond(e4,e4_5,e4_16,1).
bond(e4,e4_8,e4_17,1).
bond(e4,e4_9,e4_18,1).
bond(e4,e4_10,e4_19,1).
bond(e4,e4_10,e4_20,1).
bond(e4,e4_10,e4_21,1).
atm(e5,e5_1,c,22,-0.121).
atm(e5,e5_2,c,22,-0.121).
atm(e5,e5_3,c,22,-0.121).
atm(e5,e5_4,c,22,-0.121).
atm(e5,e5_5,c,22,-0.121).
atm(e5,e5_6,c,22,-0.121).
atm(e5,e5_7,h,3,0.139).
atm(e5,e5_8,h,3,0.139).
atm(e5,e5_9,h,3,0.139).
atm(e5,e5_10,h,3,0.139).
atm(e5,e5_11,h,3,0.14).
atm(e5,e5_12,n,38,0.81).
atm(e5,e5_13,o,40,-0.39).
atm(e5,e5_14,o,40,-0.39).

bond(e5,e5_1,e5_2,7).
bond(e5,e5_2,e5_3,7).
bond(e5,e5_3,e5_4,7).
bond(e5,e5_4,e5_5,7).
bond(e5,e5_5,e5_6,7).
bond(e5,e5_6,e5_1,7).
bond(e5,e5_1,e5_7,1).
bond(e5,e5_2,e5_8,1).
bond(e5,e5_3,e5_9,1).
bond(e5,e5_4,e5_10,1).
bond(e5,e5_6,e5_11,1).
bond(e5,e5_5,e5_12,1).
bond(e5,e5_12,e5_13,2).
bond(e5,e5_12,e5_14,2).
atm(e6,e6_1,c,22,-0.149).
atm(e6,e6_2,c,22,-0.149).
atm(e6,e6_3,c,26,0.281).
atm(e6,e6_4,c,26,0.081).
atm(e6,e6_5,c,22,-0.149).
atm(e6,e6_6,c,22,-0.149).
atm(e6,e6_7,n,34,-0.559).
atm(e6,e6_8,c,21,0.151).
atm(e6,e6_9,c,21,-0.119).
atm(e6,e6_10,c,10,0.231).
atm(e6,e6_11,n,38,0.78).
atm(e6,e6_12,o,40,-0.419).
atm(e6,e6_13,o,40,-0.419).
atm(e6,e6_14,h,3,0.111).
atm(e6,e6_15,h,3,0.081).
atm(e6,e6_16,h,3,0.111).
atm(e6,e6_17,h,3,0.111).
atm(e6,e6_18,h,3,0.081).
atm(e6,e6_19,h,3,0.031).
atm(e6,e6_20,h,3,0.031).
atm(e6,e6_21,h,3,0.031).

bond(e6,e6_1,e6_2,7).
bond(e6,e6_2,e6_3,7).
bond(e6,e6_3,e6_4,7).
bond(e6,e6_4,e6_5,7).
bond(e6,e6_5,e6_6,7).
bond(e6,e6_6,e6_1,7).
bond(e6,e6_3,e6_7,7).
bond(e6,e6_7,e6_8,7).
bond(e6,e6_4,e6_9,7).
bond(e6,e6_9,e6_8,7).
bond(e6,e6_7,e6_10,1).
bond(e6,e6_2,e6_11,1).
bond(e6,e6_11,e6_12,2).
bond(e6,e6_11,e6_13,2).
bond(e6,e6_1,e6_14,1).
bond(e6,e6_5,e6_15,1).
bond(e6,e6_6,e6_16,1).
bond(e6,e6_8,e6_17,1).
bond(e6,e6_9,e6_18,1).
bond(e6,e6_10,e6_19,1).
bond(e6,e6_10,e6_20,1).
bond(e6,e6_10,e6_21,1).
atm(e7,e7_1,c,22,-0.139).
atm(e7,e7_2,c,22,-0.139).
atm(e7,e7_3,c,26,-0.049).
atm(e7,e7_4,c,26,0.091).
atm(e7,e7_5,c,22,-0.14).
atm(e7,e7_6,c,22,-0.14).
atm(e7,e7_7,n,34,-0.37).
atm(e7,e7_8,c,21,0.16).
atm(e7,e7_9,c,21,-0.109).
atm(e7,e7_10,n,38,0.79).
atm(e7,e7_11,o,40,-0.41).
atm(e7,e7_12,o,40,-0.41).
atm(e7,e7_13,h,3,0.121).
atm(e7,e7_14,h,3,0.121).
atm(e7,e7_15,h,3,0.121).
atm(e7,e7_16,h,3,0.121).
atm(e7,e7_17,h,3,0.091).
atm(e7,e7_18,h,1,0.29).

bond(e7,e7_1,e7_2,7).
bond(e7,e7_2,e7_3,7).
bond(e7,e7_3,e7_4,7).
bond(e7,e7_4,e7_5,7).
bond(e7,e7_5,e7_6,7).
bond(e7,e7_6,e7_1,7).
bond(e7,e7_3,e7_7,7).
bond(e7,e7_7,e7_8,7).
bond(e7,e7_4,e7_9,7).
bond(e7,e7_9,e7_8,7).
bond(e7,e7_5,e7_10,1).
bond(e7,e7_10,e7_11,2).
bond(e7,e7_10,e7_12,2).
bond(e7,e7_1,e7_13,1).
bond(e7,e7_2,e7_14,1).
bond(e7,e7_6,e7_15,1).
bond(e7,e7_8,e7_16,1).
bond(e7,e7_9,e7_17,1).
bond(e7,e7_7,e7_18,1).
atm(e8,e8_1,c,22,-0.115).
atm(e8,e8_2,c,22,-0.115).
atm(e8,e8_3,c,26,0.315).
atm(e8,e8_4,c,26,0.115).
atm(e8,e8_5,c,22,-0.115).
atm(e8,e8_6,c,22,-0.115).
atm(e8,e8_7,n,34,-0.384).
atm(e8,e8_8,n,34,-0.384).
atm(e8,e8_9,c,21,-0.115).
atm(e8,e8_10,n,38,0.816).
atm(e8,e8_11,o,40,-0.384).
atm(e8,e8_12,o,40,-0.384).
atm(e8,e8_13,h,1,0.315).
atm(e8,e8_14,h,3,0.145).
atm(e8,e8_15,h,3,0.145).
atm(e8,e8_16,h,3,0.115).
atm(e8,e8_17,h,3,0.145).

bond(e8,e8_1,e8_2,7).
bond(e8,e8_2,e8_3,7).
bond(e8,e8_3,e8_4,7).
bond(e8,e8_4,e8_5,7).
bond(e8,e8_5,e8_6,7).
bond(e8,e8_6,e8_1,7).
bond(e8,e8_3,e8_7,7).
bond(e8,e8_7,e8_8,7).
bond(e8,e8_4,e8_9,7).
bond(e8,e8_9,e8_8,7).
bond(e8,e8_6,e8_10,1).
bond(e8,e8_10,e8_11,2).
bond(e8,e8_10,e8_12,2).
bond(e8,e8_7,e8_13,1).
bond(e8,e8_1,e8_14,1).
bond(e8,e8_2,e8_15,1).
bond(e8,e8_5,e8_16,1).
bond(e8,e8_9,e8_17,1).
atm(e9,e9_1,c,22,-0.121).
atm(e9,e9_2,c,22,-0.121).
atm(e9,e9_3,c,22,-0.121).
atm(e9,e9_4,c,22,0.31).
atm(e9,e9_5,c,22,-0.121).
atm(e9,e9_6,c,22,-0.121).
atm(e9,e9_7,h,3,0.139).
atm(e9,e9_8,h,3,0.139).
atm(e9,e9_9,h,3,0.139).
atm(e9,e9_10,h,3,0.139).
atm(e9,e9_11,n,38,0.81).
atm(e9,e9_12,f,92,-0.291).
atm(e9,e9_13,o,40,-0.39).
atm(e9,e9_14,o,40,-0.39).

bond(e9,e9_1,e9_2,7).
bond(e9,e9_2,e9_3,7).
bond(e9,e9_3,e9_4,7).
bond(e9,e9_4,e9_5,7).
bond(e9,e9_5,e9_6,7).
bond(e9,e9_6,e9_1,7).
bond(e9,e9_1,e9_7,1).
bond(e9,e9_2,e9_8,1).
bond(e9,e9_3,e9_9,1).
bond(e9,e9_6,e9_10,1).
bond(e9,e9_5,e9_11,1).
bond(e9,e9_4,e9_12,1).
bond(e9,e9_11,e9_13,2).
bond(e9,e9_11,e9_14,2).
atm(f1,f1_1,c,21,0.187).
atm(f1,f1_2,c,21,-0.143).
atm(f1,f1_3,c,21,-0.143).
atm(f1,f1_4,c,21,-0.013).
atm(f1,f1_5,o,52,-0.043).
atm(f1,f1_6,h,3,0.117).
atm(f1,f1_7,h,3,0.117).
atm(f1,f1_8,n,38,0.786).
atm(f1,f1_9,o,40,-0.413).
atm(f1,f1_10,o,40,-0.413).
atm(f1,f1_11,c,14,0.217).
atm(f1,f1_12,o,41,-0.443).
atm(f1,f1_13,h,3,0.187).

bond(f1,f1_1,f1_2,7).
bond(f1,f1_2,f1_3,7).
bond(f1,f1_3,f1_4,7).
bond(f1,f1_4,f1_5,7).
bond(f1,f1_5,f1_1,7).
bond(f1,f1_2,f1_6,1).
bond(f1,f1_3,f1_7,1).
bond(f1,f1_4,f1_8,1).
bond(f1,f1_8,f1_9,2).
bond(f1,f1_8,f1_10,2).
bond(f1,f1_1,f1_11,1).
bond(f1,f1_11,f1_12,2).
bond(f1,f1_11,f1_13,1).
atm(f2,f2_1,c,21,0.006).
atm(f2,f2_2,c,21,-0.124).
atm(f2,f2_3,c,21,-0.124).
atm(f2,f2_4,c,21,0.106).
atm(f2,f2_5,c,21,-0.124).
atm(f2,f2_6,h,3,0.136).
atm(f2,f2_7,h,3,0.136).
atm(f2,f2_8,h,3,0.136).
atm(f2,f2_9,n,38,0.805).
atm(f2,f2_10,o,40,-0.395).
atm(f2,f2_11,o,40,-0.395).
atm(f2,f2_12,c,16,-0.195).
atm(f2,f2_13,c,16,-0.195).
atm(f2,f2_14,c,14,0.235).
atm(f2,f2_15,o,41,-0.425).
atm(f2,f2_16,h,3,0.205).
atm(f2,f2_17,h,3,0.106).
atm(f2,f2_18,h,3,0.106).

bond(f2,f2_1,f2_2,7).
bond(f2,f2_2,f2_3,7).
bond(f2,f2_3,f2_4,7).
bond(f2,f2_4,f2_5,7).
bond(f2,f2_5,f2_1,7).
bond(f2,f2_2,f2_6,1).
bond(f2,f2_3,f2_7,1).
bond(f2,f2_5,f2_8,1).
bond(f2,f2_1,f2_9,1).
bond(f2,f2_9,f2_10,2).
bond(f2,f2_9,f2_11,2).
bond(f2,f2_12,f2_13,2).
bond(f2,f2_13,f2_14,1).
bond(f2,f2_14,f2_15,2).
bond(f2,f2_14,f2_16,1).
bond(f2,f2_13,f2_17,1).
bond(f2,f2_4,f2_12,1).
bond(f2,f2_12,f2_18,1).
atm(f3,f3_1,c,21,0.086).
atm(f3,f3_2,c,21,-0.144).
atm(f3,f3_3,c,21,-0.144).
atm(f3,f3_4,c,21,-0.014).
atm(f3,f3_5,o,52,-0.044).
atm(f3,f3_6,h,3,0.116).
atm(f3,f3_7,h,3,0.116).
atm(f3,f3_8,n,38,0.788).
atm(f3,f3_9,o,40,-0.414).
atm(f3,f3_10,o,40,-0.414).
atm(f3,f3_11,c,14,0.587).
atm(f3,f3_12,n,32,-0.414).
atm(f3,f3_13,h,3,0.036).
atm(f3,f3_14,n,32,-0.414).
atm(f3,f3_15,c,14,0.587).
atm(f3,f3_16,h,1,0.287).
atm(f3,f3_17,o,40,-0.563).
atm(f3,f3_18,n,32,-0.612).
atm(f3,f3_19,h,1,0.287).
atm(f3,f3_20,h,1,0.287).

bond(f3,f3_1,f3_2,7).
bond(f3,f3_2,f3_3,7).
bond(f3,f3_3,f3_4,7).
bond(f3,f3_4,f3_5,7).
bond(f3,f3_5,f3_1,7).
bond(f3,f3_2,f3_6,1).
bond(f3,f3_3,f3_7,1).
bond(f3,f3_4,f3_8,1).
bond(f3,f3_8,f3_9,2).
bond(f3,f3_8,f3_10,2).
bond(f3,f3_1,f3_11,1).
bond(f3,f3_11,f3_12,2).
bond(f3,f3_11,f3_13,1).
bond(f3,f3_12,f3_14,1).
bond(f3,f3_14,f3_15,1).
bond(f3,f3_14,f3_16,1).
bond(f3,f3_15,f3_17,2).
bond(f3,f3_15,f3_18,1).
bond(f3,f3_18,f3_19,1).
bond(f3,f3_18,f3_20,1).
atm(f4,f4_1,c,21,0.107).
atm(f4,f4_2,c,21,-0.123).
atm(f4,f4_3,c,21,-0.123).
atm(f4,f4_4,c,21,0.007).
atm(f4,f4_5,o,52,-0.023).
atm(f4,f4_6,h,3,0.137).
atm(f4,f4_7,h,3,0.137).
atm(f4,f4_8,n,38,0.808).
atm(f4,f4_9,o,40,-0.393).
atm(f4,f4_10,o,40,-0.393).
atm(f4,f4_11,c,14,0.608).
atm(f4,f4_12,n,32,-0.393).
atm(f4,f4_13,h,3,0.057).
atm(f4,f4_14,n,36,-0.293).
atm(f4,f4_15,n,32,-0.393).
atm(f4,f4_16,h,1,0.157).
atm(f4,f4_17,c,10,0.007).
atm(f4,f4_18,c,14,0.608).
atm(f4,f4_19,c,14,0.608).
atm(f4,f4_20,h,3,0.057).
atm(f4,f4_21,h,3,0.057).
atm(f4,f4_22,n,32,-0.393).
atm(f4,f4_23,o,40,-0.542).
atm(f4,f4_24,o,40,-0.543).
atm(f4,f4_25,h,1,0.257).

bond(f4,f4_1,f4_2,7).
bond(f4,f4_2,f4_3,7).
bond(f4,f4_3,f4_4,7).
bond(f4,f4_4,f4_5,7).
bond(f4,f4_5,f4_1,7).
bond(f4,f4_2,f4_6,1).
bond(f4,f4_3,f4_7,1).
bond(f4,f4_4,f4_8,1).
bond(f4,f4_8,f4_9,2).
bond(f4,f4_8,f4_10,2).
bond(f4,f4_1,f4_11,1).
bond(f4,f4_11,f4_12,2).
bond(f4,f4_11,f4_13,1).
bond(f4,f4_12,f4_14,1).
bond(f4,f4_14,f4_15,1).
bond(f4,f4_14,f4_16,1).
bond(f4,f4_15,f4_17,1).
bond(f4,f4_15,f4_18,1).
bond(f4,f4_17,f4_19,1).
bond(f4,f4_17,f4_20,1).
bond(f4,f4_17,f4_21,1).
bond(f4,f4_19,f4_22,1).
bond(f4,f4_22,f4_18,1).
bond(f4,f4_18,f4_23,2).
bond(f4,f4_19,f4_24,2).
bond(f4,f4_22,f4_25,1).
atm(f5,f5_1,c,21,0.046).
atm(f5,f5_2,n,34,-0.494).
atm(f5,f5_3,c,21,0.046).
atm(f5,f5_4,c,21,0.046).
atm(f5,f5_5,s,72,-0.184).
atm(f5,f5_6,n,32,-0.354).
atm(f5,f5_7,c,10,0.046).
atm(f5,f5_8,c,10,0.046).
atm(f5,f5_9,n,32,-0.354).
atm(f5,f5_10,c,14,0.646).
atm(f5,f5_11,h,3,0.146).
atm(f5,f5_12,h,3,0.096).
atm(f5,f5_13,h,3,0.096).
atm(f5,f5_14,h,3,0.096).
atm(f5,f5_15,h,3,0.096).
atm(f5,f5_16,n,38,0.846).
atm(f5,f5_17,o,40,-0.354).
atm(f5,f5_18,o,40,-0.354).
atm(f5,f5_19,h,1,0.346).
atm(f5,f5_20,o,40,-0.504).

bond(f5,f5_1,f5_2,7).
bond(f5,f5_2,f5_3,7).
bond(f5,f5_3,f5_4,7).
bond(f5,f5_4,f5_5,7).
bond(f5,f5_5,f5_1,7).
bond(f5,f5_6,f5_7,1).
bond(f5,f5_7,f5_8,1).
bond(f5,f5_8,f5_9,1).
bond(f5,f5_9,f5_10,1).
bond(f5,f5_10,f5_6,1).
bond(f5,f5_3,f5_11,1).
bond(f5,f5_7,f5_12,1).
bond(f5,f5_7,f5_13,1).
bond(f5,f5_8,f5_14,1).
bond(f5,f5_8,f5_15,1).
bond(f5,f5_4,f5_16,1).
bond(f5,f5_16,f5_17,2).
bond(f5,f5_16,f5_18,2).
bond(f5,f5_1,f5_9,1).
bond(f5,f5_6,f5_19,1).
bond(f5,f5_10,f5_20,2).
atm(f6,f6_1,c,21,0.106).
atm(f6,f6_2,c,21,-0.124).
atm(f6,f6_3,c,21,-0.124).
atm(f6,f6_4,c,21,0.006).
atm(f6,f6_5,o,52,-0.024).
atm(f6,f6_6,h,3,0.136).
atm(f6,f6_7,h,3,0.136).
atm(f6,f6_8,n,38,0.805).
atm(f6,f6_9,o,40,-0.395).
atm(f6,f6_10,o,40,-0.394).
atm(f6,f6_11,c,14,0.605).
atm(f6,f6_12,n,32,-0.395).
atm(f6,f6_13,h,3,0.056).
atm(f6,f6_14,n,32,-0.395).
atm(f6,f6_15,c,10,0.006).
atm(f6,f6_16,c,14,0.605).
atm(f6,f6_17,c,10,0.256).
atm(f6,f6_18,h,3,0.056).
atm(f6,f6_19,h,3,0.056).
atm(f6,f6_20,o,49,-0.645).
atm(f6,f6_21,o,51,-0.545).
atm(f6,f6_22,h,3,0.106).
atm(f6,f6_23,h,3,0.106).

bond(f6,f6_1,f6_2,7).
bond(f6,f6_2,f6_3,7).
bond(f6,f6_3,f6_4,7).
bond(f6,f6_4,f6_5,7).
bond(f6,f6_5,f6_1,7).
bond(f6,f6_2,f6_6,1).
bond(f6,f6_3,f6_7,1).
bond(f6,f6_4,f6_8,1).
bond(f6,f6_8,f6_9,2).
bond(f6,f6_8,f6_10,2).
bond(f6,f6_1,f6_11,1).
bond(f6,f6_11,f6_12,2).
bond(f6,f6_11,f6_13,1).
bond(f6,f6_14,f6_15,1).
bond(f6,f6_14,f6_16,1).
bond(f6,f6_15,f6_17,1).
bond(f6,f6_15,f6_18,1).
bond(f6,f6_15,f6_19,1).
bond(f6,f6_17,f6_20,1).
bond(f6,f6_20,f6_16,1).
bond(f6,f6_16,f6_21,2).
bond(f6,f6_12,f6_14,1).
bond(f6,f6_17,f6_22,1).
bond(f6,f6_17,f6_23,1).
ind1(d1,1.0).
ind1(d2,0.0).
ind1(d3,0.0).
ind1(d4,1.0).
ind1(d5,0.0).
ind1(d6,1.0).
ind1(d7,0.0).
ind1(d8,1.0).
ind1(d9,0.0).

ind1(d10,1.0).
ind1(d11,1.0).
ind1(d12,1.0).
ind1(d13,1.0).
ind1(d14,0.0).
ind1(d15,1.0).
ind1(d16,1.0).
ind1(d17,0.0).
ind1(d18,1.0).
ind1(d19,0.0).

ind1(d20,1.0).
ind1(d21,0.0).
ind1(d22,1.0).
ind1(d23,1.0).
ind1(d24,1.0).
ind1(d25,1.0).
ind1(d26,0.0).
ind1(d27,1.0).
ind1(d28,1.0).
ind1(d29,1.0).

ind1(d30,1.0).
ind1(d31,1.0).
ind1(d32,0.0).
ind1(d33,1.0).
ind1(d34,0.0).
ind1(d35,1.0).
ind1(d36,0.0).
ind1(d37,0.0).
ind1(d38,0.0).
ind1(d39,0.0).

ind1(d40,0.0).
ind1(d41,1.0).
ind1(d42,0.0).
ind1(d43,1.0).
ind1(d44,1.0).
ind1(d45,0.0).
ind1(d46,1.0).
ind1(d47,1.0).
ind1(d48,1.0).
ind1(d49,1.0).

ind1(d50,0.0).
ind1(d51,1.0).
ind1(d52,1.0).
ind1(d53,1.0).
ind1(d54,1.0).
ind1(d55,0.0).
ind1(d56,1.0).
ind1(d57,0.0).
ind1(d58,1.0).
ind1(d59,1.0).

ind1(d60,0.0).
ind1(d61,1.0).
ind1(d62,0.0).
ind1(d63,0.0).
ind1(d64,1.0).
ind1(d65,0.0).
ind1(d66,0.0).
ind1(d67,1.0).
ind1(d68,1.0).
ind1(d69,1.0).

ind1(d70,0.0).
ind1(d71,1.0).
ind1(d72,1.0).
ind1(d73,0.0).
ind1(d74,1.0).
ind1(d75,0.0).
ind1(d76,0.0).
ind1(d77,0.0).
ind1(d78,0.0).
ind1(d79,1.0).

ind1(d80,1.0).
ind1(d81,0.0).
ind1(d82,1.0).
ind1(d83,1.0).
ind1(d84,0.0).
ind1(d85,1.0).
ind1(d86,1.0).
ind1(d87,1.0).
ind1(d88,0.0).
ind1(d89,0.0).

ind1(d90,1.0).
ind1(d91,1.0).
ind1(d92,1.0).
ind1(d93,1.0).
ind1(d94,1.0).
ind1(d95,0.0).
ind1(d96,1.0).
ind1(d97,1.0).
ind1(d98,0.0).
ind1(d99,1.0).

ind1(d100,0.0).
ind1(d101,1.0).
ind1(d102,1.0).
ind1(d103,1.0).
ind1(d104,1.0).
ind1(d105,1.0).
ind1(d106,1.0).
ind1(d107,1.0).
ind1(d108,1.0).
ind1(d109,1.0).

ind1(d110,0.0).
ind1(d111,0.0).
ind1(d112,1.0).
ind1(d113,0.0).
ind1(d114,0.0).
ind1(d115,0.0).
ind1(d116,0.0).
ind1(d117,0.0).
ind1(d118,1.0).
ind1(d119,0.0).

ind1(d120,0.0).
ind1(d121,1.0).
ind1(d122,1.0).
ind1(d123,0.0).
ind1(d124,0.0).
ind1(d125,1.0).
ind1(d126,1.0).
ind1(d127,0.0).
ind1(d128,1.0).
ind1(d129,0.0).

ind1(d130,0.0).
ind1(d131,0.0).
ind1(d132,0.0).
ind1(d133,1.0).
ind1(d134,1.0).
ind1(d135,0.0).
ind1(d136,1.0).
ind1(d137,1.0).
ind1(d138,0.0).
ind1(d139,0.0).

ind1(d140,0.0).
ind1(d141,0.0).
ind1(d142,1.0).
ind1(d143,0.0).
ind1(d144,0.0).
ind1(d145,0.0).
ind1(d146,0.0).
ind1(d147,0.0).
ind1(d148,1.0).
ind1(d149,1.0).

ind1(d150,0.0).
ind1(d151,0.0).
ind1(d152,1.0).
ind1(d153,1.0).
ind1(d154,0.0).
ind1(d155,0.0).
ind1(d156,0.0).
ind1(d157,1.0).
ind1(d158,1.0).
ind1(d159,1.0).

ind1(d160,0.0).
ind1(d161,1.0).
ind1(d162,0.0).
ind1(d163,1.0).
ind1(d164,1.0).
ind1(d165,1.0).
ind1(d166,1.0).
ind1(d167,1.0).
ind1(d168,0.0).
ind1(d169,0.0).

ind1(d170,1.0).
ind1(d171,0.0).
ind1(d172,0.0).
ind1(d173,0.0).
ind1(d174,1.0).
ind1(d175,1.0).
ind1(d176,0.0).
ind1(d177,1.0).
ind1(d178,0.0).
ind1(d179,0.0).

ind1(d180,1.0).
ind1(d181,0.0).
ind1(d182,0.0).
ind1(d183,1.0).
ind1(d184,1.0).
ind1(d185,0.0).
ind1(d186,0.0).
ind1(d187,1.0).
ind1(d188,1.0).








ind1(d189,0.0).
ind1(d190,0.0).
ind1(d191,1.0).
ind1(d192,1.0).
ind1(d193,1.0).
ind1(d194,0.0).
ind1(d195,0.0).
ind1(d196,1.0).
ind1(d197,0.0).

ind1(e1,1.0).
ind1(e2,1.0).
ind1(e3,0.0).
ind1(e4,0.0).
ind1(e5,0.0).
ind1(e6,0.0).
ind1(e7,0.0).
ind1(e8,0.0).
ind1(e9,0.0).
ind1(e10,0.0).
ind1(e11,0.0).
ind1(e12,0.0).
ind1(e13,0.0).
ind1(e14,0.0).
ind1(e15,1.0).
ind1(e16,0.0).
ind1(e17,1.0).
ind1(e18,0.0).
ind1(e19,1.0).
ind1(e20,1.0).
ind1(e21,1.0).
ind1(e22,1.0).
ind1(e23,1.0).
ind1(e24,1.0).
ind1(e25,1.0).
ind1(e26,1.0).
ind1(e27,1.0).
ind1(f1,0.0).
ind1(f2,0.0).
ind1(f3,0.0).
ind1(f4,0.0).
ind1(f5,0.0).
ind1(f6,0.0).
inda(d1,0.0).
inda(d2,0.0).
inda(d3,0.0).
inda(d4,0.0).
inda(d5,0.0).
inda(d6,0.0).
inda(d7,0.0).
inda(d8,0.0).
inda(d9,0.0).

inda(d10,0.0).
inda(d11,0.0).
inda(d12,0.0).
inda(d13,0.0).
inda(d14,0.0).
inda(d15,0.0).
inda(d16,0.0).
inda(d17,0.0).
inda(d18,0.0).
inda(d19,0.0).

inda(d20,0.0).
inda(d21,0.0).
inda(d22,0.0).
inda(d23,1.0).
inda(d24,0.0).
inda(d25,0.0).
inda(d26,0.0).
inda(d27,0.0).
inda(d28,0.0).
inda(d29,0.0).

inda(d30,1.0).
inda(d31,0.0).
inda(d32,0.0).
inda(d33,0.0).
inda(d34,0.0).
inda(d35,0.0).
inda(d36,0.0).
inda(d37,0.0).
inda(d38,0.0).
inda(d39,0.0).

inda(d40,0.0).
inda(d41,0.0).
inda(d42,0.0).
inda(d43,0.0).
inda(d44,0.0).
inda(d45,0.0).
inda(d46,0.0).
inda(d47,0.0).
inda(d48,0.0).
inda(d49,0.0).

inda(d50,0.0).
inda(d51,0.0).
inda(d52,0.0).
inda(d53,0.0).
inda(d54,0.0).
inda(d55,0.0).
inda(d56,0.0).
inda(d57,0.0).
inda(d58,0.0).
inda(d59,1.0).

inda(d60,0.0).
inda(d61,0.0).
inda(d62,0.0).
inda(d63,0.0).
inda(d64,0.0).
inda(d65,0.0).
inda(d66,0.0).
inda(d67,0.0).
inda(d68,0.0).
inda(d69,0.0).

inda(d70,0.0).
inda(d71,0.0).
inda(d72,0.0).
inda(d73,0.0).
inda(d74,0.0).
inda(d75,0.0).
inda(d76,0.0).
inda(d77,0.0).
inda(d78,0.0).
inda(d79,0.0).

inda(d80,0.0).
inda(d81,0.0).
inda(d82,0.0).
inda(d83,0.0).
inda(d84,0.0).
inda(d85,1.0).
inda(d86,1.0).
inda(d87,0.0).
inda(d88,0.0).
inda(d89,0.0).

inda(d90,0.0).
inda(d91,0.0).
inda(d92,0.0).
inda(d93,0.0).
inda(d94,0.0).
inda(d95,0.0).
inda(d96,0.0).
inda(d97,0.0).
inda(d98,0.0).
inda(d99,0.0).

inda(d100,0.0).
inda(d101,0.0).
inda(d102,0.0).
inda(d103,0.0).
inda(d104,0.0).
inda(d105,0.0).
inda(d106,0.0).
inda(d107,0.0).
inda(d108,0.0).
inda(d109,0.0).

inda(d110,0.0).
inda(d111,0.0).
inda(d112,0.0).
inda(d113,0.0).
inda(d114,0.0).
inda(d115,0.0).
inda(d116,0.0).
inda(d117,0.0).
inda(d118,0.0).
inda(d119,0.0).

inda(d120,0.0).
inda(d121,0.0).
inda(d122,0.0).
inda(d123,0.0).
inda(d124,0.0).
inda(d125,0.0).
inda(d126,0.0).
inda(d127,0.0).
inda(d128,0.0).
inda(d129,0.0).

inda(d130,0.0).
inda(d131,0.0).
inda(d132,0.0).
inda(d133,0.0).
inda(d134,0.0).
inda(d135,0.0).
inda(d136,0.0).
inda(d137,0.0).
inda(d138,0.0).
inda(d139,0.0).

inda(d140,0.0).
inda(d141,0.0).
inda(d142,0.0).
inda(d143,0.0).
inda(d144,0.0).
inda(d145,0.0).
inda(d146,0.0).
inda(d147,0.0).
inda(d148,0.0).
inda(d149,0.0).

inda(d150,0.0).
inda(d151,0.0).
inda(d152,0.0).
inda(d153,0.0).
inda(d154,0.0).
inda(d155,0.0).
inda(d156,0.0).
inda(d157,0.0).
inda(d158,0.0).
inda(d159,0.0).

inda(d160,0.0).
inda(d161,0.0).
inda(d162,0.0).
inda(d163,0.0).
inda(d164,0.0).
inda(d165,0.0).
inda(d166,0.0).
inda(d167,0.0).
inda(d168,0.0).
inda(d169,0.0).

inda(d170,0.0).
inda(d171,0.0).
inda(d172,0.0).
inda(d173,0.0).
inda(d174,0.0).
inda(d175,0.0).
inda(d176,0.0).
inda(d177,0.0).
inda(d178,0.0).
inda(d179,0.0).

inda(d180,0.0).
inda(d181,0.0).
inda(d182,0.0).
inda(d183,0.0).
inda(d184,0.0).
inda(d185,0.0).
inda(d186,0.0).
inda(d187,0.0).
inda(d188,0.0).








inda(d189,0.0).
inda(d190,0.0).
inda(d191,0.0).
inda(d192,0.0).
inda(d193,0.0).
inda(d194,0.0).
inda(d195,0.0).
inda(d196,0.0).
inda(d197,0.0).

inda(e1,0.0).
inda(e2,0.0).
inda(e3,0.0).
inda(e4,0.0).
inda(e5,0.0).
inda(e6,0.0).
inda(e7,0.0).
inda(e8,0.0).
inda(e9,0.0).
inda(e10,0.0).
inda(e11,0.0).
inda(e12,0.0).
inda(e13,0.0).
inda(e14,0.0).
inda(e15,0.0).
inda(e16,0.0).
inda(e17,0.0).
inda(e18,0.0).
inda(e19,0.0).
inda(e20,0.0).
inda(e21,0.0).
inda(e22,0.0).
inda(e23,0.0).
inda(e24,0.0).
inda(e25,0.0).
inda(e26,0.0).
inda(e27,0.0).
inda(f1,0.0).
inda(f2,0.0).
inda(f3,0.0).
inda(f4,0.0).
inda(f5,0.0).
inda(f6,0.0).
act(d1,2.11).
act(d2,-1.24).
act(d3,-0.96).
act(d4,2.74).
act(d5,-0.79).
act(d6,4.99).
act(d7,-0.10).
act(d8,1.86).
act(d9,-1.83).
act(d10,2.36).
act(d11,2.25).
act(d12,2.43).
act(d13,3.78).
act(d14,-1.10).
act(d15,2.78).
act(d16,3.62).
act(d17,-3.00).
act(d18,2.13).
act(d19,-1.08).
act(d20,2.51).
act(d21,0.23).
act(d22,2.97).
act(d23,0.86).
act(d24,0.45).
act(d25,1.68).
act(d26,0.30).
act(d27,2.63).
act(d28,2.68).
act(d29,3.41).
act(d30,0.67).
act(d31,3.11).
act(d32,1.54).
act(d33,2.81).
act(d34,-0.30).
act(d35,1.56).
act(d36,0.00).
act(d37,0.69).
act(d38,-0.30).
act(d39,-1.21).
act(d40,-0.61).
act(d41,3.06).
act(d42,-1.10).
act(d43,3.01).
act(d44,1.99).
act(d45,0.15).
act(d46,0.90).
act(d47,3.50).
act(d48,2.95).
act(d49,2.75).
act(d50,0.52).
act(d51,2.41).
act(d52,2.41).
act(d53,3.22).
act(d54,1.73).
act(d55,-0.70).
act(d56,2.06).
act(d57,0.66).
act(d58,2.80).
act(d59,0.26).
act(d60,0.35).
act(d61,3.11).
act(d62,-2.40).
act(d63,2.45).
act(d64,3.19).
act(d65,-0.05).
act(d66,-1.52).
act(d67,2.26).
act(d68,2.82).
act(d69,3.20).
act(d70,-0.94).
act(d71,2.19).
act(d72,1.43).
act(d73,-2.00).
act(d74,2.76).
act(d75,1.17).
act(d76,-1.72).
act(d77,-1.52).
act(d78,-0.23).
act(d79,1.79).
act(d80,1.87).
act(d81,0.03).
act(d82,3.56).
act(d83,1.01).
act(d84,-1.30).
act(d85,0.04).
act(d86,0.92).
act(d87,1.64).
act(d88,-1.54).
act(d89,-1.00).
act(d90,1.59).
act(d91,3.08).
act(d92,1.58).
act(d93,2.95).
act(d94,3.39).
act(d95,1.51).
act(d96,4.99).
act(d97,2.79).
act(d98,-1.55).
act(d99,1.77).
act(d100,-1.70).
act(d101,0.78).
act(d102,2.93).
act(d103,3.35).
act(d104,0.70).
act(d105,0.59).
act(d106,1.89).
act(d107,3.44).
act(d108,3.67).
act(d109,3.60).
act(d110,-0.07).
act(d111,-1.64).
act(d112,1.00).
act(d113,-1.74).
act(d114,-1.70).
act(d115,0.08).
act(d116,-0.23).
act(d117,0.03).
act(d118,0.30).
act(d119,-0.19).
act(d120,-0.82).
act(d121,3.16).
act(d122,1.75).
act(d123,-0.84).
act(d124,-1.66).
act(d125,3.27).
act(d126,2.02).
act(d127,1.92).
act(d128,1.91).
act(d129,-0.07).
act(d130,-2.70).
act(d131,-1.15).
act(d132,-1.92).
act(d133,-1.40).
act(d134,4.25).
act(d135,-1.21).
act(d136,3.87).
act(d137,2.62).
act(d138,-0.41).
act(d139,-2.00).
act(d140,0.90).
act(d141,-1.51).
act(d142,-0.53).
act(d143,-2.22).
act(d144,-1.54).
act(d145,0.25).
act(d146,0.19).
act(d147,-0.51).
act(d148,1.77).
act(d149,4.34).
act(d150,-1.61).
act(d151,0.11).
act(d152,2.32).
act(d153,0.58).
act(d154,0.00).
act(d155,-1.89).
act(d156,-0.48).
act(d157,1.34).
act(d158,2.89).
act(d159,1.59).
act(d160,-1.32).
act(d161,1.99).
act(d162,2.85).
act(d163,4.58).
act(d164,5.04).
act(d165,4.05).
act(d166,4.22).
act(d167,5.06).
act(d168,-2.70).
act(d169,1.20).
act(d170,0.87).
act(d171,2.66).
act(d172,0.66).
act(d173,2.60).
act(d174,5.02).
act(d175,-0.33).
act(d176,0.96).
act(d177,1.26).
act(d178,0.23).
act(d179,-1.42).
act(d180,4.33).
act(d181,-1.11).
act(d182,-2.70).
act(d183,5.39).
act(d184,5.09).
act(d185,-1.84).
act(d186,-0.17).
act(d187,4.09).
act(d188,-0.70).
act(d189,-0.70).
act(d190,0.57).
act(d191,0.77).
act(d192,-0.22).
act(d193,-0.22).
act(d194,0.63).
act(d195,-2.94).
act(d196,-2.00).
act(d197,2.54).
act(e1,1.64).
act(e2,0.30).
act(e3,-4.0).
act(e4,-4.0).
act(e5,-4.0).
act(e6,-4.0).
act(e7,-4.0).
act(e8,-4.0).
act(e9,-4.0).
act(e10,-4.0).
act(e11,-4.0).
act(e12,-4.0).
act(e13,-4.0).
act(e14,-4.0).
act(e15,-0.52).
act(e16,-4.0).
act(e17,-0.95).
act(e18,-4.0).
act(e19,-4.0).
act(e20,-4.0).
act(e21,-4.0).
act(e22,-4.0).
act(e23,-4.0).
act(e24,-4.0).
act(e25,-4.0).
act(e26,-4.0).
act(e27,1.48).
act(f1,0.64).
act(f2,1.97).
act(f3,1.02).
act(f4,1.03).
act(f5,2.59).
act(f6,2.24).
logp(d8, 3.46).
logp(d16, 4.44).
logp(d21, 3.52).
logp(d22, 5.09).
logp(d23, 5.07).
logp(d29, 2.42).
logp(d30, 5.07).
logp(d32, 3).
logp(d33, 4.23).
logp(d35, 3.06).
logp(d37, 4.18).
logp(d44, 4.53).
logp(d47, 5.02).
logp(d74, 5.87).
logp(d81, 1.49).
logp(d85, 5.07).
logp(d91, 3.01).
logp(d92, 5.28).
logp(d93, 5.87).
logp(d97, 3.95).
logp(d99, 2.72).
logp(d102, 2.4).
logp(d103, 4.69).
logp(d106, 4.34).
logp(d108, 4.69).
logp(d118, 6.57).
logp(d121, 4.18).
logp(d134, 4.73).
logp(d145, 2.68).
logp(d158, 3.08).
logp(d161, 5.61).
logp(d162, 3).
logp(d167, 4.44).
logp(d170, 2.52).
logp(d176, 2.74).
logp(d180, 6.07).
logp(f4, -0.47).
logp(f5, 0.95).
logp(d2, 1.44).
logp(d17, 0.87).
logp(d42, 1.77).
logp(d70, 0.47).
logp(d77, 2.68).
logp(d89, 1.77).
logp(d98, 1.65).
logp(d111, 1.56).
logp(d114, 2.68).
logp(d119, 3.26).
logp(d120, 1.77).
logp(d130, 1.73).
logp(d131, 2.83).
logp(d132, 1.74).
logp(d138, 1.77).
logp(d141, 3.05).
logp(d144, 2.9).
logp(d147, 1.94).
logp(d186, 2.07).
logp(d188, 3.51).
logp(d189, 2.29).
logp(d196, 3.51).
logp(e4, 2.5).
logp(e5, 1.85).
logp(e8, 2.03).
logp(e12, 1.58).
logp(e14, 2.39).
logp(e18, 3.12).
logp(e22, 2.55).
logp(e25, 4.78).
logp(e26, 5.06).
logp(d1, 4.23).
logp(d4, 4.69).
logp(d6, 3.92).
logp(d10, 4.62).
logp(d11, 4.23).
logp(d12, 3.63).
logp(d13, 4.44).
logp(d15, 4.69).
logp(d18, 3.06).
logp(d20, 3.4).
logp(d24, 6.79).
logp(d25, 3.43).
logp(d26, 2.17).
logp(d27, 5.87).
logp(d28, 4.11).
logp(d31, 4.44).
logp(d41, 4.83).
logp(d43, 4.69).
logp(d45, 1.46).
logp(d46, 6.57).
logp(d48, 4.23).
logp(d49, 2.29).
logp(d50, 2.58).
logp(d51, 6.01).
logp(d52, 6.01).
logp(d53, 3.35).
logp(d54, 6.24).
logp(d56, 2.52).
logp(d57, 3.26).
logp(d58, 3.01).
logp(d59, 5.07).
logp(d60, 2.3).
logp(d61, 4.68).
logp(d63, 2.79).
logp(d64, 2.84).
logp(d67, 4.66).
logp(d68, 5.87).
logp(d69, 3.71).
logp(d71, 6.01).
logp(d72, 3.37).
logp(d75, 3.52).
logp(d79, 4.73).
logp(d80, 4.69).
logp(d82, 4.18).
logp(d83, 3.51).
logp(d86, 5.07).
logp(d87, 3.83).
logp(d90, 5.87).
logp(d94, 4.69).
logp(d95, 2.55).
logp(d96, 4.18).
logp(d101, 6.26).
logp(d104, 6.26).
logp(d105, 1.84).
logp(d107, 4.18).
logp(d109, 6.07).
logp(d112, 3.81).
logp(d115, 2.74).
logp(d117, 3.26).
logp(d122, 5.41).
logp(d125, 4.99).
logp(d126, 2.29).
logp(d127, 3.26).
logp(d128, 3.36).
logp(d136, 4.66).
logp(d137, 4.44).
logp(d140, 2.52).
logp(d146, 2.68).
logp(d148, 3.36).
logp(d149, 2.29).
logp(d151, 1.75).
logp(d152, 4.44).
logp(d153, 3.85).
logp(d157, 4.19).
logp(d159, 4.19).
logp(d163, 4.44).
logp(d164, 4.44).
logp(d165, 4.69).
logp(d166, 4.42).
logp(d169, 1.49).
logp(d171, 3).
logp(d172, 2.06).
logp(d173, 3.26).
logp(d174, 4.44).
logp(d177, 2.29).
logp(d178, 1.77).
logp(d182, 1.99).
logp(d183, 4.44).
logp(d184, 4.44).
logp(d187, 5.41).
logp(d190, 2.13).
logp(d191, 4.35).
logp(d194, 0.88).
logp(d197, 1.58).
logp(e1, 5.87).
logp(e2, 6.16).
logp(e27, 5.87).
logp(f1, 1.01).
logp(f2, 0.96).
logp(f3, 0.23).
logp(f6, -0.04).
logp(d3, 1.86).
logp(d5, 1.89).
logp(d7, 3.99).
logp(d9, 1.64).
logp(d14, 1.77).
logp(d19, 1.84).
logp(d34, 3.24).
logp(d36, 3.43).
logp(d38, 3.77).
logp(d39, 1.87).
logp(d40, 3.19).
logp(d55, 3.43).
logp(d62, 1.36).
logp(d65, 2.83).
logp(d66, 2.68).
logp(d73, 2.68).
logp(d76, 2.24).
logp(d78, 4.27).
logp(d84, 2.61).
logp(d88, 1.53).
logp(d100, 2.68).
logp(d110, 3).
logp(d113, 1.84).
logp(d116, 1.8).
logp(d123, 4.49).
logp(d124, 1.89).
logp(d129, 1.46).
logp(d133, 7.13).
logp(d135, 2.74).
logp(d139, 2.72).
logp(d142, 6.68).
logp(d143, 2.35).
logp(d150, -0.02).
logp(d154, 1.59).
logp(d155, 1.72).
logp(d156, 1.92).
logp(d160, 2.78).
logp(d168, 2.03).
logp(d175, 7.84).
logp(d179, 2.73).
logp(d181, 0.53).
logp(d185, 1.89).
logp(d192, 5.41).
logp(d193, 5.41).
logp(d195, 3.61).
logp(e3, 2.86).
logp(e6, 2.5).
logp(e7, 2.13).
logp(e9, 1.69).
logp(e10, 2.13).
logp(e11, 1.9).
logp(e13, 2.41).
logp(e15, 5.41).
logp(e16, 3.09).
logp(e17, 4.78).
logp(e19, 1.38).
logp(e20, 1.38).
logp(e21, 6.57).
logp(e23, 2.55).
logp(e24, 5.41).
lumo(d8, -1.437).
lumo(d16, -2.172).
lumo(d21, -1.665).
lumo(d22, -1.602).
lumo(d23, -2.164).
lumo(d29, -2.837).
lumo(d30, -2.005).
lumo(d32, -2.562).
lumo(d33, -1.591).
lumo(d35, -1.176).
lumo(d37, -1.428).
lumo(d44, -1.265).
lumo(d47, -1.88).
lumo(d74, -1.689).
lumo(d81, -1.937).
lumo(d85, -2.113).
lumo(d91, -2.032).
lumo(d92, -1.208).
lumo(d93, -1.729).
lumo(d97, -1.361).
lumo(d99, -2.159).
lumo(d102, -3.172).
lumo(d103, -1.487).
lumo(d106, -1.607).
lumo(d108, -1.676).
lumo(d118, -1.8).
lumo(d121, -2.68).
lumo(d134, -1.951).
lumo(d145, -1.178).
lumo(d158, -1.34).
lumo(d161, -2.221).
lumo(d162, -2.687).
lumo(d167, -2.31).
lumo(d170, -2.113).
lumo(d176, -1.304).
lumo(d180, -2.182).
lumo(f4, -1.645).
lumo(f5, -1.526).
lumo(d2, -1.429).
lumo(d17, -0.529).
lumo(d42, -1.19).
lumo(d70, -1.786).
lumo(d77, -1.029).
lumo(d89, -1.028).
lumo(d98, -1.598).
lumo(d111, -1.687).
lumo(d114, -1.148).
lumo(d119, -1.995).
lumo(d120, -0.937).
lumo(d130, -0.93).
lumo(d131, -1.538).
lumo(d132, -1.499).
lumo(d138, -1.157).
lumo(d141, -1.228).
lumo(d144, -1.288).
lumo(d147, -0.937).
lumo(d186, -0.574).
lumo(d188, -0.872).
lumo(d189, -3.025).
lumo(d196, -1.092).
lumo(e4, -0.746).
lumo(e5, -1.089).
lumo(e8, -1.117).
lumo(e12, -1.834).
lumo(e14, -1.36).
lumo(e18, -1.538).
lumo(e22, -1.636).
lumo(e25, -1.748).
lumo(e26, -1.82).
lumo(d1, -1.246).
lumo(d4, -1.591).
lumo(d6, -3.406).
lumo(d10, -1.387).
lumo(d11, -1.254).
lumo(d12, -1.627).
lumo(d13, -2.292).
lumo(d15, -1.698).
lumo(d18, -1.861).
lumo(d20, -1.764).
lumo(d24, -1.728).
lumo(d25, -1.398).
lumo(d26, -2.072).
lumo(d27, -1.801).
lumo(d28, -1.558).
lumo(d31, -2.055).
lumo(d41, -1.6).
lumo(d43, -1.57).
lumo(d45, -2.227).
lumo(d46, -1.804).
lumo(d48, -1.616).
lumo(d49, -2.808).
lumo(d50, -1.932).
lumo(d51, -2.184).
lumo(d52, -2.189).
lumo(d53, -2.155).
lumo(d54, -1.464).
lumo(d56, -2.234).
lumo(d57, -2.242).
lumo(d58, -1.991).
lumo(d59, -2.14).
lumo(d60, -2.468).
lumo(d61, -1.556).
lumo(d63, -3.768).
lumo(d64, -2.338).
lumo(d67, -1.536).
lumo(d68, -1.766).
lumo(d69, -1.929).
lumo(d71, -2.095).
lumo(d72, -1.448).
lumo(d75, -1.87).
lumo(d79, -1.26).
lumo(d80, -1.329).
lumo(d82, -2.71).
lumo(d83, -1.145).
lumo(d86, -1.918).
lumo(d87, -1.488).
lumo(d90, -1.62).
lumo(d94, -1.585).
lumo(d95, -2.434).
lumo(d96, -2.871).
lumo(d101, -1.598).
lumo(d104, -1.546).
lumo(d105, -1.749).
lumo(d107, -2.791).
lumo(d109, -2.284).
lumo(d112, -1.208).
lumo(d115, -1.161).
lumo(d117, -2.142).
lumo(d122, -1.61).
lumo(d125, -1.256).
lumo(d126, -2.718).
lumo(d127, -2.196).
lumo(d128, -2.149).
lumo(d136, -1.685).
lumo(d137, -2.263).
lumo(d140, -1.751).
lumo(d146, -1.102).
lumo(d148, -2.158).
lumo(d149, -2.87).
lumo(d151, -1.411).
lumo(d152, -2.191).
lumo(d153, -1.151).
lumo(d157, -1.623).
lumo(d159, -1.742).
lumo(d163, -1.974).
lumo(d164, -2.306).
lumo(d165, -1.522).
lumo(d166, -1.709).
lumo(d169, -2.17).
lumo(d171, -2.508).
lumo(d172, -1.487).
lumo(d173, -2.328).
lumo(d174, -2.209).
lumo(d177, -2.614).
lumo(d178, -1.213).
lumo(d182, -1.366).
lumo(d183, -2.294).
lumo(d184, -2.074).
lumo(d187, -1.276).
lumo(d190, -0.798).
lumo(d191, -2.138).
lumo(d194, -0.857).
lumo(d197, -1.293).
lumo(e1, -1.767).
lumo(e2, -1.35).
lumo(e27, -1.845).
lumo(f1, -1.785).
lumo(f2, -1.851).
lumo(f3, -1.412).
lumo(f6, -1.503).
lumo(d3, -1.456).
lumo(d5, -1.59).
lumo(d7, -1.144).
lumo(d9, -0.982).
lumo(d14, -1.289).
lumo(d19, -1.478).
lumo(d34, -1.451).
lumo(d36, -1.24).
lumo(d38, -1.228).
lumo(d39, -1.443).
lumo(d40, -1.266).
lumo(d55, -1.177).
lumo(d62, -0.923).
lumo(d65, -1.952).
lumo(d66, -0.959).
lumo(d73, -0.648).
lumo(d76, -1.069).
lumo(d78, -1.276).
lumo(d84, -1.256).
lumo(d88, -1.605).
lumo(d100, -1.034).
lumo(d110, -2.14).
lumo(d113, -1.491).
lumo(d116, -1.37).
lumo(d123, -1.056).
lumo(d124, -1.596).
lumo(d129, -1.592).
lumo(d133, -1.492).
lumo(d135, -1.562).
lumo(d139, -1.019).
lumo(d142, -1.474).
lumo(d143, -1.046).
lumo(d150, -0.995).
lumo(d154, -1.362).
lumo(d155, -1.737).
lumo(d156, -0.854).
lumo(d160, -1.691).
lumo(d168, -1.112).
lumo(d175, -1.616).
lumo(d179, -1.889).
lumo(d181, -0.727).
lumo(d185, -2.09).
lumo(d192, -1.429).
lumo(d193, -1.478).
lumo(d195, -1.465).
lumo(e3, -0.56).
lumo(e6, -0.868).
lumo(e7, -1.05).
lumo(e9, -1.321).
lumo(e10, -1.125).
lumo(e11, -1.358).
lumo(e13, -1.306).
lumo(e15, -1.723).
lumo(e16, -1.351).
lumo(e17, -1.755).
lumo(e19, -1.392).
lumo(e20, -1.447).
lumo(e21, -1.717).
lumo(e23, -1.808).
lumo(e24, -1.113).
benzene(d1,[d1_6,d1_1,d1_2,d1_3,d1_4,d1_5]).
benzene(d1,[d1_5,d1_14,d1_13,d1_12,d1_11,d1_4]).
benzene(d1,[d1_20,d1_12,d1_11,d1_17,d1_18,d1_19]).
ring_size_6(d1,[d1_6,d1_1,d1_2,d1_3,d1_4,d1_5]).
ring_size_6(d1,[d1_5,d1_14,d1_13,d1_12,d1_11,d1_4]).
ring_size_6(d1,[d1_20,d1_12,d1_11,d1_17,d1_18,d1_19]).
nitro(d1,[d1_19,d1_24,d1_25,d1_26]).
phenanthrene(d1,[[d1_6,d1_1,d1_2,d1_3,d1_4,d1_5],[d1_5,d1_14,d1_13,d1_12,d1_11,d1_4],[d1_20,d1_12,d1_11,d1_17,d1_18,d1_19]]).
benzene(d4,[d4_6,d4_1,d4_2,d4_3,d4_4,d4_5]).
benzene(d4,[d4_9,d4_10,d4_4,d4_3,d4_7,d4_8]).
benzene(d4,[d4_16,d4_11,d4_12,d4_13,d4_14,d4_15]).
ring_size_6(d4,[d4_6,d4_1,d4_2,d4_3,d4_4,d4_5]).
ring_size_6(d4,[d4_9,d4_10,d4_4,d4_3,d4_7,d4_8]).
ring_size_6(d4,[d4_16,d4_11,d4_12,d4_13,d4_14,d4_15]).
ring_size_5(d4,[d4_5,d4_12,d4_13,d4_10,d4_4]).
nitro(d4,[d4_9,d4_17,d4_18,d4_19]).
benzene(d6,[d6_6,d6_1,d6_2,d6_3,d6_4,d6_5]).
benzene(d6,[d6_9,d6_10,d6_5,d6_4,d6_7,d6_8]).
benzene(d6,[d6_7,d6_13,d6_12,d6_11,d6_3,d6_4]).
benzene(d6,[d6_8,d6_15,d6_14,d6_16,d6_13,d6_7]).
ring_size_6(d6,[d6_6,d6_1,d6_2,d6_3,d6_4,d6_5]).
ring_size_6(d6,[d6_9,d6_10,d6_5,d6_4,d6_7,d6_8]).
ring_size_6(d6,[d6_7,d6_13,d6_12,d6_11,d6_3,d6_4]).
ring_size_6(d6,[d6_8,d6_15,d6_14,d6_16,d6_13,d6_7]).
nitro(d6,[d6_16,d6_23,d6_27,d6_28]).
nitro(d6,[d6_2,d6_24,d6_29,d6_30]).
nitro(d6,[d6_6,d6_33,d6_31,d6_32]).
nitro(d6,[d6_15,d6_34,d6_25,d6_26]).
phenanthrene(d6,[[d6_8,d6_15,d6_14,d6_16,d6_13,d6_7],[d6_7,d6_13,d6_12,d6_11,d6_3,d6_4],[d6_6,d6_1,d6_2,d6_3,d6_4,d6_5]]).
ball3(d6,[[d6_9,d6_10,d6_5,d6_4,d6_7,d6_8],[d6_7,d6_13,d6_12,d6_11,d6_3,d6_4],[d6_6,d6_1,d6_2,d6_3,d6_4,d6_5]]).
ball3(d6,[[d6_9,d6_10,d6_5,d6_4,d6_7,d6_8],[d6_8,d6_15,d6_14,d6_16,d6_13,d6_7],[d6_7,d6_13,d6_12,d6_11,d6_3,d6_4]]).
benzene(d10,[d10_6,d10_1,d10_2,d10_3,d10_4,d10_5]).
benzene(d10,[d10_9,d10_10,d10_11,d10_12,d10_7,d10_8]).
ring_size_6(d10,[d10_6,d10_1,d10_2,d10_3,d10_4,d10_5]).
ring_size_6(d10,[d10_9,d10_10,d10_11,d10_12,d10_7,d10_8]).
ring_size_5(d10,[d10_7,d10_12,d10_4,d10_3,d10_13]).
nitro(d10,[d10_1,d10_14,d10_15,d10_16]).
benzene(d11,[d11_6,d11_1,d11_2,d11_3,d11_4,d11_5]).
benzene(d11,[d11_5,d11_14,d11_13,d11_12,d11_11,d11_4]).
benzene(d11,[d11_19,d11_12,d11_11,d11_16,d11_17,d11_18]).
ring_size_6(d11,[d11_6,d11_1,d11_2,d11_3,d11_4,d11_5]).
ring_size_6(d11,[d11_5,d11_14,d11_13,d11_12,d11_11,d11_4]).
ring_size_6(d11,[d11_19,d11_12,d11_11,d11_16,d11_17,d11_18]).
nitro(d11,[d11_14,d11_24,d11_25,d11_26]).
phenanthrene(d11,[[d11_6,d11_1,d11_2,d11_3,d11_4,d11_5],[d11_5,d11_14,d11_13,d11_12,d11_11,d11_4],[d11_19,d11_12,d11_11,d11_16,d11_17,d11_18]]).
benzene(d12,[d12_6,d12_1,d12_2,d12_3,d12_4,d12_5]).
benzene(d12,[d12_9,d12_10,d12_11,d12_12,d12_5,d12_4]).
benzene(d12,[d12_9,d12_10,d12_22,d12_21,d12_20,d12_17]).
benzene(d12,[d12_9,d12_17,d12_16,d12_15,d12_3,d12_4]).
ring_size_6(d12,[d12_6,d12_1,d12_2,d12_3,d12_4,d12_5]).
ring_size_6(d12,[d12_9,d12_10,d12_11,d12_12,d12_5,d12_4]).
ring_size_6(d12,[d12_9,d12_10,d12_22,d12_21,d12_20,d12_17]).
ring_size_6(d12,[d12_9,d12_17,d12_16,d12_15,d12_3,d12_4]).
nitro(d12,[d12_6,d12_25,d12_26,d12_27]).
phenanthrene(d12,[[d12_9,d12_10,d12_22,d12_21,d12_20,d12_17],[d12_9,d12_10,d12_11,d12_12,d12_5,d12_4],[d12_6,d12_1,d12_2,d12_3,d12_4,d12_5]]).
ball3(d12,[[d12_9,d12_17,d12_16,d12_15,d12_3,d12_4],[d12_9,d12_10,d12_11,d12_12,d12_5,d12_4],[d12_6,d12_1,d12_2,d12_3,d12_4,d12_5]]).
ball3(d12,[[d12_9,d12_17,d12_16,d12_15,d12_3,d12_4],[d12_9,d12_10,d12_22,d12_21,d12_20,d12_17],[d12_9,d12_10,d12_11,d12_12,d12_5,d12_4]]).
benzene(d13,[d13_6,d13_1,d13_2,d13_3,d13_4,d13_5]).
benzene(d13,[d13_9,d13_10,d13_4,d13_3,d13_7,d13_8]).
benzene(d13,[d13_16,d13_11,d13_12,d13_13,d13_14,d13_15]).
ring_size_6(d13,[d13_6,d13_1,d13_2,d13_3,d13_4,d13_5]).
ring_size_6(d13,[d13_9,d13_10,d13_4,d13_3,d13_7,d13_8]).
ring_size_6(d13,[d13_16,d13_11,d13_12,d13_13,d13_14,d13_15]).
ring_size_5(d13,[d13_5,d13_12,d13_13,d13_10,d13_4]).
nitro(d13,[d13_8,d13_17,d13_19,d13_20]).
nitro(d13,[d13_2,d13_18,d13_21,d13_22]).
benzene(d15,[d15_6,d15_1,d15_2,d15_3,d15_4,d15_5]).
benzene(d15,[d15_5,d15_13,d15_12,d15_11,d15_10,d15_4]).
benzene(d15,[d15_23,d15_11,d15_10,d15_18,d15_21,d15_22]).
benzene(d15,[d15_4,d15_10,d15_18,d15_17,d15_16,d15_3]).
ring_size_6(d15,[d15_6,d15_1,d15_2,d15_3,d15_4,d15_5]).
ring_size_6(d15,[d15_5,d15_13,d15_12,d15_11,d15_10,d15_4]).
ring_size_6(d15,[d15_23,d15_11,d15_10,d15_18,d15_21,d15_22]).
ring_size_6(d15,[d15_4,d15_10,d15_18,d15_17,d15_16,d15_3]).
nitro(d15,[d15_23,d15_26,d15_27,d15_28]).
phenanthrene(d15,[[d15_6,d15_1,d15_2,d15_3,d15_4,d15_5],[d15_5,d15_13,d15_12,d15_11,d15_10,d15_4],[d15_23,d15_11,d15_10,d15_18,d15_21,d15_22]]).
phenanthrene(d15,[[d15_6,d15_1,d15_2,d15_3,d15_4,d15_5],[d15_4,d15_10,d15_18,d15_17,d15_16,d15_3],[d15_23,d15_11,d15_10,d15_18,d15_21,d15_22]]).
ball3(d15,[[d15_6,d15_1,d15_2,d15_3,d15_4,d15_5],[d15_5,d15_13,d15_12,d15_11,d15_10,d15_4],[d15_4,d15_10,d15_18,d15_17,d15_16,d15_3]]).
ball3(d15,[[d15_5,d15_13,d15_12,d15_11,d15_10,d15_4],[d15_4,d15_10,d15_18,d15_17,d15_16,d15_3],[d15_23,d15_11,d15_10,d15_18,d15_21,d15_22]]).
benzene(d18,[d18_6,d18_1,d18_2,d18_3,d18_4,d18_5]).
benzene(d18,[d18_9,d18_10,d18_11,d18_12,d18_7,d18_8]).
ring_size_6(d18,[d18_6,d18_1,d18_2,d18_3,d18_4,d18_5]).
ring_size_6(d18,[d18_9,d18_10,d18_11,d18_12,d18_7,d18_8]).
ring_size_5(d18,[d18_7,d18_12,d18_4,d18_3,d18_13]).
nitro(d18,[d18_10,d18_15,d18_16,d18_17]).
benzene(d20,[d20_6,d20_1,d20_2,d20_3,d20_4,d20_5]).
benzene(d20,[d20_9,d20_10,d20_11,d20_12,d20_7,d20_8]).
ring_size_6(d20,[d20_6,d20_1,d20_2,d20_3,d20_4,d20_5]).
ring_size_6(d20,[d20_9,d20_10,d20_11,d20_12,d20_7,d20_8]).
ring_size_5(d20,[d20_7,d20_12,d20_4,d20_3,d20_13]).
nitro(d20,[d20_1,d20_14,d20_15,d20_16]).
benzene(d24,[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5]).
benzene(d24,[d24_9,d24_10,d24_11,d24_12,d24_4,d24_3]).
benzene(d24,[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]).
benzene(d24,[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4]).
benzene(d24,[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]).
benzene(d24,[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28]).
benzene(d24,[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5]).
ring_size_6(d24,[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5]).
ring_size_6(d24,[d24_9,d24_10,d24_11,d24_12,d24_4,d24_3]).
ring_size_6(d24,[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]).
ring_size_6(d24,[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4]).
ring_size_6(d24,[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]).
ring_size_6(d24,[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28]).
ring_size_6(d24,[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5]).
nitro(d24,[d24_22,d24_36,d24_37,d24_38]).
anthracene(d24,[[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]]).
anthracene(d24,[[d24_9,d24_10,d24_11,d24_12,d24_4,d24_3],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28]]).
anthracene(d24,[[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]]).
phenanthrene(d24,[[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]]).
phenanthrene(d24,[[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28]]).
phenanthrene(d24,[[d24_9,d24_10,d24_11,d24_12,d24_4,d24_3],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]]).
phenanthrene(d24,[[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23],[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]]).
phenanthrene(d24,[[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]]).
phenanthrene(d24,[[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5],[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]]).
ball3(d24,[[d24_9,d24_10,d24_11,d24_12,d24_4,d24_3],[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4]]).
ball3(d24,[[d24_9,d24_10,d24_11,d24_12,d24_4,d24_3],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]]).
ball3(d24,[[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23],[d24_18,d24_12,d24_11,d24_15,d24_16,d24_17]]).
ball3(d24,[[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28],[d24_24,d24_18,d24_17,d24_21,d24_22,d24_23]]).
ball3(d24,[[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5],[d24_6,d24_1,d24_2,d24_3,d24_4,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4]]).
ball3(d24,[[d24_6,d24_33,d24_32,d24_28,d24_29,d24_5],[d24_5,d24_29,d24_24,d24_18,d24_12,d24_4],[d24_29,d24_24,d24_23,d24_26,d24_27,d24_28]]).
benzene(d25,[d25_6,d25_1,d25_2,d25_3,d25_4,d25_5]).
benzene(d25,[d25_9,d25_10,d25_11,d25_12,d25_7,d25_8]).
ring_size_6(d25,[d25_6,d25_1,d25_2,d25_3,d25_4,d25_5]).
ring_size_6(d25,[d25_9,d25_10,d25_11,d25_12,d25_7,d25_8]).
ring_size_5(d25,[d25_7,d25_12,d25_4,d25_3,d25_13]).
nitro(d25,[d25_1,d25_15,d25_16,d25_17]).
benzene(d26,[d26_6,d26_1,d26_2,d26_3,d26_4,d26_5]).
ring_size_6(d26,[d26_6,d26_1,d26_2,d26_3,d26_4,d26_5]).
nitro(d26,[d26_4,d26_11,d26_13,d26_14]).
nitro(d26,[d26_2,d26_12,d26_15,d26_16]).
benzene(d27,[d27_6,d27_1,d27_2,d27_3,d27_4,d27_5]).
benzene(d27,[d27_4,d27_14,d27_13,d27_12,d27_11,d27_3]).
benzene(d27,[d27_19,d27_13,d27_12,d27_16,d27_17,d27_18]).
benzene(d27,[d27_24,d27_14,d27_13,d27_19,d27_22,d27_23]).
benzene(d27,[d27_29,d27_22,d27_19,d27_18,d27_27,d27_28]).
ring_size_6(d27,[d27_6,d27_1,d27_2,d27_3,d27_4,d27_5]).
ring_size_6(d27,[d27_4,d27_14,d27_13,d27_12,d27_11,d27_3]).
ring_size_6(d27,[d27_19,d27_13,d27_12,d27_16,d27_17,d27_18]).
ring_size_6(d27,[d27_24,d27_14,d27_13,d27_19,d27_22,d27_23]).
ring_size_6(d27,[d27_29,d27_22,d27_19,d27_18,d27_27,d27_28]).
nitro(d27,[d27_29,d27_32,d27_33,d27_34]).
anthracene(d27,[[d27_6,d27_1,d27_2,d27_3,d27_4,d27_5],[d27_4,d27_14,d27_13,d27_12,d27_11,d27_3],[d27_19,d27_13,d27_12,d27_16,d27_17,d27_18]]).
phenanthrene(d27,[[d27_6,d27_1,d27_2,d27_3,d27_4,d27_5],[d27_4,d27_14,d27_13,d27_12,d27_11,d27_3],[d27_24,d27_14,d27_13,d27_19,d27_22,d27_23]]).
ball3(d27,[[d27_4,d27_14,d27_13,d27_12,d27_11,d27_3],[d27_24,d27_14,d27_13,d27_19,d27_22,d27_23],[d27_19,d27_13,d27_12,d27_16,d27_17,d27_18]]).
ball3(d27,[[d27_29,d27_22,d27_19,d27_18,d27_27,d27_28],[d27_24,d27_14,d27_13,d27_19,d27_22,d27_23],[d27_19,d27_13,d27_12,d27_16,d27_17,d27_18]]).
benzene(d28,[d28_6,d28_1,d28_2,d28_3,d28_4,d28_5]).
benzene(d28,[d28_9,d28_10,d28_11,d28_12,d28_7,d28_8]).
ring_size_6(d28,[d28_6,d28_1,d28_2,d28_3,d28_4,d28_5]).
ring_size_6(d28,[d28_9,d28_10,d28_11,d28_12,d28_7,d28_8]).
ring_size_5(d28,[d28_7,d28_12,d28_4,d28_3,d28_13]).
nitro(d28,[d28_1,d28_14,d28_15,d28_16]).
benzene(d31,[d31_6,d31_1,d31_2,d31_3,d31_4,d31_5]).
benzene(d31,[d31_4,d31_13,d31_12,d31_11,d31_10,d31_3]).
benzene(d31,[d31_20,d31_15,d31_16,d31_17,d31_18,d31_19]).
ring_size_6(d31,[d31_6,d31_1,d31_2,d31_3,d31_4,d31_5]).
ring_size_6(d31,[d31_4,d31_13,d31_12,d31_11,d31_10,d31_3]).
ring_size_6(d31,[d31_20,d31_15,d31_16,d31_17,d31_18,d31_19]).
ring_size_5(d31,[d31_5,d31_15,d31_16,d31_13,d31_4]).
nitro(d31,[d31_12,d31_24,d31_26,d31_27]).
nitro(d31,[d31_11,d31_25,d31_28,d31_29]).
benzene(d41,[d41_6,d41_1,d41_2,d41_3,d41_4,d41_5]).
benzene(d41,[d41_15,d41_10,d41_11,d41_12,d41_13,d41_14]).
ring_size_6(d41,[d41_6,d41_1,d41_2,d41_3,d41_4,d41_5]).
ring_size_6(d41,[d41_15,d41_10,d41_11,d41_12,d41_13,d41_14]).
ring_size_5(d41,[d41_4,d41_15,d41_10,d41_19,d41_3]).
nitro(d41,[d41_1,d41_23,d41_24,d41_25]).
benzene(d43,[d43_6,d43_1,d43_2,d43_3,d43_4,d43_5]).
benzene(d43,[d43_4,d43_13,d43_12,d43_11,d43_10,d43_3]).
benzene(d43,[d43_21,d43_16,d43_17,d43_18,d43_19,d43_20]).
ring_size_6(d43,[d43_6,d43_1,d43_2,d43_3,d43_4,d43_5]).
ring_size_6(d43,[d43_4,d43_13,d43_12,d43_11,d43_10,d43_3]).
ring_size_6(d43,[d43_21,d43_16,d43_17,d43_18,d43_19,d43_20]).
ring_size_5(d43,[d43_5,d43_16,d43_17,d43_13,d43_4]).
nitro(d43,[d43_11,d43_25,d43_26,d43_27]).
benzene(d45,[d45_6,d45_1,d45_2,d45_3,d45_4,d45_5]).
ring_size_6(d45,[d45_6,d45_1,d45_2,d45_3,d45_4,d45_5]).
nitro(d45,[d45_5,d45_11,d45_13,d45_14]).
nitro(d45,[d45_2,d45_12,d45_15,d45_16]).
benzene(d46,[d46_9,d46_10,d46_4,d46_3,d46_7,d46_8]).
benzene(d46,[d46_9,d46_10,d46_17,d46_16,d46_15,d46_14]).
benzene(d46,[d46_9,d46_14,d46_13,d46_12,d46_11,d46_8]).
benzene(d46,[d46_20,d46_15,d46_14,d46_13,d46_18,d46_19]).
carbon_6_ring(d46,[d46_6,d46_1,d46_2,d46_3,d46_4,d46_5]).
ring_size_6(d46,[d46_6,d46_1,d46_2,d46_3,d46_4,d46_5]).
ring_size_6(d46,[d46_9,d46_10,d46_4,d46_3,d46_7,d46_8]).
ring_size_6(d46,[d46_9,d46_10,d46_17,d46_16,d46_15,d46_14]).
ring_size_6(d46,[d46_9,d46_14,d46_13,d46_12,d46_11,d46_8]).
ring_size_6(d46,[d46_20,d46_15,d46_14,d46_13,d46_18,d46_19]).
nitro(d46,[d46_20,d46_21,d46_22,d46_23]).
phenanthrene(d46,[[d46_9,d46_10,d46_4,d46_3,d46_7,d46_8],[d46_9,d46_10,d46_17,d46_16,d46_15,d46_14],[d46_20,d46_15,d46_14,d46_13,d46_18,d46_19]]).
ball3(d46,[[d46_9,d46_14,d46_13,d46_12,d46_11,d46_8],[d46_9,d46_10,d46_4,d46_3,d46_7,d46_8],[d46_9,d46_10,d46_17,d46_16,d46_15,d46_14]]).
ball3(d46,[[d46_9,d46_14,d46_13,d46_12,d46_11,d46_8],[d46_9,d46_10,d46_17,d46_16,d46_15,d46_14],[d46_20,d46_15,d46_14,d46_13,d46_18,d46_19]]).
benzene(d48,[d48_6,d48_1,d48_2,d48_3,d48_4,d48_5]).
benzene(d48,[d48_4,d48_14,d48_13,d48_12,d48_11,d48_3]).
benzene(d48,[d48_20,d48_13,d48_12,d48_17,d48_18,d48_19]).
ring_size_6(d48,[d48_6,d48_1,d48_2,d48_3,d48_4,d48_5]).
ring_size_6(d48,[d48_4,d48_14,d48_13,d48_12,d48_11,d48_3]).
ring_size_6(d48,[d48_20,d48_13,d48_12,d48_17,d48_18,d48_19]).
nitro(d48,[d48_19,d48_24,d48_25,d48_26]).
anthracene(d48,[[d48_6,d48_1,d48_2,d48_3,d48_4,d48_5],[d48_4,d48_14,d48_13,d48_12,d48_11,d48_3],[d48_20,d48_13,d48_12,d48_17,d48_18,d48_19]]).
benzene(d49,[d49_6,d49_1,d49_2,d49_3,d49_4,d49_5]).
benzene(d49,[d49_15,d49_10,d49_11,d49_12,d49_13,d49_14]).
ring_size_6(d49,[d49_6,d49_1,d49_2,d49_3,d49_4,d49_5]).
ring_size_6(d49,[d49_15,d49_10,d49_11,d49_12,d49_13,d49_14]).
ring_size_6(d49,[d49_4,d49_20,d49_15,d49_10,d49_19,d49_3]).
nitro(d49,[d49_13,d49_23,d49_25,d49_26]).
nitro(d49,[d49_6,d49_24,d49_27,d49_28]).
benzene(d50,[d50_6,d50_1,d50_2,d50_3,d50_4,d50_5]).
benzene(d50,[d50_4,d50_13,d50_12,d50_11,d50_10,d50_3]).
ring_size_6(d50,[d50_6,d50_1,d50_2,d50_3,d50_4,d50_5]).
ring_size_6(d50,[d50_4,d50_13,d50_12,d50_11,d50_10,d50_3]).
nitro(d50,[d50_12,d50_17,d50_19,d50_20]).
nitro(d50,[d50_2,d50_18,d50_21,d50_22]).
benzene(d51,[d51_6,d51_1,d51_2,d51_3,d51_4,d51_5]).
benzene(d51,[d51_9,d51_10,d51_4,d51_3,d51_7,d51_8]).
benzene(d51,[d51_5,d51_13,d51_12,d51_11,d51_10,d51_4]).
benzene(d51,[d51_9,d51_10,d51_11,d51_16,d51_15,d51_14]).
carbon_6_ring(d51,[d51_20,d51_13,d51_12,d51_17,d51_18,d51_19]).
ring_size_6(d51,[d51_6,d51_1,d51_2,d51_3,d51_4,d51_5]).
ring_size_6(d51,[d51_9,d51_10,d51_4,d51_3,d51_7,d51_8]).
ring_size_6(d51,[d51_5,d51_13,d51_12,d51_11,d51_10,d51_4]).
ring_size_6(d51,[d51_9,d51_10,d51_11,d51_16,d51_15,d51_14]).
ring_size_6(d51,[d51_20,d51_13,d51_12,d51_17,d51_18,d51_19]).
nitro(d51,[d51_16,d51_21,d51_23,d51_24]).
nitro(d51,[d51_2,d51_22,d51_25,d51_26]).
ball3(d51,[[d51_9,d51_10,d51_4,d51_3,d51_7,d51_8],[d51_6,d51_1,d51_2,d51_3,d51_4,d51_5],[d51_5,d51_13,d51_12,d51_11,d51_10,d51_4]]).
ball3(d51,[[d51_9,d51_10,d51_4,d51_3,d51_7,d51_8],[d51_9,d51_10,d51_11,d51_16,d51_15,d51_14],[d51_5,d51_13,d51_12,d51_11,d51_10,d51_4]]).
benzene(d52,[d52_6,d52_1,d52_2,d52_3,d52_4,d52_5]).
benzene(d52,[d52_9,d52_10,d52_4,d52_3,d52_7,d52_8]).
benzene(d52,[d52_5,d52_13,d52_12,d52_11,d52_10,d52_4]).
benzene(d52,[d52_9,d52_10,d52_11,d52_16,d52_15,d52_14]).
carbon_6_ring(d52,[d52_20,d52_13,d52_12,d52_17,d52_18,d52_19]).
ring_size_6(d52,[d52_6,d52_1,d52_2,d52_3,d52_4,d52_5]).
ring_size_6(d52,[d52_9,d52_10,d52_4,d52_3,d52_7,d52_8]).
ring_size_6(d52,[d52_5,d52_13,d52_12,d52_11,d52_10,d52_4]).
ring_size_6(d52,[d52_9,d52_10,d52_11,d52_16,d52_15,d52_14]).
ring_size_6(d52,[d52_20,d52_13,d52_12,d52_17,d52_18,d52_19]).
nitro(d52,[d52_16,d52_21,d52_22,d52_23]).
nitro(d52,[d52_14,d52_24,d52_25,d52_26]).
ball3(d52,[[d52_9,d52_10,d52_4,d52_3,d52_7,d52_8],[d52_6,d52_1,d52_2,d52_3,d52_4,d52_5],[d52_5,d52_13,d52_12,d52_11,d52_10,d52_4]]).
ball3(d52,[[d52_9,d52_10,d52_4,d52_3,d52_7,d52_8],[d52_9,d52_10,d52_11,d52_16,d52_15,d52_14],[d52_5,d52_13,d52_12,d52_11,d52_10,d52_4]]).
benzene(d53,[d53_6,d53_1,d53_2,d53_3,d53_4,d53_5]).
benzene(d53,[d53_15,d53_10,d53_11,d53_12,d53_13,d53_14]).
ring_size_6(d53,[d53_6,d53_1,d53_2,d53_3,d53_4,d53_5]).
ring_size_6(d53,[d53_15,d53_10,d53_11,d53_12,d53_13,d53_14]).
ring_size_5(d53,[d53_4,d53_15,d53_10,d53_19,d53_3]).
nitro(d53,[d53_12,d53_22,d53_24,d53_25]).
nitro(d53,[d53_1,d53_23,d53_26,d53_27]).
benzene(d54,[d54_6,d54_1,d54_2,d54_3,d54_4,d54_5]).
benzene(d54,[d54_15,d54_10,d54_11,d54_12,d54_13,d54_14]).
ring_size_6(d54,[d54_6,d54_1,d54_2,d54_3,d54_4,d54_5]).
ring_size_6(d54,[d54_15,d54_10,d54_11,d54_12,d54_13,d54_14]).
ring_size_6(d54,[d54_4,d54_19,d54_15,d54_10,d54_18,d54_3]).
nitro(d54,[d54_1,d54_20,d54_21,d54_22]).
benzene(d56,[d56_6,d56_1,d56_2,d56_3,d56_4,d56_5]).
benzene(d56,[d56_16,d56_11,d56_12,d56_13,d56_14,d56_15]).
ring_size_6(d56,[d56_6,d56_1,d56_2,d56_3,d56_4,d56_5]).
ring_size_6(d56,[d56_16,d56_11,d56_12,d56_13,d56_14,d56_15]).
ring_size_6(d56,[d56_4,d56_21,d56_16,d56_11,d56_20,d56_3]).
nitro(d56,[d56_14,d56_24,d56_25,d56_26]).
benzene(d57,[d57_6,d57_1,d57_2,d57_3,d57_4,d57_5]).
benzene(d57,[d57_16,d57_11,d57_12,d57_13,d57_14,d57_15]).
ring_size_6(d57,[d57_6,d57_1,d57_2,d57_3,d57_4,d57_5]).
ring_size_6(d57,[d57_16,d57_11,d57_12,d57_13,d57_14,d57_15]).
nitro(d57,[d57_16,d57_20,d57_23,d57_24]).
nitro(d57,[d57_14,d57_21,d57_25,d57_26]).
nitro(d57,[d57_1,d57_22,d57_27,d57_28]).
benzene(d58,[d58_6,d58_1,d58_2,d58_3,d58_4,d58_5]).
benzene(d58,[d58_9,d58_10,d58_4,d58_3,d58_7,d58_8]).
benzene(d58,[d58_9,d58_10,d58_11,d58_16,d58_15,d58_14]).
carbon_6_ring(d58,[d58_5,d58_13,d58_12,d58_11,d58_10,d58_4]).
ring_size_6(d58,[d58_6,d58_1,d58_2,d58_3,d58_4,d58_5]).
ring_size_6(d58,[d58_9,d58_10,d58_4,d58_3,d58_7,d58_8]).
ring_size_6(d58,[d58_5,d58_13,d58_12,d58_11,d58_10,d58_4]).
ring_size_6(d58,[d58_9,d58_10,d58_11,d58_16,d58_15,d58_14]).
nitro(d58,[d58_14,d58_17,d58_18,d58_19]).
benzene(d59,[d59_6,d59_1,d59_2,d59_3,d59_4,d59_5]).
benzene(d59,[d59_9,d59_10,d59_3,d59_2,d59_7,d59_8]).
benzene(d59,[d59_9,d59_10,d59_14,d59_13,d59_12,d59_11]).
benzene(d59,[d59_18,d59_13,d59_12,d59_15,d59_16,d59_17]).
carbon_5_aromatic_ring(d59,[d59_20,d59_18,d59_13,d59_14,d59_19]).
ring_size_6(d59,[d59_6,d59_1,d59_2,d59_3,d59_4,d59_5]).
ring_size_6(d59,[d59_9,d59_10,d59_3,d59_2,d59_7,d59_8]).
ring_size_6(d59,[d59_9,d59_10,d59_14,d59_13,d59_12,d59_11]).
ring_size_6(d59,[d59_18,d59_13,d59_12,d59_15,d59_16,d59_17]).
ring_size_5(d59,[d59_20,d59_18,d59_13,d59_14,d59_19]).
nitro(d59,[d59_20,d59_32,d59_33,d59_34]).
anthracene(d59,[[d59_9,d59_10,d59_3,d59_2,d59_7,d59_8],[d59_9,d59_10,d59_14,d59_13,d59_12,d59_11],[d59_18,d59_13,d59_12,d59_15,d59_16,d59_17]]).
benzene(d60,[d60_6,d60_1,d60_2,d60_3,d60_4,d60_5]).
benzene(d60,[d60_4,d60_13,d60_12,d60_11,d60_10,d60_3]).
ring_size_6(d60,[d60_6,d60_1,d60_2,d60_3,d60_4,d60_5]).
ring_size_6(d60,[d60_4,d60_13,d60_12,d60_11,d60_10,d60_3]).
nitro(d60,[d60_13,d60_16,d60_21,d60_22]).
nitro(d60,[d60_11,d60_17,d60_23,d60_24]).
nitro(d60,[d60_5,d60_18,d60_19,d60_20]).
benzene(d61,[d61_6,d61_1,d61_2,d61_3,d61_4,d61_5]).
benzene(d61,[d61_15,d61_10,d61_11,d61_12,d61_13,d61_14]).
ring_size_6(d61,[d61_6,d61_1,d61_2,d61_3,d61_4,d61_5]).
ring_size_6(d61,[d61_15,d61_10,d61_11,d61_12,d61_13,d61_14]).
ring_size_5(d61,[d61_4,d61_15,d61_10,d61_19,d61_3]).
nitro(d61,[d61_1,d61_23,d61_24,d61_25]).
carbon_5_aromatic_ring(d63,[d63_20,d63_16,d63_17,d63_18,d63_19]).
hetero_aromatic_5_ring(d63,[d63_5,d63_1,d63_2,d63_3,d63_4]).
ring_size_5(d63,[d63_5,d63_1,d63_2,d63_3,d63_4]).
ring_size_5(d63,[d63_20,d63_16,d63_17,d63_18,d63_19]).
nitro(d63,[d63_1,d63_6,d63_7,d63_8]).
nitro(d63,[d63_19,d63_21,d63_22,d63_23]).
benzene(d64,[d64_6,d64_1,d64_2,d64_3,d64_4,d64_5]).
benzene(d64,[d64_15,d64_10,d64_11,d64_12,d64_13,d64_14]).
ring_size_6(d64,[d64_6,d64_1,d64_2,d64_3,d64_4,d64_5]).
ring_size_6(d64,[d64_15,d64_10,d64_11,d64_12,d64_13,d64_14]).
ring_size_5(d64,[d64_4,d64_15,d64_10,d64_19,d64_3]).
nitro(d64,[d64_12,d64_20,d64_24,d64_25]).
nitro(d64,[d64_1,d64_21,d64_22,d64_23]).
benzene(d67,[d67_6,d67_1,d67_2,d67_3,d67_4,d67_5]).
benzene(d67,[d67_4,d67_13,d67_12,d67_11,d67_10,d67_3]).
benzene(d67,[d67_20,d67_15,d67_16,d67_17,d67_18,d67_19]).
ring_size_6(d67,[d67_6,d67_1,d67_2,d67_3,d67_4,d67_5]).
ring_size_6(d67,[d67_4,d67_13,d67_12,d67_11,d67_10,d67_3]).
ring_size_6(d67,[d67_20,d67_15,d67_16,d67_17,d67_18,d67_19]).
ring_size_5(d67,[d67_5,d67_15,d67_16,d67_13,d67_4]).
nitro(d67,[d67_12,d67_24,d67_26,d67_27]).
benzene(d68,[d68_6,d68_1,d68_2,d68_3,d68_4,d68_5]).
benzene(d68,[d68_4,d68_14,d68_13,d68_12,d68_11,d68_3]).
benzene(d68,[d68_19,d68_13,d68_12,d68_16,d68_17,d68_18]).
benzene(d68,[d68_24,d68_14,d68_13,d68_19,d68_22,d68_23]).
benzene(d68,[d68_29,d68_22,d68_19,d68_18,d68_27,d68_28]).
ring_size_6(d68,[d68_6,d68_1,d68_2,d68_3,d68_4,d68_5]).
ring_size_6(d68,[d68_4,d68_14,d68_13,d68_12,d68_11,d68_3]).
ring_size_6(d68,[d68_19,d68_13,d68_12,d68_16,d68_17,d68_18]).
ring_size_6(d68,[d68_24,d68_14,d68_13,d68_19,d68_22,d68_23]).
ring_size_6(d68,[d68_29,d68_22,d68_19,d68_18,d68_27,d68_28]).
nitro(d68,[d68_27,d68_32,d68_33,d68_34]).
anthracene(d68,[[d68_6,d68_1,d68_2,d68_3,d68_4,d68_5],[d68_4,d68_14,d68_13,d68_12,d68_11,d68_3],[d68_19,d68_13,d68_12,d68_16,d68_17,d68_18]]).
phenanthrene(d68,[[d68_6,d68_1,d68_2,d68_3,d68_4,d68_5],[d68_4,d68_14,d68_13,d68_12,d68_11,d68_3],[d68_24,d68_14,d68_13,d68_19,d68_22,d68_23]]).
ball3(d68,[[d68_4,d68_14,d68_13,d68_12,d68_11,d68_3],[d68_24,d68_14,d68_13,d68_19,d68_22,d68_23],[d68_19,d68_13,d68_12,d68_16,d68_17,d68_18]]).
ball3(d68,[[d68_29,d68_22,d68_19,d68_18,d68_27,d68_28],[d68_24,d68_14,d68_13,d68_19,d68_22,d68_23],[d68_19,d68_13,d68_12,d68_16,d68_17,d68_18]]).
benzene(d69,[d69_6,d69_1,d69_2,d69_3,d69_4,d69_5]).
benzene(d69,[d69_15,d69_10,d69_11,d69_12,d69_13,d69_14]).
ring_size_6(d69,[d69_6,d69_1,d69_2,d69_3,d69_4,d69_5]).
ring_size_6(d69,[d69_15,d69_10,d69_11,d69_12,d69_13,d69_14]).
ring_size_5(d69,[d69_4,d69_15,d69_10,d69_19,d69_3]).
nitro(d69,[d69_12,d69_22,d69_26,d69_27]).
nitro(d69,[d69_5,d69_23,d69_24,d69_25]).
benzene(d71,[d71_6,d71_1,d71_2,d71_3,d71_4,d71_5]).
benzene(d71,[d71_9,d71_10,d71_4,d71_3,d71_7,d71_8]).
benzene(d71,[d71_5,d71_13,d71_12,d71_11,d71_10,d71_4]).
benzene(d71,[d71_9,d71_10,d71_11,d71_16,d71_15,d71_14]).
carbon_6_ring(d71,[d71_20,d71_13,d71_12,d71_17,d71_18,d71_19]).
ring_size_6(d71,[d71_6,d71_1,d71_2,d71_3,d71_4,d71_5]).
ring_size_6(d71,[d71_9,d71_10,d71_4,d71_3,d71_7,d71_8]).
ring_size_6(d71,[d71_5,d71_13,d71_12,d71_11,d71_10,d71_4]).
ring_size_6(d71,[d71_9,d71_10,d71_11,d71_16,d71_15,d71_14]).
ring_size_6(d71,[d71_20,d71_13,d71_12,d71_17,d71_18,d71_19]).
nitro(d71,[d71_16,d71_21,d71_23,d71_24]).
nitro(d71,[d71_6,d71_22,d71_25,d71_26]).
ball3(d71,[[d71_9,d71_10,d71_4,d71_3,d71_7,d71_8],[d71_6,d71_1,d71_2,d71_3,d71_4,d71_5],[d71_5,d71_13,d71_12,d71_11,d71_10,d71_4]]).
ball3(d71,[[d71_9,d71_10,d71_4,d71_3,d71_7,d71_8],[d71_9,d71_10,d71_11,d71_16,d71_15,d71_14],[d71_5,d71_13,d71_12,d71_11,d71_10,d71_4]]).
benzene(d72,[d72_6,d72_1,d72_2,d72_3,d72_4,d72_5]).
benzene(d72,[d72_16,d72_11,d72_12,d72_13,d72_14,d72_15]).
ring_size_6(d72,[d72_6,d72_1,d72_2,d72_3,d72_4,d72_5]).
ring_size_6(d72,[d72_16,d72_11,d72_12,d72_13,d72_14,d72_15]).
ring_size_5(d72,[d72_4,d72_16,d72_11,d72_20,d72_3]).
nitro(d72,[d72_13,d72_23,d72_24,d72_25]).
benzene(d75,[d75_6,d75_1,d75_2,d75_3,d75_4,d75_5]).
benzene(d75,[d75_16,d75_11,d75_12,d75_13,d75_14,d75_15]).
ring_size_6(d75,[d75_6,d75_1,d75_2,d75_3,d75_4,d75_5]).
ring_size_6(d75,[d75_16,d75_11,d75_12,d75_13,d75_14,d75_15]).
nitro(d75,[d75_14,d75_21,d75_23,d75_24]).
nitro(d75,[d75_1,d75_22,d75_25,d75_26]).
benzene(d79,[d79_6,d79_1,d79_2,d79_3,d79_4,d79_5]).
benzene(d79,[d79_16,d79_11,d79_12,d79_13,d79_14,d79_15]).
ring_size_6(d79,[d79_6,d79_1,d79_2,d79_3,d79_4,d79_5]).
ring_size_6(d79,[d79_16,d79_11,d79_12,d79_13,d79_14,d79_15]).
ring_size_6(d79,[d79_4,d79_21,d79_16,d79_11,d79_20,d79_3]).
nitro(d79,[d79_14,d79_22,d79_23,d79_24]).
benzene(d80,[d80_6,d80_1,d80_2,d80_3,d80_4,d80_5]).
benzene(d80,[d80_4,d80_13,d80_12,d80_11,d80_10,d80_3]).
benzene(d80,[d80_22,d80_17,d80_18,d80_19,d80_20,d80_21]).
ring_size_6(d80,[d80_6,d80_1,d80_2,d80_3,d80_4,d80_5]).
ring_size_6(d80,[d80_4,d80_13,d80_12,d80_11,d80_10,d80_3]).
ring_size_6(d80,[d80_22,d80_17,d80_18,d80_19,d80_20,d80_21]).
ring_size_5(d80,[d80_5,d80_17,d80_18,d80_13,d80_4]).
nitro(d80,[d80_22,d80_26,d80_27,d80_28]).
benzene(d82,[d82_6,d82_1,d82_2,d82_3,d82_4,d82_5]).
benzene(d82,[d82_9,d82_10,d82_11,d82_12,d82_4,d82_3]).
benzene(d82,[d82_19,d82_14,d82_15,d82_16,d82_17,d82_18]).
ring_size_6(d82,[d82_6,d82_1,d82_2,d82_3,d82_4,d82_5]).
ring_size_6(d82,[d82_9,d82_10,d82_11,d82_12,d82_4,d82_3]).
ring_size_6(d82,[d82_19,d82_14,d82_15,d82_16,d82_17,d82_18]).
ring_size_5(d82,[d82_5,d82_14,d82_15,d82_12,d82_4]).
nitro(d82,[d82_11,d82_24,d82_27,d82_28]).
nitro(d82,[d82_10,d82_25,d82_29,d82_30]).
nitro(d82,[d82_2,d82_26,d82_31,d82_32]).
benzene(d83,[d83_6,d83_1,d83_2,d83_3,d83_4,d83_5]).
benzene(d83,[d83_9,d83_10,d83_11,d83_12,d83_7,d83_8]).
hetero_aromatic_5_ring(d83,[d83_7,d83_12,d83_4,d83_3,d83_13]).
ring_size_6(d83,[d83_6,d83_1,d83_2,d83_3,d83_4,d83_5]).
ring_size_6(d83,[d83_9,d83_10,d83_11,d83_12,d83_7,d83_8]).
ring_size_5(d83,[d83_7,d83_12,d83_4,d83_3,d83_13]).
nitro(d83,[d83_9,d83_14,d83_15,d83_16]).
benzene(d86,[d86_6,d86_1,d86_2,d86_3,d86_4,d86_5]).
benzene(d86,[d86_9,d86_10,d86_5,d86_4,d86_7,d86_8]).
benzene(d86,[d86_8,d86_14,d86_13,d86_12,d86_11,d86_7]).
benzene(d86,[d86_18,d86_13,d86_12,d86_15,d86_16,d86_17]).
carbon_5_aromatic_ring(d86,[d86_9,d86_19,d86_20,d86_14,d86_8]).
ring_size_6(d86,[d86_6,d86_1,d86_2,d86_3,d86_4,d86_5]).
ring_size_6(d86,[d86_9,d86_10,d86_5,d86_4,d86_7,d86_8]).
ring_size_6(d86,[d86_8,d86_14,d86_13,d86_12,d86_11,d86_7]).
ring_size_6(d86,[d86_18,d86_13,d86_12,d86_15,d86_16,d86_17]).
ring_size_5(d86,[d86_9,d86_19,d86_20,d86_14,d86_8]).
nitro(d86,[d86_10,d86_32,d86_33,d86_34]).
anthracene(d86,[[d86_9,d86_10,d86_5,d86_4,d86_7,d86_8],[d86_8,d86_14,d86_13,d86_12,d86_11,d86_7],[d86_18,d86_13,d86_12,d86_15,d86_16,d86_17]]).
benzene(d87,[d87_6,d87_1,d87_2,d87_3,d87_4,d87_5]).
benzene(d87,[d87_9,d87_10,d87_11,d87_12,d87_7,d87_8]).
hetero_aromatic_5_ring(d87,[d87_7,d87_12,d87_4,d87_3,d87_13]).
ring_size_6(d87,[d87_6,d87_1,d87_2,d87_3,d87_4,d87_5]).
ring_size_6(d87,[d87_9,d87_10,d87_11,d87_12,d87_7,d87_8]).
ring_size_5(d87,[d87_7,d87_12,d87_4,d87_3,d87_13]).
nitro(d87,[d87_9,d87_14,d87_15,d87_16]).
benzene(d90,[d90_6,d90_1,d90_2,d90_3,d90_4,d90_5]).
benzene(d90,[d90_4,d90_13,d90_12,d90_11,d90_10,d90_3]).
benzene(d90,[d90_21,d90_16,d90_13,d90_12,d90_19,d90_20]).
benzene(d90,[d90_5,d90_18,d90_17,d90_16,d90_13,d90_4]).
benzene(d90,[d90_27,d90_18,d90_17,d90_24,d90_25,d90_26]).
ring_size_6(d90,[d90_6,d90_1,d90_2,d90_3,d90_4,d90_5]).
ring_size_6(d90,[d90_4,d90_13,d90_12,d90_11,d90_10,d90_3]).
ring_size_6(d90,[d90_21,d90_16,d90_13,d90_12,d90_19,d90_20]).
ring_size_6(d90,[d90_5,d90_18,d90_17,d90_16,d90_13,d90_4]).
ring_size_6(d90,[d90_27,d90_18,d90_17,d90_24,d90_25,d90_26]).
nitro(d90,[d90_21,d90_32,d90_33,d90_34]).
phenanthrene(d90,[[d90_6,d90_1,d90_2,d90_3,d90_4,d90_5],[d90_4,d90_13,d90_12,d90_11,d90_10,d90_3],[d90_21,d90_16,d90_13,d90_12,d90_19,d90_20]]).
phenanthrene(d90,[[d90_6,d90_1,d90_2,d90_3,d90_4,d90_5],[d90_5,d90_18,d90_17,d90_16,d90_13,d90_4],[d90_21,d90_16,d90_13,d90_12,d90_19,d90_20]]).
phenanthrene(d90,[[d90_6,d90_1,d90_2,d90_3,d90_4,d90_5],[d90_5,d90_18,d90_17,d90_16,d90_13,d90_4],[d90_27,d90_18,d90_17,d90_24,d90_25,d90_26]]).
ball3(d90,[[d90_6,d90_1,d90_2,d90_3,d90_4,d90_5],[d90_5,d90_18,d90_17,d90_16,d90_13,d90_4],[d90_4,d90_13,d90_12,d90_11,d90_10,d90_3]]).
ball3(d90,[[d90_5,d90_18,d90_17,d90_16,d90_13,d90_4],[d90_4,d90_13,d90_12,d90_11,d90_10,d90_3],[d90_21,d90_16,d90_13,d90_12,d90_19,d90_20]]).
benzene(d94,[d94_6,d94_1,d94_2,d94_3,d94_4,d94_5]).
benzene(d94,[d94_4,d94_13,d94_12,d94_11,d94_10,d94_3]).
benzene(d94,[d94_22,d94_15,d94_13,d94_12,d94_20,d94_21]).
benzene(d94,[d94_5,d94_17,d94_16,d94_15,d94_13,d94_4]).
ring_size_6(d94,[d94_6,d94_1,d94_2,d94_3,d94_4,d94_5]).
ring_size_6(d94,[d94_4,d94_13,d94_12,d94_11,d94_10,d94_3]).
ring_size_6(d94,[d94_22,d94_15,d94_13,d94_12,d94_20,d94_21]).
ring_size_6(d94,[d94_5,d94_17,d94_16,d94_15,d94_13,d94_4]).
nitro(d94,[d94_11,d94_26,d94_27,d94_28]).
phenanthrene(d94,[[d94_6,d94_1,d94_2,d94_3,d94_4,d94_5],[d94_4,d94_13,d94_12,d94_11,d94_10,d94_3],[d94_22,d94_15,d94_13,d94_12,d94_20,d94_21]]).
phenanthrene(d94,[[d94_6,d94_1,d94_2,d94_3,d94_4,d94_5],[d94_5,d94_17,d94_16,d94_15,d94_13,d94_4],[d94_22,d94_15,d94_13,d94_12,d94_20,d94_21]]).
ball3(d94,[[d94_6,d94_1,d94_2,d94_3,d94_4,d94_5],[d94_5,d94_17,d94_16,d94_15,d94_13,d94_4],[d94_4,d94_13,d94_12,d94_11,d94_10,d94_3]]).
ball3(d94,[[d94_5,d94_17,d94_16,d94_15,d94_13,d94_4],[d94_4,d94_13,d94_12,d94_11,d94_10,d94_3],[d94_22,d94_15,d94_13,d94_12,d94_20,d94_21]]).
benzene(d95,[d95_6,d95_1,d95_2,d95_3,d95_4,d95_5]).
benzene(d95,[d95_4,d95_13,d95_12,d95_11,d95_10,d95_3]).
ring_size_6(d95,[d95_6,d95_1,d95_2,d95_3,d95_4,d95_5]).
ring_size_6(d95,[d95_4,d95_13,d95_12,d95_11,d95_10,d95_3]).
nitro(d95,[d95_12,d95_16,d95_19,d95_20]).
nitro(d95,[d95_11,d95_17,d95_21,d95_22]).
nitro(d95,[d95_2,d95_18,d95_23,d95_24]).
benzene(d96,[d96_6,d96_1,d96_2,d96_3,d96_4,d96_5]).
benzene(d96,[d96_9,d96_10,d96_11,d96_12,d96_4,d96_3]).
benzene(d96,[d96_22,d96_15,d96_12,d96_11,d96_20,d96_21]).
benzene(d96,[d96_5,d96_17,d96_16,d96_15,d96_12,d96_4]).
ring_size_6(d96,[d96_6,d96_1,d96_2,d96_3,d96_4,d96_5]).
ring_size_6(d96,[d96_9,d96_10,d96_11,d96_12,d96_4,d96_3]).
ring_size_6(d96,[d96_22,d96_15,d96_12,d96_11,d96_20,d96_21]).
ring_size_6(d96,[d96_5,d96_17,d96_16,d96_15,d96_12,d96_4]).
nitro(d96,[d96_22,d96_24,d96_27,d96_28]).
nitro(d96,[d96_20,d96_25,d96_29,d96_30]).
nitro(d96,[d96_2,d96_26,d96_31,d96_32]).
phenanthrene(d96,[[d96_6,d96_1,d96_2,d96_3,d96_4,d96_5],[d96_5,d96_17,d96_16,d96_15,d96_12,d96_4],[d96_22,d96_15,d96_12,d96_11,d96_20,d96_21]]).
ball3(d96,[[d96_9,d96_10,d96_11,d96_12,d96_4,d96_3],[d96_6,d96_1,d96_2,d96_3,d96_4,d96_5],[d96_5,d96_17,d96_16,d96_15,d96_12,d96_4]]).
ball3(d96,[[d96_9,d96_10,d96_11,d96_12,d96_4,d96_3],[d96_5,d96_17,d96_16,d96_15,d96_12,d96_4],[d96_22,d96_15,d96_12,d96_11,d96_20,d96_21]]).
benzene(d101,[d101_6,d101_1,d101_2,d101_3,d101_4,d101_5]).
benzene(d101,[d101_9,d101_10,d101_4,d101_3,d101_7,d101_8]).
benzene(d101,[d101_5,d101_13,d101_12,d101_11,d101_10,d101_4]).
benzene(d101,[d101_9,d101_10,d101_11,d101_16,d101_15,d101_14]).
carbon_6_ring(d101,[d101_20,d101_13,d101_12,d101_17,d101_18,d101_19]).
ring_size_6(d101,[d101_6,d101_1,d101_2,d101_3,d101_4,d101_5]).
ring_size_6(d101,[d101_9,d101_10,d101_4,d101_3,d101_7,d101_8]).
ring_size_6(d101,[d101_5,d101_13,d101_12,d101_11,d101_10,d101_4]).
ring_size_6(d101,[d101_9,d101_10,d101_11,d101_16,d101_15,d101_14]).
ring_size_6(d101,[d101_20,d101_13,d101_12,d101_17,d101_18,d101_19]).
nitro(d101,[d101_14,d101_21,d101_22,d101_23]).
ball3(d101,[[d101_9,d101_10,d101_4,d101_3,d101_7,d101_8],[d101_6,d101_1,d101_2,d101_3,d101_4,d101_5],[d101_5,d101_13,d101_12,d101_11,d101_10,d101_4]]).
ball3(d101,[[d101_9,d101_10,d101_4,d101_3,d101_7,d101_8],[d101_9,d101_10,d101_11,d101_16,d101_15,d101_14],[d101_5,d101_13,d101_12,d101_11,d101_10,d101_4]]).
benzene(d104,[d104_6,d104_1,d104_2,d104_3,d104_4,d104_5]).
benzene(d104,[d104_9,d104_10,d104_4,d104_3,d104_7,d104_8]).
benzene(d104,[d104_5,d104_13,d104_12,d104_11,d104_10,d104_4]).
benzene(d104,[d104_9,d104_10,d104_11,d104_16,d104_15,d104_14]).
carbon_6_ring(d104,[d104_20,d104_13,d104_12,d104_17,d104_18,d104_19]).
ring_size_6(d104,[d104_6,d104_1,d104_2,d104_3,d104_4,d104_5]).
ring_size_6(d104,[d104_9,d104_10,d104_4,d104_3,d104_7,d104_8]).
ring_size_6(d104,[d104_5,d104_13,d104_12,d104_11,d104_10,d104_4]).
ring_size_6(d104,[d104_9,d104_10,d104_11,d104_16,d104_15,d104_14]).
ring_size_6(d104,[d104_20,d104_13,d104_12,d104_17,d104_18,d104_19]).
nitro(d104,[d104_16,d104_21,d104_22,d104_23]).
ball3(d104,[[d104_9,d104_10,d104_4,d104_3,d104_7,d104_8],[d104_6,d104_1,d104_2,d104_3,d104_4,d104_5],[d104_5,d104_13,d104_12,d104_11,d104_10,d104_4]]).
ball3(d104,[[d104_9,d104_10,d104_4,d104_3,d104_7,d104_8],[d104_9,d104_10,d104_11,d104_16,d104_15,d104_14],[d104_5,d104_13,d104_12,d104_11,d104_10,d104_4]]).
benzene(d105,[d105_6,d105_1,d105_2,d105_3,d105_4,d105_5]).
hetero_aromatic_6_ring(d105,[d105_5,d105_13,d105_12,d105_11,d105_10,d105_4]).
hetero_aromatic_6_ring(d105,[d105_18,d105_11,d105_10,d105_15,d105_16,d105_17]).
ring_size_6(d105,[d105_6,d105_1,d105_2,d105_3,d105_4,d105_5]).
ring_size_6(d105,[d105_5,d105_13,d105_12,d105_11,d105_10,d105_4]).
ring_size_6(d105,[d105_18,d105_11,d105_10,d105_15,d105_16,d105_17]).
nitro(d105,[d105_3,d105_22,d105_23,d105_24]).
benzene(d107,[d107_6,d107_1,d107_2,d107_3,d107_4,d107_5]).
benzene(d107,[d107_9,d107_10,d107_11,d107_12,d107_4,d107_3]).
benzene(d107,[d107_19,d107_14,d107_15,d107_16,d107_17,d107_18]).
ring_size_6(d107,[d107_6,d107_1,d107_2,d107_3,d107_4,d107_5]).
ring_size_6(d107,[d107_9,d107_10,d107_11,d107_12,d107_4,d107_3]).
ring_size_6(d107,[d107_19,d107_14,d107_15,d107_16,d107_17,d107_18]).
ring_size_5(d107,[d107_5,d107_14,d107_15,d107_12,d107_4]).
nitro(d107,[d107_10,d107_24,d107_31,d107_32]).
nitro(d107,[d107_9,d107_25,d107_29,d107_30]).
nitro(d107,[d107_1,d107_26,d107_27,d107_28]).
benzene(d109,[d109_6,d109_1,d109_2,d109_3,d109_4,d109_5]).
benzene(d109,[d109_9,d109_10,d109_15,d109_19,d109_18,d109_14]).
benzene(d109,[d109_6,d109_17,d109_16,d109_15,d109_10,d109_5]).
benzene(d109,[d109_9,d109_14,d109_13,d109_12,d109_11,d109_8]).
benzene(d109,[d109_8,d109_11,d109_22,d109_21,d109_20,d109_7]).
carbon_6_ring(d109,[d109_9,d109_10,d109_5,d109_4,d109_7,d109_8]).
ring_size_6(d109,[d109_6,d109_1,d109_2,d109_3,d109_4,d109_5]).
ring_size_6(d109,[d109_9,d109_10,d109_5,d109_4,d109_7,d109_8]).
ring_size_6(d109,[d109_9,d109_10,d109_15,d109_19,d109_18,d109_14]).
ring_size_6(d109,[d109_6,d109_17,d109_16,d109_15,d109_10,d109_5]).
ring_size_6(d109,[d109_9,d109_14,d109_13,d109_12,d109_11,d109_8]).
ring_size_6(d109,[d109_8,d109_11,d109_22,d109_21,d109_20,d109_7]).
nitro(d109,[d109_22,d109_23,d109_25,d109_26]).
nitro(d109,[d109_1,d109_24,d109_27,d109_28]).
phenanthrene(d109,[[d109_9,d109_10,d109_15,d109_19,d109_18,d109_14],[d109_6,d109_17,d109_16,d109_15,d109_10,d109_5],[d109_6,d109_1,d109_2,d109_3,d109_4,d109_5]]).
phenanthrene(d109,[[d109_9,d109_14,d109_13,d109_12,d109_11,d109_8],[d109_9,d109_10,d109_15,d109_19,d109_18,d109_14],[d109_6,d109_17,d109_16,d109_15,d109_10,d109_5]]).
benzene(d112,[d112_6,d112_1,d112_2,d112_3,d112_4,d112_5]).
benzene(d112,[d112_4,d112_13,d112_12,d112_11,d112_10,d112_3]).
ring_size_6(d112,[d112_6,d112_1,d112_2,d112_3,d112_4,d112_5]).
ring_size_6(d112,[d112_4,d112_13,d112_12,d112_11,d112_10,d112_3]).
ring_size_5(d112,[d112_5,d112_16,d112_17,d112_13,d112_4]).
nitro(d112,[d112_12,d112_22,d112_23,d112_24]).
benzene(d115,[d115_6,d115_1,d115_2,d115_3,d115_4,d115_5]).
ring_size_6(d115,[d115_6,d115_1,d115_2,d115_3,d115_4,d115_5]).
ring_size_5(d115,[d115_4,d115_14,d115_11,d115_10,d115_3]).
nitro(d115,[d115_6,d115_17,d115_18,d115_19]).
benzene(d117,[d117_6,d117_1,d117_2,d117_3,d117_4,d117_5]).
benzene(d117,[d117_16,d117_11,d117_12,d117_13,d117_14,d117_15]).
ring_size_6(d117,[d117_6,d117_1,d117_2,d117_3,d117_4,d117_5]).
ring_size_6(d117,[d117_16,d117_11,d117_12,d117_13,d117_14,d117_15]).
nitro(d117,[d117_16,d117_20,d117_25,d117_26]).
nitro(d117,[d117_14,d117_21,d117_27,d117_28]).
nitro(d117,[d117_6,d117_22,d117_23,d117_24]).
benzene(d122,[d122_6,d122_1,d122_2,d122_3,d122_4,d122_5]).
benzene(d122,[d122_4,d122_14,d122_13,d122_12,d122_11,d122_3]).
benzene(d122,[d122_19,d122_14,d122_13,d122_16,d122_17,d122_18]).
benzene(d122,[d122_25,d122_17,d122_16,d122_22,d122_23,d122_24]).
ring_size_6(d122,[d122_6,d122_1,d122_2,d122_3,d122_4,d122_5]).
ring_size_6(d122,[d122_4,d122_14,d122_13,d122_12,d122_11,d122_3]).
ring_size_6(d122,[d122_19,d122_14,d122_13,d122_16,d122_17,d122_18]).
ring_size_6(d122,[d122_25,d122_17,d122_16,d122_22,d122_23,d122_24]).
nitro(d122,[d122_11,d122_30,d122_31,d122_32]).
phenanthrene(d122,[[d122_6,d122_1,d122_2,d122_3,d122_4,d122_5],[d122_4,d122_14,d122_13,d122_12,d122_11,d122_3],[d122_19,d122_14,d122_13,d122_16,d122_17,d122_18]]).
benzene(d125,[d125_6,d125_1,d125_2,d125_3,d125_4,d125_5]).
benzene(d125,[d125_5,d125_13,d125_12,d125_11,d125_10,d125_4]).
benzene(d125,[d125_9,d125_10,d125_11,d125_16,d125_15,d125_14]).
carbon_6_ring(d125,[d125_9,d125_10,d125_4,d125_3,d125_7,d125_8]).
ring_size_6(d125,[d125_6,d125_1,d125_2,d125_3,d125_4,d125_5]).
ring_size_6(d125,[d125_9,d125_10,d125_4,d125_3,d125_7,d125_8]).
ring_size_6(d125,[d125_5,d125_13,d125_12,d125_11,d125_10,d125_4]).
ring_size_6(d125,[d125_9,d125_10,d125_11,d125_16,d125_15,d125_14]).
nitro(d125,[d125_15,d125_17,d125_29,d125_30]).
benzene(d126,[d126_6,d126_1,d126_2,d126_3,d126_4,d126_5]).
benzene(d126,[d126_17,d126_12,d126_11,d126_14,d126_15,d126_16]).
ring_size_6(d126,[d126_6,d126_1,d126_2,d126_3,d126_4,d126_5]).
ring_size_6(d126,[d126_4,d126_13,d126_12,d126_11,d126_10,d126_3]).
ring_size_6(d126,[d126_17,d126_12,d126_11,d126_14,d126_15,d126_16]).
nitro(d126,[d126_17,d126_23,d126_25,d126_26]).
nitro(d126,[d126_1,d126_24,d126_27,d126_28]).
benzene(d127,[d127_6,d127_1,d127_2,d127_3,d127_4,d127_5]).
benzene(d127,[d127_16,d127_11,d127_12,d127_13,d127_14,d127_15]).
ring_size_6(d127,[d127_6,d127_1,d127_2,d127_3,d127_4,d127_5]).
ring_size_6(d127,[d127_16,d127_11,d127_12,d127_13,d127_14,d127_15]).
nitro(d127,[d127_15,d127_20,d127_23,d127_24]).
nitro(d127,[d127_6,d127_21,d127_27,d127_28]).
nitro(d127,[d127_14,d127_22,d127_25,d127_26]).
benzene(d128,[d128_6,d128_1,d128_2,d128_3,d128_4,d128_5]).
benzene(d128,[d128_9,d128_10,d128_4,d128_3,d128_7,d128_8]).
carbon_5_aromatic_ring(d128,[d128_5,d128_11,d128_12,d128_10,d128_4]).
ring_size_6(d128,[d128_6,d128_1,d128_2,d128_3,d128_4,d128_5]).
ring_size_6(d128,[d128_9,d128_10,d128_4,d128_3,d128_7,d128_8]).
ring_size_5(d128,[d128_5,d128_11,d128_12,d128_10,d128_4]).
nitro(d128,[d128_7,d128_20,d128_21,d128_22]).
benzene(d136,[d136_6,d136_1,d136_2,d136_3,d136_4,d136_5]).
benzene(d136,[d136_4,d136_13,d136_12,d136_11,d136_10,d136_3]).
benzene(d136,[d136_23,d136_16,d136_13,d136_12,d136_21,d136_22]).
benzene(d136,[d136_5,d136_18,d136_17,d136_16,d136_13,d136_4]).
ring_size_6(d136,[d136_6,d136_1,d136_2,d136_3,d136_4,d136_5]).
ring_size_6(d136,[d136_4,d136_13,d136_12,d136_11,d136_10,d136_3]).
ring_size_6(d136,[d136_23,d136_16,d136_13,d136_12,d136_21,d136_22]).
ring_size_6(d136,[d136_5,d136_18,d136_17,d136_16,d136_13,d136_4]).
nitro(d136,[d136_21,d136_26,d136_27,d136_28]).
phenanthrene(d136,[[d136_6,d136_1,d136_2,d136_3,d136_4,d136_5],[d136_4,d136_13,d136_12,d136_11,d136_10,d136_3],[d136_23,d136_16,d136_13,d136_12,d136_21,d136_22]]).
phenanthrene(d136,[[d136_6,d136_1,d136_2,d136_3,d136_4,d136_5],[d136_5,d136_18,d136_17,d136_16,d136_13,d136_4],[d136_23,d136_16,d136_13,d136_12,d136_21,d136_22]]).
ball3(d136,[[d136_6,d136_1,d136_2,d136_3,d136_4,d136_5],[d136_5,d136_18,d136_17,d136_16,d136_13,d136_4],[d136_4,d136_13,d136_12,d136_11,d136_10,d136_3]]).
ball3(d136,[[d136_5,d136_18,d136_17,d136_16,d136_13,d136_4],[d136_4,d136_13,d136_12,d136_11,d136_10,d136_3],[d136_23,d136_16,d136_13,d136_12,d136_21,d136_22]]).
benzene(d137,[d137_6,d137_1,d137_2,d137_3,d137_4,d137_5]).
benzene(d137,[d137_4,d137_13,d137_12,d137_11,d137_10,d137_3]).
benzene(d137,[d137_20,d137_15,d137_16,d137_17,d137_18,d137_19]).
ring_size_6(d137,[d137_6,d137_1,d137_2,d137_3,d137_4,d137_5]).
ring_size_6(d137,[d137_4,d137_13,d137_12,d137_11,d137_10,d137_3]).
ring_size_6(d137,[d137_20,d137_15,d137_16,d137_17,d137_18,d137_19]).
ring_size_5(d137,[d137_5,d137_15,d137_16,d137_13,d137_4]).
nitro(d137,[d137_11,d137_25,d137_27,d137_28]).
nitro(d137,[d137_10,d137_26,d137_29,d137_30]).
benzene(d140,[d140_6,d140_1,d140_2,d140_3,d140_4,d140_5]).
benzene(d140,[d140_4,d140_13,d140_12,d140_11,d140_10,d140_3]).
ring_size_6(d140,[d140_6,d140_1,d140_2,d140_3,d140_4,d140_5]).
ring_size_6(d140,[d140_4,d140_13,d140_12,d140_11,d140_10,d140_3]).
nitro(d140,[d140_13,d140_17,d140_21,d140_22]).
nitro(d140,[d140_5,d140_18,d140_19,d140_20]).
benzene(d146,[d146_6,d146_1,d146_2,d146_3,d146_4,d146_5]).
benzene(d146,[d146_16,d146_11,d146_12,d146_13,d146_14,d146_15]).
ring_size_6(d146,[d146_6,d146_1,d146_2,d146_3,d146_4,d146_5]).
ring_size_6(d146,[d146_16,d146_11,d146_12,d146_13,d146_14,d146_15]).
nitro(d146,[d146_1,d146_20,d146_21,d146_22]).
benzene(d148,[d148_6,d148_1,d148_2,d148_3,d148_4,d148_5]).
benzene(d148,[d148_9,d148_10,d148_4,d148_3,d148_7,d148_8]).
carbon_5_aromatic_ring(d148,[d148_5,d148_11,d148_12,d148_10,d148_4]).
ring_size_6(d148,[d148_6,d148_1,d148_2,d148_3,d148_4,d148_5]).
ring_size_6(d148,[d148_9,d148_10,d148_4,d148_3,d148_7,d148_8]).
ring_size_5(d148,[d148_5,d148_11,d148_12,d148_10,d148_4]).
nitro(d148,[d148_9,d148_20,d148_21,d148_22]).
benzene(d149,[d149_6,d149_1,d149_2,d149_3,d149_4,d149_5]).
benzene(d149,[d149_17,d149_12,d149_11,d149_14,d149_15,d149_16]).
ring_size_6(d149,[d149_6,d149_1,d149_2,d149_3,d149_4,d149_5]).
ring_size_6(d149,[d149_4,d149_13,d149_12,d149_11,d149_10,d149_3]).
ring_size_6(d149,[d149_17,d149_12,d149_11,d149_14,d149_15,d149_16]).
nitro(d149,[d149_16,d149_23,d149_25,d149_26]).
nitro(d149,[d149_1,d149_24,d149_27,d149_28]).
benzene(d151,[d151_6,d151_1,d151_2,d151_3,d151_4,d151_5]).
hetero_aromatic_5_ring(d151,[d151_9,d151_4,d151_3,d151_7,d151_8]).
ring_size_6(d151,[d151_6,d151_1,d151_2,d151_3,d151_4,d151_5]).
ring_size_5(d151,[d151_9,d151_4,d151_3,d151_7,d151_8]).
nitro(d151,[d151_2,d151_10,d151_11,d151_12]).
benzene(d152,[d152_6,d152_1,d152_2,d152_3,d152_4,d152_5]).
benzene(d152,[d152_9,d152_10,d152_11,d152_12,d152_4,d152_3]).
benzene(d152,[d152_20,d152_15,d152_16,d152_17,d152_18,d152_19]).
ring_size_6(d152,[d152_6,d152_1,d152_2,d152_3,d152_4,d152_5]).
ring_size_6(d152,[d152_9,d152_10,d152_11,d152_12,d152_4,d152_3]).
ring_size_6(d152,[d152_20,d152_15,d152_16,d152_17,d152_18,d152_19]).
ring_size_5(d152,[d152_5,d152_15,d152_16,d152_12,d152_4]).
nitro(d152,[d152_10,d152_25,d152_29,d152_30]).
nitro(d152,[d152_1,d152_26,d152_27,d152_28]).
benzene(d153,[d153_6,d153_1,d153_2,d153_3,d153_4,d153_5]).
benzene(d153,[d153_4,d153_13,d153_12,d153_11,d153_10,d153_3]).
ring_size_6(d153,[d153_6,d153_1,d153_2,d153_3,d153_4,d153_5]).
ring_size_6(d153,[d153_4,d153_13,d153_12,d153_11,d153_10,d153_3]).
ring_size_5(d153,[d153_5,d153_16,d153_17,d153_13,d153_4]).
nitro(d153,[d153_10,d153_22,d153_23,d153_24]).
benzene(d157,[d157_9,d157_10,d157_4,d157_3,d157_7,d157_8]).
benzene(d157,[d157_5,d157_13,d157_12,d157_11,d157_10,d157_4]).
benzene(d157,[d157_9,d157_10,d157_11,d157_16,d157_15,d157_14]).
carbon_6_ring(d157,[d157_6,d157_1,d157_2,d157_3,d157_4,d157_5]).
ring_size_6(d157,[d157_6,d157_1,d157_2,d157_3,d157_4,d157_5]).
ring_size_6(d157,[d157_9,d157_10,d157_4,d157_3,d157_7,d157_8]).
ring_size_6(d157,[d157_5,d157_13,d157_12,d157_11,d157_10,d157_4]).
ring_size_6(d157,[d157_9,d157_10,d157_11,d157_16,d157_15,d157_14]).
nitro(d157,[d157_16,d157_27,d157_28,d157_29]).
ball3(d157,[[d157_9,d157_10,d157_4,d157_3,d157_7,d157_8],[d157_9,d157_10,d157_11,d157_16,d157_15,d157_14],[d157_5,d157_13,d157_12,d157_11,d157_10,d157_4]]).
benzene(d159,[d159_9,d159_10,d159_4,d159_3,d159_7,d159_8]).
benzene(d159,[d159_5,d159_13,d159_12,d159_11,d159_10,d159_4]).
benzene(d159,[d159_9,d159_10,d159_11,d159_16,d159_15,d159_14]).
carbon_6_ring(d159,[d159_6,d159_1,d159_2,d159_3,d159_4,d159_5]).
ring_size_6(d159,[d159_6,d159_1,d159_2,d159_3,d159_4,d159_5]).
ring_size_6(d159,[d159_9,d159_10,d159_4,d159_3,d159_7,d159_8]).
ring_size_6(d159,[d159_5,d159_13,d159_12,d159_11,d159_10,d159_4]).
ring_size_6(d159,[d159_9,d159_10,d159_11,d159_16,d159_15,d159_14]).
nitro(d159,[d159_16,d159_27,d159_28,d159_29]).
ball3(d159,[[d159_9,d159_10,d159_4,d159_3,d159_7,d159_8],[d159_9,d159_10,d159_11,d159_16,d159_15,d159_14],[d159_5,d159_13,d159_12,d159_11,d159_10,d159_4]]).
benzene(d163,[d163_6,d163_1,d163_2,d163_3,d163_4,d163_5]).
benzene(d163,[d163_9,d163_10,d163_11,d163_12,d163_4,d163_3]).
benzene(d163,[d163_22,d163_15,d163_12,d163_11,d163_20,d163_21]).
benzene(d163,[d163_5,d163_17,d163_16,d163_15,d163_12,d163_4]).
ring_size_6(d163,[d163_6,d163_1,d163_2,d163_3,d163_4,d163_5]).
ring_size_6(d163,[d163_9,d163_10,d163_11,d163_12,d163_4,d163_3]).
ring_size_6(d163,[d163_22,d163_15,d163_12,d163_11,d163_20,d163_21]).
ring_size_6(d163,[d163_5,d163_17,d163_16,d163_15,d163_12,d163_4]).
nitro(d163,[d163_21,d163_25,d163_27,d163_28]).
nitro(d163,[d163_1,d163_26,d163_29,d163_30]).
phenanthrene(d163,[[d163_6,d163_1,d163_2,d163_3,d163_4,d163_5],[d163_5,d163_17,d163_16,d163_15,d163_12,d163_4],[d163_22,d163_15,d163_12,d163_11,d163_20,d163_21]]).
ball3(d163,[[d163_9,d163_10,d163_11,d163_12,d163_4,d163_3],[d163_6,d163_1,d163_2,d163_3,d163_4,d163_5],[d163_5,d163_17,d163_16,d163_15,d163_12,d163_4]]).
ball3(d163,[[d163_9,d163_10,d163_11,d163_12,d163_4,d163_3],[d163_5,d163_17,d163_16,d163_15,d163_12,d163_4],[d163_22,d163_15,d163_12,d163_11,d163_20,d163_21]]).
benzene(d164,[d164_6,d164_1,d164_2,d164_3,d164_4,d164_5]).
benzene(d164,[d164_9,d164_10,d164_11,d164_12,d164_4,d164_3]).
benzene(d164,[d164_22,d164_15,d164_12,d164_11,d164_20,d164_21]).
benzene(d164,[d164_5,d164_17,d164_16,d164_15,d164_12,d164_4]).
ring_size_6(d164,[d164_6,d164_1,d164_2,d164_3,d164_4,d164_5]).
ring_size_6(d164,[d164_9,d164_10,d164_11,d164_12,d164_4,d164_3]).
ring_size_6(d164,[d164_22,d164_15,d164_12,d164_11,d164_20,d164_21]).
ring_size_6(d164,[d164_5,d164_17,d164_16,d164_15,d164_12,d164_4]).
nitro(d164,[d164_22,d164_25,d164_27,d164_28]).
nitro(d164,[d164_20,d164_26,d164_29,d164_30]).
phenanthrene(d164,[[d164_6,d164_1,d164_2,d164_3,d164_4,d164_5],[d164_5,d164_17,d164_16,d164_15,d164_12,d164_4],[d164_22,d164_15,d164_12,d164_11,d164_20,d164_21]]).
ball3(d164,[[d164_9,d164_10,d164_11,d164_12,d164_4,d164_3],[d164_6,d164_1,d164_2,d164_3,d164_4,d164_5],[d164_5,d164_17,d164_16,d164_15,d164_12,d164_4]]).
ball3(d164,[[d164_9,d164_10,d164_11,d164_12,d164_4,d164_3],[d164_5,d164_17,d164_16,d164_15,d164_12,d164_4],[d164_22,d164_15,d164_12,d164_11,d164_20,d164_21]]).
benzene(d165,[d165_6,d165_1,d165_2,d165_3,d165_4,d165_5]).
benzene(d165,[d165_4,d165_13,d165_12,d165_11,d165_10,d165_3]).
benzene(d165,[d165_22,d165_17,d165_18,d165_19,d165_20,d165_21]).
ring_size_6(d165,[d165_6,d165_1,d165_2,d165_3,d165_4,d165_5]).
ring_size_6(d165,[d165_4,d165_13,d165_12,d165_11,d165_10,d165_3]).
ring_size_6(d165,[d165_22,d165_17,d165_18,d165_19,d165_20,d165_21]).
ring_size_5(d165,[d165_5,d165_17,d165_18,d165_13,d165_4]).
nitro(d165,[d165_21,d165_26,d165_27,d165_28]).
benzene(d166,[d166_6,d166_1,d166_2,d166_3,d166_4,d166_5]).
benzene(d166,[d166_4,d166_13,d166_12,d166_11,d166_10,d166_3]).
benzene(d166,[d166_23,d166_16,d166_13,d166_12,d166_21,d166_22]).
benzene(d166,[d166_5,d166_18,d166_17,d166_16,d166_13,d166_4]).
ring_size_6(d166,[d166_6,d166_1,d166_2,d166_3,d166_4,d166_5]).
ring_size_6(d166,[d166_4,d166_13,d166_12,d166_11,d166_10,d166_3]).
ring_size_6(d166,[d166_23,d166_16,d166_13,d166_12,d166_21,d166_22]).
ring_size_6(d166,[d166_5,d166_18,d166_17,d166_16,d166_13,d166_4]).
nitro(d166,[d166_23,d166_25,d166_26,d166_27]).
phenanthrene(d166,[[d166_6,d166_1,d166_2,d166_3,d166_4,d166_5],[d166_4,d166_13,d166_12,d166_11,d166_10,d166_3],[d166_23,d166_16,d166_13,d166_12,d166_21,d166_22]]).
phenanthrene(d166,[[d166_6,d166_1,d166_2,d166_3,d166_4,d166_5],[d166_5,d166_18,d166_17,d166_16,d166_13,d166_4],[d166_23,d166_16,d166_13,d166_12,d166_21,d166_22]]).
ball3(d166,[[d166_6,d166_1,d166_2,d166_3,d166_4,d166_5],[d166_5,d166_18,d166_17,d166_16,d166_13,d166_4],[d166_4,d166_13,d166_12,d166_11,d166_10,d166_3]]).
ball3(d166,[[d166_5,d166_18,d166_17,d166_16,d166_13,d166_4],[d166_4,d166_13,d166_12,d166_11,d166_10,d166_3],[d166_23,d166_16,d166_13,d166_12,d166_21,d166_22]]).
benzene(d169,[d169_6,d169_1,d169_2,d169_3,d169_4,d169_5]).
ring_size_6(d169,[d169_6,d169_1,d169_2,d169_3,d169_4,d169_5]).
nitro(d169,[d169_4,d169_11,d169_13,d169_14]).
nitro(d169,[d169_2,d169_12,d169_15,d169_16]).
benzene(d171,[d171_6,d171_1,d171_2,d171_3,d171_4,d171_5]).
benzene(d171,[d171_15,d171_10,d171_11,d171_12,d171_13,d171_14]).
ring_size_6(d171,[d171_6,d171_1,d171_2,d171_3,d171_4,d171_5]).
ring_size_6(d171,[d171_15,d171_10,d171_11,d171_12,d171_13,d171_14]).
nitro(d171,[d171_15,d171_19,d171_27,d171_28]).
nitro(d171,[d171_13,d171_20,d171_29,d171_30]).
nitro(d171,[d171_5,d171_21,d171_25,d171_26]).
nitro(d171,[d171_1,d171_22,d171_23,d171_24]).
benzene(d172,[d172_6,d172_1,d172_2,d172_3,d172_4,d172_5]).
hetero_aromatic_5_ring(d172,[d172_9,d172_4,d172_3,d172_7,d172_8]).
ring_size_6(d172,[d172_6,d172_1,d172_2,d172_3,d172_4,d172_5]).
ring_size_5(d172,[d172_9,d172_4,d172_3,d172_7,d172_8]).
nitro(d172,[d172_1,d172_10,d172_11,d172_12]).
benzene(d173,[d173_6,d173_1,d173_2,d173_3,d173_4,d173_5]).
benzene(d173,[d173_15,d173_10,d173_11,d173_12,d173_13,d173_14]).
ring_size_6(d173,[d173_6,d173_1,d173_2,d173_3,d173_4,d173_5]).
ring_size_6(d173,[d173_15,d173_10,d173_11,d173_12,d173_13,d173_14]).
nitro(d173,[d173_5,d173_20,d173_25,d173_26]).
nitro(d173,[d173_4,d173_21,d173_27,d173_28]).
nitro(d173,[d173_10,d173_22,d173_23,d173_24]).
benzene(d174,[d174_6,d174_1,d174_2,d174_3,d174_4,d174_5]).
benzene(d174,[d174_4,d174_13,d174_12,d174_11,d174_10,d174_3]).
benzene(d174,[d174_21,d174_16,d174_17,d174_18,d174_19,d174_20]).
ring_size_6(d174,[d174_6,d174_1,d174_2,d174_3,d174_4,d174_5]).
ring_size_6(d174,[d174_4,d174_13,d174_12,d174_11,d174_10,d174_3]).
ring_size_6(d174,[d174_21,d174_16,d174_17,d174_18,d174_19,d174_20]).
ring_size_5(d174,[d174_5,d174_16,d174_17,d174_13,d174_4]).
nitro(d174,[d174_10,d174_25,d174_27,d174_28]).
nitro(d174,[d174_19,d174_26,d174_29,d174_30]).
benzene(d177,[d177_6,d177_1,d177_2,d177_3,d177_4,d177_5]).
benzene(d177,[d177_17,d177_12,d177_11,d177_14,d177_15,d177_16]).
ring_size_6(d177,[d177_6,d177_1,d177_2,d177_3,d177_4,d177_5]).
ring_size_6(d177,[d177_4,d177_13,d177_12,d177_11,d177_10,d177_3]).
ring_size_6(d177,[d177_17,d177_12,d177_11,d177_14,d177_15,d177_16]).
nitro(d177,[d177_17,d177_23,d177_27,d177_28]).
nitro(d177,[d177_5,d177_24,d177_25,d177_26]).
benzene(d178,[d178_6,d178_1,d178_2,d178_3,d178_4,d178_5]).
ring_size_6(d178,[d178_6,d178_1,d178_2,d178_3,d178_4,d178_5]).
ring_size_5(d178,[d178_4,d178_12,d178_11,d178_10,d178_3]).
nitro(d178,[d178_2,d178_19,d178_20,d178_21]).
benzene(d183,[d183_6,d183_1,d183_2,d183_3,d183_4,d183_5]).
benzene(d183,[d183_9,d183_10,d183_11,d183_12,d183_4,d183_3]).
benzene(d183,[d183_22,d183_15,d183_12,d183_11,d183_20,d183_21]).
benzene(d183,[d183_5,d183_17,d183_16,d183_15,d183_12,d183_4]).
ring_size_6(d183,[d183_6,d183_1,d183_2,d183_3,d183_4,d183_5]).
ring_size_6(d183,[d183_9,d183_10,d183_11,d183_12,d183_4,d183_3]).
ring_size_6(d183,[d183_22,d183_15,d183_12,d183_11,d183_20,d183_21]).
ring_size_6(d183,[d183_5,d183_17,d183_16,d183_15,d183_12,d183_4]).
nitro(d183,[d183_22,d183_25,d183_29,d183_30]).
nitro(d183,[d183_6,d183_26,d183_27,d183_28]).
phenanthrene(d183,[[d183_6,d183_1,d183_2,d183_3,d183_4,d183_5],[d183_5,d183_17,d183_16,d183_15,d183_12,d183_4],[d183_22,d183_15,d183_12,d183_11,d183_20,d183_21]]).
ball3(d183,[[d183_9,d183_10,d183_11,d183_12,d183_4,d183_3],[d183_6,d183_1,d183_2,d183_3,d183_4,d183_5],[d183_5,d183_17,d183_16,d183_15,d183_12,d183_4]]).
ball3(d183,[[d183_9,d183_10,d183_11,d183_12,d183_4,d183_3],[d183_5,d183_17,d183_16,d183_15,d183_12,d183_4],[d183_22,d183_15,d183_12,d183_11,d183_20,d183_21]]).
benzene(d184,[d184_6,d184_1,d184_2,d184_3,d184_4,d184_5]).
benzene(d184,[d184_4,d184_13,d184_12,d184_11,d184_10,d184_3]).
benzene(d184,[d184_21,d184_16,d184_17,d184_18,d184_19,d184_20]).
ring_size_6(d184,[d184_6,d184_1,d184_2,d184_3,d184_4,d184_5]).
ring_size_6(d184,[d184_4,d184_13,d184_12,d184_11,d184_10,d184_3]).
ring_size_6(d184,[d184_21,d184_16,d184_17,d184_18,d184_19,d184_20]).
ring_size_5(d184,[d184_5,d184_16,d184_17,d184_13,d184_4]).
nitro(d184,[d184_10,d184_25,d184_29,d184_30]).
nitro(d184,[d184_21,d184_26,d184_27,d184_28]).
benzene(d187,[d187_6,d187_1,d187_25,d187_24,d187_23,d187_22]).
benzene(d187,[d187_9,d187_10,d187_5,d187_4,d187_7,d187_8]).
benzene(d187,[d187_3,d187_17,d187_16,d187_15,d187_14,d187_2]).
carbon_6_ring(d187,[d187_6,d187_1,d187_2,d187_3,d187_4,d187_5]).
ring_size_6(d187,[d187_6,d187_1,d187_2,d187_3,d187_4,d187_5]).
ring_size_6(d187,[d187_6,d187_1,d187_25,d187_24,d187_23,d187_22]).
ring_size_6(d187,[d187_9,d187_10,d187_5,d187_4,d187_7,d187_8]).
ring_size_6(d187,[d187_3,d187_17,d187_16,d187_15,d187_14,d187_2]).
nitro(d187,[d187_9,d187_30,d187_31,d187_32]).
benzene(d8,[d8_6,d8_1,d8_2,d8_3,d8_4,d8_5]).
benzene(d8,[d8_9,d8_10,d8_11,d8_12,d8_7,d8_8]).
ring_size_6(d8,[d8_6,d8_1,d8_2,d8_3,d8_4,d8_5]).
ring_size_6(d8,[d8_9,d8_10,d8_11,d8_12,d8_7,d8_8]).
ring_size_5(d8,[d8_7,d8_12,d8_4,d8_3,d8_13]).
nitro(d8,[d8_1,d8_14,d8_15,d8_16]).
benzene(d16,[d16_6,d16_1,d16_2,d16_3,d16_4,d16_5]).
benzene(d16,[d16_9,d16_10,d16_4,d16_3,d16_7,d16_8]).
benzene(d16,[d16_16,d16_11,d16_12,d16_13,d16_14,d16_15]).
ring_size_6(d16,[d16_6,d16_1,d16_2,d16_3,d16_4,d16_5]).
ring_size_6(d16,[d16_9,d16_10,d16_4,d16_3,d16_7,d16_8]).
ring_size_6(d16,[d16_16,d16_11,d16_12,d16_13,d16_14,d16_15]).
ring_size_5(d16,[d16_5,d16_11,d16_12,d16_10,d16_4]).
nitro(d16,[d16_7,d16_17,d16_19,d16_20]).
nitro(d16,[d16_2,d16_18,d16_21,d16_22]).
benzene(d21,[d21_6,d21_1,d21_2,d21_3,d21_4,d21_5]).
benzene(d21,[d21_16,d21_11,d21_12,d21_13,d21_14,d21_15]).
ring_size_6(d21,[d21_6,d21_1,d21_2,d21_3,d21_4,d21_5]).
ring_size_6(d21,[d21_16,d21_11,d21_12,d21_13,d21_14,d21_15]).
nitro(d21,[d21_14,d21_21,d21_23,d21_24]).
nitro(d21,[d21_6,d21_22,d21_25,d21_26]).
benzene(d22,[d22_6,d22_1,d22_2,d22_3,d22_4,d22_5]).
benzene(d22,[d22_9,d22_10,d22_11,d22_12,d22_7,d22_8]).
ring_size_6(d22,[d22_6,d22_1,d22_2,d22_3,d22_4,d22_5]).
ring_size_6(d22,[d22_9,d22_10,d22_11,d22_12,d22_7,d22_8]).
ring_size_5(d22,[d22_7,d22_12,d22_4,d22_3,d22_13]).
nitro(d22,[d22_1,d22_15,d22_16,d22_17]).
benzene(d23,[d23_6,d23_1,d23_2,d23_3,d23_4,d23_5]).
benzene(d23,[d23_9,d23_10,d23_5,d23_4,d23_7,d23_8]).
benzene(d23,[d23_8,d23_14,d23_13,d23_12,d23_11,d23_7]).
benzene(d23,[d23_18,d23_13,d23_12,d23_15,d23_16,d23_17]).
carbon_5_aromatic_ring(d23,[d23_20,d23_18,d23_13,d23_14,d23_19]).
ring_size_6(d23,[d23_6,d23_1,d23_2,d23_3,d23_4,d23_5]).
ring_size_6(d23,[d23_9,d23_10,d23_5,d23_4,d23_7,d23_8]).
ring_size_6(d23,[d23_8,d23_14,d23_13,d23_12,d23_11,d23_7]).
ring_size_6(d23,[d23_18,d23_13,d23_12,d23_15,d23_16,d23_17]).
ring_size_5(d23,[d23_20,d23_18,d23_13,d23_14,d23_19]).
nitro(d23,[d23_20,d23_32,d23_33,d23_34]).
anthracene(d23,[[d23_9,d23_10,d23_5,d23_4,d23_7,d23_8],[d23_8,d23_14,d23_13,d23_12,d23_11,d23_7],[d23_18,d23_13,d23_12,d23_15,d23_16,d23_17]]).
benzene(d29,[d29_6,d29_1,d29_2,d29_3,d29_4,d29_5]).
benzene(d29,[d29_9,d29_10,d29_11,d29_12,d29_7,d29_8]).
ring_size_6(d29,[d29_6,d29_1,d29_2,d29_3,d29_4,d29_5]).
ring_size_6(d29,[d29_9,d29_10,d29_11,d29_12,d29_7,d29_8]).
ring_size_5(d29,[d29_7,d29_12,d29_4,d29_3,d29_13]).
nitro(d29,[d29_1,d29_14,d29_15,d29_16]).
nitro(d29,[d29_9,d29_17,d29_18,d29_19]).
nitro(d29,[d29_11,d29_20,d29_21,d29_22]).
benzene(d30,[d30_6,d30_1,d30_2,d30_3,d30_4,d30_5]).
benzene(d30,[d30_9,d30_10,d30_4,d30_3,d30_7,d30_8]).
benzene(d30,[d30_8,d30_14,d30_13,d30_12,d30_11,d30_7]).
benzene(d30,[d30_18,d30_13,d30_12,d30_15,d30_16,d30_17]).
carbon_5_aromatic_ring(d30,[d30_5,d30_19,d30_20,d30_10,d30_4]).
ring_size_6(d30,[d30_6,d30_1,d30_2,d30_3,d30_4,d30_5]).
ring_size_6(d30,[d30_9,d30_10,d30_4,d30_3,d30_7,d30_8]).
ring_size_6(d30,[d30_8,d30_14,d30_13,d30_12,d30_11,d30_7]).
ring_size_6(d30,[d30_18,d30_13,d30_12,d30_15,d30_16,d30_17]).
ring_size_5(d30,[d30_5,d30_19,d30_20,d30_10,d30_4]).
nitro(d30,[d30_19,d30_32,d30_33,d30_34]).
anthracene(d30,[[d30_9,d30_10,d30_4,d30_3,d30_7,d30_8],[d30_8,d30_14,d30_13,d30_12,d30_11,d30_7],[d30_18,d30_13,d30_12,d30_15,d30_16,d30_17]]).
benzene(d32,[d32_6,d32_1,d32_2,d32_3,d32_4,d32_5]).
benzene(d32,[d32_15,d32_10,d32_11,d32_12,d32_13,d32_14]).
ring_size_6(d32,[d32_6,d32_1,d32_2,d32_3,d32_4,d32_5]).
ring_size_6(d32,[d32_15,d32_10,d32_11,d32_12,d32_13,d32_14]).
nitro(d32,[d32_15,d32_19,d32_23,d32_24]).
nitro(d32,[d32_13,d32_20,d32_25,d32_26]).
nitro(d32,[d32_6,d32_21,d32_29,d32_30]).
nitro(d32,[d32_1,d32_22,d32_27,d32_28]).
benzene(d33,[d33_6,d33_1,d33_2,d33_3,d33_4,d33_5]).
benzene(d33,[d33_15,d33_10,d33_11,d33_12,d33_13,d33_14]).
ring_size_6(d33,[d33_6,d33_1,d33_2,d33_3,d33_4,d33_5]).
ring_size_6(d33,[d33_15,d33_10,d33_11,d33_12,d33_13,d33_14]).
ring_size_5(d33,[d33_4,d33_15,d33_10,d33_19,d33_3]).
nitro(d33,[d33_1,d33_22,d33_23,d33_24]).
benzene(d35,[d35_6,d35_1,d35_2,d35_3,d35_4,d35_5]).
benzene(d35,[d35_15,d35_10,d35_11,d35_12,d35_13,d35_14]).
ring_size_6(d35,[d35_6,d35_1,d35_2,d35_3,d35_4,d35_5]).
ring_size_6(d35,[d35_15,d35_10,d35_11,d35_12,d35_13,d35_14]).
ring_size_5(d35,[d35_4,d35_15,d35_10,d35_19,d35_3]).
nitro(d35,[d35_1,d35_22,d35_24,d35_25]).
benzene(d37,[d37_6,d37_1,d37_2,d37_3,d37_4,d37_5]).
benzene(d37,[d37_22,d37_15,d37_18,d37_19,d37_20,d37_21]).
ring_size_6(d37,[d37_6,d37_1,d37_2,d37_3,d37_4,d37_5]).
ring_size_6(d37,[d37_22,d37_15,d37_18,d37_19,d37_20,d37_21]).
nitro(d37,[d37_1,d37_28,d37_29,d37_30]).
benzene(d44,[d44_6,d44_1,d44_2,d44_3,d44_4,d44_5]).
benzene(d44,[d44_8,d44_14,d44_13,d44_12,d44_11,d44_7]).
carbon_6_ring(d44,[d44_9,d44_10,d44_5,d44_4,d44_7,d44_8]).
ring_size_6(d44,[d44_6,d44_1,d44_2,d44_3,d44_4,d44_5]).
ring_size_6(d44,[d44_9,d44_10,d44_5,d44_4,d44_7,d44_8]).
ring_size_6(d44,[d44_8,d44_14,d44_13,d44_12,d44_11,d44_7]).
nitro(d44,[d44_13,d44_15,d44_16,d44_17]).
benzene(d47,[d47_6,d47_1,d47_2,d47_3,d47_4,d47_5]).
benzene(d47,[d47_9,d47_10,d47_11,d47_16,d47_15,d47_14]).
carbon_6_ring(d47,[d47_9,d47_10,d47_4,d47_3,d47_7,d47_8]).
carbon_6_ring(d47,[d47_5,d47_13,d47_12,d47_11,d47_10,d47_4]).
ring_size_6(d47,[d47_6,d47_1,d47_2,d47_3,d47_4,d47_5]).
ring_size_6(d47,[d47_9,d47_10,d47_4,d47_3,d47_7,d47_8]).
ring_size_6(d47,[d47_5,d47_13,d47_12,d47_11,d47_10,d47_4]).
ring_size_6(d47,[d47_9,d47_10,d47_11,d47_16,d47_15,d47_14]).
nitro(d47,[d47_15,d47_17,d47_19,d47_20]).
nitro(d47,[d47_1,d47_18,d47_21,d47_22]).
benzene(d74,[d74_6,d74_1,d74_2,d74_3,d74_4,d74_5]).
benzene(d74,[d74_4,d74_13,d74_12,d74_11,d74_10,d74_3]).
benzene(d74,[d74_22,d74_17,d74_18,d74_19,d74_20,d74_21]).
benzene(d74,[d74_28,d74_21,d74_20,d74_25,d74_26,d74_27]).
ring_size_6(d74,[d74_6,d74_1,d74_2,d74_3,d74_4,d74_5]).
ring_size_6(d74,[d74_4,d74_13,d74_12,d74_11,d74_10,d74_3]).
ring_size_6(d74,[d74_22,d74_17,d74_18,d74_19,d74_20,d74_21]).
ring_size_6(d74,[d74_28,d74_21,d74_20,d74_25,d74_26,d74_27]).
ring_size_5(d74,[d74_5,d74_17,d74_18,d74_13,d74_4]).
nitro(d74,[d74_26,d74_32,d74_33,d74_34]).
benzene(d81,[d81_6,d81_1,d81_2,d81_3,d81_4,d81_5]).
ring_size_6(d81,[d81_6,d81_1,d81_2,d81_3,d81_4,d81_5]).
nitro(d81,[d81_5,d81_11,d81_13,d81_14]).
nitro(d81,[d81_3,d81_12,d81_15,d81_16]).
benzene(d85,[d85_6,d85_1,d85_2,d85_3,d85_4,d85_5]).
benzene(d85,[d85_9,d85_10,d85_5,d85_4,d85_7,d85_8]).
benzene(d85,[d85_8,d85_14,d85_13,d85_12,d85_11,d85_7]).
benzene(d85,[d85_18,d85_13,d85_12,d85_15,d85_16,d85_17]).
carbon_5_aromatic_ring(d85,[d85_9,d85_19,d85_20,d85_14,d85_8]).
ring_size_6(d85,[d85_6,d85_1,d85_2,d85_3,d85_4,d85_5]).
ring_size_6(d85,[d85_9,d85_10,d85_5,d85_4,d85_7,d85_8]).
ring_size_6(d85,[d85_8,d85_14,d85_13,d85_12,d85_11,d85_7]).
ring_size_6(d85,[d85_18,d85_13,d85_12,d85_15,d85_16,d85_17]).
ring_size_5(d85,[d85_9,d85_19,d85_20,d85_14,d85_8]).
nitro(d85,[d85_19,d85_32,d85_33,d85_34]).
anthracene(d85,[[d85_9,d85_10,d85_5,d85_4,d85_7,d85_8],[d85_8,d85_14,d85_13,d85_12,d85_11,d85_7],[d85_18,d85_13,d85_12,d85_15,d85_16,d85_17]]).
benzene(d91,[d91_9,d91_10,d91_4,d91_3,d91_7,d91_8]).
benzene(d91,[d91_9,d91_10,d91_17,d91_16,d91_15,d91_14]).
benzene(d91,[d91_9,d91_14,d91_13,d91_12,d91_11,d91_8]).
benzene(d91,[d91_20,d91_15,d91_14,d91_13,d91_18,d91_19]).
carbon_6_ring(d91,[d91_6,d91_1,d91_2,d91_3,d91_4,d91_5]).
ring_size_6(d91,[d91_6,d91_1,d91_2,d91_3,d91_4,d91_5]).
ring_size_6(d91,[d91_9,d91_10,d91_4,d91_3,d91_7,d91_8]).
ring_size_6(d91,[d91_9,d91_10,d91_17,d91_16,d91_15,d91_14]).
ring_size_6(d91,[d91_9,d91_14,d91_13,d91_12,d91_11,d91_8]).
ring_size_6(d91,[d91_20,d91_15,d91_14,d91_13,d91_18,d91_19]).
nitro(d91,[d91_18,d91_21,d91_22,d91_23]).
phenanthrene(d91,[[d91_9,d91_10,d91_4,d91_3,d91_7,d91_8],[d91_9,d91_10,d91_17,d91_16,d91_15,d91_14],[d91_20,d91_15,d91_14,d91_13,d91_18,d91_19]]).
ball3(d91,[[d91_9,d91_14,d91_13,d91_12,d91_11,d91_8],[d91_9,d91_10,d91_4,d91_3,d91_7,d91_8],[d91_9,d91_10,d91_17,d91_16,d91_15,d91_14]]).
ball3(d91,[[d91_9,d91_14,d91_13,d91_12,d91_11,d91_8],[d91_9,d91_10,d91_17,d91_16,d91_15,d91_14],[d91_20,d91_15,d91_14,d91_13,d91_18,d91_19]]).
benzene(d92,[d92_6,d92_1,d92_2,d92_3,d92_4,d92_5]).
benzene(d92,[d92_9,d92_10,d92_11,d92_16,d92_15,d92_14]).
carbon_6_ring(d92,[d92_9,d92_10,d92_4,d92_3,d92_7,d92_8]).
carbon_6_ring(d92,[d92_5,d92_13,d92_12,d92_11,d92_10,d92_4]).
ring_size_6(d92,[d92_6,d92_1,d92_2,d92_3,d92_4,d92_5]).
ring_size_6(d92,[d92_9,d92_10,d92_4,d92_3,d92_7,d92_8]).
ring_size_6(d92,[d92_5,d92_13,d92_12,d92_11,d92_10,d92_4]).
ring_size_6(d92,[d92_9,d92_10,d92_11,d92_16,d92_15,d92_14]).
nitro(d92,[d92_15,d92_17,d92_18,d92_19]).
benzene(d93,[d93_6,d93_1,d93_2,d93_3,d93_4,d93_5]).
benzene(d93,[d93_4,d93_13,d93_12,d93_11,d93_10,d93_3]).
benzene(d93,[d93_21,d93_16,d93_13,d93_12,d93_19,d93_20]).
benzene(d93,[d93_5,d93_18,d93_17,d93_16,d93_13,d93_4]).
benzene(d93,[d93_27,d93_18,d93_17,d93_24,d93_25,d93_26]).
ring_size_6(d93,[d93_6,d93_1,d93_2,d93_3,d93_4,d93_5]).
ring_size_6(d93,[d93_4,d93_13,d93_12,d93_11,d93_10,d93_3]).
ring_size_6(d93,[d93_21,d93_16,d93_13,d93_12,d93_19,d93_20]).
ring_size_6(d93,[d93_5,d93_18,d93_17,d93_16,d93_13,d93_4]).
ring_size_6(d93,[d93_27,d93_18,d93_17,d93_24,d93_25,d93_26]).
nitro(d93,[d93_19,d93_32,d93_33,d93_34]).
phenanthrene(d93,[[d93_6,d93_1,d93_2,d93_3,d93_4,d93_5],[d93_4,d93_13,d93_12,d93_11,d93_10,d93_3],[d93_21,d93_16,d93_13,d93_12,d93_19,d93_20]]).
phenanthrene(d93,[[d93_6,d93_1,d93_2,d93_3,d93_4,d93_5],[d93_5,d93_18,d93_17,d93_16,d93_13,d93_4],[d93_21,d93_16,d93_13,d93_12,d93_19,d93_20]]).
phenanthrene(d93,[[d93_6,d93_1,d93_2,d93_3,d93_4,d93_5],[d93_5,d93_18,d93_17,d93_16,d93_13,d93_4],[d93_27,d93_18,d93_17,d93_24,d93_25,d93_26]]).
ball3(d93,[[d93_6,d93_1,d93_2,d93_3,d93_4,d93_5],[d93_5,d93_18,d93_17,d93_16,d93_13,d93_4],[d93_4,d93_13,d93_12,d93_11,d93_10,d93_3]]).
ball3(d93,[[d93_5,d93_18,d93_17,d93_16,d93_13,d93_4],[d93_4,d93_13,d93_12,d93_11,d93_10,d93_3],[d93_21,d93_16,d93_13,d93_12,d93_19,d93_20]]).
benzene(d97,[d97_6,d97_1,d97_2,d97_3,d97_4,d97_5]).
benzene(d97,[d97_15,d97_10,d97_11,d97_12,d97_13,d97_14]).
ring_size_6(d97,[d97_6,d97_1,d97_2,d97_3,d97_4,d97_5]).
ring_size_6(d97,[d97_15,d97_10,d97_11,d97_12,d97_13,d97_14]).
ring_size_5(d97,[d97_4,d97_15,d97_10,d97_19,d97_3]).
nitro(d97,[d97_1,d97_26,d97_27,d97_28]).
benzene(d99,[d99_6,d99_1,d99_2,d99_3,d99_4,d99_5]).
benzene(d99,[d99_9,d99_10,d99_4,d99_3,d99_7,d99_8]).
carbon_5_aromatic_ring(d99,[d99_5,d99_11,d99_12,d99_10,d99_4]).
ring_size_6(d99,[d99_6,d99_1,d99_2,d99_3,d99_4,d99_5]).
ring_size_6(d99,[d99_9,d99_10,d99_4,d99_3,d99_7,d99_8]).
ring_size_5(d99,[d99_5,d99_11,d99_12,d99_10,d99_4]).
nitro(d99,[d99_11,d99_13,d99_14,d99_15]).
benzene(d102,[d102_6,d102_1,d102_2,d102_3,d102_4,d102_5]).
benzene(d102,[d102_9,d102_10,d102_11,d102_12,d102_13,d102_14]).
ring_size_6(d102,[d102_6,d102_1,d102_2,d102_3,d102_4,d102_5]).
ring_size_6(d102,[d102_9,d102_10,d102_11,d102_12,d102_13,d102_14]).
ring_size_5(d102,[d102_9,d102_14,d102_4,d102_3,d102_17]).
nitro(d102,[d102_11,d102_19,d102_29,d102_30]).
nitro(d102,[d102_13,d102_20,d102_27,d102_28]).
nitro(d102,[d102_5,d102_21,d102_25,d102_26]).
nitro(d102,[d102_1,d102_22,d102_23,d102_24]).
benzene(d103,[d103_6,d103_1,d103_2,d103_3,d103_4,d103_5]).
benzene(d103,[d103_4,d103_13,d103_12,d103_11,d103_10,d103_3]).
benzene(d103,[d103_23,d103_16,d103_13,d103_12,d103_21,d103_22]).
benzene(d103,[d103_5,d103_18,d103_17,d103_16,d103_13,d103_4]).
ring_size_6(d103,[d103_6,d103_1,d103_2,d103_3,d103_4,d103_5]).
ring_size_6(d103,[d103_4,d103_13,d103_12,d103_11,d103_10,d103_3]).
ring_size_6(d103,[d103_23,d103_16,d103_13,d103_12,d103_21,d103_22]).
ring_size_6(d103,[d103_5,d103_18,d103_17,d103_16,d103_13,d103_4]).
nitro(d103,[d103_22,d103_26,d103_27,d103_28]).
phenanthrene(d103,[[d103_6,d103_1,d103_2,d103_3,d103_4,d103_5],[d103_4,d103_13,d103_12,d103_11,d103_10,d103_3],[d103_23,d103_16,d103_13,d103_12,d103_21,d103_22]]).
phenanthrene(d103,[[d103_6,d103_1,d103_2,d103_3,d103_4,d103_5],[d103_5,d103_18,d103_17,d103_16,d103_13,d103_4],[d103_23,d103_16,d103_13,d103_12,d103_21,d103_22]]).
ball3(d103,[[d103_6,d103_1,d103_2,d103_3,d103_4,d103_5],[d103_5,d103_18,d103_17,d103_16,d103_13,d103_4],[d103_4,d103_13,d103_12,d103_11,d103_10,d103_3]]).
ball3(d103,[[d103_5,d103_18,d103_17,d103_16,d103_13,d103_4],[d103_4,d103_13,d103_12,d103_11,d103_10,d103_3],[d103_23,d103_16,d103_13,d103_12,d103_21,d103_22]]).
benzene(d106,[d106_6,d106_1,d106_2,d106_3,d106_4,d106_5]).
benzene(d106,[d106_4,d106_13,d106_12,d106_11,d106_10,d106_3]).
benzene(d106,[d106_22,d106_16,d106_13,d106_12,d106_20,d106_21]).
benzene(d106,[d106_5,d106_18,d106_17,d106_16,d106_13,d106_4]).
ring_size_6(d106,[d106_6,d106_1,d106_2,d106_3,d106_4,d106_5]).
ring_size_6(d106,[d106_4,d106_13,d106_12,d106_11,d106_10,d106_3]).
ring_size_6(d106,[d106_22,d106_16,d106_13,d106_12,d106_20,d106_21]).
ring_size_6(d106,[d106_5,d106_18,d106_17,d106_16,d106_13,d106_4]).
nitro(d106,[d106_22,d106_25,d106_27,d106_28]).
phenanthrene(d106,[[d106_6,d106_1,d106_2,d106_3,d106_4,d106_5],[d106_4,d106_13,d106_12,d106_11,d106_10,d106_3],[d106_22,d106_16,d106_13,d106_12,d106_20,d106_21]]).
phenanthrene(d106,[[d106_6,d106_1,d106_2,d106_3,d106_4,d106_5],[d106_5,d106_18,d106_17,d106_16,d106_13,d106_4],[d106_22,d106_16,d106_13,d106_12,d106_20,d106_21]]).
ball3(d106,[[d106_6,d106_1,d106_2,d106_3,d106_4,d106_5],[d106_5,d106_18,d106_17,d106_16,d106_13,d106_4],[d106_4,d106_13,d106_12,d106_11,d106_10,d106_3]]).
ball3(d106,[[d106_5,d106_18,d106_17,d106_16,d106_13,d106_4],[d106_4,d106_13,d106_12,d106_11,d106_10,d106_3],[d106_22,d106_16,d106_13,d106_12,d106_20,d106_21]]).
benzene(d108,[d108_6,d108_1,d108_2,d108_3,d108_4,d108_5]).
benzene(d108,[d108_9,d108_10,d108_11,d108_12,d108_4,d108_3]).
benzene(d108,[d108_19,d108_14,d108_15,d108_16,d108_17,d108_18]).
ring_size_6(d108,[d108_6,d108_1,d108_2,d108_3,d108_4,d108_5]).
ring_size_6(d108,[d108_9,d108_10,d108_11,d108_12,d108_4,d108_3]).
ring_size_6(d108,[d108_19,d108_14,d108_15,d108_16,d108_17,d108_18]).
ring_size_5(d108,[d108_5,d108_14,d108_15,d108_12,d108_4]).
nitro(d108,[d108_9,d108_24,d108_25,d108_26]).
benzene(d118,[d118_9,d118_10,d118_4,d118_3,d118_7,d118_8]).
benzene(d118,[d118_9,d118_10,d118_17,d118_16,d118_15,d118_14]).
benzene(d118,[d118_9,d118_14,d118_13,d118_12,d118_11,d118_8]).
benzene(d118,[d118_20,d118_15,d118_14,d118_13,d118_18,d118_19]).
carbon_6_ring(d118,[d118_6,d118_1,d118_2,d118_3,d118_4,d118_5]).
ring_size_6(d118,[d118_6,d118_1,d118_2,d118_3,d118_4,d118_5]).
ring_size_6(d118,[d118_9,d118_10,d118_4,d118_3,d118_7,d118_8]).
ring_size_6(d118,[d118_9,d118_10,d118_17,d118_16,d118_15,d118_14]).
ring_size_6(d118,[d118_9,d118_14,d118_13,d118_12,d118_11,d118_8]).
ring_size_6(d118,[d118_20,d118_15,d118_14,d118_13,d118_18,d118_19]).
nitro(d118,[d118_18,d118_21,d118_22,d118_23]).
phenanthrene(d118,[[d118_9,d118_10,d118_4,d118_3,d118_7,d118_8],[d118_9,d118_10,d118_17,d118_16,d118_15,d118_14],[d118_20,d118_15,d118_14,d118_13,d118_18,d118_19]]).
ball3(d118,[[d118_9,d118_14,d118_13,d118_12,d118_11,d118_8],[d118_9,d118_10,d118_4,d118_3,d118_7,d118_8],[d118_9,d118_10,d118_17,d118_16,d118_15,d118_14]]).
ball3(d118,[[d118_9,d118_14,d118_13,d118_12,d118_11,d118_8],[d118_9,d118_10,d118_17,d118_16,d118_15,d118_14],[d118_20,d118_15,d118_14,d118_13,d118_18,d118_19]]).
benzene(d121,[d121_6,d121_1,d121_2,d121_3,d121_4,d121_5]).
benzene(d121,[d121_9,d121_10,d121_11,d121_12,d121_4,d121_3]).
benzene(d121,[d121_19,d121_14,d121_15,d121_16,d121_17,d121_18]).
ring_size_6(d121,[d121_6,d121_1,d121_2,d121_3,d121_4,d121_5]).
ring_size_6(d121,[d121_9,d121_10,d121_11,d121_12,d121_4,d121_3]).
ring_size_6(d121,[d121_19,d121_14,d121_15,d121_16,d121_17,d121_18]).
ring_size_5(d121,[d121_5,d121_14,d121_15,d121_12,d121_4]).
nitro(d121,[d121_11,d121_24,d121_27,d121_28]).
nitro(d121,[d121_10,d121_25,d121_29,d121_30]).
nitro(d121,[d121_1,d121_26,d121_31,d121_32]).
benzene(d134,[d134_6,d134_1,d134_2,d134_3,d134_4,d134_5]).
benzene(d134,[d134_5,d134_13,d134_12,d134_11,d134_10,d134_4]).
benzene(d134,[d134_9,d134_10,d134_11,d134_16,d134_15,d134_14]).
carbon_6_ring(d134,[d134_9,d134_10,d134_4,d134_3,d134_7,d134_8]).
ring_size_6(d134,[d134_6,d134_1,d134_2,d134_3,d134_4,d134_5]).
ring_size_6(d134,[d134_9,d134_10,d134_4,d134_3,d134_7,d134_8]).
ring_size_6(d134,[d134_5,d134_13,d134_12,d134_11,d134_10,d134_4]).
ring_size_6(d134,[d134_9,d134_10,d134_11,d134_16,d134_15,d134_14]).
nitro(d134,[d134_15,d134_27,d134_29,d134_30]).
nitro(d134,[d134_1,d134_28,d134_31,d134_32]).
benzene(d145,[d145_6,d145_1,d145_2,d145_3,d145_4,d145_5]).
benzene(d145,[d145_16,d145_11,d145_12,d145_13,d145_14,d145_15]).
ring_size_6(d145,[d145_6,d145_1,d145_2,d145_3,d145_4,d145_5]).
ring_size_6(d145,[d145_16,d145_11,d145_12,d145_13,d145_14,d145_15]).
nitro(d145,[d145_1,d145_22,d145_23,d145_24]).
benzene(d158,[d158_6,d158_1,d158_2,d158_3,d158_4,d158_5]).
benzene(d158,[d158_15,d158_10,d158_11,d158_12,d158_13,d158_14]).
ring_size_6(d158,[d158_6,d158_1,d158_2,d158_3,d158_4,d158_5]).
ring_size_6(d158,[d158_15,d158_10,d158_11,d158_12,d158_13,d158_14]).
ring_size_5(d158,[d158_4,d158_15,d158_10,d158_19,d158_3]).
nitro(d158,[d158_1,d158_22,d158_23,d158_24]).
benzene(d161,[d161_6,d161_1,d161_2,d161_3,d161_4,d161_5]).
benzene(d161,[d161_9,d161_10,d161_11,d161_12,d161_4,d161_3]).
benzene(d161,[d161_20,d161_15,d161_12,d161_11,d161_18,d161_19]).
benzene(d161,[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4]).
benzene(d161,[d161_26,d161_17,d161_16,d161_23,d161_24,d161_25]).
ring_size_6(d161,[d161_6,d161_1,d161_2,d161_3,d161_4,d161_5]).
ring_size_6(d161,[d161_9,d161_10,d161_11,d161_12,d161_4,d161_3]).
ring_size_6(d161,[d161_20,d161_15,d161_12,d161_11,d161_18,d161_19]).
ring_size_6(d161,[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4]).
ring_size_6(d161,[d161_26,d161_17,d161_16,d161_23,d161_24,d161_25]).
nitro(d161,[d161_20,d161_31,d161_33,d161_34]).
nitro(d161,[d161_2,d161_32,d161_35,d161_36]).
anthracene(d161,[[d161_9,d161_10,d161_11,d161_12,d161_4,d161_3],[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4],[d161_26,d161_17,d161_16,d161_23,d161_24,d161_25]]).
phenanthrene(d161,[[d161_6,d161_1,d161_2,d161_3,d161_4,d161_5],[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4],[d161_20,d161_15,d161_12,d161_11,d161_18,d161_19]]).
phenanthrene(d161,[[d161_6,d161_1,d161_2,d161_3,d161_4,d161_5],[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4],[d161_26,d161_17,d161_16,d161_23,d161_24,d161_25]]).
ball3(d161,[[d161_9,d161_10,d161_11,d161_12,d161_4,d161_3],[d161_6,d161_1,d161_2,d161_3,d161_4,d161_5],[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4]]).
ball3(d161,[[d161_9,d161_10,d161_11,d161_12,d161_4,d161_3],[d161_5,d161_17,d161_16,d161_15,d161_12,d161_4],[d161_20,d161_15,d161_12,d161_11,d161_18,d161_19]]).
benzene(d162,[d162_6,d162_1,d162_2,d162_3,d162_4,d162_5]).
benzene(d162,[d162_15,d162_10,d162_11,d162_12,d162_13,d162_14]).
ring_size_6(d162,[d162_6,d162_1,d162_2,d162_3,d162_4,d162_5]).
ring_size_6(d162,[d162_15,d162_10,d162_11,d162_12,d162_13,d162_14]).
nitro(d162,[d162_14,d162_19,d162_27,d162_28]).
nitro(d162,[d162_13,d162_20,d162_29,d162_30]).
nitro(d162,[d162_6,d162_21,d162_25,d162_26]).
nitro(d162,[d162_1,d162_22,d162_23,d162_24]).
benzene(d167,[d167_6,d167_1,d167_2,d167_3,d167_4,d167_5]).
benzene(d167,[d167_9,d167_10,d167_11,d167_12,d167_4,d167_3]).
benzene(d167,[d167_22,d167_15,d167_12,d167_11,d167_20,d167_21]).
benzene(d167,[d167_5,d167_17,d167_16,d167_15,d167_12,d167_4]).
ring_size_6(d167,[d167_6,d167_1,d167_2,d167_3,d167_4,d167_5]).
ring_size_6(d167,[d167_9,d167_10,d167_11,d167_12,d167_4,d167_3]).
ring_size_6(d167,[d167_22,d167_15,d167_12,d167_11,d167_20,d167_21]).
ring_size_6(d167,[d167_5,d167_17,d167_16,d167_15,d167_12,d167_4]).
nitro(d167,[d167_22,d167_25,d167_27,d167_28]).
nitro(d167,[d167_2,d167_26,d167_29,d167_30]).
phenanthrene(d167,[[d167_6,d167_1,d167_2,d167_3,d167_4,d167_5],[d167_5,d167_17,d167_16,d167_15,d167_12,d167_4],[d167_22,d167_15,d167_12,d167_11,d167_20,d167_21]]).
ball3(d167,[[d167_9,d167_10,d167_11,d167_12,d167_4,d167_3],[d167_6,d167_1,d167_2,d167_3,d167_4,d167_5],[d167_5,d167_17,d167_16,d167_15,d167_12,d167_4]]).
ball3(d167,[[d167_9,d167_10,d167_11,d167_12,d167_4,d167_3],[d167_5,d167_17,d167_16,d167_15,d167_12,d167_4],[d167_22,d167_15,d167_12,d167_11,d167_20,d167_21]]).
benzene(d170,[d170_6,d170_1,d170_2,d170_3,d170_4,d170_5]).
benzene(d170,[d170_18,d170_13,d170_12,d170_15,d170_16,d170_17]).
ring_size_6(d170,[d170_6,d170_1,d170_2,d170_3,d170_4,d170_5]).
ring_size_6(d170,[d170_4,d170_14,d170_13,d170_12,d170_11,d170_3]).
ring_size_6(d170,[d170_18,d170_13,d170_12,d170_15,d170_16,d170_17]).
nitro(d170,[d170_18,d170_24,d170_25,d170_26]).
benzene(d176,[d176_6,d176_1,d176_2,d176_3,d176_4,d176_5]).
ring_size_6(d176,[d176_6,d176_1,d176_2,d176_3,d176_4,d176_5]).
ring_size_5(d176,[d176_4,d176_14,d176_11,d176_10,d176_3]).
nitro(d176,[d176_1,d176_17,d176_18,d176_19]).
benzene(d180,[d180_6,d180_1,d180_2,d180_3,d180_4,d180_5]).
benzene(d180,[d180_9,d180_10,d180_6,d180_5,d180_7,d180_8]).
benzene(d180,[d180_8,d180_14,d180_13,d180_12,d180_11,d180_7]).
benzene(d180,[d180_18,d180_12,d180_11,d180_15,d180_16,d180_17]).
benzene(d180,[d180_22,d180_16,d180_15,d180_19,d180_20,d180_21]).
carbon_6_ring(d180,[d180_7,d180_11,d180_15,d180_19,d180_4,d180_5]).
ring_size_6(d180,[d180_6,d180_1,d180_2,d180_3,d180_4,d180_5]).
ring_size_6(d180,[d180_9,d180_10,d180_6,d180_5,d180_7,d180_8]).
ring_size_6(d180,[d180_8,d180_14,d180_13,d180_12,d180_11,d180_7]).
ring_size_6(d180,[d180_18,d180_12,d180_11,d180_15,d180_16,d180_17]).
ring_size_6(d180,[d180_7,d180_11,d180_15,d180_19,d180_4,d180_5]).
ring_size_6(d180,[d180_22,d180_16,d180_15,d180_19,d180_20,d180_21]).
nitro(d180,[d180_22,d180_23,d180_25,d180_26]).
nitro(d180,[d180_3,d180_24,d180_27,d180_28]).
phenanthrene(d180,[[d180_9,d180_10,d180_6,d180_5,d180_7,d180_8],[d180_8,d180_14,d180_13,d180_12,d180_11,d180_7],[d180_18,d180_12,d180_11,d180_15,d180_16,d180_17]]).
benzene(d2,[d2_6,d2_1,d2_2,d2_3,d2_4,d2_5]).
hetero_aromatic_6_ring(d2,[d2_4,d2_13,d2_12,d2_11,d2_10,d2_3]).
ring_size_6(d2,[d2_6,d2_1,d2_2,d2_3,d2_4,d2_5]).
ring_size_6(d2,[d2_4,d2_13,d2_12,d2_11,d2_10,d2_3]).
nitro(d2,[d2_2,d2_17,d2_18,d2_19]).
benzene(d17,[d17_6,d17_1,d17_2,d17_3,d17_4,d17_5]).
ring_size_6(d17,[d17_6,d17_1,d17_2,d17_3,d17_4,d17_5]).
nitro(d17,[d17_4,d17_10,d17_11,d17_12]).
benzene(d42,[d42_6,d42_1,d42_2,d42_3,d42_4,d42_5]).
hetero_aromatic_5_ring(d42,[d42_9,d42_4,d42_3,d42_7,d42_8]).
ring_size_6(d42,[d42_6,d42_1,d42_2,d42_3,d42_4,d42_5]).
ring_size_5(d42,[d42_9,d42_4,d42_3,d42_7,d42_8]).
nitro(d42,[d42_6,d42_11,d42_12,d42_13]).
benzene(d70,[d70_6,d70_1,d70_2,d70_3,d70_4,d70_5]).
ring_size_6(d70,[d70_6,d70_1,d70_2,d70_3,d70_4,d70_5]).
ring_size_5(d70,[d70_4,d70_12,d70_11,d70_10,d70_3]).
nitro(d70,[d70_6,d70_16,d70_17,d70_18]).
benzene(d77,[d77_6,d77_1,d77_2,d77_3,d77_4,d77_5]).
benzene(d77,[d77_16,d77_11,d77_12,d77_13,d77_14,d77_15]).
ring_size_6(d77,[d77_6,d77_1,d77_2,d77_3,d77_4,d77_5]).
ring_size_6(d77,[d77_16,d77_11,d77_12,d77_13,d77_14,d77_15]).
nitro(d77,[d77_6,d77_22,d77_23,d77_24]).
benzene(d89,[d89_6,d89_1,d89_2,d89_3,d89_4,d89_5]).
hetero_aromatic_5_ring(d89,[d89_9,d89_4,d89_3,d89_7,d89_8]).
ring_size_6(d89,[d89_6,d89_1,d89_2,d89_3,d89_4,d89_5]).
ring_size_5(d89,[d89_9,d89_4,d89_3,d89_7,d89_8]).
nitro(d89,[d89_2,d89_11,d89_12,d89_13]).
benzene(d98,[d98_6,d98_1,d98_2,d98_3,d98_4,d98_5]).
hetero_aromatic_6_ring(d98,[d98_4,d98_13,d98_12,d98_11,d98_10,d98_3]).
ring_size_6(d98,[d98_6,d98_1,d98_2,d98_3,d98_4,d98_5]).
ring_size_6(d98,[d98_4,d98_13,d98_12,d98_11,d98_10,d98_3]).
nitro(d98,[d98_5,d98_17,d98_18,d98_19]).
benzene(d111,[d111_6,d111_1,d111_2,d111_3,d111_4,d111_5]).
ring_size_6(d111,[d111_6,d111_1,d111_2,d111_3,d111_4,d111_5]).
nitro(d111,[d111_2,d111_14,d111_15,d111_16]).
benzene(d114,[d114_6,d114_1,d114_2,d114_3,d114_4,d114_5]).
benzene(d114,[d114_16,d114_11,d114_12,d114_13,d114_14,d114_15]).
ring_size_6(d114,[d114_6,d114_1,d114_2,d114_3,d114_4,d114_5]).
ring_size_6(d114,[d114_16,d114_11,d114_12,d114_13,d114_14,d114_15]).
nitro(d114,[d114_1,d114_22,d114_23,d114_24]).
benzene(d119,[d119_6,d119_1,d119_2,d119_3,d119_4,d119_5]).
benzene(d119,[d119_16,d119_11,d119_12,d119_13,d119_14,d119_15]).
ring_size_6(d119,[d119_6,d119_1,d119_2,d119_3,d119_4,d119_5]).
ring_size_6(d119,[d119_16,d119_11,d119_12,d119_13,d119_14,d119_15]).
nitro(d119,[d119_16,d119_20,d119_25,d119_26]).
nitro(d119,[d119_14,d119_21,d119_27,d119_28]).
nitro(d119,[d119_5,d119_22,d119_23,d119_24]).
benzene(d120,[d120_6,d120_1,d120_2,d120_3,d120_4,d120_5]).
hetero_aromatic_5_ring(d120,[d120_9,d120_4,d120_3,d120_7,d120_8]).
ring_size_6(d120,[d120_6,d120_1,d120_2,d120_3,d120_4,d120_5]).
ring_size_5(d120,[d120_9,d120_4,d120_3,d120_7,d120_8]).
nitro(d120,[d120_6,d120_11,d120_12,d120_13]).
benzene(d130,[d130_6,d130_1,d130_2,d130_3,d130_4,d130_5]).
ring_size_6(d130,[d130_6,d130_1,d130_2,d130_3,d130_4,d130_5]).
nitro(d130,[d130_4,d130_16,d130_17,d130_18]).
benzene(d131,[d131_6,d131_1,d131_2,d131_3,d131_4,d131_5]).
benzene(d131,[d131_21,d131_16,d131_17,d131_18,d131_19,d131_20]).
ring_size_6(d131,[d131_6,d131_1,d131_2,d131_3,d131_4,d131_5]).
ring_size_6(d131,[d131_21,d131_16,d131_17,d131_18,d131_19,d131_20]).
nitro(d131,[d131_2,d131_27,d131_28,d131_29]).
benzene(d132,[d132_6,d132_1,d132_2,d132_3,d132_4,d132_5]).
ring_size_6(d132,[d132_6,d132_1,d132_2,d132_3,d132_4,d132_5]).
nitro(d132,[d132_4,d132_14,d132_15,d132_16]).
benzene(d138,[d138_6,d138_1,d138_2,d138_3,d138_4,d138_5]).
hetero_aromatic_5_ring(d138,[d138_9,d138_4,d138_3,d138_7,d138_8]).
ring_size_6(d138,[d138_6,d138_1,d138_2,d138_3,d138_4,d138_5]).
ring_size_5(d138,[d138_9,d138_4,d138_3,d138_7,d138_8]).
nitro(d138,[d138_1,d138_11,d138_12,d138_13]).
benzene(d141,[d141_6,d141_1,d141_2,d141_3,d141_4,d141_5]).
ring_size_6(d141,[d141_6,d141_1,d141_2,d141_3,d141_4,d141_5]).
nitro(d141,[d141_5,d141_10,d141_13,d141_14]).
benzene(d144,[d144_6,d144_1,d144_2,d144_3,d144_4,d144_5]).
ring_size_6(d144,[d144_6,d144_1,d144_2,d144_3,d144_4,d144_5]).
nitro(d144,[d144_5,d144_10,d144_13,d144_14]).
benzene(d147,[d147_6,d147_1,d147_2,d147_3,d147_4,d147_5]).
hetero_aromatic_5_ring(d147,[d147_9,d147_4,d147_3,d147_7,d147_8]).
ring_size_6(d147,[d147_6,d147_1,d147_2,d147_3,d147_4,d147_5]).
ring_size_5(d147,[d147_9,d147_4,d147_3,d147_7,d147_8]).
nitro(d147,[d147_6,d147_11,d147_12,d147_13]).
benzene(d186,[d186_6,d186_1,d186_2,d186_3,d186_4,d186_5]).
ring_size_6(d186,[d186_6,d186_1,d186_2,d186_3,d186_4,d186_5]).
ring_size_5(d186,[d186_4,d186_12,d186_11,d186_10,d186_3]).
nitro(d186,[d186_6,d186_18,d186_19,d186_20]).
benzene(d188,[d188_6,d188_1,d188_2,d188_3,d188_4,d188_5]).
benzene(d188,[d188_16,d188_11,d188_12,d188_13,d188_14,d188_15]).
ring_size_6(d188,[d188_6,d188_1,d188_2,d188_3,d188_4,d188_5]).
ring_size_6(d188,[d188_16,d188_11,d188_12,d188_13,d188_14,d188_15]).
ring_size_5(d188,[d188_4,d188_16,d188_11,d188_20,d188_3]).
nitro(d188,[d188_14,d188_22,d188_23,d188_24]).
benzene(d3,[d3_6,d3_1,d3_2,d3_3,d3_4,d3_5]).
hetero_aromatic_6_ring(d3,[d3_4,d3_13,d3_12,d3_11,d3_10,d3_3]).
ring_size_6(d3,[d3_6,d3_1,d3_2,d3_3,d3_4,d3_5]).
ring_size_6(d3,[d3_4,d3_13,d3_12,d3_11,d3_10,d3_3]).
nitro(d3,[d3_5,d3_17,d3_18,d3_19]).
benzene(d5,[d5_6,d5_1,d5_2,d5_3,d5_4,d5_5]).
ring_size_6(d5,[d5_6,d5_1,d5_2,d5_3,d5_4,d5_5]).
nitro(d5,[d5_5,d5_10,d5_13,d5_14]).
benzene(d7,[d7_6,d7_1,d7_2,d7_3,d7_4,d7_5]).
benzene(d7,[d7_17,d7_12,d7_13,d7_14,d7_15,d7_16]).
ring_size_6(d7,[d7_6,d7_1,d7_2,d7_3,d7_4,d7_5]).
ring_size_6(d7,[d7_17,d7_12,d7_13,d7_14,d7_15,d7_16]).
nitro(d7,[d7_15,d7_25,d7_26,d7_27]).
benzene(d9,[d9_6,d9_1,d9_2,d9_3,d9_4,d9_5]).
hetero_aromatic_5_ring(d9,[d9_9,d9_3,d9_4,d9_7,d9_8]).
ring_size_6(d9,[d9_6,d9_1,d9_2,d9_3,d9_4,d9_5]).
ring_size_5(d9,[d9_9,d9_3,d9_4,d9_7,d9_8]).
nitro(d9,[d9_6,d9_10,d9_11,d9_12]).
benzene(d14,[d14_6,d14_1,d14_2,d14_3,d14_4,d14_5]).
hetero_aromatic_5_ring(d14,[d14_9,d14_3,d14_4,d14_7,d14_8]).
ring_size_6(d14,[d14_6,d14_1,d14_2,d14_3,d14_4,d14_5]).
ring_size_5(d14,[d14_9,d14_3,d14_4,d14_7,d14_8]).
nitro(d14,[d14_1,d14_11,d14_12,d14_13]).
benzene(d19,[d19_6,d19_1,d19_2,d19_3,d19_4,d19_5]).
hetero_aromatic_6_ring(d19,[d19_4,d19_13,d19_12,d19_11,d19_10,d19_3]).
ring_size_6(d19,[d19_6,d19_1,d19_2,d19_3,d19_4,d19_5]).
ring_size_6(d19,[d19_4,d19_13,d19_12,d19_11,d19_10,d19_3]).
nitro(d19,[d19_6,d19_17,d19_18,d19_19]).
benzene(d34,[d34_6,d34_1,d34_2,d34_3,d34_4,d34_5]).
benzene(d34,[d34_4,d34_14,d34_13,d34_12,d34_11,d34_3]).
ring_size_6(d34,[d34_6,d34_1,d34_2,d34_3,d34_4,d34_5]).
ring_size_6(d34,[d34_4,d34_14,d34_13,d34_12,d34_11,d34_3]).
nitro(d34,[d34_13,d34_18,d34_19,d34_20]).
benzene(d36,[d36_6,d36_1,d36_2,d36_3,d36_4,d36_5]).
benzene(d36,[d36_4,d36_14,d36_13,d36_12,d36_11,d36_3]).
ring_size_6(d36,[d36_6,d36_1,d36_2,d36_3,d36_4,d36_5]).
ring_size_6(d36,[d36_4,d36_14,d36_13,d36_12,d36_11,d36_3]).
nitro(d36,[d36_13,d36_21,d36_22,d36_23]).
benzene(d38,[d38_6,d38_1,d38_2,d38_3,d38_4,d38_5]).
benzene(d38,[d38_17,d38_12,d38_13,d38_14,d38_15,d38_16]).
ring_size_6(d38,[d38_6,d38_1,d38_2,d38_3,d38_4,d38_5]).
ring_size_6(d38,[d38_17,d38_12,d38_13,d38_14,d38_15,d38_16]).
nitro(d38,[d38_15,d38_22,d38_23,d38_24]).
benzene(d39,[d39_6,d39_1,d39_2,d39_3,d39_4,d39_5]).
hetero_aromatic_6_ring(d39,[d39_9,d39_10,d39_11,d39_12,d39_4,d39_3]).
ring_size_6(d39,[d39_6,d39_1,d39_2,d39_3,d39_4,d39_5]).
ring_size_6(d39,[d39_9,d39_10,d39_11,d39_12,d39_4,d39_3]).
nitro(d39,[d39_2,d39_16,d39_17,d39_18]).
benzene(d40,[d40_6,d40_1,d40_2,d40_3,d40_4,d40_5]).
benzene(d40,[d40_4,d40_14,d40_13,d40_12,d40_11,d40_3]).
ring_size_6(d40,[d40_6,d40_1,d40_2,d40_3,d40_4,d40_5]).
ring_size_6(d40,[d40_4,d40_14,d40_13,d40_12,d40_11,d40_3]).
nitro(d40,[d40_14,d40_18,d40_19,d40_20]).
benzene(d55,[d55_6,d55_1,d55_2,d55_3,d55_4,d55_5]).
benzene(d55,[d55_4,d55_14,d55_13,d55_12,d55_11,d55_3]).
ring_size_6(d55,[d55_6,d55_1,d55_2,d55_3,d55_4,d55_5]).
ring_size_6(d55,[d55_4,d55_14,d55_13,d55_12,d55_11,d55_3]).
nitro(d55,[d55_13,d55_21,d55_22,d55_23]).
benzene(d62,[d62_6,d62_1,d62_2,d62_3,d62_4,d62_5]).
ring_size_6(d62,[d62_6,d62_1,d62_2,d62_3,d62_4,d62_5]).
nitro(d62,[d62_1,d62_13,d62_14,d62_15]).
benzene(d65,[d65_6,d65_1,d65_2,d65_3,d65_4,d65_5]).
benzene(d65,[d65_4,d65_14,d65_13,d65_12,d65_11,d65_3]).
ring_size_6(d65,[d65_6,d65_1,d65_2,d65_3,d65_4,d65_5]).
ring_size_6(d65,[d65_4,d65_14,d65_13,d65_12,d65_11,d65_3]).
nitro(d65,[d65_14,d65_17,d65_19,d65_20]).
nitro(d65,[d65_12,d65_18,d65_21,d65_22]).
benzene(d66,[d66_6,d66_1,d66_2,d66_3,d66_4,d66_5]).
benzene(d66,[d66_16,d66_11,d66_12,d66_13,d66_14,d66_15]).
ring_size_6(d66,[d66_6,d66_1,d66_2,d66_3,d66_4,d66_5]).
ring_size_6(d66,[d66_16,d66_11,d66_12,d66_13,d66_14,d66_15]).
nitro(d66,[d66_6,d66_22,d66_23,d66_24]).
benzene(d73,[d73_6,d73_1,d73_2,d73_3,d73_4,d73_5]).
benzene(d73,[d73_16,d73_11,d73_12,d73_13,d73_14,d73_15]).
ring_size_6(d73,[d73_6,d73_1,d73_2,d73_3,d73_4,d73_5]).
ring_size_6(d73,[d73_16,d73_11,d73_12,d73_13,d73_14,d73_15]).
nitro(d73,[d73_5,d73_22,d73_23,d73_24]).
benzene(d76,[d76_6,d76_1,d76_2,d76_3,d76_4,d76_5]).
ring_size_6(d76,[d76_6,d76_1,d76_2,d76_3,d76_4,d76_5]).
nitro(d76,[d76_5,d76_11,d76_13,d76_14]).
benzene(d78,[d78_6,d78_1,d78_2,d78_3,d78_4,d78_5]).
benzene(d78,[d78_16,d78_11,d78_12,d78_13,d78_14,d78_15]).
ring_size_6(d78,[d78_6,d78_1,d78_2,d78_3,d78_4,d78_5]).
ring_size_6(d78,[d78_16,d78_11,d78_12,d78_13,d78_14,d78_15]).
nitro(d78,[d78_14,d78_25,d78_26,d78_27]).
benzene(d84,[d84_6,d84_1,d84_2,d84_3,d84_4,d84_5]).
ring_size_6(d84,[d84_6,d84_1,d84_2,d84_3,d84_4,d84_5]).
nitro(d84,[d84_2,d84_16,d84_17,d84_18]).
benzene(d88,[d88_6,d88_1,d88_2,d88_3,d88_4,d88_5]).
ring_size_6(d88,[d88_6,d88_1,d88_2,d88_3,d88_4,d88_5]).
nitro(d88,[d88_2,d88_17,d88_18,d88_19]).
benzene(d100,[d100_6,d100_1,d100_2,d100_3,d100_4,d100_5]).
benzene(d100,[d100_16,d100_11,d100_12,d100_13,d100_14,d100_15]).
ring_size_6(d100,[d100_6,d100_1,d100_2,d100_3,d100_4,d100_5]).
ring_size_6(d100,[d100_16,d100_11,d100_12,d100_13,d100_14,d100_15]).
nitro(d100,[d100_6,d100_22,d100_23,d100_24]).
benzene(d110,[d110_6,d110_1,d110_2,d110_3,d110_4,d110_5]).
benzene(d110,[d110_15,d110_10,d110_11,d110_12,d110_13,d110_14]).
ring_size_6(d110,[d110_6,d110_1,d110_2,d110_3,d110_4,d110_5]).
ring_size_6(d110,[d110_15,d110_10,d110_11,d110_12,d110_13,d110_14]).
nitro(d110,[d110_15,d110_19,d110_25,d110_26]).
nitro(d110,[d110_13,d110_20,d110_27,d110_28]).
nitro(d110,[d110_5,d110_21,d110_23,d110_24]).
nitro(d110,[d110_3,d110_22,d110_29,d110_30]).
benzene(d113,[d113_6,d113_1,d113_2,d113_3,d113_4,d113_5]).
ring_size_6(d113,[d113_6,d113_1,d113_2,d113_3,d113_4,d113_5]).
nitro(d113,[d113_4,d113_11,d113_13,d113_14]).
nitro(d113,[d113_2,d113_12,d113_15,d113_16]).
benzene(d116,[d116_6,d116_1,d116_2,d116_3,d116_4,d116_5]).
ring_size_6(d116,[d116_6,d116_1,d116_2,d116_3,d116_4,d116_5]).
nitro(d116,[d116_5,d116_11,d116_13,d116_14]).
benzene(d123,[d123_6,d123_1,d123_2,d123_3,d123_4,d123_5]).
benzene(d123,[d123_16,d123_11,d123_12,d123_13,d123_14,d123_15]).
ring_size_6(d123,[d123_6,d123_1,d123_2,d123_3,d123_4,d123_5]).
ring_size_6(d123,[d123_16,d123_11,d123_12,d123_13,d123_14,d123_15]).
nitro(d123,[d123_14,d123_28,d123_29,d123_30]).
benzene(d124,[d124_6,d124_1,d124_2,d124_3,d124_4,d124_5]).
ring_size_6(d124,[d124_6,d124_1,d124_2,d124_3,d124_4,d124_5]).
nitro(d124,[d124_5,d124_10,d124_13,d124_14]).
benzene(d129,[d129_6,d129_1,d129_2,d129_3,d129_4,d129_5]).
ring_size_6(d129,[d129_6,d129_1,d129_2,d129_3,d129_4,d129_5]).
nitro(d129,[d129_4,d129_15,d129_17,d129_18]).
nitro(d129,[d129_2,d129_16,d129_19,d129_20]).
benzene(d133,[d133_6,d133_1,d133_2,d133_3,d133_4,d133_5]).
benzene(d133,[d133_16,d133_11,d133_10,d133_13,d133_14,d133_15]).
ring_size_6(d133,[d133_6,d133_1,d133_2,d133_3,d133_4,d133_5]).
ring_size_6(d133,[d133_9,d133_10,d133_11,d133_12,d133_4,d133_3]).
ring_size_6(d133,[d133_16,d133_11,d133_10,d133_13,d133_14,d133_15]).
nitro(d133,[d133_15,d133_18,d133_23,d133_24]).
benzene(d135,[d135_6,d135_1,d135_2,d135_3,d135_4,d135_5]).
ring_size_6(d135,[d135_6,d135_1,d135_2,d135_3,d135_4,d135_5]).
nitro(d135,[d135_5,d135_10,d135_11,d135_12]).
benzene(d139,[d139_6,d139_1,d139_2,d139_3,d139_4,d139_5]).
ring_size_6(d139,[d139_6,d139_1,d139_2,d139_3,d139_4,d139_5]).
nitro(d139,[d139_4,d139_13,d139_15,d139_16]).
benzene(d142,[d142_6,d142_1,d142_2,d142_3,d142_4,d142_5]).
benzene(d142,[d142_16,d142_11,d142_10,d142_13,d142_14,d142_15]).
ring_size_6(d142,[d142_6,d142_1,d142_2,d142_3,d142_4,d142_5]).
ring_size_6(d142,[d142_9,d142_10,d142_11,d142_12,d142_4,d142_3]).
ring_size_6(d142,[d142_16,d142_11,d142_10,d142_13,d142_14,d142_15]).
nitro(d142,[d142_6,d142_19,d142_20,d142_21]).
benzene(d143,[d143_6,d143_1,d143_2,d143_3,d143_4,d143_5]).
benzene(d143,[d143_17,d143_12,d143_13,d143_14,d143_15,d143_16]).
ring_size_6(d143,[d143_6,d143_1,d143_2,d143_3,d143_4,d143_5]).
ring_size_6(d143,[d143_17,d143_12,d143_13,d143_14,d143_15,d143_16]).
nitro(d143,[d143_4,d143_23,d143_24,d143_25]).
hetero_aromatic_5_ring(d150,[d150_5,d150_1,d150_2,d150_3,d150_4]).
ring_size_5(d150,[d150_5,d150_1,d150_2,d150_3,d150_4]).
nitro(d150,[d150_1,d150_6,d150_7,d150_8]).
benzene(d154,[d154_6,d154_1,d154_2,d154_3,d154_4,d154_5]).
hetero_aromatic_5_ring(d154,[d154_9,d154_4,d154_3,d154_7,d154_8]).
ring_size_6(d154,[d154_6,d154_1,d154_2,d154_3,d154_4,d154_5]).
ring_size_5(d154,[d154_9,d154_4,d154_3,d154_7,d154_8]).
nitro(d154,[d154_8,d154_10,d154_11,d154_12]).
benzene(d155,[d155_6,d155_1,d155_2,d155_3,d155_4,d155_5]).
ring_size_6(d155,[d155_6,d155_1,d155_2,d155_3,d155_4,d155_5]).
nitro(d155,[d155_4,d155_15,d155_17,d155_18]).
nitro(d155,[d155_2,d155_16,d155_19,d155_20]).
benzene(d156,[d156_6,d156_1,d156_2,d156_3,d156_4,d156_5]).
ring_size_6(d156,[d156_6,d156_1,d156_2,d156_3,d156_4,d156_5]).
ring_size_5(d156,[d156_4,d156_12,d156_11,d156_10,d156_3]).
nitro(d156,[d156_1,d156_18,d156_19,d156_20]).
benzene(d160,[d160_6,d160_1,d160_2,d160_3,d160_4,d160_5]).
ring_size_6(d160,[d160_6,d160_1,d160_2,d160_3,d160_4,d160_5]).
nitro(d160,[d160_2,d160_13,d160_17,d160_18]).
nitro(d160,[d160_6,d160_14,d160_15,d160_16]).
benzene(d168,[d168_6,d168_1,d168_2,d168_3,d168_4,d168_5]).
ring_size_6(d168,[d168_6,d168_1,d168_2,d168_3,d168_4,d168_5]).
nitro(d168,[d168_2,d168_16,d168_17,d168_18]).
benzene(d175,[d175_6,d175_1,d175_2,d175_3,d175_4,d175_5]).
benzene(d175,[d175_16,d175_11,d175_10,d175_13,d175_14,d175_15]).
ring_size_6(d175,[d175_6,d175_1,d175_2,d175_3,d175_4,d175_5]).
ring_size_6(d175,[d175_9,d175_10,d175_11,d175_12,d175_4,d175_3]).
ring_size_6(d175,[d175_16,d175_11,d175_10,d175_13,d175_14,d175_15]).
nitro(d175,[d175_14,d175_17,d175_18,d175_19]).
benzene(d179,[d179_6,d179_1,d179_2,d179_3,d179_4,d179_5]).
benzene(d179,[d179_21,d179_16,d179_17,d179_18,d179_19,d179_20]).
ring_size_6(d179,[d179_6,d179_1,d179_2,d179_3,d179_4,d179_5]).
ring_size_6(d179,[d179_21,d179_16,d179_17,d179_18,d179_19,d179_20]).
nitro(d179,[d179_2,d179_26,d179_27,d179_28]).
nitro(d179,[d179_19,d179_29,d179_30,d179_31]).
benzene(d181,[d181_6,d181_1,d181_2,d181_3,d181_4,d181_5]).
ring_size_6(d181,[d181_6,d181_1,d181_2,d181_3,d181_4,d181_5]).
nitro(d181,[d181_4,d181_12,d181_13,d181_14]).
benzene(d182,[d182_6,d182_1,d182_2,d182_3,d182_4,d182_5]).
hetero_aromatic_6_ring(d182,[d182_4,d182_13,d182_12,d182_11,d182_10,d182_3]).
ring_size_6(d182,[d182_6,d182_1,d182_2,d182_3,d182_4,d182_5]).
ring_size_6(d182,[d182_4,d182_13,d182_12,d182_11,d182_10,d182_3]).
nitro(d182,[d182_2,d182_20,d182_21,d182_22]).
benzene(d185,[d185_6,d185_1,d185_2,d185_3,d185_4,d185_5]).
ring_size_6(d185,[d185_6,d185_1,d185_2,d185_3,d185_4,d185_5]).
nitro(d185,[d185_3,d185_11,d185_15,d185_16]).
nitro(d185,[d185_2,d185_12,d185_13,d185_14]).
benzene(d190,[d190_6,d190_1,d190_2,d190_3,d190_4,d190_5]).
hetero_aromatic_5_ring(d190,[d190_9,d190_4,d190_3,d190_7,d190_8]).
ring_size_6(d190,[d190_6,d190_1,d190_2,d190_3,d190_4,d190_5]).
ring_size_5(d190,[d190_9,d190_4,d190_3,d190_7,d190_8]).
nitro(d190,[d190_6,d190_10,d190_11,d190_12]).
benzene(d191,[d191_6,d191_1,d191_2,d191_3,d191_4,d191_5]).
benzene(d191,[d191_9,d191_10,d191_5,d191_4,d191_7,d191_8]).
benzene(d191,[d191_8,d191_14,d191_13,d191_12,d191_11,d191_7]).
benzene(d191,[d191_7,d191_11,d191_16,d191_15,d191_3,d191_4]).
carbon_5_aromatic_ring(d191,[d191_18,d191_16,d191_11,d191_12,d191_17]).
ring_size_6(d191,[d191_6,d191_1,d191_2,d191_3,d191_4,d191_5]).
ring_size_6(d191,[d191_9,d191_10,d191_5,d191_4,d191_7,d191_8]).
ring_size_6(d191,[d191_8,d191_14,d191_13,d191_12,d191_11,d191_7]).
ring_size_6(d191,[d191_7,d191_11,d191_16,d191_15,d191_3,d191_4]).
ring_size_5(d191,[d191_18,d191_16,d191_11,d191_12,d191_17]).
nitro(d191,[d191_18,d191_28,d191_29,d191_30]).
phenanthrene(d191,[[d191_8,d191_14,d191_13,d191_12,d191_11,d191_7],[d191_7,d191_11,d191_16,d191_15,d191_3,d191_4],[d191_6,d191_1,d191_2,d191_3,d191_4,d191_5]]).
ball3(d191,[[d191_9,d191_10,d191_5,d191_4,d191_7,d191_8],[d191_8,d191_14,d191_13,d191_12,d191_11,d191_7],[d191_7,d191_11,d191_16,d191_15,d191_3,d191_4]]).
ball3(d191,[[d191_9,d191_10,d191_5,d191_4,d191_7,d191_8],[d191_7,d191_11,d191_16,d191_15,d191_3,d191_4],[d191_6,d191_1,d191_2,d191_3,d191_4,d191_5]]).
benzene(d194,[d194_6,d194_1,d194_2,d194_3,d194_4,d194_5]).
ring_size_6(d194,[d194_6,d194_1,d194_2,d194_3,d194_4,d194_5]).
nitro(d194,[d194_2,d194_11,d194_14,d194_15]).
benzene(d197,[d197_6,d197_1,d197_2,d197_3,d197_4,d197_5]).
hetero_aromatic_5_ring(d197,[d197_9,d197_4,d197_3,d197_7,d197_8]).
ring_size_6(d197,[d197_6,d197_1,d197_2,d197_3,d197_4,d197_5]).
ring_size_5(d197,[d197_9,d197_4,d197_3,d197_7,d197_8]).
nitro(d197,[d197_8,d197_11,d197_12,d197_13]).
benzene(e1,[e1_6,e1_1,e1_2,e1_3,e1_4,e1_5]).
benzene(e1,[e1_4,e1_14,e1_13,e1_12,e1_11,e1_3]).
benzene(e1,[e1_18,e1_13,e1_12,e1_15,e1_16,e1_17]).
benzene(e1,[e1_23,e1_14,e1_13,e1_18,e1_21,e1_22]).
benzene(e1,[e1_28,e1_21,e1_18,e1_17,e1_26,e1_27]).
ring_size_6(e1,[e1_6,e1_1,e1_2,e1_3,e1_4,e1_5]).
ring_size_6(e1,[e1_4,e1_14,e1_13,e1_12,e1_11,e1_3]).
ring_size_6(e1,[e1_18,e1_13,e1_12,e1_15,e1_16,e1_17]).
ring_size_6(e1,[e1_23,e1_14,e1_13,e1_18,e1_21,e1_22]).
ring_size_6(e1,[e1_28,e1_21,e1_18,e1_17,e1_26,e1_27]).
nitro(e1,[e1_11,e1_32,e1_33,e1_34]).
anthracene(e1,[[e1_6,e1_1,e1_2,e1_3,e1_4,e1_5],[e1_4,e1_14,e1_13,e1_12,e1_11,e1_3],[e1_18,e1_13,e1_12,e1_15,e1_16,e1_17]]).
phenanthrene(e1,[[e1_6,e1_1,e1_2,e1_3,e1_4,e1_5],[e1_4,e1_14,e1_13,e1_12,e1_11,e1_3],[e1_23,e1_14,e1_13,e1_18,e1_21,e1_22]]).
ball3(e1,[[e1_4,e1_14,e1_13,e1_12,e1_11,e1_3],[e1_23,e1_14,e1_13,e1_18,e1_21,e1_22],[e1_18,e1_13,e1_12,e1_15,e1_16,e1_17]]).
ball3(e1,[[e1_28,e1_21,e1_18,e1_17,e1_26,e1_27],[e1_23,e1_14,e1_13,e1_18,e1_21,e1_22],[e1_18,e1_13,e1_12,e1_15,e1_16,e1_17]]).
benzene(e2,[e2_6,e2_1,e2_2,e2_3,e2_4,e2_5]).
benzene(e2,[e2_9,e2_10,e2_4,e2_3,e2_7,e2_8]).
benzene(e2,[e2_9,e2_10,e2_17,e2_16,e2_15,e2_14]).
benzene(e2,[e2_20,e2_15,e2_14,e2_13,e2_18,e2_19]).
carbon_6_ring(e2,[e2_9,e2_14,e2_13,e2_12,e2_11,e2_8]).
ring_size_6(e2,[e2_6,e2_1,e2_2,e2_3,e2_4,e2_5]).
ring_size_6(e2,[e2_9,e2_10,e2_4,e2_3,e2_7,e2_8]).
ring_size_6(e2,[e2_9,e2_10,e2_17,e2_16,e2_15,e2_14]).
ring_size_6(e2,[e2_9,e2_14,e2_13,e2_12,e2_11,e2_8]).
ring_size_6(e2,[e2_20,e2_15,e2_14,e2_13,e2_18,e2_19]).
nitro(e2,[e2_7,e2_21,e2_22,e2_23]).
phenanthrene(e2,[[e2_9,e2_10,e2_4,e2_3,e2_7,e2_8],[e2_9,e2_10,e2_17,e2_16,e2_15,e2_14],[e2_20,e2_15,e2_14,e2_13,e2_18,e2_19]]).
benzene(e27,[e27_6,e27_1,e27_2,e27_3,e27_4,e27_5]).
benzene(e27,[e27_5,e27_13,e27_12,e27_11,e27_10,e27_4]).
benzene(e27,[e27_29,e27_20,e27_18,e27_17,e27_27,e27_28]).
benzene(e27,[e27_23,e27_19,e27_18,e27_20,e27_21,e27_22]).
carbon_6_ring(e27,[e27_4,e27_10,e27_19,e27_18,e27_17,e27_3]).
ring_size_6(e27,[e27_6,e27_1,e27_2,e27_3,e27_4,e27_5]).
ring_size_6(e27,[e27_5,e27_13,e27_12,e27_11,e27_10,e27_4]).
ring_size_6(e27,[e27_4,e27_10,e27_19,e27_18,e27_17,e27_3]).
ring_size_6(e27,[e27_29,e27_20,e27_18,e27_17,e27_27,e27_28]).
ring_size_6(e27,[e27_23,e27_19,e27_18,e27_20,e27_21,e27_22]).
nitro(e27,[e27_29,e27_32,e27_33,e27_34]).
hetero_aromatic_5_ring(f1,[f1_5,f1_1,f1_2,f1_3,f1_4]).
ring_size_5(f1,[f1_5,f1_1,f1_2,f1_3,f1_4]).
nitro(f1,[f1_4,f1_8,f1_10,f1_9]).
carbon_5_aromatic_ring(f2,[f2_5,f2_1,f2_2,f2_3,f2_4]).
ring_size_5(f2,[f2_5,f2_1,f2_2,f2_3,f2_4]).
nitro(f2,[f2_1,f2_9,f2_10,f2_11]).
hetero_aromatic_5_ring(f3,[f3_5,f3_1,f3_2,f3_3,f3_4]).
ring_size_5(f3,[f3_5,f3_1,f3_2,f3_3,f3_4]).
nitro(f3,[f3_4,f3_8,f3_10,f3_9]).
hetero_aromatic_5_ring(f4,[f4_5,f4_1,f4_2,f4_3,f4_4]).
ring_size_5(f4,[f4_5,f4_1,f4_2,f4_3,f4_4]).
ring_size_5(f4,[f4_22,f4_18,f4_15,f4_17,f4_19]).
nitro(f4,[f4_4,f4_8,f4_10,f4_9]).
hetero_aromatic_5_ring(f5,[f5_5,f5_1,f5_2,f5_3,f5_4]).
ring_size_5(f5,[f5_5,f5_1,f5_2,f5_3,f5_4]).
ring_size_5(f5,[f5_9,f5_10,f5_6,f5_7,f5_8]).
nitro(f5,[f5_4,f5_16,f5_17,f5_18]).
hetero_aromatic_5_ring(f6,[f6_5,f6_1,f6_2,f6_3,f6_4]).
ring_size_5(f6,[f6_5,f6_1,f6_2,f6_3,f6_4]).
ring_size_5(f6,[f6_20,f6_16,f6_14,f6_15,f6_17]).
nitro(f6,[f6_4,f6_8,f6_10,f6_9]).
benzene(d189,[d189_6,d189_1,d189_2,d189_3,d189_4,d189_5]).
benzene(d189,[d189_9,d189_10,d189_11,d189_12,d189_4,d189_3]).
ring_size_6(d189,[d189_6,d189_1,d189_2,d189_3,d189_4,d189_5]).
ring_size_6(d189,[d189_9,d189_10,d189_11,d189_12,d189_4,d189_3]).
nitro(d189,[d189_12,d189_15,d189_21,d189_22]).
nitro(d189,[d189_10,d189_16,d189_23,d189_24]).
nitro(d189,[d189_5,d189_17,d189_19,d189_20]).
nitro(d189,[d189_1,d189_18,d189_25,d189_26]).
benzene(d192,[d192_6,d192_1,d192_2,d192_3,d192_4,d192_5]).
benzene(d192,[d192_4,d192_14,d192_13,d192_12,d192_11,d192_3]).
benzene(d192,[d192_19,d192_14,d192_13,d192_16,d192_17,d192_18]).
benzene(d192,[d192_25,d192_17,d192_16,d192_22,d192_23,d192_24]).
ring_size_6(d192,[d192_6,d192_1,d192_2,d192_3,d192_4,d192_5]).
ring_size_6(d192,[d192_4,d192_14,d192_13,d192_12,d192_11,d192_3]).
ring_size_6(d192,[d192_19,d192_14,d192_13,d192_16,d192_17,d192_18]).
ring_size_6(d192,[d192_25,d192_17,d192_16,d192_22,d192_23,d192_24]).
nitro(d192,[d192_12,d192_30,d192_31,d192_32]).
phenanthrene(d192,[[d192_6,d192_1,d192_2,d192_3,d192_4,d192_5],[d192_4,d192_14,d192_13,d192_12,d192_11,d192_3],[d192_19,d192_14,d192_13,d192_16,d192_17,d192_18]]).
benzene(d193,[d193_6,d193_1,d193_2,d193_3,d193_4,d193_5]).
benzene(d193,[d193_4,d193_14,d193_13,d193_12,d193_11,d193_3]).
benzene(d193,[d193_19,d193_14,d193_13,d193_16,d193_17,d193_18]).
benzene(d193,[d193_25,d193_17,d193_16,d193_22,d193_23,d193_24]).
ring_size_6(d193,[d193_6,d193_1,d193_2,d193_3,d193_4,d193_5]).
ring_size_6(d193,[d193_4,d193_14,d193_13,d193_12,d193_11,d193_3]).
ring_size_6(d193,[d193_19,d193_14,d193_13,d193_16,d193_17,d193_18]).
ring_size_6(d193,[d193_25,d193_17,d193_16,d193_22,d193_23,d193_24]).
nitro(d193,[d193_24,d193_30,d193_31,d193_32]).
phenanthrene(d193,[[d193_6,d193_1,d193_2,d193_3,d193_4,d193_5],[d193_4,d193_14,d193_13,d193_12,d193_11,d193_3],[d193_19,d193_14,d193_13,d193_16,d193_17,d193_18]]).
benzene(d195,[d195_6,d195_1,d195_2,d195_3,d195_4,d195_5]).
ring_size_6(d195,[d195_6,d195_1,d195_2,d195_3,d195_4,d195_5]).
nitro(d195,[d195_2,d195_12,d195_13,d195_14]).
benzene(d196,[d196_6,d196_1,d196_2,d196_3,d196_4,d196_5]).
benzene(d196,[d196_16,d196_11,d196_12,d196_13,d196_14,d196_15]).
ring_size_6(d196,[d196_6,d196_1,d196_2,d196_3,d196_4,d196_5]).
ring_size_6(d196,[d196_16,d196_11,d196_12,d196_13,d196_14,d196_15]).
ring_size_5(d196,[d196_4,d196_16,d196_11,d196_19,d196_3]).
nitro(d196,[d196_15,d196_22,d196_23,d196_24]).
benzene(e3,[e3_6,e3_1,e3_2,e3_3,e3_4,e3_5]).
ring_size_6(e3,[e3_6,e3_1,e3_2,e3_3,e3_4,e3_5]).
ring_size_5(e3,[e3_4,e3_12,e3_11,e3_10,e3_3]).
nitro(e3,[e3_6,e3_17,e3_18,e3_19]).
benzene(e4,[e4_6,e4_1,e4_2,e4_3,e4_4,e4_5]).
hetero_aromatic_5_ring(e4,[e4_9,e4_4,e4_3,e4_7,e4_8]).
ring_size_6(e4,[e4_6,e4_1,e4_2,e4_3,e4_4,e4_5]).
ring_size_5(e4,[e4_9,e4_4,e4_3,e4_7,e4_8]).
nitro(e4,[e4_6,e4_11,e4_12,e4_13]).
benzene(e5,[e5_6,e5_1,e5_2,e5_3,e5_4,e5_5]).
ring_size_6(e5,[e5_6,e5_1,e5_2,e5_3,e5_4,e5_5]).
nitro(e5,[e5_5,e5_12,e5_13,e5_14]).
benzene(e6,[e6_6,e6_1,e6_2,e6_3,e6_4,e6_5]).
hetero_aromatic_5_ring(e6,[e6_9,e6_4,e6_3,e6_7,e6_8]).
ring_size_6(e6,[e6_6,e6_1,e6_2,e6_3,e6_4,e6_5]).
ring_size_5(e6,[e6_9,e6_4,e6_3,e6_7,e6_8]).
nitro(e6,[e6_2,e6_11,e6_12,e6_13]).
benzene(e7,[e7_6,e7_1,e7_2,e7_3,e7_4,e7_5]).
hetero_aromatic_5_ring(e7,[e7_9,e7_4,e7_3,e7_7,e7_8]).
ring_size_6(e7,[e7_6,e7_1,e7_2,e7_3,e7_4,e7_5]).
ring_size_5(e7,[e7_9,e7_4,e7_3,e7_7,e7_8]).
nitro(e7,[e7_5,e7_10,e7_11,e7_12]).
benzene(e8,[e8_6,e8_1,e8_2,e8_3,e8_4,e8_5]).
hetero_aromatic_5_ring(e8,[e8_9,e8_4,e8_3,e8_7,e8_8]).
ring_size_6(e8,[e8_6,e8_1,e8_2,e8_3,e8_4,e8_5]).
ring_size_5(e8,[e8_9,e8_4,e8_3,e8_7,e8_8]).
nitro(e8,[e8_6,e8_10,e8_11,e8_12]).
benzene(e9,[e9_6,e9_1,e9_2,e9_3,e9_4,e9_5]).
ring_size_6(e9,[e9_6,e9_1,e9_2,e9_3,e9_4,e9_5]).
nitro(e9,[e9_5,e9_11,e9_13,e9_14]).
benzene(e10,[e10_6,e10_1,e10_2,e10_3,e10_4,e10_5]).
hetero_aromatic_5_ring(e10,[e10_9,e10_4,e10_3,e10_7,e10_8]).
ring_size_6(e10,[e10_6,e10_1,e10_2,e10_3,e10_4,e10_5]).
ring_size_5(e10,[e10_9,e10_4,e10_3,e10_7,e10_8]).
nitro(e10,[e10_2,e10_10,e10_11,e10_12]).
benzene(e11,[e11_6,e11_1,e11_2,e11_3,e11_4,e11_5]).
ring_size_6(e11,[e11_6,e11_1,e11_2,e11_3,e11_4,e11_5]).
nitro(e11,[e11_5,e11_10,e11_11,e11_12]).
benzene(e12,[e12_6,e12_1,e12_2,e12_3,e12_4,e12_5]).
ring_size_6(e12,[e12_6,e12_1,e12_2,e12_3,e12_4,e12_5]).
nitro(e12,[e12_5,e12_10,e12_11,e12_12]).
nitro(e12,[e12_4,e12_14,e12_15,e12_16]).
benzene(e13,[e13_6,e13_1,e13_2,e13_3,e13_4,e13_5]).
ring_size_6(e13,[e13_6,e13_1,e13_2,e13_3,e13_4,e13_5]).
nitro(e13,[e13_5,e13_10,e13_11,e13_12]).
benzene(e14,[e14_6,e14_1,e14_2,e14_3,e14_4,e14_5]).
ring_size_6(e14,[e14_6,e14_1,e14_2,e14_3,e14_4,e14_5]).
nitro(e14,[e14_5,e14_9,e14_10,e14_11]).
benzene(e15,[e15_6,e15_1,e15_2,e15_3,e15_4,e15_5]).
benzene(e15,[e15_4,e15_14,e15_13,e15_12,e15_11,e15_3]).
benzene(e15,[e15_19,e15_13,e15_12,e15_16,e15_17,e15_18]).
benzene(e15,[e15_25,e15_19,e15_18,e15_22,e15_23,e15_24]).
ring_size_6(e15,[e15_6,e15_1,e15_2,e15_3,e15_4,e15_5]).
ring_size_6(e15,[e15_4,e15_14,e15_13,e15_12,e15_11,e15_3]).
ring_size_6(e15,[e15_19,e15_13,e15_12,e15_16,e15_17,e15_18]).
ring_size_6(e15,[e15_25,e15_19,e15_18,e15_22,e15_23,e15_24]).
nitro(e15,[e15_11,e15_30,e15_31,e15_32]).
anthracene(e15,[[e15_6,e15_1,e15_2,e15_3,e15_4,e15_5],[e15_4,e15_14,e15_13,e15_12,e15_11,e15_3],[e15_19,e15_13,e15_12,e15_16,e15_17,e15_18]]).
benzene(e16,[e16_6,e16_1,e16_2,e16_3,e16_4,e16_5]).
ring_size_6(e16,[e16_6,e16_1,e16_2,e16_3,e16_4,e16_5]).
nitro(e16,[e16_5,e16_9,e16_10,e16_11]).
benzene(e17,[e17_6,e17_1,e17_2,e17_3,e17_4,e17_5]).
benzene(e17,[e17_4,e17_14,e17_13,e17_12,e17_11,e17_3]).
benzene(e17,[e17_19,e17_13,e17_12,e17_16,e17_17,e17_18]).
ring_size_6(e17,[e17_6,e17_1,e17_2,e17_3,e17_4,e17_5]).
ring_size_6(e17,[e17_4,e17_14,e17_13,e17_12,e17_11,e17_3]).
ring_size_6(e17,[e17_19,e17_13,e17_12,e17_16,e17_17,e17_18]).
nitro(e17,[e17_14,e17_24,e17_25,e17_26]).
anthracene(e17,[[e17_6,e17_1,e17_2,e17_3,e17_4,e17_5],[e17_4,e17_14,e17_13,e17_12,e17_11,e17_3],[e17_19,e17_13,e17_12,e17_16,e17_17,e17_18]]).
benzene(e18,[e18_6,e18_1,e18_2,e18_3,e18_4,e18_5]).
ring_size_6(e18,[e18_6,e18_1,e18_2,e18_3,e18_4,e18_5]).
nitro(e18,[e18_5,e18_9,e18_10,e18_11]).
benzene(e19,[e19_6,e19_1,e19_2,e19_3,e19_4,e19_5]).
benzene(e19,[e19_9,e19_10,e19_4,e19_3,e19_7,e19_8]).
carbon_6_ring(e19,[e19_9,e19_14,e19_13,e19_12,e19_11,e19_8]).
ring_size_6(e19,[e19_6,e19_1,e19_2,e19_3,e19_4,e19_5]).
ring_size_6(e19,[e19_9,e19_10,e19_4,e19_3,e19_7,e19_8]).
ring_size_6(e19,[e19_9,e19_14,e19_13,e19_12,e19_11,e19_8]).
nitro(e19,[e19_10,e19_15,e19_16,e19_17]).
benzene(e20,[e20_6,e20_1,e20_2,e20_3,e20_4,e20_5]).
benzene(e20,[e20_9,e20_10,e20_4,e20_3,e20_7,e20_8]).
carbon_6_ring(e20,[e20_9,e20_14,e20_13,e20_12,e20_11,e20_8]).
ring_size_6(e20,[e20_6,e20_1,e20_2,e20_3,e20_4,e20_5]).
ring_size_6(e20,[e20_9,e20_10,e20_4,e20_3,e20_7,e20_8]).
ring_size_6(e20,[e20_9,e20_14,e20_13,e20_12,e20_11,e20_8]).
nitro(e20,[e20_10,e20_15,e20_16,e20_17]).
benzene(e21,[e21_9,e21_10,e21_4,e21_3,e21_7,e21_8]).
benzene(e21,[e21_9,e21_10,e21_17,e21_16,e21_15,e21_14]).
benzene(e21,[e21_9,e21_14,e21_13,e21_12,e21_11,e21_8]).
benzene(e21,[e21_20,e21_15,e21_14,e21_13,e21_18,e21_19]).
carbon_6_ring(e21,[e21_6,e21_1,e21_2,e21_3,e21_4,e21_5]).
ring_size_6(e21,[e21_6,e21_1,e21_2,e21_3,e21_4,e21_5]).
ring_size_6(e21,[e21_9,e21_10,e21_4,e21_3,e21_7,e21_8]).
ring_size_6(e21,[e21_9,e21_10,e21_17,e21_16,e21_15,e21_14]).
ring_size_6(e21,[e21_9,e21_14,e21_13,e21_12,e21_11,e21_8]).
ring_size_6(e21,[e21_20,e21_15,e21_14,e21_13,e21_18,e21_19]).
nitro(e21,[e21_7,e21_21,e21_22,e21_23]).
phenanthrene(e21,[[e21_9,e21_10,e21_4,e21_3,e21_7,e21_8],[e21_9,e21_10,e21_17,e21_16,e21_15,e21_14],[e21_20,e21_15,e21_14,e21_13,e21_18,e21_19]]).
ball3(e21,[[e21_9,e21_14,e21_13,e21_12,e21_11,e21_8],[e21_9,e21_10,e21_4,e21_3,e21_7,e21_8],[e21_9,e21_10,e21_17,e21_16,e21_15,e21_14]]).
ball3(e21,[[e21_9,e21_14,e21_13,e21_12,e21_11,e21_8],[e21_9,e21_10,e21_17,e21_16,e21_15,e21_14],[e21_20,e21_15,e21_14,e21_13,e21_18,e21_19]]).
benzene(e22,[e22_9,e22_10,e22_4,e22_3,e22_7,e22_8]).
benzene(e22,[e22_9,e22_14,e22_13,e22_12,e22_11,e22_8]).
benzene(e22,[e22_18,e22_14,e22_13,e22_15,e22_16,e22_17]).
carbon_6_ring(e22,[e22_6,e22_1,e22_2,e22_3,e22_4,e22_5]).
ring_size_6(e22,[e22_6,e22_1,e22_2,e22_3,e22_4,e22_5]).
ring_size_6(e22,[e22_9,e22_10,e22_4,e22_3,e22_7,e22_8]).
ring_size_6(e22,[e22_9,e22_14,e22_13,e22_12,e22_11,e22_8]).
ring_size_6(e22,[e22_18,e22_14,e22_13,e22_15,e22_16,e22_17]).
nitro(e22,[e22_7,e22_19,e22_20,e22_21]).
benzene(e23,[e23_6,e23_1,e23_2,e23_3,e23_4,e23_5]).
benzene(e23,[e23_9,e23_10,e23_4,e23_3,e23_7,e23_8]).
benzene(e23,[e23_9,e23_14,e23_13,e23_12,e23_11,e23_8]).
carbon_6_ring(e23,[e23_18,e23_14,e23_13,e23_15,e23_16,e23_17]).
ring_size_6(e23,[e23_6,e23_1,e23_2,e23_3,e23_4,e23_5]).
ring_size_6(e23,[e23_9,e23_10,e23_4,e23_3,e23_7,e23_8]).
ring_size_6(e23,[e23_9,e23_14,e23_13,e23_12,e23_11,e23_8]).
ring_size_6(e23,[e23_18,e23_14,e23_13,e23_15,e23_16,e23_17]).
nitro(e23,[e23_7,e23_19,e23_20,e23_21]).
anthracene(e23,[[e23_9,e23_14,e23_13,e23_12,e23_11,e23_8],[e23_9,e23_10,e23_4,e23_3,e23_7,e23_8],[e23_6,e23_1,e23_2,e23_3,e23_4,e23_5]]).
benzene(e24,[e24_6,e24_1,e24_17,e24_16,e24_15,e24_14]).
benzene(e24,[e24_3,e24_2,e24_22,e24_23,e24_24,e24_25]).
benzene(e24,[e24_9,e24_10,e24_5,e24_4,e24_7,e24_8]).
carbon_6_ring(e24,[e24_6,e24_1,e24_2,e24_3,e24_4,e24_5]).
ring_size_6(e24,[e24_6,e24_1,e24_2,e24_3,e24_4,e24_5]).
ring_size_6(e24,[e24_6,e24_1,e24_17,e24_16,e24_15,e24_14]).
ring_size_6(e24,[e24_3,e24_2,e24_22,e24_23,e24_24,e24_25]).
ring_size_6(e24,[e24_9,e24_10,e24_5,e24_4,e24_7,e24_8]).
nitro(e24,[e24_10,e24_30,e24_31,e24_32]).
benzene(e25,[e25_9,e25_10,e25_4,e25_3,e25_7,e25_8]).
benzene(e25,[e25_9,e25_10,e25_17,e25_16,e25_15,e25_14]).
benzene(e25,[e25_9,e25_14,e25_13,e25_12,e25_11,e25_8]).
benzene(e25,[e25_20,e25_15,e25_14,e25_13,e25_18,e25_19]).
carbon_6_ring(e25,[e25_6,e25_1,e25_2,e25_3,e25_4,e25_5]).
ring_size_6(e25,[e25_6,e25_1,e25_2,e25_3,e25_4,e25_5]).
ring_size_6(e25,[e25_9,e25_10,e25_4,e25_3,e25_7,e25_8]).
ring_size_6(e25,[e25_9,e25_10,e25_17,e25_16,e25_15,e25_14]).
ring_size_6(e25,[e25_9,e25_14,e25_13,e25_12,e25_11,e25_8]).
ring_size_6(e25,[e25_20,e25_15,e25_14,e25_13,e25_18,e25_19]).
nitro(e25,[e25_7,e25_21,e25_22,e25_23]).
phenanthrene(e25,[[e25_9,e25_10,e25_4,e25_3,e25_7,e25_8],[e25_9,e25_10,e25_17,e25_16,e25_15,e25_14],[e25_20,e25_15,e25_14,e25_13,e25_18,e25_19]]).
ball3(e25,[[e25_9,e25_14,e25_13,e25_12,e25_11,e25_8],[e25_9,e25_10,e25_4,e25_3,e25_7,e25_8],[e25_9,e25_10,e25_17,e25_16,e25_15,e25_14]]).
ball3(e25,[[e25_9,e25_14,e25_13,e25_12,e25_11,e25_8],[e25_9,e25_10,e25_17,e25_16,e25_15,e25_14],[e25_20,e25_15,e25_14,e25_13,e25_18,e25_19]]).
benzene(e26,[e26_6,e26_1,e26_2,e26_3,e26_4,e26_5]).
benzene(e26,[e26_9,e26_10,e26_4,e26_3,e26_7,e26_8]).
benzene(e26,[e26_9,e26_14,e26_13,e26_12,e26_11,e26_8]).
benzene(e26,[e26_18,e26_14,e26_13,e26_15,e26_16,e26_17]).
ring_size_6(e26,[e26_6,e26_1,e26_2,e26_3,e26_4,e26_5]).
ring_size_6(e26,[e26_9,e26_10,e26_4,e26_3,e26_7,e26_8]).
ring_size_6(e26,[e26_9,e26_14,e26_13,e26_12,e26_11,e26_8]).
ring_size_6(e26,[e26_18,e26_14,e26_13,e26_15,e26_16,e26_17]).
nitro(e26,[e26_7,e26_31,e26_32,e26_33]).
anthracene(e26,[[e26_9,e26_14,e26_13,e26_12,e26_11,e26_8],[e26_9,e26_10,e26_4,e26_3,e26_7,e26_8],[e26_6,e26_1,e26_2,e26_3,e26_4,e26_5]]).


methyl(d10,[d10_9,d10_17,d10_24,d10_25,d10_26]).
methyl(d166,[d166_29,d166_31,d166_32,d166_33,d166_34]).
methyl(d178,[d178_11,d178_15,d178_16,d178_17,d178_18]).
methyl(d8,[d8_26,d8_27,d8_28,d8_29,d8_30]).
methyl(d97,[d97_12,d97_22,d97_23,d97_24,d97_25]).
methyl(d158,[d158_26,d158_27,d158_30,d158_31,d158_32]).
methyl(d42,[d42_8,d42_10,d42_18,d42_19,d42_20]).
methyl(d89,[d89_7,d89_10,d89_18,d89_19,d89_20]).
methyl(d120,[d120_7,d120_10,d120_18,d120_19,d120_20]).
methyl(d130,[d130_11,d130_12,d130_13,d130_14,d130_15]).
methyl(d138,[d138_8,d138_10,d138_18,d138_19,d138_20]).
methyl(d147,[d147_8,d147_10,d147_17,d147_18,d147_19]).
methyl(d7,[d7_16,d7_21,d7_22,d7_23,d7_24]).
methyl(d14,[d14_9,d14_10,d14_18,d14_19,d14_20]).
methyl(d36,[d36_12,d36_17,d36_18,d36_19,d36_20]).
methyl(d39,[d39_19,d39_20,d39_21,d39_22,d39_23]).
methyl(d55,[d55_14,d55_17,d55_18,d55_19,d55_20]).
methyl(d78,[d78_5,d78_21,d78_22,d78_23,d78_24]).
methyl(d88,[d88_11,d88_12,d88_13,d88_14,d88_15]).
methyl(d123,[d123_15,d123_19,d123_20,d123_21,d123_22]).
methyl(d123,[d123_5,d123_24,d123_25,d123_26,d123_27]).
methyl(d150,[d150_4,d150_12,d150_19,d150_20,d150_21]).
methyl(d155,[d155_10,d155_11,d155_12,d155_13,d155_14]).
methyl(d168,[d168_11,d168_12,d168_13,d168_14,d168_15]).
methyl(d182,[d182_11,d182_16,d182_17,d182_18,d182_19]).
methyl(d197,[d197_7,d197_10,d197_18,d197_19,d197_20]).
methyl(e3,[e3_10,e3_13,e3_14,e3_15,e3_16]).
methyl(e4,[e4_7,e4_10,e4_19,e4_20,e4_21]).
methyl(e6,[e6_7,e6_10,e6_19,e6_20,e6_21]).
%end(model(s1)).

