/* File:      view_sys.P
** Author(s): David S. Warren
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) David S. Warren 2016
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: altcdf.P,v 1.6 2010-08-19 15:03:38 warren Exp $
** 
*/

:- comment(title,"ViewSys").
:- comment(subtitle,"A View Maintenance System Library").
:- comment(author,"David S. Warren").
:- comment(copyright,"Copyright 2016 David S. Warren, XSB, Inc.").
%%:- comment(version(0*1+1,2016/06/14,16:51*00+'EST'),"Initial system; David S. Warren").

:- export
	viewsys_view_status/3,	 % viewsys_view_status(VSDir,View:VInst,-Status)
	viewsys_status/1,	 % viewsys_status(VSDir)
	viewsys_status/2,	 % viewsys_status(VSDir,Option)
	print_viewsys/1,	 % print_viewsys(VSDir)
	check_viewsys/1,	 % check_viewsys(VSDir)
	generate_new_instance/2, % generate_new_instance(VSDir,VInst)
	update_instance/2,	 % update_instance(VSDir,VInst)
	delete_instance/2,	 % delete_instance(VSDir,VInst)
	update_views/4,		 % update_views(VSDir,ViewInstList,ProcName,NProcs)
	update_views/5,		 % update_views(VSDir,ViewInstList,ProcName,NProcs,OutStream)
	invalidate_all_instances/1, % invalidate_all_instances(VSDir)
	invalidate_view_instances/2, % invalidate_view_instances(VSDir,[View:VInst])
	reset_unfinished/2,	% reset_unfinished(VSDir,ProcName)
	show_failed/2,		% show_failed(VSDir,VInst)
	reset_failed/2,		% reset_failed(VSDir,VInst)
	expand_views/1,		% expand_views(VSDir)
	set_viewsys_uservar/2,	% set_viewsys_uservar('$varname$','varvalue')
	generate_required_dirs/2, % generate_required_dirs(Substitution,XSB_LOGFILES)
	copy_required_files/2.	% copy_required_files(VSDir,FromToSubs)

:- export logfile_file/2, logfile_directory/2.

:- dynamic global_viewsys_uservar/2.  % for external user to set.

:- import create_lockfile/1, rm/1, sys_mkdir/2, process_control/2, process_status/2 from shell.
:- import rename/2 from shell.
:- import concat_atom/2, concat_atom/3, term_to_atom/3 from string.
:- import member/2, select/3, length/2, append/3, for/3 from basics.
:- import fmt_write/3, file_time/2, flush_output/1 from file_io.
:- import local_datime/1, call_c/1, file_exists/1 from standard.
:- import parsort/4 from machine.
:- import xsb_configuration/2 from xsb_configuration.
:- import predicate_defined/1 from assert.
:- import misc_error/1 from error_handler.

%% backwards compatibility
viewV(View,Type,ViewNameTemplate,InputViews,Opts,ShCmd) :-
	view(View,Type,ViewNameTemplate,_Stdout,InputViews,Opts,ShCmd).
viewV(View,Type,ViewNameTemplate,InputViews,Opts,ShCmd) :-
	view(View,Type,ViewNameTemplate,InputViews,Opts,ShCmd).

consViewV(ConsViewName, CheckedViewName, FileTemplate, Inputs, ShCmd) :-
	consView(ConsViewName, CheckedViewName, FileTemplate, _StdoutFileTemplate, Inputs, ShCmd).
consViewV(ConsViewName, CheckedViewName, FileTemplate, Inputs, ShCmd) :-
	consView(ConsViewName, CheckedViewName, FileTemplate, Inputs, ShCmd).
%%

:- comment(summary, "The ViewSys package supports workflows that can
be structured as a DAG of view definitions.  ViewSys allows a user to
define such workflows and manage their execution and maintenance.  At
this point it does not provide support for incremental view
maintenance, only full regeneration of views.").

:- comment(module," @section{Overview} A View System supports a DAG of
views.  Most simply a view is a file (or table) that is generated by a
process applied to a set of input files.  A view that has no inputs is
called a 'base view'.

More precisely we can think of a view as a data source.  Base views
are data sources from outside the system.  A non-base view is a data
source that is determined (and computed) by its process applied to its
input data sources.  The process should be idempotent, so normally it
creates a new file (or table).  (The system does allow non-idempotent
updates, but they may be problematic under some circumstances.)

A view system workflow (@bf{ViewSys} for short) describes the names of
the views, their input views, the command to be run to generate a view
from its inputs, etc.  A particular @bf{instance} of a ViewSys is
determined by the specific external data sources associated with the
base views of the ViewSys.  Instances are used to support the
situation in which the same workflow needs to be applied to different
sets of base views.  It is useful to give names to such instances,
usually indicating the external source of the base data sources.  Many
view systems will have only one instance.

Another useful component of a view system is what is called a
consistency view.  The purpose of a consistency view is to check to
see whether a regular view is 'consistent'.  The command for a
consistency view should return non-zero if the view instance is not
deemed to be consistent.

The view system will run consistency views where applicable and will
not use a view as input to another view that it supports if it is
deemed not consistemt.  A single view may have zero or more
consistency views associated with it.

@section{An Example}

Consider a situation in which we are collecting data from four
institutions of higher education and want to integrate that data into
a dataset that allows us to make coherent queries across the data from
all institutions.  We might want to answer questions about the
possibility of transferring classes between the schools, or perhaps
whether a student might take a class scheduled at one that would be
equivalent to one at another, if schedules don't conflict.

Say we have two community colleges, AJC and ACC, and two 4-year
colleges, UC and UD.  And we collect information from each of them
concerning, say, currently scheduled classes at their institutions.

To integrate data from all four institutions, we might create the
following view system:
@begin{verbatim}
    clnajc
ajc ------> ajc-cleaned--+
                         | comb2
    clnacc               |------>2-year-info--+
acc ------> acc-cleaned--+                    |
                                              |  comb24
    clnuc                                     |--------> 2-4-info
UC -------> uc-cleaned--+                     |
                        | comb4               |
    clnud               |------>4-year-info---+
UD -------> ud-cleaned--+
@end{verbatim}


For each raw-data input from an institution, we have a process to
``clean'' that data (indicated in the diagram by a name cln<inst>), that
generates a file (view) containing ``cleaned''and ``standardized'' data
(indicated in the diagram by <inst>-cleaned.)  Then we have a process,
comb2, that combines the two cleaned community college datasets to
create a view, 2-year-info; and another, comb4, that combines the two
cleaned 4-year college datasets to create the view, 4-year-info.  And
finally, we have a process, comb24, that takes those two views and
generates a fully combined dataset (i.e., view), 2-4-info.

This viewsys system has 11 views, 4 of which are base views and 7
derived views.  And it has 7 processes, one to generate each derived
view.

We can imagine what these processes might do: the cleaning processes
would do institution-specific transformations of the input data, maybe
standardizing names of equivalent classes; inferring a new variable of
the level of the classe (intro, intermediate, advanced) from the class
naming/numbering conventions of the particular institution;
standardizing class-time representations given different scheduling
conventions; etc.  The comb<?> processes might simply project and union
their inputs, but they could easily perform other more complex
inferences and transformations.

We could easily imagine having other (mostly static) data inputs (not
shown here) to these cleaning processes that provide
institution-specific information necessary to do such transformations.
We can also imagine that we have another process that uses, say, the
2-year-info view, to combine it with other information we've gleaned
from 2-year colleges to provide another view that can answer other
questions of interest.

We can imagine that the datasets we get from the source institutions
arrive at different times but we want the best data in the coherent
views to be available to any query.  So if a new file from, say, UC,
shows up, we need only run the processes clnuc, comb4, and comb24 to
be sure that all data is up to date.

@section{The ViewSys Data Model}

A ViewSys workflow is described by a set of facts of the following
predicates.  Users should put the appropriate facts for these
predicates that define their view system into a file named
@file{viewsys_view_info.P}.

@subsection{View Framework Model}

For each view (base or derived), there is a view/6 fact that describes
it:

@noindent view(View,Type,ViewNameTemplate,[InputViews],[Opts],ShCmd)
where:
@begin{itemize}
@item @var{View} is the name of the view;

@item @var{Type} is file, dir(<FileNames>) (or maybe in the future
db(...)?).  It is 'file' if the view is stored in a file (that is
generated by the @var{ShCmd}).  It is 'dir(<FileNames>)' if the view
is stored in multiple files in a directory.  <FileNames> are the
(relative) names of the files that store the view in that directory
(instance). It may also be 'table' if the view is a database table.

@item @var{ViewNameTemplate} is the path template for where instance
versions are stored.  This template string normally contains the
pattern variable $INSTANCE$ which will be replaced by the instance
name to obtain the filename (or directory name) of an instance of this
view.  (If the viewsys will ever have only one instance, the
$INSTANCE$ variable is not required.)  A file template may also
contain user-defined pattern variables of the form '$USERVARNAME$'
where USERVARNAME is any upper-case letter sequence (except those
reserved for viewsys system variables.)  User-defined pattern variable
values are defined in facts of the form
viewsys_uservar('$USERVARNAME$',VarValueString).  When instantiated by
an instance name and user-variable values, this will be the filename
that contains the view contents, or a directory name that contains the
files containing the view contents (or the name of a database table.)

@item @var{[InputViews]} is a list of the names of views that this
view directly depends on, i.e., the inputs needed to generate this
view.  This is an empty list for base views.  Normally these input
view indicators are atoms for which there is another @pred{view/6}
fact that describes it.  However, if that view generates a directory
and the input to this view is a file in that directory, then that
filename should be put as an argument to the view atom.  E.g., if the
view, m_view, generates a directory and several files in it and this
view needs to use the file 'first_file.P' from that directory, then
the input view indicator in this list should be the term
m_view('first_file.P').

@item @var{[Opts]} is a list of options. The possible options are:

@begin{itemize}				       

@item @tt{split(N)} where @tt{N} is a positive integer.  This tells
@tt{viewsys} to split the first input view file into @tt{N}
subfiles; to run this command on each of those subfiles; and to
concatenate all the resulting subfiles back together to get the output
file for this view.  Of course, this is only appropriate for view
commands for which this process gives the same answer as running it on
the large unsplit file.  When the command satisfies this property,
this option can allow the records in a large file to be processed in
parallel.

If this option is used, the user must first run
@pred{expand_views(ViewDir)} to generate a viewsys file that
implements the splitting.  It will move the @file{viewsys_view_info.P}
file to @file{viewsys_view_orig_info.P} replace it with a modified
version of the file that will drive the @tt(viewsys) processing.  (If
the file @file{viewsys_view_orig_info.P} exists, the operation will
indicate an error, in order to protect against inadvertantly
overwriting the original @file{viewsys_view_info.P} file.)

@end{itemize}				       

@item @var{ShCmd} is the shell command to execute to generate the view
instance from its input view instances. (Ignored for base views.)  The
shell command can be in one of two forms:

@begin{enumerate}

@item a string containing metavariables of the form $INP1$, $INP2$,
..., and $OUT$, which will be replaced by the filenames of the input
view instance files/directories and the output view instance
file/directory, respectively; or

@item a string containing the metavariables $INPUTFILES$ and
$OUTPUTFILE$, which will be replaces with the sequence of input
filenames and the output filename, respectively, where each filename
is enclosed in double-quotes.  This is often appropreate for shell
commands.  If the shell string doesn't contain any of the
metavariables, then it is treated as if it were: '<ShCmd> $INPUTFILES$
$OUTPUTFILE$'.

@end{enumerate}

@end{itemize}

User-defined syntactic variables can be used in filename templates and
in shell command templates to make it easier to define filenames and
commands.  The predicate @pred{viewsys_uservar/2} is used to define
user variables, and facts for this predicate should be placed in the
@file{viewsys_view_info.P} file.  For example, assume the user adds
the following facts to that file:

@begin{verbatim}
viewsys_uservar('$DATA_DIR$','C:/userfiles/project1/data').
viewsys_uservar('$SCRIPT_LIB$','c:/userfiles/project1/scripts').
@end{verbatim}

With these declarations in a @file{viewsys_view_info.P} file, a file
template string could be of the form '$DATA_DIR$/data_file_13', which
after replacement of the syntactic variable by its value would refer
to the file named 'C:/userfiles/project1/data/data_file_13'.  A shell
command string could be 'sh $SCRIPT_LIB$/script_cc.sh', which after
replacements would cause the command 'sh
c:/userfiles/project1/scripts/script_cc.sh' to be run.  User variables
are normally defined at the beginning of the view file and can be used
to allow locations to be easily changed.  The value of a user variable
may contain another user variable, but, of course, cycles are not
permitted.

The user must define a uservar of $STDOUTFILE$ which is the filename
into which the stdout streams from the execution of a view generation
will be put.  The user should use the $INSTANCE$ and $VIEW$ variables
to make it unique for each output stream.


For each consistency view, there is a consView/5 fact:

@noindent consView(ConsViewName, CheckedViewName, FileTemplate, [Inputs], ShCmd)
where

@begin{itemize}
@item @var{ConsViewName} is the name of the consistency view.

@item @var{ViewName} is the name of the view this view checks.

@item @var{FileTemplate} is the template for the output file for this
consistency check.  This file may be used to provide information as to
why the consistency check failed (or passed.)

@item @var{[InputViews]} is a list of parameter input views (maybe empty)

@item @var{ShCmd} is the shell command the executes the consistency
check.  The inputs are the the filename containing the view instance
to be checked followd by the input view file instances.  The output is
the output file instance.  These parameters are processed similarly to
the processing for shell-commands for regular views.

@end{itemize}

@subsection{View Instance Model}

A ViewSys Instance is a particular instantiation of a ViewSys workflow
that is identified by a name, usually indicating the source of the
base views.  Of course, the files (directories) that contain instances
of views must all be distinct.

View instances are described by another set of facts, which are stored
in a file named @file{viewsys_instance_info.P}.  Whereas the user is
responsible for creating the @file{viewsys_view_info.P} file, viewsys
creates and maintains the @file{viewsys_instance_info.P} file in
response to viewsys commands entered by the user.

For each view instance (base or derived), there is a viewInst/5 fact:

@noindent viewInst(View,InstName,Status,Date,Began)
where:
@begin{itemize}

@item @var{View} is the name of a view;

@item @var{InstName} is the name of the instance;
	   
@item @var{Status} is the status of this view instance
@tt{not_generated}, @tt{being_generated(ProcName)}, @tt{generated},
@tt{generation_failed}. (For base view instances this is always
@tt{generated}.)

@item @var{Date} is the date-time the view instance was generated.
(Better? the filetime of the base view last used to regenerate view
instances. Not used for non-base views.)

@item @var{Began} is the date-time at which the generation of this
view began.  (This is the same as Date above for base view instances.)
It is used to estimate how long it will take to generate this view
output given its inputs.

@end{itemize}

For each consistency view instance, there is a consViewInst/5 fact:

@noindent consViewInst(ConsViewName, InstName, Status, Date, Began)
where;

@begin{itemize}

@item @var{ConsViewName} is the name of the consistency view.

@item @var{Status} is this consistency view, same as for viewInst
status.

@item @var{Date} is the date-time the check was generated.

@item @var{Began} is the date-time at which the generation of this
view began.

@end{itemize}

The ViewSys relations, @pred{view/6}, @pred{consView/5}, and
@pred{viewOrig/6}, are stored in the file named
@file{viewsys_view_info.P}.  It is read for most commands, but not
updated.  (Only @pred{expand_views/1} generates this file from the
file namsd @file{viewsys_view_orig_info.P}.)  @pred{viewInst/5}, and
@pred{consViewInst/5} are stored in the file named
@file{viewsys_instance_info.P}, and the directory containing these
files is explicitly provided to predicates that need to operate on it.
The contents of the files are Prolog terms in canonical form.

A lockfile (named @file{lock_view} in the viewsys directory) is
obtained whenever these files are read, and it is kept until reading
and rewriting (if necessary) is completed.

@section{Using ViewSys}

The viewsys system is normally used as follows.  The user creates a
directory to hold the viewsys information.  She creates a file
@file{viewsys_view_info.P} in this directory containing the desired
@pred{view/6}, and @pred{consView/5} facts that describe the desired
view system.  Then the user consults the viewsys.P package, and runs
@pred{check_viewsys/1} to report any obvious inconsistencies in the
view system specified in the file @file{viewsys_view_info.P}.  After
the check passes, if any views have the @tt{split(N)} option, the user
should copy the @file{viewsys_view_info.P} file to a file named
@file{viewsys_orig_view_info.P} and then run @pred{expand_views/1} to
generate the appropriate file @file{viewsys_view_info.P} to contain
the views necessary to split, execute and combine the results.  This
will overwrite the @file{viewsys_view_info.P} file.  (From then on,
should the viewsys need to be modified, the user should edit the
@file{viewsys_orig_view_info.P} file, and rerun @pred{expand_views/1}
to regenerate the @file{viewsys_view_info.P} file.)  The user will
then run @pred{generate_view_instance/2} to generate an instance (or
instances) of the view system into the file
@file{viewsys_instance_info.P}.  After that the user will run
@pred{update_views/4} to run the workflow to generate all the view
contents.  Then the user checks the generated logging to determine if
there were any errors.  If so, the user corrects the programs (the
viewsys specification, whatever), executes @pred{reset_failed/2} and
reruns @pred{update_views/4}.  The user can also use
@pred{viewsys_status/1} to determine what the state of the view system
is, and to determine what needs to be fixed and what needs to be
rerun.  If the execution of @pred{update_views/4} is aborted or
somehow does not complete, the user can run @pred{reset_unfinished/2}
to reset the views that were in process, so that a subsequent
@pred{update_views/4} will try to recompute those unfinished
computations.

@section{Ideas for Possible Future Extensions}
	   
It may be useful to somehow associate or connect multiple view
systems.  This might support a base view in one ViewSys that is
defined in another ViewSys framework.

Perhaps we should support annotations/options to indicate how/when to
delete versions of intermediate views.

We might explore the integration of incrementally maintained views, by
adding difference files, and generating difference sets to be applied
to the old view.  This will probably initially have to be constrained
to views whose increments can be computed from the inserts/deletes to
a single input file.  " ).

:- comment(generate_new_instance(ViewSys,VInst), "
@pred{generate_new_instance(+ViewSys,+VInst)} creates a brand new
instance of the view system @var{ViewSys} named VInst.  It generates
new viewInst/5 facts for every view (base and derived) according to
the file templates defined in the baseView/4, and view/6 facts of the
ViewSys.  @var{VInst} may be a list of instance names, in which case
initial instances are created for each one."  ).

generate_new_instance(ViewSys,VInstList0) :-
	generate_new_instance(ViewSys,VInstList0,new).

:- comment(update_instance(ViewSys,VInst), "
@pred{update_instance(+ViewSys,+VInst)} updates an instance of the
view system @var{ViewSys} named VInst.  It is similar to
@pred{generate_new_instance/2} but doesn't change existing instance
records.  It generates a new @pred{viewInst/5} (or
@pred{consViewInst/5}) fact for every view (base and derived) that
doesn't already exist in the @file{viewsys_instance_info.P} file.  It
doesn't change instances that already exisit, thus preserving their
statuses and process times.  "  ).

update_instance(ViewSys,VInstList0) :-
	generate_new_instance(ViewSys,VInstList0,update).

generate_new_instance(ViewSys,VInstList0,NewUpd) :-
	(atom(VInstList0)
	 ->	VInstList = [VInstList0]
	 ;	VInstList = VInstList0
	),
	load_viewfile_for_update(ViewSys),
	local_datime(Datime),
	(do_all
	 member(VInst,VInstList),
	 retractall(viewInst(_,VInst,_,_,_)),
	 retractall(consViewInst(_,VInst,_,_,_)),
	 generate_all_vinsts(VInst,NewUpd,Datime)
	),
	save_viewfile(ViewSys).

generate_all_vinsts(VInst,NewUpd,Datime) :-
	(do_all
	 viewV(View,_Type0,FileTemplate,_ViewInputList0,_Opts,_Cmd0),
	 generate_vinst(VInst,Datime,View,FileTemplate,NewUpd)
	),
	(do_all
	 consViewV(View,_Type1,FileTemplate,_ViewInputList1,_Cmd1),
	 generate_const_vinst(VInst,Datime,View,NewUpd)
	).

%% generate viewInst facts for base and derived views
generate_vinst(VInst,Datime,View,FileTemplate,NewUpd) :-
	(base_view(View)
	 ->	(str_match('$INSTANCE$',FileTemplate,forward,_,_)
		 ->	GVInst = VInst
		 ;	\+ viewInst(View,all,_Stat0,_Dt,_Beg),
			GVInst = all
		),
		Status = generated
	 ;	GVInst = VInst,
		Status = not_generated
	),
	(NewUpd == update,
	 viewInst(View,VInst,_OStatus,_ODatime,_ODTBegin)
	 ->	true
	 ;	retractall(viewInst(View,VInst,_,_,_)),
		assert(viewInst(View,VInst,Status,Datime,Datime))
	 ).

base_view(View) :-
	viewV(View,_,_,[],_,_).

generate_const_vinst(VInst,Datime,View,NewUpd) :-
	(NewUpd == update,
	 consViewInst(View,VInst,_OStatus,_ODatime,_ODTBegin)
	 ->	true
	 ;	retractall(consViewInst(View,VInst,_,_,_)),
		assert(consViewInst(View,VInst,not_generated,Datime,datime(0,0,0,0,0,0)))
	).

:- comment(delete_instance(ViewSys,VInst), "
@pred{delete_instance(+ViewSys,+VInst)} removes an entire instance
from the view system.  Any files of view contents that have been
generated remain; only information concerning this instance in the
@file{viewsys_instance_info.P} file is removed, so these view
instances are no longer maintained.").

delete_instance(ViewSys,VInst) :-
	load_viewfile_for_update(ViewSys),
	retractall(viewInst(_,VInst,_,_,_)),
	retractall(consViewInst(_,VInst,_,_,_)),
	save_viewfile(ViewSys).


:- comment(update_views(ViewSys,ViewInstList,ProcName,NProcs),
"@pred{update_views(+ViewSys, +ViewInstList, +ProcName, +NProcs)} is
the predicate that runs the shell commands of view instances to create
view instance contents.  It ensures that most recent versions of the
view instances in @var{ViewInstList} (and all instances required for those
views, recursively) are up to date by executing the commands as
necessary.  A view instance is represented in this list by a term
@tt{View:InstName}.  If @var{ViewInstList} is the atom 'all', all
view instances will be processed.  This predicate will determine what
computations can be done concurrently and will use up to @var{NProcs}
concurrent processes (using spawn_process on the current machine) to
compute them.  @var{ProcName} is a user-provided process namde that
used to identify this (perhaps very long-running) process; it is used
to indicate, in @tt{Ststus=being_updated(ProcName)} that a view
instance is in the process of being computing by this update_views
invocation.  @tt{reset_unfinished/2} uses the name to identify the
view instances that a particular invocation of this process is
responsible for.  ").

update_views(ViewSys,ViewSpec,ProcName,NProcs) :-
	update_views(ViewSys,ViewSpec,ProcName,NProcs,userout).

update_views(ViewSys,all,ProcName,NProcs,OutStream) :-
	!,
	load_viewfile_for_update(ViewSys),
	findall(View:Inst,root_view_instance(View,Inst),VIs0),
	sort(VIs0,ViewInstList),
	start_available_procs(ViewSys,ViewInstList,[],ProcName,NProcs,1,OutStream).
update_views(ViewSys,ViewInstList,ProcName,NProcs,OutStream) :-
	load_viewfile_for_update(ViewSys),
	start_available_procs(ViewSys,ViewInstList,[],ProcName,NProcs,1,OutStream).

%%:- document_export start_available_procs/7.

:- comment(
start_available_procs(ViewSys,ViewInstList,ExecutingPids,ProcName,NProcs,Slp,OStr),
"@pred{start_available_procs(+ViewSys, +ViewInstList, +ExecutingPids,
+ProcName, +NProcs, +Slp, +OStr)} is an internal predicate that
supports the @pred{view_update/4} processing.  It finds all views that
can be generated (or checked), starts processes to compute
@var{NProcs} of them, and then calls @pred{monitor_running_procs/7} to
monitor their progress and start more processes as these terminate.
This is an internal predicate, not available for call from outside the
module. The parameters to @pred{start_available_procs/7} are:

@begin{enumerate}

@item @var{ViewSys} is the directory containing the
@file{viewsys_info.P} file describing the view system.

@item @var{ViewInstList} is a) an explicit list of records of the form
@tt{View:Inst} identifying the (derived) views, normally 'root'
views, that are intended to be generated by the currently running
@pred{update_view/4} invocation; or b) the constant 'all' indicating
that all view instances of the view system are intended to be
generated.

@item @var{ExecutingPids} are pid records of the currently running
processes that have been spawned.  A pid record is of the form:
@tt{pid(Pid,ShCmd,SStr,FileOut,Datime,View,File,Inst)}, where

@begin{itemize}

@item @var{Pid} is the process ID of the process (as returned by
@pred{spawn_process/5}.)

@item @var{ShCmd} is the shell command that was used to start the process.

@item @var{SStr} is the output stream of the process's stdout and stderr file.

@item @var{FileOut} is the name of the file connected to the stdout/stderr stream.

@item @var{Datime} is the datime that the process was started.

@item @var{View} is the view the process is generating.

@item @var(File) is the name of the output file to contain the
contents of the view instance.

@item @var{Inst} is the instance of the view the process is generating.

@end{itemize}

@item @var{ProcName} is the user-provided name of this entire update
process, and is used to mark views (in the
@file{viewsys_instance_info.P} file) during processing so they can be
identified as associated to this view-update process if some error
occurs.

@item @var{NProcs} is the number of 'processors' available for a
process to be scheduled on.  The 'processors' are virtual, and this is
used to control the maximum number of concurrently running processes.

@item @var{Slp} is the number of seconds to sleep if no subprocess is
available for starting before checking again to see if some subprocess
has completed in the interim.

@item @var{OStr} is the output stream used to write progress messages
when processes start and complete.

@end{enumerate}
	   "
	  ).

start_available_procs(ViewSys,ViewInstList,ExecutingPids,ProcName,NProcs,Slp,OStr) :-
	findall(Command,command_to_update_view(ViewInstList,Command),Commands0),
	sort(Commands0,Commands1), % parsort sometimes does not elim dupls??????
	parsort(Commands1,[desc(6)],1,Commands),  
	(Commands == []
	 ->	save_viewfile(ViewSys),
		NewExecPids = ExecutingPids,
		NProcsRem = NProcs
	 ;	local_datime(Datime),
		collect_commands_to_start(Commands,NProcs,Datime,ProcName,CommandsToEx,NProcsRem),
		save_viewfile(ViewSys),
		start_commands(CommandsToEx,OStr,ExecutingPids,NewExecPids)	
	),
	monitor_running_procs(NewExecPids,NProcsRem,ViewSys,ViewInstList,ProcName,Slp,OStr).

command_to_update_view(ViewInstList,Command) :-
	(ViewInstList == all
	 ->	viewInst(View,Inst,_,_,_)
	 ;	member(View:Inst,ViewInstList)
	),
	view_to_update(View,Inst,Command).

collect_commands_to_start(_Commands,0,_,_,[],0) :- !.
collect_commands_to_start([],NProcs,_,_,[],NProcs).
collect_commands_to_start([Command|Commands],NProcs,Datime,ProcName,[Command|CommandsToEx],NProcsRem) :-
	Command = cmd(_ShCmd,_StdOut,View,Inst,_File,_Cost),
	(viewInst(View,Inst,not_generated,_,_Began)
	 do_all
	 retractall(viewInst(View,Inst,not_generated,_,_)),
	 assert(viewInst(View,Inst,being_generated(ProcName),Datime,Datime))
	),
	(consViewInst(View,Inst,not_generated,_,_Began)
	 do_all
	 retractall(consViewInst(View,Inst,not_generated,_,_)),
	 assert(consViewInst(View,Inst,being_generated(ProcName),Datime,Datime))
	),
	NProcs1 is NProcs - 1,
	collect_commands_to_start(Commands,NProcs1,Datime,ProcName,CommandsToEx,NProcsRem).

start_commands([],_,Pids,Pids).
start_commands([Command|Commands],OStr,Pids0,Pids) :-
	spawn_one_cmd(Command,OStr,Pid),
	start_commands(Commands,OStr,[Pid|Pids0],Pids).

spawn_one_cmd(cmd(ShCmd,StdOut,View,Inst,File,Cost),OStr,
	      pid(Pid,ShCmd,SStr,FileOut,Datime,View,File,Inst)) :-
	generate_file_from_template(StdOut,View,Inst,FileOut),
	ensure_directories_exist(FileOut,OStr),
	open(FileOut,write,SStr),
	spawn_process(ShCmd,none,SStr,SStr,Pid),
	local_datime(Datime),
	(OStr \== none
	 ->	fmt_write(OStr,"\nSpawned Command: PID %d: view: %s:%s at %S, est %S\n",
			  args(Pid,View,Inst,Datime,Cost)),
		fmt_write(OStr,"  %s\n",args(ShCmd)),
		flush_output(OStr)
	 ;	true
	).
	
%%:- document_export monitor_running_procs/7.

:- comment(
monitor_running_procs(Pids,NProcs,ViewSys,VInstList,ProcName,Slp,OStr),
"@pred{monitor_running_procs(+Pids, +NProcs, +ViewSys, +VInstList,
+ProcName, +Slp, +OStr)} is an internal predicate that monitors
previously spawned running processes, calling
@pred{start_available_procs/7} to spawn new ones when running
processes finish.

@begin{enumerate}

@item @var{Pids} is the list of process IDs of running processes.
Each entry is a record of the form
@tt{pid(Pid,Cmd,StdStr,FileOut,Datime,View,File,Inst)} where:

@begin{itemize}

@item @var{Pid} is the process ID of the process (as returned by
@pred{spawn_process/5}.)

@item @var{ShCmd} is the shell command that was used to start the process.

@item @var{SStr} is the output stream of the process's stdout and stderr file.

@item @var{FileOut} is the name of the file connected to the stdout/stderr stream.

@item @var{Datime} is the datime that the process was started.

@item @var{View} is the view the process is generating.

@item @var(File) is the name of the output file to contain the
contents of the view instance.

@item @var{Inst} is the instance of the view the process is generating.

@end{itemize}

@item @var{NProcs} is the number of 'processors' that are currently
available for use.  starrt_available_procs can start up to this number
of new processes.

@item @var{ViewSys} is the viewsys directory;

@item @var{VInstList} is the list of view instances (or 'all') that are
being updated by this execution of update_views/4.;

@item @var{ProcName} is the caller-provided name of this update
processor used to mark views that are being updated by this update
process; and

@item @var{Slp} is the number of seconds to sleep if no process is
available for starting.

@item @var{OStr} is the output stream for writing status messages;

@end{enumerate}
"
	  ).

monitor_running_procs([],_NProcs,_ViewSys,_VInstList,_ProcName,_Slp,_OStr) :- !.
monitor_running_procs(Pids,NProcs,ViewSys,VInstList,ProcName,Slp,OStr) :-
	collect_finished_procs(Pids,FinishedPids,UnfinishedPids),
	(FinishedPids \== []
	 ->	local_datime(CurDatime),
		length(FinishedPids,NumFinished),
		NProcsRem is NProcs + NumFinished,
		load_viewfile_for_update(ViewSys),
		(do_all
		 member(PidStr,FinishedPids),
		 process_finished_pid(PidStr,CurDatime,OStr)
		),
		start_available_procs(ViewSys,VInstList,UnfinishedPids,ProcName,NProcsRem,1,OStr)
	 ;	sleep(Slp),
		monitor_running_procs(Pids,NProcs,ViewSys,VInstList,ProcName,2,OStr)
	).

collect_finished_procs([],[],[]).
collect_finished_procs([PidStr|Pids],FinishedPids,UnfinishedPids) :-
	PidStr = pid(Pid,_Pcmd,SStr,FileOut,Datime,View,File,Inst),
	process_status(Pid,PStatus),
	(PStatus \== running
	 ->	FinishedPids = [pid(Pid,PStatus,SStr,FileOut,Datime,View,File,Inst)|FinishedPids1],
		collect_finished_procs(Pids,FinishedPids1,UnfinishedPids)
	 ;	UnfinishedPids = [PidStr|UnfinishedPids1],
		collect_finished_procs(Pids,FinishedPids,UnfinishedPids1)
	).

process_finished_pid(pid(Pid,PStatus,SStr,FileOut,Datime,View,_File,Inst),CurDatime,OStr) :-
	(process_control(Pid,wait(_RetCode))
	 ->	true		% windows: finish process
	 ;	true		% linux: already gone
	),
	close(SStr),
	datime_diff(CurDatime,Datime,Elapsed),
	display_datime_diff(Elapsed,Time),
	(PStatus == exited_normally
	 ->	op_fmt_write(OStr,"\nFinished PID %d SUCCESSFUL view: %s:%s Time: %s, StdOut: %s\n",
			     args(Pid,View,Inst,Time,FileOut)),
		VStatus = generated
	 ;	op_fmt_write(OStr,"\nFinished PID %d FAILED Code: %s, view: %s:%s Time: %s, StdOut: %s\n",
			     args(Pid,PStatus,View,Inst,Time,FileOut)),
		VStatus = generation_failed
	),
	flush_output(OStr),
	(PStatus == exited_normally
	 ->	(viewInst(View,Inst,_,_,Began)
		 do_all
		 retractall(viewInst(View,Inst,_,_,_)),
		 assert(viewInst(View,Inst,VStatus,CurDatime,Began))
		),
		(consViewInst(View,Inst,_,_,Began)
		 do_all
		 retractall(consViewInst(View,Inst,_,_,_)),
		 assert(consViewInst(View,Inst,VStatus,CurDatime,Began))
		)
	 ;	(viewInst(View,Inst,_Stat1,_Dt,Began)
		 do_all
		 retractall(viewInst(View,Inst,_,_,_)),
		 assert(viewInst(View,Inst,generation_failed,CurDatime,Began))
		),
		(consViewInst(View,Inst,_,_,Began)
		 do_all
		 retractall(consViewInst(View,Inst,_,_,_)),
		 assert(consViewInst(View,Inst,generation_failed,CurDatime,Began))
		)
	).

op_fmt_write(OStr,FmtString,Args) :-
	(OStr == none
	 ->	true
	 ;	fmt_write(OStr,FmtString,Args)
	).

ensure_viewout_directories_exist(File,ViewType,OStr) :-
	ensure_directories_exist(File,OStr),
	(functor(ViewType,dir,_)
	 ->	(file_exists(File)
		 ->	true
		 ;	mkdir(File),
			fmt_write(OStr,'Created Directory: %s\n',args(File))
		)
	 ;	true
	).

ensure_directories_exist(File,OStr) :-
	(split_dir_path(File,Dir)
	 ->	(file_exists(Dir)
		 ->	true
		 ;	ensure_directories_exist(Dir,OStr),
			mkdir(Dir),
			fmt_write(OStr,'Created Directory: %s\n',args(Dir))
		)
	 ;	true
	).

mkdir(Dir) :-
	sys_mkdir(Dir,Res),
	(Res =:= 0
	 ->	true
	 ;	throw(error('trying to create directory',Dir))
	).

split_dir_path(Name,Dir) :-
	atom_codes(Name,NameC),
	(lappend(DirC,[0'/|FileC],NameC),FileC \== []
	 ->	true
	 ; lappend(DirC,[0'\\|FileC],NameC),FileC \== []
	 ->	true
	 ;	fail
	),
	atom_codes(Dir,DirC).

lappend([X|L1],L2,[X|L3]) :- lappend(L1,L2,L3).
lappend([],L,L).

%% find views that are ready to be updated
view_to_update(View,Inst,Command) :-
	viewInst(View,Inst,not_generated,Datime,Began),
	viewV(View,Type,LocTempl,ViewsDep,_Opts,ShellCmd),
	ViewsDep \== [],
	(all_view_instances_generated(ViewsDep,Inst,InputFiles)
	 ->	generate_file_from_template(LocTempl,View,Inst,File),
		ensure_viewout_directories_exist(File,Type,userout),
		generate_shell_command(ShellCmd,View,Inst,InputFiles,File,ShCommand),
		compute_cost(View,Inst,Datime,Began,Cost),
		generate_file_from_template('$STDOUTFILE$',View,Inst,Stdout),
		Command = cmd(ShCommand,Stdout,View,Inst,File,Cost)
	 ;	member(DepViewTerm,ViewsDep),
		functor(DepViewTerm,DepView,_),
		\+ viewV(DepView,_,_,[],_,_),
		view_to_update(DepView,Inst,Command)
	).
view_to_update(View,Inst,Command) :-
	viewInst(View,Inst,generated,_,_),
	consViewV(ConsView,View,CVTempl/*,Stdout*/,ViewsDep,ShellCmd),
	consViewInst(ConsView,Inst,not_generated,Datime,Began),
	(all_view_instances_generated(ViewsDep,Inst,InputFiles0)
	 ->	generate_file_from_template(CVTempl,ConsView,Inst,File),
		viewV(View,_Type,VLocTempl,_,_,_),
		generate_file_from_template(VLocTempl,ConsView,Inst,ViewFile),
		generate_shell_command(ShellCmd,ConsView,Inst,[ViewFile|InputFiles0],File,ShCommand),
		datime_diff(Datime,Began,Cost),
		generate_file_from_template('$STDOUTFILE$',ConsView,Inst,Stdout),
		Command = cmd(ShCommand,Stdout,ConsView,Inst,File,Cost)
	 ;	fail
	).
	 
compute_cost(View,Inst,Datime,Began,Cost) :-
	datime_diff(Datime,Began,Cost0),
	(compute_post_cost_e(View,Inst,Cost1)
	 ->	datime_sum(Cost0,Cost1,Cost)
	 ;	Cost = Cost0
	).

:- table compute_post_cost_e/3 as subsumptive.
compute_post_cost_e(View,Inst,Cost) :-
	(var(View),var(Inst)
	 ->	compute_post_cost(View,Inst,Cost)
	 ;	compute_post_cost_e(View1,Inst1,Cost1),
		View1 = View, Inst1 = Inst, Cost1= Cost
	).

:- table compute_post_cost(_,_,lattice(max(_,_,_))).
compute_post_cost(View,Inst,Cost) :-
	viewV(DView,_Type,_Loc,ViewsDep,_Opt,_Sh),
	ViewsDep \== [],
	member(ViewTerm,ViewsDep),
	functor(ViewTerm,View,_),
	viewInst(DView,Inst,_Stat,Datime,Began),
	datime_diff(Datime,Began,Cost1),
	(compute_post_cost(DView,Inst,Cost0)
	 ->	datime_sum(Cost0,Cost1,Cost)
	 ;	Cost = Cost1
	).

max(X,Y,Z) :- (X @>= Y -> Z = X ; Z = Y).

%% first check for individual $INP1$...$INPN$ and $OUT$, and replace them if so.
generate_shell_command(xsb(Goal),View,Inst,InputFiles,OutFile,ShCommand) :-
	!,			% xsb command with call-term
	build_substitution_list(InputFiles,1,OutFile,SubList),
	add_user_substitutions([s(View,'$VIEW$'),s(Inst,'$INSTANCE$')|SubList],AllSubs),
	substitute_terms(AllSubs,Goal,ShGoal),
	term_to_atom(ShGoal,ShGoalStr,[quoted(true)]),
	(member(s('$XSB$',XSB),AllSubs)
	 ->	true
	 ;	XSB = xsb
	),
	double_single_quotes(ShGoalStr,ShGoalStrQ),
	concat_atom([XSB,' -e "',ShGoalStrQ,'."'],ShCommand).
generate_shell_command(ShellCmd,View,Inst,InputFiles,OutFile,ShCommand) :-
	str_match('$OUT$',ShellCmd,forward,_,_), % explicit indiv file vars
	!,
	build_substitution_list(InputFiles,1,OutFile,SubList),
	add_user_substitutions([s(View,'$VIEW$'),s(Inst,'$INSTANCE$')|SubList],AllSubs),
	substitute_terms(AllSubs,ShellCmd,ShCommand).
%% use "sh script" forms, including double-quote delimiters
generate_shell_command(ShellCmd,View,Inst,InputFiles,OutFile,ShCommand) :-
	concat_atom(InputFiles,'" "',InputFileStr0),
	concat_atom(['"',InputFileStr0,'"'],InputFileStr),
	concat_atom(['"',OutFile,'"'],OutFileStr),
	(str_match('$INPUTFILES$',ShellCmd,forward,_,_)
	 ->	ShellCmd1 = ShellCmd
	 ;	concat_atom([ShellCmd,' ','$INPUTFILES$',' ','$OUTPUTFILE$'],ShellCmd1)
	),
	add_user_substitutions([s(View,'$VIEW$'),s(Inst,'$INSTANCE$'),
				s(InputFileStr,'$INPUTFILES$'),s(OutFileStr,'$OUTPUTFILE$')],AllSubs),
	substitute_terms(AllSubs,ShellCmd1,ShCommand).

double_single_quotes(Atom0,Atom) :-
	atom_codes(Atom0,AtomC0),
	dsq(AtomC0,AtomC),
	atom_codes(Atom,AtomC).
	
dsq([],[]).
dsq([0'\''|Ss],[0'\'',0'\''|Rs]) :- !,
	dsq(Ss,Rs).
dsq([C|Ss],[C|Rs]) :-
	dsq(Ss,Rs).

%% build list of substitution pairs for input files, and output file
build_substitution_list([],_,OutFile,[s(OutFile,'$OUT$')]).
build_substitution_list([InputFile|InputFiles],K,OutFile,[s(InputFile,InpVar)|SubList]) :-
	concat_atom(['$INP',K,'$'],InpVar),
	K1 is K+1,
	build_substitution_list(InputFiles,K1,OutFile,SubList).

:- comment(generate_file_from_template/4,
"@pred{generate_file_from_template(+FileTempl,+View,+Inst,-FileName)}
takes a file template string (with embedded $$ variable names), a view
name, @var{View}, an instance name, @var{Inst}, and replaces the
variable names with their values, returnning @var{FileName}.").

generate_file_from_template(FileTempl,View,Inst,File) :-
	add_user_substitutions([s(View,'$VIEW$'),s(Inst,'$INSTANCE$')],SubList),
	substitute_terms(SubList,FileTempl,File).

add_user_substitutions(SubList,AllSubList) :-
	user_substitutions(UserSubList),
	append(SubList,UserSubList,AllSubList).

:- table user_substitutions/1.
user_substitutions([s(XSBHome,'$XSB_HOME$')|UserSubList0]) :-
	xsb_configuration(install_dir,XSBHome),
	findall(s(Val,Var),get_viewsys_uservar(Var,Val),UserSubList0).

get_viewsys_uservar(Var,Val) :-
	all_viewsys_uservar(Var,Val0),
	(Val0 = lambda(Val,Goal)
	 ->	call_c(Goal)
	 ;	Val = Val0
	).

%% must generate all var-val pairs.
all_viewsys_uservar(Var,Val) :-
	global_viewsys_uservar(Var,Val).
all_viewsys_uservar(Var,Val) :-
	viewsys_uservar(Var,Val),
	\+ global_viewsys_uservar(Var,_).

set_viewsys_uservar(Var,Val) :-
	retractall(global_viewsys_uservar(Var,_)),
	assert(global_viewsys_uservar(Var,Val)).


all_view_instances_generated([],_,[]).
all_view_instances_generated([ViewTerm|Views],Inst,[InpFile|InpFiles]) :-
	functor(ViewTerm,View,_),
	\+ a_consistency_instance_ungenerated(View,Inst),
	viewV(View,_Type,LocTempl,_,_,_),
	generate_file_from_template(LocTempl,View,Inst,InpFile0),
	(viewInst(View,Inst,generated,_,_)
	 ->	add_inpfile_if_nec(InpFile0,ViewTerm,InpFile)
	 ; base_view(View),viewInst(View,Inst,_Stat0,_Dt,_)
	 ->	InpFile = InpFile0
	 ;	base_view(View),viewInst(View,all,_Stat1,_,_)
	),
	all_view_instances_generated(Views,Inst,InpFiles).

add_inpfile_if_nec(InpFile0,ViewTerm,InpFile) :-
	(ViewTerm =.. [_,FileSuff]
	 ->	concat_atom([InpFile0,'/',FileSuff],InpFile)
	 ;	InpFile = InpFile0
	).

a_consistency_instance_ungenerated(View,Inst) :-
	consViewV(ConsView,View,_,_,_),
	consViewInst(ConsView,Inst,VStatus,_,_),
	VStatus \== generated.

:- comment(invalidate_all_instances(ViewSys),
"@pred{invalidate_all_instances(+ViewSys)} invalidates all views, so a
subsequent invocation of @pred{update_views/4} would recompute them
all."  ).

invalidate_all_instances(ViewSys) :-
	load_viewfile_for_update(ViewSys),
	(viewInst(View,VInst,_State0,Datime,Began),
	 \+ base_view(View)
	 do_all
	 retractall(viewInst(View,VInst,_,_,_)),
	 assert(viewInst(View,VInst,not_generated,Datime,Began))
	),
	(consViewInst(CView,View,_State1,Datime,Began)
	 do_all
	 retractall(consViewInst(CView,View,_,_,_)),
	 assert(consViewInst(CView,View,not_generated,Datime,Began))
	),
	save_viewfile(ViewSys).

:- comment(invalidate_view_instances(ViewSys,ViewInstList), "
@pred{invalidate_view_instances(+ViewSys,+ViewInstList)} invalidates a
set of view instances indicated by @var{ViewInstList}.  If
@var{ViewInstList} is the atom 'all', this invalidates all instances
(exactly as @pred{invalidate_all_instances/1)} does.)  If
@var{ViewInstList} is a list of terms of the form
@var{View:VInst} then these indicated view instances (and all
views that depend on them) will be invalidated.  If @var{ViewInstList}
is the atom 'filetime', then the times of the instance files will be
used to invalidate view instances where the filetime of some view
instance input file is later than the filetime of the view instance
output file.  Note this does not account for the time it takes to run
the shell command that generates the view output, so for it to work,
no view instance input file should be changed while a view instance is
in the process of being generated.

This predicate can be used if a base instance file is replaced with a
new instance.  It can be used if the contents of a view instance are
found not to be correct, and the generating process has been modified
to fix it."  ).

invalidate_view_instances(ViewSys,all) :-
	!,
	invalidate_all_instances(ViewSys).
invalidate_view_instances(ViewSys,filetime) :-
	!,
	load_viewfile_for_update(ViewSys),
	findall(View:VInst,out_of_date_view_instance(View,VInst),ODVIs),
	collect_supported_view_instances(ODVIs,SupportedVIs),
	invalidate_each(SupportedVIs),
	save_viewfile(ViewSys).
invalidate_view_instances(ViewSys,ViewInstList) :-
	load_viewfile_for_update(ViewSys),
	collect_supported_view_instances(ViewInstList,SupportedVIs),
	invalidate_each(SupportedVIs),
	save_viewfile(ViewSys).

collect_supported_view_instances(ODVIs,SupportedVIs) :-
	findall(View:VInst,
		((viewInst(OOView,VInst,_,_,_)
		  ;
		  consViewInst(OOView,VInst,_,_,_)
		 ),
		 member(OOView:VInst,ODVIs),
		 rtransitively_supports(OOView,View)
		),
		SupportedVIs0),
	sort(SupportedVIs0,SupportedVIs). % elim dupls

invalidate_each(InvalVIs) :-
	(do_all
	 member(View:VInst,InvalVIs),
	 (consViewInst(View,VInst,generated,Datime,Began), %unchanged if failed somehow
	  \+ base_view(View)	%base views always generated
	  ->	 retractall(consViewInst(View,VInst,_,_,_)),
		 assert(consViewInst(View,VInst,not_generated,Datime,Began)),
		 fmt_write('Invalidated consistency view %s:%s\n',args(View,VInst))
	  ; (viewInst(View,VInst,generated,Datime,Began), \+ base_view(View))
	  ->	 retractall(viewInst(View,VInst,_,_,_)),
		 assert(viewInst(View,VInst,not_generated,Datime,Began)),
		 fmt_write('Invalidated view %s:%s\n',args(View,VInst))
	 )
	).

/* approximate, since view generation could take time.  Should use
Began time rather than filetime, but OK if we assume base file isnt
updated until after view is generated. */
out_of_date_view_instance(View,VInst) :-
	%% only if it was generated can it be out-of-date
	viewInst(View,VInst,generated,_Datime,_Began),
	\+ base_view(View), % base has nothing to compare to, cant be ood.
	viewV(View,Type,FileTempl,SuppViews,_Opts,_Cmd),
	generate_file_from_template(FileTempl,View,VInst,ViewFile),
	(Type =.. [dir|RelFiles]
	 ->	findall(t(FT,ViewFileSub),(member(RelFile,RelFiles),
			    concat_atom([ViewFile,'/',RelFile],ViewFileSub),
			    file_time(ViewFileSub,FT)),
			FTs),
	 	FTs = [FT|FTRs],
		maxList(FTRs,FT,FileTimeP),
		FileTimeP = t(FileTime,FileName)
	 ;	file_time(ViewFile,FileTime),
		FileName = ViewFile
	),
	once((member(SuppView,SuppViews),
	      filetime_later(SuppView,VInst,FileTime,SuppFile))
	    ),
	writeln([ood_file=FileName,newer_file=SuppFile]),
	true.

maxList([],Max,Max).
maxList([X|Xs],CMax,Max) :-
	(X @> CMax
	 ->	maxList(Xs,X,Max)
	 ;	maxList(Xs,CMax,Max)
	).

filetime_later(View,VInst,FileTime,ViewFileSub) :-
	viewInst(View,VInst,_Status,_Datime,_Began),
	viewV(View,Type,FileTemplate,_,_,_),
	generate_file_from_template(FileTemplate,View,VInst,ViewFile),
	((Type == file ; Type == dir)
	 ->	file_time(ViewFile,SFileTime),
		SFileTime >= FileTime,
		ViewFileSub = ViewFile
	 ; Type =.. [dir|RelFiles]
	 ->	member(RelFile,RelFiles),
		concat_atom([ViewFile,'/',RelFile],ViewFileSub),
		file_time(ViewFileSub,SFileTime),
		SFileTime > FileTime
	).

:- comment(reset_unfinished(ViewSys,ProcName), "
@pred{reset_unfinished(+ViewSys,+ProcName)} resets view instances that
are unfinished due to some abort, i.e., that are marked as
@tt{being_generated(ProcName)} after the @tt{view_update} process
named @tt{ProcName} is no longer running scripts to generate view
instances.  This should only be called when the @tt{ProcName}
@tt{view_update} process is not running.  The statuses of these view
instances will be reset to @tt{not_generated}.  After this, the next
applicable @pred{update_views/4} will try to recreate these view
instances."  ).

reset_unfinished(ViewSys,ProcName) :-
	load_viewfile_for_update(ViewSys),
	local_datime(Datime),
	(viewInst(View,ViewInst,being_generated(ProcName),_,Began)
	 do_all
	 retractall(viewInst(View,ViewInst,being_generated(ProcName),_,_)),
	 assert(viewInst(View,ViewInst,not_generated,Datime,Began)),
	 fmt_write('Unfinished view %s:%s reset\n',args(View,ViewInst))
	),
	(consViewInst(View,ViewInst,being_generated(ProcName),_DT0,Began)
	 do_all
	 retractall(consViewInst(View,ViewInst,being_generated(ProcName),_DT1,_)),
	 assert(consViewInst(View,ViewInst,not_generated,Datime,Began)),
	 fmt_write('Unfinished consistency view %s:%s reset\n',args(View,ViewInst))
	),
	save_viewfile(ViewSys).

:- comment(show_failed(VSDir,VInst),
"@pred{show_failed(+VSDir,+VInst)} displays each failed view
instance and consistency view instance, with file information to help
a user track down why the generation, or check, of the view failed.").

show_failed(VSDir,VInstInd) :-
	(VInstInd = all
	 ->	VInstList = [_]
	 ; atomic(VInstInd)
	 ->	VInstList = [VInstInd]
	 ;	VInstList = VInstInd
	),
	load_viewfile(VSDir),

	(member(VInst,VInstList),
	 viewInst(View,VInst,generation_failed,_,Began)
	 do_all
	 viewV(View,_type,FileTempl,_InpViews,_Opts,_ShCmd),
	 generate_file_from_template(FileTempl,View,VInst,File),
	 generate_file_from_template('$STDOUTFILE$',View,VInst,StdFile),
	 fmt_write('\nFailed view: %s:%s  began: %S\n',args(View,VInst,Began)),
	 fmt_write('    bad view file: %s\n',atgs(File)),
	 fmt_write('      stdout file: %s\n',atgs(StdFile))
	),
	
	(member(VInst,VInstList),
	 consViewInst(View,VInst,generation_failed,_,Began)
	 do_all
	 consViewV(View,ChkedView,CTempl,_InpViews,_ShCmd),
	 generate_file_from_template(CTempl,View,VInst,File),
	 generate_file_from_template('$STDOUTFILE$',View,VInst,StdFile),
	 fmt_write('\nFailed consistency view: %s:%s, began: %S\n',
		   args(View,VInst,Began)),
	 fmt_write('    checking view: %s\n',args(ChkedView)),
	 fmt_write('  bad output file: %s\n',atgs(File)),
	 fmt_write('      stdout file: %s\n',atgs(StdFile))
	).

:- comment(reset_failed(ViewSys,VInst), "
@pred{reset_failed(+ViewSys,+VInst)} resets view instances with name
@var{VInst} that had failed, i.e., that are marked as
@tt{generation_failed}.  Their status will be reset to
@tt{not_generated}, so after this, the next applicable call to
@pred{update_views/4} will try to regenerate the view.  If @var{VInst}
is 'all', then views of all instances will be reset."  ).

reset_failed(ViewSys,ViewInstPar) :-
	load_viewfile_for_update(ViewSys),
	reset_failed_1(ViewInstPar),
	save_viewfile(ViewSys).

reset_failed_1(all) :- !,
	local_datime(Datime),
	(viewInst(View,ViewInst,generation_failed,_,Began)
	 do_all
	 retractall(viewInst(View,ViewInst,generation_failed,_,_)),
	 assert(viewInst(View,ViewInst,not_generated,Datime,Began)),
	 fmt_write('Failed view %s:%s reset\n',args(View,ViewInst))
	),
	(consViewInst(View,ViewInst,generation_failed,_,Began)
	 do_all
	 retractall(consViewInst(View,ViewInst,generation_failed,_,_)),
	 assert(consViewInst(View,ViewInst,not_generated,Datime,Began)),
	 consViewV(View,ChkedView,_,_,_), % reset views that were checked by failed checker.
	 (viewInst(ChkedView,ViewInst,_,DT,CBegan)
	  do_all
	  retractall(viewInst(ChkedView,ViewInst,_,_,_)),
	  assert(viewInst(ChkedView,ViewInst,not_generated,DT,CBegan)),
	  fmt_write('Failed view %s:%s reset\n',args(ChkedView,ViewInst))
	 )
	).
reset_failed_1(ViewInstPar) :-
	local_datime(Datime),
	(viewInst(View,ViewInstPar,generation_failed,_,Began)
	 do_all
	 retractall(viewInst(View,ViewInstPar,generation_failed,_,_)),
	 assert(viewInst(View,ViewInstPar,not_generated,Datime,Began)),
	 fmt_write('Failed view %s:%s reset\n',args(View,ViewInstPar))
	),
	(consViewInst(View,ViewInstPar,generation_failed,_DT,Began)
	 do_all
	 retractall(consViewInst(View,ViewInstPar,generation_failed,_,_)),
	 assert(consViewInst(View,ViewInstPar,not_generated,Datime,Began)),
	 fmt_write('Failed consistency view %s:%s reset\n',args(View,ViewInstPar))
	).

:- comment(check_viewsys(ViewDir), " @pred{check_viewsys(+ViewDir)}
checks the contents of the @file{viewsys_view_info.P} file of the
@var{ViewDir} viewsys directory for consistency and completeness.").

check_viewsys(ViewDir) :-
	load_viewfile(ViewDir),
	(do_all
	 view_defined_from(View,DefView),
	 \+ view_name(DefView),
	 fmt_write('***ERROR: View "%s" used but not defined.\n',
		   args(DefView))
	),
	(do_all
	 view_name(View),
	 transitively_depends_on(View,View),
	 fmt_write('***ERROR: View "%s" depends on itself, which is illegal.\n',
		   args(View))
	),
	(do_all
	 consViewV(ConsView,View,_,_,_),
	 \+ viewV(View,_,_,ViewsDep,_,_),
	 ViewsDep \== [],
	 fmt_write('***ERROR: Consistency View "%s" checks view "%s", which does not exist.\n',
		   args(ConsView,View))
	),
	(do_all
	 viewV(View,_,_,[],_,_),
	 viewV(View,_,_,ViewsDep,_,_), ViewsDep \== [],
	 fmt_write('***ERROR: View "%s" is both a base and derived view.\n',
		   args(View))
	),
	(do_all
	 viewV(View,_,_,[],_,_),
	 consViewV(View,_,_,_,_),
	 fmt_write('***ERROR: View "%s" is both a base and consistency view.\n',
		   args(View))
	),
	(do_all
	 viewV(View,_,_,ViewsDep,_,_), ViewsDep \== [],
	 consViewV(View,_,_,_,_),
	 fmt_write('***ERROR: View "%s" is both a derived and consistency view.\n',
		   args(View))
	),
	(do_all
	 viewV(View,A,B,C,D,E),viewV(View,A1,B1,C1,D1,E1),
	 p(A,B,C,D,E) \= p(A1,B1,C1,D1,E1),
	 fmt_write('***ERROR: View "%s" has two definitions.\n',
		   args(View))
	),
	(do_all
	 consViewV(View,A,B,C,D),consViewV(View,A1,B1,C1,D1),
	 p(A,B,C,D) \= p(A1,B1,C1,D1),
	 fmt_write('***ERROR: Consistency view "%s" has two definitions.\n',
		   args(View))
	),
	(do_all
	 viewV(View,A,B,[],_,_),viewV(View,A1,B1,[],_,_),
	 p(A,B) \= p(A1,B1),
	 fmt_write('***ERROR: View "%s" has two definitions.\n',
		   args(View))
	),
	(do_all
	 viewV(View,_,_,InpViews,_,_),
	 member(SuppView,InpViews),
	 SuppView =.. [SView,SFile],
	 viewV(SView,SType,_,_,_,_),
	 \+ (SType =.. [dir|SFiles], member(SFile,SFiles)),
	 fmt_write('***ERROR: View %s uses directory subfile %s from view %s that is not defined in that view.\n',
		   args(View,SFile,SView))
	),
	(do_all
	 for(I,5,9),
	 I =\= 7,
	 functor(ViewTerm,view,I),
	 predicate_defined(ViewTerm),
	 fmt_write('***WARNING: view/%d is defined; should only be view/6\n',args(I))
	),

	fmt_write('View system: "%s" has been checked for consistency.',
		  args(ViewDir)).
	
:- comment(viewsys_view_status/3, "
@pred{viewsys_view_status(+ViewDir,+View:Inst,-Status)} returns the
Status of the indicated view in the indicated view instance.").

viewsys_view_status(ViewDir,ViewVInst,Status) :-
	load_viewfile(ViewDir),
	(is_list(ViewVInst)
	 ->	findall(Status1,
			(member(View:VInst,ViewVInst),
			 viewInst(View,VInst,Status1,_,_)
			),Status)
	 ;	ViewVInst = View:VInst,
		viewInst(View,VInst,Status,_,_)
	).

:- comment(viewsys_status/1, " @pred{viewsys_status(+ViewDir)} prints
out the status of the view system indicated in @var{ViewDir} for all
the options in @pred{viewsys_status/2}. ").

viewsys_status(ViewDir) :-
	viewsys_status(ViewDir,all).

:- comment(viewsys_status/2, " @pred{viewsys_status(+ViewDir,+Option)}
prints out a particular list of view instance statuses as indicated by
the value of @var{option} as follows:

@begin{description}

@item{active:} View instances currently in the process of being
generated.

@item{roots:} Root View instances and their current statuses.  A root
view instance is one that no other view depends on.

@item{failed:} View instances whose generation has failed

@item{waiting:} View instances whose computations are waiting until
views they depend on are successfully update.

@item{checks_waiting:} View instances that are waiting for consistency
checks to be executed.

@item{checks_failed:} View instances whose checks have executed and
failed.

@end{description}

This predicate can be called in one shell when @pred{update_views/4}
is running in another shell.  This allows the user to monitor the
status a long-running invocation of @pred{update_views/4}.
	   ").

viewsys_status(ViewDir,Opt) :-
	load_viewfile(ViewDir),
	(member(Opt,[all,active])
	 ->	write('View Instances being generated:\n'),
		(viewInst(View,VInst,being_generated(Proc),Date,_)
		 do_all
		 fmt_write("  %s:%s by %s started %S.\n",
			   args(View,VInst,Proc,Date))
		 if_none
		 writeln('  None')
		),
		nl(userout)
	 ;	true
	),
	(member(Opt,[all,roots])
	 ->	write('Root View Instances:\n'),
		(root_view_instance(View,VInst)
		 do_all
		 viewInst(View,VInst,VStatus,Date,_),
		 (\+ consViewV(_ConsView,View,_,_,_)
		  ->	 PStat = 'No'
		  ;	 (VStatus \== generated
			  ;	 
			  a_consistency_instance_ungenerated(View,VInst)
			 )
		  ->	 PStat = 'Not passed'
		  ;	 PStat = 'Passed'
		 ),
		 fmt_write("  %s:%s status %S %S (%s checks).\n",
			   args(View,VInst,VStatus,Date,PStat))
		 if_none
		 writeln('  None')
		),
		nl(userout)
	 ;	true
	),
	(member(Opt,[all,waiting])
	 ->	write('View Instances awaiting generation:\n'),
		(viewInst(View,VInst,not_generated,_Date,_)
		 do_all
		 fmt_write("  %s:%s\n",
			   args(View,VInst))
		 if_none
		 writeln('  None')
		),
		nl(userout)
	 ;
	 true
	),
	(member(Opt,[all,failed])
	 ->	write('View Instances whose generation failed:\n'),
		(viewInst(View,VInst,generation_failed,Date,_)
		 do_all
		 fmt_write("  %s:%s failed at %S.\n",
			   args(View,VInst,Date))
		 if_none
		 writeln('  None')
		),
		nl(userout)
	 ;	true
	),
	(member(Opt,[all,checks_waiting])
	 ->	write('View Instances awaiting consistency checks:\n'),
		(viewInst(View,VInst,generated,Date,_),
		 consViewV(ConsView,View,_,_,_),
		 consViewInst(ConsView,VInst,being_generated(Proc),CDate,_)
		 do_all
		 fmt_write("  %s:%s, generated at %S, is awaiting consistency check %s, by %s started %S.\n",
			   args(View,VInst,Date,ConsView,Proc,CDate))
		 if_none
		 writeln('  None')
		),
		nl(userout)
	 ;	true
	),
	(member(Opt,[all,checks_failed])
	 ->	write('View Instances whose consistency checks failed:\n'),
		(viewInst(View,VInst,generated,Date,_),
		 consViewV(ConsView,View,_,_,_),
		 consViewInst(ConsView,VInst,generation_failed,CDate,_)
		 do_all
		 fmt_write("  %s:%s has failed consistency check %s, failed at %S.\n",
			   args(View,VInst,ConsView,CDate))
		 if_none
		 writeln('  None')
		)
	 ;      true
	).


root_view_instance(View,VInst) :-
	root_view(View),
	viewInst(View,VInst,_VStatus,_Date,_Began).

:- comment(print_viewsys(ViewDir), " @pred{print_viewsys(+ViewDir)}
prints an indented hierarchy of the view definitions."  ).

print_viewsys(ViewDir) :-
	load_viewfile(ViewDir),
	findall(RootView,root_view(RootView),RootViews),
	print_viewsys_tree(RootViews,0,[],_).

print_viewsys_tree([],_,Printed,Printed).
print_viewsys_tree([ViewTerm|Views],Ind,Printed0,Printed) :-
	functor(ViewTerm,View,_),
	(member(View,Printed0)
	 ->	tab(Ind),write('*'), % previously expanded.
		writeln(View),
		Printed2 = Printed0
	 ; viewV(View,VType,VFileTempl,SuppViews,Opts,Script), SuppViews \== []
	 ->	tab(Ind),fmt_write("%s (view) %S\n",args(View,VType)),
		tab(Ind),fmt_write("        file template  : %s\n",args(VFileTempl)),
		tab(Ind),fmt_write("        options        : %S\n",args(Opts)),
		tab(Ind),fmt_write("        command        : %s\n",args(Script)),
		Ind1 is Ind+3,
		print_cons_tree(View,Ind1,[View|Printed0],Printed1),
		print_viewsys_tree(SuppViews,Ind1,Printed1,Printed2)
	 ; viewV(View,VType,VFileTempl,[],_,_)
	 ->	tab(Ind),fmt_write("%s (base) %S\n",args(View,VType)),
		tab(Ind),fmt_write("        file template  : %s\n",args(VFileTempl)),
		print_cons_tree(View,Ind,[View|Printed0],Printed2)
	),
	print_viewsys_tree(Views,Ind,Printed2,Printed).
	
print_cons_tree(View,Ind,Printed0,Printed1) :-
	findall(CView,consViewV(CView,View,_,_,_),CViews),
	print_cons_trees(CViews,Ind,Printed0,Printed1).
	
print_cons_trees([],_Ind,Printed,Printed).
print_cons_trees([CView|CViews],Ind,Printed0,Printed) :-
	consViewV(CView,View,FileTempl,SuppViews,Script),
	tab(Ind),fmt_write("%s Consistency view for %s\n",args(CView,View)),
	tab(Ind),fmt_write("        file template  : %s\n",args(FileTempl)),
	tab(Ind),fmt_write("        command        : %s\n",args(Script)),
	Ind1 is Ind+3,
	print_viewsys_tree(SuppViews,Ind1,[CView|Printed0],Printed1),
	print_cons_trees(CViews,Ind,Printed1,Printed).
	
/********************************************************/
/* "macro" expand for split...				*/
/********************************************************/

:- comment(expand_views(ViewSys), " @pred{expand_views(+ViewSys)}
processes view/6 definitions that have a split(N) option, generates
the necessary new view/6 facts to do the split, component processing,
and rejoin.  It overwrites the viewsys_view_info.P file, putting the
original view/6 facts into viewOrig/6 facts.  This must be called (if
necessary) when creating a new viewsys system and before calling
generate_view_instance/2. " ).

expand_views(ViewSys) :-
	view_orig_file_name(ViewSys,OrigViewInfoFile),
	(file_exists(OrigViewInfoFile)
	 ->	misc_error(('Will not overwrite file: ',OrigViewInfoFile,'; delete it, and rerun.'))
	 ;	true
	),	
	load_viewfile_view_only_for_update(ViewSys),
	(viewV(View,Type,NameTempl,InpViews,Opts,Cmd),
	 InpViews \== [],
	 select(split(N),Opts,RemOpts)
	 do_all
	 retractall(view(View,_,_,_,_,_)),
	 expand_split(N,View,Type,NameTempl,InpViews,RemOpts,Cmd)
	),
	view_file_name(ViewSys,ViewInfoFile),
	rename(ViewInfoFile,OrigViewInfoFile), % OrigViewInfoFile must not exists!
	save_viewfile_view(ViewSys).

%% generate split view
expand_split(N,View,Type,NameTempl,InpViews,Opts,Cmd) :-
	concat_atom([View,'_split'],SplitView),
	(concat_atom([NameBase,'.',Suff],NameTempl),
	 \+ str_sub('.',Suff)
	 ->	true
	 ;	NameBase = NameTempl
	),
	concat_atom([NameBase,'_splitdir'],SplitName),
	InpViews = [ViewToSplit|RemViews],
	
	(%% view to split
	 findall(SubFile,
		 (for_alpha3(1,N,"aaa",Alpha),concat_atom([infile_,Alpha],SubFile)),
		 SubFiles),
	 DirType =.. [dir|SubFiles],
	 assert(view(SplitView,DirType,SplitName,[ViewToSplit],Opts,'sh ./view_split.sh')),
	 fail
	 ;	%% views to do the work
	 for_alpha3(1,N,"aaa",Alpha),
	 concat_atom([View,'_each_',Alpha],EachViewName),
	 concat_atom([SplitName,'/outfile_',Alpha],CompFileName),
	 concat_atom(['infile_',Alpha],Infile),
	 SplitViewTerm =.. [SplitView,Infile],
	 assert(view(EachViewName,file,CompFileName,[SplitViewTerm|RemViews],
		     [],Cmd)),
	 fail
	 ;	%% view to recombine results
	 findall(EachView,
		 (for_alpha3(1,N,"aaa",Alpha),concat_atom([View,'_each_',Alpha],EachView)),
		 EachViews),
	 assert(view(View,Type,NameTempl,EachViews,[],
		     'sh $XSB_HOME$/packages/viewsys/cat_script.sh $OUTPUTFILE$ $INPUTFILES$')),
	 fail
	 ;
	 true
	).


for_alpha3(I,N,LAlpha,Alpha) :-
	I =< N,
	atom_codes(Alpha,LAlpha).
for_alpha3(I,N,[A1,A2,A3],Alpha) :-
	I < N,
	I1 is I+1,
	(A3 < 0'z
	 ->	A3p is A3+1,
		for_alpha3(I1,N,[A1,A2,A3p],Alpha)
	 ; A2 < 0'z
	 ->	A2p is A2+1,
		for_alpha3(I1,N,[A1,A2p,0'a],Alpha)
	 ; A1 < 0'z
	 ->	A1p is A1+1,
		for_alpha3(I1,N,[A1p,0'a,0'a],Alpha)
	 ;	misc_error('Alpha overflow; too many split components')
	).
	


/********************************************************/
/* view table utilities					*/
/********************************************************/
:- use_subsumptive_tabling transitively_depends_on(-,-).
transitively_depends_on(View,DefView) :-
	(viewV(View,_,_,SuppViews,_,_) ; consViewV(View,_,_,SuppViews,_)),
	member(DefViewTerm,SuppViews),
	functor(DefViewTerm,DefView,_).
transitively_depends_on(View,DefView) :-
	transitively_depends_on(View,DefView1),
	(viewV(DefView1,_,_,SuppViews,_,_) ; consViewV(DefView1,_,_,SuppViews,_)),
	member(DefViewTerm,SuppViews),
	functor(DefViewTerm,DefView,_).

:- use_subsumptive_tabling rtransitively_supports(-,-).
rtransitively_supports(View,View).
rtransitively_supports(View,SupportedView) :-
	transitively_depends_on(SupportedView,View).

/********************************************************/
/* file load utilities					*/
/********************************************************/
load_viewfile(ViewDir) :-
	load_viewfile_for_update(ViewDir),
	unlock_view(ViewDir).

load_viewfile_for_update(ViewDir) :-
	abolish_table_pred(transitively_depends_on(_,_)),
	abolish_table_pred(rtransitively_supports(_,_)),
	abolish_table_pred(user_substitutions(_)),
	(view_info_pred(Goal)
	 do_all
	 retractall(Goal)
	),
	(view_inst_info_pred(Goal)
	 do_all
	 retractall(Goal)
	),
	view_file_name(ViewDir,ViewFile),
	view_instance_file_name(ViewDir,ViewInstFile),
	lock_view(ViewDir),
	readin_viewfile(ViewFile),
	readin_viewfile(ViewInstFile).

readin_viewfile(ViewFile) :-
	%%writeln(readin_file(ViewFile)),
	(file_exists(ViewFile)
	 ->	open(ViewFile,read,IStr),
		repeat,
		read_canonical(IStr,Term),
		(Term \== end_of_file
		 ->	assert(Term),
			fail
		 ;	!
		),
		close(IStr)
	 ;	true
	).

load_viewfile_view_only_for_update(ViewDir) :-
	abolish_table_pred(transitively_depends_on(_,_)),
	abolish_table_pred(rtransitively_supports(_,_)),
	abolish_table_pred(user_substitutions(_)),
	(view_info_pred(Goal)
	 do_all
	 retractall(Goal)
	),
	view_file_name(ViewDir,ViewFile),
	lock_view(ViewDir),
	readin_viewfile(ViewFile).
	
:- import viewsys_uservar/2, baseView/4, view/6, consView/5, viewInst/5, consViewInst/5,
	viewsys_required_file/1
	from usermod.
:- import view/6, consView/5 from usermod.
:- import view/7, consView/6 from usermod.
:- dynamic viewsys_uservar/2.
:- dynamic viewsys_required_file/1.
:- dynamic baseView/4.
:- dynamic view/6.
:- dynamic view/7.
:- dynamic consView/5.
:- index(consView/5,[1,2]).
:- dynamic consView/6.
:- index(consView/6,[1,2]).
:- dynamic viewInst/5.
:- dynamic consViewInst/5.

view_info_pred(viewsys_required_file(_)).
view_info_pred(viewsys_uservar(_,_)).
view_info_pred(view(_,_,_,_,_,_)).
view_info_pred(view(_,_,_,_,_,_,_)).
view_info_pred(consView(_,_,_,_,_)).
view_info_pred(consView(_,_,_,_,_,_)).

view_inst_info_pred(viewInst(_,_,_,_,_)).
view_inst_info_pred(consViewInst(_,_,_,_,_)).

save_viewfile(ViewDir) :-
	view_instance_file_name(ViewDir,ViewFile),
	%%writeln(saving_file(ViewFile)),
	open(ViewFile,write,OStr),
	(view_inst_info_pred(Goal),
	 call_c(Goal),
	 write_canonical(OStr,Goal),
	 writeln(OStr,'.'),nl(OStr),
	 fail
	 ;	
	 true
	),
	close(OStr),
	unlock_view(ViewDir).

save_viewfile_view(ViewDir) :-
	view_file_name(ViewDir,ViewFile),
	open(ViewFile,write,OStr),
	(view_info_pred(Goal),
	 call_c(Goal),
	 write_canonical(OStr,Goal),
	 writeln(OStr,'.'),nl(OStr),
	 fail
	;      
	 true
	),
	close(OStr),
	unlock_view(ViewDir).
	
/********************************************************/
/* view table access predicates 			*/
/********************************************************/

root_view(RootView) :-
	viewV(RootView,_,_,_,_,_),
	\+ view_defined_from(_,RootView).

view_name(VName) :-
	viewV(VName,_,_,_,_,_).
view_name(VName) :-
	consViewV(VName,_,_,_,_).

view_defined_from(VName,SuppView) :-
	(viewV(VName,_,_,SuppViews,_,_)
	 ;
	 consViewV(VName,_,_,SuppViews,_)
	),
	member(SuppViewTerm,SuppViews),
	functor(SuppViewTerm,SuppView,_).

/********************************************************/
/* Locking routines 					*/
/********************************************************/

lock_view(Dir) :-
	view_lockfile_name(Dir,ViewLockFile),
	get_lock(ViewLockFile,1).

unlock_view(Dir) :-
	view_lockfile_name(Dir,ViewLockFile),
	%%writeln(removing_lockfile),
	rm(ViewLockFile).

view_lockfile_name(Dir,ViewLockFile) :-
	concat_atom([Dir,'/lock_view'],ViewLockFile).

view_file_name(ViewDir,ViewFile) :-
	concat_atom([ViewDir,'/viewsys_view_info.P'],ViewFile).

view_orig_file_name(ViewDir,ViewFile) :-
	concat_atom([ViewDir,'/viewsys_view_orig_info.P'],ViewFile).

view_instance_file_name(ViewDir,ViewInstFile) :-
	concat_atom([ViewDir,'/viewsys_instance_info.P'],ViewInstFile).

get_lock(LockFile,N) :-
	(create_lockfile(LockFile)
	 ->	true %,writeln(created_lockfile)
	 ;	(N mod 5 =:= 0
		 ->	writeln('Failed to get lock, still trying '(N,LockFile))
		 ;	true
		),
		N1 is N+1,
		sleep(1),
		get_lock(LockFile,N1)
	).

display_datime_diff(datime(Yr,Mo,Da,Hr,Mi,Se),Time) :-
	 (Yr=:=0, Mo=:=0
	  ->	 (Da=:=0
		  ->	 (Hr =:= 0
			  ->	 fmt_write_string(Time,"%02d:%02d",args(Mi,Se))
			  ;	 fmt_write_string(Time,"%02d:%02d:%02d",args(Hr,Mi,Se))
			 )
		  ;	 fmt_write_string(Time,"%2d days, %02d:%02d:%02d",args(Da,Hr,Mi,Se))
		 )
	  ;	 fmt_write_string(Time,"%d years, %2d months, %2d days, %02d:%02d:%02d",args(Yr,Mo,Da,Hr,Mi,Se))
	 ).

datime_diff(datime(Yr1,Mo1,Day1,Hr1,Min1,Sec1),
	    datime(Yr2,Mo2,Day2,Hr2,Min2,Sec2),
	    datime(Yr,Mo,Day,Hr,Min,Sec)) :-
	(Sec1 >= Sec2
	 ->	Sec is Sec1-Sec2, Min3 = Min1
	 ;	Sec is Sec1+60-Sec2, Min3 is Min1-1
	),
	(Min3 >= Min2
	 ->	Min is Min3-Min2, Hr3 = Hr1
	 ;	Min is Min3+60-Min2, Hr3 is Hr1-1
	),
	(Hr3 >= Hr2
	 ->	Hr is Hr3-Hr2, Day3 = Day1
	 ;	Hr is Hr3+24-Hr2, Day3 is Day1-1
	),
	(Day3 >= Day2
	 ->	Day is Day3-Day2, Mo3 = Mo1
	 ;	Mo3 is Mo1-1,
		modays(Mo3,MoDays),
		Day is Day3+MoDays-Day2
	),
	(Mo3 >= Mo2
	 ->	Mo is Mo3-Mo2, Yr3 = Yr1
	 ;	Mo is Mo3+12-Mo2, Yr3 is Yr1-1
	),
	Yr is Yr3-Yr2.

%% really datime-diff sum
datime_sum(datime(Yr1,Mo1,Day1,Hr1,Min1,Sec1),
	   datime(Yr2,Mo2,Day2,Hr2,Min2,Sec2),
	   datime(Yr,Mo,Day,Hr,Min,Sec)) :-
	SecA is Sec1 + Sec2,
	(SecA >= 60 -> Sec is SecA-60, SecC = 1 ; Sec = SecA, SecC = 0),
	MinA is Min1 + Min2 + SecC,
	(MinA >= 60 -> Min is MinA-60, MinC = 1 ; Min = MinA, MinC = 0),
	HrA is Hr1 + Hr2 + MinC,
	(HrA >= 24 -> Hr is HrA-24, HrC = 1 ; Hr = HrA, HrC = 0),
	DayA is Day1 + Day2 + HrC,
	(DayA >= 30 -> Day is DayA-30, DayC = 1 ; Day = DayA, DayC = 0),
	MoA is Mo1 + Mo2 + DayC,
	(MoA >= 12 -> Mo is MoA-12, MoC = 1 ; Mo = MoA, MoC = 0),
	Yr is Yr1 + Yr2 + MoC.

modays(0,31).
modays(1,31).
modays(2,28). %HACK,NOT RIGHT!
modays(3,31).
modays(4,30).
modays(5,31).
modays(6,30).
modays(7,31).
modays(8,31).
modays(9,30).
modays(10,31).
modays(11,30).
modays(12,31).

substitute_terms(Subs,Term0,Term) :-
	substitute_terms(Subs,100,Term0,Term).

substitute_terms(_Subs,N,_Term0,_Term) :-
	N =< 0, !,
	throw(error('ERROR: loop in substitutions in substitute_terms/2?',[])).
substitute_terms(Subs,N,Term0,Term) :-
	atom(Term0),
	!,
	(member(s(Term0,Repl),Subs)
	 ->	N1 is N-1,
		substitute_terms(Subs,N1,Repl,Term)
	 ;	substitute_strings(Subs,N,Term0,Term)
	).
substitute_terms(_Subs,_N,Term0,Term) :-
	number(Term0),
	!,
	Term = Term0.
substitute_terms(Subs,N,Term0,Term) :-
	Term0 =.. [String0|SubTerms0],
	substitute_strings(Subs,N,String0,String),
	substitute_terms_list(Subs,N,SubTerms0,SubTerms),
	Term =.. [String|SubTerms].

:- index substitute_terms_list/4-3.
substitute_terms_list(_Subs,_N,[],[]).
substitute_terms_list(Subs,N,[Term0|Terms0],[Term|Terms]) :-
	substitute_terms(Subs,N,Term0,Term),
	substitute_terms_list(Subs,N,Terms0,Terms).

substitute_strings(Subs,N,String0,String) :-
	(N =< 0
	 ->	throw(error('ERROR: loop in substitutions in substitute_strings/2?',[]))
	 ; ASub = s(Sub,Repl),
	   append(Pre,[ASub|Post],Subs),
	   atom(Repl),
	   str_match(Repl,String0,forward,Beg,End)
	 ->	string_substitute(String0,[s(Beg,End)],[Sub],String1),
		append(Post,[ASub|Pre],ReSubs),
		N1 is N-1,
		substitute_strings(ReSubs,N1,String1,String)
	 ;	String = String0
	).

:- comment(generate_required_dirs/2, " This predicate can be used to
help the user generate @pred{viewsys_required_file/1} facts that may
help in configuration and deployment of view systems.  It is not
needed to create and run normal view systems, only help configure the
viewsys_view_info.P file to support using @pred{copy_required_files/2}
to move them for deployment, when that is necessary.

@pred{generate_required_dirs(+SubstList,+LogFiles)} takes an
XSB_LOGFILE (or list of XSB_LOGFILEs), normally generated by running a
step in the view system, and generates (to userout)
viewsys_required_file/1 facts.  These can be edited and the copied
into the viewsys_view_info.P file to document what directories (XSB
code and general data files) are required for running this view
system.  The viewsys_required_file/1 facts are used by
copy_required_files/2 to generate a new set of files that can run the
view system.

@var{SubstList} is a list of substitutions of the form
@tt{s(VarString,RootDir)} that are applied to @em{generalize} each
directory name.  For example if we have a large library file
structure, in subdirectories of @file{C:/XSBSYS/XSBLIB}, the many
loaded files (in an @file{XSB_LOGFILE}) will start with this prefix,
for example, @file{C:/XSBSYS/XSBLIB/apps/app_1/proc_code.xwam}.  By
using the substitution, @tt{s('$DIR$','C:/XSBCVS/XSBLIB')}, that file
name will be abstracted to: @tt{'$DIR$/apps/app_1'} in the
@pred{viewsys_required_file/1} fact.  Then @pred{copy_required_files/2}
can replace this variable @tt{$DIR$} with different roots to determine
the source and target of the copying.

@var{LogFiles} is an @tt{XSB_LOGFILE}, that is generated by running
xsb and initially calling machine:stat_set_flag(99,1).  This will
generate a file named @file{XSB_LOGFILE.txt} (in the current
directory) that contains the names of all files loaded during that
execution of xsb.  (If the flag is set to @{tt}K > 1, then the name of
the generated file will be @tt{XSB_LOGFILE_<K>.txt} where @tt{<K>} is
the number @tt{K}.)

So, for example, after running three steps in a workflow, setting flag
99 to 2, 3, and 4 for each step respectively, one could execute:
@begin{verbatim}	   
| ?- generate_required_dirs([s('$DIR$','C:/XSBCVS/XSBLIB')],
                            ['XSB_LOGFILE_2.txt',
                             'XSB_LOGFILE_3.txt',
                             'XSB_LOGFILE_4.txt']).
@end{verbatim}	   

@noindent which would print out facts for all directories for files in
those LOGFILEs, each with the root directory abstracted.
").

generate_required_dirs(Substitutions,XSB_LFs) :-
	(do_all
	 logfile_directory(XSB_LFs,Dir0),
	 substitute_terms(Substitutions,Dir0,Dir),
	 writeq(viewsys_required_file(Dir)),
	 writeln('.')
	).

:- comment(copy_required_files/2, " This predicate can be used
(perhaps with configuration help from @pred{generate_required_dirs/2})
to copy and deploy view systems and the files they need to run.  This
predicate is not needed for normal execution of view systems.

@pred{copy_required_files(+VSDir,+FromToSubs)} uses the
@pred{viewsys_required_file/1} facts in the @file{viewsys_view_info.P}
file in the @var{VSDir} viewsys directory to copy all directories (and
files) in those facts.  @var{FromToSubs} are terms of the form
@tt{s(USERVAR,FROMVAL,TOVAL)}, where @tt{USERVAR} is a variable in the
file templates in the @pred{viewsys_required_file/1} facts.  A
recusrive @tt{cp} shell command will be generated and executed for
each template in @pred{viewsys_required_file/1}, the source file being
the template with @tt{USERVAR} replaced by @tt{FROMVAL} and the target
file being the template with @tt{USERVAR} replaced by @tt{TOVAL}.

All necessary intermediate directories will be automatically created.

E.g.,
@begin{verbatim}
copy_required_files('.',[s('$DIR$','C:/XSBSYS/XSBLIB','C:/XSBSYS/XSBTEST/XSBLIB')]).
@end{verbatim}

@noindent would copy all files/directories indicated in the
@pred{viewsys_required_file/1} facts in the local
@file{viewsys_view_info.P} file from under @tt{C:/XSB/XSBLIB} to a
(possibly) new directory @file{C:/XSBSYS/XSBTEST/XSBLIB} (assuming all file
templates were rooted with @tt{$DIR$}.)
").

copy_required_files(VSDir,FromToSubs) :-
	load_viewfile(VSDir),
	findall(s(FromVal,Var),member(s(Var,FromVal,_),FromToSubs),FromSubList),
	findall(s(ToVal,Var),member(s(Var,_,ToVal),FromToSubs),ToSubList),
	(do_all
	 viewsys_required_file(FileTempl),
	 substitute_terms(FromSubList,FileTempl,FromDirectory),
	 substitute_terms(ToSubList,FileTempl,ToDirectory),
	 ensure_directories_exist(ToDirectory,userout),
	 shell(['cp -r "',FromDirectory,'" "',ToDirectory,'"'])
	).

logfile_file(LogFiles0,File) :-
	(atom(LogFiles0)
	 ->	LogFiles = [LogFiles0]
	 ;	LogFiles = LogFiles0
	),
	findall(Fil,
		(member(LogFile,LogFiles),
		 file_codes_in(LogFile,FilCodes0),
		 to_forward_slash_codes(FilCodes0,FilCodes),
		 atom_codes(Fil,FilCodes) ),
		Fils),
	sort(Fils,UFils),
	member(File,UFils).

logfile_directory(LogFiles0,Directory) :-
	(atom(LogFiles0)
	 ->	LogFiles = [LogFiles0]
	 ;	LogFiles = LogFiles0
	),
	findall(Dir,
		(member(LogFile,LogFiles),
		 directory_codes_in(LogFile,DirCodes0),
		 atom_codes(Dir0,DirCodes0),
		 expand_filename(Dir0,Dir1),
		 atom_codes(Dir1,DirCodes1),
		 to_forward_slash_codes(DirCodes1,DirCodes),
		 atom_codes(Dir,DirCodes) ),
		Dirs),
	sort(Dirs,UDirs),
	member(Directory,UDirs).

directory_codes_in(LogFile,DirectoryCodes) :-
	file_codes_in(LogFile,FileCodes),
	once(lappend(DirectoryCodes,[0'\\'|_],FileCodes)).


file_codes_in(LogFile,FileCodes) :-
	file_line_list_file(LogFile,CodeList),
	append_list([CWDCodes,": ",FileCodesle],CodeList),
	(FileCodesle = [0'.',0'.'|RelFileCodesle]
	 ->	append(CWDCodes,[0'/'|FileCodesle],AbsFileCodesle)
	 ; FileCodesle = [0'.'|RelFileCodesle]
	 ->	append(CWDCodes,RelFileCodesle,AbsFileCodesle)
	 ;	AbsFileCodesle = FileCodesle
	),
	(append(FileCodes0,"\r\n",AbsFileCodesle)
	 ->	true
	 ; append(FileCodes0,"\n",AbsFileCodesle)
	 ->	true
	 ;	FileCodes0 = AbsFileCodesle
	),
	(append([Drive,0':'],Undrived,FileCodes0)
	 ->	(Drive >= 0'a , Drive =< 0'z
		 ->	UDrive is Drive - 32
		 ;	UDrive = Drive
		),
		FileCodes = [UDrive,0':'|Undrived]
	 ;	FileCodes = FileCodes0
	).
	 
		 

append_list([L],L) :- !.
append_list([L|Ls],FL) :-
	append_list(Ls,FL0),
	append(L,FL0,FL).

file_line_list_file(File,Line) :-
	open(File,read,IStr),
	file_line_list_file1(IStr,Line).

file_line_list_file1(IStr,Line) :-
	(file_read_line_list(IStr,Line0)
	 ->	(Line = Line0
		 ;
		 file_line_list_file1(IStr,Line)
		)
	 ;	close(IStr),
		fail
	).

to_forward_slash_codes([],[]).
to_forward_slash_codes([C|Cs],[R|Rs]) :-
	(C =:= 0'\\'
	 ->	R = 0'/'
	 ;	R = C
	),
	to_forward_slash_codes(Cs,Rs).

