/*
*  qm_streamloader.cpp
*  QUIMUP class to download streams
*  © 2008-2024 Johan Spee
*  SPDX-License-Identifier: GPL-3.0-or-later
*/

#include "qm_streamfetcher.h"


qm_streamFetcher::qm_streamFetcher()
{
    connect(&nwm_manager, SIGNAL(finished(QNetworkReply*)),
            SLOT(on_download_finished(QNetworkReply*)));
}


void qm_streamFetcher::download_this(QString input)
{
//  convert http://radio.com/?u=http://www.party.com/listen.pls&t=.pls
//  to      http://www.party.com/listen.pls
    int i = input.lastIndexOf( "http" );
    input = input.right(input.length() - i);
    i = input.lastIndexOf( '&' );
    input = input.left(i);

    const QUrl the_url = QUrl(input.simplified());

    QNetworkRequest request(the_url);
    download_time.start();
    thedownload = nwm_manager.get(request);
    connect(thedownload, SIGNAL(downloadProgress(qint64,qint64)),
            SLOT(on_download_progress(qint64,qint64)));
}


void qm_streamFetcher::on_download_progress(qint64 bytesReceived, qint64 bytesTotal)
{
    if (download_time.elapsed() > 10000) // msecs
    {
        int percent;
        if (bytesTotal > 0)
            percent = static_cast<int>(100 * (static_cast<double>(bytesReceived)/static_cast<double>(bytesTotal)));
        else
            percent = 0;

        if (percent < 50)
        {
            thedownload->abort();
            QStringList result;
            result << "timeout" << QString::number(percent);
            emit streamdownload_done(result);
        }
    }
}


void qm_streamFetcher::on_download_finished(QNetworkReply *reply)
{
    QStringList result;

    int counter = 0;
    if (reply->error())
    {
        result << "error" << "Download failed : \n" + reply->errorString();
        emit streamdownload_done(result);
        reply->deleteLater();
    }

    if (!reply->isReadable())
    {
        result << "error" << "Download is not readable ";
        emit streamdownload_done(result);
        reply->deleteLater();
    }

     // checking for http, i.e not excluding https

    QByteArray qba = reply->readAll();
    if ( qba.startsWith("#EXTM3U") )
    {
        result << "ok" << "extended m3u";
        QTextStream in(qba);
        while (!in.atEnd())
        {
            QString line = in.readLine().simplified();
            if (line.startsWith("http"))
            {
                result << line;
                counter++;
            }
        }
    }
    else
    if ( qba.startsWith("[playlist]"))
    {
        result << "ok" << "pls";
        QTextStream in(qba);
        while (!in.atEnd())
        {
            QString line = in.readLine().simplified();
            int i = line.indexOf("http");
            if (line.indexOf("File") != -1 || i != -1 )
            {
                result << line.right(line.length() - i);
                counter++;
            }
        }
    }
    else
    if ( qba.startsWith("<asx "))
    {
        result << "ok" << "asx";
        QTextStream in(qba);
        while (!in.atEnd())
        {
            QString line = in.readLine().simplified();
            int i = line.indexOf("http");
            if (line.indexOf("<ref href") != -1 && i != -1 )
            {
                line = line.right(line.length() - i);
                line.truncate(line.indexOf("\""));
                result << line;
                counter++;
            }
        }
    }
    else
    if ( qba.startsWith("<?xml") &&  qba.indexOf("xspf.org") != -1 )
    {
        result << "ok" << "xspf";
        QTextStream in(qba);
        while (!in.atEnd())
        {
            QString line = in.readLine().simplified();
            int i = line.indexOf("http");
            if (line.indexOf("<location>") != -1 && i != -1 )
            {
                line = line.right(line.length() - i);
                line.truncate(line.indexOf("<"));
                result << line;
                counter++;
            }
        }
    }
    else
    if ( qba.indexOf("#EXTM3U" ) != -1 ) // m3u (with comments)
    {
        result << "ok" << "m3u";
        QTextStream in(qba);
        while (!in.atEnd())
        {
            QString line = in.readLine().simplified();
            if (line.startsWith("http"))
            {
                result << line;
                counter++;
            }
        }
    }
    else
    if ( qba.indexOf("<html") == -1 && qba.indexOf("http") != -1)  // flat text list?
    {
        result << "ok" << "flat text m3u";
        QTextStream in(qba);
        while (!in.atEnd())
        {
            QString line = in.readLine().simplified();
            if (line.startsWith("http"))
            {
                result << line;
                counter++;
            }
        }
    }

    if (counter == 0)
    {
        result.clear();
        result << "nothing" << " No streams found" << (reply->url()).toString();
    }
    emit streamdownload_done(result);
    reply->deleteLater();
}


qm_streamFetcher::~qm_streamFetcher()
{}
