/* $Id$ */
/*
 * functions to make X11 BDF font italic.
 */
#if defined HAVE_LIMITS_H
#include <limits.h>
#endif /* HAVE_LIMITS_H */
#if !defined CHAR_BIT
#define CHAR_BIT 8
#endif /* !CHAR_BIT */
#include "common.h"
#include "italic.h"
#include "utils.h"

#define PATTERN_MAX 35 /* Number of table for pixel correction + 1 */

#define RADIXBITS CHAR_BIT*sizeof(unsigned int)
#define BDF_LEN   (RADIXBITS/8)
#define BIT_SHIFT (RADIXBITS-1)

#define BUFLEN 256

struct pattern {
  int correct;
  int height0;
  int offset0;
  char *bits0[6];
  int height;
  int offset;
  char *bits[4];
};

struct bitmap {
  char **lines;
  size_t height;
  char **tmp;
};

static Pattern ptable_0[] = {
  /* correct >= 3 */
  { 3, 5, -2, {"..@", ".@", ".@", "@..", "@."},             1, -1, {".$"}},
  { 3, 5, -4, {"...", "@@@.", "..@.", ".@.", "@."},         2, -2, {".$,", "$,"}},
  { 3, 5, -2, {".....@", "....@", "...@.", "..@.", "@@."},  1,  0, {"...@$"}},
  { 3, 4, -3, {"...@", "...@", "..@.", "@@.."},             1, -1, {".$@" }},
  { 3, 4, -3, {"..@", "..@", ".@.", "@.."},                 1,  0, {"@$"}},
  { 3, 4, -1, {".@", ".@", "@..", "@."},                    2, -1, {"$,", ",$"}},
  { 3, 4, -2, {"..@", "..@@", "@@..", ".@."},               2, -1, {".$@", "@@$"}},
  { 3, 4, -2, {".@", ".@.", "@..", "@."},                   2, -1, {"$,", ",$"}},
  /* correct >= 2 */
  { 2, 3, -2, {"....@.", "..@@.", "@@..."},                 1,  0, {"@@$..."}},
  { 2, 3, -1, {"..@", ".@", ".@."},                         1,  0, {".,$"}},
  { 2, 3, -2, {".@", ".@", "@.@"},                          1, -1, {"$@"}},
  { 2, 3, -2, {".@", ".@.", "@.."},                         1, -1, {"$."}},
  { 2, 3, -2, {".@", ".@@", "@.."},                         1, -1, {"$@"}},
  { 2, 3, -1, {"..@", ".@.", "@@."},                        1,  0, {".,$"}},
  { 2, 3, -2, {"@@.", ".@.", "@.."},                        1, -1, {"$,"}},
  { 2, 3, -1, {"@.@", ".@.", "..."},                        1,  0, {".@$"}},
  { 2, 3, -1, {"..@", ".@.", ".@@"},                        1,  0, {".,$"}},
  /* correct >= 3 */
  { 3, 6, -2, {"..@.", "...@.", "...@.", "...@.", "..@.", "@@."}, 4, -1, {"..$,", "...@", "..$,", ".$,"}},
  { 3, 4, -1, {"@..", "@.@", ".@.", ".@."},                       1,  1, {"$@"}},
  { 3, 4, -3, {"@.", "@@@", ".@", ".@"},                          1, -1, {"$."}},
  { 3, 4, -3, {"..", "@@.", "..@.", "..@."},                      2, -2, {"@,", ".$,"}},
  { 3, 3, -2, {"@..", ".@", ".@"},                                1, -1, {"$."}},
  { 3, 4, -1, {".@.", ".@.", "..@.", "..@."},                     1,  0, {".,$"}},
  { 3, 4, -1, {".@.", "@@.", "..@.", "..@."},                     2, -1, {"@,", "@@$"}},
  /* correct >= 2 */
  { 2, 3, -1, {"@..", ".@@", ".."},         1,  0, {".,@"}},
  { 2, 3, -2, {"@..", "@@.", ".@"},         1, -1, {"$."}},
  { 2, 3, -2, {"@.@", ".@", ".@"},          1, -1, {"$@"}},
  { 2, 3, -1, {".@.", ".@.", "..@."},       1,  0, {"..$"}},
  { 2, 4, -1, {"@.", "@.", ".@@", ".."},    2,  0, {",$", ".,$"}},
  { 2, 3, -1, {"@.", "@.", ".@@"},          1,  0, {",$"}},
  { 2, 3, -2, {"..", "@@.", "..@"},         1, -1, {"@,."}},
  /* least pattern */
  { 1, 4, -2, {"..@", ".@", "@@@", ".."},   2, -1, {"$@", "@,@"}},
  { 1, 3, -1, {".@..", "@.@.", ".@@."},     1, 0,  {"$"}},
  { 1, 2, -1, {".@", "@."},                 1, -1, {"$"}},
  { 0, 0, 0, {NULL}, 0, 0, {NULL}}
};

static void pattern_print(FILE *fp, Pattern *ptable);
static int search_pattern(int *x0, char **lines, int d, char **bits);
static char *bdf2bit(char *dst, const char *src, int width);
static char *bit2bdf(char *dst, const char *src, int width);
static size_t bit2hex(unsigned int *h, const char *s, unsigned int b);

/*
 * Initialize pattern table.
 */
Pattern *pattern_new(int correct)
{
  Pattern *ptable, *p;
  size_t max, i;

  max = PATTERN_MAX;
  ptable = XMALLOC(Pattern, max);
  p = ptable_0;
  i = 0;
  while (p->correct > 0) {
    if (p->correct <= correct) {
      ptable[i] = *p;
    }
    if (i < max) {
      i++;
      p++;
    }
  }
  ptable[i].correct = 0; /* sentinel */
  return ptable;
}

/*
 * Initialize pattern table.
 */
void pattern_delete(Pattern *p)
{
  XFREE(p);
}

static void pattern_print(FILE *fp, Pattern *ptable)
{
  Pattern *p;
  int n ,d, i;

  fprintf(fp,
          "=== Pattern table ===\n"
          "Following patterns is for pixel correction in slant.\n"
          "Priority between patterns depends on pattern ID.\n\n"
          );
  p = ptable;
  while (p->correct > 0) {
    fprintf(fp, "correct %d\n", p->correct);
    n = p->offset0;
    d = p->height0;
    fprintf(fp, "input (%d, %d)\n", n, d);
    for (i = 0; i < d; i++) {
      fprintf(fp, "%2d:\t%s\n", i+n, p->bits0[i]);
    }
    n = p->offset;
    d = p->height;
    fprintf(fp, "output (%d, %d)\n", n, d);
    for (i = 0; i < d; i++) {
      fprintf(fp, "%2d:\t%s\n", i+n, p->bits[i]);
    }
    fprintf(fp, "\n");
    p++;
  }
  fprintf(fp, "=== End of Pattern table ===\n");
}

void print_ptable(FILE *fp, int correct)
{
  Pattern *ptable;

  ptable = pattern_new(correct);
  pattern_print(fp, ptable);
  pattern_delete(ptable);
}

/*
 * output BITMAP image.
 */
void bitmap_output(FILE *fp, Bitmap *bitmap, int width, int height)
{
  char **lines;
  int y;

#if !defined DEBUG
  lines = bitmap_make_bdf_tmp(bitmap, width, height);
#else
  lines = bitmap->lines;
#endif /* DEBUG */
  for (y = 0; y < height; y++) {
    fprintf(fp, "%s\n", lines[y]);
  }
}

/*
 * make BDF strings.
 */
char **bitmap_make_bdf_tmp(Bitmap *bitmap, int width, int height)
{
  char **lines, **tmplines;
  char *tmp;
  int y;

  lines = bitmap->lines;
  tmplines = bitmap->tmp;
  for (y = 0; y < height; y++) {
    tmp = tmplines[y];
    bit2bdf(tmp, lines[y], width);
  }
  return tmplines;
}

/*
 * Correct the pixels to make pettern more clear after the slanting.
 * Reference the ptable at the BEGIN routine.
 */
void bitmap_correct_pixel(Bitmap *bitmap, Pattern *ptable, int width, int height, int dy)
{
  Pattern *p;
  char **lines, **tmpline;
  int x, y, j, n, d, nn, dd, rc;
  char *t;

  /* add padding pixels from both side */
  lines = bitmap->lines;
  tmpline = bitmap->tmp;
  for (y = 0; y < height; y++) {
    *tmpline[y] = '.';
    strncpy(tmpline[y]+1, lines[y], width);
    *(tmpline[y]+width+1) = '.';
    *(tmpline[y]+width+2) = '\0';
  }
  rstr(tmpline[height], '.', width+2);
  /* pattern matching with ptable */
  for (y = dy; y < height; y+=dy) {
    p = ptable;
    while (p->correct > 0) {
      n = p->offset0;
      d = p->height0;
      if (((y+n) >= 0) && ((y+n+d) <= (height+1))) {
        x = 0;
        do {
          rc = search_pattern(&x, &tmpline[y+n], d, p->bits0);
          if (rc < 0) {
            break;
          } else if (rc == 0) {
            /* matched ! */
            DEBUG_PRINT("==== MATCH with \"%d %d %s\" ==== (%d, %d)\n", p->height0, p->offset0, p->bits0[0], x, y);
            nn = p->offset;
            dd = p->height;
            for (j = 0; j < dd; j++) {
              t = p->bits[j];
              strncpy(tmpline[y+nn+j]+x, t, strlen(t));
            }
            break;
          }
          x++;
        } while (x < width-1);
      }
      p++;
    }
  }
  /* delete padding pixels from both side */
  for (y = 0; y < height; y++) {
    strncpy(lines[y], tmpline[y]+1, width);
    lines[y][width] = '\0';
  }
}

static int search_pattern(int *x0, char **lines, int d, char **bits)
{
  const char *px;
  int x;

  px = strstr(*lines+*x0, *bits);
  if (px == NULL) {
    return -1;
  }
  *x0 = px-*lines;
  while (--d > 0) {
    lines++;
    bits++;
    px = strstr(*lines+*x0, *bits);
    if (px == NULL) {
      return -1;
    }
    x = px-*lines;
    if (x != *x0) {
      break;
    }
  }
  return d;
}

/*
 * Just slant the pattern of font.
 */
void bitmap_make_slant(Bitmap *bitmap, int width, int height, int dy)
{
  char **lines, **tmplines;
  char *tmp;
  int y, len, dw, dx, dx0;

  lines = bitmap->lines;
  tmplines = bitmap->tmp;
  for (y = 0; y < height; y++) {
    tmp = tmplines[y];
    strncpy(tmp, lines[y], width);
    tmp[width] = '\0';
  }
  dw = (height-1)/dy;
  len = width+dw;
  dx0 = (dy > 0) ? dw : 0;
  for (y = 0; y < height; y++) {
    dx = dx0-y/dy;
    rstr(lines[y], '#', len);
    strncpy(lines[y]+dx, tmplines[y], width);
  }
}

/*
 * read BITMAP and convert into image.
 */
void bitmap_get_line(Bitmap *bitmap, int width, size_t h, const char *src)
{
  char **lines;

  lines = bitmap->lines;
  bdf2bit(lines[h], src, width);
}

Bitmap *bitmap_new(size_t width, size_t height)
{
  Bitmap *bitmap;
  char **p;
  size_t i;

  bitmap = XMALLOC(Bitmap, 1);
  bitmap->height = height;
  bitmap->lines = XMALLOC(char *, height);
  p = bitmap->lines;
  for (i = 0; i < height; i++) {
    *(p++) = XMALLOC(char, width+1);
  }
  bitmap->tmp = XMALLOC(char *, height+1);
  p = bitmap->tmp;
  for (i = 0; i < height+1; i++) {
    *(p++) = XMALLOC(char, width+3);
  }
  return bitmap;
}

void bitmap_delete(Bitmap *bitmap)
{
  char **p;
  size_t i, height;

  height = bitmap->height;
  p = bitmap->lines;
  for (i = 0; i < height; i++) {
    XFREE(*p);
    p++;
  }
  XFREE(bitmap->lines);
  p = bitmap->tmp;
  for (i = 0; i < height+1; i++) {
    XFREE(*p);
    p++;
  }
  XFREE(bitmap->tmp);
  XFREE(bitmap);
}

/*
 * recast BBX or FONTBOUNDINGBOX.
 */
int recast_bbx(char *dst, int *width, int *height, int *dw, int dy, const char *src)
{
  char tmp[16];
  unsigned int w, h;
  int xd, yd;

  if (sscanf(src, "%15s%u%u%d%d", tmp, &w, &h, &xd, &yd) != 5) {
    return FALSE;
  }
  *width = w;
  *height = h;
  *dw = (h-1)/abs(dy);
  if ((h > 0) && (w > 0)) {
    w += *dw;
    xd -= *dw/2;
  }
  sprintf(dst, "%s %u %u %d %d\n", tmp, w, h, xd, yd);
  return TRUE;
}

/*
 * convert BDF format into bit image.
 */
static char *bdf2bit(char *dst, const char *src, int width)
{
  const char *s, *p;
  char *d;
  int c;
  size_t num, i;

  s = src;
  d = dst;
  num = (width-1)/8+1;
  for (i = 0; i < num*2; i++) {
    c = *s;
    switch (c) {
    case '0':
      p = "....";
      break;
    case '1':
      p = "...@";
      break;
    case '2':
      p = "..@.";
      break;
    case '3':
      p = "..@@";
      break;
    case '4':
      p = ".@..";
      break;
    case '5':
      p = ".@.@";
      break;
    case '6':
      p = ".@@.";
      break;
    case '7':
      p = ".@@@";
      break;
    case '8':
      p = "@...";
      break;
    case '9':
      p = "@..@";
      break;
    case 'A':
    case 'a':
      p = "@.@.";
      break;
    case 'B':
    case 'b':
      p = "@.@@";
      break;
    case 'C':
    case 'c':
      p = "@@..";
      break;
    case 'D':
    case 'd':
      p = "@@.@";
      break;
    case 'E':
    case 'e':
      p = "@@@.";
      break;
    case 'F':
    case 'f':
      p = "@@@@";
      break;
    default:
      p = "....";
      break;
    }
    strncpy(d, p, 4);
    d += 4;
    if (c != '\0') {
      s++;
    }
  }
  *d = '\0';
  return dst;
}

/*
 * convert bit image into BDF format.
 */
static char *bit2bdf(char *dst, const char *src, int width)
{
  const char *s;
  char *d;
  unsigned int h;
  size_t num, i, l, m;

  s = src;
  d = dst;
  num = (width-1)/RADIXBITS+1;
  for (i = 0; i < num; i++) {
    l = bit2hex(&h, s, (0x1 << BIT_SHIFT));
    if (l <= RADIXBITS-8) {
      if (l > 0) {
        m = (l+7)/8;
        h >>= RADIXBITS-8*m;
        sprintf(d, "%0*X", m*2, h);
        d += m*2;
      }
      break;
    }
    sprintf(d, "%0*X", BDF_LEN*2, h);
    d += 8;
    s += l;
  }
  *d = '\0';
  return dst;
}

static size_t bit2hex(unsigned int *h, const char *s, unsigned int b)
{
  size_t i;

  i = 0;
  *h = 0x0;
  while (b > 0x0) {
    switch (*s) {
    case '\0':
      return i;
    case '@':
    case '$':
      *h |= b;
      break;
    case '.':
    case '#':
    case ',':
    default:
      break;
    }
    b >>= 1;
    s++;
    i++;
  }
  return i;
}
