/*
Licensed to the Ant-Contrib Project under one or more
 contributor license agreements.  See the NOTICE file distributed with
 this work for additional information regarding copyright ownership.
 The Ant-Contrib Project licenses this file to You under the Apache License, Version 2.0
 (the "License"); you may not use this file except in compliance with
 the License.  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

*/
package net.sf.antcontrib.taskdocs;

import java.io.File;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.Source;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;
import org.xml.sax.helpers.DefaultHandler;

import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.RootDoc;
import com.sun.javadoc.Tag;
import com.sun.javadoc.Type;

/**
 * This document writes an XML representation of the Ant related Javadoc through
 * an XSLT transform that creates xdoc files.
 *
 */
public final class TaskDoclet {
	/**
	 * Redirects parsed XHTML comment into output stream. Drops start and end
	 * document and body element.
	 */
	private static class RedirectHandler extends DefaultHandler {
		/**
		 * output handler.
		 */
		private final ContentHandler tf;

		/**
		 * Create new instance.
		 *
		 * @param tf
		 *            output handler, may not be null.
		 */
		public RedirectHandler(final TransformerHandler tf) {
			if (tf == null) {
				throw new IllegalArgumentException("tf");
			}
			this.tf = tf;
		}

		/** {@inheritDoc} */
		@Override
		public void characters(final char[] ch, final int start, final int length) throws SAXException {
			this.tf.characters(ch, start, length);
		}

		/** {@inheritDoc} */
		@Override
		public void endDocument() {
		}

		/** {@inheritDoc} */
		@Override
		public void endElement(final String namespaceURI, final String localName, final String qName)
				throws SAXException {
			if (!"body".equals(localName)) {
				this.tf.endElement(namespaceURI, localName, qName);
			}
		}

		/** {@inheritDoc} */
		@Override
		public void endPrefixMapping(final String prefix) throws SAXException {
			this.tf.endPrefixMapping(prefix);
		}

		/** {@inheritDoc} */
		@Override
		public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
			this.tf.ignorableWhitespace(ch, start, length);
		}

		/** {@inheritDoc} */
		@Override
		public void processingInstruction(final String target, final String data) throws SAXException {
			this.tf.processingInstruction(target, data);
		}

		/** {@inheritDoc} */
		@Override
		public void setDocumentLocator(final Locator locator) {
			this.tf.setDocumentLocator(locator);
		}

		/** {@inheritDoc} */
		@Override
		public void skippedEntity(String name) throws SAXException {
			this.tf.skippedEntity(name);
		}

		/** {@inheritDoc} */
		@Override
		public void startDocument() {
		}

		/** {@inheritDoc} */
		@Override
		public void startElement(final String namespaceURI, final String localName, final String qName,
				final Attributes atts) throws SAXException {
			if (!"body".equals(localName)) {
				this.tf.startElement(namespaceURI, localName, qName, atts);
			}
		}

		/** {@inheritDoc} */
		@Override
		public void startPrefixMapping(String prefix, String uri) throws SAXException {
			this.tf.startPrefixMapping(prefix, uri);
		}
	}

	/**
	 * Namespace URI for class description elements.
	 */
	private static final String NS_URI = "http://ant-contrib.sf.net/taskdocs";

	/**
	 * Namespace URI for XHTML elements.
	 */
	private static final String XHTML_URI = "http://www.w3.org/1999/xhtml";

	/**
	 * Determine if class is an Ant task.
	 *
	 * @param clazz
	 *            class to test.
	 * @return true if class is an Ant task.
	 */
	private static boolean isTask(final ClassDoc clazz) {
		if (clazz == null) {
			return false;
		}
		if ("org.apache.tools.ant.Task".equals(clazz.qualifiedTypeName())) {
			System.out.print("true");
			return true;
		}
		return TaskDoclet.isTask(clazz.superclass());

	}

	/**
	 * Determine if class is an Ant type.
	 *
	 * @param clazz
	 *            class to test.
	 * @return true if class is an Ant type.
	 */
	private static boolean isType(final ClassDoc clazz) {
		if (clazz == null) {
			return false;
		}
		if ("org.apache.tools.ant.types.DataType".equals(clazz.qualifiedTypeName())) {
			return true;
		}
		return TaskDoclet.isType(clazz.superclass());

	}

	/**
	 * Process Javadoc content.
	 *
	 * @param root
	 *            root of javadoc content.
	 * @return true if successful
	 * @throws Exception
	 *             IO exceptions and the like.
	 */
	public static boolean start(RootDoc root) throws Exception {
		final SAXTransformerFactory tf = (SAXTransformerFactory) TransformerFactory.newInstance();
		final Source typeStyle = new StreamSource(
				new File("src/taskdocs/resources/net/sf/antcontrib/taskdocs/element.xslt"));
		//
		// replace with tf.newTransformerHandler() if you want to see raw
		// generated XML.
		final TransformerHandler typeHandler = tf.newTransformerHandler(typeStyle);

		final Map<String, Type> referencedTypes = new HashMap<>();
		final Map<String, ClassDoc> documentedTypes = new HashMap<>();
		final ClassDoc[] classes = root.classes();
		for (int i = 0; i < classes.length; ++i) {
			final ClassDoc clazz = classes[i];
			if (clazz.isPublic() && !clazz.isAbstract()) {
				if (TaskDoclet.isTask(clazz) || TaskDoclet.isType(clazz)) {
					TaskDoclet.writeClass(typeHandler, clazz, referencedTypes);
					documentedTypes.put(clazz.qualifiedTypeName(), clazz);
				}
			}
		}

		final Map<String, Type> additionalTypes = new HashMap<>();
		for (final Iterator<String> iter = referencedTypes.keySet().iterator(); iter.hasNext();) {
			final String referencedName = iter.next();
			if (documentedTypes.get(referencedName) == null) {
				final ClassDoc referencedClass = root.classNamed(referencedName);
				if (referencedClass != null) {
					if (!referencedClass.qualifiedTypeName().startsWith("org.apache.tools.ant")) {
						TaskDoclet.writeClass(typeHandler, referencedClass, additionalTypes);
						documentedTypes.put(referencedClass.qualifiedTypeName(), referencedClass);
					}
				}
			}
		}

		return true;
	}

	/**
	 * Write an Ant task or type attribute (aka property).
	 *
	 * @param tf
	 *            content handler.
	 * @param method
	 *            set method for property.
	 * @throws Exception
	 *             if IO or other exception.
	 */
	private static void writeAttribute(final TransformerHandler tf, final MethodDoc method) throws Exception {
		final AttributesImpl attributes = new AttributesImpl();
		attributes.addAttribute(null, "name", "name", "CDATA", method.name().substring(3).toLowerCase(Locale.US));
		tf.startElement(TaskDoclet.NS_URI, "attribute", "attribute", attributes);
		TaskDoclet.writeType(tf, method.parameters()[0].type());
		attributes.clear();
		tf.startElement(TaskDoclet.NS_URI, "comment", "comment", attributes);
		TaskDoclet.writeDescription(tf, method.commentText());
		tf.endElement(TaskDoclet.NS_URI, "comment", "comment");
		tf.endElement(TaskDoclet.NS_URI, "attribute", "attribute");
	}

	/**
	 * Write all Ant attributes in this class and superclasses.
	 *
	 * @param tf
	 *            destination.
	 * @param clazz
	 *            class documentation.
	 * @param processed
	 *            map of processed methods.
	 * @param referencedTypes
	 *            map of referenced types.
	 * @throws Exception
	 *             if IO or other exception.
	 */
	private static void writeAttributes(final TransformerHandler tf, final ClassDoc clazz,
			final Map<String, Comparable> processed, final Map<String, Type> referencedTypes) throws Exception {
		final MethodDoc[] methods = clazz.methods();
		for (int i = 0; i < methods.length; i++) {
			final MethodDoc method = methods[i];
			if (processed.get(method.name()) == null) {
				if (method.name().startsWith("set") && method.isPublic() && method.parameters().length == 1) {
					TaskDoclet.writeAttribute(tf, method);
					referencedTypes.put(method.parameters()[0].typeName(), method.parameters()[0].type());
				}
				processed.put(method.name(), method);
			}
		}
		if (clazz.superclass() != null) {
			TaskDoclet.writeAttributes(tf, clazz.superclass(), processed, referencedTypes);
		}

	}

	/**
	 * Write an Ant nested element.
	 *
	 * @param tf
	 *            content handler.
	 * @param method
	 *            method to add element to task or type.
	 * @param name
	 *            name of nested element.
	 * @param type
	 *            type of nested element.
	 * @param referencedTypes
	 *            map of types referenced in documentation.
	 * @throws Exception
	 *             if IO or other exception.
	 */
	private static void writeChild(final TransformerHandler tf, final MethodDoc method, final String name,
			final Type type, final Map<String, Type> referencedTypes) throws Exception {
		final AttributesImpl attributes = new AttributesImpl();
		attributes.addAttribute(null, "name", "name", "CDATA", name.toLowerCase(Locale.US));
		tf.startElement(TaskDoclet.NS_URI, "child", "child", attributes);
		attributes.clear();
		tf.startElement(TaskDoclet.NS_URI, "comment", "comment", attributes);
		TaskDoclet.writeDescription(tf, method.commentText());
		tf.endElement(TaskDoclet.NS_URI, "comment", "comment");
		TaskDoclet.writeType(tf, type);
		tf.endElement(TaskDoclet.NS_URI, "child", "child");
		referencedTypes.put(type.qualifiedTypeName(), type);
	}

	/**
	 * Write all Ant nested elements in this class and superclasses.
	 *
	 * @param tf
	 *            destination.
	 * @param clazz
	 *            class documentation.
	 * @param processed
	 *            map of processed methods.
	 * @param referencedTypes
	 *            map of referenced types.
	 * @throws Exception
	 *             if IO or other exception.
	 */
	private static final void writeChildren(final TransformerHandler tf, final ClassDoc clazz,
			final Map<String, MethodDoc> processed, final Map<String, Type> referencedTypes) throws Exception {
		final MethodDoc[] methods = clazz.methods();
		for (int i = 0; i < methods.length; i++) {
			final MethodDoc method = methods[i];
			if (processed.get(method.name()) == null) {
				if (method.name().startsWith("addConfigured") && method.isPublic() && method.parameters().length == 1) {
					TaskDoclet.writeChild(tf, method, method.name().substring(13), method.parameters()[0].type(),
							referencedTypes);
				}
				if (method.name().startsWith("add") && method.isPublic() && method.parameters().length == 1) {
					TaskDoclet.writeChild(tf, method, method.name().substring(3), method.parameters()[0].type(),
							referencedTypes);
				}
				if (method.isPublic() && method.parameters().length == 0 && method.name().startsWith("create")) {
					TaskDoclet.writeChild(tf, method, method.name().substring(6), method.returnType(), referencedTypes);
				}
				processed.put(method.name(), method);
			}
		}
		if (clazz.superclass() != null) {
			TaskDoclet.writeChildren(tf, clazz.superclass(), processed, referencedTypes);
		}
	}

	/**
	 * Write Ant documentation for this class.
	 *
	 * @param tf
	 *            destination.
	 * @param clazz
	 *            class documentation.
	 * @param referencedTypes
	 *            map of referenced types.
	 * @throws Exception
	 *             if IO or other exception.
	 */
	private static void writeClass(final TransformerHandler tf, final ClassDoc clazz,
			final Map<String, Type> referencedTypes) throws Exception {
		final StreamResult result = new StreamResult(new File("src/site/xdoc/antdocs/" + clazz.name() + ".xml"));
		tf.setResult(result);
		final AttributesImpl attributes = new AttributesImpl();
		attributes.addAttribute(null, "name", "name", "CDATA", clazz.name());
		final StringBuffer firstSentence = new StringBuffer();
		final Tag[] tags = clazz.firstSentenceTags();
		for (int i = 0; i < tags.length; i++) {
			firstSentence.append(tags[i].text());
		}
		if (firstSentence.length() > 0) {
			attributes.addAttribute(null, "firstSentence", "firstSentence", "CDATA", firstSentence.toString());
		}
		tf.startDocument();
		tf.startElement(TaskDoclet.NS_URI, "class", "class", attributes);
		attributes.clear();
		tf.startElement(TaskDoclet.NS_URI, "comment", "comment", attributes);
		TaskDoclet.writeDescription(tf, clazz.commentText());
		tf.endElement(TaskDoclet.NS_URI, "comment", "comment");

		tf.startElement(TaskDoclet.NS_URI, "attributes", "attributes", attributes);
		final Map<String, Comparable> methods = new HashMap<>();
		methods.put("setProject", "setProject");
		methods.put("setRuntimeConfigurableWrapper", "setRuntimeConfigurableWrapper");
		TaskDoclet.writeAttributes(tf, clazz, methods, referencedTypes);
		tf.endElement(TaskDoclet.NS_URI, "attributes", "attributes");

		tf.startElement(TaskDoclet.NS_URI, "children", "children", attributes);
		final Map<String, MethodDoc> children = new HashMap<>();
		TaskDoclet.writeChildren(tf, clazz, children, referencedTypes);
		tf.endElement(TaskDoclet.NS_URI, "children", "children");

		tf.endElement(TaskDoclet.NS_URI, "class", "class");
		tf.endDocument();
	}

	/**
	 * Writes description.
	 *
	 * @param tf
	 *            destination.
	 * @param description
	 *            description, may contain XHTML elements.
	 * @throws SAXException
	 *             if IO or other exception.
	 */
	private static void writeDescription(final TransformerHandler tf, final String description) throws SAXException {
		if (description.indexOf('<') == -1) {
			tf.characters(description.toCharArray(), 0, description.length());
		} else {
			//
			// attempt to fabricate an XHTML fragment
			//
			final StringBuffer buf = new StringBuffer(description);
			buf.insert(0, "<body xmlns='" + TaskDoclet.XHTML_URI + "'>");
			buf.append("</body>");
			try {
				final SAXParserFactory sf = SAXParserFactory.newInstance();
				sf.setNamespaceAware(true);
				final SAXParser parser = sf.newSAXParser();
				parser.parse(new InputSource(new StringReader(buf.toString())), new RedirectHandler(tf));
			} catch (final Exception ex) {
				tf.characters(ex.toString().toCharArray(), 0, ex.toString().length());
			}
		}
	}

	/**
	 * Write a Java type.
	 *
	 * @param tf
	 *            content handler.
	 * @param type
	 *            documented type.
	 * @throws Exception
	 *             if IO or other exception.
	 */
	private static void writeType(final TransformerHandler tf, final Type type) throws Exception {
		final AttributesImpl attributes = new AttributesImpl();
		attributes.addAttribute(null, "name", "name", "CDATA", type.simpleTypeName());
		attributes.addAttribute(null, "qualifiedTypeName", "qualifiedTypeName", "CDATA", type.qualifiedTypeName());
		tf.startElement(TaskDoclet.NS_URI, "type", "type", attributes);
		final ClassDoc typeDoc = type.asClassDoc();
		if (typeDoc != null && typeDoc.commentText() != null && typeDoc.commentText().length() > 0) {
			TaskDoclet.writeDescription(tf, typeDoc.commentText());
		} else {
			tf.characters(type.typeName().toCharArray(), 0, type.typeName().length());
		}
		tf.endElement(TaskDoclet.NS_URI, "type", "type");

	}

}