/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.indexing;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.indexing.IndexerJobStats;
import org.elasticsearch.xpack.core.indexing.IndexerState;
import org.elasticsearch.xpack.core.indexing.IterationResult;

public abstract class AsyncTwoPhaseIndexer<JobPosition, JobStats extends IndexerJobStats> {
    private static final Logger logger = LogManager.getLogger((String)AsyncTwoPhaseIndexer.class.getName());
    private static final TimeValue MAX_THROTTLE_WAIT_TIME = TimeValue.timeValueHours((long)1L);
    private static final TimeValue MIN_THROTTLE_WAIT_TIME = TimeValue.timeValueMillis((long)10L);
    private final ActionListener<SearchResponse> searchResponseListener = ActionListener.wrap(this::onSearchResponse, this::finishWithSearchFailure);
    private final JobStats stats;
    private final AtomicReference<IndexerState> state;
    private final AtomicReference<JobPosition> position;
    private final ThreadPool threadPool;
    private volatile float currentMaxDocsPerSecond;
    private volatile long lastSearchStartTimeNanos = 0L;
    private volatile long lastDocCount = 0L;
    private volatile ScheduledRunnable scheduledNextSearch;

    protected AsyncTwoPhaseIndexer(ThreadPool threadPool, AtomicReference<IndexerState> initialState, JobPosition initialPosition, JobStats jobStats) {
        this.threadPool = threadPool;
        this.state = initialState;
        this.position = new AtomicReference<JobPosition>(initialPosition);
        this.stats = jobStats;
    }

    public IndexerState getState() {
        return this.state.get();
    }

    public JobPosition getPosition() {
        return this.position.get();
    }

    public JobStats getStats() {
        return this.stats;
    }

    public synchronized IndexerState start() {
        this.state.compareAndSet(IndexerState.STOPPED, IndexerState.STARTED);
        return this.state.get();
    }

    public synchronized IndexerState stop() {
        IndexerState indexerState = this.state.updateAndGet(previousState -> {
            if (previousState == IndexerState.INDEXING) {
                return IndexerState.STOPPING;
            }
            if (previousState == IndexerState.STARTED) {
                return IndexerState.STOPPED;
            }
            return previousState;
        });
        this.runSearchImmediately();
        return indexerState;
    }

    public synchronized boolean abort() {
        IndexerState prevState = this.state.getAndUpdate(prev -> IndexerState.ABORTING);
        return prevState == IndexerState.STOPPED || prevState == IndexerState.STARTED;
    }

    public synchronized boolean maybeTriggerAsyncJob(long now) {
        IndexerState currentState = this.state.get();
        switch (currentState) {
            case INDEXING: 
            case STOPPING: 
            case ABORTING: {
                logger.warn("Schedule was triggered for job [" + this.getJobId() + "], but prior indexer is still running (with state [" + (Object)((Object)currentState) + "]");
                return false;
            }
            case STOPPED: {
                logger.debug("Schedule was triggered for job [" + this.getJobId() + "] but job is stopped.  Ignoring trigger.");
                return false;
            }
            case STARTED: {
                logger.debug("Schedule was triggered for job [" + this.getJobId() + "], state: [" + (Object)((Object)currentState) + "]");
                ((IndexerJobStats)this.stats).incrementNumInvocations(1L);
                if (this.state.compareAndSet(IndexerState.STARTED, IndexerState.INDEXING)) {
                    this.threadPool.executor("generic").execute(() -> this.onStart(now, (ActionListener<Boolean>)ActionListener.wrap(r -> {
                        assert (r != null);
                        if (r.booleanValue()) {
                            this.nextSearch();
                        } else {
                            this.onFinish((ActionListener<Void>)ActionListener.wrap(onFinishResponse -> this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure), onFinishFailure -> this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure)));
                        }
                    }, this::finishWithFailure)));
                    logger.debug("Beginning to index [" + this.getJobId() + "], state: [" + (Object)((Object)currentState) + "]");
                    return true;
                }
                logger.debug("Could not move from STARTED to INDEXING state because current state is [" + (Object)((Object)this.state.get()) + "]");
                return false;
            }
        }
        logger.warn("Encountered unexpected state [" + (Object)((Object)currentState) + "] while indexing");
        throw new IllegalStateException("Job encountered an illegal state [" + (Object)((Object)currentState) + "]");
    }

    protected boolean triggerSaveState() {
        return ((IndexerJobStats)this.stats).getNumPages() > 0L && ((IndexerJobStats)this.stats).getNumPages() % 50L == 0L;
    }

    protected void rethrottle() {
        if (this.getMaxDocsPerSecond() == this.currentMaxDocsPerSecond) {
            return;
        }
        this.reQueueThrottledSearch();
    }

    protected void runSearchImmediately() {
        ScheduledRunnable runnable = this.scheduledNextSearch;
        if (runnable != null) {
            runnable.reschedule(TimeValue.ZERO);
        }
    }

    protected long getTimeNanos() {
        return System.nanoTime();
    }

    protected ScheduledRunnable getScheduledNextSearch() {
        return this.scheduledNextSearch;
    }

    protected float getMaxDocsPerSecond() {
        return -1.0f;
    }

    protected abstract String getJobId();

    protected abstract IterationResult<JobPosition> doProcess(SearchResponse var1);

    protected abstract void onStart(long var1, ActionListener<Boolean> var3);

    protected abstract void doNextSearch(long var1, ActionListener<SearchResponse> var3);

    protected abstract void doNextBulk(BulkRequest var1, ActionListener<BulkResponse> var2);

    protected abstract void doSaveState(IndexerState var1, JobPosition var2, Runnable var3);

    protected abstract void onFailure(Exception var1);

    protected abstract void onFinish(ActionListener<Void> var1);

    protected void afterFinishOrFailure() {
    }

    protected void onStop() {
    }

    protected abstract void onAbort();

    private void finishWithSearchFailure(Exception exc) {
        ((IndexerJobStats)this.stats).incrementSearchFailures();
        this.onFailure(exc);
        this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure);
    }

    private void finishWithIndexingFailure(Exception exc) {
        ((IndexerJobStats)this.stats).incrementIndexingFailures();
        this.onFailure(exc);
        this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure);
    }

    private void finishWithFailure(Exception exc) {
        this.onFailure(exc);
        this.finishAndSetState();
        this.afterFinishOrFailure();
    }

    private IndexerState finishAndSetState() {
        AtomicBoolean callOnStop = new AtomicBoolean(false);
        AtomicBoolean callOnAbort = new AtomicBoolean(false);
        IndexerState updatedState = this.state.updateAndGet(prev -> {
            callOnAbort.set(false);
            callOnStop.set(false);
            switch (prev) {
                case INDEXING: {
                    return IndexerState.STARTED;
                }
                case STOPPING: {
                    callOnStop.set(true);
                    return IndexerState.STOPPED;
                }
                case ABORTING: {
                    callOnAbort.set(true);
                    return IndexerState.ABORTING;
                }
                case STOPPED: {
                    return IndexerState.STOPPED;
                }
            }
            throw new IllegalStateException("Indexer job encountered an illegal state [" + (Object)prev + "]");
        });
        if (callOnStop.get()) {
            this.onStop();
        } else if (callOnAbort.get()) {
            this.onAbort();
        }
        return updatedState;
    }

    private void onSearchResponse(SearchResponse searchResponse) {
        block10: {
            ((IndexerJobStats)this.stats).markEndSearch();
            try {
                if (!this.checkState(this.getState())) {
                    return;
                }
                if (searchResponse == null) {
                    logger.debug("No indexing necessary for job [{}], saving state and shutting down.", (Object)this.getJobId());
                    this.onFinish((ActionListener<Void>)ActionListener.wrap(r -> this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure), e -> this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure)));
                    return;
                }
                assert (searchResponse.getShardFailures().length == 0);
                ((IndexerJobStats)this.stats).markStartProcessing();
                ((IndexerJobStats)this.stats).incrementNumPages(1L);
                long numDocumentsBefore = ((IndexerJobStats)this.stats).getNumDocuments();
                IterationResult<JobPosition> iterationResult = this.doProcess(searchResponse);
                this.lastDocCount = ((IndexerJobStats)this.stats).getNumDocuments() - numDocumentsBefore;
                if (iterationResult.isDone()) {
                    logger.debug("Finished indexing for job [{}], saving state and shutting down.", (Object)this.getJobId());
                    this.position.set(iterationResult.getPosition());
                    ((IndexerJobStats)this.stats).markEndProcessing();
                    this.onFinish((ActionListener<Void>)ActionListener.wrap(r -> this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure), e -> this.doSaveState(this.finishAndSetState(), this.position.get(), this::afterFinishOrFailure)));
                    return;
                }
                BulkRequest bulkRequest = new BulkRequest();
                iterationResult.getToIndex().forEach(arg_0 -> ((BulkRequest)bulkRequest).add(arg_0));
                ((IndexerJobStats)this.stats).markEndProcessing();
                if (bulkRequest.numberOfActions() > 0) {
                    ((IndexerJobStats)this.stats).markStartIndexing();
                    this.doNextBulk(bulkRequest, (ActionListener<BulkResponse>)ActionListener.wrap(bulkResponse -> {
                        if (bulkResponse.hasFailures()) {
                            logger.warn("Error while attempting to bulk index documents: {}", (Object)bulkResponse.buildFailureMessage());
                        }
                        ((IndexerJobStats)this.stats).incrementNumOutputDocuments(bulkResponse.getItems().length);
                        Object newPosition = iterationResult.getPosition();
                        this.position.set(newPosition);
                        this.onBulkResponse((BulkResponse)bulkResponse, newPosition);
                    }, this::finishWithIndexingFailure));
                    break block10;
                }
                try {
                    JobPosition newPosition = iterationResult.getPosition();
                    this.position.set(newPosition);
                    if (this.triggerSaveState()) {
                        this.doSaveState(IndexerState.INDEXING, newPosition, this::nextSearch);
                        break block10;
                    }
                    this.nextSearch();
                }
                catch (Exception e2) {
                    this.finishWithFailure(e2);
                }
            }
            catch (Exception e3) {
                this.finishWithSearchFailure(e3);
            }
        }
    }

    private void onBulkResponse(BulkResponse response, JobPosition jobPosition) {
        ((IndexerJobStats)this.stats).markEndIndexing();
        if (!this.checkState(this.getState())) {
            return;
        }
        try {
            if (this.triggerSaveState()) {
                this.doSaveState(IndexerState.INDEXING, jobPosition, this::nextSearch);
            } else {
                this.nextSearch();
            }
        }
        catch (Exception e) {
            this.finishWithIndexingFailure(e);
        }
    }

    protected void nextSearch() {
        TimeValue executionDelay;
        this.currentMaxDocsPerSecond = this.getMaxDocsPerSecond();
        if (this.currentMaxDocsPerSecond > 0.0f && this.lastDocCount > 0L && (executionDelay = AsyncTwoPhaseIndexer.calculateThrottlingDelay(this.currentMaxDocsPerSecond, this.lastDocCount, this.lastSearchStartTimeNanos, this.getTimeNanos())).duration() > 0L) {
            logger.debug("throttling job [{}], wait for {} ({} {})", (Object)this.getJobId(), (Object)executionDelay, (Object)Float.valueOf(this.currentMaxDocsPerSecond), (Object)this.lastDocCount);
            this.scheduledNextSearch = new ScheduledRunnable(this.threadPool, executionDelay, () -> this.triggerNextSearch(executionDelay.getNanos()));
            if (this.getState().equals((Object)IndexerState.STOPPING) || this.triggerSaveState()) {
                this.runSearchImmediately();
            }
            return;
        }
        this.triggerNextSearch(0L);
    }

    private void triggerNextSearch(long waitTimeInNanos) {
        if (!this.checkState(this.getState())) {
            return;
        }
        this.scheduledNextSearch = null;
        ((IndexerJobStats)this.stats).markStartSearch();
        this.lastSearchStartTimeNanos = this.getTimeNanos();
        this.doNextSearch(waitTimeInNanos, this.searchResponseListener);
    }

    private boolean checkState(IndexerState currentState) {
        switch (currentState) {
            case INDEXING: {
                return true;
            }
            case STOPPING: {
                logger.info("Indexer job encountered [" + (Object)((Object)IndexerState.STOPPING) + "] state, halting indexer.");
                this.doSaveState(this.finishAndSetState(), this.getPosition(), this::afterFinishOrFailure);
                return false;
            }
            case STOPPED: {
                return false;
            }
            case ABORTING: {
                logger.info("Requested shutdown of indexer for job [" + this.getJobId() + "]");
                this.onAbort();
                return false;
            }
        }
        logger.warn("Encountered unexpected state [" + (Object)((Object)currentState) + "] while indexing");
        throw new IllegalStateException("Indexer job encountered an illegal state [" + (Object)((Object)currentState) + "]");
    }

    private synchronized void reQueueThrottledSearch() {
        this.currentMaxDocsPerSecond = this.getMaxDocsPerSecond();
        ScheduledRunnable runnable = this.scheduledNextSearch;
        if (runnable != null) {
            TimeValue executionDelay = AsyncTwoPhaseIndexer.calculateThrottlingDelay(this.currentMaxDocsPerSecond, this.lastDocCount, this.lastSearchStartTimeNanos, this.getTimeNanos());
            logger.trace("[{}] rethrottling job, wait {} until next search", (Object)this.getJobId(), (Object)executionDelay);
            runnable.reschedule(executionDelay);
        }
    }

    static TimeValue calculateThrottlingDelay(float docsPerSecond, long docCount, long startTimeNanos, long now) {
        if (docsPerSecond <= 0.0f) {
            return TimeValue.ZERO;
        }
        float timeToWaitNanos = (float)docCount / docsPerSecond * (float)TimeUnit.SECONDS.toNanos(1L);
        TimeValue executionDelay = TimeValue.timeValueNanos((long)Math.min(MAX_THROTTLE_WAIT_TIME.getNanos(), Math.max(0L, (long)timeToWaitNanos + startTimeNanos - now)));
        if (executionDelay.compareTo(MIN_THROTTLE_WAIT_TIME) < 0) {
            return TimeValue.ZERO;
        }
        return executionDelay;
    }

    class ScheduledRunnable {
        private final ThreadPool threadPool;
        private final Runnable command;
        private Scheduler.ScheduledCancellable scheduled;

        ScheduledRunnable(ThreadPool threadPool, TimeValue delay, Runnable command) {
            this.threadPool = threadPool;
            this.command = new RunOnce(command);
            this.scheduled = threadPool.schedule(command::run, delay, "generic");
        }

        public void reschedule(TimeValue delay) {
            if (this.scheduled.cancel()) {
                if (delay.duration() > 0L) {
                    this.scheduled = this.threadPool.schedule(this.command::run, delay, "generic");
                } else {
                    this.threadPool.executor("generic").execute(this.command::run);
                }
            }
        }
    }
}

