/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"

#define IN_BUFFER_SIZE  4096
#define OUT_BUFFER_SIZE IN_BUFFER_SIZE

#define KEY_BUFFER_SIZE 4096

static unsigned int CHUNK_MAX;

static int init_cipher(spe_cipher_request_t *cipher)
	{
	unsigned int tag = 0;

	ALIGNED(static unsigned char key_buffer[KEY_BUFFER_SIZE], SHARED_DATA_ALIGN);

	ALIGN_HINT(cipher, SHARED_DATA_ALIGN, 0);

	/* get key and IV */
	ASSERT(cipher->size <= KEY_BUFFER_SIZE);
	mfc_get(key_buffer, cipher->in_ea,
		ALIGN_CEIL(cipher->size, SHARED_DATA_ALIGN), tag, 0, 0);
	mfc_write_tag_mask(1 << tag);
	mfc_read_tag_status_all();

	spe_cipher_init(cipher, key_buffer, key_buffer + cipher->key_size);

	/* calculate the maximum size of data chunk */
	/* it must be product of SHARED_DATA_ALIGN and block size */
	CHUNK_MAX = FLOOR(IN_BUFFER_SIZE >> 1, cipher->block_size * SHARED_DATA_ALIGN);
	if (!CHUNK_MAX)
		{
		CHUNK_MAX = cipher->block_size * SHARED_DATA_ALIGN;
		}
	ASSERT(CHUNK_MAX <= IN_BUFFER_SIZE);

	return 1;
	}

static int do_cipher(spe_cipher_request_t *cipher,
	spe_cipher_result_t *result,
	uint64_t out_data_ea,
	uint64_t in_data_ea, unsigned int in_size)
	{
	unsigned int cur;
	const int tag[2] = { 0, 1, };
	unsigned int chunk_size, next_chunk_size;
	unsigned int out_size;
	uint64_t out_data_ea_top = out_data_ea;
	ALIGNED(static unsigned char in_buffer[2][IN_BUFFER_SIZE], SHARED_DATA_ALIGN);
	ALIGNED(static unsigned char out_buffer[2][OUT_BUFFER_SIZE], SHARED_DATA_ALIGN);

	ALIGN_HINT(cipher, SHARED_DATA_ALIGN, 0);
	ALIGN_HINT(result, SHARED_DATA_ALIGN, 0);

	/* perform 'double buffering' */
	cur = 0;

	/* request the first data */
	next_chunk_size = (in_size <= CHUNK_MAX) ? in_size : CHUNK_MAX;
	mfc_get(in_buffer[cur], in_data_ea,
		ALIGN_CEIL(next_chunk_size, SHARED_DATA_ALIGN), tag[cur], 0, 0);
	in_size -= next_chunk_size;
	in_data_ea += next_chunk_size;

	chunk_size = next_chunk_size;
	while ((next_chunk_size = (in_size <= CHUNK_MAX) ? in_size : CHUNK_MAX) > 0)
		{
		/* request the next data */
		mfc_get(in_buffer[cur ^ 1], in_data_ea,
			ALIGN_CEIL(next_chunk_size, SHARED_DATA_ALIGN), tag[cur ^ 1], 0, 0);
		in_size -= next_chunk_size;
		in_data_ea += next_chunk_size;

		/* wait for the current data */
		mfc_write_tag_mask(1 << tag[cur]);
		mfc_read_tag_status_all();

		/* encrypt / decrypt */
		spe_cipher_do(cipher, out_buffer[cur], &out_size,
			in_buffer[cur], chunk_size);

		/* send result */
		mfc_put(out_buffer[cur], out_data_ea,
			ALIGN_CEIL(out_size, SHARED_DATA_ALIGN), tag[cur], 0, 0);
		out_data_ea += out_size;
    
		cur ^= 1;
		chunk_size = next_chunk_size;
		}

	/* process the last chunk */
	mfc_write_tag_mask(1 << tag[cur]);
	mfc_read_tag_status_all();

	spe_cipher_do(cipher, out_buffer[cur], &out_size,
		in_buffer[cur], chunk_size);

	mfc_put(out_buffer[cur], out_data_ea,
		ALIGN_CEIL(out_size, SHARED_DATA_ALIGN), tag[cur], 0, 0);
	out_data_ea += out_size;

	/* wait for transfer of last result */
	mfc_write_tag_mask((1 << tag[0]) | (1 << tag[1]));
	mfc_read_tag_status_all();

	result->out_ea = out_data_ea_top;
	result->size = out_data_ea - out_data_ea_top;

	return 1;
	}

int spe_main(uint64_t q_in_ea, uint64_t q_out_ea)
	{
	for ( ; ; )
		{
		ALIGNED(static spe_cipher_request_t cipher, SHARED_DATA_ALIGN);
		ALIGNED(static spe_cipher_result_t result, SHARED_DATA_ALIGN);

		DPRINTF("\n");
		cell_queue_pop(q_in_ea, &cipher);

		/* execute commands */
		switch (cipher.request)
			{
			case SPE_CIPHER_COMMAND_INIT:
				DPRINTF("INIT\n");
				init_cipher(&cipher);
				break;
			case SPE_CIPHER_COMMAND_DO:
				DPRINTF("DO\n");
				do_cipher(&cipher,
					&result,
					cipher.out_ea,
					cipher.in_ea, cipher.size);
				cell_queue_push(q_out_ea, &result);
				break;
			case SPE_CIPHER_COMMAND_QUIT:
				DPRINTF("QUIT\n");
				return 0;
			default:
				/* FIXME: error handling */
				DPRINTF("Unknown request type!!!\n");
				break;
			}
		}

	return 0;
	}
