/* tlbflush.h: TLB flushing functions
 *
 * Copyright (C) 2004 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#ifndef _ASM_TLBFLUSH_H
#define _ASM_TLBFLUSH_H

#include <linux/config.h>
#include <asm/processor.h>

#define __flush_tlb()						\
do {								\
	int w;							\
	__asm__ __volatile__					\
		("	mov %1,%0		\n"		\
		 "	or %2,%0		\n"		\
		 "	mov %0,%1		\n"		\
		 : "=d"(w)					\
		 : "m"(MMUCTR), "i"(MMUCTR_IIV|MMUCTR_DIV)	\
		 : "memory"					\
		 );						\
} while (0)

#define __flush_tlb_all() __flush_tlb()
#define __flush_tlb_one(addr) __flush_tlb()


/*
 * TLB flushing:
 *
 *  - flush_tlb() flushes the current mm struct TLBs
 *  - flush_tlb_all() flushes all processes TLBs
 *  - flush_tlb_mm(mm) flushes the specified mm context TLB's
 *  - flush_tlb_page(vma, vmaddr) flushes one page
 *  - flush_tlb_range(mm, start, end) flushes a range of pages
 *  - flush_tlb_pgtables(mm, start, end) flushes a range of page tables
 *
 * ..but the i386 has somewhat limited tlb flushing capabilities,
 * and page-granular flushes are available only on i486 and up.
 */
#define flush_tlb_all()				\
do {						\
	preempt_disable();			\
	__flush_tlb_all();			\
	preempt_enable();			\
} while(0)

#define flush_tlb_mm(mm)			\
do {						\
	preempt_disable();			\
	__flush_tlb_all();			\
	preempt_enable();			\
} while(0)

#define flush_tlb_range(vma,start,end)				\
do {								\
	unsigned long __s __attribute__((unused)) = (start);	\
	unsigned long __e __attribute__((unused)) = (end);	\
	preempt_disable();					\
	__flush_tlb_all();					\
	preempt_enable();					\
} while(0)

#define flush_tlb_page(vma,addr)		\
do {						\
	preempt_disable();			\
	__flush_tlb_all();			\
	preempt_enable();			\
} while(0)


#define __flush_tlb_global()			flush_tlb_all()
#define flush_tlb()				flush_tlb_all()
#define flush_tlb_kernel_range(start, end)	flush_tlb_all()
#define flush_tlb_pgtables(mm,start,end)	flush_tlb_all()

#endif /* _ASM_TLBFLUSH_H */
