/* wordcopy.c -- definitions for memory copy functions.
   FRV-specific C version.
   Copyright (C) 2004 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Alexandre Oliva <aoliva@redhat.com>

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.  */

#include <stddef.h>
#include <memcopy.h>

/* Make sure we don't change op_t in memcopy.h without adjusting the
   asm code below.  */
typedef op_t sanity_check[sizeof(op_t) == sizeof (unsigned long) ? 1 : -1];

/* _wordcopy_fwd_aligned -- Copy block beginning at SRCP to
   block beginning at DSTP with LEN `op_t' words (not LEN bytes!).
   Both SRCP and DSTP should be aligned for memory operations on `op_t's.  */

void _wordcopy_fwd_aligned(long dstp, long srcp, size_t len)
{
  register int opsize asm ("gr3") = sizeof (op_t);

  if (! len)
    return;

  srcp -= opsize;
  dstp -= opsize;

  /* If dstp is dword-aligned, we'll have misaligned it above, so
     invert the test.  */
  if (! (dstp & sizeof (op_t)))
    {
      asm ("ldu.p @(%1,%4), gr4\n\t"
	   "subi %0,#1,%0\n\t"
	   "stu gr4, @(%2,%4)" :
	   "+r" (len), "+r" (srcp), "+r" (dstp),
	   "=m" (*(op_t *)(dstp + opsize))
	   : "r" (opsize), "m" (*(op_t *)(srcp + opsize))
	   : "gr4");

      if (! len)
	return;
    }

  if (len < 8)
    goto by_words;

  len -= 8;
  dstp -= opsize;

  /* dstp is two words off, while srcp is one, so the test below is
     inverted.  */
  if (! ((srcp & sizeof (op_t)) != (dstp & sizeof (op_t))))
    asm ("\n0:\t"
	 "ldu @(%1,%4), gr4\n\t"
	 "ldu @(%1,%4), gr5\n\t"
	 "ldu @(%1,%4), gr6\n\t"
	 "ldu @(%1,%4), gr7\n\t"
	 "ldu @(%1,%4), gr12\n\t"
	 "ldu @(%1,%4), gr13\n\t"
	 "ldu @(%1,%4), gr14\n\t"
	 "ldu.p @(%1,%4), gr15\n\t"
	 "subicc %0,#8,%0,icc0\n\t"
	 "stdu gr4, @(%2,%5)\n\t"
	 "stdu gr6, @(%2,%5)\n\t"
	 "stdu gr12, @(%2,%5)\n\t"
	 "stdu.p gr14, @(%2,%5)\n\t"
	 "bnc icc0,#2,0b\n"
	 : "+r" (len), "+r" (srcp), "+r" (dstp),
	 "=m" (*(struct { op_t p[len]; }*)(dstp + 2*opsize))
	 : "r" (opsize), "r" (2*opsize),
	 "m" (*(struct { op_t p[len]; }*)(srcp + opsize))
	 : "gr4", "gr5", "gr6", "gr7", "gr12", "gr13", "gr14", "gr15",
	 "icc0");
  else
    {
      srcp -= opsize;
      asm ("\n0:\t"
	   "lddu @(%1,%4), gr4\n\t"
	   "lddu @(%1,%4), gr6\n\t"
	   "lddu @(%1,%4), gr12\n\t"
	   "lddu.p @(%1,%4), gr14\n\t"
	   "subicc %0,#8,%0,icc0\n\t"
	   "stdu gr4, @(%2,%4)\n\t"
	   "stdu gr6, @(%2,%4)\n\t"
	   "stdu gr12, @(%2,%4)\n\t"
	   "stdu.p gr14, @(%2,%4)\n\t"
	   "bnc icc0,#2,0b\n"
	   : "+r" (len), "+r" (srcp), "+r" (dstp),
	   "=m" (*(struct { op_t p[len]; }*)(dstp + 2*opsize))
	   : "r" (2*opsize),
	   "m" (*(struct { op_t p[len]; }*)(srcp + 2*opsize))
	   : "gr4", "gr5", "gr6", "gr7", "gr12", "gr13", "gr14", "gr15",
	   "icc0");
      srcp += opsize;
    }

  len += 8;

  if (len)
    {
      dstp += opsize;
    by_words:
      asm ("\n0:\t"
	   "ldu.p @(%1,%4), gr4\n\t"
	   "subicc %0,#1,%0,icc0\n\t"
	   "stu.p gr4, @(%2,%4)\n\t"
	   "bne icc0,#2,0b\n" :
	   "+r" (len), "+r" (srcp), "+r" (dstp),
	   "=m" (*(struct { op_t p[len]; }*)(dstp + opsize))
	   : "r" (opsize), "m" (*(struct { op_t p[len]; }*)(srcp + opsize))
	   : "gr4", "icc0");
    }
}

/* _wordcopy_fwd_aligned -- Copy block beginning at SRCP to
   block beginning at DSTP with LEN `op_t' words (not LEN bytes!).
   Both SRCP and DSTP should be aligned for memory operations on `op_t's.  */

void _wordcopy_bwd_aligned(long dstp, long srcp, size_t len)
{
  register int opsize asm ("gr3") = -sizeof (op_t);

  /* We should have opsize multipliers from the memory base addresses
     below, to tell GCC exactly which memory ranges are being read
     from and written to, but we can omit them to save useless
     computation since the compiler doesn't really know where src and
     dst point to anyway.  The asm below makes absolutely sure it
     doesn't, even if it the condition in the if command above turns
     out to be false.  */
  asm ("" : "+r" (srcp), "+r" (dstp));

  if (! len)
    return;

  /* Unlike the fwd_aligned case, here we're doing actual
     pre-decrement, instead of simulating post-increment with
     pre-increment, so we don't have to adjust the pointers.  */
  if (/*!*/ (dstp & sizeof (op_t)))
    {
      asm ("ldu.p @(%1,%4), gr4\n\t"
	   "subi %0,#1,%0\n\t"
	   "stu gr4, @(%2,%4)" :
	   "+r" (len), "+r" (srcp), "+r" (dstp),
	   "=m" (*(op_t *)(dstp + opsize))
	   : "r" (opsize), "m" (*(op_t *)(srcp + opsize))
	   : "gr4");

      if (! len)
	return;
    }

  if (len < 8)
    goto by_words;

  len -= 8;

  if (/*!*/ ((srcp & sizeof (op_t)) != (dstp & sizeof (op_t))))
    asm ("\n0:\t"
	 "ldu @(%1,%4), gr15\n\t"
	 "ldu @(%1,%4), gr14\n\t"
	 "ldu @(%1,%4), gr13\n\t"
	 "ldu @(%1,%4), gr12\n\t"
	 "ldu @(%1,%4), gr7\n\t"
	 "ldu @(%1,%4), gr6\n\t"
	 "ldu @(%1,%4), gr5\n\t"
	 "ldu.p @(%1,%4), gr4\n\t"
	 "subicc %0,#8,%0,icc0\n\t"
	 "stdu gr14, @(%2,%5)\n\t"
	 "stdu gr12, @(%2,%5)\n\t"
	 "stdu gr6, @(%2,%5)\n\t"
	 "stdu.p gr4, @(%2,%5)\n\t"
	 "bnc icc0,#2,0b\n"
	 : "+r" (len), "+r" (srcp), "+r" (dstp),
	 "=m" (*(struct { op_t p[len]; }*)(dstp + 2*opsize))
	 : "r" (opsize), "r" (2*opsize),
	 "m" (*(struct { op_t p[len]; }*)(srcp + opsize))
	 : "gr4", "gr5", "gr6", "gr7", "gr12", "gr13", "gr14", "gr15",
	 "icc0");
  else
    {
      asm ("\n0:\t"
	   "lddu @(%1,%4), gr14\n\t"
	   "lddu @(%1,%4), gr12\n\t"
	   "lddu @(%1,%4), gr6\n\t"
	   "lddu.p @(%1,%4), gr4\n\t"
	   "subicc %0,#8,%0,icc0\n\t"
	   "stdu gr14, @(%2,%4)\n\t"
	   "stdu gr12, @(%2,%4)\n\t"
	   "stdu gr6, @(%2,%4)\n\t"
	   "stdu.p gr4, @(%2,%4)\n\t"
	   "bnc icc0,#2,0b\n"
	   : "+r" (len), "+r" (srcp), "+r" (dstp),
	   "=m" (*(struct { op_t p[len]; }*)(dstp + 2*opsize))
	   : "r" (2*opsize),
	   "m" (*(struct { op_t p[len]; }*)(srcp + 2*opsize))
	   : "gr4", "gr5", "gr6", "gr7", "gr12", "gr13", "gr14", "gr15",
	   "icc0");
    }

  len += 8;

  if (len)
    {
    by_words:
      asm ("\n0:\t"
	   "ldu.p @(%1,%4), gr4\n\t"
	   "subicc %0,#1,%0,icc0\n\t"
	   "stu.p gr4, @(%2,%4)\n\t"
	   "bne icc0,#2,0b\n" :
	   "+r" (len), "+r" (srcp), "+r" (dstp),
	   "=m" (*(struct { op_t p[len]; }*)(dstp + opsize))
	   : "r" (opsize), "m" (*(struct { op_t p[len]; }*)(srcp + opsize))
	   : "gr4", "icc0");
    }
}

/* For unaligned operations, we should use word operations, since
   shifting around with double words can be expensive.  The code below
   is copied straight from ../generic/wordcopy.c.
   Copyright (C) 1991, 1996 Free Software Foundation, Inc.  */

/* _wordcopy_fwd_dest_aligned -- Copy block beginning at SRCP to
   block beginning at DSTP with LEN `op_t' words (not LEN bytes!).
   DSTP should be aligned for memory operations on `op_t's, but SRCP must
   *not* be aligned.  */

void
_wordcopy_fwd_dest_aligned (dstp, srcp, len)
     long int dstp;
     long int srcp;
     size_t len;
{
  op_t a0, a1, a2, a3;
  int sh_1, sh_2;

  /* Calculate how to shift a word read at the memory operation
     aligned srcp to make it aligned for copy.  */

  sh_1 = 8 * (srcp % OPSIZ);
  sh_2 = 8 * OPSIZ - sh_1;

  /* Make SRCP aligned by rounding it down to the beginning of the `op_t'
     it points in the middle of.  */
  srcp &= -OPSIZ;

  switch (len % 4)
    {
    case 2:
      a1 = ((op_t *) srcp)[0];
      a2 = ((op_t *) srcp)[1];
      srcp -= 1 * OPSIZ;
      dstp -= 3 * OPSIZ;
      len += 2;
      goto do1;
    case 3:
      a0 = ((op_t *) srcp)[0];
      a1 = ((op_t *) srcp)[1];
      srcp -= 0 * OPSIZ;
      dstp -= 2 * OPSIZ;
      len += 1;
      goto do2;
    case 0:
      if (OP_T_THRES <= 3 * OPSIZ && len == 0)
	return;
      a3 = ((op_t *) srcp)[0];
      a0 = ((op_t *) srcp)[1];
      srcp -=-1 * OPSIZ;
      dstp -= 1 * OPSIZ;
      len += 0;
      goto do3;
    case 1:
      a2 = ((op_t *) srcp)[0];
      a3 = ((op_t *) srcp)[1];
      srcp -=-2 * OPSIZ;
      dstp -= 0 * OPSIZ;
      len -= 1;
      if (OP_T_THRES <= 3 * OPSIZ && len == 0)
	goto do0;
      goto do4;			/* No-op.  */
    }

  do
    {
    do4:
      a0 = ((op_t *) srcp)[0];
      ((op_t *) dstp)[0] = MERGE (a2, sh_1, a3, sh_2);
    do3:
      a1 = ((op_t *) srcp)[1];
      ((op_t *) dstp)[1] = MERGE (a3, sh_1, a0, sh_2);
    do2:
      a2 = ((op_t *) srcp)[2];
      ((op_t *) dstp)[2] = MERGE (a0, sh_1, a1, sh_2);
    do1:
      a3 = ((op_t *) srcp)[3];
      ((op_t *) dstp)[3] = MERGE (a1, sh_1, a2, sh_2);

      srcp += 4 * OPSIZ;
      dstp += 4 * OPSIZ;
      len -= 4;
    }
  while (len != 0);

  /* This is the right position for do0.  Please don't move
     it into the loop.  */
 do0:
  ((op_t *) dstp)[0] = MERGE (a2, sh_1, a3, sh_2);
}

/* _wordcopy_bwd_dest_aligned -- Copy block finishing right
   before SRCP to block finishing right before DSTP with LEN `op_t'
   words (not LEN bytes!).  DSTP should be aligned for memory
   operations on `op_t', but SRCP must *not* be aligned.  */

void
_wordcopy_bwd_dest_aligned (dstp, srcp, len)
     long int dstp;
     long int srcp;
     size_t len;
{
  op_t a0, a1, a2, a3;
  int sh_1, sh_2;

  /* Calculate how to shift a word read at the memory operation
     aligned srcp to make it aligned for copy.  */

  sh_1 = 8 * (srcp % OPSIZ);
  sh_2 = 8 * OPSIZ - sh_1;

  /* Make srcp aligned by rounding it down to the beginning of the op_t
     it points in the middle of.  */
  srcp &= -OPSIZ;
  srcp += OPSIZ;

  switch (len % 4)
    {
    case 2:
      srcp -= 3 * OPSIZ;
      dstp -= 1 * OPSIZ;
      a2 = ((op_t *) srcp)[2];
      a1 = ((op_t *) srcp)[1];
      len += 2;
      goto do1;
    case 3:
      srcp -= 4 * OPSIZ;
      dstp -= 2 * OPSIZ;
      a3 = ((op_t *) srcp)[3];
      a2 = ((op_t *) srcp)[2];
      len += 1;
      goto do2;
    case 0:
      if (OP_T_THRES <= 3 * OPSIZ && len == 0)
	return;
      srcp -= 5 * OPSIZ;
      dstp -= 3 * OPSIZ;
      a0 = ((op_t *) srcp)[4];
      a3 = ((op_t *) srcp)[3];
      goto do3;
    case 1:
      srcp -= 6 * OPSIZ;
      dstp -= 4 * OPSIZ;
      a1 = ((op_t *) srcp)[5];
      a0 = ((op_t *) srcp)[4];
      len -= 1;
      if (OP_T_THRES <= 3 * OPSIZ && len == 0)
	goto do0;
      goto do4;			/* No-op.  */
    }

  do
    {
    do4:
      a3 = ((op_t *) srcp)[3];
      ((op_t *) dstp)[3] = MERGE (a0, sh_1, a1, sh_2);
    do3:
      a2 = ((op_t *) srcp)[2];
      ((op_t *) dstp)[2] = MERGE (a3, sh_1, a0, sh_2);
    do2:
      a1 = ((op_t *) srcp)[1];
      ((op_t *) dstp)[1] = MERGE (a2, sh_1, a3, sh_2);
    do1:
      a0 = ((op_t *) srcp)[0];
      ((op_t *) dstp)[0] = MERGE (a1, sh_1, a2, sh_2);

      srcp -= 4 * OPSIZ;
      dstp -= 4 * OPSIZ;
      len -= 4;
    }
  while (len != 0);

  /* This is the right position for do0.  Please don't move
     it into the loop.  */
 do0:
  ((op_t *) dstp)[3] = MERGE (a0, sh_1, a1, sh_2);
}
