package examples;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;

import com.anotherbigidea.flash.SWFConstants;
import com.anotherbigidea.flash.interfaces.SWFActionBlock;
import com.anotherbigidea.flash.interfaces.SWFActions;
import com.anotherbigidea.flash.interfaces.SWFShape;
import com.anotherbigidea.flash.interfaces.SWFTagTypes;
import com.anotherbigidea.flash.structs.Color;
import com.anotherbigidea.flash.structs.Matrix;
import com.anotherbigidea.flash.structs.Rect;
import com.anotherbigidea.flash.writers.SWFWriter;
import com.anotherbigidea.flash.writers.TagWriter;

/**
 * Example of inserting a JPEG into a streaming SWF.
 * 
 * NOTE: The JPEG data must be in the "baseline" format, not "progressive". 
 *  
 * @author nmain
 */
public class StreamJPG {

    private SWFTagTypes tags;
    private int         symbolId;
    
    /**
     * @param tags the movie to insert into
     * @param baseSymbolId the initial symbol id to use
     */
    public StreamJPG( SWFTagTypes tags, int baseSymbolId ) {
        this.tags     = tags;
        this.symbolId = baseSymbolId;
    }
    
    /**
     * Set the starting symbol id that will be used for the next image(s).
     */
    public void setSymbolId( int id ) {
        this.symbolId = id;
    }

    /**
     * Get the next available symbol id.
     */
    public int getSymbolId() {
        return symbolId;
    }
    
    /**
     * Insert an image on the stage.
     * 
     * @param jpgImage the image data in "baseline" jpeg format
     * @param pixelWidth the width of the image
     * @param pixelHeight the height of the image
     * @param pixelX where to place the image shape
     * @param pixelY where to place the image shape
     * @param depth the z-order for the image shape
     */
    public void insertJPG( InputStream jpgImage,
            			   int pixelWidth,
            			   int pixelHeight,
            			   int pixelX,
            			   int pixelY,
            			   int depth ) throws IOException {
        int jpegId  = symbolId++;
        int shapeId = symbolId++;
                
        //insert the jpeg data
        tags.tagDefineBitsJPEG2( jpegId, jpgImage );
        
        //make a shape to display the image
        int width  = pixelWidth * SWFConstants.TWIPS;
        int height = pixelHeight * SWFConstants.TWIPS;
        Rect outline = new Rect( 0, 0, width, height ); 
        SWFShape shape = tags.tagDefineShape( shapeId, outline );
                
        Matrix matrix = new Matrix( SWFConstants.TWIPS,
							        SWFConstants.TWIPS,
							        0.0, 0.0, 0.0, 0.0 );

        shape.defineFillStyle( jpegId, matrix, true /*clipped*/ );
        shape.setFillStyle1(1);  //use image fill
        shape.setLineStyle(0);   //no line
        shape.line(width,0);
        shape.line(0,height);
        shape.line(-width,0);
        shape.line(0,-height);
        shape.done();
        
        //place the shape on the stage
        int x = SWFConstants.TWIPS * pixelX;
        int y = SWFConstants.TWIPS * pixelY;
        tags.tagPlaceObject( shapeId, depth, new Matrix( x, y ), null );
    }
    
    /**
     * Insert an image on the stage, show a frame and then free up the image.
     * 
     * @param jpgImage the image data in "baseline" jpeg format
     * @param pixelWidth the width of the image
     * @param pixelHeight the height of the image
     * @param pixelX where to place the image shape
     * @param pixelY where to place the image shape
     * @param depth the z-order for the image shape
     */
    public void insertJPGForOneFrame( InputStream jpgImage,
									  int pixelWidth,
									  int pixelHeight,
									  int pixelX,
									  int pixelY,
									  int depth ) throws IOException {
        
        insertJPG( jpgImage, pixelWidth, pixelHeight, pixelX, pixelY, depth );
        
        tags.tagShowFrame();
        tags.tagRemoveObject( symbolId-1, depth );
        tags.tagFreeCharacter( symbolId-1 ); //free up shape
        tags.tagFreeCharacter( symbolId-2 ); //free up image data
    }
    
    /**
     * Insert an image on the stage, show a number of frames and then free up the image.
     * 
     * @param jpgImage the image data in "baseline" jpeg format
     * @param pixelWidth the width of the image
     * @param pixelHeight the height of the image
     * @param pixelX where to place the image shape
     * @param pixelY where to place the image shape
     * @param depth the z-order for the image shape
     * @param frameCount number of frame to display the image for
     */
    public void insertJPGForNFrames( InputStream jpgImage,
									 int pixelWidth,
									 int pixelHeight,
									 int pixelX,
									 int pixelY,
									 int depth,
									 int frameCount ) throws IOException {
        
        insertJPG( jpgImage, pixelWidth, pixelHeight, pixelX, pixelY, depth );
        
        for( int i = 0; i < frameCount; i++ ) {
            tags.tagShowFrame();            
        }
        
        tags.tagRemoveObject( symbolId-1, depth );
        tags.tagFreeCharacter( symbolId-1 ); //free up shape
        tags.tagFreeCharacter( symbolId-2 ); //free up image data
    }
    
    /**
     * Test the image inliner.
     * Makes a SWF - filename is args[1].
     * Input images are all jpg/jpeg files in the directory that is given as
     * args[0].
     * 
     * This assumes a particular size and location for each image - alter for
     * your needs.  The frame rate is 12 fps - alter this as needed.
     * 
     * The image data must be baseline jpeg.  If any image is in progressive
     * format then it will appear red in the SWF player (this is a limitation
     * of the SWF player - not JavaSWF).
     * 
     * @param args
     * @throws IOException
     */
    public static void main( String[] args ) throws IOException {
        File dir = new File( args[0] );
        if( ! dir.isDirectory() ) throw new IOException( "File must be a dir: " + dir );
        
        //get all the jpeg files in the dir
        File[] imageFiles = dir.listFiles( new FilenameFilter() {
            public boolean accept(File dir, String name) {
                name = name.toLowerCase();
                return name.endsWith( ".jpg" )
                     || name.endsWith( ".jpeg" );
            }
        });
        
        
        //start the swf
        FileOutputStream out = new FileOutputStream( args[1] );
        SWFWriter swf  = new SWFWriter( out );
        TagWriter tags = new TagWriter( swf );
        
        //write the header, size and frame count are not set to -1 (since that would
        //cause the content to be buffered by the writer) but are set to high values.
        //There is some upper limit on the acceptable values for the player - 
        //find it by trial and error.
        tags.header( 5,    //flash version 5
                     Integer.MAX_VALUE,   //set length to large unknown value (to allow streaming)
                     800 * SWFConstants.TWIPS, //width
                     500 * SWFConstants.TWIPS, //height
                     1,                       //frame rate
                     16000 );                  //large "unknown" number of frames
        
        //this tag follows the header and only appears once
        tags.tagSetBackgroundColor( new Color( 255, 255, 255 ) );
        
        //image inserter
        StreamJPG inserter = new StreamJPG( tags, 1 );
        
        //insert the image files
        for( int i = 0; i < imageFiles.length; i++ ) {
            FileInputStream in = new FileInputStream( imageFiles[i] );
            
            inserter.insertJPGForOneFrame( in, 500, 300, 150, 100, 1 );
        }
        
        //write stop action to prevent looping
        SWFActions actions = tags.tagDoAction();
        SWFActionBlock block = actions.start( 0 );
        block.stop();
        block.end();
        actions.done();
        
        //end the swf
        tags.tagShowFrame();
        tags.tagEnd();
    }
}
