package org.javaswf.tools.projector;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * Tool to make a Flash Projector (executable Flash movie).
 * 
 * @author nmain
 */
public class ProjectorMaker {

    private byte[] executableData;
    private int    magicNumber1;
    private int    magicNumber2;
    private int    magicNumber3;
    private int    magicNumber4;
    
    public ProjectorMaker() {}
    
    /**
     * Load an existing projector exe and extract the executable code and
     * magic number.  This assumes that the projector consists of the 
     * executable code followed by the SWF data, the 32 bit magic number and 
     * the 32 bit SWF data length (little endian).
     * 
     * @param filename the file name of the projector executable.
     */
    public void loadExecutable( String filename ) throws IOException {
        RandomAccessFile projector = new RandomAccessFile( filename, "r" );
        int projectorSize = (int) projector.length();

        projector.seek( projectorSize - 8 );
        magicNumber1 = projector.readUnsignedByte();
        magicNumber2 = projector.readUnsignedByte();
        magicNumber3 = projector.readUnsignedByte();
        magicNumber4 = projector.readUnsignedByte();
        int s1 = projector.readUnsignedByte();
        int s2 = projector.readUnsignedByte();
        int s3 = projector.readUnsignedByte();
        int s4 = projector.readUnsignedByte();
        
        
        int sizeOfSWFData = s1 + (s2 << 8) + (s3 << 18) + (s4 << 24);        
        int sizeOfExecutable = projectorSize - sizeOfSWFData - 8;
        
        executableData = new byte[ sizeOfExecutable ];
        projector.seek( 0 );
        projector.readFully( executableData );
        projector.close();
    }
    
    /**
     * Create a projector from an existing SWF.  The SWF should be uncompressed.
     * An existing projector should have already been loaded via loadExecutable().
     * 
     * @param swfFilename the filename of the SWF to make into a projector.
     * @param projectorFilename the filename for the resulting SWF.
     */
    public void makeProjector( String swfFilename, String projectorFilename )
    	throws IOException {
        
        if( executableData == null ) throw new IOException( "No Projector loaded." );
        
        FileOutputStream out = new FileOutputStream( projectorFilename );
        FileInputStream in = new FileInputStream( swfFilename );
        
        try {
            out.write( executableData );
            
            int swfSize = 0;
            byte[] buff = new byte[1000];
            
            int read = 0;
            while((read = in.read( buff )) >= 0 ) {
                swfSize += read;
                out.write( buff, 0, read );
            }
            
            out.write( magicNumber1 );
            out.write( magicNumber2 );
            out.write( magicNumber3 );
            out.write( magicNumber4 );
            
            out.write( swfSize & 0xff );
            out.write((swfSize >> 8 ) & 0xff );
            out.write((swfSize >> 16) & 0xff );
            out.write((swfSize >> 24) & 0xff );
            
        } finally {
            in.close();
            out.close();
        }
    }
    
    /**
     * Usage: ..ProjectorMaker <swf-in> <projector-in> <projector-out>
     */
    public static void main(String[] args) throws IOException {
        String swfFilename       = args[0];
        String projectorFilename = args[1];
        String outputFilename    = args[2];
                
        ProjectorMaker maker = new ProjectorMaker();
        maker.loadExecutable( projectorFilename );
        maker.makeProjector( swfFilename, outputFilename );
    }
    
}
