/****************************************************************
 * Copyright (c) 2004, David N. Main, All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the 
 * following conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 * copyright notice, this list of conditions and the following 
 * disclaimer. 
 * 
 * 2. Redistributions in binary form must reproduce the above 
 * copyright notice, this list of conditions and the following 
 * disclaimer in the documentation and/or other materials 
 * provided with the distribution.
 * 
 * 3. The name of the author may not be used to endorse or 
 * promote products derived from this software without specific 
 * prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************/
 
package org.javaswf.swf.model;

import java.io.IOException;

import com.anotherbigidea.flash.interfaces.SWFTags;
import com.anotherbigidea.flash.writers.TagWriter;

/**
 * Represents a SWF file
 * 
 * @author Nick
 */
public class SWFMovieFile implements SWFTimelineContainer {

    private SWFTimeline timeline = new SWFTimeline( this );
    
    private int version;
    private int frameRate;
    private int twipsWidth;
    private int twipsHeight;
    private long incomingLength = 0;
    private int  incomingFrameCount = 0;
    private boolean wasCompressed = false;
    
    /**
     * Constructor when building a SWF from scratch.
     * 
     * @param version flash player version
     * @param frameRate number of frames per second
     * @param twipsWidth movie width in twips (20 per pixel)
     * @param twipsHeight movie height in twips
     */
    public SWFMovieFile( int version, 
                         int frameRate,
                         int twipsWidth,
                         int twipsHeight ) {
        this.version     = version;
        this.frameRate   = frameRate;
        this.twipsWidth  = twipsWidth;
        this.twipsHeight = twipsHeight;        
    }
    
    /**
     * Constructor when parsing an existing SWF file.
     * 
     * @param version flash player version
     * @param frameRate number of frames per second
     * @param twipsWidth movie width in twips (20 per pixel)
     * @param twipsHeight movie height in twips
     * @param incomingLength length value in the header of the parsed SWF file.
     * @param incomingFrameCount frame-count value in the header of the parsed SWF file.
     * @param wasCompressed true if the parsed file was compressed
     */
    public SWFMovieFile( int  version, 
                         int  frameRate,
                         int  twipsWidth,
                         int  twipsHeight,
                         long incomingLength,
                         int  incomingFrameCount,
                         boolean wasCompressed ) {
        this.version            = version;
        this.frameRate          = frameRate;
        this.twipsWidth         = twipsWidth;
        this.twipsHeight        = twipsHeight;
        this.incomingLength     = incomingLength;
        this.incomingFrameCount = incomingFrameCount;
        this.wasCompressed      = wasCompressed;
    }
    
    /**
     * @see org.javaswf.swf.model.SWFTimelineContainer#getTimeline()
     */
    public SWFTimeline getTimeline() {
        return timeline;
    }
    
    /**
     * @return the frame Rate.
     */
    public int getFrameRate() {
        return frameRate;
    }
    /**
     * @return the incoming Frame Count.
     */
    public int getIncomingFrameCount() {
        return incomingFrameCount;
    }
    /**
     * @return the incoming Length.
     */
    public long getIncomingLength() {
        return incomingLength;
    }
    /**
     * @return the twips Height.
     */
    public int getTwipsHeight() {
        return twipsHeight;
    }
    /**
     * @return the twips Width.
     */
    public int getTwipsWidth() {
        return twipsWidth;
    }
    /**
     * @return the version.
     */
    public int getVersion() {
        return version;
    }
    /**
     * @return true if the original file  was Compressed.
     */
    public boolean wasCompressed() {
        return wasCompressed;
    }
    
    /**
     * Write the movie file.
     * No assertion about the compression state of the movie is passed.
     * 
     * @param tagOutput the swf header/tag writer.
     */
    public void write( SWFTags tagOutput ) throws IOException {
        
        tagOutput.header( version, -1, twipsWidth, twipsHeight, frameRate, 
                          timeline.getFrameCount() );
        
        TagWriter tagWriter = new TagWriter( tagOutput );        
        timeline.write( tagWriter );
    }
}
