/****************************************************************
 * Copyright (c) 2004, David N. Main, All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the 
 * following conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 * copyright notice, this list of conditions and the following 
 * disclaimer. 
 * 
 * 2. Redistributions in binary form must reproduce the above 
 * copyright notice, this list of conditions and the following 
 * disclaimer in the documentation and/or other materials 
 * provided with the distribution.
 * 
 * 3. The name of the author may not be used to endorse or 
 * promote products derived from this software without specific 
 * prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************/
 
package org.javaswf.swf.model;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.anotherbigidea.flash.interfaces.SWFTagTypes;

/**
 * Represents a timeline - either the main timeline of a SWF file or a 
 * movieclip.
 * 
 * @author Nick
 */
public class SWFTimeline {

    private SWFTimelineContainer timelineContainer;
    
	/** SWFFrames in a sparse array */
	private List frames = new ArrayList();

	/** SWFFrames keyed by String frame label */
	private Map framesByLabel = new HashMap();
	
	/* package */ SWFTimeline( SWFTimelineContainer container ) {
	    timelineContainer = container;
	}
	
	/**
	 * @return the file or movie clip that contains this timeline
	 */
	public SWFTimelineContainer getContainer() {
	    return timelineContainer;
	}
	
	/** 
	 * Get iterator on SWFFrames in ascending frame number order.
	 * The list may contain null entries. 
	 */
	public Iterator frames() {
		return frames.iterator();
	}
	
	/**
	 * Get the number of the given frame
	 * @return frame number, -1 if the frame is not in this timeline
	 */
	public int getFrameNumber( SWFFrame frame ) {
	    if( frame.getTimeline() != this ) return -1;
	    return frames.indexOf( frame );
	}
	
	/**
	 * Get the frame with a given number, or make a new frame instance.
	 * .
	 * 
	 * @param frameNumber zero or larger
	 * @return null if no such frame exists
	 */
	public SWFFrame getFrame( int frameNumber ) {
	    while( frameNumber >= frames.size() ) {
	        frames.add( null );
	    }
	    
		SWFFrame frame = (SWFFrame) frames.get( frameNumber );
		
		if( frame == null ) {
		    frame = new SWFFrame( this );
		    frames.set( frameNumber, frame );
		}   
		
		return frame;
	}

	/**
	 * @return number of frames in the timeline.
	 */
	public int getFrameCount() {
	    return frames.size();
	}
	
	/**
	 * Insert a frame and shift subsequent frames up one slot.
	 * 
	 * @param frameNumber zero or more - frame at this slot is shifted up.
	 * @param label null if no label
	 * @return the new frame
	 */
	public SWFFrame insertFrame( int frameNumber, String label ) {
	    SWFFrame frame = new SWFFrame( this );
	    frames.add( frameNumber, frame );
	    
	    if( label != null ) {
	        labelFrame( frame, label );
	    }
	    
	    return frame;
	}

	/**
	 * Append a frame to the timeline.
	 * @param label null for no label
	 * @return the new frame
	 */
	public SWFFrame appendFrame( String label ) {
	    SWFFrame frame = new SWFFrame( this );
	    frames.add( frame );
	    
	    if( label != null ) {
	        labelFrame( frame, label );
	    }
	    
	    return frame;
	}
	
	/**
	 * Get the frame with a given label, if it exists.
	 * 
	 * @param frameLabel label of the frame
	 * @return null if no such frame exists
	 */
	public SWFFrame getFrame( String frameLabel ) {
		return (SWFFrame) framesByLabel.get( frameLabel );
	}
 
	/**
	 * Sets the label of a frame if the frame is in this timeline.
	 */
	public void labelFrame( SWFFrame frame, String label ) {
	    if( frame.getTimeline() != this ) return;

	    String oldLabel = frame.getFrameLabel();
	    if( oldLabel != null ) {
	        framesByLabel.remove( oldLabel );
	    }

	    frame.setLabel( label );	    
	    framesByLabel.put( label, frame );
	}
	
	/**
	 * Write the timeline to the given output
	 * @param tagOutput tag output to write to
	 */
	public void write( SWFTagTypes tagOutput ) throws IOException {
	    
	    for( Iterator i = frames(); i.hasNext(); ) {
            SWFFrame frame = (SWFFrame) i.next();
        
            if( frame != null ) {                
                frame.write( tagOutput );
            } else {            
                tagOutput.tagShowFrame();
            }
        }
	    
	    tagOutput.tagEnd();
	}
}
