/**
 * License Agreement.
 *
 * Rich Faces - Natural Ajax for Java Server Faces (JSF)
 *
 * Copyright (C) 2007 Exadel, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 */

package org.richfaces.realworld.fileupload;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.core.Events;
import org.richfaces.event.UploadEvent;
import org.richfaces.model.UploadItem;
import org.richfaces.realworld.domain.Image;
import org.richfaces.realworld.domain.User;
import org.richfaces.realworld.util.SelectionHelper;

import com.drew.imaging.jpeg.JpegMetadataReader;
import com.drew.imaging.jpeg.JpegProcessingException;
import com.drew.metadata.Directory;
import com.drew.metadata.Metadata;
import com.drew.metadata.MetadataException;
import com.drew.metadata.exif.ExifDirectory;
import com.drew.metadata.jpeg.JpegDirectory;

/**
 * @author $Autor$
 *
 */
@Name("fileUploadBean")
@Scope(ScopeType.CONVERSATION)

public class FileUploadBean implements Serializable{
	
	@In("#{messages['samplename']}") private String SAMPLE_NAME;

	@In("#{messages['sampledesc']}") private String SAMPLE_DESCRIPTION;

	private static final String ADD_IMAGE_EVENT = "addImage";

	private static final long serialVersionUID = 4969087557225414955L;

	@In
	private User user;
	
	@In(create=true) @Out
	FileWrapper fileWrapper;
	
	@In(create=true)
	private FileManager fileManager;
	
	@In @Out
	private SelectionHelper selectionHelper;

	public synchronized void listener(UploadEvent event) throws Exception{
	    UploadItem item = event.getUploadItem();
	    FileItem file = new FileItem();
	    file.setLength(item.getData().length);
	    file.setData(item.getData());
	    file.setSelected(false);
	    Image image = new Image();
	    image.setUploaded(new Date());
	    image.setDescription(SAMPLE_DESCRIPTION);
	    image.setName(SAMPLE_NAME);
	    image.setSize(file.getLength() / 1024);
	    image.setPath(user.getLogin() + "/" + selectionHelper.getSelectedAlbum().getName() + "/" + item.getFileName());
	    image.setAlbumName(selectionHelper.getSelectedAlbum().getName());
	    image.setAlbum(selectionHelper.getSelectedAlbum());
	    try {
			InputStream in = new ByteArrayInputStream(file.getData());
			Metadata metadata = JpegMetadataReader.readMetadata(in);
			Directory exifDirectory = metadata.getDirectory(ExifDirectory.class);
			Directory jpgDirectory = metadata.getDirectory(JpegDirectory.class);
			
			if(exifDirectory.containsTag(ExifDirectory.TAG_MODEL)){
				String cameraModel = exifDirectory.getString(ExifDirectory.TAG_MODEL);
				image.setCameraModel(cameraModel);
			}
			if(exifDirectory.containsTag(ExifDirectory.TAG_EXIF_IMAGE_WIDTH) && exifDirectory.containsTag(ExifDirectory.TAG_EXIF_IMAGE_HEIGHT)){
				int width = exifDirectory.getInt(ExifDirectory.TAG_EXIF_IMAGE_WIDTH);
				image.setWidth(width);
				int height = exifDirectory.getInt(ExifDirectory.TAG_EXIF_IMAGE_HEIGHT);
				image.setHeight(height);
			}else{
				if(jpgDirectory.containsTag(JpegDirectory.TAG_JPEG_IMAGE_HEIGHT)){
					int width = jpgDirectory.getInt(JpegDirectory.TAG_JPEG_IMAGE_WIDTH);
					image.setWidth(width);
					int height = jpgDirectory.getInt(JpegDirectory.TAG_JPEG_IMAGE_HEIGHT);
					image.setHeight(height);
				}
			}
			if(exifDirectory.containsTag(ExifDirectory.TAG_DATETIME_ORIGINAL)){
				Date time = exifDirectory.getDate(ExifDirectory.TAG_DATETIME_ORIGINAL);
				image.setCreated(time);
			}
		}
		catch (JpegProcessingException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		} catch (MetadataException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		file.setImage(image);
	    fileWrapper.getFiles().add(file);
	}
	
	public void clearUploadData() {
		fileWrapper.getFiles().clear();
	}
	
	public void clearSelectedData() {
		for(int i = 0; i < fileWrapper.getFiles().size(); i++){
			if(fileWrapper.getFiles().get(i).isSelected()){
				fileWrapper.getFiles().remove(i);
			}
		}
	}
	
	public void storeSelectedData() {
		List<FileItem> temp = new ArrayList<FileItem>();
		for(FileItem file:fileWrapper.getFiles()){
			if(file.isSelected()){
				//Save file to disk
				String fileName = fileManager.transformToServerPath(file.getImage().getPath());
				try {
					fileManager.addImage(fileName , file.getData());
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				Events.instance().raiseEvent(ADD_IMAGE_EVENT, file.getImage());
			}else{
				temp.add(file);
			}
		}
		fileWrapper.getFiles().clear();
		fileWrapper.getFiles().addAll(temp);
	}
	
	public void storeAll() {
		for(FileItem file:fileWrapper.getFiles()){
			//Save file to disk
			String fileName = fileManager.transformToServerPath(file.getImage().getPath());
			try {
				fileManager.addImage(fileName , file.getData());
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			//Update domain model
			Events.instance().raiseEvent(ADD_IMAGE_EVENT, file.getImage());
		}
		fileWrapper.getFiles().clear();
	}
	
}