/**
 * License Agreement.
 *
 * JBoss RichFaces - Ajax4jsf Component Library
 *
 * Copyright (C) 2007 Exadel, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
package org.richfaces.testng;

import java.util.HashMap;
import java.util.Map;

import org.ajax4jsf.template.Template;
import org.richfaces.AutoTester;
import org.richfaces.SeleniumTestBase;
import org.testng.Assert;
import org.testng.annotations.Test;

public class TabPanelTest extends SeleniumTestBase {

    private static final String FORM_ID = "_form:";

    private static final String RESET_METHOD = "#{panelBean.reset}";

    private final static String INIT_AJAX_CORE_TEST = "#{panelBean.initAjaxCoreTest}";

    private static Map<String, String> params = new HashMap<String, String>();

    static {
        params.put("parameter1", "value1");
        params.put("parameter2", "value2");
        params.put("parameter3", "value3");
    }

    @Test
    public void testRichTabPanelComponent(Template template) {
        renderPage(template, RESET_METHOD);
        String parentId = getParentId() + FORM_ID;
        String linkId = parentId + "tab2_lbl";
        String tabId1 = parentId + "tab1";
        String tabId2 = parentId + "tab2";
        String tabId4 = parentId + "tab4";

        writeStatus("Click on tab2");
        clickById(linkId);
        waitForAjaxCompletion();
        AssertTextEquals(tabId2, "Tab two");
        AssertVisible(tabId2);
        AssertNotVisible(tabId4);

        writeStatus("Click on tab1");
        linkId = parentId + "tab1_lbl";
        clickCommandAndWait(linkId);
        AssertTextEquals(tabId1, "Tab one");
        AssertVisible(tabId1);
        AssertNotVisible(tabId4);

        writeStatus("Click on tab3");
        linkId = parentId + "tab3_lbl";
        clickById(linkId);
        AssertVisible(tabId1);

        writeStatus("Click on tab4");
        linkId = parentId + "tab4_lbl";
        clickById(linkId);
        AssertVisible(tabId4);
        AssertNotVisible(tabId1);
    }

    @Test
    public void testNestedParams(Template template) {
        AutoTester tester = getAutoTester(this);
        tester.renderPage(template, INIT_AJAX_CORE_TEST);
        writeStatus("Test component encodes nested f:param tags and their values are present as request parameters");
        tester.testRequestParameters(params);
    }

    @Test
    public void testReRenderAttribute(Template template) {
        AutoTester tester = getAutoTester(this);
        tester.renderPage(template, INIT_AJAX_CORE_TEST);
        writeStatus("Test component re-renders another components");
        tester.testReRender();
    }

    @Test
    public void testSubmissionModesAndListeners(Template template) {
        renderPage(template, RESET_METHOD);

        String parentId = getParentId();

        testListener(parentId);
        testSubmissionModes(parentId);

    }

    private void testListener(String parentId) {

        String linkId = parentId + FORM_ID + "tab2_lbl";
        String inputId = parentId + FORM_ID + "_value";

        writeStatus("Click on tab2");
        clickById(linkId);
        waitForAjaxCompletion();
        AssertTextEquals(inputId, "tab2", "Listener for the second tab [ajax mode] has not been called");

        writeStatus("Click on tab1");
        linkId = parentId + FORM_ID + "tab1_lbl";
        clickCommandAndWait(linkId);
        AssertTextEquals(inputId, "tab1", "Listener for the first tab [server mode] has not been called");
    }

    private void testSubmissionModes(String parentId) {
        String tabId1 = parentId + FORM_ID + "tab1";
        String tabId2 = parentId + FORM_ID + "tab2";
        String tabId4 = parentId + FORM_ID + "tab4";

        reset(parentId);

        // Set input for the first tab
        setValueById(tabId1 + "_input", "text1");
        clickById(tabId2 + "_lbl");
        waitForAjaxCompletion();
        checkDecodes(parentId, "text1", "", "");

        // Set input for the second and 4th tabs
        setValueById(tabId2 + "_input", "text2");
        setValueById(tabId4 + "_input", "text4");
        clickById(tabId1 + "_lbl");
        waitForPageToLoad();
        checkDecodes(parentId, "text1", "text2", "text4");

        // Reset model. Swtich to tyhe first tab
        reset(parentId);

        // Switch to 4th tab
        clickById(tabId4 + "_lbl");
        setValueById(tabId4 + "_input", "text");
        submit(parentId);
        AssertVisible(tabId4, "Tab4 (client) should be kept as active after form submition");
        checkDecodes(parentId, "", "", "text");

        // Switch to 2nd tab
        clickById(tabId2 + "_lbl");
        waitForAjaxCompletion();
        setValueById(tabId4 + "_input", "text4");
        setValueById(tabId2 + "_input", "text2");
        ajaxSubmit(parentId);
        checkDecodes(parentId, "", "text2", "text4");

    }

    private void checkDecodes(String parentId, String input1, String input2, String input4) {
        String inputsId = parentId + FORM_ID + "_inputs";
        String inputs = getTextById(inputsId);

        if (!inputs.contains("tab1=" + input1)) {
            Assert.fail("Decode or update model for the first tab processed incorrect. The model should contain ['"
                    + input1 + "'] value submitted from the tab");
        }
        if (!inputs.contains("tab2=" + input2)) {
            Assert.fail("Decode or update model for the second tab processed incorrect. The model should contain ['"
                    + input2 + "'] value submitted from the tab");
        }
        if (!inputs.contains("tab4=" + input4)) {
            Assert.fail("Decode or update model for the 4th tab processed incorrect. The model should contain ['"
                    + input4 + "'] value submitted from the tab");
        }
    }

    private void submit(String parentId) {
        String commandId = parentId + FORM_ID + "submit";
        clickCommandAndWait(commandId);
    }

    private void ajaxSubmit(String parentId) {
        String commandId = parentId + FORM_ID + "ajaxSubmit";
        clickAjaxCommandAndWait(commandId);
    }

    private void reset(String parentId) {
        String commandId = parentId + "controls:reset";
        clickCommandAndWait(commandId);
    }

    @Override
    public void sendAjax() {
        clickAjaxCommandAndWait(getAutoTester(this).getClientId("") + "tab2_lbl");
    }

    @Override
    public String getAutoTestUrl() {
        return "pages/tabPanel/tabPanelAutoTest.xhtml";
    }

    @Override
    public String getTestUrl() {
        return "pages/tabPanel/tabPanelTest.xhtml";
    }

}
