/**********************************************************************
This software program is available to you under a choice of one of two 
licenses. You may choose to be licensed under either the GNU General Public 
License (GPL) Version 2, June 1991, available at 
http://www.fsf.org/copyleft/gpl.html, or the Intel BSD + Patent License, the 
text of which follows:

Recipient has requested a license and Intel Corporation ("Intel") is willing
to grant a license for the software entitled Linux Base Driver for the 
Intel(R) PRO/100 Family of Adapters (e100) (the "Software") being provided 
by Intel Corporation. The following definitions apply to this license:

"Licensed Patents" means patent claims licensable by Intel Corporation which 
are necessarily infringed by the use of sale of the Software alone or when 
combined with the operating system referred to below.

"Recipient" means the party to whom Intel delivers this Software.

"Licensee" means Recipient and those third parties that receive a license to 
any operating system available under the GNU Public License version 2.0 or 
later.

Copyright (c) 1999 - 2002 Intel Corporation.
All rights reserved.

The license is provided to Recipient and Recipient's Licensees under the 
following terms.

Redistribution and use in source and binary forms of the Software, with or 
without modification, are permitted provided that the following conditions 
are met:

Redistributions of source code of the Software may retain the above 
copyright notice, this list of conditions and the following disclaimer.

Redistributions in binary form of the Software may reproduce the above 
copyright notice, this list of conditions and the following disclaimer in 
the documentation and/or materials provided with the distribution.

Neither the name of Intel Corporation nor the names of its contributors 
shall be used to endorse or promote products derived from this Software 
without specific prior written permission.

Intel hereby grants Recipient and Licensees a non-exclusive, worldwide, 
royalty-free patent license under Licensed Patents to make, use, sell, offer 
to sell, import and otherwise transfer the Software, if any, in source code 
and object code form. This license shall include changes to the Software 
that are error corrections or other minor changes to the Software that do 
not add functionality or features when the Software is incorporated in any 
version of an operating system that has been distributed under the GNU 
General Public License 2.0 or later. This patent license shall apply to the 
combination of the Software and any operating system licensed under the GNU 
Public License version 2.0 or later if, at the time Intel provides the 
Software to Recipient, such addition of the Software to the then publicly 
available versions of such operating systems available under the GNU Public 
License version 2.0 or later (whether in gold, beta or alpha form) causes 
such combination to be covered by the Licensed Patents. The patent license 
shall not apply to any other combinations which include the Software. NO 
hardware per se is licensed hereunder.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY 
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED 
AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR 
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**********************************************************************/

#ifdef IDIAG_PRO_SUPPORT

// include files
#include "e100.h"
#include "e100_config.h"
#include "idiag_pro.h"
#include "idiag_e100.h"

extern __u16 ReadEEprom(bd_config_t *bdp, __u16 reg); 

// static functions declarations
static idiag_pro_stat_t e100_diag_reset (struct net_device *dev, idiag_e100_diag_reset_test_param_t *param);
static idiag_pro_stat_t e100_diag_selftest (struct net_device *dev, idiag_e100_diag_self_test_param_t *param);
static idiag_pro_stat_t e100_diag_eeprom (struct net_device *dev, idiag_e100_diag_eeprom_test_param_t *param);
static idiag_pro_stat_t e100_diag_loopback (struct net_device *dev, idiag_e100_diag_loopback_test_param_t *param); 
static idiag_pro_stat_t e100_diag_link (struct net_device *dev, idiag_e100_diag_link_test_param_t *param);
static u32 e100_diag_rcv_loopback_pkt (bd_config_t *bdp) ;
static idiag_pro_stat_t e100_diag_perform_loopback (struct net_device *dev, u32 loopback_mode);
static unsigned char e100_diag_config_loopback(bd_config_t* bdp,unsigned char prepare_to_lb,u32 loopback_mode,
									unsigned char* tcb_extended,unsigned char* dynamic_tbd);
static unsigned char e100_diag_loopback_alloc (bd_config_t *bdp);
static void e100_diag_loopback_cu_ru_exec (bd_config_t *bdp); 
static u32 e100_diag_check_loopback_pkt (__u8 *datap);
static void e100_diag_loopback_free (bd_config_t *bdp);

#define LB_PACKET_SIZE 1500

#define E100_WAIT_SCB(bdp)	if(e100_wait_scb((bdp)) == B_FALSE){ \
	printk("%s wait_scb failed\n",bdp->device->name); }

/*****************************************************************************
 * Name:        e100_run_diag
 *
 * Description: This routine runs a diagnostic test on the hardware according 
 *              to the data received through an IOCTL from a diagnostic 
 *              application.
 *
 *
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              idiag_pro_data_t *diag_data - the data that was received from 
 *                  the diagnostic application
 *
 * Returns:
 *      idiag_pro_stat_t - the result of the operation
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/

idiag_pro_stat_t
e100_run_diag (struct net_device *dev, idiag_pro_data_t *diag_data) 
{
    idiag_pro_stat_t ret_val;
    bd_config_t* bdp = dev->priv;
    

    /* Check that the input parameters are correct */
    /* Check interface version */

    
    if (diag_data->interface_ver > IDIAG_PRO_VERSION)
        return IDIAG_PRO_STAT_NOT_SUPPORTED;
    /* Check for "identify driver" cmd */
    if (diag_data->cmd == IDIAG_PRO_IDENTIFY_DRIVER) {
        diag_data->driver_id = IDIAG_E100_DRIVER;
        diag_data->status = IDIAG_PRO_STAT_OK;
        return IDIAG_PRO_STAT_OK;
    }
    /* In all other commands, driver_id must match ours */
    if (diag_data->driver_id != IDIAG_E100_DRIVER) {
        diag_data->status = IDIAG_PRO_STAT_NOT_SUPPORTED;
        return IDIAG_PRO_STAT_NOT_SUPPORTED;
    }

	e100_isolate_driver(bdp);
    /* Check which diagnostic test needs to be run */
    switch (diag_data->cmd) {
        case IDIAG_E100_DIAG_RESET_TEST:
            ret_val = e100_diag_reset (dev, 
	        (idiag_e100_diag_reset_test_param_t *)(&diag_data->diag_param));
            break;
        case IDIAG_E100_DIAG_82559_TEST:
            ret_val = e100_diag_selftest (dev, 
                (idiag_e100_diag_self_test_param_t *)(&diag_data->diag_param));
            break;
       case IDIAG_E100_DIAG_XSUM_TEST:
            ret_val = e100_diag_eeprom (dev, 
                (idiag_e100_diag_eeprom_test_param_t *)(&diag_data->diag_param));
            break;
       case IDIAG_E100_DIAG_LOOPBACK_TEST:
            ret_val = e100_diag_loopback (dev, 
                (idiag_e100_diag_loopback_test_param_t *)(&diag_data->diag_param));
            break;
        case IDIAG_E100_DIAG_LINK_TEST:
            ret_val =  e100_diag_link (dev, 
            (idiag_e100_diag_link_test_param_t *)(&diag_data->diag_param));
            break;
        default:
            diag_data->status = IDIAG_PRO_STAT_NOT_SUPPORTED;
            ret_val = IDIAG_PRO_STAT_NOT_SUPPORTED;

    } /* switch */

	/* recover only if the device is open*/
	if(bdp->device->flags & IFF_UP)
    	e100_deisolate_driver(bdp,B_TRUE);
	else
    	e100_deisolate_driver(bdp,B_FALSE);
    return ret_val;

}


/*****************************************************************************
 * Name:        e100_diag_reset
 *
 * Description: This routine runs the reset diagnostic test on the 8255x
 *
 *
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              idiag_e100_diag_reset_test_param_t *param - the result of the
 *                   test.
 *
 * Returns:
 *      idiag_pro_stat_t - the result of the operation
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/


static idiag_pro_stat_t 
e100_diag_reset (struct net_device *dev, idiag_e100_diag_reset_test_param_t *param) 
{
    bd_config_t    *bdp;   
    bdp = dev->priv;
    

    /* SW reset */
    e100_sw_reset(bdp, PORT_SELECTIVE_RESET);


    *param = IDIAG_E100_RESET_TEST_OK;
    return IDIAG_PRO_STAT_OK;
}

/*****************************************************************************
 * Name:        e100_diag_selftest
 *
 * Description: This routine runs the self-test diagnostic test on the 8255x:
 *              Before the test is run, it stops the adapter, then runs the HW
 *              self-test, and finally starts the adapter again.
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              idiag_e100_diag_self_test_param_t *param - the result of the
 *                   test.
 *
 * Returns:
 *      idiag_pro_stat_t - the result of the operation
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static idiag_pro_stat_t 
e100_diag_selftest (struct net_device *dev, idiag_e100_diag_self_test_param_t *param) 
{
    bd_config_t    *bdp;
    idiag_pro_stat_t retval;
    u32 st_timeout, st_result;

    bdp = dev->priv;  
    
    param->SelfTestResults = 0;
    
    /* HW self-test */
    if (e100_selftest (bdp, &st_timeout, &st_result) != B_TRUE) {
        /* test failed */
        if (!st_timeout) {
            /* test did not time out - check which sub-tests failed*/
            if (st_result & CB_SELFTEST_REGISTER_BIT)
                param->SelfTestResults |= IDIAG_E100_SELF_TEST_PR; 
            if (st_result & CB_SELFTEST_DIAG_BIT)
                param->SelfTestResults |= IDIAG_E100_SELF_TEST_SER; 
            if (st_result & CB_SELFTEST_ROM_BIT)
                param->SelfTestResults |= IDIAG_E100_SELF_TEST_ROM ; 
        }
        else {
            /* test timed out */
            param->SelfTestResults = IDIAG_E100_SELF_TEST_TIMEOUT;
        }
        retval = IDIAG_PRO_STAT_TEST_FAILED;
    }
    else {
        /* test succeeded */
        param->SelfTestResults = 0;
        retval = IDIAG_PRO_STAT_OK;
    }
	/*After issuing selftest the NIC executes sw reset */
	e100_hw_reset_recover(bdp,PORT_SOFTWARE_RESET);
	
    return retval;
}


/*****************************************************************************
 * Name:        e100_diag_eeprom
 *
 * Description: This routine runs the EEPROM diagnostic test on the 8255x:
 *              It reads the EEPROM, calculates the checksum and checks that
 *              the checksum burnt in the EEPROM is correct. (the sum of all the 
 *              EEPROM words should be 0xbaba).
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              idiag_e100_diag_eeprom_test_param_t *param - the result of the
 *                   test.
 *
 * Returns:
 *      idiag_pro_stat_t - the result of the operation
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static idiag_pro_stat_t 
e100_diag_eeprom (struct net_device *dev, idiag_e100_diag_eeprom_test_param_t *param) 
{
    bd_config_t    *bdp = dev->priv;
    u16 eeprom_sum = 0, 
             eeprom_actual_csm;
    u16 idx;
    idiag_pro_stat_t retval;


    /* Read EEPROM and check validity */
    
    for (idx = 0; idx < (bdp->EEpromSize-1); idx++) {
        eeprom_sum += ReadEEprom(bdp, idx);
    } 
    eeprom_actual_csm = ReadEEprom(bdp, bdp->EEpromSize-1);

    if (eeprom_actual_csm == (EEPROM_SUM - eeprom_sum)) {
        retval = IDIAG_PRO_STAT_OK;
    }
    else {
        param->ExpectedChecksum = (EEPROM_SUM - eeprom_sum);
        param->ActualChecksum = eeprom_actual_csm;
        retval = IDIAG_PRO_STAT_TEST_FAILED;
    }
    return retval;

}

/*****************************************************************************
 * Name:        e100_diag_link
 *
 * Description: This routine runs the link diagnostic test on the 8255x:
 *
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              idiag_e100_diag_link_test_param_t *param - the result of the
 *                   test.
 *
 * Returns:
 *      idiag_pro_stat_t - the result of the operation
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static idiag_pro_stat_t 
e100_diag_link (struct net_device *dev, idiag_e100_diag_link_test_param_t *param)
{
    bd_config_t    *bdp;   
    bdp = dev->priv;

    /*  check link status */
    if (e100_GetPhyLinkState (bdp)) {
        *param = IDIAG_E100_LINK_TEST_UP;
    }
    else {
        *param = IDIAG_E100_LINK_TEST_DOWN;
    }
    return IDIAG_PRO_STAT_OK;
}


/*****************************************************************************
 * Name:        e100_diag_loopback
 *
 * Description: This routine runs the loopback diagnostic test on the 8255x:
 *
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              idiag_e100_diag_loopback_test_param_t *param - the result of the
 *                   test.
 *
 * Returns:
 *      idiag_pro_stat_t - the result of the operation
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static idiag_pro_stat_t 
e100_diag_loopback (struct net_device *dev, idiag_e100_diag_loopback_test_param_t *param) 
{
    idiag_pro_stat_t retval;
	
    param->mac = IDIAG_E100_LOOPBACK_TEST_NOT_EXEC;
    param->tcvr = IDIAG_E100_LOOPBACK_TEST_NOT_EXEC;
    param->ext = IDIAG_E100_LOOPBACK_TEST_NOT_EXEC;


    if (param->mode & IDIAG_E100_DIAG_MAC_LB) {
        if ((retval = e100_diag_perform_loopback (dev, IDIAG_E100_DIAG_MAC_LB)) != 
            IDIAG_E100_LOOPBACK_TEST_OK) {
    	    param->mac = retval;
		    return IDIAG_PRO_STAT_TEST_FAILED;
        }       
        else {
    	    param->mac = IDIAG_E100_LOOPBACK_TEST_OK;
        }
    }
    
    if (param->mode & IDIAG_E100_DIAG_TCVR_LB) {
        if ((retval = e100_diag_perform_loopback (dev, IDIAG_E100_DIAG_TCVR_LB)) != 
            IDIAG_E100_LOOPBACK_TEST_OK) {
    	    param->tcvr = retval;
		    return IDIAG_PRO_STAT_TEST_FAILED;
        }
        else {
    	    param->tcvr = IDIAG_E100_LOOPBACK_TEST_OK;
        }
    }

    if (param->mode & IDIAG_E100_DIAG_EXT_LB) {
        if ((retval = e100_diag_perform_loopback (dev, IDIAG_E100_DIAG_EXT_LB)) != 
            IDIAG_E100_LOOPBACK_TEST_OK) {
    	    param->ext = retval;
		    return IDIAG_PRO_STAT_TEST_FAILED;
        }
        else {
    	    param->ext = IDIAG_E100_LOOPBACK_TEST_OK;
        }
    }


	return IDIAG_PRO_STAT_OK;
}

/*****************************************************************************
 * Name:        e100_diag_perform_loopback
 *
 * Description: This routine puts the chip in loopback mode (internal or 
 *              external), sends a packet and checks that it was received.
 *
 * Arguments:
 *              struct net_device *dev - the net device struct associated with the device
 *              u32 loopback_mode - if equals IDIAG_E100_INTERNAL_LB, 
 *                       perform internal (MAC) loopback, otherwise perform
 *                       external (PHY) loopback.
 *              
 *
 * Returns:
 *              idiag_e100_diag_loopback_test_param_t - the param to be 
 *                       returned to the user. 
 *                 
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static idiag_pro_stat_t 
e100_diag_perform_loopback (struct net_device *dev, u32 loopback_mode)
{
        bd_config_t    *bdp = dev->priv;
        idiag_e100_diag_loopback_result_t res =IDIAG_E100_LOOPBACK_TEST_OK;

   	unsigned char dynamic_tbd = B_FALSE;
   	unsigned char extended_tcb = B_FALSE;
    

	if( ! e100_diag_loopback_alloc(bdp) )
		return IDIAG_E100_LOOPBACK_TEST_FAILED;

	/* change the config block to standard tcb and the correct loopback */
        if( e100_diag_config_loopback(bdp,B_TRUE,loopback_mode,&(extended_tcb),&(dynamic_tbd)))
                e100_config(bdp);


	e100_diag_loopback_cu_ru_exec(bdp);

	
	
        /* Check that packet was received */
        if (e100_diag_rcv_loopback_pkt (bdp) != 0) {
		res= IDIAG_E100_LOOPBACK_TEST_FAILED;
	}

        e100_diag_loopback_free(bdp);

        /* change the config block to previous tcb mode and the no loopback */
        if( e100_diag_config_loopback(bdp,B_FALSE,IDIAG_E100_DIAG_TCVR_LB,&(extended_tcb),&(dynamic_tbd) ))
                e100_config(bdp);

        return res;
}


/*****************************************************************************
 * Name:        e100_diag_config_loopback
 *
 * Description: This routine change the config block before and after
 *				preforming loopback test.
 *              
 *
 * Arguments:
 *              bdp - pointer to the device data structure
 *				prepare_to_lb - TRUE if the function is called before lb
 *				loopback_mode - the loopback mode(INTERNAL/EXTERNAL/TRCV)
 *				tcb_extended - TRUE if enable extended tcb mode
 *
 * Returns:
 *             TRUE - if the configblock was changed , FALSE otherwise.
 *
 *****************************************************************************/


static unsigned char
e100_diag_config_loopback(bd_config_t* bdp,unsigned char prepare_to_lb,u32 loopback_mode,
									unsigned char* tcb_extended,unsigned char* dynamic_tbd)
									{
	unsigned char retval =B_FALSE;

	
	if( (prepare_to_lb) ||(*tcb_extended)){
		  *tcb_extended = e100_config_tcb_ext_enable(bdp,*tcb_extended);
		  retval = *tcb_extended;
	}

	if( (prepare_to_lb) ||(*dynamic_tbd)){
		 *dynamic_tbd = e100_config_dynamic_tbd(bdp,*dynamic_tbd);
		 retval |= *dynamic_tbd;
	}	
	
	retval |= e100_config_loopback_mode(bdp,loopback_mode);

	return retval;
}

/*****************************************************************************
 * Name:        e100_diag_loopback_alloc
 *
 * Description: This routine change alloc snd initate tcb and rfd for the loopback  
 *              
 *
 * Arguments:
 *              bdp - pointer to the device data structure
 *				
 * Returns:
 *             TRUE - if allocation succeeded , otherwise FALSE .
 *
 *****************************************************************************/

static unsigned char
e100_diag_loopback_alloc (bd_config_t *bdp) 
{
        dma_addr_t dma_handle;
        tcb_t *tcb ;
        rfd_t *rfd;
        tbd_t *tbd;

        /* allocate and initiate tcb */
        if (!(tcb= pci_alloc_consistent(bdp->ppci_dev,sizeof(tcb_t)+
        	sizeof(tbd_t)+LB_PACKET_SIZE,&dma_handle)))
                return B_FALSE;
        memset(tcb, 0x00, sizeof(tcb_t)+LB_PACKET_SIZE);

	tcb->tcb_paddr = dma_handle;
         tcb->tcb_hdr.cb_status = 0;
        tcb->tcb_hdr.cb_cmd = cpu_to_le16( CB_EL_BIT | CB_TRANSMIT | CB_TX_SF_BIT);     
	tcb->tcb_hdr.cb_lnk_ptr = cpu_to_le32(tcb->tcb_paddr);
	tcb->tcb_tbd_ptr = cpu_to_le32(E100_NULL); 
        tcb->tcb_cnt = 0;
	tcb->tcb_thrshld = bdp->tx_thld;
        tcb->tcb_tbd_num = 1;
        tcb->tcb_tbd_ptr = cpu_to_le32(tcb->tcb_paddr +sizeof(tcb_t));
        tbd = (tbd_t*)((__u8*)tcb + sizeof(tcb_t));
        tbd->tbd_buf_addr = cpu_to_le32(le32_to_cpu(tcb->tcb_tbd_ptr) + sizeof(tbd_t));
        tbd->tbd_buf_cnt = cpu_to_le32(1024);

        memset((void*) ((__u8*)tbd+sizeof(tbd_t) ), 0xFF, 1024);
	memset((void* )( (__u8*)tbd+sizeof(tbd_t) +512),0xBA, 512);

	/*alocate and initiate the rfd */
        if (!((void*)rfd= pci_alloc_consistent(bdp->ppci_dev,sizeof(rfd_t),&dma_handle))){
                pci_free_consistent(bdp->ppci_dev,sizeof(tcb_t)+sizeof(tbd_t)+LB_PACKET_SIZE,tcb,
                                    tcb->tcb_paddr);
                return B_FALSE;
        }
        memset(rfd, 0x00, sizeof(rfd_t));
    
        /* init all fields in rfd */
	
        rfd->rfd_header.cb_status = 0;
        rfd->rfd_header.cb_cmd = cpu_to_le16(RFD_EL_BIT);
        //rfd->rfd_rbd_ptr = cpu_to_le32(E100_NULL);   /* not really necessary */
        rfd->rfd_act_cnt = 0;
        rfd->rfd_sz = cpu_to_le16(ETH_FRAME_LEN + CHKSUM_SIZE);
	bdp->loopback.dma_handle = dma_handle;
        bdp->loopback.tcb = tcb;
        bdp->loopback.rfd = rfd ;

        return B_TRUE; 
}

/*****************************************************************************
 * Name:        e100_diag_check_cu_ru_exec
 *
 * Description: This routine exec the cu to transmit the loopback packet 
 *					the ru to recive this packet
 *
 * Arguments:
 *             bdp- pointer to the adapter data strucure
 * Returns:
 *
 *****************************************************************************/


static void
e100_diag_loopback_cu_ru_exec (bd_config_t *bdp) 
{
    tcb_t *tcb = bdp->loopback.tcb ;
    
    /*load CU & RU base */ 
    
    E100_WAIT_SCB(bdp);
    writel(0,&bdp->scbp->scb_gen_ptr);
    e100_exec_cmd(bdp, SCB_CUC_LOAD_BASE);
    
    E100_WAIT_SCB(bdp);
    writel(0,&bdp->scbp->scb_gen_ptr);
    e100_exec_cmd(bdp, SCB_RUC_LOAD_BASE);
    
    E100_WAIT_SCB(bdp);
    writel(bdp->loopback.dma_handle, &(bdp->scbp->scb_gen_ptr));
    e100_wait_exec_cmd(bdp, SCB_RUC_START);
    
    bdp->next_cu_cmd = START_WAIT;
    
    e100_cu_start(bdp, tcb);
    bdp->last_tcbp =NULL;
}

/*****************************************************************************
 * Name:        e100_diag_check_loopback_pkt
 *
 * Description: This routine checks if a given packet is a loopback packet
 *              that was prepared as in e100_diag_prepare_loopback_pkt().
 *
 * Arguments:
 *              struct sk_buff *skb - the packet to check
 *
 * Returns:
 *              u32 - 0 if packet is a loopback packet, -1 otherwise.
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static __u32
e100_diag_check_loopback_pkt (__u8 *datap) 
{
        /* Snoop the packet for pattern */
        if( (*datap)==0xFF) {
                if(*(datap + 600) == 0xBA) {
                        return 0;
                }
        }  
        return -1;
}

/*****************************************************************************
 * Name:        e100_diag_rcv_loopback_pkt
 *
 * Description: This routine waits until one packet is received, and then 
 *              checks if it is a loopback packet.
 *
 * Arguments:
 *             bdp - pointer to the adapter data structure
 *
 * Returns:
 *              u32 - 0 if the loopback packet was received, -EAGAIN
 *                         otherwise.
 *
 * Modification log:
 * Date      Who  Description
 * --------  ---  --------------------------------------------------------
 *
 *****************************************************************************/
static u32
e100_diag_rcv_loopback_pkt (bd_config_t* bdp) 
{    
        rfd_t          *rfdp;      /* new rfd, received rfd */
        volatile u16 rfd_status;
        u32 	   retry;
    
        rfdp =bdp->loopback.rfd;

        rfd_status = le16_to_cpu(rfdp->rfd_header.cb_status);  /* get RFD's status */

        /* Poll the RFD */

        /* Wait for HW to finish Tx */
        retry = E100_CMD_WAIT;
        while ( !(rfd_status & RFD_STATUS_COMPLETE) && (retry) ) {
                mdelay(20);
                rfd_status = le16_to_cpu(rfdp->rfd_header.cb_status);  /* get RFD's status */
                retry--;
        }
    
        if (!retry)
                return 1;

        if(e100_diag_check_loopback_pkt ( ( (__u8 *) rfdp+bdp->rfd_size)) == 0)
                return 0;
        else 
                return 1;
}


/*****************************************************************************
 * Name:       e100_diag_loopback_free
 *
 * Description: This routine free the loopback structre  
 *              
 *
 * Arguments:
 *              bdp - pointer to the device data structure
 *				
 * Returns:
 *             
 *
 *****************************************************************************/

static void
e100_diag_loopback_free (bd_config_t *bdp)
{

        pci_free_consistent(bdp->ppci_dev,sizeof(tcb_t)+
        	sizeof(tbd_t)+LB_PACKET_SIZE,(bdp->loopback.tcb),
                            (bdp->loopback.tcb->tcb_paddr));
        pci_free_consistent(bdp->ppci_dev,sizeof(rfd_t),(bdp->loopback.rfd),
                            (bdp->loopback.dma_handle));
}



#endif








