package Chemistry::Isotope;

$VERSION = '0.10';
# $Id: Isotope.pm,v 1.2 2004/11/07 21:21:12 itubert Exp $

use strict;
use warnings;
use base qw(Exporter);
our @EXPORT_OK = qw(isotope_mass);
our %EXPORT_TAGS = (all => \@EXPORT_OK);

my @masses;

sub isotope_mass {
    my ($A, $Z) = @_;
    read_data() unless @masses;
    $masses[$A]{$Z};
}

sub read_data {
    local $_;
    while (<DATA>) {
        my ($A, $Z, $m) = split;
        $masses[$A]{$Z} = $m;
    }
}

1;

=head1 NAME

Chemistry::Isotope - Table of the isotopes exact mass data

=head1 SYNOPSIS

    use Chemistry::Isotope 'isotope_mass';

    my $m = isotope_mass(235, 92); # 235.043923094753

=head1 DESCRIPTION

This module contains the exact mass data from the table of the isotopes.
It has a single exportable function, isotope_mass, which returns the mass
of an atom in mass units given its mass number (A) and atomic number (Z).

The table includes masses for 2931 nuclides and is taken from
L<http://ie.lbl.gov/txt/awm95.txt> (G. Audi and A.H. Wapstra, Nucl. Phys. A595,
409, 1995)

=head1 FUNCTIONS

=head2 isotope_mass($A, $Z)

Return the mass for the atom with the given mass number and atomic number, 
or undef if the nuclide is not in the data table.

=head1 VERSION

0.10

=head1 SEE ALSO

L<Chemistry::Atom>

The PerlMol website L<http://www.perlmol.org/>

=head1 AUTHOR

Ivan Tubert-Brohman E<lt>itub@cpan.orgE<gt>

=head1 COPYRIGHT

Copyright (c) 2004 Ivan Tubert-Brohman. All rights reserved. This program is
free software; you can redistribute it and/or modify it under the same terms as
Perl itself.

=cut

__DATA__
  1	  0	1.00866492199298
  1	  1	1.00782503039393
  2	  1	2.01410177601882
  3	  1	3.01604926463989
  3	  2	3.01602930745322
  4	  1	4.027837001106
  4	  2	4.00260324913957
  4	  3	4.027182139144
  5	  1	5.03950642656
  5	  2	5.012227668438
  5	  3	5.012538996256
  5	  4	5.0407946796
  6	  1	6.04498150198
  6	  2	6.01888804400922
  6	  3	6.01512226566446
  6	  4	6.0197253011308
  7	  2	7.028030239062
  7	  3	7.01600407487034
  7	  4	7.0169292552619
  7	  5	7.029919676854
  8	  2	8.0339218496316
  8	  3	8.02248667152204
  8	  4	8.00530509043137
  8	  5	8.0246067066082
  8	  6	8.0376749601548
  9	  2	9.043822074244
  9	  3	9.02678911461238
  9	  4	9.01218215016392
  9	  5	9.01332880272394
  9	  6	9.03104013493554
 10	  2	10.052399692402
 10	  3	10.03548063581
 10	  4	10.0135337423117
 10	  5	10.0129370664454
 10	  6	10.0168531409781
 10	  7	10.04261970474
 11	  3	11.04380060336
 11	  4	11.0216576806908
 11	  5	11.0093054811256
 11	  6	11.0114337825021
 11	  7	11.026795663232
 12	  3	12.05378456442
 12	  4	12.0269201943592
 12	  5	12.0143521050554
 12	  6	12
 12	  7	12.018613216694
 12	  8	12.0344049445216
 13	  4	13.03617843954
 13	  5	13.0177802537492
 13	  6	13.003354837434
 13	  7	13.0057386305211
 13	  8	13.0248106800062
 14	  4	14.042812942696
 14	  5	14.0254043499488
 14	  6	14.0032419875412
 14	  7	14.0030740047125
 14	  8	14.0085952886955
 14	  9	14.03607108512
 15	  5	15.0310973548414
 15	  6	15.010599209241
 15	  7	15.0001088981766
 15	  8	15.0030653981087
 15	  9	15.018014071676
 16	  5	16.039807018936
 16	  6	16.0147011142748
 16	  7	16.0061009516886
 16	  8	15.9949146232717
 16	  9	16.011465452056
 16	 10	16.0257564724464
 17	  5	17.046935352424
 17	  6	17.0225841493354
 17	  7	17.0084498663982
 17	  8	16.9991315027422
 17	  9	17.0020952362151
 17	 10	17.017702743858
 18	  5	18.05614636166
 18	  6	18.026752721464
 18	  7	18.0140816792714
 18	  8	17.9991603810812
 18	  9	18.0009376335043
 18	 10	18.0056970843606
 18	 11	18.02716066826
 19	  5	19.06376852548
 19	  6	19.035244456086
 19	  7	19.017026411012
 19	  8	19.0035791963628
 19	  9	18.9984031996215
 19	 10	19.0018798832486
 19	 11	19.0138798529618
 20	  6	20.040322320152
 20	  7	20.023371057234
 20	  8	20.004076139973
 20	  9	19.9999813203309
 20	 10	19.9924401768917
 20	 11	20.007348410049
 20	 12	20.018862171594
 21	  6	21.0493830332
 21	  7	21.027085520166
 21	  8	21.0086549133404
 21	  9	20.9999488992961
 21	 10	20.993846745238
 21	 11	20.9976550574039
 21	 12	21.0117145143104
 21	 13	21.02801950362
 22	  6	22.05646842492
 22	  7	22.034439297936
 22	  8	22.009962490176
 22	  9	22.0029994824948
 22	 10	21.9913855163342
 22	 11	21.9944367866012
 22	 12	21.9995740176614
 22	 13	22.019517033556
 22	 14	22.034525181472
 23	  7	23.04047261634
 23	  8	23.015695216204
 23	  9	23.003574902186
 23	 10	22.9944673396691
 23	 11	22.9897696712815
 23	 12	22.9941248146567
 23	 13	23.0072646736014
 23	 14	23.025518145634
 24	  7	24.0504565774
 24	  8	24.0203973398
 24	  9	24.008094523268
 24	 10	23.9936145590984
 24	 11	23.9909633343421
 24	 12	23.9850419007146
 24	 13	23.999940955069
 24	 14	24.011545967871
 24	 15	24.0343534144
 25	  8	25.02909304782
 25	  9	25.012098843134
 25	 10	24.997788498948
 25	 11	24.9899543101485
 25	 12	24.9858370212263
 25	 13	24.9904286019761
 25	 14	25.004106306565
 25	 15	25.020257779054
 26	  8	26.03778875584
 26	  9	26.019635123418
 26	 10	26.000461624006
 26	 11	25.9925903979316
 26	 12	25.98259303904
 26	 13	25.986891660313
 26	 14	25.992329526691
 26	 15	26.011776779874
 26	 16	26.0279121492
 27	  9	27.02694595942
 27	 10	27.007611428378
 27	 11	26.994009623364
 27	 12	26.9843407475267
 27	 13	26.9815384428951
 27	 14	26.9867047670032
 27	 15	26.99919484185
 27	 16	27.018797758942
 28	  9	28.03564166744
 28	 10	28.012109578576
 28	 11	27.998894249474
 28	 12	27.983876654369
 28	 13	27.9819101897807
 28	 14	27.976926536733
 28	 15	27.9923123499838
 28	 16	28.004369324894
 28	 17	28.02855627572
 29	  9	29.04326383126
 29	 10	29.0193237956
 29	 11	29.002812685804
 29	 12	28.988556018828
 29	 13	28.9804448556249
 29	 14	28.9764947175347
 29	 15	28.981801386076
 29	 16	28.996607600328
 29	 17	29.014106370788
 30	 10	30.02383268124
 30	 11	30.009221744678
 30	 12	29.990466927504
 30	 13	29.9829607064576
 30	 14	29.9737702233867
 30	 15	29.9783137630335
 30	 16	29.9849027479154
 30	 17	30.004766536248
 30	 18	30.02157823842
 31	 10	31.03306516136
 31	 11	31.013591069572
 31	 12	30.996543187676
 31	 13	30.9839462200332
 31	 14	30.975363277096
 31	 15	30.9737615169432
 31	 16	30.9795544580654
 31	 17	30.992420777948
 31	 18	31.01213104946
 32	 10	32.03993584424
 32	 11	32.01964585886
 32	 12	31.99914116464
 32	 13	31.988126601148
 32	 14	31.9741480894742
 32	 15	31.9739071646849
 32	 16	31.9720706955637
 32	 17	31.9856885822698
 32	 18	31.997659673644
 32	 19	32.02190030168
 33	 11	33.0273753771
 33	 12	33.00558242984
 33	 13	32.9908748743
 33	 14	32.9780009322536
 33	 15	32.9717253149237
 33	 16	32.9714585069836
 33	 17	32.9774518143953
 33	 18	32.989930155404
 33	 19	33.007257158792
 34	 11	34.0348901865
 34	 12	34.0091251257
 34	 13	33.996929663588
 34	 14	33.9785752784006
 34	 15	33.9736359015364
 34	 16	33.9678668360372
 34	 17	33.9737619678318
 34	 18	33.9802704046924
 34	 19	33.9983896837
 34	 20	34.01417078344
 35	 11	35.04423002104
 35	 12	35.01749877046
 35	 13	34.999935587348
 35	 14	34.984583905288
 35	 15	34.9733142676941
 35	 16	34.9690321467954
 35	 17	34.9688527146179
 35	 18	34.9752567385696
 35	 19	34.9880117319186
 35	 20	35.004766536248
 36	 12	36.02243707378
 36	 13	36.00633391078
 36	 14	35.98668805192
 36	 15	35.9782596564058
 36	 16	35.967080883593
 36	 17	35.9683069462175
 36	 18	35.9675463294163
 36	 19	35.981293492315
 36	 20	35.993086375352
 36	 21	36.01492226438
 37	 12	37.03124013622
 37	 13	37.01030602432
 37	 14	36.993000491816
 37	 15	36.979613395642
 37	 16	36.971125740488
 37	 17	36.9659026044208
 37	 18	36.9667759540984
 37	 19	36.9733769626754
 37	 20	36.9858710847838
 37	 21	37.00300592376
 38	 13	38.01685464394
 38	 14	37.99602788646
 38	 15	37.984465815426
 38	 16	37.9711635292438
 38	 17	37.9680105513993
 38	 18	37.9627321278058
 38	 19	37.9690801020149
 38	 20	37.9763186884922
 38	 21	37.99473963342
 38	 22	38.00976925222
 39	 13	39.02190030168
 39	 14	39.00225444282
 39	 15	38.98641966587
 39	 16	38.975136716328
 39	 17	38.9680076313591
 39	 18	38.9643132437036
 39	 19	38.9637069059394
 39	 20	38.9707176863375
 39	 21	38.9847900257744
 39	 22	39.001320459366
 40	 14	40.00579713868
 40	 15	39.991046641372
 40	 16	39.97546951503
 40	 17	39.970413121848
 40	 18	39.9623831282483
 40	 19	39.963998695253
 40	 20	39.9625911714524
 40	 21	39.9779644317508
 40	 22	39.99049913383
 40	 23	40.01105750526
 41	 14	41.01266782156
 41	 15	40.99484698784
 41	 16	40.98003207788
 41	 17	40.970649301572
 41	 18	40.9645007918753
 41	 19	40.9618259491466
 41	 20	40.9622783406725
 41	 21	40.9692513323148
 41	 22	40.983134620618
 41	 23	40.999742349392
 42	 14	42.016103163
 42	 15	42.00010735442
 42	 16	41.98153503976
 42	 17	41.973172130442
 42	 18	41.963048608636
 42	 19	41.9624030865085
 42	 20	41.9586183064314
 42	 21	41.9655167941062
 42	 22	41.9730314961518
 42	 23	41.991229143886
 42	 24	42.0064412652
 43	 15	43.00332798702
 43	 16	42.9865806975
 43	 17	42.974202732874
 43	 18	42.965668056484
 43	 19	42.9607157970894
 43	 20	42.9587668849487
 43	 21	42.9611510119081
 43	 22	42.968523684056
 43	 23	42.980654733516
 43	 24	42.997702615412
 44	 15	44.00987660664
 44	 16	43.98829836822
 44	 17	43.978539851442
 44	 18	43.9653653170196
 44	 19	43.961556382198
 44	 20	43.9554810882158
 44	 21	43.9594030672416
 44	 22	43.9596902403151
 44	 23	43.97439597083
 44	 24	43.985464211532
 44	 25	44.00687068288
 45	 15	45.01513697322
 45	 16	44.99484698784
 45	 17	44.97971001462
 45	 18	44.968094266376
 45	 19	44.9606996939264
 45	 20	44.9561859773375
 45	 21	44.9559102911869
 45	 22	44.958124368745
 45	 23	44.9657818521692
 45	 24	44.979162507078
 45	 25	44.99452492458
 45	 26	45.01460020112
 46	 15	46.02383268124
 46	 16	45.99957058232
 46	 17	45.98411154584
 46	 18	45.968094266376
 46	 19	45.9619761379802
 46	 20	45.9536927782874
 46	 21	45.9551702971699
 46	 22	45.9526295401117
 46	 23	45.9601995296836
 46	 24	45.9683615788818
 46	 25	45.986720258246
 46	 26	46.00085883536
 47	 16	47.00762216382
 47	 17	46.98797630496
 47	 18	46.972184469778
 47	 19	46.9616776926926
 47	 20	46.9545464606353
 47	 21	46.9524080679433
 47	 22	46.9517638340689
 47	 23	46.9549069567776
 47	 24	46.9629069008016
 47	 25	46.976102906108
 47	 26	46.99291460828
 48	 16	48.01298988482
 48	 17	47.99484698784
 48	 18	47.97509377456
 48	 19	47.9655134661192
 48	 20	47.952533243197
 48	 21	47.9522347979094
 48	 22	47.9479470623746
 48	 23	47.952254121705
 48	 24	47.954036205077
 48	 25	47.9688672182
 48	 26	47.980558114538
 48	 27	48.00171767072
 49	 16	49.0220076561
 49	 17	48.99989264558
 49	 18	48.98217916628
 49	 19	48.967450139856
 49	 20	48.955673359982
 49	 21	48.9500243704016
 49	 22	48.9478708407364
 49	 23	48.948516899636
 49	 24	48.951341609135
 49	 25	48.9596229290938
 49	 26	48.973612283564
 49	 27	48.98969397568
 50	 17	50.00772951824
 50	 18	49.98593657098
 50	 19	49.97273197732
 50	 20	49.9575187824618
 50	 21	49.9521864884204
 50	 22	49.9447921306796
 50	 23	49.9471628383365
 50	 24	49.9460495730011
 50	 25	49.9542439358797
 50	 26	49.962994931426
 50	 27	49.98153503976
 50	 28	49.99592053204
 51	 17	51.01352665692
 51	 18	50.99323667154
 51	 19	50.9763820276
 51	 20	50.961470498662
 51	 21	50.9536024932202
 51	 22	50.946615974921
 51	 23	50.9439636766205
 51	 24	50.9447717333398
 51	 25	50.9482154484246
 51	 26	50.9568252729086
 51	 27	50.970724449666
 51	 28	50.98776159612
 52	 18	51.99817497486
 52	 19	51.98260858396
 52	 20	51.9651098135
 52	 21	51.956650285204
 52	 22	51.9468982096912
 52	 23	51.9447796775669
 52	 24	51.9405119099542
 52	 25	51.9455701281614
 52	 26	51.9481166823582
 52	 27	51.963585380736
 52	 28	51.97568422387
 52	 29	51.99720878508
 53	 18	53.00622655636
 53	 19	52.9871174696
 53	 20	52.97004811682
 53	 21	52.9592053204
 53	 22	52.949736660556
 53	 23	52.944342100951
 53	 24	52.9406538324975
 53	 25	52.9412947383849
 53	 26	52.9453122628445
 53	 27	52.9542251488562
 53	 28	52.968459271404
 53	 29	52.9855071533
 54	 19	53.99398815248
 54	 20	53.97466435688
 54	 21	53.9629627251
 54	 22	53.950874617408
 54	 23	53.9464441004946
 54	 24	53.9388849537191
 54	 25	53.9403632240825
 54	 26	53.9396148564207
 54	 27	53.9484641886157
 54	 28	53.957906331918
 54	 29	53.976714826302
 54	 30	53.99291460828
 55	 19	54.99935587348
 55	 20	54.98056884998
 55	 21	54.96747161074
 55	 22	54.955115116998
 55	 23	54.94723530257
 55	 24	54.9408441718841
 55	 25	54.9380496289771
 55	 26	54.938298047105
 55	 27	54.9420031702025
 55	 28	54.951336241414
 55	 29	54.96607600328
 55	 30	54.983982720536
 56	 20	55.98582921656
 56	 21	55.9726246229
 56	 22	55.95802442178
 56	 23	55.950359316192
 56	 24	55.9406448147262
 56	 25	55.9389093231725
 56	 26	55.9349421479358
 56	 27	55.939843950753
 56	 28	55.94213596762
 56	 29	55.95856119388
 56	 30	55.97240991406
 56	 31	55.99495434226
 57	 20	56.99237783618
 57	 21	56.97702615412
 57	 22	56.96285537068
 57	 23	56.952356108404
 57	 24	56.943757019362
 57	 25	56.938287311663
 57	 26	56.9353987262841
 57	 27	56.9362962092353
 57	 28	56.939801008985
 57	 29	56.949215991619
 57	 30	56.964905840102
 57	 31	56.98293064722
 58	 21	57.98303800164
 58	 22	57.96607600328
 58	 23	57.95662881432
 58	 24	57.944250849694
 58	 25	57.93998887922
 58	 26	57.933280516223
 58	 27	57.9357576121101
 58	 28	57.9353479476434
 58	 29	57.944540706628
 58	 30	57.954599815782
 58	 31	57.974245674642
 58	 32	57.99098222872
 59	 21	58.98808365938
 59	 22	58.97198049638
 59	 23	58.95931267482
 59	 24	58.94863091003
 59	 25	58.940450503226
 59	 26	58.9348805264987
 59	 27	58.9332002151169
 59	 28	58.9343515912714
 59	 29	58.9395041740137
 59	 30	58.949264301108
 59	 31	58.963370671896
 59	 32	58.9817497486
 60	 22	59.97563054666
 60	 23	59.96446568698
 60	 24	59.94975813144
 60	 25	59.94320951182
 60	 26	59.9340768713106
 60	 27	59.9338222266264
 60	 28	59.93079064516
 60	 29	59.9373683578278
 60	 30	59.9418321546114
 60	 31	59.957058232
 60	 32	59.970187677566
 60	 33	59.99312931712
 61	 22	60.98196445744
 61	 23	60.96736425632
 61	 24	60.95405230824
 61	 25	60.94449776486
 61	 26	60.9367499963686
 61	 27	60.932479437541
 61	 28	60.9310604268174
 61	 29	60.9334621599017
 61	 30	60.9395143726836
 61	 31	60.94916768213
 61	 32	60.96382156046
 61	 33	60.98056884998
 62	 23	61.973161395
 62	 24	61.95576997896
 62	 25	61.9479331063
 62	 26	61.9367703937084
 62	 27	61.9340543268824
 62	 28	61.9283487615227
 62	 29	61.932586791961
 62	 30	61.9343345219186
 62	 31	61.9441757016
 62	 32	61.954653492992
 62	 33	61.973161395
 63	 23	62.97670409086
 63	 24	62.9618891809
 63	 25	62.94975813144
 63	 26	62.940117704524
 63	 27	62.9336152473046
 63	 28	62.9296729782934
 63	 29	62.929601050832
 63	 30	62.9332155667989
 63	 31	62.939140779302
 63	 32	62.949640041578
 63	 33	62.96371420604
 64	 24	63.96425097814
 64	 25	63.95373024498
 64	 26	63.94084771458
 64	 27	63.9358138658262
 64	 28	63.9279695857112
 64	 29	63.9297678796006
 64	 30	63.9291466195721
 64	 31	63.936838026993
 64	 32	63.941577724636
 64	 33	63.9575950041
 65	 24	64.97037018008
 65	 25	64.95609204222
 65	 26	64.94492718254
 65	 27	64.9364848309512
 65	 28	64.9300880104811
 65	 29	64.9277937391713
 65	 30	64.9292450635752
 65	 31	64.9327393425918
 65	 32	64.939441371678
 65	 33	64.94943606818
 65	 34	64.96468039582
 66	 25	65.9608156367
 66	 26	65.94600072674
 66	 27	65.93977417038
 66	 28	65.9291149500182
 66	 29	65.9288730805099
 66	 30	65.9260367767335
 66	 31	65.9315926900318
 66	 32	65.933848206396
 66	 33	65.944368939556
 66	 34	65.95523320686
 67	 25	66.96382156046
 67	 26	66.94997284028
 67	 27	66.94063300574
 67	 28	66.9315701456036
 67	 29	66.92775047534
 67	 30	66.9271308256278
 67	 31	66.9282049065999
 67	 32	66.9327381616932
 67	 33	66.939194456512
 67	 34	66.950090930142
 67	 35	66.96478775024
 68	 26	67.95254934636
 68	 27	67.94439041044
 68	 28	67.9318449729188
 68	 29	67.929639913132
 68	 30	67.9248476118232
 68	 31	67.9279835417858
 68	 32	67.9280972301166
 68	 33	67.936789717504
 68	 34	67.94192125878
 68	 35	67.95823913062
 69	 26	68.95770235852
 69	 27	68.9452492458
 69	 28	68.935179401204
 69	 29	68.929425204292
 69	 30	68.9265535809114
 69	 31	68.9255808425118
 69	 32	68.9279716254452
 69	 33	68.932280831864
 69	 34	68.93955946154
 69	 35	68.95018754912
 69	 36	68.96532452234
 70	 27	69.94975813144
 70	 28	69.9361241201
 70	 29	69.932409657168
 70	 30	69.9253253389922
 70	 31	69.926027436899
 70	 32	69.9242503991847
 70	 33	69.930928166172
 70	 34	69.933504672252
 70	 35	69.94460511928
 70	 36	69.9559846878
 71	 27	70.951690511
 71	 28	70.93998887922
 71	 29	70.932624366008
 71	 30	70.9277268573676
 71	 31	70.9247050451534
 71	 32	70.9249540000534
 71	 33	70.9271149371736
 71	 34	70.932270096422
 71	 35	70.93923739828
 71	 36	70.95050961238
 71	 37	70.96532452234
 72	 27	71.95641410548
 72	 28	71.94127713226
 72	 29	71.935522935348
 72	 30	71.9268615807424
 72	 31	71.9263693607267
 72	 32	71.9220761501165
 72	 33	71.9267531527782
 72	 34	71.9271127900852
 72	 35	71.93644618336
 72	 36	71.94192125878
 72	 37	71.95909796598
 73	 28	72.94610808116
 73	 29	72.93644618336
 73	 30	72.929779473878
 73	 31	72.9251696750832
 73	 32	72.9234594118182
 73	 33	72.9238255977448
 73	 34	72.9267671088528
 73	 35	72.931797736974
 73	 36	72.938926070462
 73	 37	72.95040225796
 73	 38	72.96596864886
 74	 28	73.9479331063
 74	 29	73.94020358806
 74	 30	73.929457410618
 74	 31	73.92694531719
 74	 32	73.9211782377476
 74	 33	73.9239290874057
 74	 34	73.9224765821031
 74	 35	73.9298911224748
 74	 36	73.933257757086
 74	 37	73.94449776486
 74	 38	73.95630675106
 75	 28	74.95297876404
 75	 29	74.94170654994
 75	 30	74.932935693826
 75	 31	74.9265008698912
 75	 32	74.9228595153192
 75	 33	74.9215963832135
 75	 34	74.922523603339
 75	 35	74.9257762275562
 75	 36	74.9310333735036
 75	 37	74.9385696537876
 75	 38	74.94997284028
 76	 28	75.95534056128
 76	 29	75.94600072674
 76	 30	75.933397317832
 76	 31	75.92893137396
 76	 32	75.9214027158398
 76	 33	75.9223939191997
 76	 34	75.9192140812793
 76	 35	75.9245416517262
 76	 36	75.9259479946282
 76	 37	75.9350709732398
 76	 38	75.94159919552
 77	 28	76.9608156367
 77	 29	76.9479331063
 77	 30	76.93709030988
 77	 31	76.929285643546
 77	 32	76.9235485159868
 77	 33	76.920647692204
 77	 34	76.9199146762242
 77	 35	76.9213800640572
 77	 36	76.9246683299418
 77	 37	76.9304064236908
 77	 38	76.937766642726
 77	 39	76.94965077702
 78	 28	77.96382156046
 78	 29	77.9527640552
 78	 30	77.938571800876
 78	 31	77.931658176228
 78	 32	77.9228529666996
 78	 33	77.9218288055328
 78	 34	77.9173095065141
 78	 35	77.9211460314216
 78	 36	77.920385962128
 78	 37	77.9281412454288
 78	 38	77.932178845165
 78	 39	77.94353157508
 79	 29	78.95523320686
 79	 30	78.94267273972
 79	 31	78.932914222942
 79	 32	78.925399413542
 79	 33	78.9209484992888
 79	 34	78.918499852323
 79	 35	78.9183376397944
 79	 36	78.9200832226636
 79	 37	78.9239962912726
 79	 38	78.9297075464166
 79	 39	78.93730501872
 79	 40	78.94911400492
 80	 29	79.9618891809
 80	 30	79.944411881324
 80	 31	79.936585744106
 80	 32	79.9254445023984
 80	 33	79.9225781393844
 80	 34	79.9165218471345
 80	 35	79.9185299115606
 80	 36	79.9163784216294
 80	 37	79.9225190944534
 80	 38	79.924524475019
 80	 39	79.93429909496
 80	 40	79.94052565132
 81	 30	80.95050961238
 81	 31	80.937755907284
 81	 32	80.92882401954
 81	 33	80.9221326185414
 81	 34	80.9179929247518
 81	 35	80.9162914645492
 81	 36	80.9165920569252
 81	 37	80.9189946488448
 81	 38	80.9232126040066
 81	 39	80.929124611916
 81	 40	80.93676824662
 81	 41	80.9490066505
 82	 30	81.95480378918
 82	 31	81.94320951182
 82	 32	81.929554029596
 82	 33	81.924508371856
 82	 34	81.9167000554717
 82	 35	81.9168046186768
 82	 36	81.9134841464662
 82	 37	81.9182077409462
 82	 38	81.9184009789022
 82	 39	81.926795021002
 82	 40	81.93107846236
 82	 41	81.9431021574
 83	 31	82.9468595621
 83	 32	82.9345138038
 83	 33	82.924980731304
 83	 34	82.919119179972
 83	 35	82.9151803463022
 83	 36	82.9141357877956
 83	 37	82.9151116394734
 83	 38	82.9175550260726
 83	 39	82.922350548014
 83	 40	82.928652252468
 83	 41	82.9366608922
 83	 42	82.94879194166
 84	 31	83.95233463752
 84	 32	83.93730501872
 84	 33	83.92903872838
 84	 34	83.91846431801
 84	 35	83.9165040263008
 84	 36	83.9115066780498
 84	 37	83.91438485005
 84	 38	83.9134251015352
 84	 39	83.920385962128
 84	 40	83.923252325142
 84	 41	83.93354761402
 84	 42	83.94009623364
 85	 32	84.94267273972
 85	 33	84.9318299433
 85	 34	84.922243193594
 85	 35	84.9156076168938
 85	 36	84.9125265450398
 85	 37	84.9117893422377
 85	 38	84.9129323447474
 85	 39	84.9164267311184
 85	 40	84.92147024177
 85	 41	84.92791150697
 85	 42	84.93655353778
 85	 43	84.94889929608
 86	 32	85.94632279
 86	 33	85.93623147452
 86	 34	85.9242711185878
 86	 35	85.918797116712
 86	 36	85.9106103759972
 86	 37	85.9111671160193
 86	 38	85.9092623265453
 86	 39	85.9148872687356
 86	 40	85.916467525798
 86	 41	85.925034408514
 86	 42	85.93064904468
 86	 43	85.94288744856
 87	 33	86.93955946154
 87	 34	86.928523427164
 87	 35	86.9207112460206
 87	 36	86.913354247618
 87	 37	86.909183528401
 87	 38	86.9088792859747
 87	 39	86.9108775811486
 87	 40	86.9148164148184
 87	 41	86.920364491244
 87	 42	86.927331793102
 87	 43	86.93655353778
 87	 44	86.94922135934
 88	 33	87.94460511928
 88	 34	87.931421996504
 88	 35	87.924068218734
 88	 36	87.9144471156136
 88	 37	87.9113188078148
 88	 38	87.9056143159993
 88	 39	87.9095034445726
 88	 40	87.9102259398192
 88	 41	87.917959752236
 88	 42	87.9219522631158
 88	 43	87.93279613308
 88	 44	87.9404182969
 89	 33	88.94922135934
 89	 34	88.93601676568
 89	 35	88.926387074206
 89	 36	88.917637688976
 89	 37	88.9122796298738
 89	 38	88.9074529751506
 89	 39	88.9058479192172
 89	 40	88.9088893772902
 89	 41	88.913493808364
 89	 42	88.9194809643674
 89	 43	88.927546501942
 89	 44	88.9361241201
 89	 45	88.94932871376
 90	 34	89.93945210712
 90	 35	89.930638309238
 90	 36	89.9195239061354
 90	 37	89.914808900009
 90	 38	89.9077374643636
 90	 39	89.9071514165848
 90	 40	89.9047037358088
 90	 41	89.9112640570606
 90	 42	89.9139361085744
 90	 43	89.923552917518
 90	 44	89.92979020932
 90	 45	89.94288744856
 91	 34	90.94535660022
 91	 35	90.933966296258
 91	 36	90.923445563098
 91	 37	90.9165340855384
 91	 38	90.9102098366562
 91	 39	90.9073037525068
 91	 40	90.9056450193634
 91	 41	90.9069902776004
 91	 42	90.9117503725832
 91	 43	90.918432111684
 91	 44	90.92635486788
 91	 45	90.93655353778
 91	 46	90.94943606818
 92	 34	91.94932871376
 92	 35	91.939258869164
 92	 36	91.9261530415704
 92	 37	91.919725732445
 92	 38	91.911030024425
 92	 39	91.908947348677
 92	 40	91.9050400772067
 92	 41	91.9071931774542
 92	 42	91.906810995719
 92	 43	91.915254420852
 92	 44	91.92012831152
 92	 45	91.93193729772
 92	 46	91.9404182969
 93	 35	92.9431021574
 93	 36	92.931260964874
 93	 37	92.9220327789308
 93	 38	92.9140219921104
 93	 39	92.9095818132992
 93	 40	92.9064756205109
 93	 41	92.9063776059255
 93	 42	92.9068120692632
 93	 43	92.9102484842474
 93	 44	92.917047239666
 93	 45	92.92571074136
 93	 46	92.93590941126
 94	 35	93.94868458724
 94	 36	93.93440644938
 94	 37	93.9264074715458
 94	 38	93.9153596281836
 94	 39	93.91159363513
 94	 40	93.9063157697795
 94	 41	93.9072834625214
 94	 42	93.9050876352147
 94	 43	93.909655887849
 94	 44	93.9113596024944
 94	 45	93.92173862782
 94	 46	93.92871666512
 94	 47	93.94278009414
 95	 36	94.9398815248
 95	 37	94.9293189234162
 95	 38	94.9193585803286
 95	 39	94.9128239167832
 95	 40	94.9080427803341
 95	 41	94.9068351504635
 95	 42	94.905841477952
 95	 43	94.9076569485486
 95	 44	94.91041273651
 95	 45	94.915898547372
 95	 46	94.92463719716
 95	 47	94.93547999358
 96	 36	95.9431021574
 96	 37	95.934288359518
 96	 38	95.9216806564332
 96	 39	95.9158974738278
 96	 40	95.9082753100078
 96	 41	95.9081003223032
 96	 42	95.9046789369378
 96	 43	95.9078705838444
 96	 44	95.9075979036176
 96	 45	95.9145179695308
 96	 46	95.918217402844
 96	 47	95.93064904468
 96	 48	95.93977417038
 97	 36	96.94857723282
 97	 37	96.937347960488
 97	 38	96.9261487473936
 97	 39	96.918131519308
 97	 40	96.9109505821542
 97	 41	96.9080971016706
 97	 42	96.9060210818966
 97	 43	96.9063644013318
 97	 44	96.9075549618496
 97	 45	96.911335984522
 97	 46	96.91647826124
 97	 47	96.92399307064
 97	 48	96.93494322148
 98	 37	97.94170654994
 98	 38	97.928469749954
 98	 39	97.9222195756216
 98	 40	97.9127466216008
 98	 41	97.9103311471508
 98	 42	97.9054078734496
 98	 43	97.9072157218824
 98	 44	97.9052876364992
 98	 45	97.9107165495186
 98	 46	97.91272085654
 98	 47	97.921760098704
 98	 48	97.927578708268
 98	 49	97.94224332204
 99	 37	98.945421012872
 99	 38	98.933311434296
 99	 39	98.9246350500716
 99	 40	98.9165115411102
 99	 41	98.9116183266466
 99	 42	98.9077115919484
 99	 43	98.9062545777601
 99	 44	98.9059392778286
 99	 45	98.9081325286292
 99	 46	98.9117675492904
 99	 47	98.917594747208
 99	 48	98.92501293763
 99	 49	98.93462115822
100	 37	99.94986548586
100	 38	99.935351168276
100	 39	99.927761210782
100	 40	99.91776651428
100	 41	99.914180876652
100	 42	99.9074776666672
100	 43	99.9076575926751
100	 44	99.904219674729
100	 45	99.9081164254662
100	 46	99.9085050484666
100	 47	99.916070314444
100	 48	99.920224930498
100	 49	99.93118581678
100	 50	99.93891533502
101	 37	100.95319347288
101	 38	100.940514915878
101	 39	100.930316245978
101	 40	100.921137443068
101	 41	100.915251200219
101	 42	100.91034617677
101	 43	100.907314487949
101	 44	100.905582217028
101	 45	100.906163648566
101	 46	100.908289266082
101	 47	100.912806740076
101	 48	100.91867902685
101	 49	100.92656957672
101	 50	100.93601676568
102	 37	101.9592053204
102	 38	101.943016273864
102	 39	101.933558349462
102	 40	101.922983939092
102	 41	101.91803490033
102	 42	101.910296793736
102	 43	101.909212514094
102	 44	101.904349466223
102	 45	101.906843202045
102	 46	101.905607552671
102	 47	101.912001581926
102	 48	101.914782061404
102	 49	101.92474455158
102	 50	101.93054169026
103	 38	102.94889929608
103	 39	102.93698295546
103	 40	102.926601783046
103	 41	102.919140650856
103	 42	102.91320395143
103	 43	102.909179234224
103	 44	102.906323714007
103	 45	102.905504492428
103	 46	102.906087426928
103	 47	102.908972040194
103	 48	102.91341866027
103	 49	102.91991360268
103	 50	102.92817989302
103	 51	102.94009623364
104	 38	103.95233463752
104	 39	103.9414918411
104	 40	103.92882401954
104	 41	103.922457902434
104	 42	103.913762194414
104	 43	103.911443338942
104	 44	103.905430417878
104	 45	103.90665533181
104	 46	103.904034810418
104	 47	103.90862850605
104	 48	103.909848052261
104	 49	103.918335492706
104	 50	103.92318791249
104	 51	103.93633882894
105	 39	104.94514189138
105	 40	104.93301084192
105	 41	104.92393939343
105	 42	104.916972091572
105	 43	104.911658047782
105	 44	104.907750346894
105	 45	104.905692362663
105	 46	104.905083663101
105	 47	104.906528653594
105	 48	104.909468017614
105	 49	104.91467363344
105	 50	104.921395093676
105	 51	104.931529350924
106	 39	105.95018754912
106	 40	105.93590941126
106	 41	105.92817989302
106	 42	105.918134739941
106	 43	105.914355864357
106	 44	105.907327370479
106	 45	105.907284428711
106	 46	105.903483008699
106	 47	105.906666067252
106	 48	105.906457799677
106	 49	105.913461602038
106	 50	105.916875472594
106	 51	105.92871666512
106	 52	105.9377344364
107	 40	106.94084771458
107	 41	106.93032698142
107	 42	106.921695686052
107	 43	106.91508265378
107	 44	106.909908170736
107	 45	106.906750877244
107	 46	106.905128751958
107	 47	106.905093324999
107	 48	106.90661453713
107	 49	106.91029249956
107	 50	106.915662367648
107	 51	106.92410042506
107	 52	106.9350505759
108	 40	107.94428305602
108	 41	107.9350505759
108	 42	107.923574388402
108	 43	107.918475053452
108	 44	107.910187292228
108	 45	107.908727272116
108	 46	107.903894176128
108	 47	107.905953233903
108	 48	107.904182959517
108	 49	107.90971493278
108	 50	107.9119693756
108	 51	107.922157310058
108	 52	107.929489616944
108	 53	107.94331686624
109	 41	108.93762708198
109	 42	108.92785782976
109	 43	108.919623745746
109	 44	108.91320395143
109	 45	108.90873586047
109	 46	108.905953233903
109	 47	108.904755158576
109	 48	108.904985970579
109	 49	108.907154529863
109	 50	108.911286601489
109	 51	108.918135813485
109	 52	108.927460618406
109	 53	108.938196060406
110	 41	109.94267273972
110	 42	109.9296828549
110	 43	109.92334894412
110	 44	109.913966167812
110	 45	109.91094950861
110	 46	109.90515236993
110	 47	109.906109971356
110	 48	109.90300528153
110	 49	109.907168485938
110	 50	109.907852333593
110	 51	109.916757382732
110	 52	109.922404225224
110	 53	109.93526528474
110	 54	109.94449776486
111	 42	110.9345138038
111	 43	110.92506661484
111	 44	110.91755180544
111	 45	110.911658047782
111	 46	110.907642992474
111	 47	110.905295151309
111	 48	110.904181885973
111	 49	110.905110501706
111	 50	110.907735317275
111	 51	110.913214686872
111	 52	110.921115972184
111	 53	110.93032698142
111	 54	110.94159919552
112	 42	111.93687560104
112	 43	111.92925343722
112	 44	111.91851799522
112	 45	111.9146532361
112	 46	111.907313414405
112	 47	111.907004233675
112	 48	111.90275729282
112	 49	111.905533478121
112	 50	111.904820644772
112	 51	111.912394499103
112	 52	111.917057975108
112	 53	111.92796518418
112	 54	111.935662496094
112	 55	111.95029490354
113	 42	112.9420286132
113	 43	112.9312931712
113	 44	112.92249010876
113	 45	112.91540471704
113	 46	112.910155085902
113	 47	112.906566227641
113	 48	112.90440088899
113	 49	112.904061649023
113	 50	112.905173840814
113	 51	112.909377839901
113	 52	112.915930753698
113	 53	112.923649536496
113	 54	112.93338658239
113	 55	112.944540706628
114	 43	113.93590941126
114	 44	113.92399307064
114	 45	113.91884005848
114	 46	113.910365500565
114	 47	113.908813155652
114	 48	113.903358477572
114	 49	113.90491726375
114	 50	113.902781984336
114	 51	113.909092277144
114	 52	113.912055259136
114	 53	113.92184598224
114	 54	113.928147686694
114	 55	113.94138448668
114	 56	113.95093903006
115	 43	114.9382712085
115	 44	114.92828724744
115	 45	114.92012831152
115	 46	114.91368704632
115	 47	114.908759478442
115	 48	114.905430417878
115	 49	114.903878072965
115	 50	114.903345595041
115	 51	114.906598433967
115	 52	114.911582899688
115	 53	114.9178738687
115	 54	114.926537370394
115	 55	114.93590941126
115	 56	114.94771839746
116	 44	115.93011227258
116	 45	115.92367100738
116	 46	115.914159405768
116	 47	115.911357455406
116	 48	115.904755158576
116	 49	115.90525972435
116	 50	115.901743867095
116	 51	115.906797039644
116	 52	115.908415944298
116	 53	115.916735911848
116	 54	115.92173862782
116	 55	115.9329034875
116	 56	115.94170654994
117	 44	116.93483586706
117	 45	116.9253886781
117	 46	116.9178738687
117	 47	116.911679518666
117	 48	116.907217868971
117	 49	116.904515758219
117	 50	116.902953751408
117	 51	116.904839968568
117	 52	116.908633873771
117	 53	116.913644104552
117	 54	116.920568464642
117	 55	116.928641517026
117	 56	116.9388079806
117	 57	116.94997284028
118	 44	117.93698295546
118	 45	117.92946814606
118	 46	117.918979619226
118	 47	117.914578088006
118	 48	117.906914055962
118	 49	117.906354739434
118	 50	117.901606453437
118	 51	117.905532404577
118	 52	117.905825482143
118	 53	117.913375718502
118	 54	117.91658561566
118	 55	117.926554547101
118	 56	117.9334402596
118	 57	117.94653749884
119	 45	118.93140052562
119	 46	118.9227048176
119	 47	118.915662367648
119	 48	118.909918906178
119	 49	118.905845879483
119	 50	118.903309094539
119	 51	118.903946779793
119	 52	118.906408416644
119	 53	118.910176556786
119	 54	118.915555013228
119	 55	118.922370945354
119	 56	118.93107846236
119	 57	118.940955069
119	 58	118.9527640552
120	 45	119.93580205684
120	 46	119.92399307064
120	 47	119.91878638127
120	 48	119.909851272893
120	 49	119.907965055734
120	 50	119.902196902747
120	 51	119.905074001203
120	 52	119.904019780799
120	 53	119.910047731482
120	 54	119.912151878114
120	 55	119.92067796615
120	 56	119.92603280462
120	 57	119.93805649966
120	 58	119.94664485326
121	 45	120.93805649966
121	 46	120.92817989302
121	 47	120.919849190028
121	 48	120.912978507148
121	 49	120.907846965872
121	 50	120.904236851436
121	 51	120.903818061844
121	 52	120.904930146281
121	 53	120.90736601807
121	 54	120.911386441099
121	 55	120.917183579779
121	 56	120.92452984274
121	 57	120.93301084192
121	 58	120.9436389295
121	 59	120.95534056128
122	 46	121.92979020932
122	 47	121.923316737794
122	 48	121.913504543806
122	 49	121.910273175764
122	 50	121.903440066931
122	 51	121.90517545113
122	 52	121.903047042399
122	 53	121.907592535897
122	 54	121.908544769602
122	 55	121.916121844566
122	 56	121.92023566594
122	 57	121.9307563991
122	 58	121.93805649966
122	 59	121.951690511
123	 46	122.93429909496
123	 47	122.924851906
123	 48	122.917004297898
123	 49	122.910438501571
123	 50	122.9057224219
123	 51	122.904215702616
123	 52	122.904272922521
123	 53	122.905597890773
123	 54	122.908470695052
123	 55	122.912990316134
123	 56	122.91884005848
123	 57	122.92624751346
123	 58	122.93547999358
123	 59	122.94600072674
124	 47	123.92850195628
124	 48	123.917648424418
124	 49	123.913171745104
124	 50	123.905274646614
124	 51	123.905937560158
124	 52	123.902819451029
124	 53	123.906211421283
124	 54	123.905895799289
124	 55	123.912245276459
124	 56	123.915088021501
124	 57	123.92452984274
124	 58	123.93054169026
124	 59	123.94299480298
125	 47	124.93054169026
125	 48	124.921244797488
125	 49	124.913601162784
125	 50	124.907784915017
125	 51	124.905247915364
125	 52	124.904424721671
125	 53	124.904624186184
125	 54	124.906398217974
125	 55	124.909724594678
125	 56	124.914621029774
125	 57	124.92066508362
125	 58	124.92850195628
125	 59	124.93784179082
126	 47	125.9345138038
126	 48	125.922350548014
126	 49	125.916467525798
126	 50	125.907653727916
126	 51	125.90724578112
126	 52	125.903305551843
126	 53	125.905619361657
126	 54	125.904268843053
126	 55	125.909447620274
126	 56	125.911243659721
126	 57	125.91937683058
126	 58	125.92410042506
126	 59	125.93526528474
126	 60	125.9431021574
127	 47	126.93687560104
127	 48	126.926430015974
127	 49	126.917347832042
127	 50	126.910350470946
127	 51	126.906914055962
127	 52	126.905216782582
127	 53	126.904468522275
127	 54	126.905179208535
127	 55	126.907417548192
127	 56	126.911121275682
127	 57	126.91615619798
127	 58	126.9227048176
127	 59	126.93086375352
127	 60	126.94052565132
128	 48	127.92775047534
128	 49	127.920171253288
128	 50	127.910530826372
128	 51	127.909167425238
128	 52	127.904461436883
128	 53	127.905805084804
128	 54	127.903530459353
128	 55	127.907748199806
128	 56	127.908308589878
128	 57	127.91540471704
128	 58	127.91884005848
128	 59	127.92882401954
128	 60	127.93537263916
128	 61	127.94825516956
129	 48	128.93225936098
129	 49	128.921652744284
129	 50	128.913440131154
129	 51	128.909150248531
129	 52	128.906595213335
129	 53	128.904987044123
129	 54	128.904779420675
129	 55	128.906063808956
129	 56	128.908673594906
129	 57	128.91266717933
129	 58	128.91808857754
129	 59	128.924851906
129	 60	128.93322555076
129	 61	128.94320951182
130	 48	129.9339770317
130	 49	129.924851906
130	 50	129.91384807795
130	 51	129.911546399185
130	 52	129.906222800852
130	 53	129.906673582061
130	 54	129.903507914924
130	 55	129.906705788387
130	 56	129.906310724122
130	 57	129.912323645186
130	 58	129.9146532361
130	 59	129.92334894412
130	 60	129.92882401954
130	 61	129.9404182969
130	 62	129.94857723282
131	 49	130.926762814676
131	 50	130.916918414362
131	 51	130.911947904716
131	 52	130.908521903111
131	 53	130.90612414214
131	 54	130.90508194543
131	 55	130.905460477115
131	 56	130.906931232669
131	 57	130.910112144134
131	 58	130.91443852726
131	 59	130.9200209571
131	 60	130.92710634882
131	 61	130.93580205684
131	 62	130.94589337232
132	 49	131.932914222942
132	 50	131.917745043396
132	 51	131.914412762199
132	 52	131.908523298718
132	 53	131.907994041427
132	 54	131.904154510596
132	 55	131.906429887528
132	 56	131.905055750952
132	 57	131.910112144134
132	 58	131.91148628071
132	 59	131.919119179972
132	 60	131.92313423528
132	 61	131.93376232286
132	 62	131.94084771458
132	 63	131.95415966266
133	 49	132.93837856292
133	 50	132.923810568126
133	 51	132.915232949968
133	 52	132.910938773168
133	 53	132.907804024104
133	 54	132.905905997958
133	 55	132.905446521041
133	 56	132.906002616936
133	 57	132.908394473414
133	 58	132.911550693362
133	 59	132.916199139748
133	 60	132.9221680455
133	 61	132.9296828549
133	 62	132.93870062618
133	 63	132.94889929608
134	 49	133.9447124737
134	 50	133.928459014512
134	 51	133.920546993758
134	 52	133.91153995792
134	 53	133.909877037954
134	 54	133.905394561502
134	 55	133.906713303197
134	 56	133.904502875689
134	 57	133.908491092392
134	 58	133.909027864492
134	 59	133.91561942588
134	 60	133.91862534964
134	 61	133.92850195628
134	 62	133.9339770317
134	 63	133.94632279
135	 50	134.93472851264
135	 51	134.925163233818
135	 52	134.916446054914
135	 53	134.91004987857
135	 54	134.907207133529
135	 55	134.905971484155
135	 56	134.905682700765
135	 57	134.906970953805
135	 58	134.909145954354
135	 59	134.913139538778
135	 60	134.918238873728
135	 61	134.92463719716
135	 62	134.9323667154
135	 63	134.94170654994
136	 50	135.9393447527
136	 51	135.93064904468
136	 52	135.920106840636
136	 53	135.9146532361
136	 54	135.907220016059
136	 55	135.907305899595
136	 56	135.904570508974
136	 57	135.907653727916
136	 58	135.9071384267
136	 59	135.912645708446
136	 60	135.915018241128
136	 61	135.923445563098
136	 62	135.92828724744
136	 63	135.93945210712
136	 64	135.94707427094
137	 50	136.9457860179
137	 51	136.93526528474
137	 52	136.925324265448
137	 53	136.9178738687
137	 54	136.911562502348
137	 55	136.907083675946
137	 56	136.905821187967
137	 57	136.906462093854
137	 58	136.90778255322
137	 59	136.91068112256
137	 60	136.914642500658
137	 61	136.920708025388
137	 62	136.927041936168
137	 63	136.93515793032
137	 64	136.94460511928
138	 51	137.940955069
138	 52	137.929221230894
138	 53	137.92238275434
138	 54	137.913987638696
138	 55	137.911010700629
138	 56	137.905241474099
138	 57	137.907107293918
138	 58	137.905985440229
138	 59	137.910748755845
138	 60	137.911926433832
138	 61	137.919484185
138	 62	137.92356365296
138	 63	137.9334402596
138	 64	137.93998887922
138	 65	137.95287140962
139	 51	138.94567866348
139	 52	138.93472851264
139	 53	138.926097217272
139	 54	138.91878638127
139	 55	138.913357468251
139	 56	138.90883570008
139	 57	138.906348298169
139	 58	138.906646743456
139	 59	138.908932319058
139	 60	138.911926433832
139	 61	138.916757382732
139	 62	138.922302238525
139	 63	138.929833151088
139	 64	138.93805649966
139	 65	138.94804046072
140	 52	139.93870062618
140	 53	139.931207287664
140	 54	139.9216312734
140	 55	139.917276978125
140	 56	139.910599533201
140	 57	139.909472311791
140	 58	139.90543363851
140	 59	139.90907080626
140	 60	139.909310206617
140	 61	139.915801928394
140	 62	139.918991428212
140	 63	139.928083274042
140	 64	139.9339770317
140	 65	139.94557130906
140	 66	139.9538375994
141	 52	140.94439041044
141	 53	140.93483586706
141	 54	140.926644724814
141	 55	140.920043501528
141	 56	140.914406320934
141	 57	140.910957023419
141	 58	140.908271015831
141	 59	140.907647286651
141	 60	140.909604357727
141	 61	140.913611898226
141	 62	140.918468612187
141	 63	140.924884112326
141	 64	140.93225936098
141	 65	140.94116977784
141	 66	140.9511537389
142	 52	141.9484698784
142	 53	141.94020358806
142	 54	141.929704325784
142	 55	141.924292589472
142	 56	141.916448202002
142	 57	141.914074595776
142	 58	141.909239352699
142	 59	141.910040216673
142	 60	141.907718140568
142	 61	141.912946300822
142	 62	141.915193228833
142	 63	141.92340262133
142	 64	141.92817989302
142	 65	141.9388079806
142	 66	141.94621543558
142	 67	141.95984944692
143	 53	142.94406834718
143	 54	142.93488954427
143	 55	142.927330719558
143	 56	142.920616774131
143	 57	142.916058505458
143	 58	142.912381616573
143	 59	142.910812094952
143	 60	142.90980940467
143	 61	142.910928037726
143	 62	142.914623176862
143	 63	142.920286122517
143	 64	142.926741343792
143	 65	142.93472851264
143	 66	142.94385363834
143	 67	142.95469643476
144	 53	143.94965077702
144	 54	143.9382712085
144	 55	143.932023181256
144	 56	143.922940997324
144	 57	143.91959153942
144	 58	143.913643031008
144	 59	143.913300570408
144	 60	143.910083158441
144	 61	143.912585589971
144	 62	143.911995140661
144	 63	143.918774572284
144	 64	143.922790701136
144	 65	143.93258142424
144	 66	143.93902268944
144	 67	143.951690511
144	 68	143.96060092786
145	 54	144.9436389295
145	 55	144.935383374602
145	 56	144.926923846306
145	 57	144.921642008842
145	 58	144.91722974218
145	 59	144.914507234089
145	 60	144.912568413264
145	 61	144.912743400968
145	 62	144.91340577774
145	 63	144.916261405312
145	 64	144.92168495061
145	 65	144.92887769675
145	 66	144.93698295546
145	 67	144.9468595621
145	 68	144.95748764968
146	 54	145.94728897978
146	 55	145.940160646292
146	 56	145.930101537138
146	 57	145.925700005918
146	 58	145.918689762292
146	 59	145.917584011766
146	 60	145.913111626629
146	 61	145.914691883691
146	 62	145.913036478535
146	 63	145.917199682942
146	 64	145.918305433468
146	 65	145.927181496914
146	 66	145.932720984986
146	 67	145.94406834718
146	 68	145.95211992868
146	 69	145.96650542096
147	 54	146.95297876404
147	 55	146.943864373782
147	 56	146.933987767142
147	 57	146.927814887992
147	 58	146.922511579644
147	 59	146.918979619226
147	 60	146.916096079505
147	 61	146.915134183902
147	 62	146.914893710001
147	 63	146.916741279569
147	 64	146.919089120734
147	 65	146.924037085952
147	 66	146.930874488962
147	 67	146.9398815248
147	 68	146.94932871376
147	 69	146.96103034554
148	 55	147.94889929608
148	 56	147.93768075919
148	 57	147.932194948328
148	 58	147.924390281994
148	 59	147.922178780942
148	 60	147.916888355124
148	 61	147.917468068992
148	 62	147.914817488363
148	 63	147.918154063736
148	 64	147.918110048424
148	 65	147.924293663016
148	 66	147.927181496914
148	 67	147.93730501872
148	 68	147.94439041044
148	 69	147.9575950041
148	 70	147.9667201298
149	 55	148.9527640552
149	 56	148.94245803088
149	 57	148.93440644938
149	 58	148.92828724744
149	 59	148.92379124433
149	 60	148.920144414683
149	 61	148.918329051441
149	 62	148.917179285603
149	 63	148.917925398822
149	 64	148.9193360359
149	 65	148.9232415897
149	 66	148.92733394019
149	 67	148.933790235009
149	 68	148.94213596762
149	 69	148.95265670078
149	 70	148.9634994972
150	 55	149.95791706736
150	 56	149.94557130906
150	 57	149.93859327176
150	 58	149.930230362442
150	 59	149.9269989944
150	 60	149.920886233725
150	 61	149.920979632071
150	 62	149.917271610404
150	 63	149.919697820296
150	 64	149.918655408878
150	 65	149.923653830673
150	 66	149.925579768968
150	 67	149.933354376064
150	 68	149.937766642726
150	 69	149.94965077702
150	 70	149.95802442178
150	 71	149.9726246229
151	 55	150.96199653532
151	 56	150.95072432122
151	 57	150.94159919552
151	 58	150.93408438612
151	 59	150.928222834788
151	 60	150.923824524201
151	 61	150.921202929264
151	 62	150.919928632299
151	 63	150.919845969395
151	 64	150.920344093904
151	 65	150.923097734777
151	 66	150.926179880175
151	 67	150.931680720656
151	 68	150.93741237314
151	 69	150.945431748314
151	 70	150.95523320686
151	 71	150.96714954748
152	 56	151.95415966266
152	 57	151.94610808116
152	 58	151.93633882894
152	 59	151.93161523446
152	 60	151.924680138928
152	 61	151.923488504866
152	 62	151.919727879533
152	 63	151.921740774908
152	 64	151.919787998009
152	 65	151.924068218734
152	 66	151.924713418798
152	 67	151.931744059764
152	 68	151.935082782226
152	 69	151.94428305602
152	 70	151.95018754912
152	 71	151.96360685162
153	 56	152.95963473808
153	 57	152.94943606818
153	 58	152.94063300574
153	 59	152.93365496844
153	 60	152.92769679813
153	 61	152.92411330759
153	 62	152.92209397095
153	 63	152.921226547237
153	 64	152.921746142629
153	 65	152.923430533479
153	 66	152.925761197937
153	 67	152.930194935483
153	 68	152.935093517668
153	 69	152.942027539656
153	 70	152.94922135934
153	 71	152.9586685483
154	 57	153.9543743715
154	 58	153.94331686624
154	 59	153.93741237314
154	 60	153.929478881502
154	 61	153.926548105836
154	 62	153.922205619547
154	 63	153.922975350738
154	 64	153.920862615753
154	 65	153.92469087437
154	 66	153.92442248832
154	 67	153.930596441014
154	 68	153.932776809284
154	 69	153.941427428448
154	 70	153.946236906464
154	 71	153.957058232
154	 72	153.96425097814
155	 57	154.9581317762
155	 58	154.94804046072
155	 59	154.93998887922
155	 60	154.932624366008
155	 61	154.928094009484
155	 62	154.924636123616
155	 63	154.922889467202
155	 64	154.922618934064
155	 65	154.923500313852
155	 66	154.925749388951
155	 67	154.92907952306
155	 68	154.933204079876
155	 69	154.939191235879
155	 70	154.9457860179
155	 71	154.954234810754
155	 72	154.96274801626
156	 58	155.95126109332
156	 59	155.94406834718
156	 60	155.93515793032
156	 61	155.931056991476
156	 62	155.925526091758
156	 63	155.924750992845
156	 64	155.922119736011
156	 65	155.924743478036
156	 66	155.924278633397
156	 67	155.929715061226
156	 68	155.931014049708
156	 69	155.939011953998
156	 70	155.942844506792
156	 71	155.95287140962
156	 72	155.9592053204
156	 73	155.97165843312
157	 58	156.95630675106
157	 59	156.94718162536
157	 60	156.93923739828
157	 61	156.93322555076
157	 62	156.928351660092
157	 63	156.925419810882
157	 64	156.923956570137
157	 65	156.924020982789
157	 66	156.925461679106
157	 67	156.928190628462
157	 68	156.931948033162
157	 69	156.936757511178
157	 70	156.942662004278
157	 71	156.950101665584
157	 72	156.9581317762
157	 73	156.96811573726
158	 59	157.95179786542
158	 60	157.94192125878
158	 61	157.9366608922
158	 62	157.929983447276
158	 63	157.927847094318
158	 64	157.92410042506
158	 65	157.925410148984
158	 66	157.924404238069
158	 67	157.928942109402
158	 68	157.929908299182
158	 69	157.936993690902
158	 70	157.939857906828
158	 71	157.94916768213
158	 72	157.95464275755
158	 73	157.96639806654
158	 74	157.97391287594
159	 59	158.95523320686
159	 60	158.94643014442
159	 61	158.93913004386
159	 62	158.93322555076
159	 63	158.929084890781
159	 64	158.926384927118
159	 65	158.925343589244
159	 66	158.925735432877
159	 67	158.927708607116
159	 68	158.930681251006
159	 69	158.934803660734
159	 70	158.94014991085
159	 71	158.946612646934
159	 72	158.95405230824
159	 73	158.96290904789
159	 74	158.97230255964
160	 60	159.94943606818
160	 61	159.94299480298
160	 62	159.93515793032
160	 63	159.931969504046
160	 64	159.927050524522
160	 65	159.927164320207
160	 66	159.925193293056
160	 67	159.928725253474
160	 68	159.929081670148
160	 69	159.9350505759
160	 70	159.937562669328
160	 71	159.946022197624
160	 72	159.950713585778
160	 73	159.9613524088
160	 74	159.9683304461
160	 75	159.98153503976
161	 60	160.9543743715
161	 61	160.94589337232
161	 62	160.9388079806
161	 63	160.93365496844
161	 64	160.929665678193
161	 65	160.927565825738
161	 66	160.926929214027
161	 67	160.927851388495
161	 68	160.930001697527
161	 69	160.933397317832
161	 70	160.937852526262
161	 71	160.943542310522
161	 72	160.950327109866
161	 73	160.958367955924
161	 74	160.96704219306
161	 75	160.97767028064
162	 61	161.95029490354
162	 62	161.94116977784
162	 63	161.93709030988
162	 64	161.930980769838
162	 65	161.929489616944
162	 66	161.926795021002
162	 67	161.92909240559
162	 68	161.928774636507
162	 69	161.933966296258
162	 70	161.93574837963
162	 71	161.943220247262
162	 72	161.947203096244
162	 73	161.957144115536
162	 74	161.96333846557
162	 75	161.97573790108
162	 76	161.98378948258
163	 61	162.95351553614
163	 62	162.94535660022
163	 63	162.93923739828
163	 64	162.9339770317
163	 65	162.930643676959
163	 66	162.928727400562
163	 67	162.928730621195
163	 68	162.930029609677
163	 69	162.93264798398
163	 70	162.936263680846
163	 71	162.941201984166
163	 72	162.94707427094
163	 73	162.95432069429
163	 74	162.96253330742
163	 75	162.971969760938
163	 76	162.98207181186
164	 62	163.94825516956
164	 63	163.94299480298
164	 64	163.93590941126
164	 65	163.933343640622
164	 66	163.929170774317
164	 67	163.930230362442
164	 68	163.929196539377
164	 69	163.933450995042
164	 70	163.934524539242
164	 71	163.941212719608
164	 72	163.944422616766
164	 73	163.95362289056
164	 74	163.958979876118
164	 75	163.97037018008
164	 76	163.97788498948
165	 62	164.95297876404
165	 63	164.94567866348
165	 64	164.9393447527
165	 65	164.934878808828
165	 66	164.931700044452
165	 67	164.930319466611
165	 68	164.93072311923
165	 69	164.932432201596
165	 70	164.935397330677
165	 71	164.939602403308
165	 72	164.94449776486
165	 73	164.950820940198
165	 74	164.958335749598
165	 75	164.967052928502
165	 76	164.97648938202
165	 77	164.98754688728
166	 63	165.94997284028
166	 64	165.94159919552
166	 65	165.93805649966
166	 66	165.932803647889
166	 67	165.932280831864
166	 68	165.930290480917
166	 69	165.933552981741
166	 70	165.933879339178
166	 71	165.939763434938
166	 72	165.94224332204
166	 73	165.95050961238
166	 74	165.955019571564
166	 75	165.965796881788
166	 76	165.972528003922
166	 77	165.9855071533
167	 63	166.95308611846
167	 64	166.94557130906
167	 65	166.94009623364
167	 66	166.935651760652
167	 67	166.933125711149
167	 68	166.932045725684
167	 69	166.932848736746
167	 70	166.934946442113
167	 71	166.938303414826
167	 72	166.942597591626
167	 73	166.9479331063
167	 74	166.95469643476
167	 75	166.962565513746
167	 76	166.9715510787
167	 77	166.981545775202
168	 64	167.94836252398
168	 65	167.9436389295
168	 66	167.9371976643
168	 67	167.935501464464
168	 68	167.932367788944
168	 69	167.934170269656
168	 70	167.933894368797
168	 71	167.93870062618
168	 72	167.94063300574
168	 73	167.94782575188
168	 74	167.951862278072
168	 75	167.96156711764
168	 76	167.967836615768
168	 77	167.97992472346
168	 78	167.98808365938
169	 64	168.95287140962
169	 65	168.94621543558
169	 66	168.94031094248
169	 67	168.936868086231
169	 68	168.93458787835
169	 69	168.934211064336
169	 70	168.935186916013
169	 71	168.937648552864
169	 72	168.941159042398
169	 73	168.945914843204
169	 74	168.95179786542
169	 75	168.95882957993
169	 76	168.967074399386
169	 77	168.976392763042
169	 78	168.98647334308
170	 65	169.95029490354
170	 66	169.94267273972
170	 67	169.93961313875
170	 68	169.935460669784
170	 69	169.935797762663
170	 70	169.934758571878
170	 71	169.938471961265
170	 72	169.939645345076
170	 73	169.946086610276
170	 74	169.94932871376
170	 75	169.9581317762
170	 76	169.963569277573
170	 77	169.975029361908
170	 78	169.982329462468
171	 65	170.9533008273
171	 66	170.94643014442
171	 67	170.9414918411
171	 68	170.938025366878
171	 69	170.93642578602
171	 70	170.936322725777
171	 71	170.937909424105
171	 72	170.940493444994
171	 73	170.944454823092
171	 74	170.94943606818
171	 75	170.95555527012
171	 76	170.96307007952
171	 77	170.971776522982
171	 78	170.9812129765
171	 79	170.991776651428
172	 66	171.94911400492
172	 67	171.94481982812
172	 68	171.939352267509
172	 69	171.938395739627
172	 70	171.936377476531
172	 71	171.939081734371
172	 72	171.939462842562
172	 73	171.944744680026
172	 74	171.9473963342
172	 75	171.95523320686
172	 76	171.960074891202
172	 77	171.97069224334
172	 78	171.977380423706
172	 79	171.99012339336
173	 66	172.95340818172
173	 67	172.94728897978
173	 68	172.94240435367
173	 69	172.93960025622
173	 70	172.938206795848
173	 71	172.938927144006
173	 72	172.940654476624
173	 73	172.943542310522
173	 74	172.94782575188
173	 75	172.95308611846
173	 76	172.9597420925
173	 77	172.967707790464
173	 78	172.976500117462
173	 79	172.986398194986
174	 67	173.9511537389
174	 68	173.94439041044
174	 69	173.942168173946
174	 70	173.938858437177
174	 71	173.940333486908
174	 72	173.940040409342
174	 73	173.944164966158
174	 74	173.94610808116
174	 75	173.95308611846
174	 76	173.95716558642
174	 77	173.96682748422
174	 78	173.972811419591
174	 79	173.98491670399
175	 67	174.95405230824
175	 68	174.9479331063
175	 69	174.943832167456
175	 70	174.941272838083
175	 71	174.940768272309
175	 72	174.941502576542
175	 73	174.943649664942
175	 74	174.946773678564
175	 75	174.95136844774
175	 76	174.957058232
175	 77	174.96425097814
175	 78	174.97230255964
175	 79	174.981545775202
175	 80	174.9914116464
176	 68	175.95029490354
176	 69	175.946988387404
176	 70	175.942568605933
176	 71	175.942682401618
176	 72	175.941401663387
176	 73	175.944744680026
176	 74	175.945592779944
176	 75	175.951572421138
176	 76	175.954954085368
176	 77	175.9634994972
176	 78	175.968996043504
176	 79	175.98024678672
176	 80	175.987418061976
177	 68	176.9543743715
177	 69	176.9490066505
177	 70	176.945256760609
177	 71	176.943754872274
177	 72	176.943220247262
177	 73	176.944471999799
177	 74	176.94664485326
177	 75	176.950273432656
177	 76	176.95501849802
177	 77	176.96113769996
177	 78	176.96843780052
177	 79	176.977219392076
177	 80	176.986333782334
177	 81	176.996875986378
178	 69	177.95265670078
178	 70	177.946643779716
178	 71	177.945951343707
178	 72	177.943697974431
178	 73	177.945753811574
178	 74	177.945850430552
178	 75	177.950853146524
178	 76	177.953343769068
178	 77	177.96103034554
178	 78	177.96575394002
178	 79	177.97595260992
178	 80	177.982476538023
178	 81	177.99522272831
179	 69	178.95534056128
179	 70	178.95018754912
179	 71	178.947324406739
179	 72	178.945815003593
179	 73	178.945934167
179	 74	178.947072123852
179	 75	178.949983575722
179	 76	178.953955689262
179	 77	178.95909796598
179	 78	178.96543187676
179	 79	178.97337610384
179	 80	178.9817497486
179	 81	178.99146532361
180	 70	179.95233463752
180	 71	179.949876221302
180	 72	179.946548234282
180	 73	179.947466114573
180	 74	179.946706045279
180	 75	179.950788733872
180	 76	179.952345372962
180	 77	179.959248262168
180	 78	179.963209640266
180	 79	179.97240991406
180	 80	179.978325142602
180	 81	179.99023074778
181	 70	180.95619939664
181	 71	180.95201257426
181	 72	180.949098975301
181	 73	180.947996445408
181	 74	180.948198271717
181	 75	180.950064091537
181	 76	180.953279356416
181	 77	180.957637945868
181	 78	180.96317743394
181	 79	180.9699407624
181	 80	180.97777763506
181	 81	180.98690276076
181	 82	180.996714954748
182	 71	181.95523320686
182	 72	181.950552554148
182	 73	181.950152122161
182	 74	181.948205786527
182	 75	181.95120741611
182	 76	181.95218648842
182	 77	181.9581317762
182	 78	181.961266525264
182	 79	181.96961869914
182	 80	181.9747717113
182	 81	181.98561450772
182	 82	181.992676281468
183	 71	182.9575950041
183	 72	182.953526271582
183	 73	182.951372741917
183	 74	182.950224049623
183	 75	182.950820940198
183	 76	182.953107589344
183	 77	182.956811316834
183	 78	182.96172814927
183	 79	182.96757896516
183	 80	182.97455700246
183	 81	182.98271593838
183	 82	182.9919484185
184	 71	183.96113769996
184	 72	183.9554479157
184	 73	183.954009366472
184	 74	183.950932588795
184	 75	183.952524654843
184	 76	183.952490301429
184	 77	183.95738029526
184	 78	183.959892388688
184	 79	183.96747161074
184	 80	183.971894612844
184	 81	183.9817497486
184	 82	183.988201749242
185	 72	184.95877590272
185	 73	184.955559564297
185	 74	184.95342106425
185	 75	184.952956219612
185	 76	184.954042646342
185	 77	184.956585872552
185	 78	184.960751224048
185	 79	184.96580761723
185	 80	184.97198049638
185	 81	184.9790658881
185	 82	184.98754688728
185	 83	184.997702615412
186	 72	185.96092299112
186	 73	185.958550458438
186	 74	185.954362562514
186	 75	185.954986291694
186	 76	185.953838672944
186	 77	185.957951420774
186	 78	185.959430764682
186	 79	185.966000855186
186	 80	185.96945766751
186	 81	185.978529116
186	 82	185.98432625468
186	 83	185.99645730414
187	 73	186.96038621902
187	 74	186.957158071611
187	 75	186.955750655164
187	 76	186.955747434532
187	 77	186.957360971464
187	 78	186.960557986092
187	 79	186.964562305958
187	 80	186.96977973077
187	 81	186.97616731876
187	 82	186.98400419142
187	 83	186.99345138038
188	 73	187.96371420604
188	 74	187.95848711933
188	 75	187.958112452404
188	 76	187.955835465156
188	 77	187.958852124358
188	 78	187.959395337723
188	 79	187.965088342616
188	 80	187.967557494276
188	 81	187.975920403594
188	 82	187.981062680312
188	 83	187.99216312734
189	 74	188.961910651784
189	 75	188.959227864828
189	 76	188.95814465873
189	 77	188.958716857789
189	 78	188.960831739863
189	 79	188.963885973112
189	 80	188.96811573726
189	 81	188.9736981671
189	 82	188.98089091324
189	 83	188.98947926684
190	 74	189.96317743394
190	 75	189.961814032806
190	 76	189.958445251106
190	 77	189.960590192418
190	 78	189.959929962735
190	 79	189.964698646071
190	 80	189.966279976678
190	 81	189.97380552152
190	 82	189.978174846414
190	 83	189.98851307706
190	 84	189.99506169668
191	 75	190.96312375673
191	 76	190.960928358841
191	 77	190.960591265962
191	 78	190.961684133958
191	 79	190.963649793388
191	 80	190.967063663944
191	 81	190.971883877402
191	 82	190.978196317298
191	 83	190.9860439254
191	 84	190.994632279
192	 75	191.965957913418
192	 76	191.961479087016
192	 77	191.962602014249
192	 78	191.961034639717
192	 79	191.964810294668
192	 80	191.96553923118
192	 81	191.97214152801
192	 82	191.975759371964
192	 83	191.985367592554
192	 84	191.99151900082
193	 76	192.964147917897
193	 77	192.962924077509
193	 78	192.962984195984
193	 79	192.964131814734
193	 80	192.966643908162
193	 81	192.970552682594
193	 82	192.976081435224
193	 83	192.98303800164
193	 84	192.99108958314
193	 85	193.00021470884
194	 76	193.965179593873
194	 77	193.965075460086
194	 78	193.962663206268
194	 79	193.965338478415
194	 80	193.965381420183
194	 81	193.971057248368
194	 82	193.97396655315
194	 83	193.98271593838
194	 84	193.988287632778
194	 85	193.9989264558
195	 76	194.96811573726
195	 77	194.965977237214
195	 78	194.96477486771
195	 79	194.965017488699
195	 80	194.966634246264
195	 81	194.969650905466
195	 82	194.97444964804
195	 83	194.980751352494
195	 84	194.988040717612
195	 85	194.99656465856
196	 76	195.96961869914
196	 77	195.96838412331
196	 78	195.964934825795
196	 79	195.966551583361
196	 80	195.965815132039
196	 81	195.970509740826
196	 82	195.972710506436
196	 83	195.980611791748
196	 84	195.9855071533
196	 85	195.9957058232
196	 86	196.00230812003
197	 77	196.969636949391
197	 78	196.96732346164
197	 79	196.966551583361
197	 80	196.967195709881
197	 81	196.969532815604
197	 82	196.97337610384
197	 83	196.978937062796
197	 84	196.98556083051
197	 85	196.99323667154
197	 86	197.0016103163
198	 77	197.972281088756
198	 78	197.967876336903
198	 79	197.968225238768
198	 80	197.966752336126
198	 81	197.970466799058
198	 82	197.97198049638
198	 83	197.979022946332
198	 84	197.983338594016
198	 85	197.99269989944
198	 86	197.998776159612
199	 77	198.973784050636
199	 78	198.970576300566
199	 79	198.968748054794
199	 80	198.968262812815
199	 81	198.969811937096
199	 82	198.972914479834
199	 83	198.977573661662
199	 84	198.98357477374
199	 85	198.990627959134
199	 86	198.998303800164
200	 78	199.971424400484
200	 79	199.970713714224
200	 80	199.968308975216
200	 81	199.970945599771
200	 82	199.971815170573
200	 83	199.978142640088
200	 84	199.981739013158
200	 85	199.990295160432
200	 86	199.995673616874
200	 87	200.00649494241
201	 78	200.974492589808
201	 79	200.971641256413
201	 80	200.970285370088
201	 81	200.970803891937
201	 82	200.972850067182
201	 83	200.97697247691
201	 84	200.982211372606
201	 85	200.988491606176
201	 86	200.995534056128
201	 87	201.00397211354
202	 78	201.97573790108
202	 79	201.973784050636
202	 80	201.9706256836
202	 81	201.972091071433
202	 82	201.972143675098
202	 83	201.97767028064
202	 84	201.980697675284
202	 85	201.988448664408
202	 86	201.993215200656
202	 87	202.00332798702
203	 79	202.975137789872
203	 80	202.972857581991
203	 81	202.972329398245
203	 82	202.973375030296
203	 83	202.976868343123
203	 84	202.981416949898
203	 85	202.98684908355
203	 86	202.99334402596
203	 87	203.001052073316
203	 88	203.009211009236
204	 79	203.977702486966
204	 80	203.973475943451
204	 81	203.973848463288
204	 82	203.973028275519
204	 83	203.977809841386
204	 84	203.980306905195
204	 85	203.987257030346
204	 86	203.991368704632
204	 87	204.00059044931
204	 88	204.006473471526
205	 79	204.9796026602
205	 80	204.976055670163
205	 81	204.974412073993
205	 82	204.974466824747
205	 83	204.977375055985
205	 84	204.981170034732
205	 85	204.986033189958
205	 86	204.991669297008
205	 87	204.998668805192
205	 88	205.006183614592
206	 80	205.977498513568
206	 81	205.976095391299
206	 82	205.974448574496
206	 83	205.978482953599
206	 84	205.980464716193
206	 85	205.986602168384
206	 86	205.990155599686
206	 87	205.998486302678
206	 88	206.003778875584
207	 80	206.982576377634
207	 81	206.977408335855
207	 82	206.975880682459
207	 83	206.97845504145
207	 84	206.981577981528
207	 85	206.98577553935
207	 86	206.990724578112
207	 87	206.996854515494
207	 88	207.0037574047
207	 89	207.012098843134
208	 80	207.98593657098
208	 81	207.982004178575
208	 82	207.976635384031
208	 83	207.979727191327
208	 84	207.981231226751
208	 85	207.9865806975
208	 86	207.989631710116
208	 87	207.997133636986
208	 88	208.00177134793
208	 89	208.01148692294
209	 81	208.985349342303
209	 82	208.981074489298
209	 83	208.980383126833
209	 84	208.982415346004
209	 85	208.986158794629
209	 86	208.990381043968
209	 87	208.99592053204
209	 88	209.001943115002
209	 89	209.009565278822
210	 81	209.990065421973
210	 82	209.984172737859
210	 83	209.984105104575
210	 84	209.982857646214
210	 85	209.987131425675
210	 86	209.989680019605
210	 87	209.996398259209
210	 88	210.000450888564
210	 89	210.009253951004
210	 90	210.0150296188
211	 82	210.988731006533
211	 83	210.98725810389
211	 84	210.986636521798
211	 85	210.98748032754
211	 86	210.990585017366
211	 87	210.995529761951
211	 88	211.000891041686
211	 89	211.007643634704
211	 90	211.01481490996
212	 82	211.991887226481
212	 83	211.99127101211
212	 84	211.988851243483
212	 85	211.99073424001
212	 86	211.990689151153
212	 87	211.996199653532
212	 88	211.999783144072
212	 89	212.007815401776
212	 90	212.012914736726
213	 82	212.996500245908
213	 83	212.994374628392
213	 84	212.992842680819
213	 85	212.992921049545
213	 86	212.99386791553
213	 87	212.996174962015
213	 88	213.000343534144
213	 89	213.006570090504
213	 90	213.012957678494
213	 91	213.021181027066
214	 82	213.99979817369
214	 83	213.99869886443
214	 84	213.995186227807
214	 85	213.996356390985
214	 86	213.995346185893
214	 87	213.998954367949
214	 88	214.000091251257
214	 89	214.006892153764
214	 90	214.011454716614
214	 91	214.020740873944
215	 83	215.001835760582
215	 84	214.999414918411
215	 85	214.998640893043
215	 86	214.998728923667
215	 87	215.000326357437
215	 88	215.00270425784
215	 89	215.006452000642
215	 90	215.011723102664
215	 91	215.019098351318
216	 83	216.006205085476
216	 84	216.001905540955
216	 85	216.002409033185
216	 86	216.000257650608
216	 87	216.00318735273
216	 88	216.003518004343
216	 89	216.008717178904
216	 90	216.011051063995
216	 91	216.01910908676
217	 84	217.006258762686
217	 85	217.004709638405
217	 86	217.003914142153
217	 87	217.00461624006
217	 88	217.006305998631
217	 89	217.009332319731
217	 90	217.013065032914
217	 91	217.018293193168
218	 84	218.008966241158
218	 85	218.008681751945
218	 86	218.005586724017
218	 87	218.007563118889
218	 88	218.007124039311
218	 89	218.011626483686
218	 90	218.013267932768
218	 91	218.020010863888
218	 92	218.023489147096
219	 85	219.011293684984
219	 86	219.009475101109
219	 87	219.009241068474
219	 88	219.010068771052
219	 89	219.012410170952
219	 90	219.015523449132
219	 91	219.019882038584
219	 92	219.024916960882
220	 85	220.01529800485
220	 86	220.011383862697
220	 87	220.01231247843
220	 88	220.011014563492
220	 89	220.014750497308
220	 90	220.015732790251
220	 91	220.021878830796
220	 92	220.024712987484
221	 85	221.01814289698
221	 86	221.01545903648
221	 87	221.014245931534
221	 88	221.013907765111
221	 89	221.015577126342
221	 90	221.018171882673
221	 91	221.021868095354
221	 92	221.02635551011
222	 85	222.02232971936
222	 86	222.017570697921
222	 87	222.017543859316
222	 88	222.015361343958
222	 89	222.017828348529
222	 90	222.018454224798
222	 91	222.02372532682
222	 92	222.026065653176
223	 85	223.02533564312
223	 86	223.02179294726
223	 87	223.019730668852
223	 88	223.018497166566
223	 89	223.019126263467
223	 90	223.020795624698
223	 91	223.023961506544
223	 92	223.027718911244
224	 86	224.02404739008
224	 87	224.023231496488
224	 88	224.020201954756
224	 89	224.021708137268
224	 90	224.021459075014
224	 91	224.025614764612
224	 92	224.02759008594
225	 86	225.0284489213
225	 87	225.025607249803
225	 88	225.023604016325
225	 89	225.023220761046
225	 90	225.023941109204
225	 91	225.026119330386
225	 92	225.029382904754
225	 93	225.033902525836
226	 86	226.03091807296
226	 87	226.029339962986
226	 88	226.025402524924
226	 89	226.026090344693
226	 90	226.024891195821
226	 91	226.02793254654
226	 92	226.029339962986
226	 93	226.035126366224
227	 86	227.0354269586
227	 87	227.03183058553
227	 88	227.029170343002
227	 89	227.027746823393
227	 90	227.027698513904
227	 91	227.028793528988
227	 92	227.031140296609
227	 93	227.034954599152
228	 86	228.0381108191
228	 87	228.035727550976
228	 88	228.031064074971
228	 89	228.031014691938
228	 90	228.028731263425
228	 91	228.031037236366
228	 92	228.031366814436
228	 93	228.03617843954
228	 94	228.038722739294
229	 87	229.03843288236
229	 88	229.034815038406
229	 89	229.032925600614
229	 90	229.031755437436
229	 91	229.032088236138
229	 92	229.033495652584
229	 93	229.036242852192
229	 94	229.040139817638
230	 87	230.04251235032
230	 88	230.037080216668
230	 89	230.036028143352
230	 90	230.033126568088
230	 91	230.034532696281
230	 92	230.033927217353
230	 93	230.037810226724
230	 94	230.039645987306
231	 87	231.04541091966
231	 88	231.04122409728
231	 89	231.038550972222
231	 90	231.036297066174
231	 91	231.035878920708
231	 92	231.036289014593
231	 93	231.038228908962
231	 94	231.041256303606
231	 95	231.04551827408
232	 87	232.04970509646
232	 88	232.04369324894
232	 89	232.042018519988
232	 90	232.038050378562
232	 91	232.038582105004
232	 92	232.037146776408
232	 93	232.04009687587
232	 94	232.041179008424
232	 95	232.04659181828
233	 88	233.04798742574
233	 89	233.0445520843
233	 90	233.041576863904
233	 91	233.040240194021
233	 92	233.039627737055
233	 93	233.040730266948
233	 94	233.042984709768
233	 95	233.046473728418
233	 96	233.05077864066
234	 88	234.05056393182
234	 89	234.04841684342
234	 90	234.043595556418
234	 91	234.043302478851
234	 92	234.040945619915
234	 93	234.04288809079
234	 94	234.04330462594
234	 95	234.047794187784
234	 96	234.05024186856
235	 89	235.05110070392
235	 90	235.04750433085
235	 91	235.045432390544
235	 92	235.043923094753
235	 93	235.044055892171
235	 94	235.045281020812
235	 95	235.048030367508
235	 96	235.051594534252
235	 97	235.05657577934
236	 89	236.05518017188
236	 90	236.04970509646
236	 91	236.048674494028
236	 92	236.045561859975
236	 93	236.046559611954
236	 94	236.046048604915
236	 95	236.049565535714
236	 96	236.051401296296
236	 97	236.05732726028
237	 90	237.05389191884
237	 91	237.051143645688
237	 92	237.048723984416
237	 93	237.048167244394
237	 94	237.048403746181
237	 95	237.04997348251
237	 96	237.052893522734
237	 97	237.05711255144
237	 98	237.06205085476
238	 90	238.05625371608
238	 91	238.054493103592
238	 92	238.050782612774
238	 93	238.050940423771
238	 94	238.049553404665
238	 95	238.051981010164
238	 96	238.053011612596
238	 97	238.05829345006
238	 98	238.06140672824
239	 91	239.05711255144
239	 92	239.054287734587
239	 93	239.052931418844
239	 94	239.052156534641
239	 95	239.053018053861
239	 96	239.054954727598
239	 97	239.05840080448
239	 98	239.062576891418
240	 91	240.06097731056
240	 92	240.056585441238
240	 93	240.056168906088
240	 94	240.053807430911
240	 95	240.0552875263
240	 96	240.055519411847
240	 97	240.059753470172
240	 98	240.062297769926
240	 99	240.06892153764
241	 92	241.06033318404
241	 93	241.058250508292
241	 94	241.056845238934
241	 95	241.056822909215
241	 96	241.057646747034
241	 97	241.06022582962
241	 98	241.06376852548
241	 99	241.0687068288
242	 92	242.062920425562
242	 93	242.061632172522
242	 94	242.058736823815
242	 95	242.059543055509
242	 96	242.058829363325
242	 97	242.06205085476
242	 98	242.063693377386
242	 99	242.06967301858
242	100	242.07343042328
243	 93	243.064273091254
243	 94	243.06199717755
243	 95	243.061372696889
243	 96	243.061382251432
243	 97	243.063002014921
243	 98	243.065421783548
243	 99	243.06967301858
243	100	243.074514702922
244	 93	244.06784799344
244	 94	244.06419794316
244	 95	244.064279425165
244	 96	244.062746296693
244	 97	244.065167353573
244	 98	244.065990761974
244	 99	244.070972007062
244	100	244.0740745498
245	 94	245.067738491932
245	 95	245.066444871171
245	 96	245.065485122656
245	 97	245.066355337584
245	 98	245.068041231396
245	 99	245.071315541206
245	100	245.07536280284
245	101	245.0810525871
246	 94	246.070197981694
246	 95	246.069768564014
246	 96	246.067217500931
246	 97	246.068663887032
246	 98	246.068798831538
246	 99	246.072968799274
246	100	246.075276919304
246	101	246.08191142246
247	 94	247.0740745498
247	 95	247.07208849303
247	 96	247.070347204338
247	 97	247.070298894849
247	 98	247.070992404402
247	 99	247.07364513212
247	100	247.076822822952
247	101	247.08180406804
248	 95	248.075749278752
248	 96	248.072341849461
248	 97	248.073076153694
248	 98	248.072177597199
248	 99	248.075459421818
248	100	248.077184607347
248	101	248.082909818566
249	 95	249.07847608102
249	 96	249.075946810885
249	 97	249.074979547561
249	 98	249.07484642808
249	 99	249.076404140714
249	100	249.079023588562
249	101	249.083006437544
249	102	249.08781591556
250	 96	250.078350476349
250	 97	250.078310755213
250	 98	250.076399953892
250	 99	250.078658583534
250	100	250.079515271806
250	101	250.08448792854
250	102	250.0874938523
251	 96	251.082277501032
251	 97	251.080753068268
251	 98	251.079579684458
251	 99	251.079983337077
251	100	251.081566814772
251	101	251.08491734622
251	102	251.088964607854
251	103	251.09436453518
252	 96	252.08491734622
252	 97	252.084305426026
252	 98	252.081619418438
252	 99	252.082974231218
252	100	252.082460003546
252	101	252.08663501694
252	102	252.088965681398
252	103	252.09533072496
253	 97	253.08684972578
253	 98	253.085126687339
253	 99	253.084817506609
253	100	253.085176070372
253	101	253.08727914346
253	102	253.090650072248
253	103	253.095255576866
253	104	253.10069844596
254	 97	254.09060713048
254	 98	254.087316717507
254	 99	254.088015594781
254	100	254.086847578692
254	101	254.089726824236
254	102	254.090948517536
254	103	254.096618978
254	104	254.10016167386
255	 98	255.09103654816
255	 99	255.090266816969
255	100	255.089955489151
255	101	255.091075195751
255	102	255.093231946049
255	103	255.096769274188
255	104	255.101492868668
255	105	255.10735442
256	 98	256.0933983454
256	 99	256.093591583356
256	100	256.091766558216
256	101	256.094053207362
256	102	256.094275431011
256	103	256.0987660664
256	104	256.10118154085
256	105	256.10810590094
257	 99	257.09597485148
257	100	257.095098839413
257	101	257.095534698358
257	102	257.096855157724
257	103	257.099603430876
257	104	257.1030602432
257	105	257.107858985774
258	100	258.097069866564
258	101	258.098425752889
258	102	258.098197087974
258	103	258.10187934458
258	104	258.103564808974
258	105	258.10939415398
258	106	258.11315155868
259	100	259.10059109154
259	101	259.100505208004
259	102	259.10102050922
259	103	259.102995830548
259	104	259.105626013838
259	105	259.10971621724
259	106	259.11465452056
260	101	260.1035970153
260	102	260.102641560962
260	103	260.105572336628
260	104	260.106431171988
260	105	260.111423152518
260	106	260.11443981172
260	107	260.1218472667
261	102	261.1057441037
261	103	261.106946473204
261	104	261.10875002746
261	105	261.112110220806
261	106	261.11615748244
261	107	261.121804324932
262	102	262.10756912884
262	103	262.10971621724
262	104	262.10993092608
262	105	262.114149954786
262	106	262.1164795457
262	107	262.12302816532
263	103	263.11143388796
263	104	263.112539638486
263	105	263.115073202798
263	106	263.118315306282
263	107	263.12313551974
263	108	263.12871794958
264	104	264.11401039404
264	105	264.117477941806
264	106	264.11894869736
264	107	264.12474583604
264	108	264.128406621762
265	105	265.1186266341
265	106	265.121063579434
265	107	265.12517525372
265	108	265.13000620262
265	109	265.13655482224
266	106	266.12195462112
266	107	266.12700027886
266	108	266.13000620262
266	109	266.1379504297
267	107	267.1277517598
267	108	267.13177755055
267	109	267.13752101202
267	110	267.14396227722
268	108	268.13215329102
268	109	268.13880926506
268	110	268.14353285954
269	108	269.13408567058
269	109	269.13913132832
269	110	269.14514317584
270	109	270.14074164462
270	110	270.14460640374
271	109	271.14127841672
271	110	271.146077159294
272	110	272.14632407446
272	111	272.1535168206
273	110	273.1492226438
