#!/usr/local/bin/perl
#
# cafDBI.pm 25/08/2002
#
# cafeterra : data flow and data replication management
# Copyright (C) 2001  Abdellaziz TALEB
#
#This program is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public License
#as published by the Free Software Foundation; either version 2
#of the License, or (at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#
use 5.005;
use strict;
package cafDBI;

use DBI;

our $lastquery;
our $lastdatabase;

sub init {
	my $self = shift;
	$self->{queries} = {};
}

sub DbhInactiveDestroy {
	my $self = shift;
	if (@_) { $self->{dbh}{InactiveDestroy} = shift; }
	$self->{dbh}{InactiveDestroy},
}

sub untrace {
	shift->{dbh}->trace(0);
}

sub trace {
	shift->{dbh}->trace(3, "/tmp/cafdbi.trc");
}

sub dbh {
	my $self = shift;

	if (@_) { $self->{dbh} = shift; }
	$self->{dbh};
}

sub showstatement {
	my $self = shift;
	defined(@_) ? $self->{dbh}->{ShowErrorStatement} = shift: $self->{dbh}->{ShowErrorStatement};
}

sub autocommit {
	my $self = shift;
	return undef unless (ref ($self->{dbh}));
	defined(@_) ? $self->{dbh}->{AutoCommit} = shift: $self->{dbh}->{AutoCommit};
}

sub commit {
	my $self = shift;
	$self->{dbh}->commit() if ($self->{dbh} and (!$self->{dbh}->{AutoCommit}));
}

sub finalcommit {
	my $self = shift;
#	print "final committement of  $self->{db}{dbidsn} \n";
	$self->commit();
}

sub partialcommit {
	my $self = shift;
#	print "partial committement of  $self->{db}{dbidsn} \n";
	$self->commit();
}

sub rollback {
	my $self = shift;
	$self->{dbh}->rollback() if ($self->{dbh} and (!$self->{dbh}->{AutoCommit}));
}

sub disconnect {
	my $self = shift;
#	print "disconnecting from $self->{db}{dbidsn} / $self->{dbh}->{AutoCommit}\n";
	$self->finalcommit();
	$self->{dbh}->disconnect();
}

sub queries {
	my $self = shift;

	$self->{queries} = {} unless ($self->{queries});

	keys %{$self->{queries}};
}

sub query {
	my $self = shift;
	my $qlabel = shift;

	if (@_) { $self->{queries}{$qlabel} = shift; }
	else { $self->{queries}{$qlabel}; }
}

sub lastquery {
	my $self = shift;

	if (@_) {
		my $db = $self->{db};
		$lastquery = shift;
		$lastdatabase = $db->{connector}{name} . "@" . $db->{user}{username} . ":" . $db->{connector}{name};
	}
	else { $lastquery; }
}

sub lastdatabase {
	my $self = shift;

	if (@_) {
		my $db = $self->{db};
		$lastdatabase = $db->{connector}{name} . "@" . $db->{user}{username} . ":" . $db->{connector}{name};
	}
	else { $lastdatabase; }
}

sub finish {
	my $self = shift;
	my $q = shift;

	return 1 if ($q and $q->_attribute("finished"));
	if ($self->lastquery()) { $self->lastquery(undef) if ($q->mylabel() eq ($self->lastquery())->mylabel()); }

	my $sth = $q->_attribute("sth");
	if ($sth) { $sth->finish(); $q->_attribute("sth", undef); };
	$q->_attribute("finished", 1);
	$q->_attribute("prepared", undef);
	$q->_attribute("executed", undef);
	$q->_attribute("fetch", undef);

	delete $self->{queries}{$q->mylabel()};

	cafDbg->popstackdump();
	cafDbg->popstackdump();
}

sub prepare {
	my $self = shift;
	my $q = shift;

	cafDbg->pushstackdump(1);
	$self->lastquery($q);

	$q->mylabel ("undefined") unless ($q->mylabel());

	if ($self->query($q->mylabel())) { my $qt = $self->query($q->mylabel()); $self->finish($qt); }

	my $dbh = $self->dbh();
	my $sth = $dbh->prepare ($q->query());
	my $ret;

	if ($sth) {
		$q->_attribute("sth", $sth);
		$q->_attribute("prepared", 1);
		##$self->{queries}{$q->mylabel()} = $q;
		$self->query($q->mylabel(), $q);
		$ret = $q;
	}
	else { $ret = undef; }
	cafDbg->popstackdump();
	return $ret;
}

sub execute {
	my $self = shift;

	my $q = shift;
	cafDbg->pushstackdump(1);

	$self->lastquery($q);

	$self->prepare($q) unless ($q->_attribute("prepared") and $self->query($q->mylabel()));

	my $sth = $q->_attribute("sth");

	my $bindvars = $q->bindvars();
	my @bindvars;
	if ($bindvars) {
		@bindvars = @{$bindvars};
		foreach my $b (@bindvars) { $b = undef unless ($b or (length($b) > 0)); }
	}
	my $ret = $sth->execute(@bindvars);

	$q->_attribute("executed", 1);

	cafDbg->popstackdump();
	return $ret;
}

sub getcolnames {
	my $self = shift;
	my $q = shift;

	my $sth = $q->_attribute("sth");

	my $colnames = ($sth ? $sth->{NAME_lc} : undef);

	$colnames = undef unless ($colnames && defined($colnames->[0]));
	$q->columns($colnames) if ($colnames);

	$colnames;
}

sub executefinish {
	my $self = shift;
	my $q = shift;

	cafDbg->pushstackdump(1);

	my $ret = $self->execute($q);
	$self->finish($q);

	cafDbg->popstackdump();


	$ret;
}

sub fetchall {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	$self->lastquery($q);

	my $sth = $q->_attribute("sth");

	my $rows = $sth->fetchall_arrayref();

	$q->_rows($rows);

	$self->finish($q);

	cafDbg->popstackdump();

	return $rows;
}

sub fetchrow {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	my $finish = shift;

	$self->lastquery($q);

	my $sth = $q->_attribute("sth");

	my @row = $sth->fetchrow_array();

	my $ret;

	if (@row and ($#row >= 0)) { 
		$q->_pushrow(\@row);

		$q->_attribute("fetch", 1);

		$ret = \@row;
	}
	else { $ret = undef; $finish = 1; }


	$self->finish($q) if ($finish);

	cafDbg->popstackdump();

	return $ret;
}

sub fetchrownop {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	my $finish = shift;

	$self->lastquery($q);

	my $sth = $q->_attribute("sth");

	my @row = $sth->fetchrow_array();

	my $ret;

	if (@row and ($#row >= 0)) { 
		$q->_rows([ \@row ]);

		$q->_attribute("fetch", 1);

		$ret = \@row;

	}
	else { $ret = undef; $finish = 1; }

	$self->finish($q) if ($finish);

	cafDbg->popstackdump();

	return $ret;
}

sub execfetchrow {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	my $finish = shift;

	my $ret = $self->execute($q);

	$ret = $self->fetchrow($q, $finish) if ($ret);

	cafDbg->popstackdump();

	return $ret;
}

sub execfetchrownop {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	my $finish = shift;

	my $ret = $self->execute($q);

	$ret = $self->fetchrownop($q, $finish) if ($ret);

	cafDbg->popstackdump();

	return $ret;
}

sub execfetchall {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	$self->execute($q);

	my $rows = $self->fetchall ($q, 1);

	cafDbg->popstackdump();

	return $rows;
}

sub execfetchset {
	my $self = shift;
	my $q = shift;
	my $start = shift || 1;
	my $count = shift || -1;

	cafDbg->pushstackdump(1);

	$self->execute($q);

	my $fetched = 1;
	my $row;
	my $sth = $q->_attribute("sth");
	while ($fetched < $start) {
		$row = $sth->fetchrow_arrayref();
		last unless ($row);
		$fetched++;
	}

	$fetched = 0;
	while (1) {
		my @row = $sth->fetchrow_array();
		last unless (@row);

		$q->_pushrow(\@row);
		$fetched++;
		last if (($fetched >= $count) && ($count > -1));
	}

	$self->finish($q);

	cafDbg->popstackdump();

	return $q->_rows();
}

1;

# Fetching in hash structures
sub hfetchall {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	$self->lastquery($q);

	my $sth = $q->_attribute("sth");

# my $rows = $sth->fetchall_hashref();
	$q->_hrows(undef);

	while (1) {
		my $row = $sth->fetchrow_hashref();
		last unless ($row);
		my %row = %{ $row };

		$q->_hpushrow(\%row);
	}

	$self->finish($q);

	cafDbg->popstackdump();

	return $q->_hrows();
}

sub hfetchrow {
	my $self = shift;

	my $q = shift;
	my $finish = shift;

	cafDbg->pushstackdump(1);

	$self->lastquery($q);

	my $sth = $q->_attribute("sth");

	my $hrow = $sth->fetchrow_hashref();

	my $ret =  undef;

	if ($hrow) {
		my %row;
		%row = %{  $hrow };

		$q->_hpushrow(\%row);

		$q->_attribute("fetch", 1);

		$self->finish($q) if ($finish);

		$ret =\%row;
	}

	cafDbg->popstackdump();

	return $ret;
}

sub hfetchrownop {
	my $self = shift;

	my $q = shift;
	my $finish = shift;

	cafDbg->pushstackdump(1);

	$self->lastquery($q);

	my $sth = $q->_attribute("sth");

	my $hrow = $sth->fetchrow_hashref();
	my %row;
	my %row = %{ $hrow } if ($hrow);

	$q->_hrows([ \%row ]);

	$q->_attribute("fetch", 1);

	$self->finish($q) if ($finish);

	cafDbg->popstackdump();

	return (\%row)  if ($hrow);
	return undef;
}

sub hexecfetchrow {
	my $self = shift;

	my $q = shift;

	my $finish = shift;

	cafDbg->pushstackdump(1);

	my $ret = $self->execute($q);

	$ret = $self->hfetchrow($q, $finish) if ($ret);

	cafDbg->popstackdump();

	return $ret;
}

sub hexecfetchrownop {
	my $self = shift;

	my $q = shift;

	my $finish = shift;

	cafDbg->pushstackdump(1);

	my $ret = $self->execute($q);

	$ret = $self->hfetchrownop($q, $finish) if ($ret);

	cafDbg->popstackdump();

	return $ret;
}

sub hexecfetchall {
	my $self = shift;

	my $q = shift;

	cafDbg->pushstackdump(1);

	$self->execute($q);

	my $rows = $self->hfetchall ($q, 1);

	cafDbg->popstackdump();

	return $rows;
}

sub hexecfetchset {
	my $self = shift;
	my $q = shift;
	my $start = shift || 1;
	my $count = shift || -1;

	cafDbg->pushstackdump(1);

	$self->execute($q);
	
	my $fetched = 1;
	my $row = [];
	my $sth = $q->_attribute("sth");
	my $ret;
	while ($fetched < $start) {
		$row = $sth->fetchrow_arrayref();
		last unless ($row);
		$fetched++;
	}

	if ($row) {
		$fetched = 0;
		while ($row) {
			$row = $sth->fetchrow_hashref();
			last unless ($row);
			my %row = %{ $row };

			$q->_hpushrow(\%row);
			$fetched++;
			last if (($fetched >= $count) && ($count > -1));
		}
		$ret = $q->_hrows();
	}
	else { $ret = undef; }

	$self->finish($q);

	cafDbg->popstackdump();

	return $ret;
}

sub listtables {
	my $self = shift;
	my $dir = shift;
	my $pattern = shift;
	my $type    = shift || "'TABLE','VIEW'";

	my $lpattern = $pattern;
	$lpattern =~ s/%/.*/g;
	$lpattern =~ s/_/./g;

	my @rows;
	my $sth = $self->{dbh}->table_info( {
                  TABLE_CAT   => '%'  # String value of the catalog name
                , TABLE_SCHEM => '%'   # String value of the schema name
                , TABLE_NAME  => $pattern    # String value of the table name
                , TABLE_TYPE  => $type     # String value of the table type(s)
             });
	my $i = -1;
	while (my $row = $sth->fetchrow_hashref()) {
		next unless ($row->{TABLE_NAME} =~ /$lpattern/);
		$i++;
		$rows[$i] = { 	externalname => $row->{TABLE_NAME}, 
				name         => $row->{TABLE_NAME},
				Type         => $row->{TABLE_TYPE},
				externaltype   => $row->{TABLE_TYPE},
				Owner        => $row->{TABLE_SCHEM},
				externalschema => $row->{TABLE_SCHEM},
				Remarks      => $row->{REMARKS}
				};
	}

	return \@rows
}

sub protlisttables {
	listtable(@_);
}

sub describe {
	my $self = shift;
	my $tablename = shift;
	my $schema = shift || "%";
	my $type   = shift;

	my @rows;
	my %c;
	my $sth = $self->{dbh}->column_info(undef, $schema, $tablename, "%");
	my $i = -1;

	while (my $row = $sth->fetchrow_hashref()) {
#		print "<BR> $row->{TABLE_NAME} - $row->{TABLE_TYPE} - $row->{REMARKS}";
		unless ($c{$row->{COLUMN_NAME}}) {
			$c{$row->{COLUMN_NAME}}=1;
			$i++;
			$rows[$i] = { 	externalname => $row->{COLUMN_NAME}, 
				name         => $row->{COLUMN_NAME},
				datatypeid   => $row->{SQL_DATA_TYPE},
				datalength   => $row->{COLUMN_SIZE},
				datascale    => $row->{DECIMAL_DIGITS},
				fieldorder   => $row->{ORDINAL_POSITION},
				nullable     => $row->{IS_NULLABLE},
				defaultvalue => $row->{OLUMN_DEF},
				Remarks      => $row->{REMARKS},
#{ name => $col, externalname => $col, datatypeid => 'VARCHAR', datalength => 100,  fieldorder => $i*10};
			};
		}
	}

	return \@rows
}

sub gencolalias {
	my $self = shift;
	shift;
}

sub columnnameformat {
	my $self = shift;
	my $col = shift;
	return $col->{externalname};
}

sub tablenameformat {
	my $self = shift;
	my $container = shift;
	return $container->{externalname};
}

sub generateselect {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $class = ref($self) || $self;

	my $query = "#select statement generated by $class\n\nSELECT ";
	my $sep = "";
	my $where = "WHERE ";
	my $chunk;
	my $wsep = "";
	my $qlen = 20;
	my $wlen = 10;

	foreach my $col (@$fields) {
		next if ($col->{localfield} eq "yes");
		if ($qlen > 50) { $query .= "$sep\n\t\t"; $qlen = 16; $sep = ""; }
		$chunk = "$sep" . $self->generatedatetochar($col) . " " . $self->gencolalias("\@c_$col->{name}");

		$qlen += length($chunk);

		$query .= $chunk;
		$sep = ", ";

		if ($col->{keyposition}) {
			my $colname = $self->columnnameformat($col);
			$chunk = "$wsep$colname = " . $self->generatechartodate($col);
			$where .= $chunk;
			$wlen += length($chunk);
			if ($wlen > 50) { $where .= "\n\t\t"; $wlen = 16; }
			$wsep = " and ";
		}
	}

	my $tablename = $self->tablenameformat($container);
	return "$query\n\tFROM $tablename\n\t$where";
}

sub generateupdate {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $class = ref($self) || $self;

	my $tablename = $self->tablenameformat($container);
	my $query = "#update statement generated by $class\n\nUPDATE $tablename SET ";
	my $sep = "";
	my $where = "";
	my $chunk;
	my $wsep = " WHERE ";
	my $qlen = 20;
	my $wlen = 20;

	foreach my $col (@$fields) {
		next if ($col->{localfield} eq "yes");
		if ($qlen > 50) { $query .= "$sep\n\t\t"; $qlen = 16; $sep = ""; }
		my $colname = $self->columnnameformat($col);
		$chunk = "$sep$colname = " . $self->generatechartodate($col);

		$qlen += length($chunk);

		$query .= $chunk;
		$sep = ", ";

		if ($col->{keyposition}) {
			$chunk = "$wsep$colname = " . $self->generatechartodate($col);
			$wlen += length($chunk);
			if ($wlen > 50) { $where .= "\n\t\t"; $wlen = 16; }
			$where .= $chunk;
			$wsep = " and ";
		}
	}

	return "$query\n\t$where";
}

sub generatedelete {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $class = ref($self) || $self;

	my $tablename = $self->tablenameformat($container);
	my $query = "#delete statement generated by $class\n\nDELETE from $tablename";
	my $sep = "";
	my $where = "";
	my $chunk;
	my $wsep = "WHERE ";
	my $qlen = 20;
	my $wlen = 20;

	foreach my $col (@$fields) {
		next if ($col->{localfield} eq "yes");

		if ($col->{keyposition}) {
			my $colname = $self->columnnameformat($col);
			$chunk = "$wsep$colname = " . $self->generatechartodate($col);
			$wlen += length($chunk);
			if ($wlen > 50) { $where .= "\n\t\t"; $wlen = 10; }
			$where .= $chunk;
			$wsep = " and ";
		}
	}

	return "$query\n\t$where";
}

sub generateinsert {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $class = ref($self) || $self;

	my $tablename = $self->tablenameformat($container);
	my $query = "#insert statement generated by $class\n\nINSERT INTO $tablename (";
	my $sep = "";
	my $values = " VALUES (";
	my $chunk;
	my $vsep = "";
	my $qlen = 20;
	my $vlen = 10;

	foreach my $col (@$fields) {
		next if ($col->{localfield} eq "yes");
		if ($qlen > 50) { $query .= "$sep\n\t\t"; $qlen = 16; $sep = ""; }
		my $colname = $self->columnnameformat($col);
		$chunk = "$sep$colname";

		$qlen += length($chunk);

		$query .= $chunk;
		$sep = ", ";

		if ($vlen > 50) { $values .= "$vsep\n\t\t"; $vlen = 16; $vsep = ""; }
		$chunk = "$vsep" . $self->generatechartodate($col);
		$vlen += length($chunk);
		$values .= $chunk;
		$vsep = ", ";
	}

	return "$query)\n\t$values)";
}


sub generatecreate {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $datatypes = shift;
	my $class = ref($self) || $self;

	my %datatypeconv = map { $_->{stdname} => $_->{datatypeid} } @$datatypes;

	my $tablename = $self->tablenameformat($container);
	my $query = "#WARNING THIS IS A DDL STATEMENT - SEE WITH YOUR DBA BEFORE USING SUCH COMMAND
#CREATE TABLE statement generated by $class\n\nCREATE TABLE $tablename (";


	my $sep = "";
	my $keys = "";
	my $ksep = "";
	my $chunk = "";

	foreach my $col (@$fields) {
		my $colname = $self->columnnameformat($col);
		my $datatype = $datatypeconv{$col->{datatypeid}};
		$query .= "$sep\n\t$colname $datatype";
		if (($datatype =~ /VARCHAR|CHAR/i) and ($col->{datalength})) { $query .= "($col->{datalength})"; }
		elsif (($datatype =~ /NUMBER/i) and ($col->{datalength})) {
			$query .= "($col->{datalength}";
			$query .=  ", $col->{datascale}" if ($col->{datascale});
			$query .= ")";
		}
		$sep = ",";

		if ($col->{allownull} eq 'no') { $query .= " NOT NULL"; }

		if ($col->{keyposition}) {
			$keys .= "$ksep$colname";
			$ksep = ", ";
		}

	}
	$keys = "$keys\n\tCONSTRAINT " . $tablename . "_pkey PRIMARY KEY ($keys)" if ($ksep);
	return "$query) $keys";
}

sub generatedrop {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $datatypes = shift;
	my $class = ref($self) || $self;

	my $tablename = $self->tablenameformat($container);
	my $query = "#WARNING THIS IS A DDL STATEMENT - SEE WITH YOUR DBA BEFORE USING SUCH COMMAND
#DROP TABLE statement generated by $class

DROP TABLE $tablename";
	return $query;
}

sub generatetruncate {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $datatypes = shift;
	my $class = ref($self) || $self;

	my $tablename = $self->tablenameformat($container);
	my $query = "#WARNING THIS IS A DDL STATEMENT - SEE WITH YOUR DBA BEFORE USING SUCH COMMAND
#TRUNCATE TABLE statement generated by $class

TRUNCATE TABLE $tablename";
	return $query;
}

sub generatequery {
	my $self = shift;
	my $command = shift;
	my $connector = shift;
	my $container = shift;
	my $fields = shift;
	my $datatypes = shift;

	my $sub = "generate$command";
	return $self->$sub($command, $connector, $container, $fields, $datatypes);
}

1;
