#!/usr/local/bin/perl
#
# refQry.pm 01/08/2002
#
# cafeterra : data flow and data replication management
# Copyright (C) 2001  Abdellaziz TALEB
#
#This program is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public License
#as published by the Free Software Foundation; either version 2
#of the License, or (at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#

use 5.005;
use strict;

package refQry;

use connectors::cafQry;

@refQry::ISA = ('cafQry');

use tools::cafUtils;


# Sessions management queries
sub suserauth {
	my $p = shift;
 
	$p->{_attr}{bindvars} = [ $p->{userid}, $p->{userpass} ];

	$p->{_attr}{mylabel} = "suserauth";
	$p->{_attr}{columns} = [ "userid", "userlabel", "userpass", "status", "hostsallow", "hostsdeny" ];

	$p->{_attr}{query} = "select userid, userlabel, userpass, status, hostsallow, hostsdeny from cafusers
where userid = ? and userpass = ? and status like 'online'";
}

sub susersess {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "susersess";
	$p->{_attr}{columns} = [];
	push @{$p->{_attr}{columns}}, qw (sessionid userid userlabel userstatus hostsallow hostsdeny connecttime ipaddress lastaccess);

	$p->{_attr}{bindvars} = [ $p->{sessionid} ];

	$p->{_attr}{query} = "select sess.sessionid, usr.userid, usr.userlabel, usr.status, usr.hostsallow, usr.hostsdeny,
to_char(sess.connecttime, 'DD/MM/YYYY HH24:MI:SS') as connectime,
sess.ipaddress, to_char(sess.lastaccess, 'DD/MM/YYYY HH24:MI:SS') as lastaccess
from cafusers usr, sessions sess
where sess.sessionid = ? and sess.userid = usr.userid";
}

sub iusersess {
	my $p = shift; # self
	$p->{_attr}{mylabel} = "iusersess";
	$p->{sessionid} = $p->{ipaddress} . ":" . cafUtils->datetime2() . $p->{sessionseq};
	$p->{sessionid} =~ s/\.//g;

	$p->{_attr}{bindvars} = [ $p->{sessionid}, $p->{userid}, $p->{ipaddress}, $p->{vars} ];

	$p->{_attr}{query} = "insert into sessions (sessionid, userid, connecttime, ipaddress, lastaccess, vars) values (?, ? || '', now(), ?, now(), ?)";
}

sub uusersess {
	my $p = shift; # self

	$p->{_attr}{bindvars} = [ $p->{vars}, $p->{sessionid} ];

	$p->{_attr}{query} = "update sessions set lastaccess = now(), vars = ? where sessionid = ?";
}

#Cafeterra users administration

sub scafusrlist {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "scafusrlist";
	$p->{_attr}{columns} = [ "userid", "userlabel", "userpass", "status", "hostsallow", "hostsdeny" ];
	$p->{_attr}{bindvars} = [];

	my $where = " where ";
	my $q = "select userid, userlabel, userpass, status, hostsallow, hostsdeny from cafusers";
	if ($p->{userid} && ($p->{userid} ne '%')) {
		$q .= "$where userid like ?";
		$where = " and ";
		push @{$p->{_attr}{bindvars}}, $p->{userid};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= "$where status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	$p->{_attr}{query} = "$q order by userid";
}

sub icafusr {
	my $p = shift;

	$p->{_attr}{mylabel} = "icafusr";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(userid userlabel userpass status hostsallow hostsdeny));
	# [ $p->{userid}, $p->{userlabel}, $p->{userpass}, $p->{status}, $p->{hostsallow}, $p->{hostsdeny} ];
	$p->{_attr}{query} = "insert into cafusers (userid, userlabel, userpass, status, hostsallow, hostsdeny) values (?, ?, ?, ?, ?, ?)";
}

sub ucafusr {
	my $p = shift;

	$p->{_attr}{mylabel} = "ucafusr";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(userlabel userpass status hostsallow hostsdeny userid));
	$p->{_attr}{query} = "update cafusers set userlabel = ?, userpass = ?, status = ?, hostsallow = ?, hostsdeny = ? where userid = ?";
}

sub dcafusr {
	my $p = shift;

	$p->{_attr}{mylabel} = "dcafusr";
	$p->{_attr}{bindvars} = [ $p->{userid} ];
	$p->{_attr}{query} = "delete from cafusers where userid = ?";
}

# Servers queries

sub sserverlist {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sserverlist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "host_name", "host_address" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT id, type, name, parent_id, created, modified, deleted, comments,
		status, createdby, modifiedby, object_id, host_name, host_address FROM objects, server
		where id = object_id and type = 'server'";
	if ($p->{object_id}) {
		$q .= " and object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((name like ?) or (host_name like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
#	if ($p->{object_id} && ($p->{object_id} ne '%')) {
#		$q .= " and object_id = ?";
#		push @{$p->{_attr}{bindvars}}, $p->{object_id};
#	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", id" }
	else { $p->{_attr}{query} = "$q order by id"; }
#	$p->{_attr}{query} = "$q order by id";
}

sub iserver {
	my $p = shift;

	$p->{_attr}{mylabel} = "iserver";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id host_name host_address));
	$p->{_attr}{query} = "insert into server (object_id, host_name, host_address) values (?, ?, ?)";
}

sub userver {
	my $p = shift;

	$p->{_attr}{mylabel} = "userver";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(host_name host_address object_id));

	$p->{_attr}{query} = "update server set host_name = ?, host_address = ? where object_id = ?";
}

sub dserver {

	my $p = shift;

	$p->{_attr}{mylabel} = "dserver";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from server where object_id = ?";
}
# Proxy queries 

sub scafproxylist {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "scafproxylist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "user_id", "port", 
				 "dbidsn", "cipher", "key", "usercipher", "userkey", "compression", "objectlabel" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT id, type, name, parent_id, created, modified, deleted, comments,
		status, createdby, modifiedby, object_id, user_id, port, 
		dbidsn, cipher, key, usercipher, userkey, compression, objectlabel
		FROM objects, cafproxy
		where id = object_id and type = 'cafproxy'";
	if ($p->{object_id}) {
		$q .= " and object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((name like ?) or (objectlabel like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	if ($p->{parent_id} && ($p->{parent_id} ne '%')) {
		$q .= " and parent_id = ?";
		push @{$p->{_attr}{bindvars}}, $p->{parent_id};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", id" }
	else { $p->{_attr}{query} = "$q order by id"; }
}

sub icafproxy {
	my $p = shift;

	$p->{_attr}{mylabel} = "icafproxy";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id user_id port dbidsn cipher key usercipher userkey compression));
	$p->{_attr}{query} = " insert into cafproxy
			(object_id, user_id, port, dbidsn, cipher, key, usercipher, userkey, compression)
			values (?, ?, ?, ?, ?, ?, ?, ?, ?)";
}

sub ucafproxy {
	my $p = shift;

	$p->{_attr}{mylabel} = "ucafproxy";
	$p->{port} = 0 unless($p->{port});
	$p->{dbidsn} = 0 unless($p->{port});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(user_id port dbidsn cipher key usercipher userkey compression object_id));

	$p->{_attr}{query} = "update cafproxy set user_id = ?, port = ?, dbidsn = ?, 
					cipher = ?, key = ?, usercipher = ?, userkey = ?, compression = ?
					 where object_id = ?";
}

sub dcafproxy {

	my $p = shift;

	$p->{_attr}{mylabel} = "dcafproxy";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from cafproxy where object_id = ?";
}

# Connector queries

sub sconnectorlist {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sconnectorlist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "ismaster", "master_id", "context_id",
				 "userid", "protoname", "driverid", "port", "dbidsn", "externalname", "proxy_id" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT id, type, name, parent_id, created, modified, deleted, comments,
		status, createdby, modifiedby, object_id, ismaster, master_id, contextid, userid,
		protocolid, driverid, port, dbidsn, externalname, proxy_id FROM objects, connector
		where id = object_id and type = 'connector'";
	if ($p->{object_id}) {
		$q .= " and object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= $p->{object_id}"; }
# push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((name like ?) or (externalname like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	if ($p->{parent_id} && ($p->{parent_id} ne '%')) {
		$q .= " and parent_id = ?";
		push @{$p->{_attr}{bindvars}}, $p->{parent_id};
	}
	if ($p->{driverid} && $p->{driverid} ne '%') {
		$q .= "and driverid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{driverid};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", id" }
	else { $p->{_attr}{query} = "$q order by id"; }
}

sub iconnector {
	my $p = shift;

	$p->{_attr}{mylabel} = "iconnector";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id master_id contextid userid ismaster protocolid driverid port dbidsn externalname proxy_id));
	$p->{_attr}{query} = " insert into connector
			(object_id, master_id, contextid, userid, ismaster, protocolid, driverid, port, dbidsn, externalname, proxy_id)
			values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
}

sub uconnector {
	my $p = shift;

	$p->{_attr}{mylabel} = "uconnector";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(master_id contextid userid ismaster protocolid driverid port dbidsn externalname proxy_id object_id));

	$p->{_attr}{query} =  "update connector set master_id = ?, contextid = ?, userid = ?, 
		 ismaster = ?, protocolid = ?, driverid = ?, port = ?, dbidsn = ?, externalname = ?, proxy_id = ? where object_id = ?";
}

sub dconnector {

	my $p = shift;

	$p->{_attr}{mylabel} = "dconnector";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from connector where object_id = ?";
}

# Fields Queries

sub smappinglist {
	my $p = shift;

	$p->{_attr}{mylabel} = "smappinglist";
	$p->{_attr}{columns} = [ "subflow_id", "incomfield_id", "outgofield_id", "script_id", "formula", "pformula" ];
	$p->{_attr}{bindvars} = [ $p->{subflow_id} ];

	$p->{_attr}{query} = "select subflow_id, incomfield_id, outgofield_id, script_id, formula, pformula 
		from mapping where subflow_id = ?";

}

sub dmapping {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "dmapping";
	$p->{_attr}{bindvars} = [ $p->{subflow_id} ];

	$p->{_attr}{query} = "delete from mapping where subflow_id = ?";

}

sub imapping {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "imapping";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(subflow_id incomfield_id outgofield_id script_id formula pformula));

	$p->{_attr}{query} = "insert into mapping (subflow_id, incomfield_id, outgofield_id, script_id, formula, pformula)
				values (?, ?, ?, ?, ?, ?)";

}

sub sflowincomfields {
	my $p = shift; # self

# SELECT "object_id", "externalname", "flowdirection" FROM "field"

	$p->{_attr}{mylabel} = "sflowincomfields";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby",  "object_id", "externalname", "datatypeid",
				 "datalength", "datascale", "dataformat", "fieldorder", "fieldpos", "keyposition", "localfield",
				 "referto_id", "xmlnode", "defaultvalue", "parent_id_name", "driverid", "protocolid" ];

	$p->{_attr}{bindvars} = [$p->{flow_id}];

	my $q = " SELECT o1.id, o1.type, o1.name, o1.parent_id, o1.created, o1.modified, o1.deleted, o1.comments,
		o1.status, o1.createdby, o1.modifiedby, f.object_id, f.externalname, f.datatypeid,
		f.datalength, f.datascale, f.dataformat, f.fieldorder, f.fieldpos, f.keyposition, f.localfield,
		f.referto_id, f.xmlnode, f.defaultvalue, o2.name as parent_id_name, c.driverid, c.protocolid
		FROM objects o1, field f, objects o2, connector c, subflow sf, objects o3
		where o1.id = f.object_id and o1.type = 'field'
		and o1.parent_id = o2.id
		and c.object_id = o2.parent_id
		and sf.container_id = o1.parent_id
		and sf.flow_id = ?
		and sf.flowdirection = 'incom'
		and sf.object_id = o3.id
		and o3.status = 'online'";

	if (($p->{incomrel} ne "dependant")) {
		$q .= " and sf.floworder in  (select min (floworder) from subflow, objects where flow_id=? and object_id=id and flowdirection = 'incom' and status='online')";
		push @{$p->{_attr}{bindvars}}, $p->{flow_id};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", o1.parent_id, o1.id"; }
	else { $p->{_attr}{query} = "$q order by o1.parent_id, o1.id"; }
}


sub sfieldlist {
	my $p = shift; # self

# SELECT "object_id", "externalname", "flowdirection" FROM "field"

	$p->{_attr}{mylabel} = "sfieldlist";
	$p->{_attr}{columns} = [ "id", "type", "name", "objectlabel", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby",  "object_id", "externalname", "datatypeid",
				 "datalength", "datascale", "dataformat", "fieldorder", "fieldpos", "keyposition", "localfield",
				 "referto_id", "xmlnode", "defaultvalue", "nullable", "parent_id_name", "driverid", "protocolid" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT o1.id, o1.type, o1.name, o1.objectlabel, o1.parent_id, o1.created, o1.modified, o1.deleted, o1.comments,
		o1.status, o1.createdby, o1.modifiedby, f.object_id, f.externalname, f.datatypeid,
		f.datalength, f.datascale, f.dataformat, f.fieldorder, f.fieldpos, f.keyposition, f.localfield,
		f.referto_id, f.xmlnode, f.defaultvalue, f.nullable, o2.name as parent_id_name, c.driverid, c.protocolid
		FROM objects o1, field f, objects o2, connector c
		where o1.id = f.object_id and o1.type = 'field'
		and o1.parent_id = o2.id
		and c.object_id = o2.parent_id";

	if ($p->{object_id}) {
		$q .= " and f.object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((o1.name like ?) or (f.externalname like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and o1.status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	if ($p->{parent_id}) {
		$q .= " and o2.id = ?";
		push @{$p->{_attr}{bindvars}}, $p->{parent_id};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", o1.parent_id, o1.id"; }
	else { $p->{_attr}{query} = "$q order by o1.parent_id, f.fieldorder, o1.id"; }
}

sub ifieldcopy {
	my $p = shift;

	$p->{_attr}{mylabel} = "ifieldcopy";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id copyobject_id ));

	$p->{_attr}{query} = " insert into field (object_id, externalname, datatypeid, datalength, datascale,
				 dataformat, fieldorder, fieldpos, keyposition, localfield, referto_id, xmlnode, defaultvalue, nullable)
				 (select ?, externalname, datatypeid, datalength, datascale, dataformat, fieldorder,
					fieldpos, keyposition, localfield, referto_id, xmlnode, defaultvalue, nullable
					from field where object_id = ?)";
}

sub ifield {
	my $p = shift;

	$p->{_attr}{mylabel} = "ifield";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id externalname datatypeid datalength datascale
							 dataformat fieldorder fieldpos keyposition localfield referto_id xmlnode defaultvalue nullable));

	$p->{_attr}{query} = " insert into field (object_id, externalname, datatypeid, datalength, datascale,
				 dataformat, fieldorder, fieldpos, keyposition, localfield, referto_id, xmlnode, defaultvalue, nullable)
			values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
}

sub ufield {
	my $p = shift;

	$p->{_attr}{mylabel} = "ufield";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(externalname datatypeid datalength datascale
							 dataformat fieldorder fieldpos keyposition localfield referto_id xmlnode defaultvalue nullable object_id));

	$p->{_attr}{query} = "update field set externalname = ?, datatypeid = ?, datalength = ?, datascale = ?,
             dataformat = ?, fieldorder = ?, fieldpos = ?, keyposition = ?, localfield = ?, referto_id = ?,
				 xmlnode = ?, defaultvalue = ?, nullable = ?  where object_id = ?";
}

sub dfield {

	my $p = shift;

	$p->{_attr}{mylabel} = "dfield";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from field where object_id = ?";
}

# Containers Queries

sub ssubflowcontainer {
	my $p = shift; # self

# SELECT "object_id", "externalname", "flowdirection" FROM "container"

	$p->{_attr}{mylabel} = "ssubflowcontainer";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "externalname", "flowdirection", "rollbackspace", "xcreate",
				 "externaltype", "externalschema", "connector_externalname", "driverid", "protocolid" ];

	$p->{_attr}{bindvars} = [ $p->{ object_id } ];

	my $q = " SELECT o1.id, o1.type, o1.name, o1.parent_id, o1.created, o1.modified, o1.deleted, o1.comments,
		o1.status, o1.createdby, o1.modifiedby, c1.object_id, c1.externalname, c1.flowdirection, c1.rollbackspace, c1.xcreate,
		c1.externaltype, c1.externalschema, c2.externalname as connector_externalname, c2.driverid, c2.protocolid
		FROM objects o1, container c1, connector c2, subflow sf
		where o1.id = c1.object_id and o1.type = 'container'
		and c2.object_id = o1.parent_id
		and sf.container_id = o1.id
		and sf.object_id = ?";

	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", o1.id" }
	else { $p->{_attr}{query} = "$q order by o1.id"; }
}

sub scontainerlist {
	my $p = shift; # self

# SELECT "object_id", "externalname", "flowdirection" FROM "container"

	$p->{_attr}{mylabel} = "scontainerlist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "externalname", "flowdirection", "rollbackspace", "xcreate",
				 "externaltype", "externalschema", "connector_name", "connector_externalname", "driverid", "protocolid" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT o1.id, o1.type, o1.name, o1.parent_id, o1.created, o1.modified, o1.deleted, o1.comments,
		o1.status, o1.createdby, o1.modifiedby, c1.object_id, c1.externalname, c1.flowdirection, c1.rollbackspace, c1.xcreate,
		c1.externaltype, c1.externalschema, o2.name as connector_name, c2.externalname AS connector_externalname, c2.driverid, c2.protocolid
		FROM objects o1, container c1, objects o2, connector c2
		where o1.id = c1.object_id and o1.type = 'container'
		and c2.object_id = o1.parent_id
		and c2.object_id = o2.id";

	if ($p->{object_id}) {
		$q .= " and c1.object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((o1.name like ?) or (c1.externalname like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and o1.status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	if ($p->{parent_id} && ($p->{parent_id} ne '%')) {
		$q .= " and o1.parent_id = ?";
		push @{$p->{_attr}{bindvars}}, $p->{parent_id};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", o1.id" }
	else { $p->{_attr}{query} = "$q order by o1.id"; }
}

sub icontainer {
	my $p = shift;

	$p->{_attr}{mylabel} = "icontainer";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id externalname flowdirection rollbackspace xcreate externaltype externalschema));
	$p->{_attr}{query} = " insert into container
			(object_id, externalname, flowdirection, rollbackspace, xcreate, externaltype, externalschema)
			values (?, ?, ?, ?, ?, ?, ?);";
}

sub ucontainer {
	my $p = shift;

	$p->{_attr}{mylabel} = "ucontainer";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(externalname flowdirection rollbackspace xcreate externaltype externalschema object_id ));

	$p->{_attr}{query} = "update container set externalname = ?, flowdirection = ?, rollbackspace = ?, xcreate = ?,
			externaltype = ?, externalschema = ?
			where object_id = ?";
}

sub dcontainer {

	my $p = shift;

	$p->{_attr}{mylabel} = "dcontainer";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from container where object_id = ?";
}

# Users queries

sub suserslist {
	my $p = shift; # self

# SELECT "object_id", "username", "password" FROM "users"

	$p->{_attr}{mylabel} = "suserslist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "username", "password" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT id, type, name, parent_id, created, modified, deleted, comments,
		status, createdby, modifiedby, object_id, username, password
		FROM objects, users
		where id = object_id and type = 'users'";
	if ($p->{object_id}) {
		$q .= " and object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((name like ?) or (username like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", id" }
	else { $p->{_attr}{query} = "$q order by id"; }
}

sub iusers {
	my $p = shift;

	$p->{_attr}{mylabel} = "iusers";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id username password));
	$p->{_attr}{query} = " insert into users
			(object_id, username, password)
			values (?, ?, ?);";
}

sub uusers {
	my $p = shift;

	$p->{_attr}{mylabel} = "uusers";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(username password object_id ));

	$p->{_attr}{query} = "update users set username = ?, password = ? where object_id = ?";
}

sub dusers {

	my $p = shift;

	$p->{_attr}{mylabel} = "dusers";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from users where object_id = ?";
}


#Subflow queries

sub ssubflowlist {
	my $p = shift; # self

# SELECT "object_id", "flow_id", "container_id", "floworder", "flowdirection", "flowmethod", FROM "subflow"

	$p->{_attr}{mylabel} = "ssubflowlist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_label", "object_id", "floworder", "flowmethod", "flowdirection",
				 "onoutgoerr", "keepalive", "container_id", "container_name" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT o1.id, o1.type, o1.name, o1.parent_id, o1.created, o1.modified, o1.deleted,
		o1.comments, o1.status, o1.createdby, o1.modifiedby, o1.objectlabel, sf.object_id, sf.flow_id, sf.floworder,
		sf.flowdirection, sf.flowmethod, sf.onoutgoerr, keepalive, sf.container_id,o2.name as container_name,
		o3.id as flow_id, o3.name as flow_name
		FROM objects o1, subflow sf, objects o2, objects o3
		where o1.id = sf.object_id and o1.type = 'subflow'
		and sf.container_id = o2.id
		and o1.parent_id = sf.flow_id
		and o3.id = o1.parent_id";
	if ($p->{object_id}) {
		$q .= " and sf.object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{parent_id}) {
		$q .= " and o1.parent_id ";
		if (ref($p->{parent_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{parent_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{parent_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and o1.name like ? ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{flowdirection} && ($p->{flowdirection} ne '%')) {
		$q .= " and sf.flowdirection like ? ";
		push @{$p->{_attr}{bindvars}}, $p->{flowdirection};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and o1.status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	else { $q .= " and o1.status not like 'deleted'"; }

	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", o1.id" }
	else { $p->{_attr}{query} = "$q order by o1.parent_id, sf.floworder"; }
}

sub isubflow {
	my $p = shift;

	$p->{_attr}{mylabel} = "isubflow";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id flow_id container_id floworder flowdirection flowmethod onoutgoerr keepalive));
	$p->{_attr}{query} = " insert into subflow
			(object_id, flow_id, container_id, floworder, flowdirection, flowmethod, onoutgoerr, keepalive)
			values (?, ?, ?, ?, ?, ?, ?, ?)";
}

sub usubflow {
	my $p = shift;

	$p->{_attr}{mylabel} = "usubflow";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(flow_id container_id floworder flowdirection flowmethod onoutgoerr keepalive object_id));

	$p->{_attr}{query} = "update subflow set flow_id = ?, container_id = ?, floworder = ?,
				flowdirection = ?, flowmethod = ?, onoutgoerr = ?, keepalive = ? where object_id = ?";
}

sub dsubflow {

	my $p = shift;

	$p->{_attr}{mylabel} = "dsubflow";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from subflow where object_id = ?";
}


#Flow queries 


sub sflowlist {
	my $p = shift; # self

# SELECT "object_id", "bufferedflow", "incomrel" FROM "flow"

	$p->{_attr}{mylabel} = "sflowlist";
	$p->{_attr}{columns} = [ "id", "type", "name", "objectlabel", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "object_id", "bufferedflow", "incomrel", "commitinterv", "flowtype" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT id, type, name, objectlabel, parent_id, created, modified, deleted, comments,
		status, createdby, modifiedby, object_id, bufferedflow, incomrel, commitinterv, flowtype
		FROM objects, flow
		where id = object_id and type = 'flow'";
	if ($p->{object_id}) {
		$q .= " and object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((name like ?) or (objectlabel like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	else { $q .= " and status not like 'deleted'"; }

	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", id" }
	else { $p->{_attr}{query} = "$q order by id"; }
}

sub iflow {
	my $p = shift;

	$p->{_attr}{mylabel} = "iflow";
	$p->{flowtype} = 'simple' unless ($p->{flowtype});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id bufferedflow incomrel commitinterv flowtype));
	$p->{_attr}{query} = " insert into flow
			(object_id, bufferedflow, incomrel, commitinterv, flowtype)
			values (?, ?, ?, ?, ?);";
}

sub uflow {
	my $p = shift;

	$p->{_attr}{mylabel} = "uflow";
	$p->{flowtype} = 'simple' unless ($p->{flowtype});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(bufferedflow incomrel commitinterv flowtype object_id));

	$p->{_attr}{query} = "update flow set bufferedflow = ?, incomrel = ?, commitinterv = ?, flowtype = ? where object_id = ?";
}

sub dflow {

	my $p = shift;

	$p->{_attr}{mylabel} = "dflow";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from flow where object_id = ?";
}

#PScripts queries (parsed scripts)

sub spscriptslist {
	my $p = shift; # self

# SELECT "object_id", "pscriptstype", "pscriptsmethod" FROM "pscripts"

	$p->{_attr}{mylabel} = "spscriptslist";
	$p->{_attr}{columns} = ["object_id", "parsetext" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT object_id, parsetext FROM pscripts";
	if ($p->{object_id}) {
		$q .= " where object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	$p->query($q);
}

sub ipscripts {
	my $p = shift;

	$p->{_attr}{mylabel} = "ipscripts";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id parsetext));
	$p->{_attr}{query} = " insert into pscripts (object_id, parsetext) values (?, ?);";
}

sub upscripts {
	my $p = shift;

	$p->{_attr}{mylabel} = "upscripts";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(parsetext object_id ));

	$p->{_attr}{query} = "update pscripts set parsetext = ? where object_id = ?";
}

sub dpscripts {

	my $p = shift;

	$p->{_attr}{mylabel} = "dpscripts";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from pscripts where object_id = ?";
}

# Other Objects object relationship (parent-child)

sub sotherobject {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "sotherobject";
	$p->{_attr}{columns} = [ "parent_id", "child_id", "name", "objectlabel", "status", "type" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT oth.parent_id, oth.child_id, o.name, o.objectlabel, o.status, o.type from otherobjects oth, objects o
				where o.id = oth.child_id";

	if ($p->{parent_id}) { $q .= " and oth.parent_id = ?"; push @{$p->{_attr}{bindvars}}, $p->{parent_id}; }
	if ($p->{type} and ($p->{type} ne "%")) { $q .= " and o.type like ?"; push @{$p->{_attr}{bindvars}}, $p->{type}; }
	$p->query($q);
}

sub iotherobject {
	my $p = shift;

	$p->{_attr}{mylabel} = "iotherobject";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(parent_id child_id));

	$p->{_attr}{query} = " insert into otherobjects (parent_id, child_id) values (?, ?);";
}

sub dotherobject {
	my $p = shift;

	$p->{_attr}{mylabel} = "dotherobject";
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = " delete from otherobjects";

	my $w = " WHERE ";
	if ($p->{parent_id}) { $p->{_attr}{query} .= " $w parent_id = ?"; push @{$p->{_attr}{bindvars}}, $p->{parent_id}; $w = " AND "; }
	if ($p->{child_id}) { $p->{_attr}{query} .= " $w child_id = ?"; push @{$p->{_attr}{bindvars}}, $p->{child_id}; $w = " AND "; }
}
 
#Object's scripts queries

sub sscriptobject {
	my $p = shift; # self

# SELECT "object_id", "scriptstype", "scriptsmethod" FROM "scripts"

	$p->{_attr}{mylabel} = "sscriptobject";
	$p->{_attr}{columns} = [ "object_id", "script_id", "name", "step", "usedfor", "status" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT s.object_id, s.script_id, o.name, s.step, s.usedfor, o.status from scriptobject s, objects o
				where o.id = s.script_id";

	if ($p->{object_id}) { $q .= " and s.object_id = ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	if ($p->{script_id}) { $q .= " and s.script_id = ?"; push @{$p->{_attr}{bindvars}}, $p->{script_id}; }
	if ($p->{step}) { $q .= " and s.step like ?"; push @{$p->{_attr}{bindvars}}, $p->{step}; }
	$p->query($q);
}

sub iscriptobject {
	my $p = shift;

	$p->{_attr}{mylabel} = "iscriptobject";
	$p->{parsed} = 'no' unless ($p->{parsed});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id script_id step usedfor));
	$p->{_attr}{query} = " insert into scriptobject
			(object_id, script_id, step, usedfor)
			values (?, ?, ?, ?);";
}

sub uscriptobject_nonono {
	my $p = shift;

	$p->{_attr}{mylabel} = "uscriptobject";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(scripttype parsed scripttext object_id ));

	$p->{_attr}{query} = "update scripts set scripttype = ?, parsed = ?, scripttext = ? where object_id = ?";
}

sub dscriptobject {

	my $p = shift;

	$p->{_attr}{mylabel} = "dscriptobject";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from scriptobject where object_id = ?";
}

	
#Scripts queries 


sub sscriptslist {
	my $p = shift; # self

# SELECT "object_id", "scriptstype", "scriptsmethod" FROM "scripts"

	$p->{_attr}{mylabel} = "sscriptslist";
	$p->{_attr}{columns} = [ "id", "type", "name", "parent_id", "created", "modified", "deleted", "comments",
				 "status", "createdby", "modifiedby", "objectlabel", "object_id", "scripttype", "parsed", "scripttext" ];

	$p->{_attr}{bindvars} = [];

	my $q = " SELECT id, type, name, parent_id, created, modified, deleted, comments,
		status, createdby, modifiedby, objectlabel, object_id, scripttype, parsed, scripttext
		FROM objects, scripts
		where id = object_id and type in ('sql', 'perl')";
	if ($p->{object_id}) {
		$q .= " and object_id ";
		if (ref($p->{object_id}) eq "ARRAY") { $q .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $q .= "= ?"; push @{$p->{_attr}{bindvars}}, $p->{object_id}; }
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= " and ((name like ?) or (objectlabel like ?)) ";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
	}
	if ($p->{type} && ($p->{type} ne '%')) {
		$q .= " and (type like ?) ";
		push @{$p->{_attr}{bindvars}}, $p->{type};
	}
	if ($p->{scripttype} && ($p->{scripttype} ne '%')) {
		$q .= " and (scripttype like ?) ";
		push @{$p->{_attr}{bindvars}}, $p->{scripttype};
	}
	if ($p->{status} && ($p->{status} ne '%')) {
		$q .= " and status like ?";
		push @{$p->{_attr}{bindvars}}, $p->{status};
	}
	else { $q .= " and status not like 'deleted'"; }

	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() . ", id" }
	else { $p->{_attr}{query} = "$q order by id"; }
}

sub iscripts {
	my $p = shift;

	$p->{_attr}{mylabel} = "iscripts";
	$p->{parsed} = 'no' unless ($p->{parsed});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id scripttype parsed scripttext));
	$p->{_attr}{query} = " insert into scripts
			(object_id, scripttype, parsed, scripttext)
			values (?, ?, ?, ?);";
}

sub uscripts {
	my $p = shift;

	$p->{_attr}{mylabel} = "uscripts";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(scripttype parsed scripttext object_id ));

	$p->{_attr}{query} = "update scripts set scripttype = ?, parsed = ?, scripttext = ? where object_id = ?";
}

sub dscripts {

	my $p = shift;

	$p->{_attr}{mylabel} = "dscripts";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id));

	$p->{_attr}{query} = "delete from scripts where object_id = ?";
}

# objectobjects queries

sub swobjectobjects {
	my $p = shift;

	$p->{_attr}{mylabel} = "swobjectobjects";
	$p->{_attr}{columns} = [ "owner_id", "owner_type", "owner_name", "owned_id", "owned_type", "owned_name" ];
	$p->{_attr}{bindvars} = [];

	my $q = "SELECT oo.owner_id, owr.type as owner_type, owr.name as owner_name,
				oo.owned_id, od.type as owned_type, od.name as owned_name
				FROM wobjectobjects oo, objects owr, objects od
				WHERE oo.owner_id = owr.id and oo.owned_id = od.id";

	if ($p->{owner_id}) {
		$q .= " AND oo.owner_id = $p->{owner_id}";
	}
	if ($p->{modified} and ($p->{modified} ne "%")) {
		$q .= " AND od.modified > to_timestamp(?, 'DD/MM/YYYY HH24:MI:SS')";
		push @{$p->{_attr}{bindvars}}, $p->{modified};
	}

	$p->{_attr}{query} = $q;
}

sub iwobjectobjects {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwobjectobjects";
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "INSERT INTO wobjectobjects (owner_id, owned_id) VALUES ($p->{owner_id}, $p->{owned_id})";
}

sub dwobjectobjects {
	my $p = shift;

	$p->{_attr}{mylabel} = "dwobjectobjects";
	$p->{_attr}{bindvars} = [];

	my $q = "DELETE FROM wobjectobjects WHERE ";

	if (ref($p->{owner_id}) eq "ARRAY") { $q .= "owner_id in (" . join(",", @{$p->{owner_id}}) . ")"; }
	else { $q .= "owner_id = $p->{owner_id}"; } 

	if ($p->{owned_id}) { $p->{_attr}{query} .= " AND owned_id = $p->{owned_id}"; }

	$p->{_attr}{query} = $q;

}


#deployed objects queries

sub sdeployedtoupdate {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sdeployedtoupdate";
	$p->{_attr}{columns} = [ "object_id", "type", "object_exist", "committed" ];

#	$p->{_attr}{query} = 
	my $q1 = "SELECT d.object_id, o.type, '1'::varchar as object_exist, d.committed from objects o, deployedobjects d
				WHERE o.id = d.object_id and o.modified > d.modified";
	my $q2 = "SELECT o1.id as object_id, o1.type, NULL as object_exist, NULL from objects o1
				where o1.id not in (select d2.object_id from deployedobjects d2";
	if ($p->{contextid} and ($p->{contextid} ne "%")) {
		$q1 .= " AND d.contextid = '$p->{contextid}'";
		$q2 .= " WHERE d2.contextid = '$p->{contextid}'";
	}
	$q2 .= ")";

	if ($p->{object_id}) {
		$q1 .= " AND o.id ";
		$q2 .= " AND o1.id ";
		if (ref($p->{object_id}) eq "ARRAY") {
			my $objlist = join (",", @{$p->{object_id}});
			$q1 .= "in ($objlist)";
			$q2 .= "in ($objlist)";
		}
		else { $q1 .= " = $p->{object_id}"; $q2 .= " = $p->{object_id}"; }
	}
	$p->{_attr}{query} = "$q1 UNION $q2 ORDER BY 1";
}

sub sdeployeduptodate {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "supdeployedtodate";
	$p->{_attr}{columns} = [ "object_id" ];

	$p->{_attr}{query} = "SELECT d.object_id, d.committed from objects o, deployedobjects d
				WHERE o.id = d.object_id and o.modified < d.modified";
	if ($p->{contextid} and ($p->{contextid} ne "%")) {
		$p->{_attr}{query} .= " AND d.contextid = '$p->{contextid}'";
	}
	if ($p->{object_id}) {
		$p->{_attr}{query} .= " AND o.id ";
		if (ref($p->{object_id}) eq "ARRAY") { $p->{_attr}{query} .= "in (" . join (",", @{$p->{object_id}}) . ")"; }
		else { $p->{_attr}{query} .= " = $p->{object_id}"; }
	}
	$p->{_attr}{query} .= " ORDER BY 1";
	$p->{_attr}{query};
}

sub sdeployedobjects {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sdeployedobjects";
	$p->{_attr}{columns} = [ "contextid", "contextlabel", "object_id", "object_name", "objectlabel",
				"object_type", "created", "modified", "committed", "parent_id" ];

	$p->{_attr}{bindvars} = [];

	my $q = "SELECT c.contextid, c.contextlabel, d.object_id, o.name as object_name, o.objectlabel, o.type as object_type,
		to_char(d.created, 'DD/MM/YYYY HH24/MI:SS') as created,
		to_char(d.modified, 'DD/MM/YYYY HH24:MI:SS') as modified, committed,
                CASE WHEN (parent_id <= 0) THEN object_id ELSE parent_id END AS parent_id
		FROM deployedobjects d, objects o, context c
		WHERE d.object_id = o.id AND c.contextid = d.contextid";

	if ($p->{object_id}) {
		$q .= " AND d.object_id = ?";
		push @{$p->{_attr}{bindvars}}, $p->{object_id};
	}
	else { $p->{contextid} = "." unless ($p->{contextid}); }

	if ($p->{contextid}) {
		$q .= " AND d.contextid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{contextid};
	}
	if ($p->{object_type}) {
		if (ref($p->{object_type}) eq "ARRAY") {
			$q .= " and o.type in ('" . join("','",@{$p->{object_type}}) . "')";
		}
		else {
			$q .= " and o.type like ?";
			push (@{$p->{_attr}{bindvars}}, $p->{object_type});
		}
	}
	$p->{_attr}{query} = "$q ORDER BY parent_id, object_id";
}

sub sdeployedobjectscount {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sdeployedobjectscount";
	$p->{_attr}{columns} = [ "objectscount" ];

	$p->{_attr}{bindvars} = [];

	my $q = "SELECT count(*) AS objectscount FROM deployedobjects d, objects o, context c
		WHERE d.object_id = o.id AND c.contextid = d.contextid";

	if ($p->{object_id}) {
		$q .= " AND d.object_id = ?";
		push @{$p->{_attr}{bindvars}}, $p->{object_id};
	}
	else { $p->{contextid} = "." unless ($p->{contextid}); }

	if ($p->{contextid}) {
		$q .= " AND d.contextid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{contextid};
	}
	if ($p->{object_type}) {
		if (ref($p->{object_type}) eq "ARRAY") {
			$q .= " and o.type in ('" . join("','",@{$p->{object_type}}) . "')";
		}
		else {
			$q .= " and o.type like ?";
			push (@{$p->{_attr}{bindvars}}, $p->{object_type});
		}
	}
	$p->{_attr}{query} = "$q"; # ORDER BY parent_id, object_id";
}

sub ideployedobjects {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "ideployedobjects";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(contextid object_id committed));
	$p->{_attr}{query} = " insert into deployedobjects (contextid, object_id, created, modified, committed)
			values (?, ?, now(), now(), ?);";
}

sub udeployedobjects {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "udeployedobjects";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(committed contextid object_id));
	$p->{_attr}{query} = " update deployedobjects set modified = now(), committed = ? where contextid = ? and object_id = ?";
}

sub ddeployedobjects () {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "udeployedobjects";
	$p->{_attr}{bindvars} = [];

	my $q = "DELETE FROM deployedobjects WHERE contextid = '$p->{contextid}' AND ";

	if (ref($p->{object_id}) eq "ARRAY") { $q .= "object_id in (" . join (",", @{$p->{object_id}}) . ")"; }
	else { $q .= "object_id = $p->{object_id}"; }

}
# Context queries

sub scontextlist {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "scontextlist";
	$p->{_attr}{columns} = [ "contextid", "contextlabel", "connector_id", "workdir", "programdir", "ftpconnector_id",
				"mailconnector_id", "pop3connector_id", "adminmail", "archconnector_id", "archdelay" ];

	$p->{_attr}{bindvars} = [];

	my $q = "  SELECT contextid, contextlabel, connector_id, workdir, programdir, ftpconnector_id,
		mailconnector_id, pop3connector_id, adminmail, archconnector_id, archdelay
		FROM context";
	my $where = " where ";
	if ($p->{contextid} && ($p->{contextid} ne '%')) {
		$q .= " where contextid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{contextid};
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() }
	else { $p->{_attr}{query} = "$q order by contextid"; }
}

sub icontext {
	my $p = shift;

	$p->{_attr}{mylabel} = "icontext";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(contextid contextlabel connector_id workdir programdir ftpconnector_id mailconnector_id pop3connector_id adminmail archconnector_id archdelay));
	$p->{_attr}{query} = " insert into context ( contextid,contextlabel,connector_id,workdir,programdir,ftpconnector_id,mailconnector_id,pop3connector_id,adminmail,archconnector_id, archdelay)
			values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
}

sub ucontext {
	my $p = shift;

	$p->{_attr}{mylabel} = "ucontext";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(contextlabel connector_id workdir programdir ftpconnector_id mailconnector_id pop3connector_id adminmail archconnector_id archdelay contextid));

	$p->{_attr}{query} = "update context set contextlabel = ?, connector_id = ?, workdir = ?, programdir = ?,
			ftpconnector_id = ?, mailconnector_id = ?, pop3connector_id = ?, adminmail = ?, archconnector_id = ?,
			archdelay = ?
			where contextid = ?";
}

sub dcontext {

	my $p = shift;

	$p->{_attr}{mylabel} = "dcontext";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(contextid));

	$p->{_attr}{query} = "delete from context where contextid = ?";
}

#Events queries

sub seventslist {
	my $p = shift;
#"object_id", "eventid", "genorwait", "object_id_source", "waitlast", "comments",   
	$p->{_attr}{mylabel} = "seventslist";
	$p->{_attr}{columns} = [ "object_id", "name", "type", "eventid", "genorwait", "object_id_source", "waitlast", "comments" ];

	$p->{_attr}{bindvars} = [];

	my (@bv1, @bv2);
	my $q1 =  " SELECT oe.object_id, o.name, o.type, oe.eventid, oe.genorwait, oe.object_id_source, oe.waitlast, oe.expiredelay, oe.comments,
				NULL as object_name
			from objectevents oe, objects o
			where oe.object_id = o.id and oe.genorwait = 'generate'";

	my $q2 =  " SELECT oe.object_id, o.name, o.type, oe.eventid, oe.genorwait, oe.object_id_source, oe.waitlast, oe.expiredelay, oe.comments,
				o2.name as source_name
			from objectevents oe, objects o, objects o2
			where oe.object_id = o.id
			and oe.genorwait = 'wait'
			and oe.object_id_source = o2.id";

	if ($p->{object_id}) {
      if (ref($p->{object_id}) eq "ARRAY") {
			 $q1 .= " and oe.object_id in (" . join (",", @{$p->{object_id}}) . ")";
			 $q2 .= " and oe.object_id in (" . join (",", @{$p->{object_id}}) . ")";
		}
      else {
			$q1 .= " and oe.object_id = ?";
			$q2 .= " and oe.object_id = ?";
			 push @bv1, $p->{object_id};
			 push @bv2, $p->{object_id};
		}
	}
	if ($p->{event_id} and ($p->{event_id} ne '%')) {
		$q1 .= " and oe.eventid like ?";
		$q2 .= " and oe.eventid like ?";
		 push @bv1, $p->{event_id};
		 push @bv2, $p->{event_id};
	}
	if ($p->{genorwait}) {
		if ($p->{genorwait} eq 'generate') {
			$p->{_attr}{query} = $q1;
			push @{$p->{_attr}{bindvars}}, @bv1 if (@bv1 and ($#bv1 > -1));
		}
		else {
			$p->{_attr}{query} = $q2;
			push @{$p->{_attr}{bindvars}}, @bv2 if (@bv2 and ($#bv2 > -1));
		}
	}
	else {
		$p->{_attr}{query} = "$q1 UNION $q2";
		push @{$p->{_attr}{bindvars}}, (@bv1, @bv2) if (@bv1 and ($#bv1 > -1));
	}

	$p->{_attr}{query};
}

sub ievents {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "ievents";
	$p->{object_id_source} = 0 unless ($p->{object_id_source});
	$p->{waitlast} = "yes" unless ($p->{waitlast});
	$p->{expiredelay} = 1 unless ($p->{expiredelay});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id eventid genorwait object_id_source comments waitlast expiredelay));
# [$p->{object_id}, $p->{eventid}, $p->{genorwait}, $p->{object_id_source}, $p->{waitlast}, $p->{expiredelay} ];

	$p->{_attr}{query} = " INSERT into objectevents (object_id, eventid, genorwait, object_id_source, comments, waitlast, expiredelay)
				values (?, ?, ?, ?, ?, ?, ?)";
}

sub devents {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "devents";
	$p->{_attr}{bindvars} = [$p->{object_id} ];

	$p->{_attr}{query} = " DELETE from objectevents where object_id = ?";
}
	

# Attributes values

sub sgetmyenvvars {
	my $p = shift;

	$p->{_attr}{mylabel} = "sgetmyenvvars";
	$p->{_attr}{columns} = [ "varname", "varvalue", "vartype" ];

	$p->{_attr}{bindvars} = [$p->{object_id}];

	$p->{_attr}{query} = "  SELECT varname, varvalue, vartype from envvars where object_id = ?";

}

sub ienvvars {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "ienvvars";
	$p->{vartype} = 'env' unless ($p->{vartype});
	$p->{_attr}{bindvars} = [$p->{object_id}, $p->{varname}, $p->{varvalue}, $p->{vartype} ];

	$p->{_attr}{query} = " INSERT into envvars (object_id, varname, varvalue, vartype) values (?, ?, ?, ?)";
}

sub uenvvars {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "uenvvars";
	$p->{vartype} = 'env' unless ($p->{vartype});
	$p->{_attr}{bindvars} = [$p->{varvalue}, $p->{object_id}, $p->{varname}, $p->{vartype} ];

	$p->{_attr}{query} = " UPDATE envvars set varvalue = ? where object_id = ? and varname = ? and vartype = ?";
}

sub denvvars {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "denvvars";
	$p->{_attr}{bindvars} = [$p->{object_id}];

	$p->{_attr}{query} = " DELETE FROM envvars where object_id = ?";

	if ($p->{varname} && ($p->{varname} ne '%')) {
		$p->{_attr}{query} .= " and varname like ?";
		push @{$p->{_attr}{bindvars}}, $p->{varname};
	}
	if ($p->{vartype} && ($p->{vartype} ne '%')) {
		$p->{_attr}{query} .= " and vartype like ?";
		push @{$p->{_attr}{bindvars}}, $p->{vartype};
	}
}


# Attributes values

#waddresses queries
sub swaddressmembers {
	my $p = shift;

	$p->{_attr}{mylabel} = "swaddressmembers";
	$p->{_attr}{columns} = [ "addressid", "addresslabel", "addresstype", "email", "faxnumber", "mobilenumber", "printercmd",
		"address_line1", "address_line2", "address_pc", "address_city", "address_status", "modified", "created", "groupid", "memebr_status" ];
	$p->{_attr}{bindvars} = [$p->{addressid}];

	$p->{_attr}{query} = "SELECT waddresses.addressid, addresslabel, addresstype, email, faxnumber, mobilenumber, printercmd,
		address_line1, address_line2, address_pc, address_city, address_status, modified, created, groupid, member_status
		FROM waddresses, waddresslist
		WHERE waddresses.addressid = waddresslist.addressid
		AND waddresslist.groupid like ?";
}
		
sub swaddressgroups {
	my $p = shift;

	$p->{_attr}{mylabel} = "swaddressgroups";
	$p->{_attr}{columns} = [ "addressid", "addresslabel", "addresstype", "email", "faxnumber", "mobilenumber", "printercmd",
		"address_line1", "address_line2", "address_pc", "address_city", "address_status", "modified", "created", "groupid", "memebr_status" ];
	$p->{_attr}{bindvars} = [$p->{addressid}];

	$p->{_attr}{query} = "SELECT waddresses.addressid, addresslabel, addresstype, email, faxnumber, mobilenumber, printercmd,
		address_line1, address_line2, address_pc, address_city, address_status, modified, created, groupid, member_status
		FROM waddresses, waddresslist
		WHERE waddresses.addressid = waddresslist.groupid
		AND waddresslist.addressid like ?";
}
		
sub swaddresseslist {
	my $p = shift;

	$p->{_attr}{mylabel} = "swaddresseslist";
	$p->{_attr}{columns} = [ "addressid", "addresslabel", "addresstype", "email", "faxnumber", "mobilenumber", "printercmd",
		"address_line1", "address_line2", "address_pc", "address_city", "address_status", "modified", "created" ];
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "SELECT addressid, addresslabel, addresstype, email, faxnumber, mobilenumber, printercmd,
		address_line1, address_line2, address_pc, address_city, address_status, modified, created FROM waddresses";

	my $w = " WHERE";
	if ($p->{addressid}) {
		$p->{_attr}{query} .= "$w addressid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{addressid};
		$w = " AND";
	}
# The members of the group idenfied by group id
	if ($p->{groupid}) {
		$p->{_attr}{query} .= "$w addressid IN (SELECT addressid from waddresslist WHERE groupid like ?)";
		push @{$p->{_attr}{bindvars}}, $p->{groupid};
		$w = " AND";
	}
# The groups of this address addressid
	if ($p->{member_of_group}) {
		$p->{_attr}{query} .= "$w addressid IN (SELECT groupid from waddresslist WHERE addressid like ?)";
		push @{$p->{_attr}{bindvars}}, $p->{addressid};
		$w = " AND";
	}
	if ($p->{address_status}) {
		$p->{_attr}{query} .= "$w address_status LIKE ?";
		push @{$p->{_attr}{bindvars}}, $p->{address_status};
		$w = " AND";
	}

	if ($p->{addresstype}) {
		$p->{_attr}{query} .= "$w addresstype LIKE ?";
		push @{$p->{_attr}{bindvars}}, $p->{addresstype};
		$w = " AND";
	}


#	print "<BR> $p->{_attr}{query}";
	$p->{_attr}{query};
}

sub iwaddresses {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "iwaddresses";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(addressid addresslabel addresstype email faxnumber mobilenumber printercmd
      address_line1 address_line2 address_pc address_city address_status));

	$p->{_attr}{query} = " INSERT into waddresses (addressid, addresslabel, addresstype, email, faxnumber, mobilenumber, printercmd,
      address_line1, address_line2, address_pc, address_city, address_status, modified, created)
		values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, now(), now())";
}

sub uwaddresses {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "uwaddresses";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(addresslabel addresstype email faxnumber mobilenumber printercmd
      address_line1 address_line2 address_pc address_city address_status addressid));

	$p->{_attr}{query} = " UPDATE waddresses SET addresslabel = ?, addresstype = ?, email = ?, faxnumber = ?, mobilenumber = ?, printercmd = ?,
		address_line1 = ?, address_line2 = ?, address_pc = ?, address_city = ?, address_status = ?, modified = now()
		WHERE addressid = ?";
}

sub dwaddresses {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "dwaddresses";
	$p->{_attr}{bindvars} = [$p->{addressid}];

	$p->{_attr}{query} = " DELETE FROM waddresses where addressid = ?";
}

sub iwaddressesgroup {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwaddressesgroup";
	$p->{_attr}{bindvars} = [$p->{addressid}, $p->{groupid}, $p->{member_status}];

	$p->{_attr}{query} = " INSERT INTO waddresslist (addressid, groupid, member_status) values (?, ?, ?)";
}

sub dwaddressesgroup {
	my $p = shift;

	$p->{_attr}{mylabel} = "dwaddressesgroup";
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = " DELETE FROM waddresslist";
	my $w = " WHERE";

	if ($p->{groupid}) { $p->{_attr}{query} = "$w groupid = ?"; $w = " AND"; push @{$p->{_attr}{bindvars}}, $p->{groupid}; }
	if ($p->{addressid}) { $p->{_attr}{query} = "$w addressid = ?"; $w = " AND"; push @{$p->{_attr}{bindvars}}, $p->{addressid}; }

}

sub dwaddressesbyattrdef {
	my $p = shift;

	$p->{_attr}{mylabel} = "dattribute";
	$p->{_attr}{bindvars} = [$p->{attrdefid}];

	$p->{_attr}{query} = " DELETE FROM waddresses where attrdefid like ?";
}
	
# Attributs definition
sub sgetmyattributes {
	my $p = shift;

	$p->{_attr}{mylabel} = "sgetmyattributes";
	$p->{_attr}{columns} = [ "attrdefid", "attrvalue" ];

	$p->{_attr}{bindvars} = [$p->{object_id}];

	$p->{_attr}{query} = "  SELECT attrdefid, attrvalue from attributes where object_id = ?";

}

sub iattributes {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "iattribute";
	$p->{_attr}{bindvars} = [$p->{object_id}, $p->{attrdefid}, $p->{attrvalue}];

	$p->{_attr}{query} = " INSERT into attributes (object_id, attrdefid, attrvalue) values (?, ?, ?)";
}

sub uattributes {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "uattribute";
	$p->{_attr}{bindvars} = [$p->{attrvalue}, $p->{object_id}, $p->{attrdefid}];

	$p->{_attr}{query} = " UPDATE attributes set attrvalue = ? where object_id = ? and attrdefid = ?";
}

sub dattributes {
	my $p = shift;
	
	$p->{_attr}{mylabel} = "dattribute";
	$p->{_attr}{bindvars} = [$p->{object_id}];

	$p->{_attr}{query} = " DELETE FROM attributes where object_id = ?";

	if ($p->{attrdefid} && ($p->{attrdefid} ne '%')) {
		$p->{_attr}{query} .= " and attrdefid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{attrdefid};
	}
}

sub dattributesbyattrdef {
	my $p = shift;

	$p->{_attr}{mylabel} = "dattribute";
	$p->{_attr}{bindvars} = [$p->{attrdefid}];

	$p->{_attr}{query} = " DELETE FROM attributes where attrdefid like ?";
}
	
# Attributs definition

sub sattrdeflistprotdriv {
	my $p = shift; # self

#SELECT "attrdefid", "protocolid", "driverid", "objtype", "datatype", "attrdeflabel", "attrdefault" FROM "attrdef"


	$p->{_attr}{mylabel} = "sattrdeflist";
	$p->{_attr}{columns} = [ "attrdefid", "protocolid", "driverid", "objtype", "datatype", "attrdeflabel", "attrdefault" ];

	$p->{_attr}{bindvars} = [ $p->{protocolid}, $p->{driverid} ];

#	my $q = "  SELECT attrdefid, protocolid, driverid, objtype, datatype, attrdeflabel, attrdefault FROM attrdef
#		WHERE (protocolid = ? and driverid = ?) or (protocolid = ? and driverid = 'all') or
#			(protocolid = 'all' and driverid = ?) or (protocolid = 'all' and driverid = 'all') ";
	my $q = "  SELECT attrdefid, protocolid, driverid, objtype, datatype, attrdeflabel, attrdefault FROM attrdef
		WHERE (protocolid = ? or protocolid = 'all') and (driverid = ? or driverid = 'all')";
	my $where = " and ";
	if ($p->{attrobjtype} && ($p->{attrobjtype} ne '%')) {
		$q .= "$where objtype like ?";
		push @{$p->{_attr}{bindvars}}, $p->{attrobjtype};
		$where = " and ";
	}
	elsif ($p->{objtype} && ($p->{objtype} ne '%')) {
		$q .= "$where objtype like ?";
		push @{$p->{_attr}{bindvars}}, $p->{objtype};
		$where = " and ";
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() }
	else { $p->{_attr}{query} = "$q order by attrdefid"; }
}

sub sattrdeflist {
	my $p = shift; # self

#SELECT "attrdefid", "protocolid", "driverid", "objtype", "datatype", "attrdeflabel", "attrdefault" FROM "attrdef"


	$p->{_attr}{mylabel} = "sattrdeflist";
	$p->{_attr}{columns} = [ "attrdefid", "protocolid", "driverid", "objtype", "datatype", "attrdeflabel", "attrdefault" ];

	$p->{_attr}{bindvars} = [];

	my $q = "  SELECT attrdefid, protocolid, driverid, objtype, datatype, attrdeflabel, attrdefault
		FROM attrdef";
	my $where = " where ";
	if ($p->{attrdefid} && ($p->{attrdefid} ne '%')) {
		$q .= "$where attrdefid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{attrdefid};
		$where = " and ";
	}
	if ($p->{protocolid} && ($p->{protocolid} ne '%')) {
		$q .= "$where protocolid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{protocolid};
		$where = " and ";
	}
	if ($p->{driverid} && ($p->{driverid} ne '%')) {
		$q .= "$where driverid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{driverid};
		$where = " and ";
	}
	if ($p->{attrobjtype} && ($p->{attrobjtype} ne '%')) {
		$q .= "$where objtype like ?";
		push @{$p->{_attr}{bindvars}}, $p->{attrobjtype};
		$where = " and ";
	}
	elsif ($p->{objtype} && ($p->{objtype} ne '%')) {
		$q .= "$where objtype like ?";
		push @{$p->{_attr}{bindvars}}, $p->{objtype};
		$where = " and ";
	}
	if ($p->{attrdeflabel} && ($p->{attrdeflabel} ne '%')) {
		$q .= "$where attrdeflabel like ?";
		push @{$p->{_attr}{bindvars}}, $p->{attrdeflabel};
		$where = " and ";
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= "$where ((attrdefid like ?) or (attrdeflabel like ?))";
		push @{$p->{_attr}{bindvars}}, $p->{objtype};
		push @{$p->{_attr}{bindvars}}, $p->{objtype};
		$where = " and ";
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() }
	else { $p->{_attr}{query} = "$q order by attrdefid"; }
}

sub iattrdef {
	my $p = shift;

	$p->{_attr}{mylabel} = "iattrdef";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(attrdefid protocolid driverid objtype datatype attrdeflabel attrdefault));
	$p->{_attr}{query} = " insert into attrdef (attrdefid, protocolid, driverid, objtype, datatype, attrdeflabel, attrdefault)
			values (?, ?, ?, ?, ?, ?, ?);";
}

sub uattrdef {
	my $p = shift;

	$p->{_attr}{mylabel} = "uattrdef";
	if (! $p->{attrdefault} and (length ($p->{attrdefault}) == 0)) { $p->{attrdefault} = "--nodefault--"; }
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(protocolid driverid objtype datatype attrdeflabel attrdefault attrdefid));

	$p->{_attr}{query} = "update attrdef set protocolid = ?, driverid = ?, objtype = ?, datatype = ?, attrdeflabel = ?, attrdefault = ?
			where attrdefid = ?";
}

sub dattrdef {

	my $p = shift;

	$p->{_attr}{mylabel} = "dattrdef";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(attrdefid));

	$p->{_attr}{query} = "delete from attrdef where attrdefid = ?";
}

# Datatypes queries


sub sdatatypeliststd {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sdatatypelist";
	$p->{_attr}{columns} = [ "datatypeid", "driverid", "stdname" ];

	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "SELECT datatypeid, driverid, stdname FROM datatype where driverid = '==standardsql==' order by int4(datatypeid) desc";

}

sub sdatatypelist {
	my $p = shift; # self

	$p->{_attr}{mylabel} = "sdatatypelist";
	$p->{_attr}{columns} = [ "datatypeid", "driverid", "stdname" ];

	$p->{_attr}{bindvars} = [];

	my $q = "  SELECT datatypeid, driverid, stdname FROM datatype where driverid != '==standardsql=='";
	my $where = " and ";
	if ($p->{datatypeid} && ($p->{datatypeid} ne '%')) {
		$q .= "$where datatypeid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{datatypeid};
		$where = " and ";
	}
	if ($p->{stdname} && ($p->{stdname} ne '%')) {
		$q .= "$where stdname like ?";
		push @{$p->{_attr}{bindvars}}, $p->{stdname};
		$where = " and ";
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= "$where ((datatypeid like ?) or (stdname like ?))";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
		$where = " and ";
	}
	if ($p->{driverid} && ($p->{driverid} ne '%')) {
		$q .= "$where driverid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{driverid};
		$where = " and ";
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() }
	else { $p->{_attr}{query} = "$q order by driverid, datatypeid"; }
}

sub idatatype {
	my $p = shift;

	$p->{_attr}{mylabel} = "idatatype";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(datatypeid driverid stdname));
	$p->{_attr}{query} = " insert into datatype (datatypeid, driverid, stdname) values (?, ?, ?);";
}

sub udatatype {
	my $p = shift;

	$p->{_attr}{mylabel} = "udatatype";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(stdname datatypeid driverid));

	$p->{_attr}{query} = "update datatype set stdname = ? where datatypeid = ? and driverid = ?";
}

sub ddatatype {

	my $p = shift;

	$p->{_attr}{mylabel} = "ddatatype";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(datatypeid driverid));

	$p->{_attr}{query} = "delete from datatype where datatypeid = ? and driverid = ?";
}


sub sprotocollist {
	my $p = shift; # self

#SELECT "protocolid", "protocollabel" FROM "protocol"


	$p->{_attr}{mylabel} = "sprotocollist";
	$p->{_attr}{columns} = [ "protocolid", "protocollabel", "inoutdir" ];

	$p->{_attr}{bindvars} = [];

	my $q = "  SELECT protocolid, protocollabel, inoutdir FROM protocol";
	my $where = " where ";
	if ($p->{protocolid} && ($p->{protocolid} ne '%')) {
		$q .= "$where protocolid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{protocolid};
		$where = " and ";
	}
	if ($p->{protocollabel} && ($p->{protocollabel} ne '%')) {
		$q .= "$where protocollabel like ?";
		push @{$p->{_attr}{bindvars}}, $p->{protocollabel};
		$where = " and ";
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= "$where ((protocolid like ?) or (protocollabel like ?))";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
		$where = " and ";
	}
	if ($p->{inoutdir} && ($p->{inoutdir} ne '%')) {
		$q .= "$where inoutdir like ?";
		push @{$p->{_attr}{bindvars}}, $p->{inoutdir};
		$where = " and ";
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() }
	else { $p->{_attr}{query} = "$q order by protocolid"; }
}

sub iprotocol {
	my $p = shift;

	$p->{_attr}{mylabel} = "iprotocol";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(protocolid protocollabel inoutdir));
	$p->{_attr}{query} = " insert into protocol (protocolid, protocollabel, inoutdir) values (?, ?, ?);";
}

sub uprotocol {
	my $p = shift;

	$p->{_attr}{mylabel} = "uprotocol";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(protocollabel inoutdir protocolid));

	$p->{_attr}{query} = "update protocol set protocollabel = ?, inoutdir = ? where protocolid = ?";
}

sub dprotocol {

	my $p = shift;

	$p->{_attr}{mylabel} = "dprotocol";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(protocolid));

	$p->{_attr}{query} = "delete from protocol where protocolid = ?";
}

# Drivers queries

sub sdriverlistuniq {
	my $p = shift; # self

#SELECT "driverid", "driverlabel" FROM "driver"


	$p->{_attr}{mylabel} = "sdriverlistuniq";
	$p->{_attr}{columns} = [ "driverid", "driverlabel", "cantransact" ];

	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "  SELECT distinct driverid, driverlabel, cantransact FROM driver order by driverid";
}

sub sdriverlist {
	my $p = shift; # self

#SELECT "driverid", "driverlabel" FROM "driver"


	$p->{_attr}{mylabel} = "sdriverlist";
	$p->{_attr}{columns} = [ "driverid", "protocolid", "driverlabel", "cantransact" ];

	$p->{_attr}{bindvars} = [];

	my $q = "  SELECT driverid, protocolid, driverlabel, cantransact FROM driver";
	my $where = " where ";
	if ($p->{driverid} && ($p->{driverid} ne '%')) {
		$q .= "$where driverid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{driverid};
		$where = " and ";
	}
	if ($p->{name} && ($p->{name} ne '%')) {
		$q .= "$where ((driverid like ?) or (driverlabel like ?))";
		push @{$p->{_attr}{bindvars}}, $p->{name};
		push @{$p->{_attr}{bindvars}}, $p->{name};
		$where = " and ";
	}
	if ($p->{protocolid} && ($p->{protocolid} ne '%')) {
		$q .= "$where protocolid like ?";
		push @{$p->{_attr}{bindvars}}, $p->{protocolid};
		$where = " and ";
	}
	if ($p->orderby()) { $p->{_attr}{query} = "$q order by " . $p->orderby() }
	else { $p->{_attr}{query} = "$q order by protocolid, driverid"; }
}

sub idriver {
	my $p = shift;

	$p->{_attr}{mylabel} = "idriver";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(driverid protocolid driverlabel cantransact));
	$p->{_attr}{query} = " insert into driver (driverid, protocolid, driverlabel, cantransact) values (?, ?, ?, ?);";
}

sub udriver {
	my $p = shift;

	$p->{_attr}{mylabel} = "udriver";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(driverlabel cantransact driverid protocolid));

	$p->{_attr}{query} = "update driver set driverlabel = ?, cantransact = ? where driverid = ? and protocolid = ?";
}

sub ddriver {

	my $p = shift;

	$p->{_attr}{mylabel} = "ddriver";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(driverid protocolid));

	$p->{_attr}{query} = "delete from driver where driverid = ? and protocolid = ?";
}


# Objects queries

sub sobjectlist {
	shift->sobject(@_);
}

sub sobject {
	my $p = shift; # self
	my $w = " where";

	$p->{_attr}{mylabel} = "sobject";
	$p->{_attr}{columns} = [];
	push @{$p->{_attr}{columns}}, qw (objectid id type name parent_id created modified deleted comments status objectlabel createdby modifiedby);

	my $query = "select id as object_id, id, type, name, parent_id,
			to_char(created, 'DD/MM/YYYY HH24:MI:SS') AS created,
			to_char(modified, 'DD/MM/YYYY HH24:MI:SS') AS modified,
			to_char(deleted, 'DD/MM/YYYY HH24:MI:SS') AS deleted,
			comments, status, objectlabel, createdby, modifiedby
			from objects";

	$p->{_attr}{bindvars} = [];

	$p->{id} = $p->{object_id} unless($p->{id});
	if (($p->{id})) {
		if (ref($p->{id}) eq "ARRAY") { $query .= "$w id in (" . join (",", @{$p->{id}}) . ")"; }
		else { $query .= "$w id = ?"; push @{$p->{_attr}{bindvars}}, $p->{id}; }
		$w = " and ";
	}
	if (($p->{parent_id})) {
		if (ref($p->{parent_id}) eq "ARRAY") { $query .= "$w parent_id in (" . join (",", @{$p->{parent_id}}) . ")"; }
		else { $query .= "$w parent_id = ?"; push @{$p->{_attr}{bindvars}}, $p->{parent_id}; }
		$w = " and ";
	}
	if (($p->{name})) { $query .= "$w name like ?"; $w = " and "; push @{$p->{_attr}{bindvars}}, $p->{name}; }
	if (($p->{namelabel})) {
		$query .= "$w (upper(name) like upper(?) or upper(objectlabel) like upper(?))";
		$w = " and ";
		push @{$p->{_attr}{bindvars}}, $p->{namelabel}, $p->{namelabel};
	}
	if (($p->{objectlabel})) { $query .= "$w objectlabel like ?"; $w = " and "; push @{$p->{_attr}{bindvars}}, $p->{objectlabel}; }
	if (($p->{type})) {
		if (ref($p->{type}) eq "ARRAY") { $query .= "$w type in ('" . join("', '", @{$p->{type}}) . "')"; }
		else {
			$query .= "$w type like ?";
			push @{$p->{_attr}{bindvars}}, $p->{type};
		}
		$w = " and ";
	}
	if (($p->{status})) { $query .= "$w status like ?"; $w = " and "; push @{$p->{_attr}{bindvars}}, $p->{status}; }

	$p->{_attr}{query} = $query;
}

sub iobjectcopy {
	my $p = shift;

	$p->{_attr}{mylabel} = "iobjectcopy";
	$p->{parent_id} = -1 unless ($p->{parent_id});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id parent_id copyobject_id));
	$p->{_attr}{query} = "insert into objects (id, type, parent_id, name, created, modified, comments,
			status, createdby, modifiedby, objectlabel)
			(select ?, type, ?, name, created, modified, comments, status, now(), now(), objectlabel
				from objects where id = ?)";
}

sub iobject {
	my $p = shift;

	$p->{_attr}{mylabel} = "iobject";
	$p->{parent_id} = -1 unless ($p->{parent_id});
	$p->{name} =~ s/[^[:alnum:]]/_/g;
	$p->{name} =~ s/__*/_/g;
	if ($p->{name} !~ /^[[:alpha:]]/) { $p->{name} = "a$p->{name}"; }
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id type parent_id name comments status currentuser currentuser objectlabel));
	$p->{_attr}{query} = "insert into objects (id, type, parent_id, name, created, modified, comments,
			status, createdby, modifiedby, objectlabel)
			values (?, ?, ?, ?, now(), now(), ?, ?, ?, ?, ?)";
}

sub uobjectmodified {
	my $p = shift;

	$p->{_attr}{mylabel} = "uobject";
	$p->{object_id} = $p->{id} unless($p->{object_id});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(currentuser object_id));

	$p->{_attr}{query} = "update objects set modified = now(), modifiedby = ? where id = ?";
}

sub uobject {
	my $p = shift;

	$p->{_attr}{mylabel} = "uobject";
	$p->{parent_id} = -1 unless ($p->{parent_id});
	$p->{object_id} = $p->{id} unless($p->{object_id});
	$p->{name} =~ s/[^[:alnum:]]/_/g;
	$p->{name} =~ s/__*/_/g;
	if ($p->{name} !~ /^[_[:alpha:]]/) { $p->{name} = "a$p->{name}"; }
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(name parent_id comments status currentuser objectlabel object_id));

	$p->{_attr}{query} = "update objects set name = ?, parent_id = ?, modified = now(), comments = ?,
				status = ?, modifiedby = ?, objectlabel = ? where id = ?";
}

sub dobject {

	my $p = shift;

	$p->{_attr}{mylabel} = "dobject";
	$p->{object_id} = $p->{id} unless($p->{object_id});
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(currentuser object_id));

	$p->{_attr}{query} = "update objects modified = now (), status = 'deleted', modifiedby = ? where id = ?";
}


# Users definitions
sub suserdef {
	my $p = shift; # self

	$p->{_attr}{bindvars} = [ $p->{username}, $p->{password}, $p->{status} ];

	$p->{_attr}{mylabel} = "suserdef";
	$p->{_attr}{columns} = [];
	push @{$p->{_attr}{columns}}, qw (id name username);

	$p->{_attr}{query} = "select obj.id, obj.name, usr.username from objects obj, users usr
where obj.id = usr.id and type = 'user', and usr.username = ? and usr.password = ? and obj.status like ?";
}

#Scheduler Queries

sub swschedulescheck {
	my $p =shift;

	$p->{_attr}{mylabel} = "swschedulescheck";

	$p->{_attr}{columns} = ["sched_id", "sched_label", "object_id", "sched_created", "sched_modified", "sched_min", "sched_hour",
									"sched_mday", "sched_month", "sched_wday", "sched_mode", "sched_status", "tracefile", "errorfile",
									"object_id", "rt_status", "status_date", "sched_id", "system_pid" ];
	$p->{_attr}{bindvars} = [];

# "sched_id", "object_id", "system_pid" ];
#SELECT wsch.*, wfst.* from wscheduler wsch, wflowsstatus wfst

	$p->{_attr}{query} = "SELECT wsch.sched_id, wsch.sched_label, wsch.object_id, wsch.sched_created, wsch.sched_modified,
				wsch.sched_min, wsch.sched_hour, wsch.sched_mday, wsch.sched_month, wsch.sched_wday,
				wsch.sched_mode, wsch.sched_status, wsch.tracefile, wsch.errorfile, wfst.object_id AS wfst_object_id,
				wfst.rt_status, wfst.status_date, wfst.sched_id AS wfst_sched_id, wfst.system_pid
			FROM wscheduler wsch, wflowsstatus wfst
			WHERE wsch.object_id = wfst.object_id
			AND wsch.sched_mode = 'daemon'
			AND wsch.sched_status = 'online'
			AND wsch.object_id > 0
			UNION
			SELECT wsch2.sched_id, wsch2.sched_label, wsch2.object_id, wsch2.sched_created, wsch2.sched_modified,
					 wsch2.sched_min, wsch2.sched_hour, wsch2.sched_mday, wsch2.sched_month, wsch2.sched_wday,
					 wsch2.sched_mode, wsch2.sched_status, wsch2.tracefile, wsch2.errorfile, wsch2.object_id AS wfst_object_id,
					 'online' AS rt_status, now() AS status_date, wsch2.sched_id AS wfst_sched_id, NULL AS system_pid
			 FROM wscheduler wsch2
			 WHERE wsch2.object_id not in (select wfst2.object_id from wflowsstatus wfst2)";
}

sub swrecentschedule {
	my $p = shift; # self
 
	$p->{_attr}{mylabel} = "swrecentschedule";
	$p->{_attr}{columns} = [ "countschedules" ];
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "SELECT count(sched_id) AS countschedules from wscheduler WHERE sched_modified > to_timestamp('$p->{sched_modified}', 'DD/MM/YYYY HH24:MI:SS')";

	if ($p->{object_id} > 0) {
		$p->{_attr}{query} .= " AND object_id = $p->{object_id}";
	}
	elsif ($p->{object_id} < 0) {
		$p->{_attr}{query} .= " AND ((sched_id in ('archiver', 'monitor')) OR (sched_mode = 'once'))";
	}

}	

sub swschedulerlist {
	my $p = shift;

	
	$p->{_attr}{mylabel} = "swschedulerlist";
	$p->{_attr}{columns} = [ "sched_id", "sched_label", "object_id", "sched_created", "sched_modified", "sched_min", "sched_hour",
				 "sched_mday", "sched_month", "sched_wday", "sched_mode", "sched_status", "tracefile", "errorfile" ];
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "SELECT sched_id, sched_label, object_id, sched_created, sched_modified, sched_min,
		sched_hour, sched_mday, sched_month, sched_wday, sched_mode, sched_status, tracefile, errorfile
		from wscheduler";

	my $w = " WHERE ";
	if ($p->{object_id} > 0) {
		$p->{_attr}{query} .= " $w object_id = $p->{object_id}";
		$w = " AND ";
	}
	elsif ($p->{object_id} < 0) {
		$p->{_attr}{query} .= " $w ((sched_id in ('archiver', 'monitor')) OR (sched_mode = 'once'))";
	}

}	


1;
sub swschedulerlist2 {
	my $p = shift;

	
	$p->{_attr}{mylabel} = "swschedulerlist";
	$p->{_attr}{columns} = [ "sched_id", "sched_label", "object_id", "object_id_name", "sched_created", "sched_modified", 
				 "sched_min", "sched_hour","sched_mday", "sched_month", "sched_wday", "sched_mode", "sched_status",
				 "tracefile", "errorfile" ];
	$p->{_attr}{bindvars} = [];

	my $q1 = "SELECT ws.sched_id, ws.sched_label, ws.object_id, o.objectlabel AS object_id_name, ws.sched_created,
		ws.sched_modified, ws.sched_min, ws.sched_hour, ws.sched_mday, ws.sched_month, ws.sched_wday, ws.sched_mode,
		ws.sched_status, ws.tracefile, ws.errorfile
		from wscheduler ws, objects o
		where ws.object_id = o.id";
	my $q2 = "SELECT ws.sched_id, ws.sched_label, ws.object_id, '' AS object_id_name, ws.sched_created,
		ws.sched_modified, ws.sched_min, ws.sched_hour, ws.sched_mday, ws.sched_month, ws.sched_wday, ws.sched_mode,
		ws.sched_status, ws.tracefile, ws.errorfile
		from wscheduler ws
		where (ws.object_id <= 0 or  ws.object_id is null)";

	my (@bv1, @bv2);

	my $w = " AND ";
	if ($p->{sched_id} and ($p->{sched_id} ne '%')) {
		$q1 .= " $w sched_id like ?";
		$q2 .= " $w sched_id like ?";
		push @bv1, $p->{sched_id};
		push @bv2, $p->{sched_id};
	}
	if ($p->{sched_status} and ($p->{sched_status} ne '%')) {
		$q1 .= " $w sched_status like ?";
		$q2 .= " $w sched_status like ?";
		push @bv1, $p->{sched_status};
		push @bv2, $p->{sched_status};
	}
	else {
		$q1 .= " $w sched_status != 'deleted'";
		$q2 .= " $w sched_status != 'deleted'";
	}
	if ($p->{object_id}) {
		if ($p->{object_id} > 0) {
			$q1 .= " $w object_id = $p->{object_id}";
			$p->{_attr}{query} = $q1;
			push @{$p->{_attr}{bindvars}}, @bv1;
		}
		elsif ($p->{object_id} < 0) {
			$q2 .= " $w (sched_id in ('archiver', 'monitor'))";
			$p->{_attr}{query} = $q2;
			push @{$p->{_attr}{bindvars}}, @bv2;
		}
	}
	else {
		$p->{_attr}{query} = "$q1 UNION $q2";
		push @{$p->{_attr}{bindvars}}, @bv1, @bv2;
	}
	$p->{_attr}{query};
}	

sub iwscheduler {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwscheduler";
	$p->{object_id} = -1 unless ($p->{object_id});
	$p->{sched_id} =~ s/[^[:alnum:]]/_/g;
	$p->{sched_id} =~ s/__*/_/g;
	if ($p->{sched_id} !~ /^[[:alpha:]]/) { $p->{sched_id} = "a$p->{sched_id}"; }


	$p->{_attr}{bindvars} = $p->attrs2aref (qw(sched_id sched_label object_id sched_min sched_hour sched_mday sched_month sched_wday sched_mode sched_status tracefile errorfile));
	$p->{_attr}{query} = " insert into wscheduler (sched_id, sched_label, object_id, sched_created, sched_modified, sched_min,
			sched_hour, sched_mday, sched_month, sched_wday, sched_mode, sched_status, tracefile, errorfile)
			values (?, ?, ?, now(), now(), ?, ?, ?, ?, ?, ?, ?, ?, ?);";
}

sub uwscheduler {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwscheduler";
	$p->{object_id} = -1 unless ($p->{object_id});
	$p->{sched_id} =~ s/[^[:alnum:]]/_/g;
	$p->{sched_id} =~ s/__*/_/g;
	if ($p->{sched_id} !~ /^[[:alpha:]]/) { $p->{sched_id} = "a$p->{sched_id}"; }
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(sched_label object_id sched_min sched_hour sched_mday sched_month sched_wday sched_mode sched_status tracefile errorfile sched_id));

	$p->{_attr}{query} = "update wscheduler set sched_label = ?, object_id = ?, sched_modified = now(), sched_min = ?,
                        sched_hour = ?,sched_mday = ?, sched_month = ?, sched_wday = ?, sched_mode = ?, sched_status = ?,
			tracefile = ?, errorfile = ?
			where sched_id = ?";
}

sub dwscheduler {
	my $p = shift;

	$p->{_attr}{mylabel} = "dwscheduler";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(sched_id));

	$p->{_attr}{query} = "delete from wscheduler where where sched_id = ?";
}

#Queue Vars queries

sub swvarslist {
	my $p = shift;

	$p->{_attr}{mylabel} = "swvarslist";
	$p->{_attr}{columns} = [ "object_id", "varid", "specialvar", "datatype", "created", "modified", "varvalue" ];
	$p->{_attr}{bindvars} = [];

	my $q = "SELECT object_id, varid, specialvar, datatype, to_char(created, 'DD/MM/YYYY HH24:MI:SS') AS created,
				to_char(modified, 'DD/MM/YYYY HH24:MI:SS') AS modified, varvalue FROM wvars";

	my $w = " WHERE ";

	if ($p->{object_id}) {
		if (ref($p->{object_id}) eq "ARRAY") {
			$q .= " $w object_id in (" . join(',', @{$p->{object_id}}) . ")";
		}
		else { $q .= " $w object_id = $p->{object_id}"; }

		$w = " AND ";
	}
	if ($p->{varid}) {
		if (ref($p->{varid}) eq "ARRAY") { $q .= " $w varid in ('" . join("','", @{$p->{varid}}) . "')"; }
		else {$q .= " $w varid like $p->{varid}"; }
		$w = " AND ";
	}

	$p->{_attr}{query} = $q;
}

sub iwvars {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwvars";
	$p->{_attr}{bindvars} = $p->attrs2aref("object_id", "varid", "specialvar", "datatype", "varvalue");

	$p->{_attr}{query} = "INSERT INTO wvars (object_id, varid, specialvar, datatype, created, modified, varvalue) 
				values (?, ?, ?, ?, now(), now(), ?)";
}

sub uwvars {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwvars";
	$p->{_attr}{bindvars} = $p->attrs2aref("specialvar", "datatype", "varvalue", "object_id", "varid" );

	$p->{_attr}{query} = "UPDATE wvars SET specialvar = ?, datatype = ?, modified = now(), varvalue = ?
					WHERE object_id = ? and varid = ?";
}

#wrunerrors queries


sub sflowlasterrorscount {
	my $p = shift;

	$p->{_attr}{mylabel} = "sflowlasterrorscount";
	$p->{_attr}{columns} = [ "errorscount" ];
	my $q = "SELECT count(b.error_id) AS errorscount FROM wrunerrors b, wflowsstatus c
			WHERE b.object_id = c.object_id
			AND c.object_id = ?
			AND b.error_date > date_trunc('minute', c.status_date)";

	push @{$p->{_attr}{bindvars}}, $p->{object_id};
	$p->{_attr}{query} = $q;
}
	
sub sflowerrorscount {
	my $p = shift;

	$p->{_attr}{mylabel} = "sflowerrorscount";
	$p->{_attr}{columns} = [ "errorscount" ];
	my $q = "SELECT count(a.error_id) AS errorscount from wrunerrors a where a.object_id = ?";
	push @{$p->{_attr}{bindvars}}, $p->{object_id};
	$p->{_attr}{query} = $q;
}


sub swrunerrorslist {
	my $p = shift;

	$p->{_attr}{mylabel} = "swrunerrorslist";
	$p->{_attr}{columns} = [ "error_id", "object_id", "msg_id", "run_id", "error_date", "error_context", "error_code", "driver_error", "error_text", "last_query", "stack_dump" ];
	my $bv = $p->{_attr}{bindvars} = [];
	
	my $q = "SELECT error_id, object_id, msg_id, run_id, error_date, error_context, error_code, driver_error,
			error_text, last_query, stack_dump
		FROM wrunerrors";

	my $w = " WHERE";
	my $v;
	if ($v = $p->{error_date}) { $q .= "$w error_date = to_timestamp(?, 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; push @$bv, $v; }
	if ($v = $p->{error_date1}) { $q .= "$w error_date >= to_timestamp(?, 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; push @$bv, $v; }
	if ($v = $p->{error_date2}) { $q .= "$w error_date <= to_timestamp(?, 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; push @$bv, $v; }
	if ($v = $p->{object_id}) { $q .= "$w object_id = $v"; $w = " AND"; }
	if ($v = $p->{objectlabel}) {
		$q .= "$w object_id in (SELECT id from objects where name like ? or objectlabel like ?)";
		$w = " AND";
		push @$bv, $v, $v;
	}
	if ($v = $p->{error_id}) {
		$q .= " $w error_id = $v"; $w = " AND";
	}
	$p->{_attr}{query} = $q;
}

sub iwrunerrors {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwrunerrors";
	$p->{_attr}{bindvars} = $p->attrs2aref("error_id", "object_id", "msg_id", "run_id", "error_context", "error_code", "driver_error", "error_text", "last_query", "stack_dump");

	$p->{_attr}{query} = "INSERT INTO wrunerrors (error_id, object_id, msg_id, run_id, error_date, error_context, error_code,
			driver_error, error_text, last_query, stack_dump)
			VALUES (?, ?, ?, ?, now(), ?, ?, ?, ?, ?, ?)";
}

#wrunhisto queries 

sub iwrunhisto {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwrunhisto";
	$p->{_attr}{bindvars} = $p->attrs2aref("run_id","object_id","syspid","duration","onerror","incoms","transforms","outgos","rollbaks");

	$p->{_attr}{query} = "INSERT INTO wrunhisto 
		(run_id, object_id, syspid, started, duration, onerror, incoms, transforms, outgos, rollbaks)
		VALUES (?, ?, ?, now(), ?, ?, ?, ?, ?, ?)";
}

sub uwrunhisto {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwrunhisto";
	$p->{_attr}{bindvars} = $p->attrs2aref("duration", "onerror", "incoms", "transforms", "outgos", "rollbaks", "run_id", "object_id");

	$p->{_attr}{query} = "UPDATE wrunhisto SET
		duration = ?, onerror = ?, incoms = ?, transforms = ?, outgos = ?, rollbaks = ?
		WHERE run_id = ? AND object_id = ?";
}

#wevents list;

sub sweventslist {
	my $p = shift;

	$p->{_attr}{mylabel} = "sweventslist";
	$p->{_attr}{columns} = [ "object_id", "eventid", "eventdate", "expiredate" ];
	$p->{_attr}{bindvars} = [];

	my $q = "SELECT object_id, eventid, to_char(eventdate, 'DD/MM/YYYY HH24:MI:SS') AS eventdate,
				to_char(expiredate, 'DD/MM/YYYY HH24:MI:SS') AS expiredate
				FROM wevents WHERE expiredate > now() AND (now() - eventdate) < 7";

	my $w = " AND ";

	if ($p->{object_id}) { $q .= " $w object_id = $p->{object_id}"; $w = " AND "; }
	if ($p->{eventid}) { $q .= " $w eventid = $p->{eventid}"; $w = " AND "; }
	if ($p->{afterdate}) {
			$q .= " $w eventdate > to_timestamp(?, 'DD/MM/YYYY HH24:MI:SS')";
			push @{$p->{_attr}{bindvars}}, $p->{afterdate};
			$w = " AND ";
	}

	$p->{_attr}{query} = $q;
}

sub iwevents {
	my $p = shift;	

	$p->{_attr}{mylabel} = "iwevents";
	$p->{_attr}{bindvars} = $p->attrs2aref("object_id", "eventid", "expiredelay");

	$p->{_attr}{query} = "INSERT INTO wevents (object_id, eventid, eventdate, expiredate)
				values (?, ?, now(), now() + ?)";
}

sub uwevents {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwevents";
	$p->{_attr}{bindvars} = $p->attrs2aref("expiredelay", "object_id", "eventid");

	$p->{_attr}{query} = "UPDATE wevents SET eventdate = now(), expiredate = now() + ?
					WHERE object_id = ? and eventid = ?";
}

sub dwevents {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwevents";
	$p->{_attr}{bindvars} = $p->attrs2aref("object_id", "eventid");

	$p->{_attr}{query} = "DELETE FROM wevents WHERE object_id = ? and eventid = ?";

}

#wwaitevents queries

sub swwaiteventslist {
	my $p = shift;

	$p->{_attr}{mylabel} = "swwaiteventslist";
	$p->{_attr}{columns} = [ "object_id", "object_id_source", "eventid", "received" ];
	$p->{_attr}{bindvars} = [];

	my $q = "SELECT object_id, object_id_source, eventid, to_char(received, 'DD/MM/YYYY HH24:MI:SS') AS received FROM wwaitevents";

	my $w = " WHERE ";
	if ($p->{object_id}) {
		$q .= " $w object_id = $p->{object_id}";
		$w = "AND ";
	}
	if ($p->{object_id_source}) {
		$q .= " $w object_id_source = $p->{object_id_source}";
		$w = "AND ";
	}
	if ($p->{event_id} and ($p->{event_id} ne '%')) {
		$q .= " $w eventid like $p->{eventid}";
		$w = "AND ";
	}

	$p->{_attr}{query} = $q;
}

sub iwwaitevents {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwwaitevents";
	$p->{_attr}{bindvars} = $p->attrs2aref("object_id", "object_id_source", "eventid");

	$p->{_attr}{query} = "INSERT INTO wwaitevents (object_id, object_id_source, eventid, received) values (?, ?, now())";
}


sub dwwaitevents {
	my $p = shift;

	$p->{_attr}{mylabel} = "dwwaitevents";
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "DELETE FROM wwaitevents WHERE object_id = $p->{object_id}";

	if ($p->{object_id_source}) {
		$p->{_attr}{query} .= " AND object_id_source = $p->{object_id_source}";
	}
	if ($p->{event_id} and ($p->{event_id} ne '%')) {
		$p->{_attr}{query} .= " AND eventid like $p->{eventid}";
	}
		
	$p->{_attr}{query};
}

# wstatus

sub swflowsstatuslistrt {
	my $p = shift;

	$p->{_attr}{mylabel} = "swstatuslistrt";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "objectid", "rt_status", "status_date", "sched_id", "system_pid" ];

	my $q = "SELECT object_id, rt_status, status_date, sched_id, system_pid FROM wflowsstatus";

	my $w = " WHERE ";
	if ($p->{object_id}) {
		if (ref($p->{object_id}) eq "ARRAY") { $q .= " $w object_id in (" . join(',', @{$p->{object_id}}) . ")"; $w = " AND "; }
		else { $q .= " $w object_id = $p->{object_id}"; $w = " AND "; }
	}
	if ($p->{FORUPDATE}) { $q .= " FOR UPDATE"; }


	$p->{_attr}{query} = $q;
}

sub swflowsstatuslistmonitor {
	my $p = shift;

	$p->{_attr}{mylabel} = "swflowsstatuslistmonitor";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "object_id", "parent_id", "objectlabel", "name", "type", "object_status", "rt_status", "status_date", "sched_id", "system_pid", "next_exectime" ];

	my $q = "SELECT object_id, object_id AS parent_id, sched_id AS name, 'job monitor' AS type, sched_id AS objectlabel,
		'online' as object_status, rt_status, to_char(status_date,'DD/MM/YYYY HH24:MI') AS status_date, sched_id, system_pid,
		to_char(next_exectime, 'DD/MM/YYYY HH24:MI') AS next_exectime
		FROM wflowsstatus
		where object_id = -1";

	$p->{_attr}{query} = $q;
}

sub swflowsstatuslist {
	my $p = shift;

	$p->{_attr}{mylabel} = "swstatuslist";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "object_id", "parent_id", "objectlabel", "name", "type", "object_status", "rt_status", "status_date", "sched_id", "system_pid", "next_exectime" ];

	#my $q = "SELECT object_id, CASE WHEN (parent_id <= 0) THEN object_id ELSE parent_id END AS parent_id, name, type, objectlabel,
	my $q = "SELECT object_id, CASE WHEN (parent_id <= 0) THEN object_id ELSE parent_id END AS parent_id, name, type, objectlabel,
		status as object_status, rt_status, to_char(status_date,'DD/MM/YYYY HH24:MI') AS status_date, sched_id, system_pid,
		to_char(next_exectime, 'DD/MM/YYYY HH24:MI') AS next_exectime
		FROM wflowsstatus, objects
		where ((object_id = id and type in ('flow', 'subflow')))"; # OR (object_id = parent_id and type = 'subflow'))";

	if ($p->{object_id}) {
		if (ref($p->{object_id}) eq "ARRAY") { $q .= " AND object_id in (" . join(',', @{$p->{object_id}}) . ")"; }
		else { $q .= " AND (object_id = $p->{object_id} OR parent_id = $p->{object_id})"; }
	}

	if ($p->{name} and ($p->{name} ne "%")) {
		$q .= " AND name like ? or objectlabel like ?";
		push @{$p->{bindvars}}, $p->{objectlabel};
		push @{$p->{bindvars}}, $p->{objectlabel};
	}
	if ($p->{objectlabel} and ($p->{objectlabel} ne "%")) { $q .= " AND objectlabel like ?", push @{$p->{bindvars}}, $p->{objectlabel}; }
	if ($p->{rt_status} and ($p->{rt_status} ne '%')) { $q .= " AND rt_status = '$p->{rt_status}'"; }

	$p->{_attr}{query} = $q . " order by parent_id, object_id";
}

sub iwflowsstatusinitial {
	my $p = shift;
	$p->{_attr}{mylabel} = "iwflowsstatusinitial";
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "INSERT INTO wflowsstatus (object_id, rt_status, status_date) SELECT id, 'online', now() FROM objects WHERE type IN ('flow', 'subflow') AND id NOT IN (SELECT object_id FROM wflowsstatus);"
}

sub iwflowsstatusinitialScheduler {
	my $p = shift;
	$p->{_attr}{mylabel} = "iwflowsstatusinitial";
	$p->{_attr}{bindvars} = [];

	$p->{_attr}{query} = "INSERT INTO wflowsstatus SELECT id, 'online', now() FROM objects WHERE type IN ('flow', 'subflow') AND id NOT IN (SELECT object_id FROM wflowsstatus);"
}

sub iwflowsstatus {
	my $p = shift;

	$p->{_attr}{mylabel} = "iwstatus";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(object_id rt_status sched_id system_pid));

	my $q = "INSERT INTO wflowsstatus (object_id, rt_status, status_date, sched_id, system_pid) values (?, ?, now(), ?, ?)";

	$p->{_attr}{query} = $q;
}

sub uwflowsstatusnextexec {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwflowsstatus";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(next_exectime object_id));

	my $q = "UPDATE wflowsstatus set next_exectime = to_timestamp(?, 'DD/MM/YYYY HH24:MI:SS') where object_id = ?";

	if ($p->{rt_status_cond}) {
		if (ref($p->{rt_status_cond}) eq "ARRAY") { $q .= " AND rt_status in ('" . join("','", @{$p->{rt_status_cond}}) . "')"; }
		else { $q .= " AND rt_status like ?"; push @{$p->{_attr}{bindvars}}, $p->{rt_status_cond}; }
	}
	$p->{_attr}{query} = $q;
}

sub uwflowsstatus {
	my $p = shift;

	$p->{_attr}{mylabel} = "uwflowsstatus";
	$p->{_attr}{bindvars} = $p->attrs2aref (qw(rt_status sched_id system_pid object_id));

	my $q = "UPDATE wflowsstatus set rt_status = ?, status_date = now(), sched_id = ?, system_pid = ? where object_id = ?";

	if ($p->{rt_status_cond}) {
		if (ref($p->{rt_status_cond}) eq "ARRAY") { $q .= " AND rt_status in ('" . join("','", @{$p->{rt_status_cond}}) . "')"; }
		else { $q .= " AND rt_status like ?"; push @{$p->{_attr}{bindvars}}, $p->{rt_status_cond}; }
	}
	$p->{_attr}{query} = $q;
}

# Queue Read and write

sub smastertable {
	my $p = shift;


	$p->{_attr}{mylabel} = "s$p->{mastertable}";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "msgseq", "msgdate" ];

	my $q = "SELECT i.msgseq FROM $p->{mastertable} i";

	my $w = " WHERE";

	if ($p->{msgseq}) { $q .= "$w i.msgseq = $p->{msgseq}"; $w = " AND"; }
	if ($p->{l_msgseq}) { $q .= "$w i.msgseq < $p->{l_msgseq}"; $w = " AND"; }
	if ($p->{g_msgseq}) { $q .= "$w i.msgseq > $p->{g_msgseq}"; $w = " AND"; }
	if ($p->{le_msgseq}) { $q .= "$w i.msgseq <= $p->{le_msgseq}"; $w = " AND"; }
	if ($p->{ge_msgseq}) { $q .= "$w i.msgseq >= $p->{ge_msgseq}"; $w = " AND"; }
	if ($p->{msgdate}) { $q .= "$w i.msgdate = to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{l_msgdate}) { $q .= "$w i.msgdate < to_timestamp('$p->{l_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{g_msgdate}) { $q .= "$w i.msgdate > to_timestamp('$p->{g_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{le_msgdate}) { $q .= "$w i.msgdate <= to_timestamp('$p->{le_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{ge_msgdate}) { $q .= "$w i.msgdate >= to_timestamp('$p->{ge_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	$q .= " ORDER BY i.msgseq";

	$p->{_attr}{query} = $q;
}

sub sdatatable {
	my $p = shift;


	$p->{_attr}{mylabel} = "s$p->{datatable}";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "msgseq", "msgdate" ];

	my $q = "SELECT i.*, id.* FROM $p->{mastertable} i, $p->{datatable} id WHERE i.msgseq = id.msgseq";

	my $w = " AND";

	if ($p->{msgseq}) {
		my $v = $p->{msgseq};
		$v = [ $v ] unless (ref($v) eq "ARRAY");
		$q .= "$w i.msgseq in (" . join(",", @$v) . ")";
		$w = " AND";
	}
	if ($p->{l_msgseq}) { $q .= "$w i.msgseq < $p->{l_msgseq}"; $w = " AND"; }
	if ($p->{g_msgseq}) { $q .= "$w i.msgseq > $p->{g_msgseq}"; $w = " AND"; }
	if ($p->{le_msgseq}) { $q .= "$w i.msgseq <= $p->{le_msgseq}"; $w = " AND"; }
	if ($p->{ge_msgseq}) { $q .= "$w i.msgseq >= $p->{ge_msgseq}"; $w = " AND"; }
	if ($p->{msgdate}) { $q .= "$w i.msgdate = to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{l_msgdate}) { $q .= "$w i.msgdate < to_timestamp('$p->{g_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{g_msgdate}) { $q .= "$w i.msgdate > to_timestamp('$p->{l_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{le_msgdate}) { $q .= "$w i.msgdate <= to_timestamp('$p->{ge_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	if ($p->{ge_msgdate}) { $q .= "$w i.msgdate >= to_timestamp('$p->{le_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; $w = " AND"; }
	$q .= " ORDER BY i.msgseq";

	$p->{_attr}{query} = $q;
}

sub snextincommsg {
	my $p =shift;

	$p->{_attr}{mylabel} = "si$p->{mastertable}";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "msgseq", "msgdate", "field_id", "field_value" ];

	my $q = "SELECT i.msgseq, i.msgdate, id.field_id, id.field_value FROM $p->{mastertable} i, $p->{datatable} id
		WHERE i.msgseq = id.msgseq";

	if ($p->{msgseq}) { $q .= " AND i.msgseq = $p->{msgseq}"; }
	if ($p->{l_msgseq}) { $q .= " AND i.msgseq < $p->{l_msgseq}"; }
	if ($p->{g_msgseq}) { $q .= " AND i.msgseq > $p->{g_msgseq}"; }
	if ($p->{le_msgseq}) { $q .= " AND i.msgseq <= $p->{le_msgseq}"; }
	if ($p->{ge_msgseq}) { $q .= " AND i.msgseq >= $p->{ge_msgseq}"; }
	if ($p->{msgdate}) { $q .= " AND i.msgdate = to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{l_msgdate}) { $q .= " AND i.msgdate < to_timestamp('$p->{l_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{g_msgdate}) { $q .= " AND i.msgdate > to_timestamp('$p->{g_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{le_msgdate}) { $q .= " AND i.msgdate <= to_timestamp('$p->{le_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{ge_msgdate}) { $q .= " AND i.msgdate >= to_timestamp('$p->{ge_msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{field_id}) { $q .= " AND id.field_id like '$p->{field_id}'"; }
	if ($p->{field_value}) { $q .= " AND id.field_value like ?"; push @{$p->{_attr}{bindvars}}, $p->{field_value};}
	$q .= " ORDER BY i.msgseq";

	$p->{_attr}{query} = $q;
}

sub snextoutgomsg {
	my $p =shift;

	$p->{_attr}{mylabel} = "so$p->{mastertable}";
	$p->{_attr}{bindvars} = [];
	$p->{_attr}{columns} = [ "msgseq", "inmsgseq", "msgdate", "msgstatus", "outdate", "field_id", "field_value" ];

	my $q = "SELECT o.msgseq, o.inmsgseq, o.msgdate, o.msgstatus, o.outdate, od.field_id, od.field_value
		FROM $p->{mastertable} o, $p->{datatable} od
		WHERE o.msgseq = od.msgseq";

	if ($p->{msgseq}) { $q .= " AND o.msgseq = $p->{msgseq}"; }
	if ($p->{msgstatus}) { $q .= " AND o.msgstatus like '$p->{msgstatus}'"; }
	if ($p->{l_msgseq}) { $q .= " AND o.msgseq < $p->{msgseq}"; }
	if ($p->{g_msgseq}) { $q .= " AND o.msgseq > $p->{msgseq}"; }
	if ($p->{le_msgseq}) { $q .= " AND o.msgseq <= $p->{msgseq}"; }
	if ($p->{ge_msgseq}) { $q .= " AND o.msgseq >= $p->{msgseq}"; }
	if ($p->{msgdate}) { $q .= " AND o.msgdate = to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{l_msgdate}) { $q .= " AND o.msgdate > to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{g_msgdate}) { $q .= " AND o.msgdate < to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{le_msgdate}) { $q .= " AND o.msgdate >= to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{ge_msgdate}) { $q .= " AND o.msgdate <= to_timestamp('$p->{msgdate}', 'DD/MM/YYYY HH24:MI:SS')"; }
	if ($p->{field_id}) { $q .= " AND od.field_id like '$p->{field_id}'"; }
	if ($p->{field_value}) { $q .= " AND od.field_value like ?"; push @{$p->{_attr}{bindvars}}, $p->{field_value};}

	$p->{_attr}{query} = $q;
}

sub iincommsg {
	my $p = shift;

	$p->{_attr}{mylabel} = "iim$p->{mastertable}";
	$p->{_attr}{bindvars} = [];

	my $q = "INSERT INTO $p->{mastertable} (msgseq, msgdate) VALUES ($p->{msgseq}, now())";

	$p->{_attr}{query} = $q;
}

sub iincommsgdata {
	my $p = shift;

	$p->{_attr}{mylabel} = "iid$p->{datatable}";
	$p->{_attr}{bindvars} = [$p->{field_value}];

	my $q = "INSERT INTO $p->{datatable} (msgseq, field_id, field_value) VALUES ($p->{msgseq}, '$p->{field_id}', ?)";

	$p->{_attr}{query} = $q;
}

sub itransformedmsg {
	my $p = shift;

	$p->{_attr}{mylabel} = "it$p->{mastertable}";
	$p->{_attr}{bindvars} = [];

	my $q = "INSERT INTO $p->{mastertable} (msgseq, inmsgseq, msgdate, msgstatus, outdate)
		VALUES ($p->{msgseq}, $p->{inmsgseq}, now(), 'transformed', null)";

	$p->{_attr}{query} = $q;
}

sub itransformedmsgdata {
	my $p = shift;

	$p->{_attr}{mylabel} = "itd$p->{datatable}";
	$p->{_attr}{bindvars} = [$p->{field_value}];

	my $q = "INSERT INTO $p->{datatable} (msgseq, field_id, field_value) VALUES ($p->{msgseq}, '$p->{field_id}', ?)";

	$p->{_attr}{query} = $q;
}

sub usentmsg {
	my $p = shift;

	$p->{_attr}{mylabel} = "uo$p->{mastertable}";
	$p->{_attr}{bindvars} = [];

	my $q = "UPDATE $p->{mastertable} SET outdate = now(), msgstatus = '$p->{msgstatus}' WHERE msgseq = $p->{msgseq}";

	$p->{_attr}{query} = $q;
}
# Create external Objects

sub sdbobject {
	my $p = shift;

	$p->{_attr}{mylabel} = "sdbobject";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} || '%';
	my $seqname = $tableprefix;
	my $q = "SELECT relname as object_name, 'SEQ'::varchar as object_type FROM pg_class WHERE relkind ='S' and relname like '$seqname and '$p->{dbobjtype}' like 'SEQ'
		UNION
		SELECT relname as object_name, 'INDEX'::varchar as object_type  FROM pg_class WHERE relkind ='i' and relname like '$tableprefix' and '$p->{dbobjtype}' like 'INDEX'
		UNION
		SELECT tablename as object_name, 'TABLE'::varchar as object_type  FROM pg_tables WHERE tablename like '$tableprefix' and '$p->{dbobjtype}' like 'TABLE'
		ORDER BY object_name";
	$p->{_attr}{query} = $q;
}

sub sincomobjects {
	my $p = shift;

	$p->{_attr}{mylabel} = "sincomobjects";
	$p->{_attr}{bindvars} = [];

	my $seqname = $p->{tableprefix} . '_seq';
	my $tableprefix = $p->{tableprefix} . '_i%';
	my $q = "SELECT relname as object_name, 'SEQ'::varchar as object_type FROM pg_class WHERE relkind ='S' and relname like '$seqname'
		UNION
		SELECT relname as object_name, 'INDEX'::varchar as object_type  FROM pg_class WHERE relkind ='i' and relname like '$tableprefix'
		UNION
		SELECT tablename as object_name, 'TABLE'::varchar as object_type  FROM pg_tables WHERE tablename like '$tableprefix'
		ORDER BY object_name";
	$p->{_attr}{query} = $q;
}

sub soutgoobjects {
	my $p = shift;

	$p->{_attr}{mylabel} = "soutgoobjects";
	$p->{_attr}{bindvars} = [];

	my $seqname = $p->{tableprefix} . '_seq';
	my $tableprefix = $p->{tableprefix} . '_o%';
	my $q = "SELECT relname as object_name, 'SEQ'::varchar as object_type FROM pg_class WHERE relkind ='S' and relname like '$seqname'
		UNION
		SELECT relname as object_name, 'INDEX'::varchar as object_type FROM pg_class WHERE relkind ='i' and relname like '$tableprefix'
		UNION
		SELECT tablename as object_name, 'TABLE'::varchar as object_type FROM pg_tables WHERE tablename like '$tableprefix'
		ORDER BY object_name";
	$p->{_attr}{query} = $q;
}

sub cobject_seq {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_seq";
	$p->{_attr}{bindvars} = [];

	my $seqname = $p->{tableprefix} . '_seq';
	$p->{_attr}{query} = "CREATE SEQUENCE $seqname";
}

sub cobject_o {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_o";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_o';
	$p->{_attr}{query} = "CREATE TABLE $tableprefix (
		msgseq int4 not null,
		inmsgseq int4 not null,
		msgdate  timestamp default now() not null,
		msgstatus varchar(20) not null,
		outdate timestamp)";
}

sub cobject_od {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_od";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_od';
	$p->{_attr}{query} = "CREATE TABLE $tableprefix (
		msgseq int4 not null,
		field_id  varchar(30) not null,
		field_value  varchar(4000))";
}

sub cobject_or {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_or";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_or';
	$p->{_attr}{query} = "CREATE TABLE $tableprefix (
		msgseq int4 not null,
		field_id  varchar(30) not null,
		field_value  varchar(4000))";
}

sub cobject_opk {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_opk";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_o';
	my $indexprefix = $p->{tableprefix} . '_opk';
	$p->{_attr}{query} = "CREATE  UNIQUE INDEX $indexprefix ON $tableprefix (msgseq)";
}

sub cobject_odpk {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_odpk";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_od';
	my $indexprefix = $p->{tableprefix} . '_odpk';
	$p->{_attr}{query} = "CREATE  UNIQUE INDEX $indexprefix ON $tableprefix (msgseq, field_id)";
}

sub cobject_orpk {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_orpk";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_or';
	my $indexprefix = $p->{tableprefix} . '_orpk';
	$p->{_attr}{query} = "CREATE  UNIQUE INDEX $indexprefix ON $tableprefix (msgseq, field_id)";
}



sub cobject_i {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_i";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_i';
	$p->{_attr}{query} = "CREATE TABLE $tableprefix (
		msgseq int4 not null,
		msgdate  timestamp default now() not null)";
}

sub cobject_id {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_id";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_id';
	$p->{_attr}{query} = "CREATE TABLE $tableprefix (
		msgseq int4 not null,
		field_id  varchar(30) not null,
		field_value  varchar(4000))";
}

sub cobject_ipk {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_ipk";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_i';
	my $indexprefix = $p->{tableprefix} . '_ipk';
	$p->{_attr}{query} = "CREATE  UNIQUE INDEX $indexprefix ON $tableprefix (msgseq)";
}

sub cobject_idpk {
	my $p = shift;

	$p->{_attr}{mylabel} = "cobject_idpk";
	$p->{_attr}{bindvars} = [];

	my $tableprefix = $p->{tableprefix} . '_id';
	my $indexprefix = $p->{tableprefix} . '_idpk';
	$p->{_attr}{query} = "CREATE  UNIQUE INDEX $indexprefix ON $tableprefix (msgseq, field_id)";
}


sub optimizeall {
	my $p = shift;

	return ($p->{_attr}{query} = "VACUUM ANALYZE");
}

1;
