# $Id: common.pm,v 1.13 2004/01/21 08:34:05 mig Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::HTML::common
# Common Functions for the HTML Front End.
######################################

# This module is not meant to be called on its own, it should be called from
# Comas::HTML.

package Comas::HTML::common;

use strict;
use warnings;
use Carp;

use Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(_catalog2html_opts _list2html_opts _validator2html _id2descr
                    _is_privileged _get_file_info);
our %EXPORT_TAGS = (all => [@EXPORT_OK]);

=head1 NAME

Comas::HTML::common - Common functions for the HTML Front End.

=head1 SYNOPSIS

This module is not meant to be used by itself, but as an auxiliary to
L<Comas::HTML|Comas::HTML>. Please check L<Comas::HTML|Comas::HTML>'s
documentation, in the B<Front End Tasks> section, for further details on the
usage for this functions.

=head1 SEE ALSO

L<Comas::HTML> module documentation

=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx

Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

sub _id2descr {
    my $db = shift;
    my $catalog = shift;
    my $id = shift;
    my $sth;

    unless ($sth = $db->prepare("SELECT descr FROM $catalog WHERE id = $id")) {
        return undef;
    }

    if ($sth->execute eq '0E0') {
        return undef;
    }

    my @ret = $sth->fetchrow_array;
    return $ret[0];
}

sub _catalog2html_opts {
    # This function is used to create an html dropbox  from a catalog of the
    # database
    
    # The fist argument is a Comas::DB objetct, the second is the name of the
    # catalog,
    # The third (and optional) is a id of the option selected by
    # default, if it is not present and empty option will be added and selected.

    # The fourth argument is also optional and it's the name of a configuration
    # option with the privileged values that we don't want to have in the
    # options box

    # The fifth argument if it's set to 0 or isn't set the dropbox will have a
    # blank (or unset) option,  if it's set to 1, the box will not hae the blank
    # option
    
    my $db = shift;
    my $catalog = shift;
    my $selected_id = shift;
    my $privileges = shift;
    my $type = shift;
    my ($sth, $html, $catalog_id);

    unless (defined $type) {
        $type = 0;
    }

    unless ($sth = $db->prepare("SELECT id, descr FROM $catalog WHERE id != 0")) {
        return undef;
    }
    
    if ($sth->execute eq '0E0') {
        return undef;
    }
    
    my $r = $sth->fetchall_hashref('id');

    if ($type == 0) {
        if (defined $selected_id) {
            if ($selected_id eq '') {
                $html = '<option name="unset" value="" selected></option>';
            } else {
                $html = '<option name="unset" value=""></option>';
            }
        } else {
            $html = '<option name="unset" value=""></option>';
        }
    }

    if (defined $privileges) {
        my ($priv, $p_id);
        unless ($sth = $db->prepare('SELECT value FROM config where name= ?')) {
            return undef;
        }
        unless ($sth->execute($privileges)) {
            return undef;
        }
        foreach $p_id (split(/\,/, $sth->fetchrow_array)) {
            delete $r->{$p_id};
        }
    }

    foreach $catalog_id (sort {$a <=> $b} keys %$r) {
        $html .='<option ';
        if (defined $selected_id && $catalog_id eq $selected_id) {
            $html .= 'selected ';
        }
        $html .= 'value="'. $catalog_id .'">';
        $html .= $r->{$catalog_id}{'descr'} .'</option>';
    }
    return $html;	
}

sub _is_privileged {
    # Three arguments: a Comas::DB object, the second is and id and the third
    # the name of the config option to check if is privileged the id.

    # Will return '1' if its privileged and '0' if isn't

    my ($db, $id, $priv_catalog) = (shift, shift, shift);
    my $sth;

    unless ($sth = $db->prepare('SELECT value FROM config where name= ?')) {
        return undef;
    }
    unless ($sth->execute($priv_catalog)) {
        return undef;
    }
    foreach my $p_id (split(/\,/, $sth->fetchrow_array)) {
        if ($p_id eq $id) {
            return 1;
        }
    }
    return 0;
}

sub _list2html_opts {
    # This function is used to create an html dropbox from a list of values.
    
    # The fist is the list (as a reference) and the second  (and optional) is a
    # id of the option selected by default, if it is not present and empty
    # option will be added and selected.
 
    my ($list, $selected_id) = (shift, shift);
    my ($html, $list_item);
   
    if (defined $selected_id) {
        $selected_id =~ s/^0//;    
        if ($selected_id ne '') {
            $html = '<option name="unset" value=""></option>';
        } else {
            $html = '<option name="unset" value="" selected></option>';
        }
    } else {
        $html = '<option name="unset" value="" selected></option>';
    }
    
    foreach $list_item (@$list) {
        $html .='<option ';
        if (defined $selected_id) {
            if ($list_item eq $selected_id) {
                $html .= 'selected ';
            }
        }
        $html .= 'value="'. $list_item .'">'. $list_item .'</option>';
    }
    return $html;	
}

sub _validator2html {
    # This functions parses the results of Data::FormValidator to send them 
    # to HTML::Template

    # Only have one arguments that is the object of Data::FormValidator and
    # returns a nice hash that HTML::Template understand well.

    my $results = shift;
    my %ret;

    if ( $results->has_missing ) {
        $ret{HAS_MISSING} = '1';
        foreach my $f ( $results->missing ) {
            $f =~ s/^-//;
            $ret{$f . '_MISSING'} = '1';
        }
    }    
    
    if ( $results->has_invalid ) {
        $ret{HAS_INVALID} = '1';
        foreach my $f ( $results->invalid ) {
            $f =~ s/^-//;
            $ret{$f . '_INVALID'} = '1';
        }
    }
    return %ret;
}

sub _get_file_info {
    my $h = shift;
    my $proposal = shift;
    my (%info, $mtime);

    (undef, undef, undef, undef, undef, undef, undef, $info{-size},
     undef,$mtime,undef) = stat($h->{-config}->{-prop_file_path}.
                                $proposal->{-id}.'/'.
                                $proposal->get_filename);
    (undef, $info{-min}, $info{-hour}, $info{-day}, $info{-month}, $info{-year}) = localtime($mtime);
    
    $info{-year} += 1900;
    $info{-month} += 1;

    if($info{-hour} < 10) {
        $info{-hour} = "0$info{-hour}";
    }
    if($info{-min} < 10) {
        $info{-min} = "0$info{-min}";
    }
    if($info{-size} < 1024) {
        $info{-size} = "$info{-size} bytes";
    } elsif ($info{-size} < 1048576) {
        $info{-size} = int($info{-size}/1024) . ' Kb';
    } elsif ($info{-size} < 1073741824) {
        $info{-size} = int($info{-size}/1048576) . ' Mb';
    }

    return %info;
}

1;

# $Log: common.pm,v $
# Revision 1.13  2004/01/21 08:34:05  mig
# - El id '0' en los catalogos es utilitario y no lo despliega nunca.
#
# Revision 1.12  2004/01/11 04:57:19  mig
# - Agrego _get_file_info que devuelve un hash con la info de un archivo.
#
# Revision 1.11  2003/12/20 04:14:51  mig
# - Agrego tags Id y Log que expanda el CVS
#
