#!/usr/bin/wish
#############################################################################
#
# This is Postgres Forms (pfm), a client application for PostgreSQL.
#
# Copyright (C) 2004 Willem Herremans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
# 
# Please send bug reports and feature requests via the corresponding
# facilities on the home page of Postgres Forms (pfm):
# 
# http://gborg.postgresql.org/project/pfm/
# 
# Please send all other comments or questions to the mailing list:
# 
# pfm-comments@gborg.postgresql.org

###################################################################
# Intialisation                                                   #
###################################################################

global pfmVersion
set pfmVersion {1.0.2}
global API
global installDir

package require Iwidgets

set installDir [file dirname [info script]]

if { [catch {source [file join $installDir pgin.tcl]} errMsg1 ] } then {
    if { [catch {package require Pgtcl} errMsg2] } then {
	set errMsg "$errMsg1\n$errMsg2\nNeither Pgtcl nor pgin.tcl were found.\npfm cannot connect to postgreSQL"
	tk_messageBox -message $errMsg -type ok
	set API {pfm cannot communicate with postgreSQL. Neither Pgtcl nor pgin.tcl are present.}
    } else {
	set API {pfm is using Pgtcl to communicate with postgreSQL.}
    }
} else {
    set API {pfm is using pgin.tcl to communicate with postgreSQL.}
}

########################################################################
# Begin namespace options                                              #
########################################################################

# pfmOptions is an array with all the options for Postgres Forms.
# Up to now, the options are:
#
#     - dblist : a list of data base names from which the user
#       can choose one at Open data base.
#
#     - host, hostaddr, port, user and dbname: the default values for the
#       connection parameters proposed by Open data base.
#
# pfmOptions are stored in file ~/.pfmrc. This file is read by proc initOptions.
#
# pfmOptions can be modified by menu Tools -> Options.
#
# connInfoList: the list of parameters for the conninfo in pg_connect.
#


namespace eval options {
    variable pfmOptions
    variable newOptions
    variable connInfoList {host hostaddr port user password}

    proc setDefaultOptions {OptionsName} {

	upvar $OptionsName Options
	variable connInfoList
	set Options(printcmd) {a2ps --printer=display --$(portrait_or_landscape=portrait) --rows=$(nr-of-rows=1) --columns=$(nr-of-columns=1) --major=rows --chars-per-line=$(nr_of_chars_per_line=90) --center-title=$(title=Report)}
	if { [catch pg_conndefaults conndef] } then {
	    tk_messageBox -message $conndef -type ok
	    set conndef {}
	} else {
	    foreach item $conndef {
		switch [lindex $item 0] {
		    "host" {
			set Options(host) [lindex $item 4]
		    }
		    "hostaddr" {
			set Options(hostaddr) [lindex $item 4]
		    }
		    "port" {
			set Options(port) [lindex $item 4]
		    }
		    "user" {
			set Options(user) [lindex $item 4]
		    }
		    "dbname" {
			set Options(dbname) [lindex $item 4]
		    }
		}
	    }
	    set connInfo "dbname=$Options(dbname) "
	    foreach item $connInfoList {
		if { [info exists Options($item)] } then {
		    if { ![string equal $Options($item) {}] } then {
			set connInfo "$connInfo $item=$Options($item)"
		    }
		}
	    }
	    if { [catch {set db [pg_connect -conninfo $connInfo]} errorMsg]} then {
		tk_messageBox -message $errorMsg -type ok
		set Options(dblist) {}
	    } else {
		set queryRes [pg_exec $db "SELECT datname FROM pg_database ORDER BY datname"]
		set numTuples [pg_result $queryRes -numTuples]
		set Options(dblist) {}
		for {set tupleNr 0} { $tupleNr < $numTuples} {incr tupleNr} {
		    set datname [pg_result $queryRes -getTuple $tupleNr]
		    lappend Options(dblist) $datname
		}
		pg_result $queryRes -clear
		pg_disconnect $db
	    }
	}
	return
    }

    proc saveOptions {} {

	variable pfmOptions
	set rcFile [open ~/.pfmrc w]
	set optionList [lsort [array names pfmOptions]]
	foreach option $optionList {
	    puts $rcFile "$option \{$pfmOptions($option)\}"
	}
	close $rcFile
	return
    }

    proc initOptions { } {

	variable pfmOptions
	if { [file exists ~/.pfmrc]} then {
	    set rcFile [open ~/.pfmrc r]
	    while {![eof $rcFile]} {
		set line [gets $rcFile]
		if {[string length $line] != 0} then {
		    set pfmOptions([lindex $line 0]) [lindex $line 1]
		}
	    }
	    close $rcFile
	} else {
	    setDefaultOptions pfmOptions
	    saveOptions
	}
	return
    }

    initOptions

    proc cmdOptions {} {

	variable pfmOptions
	variable newOptions
	destroy .options
	toplevel .options -class Toplevel
	wm transient .options .
	set x [winfo pointerx .]
	set y [winfo pointery .]
	wm geometry .options 400x270+$x+$y
	# wm maxsize .options 785 570
	wm minsize .options 1 1
	wm overrideredirect .options 0
	wm resizable .options 1 1
	wm title .options "pfm - Options"
	set optionList [lsort [array names pfmOptions]]
	set rowidx 0
	foreach option $optionList {
	    set newOptions($option) $pfmOptions($option)
	    label .options.lb$option -text $option
	    entry .options.ent$option -textvar ::options::newOptions($option) \
		-width 40 -background white
	    button .options.bn$option -text ">>" -pady 0 \
		-command "::options::cmdExpand $option .options.ent$option"
	    grid .options.lb$option -in .options -column 0 -row $rowidx \
		    -columnspan 1 -rowspan 1
	    grid .options.ent$option -in .options -column 1 -row $rowidx \
		    -columnspan 1 -rowspan 1
	    grid .options.bn$option -in .options -column 2 -row $rowidx \
		    -columnspan 1 -rowspan 1
	    incr rowidx
	}
	incr rowidx
	button .options.btnOK -text {OK} -command [namespace code cmdOptionOK]
	button .options.btnCancel -text {Cancel} -command [namespace code cmdOptionCancel]
	button .options.btnDefault -text {Defaults} -command [namespace code \
		{setDefaultOptions newOptions}]
	place .options.btnOK -relx 0.2 -rely 1 -y {-5} -anchor s -width 60
	place .options.btnDefault -relx 0.5 -rely 1 -y {-5} -anchor s -width 60
	place .options.btnCancel -relx 0.8 -rely 1 -y {-5} -anchor s -width 60
	return
    }


    proc cmdExpand { option optionWidget } {
	variable wrapOn

	set wrapOn 1
	destroy .options.expand
	toplevel .options.expand -class Toplevel
	wm transient .options.expand .options
	set x [expr [winfo pointerx .options] - 400]
	set y [expr [winfo pointery .options] - 100]
	wm geometry .options.expand 600x400+$x+$y
	wm title .options.expand "$option"
	::iwidgets::scrolledtext .options.expand.text -textbackground white -wrap word
	button .options.expand.bnOK -text {OK} -command "::options::cmdExpandOK $optionWidget"
	place .options.expand.bnOK -x 0 -y 0 -relx 0.6 -rely 1 -anchor s
	radiobutton .options.expand.rbWrap -text {Wrap} -value 1 -variable ::options::wrapOn \
	    -command {.options.expand.text configure -wrap word}
	radiobutton .options.expand.rbTruncate -text {Truncate} -value 0 \
	    -variable ::options::wrapOn \
	    -command {.options.expand.text configure -wrap none}
	button .options.expand.bnCancel -text {Cancel} -command {destroy .options.expand}
	place .options.expand.text -x 0 -y 0 -height -30 -relwidth 1 -relheight 1 -anchor nw
	place .options.expand.rbWrap -x 0 -y 0 -relx 0.2 -rely 1 -anchor s
	place .options.expand.rbTruncate -x 0 -y 0 -relx 0.4 -rely 1 -anchor s
	place .options.expand.bnCancel -x 0 -y 0 -relx 0.8 -rely 1 -anchor s
	.options.expand.text insert end [$optionWidget get]
	return
    }

    proc cmdExpandOK { optionWidget} {

	$optionWidget delete 0 end
	$optionWidget insert 0 [.options.expand.text get 1.0 "end -1 chars"]
	destroy .options.expand
	return
    }



    proc cmdOptionOK {} {

	variable pfmOptions
	variable newOptions
	variable ::pfm::currentDB
	destroy .options
	set optionList [array names newOptions]
	foreach option $optionList {
	    set pfmOptions($option) $newOptions($option)
	}
	saveOptions
	if {[info exists currentDB]} {
	    ::pfm::refreshFormsList
	}
	return
    }

    proc cmdOptionCancel {} {

	destroy .options
	return
    }

}

###############################################################
# End of namespace options                                    #
###############################################################


################################################################
#                                                              #
# Begin of namespace pfm                                       #
#                                                              #
################################################################

# widget is an associative array containing aliases for widget path names
#
# dbname is the name of the currently open data base. It is filled out
# by proc cmdOpenOK.
#
# hostname is the name or address of the host on which the currently
# opened data base resides. It is filled out by proc cmdOpenOK.
#
# currentDB contains the postgres data base handle for the open data base
# It is filled in by proc cmdOpenDataBase, and it is used throughout this
# application.
#
# formsArray contains all the tuples of pfm_form, where
# formsArray($name,$attribute) contains the value of attribute '$attribute'
# in the tuple of pfm_form for which name=$name. It is also filled by 
# refreshFormsList.
#
# formsIndex is an array for which formsIndex($n) contains the
# name of the form in the n-th form in the forms listbox. It is
# filled by refreshFormsList.
#


namespace eval pfm {

    variable currentDB
    variable formsArray
    variable formsIndex
    variable widget
    variable dbname
    variable hostname
    variable pfmMode {normal}

    proc initRootWindow { } {

	variable widget
	wm focusmodel . passive
	wm geometry . 300x300
	# wm maxsize . 785 570
	wm minsize . 1 1
	wm overrideredirect . 0
	wm resizable . 1 1
	wm deiconify .
	wm title . "pfm - No data base opened"
	frame .fmMenu \
		-borderwidth 2 -height 75 -relief raised -width 125 
	menubutton .fmMenu.mbFile \
		-menu .fmMenu.mbFile.m -text Database 
	menu .fmMenu.mbFile.m \
		-tearoff 0 
	.fmMenu.mbFile.m add command \
	    -command [namespace code {cmdOpenDataBase}] -label "Open ..."
	.fmMenu.mbFile.m add command \
		-command [namespace code cmdCloseDataBase] -label Close 
	.fmMenu.mbFile.m add command \
		-command [namespace code cmdExit] -label Exit 
	menubutton .fmMenu.mbReports \
	    -menu .fmMenu.mbReports.m -text "Reports/Queries"
	menu .fmMenu.mbReports.m -tearoff 0
	.fmMenu.mbReports.m add command \
	    -command {::report::cmdReportSQL sql} -label "Run SQL"
	.fmMenu.mbReports.m add command \
	    -command {::report::cmdReportSQL report} -label "Run report"
       	menubutton .fmMenu.mbTools \
		-menu .fmMenu.mbTools.m -text Tools
	menu .fmMenu.mbTools.m \
		-tearoff 0 
	.fmMenu.mbTools.m add command \
		-command ::options::cmdOptions -label "Options"
	.fmMenu.mbTools.m add command \
		-command [namespace code cmdInstall] -label {Install pfm_* tables} 
	menubutton .fmMenu.mbHelp \
		-menu .fmMenu.mbHelp.m -text Help 
	menu .fmMenu.mbHelp.m \
		-tearoff 0 
	.fmMenu.mbHelp.m add command \
		-command ::help::cmdDisplayManual -label {Help file} 
	.fmMenu.mbHelp.m add command \
	    -command ::help::cmdLicense -label {License} 
	.fmMenu.mbHelp.m add command \
		-command [namespace code cmdAbout] -label About 
	frame .frmMain \
		-borderwidth 2 -height 100 -relief groove -width 125 
	radiobutton .frmMain.rbNormal -text {Normal mode} -value {normal} \
	    -variable ::pfm::pfmMode -command ::pfm::refreshFormsList \
	    -state disabled
	radiobutton .frmMain.rbDesign -text {Design mode} -value {design} \
	    -variable ::pfm::pfmMode -command ::pfm::refreshFormsList \
	    -state disabled
	iwidgets::scrolledlistbox .frmMain.lsbForms -hscrollmode none \
	    -visibleitems 50x12 -labeltext "List of forms"
	set widget(lsbForms) .frmMain.lsbForms
	button .frmMain.btnOpen \
		-command {::pfm::cmdOpenQuery} -padx 0 -text {Open form} -state disabled
	###################
	# SETTING GEOMETRY
	###################
	place .fmMenu \
		-x 0 -y 0 -relwidth 1 -height 30 -anchor nw -bordermode ignore 
	pack .fmMenu.mbFile -side left
	pack .fmMenu.mbReports -side left
	pack .fmMenu.mbTools -side left
	pack .fmMenu.mbHelp -side right
	place .frmMain \
		-x 0 -y 30 -relwidth 1 -height -30 -relheight 1 -anchor nw \
		-bordermode ignore 
	place .frmMain.rbNormal -x 0 -y 0 -relx 0.25 -anchor n
	place .frmMain.rbDesign -x 0 -y 0 -relx 0.75 -anchor n
	place .frmMain.lsbForms -x 0 -y 30 -relwidth 1 -height -60 -relheight 1 -anchor nw
	place .frmMain.btnOpen -x 0 -y 0 -relx 0.5 -rely 1 -relwidth 1 -anchor s
	return
    }

    initRootWindow


    proc cmdAbout {} {

	global API
	global pfmVersion
	set aboutMsg \
"Postgres Forms (pfm) Version $pfmVersion
Copyright (C) Willem Herremans 2004

Postgres Forms comes with ABSOLUTELY NO WARRANTY; see 'Help -> License' for details.

This is free software, and you are welcome to redistribute it under certain conditions; see 'Help -> License' for details"
        set aboutMsg "$aboutMsg\n\n$API"
        tk_messageBox -message $aboutMsg -type ok -parent .
	return
    }


    proc cmdOpenDataBase { } {
	# Let the user specify a data base to open

	variable ::options::pfmOptions
	variable ::options::connInfoList
	variable currentDB
	variable dbname
	if { ![info exists currentDB]} then {
	    set dbname $pfmOptions(dbname)
	    destroy .opendb
	    toplevel .opendb -class Toplevel
	    wm transient .opendb .
	    set x [winfo pointerx .]
	    set y [winfo pointery .]
	    wm geometry .opendb 316x210+$x+$y
	    # wm maxsize .opendb 785 570
	    wm minsize .opendb 1 1
	    wm overrideredirect .opendb 0
	    wm resizable .opendb 1 1
	    wm title .opendb "pfm - Open data base"
	    set rowidx 0
	    foreach connItem $connInfoList {
		if { [info exists pfmOptions($connItem)] } then {
		    set connInfo $pfmOptions($connItem)
		} else {
		    set connInfo {}
		}
		label .opendb.lbl$connItem -text $connItem
		entry .opendb.val$connItem -width 30 -background white
		grid .opendb.lbl$connItem -in .opendb -row $rowidx -column 0 \
			-rowspan 1 -columnspan 1
		grid .opendb.val$connItem -in .opendb -row $rowidx -column 1 \
			-rowspan 1 -columnspan 1 -sticky {we}
		.opendb.val$connItem insert end $connInfo
		incr rowidx
	    }
	    label .opendb.lbldbname -text {dbname}
	    grid .opendb.lbldbname -in .opendb -row $rowidx -column 0 \
		    -rowspan 1 -columnspan 1
	    iwidgets::combobox .opendb.cbx -editable 0 \
		    -width 30 -textvariable ::pfm::dbname
	    foreach item $::options::pfmOptions(dblist) {
		.opendb.cbx insert list end $item
	    }
	    grid .opendb.cbx -in .opendb -row $rowidx -column 1 \
		    -rowspan 1 -columnspan 1
	    button .opendb.btnOK -text OK -command {::pfm::cmdOpenOK}
	    button .opendb.btnCancel -text Cancel -command {::pfm::cmdOpenCancel}
	    place .opendb.btnOK -relx 0.3 -rely 1 -y {-5} -anchor s
	    place .opendb.btnCancel -relx 0.6 -rely 1 -y {-5} -anchor s
	} else {
	    tk_messageBox -message "First close data base $dbname" -type ok
	}
	return
    }

    proc cmdOpenOK {} {

	variable ::options::connInfoList
	variable currentDB
	variable dbname
	variable hostname

	set connInfo "dbname=$dbname "
	foreach connItem $connInfoList {
	    set connItemValue [.opendb.val$connItem get]
	    if { ![string equal $connItemValue {}] } then {
		set connInfo "$connInfo $connItem=$connItemValue"
	    }
	    if { [string equal $connItem {hostaddr}] } then {
		set hostname $connItemValue
	    }
	}
	if { [catch {set currentDB [pg_connect -conninfo $connInfo]} errorMsg]} then {
	    tk_messageBox -message $errorMsg -type ok
	} else {
	    refreshFormsList
	    wm title . "pfm - Database : $dbname"
	    .frmMain.btnOpen configure -state normal
	    .frmMain.rbNormal configure -state normal
	    .frmMain.rbDesign configure -state normal
	}
	destroy .opendb
	return
    }

    proc cmdOpenCancel {} {

	destroy .opendb
	return
    }

    proc cmdCloseDataBase {} {
	#Close data base that is currently open

	variable currentDB
	variable dbname
	variable formsArray
	variable widget
	pg_disconnect $currentDB
	unset currentDB
	array unset formsArray
	$widget(lsbForms) clear
	$widget(lsbForms) see 0
	wm title . "pfm - No data base opened"
	.frmMain.btnOpen configure -state disabled
	.frmMain.rbNormal configure -state disabled
	.frmMain.rbDesign configure -state disabled
	destroy .form
	destroy .query
	destroy .report
	destroy .install
	return
    }

    proc cmdExit {} {

	variable currentDB
	if {[info exists currentDB]} {
	    cmdCloseDataBase
	}
	exit
    }

    proc refreshFormsList {} {

	variable currentDB
	variable ::options::pfmOptions
	variable formsArray
	variable formsIndex
	variable widget
	variable pfmMode

	array unset formsArray
	array unset formsIndex
	switch $pfmMode {
	    design {
		set sqlwhere "WHERE showform='f'"
	    }
	    normal -
	    default {
		set sqlwhere "WHERE showform='t'"
	    }
	}
	set formQuery "SELECT name,tablename,showform,view,sqlselect,sqlfrom,groupby_having"
	set formQuery "$formQuery FROM pfm_form $sqlwhere ORDER BY name"
	set resQuery [pg_exec $currentDB $formQuery]
	pg_result $resQuery -assignbyidx formsArray
	set lastTuple [expr [pg_result $resQuery -numTuples] -1]
	$widget(lsbForms) clear
	set listIndex 0
	for {set tupleNr 0} {$tupleNr <= $lastTuple} {incr tupleNr} {
	    set tuple [pg_result $resQuery -getTuple $tupleNr]
	    $widget(lsbForms) insert end [lindex $tuple 0]
	    set formsIndex($listIndex) [lindex $tuple 0]
	    incr listIndex
	}
	pg_result $resQuery -clear
	$widget(lsbForms) selection clear 0 end
	$widget(lsbForms) selection set 0 0
	$widget(lsbForms) see 0
	return
    }


    proc cmdInstall {} {

	global installDir
	variable currentDB
	variable dbname
	if { [info exists currentDB] } then {
	    if { [NotInstalledYet] } then {
		destroy .install
		toplevel .install -class Toplevel
		wm focusmodel .install passive
		wm geometry .install 600x400
		# wm maxsize .install 785 570
		wm minsize .install 1 1
		wm overrideredirect .install 0
		wm resizable .install 1 1
		wm title .install "pfm - Install forms"
		text .install.txtLog 
		button .install.btnQuit -text {Quit} -command [namespace code {
		    destroy .install
		    ::pfm::refreshFormsList
		}]
		place .install.btnQuit -relx 0.5 -rely 1 -y {-5} -anchor s -width 60
		place .install.txtLog -x 0 -y 0 -relwidth 1 -relheight 1 -height {-40}
		set Pipe "|psql -e $dbname < $installDir/install_pfm.sql |& cat"
		if { [catch {open $Pipe r} ch_psql] } then {
		    .install.txtLog insert end $ch_psql
		    .install.txtLog see end
		} else {
		    fileevent $ch_psql readable "::pfm::Log $ch_psql .install.txtLog"
		}
	    } else {
		tk_messageBox -message \
			{Some pfm_* tables are already installed in this data base. First drop all pfm_* tables if you want to re-install them.} -type ok
	    }
	} else {
	    tk_messageBox -message {First open a data base!} -type ok	
	}
	return
    }

    proc Log {channel txtwidget} {
	# This procedure is called any time the channel becomes readable.
	# See fileevent in cmdInstall procedure

	if { [eof $channel] } then {
	    if { [catch {close $channel} errmsg] } then {
		$txtwidget insert end $errmsg\n
		$txtwidget see end
	    }
	} else {
	    gets $channel line
	    $txtwidget insert end $line\n
	    $txtwidget see end
	}
	return
    }

    proc NotInstalledYet { } {

	variable currentDB
	set queryDef {
	    SELECT COUNT(*) AS pfm_exists FROM pg_tables WHERE
	    tablename IN ('pfm_form','pfm_attribute','pfm_value','pfm_value_list',
			  'pfm_link','pfm_report','pfm_section')
	}
	set queryRes [pg_exec $currentDB $queryDef]
	set nrInstalled [lindex [pg_result $queryRes -getTuple 0] 0]
	pg_result $queryRes -clear
	return [expr $nrInstalled == 0]
    }

    proc cmdOpenQuery {} {
	variable formsIndex
	variable widget
	set formNr [$widget(lsbForms) curselection]
        if { $formNr == {} } then {
            set formNr 0
        }
        set formName $formsIndex($formNr)
	::form::OpenQuery $formName
	return
    }

}

#############################################################################
# end of namespace pfm                                                      #
#############################################################################

#############################################################################
# begin of namespace help                                                   #
#############################################################################
#
# searchPosition is the index in .help.html widget from where
# text searches start.

namespace eval help {

    variable searchPosition

    proc cmdDisplayManual {} {

	global installDir
	variable searchPosition
	if { [catch { open $installDir/doc/help.html } help_ch] } then {
	    fileNotFound {help} $help_ch
	} else {
	    set searchPosition 1.0
	    destroy .help
	    toplevel .help -class Toplevel
	    wm focusmodel .help passive
	    wm geometry .help 750x550
	    wm minsize .help 1 1
	    wm overrideredirect .help 0
	    wm resizable .help 1 1
	    wm title .help "pfm - Help file"
	    ::iwidgets::scrolledhtml .help.html -labeltext "Help file" -wrap none \
		-fontname helvetica -fontsize large -textbackground white
	    bind .help <KeyPress-Next> {.help.html yview scroll 1 pages}
	    bind .help <KeyPress-Prior> {.help.html yview scroll -1 pages}
	    bind .help <KeyPress-Down> {.help.html yview scroll 1 units}
	    bind .help <KeyPress-Up> {.help.html yview scroll -1 units}
	    button .help.btnQuit -text {Quit} -command {destroy .help}
	    set pattern {}
	    entry .help.entSearch -text {} -width 40 -background white
	    button .help.btnSearch -text {Search} -command {::help::cmdSearch}
	    button .help.btnTop -text {Top} -command {::help::cmdTop}    
	    place .help.html -x 0 -y 0 -relwidth 1 -relheight 1 -height -30
	    place .help.btnQuit -relx 1 -rely 1 -y {-15} -anchor e
	    place .help.entSearch -relx 0 -rely 1 -y {-15} -anchor w
	    place .help.btnSearch -x 300 -rely 1 -y {-15} -anchor w
	    place .help.btnTop -x 380 -rely 1 -y {-15} -anchor w
	    set help_text [read $help_ch]
	    close $help_ch
	    catch {.help.html render $help_text "$installDir/doc"}
	}
	return
    }

    proc cmdSearch {} {

	variable searchPosition
	set pattern [.help.entSearch get]
	.help.html tag delete match
	set searchPosition [.help.html search -nocase $pattern $searchPosition end]
	if { ![string equal $searchPosition {}] } then {
	    set endmatch [.help.html index "$searchPosition + [string length $pattern] chars"]
	    .help.html tag add match $searchPosition $endmatch
	    .help.html tag configure match -background yellow
	    .help.html see $searchPosition
	    set searchPosition $endmatch
	} else {
	    set searchPosition 1.0
	    .help.html see $searchPosition
	}
	return
    }

    proc cmdTop {} {

	variable searchPosition
	set searchPosition 1.0
	.help.html see $searchPosition
	return
    }

    proc cmdLicense {} {

	global installDir
	if { [catch { open $installDir/doc/gpl.txt } license_ch] } then {
	    fileNotFound {license} $license_ch
	} else {
	    toplevel .license -class Toplevel
	    wm geometry .license 600x600
	    wm title .license "pfm - License"
	    iwidgets::scrolledtext .license.txt -hscrollmode none -textbackground white \
		-labeltext {GNU GENERAL PUBLIC LICENSE - Version 2, June 1991}
	    button .license.bnOK -command {destroy .license} -text {OK}
	    place .license.txt -x 0 -y 0 -relwidth 1 -relheight 1 -height {-30}
	    place .license.bnOK -x 0 -y 0 -relx 0.5 -rely 1 -anchor s
	    set license_ch [open $installDir/doc/gpl.txt]
	    set license_text [read $license_ch]
	    close $license_ch
	    .license.txt insert end $license_text
	}
	return
    }

    proc fileNotFound {what reason} {

	switch $what {
	    help {
		set msgText "Normally you should see the help file now, but $reason"
	    }
	    license {
		set msgText "Normally you should see the GNU General Public License now, but $reason"
	    }
	    default {
		set msgText $reason
	    }
	}
	tk_messageBox -message $msgText -type ok
	return
    }

}
######################################################################
# End of namespace help                                              #
######################################################################

######################################################################
# Begin of namespace form                                            #
######################################################################

# activeForm contains the name of the currently opened form.
# It is filled by several procedures:
#     1. OpenQuery
#     2. cmdFollowLink
#     3. cmdBack
#
# formAttribList contains the list of attributes of the active form,
# as defined in data base table pfm_attributes.
#
# attributeArray contains the complete attributes defintion of the
# active form as defined by pfm_attributes.
#
# tableAttribList contains only the attributes that are not "tgReadOnly".
# The purpose is to include only the attributes of the table referred
# to by pfm_form.tablename.
#
# formAttribList, tableAttribList and attributeArray are filled by 
# proc getAttributes.
#
# getAttributes is called by:
#     1. OpenQuery
#     2. cmdFollowLink
#     3. cmdBack
#
# WhereSelected is a boolean which indicates whether the user is
# pasting into the "where" or into the "order by" text entry
# of the query. Its value is controlled by the radio buttons
# on the query window.
#
# recordArray contains all the records selected by the query defined
# by the user in the query window. recordArray($tupleNr,$attribute)
# indicates the value of $attribute of $tupleNr. It is filled
# by proc OpenForm. It is the so called "internal buffer".
# On top of the attribute values, recordArray also contains
# a status for each record: recordArray($tupleNr,23status47)
# contains the status the record indicated by $tupleNr. The
# status can be : "Not Modified", "Updated", "Deleted", "Added",
# "After Last". The attribute name "23status47" has been chosen
# to avoid name conflicts with real table attributes.
#
# lastRecord is the tupleNr of the last tuple in recordArray. This
# is in fact a dummy, empty record, functioning as a sentinel.
#
# curRecord is the tupleNr of the record that is currently displayed
# on the screen.
#
# txtRecord is the textvar linked to the attribute labels,
# entries or buttons on the form. When the current record
# is displayed, the values of recordArray($curRecord,$attribute)
# are copied to txtRecord($attribute).
#
# formStack contains the subsequent queries issued by the user
# as a result of following links. lastFormOnStack is a stack
# pointer on this stack. The first query is pushed on the stack
# by proc cmdExecuteQuery. If the user clicks on a link button
# another query is pushed on the stack by proc cmdFollowLink.
# Any time the user presses back, a query defintion is popped
# of the stack. The elements that are kept on the stack are:
#     formId
#     queryDef
#     showQuery (the information displayed on top of the form)
#     displayOid (the oid of the record that was displayed at
#                 the time a link button was pressed and which
#                 is displayed again when the user presses "Back")
#
# linksArray is loaded with all the links originating from the
# active form. It is filled by proc displayLinks, which is called
# from OpenForm. Its structure is linksArray($link,$attribute)
# where $link is an index for the link (starting from 0) and
# where $attribute is any attribute of pfm_link.
#
# widget is an array containing aliases for some widget
# path names
#

namespace eval form {
    variable activeForm
    variable formState
    variable formAttribList
    variable tableAttribList
    variable attributeArray
    variable WhereSelected
    variable lastRecord
    variable recordArray
    variable curRecord
    variable txtRecord
    variable formStack
    variable lastFormOnStack
    variable linksArray
    variable widget

    #############################################################
    # Procedures that are called from the corresponding cmdXXX  #
    # procedures in namespace pfm                               #    
    #############################################################


    proc OpenQuery {formName} {

	variable ::pfm::formsArray
	variable activeForm
	variable WhereSelected
	variable formAttribList
	variable attributeArray
	variable widget
	set WhereSelected true
	set activeForm $formName
	showQueryWindow
	wm title .query "pfm - Query : $formName"
	$widget(lbSelect) insert end \
	    "SELECT $formsArray($activeForm,sqlselect)\nFROM $formsArray($activeForm,sqlfrom)"
	if { ![string equal $formsArray($activeForm,groupby_having) {}] } then {
	    $widget(lbSelect) insert end \
		"\nGROUP BY $formsArray($activeForm,groupby_having)"
	}
	getAttributes $activeForm
	label $widget(attributes).lblAttributes -text {Attributes} -relief flat
	grid $widget(attributes).lblAttributes -in $widget(attributes) -column 0 \
		-row 0 -columnspan 1 -rowspan 1
	label $widget(attributes).lblValues -text {Values} -relief flat -width 40
	grid $widget(attributes).lblValues -in $widget(attributes) -column 1 \
		-row 0 -columnspan 1 -rowspan 1
	set rowidx 1
	foreach attribute $formAttribList {
	    button $widget(attributes).btn$attribute -text $attribute -pady 0 -anchor w \
		    -command "::form::cmdPasteAttribute $attribute"
	    grid $widget(attributes).btn$attribute -in $widget(attributes) -column 0 \
		    -row $rowidx -columnspan 1 -rowspan 1 -sticky we
	    switch $attributeArray($attribute,typeofget) {
		"tgDirect" -
		"tgReadOnly" -
		"tgExpression" { }
		"tgList" -
		"tgLink" {
		    button $widget(attributes).ent$attribute -pady 0 \
			-text {Select and paste value} -relief raised -command \
			"::form::cmdSelectFromList $attribute .query paste"
		    grid $widget(attributes).ent$attribute -in $widget(attributes) -column 1 \
			    -row $rowidx -columnspan 1 -rowspan 1
		}
	    }
	    incr rowidx
	}
	return
    }


    ################################################################
    #                                                              #
    # Procedures for query window                                  #
    #                                                              #
    ################################################################

    proc showQueryWindow { } {

	variable widget
	destroy .query
	destroy .form
	toplevel .query -class Toplevel
	wm focusmodel .query passive
	wm geometry .query 750x550
	# wm maxsize .query 785 570
	wm minsize .query 1 1
	wm overrideredirect .query 0
	wm resizable .query 1 1
	iwidgets::scrolledtext .query.lbSelect -textfont {Helvetica 12 {}} -wrap word \
	    -hscrollmode none -height 75
	set widget(lbSelect) .query.lbSelect
	radiobutton .query.rabWhere  -font {Helvetica 12 {}} \
		-text WHERE -value true  -variable ::form::WhereSelected 
	entry .query.entWhere -background white
	radiobutton .query.rabOrderBy  -font {Helvetica 12 {}} \
		-text {ORDER BY} -value false  -variable ::form::WhereSelected 
	entry .query.entOrderBy -background white
	label .query.lbAttributes  -borderwidth 1 -text {Paste buttons} -font {Helvetica 12}
	::iwidgets::scrolledframe .query.frmAttributes -borderwidth 2 -height 75 \
		-relief sunken -width 125
	set widget(attributes) [.query.frmAttributes childsite]
	button .query.btnExecute  -command \
		{::form::cmdExecuteQuery [.query.entWhere get] [.query.entOrderBy get] \
		    "User defined query:"} -text Execute 
	button .query.btnQuit -command ::form::cmdQuitQuery -text Quit 
	###################
	# SETTING GEOMETRY
	###################
	place .query.lbSelect  -x 0 -y 0 -relwidth 1 -height {-65} -relheight 0.5 -anchor nw
	place .query.rabWhere  -x 0 -y {-50} -rely 0.5 -anchor w
	place .query.entWhere  -x 105 -y {-50} -rely 0.5 -width {-105} -relwidth 1 \
	    -height 22 -anchor w
	place .query.rabOrderBy  -x 0 -y {-25} -rely 0.5  -anchor w
	place .query.entOrderBy  -x 105 -y {-25} -rely 0.5 -width {-105} -relwidth 1 \
	    -height 22 -anchor w
	place .query.lbAttributes  -x 0 -y 0 -rely 0.7  -anchor w
	place .query.frmAttributes  -x 105 -y {-10} -rely 0.5 -width {-105} -relwidth 1 \
	    -height {-25} -relheight 0.5 -anchor nw
	place .query.btnExecute  -x {-65} -y 0 -relx 1 -rely 1 -anchor se
	place .query.btnQuit  -x {-10} -y 0 -relx 1 -rely 1 -anchor se
	return
    }


    proc cmdExecuteQuery {sqlWhere sqlOrderBy Intro} {
	# This procedure prepares the SQL SELECT statement of the
	# query to be made, it initialises the formStack and
	# it calls OpenForm. It is called when the user presses
	# the Execute button on the query window.

	variable ::pfm::formsArray
	variable activeForm
	variable formStack
	variable lastFormOnStack
	set sqlAttrib $formsArray($activeForm,sqlselect)
	set sqlFrom $formsArray($activeForm,sqlfrom)
	set groupby_having $formsArray($activeForm,groupby_having)
	set queryDef "SELECT $sqlAttrib FROM $sqlFrom"
	set showQuery "$Intro\nSELECT $sqlAttrib\nFROM $sqlFrom"
	if { ![string equal $sqlWhere {}] } then {
	    set queryDef "$queryDef WHERE $sqlWhere"
	    set showQuery "$showQuery\nWHERE $sqlWhere"
	}
	if { ![string equal $groupby_having {}] } then {
	    set queryDef "$queryDef GROUP BY $groupby_having"
	    set showQuery "$showQuery\nGROUP BY $groupby_having"
	}
	if { ![string equal $sqlOrderBy {}] } then {
	    set queryDef "$queryDef ORDER BY $sqlOrderBy"
	    set showQuery "$showQuery\nORDER BY $sqlOrderBy"
	}
	array unset formStack
	set lastFormOnStack 0
	set formStack($lastFormOnStack,formId) $activeForm
	set formStack($lastFormOnStack,queryDef) $queryDef
	set formStack($lastFormOnStack,showQuery) $showQuery
	set formStack($lastFormOnStack,displayOid) 0
	destroy .query
	OpenForm $queryDef $showQuery 0
	return
    }

    proc cmdPasteAttribute {attribute} {
	# The user pastes an attribute name into either the "where" or
	# "order by" text entries, depending on WhereSelect, the value
	# of which is determined by the radio buttons .query.rabWhere
	# and .query.rabOrderBy

	variable WhereSelected
	if {$WhereSelected} then {
	    .query.entWhere insert insert \"$attribute\"
	} else {
	    .query.entOrderBy insert insert \"$attribute\"
	}
	return
    }


    proc cmdQuitQuery {} {

	destroy .query
	::pfm::refreshFormsList
	return
    }



    ################################################################
    #                                                              #
    # Procedures for form window                                   #
    #                                                              #
    ################################################################

    proc showFormWindow { } {

	variable widget
	destroy .query
	if { [winfo exists .form] } then {
	    destroy .form.frmRecord
	    destroy .form.frmLinkBtn
	    ::iwidgets::scrolledframe .form.frmRecord \
		    -borderwidth 2 -height 75 -relief sunken -width 125
	    set widget(record) [.form.frmRecord childsite]
	    place .form.frmRecord \
		    -x 0 -y 30 -rely 0.2 -relwidth 0.8 -height -60 -relheight 0.6 -anchor nw \
		    -bordermode ignore 
	    frame .form.frmLinkBtn \
		    -borderwidth 2 -height 75 -relief sunken -width 125 
	    place .form.frmLinkBtn \
		    -x 0 -relx 0.8 -y 30 -rely 0.2 -relwidth 0.2 -height -30 -relheight 0.6 \
		    -anchor nw -bordermode ignore	
	    grid columnconfigure .form.frmRecord 0 -weight 1
	    grid columnconfigure .form.frmRecord 1 -weight 2
	    .form.txtQuery delete 1.0 end
	    .form.txtResult delete 1.0 end
	} else {
	    toplevel .form -class Toplevel
	    wm focusmodel .form passive
	    wm geometry .form 750x550
	    # wm maxsize .form 785 570
	    wm minsize .form 1 1
	    wm overrideredirect .form 0
	    wm resizable .form 1 1
	    iwidgets::scrolledtext .form.txtQuery -textfont {Helvetica 12} \
		-wrap word -hscrollmode none
	    iwidgets::scrolledtext .form.txtResult -textfont {Helvetica 12} \
		-wrap word -hscrollmode none
	    frame .form.frmStatus \
		    -borderwidth 2 -height 75 -relief groove -width 125 
	    label .form.frmStatus.lblRecord \
		    -borderwidth 1 -textvar ::form::txtRecord(23nr47) 
	    label .form.frmStatus.lblFormName \
		    -borderwidth 1 -textvar ::form::activeForm
	    label .form.frmStatus.lblStatus \
		    -borderwidth 1 -textvar ::form::txtRecord(23status47)
	    ::iwidgets::scrolledframe .form.frmRecord \
		    -borderwidth 2 -height 75 -relief sunken -width 125
	    set widget(record) [.form.frmRecord childsite]
	    frame .form.frmButtons \
		    -borderwidth 2 -height 75 -relief groove -width 125 
	    button .form.frmButtons.btnHelp \
		-text Help -command ::form::cmdHelp
	    button .form.frmButtons.btnPrev \
		-text Prev -command ::form::cmdPrev
	    button .form.frmButtons.btnNext \
		-text Next -command ::form::cmdNext
	    button .form.frmButtons.btnUpdate \
		-text Update -command ::form::cmdUpdate
	    button .form.frmButtons.btnAdd \
		-text Add -command ::form::cmdAdd
	    button .form.frmButtons.btnDelete \
		-text Delete -command ::form::cmdDelete
	    button .form.frmButtons.btnQuit \
		-command {::form::cmdQuitForm} -text Quit
	    button .form.frmButtons.btnOK \
		-text OK -command ::form::cmdOK
	    button .form.frmButtons.btnCancel \
		-text Cancel -command ::form::cmdCancel
	    frame .form.frmLink1 \
		-borderwidth 2 -height 75 -relief groove -width 125 
	    label .form.frmLink1.lblLinks \
		-borderwidth 1 -text Links 
	    frame .form.frmLinkBtn \
		-borderwidth 2 -height 75 -relief sunken -width 125 
	    ###################
	    # SETTING GEOMETRY
	    ###################
	    place .form.txtQuery \
		    -x 0 -y 0 -width 0 -relwidth 1 -height 0 -relheight 0.2 -anchor nw \
		    -bordermode ignore 
	    place .form.txtResult \
		    -x 0 -y 0 -rely 1 -width 0 -relwidth 1 -height 0 -relheight 0.2 -anchor sw \
		    -bordermode ignore 
	    place .form.frmStatus \
		    -x 0 -y 0 -rely 0.2  -width 0 -relwidth 0.8 -height 30 -anchor nw \
		    -bordermode ignore 
	    place .form.frmLink1 \
		    -x 0 -relx 0.8 -y 0 -rely 0.2 -relwidth 0.2 -height 30 -anchor nw \
		    -bordermode ignore 
	    place .form.frmRecord \
		    -x 0 -y 30 -rely 0.2 -relwidth 0.8 -height -60 -relheight 0.6 -anchor nw \
		    -bordermode ignore 
	    place .form.frmLinkBtn \
		    -x 0 -relx 0.8 -y 30 -rely 0.2 -relwidth 0.2 -height -30 -relheight 0.6 \
		    -anchor nw -bordermode ignore 
	    place .form.frmButtons \
		    -x 0 -y 0 -rely 0.8 -relwidth 0.8 -height 30 -anchor sw \
		    -bordermode ignore 
	    place .form.frmStatus.lblRecord \
		    -x 10 -y 0 -rely 0.5 -anchor w -bordermode ignore 
	    place .form.frmStatus.lblFormName \
		    -x 0 -y 0 -rely 0.5 -relx 0.5 -anchor center -bordermode ignore 
	    place .form.frmStatus.lblStatus \
		    -x -10 -relx 1 -y 0 -rely 0.5 -anchor e -bordermode ignore 
	    place .form.frmLink1.lblLinks \
		    -x 0 -relx 0.5 -y 0 -rely 0.5 -anchor center -bordermode ignore 
	    place .form.frmButtons.btnHelp -x 0 -y 0 -rely 0.5 -anchor w
	    grid columnconfigure .form.frmRecord 0 -weight 1
	    grid columnconfigure .form.frmRecord 1 -weight 2
	}
	return
    }

    ################################################################
    # Commands for browse, edit, add and quit buttons              #
    ################################################################

    proc cmdHelp {} {

	variable wrapOn
	variable ::pfm::currentDB
	variable activeForm

	set queryDef "SELECT help FROM pfm_form WHERE name='$activeForm'"
	set queryRes [pg_exec $currentDB $queryDef]
	set helpText [lindex [pg_result $queryRes -getTuple 0] 0]
	if { [string length $helpText] == 0 } then {
	    set helpText "No help available for $activeForm."
	}
	pg_result $queryRes -clear
	set wrapOn 1
	destroy .form.help
	toplevel .form.help -class Toplevel
	wm transient .form.help .form
	#	set x [expr [winfo pointerx .form] - 400]
	#	set y [expr [winfo pointery .form] - 100]
	wm geometry .form.help 600x400
	wm title .form.help "Help for $activeForm"
	::iwidgets::scrolledtext .form.help.text -wrap word -textbackground white
	radiobutton .form.help.rbWrap -text {Wrap} -value 1 -variable ::form::wrapOn \
	    -command {.form.help.text configure -wrap word}
	radiobutton .form.help.rbTruncate -text {Truncate} -value 0 -variable ::form::wrapOn \
	    -command {.form.help.text configure -wrap none}
	button .form.help.bnCancel -text {Cancel} -command {destroy .form.help}
	place .form.help.text -x 0 -y 0 -height -30 -relwidth 1 -relheight 1 -anchor nw
	place .form.help.rbWrap -x 0 -y 0 -relx 0.2 -rely 1 -anchor s
	place .form.help.rbTruncate -x 0 -y 0 -relx 0.4 -rely 1 -anchor s
	place .form.help.bnCancel -x 0 -y 0 -relx 0.8 -rely 1 -anchor s
	.form.help.text insert end $helpText
	.form.help.text configure -state disabled
	return
    }

    proc cmdAdd {} {

	.form.txtResult delete 1.0 end
	newFormState add
	return
    }

    proc cmdUpdate {} {

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable txtRecord
	variable recordArray
	variable curRecord
	variable tableAttribList
	variable activeForm
	# Bug 679 : The start of the transaction is postponed until
	# the user presses [OK]. Instead a 'reload record' is executed
	# to minimize the time window during which another user can
	# modify or delete the current record.

	if { ![string equal $txtRecord(23status47) {After Last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    .form.txtResult delete 1.0 end
	    if { [reloadRecord] } then {
		newFormState update   
	    }	
	}
	return
    }


    proc cmdDelete {} {

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable curRecord
	variable activeForm
	variable txtRecord
	variable recordArray
	variable formAttribList
	if { ![string equal $txtRecord(23status47) {After Last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    set queryDef "DELETE FROM \"$formsArray($activeForm,tablename)\" WHERE oid = $recordArray($curRecord,oid)"
	    set queryRes [pg_exec $currentDB $queryDef]
	    set status [pg_result $queryRes -status]
	    if { [string equal $status {PGRES_COMMAND_OK}] } then {
		set recordArray($curRecord,23status47) "Deleted"
		set txtRecord(23status47) "Deleted"
		foreach attribute $formAttribList {
		    set recordArray($curRecord,$attribute) ""
		    set txtRecord($attribute) ""
		}	
	    } else {
		set status $status\n[pg_result $queryRes -error]
		bell
	    }
	    pg_result $queryRes -clear
	    .form.txtResult delete 1.0 end
	    .form.txtResult insert end $queryDef\n$status
	}
	return
    }

    proc cmdNext {} {

	variable curRecord
	variable lastRecord
	if {$curRecord < $lastRecord} then {
	    incr curRecord
	    filltxtRecord $curRecord
	    .form.txtResult delete 1.0 end
	}
	return
    }

    proc cmdPrev {} {

	variable curRecord
	if {$curRecord > 0} then {
	    incr curRecord -1
	    filltxtRecord $curRecord
	    .form.txtResult delete 1.0 end
	}
	return
    }

    proc cmdQuitForm {} {

	destroy .form
	::pfm::refreshFormsList
	return
    }



    ################################################################
    # Commands for OK and Cancel buttons                           #
    ################################################################

    proc cmdOK {} {

	variable ::pfm::currentDB
	variable formState

	# reworked because of bug 679
	.form.txtResult delete 1.0 end
	switch $formState {
	    "update" {
		set result [updateRecord]
	    }
	    "add" {
		set result [addRecord]
	    }
	}
	.form.txtResult insert end "$result\n"
	reloadRecord
	newFormState browse
	return
    }

    proc cmdCancel {} {

	variable ::pfm::currentDB
	variable curRecord
	variable formState

	# reworked because of bug 679
	filltxtRecord $curRecord
	if { [string equal $formState {update}] } then {
	    set rollbackStatus "Update cancelled."
	} else {
	    set rollbackStatus "No record inserted."
	}
	.form.txtResult delete 1.0 end
	.form.txtResult insert end $rollbackStatus
	newFormState browse
	return
    }

    ################################################################
    # Commands for selecting a value from a list                   #
    ################################################################

    
    proc cmdSelectFromList {attribute base action} {

	variable ::pfm::currentDB
	variable activeForm
	variable attributeArray
	variable displayList
	variable searchString

	set searchString {}
	set xy [winfo pointerxy $base]
	set x [lindex $xy 0]
	set y [lindex $xy 1]
	set window $base.select
	destroy $window
	toplevel $window -class Toplevel
	wm transient $window $base
	wm geometry $window 300x300+$x+$y
	wm minsize $window 1 1
	wm overrideredirect $window 0
	wm resizable $window 1 1
	wm title $window "pfm - Select value for $attribute"
	iwidgets::scrolledlistbox $window.lsb
	switch $attributeArray($attribute,typeofget) {
	    "tgList" {
		set queryDef "SELECT value,description FROM pfm_value"
		set queryDef "$queryDef WHERE valuelist='$attributeArray($attribute,valuelist)'"
	    }
	    "tgLink" {
		set queryDef $attributeArray($attribute,sqlselect)
	    }
	}
	set queryRes [pg_exec $currentDB $queryDef]
	set lastItem [expr [pg_result $queryRes -numTuples] - 1]
	set valueList {}
	set displayList {}
	for {set item 0} {$item <= $lastItem} {incr item } {
	    set listItem [pg_result $queryRes -getTuple $item]
	    lappend valueList [lindex $listItem 0]
	    set displayItem {}
	    set itemNr 0
	    foreach subItem $listItem {
		set displayItem "$displayItem $subItem"
		if {$itemNr == 0} then {
		    set displayItem "$displayItem :"
		}
		incr itemNr
	    }
	    lappend displayList $displayItem
	    $window.lsb insert end $displayItem
	}
	pg_result $queryRes -clear
	$window.lsb configure -selectioncommand \
		"::form::cmdValueSelected \{$valueList\} $attribute $window $action"
	$window.lsb selection set 0 0
	label $window.lblSearch -text {Search for:}
	entry $window.entSearch -textvariable ::form::searchString -background white
	button $window.btnSearch -text {Go} -command \
	    "::form::cmdSearchInList $window"
	button $window.btnOK -text {OK} -command \
	    "::form::cmdValueSelected \{$valueList\} $attribute $window $action"
	button $window.btnCancel -text {Cancel} -command "destroy $window"
	place $window.lblSearch -x 0 -y 0 -height 30 -width 75
	place $window.entSearch -x 75 -y 0 -height 30 -width {-115} -relwidth 1
	place $window.btnSearch -x {-40} -y 0 -relx 1 -height 30 -width 40
	place $window.lsb -x 0 -y 30 -height {-60} -relheight 1 -relwidth 1
	place $window.btnOK -x 0 -y 0 -rely 1 -relwidth 0.5 -anchor sw
	place $window.btnCancel -x 0 -y 0 -rely 1 -relx 0.5 -relwidth 0.5 -anchor sw
	return
    }

    proc cmdValueSelected {valueList attribute base action} {

	variable txtRecord
	variable WhereSelected
	set item [$base.lsb curselection]
	switch $action {
	    "fillout" {
		set txtRecord($attribute) [lindex $valueList $item]
	    }
	    "paste" {
		if {$WhereSelected} then {
		    .query.entWhere insert insert [lindex $valueList $item]
		} else {
		    .query.entOrderBy insert insert [lindex $valueList $item]
		}
	    }
	}
	destroy $base
	return
    }

    proc cmdSearchInList {window} {
	variable displayList
	variable searchString
	
	set startPosition [expr [lindex [$window.lsb curselection] 0] + 1]
	if { $startPosition >= [llength $displayList] } then {
	    $window.lsb selection clear 0 end
	    $window.lsb selection set 0 0
	    $window.lsb see 0
	} else {
	    set newPosition [lsearch -glob -start $startPosition $displayList "*$searchString*"]
	    if { $newPosition >= 0 } then {
		$window.lsb selection clear 0 end
		$window.lsb selection set $newPosition $newPosition
		$window.lsb see $newPosition
	    } else {
		$window.lsb selection clear 0 end
		$window.lsb selection set 0 0
		$window.lsb see 0
	    }
	}
	return
    }

    ###############################################################################
    # Procedures for opening the active form with a defined query                 #
    ###############################################################################

    proc OpenForm {queryDef showQuery displayOid} {

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable activeForm
	variable lastRecord
	variable recordArray
	variable curRecord
	variable formAttribList
	showFormWindow
	wm title .form "pfm - Form : $activeForm"
	.form.txtQuery insert end $showQuery
	.form.txtQuery tag add firstLine 1.0 1.end
	.form.txtQuery tag configure firstLine -font {Helvetica 12 bold}
	set queryRes [pg_exec $currentDB $queryDef]
	# lastRecord is a dummy empty record.
	set lastRecord [pg_result $queryRes -numTuples]
	array unset recordArray
	pg_result $queryRes -assign recordArray
	set status [pg_result $queryRes -status]
	if { ![string equal $status {PGRES_TUPLES_OK}] } then {
	    set status $status\n[pg_result $queryRes -error]
	}
	pg_result $queryRes -clear
	.form.txtQuery insert end \n$status
	for {set recordNr 0} {$recordNr < $lastRecord} {incr recordNr} {
	    set recordArray($recordNr,23status47) "Not Modified"
	}
	foreach attribute $formAttribList {
	    set recordArray($lastRecord,$attribute) {}
	}
	set recordArray($lastRecord,23status47) "After Last"
	if {$displayOid == 0} then {
	    set curRecord 0
	} else {
	    set curRecord 0
	    for {set recordNr 0} {$recordNr < $lastRecord} {incr recordNr} {
		if {$recordArray($recordNr,oid) == $displayOid} then {
		    set curRecord $recordNr
		}
	    }
	}
	hideBrowseButtons
	hideEditButtons
	displayBrowseButtons
	displayAttribNames
	filltxtRecord $curRecord
	displayAttribLabels
	displayLinks
	return
    }

    proc displayAttribNames {} {

	variable formAttribList
	variable widget
	set rowidx 0
	foreach attribute $formAttribList {
	    label $widget(record).lb1$attribute -borderwidth 1 -text $attribute
	    grid $widget(record).lb1$attribute -in $widget(record) -column 0 -row $rowidx \
		    -columnspan 1 -rowspan 1 -pady 2
	    incr rowidx
	}
	return
    }


    proc displayAttribLabels {} {

	variable formAttribList
	variable txtRecord
	variable attributeArray
	variable widget
	set rowidx 0
	foreach attribute $formAttribList {
	    entry $widget(record).lb2$attribute -state readonly -width 55 \
		    -textvar ::form::txtRecord($attribute)
	    grid $widget(record).lb2$attribute -in $widget(record) -column 1 -row $rowidx \
		    -columnspan 1 -rowspan 1
	    button $widget(record).bn$attribute -text {>>} -pady 0 \
		-command "::form::cmdExpand $attribute 1"
	    grid $widget(record).bn$attribute -in $widget(record) -column 2 -row $rowidx \
		-columnspan 1 -rowspan 1
	    incr rowidx
	}
	return
    }

    proc hideAttribLabels {} {

	variable formAttribList
	variable widget
	foreach attribute $formAttribList {
	    destroy $widget(record).lb2$attribute
	    destroy $widget(record).bn$attribute
	}
	return
    }

    proc displayAttribEntries {} {

	variable formAttribList
	variable txtRecord
	variable attributeArray
	variable widget
	set rowidx 0
	foreach attribute $formAttribList {
	    switch $attributeArray($attribute,typeofget) {
		"tgDirect" -
		"tgExpression" {
		    entry $widget(record).ent$attribute -borderwidth 1 -textvar \
			::form::txtRecord($attribute) -relief sunken -width 55 \
			-background white
		    button $widget(record).bn$attribute -text {>>} -pady 0 \
			-command "::form::cmdExpand $attribute 0"
		    grid $widget(record).bn$attribute -in $widget(record) -column 2 -row $rowidx \
			-columnspan 1 -rowspan 1
		}
		"tgReadOnly" {
		    entry $widget(record).ent$attribute -borderwidth 1 -textvar \
			::form::txtRecord($attribute) -relief sunken -width 55 \
			-state readonly
		    button $widget(record).bn$attribute -text {>>} -pady 0 \
			-command "::form::cmdExpand $attribute 1"
		    grid $widget(record).bn$attribute -in $widget(record) -column 2 -row $rowidx \
			-columnspan 1 -rowspan 1
		}
		"tgList" -
		"tgLink" {
		    button $widget(record).ent$attribute -pady 0 -padx 0 -anchor w -textvar \
			::form::txtRecord($attribute) -relief raised -width 55 -command \
			"::form::cmdSelectFromList $attribute .form fillout"
		    button $widget(record).bn$attribute -text {>>} -pady 0 \
			-command "::form::cmdExpand $attribute 0"
		    grid $widget(record).bn$attribute -in $widget(record) -column 2 -row $rowidx \
			-columnspan 1 -rowspan 1
		}
	    }
	    grid $widget(record).ent$attribute -in $widget(record) -column 1 -row $rowidx \
		    -columnspan 1 -rowspan 1
	    incr rowidx
	}
	return
    }

    proc cmdExpand { attribute readonly } {
	variable wrapOn
	variable txtRecord

	set wrapOn 1
	destroy .form.expand
	toplevel .form.expand -class Toplevel
	wm transient .form.expand .form
	set x [expr [winfo pointerx .form] - 400]
	set y [expr [winfo pointery .form] - 100]
	wm geometry .form.expand 600x400+$x+$y
	wm title .form.expand "$attribute"
	if { $readonly } then {
	    ::iwidgets::scrolledtext .form.expand.text -wrap word
	} else {
	    ::iwidgets::scrolledtext .form.expand.text -textbackground white -wrap word
	    button .form.expand.bnOK -text {OK} -command "::form::cmdExpandOK $attribute"
	    place .form.expand.bnOK -x 0 -y 0 -relx 0.6 -rely 1 -anchor s
	}
	radiobutton .form.expand.rbWrap -text {Wrap} -value 1 -variable ::form::wrapOn \
	    -command {.form.expand.text configure -wrap word}
	radiobutton .form.expand.rbTruncate -text {Truncate} -value 0 -variable ::form::wrapOn \
	    -command {.form.expand.text configure -wrap none}
	button .form.expand.bnCancel -text {Cancel} -command {destroy .form.expand}
	place .form.expand.text -x 0 -y 0 -height -30 -relwidth 1 -relheight 1 -anchor nw
	place .form.expand.rbWrap -x 0 -y 0 -relx 0.2 -rely 1 -anchor s
	place .form.expand.rbTruncate -x 0 -y 0 -relx 0.4 -rely 1 -anchor s
	place .form.expand.bnCancel -x 0 -y 0 -relx 0.8 -rely 1 -anchor s
	.form.expand.text insert end $txtRecord($attribute)
	if { $readonly } then {
	    .form.expand.text configure -state disabled
	}
	return
    }

    proc cmdExpandOK { attribute} {
	variable txtRecord

	set txtRecord($attribute) [.form.expand.text get 1.0 "end -1 chars"]
	destroy .form.expand
	return
    }

    proc hideAttribEntries {} {

	variable formAttribList
	variable widget
	foreach attribute $formAttribList {
	    destroy $widget(record).ent$attribute
	    destroy $widget(record).bn$attribute
	}
	return
    }

    proc displayBrowseButtons {} {

	variable activeForm
	variable ::pfm::formsArray
	set view $formsArray($activeForm,view)
	grid .form.frmButtons.btnPrev \
		-in .form.frmButtons -column 0 -row 0 -columnspan 1 -rowspan 1 
	grid .form.frmButtons.btnNext \
		-in .form.frmButtons -column 1 -row 0 -columnspan 1 -rowspan 1 
	if { [string equal $view {f}] } then {
	    grid .form.frmButtons.btnUpdate \
		    -in .form.frmButtons -column 2 -row 0 -columnspan 1 -rowspan 1 
	    grid .form.frmButtons.btnAdd \
		    -in .form.frmButtons -column 3 -row 0 -columnspan 1 -rowspan 1 
	    grid .form.frmButtons.btnDelete \
		    -in .form.frmButtons -column 4 -row 0 -columnspan 1 -rowspan 1 
	}
	place .form.frmButtons.btnQuit -x 0 -y 0 -relx 1 -rely 0.5 -anchor e
	# also bind PgUp, PgDn, Up and Dn
	bind .form <KeyPress-Next> ::form::cmdNext
	bind .form <KeyPress-Down> ::form::cmdNext
	bind .form <KeyPress-Prior> ::form::cmdPrev
	bind .form <KeyPress-Up> ::form::cmdPrev
	return
    }

    proc hideBrowseButtons {} {

	grid forget .form.frmButtons.btnPrev
	grid forget .form.frmButtons.btnNext
	grid forget .form.frmButtons.btnUpdate
	grid forget .form.frmButtons.btnAdd
	grid forget .form.frmButtons.btnDelete
	place forget .form.frmButtons.btnQuit
	# also unbind PgUp, PgDn, Up and Dn
	bind .form <KeyPress-Next> {}
	bind .form <KeyPress-Down> {}
	bind .form <KeyPress-Prior> {}
	bind .form <KeyPress-Up> {}
	return
    }

    proc displayEditButtons {} {

	grid .form.frmButtons.btnOK \
		-in .form.frmButtons -column 0 -row 0 -columnspan 1 -rowspan 1 
	grid .form.frmButtons.btnCancel \
		-in .form.frmButtons -column 1 -row 0 -columnspan 1 -rowspan 1 
	return
    }

    proc hideEditButtons {} {

	grid forget .form.frmButtons.btnOK
	grid forget .form.frmButtons.btnCancel
	return
    }

    proc newFormState {newState} {

	variable formState
	set formState $newState
	switch $newState {
	    "update" -
	    "add" {
		hideAttribLabels
		displayAttribEntries
		hideBrowseButtons
		displayEditButtons
	    }
	    "browse" {
		hideAttribEntries
		displayAttribLabels
		hideEditButtons
		displayBrowseButtons
	    }
	}
	return
    }

    proc filltxtRecord {recordNr} {

	variable recordArray
	variable txtRecord
	variable formAttribList
	variable lastRecord
	variable widget
	foreach attribute $formAttribList {
	    set txtRecord($attribute) $recordArray($recordNr,$attribute)
	}
	set txtRecord(23nr47) "Record [expr $recordNr + 1]/$lastRecord"
	set txtRecord(23status47) $recordArray($recordNr,23status47)
	return
    }

    proc getAttributes {formName} {

	variable ::pfm::currentDB
	variable attributeArray
	variable formAttribList
	variable tableAttribList
	array unset attributeArray
	set fields "attribute,typeofattrib,typeofget,valuelist,sqlselect"
	set queryDef "SELECT $fields FROM pfm_attribute WHERE form = '$formName' ORDER BY nr"
	set queryRes [pg_exec $currentDB $queryDef]
	set formAttribList [list]
	set tableAttribList [list]
	set lastAttribute [expr [pg_result $queryRes -numTuples] - 1]
	pg_result $queryRes -assign attribRecords
	for {set attribNr 0} {$attribNr <= $lastAttribute} {incr attribNr} {
	    set attribute [string trim $attribRecords($attribNr,attribute)]
	    lappend formAttribList $attribute
	    set typeofattrib [string trim $attribRecords($attribNr,typeofattrib)]
	    set attributeArray($attribute,typeofattrib) $typeofattrib
	    set typeofget [string trim $attribRecords($attribNr,typeofget)]
	    set attributeArray($attribute,typeofget) $typeofget
	    set valuelist [string trim $attribRecords($attribNr,valuelist)]
	    set attributeArray($attribute,valuelist) $valuelist
	    set sqlselect [string trim $attribRecords($attribNr,sqlselect)]
	    set attributeArray($attribute,sqlselect) $sqlselect
	    if { ![string equal $typeofget {tgReadOnly}] } then {
		lappend tableAttribList $attribute
	    }
	}
	pg_result $queryRes -clear
	array unset attribRecords
	return
    }

    ########################################################################
    # Procedures that modify the data base                                 #
    ########################################################################

    proc addRecord {} {
	# insert txtRecord to data base and to recordArray

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable recordArray
	variable txtRecord
	variable curRecord
	variable activeForm
	variable lastRecord
	variable attributeArray
	variable tableAttribList
	variable formAttribList
	set colDef "("
	set valueDef "("
	foreach attribute $tableAttribList {
	    set colDef "$colDef \"$attribute\","
	    if { [string equal $attributeArray($attribute,typeofget) {tgExpression}] } then {
		set txtRecord($attribute) [expr $txtRecord($attribute)]
	    }
	    if { [string equal $txtRecord($attribute) {=}] } then {
		set valueDef "$valueDef DEFAULT, "
	    } else {
		switch $attributeArray($attribute,typeofattrib) {
		    "taQuoted" {
			set convertedValue [string map {' ''} $txtRecord($attribute)]
			set valueDef "$valueDef '$convertedValue',"
		    }
		    "taNotQuoted" {
			set valueDef "$valueDef $txtRecord($attribute),"
		    }
		}
	    }
	}
	set colDef "[string trimright $colDef ","])"
	set valueDef "[string trimright $valueDef ","])"
	set queryDef "INSERT INTO \"$formsArray($activeForm,tablename)\" $colDef\nVALUES $valueDef"
	set queryRes [pg_exec $currentDB $queryDef]
	set status [pg_result $queryRes -status]
	if { [string equal $status {PGRES_COMMAND_OK}] } then {
	    set oid [pg_result $queryRes -oid]
	    set curRecord $lastRecord
	    incr lastRecord
	    set recordArray($curRecord,23status47) "Added"
	    set recordArray($lastRecord,23status47) "After last"
	    set txtRecord(23status47) "Added"
	    set txtRecord(23nr47) "Record [expr $curRecord + 1]/$lastRecord"
	    set recordArray($curRecord,oid) $oid
	    foreach attribute $tableAttribList {
		set recordArray($curRecord,$attribute) $txtRecord($attribute)
	    }
	    foreach attribute $formAttribList {
		set recordArray($lastRecord,$attribute) {}
	    }
	} else {
	    set status $status\n[pg_result $queryRes -error]
	    bell
	    # filltxtRecord $curRecord
	    set curRecord $lastRecord
	    set txtRecord(23nr47) "Record [expr $lastRecord + 1]/$lastRecord"
	    set txtRecord(23status47) "Not added"
	}
	pg_result $queryRes -clear
	set status "$queryDef\n$status"
	return $status
    }


    proc updateRecord {} {
	# copy txtRecord to data base and to recordArray

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable recordArray
	variable txtRecord
	variable curRecord
	variable activeForm
	variable attributeArray
	variable tableAttribList

	# Reworked because of bugs 679 and 680.
	# selectForUpdate starts a transaction and checks that the record
	# still exists and has not been modified by another user. If that check
	# fails, selectForUpdate returns 'false' and the update operation is
	# canceled.
	if { [selectForUpdate] } then {
	    set queryDef ""
	    foreach attribute $tableAttribList {
		if { ![string equal $txtRecord($attribute) \
			   $recordArray($curRecord,$attribute)] } then {
		    if { [string equal $attributeArray($attribute,typeofget) {tgExpression}] } \
			then {
			    set txtRecord($attribute) [expr $txtRecord($attribute)]
			}
		    switch $attributeArray($attribute,typeofattrib) {
			"taQuoted" {
			    set quotesDoubled [string map {' ''} $txtRecord($attribute)]
			    set queryDef "$queryDef \"$attribute\"='$quotesDoubled',"
			}
			"taNotQuoted" {
			    set queryDef "$queryDef \"$attribute\"=$txtRecord($attribute),"
			}
		    }
		}
	    }
	    if { ![string equal $queryDef {}] } then {
		set queryDef [string trimright $queryDef ","]
		set queryDef "UPDATE \"$formsArray($activeForm,tablename)\"\nSET $queryDef"
		set queryDef "$queryDef\nWHERE oid = $recordArray($curRecord,oid)"
		set queryRes [pg_exec $currentDB $queryDef]
		set status [pg_result $queryRes -status]
		if { [string equal $status {PGRES_COMMAND_OK}] } then {
		    set recordArray($curRecord,23status47) "Updated"
		    set txtRecord(23status47) "Updated"
		    foreach attribute $tableAttribList {
			set recordArray($curRecord,$attribute) $txtRecord($attribute)
		    }
		} else {
		    set status $status\n[pg_result $queryRes -error]
		    bell
		}
		pg_result $queryRes -clear
		set status "$queryDef\n$status"
		filltxtRecord $curRecord
		set endTransaction {COMMIT WORK}
	    } else {
		set status {No updates.}
		bell
		set endTransaction {ROLLBACK WORK}
	    }
	} else {
	    set status {Your update is cancelled.}
	    set endTransaction {ROLLBACK WORK}
	}
	set commitResult [pg_exec $currentDB $endTransaction]
	set commitStatus [pg_result $commitResult -status]
	if { ![string equal $commitStatus {PGRES_COMMAND_OK}] } then {
	    set commitStatus "$commitStatus\n[pg_result $commitResult -error]"
	    bell
	}
	set status "$status\n$endTransaction\n$commitStatus"
	pg_result $commitResult -clear
	return $status
    }

    proc selectForUpdate { } {

	# Introduced because of bugs 679 and 680.
	# selectForUpdate starts a transaction and checks that the record
	# still exists and has not been modified by another user. If that check
	# fails, selectForUpdate returns 'false'.

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable txtRecord
	variable recordArray
	variable curRecord
	variable tableAttribList
	variable activeForm
	set success 1
	set status {}
	set queryDef {}
	set sqlAttrib \"[join $tableAttribList "\", \""]\"
	if {[lsearch -exact $tableAttribList oid] == -1} then {
	    set sqlAttrib "\"oid\", $sqlAttrib"
	}
	set beginWork [pg_exec $currentDB "BEGIN WORK"]
	set beginStatus [pg_result $beginWork -status]
	if { [string equal $beginStatus {PGRES_COMMAND_OK}] } then {
	    set queryDef "SELECT $sqlAttrib FROM \"$formsArray($activeForm,tablename)\""
	    set queryDef "$queryDef\nWHERE oid=$recordArray($curRecord,oid) FOR UPDATE"
	    set queryRes [pg_exec $currentDB $queryDef]
	    set status [pg_result $queryRes -status]
	    if { [string equal $status {PGRES_TUPLES_OK}] } then {
		if { [pg_result $queryRes -numTuples]==1 } then {
		    pg_result $queryRes -assign recordForUpdate
		    foreach attribute $tableAttribList {
			if { ![string equal $recordArray($curRecord,$attribute) \
				   $recordForUpdate(0,$attribute)] } then {
			    set success 0
			    bell
			    set status \
				"$status\nRecord was modified by another user after you pressed the \[Update\] button.\nReconsider your input."
			    break
			}
		    }
		} else {
		    set success 0
		    bell
		    # Bug 680 : selectForUpdate has to take into account that the the
		    # current record may have been deleted by another user.
		    # If the database is still sane, numTuples can only be 0 or 1 for this query,
		    # because the oid, is unique by defintion.
		    # So, if numTuples != 1, it is assumed to be 0, i.e. it is assumed
		    # that the record has been deleted.
		    set status \
			"$status\nRecord was deleted by another user after you pressed \[Update\]."
		}
	    } else {
		set success 0
		set status $status\n[pg_result $queryRes -error]
		bell
	    }
	    pg_result $queryRes -clear
	} else {
	    set success 0
	    set beginStatus "$beginStatus\n[pg_result $beginWork -error]"
	    bell
	}
	pg_result $beginWork -clear
	set status "BEGIN WORK\n$beginStatus\n$queryDef\n$status\n"
	.form.txtResult delete 1.0 end
	.form.txtResult insert end $status
	return $success
    }

    #############################################################################
    #                                                                           #
    #  Procedures treating the links                                            #
    #                                                                           #
    #############################################################################

    proc displayLinks {} {

	variable ::pfm::currentDB
	variable activeForm
	variable linksArray
	variable lastFormOnStack
	if {$lastFormOnStack != 0} then {
	    button .form.frmLinkBtn.btnBack -text "Back" -pady 0 \
		-command ::form::cmdBack
	    grid .form.frmLinkBtn.btnBack -in .form.frmLinkBtn -row 0 \
		    -column 0 -rowspan 1 -columnspan 1
	}
	array unset linksArray
	set queryDef "SELECT * from pfm_link WHERE fromform = '$activeForm'"
	set queryRes [pg_exec $currentDB $queryDef]
	pg_result $queryRes -assign linksArray
	set lastLink [expr [pg_result $queryRes -numTuples] -1]
	pg_result $queryRes -clear
	set rowidx 1
	for {set link 0} {$link <= $lastLink} {incr link} {
	    set linkName $linksArray($link,linkname)
	    button .form.frmLinkBtn.btn$link -text $linkName -pady 0 \
		-command "::form::cmdFollowLink $link"
	    grid .form.frmLinkBtn.btn$link -in .form.frmLinkBtn -row $rowidx \
		    -column 0 -rowspan 1 -columnspan 1
	    incr rowidx
	}
	return
    }


    proc cmdFollowLink {link} {

	variable ::pfm::formsArray
	variable linksArray
	variable txtRecord
	variable attributeArray
	variable activeForm
	variable recordArray
	variable curRecord
	variable lastFormOnStack
	variable formStack

	if { ![string equal $txtRecord(23status47) {After Last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    set whereDef [expandSqlWhere $linksArray($link,sqlwhere)]
	    set orderDef $linksArray($link,orderby)
	    set dispAttribList $linksArray($link,displayattrib)
	    set displayDef " "
	    foreach attribute $dispAttribList {
		set displayDef "$displayDef$txtRecord($attribute) "
	    }
	    set From "$activeForm \($displayDef\)"
	    set To "$linksArray($link,toform)"
	    # remember view attribute of 'fromform'
	    set view $formsArray($activeForm,view)
	    # prepare form pointed to by the link
	    set activeForm $linksArray($link,toform)
	    getAttributes $activeForm
	    set sqlAttrib $formsArray($activeForm,sqlselect)
	    set sqlFrom $formsArray($activeForm,sqlfrom)
	    set groupby_having $formsArray($activeForm,groupby_having)
	    set queryDef "SELECT $sqlAttrib FROM $sqlFrom WHERE $whereDef"
	    set showQuery "Link \'$linksArray($link,linkname) : $From -> $To\'"
	    set showQuery "$showQuery\nSELECT $sqlAttrib\nFROM $sqlFrom\nWHERE $whereDef"
	    if { [string length $groupby_having] != 0 } then {
		set queryDef "$queryDef GROUP BY $groupby_having"
		set showQuery "$showQuery\nGROUP BY $groupby_having"
	    }
	    if { [string length $orderDef] != 0 } then {
		set queryDef "$queryDef ORDER BY $orderDef"
		set showQuery "$showQuery\nORDER BY $orderDef"
	    }
	    if { [string equal $view {f}] } then {
		set formStack($lastFormOnStack,displayOid) $recordArray($curRecord,oid)
	    } else {
		set formStack($lastFormOnStack,displayOid) 0
	    }
	    incr lastFormOnStack
	    set formStack($lastFormOnStack,formId) $activeForm
	    set formStack($lastFormOnStack,queryDef) $queryDef
	    set formStack($lastFormOnStack,showQuery) $showQuery
	    OpenForm $queryDef $showQuery 0
	}
	return
    }

    proc cmdBack {} {

	variable activeForm
	variable lastFormOnStack
	variable formStack
	if {$lastFormOnStack >= 1} then {
	    incr lastFormOnStack -1
	    set activeForm $formStack($lastFormOnStack,formId)
	    getAttributes $activeForm
	    set queryDef $formStack($lastFormOnStack,queryDef)
	    set showQuery $formStack($lastFormOnStack,showQuery)
	    set displayOid $formStack($lastFormOnStack,displayOid)
	    OpenForm $queryDef $showQuery $displayOid
	}
	return
    }

    proc reloadRecord { } {

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable txtRecord
	variable recordArray
	variable curRecord
	variable formAttribList
	variable activeForm
	set success 1
	if { ![string equal $txtRecord(23status47) {After Last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    set sqlAttrib $formsArray($activeForm,sqlselect)
	    set sqlFrom $formsArray($activeForm,sqlfrom)
	    set groupby_having $formsArray($activeForm,groupby_having)
	    set tableName $formsArray($activeForm,tablename)
	    set queryDef "SELECT $sqlAttrib\nFROM $sqlFrom"
	    set queryDef "$queryDef\nWHERE \"$tableName\".oid=$recordArray($curRecord,oid)"
	    if { [string length $groupby_having] != 0 } then {
		set queryDef "$queryDef\nGROUP BY $groupby_having"
	    }
	    set queryRes [pg_exec $currentDB $queryDef]
	    set status [pg_result $queryRes -status]
	    if { [string equal $status {PGRES_TUPLES_OK}]} then {
		if { [pg_result $queryRes -numTuples]==1 } then {
		    pg_result $queryRes -assign reloadedRecord
		    foreach attribute $formAttribList {
			set recordArray($curRecord,$attribute) $reloadedRecord(0,$attribute)
			set txtRecord($attribute) $recordArray($curRecord,$attribute)
		    }
		} else {
		    set success 0
		    bell
		    # Bug 680 : reloadRecord has to take into account that the the
		    # current record may have been deleted by another user.
		    # If the database is still sane, numTuples can only be 0 or 1 for this query,
		    # because the oid, is unique by defintion.
		    # So, if numTuples != 1, it is assumed to be 0, i.e. it is assumed
		    # that the record has been deleted.
		    set status "$status\nRecord has been deleted by another user."
		    set recordArray($curRecord,23status47) "Deleted"
		    set txtRecord(23status47) "Deleted"
		    foreach attribute $formAttribList {
			set recordArray($curRecord,$attribute) ""
			set txtRecord($attribute) ""
		    }	

		}
	    } else {
		set success 0
		set status "$status\n[pg_result $queryRes -error]"
		bell
	    }
	    pg_result $queryRes -clear
	    set status "Reload record:\n$queryDef\n$status"
	} else {
	    set success 0
	    set status {Record not reloaded}
	}
	.form.txtResult insert end $status
	return $success
    }

    proc expandSqlWhere {sqlWhere} {
	# This procedure first replaces "$(" with "$txtRecord(".
	# Then substitutes all the $variables with their value.

	variable txtRecord
	#    set expandWhere [string map {$( $recordArray($curRecord,} $sqlWhere]
	set expandWhere [string map {$( $txtRecord(} $sqlWhere]
	set expandWhere [string map {{"} {\"}} $expandWhere]
        # " Extra quote to release emacs from its misery
	eval "set expandWhere \"$expandWhere\""
	return $expandWhere
    }

}

##################################################################
# End of namespace form                                          #
##################################################################

##################################################################
# Begin namespace report                                         #
##################################################################

# reportDef is an array which contains the data stored in table
#           pfm_report. It is filled out by cmdDisplayReportList.
#
# groupLevel is an arry where groupLevel($level) = true/false
#           It indicates for each level whether or not it is
#           a group level. It is filled out by collectReportLayout.
#
# layout is an array where layout($level) = row, column or table
#         - "row" means that the labels and values are printed on 1 row
#         - "column" means that the labels are printed in the
#            first column and the values in the second column
#         - "table" means that the values are printed in a
#            table with the labels as table header
#               
#           It is filled out by collectReportLayout.
#
# fieldNamesList($level) contains, for each level, the list
# of field names to be displayed in that level.
#
# printInfo($name,label) contains the label to be used for the
# field "$name". It is filled out by collectReportLayout
#
# printInfo($name,alignment) conatains the alignment info:
#    l: left
#    r: right
#
# printInfo($name,width) contains the maximum width for the
# field $name. It is filled out by cmdRunReport.
#
# printInfo($name,columnWidth) contains the maximum of
# the length of the field's label and printInfo($name,with).
# It is the optimum width for a table column displaying the field data.
#
# maxLabelWidth($level) contains, for each level, the maximum label width.
# It is filled out by collectReportLayout.
#
# reportData contains the result of the query associated with the
# report. It is filled out by cmdRunReport.


namespace eval report {

    variable ::pfm::currentDB
    variable ::pfm::dbname
    variable ::pfm::hostname
    variable reportDef
    variable layout
    variable groupLevel
    variable fieldNamesList
    variable printInfo
    variable maxLabelWidth
    variable reportData
    variable reportMode {sql}
    variable printCommand
    variable parmlist

########################################################################
#  This is the entry point for this namespace                          #
########################################################################

    proc cmdReportSQL {requestedMode} {
	variable ::pfm::currentDB
	variable ::pfm::dbname
	variable reportMode
	variable wrapOn

	set wrapOn 0
	set reportMode $requestedMode
	if { [info exists currentDB] } then {
	    if {[winfo exists .report]} {
		destroy .report
	    }
	    toplevel .report -class Toplevel
	    wm focusmodel .report passive
	    wm geometry .report 750x550
	    wm minsize .report 1 1
	    wm overrideredirect .report 0
	    wm resizable .report 1 1
	    wm deiconify .report
	    wm title .report "pfm - Reports and Queries : $dbname"
	    
	    # define widgets

	    # define upper frame for radiobuttons
	    
	    frame .report.fmUpper -borderwidth 2 -relief groove
	    place .report.fmUpper -x 0 -y 0 -relwidth 1 -height 30 -anchor nw
	    radiobutton .report.fmUpper.rbSQL -text {SQL} -value {sql} \
		-variable ::report::reportMode -command ::report::cmdEnterSQLmode
	    radiobutton .report.fmUpper.rbReport -text {Report} -value {report} \
		-variable ::report::reportMode \
		-command ::report::cmdEnterReportMode
	    grid .report.fmUpper.rbSQL -column 0 -row 0
	    grid .report.fmUpper.rbReport -column 1 -row 0

	    # define scrolled text widget for SQL statement

	    ::iwidgets::scrolledtext .report.txtSQL \
		-labeltext {SQL statement} -wrap none -textbackground white \
		-textfont {Courier -12 normal}

	    # define middle frame and buttons for SQL command

	    frame .report.fmMiddle -borderwidth 2 -relief groove
	    place .report.fmMiddle -x 0 -y -30 -rely 0.4 -height 30 -relwidth 1
	    button .report.fmMiddle.bnRun -text {Run} -command {::report::cmdRun}
	    button .report.fmMiddle.bnSave -text {Save SQL} \
		-command {::report::cmdSaveFile .report.txtSQL "Save SQL as" "sql"}
	    button .report.fmMiddle.bnImport -text {Import SQL} \
		-command {::report::cmdImportFile .report.txtSQL "Open SQL-file" "sql"}
	    button .report.fmMiddle.bnClear -text {Clear} -command ::report::cmdClear
	    button .report.fmMiddle.bnForward -text {Forward} -command ::report::cmdForward
	    button .report.fmMiddle.bnBack -text {Back} -command ::report::cmdBack
	    pack .report.fmMiddle.bnRun -anchor center -expand 0 -fill none -side right
	    pack .report.fmMiddle.bnSave -anchor center -expand 0 -fill none -side right
	    pack .report.fmMiddle.bnImport -anchor center -expand 0 -fill none -side right
	    pack .report.fmMiddle.bnClear -anchor center -expand 0 -fill none -side right
	    pack .report.fmMiddle.bnForward -anchor center -expand 0 -fill none -side right
	    pack .report.fmMiddle.bnBack -anchor center -expand 0 -fill none -side right

	    # define scrolled text widget for displaying results

	    ::iwidgets::scrolledtext .report.txtResult \
		-labeltext {Result} -wrap none -textbackground white -textfont {Courier -12}
	    place .report.txtResult -x 0 -y 0 -rely 0.4 -height -30 -relheight 0.6 -relwidth 1

	    # define lower frame and buttons

	    frame .report.fmLower -borderwidth 2 -relief groove
	    place .report.fmLower -x 0 -y -30 -rely 1 -height 30 -relwidth 1
	    button .report.fmLower.bnQuit -text {Quit} -command ::report::cmdQuitReport
	    button .report.fmLower.bnSave -text {Save result} \
		-command {::report::cmdSaveFile .report.txtResult "Save result as" "text"}
	    button .report.fmLower.bnPrint -text {Print result} \
		-command {::report::cmdPrint .report.txtResult}
	    button .report.fmLower.bnClear -text {Clear} \
		-command {.report.txtResult delete 1.0 end}
	    radiobutton .report.fmLower.rbWrap -text {Wrap} -value 1 \
		-variable ::report::wrapOn -command {.report.txtResult configure -wrap char}
	    radiobutton .report.fmLower.rbTruncate -text {Truncate} -value 0 \
		-variable ::report::wrapOn -command {.report.txtResult configure -wrap none}

	    pack .report.fmLower.bnQuit -anchor center -expand 0 -fill none -side right
	    pack .report.fmLower.bnSave -anchor center -expand 0 -fill none -side right
	    pack .report.fmLower.bnPrint -anchor center -expand 0 -fill none -side right
	    pack .report.fmLower.bnClear -anchor center -expand 0 -fill none -side right
	    pack .report.fmLower.rbTruncate -anchor center -expand 0 -fill none -side right
	    pack .report.fmLower.rbWrap -anchor center -expand 0 -fill none -side right

	    # Initially enter mode according to $reportMode

	    switch $reportMode {
		sql {
		    cmdEnterSQLmode
		}
		report {
		    cmdEnterReportMode
		}
	    }
	} else {
	    tk_messageBox -message "There is no data base opened!" -type ok
	}
	return
    }

###########################################################################
# Common procedures, i.e. for both SQL and report mode                    #
###########################################################################

    proc cmdSaveFile {txtWidget title type} {
	
	set textToSave [$txtWidget get 1.0 end]
	switch $type {
	    "sql" {
		set fileTypes {
		    {{SQL statements} {.sql} }
		    {{All files} *}
		}
		set defaultExt ".sql"
	    }
	    "text" {
		set fileTypes {
		    {{Text files} {.txt} }
		    {{All files} *}
		}
		set defaultExt ".txt"
	    }
	}
	set fileName [tk_getSaveFile -title $title -filetypes $fileTypes \
			  -defaultextension $defaultExt]
	if { $fileName !=  "" } then {
	    set file_ch [open $fileName w]
	    puts $file_ch $textToSave
	    close $file_ch
	}
	return
    }


    proc cmdImportFile {txtWidget title type} {
	
	switch $type {
	    "sql" {
		set fileTypes {
		    {{SQL statements} {.sql} }
		    {{All files} *}
		}
		set defaultExt ".sql"
	    }
	    "text" {
		set fileTypes {
		    {{Text files} {.txt} }
		    {{All files} *}
		}
		set defaultExt ".txt"
	    }
	}
	set fileName [tk_getOpenFile -title $title -filetypes $fileTypes \
			  -defaultextension $defaultExt]
	if { $fileName !=  "" } then {
	    set file_ch [open $fileName r]
	    $txtWidget insert end [read $file_ch]
	    close $file_ch
	}
	return
    }

    proc cmdPrint {txtWidget} {
	variable ::options::pfmOptions
	variable printCommand
	variable parmlist

	set x [expr [winfo pointerx .report] -500]
	set y [expr [winfo pointery .report] -300]
	toplevel .report.tpPrint -class Toplevel
	wm transient .report.tpPrint .report
	wm geometry .report.tpPrint 600x400+$x+$y
	wm title .report.tpPrint "pfm - Print result"
	set printCommand $pfmOptions(printcmd)
	label .report.tpPrint.lbPrintcmd -text \
	    "$printCommand\n\nLongest line is: [longestLine .report.txtResult]" -wraplength 590
	place .report.tpPrint.lbPrintcmd -x 0 -y 0 -relx 0.5 -anchor n

	# Get the parameters for the printcommand
	set parmlist {}
	set startOfParm [string first "\$(" $printCommand 0]
	if { $startOfParm >= 0 } then {
	    set n 1
	    while { $startOfParm >= 0 } {
		set endOfParm [string first ")" $printCommand $startOfParm]
		if { $endOfParm >= 0 } then {
		    set parm [string range $printCommand $startOfParm $endOfParm]
		    set equalSign [string first "=" $parm 0]
		    if { $equalSign >= 0 } then {
			set defVal [string range $parm [expr $equalSign + 1] "end-1"]
			set parmName [string range $parm 2 [expr $equalSign - 1]]
		    } else {
			set defVal {}
			set parmName [string range $parm 2 "end-1"]
		    }
		    label .report.tpPrint.lb$n -text $parmName
		    grid .report.tpPrint.lb$n -column 0 -row $n
		    entry .report.tpPrint.en$n -width 40 -background white
		    .report.tpPrint.en$n insert end $defVal
		    grid .report.tpPrint.en$n -column 1 -row $n
		    lappend parmlist $parm
		    set startOfParm [string first "\$(" $printCommand [expr $endOfParm + 1]]
		    incr n
		} else {
		    set startOfParm -1
		}
	    }
	}
	button .report.tpPrint.bnOK -text OK \
	    -command "::report::cmdPrintOK $txtWidget"
	place .report.tpPrint.bnOK -x -30 -y 0 -relx 0.5 -rely 1 -anchor s
	button .report.tpPrint.bnCancel -text Cancel \
	    -command {destroy .report.tpPrint}
	place .report.tpPrint.bnCancel -x 30 -y 0 -relx 0.5 -rely 1 -anchor s
	return
    }

    proc cmdPrintOK {txtWidget} {
	variable printCommand
	variable parmlist

	set n 1
	foreach parm $parmlist {
	    set value [.report.tpPrint.en$n get]
	    set printCommand [string map "$parm \"$value\"" $printCommand]
	    incr n
	}
	destroy .report.tpPrint
	set printch [open "|$printCommand" w]
	puts $printch [$txtWidget get 1.0 end]
	if { [catch {close $printch} errMsg] } then {
	    tk_messageBox -message $errMsg -type ok
	}

	return
    }

    proc longestLine {txtWidget} {

	set longest 0
	set lastIndex [$txtWidget index end]
	set index [$txtWidget index 1.0]
	while { $index < $lastIndex } {
	    set thisLineLength [string length [$txtWidget get $index "$index lineend"]]
	    if { $longest < $thisLineLength } then {
		set longest $thisLineLength
	    }
	    set index [$txtWidget index "$index +1 lines"]
	}
	return $longest
    }

    proc cmdRun {} {
	variable reportMode

	switch $reportMode {
	    sql {
		cmdRunSQL
	    }
	    report {
		cmdRunReport [.report.lsb curselection]
	    }
	}
	return
    }


    proc {cmdQuitReport} {} {
	# destroy window .topReport

	destroy .report
    }

###########################################################################
#  Procedures for queries (SQL mode)                                      #
###########################################################################

    proc cmdEnterSQLmode { } {
	variable reportMode
	variable commandHistory

	destroy .report.lsb
	place .report.txtSQL -x 0 -y 30 -relwidth 1 -height -60 -relheight 0.4
	.report.fmMiddle.bnSave configure -state normal
	.report.fmMiddle.bnImport configure -state normal
	.report.fmMiddle.bnClear configure -state normal
	set commandHistory(cursor) $commandHistory(top)
	.report.fmMiddle.bnForward configure -state disabled
	if { $commandHistory(cursor) > 0 } then {
	    .report.fmMiddle.bnBack configure -state normal
	} else {
	    .report.fmMiddle.bnBack configure -state disabled
	}
	return
    }

    proc cmdClear {} {

	.report.txtSQL delete 1.0 end
	return
    }

    proc initCommandHistory {} {
	variable commandHistory

	set commandHistory(top) 0
	set commandHistory(cursor) 0
	set commandHistory(0) {}
	return
    }

    initCommandHistory

    proc storeCommand {} {
	variable commandHistory

	incr commandHistory(top)
	set commandHistory(cursor) $commandHistory(top)
	set commandHistory($commandHistory(top)) [.report.txtSQL get 1.0 "end -1 chars"]
	.report.fmMiddle.bnBack configure -state normal
	.report.fmMiddle.bnForward configure -state disabled
	return
    }

    proc cmdBack {} {
	variable commandHistory

	if { $commandHistory(cursor) > 1 } then {
	    incr commandHistory(cursor) -1
	    .report.txtSQL delete 1.0 end
	    .report.txtSQL insert end $commandHistory($commandHistory(cursor))
	    if { $commandHistory(cursor) == 1 } then {
		.report.fmMiddle.bnBack configure -state disabled
	    }
	    .report.fmMiddle.bnForward configure -state normal
	} else {
	    bell
	}
	return
    }

    proc cmdForward {} {
	variable commandHistory

	if { $commandHistory(cursor) < $commandHistory(top) } then {
	    incr commandHistory(cursor)
	    .report.txtSQL delete 1.0 end
	    .report.txtSQL insert end $commandHistory($commandHistory(cursor))	    
	    if { $commandHistory(cursor) == $commandHistory(top) } then {
		.report.fmMiddle.bnForward configure -state disabled
	    }
	    .report.fmMiddle.bnBack configure -state normal
	} else {
	    bell
	}
	return
    }

    proc cmdRunSQL {} {
	variable ::pfm::dbname
	variable ::pfm::hostname

	storeCommand
	# When psql is called from tcl, it puts client_encoding to UNICODE.
	# Although I am not sure, I assume that it means in fact utf-8. At any rate,
	# without the conversions to/from utf-8 the special characters like
	# '' are not handled correctly. It seems to be alright with server_encoding
	# LATIN1 and UNICODE, but I don't know exactly why, and I am still not convinced
	# that this is the right solution.
        set sqlCmd [encoding convertto utf-8 [.report.txtSQL get 1.0 end]]
	if { $hostname == "" } then {
	    .report.txtResult insert end \
		[encoding convertfrom utf-8 \
		     [exec echo "$sqlCmd" | psql --dbname $dbname --echo-all |& cat]]
	} else {
	    .report.txtResult insert end \
		[encoding convertfrom utf-8 \
		     [exec echo "$sqlCmd" | psql --dbname $dbname --host $hostname --echo-all |& cat]]
	}
	.report.txtResult insert end "\n\n"
	.report.txtResult see end
	return
    }


###################################################################
#  Procedures for report mode                                     #
###################################################################

    proc cmdEnterReportMode { } {
	variable reportMode

	place forget .report.txtSQL
	.report.fmMiddle.bnSave configure -state disabled
	.report.fmMiddle.bnImport configure -state disabled
	.report.fmMiddle.bnClear configure -state disabled
	.report.fmMiddle.bnForward configure -state disabled
	.report.fmMiddle.bnBack configure -state disabled
	cmdDisplayReportList
	return
    }


    proc cmdDisplayReportList {} {
	variable ::pfm::currentDB
	variable reportDef

	if { ![info exists currentDB] } then {
	    tk_messageBox -message "There is no data base opened!" -type ok
	} else {
	    set queryDef "SELECT * FROM pfm_report ORDER BY name"
	    set queryRes [pg_exec $currentDB $queryDef]
	    set lastTuple [expr [pg_result $queryRes -numTuples] - 1]
	    pg_result $queryRes -assign reportDef
	    pg_result $queryRes -clear
	    if { $lastTuple < 0} then {
		tk_messageBox -message "There are no reports registered!" -type ok
	    } else {
		set maxNameWidth 0
		for {set tuple 0} {$tuple <= $lastTuple} {incr tuple } {
		    set nameLength [string length $reportDef($tuple,name)]
		    if { $nameLength > $maxNameWidth } then {
			set maxNameWidth $nameLength
		    }
		}
		set reportList {}
		for {set tuple 0} {$tuple <= $lastTuple} {incr tuple } {
		    set name [format "%-$maxNameWidth\s" $reportDef($tuple,name)]
		    lappend reportList \
			    "$name : $reportDef($tuple,description)"
		}
		iwidgets::scrolledlistbox .report.lsb -labeltext "List of reports" \
		    -textfont {Courier -12 bold}
		foreach report $reportList {
		    .report.lsb insert end $report
		}
		.report.lsb selection clear 0 end
		.report.lsb selection set 0 0
		place .report.lsb -x 0 -y 30 -relwidth 1 -height -60 -relheight 0.4
	    }
	}
	return
    }

    proc cmdRunReport { selectedReport } {
	variable reportDef
	variable ::pfm::currentDB
	variable reportData
	variable parmlist

	# Get the parameters necessary to execute the query
	set sqlwhere $reportDef($selectedReport,sqlwhere)
	set startOfParm [string first "\$(" $sqlwhere 0]
	if { $startOfParm >= 0 } then {
	    set parmlist {}
	    set x [expr [winfo pointerx .report] - 600]
	    set y [expr [winfo pointery .report] -100]
	    toplevel .report.getparm -class Toplevel
	    wm transient .report.getparm .report
	    wm geometry .report.getparm 600x300+$x+$y
	    wm title .report.getparm "pfm - Get report parameters"
	    label .report.getparm.lbselect \
		-text "SELECT * FROM $reportDef($selectedReport,table_or_view)"
	    place .report.getparm.lbselect -x 0 -y 0 -relx 0.5 -anchor n
	    label .report.getparm.lbsqlwhere -text "WHERE $sqlwhere"
	    place .report.getparm.lbsqlwhere -x 0 -y 20 -relx 0.5 -anchor n
	    set n 1
	    while { $startOfParm >= 0 } {
		set endOfParm [string first ")" $sqlwhere $startOfParm]
		if { $endOfParm >= 0 } then {
		    set parm [string range $sqlwhere $startOfParm $endOfParm]
		    set labelText [string range $sqlwhere \
				       [expr $startOfParm +2] [expr $endOfParm - 1]]
		    label .report.getparm.lb$n -text $labelText
		    grid .report.getparm.lb$n -column 0 -row $n
		    entry .report.getparm.en$n -width 40 -background white
		    grid .report.getparm.en$n -column 1 -row $n
		    lappend parmlist $parm
		    set startOfParm [string first "\$(" $sqlwhere [expr $endOfParm + 1]]
		    incr n
		} else {
		    set startOfParm -1
		}
	    }
	    button .report.getparm.bnOK -text OK \
		-command "::report::completeSqlwhere $selectedReport"
	    place .report.getparm.bnOK -x -30 -y 0 -relx 0.5 -rely 1 -anchor s
	    button .report.getparm.bnCancel -text Cancel \
		-command {destroy .report.getparm}
	    place .report.getparm.bnCancel -x 30 -y 0 -relx 0.5 -rely 1 -anchor s
	} else {
	    executeQuery $selectedReport $sqlwhere
	}
	return
    }

    proc completeSqlwhere { selectedReport } {
	variable parmlist
	variable reportDef

	set sqlwhere $reportDef($selectedReport,sqlwhere)
	set n 1
	foreach parm $parmlist {
	    set value [.report.getparm.en$n get]
	    set sqlwhere [string map "$parm \"$value\"" $sqlwhere]
	    incr n
	}
	destroy .report.getparm
	executeQuery $selectedReport $sqlwhere
	return
    }

    proc executeQuery {selectedReport sqlwhere } {
	variable reportDef
	variable ::pfm::currentDB
	variable reportData

	# Execute the query for the report and store the result in reportData

	set tableOrView $reportDef($selectedReport,table_or_view)
	if { [string equal $sqlwhere ""] } then {
	    set queryDef "SELECT * FROM $tableOrView"
	} else {
	    set queryDef "SELECT * FROM $tableOrView WHERE $sqlwhere"
	}
	set orderby $reportDef($selectedReport,orderby)
	if { [string length $orderby] } then {
	    set queryDef "$queryDef ORDER BY $orderby"
	}
	set queryRes [pg_exec $currentDB $queryDef]
	set queryStatus [pg_result $queryRes -status]
	if { [string equal $queryStatus "PGRES_TUPLES_OK"] } then {
	    pg_result $queryRes -assign reportData
	    set lastTuple [expr [pg_result $queryRes -numTuples] - 1]
	    printReport $selectedReport $lastTuple $sqlwhere $orderby
	} else {
	    set errmsg "$queryDef failed\n"
	    set errmsg "$errmsg [pg_result $queryRes -error]\n"
	    .report.txtResult insert end $errmsg
	    .report.txtResult see end
	}
	pg_result $queryRes -clear
	return
    }


    proc printReport { selectedReport lastTuple filter orderby} {
	variable reportDef
	variable groupLevel
	variable fieldNamesList
	variable printInfo
	variable reportData
	variable layout
	variable maxLabelWidth

	set lastLevel [collectReportLayout $selectedReport]

	# Calculate maximum field width and the optimum columwidth
	# for a table layout. Store result in printInfo.

	for { set level 1 } { $level <= $lastLevel} { incr level} {
	    foreach field $fieldNamesList($level) {
		set printInfo($field,width) 0
		for {set tuple 0} { $tuple <= $lastTuple } { incr tuple } {
		    set width [string length $reportData($tuple,$field)]
		    if { $printInfo($field,width) < $width } then {
			set printInfo($field,width) $width
		    }
		}
		set labelLength [string length $printInfo($field,label)]
		if { $printInfo($field,width) < $labelLength } then {
		    set printInfo($field,columnWidth) $labelLength
		} else {
		    set printInfo($field,columnWidth) $printInfo($field,width)
		}
	    }
	}

	# Print report title

	.report.txtResult insert end "$reportDef($selectedReport,description)\n"
	.report.txtResult insert end \
	    "[string repeat - [string length $reportDef($selectedReport,description)]]\n\n"

	.report.txtResult insert end "Report   : $reportDef($selectedReport,name)\n"
	.report.txtResult insert end "View     : $reportDef($selectedReport,table_or_view)\n"
	if { ![string equal $filter {}] } then {
	    .report.txtResult insert end "Where    : $filter\n"
	}
	if { ![string equal $orderby {}] } then {
	    .report.txtResult insert end "Order by : $orderby\n"
	}
	.report.txtResult insert end \
	    "Date     : [clock format [clock seconds] -format {%d-%b-%Y}]\n\n"

	# print the reportData

	set firstRecordOfTable 1
	for {set tuple 0} { $tuple <= $lastTuple } { incr tuple } {
	    for { set level 1 } { $level <= $lastLevel } { incr level } {
		if { $groupLevel($level) } then {
		    if { [newValues $level $tuple] } then { 
			.report.txtResult insert end [printLevel 1 $level $tuple]
			if { $level == [expr $lastLevel - 1] } then {
			    set firstRecordOfTable 1
			}
		    }
		} else {
		    .report.txtResult insert end [printLevel $firstRecordOfTable $level $tuple]
		    set firstRecordOfTable 0
		}
	    }
	}
	.report.txtResult insert end "\n\n"
	.report.txtResult see end
	array unset groupLevel
	array unset fieldNamesList
	array unset printInfo
	array unset reportData
	array unset layout
	array unset maxLabelWidth
	return
    }

    proc collectReportLayout {selectedReport } {
	variable ::pfm::currentDB
	variable groupLevel
	variable layout
	variable fieldNamesList
	variable printInfo
	variable reportDef
	variable maxLabelWidth

	# Get data from pfm_section and store them in sectionArray

	set queryDef "SELECT * FROM pfm_section WHERE report='$reportDef($selectedReport,name)'"
	set queryDef "$queryDef ORDER BY level"
	set queryRes [pg_exec $currentDB $queryDef]
	set lastTuple [expr [pg_result $queryRes -numTuples] - 1]
	pg_result $queryRes -assign sectionArray
	pg_result $queryRes -clear
	set lastLevel 0

	# Store data in groupLevel, layout, fieldNamesList and printInfo

	for { set tuple 0} {$tuple <= $lastTuple} {incr tuple } {
	    if { $sectionArray($tuple,level) > $lastLevel} then {
		set lastLevel $sectionArray($tuple,level)
	    }
	    set groupLevel($sectionArray($tuple,level)) 1
	    set layout($sectionArray($tuple,level)) $sectionArray($tuple,layout)
	    set fieldDefList $sectionArray($tuple,fieldlist)
	    set fieldNamesList($sectionArray($tuple,level)) {}
	    foreach item $fieldDefList {
		set attribName [lindex $item 0]
		lappend fieldNamesList($sectionArray($tuple,level)) $attribName
		set printInfo($attribName,label) [lindex $item 1]
		set printInfo($attribName,alignment) [lindex $item 2]
	    }
	}
	set groupLevel($lastLevel) 0

	# Calculate maxLabelWidth per level

	for { set level 1 } { $level <= $lastLevel } { incr level } {
	    set maxLabelWidth($level) 0
	    foreach field $fieldNamesList($level) {
		set width [string length $printInfo($field,label)]
		if { $width > $maxLabelWidth($level) } then {
		    set maxLabelWidth($level) $width
		}
	    }
	}

	array unset sectionArray
	return $lastLevel
    }

    proc newValues { level tuple } {
	variable fieldNamesList
	variable reportData

	if { $tuple == 0 } then {
	    set returnValue 1
	} else {
	    set returnValue 0
	    foreach field $fieldNamesList($level) {
		if { $reportData($tuple,$field) != $reportData([expr $tuple - 1],$field) } then {
		    set returnValue 1
		}
	    }
	}
	return $returnValue
    }


    proc printLevel {firstRecordOfTable level tuple} {
	variable layout

	switch $layout($level) {
	    row {
		set lineToPrint [printRow $level $tuple]
	    }
	    column {
		set lineToPrint [printColumn $firstRecordOfTable $level $tuple]
	    }
	    table {
		set lineToPrint [printTable $firstRecordOfTable $level $tuple]
	    }
	}
	if { $firstRecordOfTable } then {
	    set lineToPrint "\n$lineToPrint"
	}
	return $lineToPrint
    }



    proc printRow { level tuple } {
	variable reportData
	variable fieldNamesList
	variable printInfo

	set offset [string repeat " " [expr 4 * ($level - 1)]]
	set lineToPrint $offset
	foreach field $fieldNamesList($level) {
	    set lineToPrint \
		"$lineToPrint$printInfo($field,label):"
	    set lineToPrint \
		"$lineToPrint $reportData($tuple,$field); "
	}
	return "$lineToPrint\n"
    }


    proc printColumn {firstRecordOfTable level tuple} {
	variable reportData
	variable fieldNamesList
	variable printInfo
	variable maxLabelWidth

	set offset [string repeat " " [expr 4 * ($level - 1)]]
	# if { $firstRecordOfTable } then {
	set textToPrint "$offset----------------------------\n"
	# } else {
	#    set textToPrint ""
	#}
	foreach field $fieldNamesList($level) {
	    set formatString "%-$maxLabelWidth($level)\s"
	    set labelToPrint \
		"$offset[format $formatString $printInfo($field,label)] : "
	    set textToPrint "$textToPrint$labelToPrint"
	    set nextLineOffset [string repeat { } [string length $labelToPrint]]
	    set startIdx 0
	    set nlIdx [string first "\n" $reportData($tuple,$field) $startIdx]
	    while { $nlIdx >= 0 } {
		set textToPrint \
		    "$textToPrint[string range $reportData($tuple,$field) $startIdx $nlIdx]"
		set textToPrint "$textToPrint$nextLineOffset"
		set startIdx [expr $nlIdx + 1]
		set nlIdx [string first "\n" $reportData($tuple,$field) $startIdx]
	    }
	    set textToPrint \
		"$textToPrint[string range $reportData($tuple,$field) $startIdx end]\n"
	}
	return "$textToPrint\n"
    }


    proc printHeader {level} {
	variable reportData
	variable fieldNamesList
	variable printInfo

	set offset [string repeat " " [expr 4 * ($level - 1)]]
	set header ""
	set underline ""
	foreach field $fieldNamesList($level) {
	    switch $printInfo($field,alignment) {
		r {
		    set formatString "%$printInfo($field,columnWidth)\s"
		}
		l -
		default {
		    set formatString "%-$printInfo($field,columnWidth)\s"
		}
	    }
	    set header "$header| [format $formatString $printInfo($field,label)] "
	    set underline "$underline\+-[string repeat - $printInfo($field,columnWidth)]-"
	}
	set header [string replace $header 0 0 ""]
	set underline [string replace $underline 0 0 ""]
	return "$offset$header\n$offset$underline\n"
    }

    proc printTable {firstRecordOfTable level tuple} {
	variable reportData
	variable fieldNamesList
	variable printInfo


	if { $firstRecordOfTable } then {
	    set header [printHeader $level]
	} else {
	    set header ""
	}
	set offset [string repeat " " [expr 4 * ($level - 1)]]
	set lineToPrint ""
	foreach field $fieldNamesList($level) {
	    switch $printInfo($field,alignment) {
		r {
		    set formatString "%$printInfo($field,columnWidth)\s"
		}
		l -
		default {
		    set formatString "%-$printInfo($field,columnWidth)\s"
		}
	    }
	    set lineToPrint \
		"$lineToPrint| [format $formatString $reportData($tuple,$field)] "
	}
	set lineToPrint [string replace $lineToPrint 0 0 ""]
	return "$header$offset$lineToPrint\n"
    }
}