/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "tableschemaform.h"			
#include "comboboxtableitem.h"
#include "tablegrouplistviewitem.h"
#include "tablegroupitempopup.h"
#include "tableitempopup.h"
#include "../common/infodialogs.h"
#include "../../dataaccess/tables/dbtableschema.h"
#include "../common/constants.h"
#include "../../utils/debugger.h"
#include "../../utils/converter.h"
#include <qheader.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qstring.h>	
#include <qtable.h>	
#include <string.h>

	const int TableSchemaForm::COLUMN_NAME_IDX = 0;
	const int TableSchemaForm::COLUMN_TYPE_IDX = 1;
	const int TableSchemaForm::COLUMN_SIZE_IDX = 2;
	const int TableSchemaForm::COLUMN_PRIMARY_IDX = 3;	
	const int TableSchemaForm::COLUMN_NULLABLE_IDX = 4;	
	const int TableSchemaForm::COLUMN_DEFAULT_IDX = 5;	
	const int TableSchemaForm::COLUMN_COMMENTS_IDX = 6;			

	/**
 	 * Constructor
   */		
  TableSchemaForm::TableSchemaForm(QWidget * pqoParent, const DBConnection & roDBConn, const string & rstrTableName)
  	: UicTableSchemaForm(pqoParent)
  {
  	// all temporary values
  	QString qstrTemp1, qstrTemp2;
  	QCheckBox * pqchkTemp = 0;	// used to setup check boxes
  	QTableItem * pqoTableItem = 0;	// used to setup inputs
  	bool bTemp = false;
  	int nTemp = 0;
  	
  	// setup all instance variables
  	m_nRowCount = 0;
  	m_poTableListViewItem = 0;
  	m_bNewTable = true;
  	m_bSaved = false;
  	m_strTableName = rstrTableName;
  	
  	// set table name
  	m_pqtxtTableName->setText(m_strTableName.c_str());
  	setCaption("Table Schema [New]");
  	
  	// setup type list
  	// text types
  	m_qoTypeList << "text" << "varchar" << "char";
  	// number types
  	m_qoTypeList << "integer" << "int2" << "int8" << "oid" << "numeric" << "float" << "float4";
  	// date-time types
  	m_qoTypeList << "date" << "time" << "timestamp" << "interval";
  	// others
  	m_qoTypeList << "bool";
  	
  	// setup yes/no list
  	m_qoYesNoList << Constants::Q_YES << Constants::Q_NO;
  	
  	m_qoTypeList.sort();	// sort types
  	
  	QHeader * pqoVHeader = m_pqtblSchema->verticalHeader();
  	QHeader * pqoHHeader = m_pqtblSchema->horizontalHeader();
  	  	
  	// connect signals and slots
    connect(pqoVHeader, SIGNAL(clicked(int)), this, SLOT(processVerticalHeaderClicked(int)));
    connect(pqoVHeader, SIGNAL(pressed(int)), this, SLOT(processVerticalHeaderPressed(int)));
    connect(pqoVHeader, SIGNAL(released(int)), this, SLOT(processVerticalHeaderReleased(int)));
    connect(pqoHHeader, SIGNAL(clicked(int)), this, SLOT(processHorizontalHeaderClicked(int)));
    connect(pqoHHeader, SIGNAL(pressed(int)), this, SLOT(processHorizontalHeaderPressed(int)));
    connect(pqoHHeader, SIGNAL(released(int)), this, SLOT(processHorizontalHeaderReleased(int)));
                  	
  	// setup table headers

  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_NAME_IDX, QString("Column Name"), 100);
  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_TYPE_IDX, QString("Type"), 80);  	
  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_SIZE_IDX, QString("Size"), 40);  	  	
  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_PRIMARY_IDX, QString("Primary Key"), 80);  	  	  	
  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_NULLABLE_IDX, QString("Nullable"), 60);  	  	  	
  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_DEFAULT_IDX, QString("Default"), 120);  	  	  	  	
  	pqoHHeader->setLabel(TableSchemaForm::COLUMN_COMMENTS_IDX, QString("Comment"), 150);  	  	  	  	  	
  	  	
		// add one row
		addRow();

		// disable these buttons
		m_pqbtnRemoveField->setEnabled(false);
		m_pqbtnMoveUp->setEnabled(false);
		m_pqbtnMoveDown->setEnabled(false);
				  	  	
  	// setup new database connection
		m_oDBConn = roDBConn;		// copies all connection properties		
		// do connection
		try
		{
			m_oDBConn.connect();
		} // end try to connect
		catch (DBConnectionException e)
		{
			// report error to user
			InfoDialogs::infoDBConnectionException(this, e);		
			accept();		// lets get out now
			return;
		} // end catch exception
				
		m_oDBTableMgr.setDBConnection(&m_oDBConn);
		// ERROR HANDLING??
		
  } // end constructor

  /**
   * This function is used to populate this form with the schema details of an existing database table.
   */
  void TableSchemaForm::getSchemaDetails()
  {
  	// all temporary values
  	QString qstrTemp1, qstrTemp2;
  	//QCheckBox * pqchkTemp = 0;	// used to setup check boxes
  	QTableItem * pqoTableItem = 0;	// used to setup inputs
  	bool bTemp = false;
  	int nTemp = 0;
  	
  	setCaption("Table Schema");  	
  	// set to 0
  	m_nRowCount = 0;

  	try
  	{
	 		m_oDBTableMgr.retrieveTableSchema(m_strTableName, m_oDBTableSchema);
	 	} // end try to get table schema
	 	catch (SQLException e)
	 	{
			// report error to user
			InfoDialogs::infoSQLException(this, e);			
			return;	 	
	 	} // end catch exception	
		
 		// set the number of columns
 		m_pqtblSchema->setNumRows(m_oDBTableSchema.getNumberOfColumns());
		// loop through results and set table
		for (int nFieldIdx = 0; nFieldIdx < m_oDBTableSchema.getNumberOfColumns(); nFieldIdx++)
		{
			DBTableColumn oDBTableCol = m_oDBTableSchema.getColumn(nFieldIdx);
			qstrTemp1 = oDBTableCol.getFieldName().c_str();				
			m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_NAME_IDX, qstrTemp1);
			qstrTemp1 = oDBTableCol.getFieldType().c_str();				
			m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_TYPE_IDX, qstrTemp1);
			// check to see if numeric type
			if (oDBTableCol.isNumeric())
			{
				nTemp = oDBTableCol.getNumericPrecision();
				qstrTemp1.setNum(nTemp);
				qstrTemp1 += ", ";
				nTemp = oDBTableCol.getNumericDecimal();
				qstrTemp2.setNum(nTemp);
				qstrTemp1 += qstrTemp2;
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_SIZE_IDX, qstrTemp1);				
			} // end if numeric
			else if (oDBTableCol.getFieldType() == "varchar"
							 || oDBTableCol.getFieldType() == "char")
			{
				nTemp = oDBTableCol.getFieldSize();
				qstrTemp1.setNum(nTemp);
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_SIZE_IDX, qstrTemp1);								
			} // end else if varchar or char
			else
			{
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_SIZE_IDX, "");											
			} // end else

			bTemp = oDBTableCol.isPrimaryKey();				
			if (bTemp == true)
			{
				// set check box as yes
				qstrTemp1 = Constants::Q_YES;
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_PRIMARY_IDX, qstrTemp1);
			} // end if primary
			else
			{
				// set check box as no
				qstrTemp1 = Constants::Q_NO;
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_PRIMARY_IDX, qstrTemp1);
			} // end else not primary

			bTemp = oDBTableCol.isFieldNullable();				
			if (bTemp == true)
			{
				// set check box as YES
				qstrTemp1 = Constants::Q_YES;
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_NULLABLE_IDX, qstrTemp1);
			} // end if nullable
			else
			{
				// set check box as NO
				qstrTemp1 = Constants::Q_NO;
				m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_NULLABLE_IDX, qstrTemp1);
			} // end else not nullable

			string strDefault = oDBTableCol.getFieldDefault();
			// remove the quotes (if they exists) around the default field		
			if ((strDefault[0] == '\'' && strDefault[strDefault.size() - 1] == '\'')
					|| (strDefault[0] == '"' && strDefault[strDefault.size() - 1] == '"'))
			{
				strDefault = strDefault.substr(1, strDefault.size() - 2);
			} // end if default
			
			qstrTemp1 = strDefault.c_str();		
			m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_DEFAULT_IDX, qstrTemp1);
			qstrTemp1 = oDBTableCol.getFieldComment().c_str();				
			m_pqtblSchema->setText(nFieldIdx, TableSchemaForm::COLUMN_COMMENTS_IDX, qstrTemp1);
			
			m_nRowCount++;
		} // end for more fields		
		
		// lets update all flags
  	m_bNewTable = false;
  	m_bSaved = true;
  	m_pqbtnSave->setEnabled(false);
  			
  } // end getSchemaDetails

	/**
	 * This method is used to create a new table.
	 */
	void TableSchemaForm::createTableSchema()
	{	
		// clear all table selections
		m_pqtblSchema->clearSelection();	
		DBTableSchema oNewTableSchema;
		m_strTableName = (m_pqtxtTableName->text()).stripWhiteSpace().latin1();
		oNewTableSchema.setTableName(m_strTableName);
		// go through selected options and convert into db table schema object
		for (int nRowIdx = 0; nRowIdx < m_nRowCount; nRowIdx++)
		{
			// get field details out
			DBTableColumn oTableCol;
			getTableColumn(nRowIdx, oTableCol);
						
			// add column to table
			oNewTableSchema.addColumn(oTableCol);
		} // end for more rows
		
		try
		{
			m_oDBTableMgr.createTableSchema(oNewTableSchema);
		} // end try to create table schema
		catch (SQLException e)
		{
			// report error to user
			InfoDialogs::infoSQLException(this, e);					
			return;
		} // end catch exception		
		
		// lets add the table to the table list view item
		TableGroupListViewItem * poTableGroupItem = ((TableGroupItemPopup*)parentWidget())->getTableGroupListViewItem();
		DBTableSet oDBTableSet;	// used to retrieve the list of tables
		// lets retrieve the table just created
		m_oDBTableMgr.retrieveListOfTables(oDBTableSet, m_strTableName);
		if (oDBTableSet.next())
		{
			m_poTableListViewItem = new TableListViewItem(poTableGroupItem);
			oDBTableSet.getDBTable(*m_poTableListViewItem);
			// lets expand the table list too
			poTableGroupItem->setOpen(true);
			// if not expanded before, then can just expand
			if (poTableGroupItem->hasExpanded() == false)
			{
				poTableGroupItem->initGrandChildren();
			} // end if not expanded before
			
			m_poTableListViewItem->setText(0, QString(m_poTableListViewItem->getTableName().c_str()));					
			// set database connection
			m_poTableListViewItem->setDBMasterManager(poTableGroupItem->getDBMasterManager());
			m_poTableListViewItem->updateDescription();			
			m_poTableListViewItem->initChildren();
			// and set the selected item to the new table
			poTableGroupItem->listView()->setSelected(m_poTableListViewItem, true);
				
		} // end if table exists
	
		getSchemaDetails();
	} // end createTableSchema
	
	/**
	 * This method is used to update a table.
	 */
	void TableSchemaForm::updateTableSchema()
	{
		string strMethodName = "TableSchemaForm::updateTableSchema";
		Debugger::entered(strMethodName);
		
		// clear all table selections
		m_pqtblSchema->clearSelection();	
		// lets see if we need to get a reference to a table list view item
		if (m_poTableListViewItem == 0)
		{
			// lets get a reference to the table list view item for this table
			m_poTableListViewItem = ((TableItemPopup*)parentWidget())->getTableListViewItem();
		} // end if no reference to table list view item			
		
		// check to see if the table name has changed
		string strTempTableName = (m_pqtxtTableName->text()).stripWhiteSpace().latin1();
		if (m_oDBTableSchema.getTableName() != strTempTableName)
		{
			m_oDBTableMgr.alterTableName(strTempTableName, m_oDBTableSchema.getTableName());
			// update table name
			m_strTableName = strTempTableName;
			
			// lets update the table name in the list view item
			m_poTableListViewItem->setTableName(m_strTableName);
			m_poTableListViewItem->setText(0, m_strTableName.c_str());								
		} // end if table name has changed
		
		// first go through existing columns and run column alters
		for (int nRowIdx = 0; nRowIdx < m_oDBTableSchema.getNumberOfColumns(); nRowIdx++)
		{
			DBTableColumn oTableColNew;
			getTableColumn(nRowIdx, oTableColNew);
			DBTableColumn oTableColOld = m_oDBTableSchema.getColumn(nRowIdx);
			try
			{
		 		m_oDBTableMgr.alterTableColumn(m_strTableName, oTableColNew, oTableColOld);			
		 	} // end try to update table column
			catch (SQLException e)
			{
				// report error to user
				InfoDialogs::infoSQLException(this, e);					
				return;
			} // end catch exception		

		} // end for more existing rows to alter
		
		// go through new columns and add them
		for (int nRowIdx = m_oDBTableSchema.getNumberOfColumns(); nRowIdx < m_nRowCount; nRowIdx++)
		{
			DBTableColumn oTableColNew;
			getTableColumn(nRowIdx, oTableColNew);
			try
			{
				m_oDBTableMgr.addTableColumn(m_strTableName, oTableColNew);
			} // end try to add table column
			catch (SQLException e)
			{
				// report error to user
				InfoDialogs::infoSQLException(this, e);					
				return;
			} // end catch exception		
			
		} // end for more new columns

		// and set the selected item to the this one, this will repaint table name
		m_poTableListViewItem->updateDescription();
		m_poTableListViewItem->listView()->setSelected(m_poTableListViewItem, false);			
		m_poTableListViewItem->listView()->setSelected(m_poTableListViewItem, true); // so selected changed is emitted
		m_poTableListViewItem->repaint();		
		// need to retrieve results again from backend
		getSchemaDetails();
	
		Debugger::exited(strMethodName);
	} // end updateTableSchema
	
	/**
	 * Converts a row in the table into a DBTableColumn object.
	 */
	void TableSchemaForm::getTableColumn(int nRowIdx, DBTableColumn & roTableColumn) const
					throw (IndexOutOfRangeException)
	{
		string strMethodName = "TableSchemaForm::getTableColumn";
		Debugger::entered(strMethodName);
		
		// check to see if index is in range
		if (nRowIdx < 0 || nRowIdx >= m_nRowCount)
		{
			throw IndexOutOfRangeException("Index out of range.", "TableSchemaForm", "getTableColumn");			
		} // end if index is out of range
 		// get field details out
 		bool bIsNumeric = false;
 		string strFieldName = (m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_NAME_IDX)).stripWhiteSpace().latin1();
 		string strFieldType = (m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_TYPE_IDX)).stripWhiteSpace().latin1();
 		int nFieldSize = 0;
 		int nNumPrecision = 0;
 		int nNumDecimal = 0;
 		bool bIsPrimary = false;
 		bool bIsNullable = false;			
 		QString qstrFieldSize = (m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_SIZE_IDX)).stripWhiteSpace();
 		if (strFieldType == DBTableColumn::NUMERIC_FIELD_NAME)
 		{
 			bIsNumeric = true;
 			// need to find the comma seperator
 			int nCommaIdx = qstrFieldSize.find(",");
 			nNumPrecision = qstrFieldSize.mid(0, nCommaIdx).stripWhiteSpace().toInt();
 			nNumDecimal = qstrFieldSize.mid(nCommaIdx + 1).stripWhiteSpace().toInt();
 		} // end if type is numeric
 		else if (qstrFieldSize != "")
 		{
 			nFieldSize = qstrFieldSize.toInt();
 		} // end else if size exists for this type			
			
 		if ((m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_PRIMARY_IDX)).stripWhiteSpace().latin1()
 				== Constants::Q_YES)
 		{
 			bIsPrimary = true;
 		} // end if primary key
 		if ((m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_NULLABLE_IDX)).stripWhiteSpace().latin1()
 				== Constants::Q_YES)
 		{
 			bIsNullable = true;
 		} // end if primary key
 		string strDefault = (m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_DEFAULT_IDX)).stripWhiteSpace().latin1();								
 		string strComments = (m_pqtblSchema->text(nRowIdx, TableSchemaForm::COLUMN_COMMENTS_IDX)).stripWhiteSpace().latin1();											
			
 		if (bIsNumeric)
 		{
 			roTableColumn.setNumericFieldDetails(strFieldName, nNumPrecision, nNumDecimal, bIsPrimary,
 																				bIsNullable, strDefault, strComments);
 		} // end if numeric
 		else
 		{
 			roTableColumn.setFieldDetails(strFieldName, strFieldType, nFieldSize, bIsPrimary,
 																bIsNullable, strDefault, strComments);
 		} // end else not numeric			

	} // end getTableColumn
	
	/**
	 * This function is used to verify whether a row can be removed.
	 */
	bool TableSchemaForm::isRemovable(int nRow) const throw (IndexOutOfRangeException)
	{
		// check to see if index is in range
		if (nRow < 0 || nRow >= m_nRowCount)
		{
			throw IndexOutOfRangeException("Index out of range.", "TableSchemaForm", "isRemovable");			
		} // end if index is out of range
		
		// if the table is new then can always remove row
	  if (m_bNewTable == true)
	  {
	  	return true;
	  } // end if new table
	
	  // if the row is a new row then can remove, otherwise no
		if (nRow >= m_oDBTableSchema.getNumberOfColumns())
		{
			return true;
		}
	
		return false;
	} // end isRemovable

	/**
	 * This function is used to verify whether a row can move up.
	 */
	bool TableSchemaForm::isMoveableUp(int nRow) const throw (IndexOutOfRangeException)
	{
		// check to see if index is in range
		if (nRow < 0 || nRow >= m_nRowCount)
		{
			throw IndexOutOfRangeException("Index out of range.", "TableSchemaForm", "isMoveableUp");			
		} // end if index is out of range
	
		if (m_bNewTable == true)
		{
			if (nRow != 0)
			{
				return true;
			} // end if not the first row
		} // end if table is new			
		else
		{
			if (nRow > m_oDBTableSchema.getNumberOfColumns())
			{
				return true;
			} // end if row can be moved up	
		} // end else table is not new				
		
		// otherwise every other case if false
		return false;
	} // end isMoveableUp

	/**
	 * This function is used to verify whether a row can move down.
	 */
	bool TableSchemaForm::isMoveableDown(int nRow) const throw (IndexOutOfRangeException)
	{
		// check to see if index is in range
		if (nRow < 0 || nRow >= m_nRowCount)
		{
			throw IndexOutOfRangeException("Index out of range.", "TableSchemaForm", "isMoveableDown");			
		} // end if index is out of range
	
		if (m_bNewTable == true)
		{
			if (nRow != (m_nRowCount - 1))
			{
				return true;
			} // end if not the first row
		} // end if table is new			
		else
		{
			if (nRow >= m_oDBTableSchema.getNumberOfColumns()
					&& nRow != (m_nRowCount - 1))
			{
				return true;
			} // end if row can be moved up	
		} // end else table is not new				
		
		// otherwise every other case if false
		return false;	
	} // end isMoveableDown
	

	/**
	 * This slot is called when the close button is clicked.
	 */
	void TableSchemaForm::processCloseClicked()
	{
		close(true);
	} // end processCloseClicked

	/**
	 * This slot is called when the save button is clicked.
	 * This will call either createTableSchema or updateTableSchema
	 * depending on whether the table is new or modified.
	 */
	void TableSchemaForm::processSaveClicked()
	{
		if (m_bNewTable == true)
		{
			createTableSchema();
		} // end if table is new
		else
		{
			updateTableSchema();
		} // end else table has been modified
	} // end processSaveClicked	
				
  /**
 	 * This slot is called when a new row needs to inserted.
 	 * The new row is always appended to the end of the table.
 	 */
 	void TableSchemaForm::addRow()
 	{
 	  	// all temporary values
  	QString qstrTemp1, qstrTemp2;
  	QCheckBox * pqchkTemp = 0;	// used to setup check boxes
  	ComboBoxTableItem * poComboItem = 0;	// used to setup up combo options
  	QTableItem * pqoTableItem = 0;	// used to setup inputs

  	// adjust row size
  	m_pqtblSchema->setNumRows(m_nRowCount + 1);
  	
  	// need to clear out all text fields
		qstrTemp1 = "";				
		m_pqtblSchema->setText(m_nRowCount, TableSchemaForm::COLUMN_NAME_IDX, qstrTemp1);		
		m_pqtblSchema->setText(m_nRowCount, TableSchemaForm::COLUMN_TYPE_IDX, qstrTemp1);
		m_pqtblSchema->setText(m_nRowCount, TableSchemaForm::COLUMN_SIZE_IDX, qstrTemp1);				
		m_pqtblSchema->setText(m_nRowCount, TableSchemaForm::COLUMN_DEFAULT_IDX, qstrTemp1);
		m_pqtblSchema->setText(m_nRowCount, TableSchemaForm::COLUMN_COMMENTS_IDX, qstrTemp1);
  	
		// set up type list
		poComboItem = new ComboBoxTableItem(m_pqtblSchema, QTableItem::WhenCurrent, "");
		poComboItem->setStringList(m_qoTypeList);
 		m_pqtblSchema->setItem(m_nRowCount, TableSchemaForm::COLUMN_TYPE_IDX, poComboItem);		
		
 		// set yes no for primary key
		poComboItem = new ComboBoxTableItem(m_pqtblSchema, QTableItem::WhenCurrent, "");
		poComboItem->setStringList(m_qoYesNoList);
		poComboItem->setManualEdit(false);
 		m_pqtblSchema->setItem(m_nRowCount, TableSchemaForm::COLUMN_PRIMARY_IDX, poComboItem);		
 			
 		// set yes/no for not null
		poComboItem = new ComboBoxTableItem(m_pqtblSchema, QTableItem::WhenCurrent, "");
		poComboItem->setStringList(m_qoYesNoList);
		poComboItem->setManualEdit(false);		
 		m_pqtblSchema->setItem(m_nRowCount, TableSchemaForm::COLUMN_NULLABLE_IDX, poComboItem);		
 		
 		m_nRowCount++;
 		 		
 		// need to update buttons
 		processSelectionChanged();
 	} // end addRow

 	/**
 	 * This slot is called when a row needs to be deleted.
 	 */
 	void TableSchemaForm::removeRow()
 	{
		int nNumSelections = m_pqtblSchema->numSelections();

		int nRowsRemoved = 0;
		
		// go through all selections and move rows to the end
		for (int nSelection = 0; nSelection < nNumSelections; nSelection++)
		{
			QTableSelection qoTableSel = m_pqtblSchema->selection(nSelection);
			
			// go through all rows in selection and move each one to the end
			for (int nRowInSel = qoTableSel.topRow(); nRowInSel <= qoTableSel.bottomRow()
																								&& nRowInSel < m_nRowCount; nRowInSel++)
			{
				int nRowToMoveTo = nRowInSel + 1;
				while (nRowToMoveTo < m_nRowCount)
				{
					m_pqtblSchema->swapRows(nRowToMoveTo - 1, nRowToMoveTo);
					nRowToMoveTo++;	// keep moving till end
				} // end while rows to move down
				nRowsRemoved++;
			} // end for more rows in selection						
		} // end for more selections
				
		m_nRowCount = m_nRowCount - nRowsRemoved;
		// adjust row size
		m_pqtblSchema->setNumRows(m_nRowCount);
		
		// clear all selections after removed
		m_pqtblSchema->clearSelection();
		
 		// need to update buttons
 		processSelectionChanged();		
 	} // end removeRow
 	
 	/**
 	 * This will move the selected the row up.
 	 */
 	void TableSchemaForm::moveRowUp()
 	{
 		QTableSelection qoTableSel = m_pqtblSchema->selection(0);
 		int nRowSel = qoTableSel.topRow();
 		m_pqtblSchema->swapRows(nRowSel, nRowSel - 1);
 		// clear selection and move up to follow row
 		m_pqtblSchema->removeSelection(qoTableSel);
 		qoTableSel.init(nRowSel - 1, TableSchemaForm::COLUMN_NAME_IDX);
 		qoTableSel.expandTo(nRowSel - 1, TableSchemaForm::COLUMN_COMMENTS_IDX);
 		m_pqtblSchema->addSelection(qoTableSel);
 		// need to update buttons
 		processSelectionChanged(); 		
 	} // end moveRowUp
 	
 	/**
 	 * This will move the selected the row down.
 	 */
 	void TableSchemaForm::moveRowDown()
 	{
 		QTableSelection qoTableSel = m_pqtblSchema->selection(0);
 		int nRowSel = qoTableSel.topRow();
 		m_pqtblSchema->swapRows(nRowSel, nRowSel + 1);
 		// clear selection and move up to follow row
 		m_pqtblSchema->removeSelection(qoTableSel);
 		qoTableSel.init(nRowSel + 1, TableSchemaForm::COLUMN_NAME_IDX);
 		qoTableSel.expandTo(nRowSel + 1, TableSchemaForm::COLUMN_COMMENTS_IDX);
 		m_pqtblSchema->addSelection(qoTableSel);
 		// need to update buttons
 		processSelectionChanged(); 		
 	} // end moveRowDown
 	
 	/**
 	 * This will be called when the table selection has changed.
 	 */
 	void TableSchemaForm::processSelectionChanged()
 	{
		int nNumSelections = m_pqtblSchema->numSelections();
								
		// if no selections then, must turn off remove and move buttons
		if (nNumSelections == 0)
		{
			m_pqbtnRemoveField->setEnabled(false);
			m_pqbtnMoveUp->setEnabled(false);
			m_pqbtnMoveDown->setEnabled(false);
			return;			
		} // end if no selections		
		
		// check whether move buttons should be enabled or disabled
		if (nNumSelections != 1)
		{
			// can only have singular selection for move buttons
			m_pqbtnMoveUp->setEnabled(false);
			m_pqbtnMoveDown->setEnabled(false);		
		} // end if not single selection
		else
		{
			// get table selection
			QTableSelection qoTableSel = m_pqtblSchema->selection(0);
			if (qoTableSel.topRow() == qoTableSel.bottomRow()
					&& qoTableSel.leftCol() == TableSchemaForm::COLUMN_NAME_IDX
					&& qoTableSel.rightCol() == TableSchemaForm::COLUMN_COMMENTS_IDX)
			{
				if (isMoveableUp(qoTableSel.topRow()))
				{
					m_pqbtnMoveUp->setEnabled(true);
				} // end if can move up
				else
				{
					m_pqbtnMoveUp->setEnabled(false);
				} // end else can't move up
				if (isMoveableDown(qoTableSel.bottomRow()))
				{
					m_pqbtnMoveDown->setEnabled(true);		
				} // end if can move down
				else
				{
					m_pqbtnMoveDown->setEnabled(false);		
				} // end else can't move down
			} // end if only one row selected (all of row)
			else
			{
				m_pqbtnMoveUp->setEnabled(false);
				m_pqbtnMoveDown->setEnabled(false);				
			} // end else not one row selected
		}
		
		bool bRemoveEnabled = true;
		// for remove button, just look for any selection that cannot be removed
		// or look for any selection that is not an entire row
		for (int nSelection = 0; nSelection < nNumSelections; nSelection++)
		{
			QTableSelection qoTableSel = m_pqtblSchema->selection(nSelection);
			
			// go through all rows in selection
			for (int nRowInSel = qoTableSel.topRow(); nRowInSel <= qoTableSel.bottomRow()
																								&& nRowInSel < m_nRowCount; nRowInSel++)
			{
				if (qoTableSel.leftCol() != TableSchemaForm::COLUMN_NAME_IDX
						|| qoTableSel.rightCol() != TableSchemaForm::COLUMN_COMMENTS_IDX)
				{
					bRemoveEnabled = false;
					break;				
				} // end if not an entire row was selected						
				if (isRemovable(nRowInSel) == false)
				{
					bRemoveEnabled = false;
					break;
				} // end if this row is not removable
			} // end for more rows in selection
			
			if (bRemoveEnabled == false)
			{
				break;
			} // end if remove button is disabled
		} // end for more selections
 	
		m_pqbtnRemoveField->setEnabled(bRemoveEnabled);		
 	} // end processSelectionChanged

	/**
	 * This slot is called whenever the value of the table name changes.
	 */
  void TableSchemaForm::processTableNameChanged(const QString &)
  {
 		if (m_bSaved == true)
 		{
 			m_bSaved = false;
 			m_pqbtnSave->setEnabled(true);
 		} // end if need previously saved  	
  } // end processTableNameChanged
 	 	
 	/**
 	 * This will be called when the value of a table column is changed.
 	 */
 	void TableSchemaForm::processValueChanged(int nRow, int nCol)
 	{
 		if (m_bSaved == true)
 		{
 			m_bSaved = false;
 			m_pqbtnSave->setEnabled(true);
 		} // end if need previously saved
 	} // end processValueChanged 	 	

 	/**
 	 * This will be called when any of the headers are clicked
 	 */
 	void TableSchemaForm::processVerticalHeaderClicked(int nRow)
 	{
 		processSelectionChanged();
 	} // end processVerticalHeaderClicked

 	/**
 	 * This will be called when the header is pressed.
 	 */
 	void TableSchemaForm::processVerticalHeaderPressed(int nRow)
 	{
 	} // end processVerticalHeaderPressed  	 	 	

 	/**
 	 * This will be called when the header is released.
 	 */
 	void TableSchemaForm::processVerticalHeaderReleased(int nRow)
 	{
 		processSelectionChanged();
 	} // end processVerticalHeaderReleased  	 	 	
 	
 	/**
 	 * This will be called when any of the headers are clicked.
 	 */
 	void TableSchemaForm::processHorizontalHeaderClicked(int nColumn)
 	{
 		processSelectionChanged();
 	} // end processHeaderClicked

 	/**
 	 * This will be called when the header is pressed.
 	 */
 	void TableSchemaForm::processHorizontalHeaderPressed(int nColumn)
 	{
 	} // end processHorizontalHeaderPressed

 	/**
 	 * This will be called when the header is released.
 	 */
 	void TableSchemaForm::processHorizontalHeaderReleased(int nColumn) 	
 	{
 		processSelectionChanged(); 		 	 	 	 	
 	} // end processHorizontalHeaderReleased