/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbtypeset.h"
#include "../dbconnection.h"
#include "../../utils/debugger.h"
#include "../../utils/converter.h"

	/**
 	 * Constructor 	
   */		
  DBTypeSet::DBTypeSet()
  {
  	// do nothing
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBTypeSet::~DBTypeSet()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBTypeSet::isEmpty()
	{
		return m_oTypeList.isEmpty();
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBTypeSet::getCurrentIndex()
	{
		return m_oTypeList.getCurrentIndex();
	} // end getCurrentIndex
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBTypeSet::next()
	{
		return m_oTypeList.next();	
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBTypeSet::previous()
	{
		return m_oTypeList.previous();
	} // end previous

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBTypeSet::reset()
	{
		m_oTypeList.reset();
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBTypeSet::first() throw (NoRecordException)
	{
		return m_oTypeList.first();
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBTypeSet::last() throw (NoRecordException)
	{
		return m_oTypeList.last();	
	} // end last
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBTypeSet::getRecordCount()
	{
		return m_oTypeList.getRecordCount();		
	} // end getRecordCount

	/**
	 * Used to get the current Type.
	 * @param	roDBType the a DBType object used to return output for current record
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBTypeSet::getDBType(DBType & roDBType) throw (NoRecordException)
	{
		string strMethodName = "DBTypeSet::getDBType";
		
		Debugger::entered(strMethodName);
		
		if (isEmpty())
		{
			throw NoRecordException("Cannot retrieve DBType object, no records exist.", "DBTypeSet", "getDBType");
		} // end if empty
		try
		{
			roDBType.setTypeName(m_oTypeList.getFieldValue("typename"));
			roDBType.setTypeOid(Converter::stringToLong(m_oTypeList.getFieldValue("typeoid")));			
		} // end try
		catch (VPException e)
		{
			Debugger::logException(strMethodName, e);
		} // end catch	
		
		Debugger::exited(strMethodName);
		
	} // end getDBType

