/*
	dbagent.cpp

	The DbAgent class implements functionality specific to a database agent.

	Project: pgjobs
	Author: Zlatko Michailov
	Created: 13-Oct-2003
	Updated: 14-Oct-2003
	Updated: 28-Oct-2003
	Updated:  5-May-2004
	Updated: 26-May-2004
	Updated: 27-May-2004
	Updated:  1-Jun-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003-2004, Zlatko Michailov

*/



//--------------------------------------------------------------------------------

#include <stdlib.h>
#include <errno.h>
#include "dbagent.h"


using namespace msdk;



//--------------------------------------------------------------------------------
// Construction

DbAgent::DbAgent( const DbInfo& dbInfo ) :
	Agent( dbInfo.ID.c_str(), atoi( dbInfo.LogLevel.c_str() ) ),
	Thread()
{
	DbConfig = dbInfo;

	// Translate default user name
	if ( DbConfig.User == Default )
	{
		DbConfig.User = Postgres;
	}

	ComputePsqlOptions();
}



//--------------------------------------------------------------------------------
// Overridables

void DbAgent::Startup()
{
	// Init log
	IsReadyToRun = InitLog();
}



void DbAgent::Shutdown()
{
	Status = StatusShutdown;
	Record( "Down", LogLevelSystemInfo );
}



void DbAgent::Run()
{
	if ( IsReadyToRun )
	{
		Status = StatusRunning;
		Record( "Up and running", LogLevelSystemInfo );

		SQL sql( PsqlOptions.c_str(), DbConfig.User.c_str() );

		// Init session
		InitSession( sql );

		// Execute startup jobs
		ExecuteStartupJobs( sql );

		// Infinite loop until stopped by MasterAgent
		for ( ;; )
		{
			int  sleepSeconds	= ComputeSleepSeconds( sql );
			int  sleepHH		= sleepSeconds / 3600;
			int  sleepMM		= ( sleepSeconds / 60 ) % 60;
			int  sleepSS		= sleepSeconds % 60;
			char sleepRecord[ 256 ];

			// Record sleeping
			sprintf( sleepRecord, "Sleep for %u:%2.2u:%2.2u",
									sleepHH, sleepMM, sleepSS );
			Record( sleepRecord, LogLevelSystemVerbose );

			// Sleep
			Sleep( sleepSeconds );

			// Execute jobs
			ExecuteJobs( sql );
		}
	}
	else
	{
		Record( "Not able to run! Shutting down...", LogLevelError );
	}
}



//--------------------------------------------------------------------------------
// Support

void DbAgent::ComputePsqlOptions()
{
	PsqlOptions.clear();

	PsqlOptions  = " -d \"";
	PsqlOptions += DbConfig.Database;
	PsqlOptions += "\" ";

	if ( DbConfig.Host != Default )
	{
		PsqlOptions += " -h \"";
		PsqlOptions += DbConfig.Host;
		PsqlOptions += "\" ";
	}

	if ( DbConfig.Port != Default )
	{
		PsqlOptions += " -p \"";
		PsqlOptions += DbConfig.Port;
		PsqlOptions += "\" ";
	}
}



void DbAgent::InitSession( SQL& sql )
{
	Record( "Initializing session...", LogLevelSystemVerbose );
	
	sql.ExecCmd( "select pgj_startup();" );

	Record( "Session initialized", LogLevelSystemInfo );
}



int DbAgent::ComputeSleepSeconds( SQL& sql )
{
	const int MAX_SLEEP_SECONDS = 60 * 60; // 1 hour

	int			seconds = MAX_SLEEP_SECONDS;
	SQL::Rowset	rowset;
	bool		ok;

	// Recompute schedules
	ok = sql.ExecCmd( "select pgj_recompute_schedules();" );

	// Fetch 1 row x 1 column: trigger seconds
	if ( ok )
	{
		ok = sql.FetchCmdRowset( "select pgj_trigger_seconds();", rowset );
	}

	// Process rowset
	if ( ok && rowset.size() != 0 )
	{
		int triggerSeconds = atoi( rowset[ 0 ][ 0 ].c_str() );

		if ( triggerSeconds > 0 && triggerSeconds < MAX_SLEEP_SECONDS )
		{
			seconds = triggerSeconds;
		}
	}

	return seconds;
}



void DbAgent::ExecuteStartupJobs( SQL& sql )
{
	SQL::Rowset	rowset;
	bool		ok;

	Record( "Executing startup jobs...", LogLevelSystemVerbose );

	// Fetch n rows x 3 columns: jobid, jobname, jobfunction
	ok = sql.FetchCmdRowset( "select * from pgj_jobs_startup;", rowset );

	// Process rowset
	for ( int i = 0; ok && i < rowset.size(); i++ )
	{
		string cmd;

		cmd  = "select ";
		cmd += rowset[ i ][ 2 ];
		cmd += ";";

		// Execute function
		sql.ExecCmd( cmd.c_str() );

		Record( rowset[ i ][ 2 ].c_str(), LogLevelSystemVerbose );
	}

	Record( "Startup jobs executed", LogLevelSystemVerbose );
}



void DbAgent::ExecuteJobs( SQL& sql )
{
	SQL::Rowset	rowset;
	bool		ok;

	Record( "Executing jobs...", LogLevelSystemVerbose );

	// Recompute schedules
	ok = sql.ExecCmd( "select pgj_recompute_schedules();" );

	// Fetch n rows x 3 columns: jobid, jobname, jobfunction
	if ( ok )
	{
		ok = sql.FetchCmdRowset( "select * from pgj_jobs_up;", rowset );
	}

	// Process rowset
	for ( int i = 0; ok && i < rowset.size(); i++ )
	{
		string cmd;

		cmd  = "select ";
		cmd += rowset[ i ][ 2 ];
		cmd += ";";

		// Execute function
		sql.ExecCmd( cmd.c_str() );

		Record( rowset[ i ][ 2 ].c_str(), LogLevelSystemVerbose );
	}

	Record( "Jobs executed", LogLevelSystemVerbose );
}


