/*-------------------------------------------------------------------------
 *
 * proc.c--
 *	  routines to manage per-process shared memory data structure
 *
 * Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  $Header: /home/postgres/cvs_root/postgres-r/src/backend/storage/lmgr/proc.c,v 1.2 2001/06/12 14:51:59 johnsond Exp $
 *
 *-------------------------------------------------------------------------
 */
/*
 *	Each postgres backend gets one of these.  We'll use it to
 *	clean up after the process should the process suddenly die.
 *
 *
 * Interface (a):
 *		ProcSleep(), ProcWakeup(), ProcWakeupNext(),
 *		ProcQueueAlloc() -- create a shm queue for sleeping processes
 *		ProcQueueInit() -- create a queue without allocing memory
 *
 * Locking and waiting for buffers can cause the backend to be
 * put to sleep.  Whoever releases the lock, etc. wakes the
 * process up again (and gives it an error code so it knows
 * whether it was awoken on an error condition).
 *
 * Interface (b):
 *
 * ProcReleaseLocks -- frees the locks associated with this process,
 * ProcKill -- destroys the shared memory state (and locks)
 *		associated with the process.
 *
 * 5/15/91 -- removed the buffer pool based lock chain in favor
 *		of a shared memory lock chain.	The write-protection is
 *		more expensive if the lock chain is in the buffer pool.
 *		The only reason I kept the lock chain in the buffer pool
 *		in the first place was to allow the lock table to grow larger
 *		than available shared memory and that isn't going to work
 *		without a lot of unimplemented support anyway.
 *
 * 4/7/95 -- instead of allocating a set of 1 semaphore per process, we
 *		allocate a semaphore from a set of PROC_NSEMS_PER_SET semaphores
 *		shared among backends (we keep a few sets of semaphores around).
 *		This is so that we can support more backends. (system-wide semaphore
 *		sets run out pretty fast.)				  -ay 4/95
 *	1/19/99 -- (wb) Altered process management in order to allow one process to
 *		wait in several lock wait queues. This is necessary when running
 *		in replication mode in order to acquire all locks associated with
 *		a transaction's write set in one atomic step. Wait queues now no longer
 *		contain process headers, but process shadows, which are structures
 *		pointing to the actual process header. Shadows are kept in a hashtable
 *		which uses as key the pid of the process and the offset of the lock 
 *		record in shared memory the shadow waits on. Thus, for each process, 
 *		there can be only one shadow per lock. Generally speaking, operations on
 *		PROC types have been replaced by operations on PROC_SHADOW types. The PROC 
 *		struct for each process still exists, it can be accessed via its shmem offset
 *		stored in its shadow. Data originally stored in the PROC struct is now 
 *		partially stored in the PROC_SHADOW struct (see proc.h)
 *			 ATTN: Requesting a shadow twice is considered as error behaviour
 *			 As of today, this situation never occurs at runtime, since a process
 *			 either sleeps on a lock, or requests several write locks at once.
 *			 In the latter case, the requesting procedure avoids requesting
 *			 the same lock twice. However, when implementing a read lock upgrade
 *			 mechanism, this situation will occur...
 * $Header: /home/postgres/cvs_root/postgres-r/src/backend/storage/lmgr/proc.c,v 1.2 2001/06/12 14:51:59 johnsond Exp $
 */
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>


#if defined(solaris_sparc)
#include <sys/ipc.h>
#include <sys/sem.h>
#endif

#include "postgres.h"
#include "miscadmin.h"
#include "libpq/pqsignal.h"

#include "access/xact.h"
#include "utils/hsearch.h"

#include "storage/ipc.h"
/* In Ultrix, sem.h must be included after ipc.h */
#include <sys/sem.h>
#include "storage/buf.h"
#include "storage/lock.h"
#include "storage/lmgr.h"
#include "storage/multilev.h"
#include "storage/shmem.h"
#include "storage/spin.h"
#include "storage/proc.h"
#include "utils/trace.h"
#include "replication/replication.h"

static void HandleAlarm(int sig);
static void HandleDeadLock(int sig);
static PROC_SHADOW *ProcWakeup(PROC_SHADOW *proc, int errType);
static void ProcAdjustSemaphore(SEMA *sem, int exceedUps);
void ProcRemoveAllWaitingLocks();

#define DeadlockCheckTimer pg_options[OPT_DEADLOCKTIMEOUT]



/* --------------------
 * Spin lock for manipulating the shared process data structure:
 * ProcGlobal.... Adding an extra spin lock seemed like the smallest
 * hack to get around reading and updating this structure in shared
 * memory. -mer 17 July 1991
 * --------------------
 */
SPINLOCK	ProcStructLock;



/*
 * For cleanup routines.  Don't cleanup if the initialization
 * has not happened.
 */
static bool ProcInitialized = FALSE;

static PROC_HDR *ProcGlobal = NULL;

static HTAB *shadow_table = NULL;

PROC	   *MyProc = NULL;

static void ProcKill(int exitStatus, int pid);
static void ProcGetNewSemKeyAndNum(IPCKey *key, int *semNum);
static void ProcFreeSem(IpcSemaphoreKey semKey, int semNum);

/* (wb) Prototypes for process shadow creation and deletion*/
static PROC_SHADOW * GetShadow(const LOCK *lock);
static int DeleteShadow(int pid, const LOCK *lock);

/*
 * InitProcGlobal -
 *	  initializes the global process table. We put it here so that
 *	  the postmaster can do this initialization. (ProcFreeAllSem needs
 *	  to read this table on exiting the postmaster. If we have the first
 *	  backend do this, starting up and killing the postmaster without
 *	  starting any backends will be a problem.)
 */
void
InitProcGlobal(IPCKey key)
{
	bool		found = false;

	/* attach to the free list */
	ProcGlobal = (PROC_HDR *)
		ShmemInitStruct("Proc Header", (unsigned) sizeof(PROC_HDR), &found);

	/* --------------------
	 * We're the first - initialize.
	 * --------------------
	 */
	if (!found)
	{
		int			i;

		ProcGlobal->freeProcs = INVALID_OFFSET;
		ProcGlobal->currKey = IPCGetProcessSemaphoreInitKey(key);
		for (i = 0; i < MAX_PROC_SEMS / PROC_NSEMS_PER_SET; i++)
			ProcGlobal->freeSemMap[i] = 0;
	}
	
}

/* ------------------------
 * InitProc -- create a per-process data structure for this process
 * used by the lock manager on semaphore queues.
 *
 * (wb) Attaches to the process shadow table
 * ------------------------
 */
void
InitProcess(IPCKey key)
{
	bool		found = false;
	int			semstat;
	unsigned long location,
				myOffset;
	HASHCTL		info;
	int			hash_flags;

	/* ------------------
	 * Routine called if deadlock timer goes off. See ProcSleep()
	 * ------------------
	 */



	pqsignal(SIGALRM, HandleAlarm);

	SpinAcquire(ProcStructLock);


	/* attach to the free list */
	ProcGlobal = (PROC_HDR *)
		ShmemInitStruct("Proc Header", (unsigned) sizeof(PROC_HDR), &found);
	if (!found)
	{
		/* this should not happen. InitProcGlobal() is called before this. */
		elog(ERROR, "InitProcess: Proc Header uninitialized");
	}
	
	
	/*(wb) attach to the process shadow table*/
	
	info.keysize = sizeof(SHDWTAG);
	info.datasize = sizeof(PROC_SHADOW);
	info.hash = tag_hash;
	hash_flags = (HASH_ELEM | HASH_FUNCTION);
	/*
	* (wb)
	* The lockMgr must be locked since we are attaching to a data structure 
	* used by the lockMgr functions.
	* This should not be a performance problem since this function is only called once
	* for each backend at backend startup time
	*/
	LockLockTable();
	shadow_table = (HTAB *) ShmemInitHash("shadow hash", INIT_TABLE_SIZE, 
										  MAX_TABLE_SIZE, &info, hash_flags);
	if(!shadow_table){
		elog(FATAL, "InitProcGlobal: could not initialize/attach shadow hash table");
	} 
	UnlockLockTable();
	
	if (MyProc != NULL)
	{
		SpinRelease(ProcStructLock);
		elog(ERROR, "ProcInit: you already exist");
		return;
	}

	/* try to get a proc from the free list first */

	myOffset = ProcGlobal->freeProcs;

	if (myOffset != INVALID_OFFSET)
	{
		MyProc = (PROC *) MAKE_PTR(myOffset);
		ProcGlobal->freeProcs = MyProc->links.next;
	}
	else
	{
		/*
		 * have to allocate one.  We can't use the normal shmem index
		 * table mechanism because the proc structure is stored by PID
		 * instead of by a global name (need to look it up by PID when we
		 * cleanup dead processes).
		 */

		MyProc = (PROC *) ShmemAlloc((unsigned) sizeof(PROC));
		if (!MyProc)
		{
			SpinRelease(ProcStructLock);
			elog(FATAL, "cannot create new proc: out of memory");
		}

		/* this cannot be initialized until after the buffer pool */
		SHMQueueInit(&(MyProc->lockQueue));
		SHMQueueInit(&(MyProc->shadowList));
	}

	/*
	 * zero out the spin lock counts and set the sLocks field for
	 * ProcStructLock to 1 as we have acquired this spinlock above but
	 * didn't record it since we didn't have MyProc until now.
	 */
	MemSet(MyProc->sLocks, 0, sizeof(MyProc->sLocks));
	MyProc->sLocks[ProcStructLock] = 1;
	
	/* bk: initialize connection state */
	if (txn_type != NOT_REPLICATED);
	  MyProc->connection_state = NOT_YET_CONNECTED;


	if (IsUnderPostmaster)
	{
		IPCKey		semKey;
		int			semNum;
		int			semId;
		union semun semun;


		ProcGetNewSemKeyAndNum(&semKey, &semNum);

		
		semId = IpcSemaphoreCreate(semKey,
								   PROC_NSEMS_PER_SET,
								   IPCProtection,
								   IpcSemaphoreDefaultStartValue,
								   0,
								   &semstat);


		/*
		 * we might be reusing a semaphore that belongs to a dead backend.
		 * So be careful and reinitialize its value here.
		 */
		semun.val = IpcSemaphoreDefaultStartValue;
		semctl(semId, semNum, SETVAL, semun);

		IpcSemaphoreLock(semId, semNum, IpcExclusiveLock);
		MyProc->sem.semId = semId;
		MyProc->sem.semNum = semNum;
		MyProc->sem.semKey = semKey;
		
		
		/* bk: now do the same for semaphore for multiple acquisition */
		ProcGetNewSemKeyAndNum(&semKey, &semNum);

		semId = IpcSemaphoreCreate(semKey,
								   PROC_NSEMS_PER_SET,
								   IPCProtection,
								   IpcSemaphoreDefaultStartValue,
								   0,
								   &semstat);

		semun.val = IpcSemaphoreDefaultStartValue;
		semctl(semId, semNum, SETVAL, semun);



		IpcSemaphoreLock(semId, semNum, IpcExclusiveLock);
		MyProc->sem_multiple.semId = semId;
		MyProc->sem_multiple.semNum = semNum;
		MyProc->sem_multiple.semKey = semKey;
		

	}
	else
	{
		MyProc->sem.semId = -1;
		MyProc->sem_multiple.semId = -1;
	}


	/* ----------------------
	 * Release the lock.
	 * ----------------------
	 */
	SpinRelease(ProcStructLock);

	MyProc->pid = MyProcPid;
	MyProc->abortFlag = NO_ABORT;
	MyProc->xid = InvalidTransactionId;
#ifdef LowLevelLocking
	MyProc->xmin = InvalidTransactionId;
#endif
	/* ----------------
	 * Start keeping spin lock stats from here on.	Any botch before
	 * this initialization is forever botched
	 * ----------------
	 */
	MemSet(MyProc->sLocks, 0, MAX_SPINS * sizeof(*MyProc->sLocks));

	/* -------------------------
	 * Install ourselves in the shmem index table.	The name to
	 * use is determined by the OS-assigned process id.  That
	 * allows the cleanup process to find us after any untimely
	 * exit.
	 * -------------------------
	 */
	location = MAKE_OFFSET(MyProc);
	if ((!ShmemPIDLookup(MyProcPid, &location)) || (location != MAKE_OFFSET(MyProc)))
		elog(FATAL, "InitProc: ShmemPID table broken");

	MyProc->errType = NO_ERROR;

	on_shmem_exit(ProcKill, (caddr_t) MyProcPid);

	ProcInitialized = TRUE;
}

/*
*	(wb) GetShadow
*
*	initializes a new shadow struct for the given process and a given
*	lock queue. ATTN: Assumes the LockMgr spinlock is already acquired!
*
*	param:
*		lock - the lock to create the shadow for
*	return:
*		pointer to the shadow
*/
static PROC_SHADOW *
GetShadow(const LOCK *lock)
{
	bool		 	found = false;
	PROC_SHADOW		search_shdw,
					*result_shdw;
					
	if(!ProcInitialized){
			elog(ERROR, "GetShadow: no shadow table.");
			return NULL;
		}



#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]getting shadow... ", MyProcPid);
#endif

	/* get a shadow for the specified lock*/

	MemSet(&search_shdw, 0, sizeof(SHDWTAG));
	search_shdw.tag.pid = MyProcPid;
	search_shdw.tag.lock = MAKE_OFFSET(lock);
	result_shdw = (PROC_SHADOW *) hash_search(shadow_table, (Pointer) &search_shdw,
													HASH_ENTER, &found); 
	
	if(found){
		/*not supposed to happen, EXCEPT with read lock upgrade (not implemented)*/
		return NULL;
	}
	SHMQueueElemInit(&(result_shdw->waitLinks));
	SHMQueueElemInit(&(result_shdw->shadowLinks));
	result_shdw->proc = MAKE_OFFSET(MyProc);	
#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]inserting into shadowList no %d",MyProcPid, MAKE_OFFSET(result_shdw));
	dumpQ(&MyProc->shadowList, "\tbefore:");
#endif
	SHMQueueInsertTL(&(MyProc->shadowList), &(result_shdw->shadowLinks));
#ifdef PROC_DEBUG
	dumpQ(&MyProc->shadowList, "\tafter:");
#endif
//	printf("generated shadow no. %d (pid %d, lock %d\n", MAKE_OFFSET(result_shdw), result_shdw->tag.pid, result_shdw->tag.lock);
//	fflush(stdout);
	return result_shdw;
}

/*
*	(wb)
*	DeleteShadow - deletes a process'es shadow
*		ATTN: Assumes the LockMgr spinlock is acquired
*	
*	params:
*		proc - the proc associated to the shadow
*		lock - the lock this shadow was waiting on
*	return:
*		int - 0 error, 1 sucess
*						
*	Note: 	Be sure to remove the shadow from its lock queue  
*	before calling delete.
*/
static int
DeleteShadow(int pid, const LOCK *lock){
	
	bool			found = FALSE ;
	PROC_SHADOW 	search_shdw,
					*result_shdw;
	PROC			*proc;
	
	/*look if shadow exists*/
	MemSet(&search_shdw, 0, sizeof(SHDWTAG));
	search_shdw.tag.pid = pid;
	search_shdw.tag.lock = MAKE_OFFSET(lock);
	
	result_shdw = (PROC_SHADOW *) hash_search(shadow_table, (Pointer) &search_shdw,
													HASH_FIND, &found);
										
	if(!found || !result_shdw || (result_shdw->waitLinks.next != INVALID_OFFSET &&
				  result_shdw->waitLinks.prev != INVALID_OFFSET)){
#ifdef PROC_DEBUG
		elog(NOTICE, "[%d]shadow to be deleted not found!", MyProcPid);
#endif
		return 0;
	}
	
	proc = (PROC *) MAKE_PTR(result_shdw->proc);
	/*remove shadow from shadowList*/
	/* shadow->shadowLinks == proc->shadowList */
#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]removing from shadowList %d",MyProcPid,MAKE_OFFSET(result_shdw));
	dumpQ(&proc->shadowList, "\tbefore:");
#endif
	SHMQueueDelete(&(result_shdw->shadowLinks));
	SHMQueueElemInit(&(result_shdw->shadowLinks));
#ifdef PROC_DEBUG
	dumpQ(&proc->shadowList, "\tafter:");
#endif
	/*delete shadow*/
#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]deleting shadow no. %d", MyProcPid, MAKE_OFFSET(result_shdw));
#endif
	result_shdw = (PROC_SHADOW *) hash_search(shadow_table, (Pointer) &search_shdw,
															HASH_REMOVE, &found);
	if(!found || !result_shdw){
		elog(FATAL, "deleteShadow: shadow_table corrupted");
		return 0;
	}
	
#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]...done (shadow is dead)",MyProcPid);
#endif
//	printf("deleted shadow %d (pid %d, lock %d)\n", MAKE_OFFSET(result_shdw), result_shdw->tag.pid, result_shdw->tag.lock);
//	fflush(stdout);
	return 1;
}

/*
*	ProcMarkAbortedReaders (wb)
*
*	Readers may be aborted by a writer if their own write set 
*	has not yet been processed. This procedure goes through the
*	list of lock holders and sets the abort flag if
*	they have granted read ops and their write set has not yet 
*	been received.
*
*	param:
*		lock - lock  to be checked for internal aborts
*	return:
*		void  
*/
static void
ProcMarkAbortedReaders(LOCK *lock)
{
	SHM_QUEUE   *grantedQueue;
	SHMEM_OFFSET	proc_offset;
	XIDLookupEnt	*grantedQ_XLE;
	PROC			*proc;
	int				i;
	abortFlagType			old_flag;

	grantedQueue = &lock->grantedQueue.links;
	for(i = 0;i < lock->grantedQueue.size; i++)
	{
		SHMQueueFirst(grantedQueue, (Pointer *) &grantedQ_XLE, &grantedQ_XLE->grantedLink);
		if(grantedQ_XLE->abortpid != MyProcPid &&
		   (grantedQ_XLE->holders[READ_LOCK] > 0 || 
			 grantedQ_XLE->holders[READ_FOR_WRITE] > 0))
		{
			if(ShmemPIDLookup(grantedQ_XLE->abortpid, &proc_offset))
			{
				proc = (PROC *) MAKE_PTR(proc_offset);
				if(proc->abortFlag == NO_ABORT && (proc->state == READ_PHASE ||
				   proc->state == SEND_PHASE))
				{
					elog(NOTICE,"Setting Abort flag for txn %x (pid %d)\n", 
					  grantedQ_XLE->tag.xid, grantedQ_XLE->abortpid);
					fflush(stdout);
					old_flag = proc->abortFlag;
					proc->abortFlag = MUST_ABORT;
					/*
					* remove the aborted proc's shadows from the wait queues, or
					* next wakeup will fail...(the process will try to wake itself)
					*/
					/* bk: not ProcRemoveWaiting must derecement also the
					* nholding counters if the proc was waiting for a lock
					*/
					ProcRemoveWaiting(proc);
					/*elog(NOTICE,"before sem 3");*/
					if(IpcSemaphoreGetCount(proc->sem.semId, 
											proc->sem.semNum) > 0)
					{

						proc->errType = ERR_INT_ABORT;
						
						IpcSemaphoreUnlock(proc->sem.semId, 
										   proc->sem.semNum, 
										   IpcExclusiveLock);
					}
					if (proc->state == SEND_PHASE && old_flag == NO_ABORT)
						  {
							elog(NOTICE,"is really in send phase %d", proc->pid);
							kill(proc->pid, SIGQUIT);
						  }

				}
			}
			else
			{
				proc_exit(600);
			}
		}
		grantedQueue = &grantedQ_XLE->grantedLink;
	}//end for
	
}


/*
 * ProcReleaseLocks() -- release all locks associated with this process
 *
 */
void
ProcReleaseLocks()
{
	if (!MyProc)
		return;
	LockReleaseAll(1, &MyProc->lockQueue);
	
	/* bk: reset semaphores */
	if (txn_type != NOT_REPLICATED)
	{
		int			semNum;
		int			semId;
		union semun semun;
	
		semun.val = IpcSemaphoreDefaultStartValue;
		semId = MyProc->sem.semId;
		semNum = MyProc->sem.semNum;
		semctl(semId, semNum, SETVAL, semun);

		IpcSemaphoreLock(semId, semNum, IpcExclusiveLock);
  }
}


/* bk: to do be done for funny race conditions */
void
ProcRemoveAllWaitingLocks()
{
	PROC_SHADOW *curr_shdw = NULL; 
	LOCK		 *lock;
	SHM_QUEUE 	*queue;
	
	queue = (SHM_QUEUE *) &MyProc->shadowList;
	if(!SHMQueueEmpty(queue)){
		do{
			SHMQueueFirst(queue, (Pointer *) &curr_shdw, &curr_shdw->shadowLinks);
			if(curr_shdw->waitLinks.next == INVALID_OFFSET &&
			   curr_shdw->waitLinks.prev == INVALID_OFFSET){
			   elog(NOTICE,"remove wait: there is a shadow where the lock was already granted");
			   /* this is a granted shadow and we just had not deleted it before
				* the abort took place */
				DeleteShadow(MyProcPid, curr_shdw->waitLock);
			}else{
				elog(NOTICE,"there is a non-granted shadow floating around!");
				elog(NOTICE,"in this case i also should decrease the holdings!");
				Assert(curr_shdw->waitLock->waitProcs.size > 0);
				SHMQueueDelete(&curr_shdw->waitLinks);
				SHMQueueElemInit(&curr_shdw->waitLinks); //important for DeleteShadow!!
				lock = curr_shdw->waitLock;
				--lock->waitProcs.size;
			/* bk: decrement also the nHolding counters */
			/* Decrement the lock nHolding and holders fields as
			* we are no longer waiting on this lock.
			 */
				lock->nHolding--;
				lock->holders[curr_shdw->token]--;
				Assert((lock->nHolding >= 0) && (lock->holders[&curr_shdw->token] >= 0));
				Assert(lock->nActive <= lock->nHolding);
				DeleteShadow(MyProcPid, curr_shdw->waitLock);
			}
		}while(queue->next != MAKE_OFFSET(&MyProc->shadowList));
	}
}

/*
*	ProcRemoveWaiting - removes process shadow from wait queues
*   (bk) note: this can only happen in read phase. Then, the waiting
*    process can only wait for one single lock, and hence, there is
*   only one shadow in the queue. 
*   hence, the while is turned to a if-statement 
*  all this is possible because, a process p can wait on a lock l, the lock l
*  is granted at the same time as p's abort flag is set due to another lock conflict
*  the wakeup of p is somehow funny, sometimes it simply runs to code that it should
*  not run through
*/
void
ProcRemoveWaiting(PROC *proc)
{
	PROC_SHADOW *curr_shdw = NULL; 
	LOCK		 *lock;
	/*XIDLookupEnt  *xidentP;  */
	
	/*while(proc->shadowList.next != MAKE_OFFSET(&proc->shadowList))*/
	if (proc->shadowList.next != MAKE_OFFSET(&proc->shadowList))
	{
	  SHMQueueFirst(&proc->shadowList, (Pointer *) &curr_shdw, &curr_shdw->shadowLinks);
	  if(curr_shdw->waitLinks.prev != INVALID_OFFSET)
	  {
#ifdef PROC_DEBUG
		elog(NOTICE, "[%d]ProcRemoveWaiting procid %d", MyProcPid, proc->pid);
		dumpQ(&curr_shdw->waitLock->waitProcs.links, "\tbefore:");
#endif
			
		/* delete it from lock->waitProcs.links == shadow->waitLinks */
		SHMQueueDelete(&curr_shdw->waitLinks);
		SHMQueueElemInit(&curr_shdw->waitLinks);
		lock = curr_shdw->waitLock;
		--lock->waitProcs.size;
		/*--curr_shdw->waitLock->waitProcs.size;*/
		/* bk: de236crement also the nHolding counters */
		/* we want to do this immediately and not when the process wakes up */
		/* Decrement the lock nHolding and holders fields as
		* we are no longer waiting on this lock.
		 */
		lock->nHolding--;
		/* bk: will be read or readforwrite lock */
		lock->holders[curr_shdw->token]--;
		Assert((lock->nHolding >= 0) && (lock->holders[&curr_shdw->token] >= 0));
		Assert(lock->nActive <= lock->nHolding);
		
		/* try that out: you also have to decrease the xid entries */
		/*elog(NOTICE,"i decrease also the xidentP");
		xidentP = curr_shdw->xidentP;
		xidentP->holders[curr_shdw->token]--;
		xidentP->nHolding--;*/
#ifdef PROC_DEBUG
		dumpQ(&curr_shdw->waitLock->waitProcs.links, "\tafter:");
#endif	
			 
		/* bk: before all shadows were deleted (whether still waiting for a lock, i.e. where waitLinks
		 * not empty, or not waiting for locks, i.e. in the meanwhile the lock was
		 * granting and the shadow is simply still floating around because the
		 * process itself has not waken up yet).
		 * now, we only delete them when they were not yet granted, hence, the parentheses
		 * is not commented out and put later */
		/* }*/


		/* bk: we also have to delete the xid entry in the proc's lockqueye! */
		/* shadow->xidentP->queue == proc->lockQueue */
		/* bk: no, don't do that, xidentP might hold other locks, e.g. read locks */
		/* xidentP = curr_shdw->xidentP;
		SHMQueueDelete(&xidentP->queue); */


		/* bk: new we cannot delete shadow here, because we have to behave
		 * same as grant lock, we don't know which will be the last
		 * before the process wakes up: ProcLockWakeup or ProcRemoveWaiting
		 * and hence the shadow must be deleted in WaitOnLock in any of the cases */
		/* and now we have to delete it from the proc's shadowlist and
		* we have to delete the shadow itself
		* shadow->shadowLinks == proc->shadowList */
		/* DeleteShadow(proc->pid, curr_shdw->waitLock);*/
	  }
	}
}

/*
 * ProcRemove -
 *	  used by the postmaster to clean up the global tables. This also frees
 *	  up the semaphore used for the lmgr of the process. (We have to do
 *	  this is the postmaster instead of doing a IpcSemaphoreKill on exiting
 *	  the process because the semaphore set is shared among backends and
 *	  we don't want to remove other's semaphores on exit.)
 */
bool
ProcRemove(int pid)
{
	SHMEM_OFFSET location;
	PROC	   *proc;

	location = INVALID_OFFSET;

	location = ShmemPIDDestroy(pid);
	if (location == INVALID_OFFSET)
		return FALSE;
	proc = (PROC *) MAKE_PTR(location);

	SpinAcquire(ProcStructLock);

	ProcFreeSem(proc->sem.semKey, proc->sem.semNum);
	/* bk: now also for multiple */
	ProcFreeSem(proc->sem_multiple.semKey, proc->sem_multiple.semNum);

	proc->links.next = ProcGlobal->freeProcs;
	ProcGlobal->freeProcs = MAKE_OFFSET(proc);

	SpinRelease(ProcStructLock);

	return TRUE;
}

/*
 * ProcKill() -- Destroy the per-proc data structure for
 *		this process. Release any of its held spin locks.
 */
static void
ProcKill(int exitStatus, int pid)
{
	PROC	   *proc;
	SHMEM_OFFSET location;

	/* --------------------
	 * If this is a FATAL exit the postmaster will have to kill all the
	 * existing backends and reinitialize shared memory.  So all we don't
	 * need to do anything here.
	 * --------------------
	 */
	if (exitStatus != 0)
		return;

	ShmemPIDLookup(MyProcPid, &location);
	if (location == INVALID_OFFSET)
		return;

	proc = (PROC *) MAKE_PTR(location);

	Assert(proc == MyProc || pid != MyProcPid);

	MyProc = NULL;

	/* ---------------
	 * Assume one lock table.
	 * ---------------
	 */
	ProcReleaseSpins(proc);
	LockReleaseAll(DEFAULT_LOCKMETHOD, &proc->lockQueue);

#ifdef USER_LOCKS

	/*
	 * Assume we have a second lock table.
	 */
	LockReleaseAll(USER_LOCKMETHOD, &proc->lockQueue);
#endif

	/* ----------------
	 * (wb)
	 * get off the wait queues:
	 * Go through the shadow list of this process
	 * and delete all shadows currently waiting for a lock. 
	 * ----------------
	 */
	LockLockTable();
	ProcRemoveWaiting(proc);
	UnlockLockTable();
	

	return;
}

/*
 * ProcQueue package: routines for putting processes to sleep
 *		and  waking them up
 */

/*
 * ProcQueueAlloc -- alloc/attach to a shared memory process queue
 *
 * Returns: a pointer to the queue or NULL
 * Side Effects: Initializes the queue if we allocated one
 */
#ifdef NOT_USED
PROC_QUEUE *
ProcQueueAlloc(char *name)
{
	bool		found;
	PROC_QUEUE *queue = (PROC_QUEUE *)
	ShmemInitStruct(name, (unsigned) sizeof(PROC_QUEUE), &found);

	if (!queue)
		return NULL;
	if (!found)
		ProcQueueInit(queue);
	return queue;
}

#endif

/*
 * ProcQueueInit -- initialize a shared memory process queue
 */
void
ProcQueueInit(PROC_QUEUE *queue)
{
	SHMQueueInit(&(queue->links));
	queue->size = 0;
}



/*
 * (wb)
 * ProcInsertWaiting-- insert a process into the lock wait queue
 *
 * A process that waits for a lock is registered by inserting
 * a process shadow into the lock wait queue for the corresponding
 * lock. Originally, this code was integrated into the ProcSleep
 * procedure. This implied that a process waiting for a lock was 
 * ALWAYS put to sleep. However, when processing a write set in 
 * a replicated database, we do not always want to sleep on a lock.
 * To do this, the ProcSleep function has been splitted in two 
 * distinct functions: ProcInsertWaiting to insert the process 
 * into the wait queue and ProcSleep to put the process to sleep
 * by issuing a "down" on its semaphore.
 *
 * (original notes)
 * ASSUME: that no one will fiddle with the queue until after
 *		we release the spin lock.
 *
 * NOTES: The process queue is now a priority queue for locking.
 */

void
ProcInsertWaiting(PROC_QUEUE *waitQueue,/* lock->waitProcs */
				  int token,			/* lockmode */
				  int prio,
				  LOCK *lock,
				  TransactionId xid,  /* needed by user locks, see below */
				  REQ_MODE reqmode,
				  XIDLookupEnt *xidentP)
{
	int			i;
	PROC_SHADOW *curr_shdw = NULL,
				*my_shdw = NULL;
	/*
	 * If the first entries in the waitQueue have a greater priority than
	 * we have, we must be a reader, and they must be a writers, and we
	 * must be here because the current holder is a writer or a reader but
	 * we don't share shared locks if a writer is waiting. We put
	 * ourselves after the writers.  This way, we have a FIFO, but keep
	 * the readers together to give them decent priority, and no one
	 * starves.  Because we group all readers together, a non-empty queue
	 * only has a few possible configurations:
	 *
	 * [readers] [writers] [readers][writers] [writers][readers]
	 * [writers][readers][writers]
	 *
	 * In a full queue, we would have a reader holding a lock, then a writer
	 * gets the lock, then a bunch of readers, made up of readers who
	 * could not share the first readlock because a writer was waiting,
	 * and new readers arriving while the writer had the lock.
	 *
	 */
	if(reqmode == REQ_MULTIPLE)
	{
	    ProcMarkAbortedReaders(lock);
	}
	if(waitQueue->size>0){
		curr_shdw = (PROC_SHADOW *) WL_2_SHDWPTR(waitQueue->links.prev);
		/* If we are a reader, and they are writers, skip past them */
		for (i = 0; i < waitQueue->size && curr_shdw->prio > prio; i++)
			curr_shdw = (PROC_SHADOW *) WL_2_SHDWPTR(curr_shdw->waitLinks.prev);
		
		/* 
		* (wb) The wait queue discipline has been changed. When inserting a
		* request from a write set, we must not insert after the waiting writers
		* since this would mean that all readers would be aborted.
		* When such a write request arrives, it is inserted before the bunch
		* of readers in the queue.
		*	ATTN: This introduces the possibility of reader starvation.
		* Example:
		*	[W] -> [writers][readers]
		*   old discipline: [writers][readers][W}
		*	new discipline: [writers][W][readers]
		*/

		/* The rest of the queue is FIFO, with readers first, writers last */
		for (; i < waitQueue->size && curr_shdw->prio <= prio; i++)
		{
			if(reqmode == REQ_MULTIPLE && curr_shdw->prio < prio)
			{
				break; //(wb) handles the special case (see above comment)
			}
			curr_shdw = (PROC_SHADOW *) WL_2_SHDWPTR(curr_shdw->waitLinks.prev);
		}
	}
	/*Get a shadow for this process and initialize it*/
	my_shdw = GetShadow(lock);
	my_shdw->waitLock = lock;
	my_shdw->prio = prio;
	my_shdw->token = token;
	my_shdw->xidentP = xidentP;
	/* bk: now the shadow has the reqmode strored to unlock the right semaphore */
	my_shdw->req_mode = reqmode;

#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]shadow no %d (%d, %d, %d)", MyProcPid, MAKE_OFFSET(my_shdw),
												MAKE_OFFSET(my_shdw->waitLock),
												my_shdw->token, my_shdw->proc);

#endif
#ifdef USER_LOCKS
	/* -------------------
	 * Currently, we only need this for the ProcWakeup routines.
	 * This must be 0 for user lock, so we can't just use the value
	 * from GetCurrentTransactionId().
	 * -------------------
	 */
	TransactionIdStore(xid, &MyProc->xid);
#else
#ifndef LowLevelLocking
	/* -------------------
	 * currently, we only need this for the ProcWakeup routines
	 * -------------------
	 */
	TransactionIdStore((TransactionId) GetCurrentTransactionId(), &MyProc->xid);
#endif
#endif

	/* -------------------
	 * assume that these two operations are atomic (because
	 * of the spinlock).
	 * -------------------
	 */
#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]inserting myself into lock wait queue shd no %d", MyProcPid, MAKE_OFFSET(my_shdw));
	dumpQ(&waitQueue->links, "\tbefore:");
#endif
	if(waitQueue->size>0){
		SHMQueueInsertTL(&(curr_shdw->waitLinks), &(my_shdw->waitLinks));
	}else{
		SHMQueueInsertTL(&(waitQueue->links), &(my_shdw->waitLinks));
	}
#ifdef PROC_DEBUG
	dumpQ(&waitQueue->links, "\tafter:");
#endif
	waitQueue->size++;
	
	return;
}

/*
 * ProcSleep -- put a process to sleep
 *
 * P() on the semaphore should put us to sleep.  The process
 * semaphore is cleared by default, so the first time we try
 * to acquire it, we sleep.
 *
 * ASSUME: that no one will fiddle with the queue until after
 *		we release the spin lock.
 *
 * (wb) This function is now only responsible for putting
 *	a process to sleep and checking periodically for deadlocks
*/
int
ProcSleep(SPINLOCK spinlock, LOCK * lock)
{
	struct itimerval	timeval,
						dummy;
						
	
	SpinRelease(spinlock);
	if(txn_type == REPLICATED_LOCAL && MyProc->abortFlag == MUST_ABORT && !may_not_abort_right_now)
	{
		MyProc->abortFlag = ABORTING; 
		elog(ERROR, "detected internal abort - aborting transaction (2)");
		/*not reached*/
	}
	else if (txn_type == REPLICATED_LOCAL && MyProc->abortFlag == MUST_ABORT && may_not_abort_right_now)
	{
	  elog(NOTICE,"should abort but maynot 4");
	}

	/* --------------
	 * We set this so we can wake up periodically and check for a deadlock.
	 * If a deadlock is detected, the handler releases the processes
	 * semaphore and aborts the current transaction.
	 *
	 * Need to zero out struct to set the interval and the micro seconds fields
	 * to 0.
	 * --------------
	 */
	MemSet(&timeval, 0, sizeof(struct itimerval));
	/*timeval.it_value.tv_sec = \
		(DeadlockCheckTimer ? DeadlockCheckTimer : DEADLOCK_CHECK_TIMER);
	timeval.it_interval.tv_sec = \
		(DeadlockCheckTimer ? DeadlockCheckTimer : DEADLOCK_CHECK_TIMER);*/


	do
	{
		//elog(NOTICE,"i am here");
		MyProc->errType = NO_ERROR;		/* reset flag after deadlock check */


		MemSet(&timeval, 0, sizeof(struct itimerval));
		timeval.it_value.tv_usec = \
		  (DeadlockCheckTimer ? DeadlockCheckTimer : DEADLOCK_CHECK_TIMER);
		timeval.it_interval.tv_usec = \
		(DeadlockCheckTimer ? DeadlockCheckTimer : DEADLOCK_CHECK_TIMER);
		if (setitimer(ITIMER_REAL, &timeval, &dummy))
			elog(FATAL, "ProcSleep: Unable to set timer for process wakeup");

		/*elog(NOTICE,"go to sleep time %ld interval", timeval.it_value.tv_usec, timeval.it_interval.tv_usec);*/
		/* --------------
		 * if someone wakes us between SpinRelease and IpcSemaphoreLock,
		 * IpcSemaphoreLock will not block.  The wakeup is "saved" by
		 * the semaphore implementation.
		 * --------------
		 */
		 /* bk: procsleep is used for request mode REQ_SINGLE, hence the old
		  * semaphore is used */
		/*elog(NOTICE,"before sem4");*/
		IpcSemaphoreLock(MyProc->sem.semId, MyProc->sem.semNum,
						 IpcExclusiveLock);
		  
		/*elog(NOTICE,"after sem4");*/
	} while (MyProc->errType == STATUS_NOT_FOUND);		/* sleep after deadlock
														 * check */

	/* ---------------
	 * We were awoken before a timeout - now disable the timer
	 * ---------------
	 */
	 
	
	timeval.it_value.tv_sec = 0;
	timeval.it_value.tv_usec = 0;
	if (setitimer(ITIMER_REAL, &timeval, &dummy))
		elog(FATAL, "ProcSleep: Unable to diable timer for process wakeup");

	/* ----------------
	 * We were assumed to be in a critical section when we went
	 * to sleep.
	 * ----------------
	 */
	/*elog(NOTICE,"before spinlock acquire");*/
	SpinAcquire(spinlock);
	/*elog(NOTICE, "[%d]awaking! errType %d", MyProcPid, MyProc->errType);*/
#ifdef PROC_DEBUG	
	elog(NOTICE, "[%d]awaking!", MyProcPid);
#endif
	/* -----------------
	* (wb) 
	* If there was a deadlock (errType=STATUS_ERROR), the shadow 
	* has already been deleted by the deadlock handling routine 
	* (see HandleDeadLock)
	* If the txn is internally aborted (errtype == ERR_INT_ABORT, the shadow
	* has also been deleted already
	* ------------------
	*/

	/*if(MyProc->errType == NO_ERROR)
		DeleteShadow(MyProcPid, lock); */

	/* bk: delete the shadow out of the lists both in NO_ERROR and ERROR case */
	DeleteShadow(MyProcPid, lock); 

	return MyProc->errType;
}

/*
*	(wb)
*	ProcSleepOnWriteSet
*	
*	This procedure is responsible for putting a process to sleep if it 
*	is waiting for locks requested for write set processing. If the process
*	is awakened, it goes through its shadow list to see which locks have 
*	been granted (there can be several locks granted).
*
*	param:
*		wokenRelIds - an array which , on return of ProcSleepOnWriteSet,
*			contains the relation oids of the locks that have been granted
*	return:
*		int - the number of granted locks
*/
int
ProcSleepOnWriteSet(Oid wokenRelIds[])
{
	PROC_SHADOW	*curr_shdw = NULL;
	SHM_QUEUE 	*queue;
	int 		i = 0;

	MyProc->errType = NO_ERROR;
	/* bk: new we take now the multiple semaphore */
	//elog(NOTICE,"before sem6");
	/*IpcSemaphoreLock(MyProc->sem.semId, MyProc->sem.semNum, IpcExclusiveLock);*/
	IpcSemaphoreLock(MyProc->sem_multiple.semId, MyProc->sem_multiple.semNum, IpcExclusiveLock);
	//elog(NOTICE,"after sem6");
	LockLockTable();

	
	queue = (SHM_QUEUE *) &MyProc->shadowList;
	if(!SHMQueueEmpty(queue)){
		do{
			SHMQueueFirst(queue, (Pointer *) &curr_shdw, &curr_shdw->shadowLinks);
			if(curr_shdw->waitLinks.next == INVALID_OFFSET &&
			   curr_shdw->waitLinks.prev == INVALID_OFFSET){
				wokenRelIds[i++] = curr_shdw->waitLock->tag.relId;
				DeleteShadow(MyProcPid, curr_shdw->waitLock);
			}else{
				queue = &curr_shdw->shadowLinks;
			}
		}while(queue->next != MAKE_OFFSET(&MyProc->shadowList));
	}
	UnlockLockTable();
	/*
	*	The semaphore has to be adjusted if several locks have been released
	*	at the same time (like in the LockReleaseAll function). For each
	*   released lock, an 'up' operation is issued on the semaphore. Semaphores
	*	save all 'up' operations, meaning that e.g. 3 consecutive 'up's imply
	*	that 3 consecutive calls to IpcSemaphoreLock will not
	*	block. This implies that, if several locks have been released while I
	*	was awake, I must adjust my semaphore by the count of locks released 
	*	minus 1, or I will not block at the next calls of IpcSemaphoreLock.
	*	ProcAdjustSemaphore does nothing else than calling IpcSemaphoreLock
	*	a given number of times. 
	*/
	ProcAdjustSemaphore(&(MyProc->sem_multiple), i - 1);
	/*ProcAdjustSemaphore(&(MyProc->sem), i - 1);*/
	return i;
}

/*
 * ProcWakeup -- wake up a process by releasing its private semaphore.
 *
 *	 remove the process from the wait queue and set its links invalid.
 *	 RETURN: the next process in the wait queue.
 *	 
 *	 (wb) This operation is now performed on shadows
 */
static PROC_SHADOW *
ProcWakeup(PROC_SHADOW *shdw, int errType)
{
	PROC_SHADOW		*ret_shdw;
	PROC 			*proc;
	LOCK			*lock;

	/* assume that spinlock has been acquired */
	proc = (PROC *) MAKE_PTR(shdw->proc);
	proc->errType = errType;
	
	lock = shdw->waitLock;
	
	if (shdw->waitLinks.prev == INVALID_OFFSET ||
		shdw->waitLinks.next == INVALID_OFFSET)
		{
		elog(NOTICE,"wake up, shadow not in waitLink list");
		return (PROC_SHADOW *) NULL;
		}
		
	ret_shdw = (PROC_SHADOW *) WL_2_SHDWPTR(shdw->waitLinks.prev);

	/* you have to update waitLock->waitProcs.size yourself */
	
	SHMQueueDelete(&(shdw->waitLinks));
	SHMQueueElemInit(&(shdw->waitLinks));
	
	
	if (shdw->req_mode == REQ_SINGLE)
	{
	  /*elog(NOTICE,"before sem 8");*/
	  IpcSemaphoreUnlock(proc->sem.semId, proc->sem.semNum, IpcExclusiveLock);
	}
	else
	{
	//elog(NOTICE,"before sem 9");
	/*  IpcSemaphoreUnlock(proc->sem.semId, proc->sem.semNum, IpcExclusiveLock);*/
	IpcSemaphoreUnlock(proc->sem_multiple.semId, proc->sem_multiple.semNum, IpcExclusiveLock);
	}
	return ret_shdw;
}

/*
 * ProcLockWakeup -- routine for waking up processes when a lock is
 *		released.
 *	
 *	(wb) modified to support process shadows
 */
int
ProcLockWakeup(PROC_QUEUE *queue, LOCKMETHOD lockmethod, LOCK *lock)
{
	PROC_SHADOW *shdw = NULL;
	PROC	   	*proc = NULL;
	int			count;
	int			trace_flag;
	int			last_locktype = -1;
	int			queue_size = queue->size;
		
	Assert(queue->size >= 0);

	
	if (!queue->size)
	{
		return STATUS_NOT_FOUND;
	}

	shdw = (PROC_SHADOW *) WL_2_SHDWPTR(queue->links.prev);


#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]initiates wakeup(qsize: %d)", MyProcPid, queue->size);
	elog(NOTICE, "[%d]shadow no %d (%d, %d, %d)", MyProcPid, MAKE_OFFSET(shdw),
												MAKE_OFFSET(lock),
												shdw->token, shdw->proc);
#endif
	count = 0;
	
	while ((queue_size--) && (shdw))
	{
		
		proc = (PROC *) MAKE_PTR(shdw->proc);
		
		/*elog(NOTICE,"check:%d (on pid %x / %d, me = %x / %d)",count, proc->xid, proc->pid, 
				GetCurrentTransactionId(), MyProcPid );*/
		/*
		 * This proc will conflict as the previous one did, don't even
		 * try.
		 */
		if (shdw->token == last_locktype)
		{
			/*elog(NOTICE,"same as last lock type");*/
			continue;
		}

		/*
		 * This proc conflicts with locks held by others, ignored.
		 */

		if (LockResolveConflicts(lockmethod,
								 lock,
								 shdw->token,
								 proc->xid,
								 (XIDLookupEnt *) NULL) != STATUS_OK)
		{
			/*elog(NOTICE,"conflicts:%d (on pid %x / %d, me = %x / %d)",count, proc->xid, proc->pid, 
							GetCurrentTransactionId(), MyProcPid );*/
			last_locktype = shdw->token;
			continue;
		}
		/*
		 * there was a waiting process, grant it the lock before waking it
		 * up.	This will prevent another process from seizing the lock
		 * between the time we release the lock master (spinlock) and the
		 * time that the awoken process begins executing again.
		 */

		/* bk: weird race condition: if the txn belonging to this
		 * shadow must be aborted (abortFlag == TRUE) we should
		 * not grant this lock but simply ignore it, the corresponding
		 * process has been woken up already by the one who set
		 * the abort flag to true, when the process finally wakes
		 * up it will release it from the wait queue
		*/
		/* (bk) now i undo this again, the case where the abort
		* flag is set should not exist anymore, because whoever
		* sets the abort flag of another process removes all entries
		* correlated to any waiting lock out of any queue! */
		
		/* bk: we have to check aborting too, because txn requires lock on pg_log when in aborting state */
		GrantLock(lock, proc->xid, lockmethod, shdw->token);
		  /*
		   * ProcWakeup removes proc from the lock waiting process queue and
		   * returns the next proc in chain.
		   */

		  count++;
		  queue->size--;
		

#ifdef PROC_DEBUG
		  elog(NOTICE, "[%d]removing from waitQueue shd no %d",MyProcPid,MAKE_OFFSET(shdw));
		  dumpQ(&queue->links, "\tbefore:");
#endif

		  shdw = ProcWakeup(shdw, NO_ERROR);
		  /*elog(NOTICE,"shadow removed");*/
		
#ifdef PROC_DEBUG
		  dumpQ(&queue->links, "\tafter:");
#endif
	  }
		

	  Assert(queue->size >= 0);

	  if (count){
		  return STATUS_OK;
	  }
	  else
	  {
		  /* Something is still blocking us.	May have deadlocked. */
		  /* bk wakeup might fail because of two correct situations:
		   * the process will be woken up by the one who forces it to
		   * abort or -- and another reason I just have forgotten
		   * but also only in the case of remote txn */
		  trace_flag = (lock->tag.lockmethod == USER_LOCKMETHOD) ? \
			  TRACE_USERLOCKS : TRACE_LOCKS;
		  TPRINTF(trace_flag,
			  	"ProcLockWakeup: lock(%x) can't wake up any process",
				  MAKE_OFFSET(lock));
#ifdef DEADLOCK_DEBUG
		  if (pg_options[trace_flag] >= 2)
			  DumpAllLocks();
#endif	
		  /*elog(NOTICE,"wakeup failed:%d (on pid %x / %d, me = %x / %d)",count, proc->xid, proc->pid, GetCurrentTransactionId(), MyProcPid );*/
		  return STATUS_NOT_FOUND;
	}
}

void
ProcAddLock(SHM_QUEUE *elem)
{
	SHMQueueInsertTL(&MyProc->lockQueue, elem);
}

/*
*	HandleAlarm - handles a SIGALRM signal. In case of non-replicated transactions, we do
*		deadlock checking, and in case of replicated transactions, we have to react
*		in different ways: remote backends are NEVER deadlocked, but if we have to wait
*		for the decision message before starting to abort readers, we have to check if any 
*		message arrived. Local backends should not check for deadlocks, since they are never 
*		deadlocked. (Remember that we do not run replicated and non-replicated txn's in parallel)
*	
*/

static void
HandleAlarm(int sig)
{
	
		
	if(txn_type == NOT_REPLICATED)
	{
		HandleDeadLock(sig);
	}
	else if (txn_type == REPLICATED_LOCAL && MyProc->state == READ_PHASE)
	{
	  if(MyProc->abortFlag == MUST_ABORT && !may_not_abort_right_now)
	  {
		struct itimerval	timeval,     // bk: set timer to zero so that it does not start again
							dummy;
							
		int			semNum;              // reset semaphore otherwise funny race condition
		int			semId;
		union semun semun;
			
		MemSet(&timeval, 0, sizeof(struct itimerval));
		timeval.it_value.tv_sec = 0;
		timeval.it_value.tv_usec = 0;
		if (setitimer(ITIMER_REAL, &timeval, &dummy))
		  elog(FATAL, "ProcSleep: Unable to diable timer for process wakeup");
		MyProc->abortFlag = ABORTING;
		/*printf("Txn %x has abortflag set, aborts now", GetCurrentTransactionId());*/
		
		/*semun.val = IpcSemaphoreDefaultStartValue;
		semId = MyProc->sem.semId;
		semNum = MyProc->sem.semNum;
		semctl(semId, semNum, SETVAL, semun);
		IpcSemaphoreLock(semId, semNum, IpcExclusiveLock);*/

		elog(ERROR, "Detected internal abort, aborting(8)");
	  }		
	  else if (MyProc->abortFlag == NO_ABORT || may_not_abort_right_now)
		HandleDeadLock(sig);
	}
}


/* --------------------
*	HandleDeadlock
*
 * We only get to this routine if we got SIGALRM after DEADLOCK_CHECK_TIMER
 * while waiting for a lock to be released by some other process.  If we have
 * a real deadlock, we must also indicate that I'm no longer waiting
 * on a lock so that other processes don't try to wake me up and screw
 * up my semaphore.
 *
 *	(wb) modified to work on shadows...
 * --------------------
 */
static void
HandleDeadLock(int sig)
{
	PROC_SHADOW *curr_shdw = NULL;
	SHM_QUEUE   *queue;


		XIDLookupEnt *result,
				item;
	bool		found;

	elog(NOTICE,"we look for deadlock");

	LockLockTable();

	/* ---------------------
	 * Check to see if we've been awoken by anyone in the interim.
	 *
	 * If we have we can return and resume our transaction -- happy day.
	 * Before we are awoken the process releasing the lock grants it to
	 * us so we know that we don't have to wait anymore.
	 *
	 * Damn these names are LONG! -mer
	 * ---------------------
	 */
	/*elog(NOTICE,"before sem9");*/
	if (IpcSemaphoreGetCount(MyProc->sem.semId, MyProc->sem.semNum) ==
		IpcSemaphoreDefaultStartValue)
	{
		UnlockLockTable();
		elog(NOTICE,"semaphore counts are strange");
		return;
	}

	/*
	 * you would think this would be unnecessary, but...
	 *
	 * this also means we've been removed already.  in some ports (e.g.,
	 * sparc and aix) the semop(2) implementation is such that we can
	 * actually end up in this handler after someone has removed us from
	 * the queue and bopped the semaphore *but the test above fails to
	 * detect the semaphore update* (presumably something weird having to
	 * do with the order in which the semaphore wakeup signal and SIGALRM
	 * get handled).
	 *
	 *	Note(wb, 12/03/98): 	
	 *				We must check if there is any shadow in shadowList that is
	 *				no longer engaged in a lock wait queue. This would indeed
	 *				mean that somebody removed a shadow from a waiting list
	 *				after this handler was called.
	 */
	if (SHMQueueEmpty(&MyProc->shadowList))
	{
		UnlockLockTable();
		elog(NOTICE,"somebody just woke us up");
		return;

	}else{
		queue = (SHM_QUEUE *) &MyProc->shadowList;
		do{
			SHMQueueFirst(queue, (Pointer *) &curr_shdw, &curr_shdw->shadowLinks);
			if(curr_shdw->waitLinks.next == INVALID_OFFSET &&
			   curr_shdw->waitLinks.prev == INVALID_OFFSET){
				elog(NOTICE,"we got woken up");
				UnlockLockTable();
				return;
			}
			queue = &curr_shdw->shadowLinks;
		}while(queue != &MyProc->shadowList);
	}
	
	

#ifdef DEADLOCK_DEBUG
	DumpAllLocks();
#endif
	MyProc->errType = STATUS_NOT_FOUND;
	
#ifdef PROC_DEBUG
	elog(NOTICE, "[%d]shadowList dump", MyProcPid);
	dumpQ(&MyProc->shadowList, "\t");
#endif

	queue = (SHM_QUEUE *) &MyProc->shadowList;
	/* bk change from while to if, because there can only be one shadow!! */
/*	do{ */

		SHMQueueFirst(queue, (Pointer *) &curr_shdw, &curr_shdw->shadowLinks);
	
	if(&curr_shdw->shadowLinks != queue)  /* bk */
	{									  /* bk */
#ifdef PROC_DEBUG
		elog(NOTICE, "[%d]shadowList dump in for loop", MyProcPid);
		dumpQ(queue, "\t");
		elog(NOTICE, "[%d]shadow no %d (%d, %d, %d)", MyProcPid, MAKE_OFFSET(curr_shdw),
												MAKE_OFFSET(curr_shdw->waitLock),
												curr_shdw->token, curr_shdw->proc);
#endif


	  /* bk test */
	  MemSet(&item, 0, XID_TAGSIZE);
	  TransactionIdStore(MyProc->xid, &item.tag.xid);
	  item.tag.lock = MAKE_OFFSET(curr_shdw->waitLock);
#if 0
	  item.tag.pid = pid;
#endif

	  if (!(result = (XIDLookupEnt *)
		hash_search(LockMethodTable[DEFAULT_LOCKMETHOD]->xidHash, 
		  (Pointer) &item, HASH_FIND, &found)) || !found)
	  {
		if(!found)
		  elog(NOTICE, "handle deadlock: xid table corrupted( %d)", MyProc->xid);
	  }
	  else
	  {
	  /*log(NOTICE,"xid found");*/
	  }
	  if (DeadLockCheck(&(MyProc->lockQueue), curr_shdw->waitLock, true))
	  {
		  MyProc->errType = STATUS_ERROR;
	/*		break;*/ /* bk */
	  }
	  queue = &curr_shdw->shadowLinks;
	  /*elog(NOTICE, "[%d]shadowList dump in for loop end", MyProcPid);*/
	  dumpQ(&MyProc->shadowList, "\t");
	/*}while(queue != &MyProc->shadowList);*/
	}
	
	if(MyProc->errType == STATUS_NOT_FOUND){
		UnlockLockTable();
		/*elog(NOTICE,"no deadlock found");*/
		return;
	}	


	elog(NOTICE,"have found a deadlock");

	/* ------------------------
	 * Get this process off the lock's wait queue
	 * ------------------------
	 */
	/* bk: there can only be one waiting shadow  in the read phase*/
	/* we cannot use while because now we do not delete the shadow anymore */
	if (!SHMQueueEmpty(&MyProc->shadowList))
	/*while(!SHMQueueEmpty(&MyProc->shadowList))*/
	{
		static PROC_SHADOW *curr_shdw = NULL;
		
		/*delete the shadow from the lock wait queue*/
		SHMQueueFirst(&MyProc->shadowList, (Pointer *) &curr_shdw, &curr_shdw->shadowLinks);
		Assert(curr_shdw->waitLock->waitProcs.size > 0);
		SHMQueueDelete(&curr_shdw->waitLinks);
		SHMQueueElemInit(&curr_shdw->waitLinks); //important for DeleteShadow!!
		--curr_shdw->waitLock->waitProcs.size;
		
		/*delete the shadow from the process's shadow list*/
		/* bk: now, do that now in WaitonLock */
		/* DeleteShadow(MyProcPid, curr_shdw->waitLock);*/
		
	
		/* ------------------
		 * Unlock my semaphore so that the count is right for next time.
		 * I was awoken by a signal, not by someone unlocking my semaphore.
		 * ------------------
		 */
		/*elog(NOTICE,"before sem10");*/
		IpcSemaphoreUnlock(MyProc->sem.semId, MyProc->sem.semNum,
					   IpcExclusiveLock);
	}


	/* -------------
	 * Set MyProc->errType to STATUS_ERROR so that we abort after
	 * returning from this handler.
	 * -------------
	 */
	MyProc->errType = STATUS_ERROR;

	/*
	 * if this doesn't follow the IpcSemaphoreUnlock then we get lock
	 * table corruption ("LockReplace: xid table corrupted") due to race
	 * conditions.	i don't claim to understand this...
	 */
	UnlockLockTable();

	elog(NOTICE, "Deadlock detected -- See the lock(l) manual page for a possible cause.");
	return;
}

void
ProcReleaseSpins(PROC *proc)
{
	int			i;

	if (!proc)
		proc = MyProc;

	if (!proc)
		return;
	for (i = 0; i < (int) MAX_SPINS; i++)
	{
		if (proc->sLocks[i])
		{
			Assert(proc->sLocks[i] == 1);
			SpinRelease(i);
		}
	}
}

/*****************************************************************************
 *
 *****************************************************************************/

/*
 * ProcGetNewSemKeyAndNum -
 *	  scan the free semaphore bitmap and allocate a single semaphore from
 *	  a semaphore set. (If the semaphore set doesn't exist yet,
 *	  IpcSemaphoreCreate will create it. Otherwise, we use the existing
 *	  semaphore set.)
 */
static void
ProcGetNewSemKeyAndNum(IPCKey *key, int *semNum)
{
	int			i;
	int32	   *freeSemMap; /*= ProcGlobal->freeSemMap;*/
	unsigned int fullmask;


	/*
	 * we hold ProcStructLock when entering this routine. We scan through
	 * the bitmap to look for a free semaphore.
	 */
	 
	 	
	freeSemMap = ProcGlobal->freeSemMap;
	

	fullmask = ~0 >> (32 - PROC_NSEMS_PER_SET);
	for (i = 0; i < MAX_PROC_SEMS / PROC_NSEMS_PER_SET; i++)
	{
		int			mask = 1;
		int			j;

		if (freeSemMap[i] == fullmask)
		{
			continue;			/* none free for this set */
		}

		for (j = 0; j < PROC_NSEMS_PER_SET; j++)
		{
			if ((freeSemMap[i] & mask) == 0)
			{

				/*
				 * a free semaphore found. Mark it as allocated.
				 */
				freeSemMap[i] |= mask;

				*key = ProcGlobal->currKey + i;
				*semNum = j;
				return;
			}
			mask <<= 1;
		}
	}
	/* if we reach here, all the semaphores are in use. */
	elog(ERROR, "InitProc: cannot allocate a free semaphore");
}

/*
*	(wb)
*	ProcAdjustSemaphore
*
*	Performs a number of 'down' ops on the processes semaphore in order to 
*	neutralize previous 'up' ops that are no longer needed.
*
*	param:
*		exceedUps - the number of 'down' ops to perform
*	return:
*		void
*/
/* bk: new, give as input also the SEMA */
static void
ProcAdjustSemaphore(SEMA *sem, int exceedUps)
{
	//elog(NOTICE, "adjusting semaphore %d times", exceedUps);
	while(exceedUps > 0){
	  IpcSemaphoreLock(sem->semId, sem->semNum, IpcExclusiveLock);
		/*IpcSemaphoreLock(MyProc->sem.semId, MyProc->sem.semNum, IpcExclusiveLock);*/
		exceedUps--;
	}
	//elog(NOTICE, "done adjusting");
	
}

/*
 * ProcFreeSem -
 *	  free up our semaphore in the semaphore set. If we're the last one
 *	  in the set, also remove the semaphore set.
 */
static void
ProcFreeSem(IpcSemaphoreKey semKey, int semNum)
{
	int			mask;
	int			i;
	int32	   *freeSemMap = ProcGlobal->freeSemMap;

	i = semKey - ProcGlobal->currKey;
	mask = ~(1 << semNum);
	freeSemMap[i] &= mask;

	if (freeSemMap[i] == 0)
		IpcSemaphoreKill(semKey);
}

/*
 * ProcFreeAllSemaphores -
 *	  on exiting the postmaster, we free up all the semaphores allocated
 *	  to the lmgrs of the backends.
 */
void
ProcFreeAllSemaphores()
{
	int			i;
	int32	   *freeSemMap = ProcGlobal->freeSemMap;

	elog(NOTICE,"in all free");

	for (i = 0; i < MAX_PROC_SEMS / PROC_NSEMS_PER_SET; i++)
	{
		if (freeSemMap[i] != 0)
			IpcSemaphoreKill(ProcGlobal->currKey + i);
	}
}
