/* pgxload.c,v 1.1 2003/07/17 14:10:45 myui Exp
 *
 * ----------------------
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *	  products derived from this software without specific prior written
 *	  permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License (see COPYING.GPL), in which case the provisions
 * of the GPL are required INSTEAD OF the above restrictions.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.	IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <string.h>
#include <ctype.h>				/* isspace() */
#include <getopt.h>
#include <iconv.h>
#include <expat.h>
#include <glib.h>
#include "libpq-fe.h"
#include "stack/stack.h"

/* Debug Mode */
//#define __DEBUG__ 

/* skip space symbol ? -> affect to XPath position() function */
//#define STRIP_SPACE 

#ifdef BUFSIZ
#undef BUFSIZ
#endif
#define BUFSIZ 1048576		// 1M
//#define BUFSIZ 2097152    // 2M 

#define __DBNAME "xpsql"
#define __HOST NULL
#define __PORT NULL
#define __OPTION NULL
#define __TTY NULL
#define __LOGIN "postgres"
#define __PWD NULL
#define __FROM_ENC "UTF-8"
#define __TO_ENC "EUC-JP"

#define MAX_CONTENT_SIZE 65000

#define EXIT_DISCONN() PQfinish(conn); \
					   exit(1)

#define TRANZ_BEGIN		res = PQexec(conn, "BEGIN");				 \
				if (!res || PQresultStatus(res) != PGRES_COMMAND_OK) \
				{													 \
					fprintf(stderr, "BEGIN command failed\n");		 \
					PQclear(res);									 \
					EXIT_DISCONN();									 \
				}													 \
				PQclear(res)

#define TRANZ_COMMIT		res = PQexec(conn, "COMMIT");			 \
							PQclear(res)

#define EXIT_STARTELEMENT	free(p);								\
							free(a1);free(a2);						\
							g_free(CurrentNode);					\
							g_string_free(query, TRUE);				\
							PQclear(res);							\
							EXIT_DISCONN()

/**
 * ############################################################
 * Memory management - we make expat use standard pg MM
 * ############################################################
 */
XML_Memory_Handling_Suite mhs;

/**
 * passthrough functions (malloc is a macro)  
 */
static void *
pgxml_malloc(size_t size)
{
	return malloc(size);
}

static void *
pgxml_remalloc(void *ptr, size_t size)
{
	return realloc(ptr, size);
}

static void
pgxml_free(void *ptr)
{
	return free(ptr);
}

static void
pgxml_mhs_init()
{
	mhs.malloc_fcn = pgxml_malloc;
	mhs.realloc_fcn = pgxml_remalloc;
	mhs.free_fcn = pgxml_free;
}

// ############################################################

_Bool strip_space;
static PGconn *conn;
static signed int CurrentParent;	// int32
static _Bool StackIsLoaded;
static Stack *textNodeStack;
static _Bool StackNsIsLoaded;
static Stack *nsprefixStack;
static Stack *nsuriStack;

char *from_codeset;
char *to_codeset;

/**
 * expat
 */
void startElement(void *, const char *, const char **);
void endElement(void *, const char *);
void charData(void *, const XML_Char *, int);
void commentData(void *, const XML_Char *);
void piData(void *, const XML_Char *, const XML_Char *);
//void startNS(void *, const XML_Char *, const XML_Char *); 
void xmlDecl(void *, const XML_Char *, const XML_Char *, int);
//void endNS(void *, const XML_Char *); 
void defaultData(void *, const XML_Char *, int);

static void trim(char *);
static char *strcompress(const char *);
static int StrReplace(char *, const char *, const char *);
static char *basename(char *);
static _Bool setTextNode(void);
//static _Bool setNsNode(void); 
static void showUsage();

// ############################################################
void
startElement(void *userData, const char *name, const char **attrs)
{
	int i = 0;
	char *att_id;
	char *CurrentNode;
	char *tag_name = (char *) malloc(strlen(name) + 1);
	char *p = tag_name;
	char *att_name = (char *) malloc(MAX_CONTENT_SIZE + 1);
	char *a1 = att_name;
	char *att_value = (char *) malloc(MAX_CONTENT_SIZE + 1);
	char *a2 = att_value;
	PGresult *res;
	GString *query = g_string_sized_new(100);

	if (StackIsLoaded)
		setTextNode();

	/**
	 * convert UTF8 to EUC
	 */
	memcpy(tag_name, name, strlen(name) + 1);
	g_strstrip(tag_name);
	tag_name =
		g_convert(tag_name, strlen(tag_name), to_codeset, from_codeset,
				  NULL, NULL, NULL);

#ifdef __DEBUG__
	printf("[XML_StartElementHandler]\n");
	printf("element:%s\n", tag_name);
#endif

	/**
	 * create element
	 */
	g_string_sprintf(query, "SELECT new_childelement(%d,'%s')",
					 CurrentParent, tag_name);

#ifdef __DEBUG__
	fprintf(stdout, "SELECT new_childelement(%d,'%s')\n", CurrentParent,
			tag_name);
#endif

	res = PQexec(conn, query->str);
	CurrentNode = g_strdup(PQgetvalue(res, 0, 0));
	if (!res || PQresultStatus(res) != PGRES_TUPLES_OK || CurrentNode == NULL) {
		fprintf(stderr, "Error\tnew_childelement(%d,%s)\n", CurrentParent,
				tag_name);
		EXIT_STARTELEMENT;
	}
	PQclear(res);

	CurrentParent = atoi(CurrentNode);

#ifdef __DEBUG__
	fprintf(stdout, "CurrentParent=%d,CurrentNode=%s\n", CurrentParent,
			CurrentNode);
#endif

	/**
	 * create namespacce decl
	 *
	if(StackNsIsLoaded)
		setNsNode();
	*/

	/**
	 * set attribute
	 */
	while (attrs[i] != '\0') {
		att_name = a1;
		strcpy(att_name, attrs[i]);
		g_strstrip(att_name);
		att_name =
			g_convert(att_name, strlen(att_name), to_codeset, from_codeset,
					  NULL, NULL, NULL);
		att_value = a2;
		strcpy(att_value, attrs[i + 1]);
		g_strstrip(att_value);
		att_value =
			g_convert(att_value, strlen(att_value), to_codeset,
					  from_codeset, NULL, NULL, NULL);

		if (strstr(att_name, "xmlns:") != NULL) {
			att_name = att_name + 6;
			if (att_name != NULL)
			{
				g_string_sprintf(query, "SELECT setnamespace(%s,'%s','%s')",
								 CurrentNode, att_name, att_value);
			}
			else
			{
				EXIT_STARTELEMENT;
			}
		}
		else if (strstr(att_name, "xmlns=") != NULL) {
			g_string_sprintf(query, "SELECT setnamespace(%s,'%s')",
							 CurrentNode, att_value);
		}
		else {
			g_string_sprintf(query,
							 "SELECT setattribute(%s,'%s','%s')",
							 CurrentNode, att_name, att_value);
		}

		res = PQexec(conn, query->str);
		att_id = PQgetvalue(res, 0, 0);
		if (!res || PQresultStatus(res) != PGRES_TUPLES_OK || att_id == NULL) {
			fprintf(stdout, "error is occuring at...\n%s\n", query->str);
			EXIT_STARTELEMENT;
		}
		PQclear(res);

#ifdef __DEBUG__
		fprintf(stdout, "create attribute ...name='%s' value='%s'\n",
				attrs[i], attrs[i + 1]);
#endif
		i += 2;
	}

	/**
	 * set current parent to current node
	 */
	g_string_free(query, TRUE);
	g_free(CurrentNode);
	free(p);
	free(a1);
	free(a2);
}

void
endElement(void *userData, const char *name)
{
	char *parent;
	PGresult *res;
	GString *query = g_string_sized_new(100);

#ifdef __DEBUG__
	fprintf(stdout, "End element...\n");
#endif

	if (StackIsLoaded)
		setTextNode();

	/**
	 * get parent id
	 */
	g_string_sprintf(query, "SELECT parent FROM xml_node WHERE id=%d",
					 CurrentParent);
	res = PQexec(conn, query->str);
	parent = PQgetvalue(res, 0, 0);
	if (!res || PQresultStatus(res) != PGRES_TUPLES_OK || parent == NULL) {
		fprintf(stderr, "Can't select parent of '%d'\n", CurrentParent);
		g_string_free(query, TRUE);
		PQclear(res);
		EXIT_DISCONN();
	}

	CurrentParent = atoi(parent);
	g_string_free(query, TRUE);
	PQclear(res);
}

void
defaultData(void *userData, const XML_Char * s, int len)
{
}

void
commentData(void *userData, const XML_Char * data)
{
	char *comment = (char *) malloc(strlen(data) + 1);
	char *p = comment;
	PGresult *res;
	GString *query = g_string_sized_new(100);

	if (StackIsLoaded)
		setTextNode();

	/**
	 * convert UTF8 -> EUC
	 */
	memcpy(comment, data, strlen(data) + 1);
	comment =
		g_convert(comment, strlen(comment), from_codeset, to_codeset, NULL,
				  NULL, NULL);

#ifdef __DEBUG__
	printf("[commentData]\n");
#endif
	g_string_sprintf(query, "SELECT new_childcomment(%d,'%s')",
					 CurrentParent, comment);
	res = PQexec(conn, query->str);
	if (!res ||
		PQresultStatus(res) != PGRES_TUPLES_OK ||
		PQgetvalue(res, 0, 0) == NULL) {
		fprintf(stderr, "create comment under element...='%d'\n",
				CurrentParent);
		free(p);
		g_string_free(query, TRUE);
		PQclear(res);
		EXIT_DISCONN();
	}
	PQclear(res);

	g_string_free(query, TRUE);
	free(p);
}

void
piData(void *userData, const XML_Char * target, const XML_Char * data)
{
	char *pitarget = (char *) malloc(strlen(target) + 1);
	char *pidata = (char *) malloc(strlen(data) + 1);
	char *p1 = pitarget;
	char *p2 = pidata;
	PGresult *res;
	GString *query = g_string_sized_new(100);

#ifdef __DEBUG__
	printf("[piData]\n");
#endif

	if (StackIsLoaded)
		setTextNode();

	memcpy(pitarget, target, strlen(target) + 1);
	memcpy(pidata, data, strlen(data) + 1);
	pitarget =
		g_convert(pitarget, strlen(pitarget), from_codeset, to_codeset,
				  NULL, NULL, NULL);
	pidata =
		g_convert(pidata, strlen(pidata), from_codeset, to_codeset, NULL,
				  NULL, NULL);

	g_string_sprintf(query, "SELECT new_childpi(%d,'%s','%s')",
					 CurrentParent, pitarget, pidata);
	res = PQexec(conn, query->str);
	if (!res ||
		PQresultStatus(res) != PGRES_TUPLES_OK ||
		PQgetvalue(res, 0, 0) == NULL) {
		fprintf(stderr, "Can't Create PI ...'%s'\n", pidata);
		free(p1);
		free(p2);
		g_string_free(query, TRUE);
		PQclear(res);
		EXIT_DISCONN();
	}
	PQclear(res);

#ifdef __DEBUG__
	printf("create pi under...='%d'\n", CurrentParent);
#endif

	g_string_free(query, TRUE);
	free(p1);
	free(p2);
}

/*
void
startNS(void *userData, const XML_Char * prefix, const XML_Char * uri)
{
	char *nsprefix = (char *) malloc(strlen(prefix) + 1);
	char *nsuri = (char *) malloc(strlen(uri) + 1);

	memcpy(nsprefix, prefix, strlen(prefix) + 1);
	memcpy(nsuri, uri, strlen(uri) + 1);
	nsprefix =
		g_convert(nsprefix, strlen(nsprefix), to_codeset, from_codeset,
				  NULL, NULL, NULL);
	nsuri =
		g_convert(nsuri, strlen(nsuri), to_codeset, from_codeset, NULL,
				  NULL, NULL);

#ifdef __DEBUG__
	printf("[startNS] create namespace ...uri='%s',prefix='%s'\n", nsuri,
		   nsprefix);
#endif
	stack_push_pointer(nsprefixStack, nsprefix);
	stack_push_pointer(nsuriStack, nsuri);

	StackNsIsLoaded = TRUE;
}
*/

/*
void
endNS(void *userData, const XML_Char * prefix)
{
}
*/

void
xmlDecl(void *userData, const XML_Char * version,
		const XML_Char * encoding, int standalone)
{
}

void
charData(void *userData, const XML_Char * s, int len)
{
	char *str = (char *) malloc(len * 2 + 1);
	GError *err = NULL;

#ifdef __DEBUG__
	printf("[XML_CharacterDataHandler]\n");
#endif

	if (str != NULL) {
		PQescapeString((char *) str, (char *) s, len);
	 	//StrReplace(str,"'","''");        // select new_childText(1,'')
		StrReplace(str, "\n", "\\n");
		StrReplace(str, "\t", "\\t");

#ifdef __DEBUG__
		printf("before conversion:%s\n", str);
#endif

		if (*str != '\0') {
			str =
				g_convert(str, strlen(str), to_codeset, from_codeset, NULL,
						  NULL, &err);
			if (err != NULL) {
				fprintf(stderr, "error:%s\n", err->message);
				g_error_free(err);
				exit(1);
			}
#ifdef __DEBUG__
			printf("content:%s\n", str);
#endif
			stack_push_pointer(textNodeStack, str);
			StackIsLoaded = TRUE;
		}
	}
	return;
}

/*
 *  StrReplace - replace string
 *
 *  this function needs care for handring, buffer overflow
 */
static int
StrReplace(char *ap, const char *bp, const char *cp)
{
	int bs, cs, rc;
	char *limit;

	rc = 0;
	bs = strlen(bp);
	cs = strlen(cp);
	limit = ap + strlen(ap) - bs;

	while (ap <= limit) {
		if (strncmp(ap, bp, bs) == 0) {
			memmove(ap + cs, ap + bs, limit - ap + 1);
			strncpy(ap, cp, cs);
			limit += cs - bs;
			rc++;
			ap += cs;
		}
		else
			ap++;
	}
	return rc;
}// streplace

/*
 *  trim - Remove leading, trailing, & excess embedded spaces
 */
static void
trim(char *s)
{
	int len = strlen(s);

	/*
	 * trim trailing whitespace
	 */
	while (len > 0 && isspace(s[len - 1]))
		s[--len] = '\0';

	/*
	 * trim leading whitespace
	 */
	memmove(s, &s[strspn(s, " \n\r\t\v")], len);
}// trim

static char *
strcompress(const char *source)
{
	const char *p = source;
	char *dest = g_malloc(strlen(source) + 1);
	char *q = dest;

	while (*p) {
		if (*p == '\\') {
			p++;
			switch (*p) {
			  case 'b':
				  *q++ = '\b';
				  break;
			  case 'f':
				  *q++ = '\f';
				  break;
			  case 'n':
				  *q++ = '\n';
				  break;
			  case 'r':
				  *q++ = '\r';
				  break;
			  case 't':
				  *q++ = '\t';
				  break;
			  default:			/* Also handles \" and \\ */
				  *q++ = *p;
				  break;
			}
		}
		else
			*q++ = *p;
		p++;
	}
	*q = 0;

	return dest;
}

static char *
basename(char *fn)
{
	char *p = strrchr(fn, '/');

	if (p != NULL)
		return p + 1;

	return fn;
}//basename

static _Bool
setTextNode(void)
{
	GString *query = g_string_sized_new(120);
	int i;
	char *str;
	char *text = NULL;
	PGresult *res;
	char *tmp;

	for (i = 0; textNodeStack->used > 0; i++) {
		str = stack_pop(textNodeStack);
		if (i == 0)
			text = g_strconcat(str, NULL);

		else
			text = g_strconcat(str, text, NULL);

		g_free(str);
	}

	if (strip_space) {
		if (text == NULL)
			return FALSE;
		tmp = strcompress(text);
		trim(tmp);
		if (*tmp == '\0')
			return FALSE;		//equals
		g_free(tmp);
	}

	g_string_sprintf(query, "SELECT new_childtext(%d,'%s')",
					 CurrentParent, text);
	res = PQexec(conn, query->str);
	if (!res ||
		PQresultStatus(res) != PGRES_TUPLES_OK ||
		PQgetvalue(res, 0, 0) == NULL) {
		fprintf(stderr, "Could not create TextNode\n");
		fprintf(stderr, "SELECT new_childtext(%d,'%s')\n",
				CurrentParent, text);
		PQclear(res);
		g_string_free(query, TRUE);
		EXIT_DISCONN();
	}

#ifdef __DEBUG__
	printf("create text_node under...='%d'\n", CurrentParent);
#endif

	/*
	 * clean up
	 */
	PQclear(res);
	g_free(text);

	g_string_free(query, TRUE);
	stack_init(textNodeStack);
	StackIsLoaded = FALSE;

	return TRUE;
}//setTextNode

/*
   static _Bool setNsNode(void)
   {
   int i;
   char *uri,*prefix;
   PGresult *res;
   GString *query = g_string_sized_new(100);

   for( i=0; nsuriStack->used > 0;i++)
   {
   uri = stack_pop(nsuriStack);
   prefix = stack_pop(nsprefixStack);

   if (*prefix == '\0')
   {
   g_string_append_printf(query, "SELECT setnamespace(%d,'%s')",
   CurrentParent, uri);
   }
   else
   {
   g_string_append_printf(query, "SELECT setnamespace(%d,'%s','%s')",
   CurrentParent, prefix, uri);
   }
   g_free(uri);
   g_free(prefix);
   }

   res = PQexec(conn, query->str);
   if (!res ||
   PQresultStatus(res) != PGRES_TUPLES_OK ||
   PQgetvalue(res, 0, 0) == NULL)
   {
   fprintf(stderr, "Could not create namespace\n");
   PQclear(res);
   g_string_free(query, TRUE);
   EXIT_DISCONN();
   }
   PQclear(res);

   g_string_free(query, TRUE);
   stack_init(nsuriStack);
   stack_init(nsprefixStack);
   StackNsIsLoaded = FALSE;

   return TRUE;
   }
 */

static void
showUsage()
{
	fprintf(stderr, "usage: pgxload [-u user] [-p password] [-d dbname] "
			"[-h host] [-o port] \n\t"
			"[-f file_encoding] [-t db_encoding] " "[-s] [-?] [file]\n\n");

	fprintf(stderr, "option [-s] enables strip_space"
			"(Get detail on XSLT reference page),\n"
			"defalut values is false.\n\n");

	fprintf(stderr, "--- default paramator ---\n"
			"DB user set to postgres.\n"
			"Input file encoding set to UTF-8.\n"
			"DB encoding set to your locale encoding.\n");
	return;
}

int
main(int argc, char *argv[])
{
	char buf[BUFSIZ];
	size_t len;
	int done;
	char *parent;
	//char *CurrentDocument;
	void *ret;
	XML_Parser parser;
	FILE *fi;
	PGresult *res;
	GString *query;
	extern int optind;
	int ch;
	char *host = __HOST;
	char *port = __PORT;
	char *user = __LOGIN;
	char *dbname = __DBNAME;
	char *password = __PWD;
	char *fname;
	_Bool has_from = FALSE;
	_Bool has_to = FALSE;

#ifdef STRIP_SPACE
	strip_space = TRUE;
#else //STRIP_SPACE
	strip_space = FALSE;
#endif //STRIP_SPACE
	from_codeset = __FROM_ENC;
	to_codeset = __TO_ENC;

	while ((ch = getopt(argc, argv, "h:o:u:d:p:f:t:s?")) != EOF) {
		switch ((char) ch) {
		  case 'h':
			  host = strdup(optarg);
			  break;
		  case 'o':
			  port = strdup(optarg);
			  break;
		  case 'u':
			  user = strdup(optarg);
			  break;
		  case 'd':
			  dbname = strdup(optarg);
			  break;
		  case 'p':
			  password = strdup(optarg);
			  break;
		  case 'f':
			  from_codeset = strdup(optarg);
			  has_from = TRUE;
			  break;
		  case 't':
			  to_codeset = strdup(optarg);
			  has_to = TRUE;
			  break;
		  case 's':
			  strip_space = TRUE;
			  break;
		  case '?':
		  default:
			  showUsage();
			  exit(1);
		}
	}
	argc -= optind;
	argv += optind;

	if (argc > 0)
		fname = argv[0];
	else {
		showUsage();
		exit(1);
	}

	/**
	 *	open document
	 */
	fi = fopen(fname, "r");
	if (fi == NULL) {
		fprintf(stderr, "%s cannot be opened.\n", fname);
		exit(1);
	}

	/**
	 *	connect db
	 */
	conn = PQsetdbLogin(host, port, __OPTION, __TTY, dbname, user, password);
	if (PQstatus(conn) == CONNECTION_BAD) {
		fprintf(stderr, "Connection to database '%s' failed.\n", __DBNAME);
		fprintf(stderr, "%s\n", PQerrorMessage(conn));
		EXIT_DISCONN();
	}

	g_free(host);
	g_free(port);
	g_free(user);
	g_free(dbname);
	g_free(password);

	query = g_string_sized_new(100);

	TRANZ_BEGIN;

	/**
	 * create document
	 */
	g_string_sprintf(query, "SELECT create_new_document('%s')",
					 basename(fname));
	res = PQexec(conn, query->str);
	parent = PQgetvalue(res, 0, 0);
	if (!res || PQresultStatus(res) != PGRES_TUPLES_OK || parent == NULL) {
		fprintf(stderr, "create document failed\n");
		g_string_free(query, TRUE);
		PQclear(res);
		EXIT_DISCONN();
	}
	PQclear(res);

	CurrentParent = atoi(parent);
	g_string_free(query, TRUE);

	/**
	 * create parser handler
	 *
	 * Construct a new parser using the suite of memory handling functions
	 * specified in ms.
	 */
	pgxml_mhs_init();
	parser = XML_ParserCreate_MM(NULL, &mhs, NULL);
	//parser = XML_ParserCreate_MM(NULL, &mhs, ":");
	//parser = XML_ParserCreate(NULL);

	if (!parser) {
		fprintf(stderr, "pgxload: Could'nt create expat parser handler\n");
		EXIT_DISCONN();
	}

	/**
	 * discriminate
	 */
	XML_SetUserData(parser, (void *) &ret);

	/**
	 * create event handler
	 */
	XML_SetElementHandler(parser, startElement, endElement);
	XML_SetCharacterDataHandler(parser, charData);
	XML_SetCommentHandler(parser, commentData);
	XML_SetProcessingInstructionHandler(parser, piData);
 	//XML_SetNamespaceDeclHandler(parser, startNS, endNS);
	XML_SetXmlDeclHandler(parser, xmlDecl);
	XML_SetDefaultHandler(parser, defaultData);
	//XML_SetReturnNSTriplet(parser, TRUE);

	/**
	 * setting for stack
	 */
	StackIsLoaded = FALSE;
	textNodeStack = stack_create();
	StackNsIsLoaded = FALSE;
	nsprefixStack = stack_create();
	nsuriStack = stack_create();

	/**
	 * parse xml
	 *
	 * buf : read data
	 * len : data size
	 * done: is done ? true or false
	 */
	do {
		len = fread(buf, 1, BUFSIZ, fi);
		done = len < sizeof(buf);
		printf("read:%d buffer:%d done:%s\n", len, sizeof(buf),
			   done ? "true" : "false");
		if (!XML_Parse(parser, buf, len, done)) {
			fprintf(stderr,
					"%s at line %d\n",
					XML_ErrorString(XML_GetErrorCode(parser)),
					XML_GetCurrentLineNumber(parser));
			XML_ParserFree(parser);
			fclose(fi);
			EXIT_DISCONN();
		}
	}
	while (!done);

	TRANZ_COMMIT;

	/**
	 * clean up
	 */
	if (has_from)
		g_free(from_codeset);
	if (has_to)
		g_free(to_codeset);
	XML_ParserFree(parser);
	PQfinish(conn);
	fclose(fi);
	stack_destroy(textNodeStack);
	fprintf(stdout, "[pgxload]finished\n");

	return 0;
}
