/**
 * xpsql.c,v 1.5 2003/07/17 14:10:45 myui Exp
 * 
 * ----------------------
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *      products derived from this software without specific prior written
 *   permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License (see COPYING.GPL), in which case the provisions
 * of the GPL are required INSTEAD OF the above restrictions.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.    IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

//#define __DEBUG__
//#define __INFO__	/* show infomations */
// #define __OFF_ESCAPING__

#include <string.h>
#include "xpsql.h"
#include "executor/spi.h"	/* for SPI_exec */
#include "commands/sequence.h"	/* for nextval() */
#include "lib/stringinfo.h"	/* for makeStringInfo() */
#include "mb/pg_wchar.h"
#include "commands/trigger.h"	/* -"- and triggers */

#define MAX_PLANS 24

#ifndef INT4ARRAYOID
#define INT4ARRAYOID 1007	/* see bootstrap.c, defined in there. */
#endif

#define _SPI_CONN() connected=(SPI_connect()==SPI_ERROR_CONNECT)
#define _SPI_FIN() if(!connected)SPI_finish()

#define R_NORMAL(r) \
		switch(r) \
		{ \
			case 1: elog(NOTICE, "Status: SPI_OK_CONNECT");break; \
			case 2: elog(NOTICE, "Status: SPI_OK_FINISH");break; \
			case 3: elog(NOTICE, "Status: SPI_OK_FETCH");break; \
			case 4: elog(NOTICE, "Status: SPI_OK_UTILITY");break; \
			case 5: elog(NOTICE, "Status: SPI_OK_SELECT");break; \
			case 6: elog(NOTICE, "Status: SPI_OK_SELINTO");break; \
			case 7: elog(NOTICE, "Status: SPI_OK_INSERT");break; \
			case 8: elog(NOTICE, "Status: SPI_OK_DELETE");break; \
			case 9: elog(NOTICE, "Status: SPI_OK_UPDATE");break; \
			case 10: elog(NOTICE, "Status: SPI_OK_CURSOR");break; \
			default: elog(NOTICE, "Unknown status: %d", ret); \
		}

#define R_ERROR(r) \
		switch (r) \
		{ \
			case -1: elog(ERROR, "[SPI_ERROR_CONNECT] %s", query);break; \
			case -2: elog(ERROR, "[SPI_ERROR_COPY] %s", query);break; \
			case -3: elog(ERROR, "[SPI_ERROR_OPUNKNOWN] %s", query);break; \
			case -4: elog(ERROR, "[SPI_ERROR_UNCONNECTED] %s", query);break; \
 			case -5: elog(ERROR, "[SPI_ERROR_CURSOR] %s", query);break; \
			case -6: elog(ERROR, "[SPI_ERROR_ARGUMENT] %s", query);break; \
			case -7: elog(ERROR, "[SPI_ERROR_PARAM] %s", query);break; \
			case -8: elog(ERROR, "[SPI_ERROR_TRANSACTION] %s", query);break; \
			case -9: elog(ERROR, "[SPI_ERROR_NOATTRIBUTE] %s", query);break; \
			case -10: elog(ERROR, "[SPI_ERROR_NOOUTFUNC] %s", query);break; \
			case -11: elog(ERROR, "[SPI_ERROR_TYPUNKNOWN] %s", query);break; \
			default: elog(ERROR, "[SPI_ERROR_OTHER] %s", query); \
		}

/**
 * exec_query
 *		-- internal function
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
int
exec_query(char *query)
{
	int ret;

#ifdef __DEBUG__
	elog(NOTICE, "[exec_query] Query:%s", query);
#endif
	if ((ret = SPI_exec(query, 0)) < 0) {
		R_ERROR(ret);
	}
#ifdef __DEBUG__
	R_NORMAL(ret);
#endif

	return ret;
}

/**
 * execp_query
 * 
 * @author Makoto Yui <yuin@bb.din.or.jp>
 * @see spi.h
 * @params qstr 
 * 		query skelton
 * @params nargs
 * 		number of input parameters ($1, $2, etc.)
 * @params argtypes
 * 		pointer to an array containing the OIDs of the data types of the parameters
 * @params nulls
 * 		 An array describing which parameters are null.
 * @params ptype
 * 		specify the type of plan.
 */
int
execp_query(char *qstr, int nargs, Oid * argtypes, Datum * pvals, char *nulls)
{
	EPlan *qplan;				/* prepared plan */
	int stat;

#ifdef __DEBUG__
	elog(NOTICE, "[execp_query] Query:%s", qstr);
#endif

/*
 * find query plan which stored in memory 
 */
	qplan = find_plan(qstr, &sPlans, &n_sPlans);

/*
 * if there is no plan, we have to prepare plan ... 
 */
	if (qplan->splan == NULL) {
#ifdef __INFO__
		elog(NOTICE, "There is no allocated plan. create new plan.");
#endif
		create_plan(&qplan, qstr, nargs, argtypes);
	}
#ifdef __INFO__
	else {
		elog(NOTICE, "Reuse already prepared plan.");
	}
#endif

/*
 * Ok, execute prepared plan. 
 */
	if (qplan->splan == NULL)
		elog(ERROR, "execute plan is NULL");
#ifdef __DEBUG__
	elog(NOTICE, "execute prepared plan.");
#endif
	stat = SPI_execp(qplan->splan, pvals, NULL, 0);
	return stat;
}

/**
 * va_exec_query:
 * 
 */
static int
va_exec_query(char *query_str, ...)
{
	va_list args;
	char query[XBUFSIZE + 1];

	va_start(args, query_str);
	vsnprintf(query, XBUFSIZE, query_str, args);
	va_end(args);

	return exec_query(query);
}

/**
 * h_replace:
 *
 * warning : undefined when buffer over flows.
 */
PG_FUNCTION_INFO_V1(h_replace);
Datum
h_replace(PG_FUNCTION_ARGS)
{
	char *str =
		DatumGetCString(DirectFunctionCall1
						(textout, PointerGetDatum(PG_GETARG_TEXT_P(0))));
	char *pat =
		DatumGetCString(DirectFunctionCall1
						(textout, PointerGetDatum(PG_GETARG_TEXT_P(1))));
	char *sub =
		DatumGetCString(DirectFunctionCall1
						(textout, PointerGetDatum(PG_GETARG_TEXT_P(2))));
	char *buf, *match_p;
	size_t lenpat, lensub, lenstr, lenres;
	struct varlena *res;

	if (*str == '\0')
		PG_RETURN_NULL();
	if (pat == NULL || *pat == '\0')
		PG_RETURN_TEXT_P(PG_GETARG_DATUM(0));

	lenstr = strlen(str);
	lenpat = strlen(pat);	// from
	lensub = strlen(sub);	// to
	lenres = lenstr + lensub - lenpat;

	buf = (char *) repalloc(str, lenres);

/*
 * make room for substituted string, or remove slack after it 
 */
	if ((match_p = strstr(buf, pat)) != NULL && lensub != lenpat)
		memmove(match_p + lensub, match_p + lenpat, lenstr + 1 - lenpat);

	memcpy(match_p, sub, lensub);

	res = palloc(lenres + VARHDRSZ);
	res->vl_len = lenres + VARHDRSZ;

	memcpy(VARDATA(res), buf, lenres);
	PG_RETURN_TEXT_P(res);
}

/**
 * xml_tagid
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(xml_tagid);
Datum
xml_tagid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum tag_id, seqname;
	Datum pval1[1];
	Datum pval2[2];
	Datum tag_name = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { VARCHAROID };
	Oid argtypes2[2] = { INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[xml_tagid] tagname=%s", GET_STR(tag_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_tag WHERE name = $1";
	pval1[0] = tag_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0) {
		tag_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
	}
	else {
		seqname =
			DirectFunctionCall1(textin, CStringGetDatum("xml_tagid_seq"));
	/*
	 * nextval now returns int64; coerce down to int32 
	 */
		tag_id = Int32GetDatum((int32)
							   DatumGetInt64(DirectFunctionCall1
											 (nextval, seqname)));
		pfree(DatumGetTextP(seqname));

		qstr = "INSERT INTO xml_tag VALUES($1,$2)";
		pval2[0] = tag_id;
		pval2[1] = tag_name;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}
	_SPI_FIN();
	PG_RETURN_DATUM(tag_id);
}

/**
 * xml_attid:
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(xml_attid);
Datum
xml_attid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum att_id, seqname;
	Datum pval1[1];
	Datum pval2[2];
	Datum att_name = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { VARCHAROID };
	Oid argtypes2[2] = { INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[xml_attid] attname=%s", GET_STR(att_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_attribute WHERE name = $1";
	pval1[0] = att_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0)
		att_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
	else {
		seqname =
			DirectFunctionCall1(textin, CStringGetDatum("xml_attid_seq"));
		att_id = Int32GetDatum((int32)
							   DatumGetInt64(DirectFunctionCall1
											 (nextval, seqname)));
		pfree(DatumGetTextP(seqname));

		qstr = "INSERT INTO xml_attribute VALUES($1,$2)";
		pval2[0] = att_id;
		pval2[1] = att_name;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}

	_SPI_FIN();
	PG_RETURN_DATUM(att_id);
}


/**
 * xml_nsid
 *		-- unspecified in DOM level 2, proprietary extention 
 *
 *  Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(xml_nsid);
Datum
xml_nsid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum ns_id, seqname;
	Datum pval1[1];
	Datum pval2[2];
	Datum ns_name = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { VARCHAROID };
	Oid argtypes2[2] = { INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[xml_nsid] nsname=%s", GET_STR(ns_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_namespace WHERE name = $1";
	pval1[0] = ns_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0)
		ns_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
	else {
		seqname =
			DirectFunctionCall1(textin, CStringGetDatum("xml_nsid_seq"));
		ns_id = Int32GetDatum((int32)
							  DatumGetInt64(DirectFunctionCall1
											(nextval, seqname)));
		pfree(DatumGetTextP(seqname));

		qstr = "INSERT INTO xml_namespace VALUES($1,$2)";
		pval2[0] = ns_id;
		pval2[1] = ns_name;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}

	_SPI_FIN();
	PG_RETURN_DATUM(ns_id);
}


/**
 * xml_piid
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(xml_piid);
Datum
xml_piid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum pi_id, seqname;
	Datum pval1[1];
	Datum pval2[2];
	Datum pi_name = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { VARCHAROID };
	Oid argtypes2[2] = { INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[xml_piid] piname=%s", GET_STR(pi_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_pi WHERE name = $1";
	pval1[0] = pi_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0)
		pi_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
	else {
		seqname =
			DirectFunctionCall1(textin, CStringGetDatum("xml_piid_seq"));
		pi_id = Int32GetDatum((int32)
							  DatumGetInt64(DirectFunctionCall1
											(nextval, seqname)));
		pfree(DatumGetTextP(seqname));

		qstr = "INSERT INTO xml_pi VALUES($1,$2)";
		pval2[0] = pi_id;
		pval2[1] = pi_name;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}

	_SPI_FIN();
	PG_RETURN_DATUM(pi_id);
}

/**
 * get_tagid:
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(get_tagid);
Datum
get_tagid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum tag_id;
	Datum pval1[1];
	Datum tag_name = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[get_tagid] tagname:%s", GET_STR(tag_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_tag WHERE name = $1";
	pval1[0] = tag_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0) {
		tag_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
		_SPI_FIN();
		PG_RETURN_DATUM(tag_id);
	}
	else {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
}

/**
 * get_attid:
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(get_attid);
Datum
get_attid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum att_id;
	Datum pval1[1];
	Datum att_name = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[get_attid] attname=%s", GET_STR(att_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_attribute WHERE name = $1";
	pval1[0] = att_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0) {
		att_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
		_SPI_FIN();
		PG_RETURN_DATUM(att_id);
	}
	else {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
}

/**
 * get_nsid:
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(get_nsid);
Datum
get_nsid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum ns_id;
	Datum pval1[1];
	Oid argtypes1[1] = { VARCHAROID };
	Datum ns_name = PG_GETARG_DATUM(0);

#ifdef __DEBUG__
	elog(NOTICE, "[get_nsid] nsname=%s", GET_STR(ns_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_namespace WHERE name = $1";
	pval1[0] = ns_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0) {
		ns_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
		_SPI_FIN();
		PG_RETURN_DATUM(ns_id);
	}
	else {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
}

/**
 * get_piid:
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(get_piid);
Datum
get_piid(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum pi_id;
	Datum pval1[1];
	Oid argtypes1[1] = { VARCHAROID };
	Datum pi_name = PG_GETARG_DATUM(0);

#ifdef __DEBUG__
	elog(NOTICE, "[get_piid] piname=%s", GET_STR(pi_name));
#endif
	_SPI_CONN();

	qstr = "SELECT tagid FROM xml_pi WHERE name = $1";
	pval1[0] = pi_name;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0) {
		pi_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
		_SPI_FIN();
		PG_RETURN_DATUM(pi_id);
	}
	else {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
}

/**
 * addChildElement
 *		-- unspecified in DOM level 2, proprietary extention
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(addchildelement);
Datum
addchildelement(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum doc_id, node_id, seqname;
	Datum pval1[1];
	Datum pval5[5];
	Datum parent_id = PG_GETARG_DATUM(0);
	Datum tag_name = PG_GETARG_DATUM(1);
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes5[5] = { INT4OID, INT2OID, INT4OID, INT4OID, INT4OID };

#ifdef __DEBUG__
	elog(NOTICE, "[addChildElement]");
#endif
	_SPI_CONN();

/*
 * Error check routine for DOMException, HIERARCHY_REQUEST_ERR
 */
	pval1[0] = parent_id;
	qstr = "SELECT docid FROM xml_node "
		"WHERE id = $1 AND (kind = '1' OR (kind = '0' AND child IS NULL))";
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
	doc_id = SPI_getbinval(SPI_tuptable->vals[0],
						   SPI_tuptable->tupdesc, 1, &isnull);

	seqname = DirectFunctionCall1(textin, CStringGetDatum("xml_nodeid_seq"));
	node_id = Int32GetDatum((int32)
							DatumGetInt64(DirectFunctionCall1
										  (nextval, seqname)));
	pfree(DatumGetTextP(seqname));

/*
 * SET node info
 */
	pval5[0] = pval5[3] = node_id;
	pval5[1] = Int16GetDatum((int16) KIND_ELEMENT);
	pval5[2] = doc_id;
	pval5[4] = DirectFunctionCall1(xml_tagid, tag_name);
	qstr =
		"INSERT INTO xml_node(id,kind,docid,pathid,tagid) VALUES($1,$2,$3,$4,$5)";
	execp_query(qstr, 5, argtypes5, pval5, NULL);

/*
 * SET parent-child link
 */
	DirectFunctionCall2(appendchild, parent_id, node_id);

	_SPI_FIN();
	PG_RETURN_DATUM(node_id);
}

/**
 * addChildPi:
 *		-- unspecified in DOM level 2, proprietary extention
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(addchildpi);
Datum
addchildpi(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum doc_id, node_id, seqname;
	Datum pval1[1];
	Datum pval6[6];
	Datum parent_id = PG_GETARG_DATUM(0);
	Datum pi_name = PG_GETARG_DATUM(1);
	Datum pi_data = PG_GETARG_DATUM(2);
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes6[6] =
		{ INT4OID, INT2OID, INT4OID, INT4OID, INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[addChildPi]");
#endif
	_SPI_CONN();

	qstr = "SELECT docid FROM xml_node WHERE id = $1 AND kind = '1'";
	pval1[0] = parent_id;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
	doc_id = SPI_getbinval(SPI_tuptable->vals[0],
						   SPI_tuptable->tupdesc, 1, &isnull);

	seqname = DirectFunctionCall1(textin, CStringGetDatum("xml_nodeid_seq"));
	node_id = Int32GetDatum((int32)
							DatumGetInt64(DirectFunctionCall1
										  (nextval, seqname)));
	pfree(DatumGetTextP(seqname));
/*
#ifndef __OFF_ESCAPING__
	pi_data = escapeString(pi_data);
#endif
*/
	qstr = "INSERT INTO xml_node(id,kind,docid,pathid,tagid,value) "
		"VALUES($1,$2,$3,$4,$5,$6)";
	pval6[0] = pval6[3] = node_id;
	pval6[1] = Int16GetDatum((int16) KIND_PI);
	pval6[2] = doc_id;
	pval6[4] = DirectFunctionCall1(xml_piid, pi_name);
	pval6[5] = pi_data;
	execp_query(qstr, 6, argtypes6, pval6, NULL);

	DirectFunctionCall2(appendchild, parent_id, node_id);

	_SPI_FIN();
	PG_RETURN_DATUM(node_id);
}

/**
 * addChildText:
 *		-- unspecified in DOM level 2, proprietary extention
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(addchildtext);
Datum
addchildtext(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum doc_id, node_id, seqname;
	Datum pval1[1];
	Datum pval5[5];
	Datum parent_id = PG_GETARG_DATUM(0);
	Datum content = PG_GETARG_DATUM(1);
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes5[5] = { INT4OID, INT2OID, INT4OID, INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[addChildText]");
#endif
	_SPI_CONN();

	qstr = "SELECT docid FROM xml_node WHERE id = $1 AND kind = '1'";
	pval1[0] = parent_id;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
	doc_id = SPI_getbinval(SPI_tuptable->vals[0],
						   SPI_tuptable->tupdesc, 1, &isnull);

	seqname = DirectFunctionCall1(textin, CStringGetDatum("xml_nodeid_seq"));
	node_id = Int32GetDatum((int32)
							DatumGetInt64(DirectFunctionCall1
										  (nextval, seqname)));
	pfree(DatumGetTextP(seqname));

//	content = escapeString(content);

	qstr = "INSERT INTO xml_node(id,kind,docid,pathid,value) "
		"VALUES($1,$2,$3,$4,$5)";
	pval5[0] = pval5[3] = node_id;
	pval5[1] = Int16GetDatum((int16) KIND_TEXT);
	pval5[2] = doc_id;
	pval5[4] = content;
	execp_query(qstr, 5, argtypes5, pval5, NULL);

	DirectFunctionCall2(appendchild, parent_id, node_id);

	_SPI_FIN();
	PG_RETURN_DATUM(node_id);
}

/**
 * addChildComment:
 *		-- unspecified in DOM level 2, proprietary extention
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(addchildcomment);
Datum
addchildcomment(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum doc_id, node_id, seqname;
	Datum pval1[1];
	Datum pval5[5];
	Datum parent_id = PG_GETARG_DATUM(0);
	Datum content = PG_GETARG_DATUM(1);
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes5[5] = { INT4OID, INT2OID, INT4OID, INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[addChildComment]");
#endif
	_SPI_CONN();

	qstr = "SELECT docid FROM xml_node WHERE id = $1 AND kind = '2'";
	pval1[0] = parent_id;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}
	doc_id = SPI_getbinval(SPI_tuptable->vals[0],
						   SPI_tuptable->tupdesc, 1, &isnull);

	seqname = DirectFunctionCall1(textin, CStringGetDatum("xml_nodeid_seq"));
	node_id = Int32GetDatum((int32)
							DatumGetInt64(DirectFunctionCall1
										  (nextval, seqname)));
	pfree(DatumGetTextP(seqname));

/*
#ifndef __OFF_ESCAPING__
	content = escapeString(content);
#endif
*/
	qstr = "INSERT INTO xml_node(id,kind,docid,pathid,value) "
		"VALUES($1,$2,$3,$4,$5)";
	pval5[0] = pval5[3] = node_id;
	pval5[1] = Int16GetDatum((int16) KIND_COMMENT);
	pval5[2] = doc_id;
	pval5[4] = content;
	execp_query(qstr, 5, argtypes5, pval5, NULL);

	DirectFunctionCall2(appendchild, parent_id, node_id);

	_SPI_FIN();
	PG_RETURN_DATUM(node_id);
}

/**
 * setAttribute:
 * 
 * Description: 
 * 		Adds a new attribute. 
 * 
 * Parameters:  
 * 		DOMString name  -  The name of the attribute to create or alter.
 * 		DOMString value  -  Value to set in string form.
 * 
 * Returns:  
 * 		attid
 * 
 * Modified by Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(setattribute);
Datum
setattribute(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum doc_id, node_id, seqname, att_id;
	Datum pval1[1];
	Datum pval2[2];
	Datum pval6[6];
	Datum parent_id = PG_GETARG_DATUM(0);
	Datum att_name = PG_GETARG_DATUM(1);
	Datum att_value = PG_GETARG_DATUM(2);
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes2[2] = { INT4OID, INT4OID };
	Oid argtypes6[6] =
		{ INT4OID, INT2OID, INT4OID, INT4OID, INT4OID, VARCHAROID };

#ifdef __DEBUG__
	elog(NOTICE, "[setAttribute]");
#endif
	_SPI_CONN();

	qstr = "SELECT id FROM xml_node "
		"WHERE kind = '5' AND parent = $1 AND tagid = $2";
	pval2[0] = parent_id;
	pval2[1] = att_id = DirectFunctionCall1(xml_attid, att_name);
	execp_query(qstr, 2, argtypes2, pval2, NULL);

/*
 * #ifndef __OFF_ESCAPING__ att_value = escapeString(att_value); #endif 
 */

	if (SPI_processed > 0) {
		node_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
		qstr = "UPDATE xml_node SET value = $1 WHERE id = $2";
		argtypes2[0] = VARCHAROID;
		pval2[0] = att_value;
		pval2[1] = node_id;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	/*
	 * va_exec_query("UPDATE xml_node SET value = '%s' WHERE id = %d",
	 * att_value, DatumGetInt32(node_id)); 
	 */
	}
	else {
		qstr = "SELECT docid FROM xml_node WHERE id = $1";
		pval1[0] = parent_id;
		execp_query(qstr, 1, argtypes1, pval1, NULL);

		if (SPI_processed == 0) {
			_SPI_FIN();
			PG_RETURN_NULL();
		}
		doc_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);

		seqname =
			DirectFunctionCall1(textin, CStringGetDatum("xml_nodeid_seq"));
		node_id = Int32GetDatum((int32)
								DatumGetInt64(DirectFunctionCall1
											  (nextval, seqname)));
		pfree(DatumGetTextP(seqname));

		qstr = "INSERT INTO xml_node(id,kind,docid,pathid,tagid,value) "
			"VALUES($1,$2,$3,$4,$5,$6)";
		pval6[0] = pval6[3] = node_id;
		pval6[1] = Int16GetDatum((int16) KIND_ATTRIBUTE);
		pval6[2] = doc_id;
		pval6[4] = att_id;
		pval6[5] = att_value;
		execp_query(qstr, 6, argtypes6, pval6, NULL);

		qstr = "UPDATE xml_node SET parent = $1 WHERE id = $2";
		pval2[0] = parent_id;
		pval2[1] = node_id;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	/*
	 * va_exec_query("UPDATE xml_node SET parent = %d WHERE id = %d",
	 * DatumGetInt32(parent_id), DatumGetInt32(node_id)); 
	 */
	}

	_SPI_FIN();
	PG_RETURN_DATUM(node_id);
}

/**
 * setNamespace
 *		-- unspecified in DOM level 2, proprietary extention 
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(setnamespace);
Datum
setnamespace(PG_FUNCTION_ARGS)
{
	char *qstr;
	bool connected, isnull;
	Datum doc_id, node_id, ns_name, seqname, ns_id, ns_value;
	Datum pval1[1];
	Datum pval2[2];
	Datum pval5[5];
	Datum parent_id = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes2[2] = { INT4OID, INT4OID };
	Oid argtypes5[5] = { INT4OID, INT2OID, INT4OID, INT4OID, VARCHAROID };
	StringInfo str = makeStringInfo();

	if (fcinfo->nargs == 3) {
		appendStringInfo(str, "xmlns:%s",
						 DatumGetCString(DirectFunctionCall1(varcharout, PG_GETARG_DATUM(1))));
		ns_name = DirectFunctionCall1(varcharin, CStringGetDatum(str->data));
		ns_value = PG_GETARG_DATUM(2);
	}
	else if (fcinfo->nargs == 2) {
		ns_name = DirectFunctionCall1(varcharin, CStringGetDatum("xmlns"));
		ns_value = PG_GETARG_DATUM(1);
	}
	else {
		PG_RETURN_NULL();
	}

#ifdef __DEBUG__
	elog(NOTICE, "[setNamespace]");
#endif
	_SPI_CONN();

	qstr = "SELECT id FROM xml_node "
		"WHERE kind = '6' AND parent = $1 AND tagid = $2";
	pval2[0] = parent_id;
	pval2[1] = ns_id = DirectFunctionCall1(xml_nsid, ns_name);
	execp_query(qstr, 2, argtypes2, pval2, NULL);

/*
 * #ifndef __OFF_ESCAPING__ ns_value = escapeString(ns_value); #endif 
 */

	if (SPI_processed > 0) {
		node_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);
		qstr = "UPDATE xml_node SET value = $1 WHERE id = $2";
		argtypes2[0] = VARCHAROID;
		pval2[0] = ns_value;
		pval2[1] = node_id;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}
	else {
		qstr = "SELECT docid FROM xml_node WHERE id = $1";
		pval1[0] = parent_id;
		execp_query(qstr, 1, argtypes1, pval1, NULL);

		if (SPI_processed == 0) {
			_SPI_FIN();
			PG_RETURN_NULL();
		}
		doc_id = SPI_getbinval
			(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull);

		seqname =
			DirectFunctionCall1(textin, CStringGetDatum("xml_nodeid_seq"));
		node_id = Int32GetDatum((int32)
								DatumGetInt64(DirectFunctionCall1
											  (nextval, seqname)));
		pfree(DatumGetTextP(seqname));
		qstr = "INSERT INTO xml_node(id,kind,docid,tagid,value) "
			"VALUES($1,$2,$3,$4,$5)";
		pval5[0] = node_id;
		pval5[1] = Int16GetDatum((int16) KIND_NAMESPACE);
		pval5[2] = doc_id;
		pval5[3] = DirectFunctionCall1(xml_nsid, ns_name);
		pval5[4] = ns_value;
		execp_query(qstr, 5, argtypes5, pval5, NULL);

		qstr = "UPDATE xml_node SET parent = $1 WHERE id = $2";
		pval2[0] = parent_id;
		pval2[1] = node_id;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	/*
	 * va_exec_query("UPDATE xml_node SET parent = %d WHERE id = %d",
	 * DatumGetInt32(parent_id), DatumGetInt32(node_id)); 
	 */
	}

	pfree(str->data);
	_SPI_FIN();
	PG_RETURN_INT32(node_id);
}

/**
 * appendChild()
 * 		-- DOM Level2 Core
 * 
 * Description:
 * 		Adds the node newChild to the end of the list of children of this node. 
 * 
 * Parameters:  
 * 		Node newChild  -  The node to add. If it is a DocumentFragment object, 
 * 						the entire contents of the document fragment are moved 
 * 						into the child list of this node.
 * 
 * Returns:	
 * 		Node -  The node added.
 * 
 * Exceptions:  
 * 		DOMException HIERARCHY_REQUEST_ERR
 * 			Raised if this node is of a type that does not allow children 
 * 			of the type of the newChild node, or if the node to append is 
 * 			one of this node's ancestors or this node itself. 
 * 		DOMException WRONG_DOCUMENT_ERR
 * 			Raised if newChild was created from a different document than 
 * 			the one that created this node. 
 * 		DOMException NO_MODIFICATION_ALLOWED_ERR	-- not yet imple'd
 * 			Raised if this node is readonly or if the previous parent of 
 * 			the node being inserted is readonly. 
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(appendchild);
Datum
appendchild(PG_FUNCTION_ARGS)
{
	char *qstr;
	int kind_code;
	bool connected;
	Datum pval1[1];
	Datum pval2[2];
	Datum pval5[5];
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes2[2] = { INT4OID, INT4OID };
	Oid argtypes5[5] = { INT4OID, INT4OID, INT4OID, INT2OID, INT2OID };
	Datum parent_node_id = PG_GETARG_DATUM(0);
	Datum node_id = PG_GETARG_DATUM(1);

#ifdef __DEBUG__
	elog(NOTICE, "[appendChild] PARENT_ID:%d CHILD_ID:%d",
		 (int) DatumGetInt32(parent_node_id), (int) DatumGetInt32(node_id));
#endif
	_SPI_CONN();

	qstr = "SELECT kind FROM xml_node WHERE id = $1";
	pval1[0] = node_id;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_BOOL(FALSE);
	}
	kind_code =
		atoi(SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1));

	if ((kind_code == KIND_ROOT) || (kind_code == KIND_ATTRIBUTE)
		|| (kind_code == KIND_NAMESPACE)) {
		_SPI_FIN();
		elog(ERROR,
			 "[appendChild] Raising a DOMException, HIERARCHY_REQUEST_ERR.");
	}

	qstr = "UPDATE xml_node SET parent = $1 WHERE id = $2";
	pval2[0] = parent_node_id;
	pval2[1] = node_id;
	execp_query(qstr, 2, argtypes2, pval2, NULL);

/*
 * parent node has child ?
 */
	qstr = "SELECT id FROM xml_node WHERE id = $1 AND child IS NOT NULL";
	pval1[0] = parent_node_id;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed > 0) {
	/*
	 * link from older brother
	 */
		qstr = "UPDATE xml_node SET next = $1 "
			"WHERE parent = $2 AND id != $3 AND "
			"kind <> $4 AND kind <> $5 AND next IS NULL";
		pval5[0] = pval5[2] = node_id;
		pval5[1] = parent_node_id;
		pval5[3] = Int16GetDatum((int16) KIND_ATTRIBUTE);
		pval5[4] = Int16GetDatum((int16) KIND_NAMESPACE);
		execp_query(qstr, 5, argtypes5, pval5, NULL);
	}
	else {
	/*
	 * link from parent
	 */
		qstr = "UPDATE xml_node SET child = $1 WHERE id = $2";
		pval2[0] = node_id;
		pval2[1] = parent_node_id;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}

	_SPI_FIN();
	PG_RETURN_BOOL(TRUE);
}

/**
 * insertBefore()
 * 		-- DOM Level2 Core
 * 
 * Description:
 * 		inserts the node newChild before the existing child node refChild. 
 * 		If refChild is null, insert newChild at the end of the list of children.
 * 
 * Parameters:  
 * 		Node newChild  -  The node to insert.
 *		Node refChild  -  The reference node, 
 * 							i.e., the node before which the new node must be inserted.
 * Returns:	
 * 		Node -  The node being inserted.  
 * 
 * Exceptions:
 * 		DOMException HIERARCHY_REQUEST_ERR
 * 			Raised if this node is of a type that does not allow children of the type 
 * 			of the newChild node, or if the node to insert is one of this node's ancestors 
 * 			or this node itself. 
 * 		DOMException WRONG_DOCUMENT_ERR			 -- diffrent specification with DOM
 * 			Raised if newChild was created from a different document than the one that 
 * 			created this node. 
 * 		DOMException NO_MODIFICATION_ALLOWED_ERR -- not yet implemented
 * 			Raised if this node is readonly or if the parent of the node being inserted is 
 * 			readonly. 
 * 		DOMException NOT_FOUND_ERR				 -- not yet implemented
 * 			Raised if refChild is not a child of this node. 
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(insertbefore);
Datum
insertbefore(PG_FUNCTION_ARGS)
{
	char *qstr;
	int kind_code;
	bool connected, isnull;
	Datum parent_id, ref_doc, child_id;
	Datum pval1[1], pval2[2], pval4[4];
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes2[2] = { INT4OID, INT4OID };
	Oid argtypes4[4] = { INT4OID, INT4OID, INT4OID, INT4OID };
	Datum newChild = PG_GETARG_DATUM(0);
	Datum refChild = PG_GETARG_DATUM(1);

#ifdef __DEBUG__
	elog(NOTICE, "[insertBefore] InsertNode:%d BeforeNode:%d",
		 (int) DatumGetInt32(newChild), (int) DatumGetInt32(refChild));
#endif
	_SPI_CONN();

	/**
	 * HIERARCHY_REQUEST_ERR 
	 */
	qstr = "SELECT kind, docid, parent FROM xml_node WHERE id = $1";
	pval1[0] = refChild;
	execp_query(qstr, 1, argtypes1, pval1, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_BOOL(FALSE);
	}
	kind_code =
		atoi(SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1));

	if ((kind_code == KIND_ROOT) || (kind_code == KIND_ATTRIBUTE)
		|| (kind_code == KIND_NAMESPACE)) {
		_SPI_FIN();
		PG_RETURN_BOOL(FALSE);
	}

	ref_doc =
		SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 2,
					  &isnull);
	parent_id =
		SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 3,
					  &isnull);

	if (isnull) {
#ifdef __DEBUG__
		elog(NOTICE, "Parent is NULL:%d", (int) DatumGetInt32(refChild));
#endif
		_SPI_FIN();
		PG_RETURN_BOOL(FALSE);
	}

/*
 * new node will be first-child of the parent ? 
 */
	qstr = "SELECT child FROM xml_node WHERE id = $1 AND docid = $2";
	pval2[0] = parent_id;
	pval2[1] = ref_doc;
	execp_query(qstr, 2, argtypes2, pval2, NULL);

	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_BOOL(FALSE);
	}

	child_id =
		SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1,
					  &isnull);

	if (refChild == child_id) {
	/*
	 * if firstchild 
	 */
		qstr = "UPDATE xml_node SET child = $1 WHERE id = $2";
		pval2[0] = newChild;
		pval2[1] = parent_id;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}
	else {
	/*
	 * modify older brother's link info 
	 */
		qstr = "UPDATE xml_node SET next = $1 "
			"WHERE id = (SELECT id FROM xml_node WHERE next = $2)";
		pval2[0] = refChild;
		pval2[1] = newChild;
		execp_query(qstr, 2, argtypes2, pval2, NULL);
	}

	qstr =
		"UPDATE xml_node SET docid = $1, parent = $2, next = $3 WHERE id = $4";
	pval4[0] = ref_doc;
	pval4[1] = parent_id;
	pval4[2] = refChild;
	pval4[3] = newChild;
	execp_query(qstr, 4, argtypes4, pval4, NULL);

	_SPI_FIN();
	PG_RETURN_BOOL(TRUE);
}



/**
 * stringValue
 * 		-- XPath 1.0
 *
 * This function is too stupid, might be replaced.
 * I recommeds you don't use this syntax. '/aaa[bbb="ccc"]'
 * 
 * we have alternative way using this syntax, '/aaa[bbb=="ccc]'.
 * 
 * Author: Makoto Yui <yuin@bb.din.or.jp>
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>
 */
PG_FUNCTION_INFO_V1(stringvalue);
Datum
stringvalue(PG_FUNCTION_ARGS)
{
	int i, kind;
	int needed, newlen, datalen;
	char *qstr, *dewey;
	bool connected, isnull;
	VarChar *result;
	Datum value;
	Datum pval1[1];
	char *ret_text = NULL;
	StringInfo str = makeStringInfo();
	Datum node_id = PG_GETARG_DATUM(0);
	Oid argtypes1[1] = { INT4OID };

	pval1[0] = node_id;

#ifdef __DEBUG__
	elog(NOTICE, "[stringValue] nodeid:%d", (int) DatumGetInt32(node_id));
#endif

	_SPI_CONN();
	qstr = "SELECT kind, dewey FROM xml_node WHERE id = $1";
	execp_query(qstr, 1, argtypes1, pval1, NULL);
	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}

	kind =
		(int)
		DatumGetInt16(SPI_getbinval
					  (SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1,
					   &isnull));
	dewey = SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 2);

	switch (kind) {
	  case KIND_ROOT:
	  case KIND_ELEMENT:
		  va_exec_query("SELECT value FROM xml_node "
						"WHERE kind = '2' AND dewey > '%s' AND dewey < nextSibling('%s') "
						"ORDER BY dewey", dewey, dewey);
		  if (SPI_processed == 0) {
			  _SPI_FIN();
			  PG_RETURN_NULL();
		  }
	  /*
	   * get(enlarge) enough buffer 
	   */
		  newlen = 128 + 32 * SPI_processed;
		  str->data = (char *) malloc(newlen);
		  str->maxlen = newlen;

#ifdef __DEBUG__
		  elog(NOTICE, "SPI_processed=%d, expected len=%d", SPI_processed,
			   newlen);
#endif

		  for (i = 0; i < SPI_processed; i++) {
		  // ret_text have to be NULL terminated.
			  ret_text =
				  SPI_getvalue(SPI_tuptable->vals[i], SPI_tuptable->tupdesc,
							   1);

#ifdef __DEBUG__
			  elog(NOTICE, "i=%d, SPI_processed=%d", i, SPI_processed);
#endif

			  if (ret_text != NULL) {
				  datalen = strlen(ret_text);
				  needed = str->len + datalen + 1;

				  if (needed > str->maxlen) {
					  if (i + 2 == SPI_processed)	// when the next is last
													// entry.
						  newlen += datalen;
					  else
						  newlen = 2 * needed;
					  str->data = (char *) realloc(str->data, newlen);
					  str->maxlen = newlen;
				  }

			  /*
			   * OK, append the data 
			   */
				  memcpy(str->data + str->len, ret_text, datalen);
				  str->len += datalen;
				  str->data[str->len] = '\0';

				  pfree(ret_text);
			  }
		  }

#ifdef __DEBUG__
		  elog(NOTICE, "actual len=%d", str->len);
#endif
		  _SPI_FIN();
		  if (str->data == NULL)
			  PG_RETURN_NULL();
		  else {
			  result = DatumGetVarCharP(DirectFunctionCall1
										(varcharin,
										 CStringGetDatum(str->data)));
			  free(str->data);
			  PG_RETURN_VARCHAR_P(result);
		  }
	  case KIND_TEXT:
	  case KIND_COMMENT:
	  case KIND_ATTRIBUTE:
	  case KIND_NAMESPACE:
	  case KIND_PI:
		  qstr = "SELECT value FROM xml_node WHERE id = $1";
		  execp_query(qstr, 1, argtypes1, pval1, NULL);
		  if (SPI_processed == 0) {
			  _SPI_FIN();
			  PG_RETURN_NULL();
		  }
		  value =
			  SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1,
							&isnull);
		  if (isnull) {
			  _SPI_FIN();
			  PG_RETURN_NULL();
		  }
		  _SPI_FIN();
		  PG_RETURN_VARCHAR_P(value);
	  default:
		  _SPI_FIN();
		  PG_RETURN_NULL();
	}
}

PG_FUNCTION_INFO_V1(toxml);
Datum
toxml(PG_FUNCTION_ARGS)
{
	Datum node_id = PG_GETARG_DATUM(0);
	int i, needed, newlen, datalen;
	char *qstr;
	bool connected, isnull;
	Datum d_start, d_nextsibling;
	Datum pvals[10];
	VarChar *result;
	StringInfo str = makeStringInfo();
	char *ret_text = NULL;
	Oid argtypes1[1] = { INT4OID };
	Oid argtypes[10] =
		{ INT4ARRAYOID, INT4ARRAYOID, INT4ARRAYOID, INT4ARRAYOID,
		INT4ARRAYOID,
		INT4ARRAYOID, INT4ARRAYOID, INT4ARRAYOID, INT4ARRAYOID, INT4ARRAYOID
	};
/*
 * have to allocate to accesable space 
 */
	Datum *pvals1 = (Datum *) palloc(sizeof(Datum));

	pvals1[0] = node_id;

#ifdef __DEBUG__
	elog(NOTICE, "[toXML]");
#endif

	_SPI_CONN();
	qstr = "SELECT dewey, nextSibling(dewey) FROM xml_node WHERE id = $1";
	execp_query(qstr, 1, argtypes1, pvals1, NULL);
	pfree(pvals1);
	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}

	d_start =
		SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1,
					  &isnull);
	d_nextsibling =
		SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 2,
					  &isnull);
	pvals[0] = pvals[2] = pvals[4] = pvals[6] = pvals[8] = d_start;
	pvals[1] = pvals[3] = pvals[5] = pvals[7] = pvals[9] = d_nextsibling;

	qstr = "SELECT "
		"xn.dewey as dewey, 1 as seq, '<' || xt.name as parsed_text "
		"FROM xml_node xn, xml_tag xt "
		"WHERE xn.tagid = xt.tagid AND xn.kind = '1' "
		"AND (xn.dewey >= $1 AND xn.dewey < $2) "
		"UNION ALL "
		"SELECT "
		"xn.dewey, xn.kind - 3 as seq, "
		"CASE WHEN xn.kind = '5' "
		"THEN ' ' || xa.name || '=\"' || xn.value || '\"' "
		"ELSE ' ' || ns.name || '=\"' || xn.value || '\"' END "
		"FROM xml_node xn LEFT JOIN xml_attribute xa USING(tagid) "
		"LEFT JOIN xml_namespace ns USING(tagid) "
		"WHERE (xn.kind = '5' OR xn.kind = '6') "
		"AND (xn.dewey >= $3 AND xn.dewey < $4) "
		"UNION ALL "
		"SELECT "
		"xn.dewey + 0, 4 as seq, '>' "
		"FROM xml_node xn "
		"WHERE xn.kind = '1' AND (xn.dewey >= $5 and xn.dewey < $6) "
		"UNION ALL "
		"SELECT "
		"xn.dewey, 5 as seq, "
		"CASE WHEN xn.kind = '2' THEN xn.value "
		"WHEN xn.kind = '3' THEN '<!-- ' || xn.value || ' -->' "
		"ELSE '<?' || pi.name || xn.value || '?>' END "
		"FROM xml_node xn LEFT JOIN xml_pi pi USING(tagid) "
		"WHERE (xn.kind = '2' OR xn.kind = '3' OR xn.kind = '4') "
		"AND (xn.dewey >= $7 AND xn.dewey < $8) "
		"UNION ALL "
		"SELECT "
		"xn.dewey + 2147483647, 6 as seq, '</' || xt.name || '>' "
		"FROM xml_node xn, xml_tag xt "
		"WHERE xn.tagid = xt.tagid AND xn.kind = '1' "
		"AND (xn.dewey >= $9 AND xn.dewey < $10) ORDER BY dewey, seq";

#ifdef __DEBUG__
	elog(NOTICE, "%s", qstr);
#endif
	execp_query(qstr, 10, argtypes, pvals, NULL);
	if (SPI_processed == 0) {
		_SPI_FIN();
		PG_RETURN_NULL();
	}

/*
 * get enough Buffer 
 */
	str->data = (char *) repalloc(str->data, XBUFSIZE);
	str->maxlen = XBUFSIZE;

	for (i = 0; i < SPI_processed; i++) {
		ret_text =
			SPI_getvalue(SPI_tuptable->vals[i], SPI_tuptable->tupdesc, 3);
		if (ret_text != NULL) {
			datalen = strlen(ret_text);
			needed = str->len + datalen + 1;

			if (needed > str->maxlen) {
			/*
			 * There is not enough buffer, have to make more room. 
			 *
			 * [note]
			 *   if data size have lots variation, this strategy
			 *   is not efficient. especially, enlarging is caused
			 *   in the last call.
			 *
			 *   1. len=0; maxlen=5; needed=100;    
			 *        -> newlen=200; actual=100;
			 *   2. len=100; maxlen=200; needed=1000;
			 *       -> newlen=2000; actual=1100;
			 *   3. len=1000; maxlen=2000; needed=1;
			 *       -> newlen=2000; actual=1101;
			 *        cause 899 bytes of futility.
			 *
			 *   influence a remained percentage and differ-size of
			 *   newlen and actual. 
			 */
				newlen = 2 * needed;

				str->data = (char *) repalloc(str->data, newlen);
				str->maxlen = newlen;
			}

		/*
		 * OK, append the data 
		 */
			memcpy(str->data + str->len, ret_text, datalen);
			str->len += datalen;
			str->data[str->len] = '\0';

			pfree(ret_text);
		}
	}

	_SPI_FIN();	// disconenct

	if (str->data == NULL)
		PG_RETURN_NULL();

	result = DatumGetVarCharP(DirectFunctionCall1
							  (varcharin, CStringGetDatum(str->data)));
	pfree(str->data);

	PG_RETURN_VARCHAR_P(result);
}

PG_FUNCTION_INFO_V1(trg_xorder);
Datum
trg_xorder(PG_FUNCTION_ARGS)
{
	TriggerData *trigdata = (TriggerData *) fcinfo->context;
	Trigger *trigger;			/* to get trigger name */
	TupleDesc tupdesc;			/* tuple descriptor */
	Relation rel;				/* triggered relation */
	HeapTuple trigtuple;		/* triggered NEW tuple */
	HeapTuple trigtuple_old;	/* triggered OLD tuple */
	bool connected;				/* SPI connection */
	bool isnull;				/* getbinval() */
	char *qstr;

/*
 * arguments 
 */
	int kind;
	char *dewey, *old_dewey;
	char *next_id, *old_next;
	int32 node_id, doc_id, pathid, parent_id, child_id, tag_id;
	int32 old_pathid, old_parent, old_tag;

/*
 * select into variable 
 */
	char *path, *xorder;
	int dims;
	int32 l_order;

/*
 * Plans
 */
	Datum pval1[1];

// Datum pval2[2];
	Oid argtypes1[1] = { INT4OID };
// Oid argtypes2[2] = { INT4OID, VARCHAROID };

// initialize
	child_id = -1;
	path = xorder = NULL;
	dims = 0;
	l_order = 1;

#ifdef __DEBUG__
	elog(NOTICE, "[trg_xorder] enter");
#endif

/*
 * Some checks first...
 */
/*
 * Called by trigger manager ? 
 */
	if (!CALLED_AS_TRIGGER(fcinfo))
		elog(ERROR, "trg_xorder: not fired by trigger manager");

/*
 * check for null values 
 */
	if (!TRIGGER_FIRED_BY_UPDATE(trigdata->tg_event)
		&& !TRIGGER_FIRED_AFTER(trigdata->tg_event))
		elog(ERROR, "trg_xorder: not fired by after update trigger");

	trigtuple = trigdata->tg_newtuple;
	trigtuple_old = trigdata->tg_trigtuple;

	rel = trigdata->tg_relation;
	trigger = trigdata->tg_trigger;
	tupdesc = rel->rd_att;

/*
 * Connect to SPI manager 
 */
	_SPI_CONN();

	/**
     *  Well, the string value of attribute
     * 
     *  aval[1]=id, aval[2]=kind, aval[3]=document, aval[4]=pathid,
     *  aval[5]=dewey, aval[6]=parent, aval[7]=child, a_val[8]=next
     *  aval[9]=tag
     */
	parent_id = DatumGetInt32(SPI_getbinval(trigtuple, tupdesc, 6, &isnull));
	if (isnull) {
#ifdef __DEBUG__
		elog(NOTICE, "parent is null");
#endif
		_SPI_FIN();
		return PointerGetDatum(trigtuple);
	}

	old_parent =
		DatumGetInt32(SPI_getbinval(trigtuple_old, tupdesc, 6, &isnull));
	old_tag =
		DatumGetInt32(SPI_getbinval(trigtuple_old, tupdesc, 9, &isnull));
	tag_id = DatumGetInt32(SPI_getbinval(trigtuple, tupdesc, 9, &isnull));
	if (parent_id == old_parent && (tag_id == old_tag || isnull)) {
#ifdef __DEBUG__
		elog(NOTICE, "no need changing");
#endif
		_SPI_FIN();
		return PointerGetDatum(trigtuple);
	}
	else {
		next_id = SPI_getvalue(trigtuple, tupdesc, 8);
		old_next = SPI_getvalue(trigtuple_old, tupdesc, 8);
	}

	pathid = DatumGetInt32(SPI_getbinval(trigtuple, tupdesc, 4, &isnull));
	old_pathid =
		DatumGetInt32(SPI_getbinval(trigtuple_old, tupdesc, 4, &isnull));
	dewey = SPI_getvalue(trigtuple, tupdesc, 5);
	old_dewey = SPI_getvalue(trigtuple_old, tupdesc, 5);
	if (pathid != old_pathid) {
#ifdef __DEBUG__
		elog(NOTICE, "already changed\nnew_pathid=%d,old_pathid=%d", pathid,
			 old_pathid);
#endif
		_SPI_FIN();
		return PointerGetDatum(trigtuple);
	}
	if (old_dewey != NULL && strcmp(dewey, old_dewey) != 0) {
#ifdef __DEBUG__
		elog(NOTICE, "already changed\nnew_dewey=%s,old_dewey=%s", dewey,
			 old_dewey);
#endif
		_SPI_FIN();
		return PointerGetDatum(trigtuple);
	}

	node_id = DatumGetInt32(SPI_getbinval(trigtuple, tupdesc, 1, &isnull));
	kind =
		(int16) DatumGetInt16(SPI_getbinval(trigtuple, tupdesc, 2, &isnull));
	doc_id = DatumGetInt32(SPI_getbinval(trigtuple, tupdesc, 3, &isnull));

#ifdef __DEBUG__
	elog(NOTICE,
		 "node_id=%d,kind=%d,doc_id=%d,pathid=%d,dewey=%s,parent_id=%d\n"
		 "child_id=%d,next_id=%s,tag_id=%d", node_id, kind, doc_id, pathid,
		 dewey, parent_id, child_id, next_id, tag_id);
	elog(NOTICE, "old_pathid=%d,old_parent=%d,old_next=%s,old_tag=%d",
		 old_pathid, old_parent, old_next, old_tag);
#endif

	if (kind == KIND_ELEMENT || kind == KIND_TEXT || kind == KIND_COMMENT
		|| kind == KIND_PI) {
		qstr = "SELECT last(dewey) + 1 FROM xml_node WHERE next = $1";
		pval1[0] = Int32GetDatum(node_id);
		execp_query(qstr, 1, argtypes1, pval1, NULL);

		if (SPI_processed > 0) {
			l_order =
				DatumGetInt32(SPI_getbinval
							  (SPI_tuptable->vals[0], SPI_tuptable->tupdesc,
							   1, &isnull));
			if (isnull)
				l_order = 1;
		}
	}

	if (kind == KIND_ELEMENT) {
	/*
	 * select pathexp and calculate dewey 
	 */
		va_exec_query("SELECT "
					  "(xp.pathexp || '#/' || (SELECT name FROM xml_tag WHERE tagid = %d)),"
					  "pn.dewey + %d,"
					  "icount(pn.dewey) + 1 "
					  "FROM xml_node pn, xml_path xp "
					  "WHERE pn.id = %d AND xp.pathid = pn.pathid",
					  tag_id, l_order, parent_id);

		if (SPI_processed > 0) {
			path =
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1);
			xorder =
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 2);
			dims = (int)
				DatumGetInt32(SPI_getbinval
							  (SPI_tuptable->vals[0], SPI_tuptable->tupdesc,
							   3, &isnull));
		}
		else {
			elog(ERROR, "[SPI ERROR] path & label could't calculate");
		}

		if (dewey == NULL) {
		/*
		 * set current node's path 
		 */
			va_exec_query
				("INSERT INTO xml_path(pathid,pathexp) VALUES(%d,'%s')",
				 node_id, path);
		/*
		 * qstr = "INSERT INTO xml_path(pathid,pathexp) VALUES($1,$2)";
		 * pval2[0] = Int32GetDatum(node_id); pval2[1] =
		 * CStringGetDatum(path); execp_query(qstr, 2, argtypes2, pval2,
		 * NULL); 
		 */
		}
		else {
		/*
		 * update influenced node's label 
		 */
			if (next_id != NULL) {
				va_exec_query("UPDATE xml_node "
							  "SET dewey[%d] = dewey[%d] + 1 "
							  "WHERE "
							  "subarray(dewey,1,%d) = subarray('%s'::_int4,1,%d) AND "
							  "dewey[%d] >= %d",
							  dims, dims, dims - 1, xorder, dims - 1, dims,
							  l_order);
			}

		/*
		 * update insert-node's descendant path and label 
		 */
			SPI_getbinval(trigtuple, tupdesc, 7, &isnull);
			if (!isnull) {
			/*
			 * update insert-node's descendant label 
			 */
				va_exec_query("UPDATE xml_node "
							  "SET dewey = ('%s' + subarray(dewey,icount('%s') + 1)) "
							  "WHERE dewey > '%s' AND dewey < nextSibling('%s')",
							  xorder, dewey, dewey, dewey);
			/*
			 * need to modify desendant's pathexp... 
			 */
				va_exec_query("UPDATE xml_path "
							  "SET "
							  "pathexp = h_replace(pathexp,"
							  "(SELECT pathexp FROM xml_path WHERE pathid = %d),"
							  "'%s'"
							  ") "
							  "FROM xml_node cn "
							  "WHERE "
							  "cn.dewey > '%s' AND cn.dewey < nextSibling('%s') AND "
							  "xml_path.pathid = cn.pathid",
							  pathid, path, xorder, xorder);

			}

		/*
		 * set current node's path 
		 */
			va_exec_query
				("UPDATE xml_path SET pathexp = '%s', united = NULL WHERE pathid = %d",
				 path, node_id);
		}

	/*
	 * set current node's label 
	 */
		va_exec_query("UPDATE xml_node SET dewey = '%s' WHERE id = %d",
					  xorder, node_id);
	}
	else if (kind == KIND_TEXT || kind == KIND_COMMENT || kind == KIND_PI) {

	/*
	 * select pathexp and calculate dewey 
	 */
		va_exec_query("SELECT "
					  "xp.pathexp || '#/%s',"
					  "pn.dewey + %d,"
					  "icount(pn.dewey) + 1 "
					  "FROM xml_node pn, xml_path xp "
					  "WHERE pn.id = %d AND xp.pathid = pn.pathid",
					  kind == KIND_TEXT ? "text()" : kind ==
					  KIND_COMMENT ? "comment()" : kind ==
					  KIND_PI ? "processing-instruction()" : "", l_order,
					  parent_id);

		if (SPI_processed > 0) {
			path =
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1);
			xorder =
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 2);
			dims = (int)
				DatumGetInt32(SPI_getbinval
							  (SPI_tuptable->vals[0], SPI_tuptable->tupdesc,
							   3, &isnull));
		}
		else {
			elog(ERROR, "[SPI ERROR]\n");
		}

		if (dewey == NULL) {
		/*
		 * set current node's path 
		 */
			va_exec_query
				("INSERT INTO xml_path(pathid,pathexp) VALUES(%d,'%s')",
				 node_id, path);
		/*
		 * qstr = "INSERT INTO xml_path(pathid,pathexp) VALUES($1,$2)";
		 * pval2[0] = Int32GetDatum(node_id); pval2[1] =
		 * CStringGetDatum(path); execp_query(qstr, 2, argtypes2, pval2,
		 * NULL); 
		 */
		}
		else {
		/*
		 * update influenced node's label 
		 */
			if (next_id != NULL) {
				va_exec_query("UPDATE xml_node SET dewey[%d] = dewey[%d] + 1 "
							  "WHERE subarray(dewey,1,%d) = subarray('%s'::_int4,1,%d) AND "
							  "dewey[%d] >= %d",
							  dims, dims, dims - 1, xorder, dims - 1, dims,
							  l_order);
			}

		/*
		 * set current node's path 
		 */
			va_exec_query
				("UPDATE xml_path SET pathexp = '%s', united = NULL WHERE pathid=%d",
				 path, node_id);
		}

	/*
	 * set current node's label 
	 */
		va_exec_query("UPDATE xml_node SET dewey = '%s' WHERE id = %d",
					  xorder, node_id);
	}
	else if (kind == KIND_ATTRIBUTE) {

	/*
	 * select pathexp and calculate dewey 
	 */
		va_exec_query("SELECT "
					  "(xp.pathexp || '#/@' || (SELECT name FROM xml_attribute WHERE tagid = %d)),"
					  "pn.dewey + 0 "
					  "FROM xml_node pn, xml_path xp "
					  "WHERE pn.id = %d AND xp.pathid = pn.pathid",
					  tag_id, parent_id);

		if (SPI_processed > 0) {
			path =
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1);
			xorder =
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 2);
		}
		else {
			elog(ERROR, "[SPI ERROR] result not found\n");
		}

	/*
	 * set current node's path 
	 */
		if (dewey == NULL) {
			va_exec_query
				("INSERT INTO xml_path(pathid,pathexp) VALUES(%d,'%s')",
				 node_id, path);
		/*
		 * qstr = "INSERT INTO xml_path(pathid,pathexp) VALUES($1,$2)";
		 * pval2[0] = Int32GetDatum(node_id); pval2[1] =
		 * CStringGetDatum(path); execp_query(qstr, 2, argtypes2, pval2,
		 * NULL); 
		 */
		}
		else {
			va_exec_query
				("UPDATE xml_path SET pathexp = '%s', united IS NULL WHERE pathid = %d",
				 path, node_id);
		}

	/*
	 * set current node's label 
	 */
		va_exec_query("UPDATE xml_node SET dewey = '%s' WHERE id = %d",
					  xorder, node_id);
	}
	else if (kind == KIND_NAMESPACE) {
	/*
	 * set current node's label 
	 */
		va_exec_query("UPDATE xml_node "
					  "SET dewey = (SELECT dewey + 0 FROM xml_node WHERE id = %d) "
					  "WHERE id = %d", parent_id, node_id);
	}
	else {
	// When node-type is invalid one
		_SPI_FIN();
		return PointerGetDatum(trigtuple);
	}

#ifdef __DEBUG__
	elog(NOTICE, "[trg_xorder] finish");
#endif

/*
 * Clean up 
 */
	_SPI_FIN();

	return PointerGetDatum(trigtuple);
}

/**
 * escapeString
 * 
 */
/*
static char *
escapeString(char *s)
{
	char *p1, *ret;
	size_t len;
	char *p2 = NULL;

#ifdef __DEBUG__
	elog(NOTICE, "escapeString()");
#endif

	if ((p1 = strchr(s, '\'')) != NULL || (p2 = strchr(s, '\\')) != NULL) {
	// escape string
		len = strlen(s);
#ifdef __DEBUG__
		elog(NOTICE, "alloced memory size is..%d", (int) (len * 1.4));
#endif
		ret = (char *) repalloc(s, (int) (len * 1.4));

		if (p1 != NULL)
			StrReplace(ret, "\'", "\'\'");
		if (p2 != NULL)
			StrReplace(ret, "\\", "\\\\");

		return ret;
	}

	return s;	// needs no changing
}
*/

/*
 *  StrReplace - replace string
 *  
 *  this function needs care for handring, segfault.
 */
/*
static int
StrReplace(char *ap, const char *bp, const char *cp)
{
	int bs, cs, rc;
	char *limit;

	rc = 0;
	bs = strlen(bp);
	cs = strlen(cp);
	limit = ap + strlen(ap) - bs;

	while (ap <= limit) {
		if (strncmp(ap, bp, bs) == 0) {
			memmove(ap + cs, ap + bs, limit - ap + 1);
			strncpy(ap, cp, cs);
			limit += cs - bs;
			rc++;
			ap += cs;
		}
		else
			ap++;
	}
	return rc;
}  // streplace
*/

/*
 * for caching of query plans, stolen from contrib/spi/\*.c 
 */
static EPlan *
find_plan(char *ident, EPlan ** eplan, int *nplans)
{
// static values for LRU map, minimux value is the oldest.
	static uint32 LRUMap[MAX_PLANS];
	int j;
	int i = 0;
	int replaceIdx = 0;
	uint32 max = 1;
	EPlan *newp = NULL;

#ifdef __INFO__
	elog(NOTICE, "[Before] Show counters of prepared plans..%d", n_sPlans);
#endif

/*
 * find a plan if there is prepared plans.
 */
	if (*nplans > 0) {
	// find a plan which matched to the specified label first.
		for (i = 0; i < *nplans; i++) {
			if (strcmp((*eplan)[i].ident, ident) == 0)
				break;
		}
	// found
		if (i < *nplans) {
			for (j = 0; j < (*nplans); j++) {
				max = (LRUMap[j] > max) ? LRUMap[j] : max;
			}
			if (LRUMap[i] == max)
				LRUMap[i] = max + 1;
#ifdef __INFO__
			elog(NOTICE, "found..replaced index = %d", i);
#endif
			return (*eplan + i);
		}
	// set replace index.
		replaceIdx = i;
	}

	if (*nplans >= MAX_PLANS)	// if exceeds MAX_PLANS, replace plan-Map
	// with LRU strategy.
	{
	/*
	 * restrict the number of plans.
	 * if there is no more space, the oldest place is assigned newly.
	 * 
	 * The replacement strategy is almost likey to Least Reacently Used(LRU).
	 * 
	 * [Caution] Current Limitation.
	 * Don't call more than 2^32(=4294967296) times in a session.
	 */
	// find the least recently used index to replace.
		for (i = 1, replaceIdx = 0, j = LRUMap[0]; i < MAX_PLANS; i++) {
			if (LRUMap[i] < j) {
				replaceIdx = i;
				j = LRUMap[i];
			}
		}
#ifdef __INFO__
		elog(NOTICE, "relace index = %d", replaceIdx);
#endif
	// decrement plan counter
		(*nplans)--;
	// free LRU one.
		newp = *eplan + replaceIdx;
		if (newp->ident != NULL)
			free(newp->ident);

#ifdef __INFO__
		elog(NOTICE, "There are too many plans. "
			 "The oldest plan is replaced to this new plan.");
#endif
	}
	else if (*nplans == 0)	// there is no allocated space for plan.
	{
		newp = *eplan = (EPlan *) malloc(MAX_PLANS * sizeof(EPlan));
	}

// expand the space and create new space for a plan.
	if (*nplans > 0)
		newp = *eplan + replaceIdx;

// set identifier.
	newp->ident = (char *) malloc(strlen(ident) + 1);
	strcpy(newp->ident, ident);
	newp->splan = NULL;

// set LRU.
	for (i = 0; i < MAX_PLANS; i++) {
		max = (LRUMap[i] > max) ? LRUMap[i] : max;
	}
	LRUMap[replaceIdx] = max;

// increment plan counter
	(*nplans)++;

#ifdef __DEBUG__
	elog(NOTICE, "created plan..replaced index = %d", replaceIdx);
	for (i = 0; i < MAX_PLANS; i++) {
		elog(NOTICE, "%d -", LRUMap[i]);
	}
#endif

	return (newp);	// return space for a plan which specified by the
// identifier
}

/*
 * prepare plan and save it.
 * 
 * @return prepared plan
 */
static void
create_plan(EPlan ** qplan, char *querystr, int nargs, Oid * argtypes)
{
	void *pplan;

#ifdef __DEBUG__
	elog(NOTICE, "[create_plan] %s", querystr);
#endif

/*
 * Prepare plan for query 
 */
	pplan = SPI_prepare(querystr, nargs, argtypes);
	if (pplan == NULL)
		elog(ERROR, "SPI_prepare returned %d", SPI_result);

/*
 * Remember that SPI_prepare places plan in current memory context
 * - so, we have to save plan in Top memory context for latter
 * use.
 */
	pplan = SPI_saveplan(pplan);
	if (pplan == NULL)
		elog(ERROR, "SPI_saveplan returned %d", SPI_result);

	(*qplan)->splan = (void *) malloc(sizeof(void *));
	(*qplan)->splan = pplan;	// saving plan

	return;
}


/**
 * indent option
 * -bad -nbap -cdb -br -nce -npcs -nbc -lp -fca -fc1 -i4 -ci4 -cli2 -d4 -ts4 -sc -cp1 -c4
 * 
 * [c.h]
 * typedef signed int int32;               // Size 32bits 4byte (int4)
 * 
 * [limits.h]
 * define INT_MAX 2147483647
 */
