#include "connection.h"
#include "statement.h"
#include "protocol.h"

#define QP_PREPARE         "PREPARE\""
#define QP_DECLARE         "DECLARE\""
#define QP_EXECUTE         "EXECUTE\""
#define QP_QUOTE           "\""
#define QP_QUOTE_AS        "\"AS "
#define QP_QUOTE_FOR       "\"CURSOR FOR "
#define QP_QUOTE_BRACKET   "\"("
#define QP_BRACKET_AS      ")AS "
#define QP_SPACE           "\0,"
#define QP_OBRACKET        "("
#define QP_CBRACKET        ")"

#define RESET_FETCH(stmt) { stmt->fetch_position = stmt->fetch_first = stmt->fetch_last = -1; pStatement->saved_status_ptr = NULL; }

Statement* AllocStatement(Connection* pConnection)
{
	Statement* pStatement = (Statement*)malloc(sizeof(Statement));
	if (pStatement)
	{
		memset(pStatement, '\0', sizeof(Statement));

		pStatement->connection = pConnection;
		pStatement->state = SS_ALLOCATED;

		pStatement->use_buffering = DBC_USE_BUFFERING(pConnection) ? TRUE : FALSE;
		/* results */
		pStatement->results.pHead     = NULL;
		pStatement->results.pTail     = NULL;
		pStatement->results.pLastHead = NULL;

		pStatement->results.unFreeSpaceInTailBlock = 0;
		
		/* Fetch */
		RESET_FETCH(pStatement);

		pStatement->need_data           = _T('\0');
		pStatement->data_at_exec.buffer = NULL;
		/* resultsets */
		pStatement->results.irds.allocated =  0;
		pStatement->results.irds.used      =  0;
		pStatement->results.current        = -1; /* use predefined resultset */

		/* default attributes */
		pStatement->attributes.no_scan         = SQL_NOSCAN_DEFAULT;
		pStatement->attributes.concurrency     = SQL_CONCUR_DEFAULT;
		pStatement->attributes.use_bookmarks   = SQL_UB_DEFAULT;
		pStatement->attributes.retrieve_data   = SQL_RD_DEFAULT;
		pStatement->attributes.enable_auto_ipd = SQL_FALSE;
		pStatement->attributes.bookmark_ptr    = NULL;
		pStatement->attributes.keyset_size     = 0;
		pStatement->attributes.max_length      = 0;

		/* attributes from parent Connection */
		pStatement->attributes.query_timeout   = pConnection->attributes.query_timeout;
		pStatement->attributes.metadata_id     = pConnection->attributes.metadata_id;
		pStatement->attributes.max_rows        = pConnection->attributes.max_rows;
		pStatement->attributes.cursor          = pConnection->attributes.cursor;

		/* query */
		pStatement->query.prepareType        = PT_SIMPLE_QUERY;
		pStatement->query.nParametersNumber  = 0;
		pStatement->query.execute_query      = NULL;
		pStatement->query.query              = NULL;
		pStatement->query.query_size         = 0;
		pStatement->query.execute_query_size = 0;

		/* prepare implicitly allocated descriptors */
		pStatement->d_apd.type = DT_APD;
		pStatement->d_ard.type = DT_ARD;
		pStatement->d_ipd.type = DT_IPD;
		pStatement->d_ird.type = DT_IRD;

		pStatement->d_apd.connection = pConnection;
		pStatement->d_ard.connection = pConnection;
		pStatement->d_ipd.connection = pConnection;
		pStatement->d_ird.connection = pConnection;

		InitDescriptor(pStatement->apd = &pStatement->d_apd, SQL_DESC_ALLOC_AUTO);
		InitDescriptor(pStatement->ard = &pStatement->d_ard, SQL_DESC_ALLOC_AUTO);
		InitDescriptor(pStatement->ipd = &pStatement->d_ipd, SQL_DESC_ALLOC_AUTO);
		InitDescriptor(pStatement->ird = &pStatement->d_ird, SQL_DESC_ALLOC_AUTO);

		/* add to connection's list */
		AddItem(&pConnection->statements, pStatement);

		/* autogenerate cursor name */
		SetCursorName(pStatement, NULL, 0, TRUE);
		pStatement->query.cursorState = CS_UNINITIALIZED;

		InitDiag(&pStatement->diag);
		_INIT_CRITICAL_SECTION(pStatement);
	}

	return pStatement;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ResetStatement
 *
 * DESCRIPTION: closes result and prepared statement if any
 *-----------------------------------------------------------------------------
 */
SQLRETURN ResetStatement(Statement* pStatement)
{
	Descriptor* pIRD;

	/* clear statement results */
	pStatement->need_data = _T('\0');
	EmptyStatementResults(pStatement);

	pIRD = GET_DESCRIPTOR(pStatement->ird);
	if (0 < pIRD->header.count)
		SetDescField(pIRD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLSMALLINT)0, SQL_IS_SMALLINT);
	RET_DESCRIPTOR(pIRD);

	/* close previously prepared statement if needed  */
	if (SQL_ERROR == CloseDeclared(pStatement, _T('s')))
		return SQL_ERROR;

	pStatement->use_buffering = DBC_USE_BUFFERING(pStatement->connection) ? TRUE : FALSE;
	pStatement->state         = SS_ALLOCATED;
	
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: BindParameter
 *
 * DESCRIPTION: No need to check ParameterNumber >= 1, Driver Manger does it.
 *-----------------------------------------------------------------------------
 */
SQLRETURN
BindParameter(Statement* pStatement, SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType,
              SQLSMALLINT ValueType, SQLSMALLINT ParameterType, SQLULEN ColumnSize,
              SQLSMALLINT DecimalDigits, SQLPOINTER ParameterValuePtr, SQLLEN BufferLength,
							SQLLEN* StrLen_or_IndPtr)
{
	SQLRETURN   nRet;
	SQLSMALLINT savedAPDCount;
	SQLSMALLINT savedIPDCount;

	Descriptor* pAPD;
	Descriptor* pIPD;

	/* get descriptors and save there COUNT field */
	pAPD = GET_DESCRIPTOR(pStatement->apd);
	pIPD = GET_DESCRIPTOR(pStatement->ipd);
	savedAPDCount = pAPD->header.count;
	savedIPDCount = pIPD->header.count;
	
	if (SQL_ERROR == ReallocDescriptorRecords(pAPD, ParameterNumber) ||
	    SQL_ERROR == ReallocDescriptorRecords(pIPD, ParameterNumber)
	   )
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
		nRet = SQL_ERROR;
	}
	else
	{
		AD_REC* pAD_REC;
		ID_REC* pID_REC;

		ParameterNumber--;
		nRet = SQL_SUCCESS;
		pAD_REC = &pAPD->ad_records[ParameterNumber];
		pID_REC = &pIPD->id_records[ParameterNumber];
	
		/* set APD's & IPD's requested parameter data */
		pAD_REC->bound               = SQL_TRUE;
		pAD_REC->common.octet_length = BufferLength;
		pAD_REC->common.data_ptr     = (SQLPOINTER*)ParameterValuePtr;
		pAD_REC->octet_length_ptr    = StrLen_or_IndPtr;
		pAD_REC->indicator_ptr       = StrLen_or_IndPtr;

		pID_REC->common.scale        = DecimalDigits;
		pID_REC->parameter_type      = InputOutputType;

		if (SQL_C_DEFAULT == ValueType && SQL_DEFAULT != ParameterType)
			ValueType = GetCDefaultType(ParameterType);

		TranslateType(&pAD_REC->common, ValueType,     DecimalDigits, 0,          C_TYPE);
		TranslateType(&pID_REC->common, ParameterType, DecimalDigits, ColumnSize, SQL_TYPE);
		SQLTypeDescriptor(ValueType, 0, &BufferLength, NULL, &pID_REC->common.scale, NULL, &pID_REC->display_size, NULL, NULL);

		if (SQL_NUMERIC == pAD_REC->common.type)
		{
			pID_REC->common.precision = 0; /* andyk change */
			pID_REC->common.scale = 0;
		}

		/* if an error occured - restore saved COUNT fields */
		if (SQL_ERROR == nRet)
		{
			pAPD->header.count = savedAPDCount;
			pIPD->header.count = savedIPDCount;
		}
	}
	/* end */
	RET_DESCRIPTOR(pAPD);
	RET_DESCRIPTOR(pIPD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FreeStatement
 *-----------------------------------------------------------------------------
 */
SQLRETURN FreeStatement(Statement* pStatement, SQLUSMALLINT Option)
{
	Descriptor* pDesc;

	switch (Option)
	{
		/* unbind all bound columns */
		case SQL_UNBIND:
			pDesc = GET_DESCRIPTOR(pStatement->ard);
			pDesc->header.count = 0;
			RET_DESCRIPTOR(pDesc);
			break;
		/* unbind all bound parameters */
		case SQL_RESET_PARAMS:
			pDesc = GET_DESCRIPTOR(pStatement->apd);
			pDesc->header.count = 0;
			RET_DESCRIPTOR(pDesc);
			break;
		/* close cursor on this statement */
		case SQL_CLOSE:
			EmptyStatementResults(pStatement);
			pStatement->state ^= SS_EXECUTED;

			/* we should commit transaction if it was opened only to process SELECT query
			 * when USEBUFFERING_PARAM = N
			 */
			if (STMT_USE_BUFFERING(pStatement) &&
			    0 != (SS_TRANSACTION_OPENED & pStatement->state)
			   )
				EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT, TO_DRIVER);
			break;
		default:
			ENTER_CRITICAL_SECTION(pStatement);
			EmptyStatementResults(pStatement);

			/* we should commit any transaction if it was opened by DRIVER */
			if (0 != (SS_TRANSACTION_OPENED & pStatement->state))
				EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT, TO_DRIVER);

			/* full close */
			RemoveItem(&pStatement->connection->statements, pStatement);
			/* free results */

			while (NULL != pStatement->results.pHead)
			{
				pStatement->results.pTail = pStatement->results.pHead;
				pStatement->results.pHead = pStatement->results.pHead->next;
				FreeBlock(pStatement->results.pTail);
			}

			FREE(pStatement->query.query);

			
			FreeDescriptor(pStatement->ird);
			FreeDescriptor(pStatement->ipd);
			FreeDescriptor(pStatement->ard);
			FreeDescriptor(pStatement->apd);

			FreeDiag(&pStatement->diag);
			_DELETE_CRITICAL_SECTION(pStatement);
			FREE(pStatement);
			return SQL_SUCCESS;
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * EmptyStatementResults
 *-----------------------------------------------------------------------------
 */
void EmptyStatementResults(Statement* pStatement)
{
	int i;
	Descriptor* pIRD;

	/* do not touch here bound parameters */
	if (NULL != pStatement->results.pHead)
	{
		*(int*)pStatement->results.pHead->data = FIELD_TERMINATOR;
		pStatement->results.pLastHead = pStatement->results.pTail = pStatement->results.pHead;
		pStatement->results.pLastHeadData = (int*) pStatement->results.pHead->data;
		pStatement->results.pTailField = (int*) pStatement->results.pTail->data;
		pStatement->results.unFreeSpaceInTailBlock = RESULTED_ROWS_BLOCK_SIZE;
	}

	pIRD = GET_DESCRIPTOR(pStatement->ird);
	if (NULL != pIRD->id_records && NULL != pIRD->id_records[0].common.data_ptr)
	{
		pIRD->header.array_size = 0;
		pIRD->header.bookmarks  = 0;
		FREE(pIRD->id_records[0].common.data_ptr);
		pIRD->id_records[0].common.data_ptr = NULL;
	}

	pIRD->bookmark.delete_query = NULL;
	pIRD->bookmark.insert_query = NULL;
	pIRD->bookmark.update_query = NULL;

	RET_DESCRIPTOR(pIRD);

	for (i=0;i<pStatement->results.irds.used;i++)
		FreeDescriptor((Descriptor*)pStatement->results.irds.handles[i]);
	
	pStatement->ird->header.rows_affected =  0;
	pStatement->results.irds.used  =  0;
	pStatement->results.current    = -1;
	RESET_FETCH(pStatement);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareStatement
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareStatement(Statement* pStatement, SQLTCHAR* StatementText, SQLINTEGER TextLength)
{
	SQLRETURN nRet;

	if (0 != (SS_ALLOCATED & pStatement->state) && SQL_SUCCEEDED(nRet = PrepareQuery(pStatement, StatementText, TextLength, QO_SERVER_SIDE_PREPARE)))
	{
		return DeclareStatement(pStatement, TRUE);
	}
	else
		nRet = SQL_ERROR;
	
	/* andyk wrong statement's state */
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PopulateID
 *
 * DESCRIPTION: converts information about columns, retrived from backend into
 *              data, usefull for application
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PopulateID(Descriptor* pDescriptor, SQLINTEGER odbc_version, BOOL mapp)
{
	int i;

	/* populate bookmark */
	pDescriptor->bookmark.type = SQL_BINARY;

	/* fill implementation descriptor fields with data */
	for (i=0;i<pDescriptor->header.count;i++)
	{
		TCHAR*      name;
		SQLINTEGER  length;
		SQLSMALLINT precision;

		ID_REC*     pID_REC = &pDescriptor->id_records[i];

		pID_REC->unnamed        = SQL_UNNAMED;
		pID_REC->is_unsigned    = SQL_FALSE;
		pID_REC->nullable       = SQL_NULLABLE_UNKNOWN;
		pID_REC->parameter_type = SQL_PARAM_INPUT;
		pID_REC->common.scale   = 0;

		pID_REC->type_name = NULL;
		
		/* translate type retrieved from PostgreSQL backend into SQL type */
		PostgreTypeToSQLType(pID_REC->type_oid, pID_REC->type_modifier, odbc_version, &name, &pID_REC->common.concise_type, &length, &precision, mapp);
		/* set all descriptor's fields */
		DescribeSQLType(pID_REC->common.concise_type, length, precision, &pID_REC->common.type, (SQLINTEGER*)&pID_REC->common.length, &pID_REC->common.num_prec_radix, &pID_REC->display_size);
	}

	return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: UpdateParameters
 *
 * RETURN: 
 *-----------------------------------------------------------------------------
 */
static SQLRETURN
UpdateParameters(Statement* pStatement, Descriptor* pAPD, int* pParametersSize)
{
	SQLRETURN   nRet;
	int         i;
	int         length = 0;
	Descriptor* pIPD = GET_DESCRIPTOR(pStatement->ipd);


	if (SQL_ERROR != (nRet = ReallocDescriptorRecords(pIPD, pStatement->query.nParametersNumber)))
	{
		for (i=0;i<pStatement->query.nParametersNumber;i++)
		{/* convert parameters from SQL_C_TYPE into backend's specific format(text) */
			CD_REC* common = &pAPD->ad_records[i].common;

			if (NULL == (pIPD->id_records[i].common.data_ptr = (SQLPOINTER*)PrepareParameter(pStatement, (0 <= pAPD->ad_records[i].data_at_exec_indicator) ? (SQLPOINTER) pAPD->ad_records[i].data_at_exec : (SQLPOINTER) common->data_ptr, (0 <= pAPD->ad_records[i].data_at_exec_indicator) ? pAPD->ad_records[i].data_at_exec_indicator : common->octet_length, (SQLSMALLINT)((SQL_C_DEFAULT == common->concise_type) ? GetCDefaultType(pIPD->id_records[i].common.concise_type) : common->concise_type), pAPD->ad_records[i].indicator_ptr, pIPD->id_records[i].common.concise_type, common->scale)))
			{/* no more memory */
				nRet = SQL_ERROR;
				break;
			}
			else
			{
				SQLINTEGER temp_length = SQL_NTS;
				SQLTCHAR*  temp_name   = (SQLTCHAR*)(pIPD->id_records[i].common.data_ptr + sizeof(int)/sizeof(void**));
				ReplaceEscapes(pStatement, &temp_name, *(int*)pIPD->id_records[i].common.data_ptr, &temp_length, FALSE, NULL, NULL, TRUE);
				length += *((int*)pIPD->id_records[i].common.data_ptr);
			}
		}
		pStatement->results.pLastHeadData = pStatement->results.pTailField;
	}

	if (NULL != pParametersSize)
		*pParametersSize = length;

	RET_DESCRIPTOR(pIPD);
	return nRet;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: AllParametersBound
 *
 * RETURN: SQL_SUCCESS - no parameters, or no need to recend parameters, or some
 *                       parameters were changed and rebind to backend successfully
 *                       completed
 *-----------------------------------------------------------------------------
 */
SQLRETURN AllParametersBound(Statement* pStatement, TCHAR** pExecuteQuery)
{
	int         i     = 0;
	int         nParametersSize = 0;
	SQLRETURN   nRet  = SQL_SUCCESS;
	SQLRETURN   nRet1 = SQL_SUCCESS;
	Descriptor* pAPD  = GET_DESCRIPTOR(pStatement->apd);

	/* -- check - 'all parameters bound to the driver by the application?' -- */
	if (pAPD->header.count >= pStatement->query.nParametersNumber)
		for (;i<pStatement->query.nParametersNumber;i++)
			if ((NULL == pAPD->ad_records[i].common.data_ptr) &&
			    (NULL == pAPD->ad_records[i].indicator_ptr || (SQL_NULL_DATA != *pAPD->ad_records[i].indicator_ptr &&  SQL_DATA_AT_EXEC != *pAPD->ad_records[i].indicator_ptr))
			   )
				break;

	if (i < pStatement->query.nParametersNumber)
	{/* not all parameters were bound to the driver by the application */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_NOT_ENOUGH_PARAMETERS, NULL);
		RET_DESCRIPTOR(pAPD);
		return SQL_ERROR;
	}

	if (0 < pStatement->query.nParametersNumber)
		SET_PORTAL_NEEDS_REDECLARATION(pStatement);
	
	/* check parameters for SQL_DATA_AT_EXEC */
	for (i--;i>=0;i--)
		if IS_DATA_AT_EXEC(pAPD, i)
		{/* there is still some unprepared parameters */
			pStatement->need_data = _T('p');
			pStatement->data_at_exec.index = i;

			RET_DESCRIPTOR(pAPD);
			return SQL_NEED_DATA;
		}


	switch(pStatement->query.prepareType)
	{
		case PT_SIMPLE_QUERY:
		case PT_SERVER_SIDE:
			/* bind here means - remake execute query */
			
			/* if we have no parameters - we shouldn't modify execute_query, */
			if (0 < pStatement->query.nParametersNumber)
			{
				nRet = UpdateParameters(pStatement, pAPD, &nParametersSize);
				if (SQL_SUCCEEDED(nRet) && (0 < nParametersSize))
				{
					TCHAR* source = (PT_SIMPLE_QUERY == pStatement->query.prepareType) ? pStatement->query.query : pStatement->query.execute_query;
					TCHAR* parameter;
					TCHAR* dest_query;
					TCHAR* destination;

					int i;
					int length = (PT_SIMPLE_QUERY == pStatement->query.prepareType) ? pStatement->query.query_size : pStatement->query.execute_query_size;
					int parameter_length;

					ID_REC*     pID_REC;
					Descriptor* pIPD = GET_DESCRIPTOR(pStatement->ipd);

					pID_REC = pIPD->id_records;

					/* calculate query size for parameters */
					for (i = 0; i < pStatement->query.nParametersNumber; ++i)
					{
						parameter_length = *(int*) pID_REC[i].common.data_ptr;
						if (FIELD_NULL == parameter_length)
							length += STR_SIZEOF("NULL") - 1;
						else
							length += parameter_length;
					}

					/* allocate space */
					destination = dest_query = (TCHAR*) malloc (length * sizeof(TCHAR));

					/* place parameters */
					for (i = 0; i < pStatement->query.nParametersNumber; ++i)
					{
						parameter        = (TCHAR*) ((int*)pID_REC[i].common.data_ptr + 1);
						parameter_length = _tcslen(parameter);

						/* before part */
						length = _tcslen(source);
						_tcsncpy(destination, source, length);
						destination += length;
						source      += length;

						/* check - is this a NULL? */
						if (FIELD_NULL == *(int*)pID_REC[i].common.data_ptr)
						{
							_tcsncpy(destination, _T("NULL"), STR_SIZEOF("NULL"));
							destination += STR_SIZEOF("NULL");						
						}
						else
						{ /* open ' */
							*destination++ = _T('\'');
							/* parameter */
							_tcsncpy(destination, parameter, parameter_length);
							destination += parameter_length;
							/* close ' */
							*destination++ = _T('\'');
						}
						++source;
					}

					_tcscpy(destination, source);

					RET_DESCRIPTOR(pIPD);

					*pExecuteQuery = dest_query;
				}
			}
			break;
		case PT_FR_BK_PROTOCOL:
			/* bind by declaring new portal */
			/* -- check - 'should we resend query text to parse (it changed) to the backend?' -- */
			if (STATEMENT_NEEDS_REDECLARATION(pStatement))
				nRet = DeclareStatement(pStatement, FALSE);
		
			/* -- check - 'should we resend parameters to the backend?' -- */
			if (SQL_SUCCEEDED(nRet) && PORTAL_NEEDS_REDECLARATION(pStatement))
			{/* we need to reprepare portal because of new query text or parameter values */
				nRet1 = UpdateParameters(pStatement, pAPD, &nParametersSize);

				if (SQL_ERROR == nRet1)
					SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
				else if (PT_FR_BK_PROTOCOL == pStatement->query.prepareType)
					nRet1 = DeclarePortal(pStatement);

				nRet = MAX_ERROR(nRet, nRet1);
			}
			break;
		default:
			/* error - we shouldn't get here */
			break;
	}

	if (!SQL_SUCCEEDED(nRet))
		/* can't bind parameters to the server */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CANT_BIND_PARAMETERS, NULL);

	RET_DESCRIPTOR(pAPD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ExecuteStatement
 *-----------------------------------------------------------------------------
 */
SQLRETURN ExecuteStatement(Statement* pStatement, BOOL ForceBuffering)
{
	TCHAR* execute_query = NULL;

	/* first of all - control transaction state */
	SQLRETURN nRet = (AUTOCOMMIT_MODE_OFF(pStatement)) ? BeginTransaction(pStatement, TO_APPLICATION) : SQL_SUCCESS;

	pStatement->use_buffering |= ForceBuffering;
	/* check for the corresponding parameters bound by the application */
	if (SQL_SUCCEEDED(nRet) &&
	    SQL_SUCCEEDED(nRet = AllParametersBound(pStatement, &execute_query))
	   )
	{
		switch (pStatement->query.prepareType)
		{
			case PT_SERVER_SIDE:
				/* send PREPARE query to the backend if this is the fist Execute */
				if (NULL != pStatement->query.query)
				{
					TCHAR  prepare_buffer[MAX_QUERY_LENGTH + 1];
					TCHAR* prepare_query;
					TCHAR* prepare_ptr;

					if (0 < pStatement->query.nParametersNumber)
					{
						TCHAR* source_ptr = pStatement->query.query;
						Descriptor* pIPD = pStatement->ipd;
						Descriptor* pAPD = pStatement->apd;
						int    total_length = 1;
						int    length;
						int    i, j;

						/* calculate required size */
						length = _tcslen(source_ptr);
						total_length += length;
						source_ptr += length;

						for (i=0;i<pStatement->query.nParametersNumber;++i)
						{
							SQLSMALLINT type = (SQL_DEFAULT == pIPD->id_records[i].common.concise_type) ? GetDefaultType(pAPD->ad_records[i].common.concise_type) : pIPD->id_records[i].common.concise_type;

							for (j=0;j<sizeof(c_PostgreSQLDataTypes)/sizeof(PostgreSQLDataType);++j)
							{
								if (c_PostgreSQLDataTypes[j].sql_type == type)
								{
									length = _tcslen(c_PostgreSQLDataTypes[j].local_name);
									total_length += length;
									++source_ptr;
									length = _tcslen(source_ptr);
									total_length += length;
									source_ptr += length;
									break;
								}
							}
						}

						prepare_query = (MAX_QUERY_LENGTH < total_length) ? (TCHAR*)malloc(total_length * sizeof(TCHAR)) : prepare_buffer;

						prepare_ptr = prepare_query;
						source_ptr = pStatement->query.query;

						/* get all types for prepare from bound data */
						length = _tcslen(source_ptr);
						_tcsncpy(prepare_ptr, source_ptr, length);
						prepare_ptr += length;
						source_ptr += length;

						for (i=0;i<pStatement->query.nParametersNumber;++i)
						{
							SQLSMALLINT type = (SQL_DEFAULT == pIPD->id_records[i].common.concise_type) ? GetDefaultType(pAPD->ad_records[i].common.concise_type) : pIPD->id_records[i].common.concise_type;

							for (j=0;j<sizeof(c_PostgreSQLDataTypes)/sizeof(PostgreSQLDataType);++j)
							{
								if (c_PostgreSQLDataTypes[j].sql_type == type)
								{
									length = _tcslen(c_PostgreSQLDataTypes[j].local_name);
									_tcsncpy(prepare_ptr, c_PostgreSQLDataTypes[j].local_name, length);
									prepare_ptr += length;
									++source_ptr;
									length = _tcslen(source_ptr);
									_tcsncpy(prepare_ptr, source_ptr, length);
									prepare_ptr += length;
									source_ptr += length;
									break;
								}
							}
							*prepare_ptr = _T('\0');
						}

						/* execute PREPARE query */
						nRet = (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, prepare_query)
									 ) ? WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement) : SQL_ERROR;
						FREE(pStatement->query.query);

						if (prepare_buffer != prepare_query)
							FREE(prepare_query);

						if (!SQL_SUCCEEDED(nRet))
							break;
					}					
				}

				if (NULL == execute_query)
					execute_query = pStatement->query.execute_query;

				/* simply call prepared sql_execute query - if we are not using buffering results */
				if (STMT_USE_BUFFERING(pStatement) || (ST_SELECT != pStatement->query.type))
				{
					nRet = (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, execute_query)
								 ) ? WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement) : SQL_ERROR;
				}
				break;
			case PT_SERVER_SIDE_CURSOR:
			case PT_SIMPLE_QUERY:
				if (0 == pStatement->query.nParametersNumber)
					execute_query = pStatement->query.query;
				/* no break! */
			case PT_FR_BK_PROTOCOL:
			default:
				if (PT_SERVER_SIDE_CURSOR == pStatement->query.prepareType)
				{
					BeginTransaction(pStatement, TO_DRIVER);
					nRet = (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, execute_query) &&
					        SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Describe, pStatement) &&
					        SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL)
								 ) ? WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement) : SQL_ERROR;

					SET_STATEMENT_DECLARED(pStatement);
				}
				else
				{
					if (PORTAL_DECLARED(pStatement))
					{/* execute prepared portal */
						nRet = ((SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Describe, pStatement)) &&
										(SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, pStatement)) &&
										(SQL_SUCCESS == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement)) &&
						        (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Execute, pStatement)) &&
						        (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL))
									 ) ? WaitForBackendReply(pStatement->connection,  MSG_ReadyForQuery, pStatement) : SQL_ERROR;
					}
					else
					{/* use simple query protocol to get resultset */
						nRet = (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, execute_query)
									 ) ? WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement) : SQL_ERROR;
					}
				}

				break;
			/* nothing to declare */
			break;
		}

		if (execute_query != pStatement->query.query && execute_query != pStatement->query.execute_query)
			FREE(execute_query);

/*
		if (SQL_SUCCESS != nRet)
			SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CANT_EXECUTE, NULL);
*/
	}

	/* close transaction if needed */
	if ((STMT_USE_BUFFERING(pStatement) || (ST_DECLARE_CURSOR == pStatement->ird->header.query_type)) && (AUTOCOMMIT_MODE_ON(pStatement)))
	{
		SQLRETURN nRet1 = EndTransaction(SQL_HANDLE_STMT, pStatement, (SQLSMALLINT)((SQL_SUCCESS == nRet) ? SQL_COMMIT : SQL_ROLLBACK), TO_DRIVER);
		nRet = MAX_ERROR(nRet, nRet1);
	}

	return nRet;
}

/*-----------------------------------------------------------------------------
 * CreateBlock
 *-----------------------------------------------------------------------------
 */
Block* AllocBlock(Block* prev, int size)
{
	Block* pBlock;
	unsigned int resulted = 0;
	if (NULL != (pBlock = (Block*) malloc((resulted = (RESULTED_ROWS_BLOCK_SIZE < size) ? size : RESULTED_ROWS_BLOCK_SIZE)+sizeof(Block)-RESULTED_ROWS_BLOCK_SIZE)))
	{
		pBlock->next = NULL;
		pBlock->prev = prev;
		pBlock->size = resulted;
		if (NULL != prev)
			prev->next = pBlock;
		*(int*) pBlock->data = 0;     /* no more fields avaible */
	}

	return pBlock;
}

/*-----------------------------------------------------------------------------
 * FreeBlock
 *-----------------------------------------------------------------------------
 */
void FreeBlock(Block* block)
{
	FREE(block);
}

/*-----------------------------------------------------------------------------
 * AddField
 *-----------------------------------------------------------------------------
 */
TCHAR* AddField(Statement* pStatement, unsigned int length)
{
	TCHAR* pRet;
	int    resulted_length = length*sizeof(TCHAR);

	if (NULL == pStatement->results.pTail)
	{
		pStatement->results.pTail = AllocBlock(NULL, resulted_length + 2*sizeof(int));
		pStatement->results.pHead = pStatement->results.pTail;
		pStatement->results.pLastHead = pStatement->results.pTail;
		pStatement->results.pTailField = (int*) pStatement->results.pTail->data;
		pStatement->results.unFreeSpaceInTailBlock = pStatement->results.pTail->size;
		pStatement->results.pLastHeadData = (int*) pStatement->results.pLastHead->data;
	}

	if (pStatement->results.unFreeSpaceInTailBlock < resulted_length + 2*sizeof(int))
	{
		/* add new block*/
		*pStatement->results.pTailField = FIELD_IN_NEXT_BLOCK;
		if (NULL == pStatement->results.pTail->next)
			pStatement->results.pTail = AllocBlock(pStatement->results.pTail, resulted_length + 2*sizeof(int));
		else
			pStatement->results.pTail = pStatement->results.pTail->next;
		pStatement->results.pTailField = (int*) pStatement->results.pTail->data;
		pStatement->results.unFreeSpaceInTailBlock = pStatement->results.pTail->size;
	}

	*pStatement->results.pTailField = length;
	pStatement->results.pTailField += 1;
	pRet = pStatement->results.pTailField_chr;
	pStatement->results.pTailField_chr += length;
	*pStatement->results.pTailField = FIELD_TERMINATOR;
	pStatement->results.unFreeSpaceInTailBlock -= length*sizeof(TCHAR) + sizeof(int);
	return pRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: Fetch
 *
 * DESCRIPTION: No need to check pStatement's state, Driver Manager does it.
 *              
 *-----------------------------------------------------------------------------
 */
SQLRETURN
Fetch(Statement* pStatement, SQLSMALLINT FetchOrientation, SQLLEN FetchOffset, SQLULEN* pcrow, SQLUSMALLINT* rgfRowStatus, SQLINTEGER called_from)
{
	SQLUINTEGER i = 0;
	SQLSMALLINT fill_bookmark = SQL_TRUE;

	/* check for cursor and orientation consistensy */
	if ((SQL_API_SQLEXTENDEDFETCH != called_from) &&
	    ((SQL_CURSOR_STATIC == pStatement->attributes.cursor) &&
	     (!STMT_USE_BUFFERING(pStatement)) &&
	     (((SQL_FETCH_FIRST    == FetchOrientation) && (-1 < pStatement->fetch_position)) ||
	      ((SQL_FETCH_ABSOLUTE == FetchOrientation) && (FetchOffset <= pStatement->fetch_position)) ||
	      ((SQL_FETCH_RELATIVE == FetchOrientation) && (FetchOffset < 0)) ||
	      (SQL_FETCH_BOOKMARK == FetchOrientation) ||
	      (SQL_FETCH_PRIOR    == FetchOrientation)
			 )
      )/* ||

	    ((SQL_CURSOR_KEYSET_DRIVEN == pStatement->attributes.cursor)) ||
	    ((SQL_CURSOR_FORWARD_ONLY  == pStatement->attributes.cursor)) ||
	    ((SQL_CURSOR_DYNAMIC       == pStatement->attributes.cursor)) */
	   )
	{/* bad orientation requested */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_FETCH_IMPOSSIBLE_ORIENTATION, NULL);
		return SQL_ERROR;
	}

	/* check buffering 'OFF' */
	if (!STMT_USE_BUFFERING(pStatement) && (-1 != pStatement->fetch_last))
	{
		/* empty results buffer */
		EmptyStatementResults(pStatement);
		/* get portal's row */
		if (SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Execute, pStatement) ||
		    SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL) ||
		    SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement)
			 )
			return SQL_ERROR;
	}

	pStatement->fetch_last = pStatement->ird->header.rows_affected;
	/* set cursor position */
	switch(FetchOrientation)
	{
		case SQL_FETCH_FIRST:        /* 2 */
			pStatement->fetch_position = 0;
			break;
		case SQL_FETCH_LAST:
			pStatement->fetch_position = pStatement->fetch_last-1;
			break;
		case SQL_FETCH_PRIOR:
			pStatement->fetch_position = (0 < pStatement->fetch_position) ?	pStatement->fetch_position-1 : -1;
			break;
		case SQL_FETCH_NEXT:         /* 1 */
			pStatement->fetch_position = (pStatement->fetch_position < pStatement->fetch_last-1) ? pStatement->fetch_position+1 : pStatement->fetch_last+1;
			break;
		case SQL_FETCH_RELATIVE:
			pStatement->fetch_position = 1;
			break;
		case SQL_FETCH_ABSOLUTE:
			if (0 == FetchOffset)
				pStatement->fetch_position = -1;
			else if (FetchOffset < 0)
			{
				if ((-FetchOffset) <= pStatement->fetch_last)
				{
					pStatement->fetch_position = pStatement->fetch_last + FetchOffset;
				}
				else
				{
					if ((-FetchOffset) <= (SQLINTEGER)pStatement->ard->header.array_size)
						pStatement->fetch_position = 0;					
					else
						pStatement->fetch_position = -1;
				}
			}
			else if (FetchOffset <= pStatement->fetch_last)
				pStatement->fetch_position = FetchOffset - 1;
			else
				pStatement->fetch_position = pStatement->fetch_last + 1;
			break;
		case SQL_FETCH_BOOKMARK:
			{
				SQLLEN row = -1;
				if (SQL_API_SQLEXTENDEDFETCH == called_from)
				{
					row = FetchOffset;
					FetchOffset = 0;
				}
				else if (NULL == pStatement->attributes.bookmark_ptr)
				{/* bad pointer to array of bookmarks */
					SetError(SQL_HANDLE_STMT, pStatement, ERR_FETCH_INVALID_BOOKMARK_VALUE, NULL);
					return SQL_ERROR;
				}
				else
				{
					row = FindRow(pStatement->ird, (SQLCHAR*)pStatement->attributes.bookmark_ptr, pStatement->ard->bookmark.type);
				}

				if (0 <= row)
				{/* bookmark exists - check FetchOffset */
					fill_bookmark = SQL_FALSE;
					row += FetchOffset;
					if (row < 0)
						pStatement->fetch_position = -1;
					else if (row < pStatement->fetch_last)
						pStatement->fetch_position = row;
					else
						pStatement->fetch_position = pStatement->fetch_last + 1;
				}
				else
					return SQL_ERROR;
			}
	}

	if (0 > pStatement->fetch_position)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_FETCH_NO_ROW_IN_RESULTSET, NULL);
		return SQL_ERROR;
	}
	else
	{
		Descriptor* pIRD;
		Descriptor* pARD;

		pIRD = GET_DESCRIPTOR(pStatement->ird);
		pARD = GET_DESCRIPTOR(pStatement->ard);

		/* fill bound columns */
		for (i=0;i<pARD->header.array_size;i++)
		{
			if (pStatement->fetch_last <= pStatement->fetch_position)
			{
				if (!STMT_USE_BUFFERING(pStatement) && AUTOCOMMIT_MODE_ON(pStatement))
					EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT, TO_DRIVER);
				break;
			}
			else
				FillBoundColumns(pARD, pIRD, i, pStatement->fetch_position++, fill_bookmark);
		}

		if (pIRD->header.rows_processed_ptr)
			*pIRD->header.rows_processed_ptr = i;
		
		RET_DESCRIPTOR(pIRD);
		RET_DESCRIPTOR(pARD);

		pStatement->fetch_position--;

		if (SQL_API_SQLEXTENDEDFETCH == called_from)
		{
			if (NULL != pcrow)
				*pcrow = i;

			if (NULL != rgfRowStatus)
			{
				unsigned int j;

				for (j=0;j<i;++j)
					rgfRowStatus[j] = SQL_ROW_SUCCESS;

				pStatement->saved_status_ptr = rgfRowStatus;
			}
		}

	}
	return (0 == i) ? SQL_NO_DATA : SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * PrepareCursor
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareCursor(Statement* pStatement)
{
	TCHAR* pPreparedQuery;

	/* check was the cursor declared in current transaction */
	if (0 != ((CS_DECLARED | CS_ADDED_TO_QUERY) & pStatement->query.cursorState))
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CURSOR_DECLARED, pStatement->query.cursorName, NULL);
		return SQL_ERROR;
	}
	/* create unique cursor name if needed */
	SetCursorName(pStatement, NULL, 0, TRUE);
	/* compile new SELECT statement - with cursor declaration */
	pPreparedQuery = GetText(_T("DECLARE ? CURSOR FOR ?"), pStatement->query.cursorName, pStatement->query.query, NULL);
	FREE(pStatement->query.query);
	pStatement->query.query = pPreparedQuery;
	pStatement->query.cursorState = CS_ADDED_TO_QUERY;

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * SetCursorName
 *-----------------------------------------------------------------------------
 */
SQLRETURN
SetCursorName(Statement* pStatement, TCHAR* CursorName, SQLSMALLINT NameLength, BOOL bAutoGenerate)
{
	int i;

	/* create unique cursor name if there is no valid name and Auto Generate specified */
	if (TRUE == bAutoGenerate)
	{
		if (_T('\0') == pStatement->query.cursorName[0])
		{
			_tcsncpy(pStatement->query.cursorName, _T("SQL_CUR"), STR_SIZEOF("SQL_CUR"));
			_itot(++pStatement->connection->cursor_count, pStatement->query.cursorName + STR_SIZEOF("SQL_CUR"), 10 /* space economy */);
		}
		return SQL_SUCCESS;
	}

	/* check cursor's presence */
	if (CS_DECLARED == pStatement->query.cursorState)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CURSOR_ALREADY_DECLARED, pStatement->query.cursorName, NULL);
		return SQL_ERROR;
	}

	if (SQL_NTS == NameLength)
		NameLength = _tcslen(CursorName);

	/* check cursor name validity */
	if (	(MAX_CURSOR_NAME_LENGTH < NameLength)
	      ||(STR_SIZEOF("SQLCUR") <= NameLength && (0 == _tcsncmp(CursorName, _T("SQLCUR"),  STR_SIZEOF("SQLCUR"))))
	      ||(0 == _tcsncmp(CursorName, _T("SQL_CUR"), STR_SIZEOF("SQL_CUR")))
	   )
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_INVALID_CURSOR_NAME, NULL);
		return SQL_ERROR;
	}

	/* check cursor name originality within connection */
	for (i=pStatement->connection->statements.used-1;i>=0;i--)
	{
		if (0 == _tcsncmp(((Statement*)pStatement->connection->statements.handles[i])->query.cursorName, CursorName, NameLength) &&
			  NameLength == (SQLSMALLINT) _tcslen(((Statement*)pStatement->connection->statements.handles[i])->query.cursorName)
		   )
		{
			SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CURSOR_NAME_IN_USE, NULL);
			return SQL_ERROR;
		}
	}

	/* set new name */
	_tcsncpy(pStatement->query.cursorName, CursorName, NameLength);
	pStatement->query.cursorName[NameLength] = _T('\0');
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: CheckQuery
 *-----------------------------------------------------------------------------
 */
SQLRETURN
CheckQuery(TCHAR* begin, SQLINTEGER length, Restrictions* restrictions)
{
	POSITION   position    = POS_OUTSIDE_TEXT;
	BOOL       inside_from = FALSE;

	/* for every FROM-clause apply restrictions */
	for (;0<length;++begin,--length)
	{
		TCHAR* begining;
		TCHAR* ending;

		switch (*begin)
		{
			/* end of FROM-clause */
			case _T('w'): /* WHERE */
			case _T('W'):
				if ((STR_SIZEOF("WHERE")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("WHERE"), STR_SIZEOF("WHERE"))))
				{
					begin   += STR_SIZEOF("WHERE")-1;
					length  -= STR_SIZEOF("WHERE")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T('g'): /* GROUP BY */
			case _T('G'):
				if ((STR_SIZEOF("GROUP")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("GROUP"), STR_SIZEOF("GROUP"))))
				{
					begin   += STR_SIZEOF("GROUP")-1;
					length  -= STR_SIZEOF("GROUP")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T('h'): /* HAVING */
			case _T('H'):
				if ((STR_SIZEOF("HAVING")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("HAVING"), STR_SIZEOF("HAVING"))))
				{
					begin   += STR_SIZEOF("HAVING")-1;
					length  -= STR_SIZEOF("HAVING")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T('l'): /* LIMIT */
			case _T('L'):
				if ((STR_SIZEOF("LIMIT")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("LIMIT"), STR_SIZEOF("LIMIT"))))
				{
					begin   += STR_SIZEOF("LIMIT")-1;
					length  -= STR_SIZEOF("LIMIT")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T('o'): /* ORDER BY, OFFSET */
			case _T('O'):
				if ((STR_SIZEOF("ORDER")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("ORDER"), STR_SIZEOF("ORDER"))))
				{
					begin   += STR_SIZEOF("ORDER")-1;
					length  -= STR_SIZEOF("ORDER")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else if ((STR_SIZEOF("OFFSET")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("OFFSET"), STR_SIZEOF("OFFSET"))))
				{
					begin   += STR_SIZEOF("OFFSET")-1;
					length  -= STR_SIZEOF("OFFSET")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T('f'): /* FOR */
			case _T('F'):
				if ((STR_SIZEOF("FROM")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("FROM"), STR_SIZEOF("FROM"))))
				{
					begin   += STR_SIZEOF("FROM")-1;
					length  -= STR_SIZEOF("FROM")-1;
					position = POS_OUTSIDE_FROM;
				}
				else if ((STR_SIZEOF("FOR")-1 <= length) && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("FOR"), STR_SIZEOF("FOR"))))
				{
					begin   += STR_SIZEOF("FOR")-1;
					length  -= STR_SIZEOF("FOR")-1;
					position = POS_OUTSIDE_FROM_CLAUSE;		
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T('j'):
			case _T('J'):
				if ((STR_SIZEOF("JOIN")-1 <= length) && inside_from && (POS_OUTSIDE_READY == position) && (0 == _tcsnicmp(begin, _T("JOIN"), STR_SIZEOF("JOIN"))))
				{
					begin   += STR_SIZEOF("JOIN")-1;
					length  -= STR_SIZEOF("JOIN")-1;
					position = POS_OUTSIDE_FROM;
				}
				else
				{
					position = POS_OUTSIDE_TEXT;
				}
				break;
			case _T(','):
				if (inside_from && (POS_OUTSIDE_READY == position || POS_OUTSIDE_TEXT == position))
				{
					position = POS_ITEM_LOCATED;
					--length;
					++begin;
				}
				else if (POS_OUTSIDE_FROM_CLAUSE==position)
					return SQL_SUCCESS;
				break;
			case _T('\''):
				switch (position)
				{
					case POS_OUTSIDE_FROM_CLAUSE:
						return SQL_SUCCESS;
					case POS_INSIDE_VAR:
						position = POS_OUTSIDE_READY;
						break;
					case POS_INSIDE_SYS:
						break;
					default:
						position = POS_INSIDE_VAR;
				}
				break;
			case _T('('):
				switch (position)
				{
					case POS_OUTSIDE_FROM_CLAUSE:
						return SQL_SUCCESS;
					case POS_OUTSIDE_FROM:
						position = POS_ITEM_LOCATED;
						break;
					default:
						;
				}
				break;
			case _T('"'):
				switch (position)
				{
					case POS_OUTSIDE_FROM_CLAUSE:
						return SQL_SUCCESS;
					case POS_OUTSIDE_FROM:
						position = POS_ITEM_LOCATED;
						break;
					case POS_INSIDE_SYS:
						position = POS_OUTSIDE_READY;
						break;
					case POS_INSIDE_VAR:
						break;
					default:
						position = POS_INSIDE_SYS;
				}
				break;
			case _T(' '):
			case _T('\t'):
				switch (position)
				{
					case POS_OUTSIDE_FROM_CLAUSE:
						return SQL_SUCCESS;
					case POS_OUTSIDE_FROM:
						position = POS_ITEM_LOCATED;
						break;
					case POS_OUTSIDE_TEXT:
						position = POS_OUTSIDE_READY;
					default:
						;
				}
				break;
			default:
				;
		}

		if (POS_ITEM_LOCATED == position)
		{/* process FROM-clause by applying restrictions,
			* here can be more than one item - parse them all
			*/
			while (0<length && _istspace(*begin))
			{
				--length;
				++begin;
			}
				
			if (_T('(') == *begin)
			{
				TCHAR*     ptr        = ++begin;
				SQLINTEGER sub_length = 0;
				SQLINTEGER counter    = 1;
				for (--length;(0<length && counter != 0);++begin,--length,++sub_length)
				{
					if (_T(')') == *begin)
						--counter;
					else if (_T('(') == *begin)
						++counter;
				}

				if (!SQL_SUCCEEDED(CheckQuery(ptr, sub_length-1, restrictions)))
					return SQL_ERROR;
				position = POS_OUTSIDE_READY;
			}
			else
			{


			/* possible 'only' word presets */
			if ((STR_SIZEOF("ONLY")-1 <= length) && (0 == _tcsnicmp(begin, _T("ONLY"), STR_SIZEOF("ONLY"))))
			{
				begin  += STR_SIZEOF("ONLY")-1;
				length -= STR_SIZEOF("ONLY")-1;
		
				while (0<length && _istspace(*begin))
				{
					--length;
					++begin;
				}
			}

			/* this must be the table name (or function name) */
			begining = begin;

			while (TRUE)
			{
				TCHAR* temp;

				while (0<length && _istspace(*begin))
				{
					--length;
					++begin;
				}

				if (_T('"') == *begin)
				{
					for(--length,++begin;0<length;--length,++begin)
						if (_T('"') == *begin)
						{
							++begin;
							--length;
							break;
						}
				}
				else
					for(;(0<length && _T('.') != *begin && _T(',') != *begin && !_istspace(*begin));--length,++begin);

				temp = begin;

				while (0<length && _istspace(*begin))
				{
					--length;
					++begin;
				}

				if (_T('(') == *begin)
				{/* this is function_name - ignore it */
					SQLINTEGER counter = 1;
					++begin;
					for (--length;(0<length && counter != 0);++begin,--length)
					{
						if (_T(')') == *begin)
							--counter;
						else if (_T('(') == *begin)
							++counter;
					}
					begining = NULL;
					break;
				}
				else if (_T('.') != *begin)
				{
					begin = temp;
					break; /* while */
				}
				else
				{
					++begin;
					--length;
				}
			}
			ending = begin;

			if ((NULL != begining) && (SQL_SUCCESS != CheckPatterns(begining, ending, restrictions)))
				return SQL_ERROR;

			position    = POS_OUTSIDE_READY;
			inside_from = TRUE;

			/* here can be only some type of 'JOIN' or ',' */
			while (0<length && _istspace(*begin))
			{
				--length;
				++begin;
			}

			++length;
			--begin;
		}	
			}
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetParametersLength
 *
 * DESCRIPTION: calculates the size of the prepared query text, it depends on
 *              the number of parameters, every '?' must be replaced with it's
 *              number - for example ' $19 '
 *-----------------------------------------------------------------------------
 */
static unsigned int
GetParametersLength(unsigned int unParametersNumber)
{
	if (0 == unParametersNumber)
		return 0;
	else
	{
		unsigned int unPreparedQueryLength = 0;
		unsigned int unCurrentLevel = 9;
		unsigned int unI = 2;

		while (unParametersNumber > unCurrentLevel)
		{
			unParametersNumber -= unCurrentLevel;
			unCurrentLevel *= 10;
			unPreparedQueryLength += unCurrentLevel*(unI++);
		}

		unPreparedQueryLength += unParametersNumber * unI;

		return unPreparedQueryLength;
	}
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareParameterQuery
 *
 * DESCRIPTION: 
 *-----------------------------------------------------------------------------
 */
static void
PrepareParameterQuery(TCHAR* szDestinationQuery, const TCHAR* szSourceQuery, int nSourceLength)
{
	SQLINTEGER   i, j, k;
	TCHAR        parameter[SQLSMALLINT_LENGTH];
	unsigned int unParameterLength = 0;
	unsigned int unParameter;
	unsigned int l;

	/* prepare query */
	for(i=0;i<SQLSMALLINT_LENGTH;i++)
		parameter[i]=_T('0');

	parameter[0]++;

	for(i=0,j=0,unParameter=1;i<nSourceLength;i++)
	{
		if(_T('\0') == szSourceQuery[i])
		{
//			szDestinationQuery[j++] = _T(' ');
			szDestinationQuery[j++] = _T('$');
			for (k=unParameterLength;k>=0;k--)
				szDestinationQuery[j++] = parameter[k];

			/* increment parameter's number */
			for (l=0;l<=unParameterLength;l++)
			{
				if (_T(':') != ++parameter[l])
					break;
				else
				{
					parameter[l] = _T('0');
					if (l == unParameterLength)
						unParameterLength++;
				}
			}
//			szDestinationQuery[j] = _T(' ');
		}
		else
			szDestinationQuery[j++] = szSourceQuery[i];
	}
	
	szDestinationQuery[j] = _T('\0');
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareQuery
 *
 * DESCRIPTION: Checks the type of the query and sets statement's query type,
 *              
 *
 * RETURN: SQL_SUCCESS           - type recognized
 *         SQL_SUCCESS_WITH_INFO - empty query or unknown type, type was set to
 *                                 ST_UNKNOWN
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareQuery(Statement* pStatement, SQLTCHAR *StatementText, SQLINTEGER TextLength,
             QueryOptions Options)
{
	/* in any case - this function changes the statement's query text */
	SET_STATEMENT_NEEDS_REDECLARATION(pStatement);

	/* do we have a text? */
	if (NULL != StatementText)
	{
		SQLINTEGER   LengthTemp;
		unsigned int uI;
		unsigned int unParametersNumber;
		unsigned int unPreparedQueryLength;
		TCHAR*       szExecuteQuery = NULL;
		TCHAR*       szQuery;
		TCHAR*       pText;

		int          nQuerySize = 0;		
		int          nExecuteQuerySize = 0;

		if (SQL_NTS == TextLength)
			TextLength = _tcslen((TCHAR*)StatementText);

		/* -- remove leading spaces -- */
		while (_istspace(*StatementText))
		{
			StatementText++;
			TextLength--;
		}

		/* -- remove trailing spaces -- */
		while ((0 < TextLength) && _istspace(StatementText[--TextLength]));
		TextLength++;

		if (0 < TextLength)
		{/* determine - which type of PREPARE use for this query:
		  * 1) without parameters - USE server-side
			* 2) some parameters - a) PostgreSQL version 8.0.x, 8.1.x - 'PREPARE SQL_CUR... AS '
			*                      b) PostgreSQL version 8.2          - currently not available
			*/
			TCHAR  query_copy[MAX_QUERY_LENGTH + 1];
			TCHAR* query_ptr = (MAX_QUERY_LENGTH > TextLength) ? query_copy : ((TCHAR*) malloc(sizeof(TCHAR)*(TextLength + 1)));

			_tcsncpy(query_ptr, StatementText, TextLength + 1);

			/* -- parse StatementText and set corresponding statement's type -- */
			/* uppercase first word in the query */
			pText = query_ptr;
			LengthTemp = TextLength + 1;
			while(LengthTemp)
			{
				if (_T('a') <= *pText && _T('z') >= *pText)
					*pText += 'A'-'a';
				else
					if (_T('A') > *pText || _T('Z') < *pText)
						break;
				pText++;
				LengthTemp--;
			}

			switch(query_ptr[0])
			{
				case _T('A'): /* ALTER */
					pStatement->query.type = ST_ALTER;
					break;
				case _T('B'): /* BEGIN */
					pStatement->query.type = ST_BEGIN;
					break;
				case _T('C'):	/* COMMIT, CREATE */
					pStatement->query.type = (_T('O') == query_ptr[1]) ? ST_COMMIT : ST_CREATE;
					break;
				case _T('D'): /* DELETE */
					pStatement->query.type = ST_DELETE;
					break;
				case _T('E'): /* EXECUTE */
					pStatement->query.type = ST_EXECUTE;
					break;
				case _T('F'): /* FETCH */
					pStatement->query.type = ST_FETCH;
					break;
				case _T('I'):	/* INSERT */
					pStatement->query.type = ST_INSERT;
					break;
				case _T('M'): /* MOVE */
					pStatement->query.type = ST_MOVE;
					break;
				case _T('P'): /* PREPARE */
					pStatement->query.type = ST_PREPARE;
					break;
				case _T('S'): /* SELECT, SET */
					pStatement->query.type = (_T('L') == query_ptr[2]) ? ST_SELECT : ST_SET;
					break;
				case _T('U'): /* UPDATE */
					pStatement->query.type = ST_UPDATE;
					break;
				default:
					pStatement->query.type = ST_UNKNOWN;
			}

			/* -- replace escape sequencies if SQL_ATTR_NOSCAN = SQL_NOSCAN_OFF -- */
			unParametersNumber = ReplaceEscapes(pStatement, (SQLTCHAR**)&query_ptr, (query_ptr == query_copy) ? sizeof(query_copy)/sizeof(TCHAR) : (TextLength + 1), &TextLength, TRUE, NULL, NULL, FALSE);

			/* determine */
			pStatement->query.prepareType = (Options & QO_SERVER_SIDE_PREPARE) ? PT_SERVER_SIDE : PT_SIMPLE_QUERY;

			if ((ST_SELECT == pStatement->query.type) && !(STMT_USE_BUFFERING(pStatement)) && !(Options & QO_FORCE_BUFFERING))
			{
				pStatement->query.prepareType = PT_FR_BK_PROTOCOL;
				pStatement->use_buffering = SQL_FALSE;
			}

			/* modify query in dependensy of prepare type */
			switch (pStatement->query.prepareType)
			{
				case PT_SERVER_SIDE_CURSOR:
				{
					TCHAR* dest_prepare;

					int    cursor_length = _tcslen(pStatement->query.cursorName);
					
					nQuerySize = STR_SIZEOF(QP_DECLARE) + cursor_length + STR_SIZEOF(QP_QUOTE_FOR) + 1 + TextLength;
					dest_prepare = szQuery = (TCHAR*) malloc(nQuerySize * sizeof(TCHAR));

					/* prepare cursor part */
					_tcsncpy(dest_prepare, _T(QP_DECLARE), STR_SIZEOF(QP_DECLARE));
					dest_prepare += STR_SIZEOF(QP_DECLARE);

					_tcsncpy(dest_prepare, pStatement->query.cursorName, cursor_length);
					dest_prepare += cursor_length;

					_tcsncpy(dest_prepare, _T(QP_QUOTE_FOR), STR_SIZEOF(QP_QUOTE_FOR));
					dest_prepare += STR_SIZEOF(QP_QUOTE_FOR);

					memcpy(dest_prepare, query_ptr, TextLength * sizeof(TCHAR));

					szQuery[nQuerySize - 1] = _T('\0');
					break;
				}
				/* use server-side PREPARE */
				case PT_SERVER_SIDE:
				{
					TCHAR* dest_prepare;
					TCHAR* dest_execute;

					/* calculate the size of the future PREPARE and EXECUTE queries */
					int    cursor_length     = _tcslen(pStatement->query.cursorName),
					       parameters_length = GetParametersLength(unParametersNumber);

 					nQuerySize        = STR_SIZEOF(QP_PREPARE) + cursor_length + STR_SIZEOF(QP_QUOTE_AS) + 1 + TextLength,
					nExecuteQuerySize = STR_SIZEOF(QP_EXECUTE) + cursor_length + STR_SIZEOF(QP_QUOTE)    + 1;

					if (0 < unParametersNumber)
					{
						int spaces = STR_SIZEOF(QP_SPACE) * unParametersNumber;

						nQuerySize        += STR_SIZEOF(QP_QUOTE_BRACKET) + spaces + STR_SIZEOF(QP_BRACKET_AS) \
														   + parameters_length \
														   - STR_SIZEOF(QP_QUOTE_AS) \
												       - 1 /* remove excess ',' */;

						nExecuteQuerySize += STR_SIZEOF(QP_QUOTE_BRACKET) + spaces + STR_SIZEOF(QP_CBRACKET) \
							                 - STR_SIZEOF(QP_QUOTE) \
														   - 1 /* remove excess ',' */;
					}

					/* allocate memory for these queries */
					dest_prepare = szQuery        = (TCHAR*) malloc(nQuerySize * sizeof(TCHAR));
					dest_execute = szExecuteQuery = (TCHAR*) malloc(nExecuteQuerySize * sizeof(TCHAR));

					/* prepare 'PREPARE' and 'EXECUTE' queries at the same time */
					_tcsncpy(dest_prepare, _T(QP_PREPARE), STR_SIZEOF(QP_PREPARE));
					dest_prepare += STR_SIZEOF(QP_PREPARE);

					_tcsncpy(dest_execute, _T(QP_EXECUTE), STR_SIZEOF(QP_EXECUTE));
					dest_execute += STR_SIZEOF(QP_EXECUTE);

					/* add cursor name */
					_tcsncpy(dest_prepare, pStatement->query.cursorName, cursor_length);
					_tcsncpy(dest_execute, pStatement->query.cursorName, cursor_length);
					dest_prepare += cursor_length;
					dest_execute += cursor_length;

					if (0 == unParametersNumber)
					{/* AS */
						_tcsncpy(dest_prepare, _T(QP_QUOTE_AS), STR_SIZEOF(QP_QUOTE_AS));
						dest_prepare += STR_SIZEOF(QP_QUOTE_AS);

						*dest_execute++ = _T('\"');

						/* copy actual query after */
						_tcsncpy(dest_prepare, query_ptr, TextLength);
						dest_prepare += TextLength;

						*dest_prepare = _T('\0');
						*dest_execute = _T('\0');
					}
					else
					{/* try to get parameters types and place them into PREPARE query */
						_tcsncpy(dest_prepare, _T(QP_QUOTE_BRACKET), STR_SIZEOF(QP_QUOTE_BRACKET));
						dest_prepare += STR_SIZEOF(QP_QUOTE_BRACKET);

						_tcsncpy(dest_execute, _T(QP_QUOTE_BRACKET), STR_SIZEOF(QP_QUOTE_BRACKET));
						dest_execute += STR_SIZEOF(QP_QUOTE_BRACKET);

						/* we do not put here actual types for parameters - we will be able to do
						 * this only at a fist execution time
						 */
						for (uI=0;uI<unParametersNumber;++uI)
						{
							memcpy(dest_prepare, _T(QP_SPACE), STR_SIZEOF(QP_SPACE) * sizeof(TCHAR));
							memcpy(dest_execute, _T(QP_SPACE), STR_SIZEOF(QP_SPACE) * sizeof(TCHAR));
							dest_prepare += STR_SIZEOF(QP_SPACE);
							dest_execute += STR_SIZEOF(QP_SPACE);
						}

						_tcsncpy(--dest_prepare, _T(QP_BRACKET_AS), STR_SIZEOF(QP_BRACKET_AS));
						dest_prepare += STR_SIZEOF(QP_BRACKET_AS);
						*--dest_execute = _T(')');

						/* replace '\0' with actual parameters' numbers and complete query */
						PrepareParameterQuery(dest_prepare, query_ptr, TextLength);
						
						*++dest_execute = _T('\0');
					}
					break;
				}	/* frontend/backend implementaion */
				case PT_FR_BK_PROTOCOL:
					if (0 < unParametersNumber)
					{
						unPreparedQueryLength = TextLength + 3 - unParametersNumber + GetParametersLength(unParametersNumber);
						szQuery = (TCHAR*) malloc(unPreparedQueryLength*sizeof(TCHAR));
						PrepareParameterQuery(szQuery, query_ptr, TextLength);
					}
					else
					{
						szQuery = (TCHAR*) malloc((TextLength + 1)*sizeof(TCHAR));
						_tcsncpy(szQuery, query_ptr, TextLength);
						szQuery[TextLength] = _T('\0');
					}
					nQuerySize = TextLength;
					break;

				/* this is a simple query, parameters should be replaced only once */
				case PT_SIMPLE_QUERY:
					/* everything is done */
					if (query_ptr == query_copy)
					{
						szQuery = ((TCHAR*) malloc((TextLength + 1) * sizeof(TCHAR)));
						memcpy(szQuery, query_ptr, TextLength * sizeof(TCHAR));
					}
					else
					{
						szQuery = query_ptr;
					}
					
					szQuery[TextLength] = _T('\0');
					nQuerySize = TextLength;
					break;
				default:
					/* unsupported prepare type - hm... */
					;
			}

			if (NULL != pStatement->query.query)
				FREE(pStatement->query.query);

			if (NULL != pStatement->query.execute_query)
				FREE(pStatement->query.execute_query);

			pStatement->query.nParametersNumber  = unParametersNumber;
			pStatement->query.execute_query      = szExecuteQuery;
			pStatement->query.query              = szQuery;
			pStatement->query.query_size         = TextLength + 1;
			pStatement->query.execute_query_size = nExecuteQuerySize + 1;
	
			/* check for passing restrictions */
			if (!(QO_IGNORE_BOOKMARK & Options) && (0 < pStatement->connection->restrictions.list.used))
			{/* we have restrictions for this connection */
				if (!SQL_SUCCEEDED(CheckQuery(pStatement->query.query, _tcslen(pStatement->query.query), &pStatement->connection->restrictions)))
				{
					SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ALLOWED_OBJECT, NULL);
					return SQL_ERROR;
				}
			}

			/* check for bookmarks */
			if (STMT_USE_BOOKMARKS(pStatement) &&
			    (QO_IGNORE_BOOKMARK & Options)
			   )
				return PrepareBookmark(pStatement);

			if (0 < pStatement->query.nParametersNumber)
				SET_STATEMENT_NEEDS_REDECLARATION(pStatement);

			return SQL_SUCCESS;
		}
	}

	/* empty query text to prepare */
	if(NULL != pStatement->query.query)
	{
		FREE(pStatement->query.query);
		pStatement->query.query_size = 0;
		pStatement->query.nParametersNumber = 0;
		pStatement->query.prepareType = PT_FR_BK_PROTOCOL;
	}

	pStatement->query.type = ST_EMPTY;
	SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_EMPTY_QUERY, NULL);
	return SQL_SUCCESS_WITH_INFO;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SetApplicationDescriptor
 *-----------------------------------------------------------------------------
 */
SQLRETURN
SetApplicationDescriptor(Statement* pStatement, Descriptor** ppDescInUse, Descriptor* pDescImplicit, Descriptor* pDescToSet)
{
	/* set back implicitly allocated descriptor? */
	if (SQL_NULL_HDESC == pDescToSet)
	{
		*ppDescInUse = pDescImplicit;
		return SQL_SUCCESS;
	}

	/* cannot set implicitly allocated descriptor */
	if (SQL_DESC_ALLOC_AUTO == pDescToSet->header.alloc_type)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_DESC_WRONG_ALLOC_TYPE, NULL);
		return SQL_ERROR;
	}

	/* descriptor and statement must be allocated on the same connection */
	if (pStatement->connection != pDescToSet->connection)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_DESC_WRONG_CONNECTION, NULL);
		return SQL_ERROR;
	}

	/* ok, set descriptor */
	*ppDescInUse = pDescToSet;
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetData
 *-----------------------------------------------------------------------------
 */
SQLRETURN
GetData(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValue, SQLLEN BufferLength, SQLLEN* StrLen_or_Ind)
{
	SQLRETURN   nRet = SQL_SUCCESS;
	Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

	/* is this a valid column number? */
	if (ColumnNumber <= pIRD->header.count)
	{
		if (0 == ColumnNumber)
		{/* bookmark column */
			if STMT_USE_BOOKMARKS(pStatement)
			{/* get data from bookmark column */
				SQLLEN ret;
				if (0 < (ret = FillBufferWithValue(TargetValue, BufferLength, TargetType,
				                                   (TCHAR*)((int*)pIRD->id_records[pIRD->header.count-1].common.data_ptr[pStatement->fetch_position]+1),
				                                   *(int*)pIRD->id_records[pIRD->header.count-1].common.data_ptr[pStatement->fetch_position],
				                                   pIRD->id_records[pIRD->header.count-1].common.concise_type)))
					pIRD->id_records[pIRD->header.count-1].bytes_left -= ret;
				if (NULL != StrLen_or_Ind)
					*StrLen_or_Ind = ret;
			}
			else
			{
				/* no bookmarks used */
				SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_BOOKMARKS_UNUSED, NULL);
				nRet = SQL_ERROR;
			}
		}
		else
		{/* column exists in result set */
			ColumnNumber--;
			if (FIELD_NULL == *(int*)pIRD->id_records[ColumnNumber].common.data_ptr[pStatement->fetch_position])
			{/* null */
				*StrLen_or_Ind = SQL_NULL_DATA;
			}
			else if (0 <= pIRD->id_records[ColumnNumber].bytes_left)
			{
				SQLLEN ret_size;
				if (0 < (ret_size = FillBufferWithValue(TargetValue, BufferLength, TargetType, (TCHAR*)((int*)pIRD->id_records[ColumnNumber].common.data_ptr[pStatement->fetch_position]+1), pIRD->id_records[ColumnNumber].bytes_left, pIRD->id_records[ColumnNumber].common.concise_type)))
					pStatement->ird->id_records[ColumnNumber].bytes_left -= ret_size;

				if (NULL != StrLen_or_Ind)
					*StrLen_or_Ind = GetCTypeLength((SQLSMALLINT)((SQL_DEFAULT == TargetType) ? GetCDefaultType(pIRD->id_records[ColumnNumber].common.concise_type) : TargetType), ret_size);
			}
			else
			{/* all avaible data already fetched */
				nRet = SQL_NO_DATA;
			}
		}
	}
	else
	{
		/* column with such number doesn't exist in result set */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_COLUMN_NUMBER_TOO_BIG, NULL);
		nRet = SQL_ERROR;
	}

	RET_DESCRIPTOR(pIRD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ColAttribute
 *-----------------------------------------------------------------------------
 */
SQLRETURN
ColAttribute(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier,
             SQLPOINTER CharacterAttribute, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength,
             SQLLEN* NumericAttribute)
{
	SQLRETURN   nRet = SQL_SUCCESS;
	SQLINTEGER  integer  = 0;
	SQLLEN      len      = 0;
	SQLSMALLINT ret_type = 0;

	Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

	if (NULL != NumericAttribute)
		*NumericAttribute = 0;

	if (0 == ColumnNumber)
	{
		/* bookmark column */
		/* Driver Manager won't get us here if SQL_ATTR_USE_BOOKMARKS = SQL_UB_OFF */
		switch(FieldIdentifier)
		{
			case SQL_DESC_TYPE:               /* 1002 */
				integer  = pIRD->bookmark.type; /* SQL_BINARY or SQL_INTEGER */
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_DESC_OCTET_LENGTH:       /* 1013 */
				len = pIRD->bookmark.octet_length;
				ret_type = SQL_IS_LEN;
				break;
			/* all other fields here can return underfined values */
		}
	}
	else
	{
		BOOL         translate = FALSE;
		SQLPOINTER   value     = NULL;
		SQLSMALLINT* length    = NULL;
		SQLSMALLINT  smallint  = 0;
		SQLSMALLINT  resLength = 0;

		switch(FieldIdentifier)
		{
			/* 64-bit numeric values in 64-bit mode */
			case SQL_DESC_DISPLAY_SIZE:       /* 6 */
			case SQL_DESC_LENGTH:          /* 1003 */
			case SQL_DESC_OCTET_LENGTH:    /* 1013 */
			case SQL_DESC_COUNT:           /* 1001 */
			/* numeric values */
			case SQL_DESC_AUTO_UNIQUE_VALUE: /* 11 */
			case SQL_DESC_CASE_SENSITIVE:    /* 12 */
			case SQL_DESC_NUM_PREC_RADIX:  /*   32 */
				value  = &len;
				length = &resLength;
				ret_type = SQL_IS_LEN;
				break;
			case SQL_DESC_CONCISE_TYPE:       /* 2 */
			case SQL_DESC_UNSIGNED:           /* 8 */
			case SQL_DESC_FIXED_PREC_SCALE:   /* 9 */
			case SQL_DESC_UPDATABLE:         /* 10 */
			case SQL_DESC_SEARCHABLE:        /* 13 */
			case SQL_DESC_NULLABLE:        /* 1008 */
			case SQL_DESC_PRECISION:       /* 1005 */
			case SQL_DESC_SCALE:           /* 1006 */
			case SQL_DESC_TYPE:            /* 1002 */
			case SQL_DESC_UNNAMED:         /* 1012 */
				value  = &smallint;
				length = &resLength;
				translate = TRUE;
				ret_type = SQL_IS_SMALLINT;
				break;
			/* string values */
			case SQL_DESC_BASE_COLUMN_NAME:  /* 22 */
			case SQL_DESC_BASE_TABLE_NAME:   /* 23 */
			case SQL_DESC_CATALOG_NAME:      /* 17 */
			case SQL_DESC_LABEL:             /* 18 */
			case SQL_DESC_LITERAL_PREFIX:    /* 27 */
			case SQL_DESC_LITERAL_SUFFIX:    /* 28 */
			case SQL_DESC_LOCAL_TYPE_NAME:   /* 29 */
			case SQL_DESC_NAME:            /* 1011 */
			case SQL_DESC_SCHEMA_NAME:       /* 16 */
			case SQL_DESC_TABLE_NAME:        /* 15 */
			case SQL_DESC_TYPE_NAME:         /* 14 */
				value  = CharacterAttribute;
				length = StringLength;
				ret_type = SQL_IS_POINTER;
				break;
		}
		nRet = GetDescField(pIRD, ColumnNumber, FieldIdentifier, value, BufferLength, length, pStatement);
		if (translate)
		{
			integer = smallint;
			ret_type = SQL_IS_INTEGER;
		}
	}

	if (NumericAttribute) {
		if (SQL_IS_INTEGER == ret_type)
			memcpy(NumericAttribute, &integer, sizeof(integer));
		else if (SQL_IS_LEN == ret_type)
			memcpy(NumericAttribute, &len, sizeof(len));
	}

	RET_DESCRIPTOR(pIRD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: DescribeCol
 *-----------------------------------------------------------------------------
 */
SQLRETURN
DescribeCol(Statement* pStatement, SQLUSMALLINT ColumnNumber,  SQLTCHAR* ColumnName,
            SQLSMALLINT BufferLength, SQLSMALLINT* NameLength, SQLSMALLINT* DataType,
            SQLULEN* ColumnSize, SQLSMALLINT* DecimalDigits, SQLSMALLINT* Nullable)
{
	SQLRETURN   nRet = SQL_SUCCESS;
	Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

	if (ColumnNumber <= pIRD->header.count)
	{
		if (0 < ColumnNumber)
		{
			ID_REC* pID_REC = &pIRD->id_records[ColumnNumber-1];
			
			SQLSMALLINT concise_type;
			SQLSMALLINT precision;
			SQLINTEGER  data_size;
			SQLINTEGER  length;

			PostgreTypeToSQLType(pID_REC->type_oid, pID_REC->type_modifier, pStatement->connection->environment->attributes.odbc_version, NULL, &concise_type, &length, &precision, pStatement->connection->mjet);
			DescribeSQLType(concise_type, length, precision, NULL, &data_size, NULL, NULL);

			/* SQL data type */
			if (NULL != DataType)
				*DataType = concise_type;

			/* column display size - for sharp numeric types - this is precision */
			if (NULL != ColumnSize)
				*ColumnSize = data_size;

			/* now driver can't determine nullability */
			if (NULL != Nullable)
				*Nullable = pID_REC->nullable;

			/* Decimal digits - this value based on data, retrived from server */
			if (NULL != DecimalDigits)
				*DecimalDigits = (precision < 0) ? 0 : precision;

			/* column name and it's length, we don't call GetDescField - because it sets pDescritor's error */
			if (NULL != ColumnName)
				nRet = ReturnString(ColumnName, BufferLength, NameLength, (TCHAR*)pID_REC->name, SQL_NTS, SQL_FALSE /* I have no idea why! MSDN says - this value should be in bytes, not in characters */);
		}
		else
		{
			/* bookmark column */
			if (NULL != ColumnSize)
				*ColumnSize = pIRD->bookmark.octet_length;

			if (NULL != Nullable)
				*Nullable = SQL_NO_NULLS; /* bookmark column can't be NULL */

			if (NULL != DecimalDigits)
				*DecimalDigits = 0;       /* No decimal digits in SQL_BINARY */

			if (NULL != DataType)
				*DataType = pIRD->bookmark.type; /* SQL_BINARY or SQL_INTEGER */

	    nRet = ReturnString(ColumnName, BufferLength, NameLength, c_szBookmarkName, SQL_NTS, SQL_TRUE);
		}

		if (SQL_SUCCESS != nRet)
			SetError(SQL_HANDLE_STMT, pStatement, ERR_TOO_SMALL_BUFFER, _T("ColumnName"), NULL);
	}
	else
	{
		/* column with such number doesn't exist in result set */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_COLUMN_NUMBER_TOO_BIG, NULL);
		nRet = SQL_ERROR;
	}

	RET_DESCRIPTOR(pIRD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: BindCol
 *
 * DESCRIPTION: columns numbered from 1, bookmark column = 0
 *-----------------------------------------------------------------------------
 */
SQLRETURN 
BindCol(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValue, SQLLEN BufferLength, SQLLEN* StrLen_or_Ind)
{
	SQLRETURN   nRet = SQL_SUCCESS;
	Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);

	if (0 < ColumnNumber)
	{/* application data column */
		SQLSMALLINT savedARDCount = pARD->header.count;

		if (NULL == TargetValue)
		{/* unbind column */
			/* if this is the last record - just decrement 'count' field, else - empty it */
			nRet = (ColumnNumber == pARD->header.count) ? SetDescField(pARD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLULEN_TO_POINTER)(ColumnNumber-1), SQL_IS_SMALLINT) : EmptyDescriptorRecord(pARD, (SQLSMALLINT)(ColumnNumber-1));
		}
		else if (SQL_ERROR != ReallocDescriptorRecords(pARD, ColumnNumber))
		{/* bind or rebind column */
			AD_REC* pAD_REC	= &pARD->ad_records[--ColumnNumber];

			/* fills ARD's fields with data */
			pAD_REC->bound               = SQL_TRUE;
			pAD_REC->common.octet_length = BufferLength;
			pAD_REC->common.data_ptr     = (SQLPOINTER*)TargetValue;
			pAD_REC->indicator_ptr       = StrLen_or_Ind;
			pAD_REC->octet_length_ptr    = StrLen_or_Ind;

			TranslateType(&pAD_REC->common, TargetType, 0, BufferLength, C_TYPE);
		}
		else
		{/* unable to realloc descriptor's records */
			SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
			nRet = SQL_ERROR;
		}

		/* if an error occured - restore saved COUNT field */
		if (SQL_ERROR == nRet)
			pARD->header.count = savedARDCount;
	}
	else if STMT_USE_BOOKMARKS(pStatement)
	{/* bind bookmark's column */
		Bookmark* pBookmark = &pARD->bookmark;

		if (NULL == TargetValue)
		{
			/* -- unbind bookmark column -- */
			pBookmark->data_ptr      = NULL;
			pBookmark->indicator_ptr = NULL;
		}
		else
		{
			pBookmark->data_ptr      = TargetValue;
			pBookmark->indicator_ptr = StrLen_or_Ind;
			pBookmark->octet_length  = BufferLength;
			pBookmark->type          = TargetType;
		}
	}
	else
	{/* no bookmarks used */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_BOOKMARKS_UNUSED, NULL);
		nRet = SQL_ERROR;
	}
	
	RET_DESCRIPTOR(pARD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetStmtAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN
GetStmtAttr(Statement* pStatement, SQLINTEGER Attribute, SQLPOINTER Value)
{
	SQLINTEGER  uInteger;
	SQLLEN      nLen;
	SQLPOINTER  pointer;
	SQLHDESC    hDesc;
	SQLSMALLINT ret_type;

	if (NULL == Value)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_NULL_POINTER_BUFFER, NULL);
		return SQL_ERROR;
	}
	else
	{
		SQLRETURN nRet = SQL_SUCCESS;
		switch(Attribute)
		{
			case SQL_ATTR_QUERY_TIMEOUT:            /* 0 */
				uInteger = pStatement->attributes.query_timeout;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_MAX_ROWS:                 /* 1 */
				nLen = pStatement->attributes.max_rows;
				ret_type = SQL_IS_LEN;
				break;
			case SQL_ATTR_NOSCAN:                   /* 2 */
				uInteger = pStatement->attributes.no_scan;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_MAX_LENGTH:               /* 3 */
				nLen = pStatement->attributes.max_length;
				ret_type = SQL_IS_LEN;
				break;
			case SQL_ATTR_CURSOR_TYPE:              /* 6 */
				uInteger = pStatement->attributes.cursor;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_CONCURRENCY:              /* 7 */
				uInteger = pStatement->attributes.concurrency;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_USE_BOOKMARKS:           /* 12 */
				uInteger = pStatement->attributes.use_bookmarks;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_ENABLE_AUTO_IPD:         /* 15 */
				uInteger = pStatement->attributes.enable_auto_ipd;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_PARAMS_PROCESSED_PTR:    /* 21 */
				pointer  = pStatement->ipd->header.rows_processed_ptr;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_ATTR_ROWS_FETCHED_PTR:        /* 26 */
				pointer = pStatement->ird->header.rows_processed_ptr;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_ATTR_ROW_ARRAY_SIZE:          /* 27 */
			/* ODBC 2.0 */
			case SQL_ROWSET_SIZE:                   /* 9 */
				nLen = pStatement->ard->header.array_size;
				ret_type = SQL_IS_LEN;
				break;
			case SQL_ATTR_APP_PARAM_DESC:       /* 10011 */
				hDesc    = pStatement->apd;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_APP_ROW_DESC:         /* 10010 */
				hDesc    = pStatement->ard;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_IMP_PARAM_DESC:       /* 10013 */
				hDesc    = pStatement->ipd;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_IMP_ROW_DESC:         /* 10012 */
				hDesc    = pStatement->ird;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_METADATA_ID:          /* 10014 */
				uInteger = pStatement->attributes.metadata_id;	
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_ROW_BIND_TYPE:            /* 5 */
				return GetDescField(pStatement->ard, 0, SQL_DESC_BIND_TYPE, Value, SQL_IS_INTEGER, NULL, pStatement);
			case SQL_ATTR_RETRIEVE_DATA:           /* 11 */
				uInteger = pStatement->attributes.retrieve_data;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_FETCH_BOOKMARK_PTR:      /* 16 */
				pointer  = pStatement->attributes.bookmark_ptr;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_ATTR_PARAM_BIND_OFFSET_PTR:   /* 17 */
				return GetDescField(pStatement->apd, 0, SQL_DESC_BIND_OFFSET_PTR, Value, SQL_IS_POINTER, NULL, pStatement);
			case SQL_ATTR_PARAM_BIND_TYPE:         /* 18 */
				return GetDescField(pStatement->apd, 0, SQL_DESC_BIND_TYPE, Value, SQL_IS_UINTEGER, NULL, pStatement);
			case SQL_ATTR_PARAMSET_SIZE:           /* 22 */
				return GetDescField(pStatement->apd, 0, SQL_DESC_ARRAY_SIZE, Value, SQL_IS_UINTEGER, NULL, pStatement);
			case SQL_ATTR_ROW_BIND_OFFSET_PTR:     /* 23 */
				return GetDescField(pStatement->ard, 0, SQL_DESC_BIND_OFFSET_PTR, Value, SQL_IS_POINTER, NULL, pStatement);
			case SQL_ATTR_ROW_STATUS_PTR:          /* 25 */
				return GetDescField(pStatement->ird, 0, SQL_DESC_ARRAY_STATUS_PTR, Value, SQL_IS_POINTER, NULL, pStatement);

			case SQL_ATTR_ASYNC_ENABLE:             /* 4 */
			case SQL_ATTR_CURSOR_SCROLLABLE:       /* -1 */
			case SQL_ATTR_CURSOR_SENSITIVITY:      /* -2 */
			case SQL_ATTR_KEYSET_SIZE: /* 64 bit value in 64-bit mode */
			case SQL_ATTR_PARAM_OPERATION_PTR:     /* 19 */
			case SQL_ATTR_PARAM_STATUS_PTR:        /* 20 */
			case SQL_ATTR_ROW_NUMBER: /* 64 bit value in 64-bit mode */
			case SQL_ATTR_ROW_OPERATION_PTR:       /* 24 */
			case SQL_ATTR_SIMULATE_CURSOR:         /* 10 */
			default:
				SetError(SQL_HANDLE_STMT, pStatement, ERR_UNKNOWN_ATTRIBUTE, NULL);
				return SQL_ERROR;
		}

		switch (ret_type)
		{
			case SQL_IS_LEN:
				memcpy(Value, &nLen, sizeof(nLen));
				break;
			case SQL_IS_SMALLINT: /* HDESC */
				memcpy(Value, &hDesc, sizeof(hDesc));
				break;
			case SQL_IS_INTEGER:
				memcpy(Value, &uInteger, sizeof(uInteger));
				break;
			case SQL_IS_POINTER:
				memcpy(Value, &pointer, sizeof(pointer));
				break;
		}
		return nRet;
	}
}

/*-----------------------------------------------------------------------------
 * FUNCTION: BulkOperations
 *-----------------------------------------------------------------------------
 */
SQLRETURN
BulkOperations(Statement* pStatement, SQLSMALLINT Operation, BOOL continue_aborted)
{
	SQLRETURN nRet = SQL_SUCCESS;
	SQLUSMALLINT* status_ptr = NULL;

	switch(Operation)
	{
		case SQL_ADD:
			status_ptr = pStatement->saved_status_ptr;
		case SQL_UPDATE_BY_BOOKMARK:
		case SQL_DELETE_BY_BOOKMARK:
		{/* -- insert data --
			* SQL_ATTR_ROW_ARRAY_SIZE containes the number of rows that application wants to insert,
			* SQL_ATTR_ROW_STATUS_PTR can be checked for results
			* data bound by the SQLBindCol
			*/
			uint32      table_oid;
			SQLSMALLINT columns;
			SQLSMALLINT i;
			Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

			/* check alter possibility */
			if (0 >= (columns = pIRD->header.count - pIRD->header.bookmarks))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_BULK_NO_COLUMNS_TO, (SQL_ADD == Operation) ? _T("add") : ((SQL_UPDATE_BY_BOOKMARK == Operation) ? _T("update") : _T("delete")), NULL);
				nRet = SQL_SUCCESS_WITH_INFO;
			}
			else
			{
				if (0 >= (table_oid = pIRD->id_records[0].table_oid))
					nRet = SQL_ERROR;
				else for(i=columns-1;i>=0;i--)
					if (table_oid != pIRD->id_records[i].table_oid)
					{
						nRet = SQL_ERROR;
						break;
					}
				if (SQL_ERROR == nRet)
					SetError(SQL_HANDLE_STMT, pStatement, ERR_BULK_DIFFERENT_TABLES, NULL);
				else
				{
					Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);

					/* -- check - 'all columns bound to the driver by the application?' -- */
					if (pARD->header.count >= pIRD->header.count)
						for (i=columns-1;i>=0;i--)
						{
							if (SQL_FALSE == pARD->ad_records[i].bound)
								break;
						}

					if (0 <= i)
					{/* not all columns were bound to the driver by the application */
						SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_NOT_ENOUGH_PARAMETERS, NULL);
						nRet = SQL_ERROR;
					}
					else
					{
						/* check parameters for SQL_DATA_AT_EXEC */
						for (i=(continue_aborted) ? pStatement->data_at_exec.index : (columns-1);i>=0;i--)
						{
							if IS_DATA_AT_EXEC(pARD, i)
							{
								pStatement->need_data = _T('c');
								pStatement->data_at_exec.aborted_function         = SQL_API_SQLBULKOPERATIONS;
								pStatement->data_at_exec.aborted_function_options = SQL_ADD;
								pStatement->data_at_exec.index                    = i;
								nRet = SQL_NEED_DATA;
								break;
							}
						}

						if (SQL_NEED_DATA != nRet)
						{
							TCHAR         schema[SCHEMA_MAX_LENGTH+1];
							TCHAR         table[TABLE_MAX_LENGTH+1];
							SQLUSMALLINT* status_ptr;
							SQLINTEGER    rows;
							Statement*    pAddStatement;

							GetStmtAttr(pStatement, SQL_ATTR_ROW_ARRAY_SIZE, &rows);       /* number of rows to insert */
							GetStmtAttr(pStatement, SQL_ATTR_ROW_STATUS_PTR, &status_ptr); /* info array to return */

							GetDescField(pIRD, 1, SQL_DESC_SCHEMA_NAME, schema, sizeof(schema), NULL, NULL); /* schema name */
							GetDescField(pIRD, 1, SQL_DESC_TABLE_NAME,  table, sizeof(table), NULL, NULL);   /* table name */

							/* alloc new statement */
							if (NULL != (pAddStatement = AllocStatement(pStatement->connection)))
							{
								TCHAR  questions[(COLUMN_MAX_LENGTH+CHR_SIZEOF(",?,"))*MAX_COLUMNS_IN_TABLE+1];
								TCHAR* col_name = questions;

								SQLSMALLINT col_length;
								TCHAR*      query = NULL;
								
								switch(Operation)
								{
									case SQL_ADD:
									{/*  prepare insert query */
										for (i=columns-1;i>=0;i--)
										{/* prepare questions */
											*(col_name++) = _T('?');
											*(col_name++) = _T(',');
										}
										*(--col_name) = _T('\0');

										for (i=1;i<=columns;i++)
										{/* prepare column-name's list */
											GetDescField(pIRD, i, SQL_DESC_BASE_COLUMN_NAME, ++col_name, COLUMN_MAX_LENGTH+1, &col_length, NULL);
											col_name += col_length/sizeof(TCHAR);
											*col_name = _T(',');
										}
										*col_name = _T('\0');
										query = GetText(_T("INSERT INTO \"\?\".\"\?\"(\?)VALUES(\?)"), schema, table, questions+2*columns/* column names */, questions, NULL);
										break;
									}
									case SQL_UPDATE_BY_BOOKMARK:
									{/* prepare update query */
										*col_name = _T('\"');
										for (i=1;i<=columns;i++)
										{/* prepare column-name's list */
											GetDescField(pIRD, i, SQL_DESC_BASE_COLUMN_NAME, ++col_name, COLUMN_MAX_LENGTH+1, &col_length, NULL);
											col_name += col_length/sizeof(TCHAR);
											_tcsncpy(col_name, _T("\"=\?,\""), STR_SIZEOF("\"=\?,\""));
											col_name += STR_SIZEOF("\"=\?,\"")-1;
										}

										col_name -= STR_SIZEOF(",\"")-1;
										*col_name = _T('\0');

										query = GetText(_T("UPDATE \"\?\".\"\?\" SET \? WHERE \?"), schema, table, questions, _T("ctid = ?"), NULL);
										break;
									}
									case SQL_DELETE_BY_BOOKMARK:
									{/* prepare delete query */
										query = GetText(_T("DELETE FROM \"\?\".\"\?\" WHERE \?"), schema, table, _T("ctid = ?"), NULL);
										break;
									}
								}

								if (NULL != query)
								{/* process prepared query */
									Descriptor* pIPD = GET_DESCRIPTOR(pAddStatement->ipd);
									PrepareStatement(pAddStatement, (SQLTCHAR*)query, SQL_NTS);
									FREE(query);

									if (SQL_ERROR != ReallocDescriptorRecords(pIPD, (SQLSMALLINT)((SQL_DELETE_BY_BOOKMARK == Operation) ? 1 : columns)))
									{
										SQLSMALLINT j;
										SQLINTEGER offset = ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr);

										/* transaction */
										BeginTransaction(pAddStatement, TO_DRIVER);

										/* bind parameters and execute prepared query */
										for(j=0;j<rows;j++)
										{
											/* translate parameters from application to update or insert */
											if (SQL_ADD == Operation || SQL_UPDATE_BY_BOOKMARK == Operation)
											{
												/* prepare insert parameters */												
												for (i=0;i<columns;i++)
												{/* convert parameters from SQL_C_TYPE into backend's specific format(text) */
													CD_REC* common = &pARD->ad_records[i].common;
													SQLSMALLINT  type = (SQL_DEFAULT == common->concise_type) ? GetCDefaultType(pIRD->id_records[i].common.concise_type) : common->concise_type;
													SQLPOINTER buffer = (BYTE*)common->data_ptr + offset +													                    
													                    j * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? GetCTypeLength(type, (pARD->ad_records[i].octet_length_ptr) ? *pARD->ad_records[i].octet_length_ptr : 0) : pARD->header.bind_type);

													if (NULL == (pIPD->id_records[i].common.data_ptr = (SQLPOINTER*)PrepareParameter(pStatement, buffer, common->octet_length, common->concise_type, pARD->ad_records[i].indicator_ptr, pIPD->id_records[i].common.concise_type, common->scale)))
													{/* no more memory */
														nRet = SQL_ERROR;
														break;
													}
												}
											}

											/* prepare bookmark parameters */
											if (SQL_DELETE_BY_BOOKMARK == Operation || SQL_UPDATE_BY_BOOKMARK == Operation)
											{
												SQLPOINTER buffer = (BYTE*)pARD->bookmark.data_ptr + offset +													                    
												                    j * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? GetCTypeLength(pARD->bookmark.type, (pARD->bookmark.indicator_ptr) ? *pARD->bookmark.indicator_ptr : 0) : pARD->header.bind_type);

												/* convert bookmarks parameters */		
												if (NULL == (pIPD->id_records[(SQL_DELETE_BY_BOOKMARK == Operation) ? 0 : columns].common.data_ptr = (SQLPOINTER*)PrepareParameter(pStatement, buffer, pARD->bookmark.octet_length, pARD->bookmark.type, pARD->bookmark.indicator_ptr, SQL_DEFAULT, 0)))
												{/* no more memory */
													nRet = SQL_ERROR;
												}
											}

											nRet = ((SQL_SUCCESS == (nRet = DeclarePortal(pAddStatement))) &&
											        (SQL_SUCCESS == Stmt_SendMessageToBackend(pAddStatement->connection, MSG_Execute, pAddStatement)) &&
											        (SQL_SUCCESS == Stmt_SendMessageToBackend(pAddStatement->connection, MSG_Sync, pAddStatement)) &&
											        (SQL_SUCCESS == WaitForBackendReply(pAddStatement->connection,  MSG_ReadyForQuery, pAddStatement))
											       ) ? SQL_ROW_ADDED : SQL_ROW_ERROR;
												
											/* update status array */
											if (status_ptr)
												status_ptr[j] = nRet;
										}
										EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT /* in any case */, TO_DRIVER);
										nRet = SQL_SUCCESS;
									}
									else
									{
										SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
									}
									RET_DESCRIPTOR(pIPD);
								}
							}
							FreeStatement(pAddStatement, SQL_DROP);
						}
					}
					RET_DESCRIPTOR(pARD);
				}
			}
			RET_DESCRIPTOR(pIRD);
			break;
		}
		case SQL_FETCH_BY_BOOKMARK:
		{/* fetch SQL_ATTR_ROW_ARRAY_SIZE columns into bound columns */
			SQLSMALLINT i          = 0;
			SQLINTEGER  row        = 0;
			SQLINTEGER  array_size = 0;
			SQLINTEGER  increment  = 0;
			SQLRETURN   status     = SQL_SUCCESS;
			SQLCHAR*    bookmarks  = NULL;
			SQLLEN*     status_arr = NULL;
			Descriptor* pIRD       = NULL;
			
			/* get data for process */
			Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);
			array_size = pARD->header.array_size;
			bookmarks  = (BYTE*)pARD->bookmark.data_ptr + ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr);
			increment  = ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? GetCTypeLength(pARD->bookmark.type, (pARD->bookmark.indicator_ptr) ? *pARD->bookmark.indicator_ptr : 0) : pARD->header.bind_type) / sizeof(SQLCHAR);
			status_arr = pARD->bookmark.indicator_ptr;
			pIRD = GET_DESCRIPTOR(pStatement->ird);

			for (i=0;i<array_size;i++)
			{/* for every found bookmark in bound array fill bound columns and status array */
				if (0 <= (row = FindRow(pIRD, bookmarks, pARD->bookmark.type)))
				{/* bookmark exists */
					switch(status = FillBoundColumns(pARD, pIRD, i, row, SQL_FALSE))
					{
						case SQL_SUCCESS:
							status_arr[i] = SQL_ROW_SUCCESS;
							break;
						case SQL_SUCCESS_WITH_INFO:
							status_arr[i] = SQL_ROW_SUCCESS_WITH_INFO;
							break;
						case SQL_ERROR:
						default:
							status_arr[i] = SQL_ROW_ERROR;
					}
					bookmarks += increment;
				}
				else
				{/* failed to find column corresponding to bookmark value */
					status_arr[i] = SQL_ROW_ERROR;
				}
			}
			RET_DESCRIPTOR(pIRD);
			RET_DESCRIPTOR(pARD);
			break;
		}
		default:
			/* Driver Manager is not supposed to let us come here, even more - it's impossible! */
			SetError(SQL_HANDLE_STMT, pStatement, ERR_INVALID_BULK_OPERATION, NULL);
			nRet = SQL_ERROR;
	}
	return nRet;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareBookmark
 *
 * REMARK: Adding column before 'FROM' allows - do not recalculate coordinates in
 *         resultset and add as many columns as we need
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareBookmark(Statement* pStatement)
{
	/* Discover - is the query 'SELECT' */
	if (ST_SELECT != pStatement->query.type)
		/* don't need bookmark */
		return SQL_SUCCESS;
	else
	{/* Determine the method to use in preparing bookmark:
	  *
		* - CID - this is the only method for now
	  * - OID
	  * - PRIMARY KEY
	  *
	  */
		TCHAR*     query     = pStatement->query.query;
		SQLINTEGER length    = _tcslen(pStatement->query.query);
		TCHAR*     ending    = query + length;
		SQLINTEGER counter   = 0;
		TCHAR      in_quotes = _T('\0'); /* use this character as indicator and quotation mark simultaneously */

		/* check for only one query in statement's text */
		for(;query <= ending;query++)
			if (_T(';') == *query)
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_BOOKMARK_ONE_QUERY, NULL);
				return SQL_ERROR;
			}
		
		/* Find corresponding 'FROM' for the first 'SELECT' */
		for(query = pStatement->query.query;query <= ending;query++)
		{
			if (in_quotes)
			{/* ignore all text in quotes */
				if (*query == in_quotes)
					in_quotes = _T('\0');
			}
			else
			{
				if (_T('\'') == *query || _T('\"') == *query)
					in_quotes = *query;
				else
				{/* not in quotes - look for 'SELECT' and  'FROM' */
					if (0 == _tcsnicmp(query, _T("select"), STR_SIZEOF("select")))
					{
						counter++;
					}
					else if (0 == _tcsnicmp(query, _T("from"), STR_SIZEOF("from")))
					{
						if (0 == --counter)
						{/* we've found it */
							TCHAR*    bookmark = _T(",ctid ");
							TCHAR*      comma  = query;
							SQLSMALLINT tables = 1;
							pStatement->ird->header.bookmarks = 1; /* this value depands on 'bookmark' */
							
							/* calculate number of tables in use */
							for(;comma <= ending;comma++)
								if (_T(',') == *comma)
									tables++;

							if (1 < tables)
							{/* unsupported for now */
								SetError(SQL_HANDLE_STMT, pStatement, ERR_BOOKMARK_IN_PARSE, NULL);
								return SQL_ERROR;
							}
							else
							{/* update query with additional column */
								SQLSMALLINT len_bookma = _tcslen(bookmark);
								SQLSMALLINT len_before = query-pStatement->query.query;
								TCHAR*      new_query  = (TCHAR*) malloc((length + len_bookma + 1)*sizeof(TCHAR));
								_tcsncpy(new_query, pStatement->query.query, len_before);
								_tcsncpy(new_query+len_before, bookmark, len_bookma);
								_tcscpy(new_query+len_before+len_bookma, query);
								FREE(pStatement->query.query);
								pStatement->query.query = new_query;
							}
							return SQL_SUCCESS;
						}
					}
				}
			}
		}
	}

	return SQL_ERROR;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SetPos
 *
 * DESCRIPTION: Functionality reflects in SQLGetInfo(78,79,80,83)
 *-----------------------------------------------------------------------------
 */
SQLRETURN
SetPos(Statement*    pStatement,
       SQLSETPOSIROW RowNumber, /* 0 - the operation applies to every row in the rowset */
       SQLUSMALLINT  Operation, /* SQL_POSITION, SQL_REFRESH, SQL_UPDATE or SQL_DELETE
                                * SQL_ADD has been deprecated for ODBC 3.x */
       SQLUSMALLINT LockType   /* SQL_LOCK_NO_CHANGE, SQL_LOCK_EXCLUSIVE, SQL_LOCK_UNLOCK */
      )
{
	SQLRETURN   nRet = SQL_SUCCESS;
	Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

	/* check RowNumber */
	if ((SQLUINTEGER)RowNumber > pIRD->header.array_size)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_INVALID_ROWNUMBER, NULL);
		nRet = SQL_ERROR;
	}
	/* check LockType - now driver can support only SQL_LOCK_NO_CHANGE */
	else if (SQL_LOCK_NO_CHANGE != LockType)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_UNSUPPORTED_LOCKTYPE, NULL);
		nRet = SQL_ERROR;
	}
	else
	{
		Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);
		
		/* row can't be in SQL_ROW_DELETED state */
		switch(Operation)
		{
			case SQL_POSITION:
				pStatement->fetch_position = RowNumber-1;
				break;
			case SQL_DELETE:
			case SQL_UPDATE:
				/* we can use these operations only with BOOKMARKS ON */
				if STMT_USE_BOOKMARKS(pStatement)
				{/* try to process operation */
					SQLINTEGER columns        = 0; /* number of columns, bound by application */
					SQLINTEGER first_bookmark = 0; /* number of the first bookmark column in resultset */
					SQLTCHAR*  query = GetQuery(pARD, pIRD, &columns, &first_bookmark, Operation);
					if (query)
					{/* make request */
						Statement* pQueryStmt = AllocStatement(pStatement->connection);
						if (pQueryStmt &&
						    SQL_SUCCEEDED(PrepareStatement(pQueryStmt, query, SQL_NTS)) &&
								SQL_SUCCEEDED(BeginTransaction(pQueryStmt, TO_DRIVER))
						   )
						{
							SQLINTEGER nRes, i;
							SQLINTEGER offset = ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr);

							/* if we should process all rows - set its real count for use in 'while' */
							SQLINTEGER UpperLimit = (0 == RowNumber) ? pIRD->header.array_size : RowNumber--;
							Descriptor* pIPD = GET_DESCRIPTOR(pQueryStmt->ipd);

							for(;RowNumber<UpperLimit;RowNumber++)
							{/* prepare parameters */

								/* translate parameters from application to update or insert */
								if (SQL_UPDATE == Operation)
								{
									/* prepare insert parameters */												
									for (i=0;i<columns;i++)
									{/* convert parameters from SQL_C_TYPE into backend's specific format(text) */
										CD_REC* common = &pARD->ad_records[i].common;
										SQLSMALLINT  type = (SQL_DEFAULT == common->concise_type) ? GetCDefaultType(pIRD->id_records[i].common.concise_type) : common->concise_type;
										SQLPOINTER buffer = (BYTE*)common->data_ptr + offset +													                    
										                    RowNumber * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? GetCTypeLength(type, (pARD->ad_records[i].octet_length_ptr) ? *pARD->ad_records[i].octet_length_ptr : 0) : pARD->header.bind_type);
										SQLPOINTER indicator = (BYTE*)pARD->ad_records[i].indicator_ptr + offset + 
										                       RowNumber * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? sizeof(SQLINTEGER) : pARD->header.bind_type);

										if (NULL == (pIPD->id_records[i].common.data_ptr = (SQLPOINTER*)PrepareParameter(pStatement, buffer, common->octet_length, common->concise_type, (SQLINTEGER*)indicator, pIPD->id_records[i].common.concise_type, common->scale)))
										{/* no more memory */
											nRet = SQL_ERROR;
											break;
										}
									}
								}

								/* prepare bookmark parameters */
								if (SQL_UPDATE == Operation || SQL_DELETE == Operation)
								{/* get bookmarks in driver's format */
									if (NULL == (pIPD->id_records[(SQL_DELETE == Operation) ? 0 : columns].common.data_ptr = (SQLPOINTER*)PrepareParameter(pStatement, (int*)pIRD->id_records[first_bookmark].common.data_ptr[RowNumber]+1, *(int*)pIRD->id_records[first_bookmark].common.data_ptr[RowNumber], SQL_C_TCHAR, NULL, SQL_DEFAULT, 0)))
									{/* no more memory */
										nRet = SQL_ERROR;
									}
								}

								/* execute query */
								nRes = ((SQL_SUCCESS == (nRet = DeclarePortal(pQueryStmt))) &&
								        (SQL_SUCCESS == Stmt_SendMessageToBackend(pQueryStmt->connection, MSG_Execute, pQueryStmt)) &&
								        (SQL_SUCCESS == Stmt_SendMessageToBackend(pQueryStmt->connection, MSG_Sync, pQueryStmt)) &&
								        (SQL_SUCCESS == WaitForBackendReply(pQueryStmt->connection,  MSG_ReadyForQuery, pQueryStmt))
								       ) ? SQL_SUCCESS : SQL_ERROR;
							}

							EndTransaction(SQL_HANDLE_STMT, pQueryStmt, SQL_COMMIT, TO_DRIVER);
							RET_DESCRIPTOR(pIPD);
							FreeStatement(pQueryStmt, SQL_DROP);
						}
						FREE(query);
					}
					else
					{
						/* error */
						nRet = SQL_ERROR;
					}
				}
				else
				{/* unable to do this without bookmarks retrieved */
					SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_BOOKMARKS_UNUSED, NULL);
					nRet = SQL_ERROR;
				}
				break;
			case SQL_REFRESH:
			default:
				/* there is no way to update bookmark */
				SetError(SQL_HANDLE_STMT, pStatement, ERR_UNSUPPORTED_OPERATION, NULL);
				nRet = SQL_ERROR;
		}
		RET_DESCRIPTOR(pARD);
	}

	RET_DESCRIPTOR(pIRD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: FillBoundColumns
 *
 * DESCRIPTION: Fills all bound columns in 'index' position with 'row' data
 *
 * WARNING: 'row' - zero-based index!
 *-----------------------------------------------------------------------------
 */
SQLRETURN
FillBoundColumns(Descriptor* pARD, Descriptor* pIRD, SQLINTEGER index, SQLLEN row, SQLSMALLINT fill_bookmark)
{
	SQLRETURN   nRet    = SQL_SUCCESS;
	SQLPOINTER  buffer  = NULL;
	SQLINTEGER  is_null = SQL_FALSE;
	SQLSMALLINT count   = pIRD->header.count - pIRD->header.bookmarks - 1;

	/* fill bound bookmark */
	if (SQL_TRUE == fill_bookmark)
	{/* we should calculate unique bookmark value, depanding on values from recived */
		if (pARD->bookmark.data_ptr)
		{/* fill buffer with data */
			buffer = (BYTE*) pARD->bookmark.data_ptr +
					     ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr) +
					     index * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? GetCTypeLength(pARD->bookmark.type, pARD->bookmark.octet_length) : pARD->header.bind_type);

			if (SQL_C_BOOKMARK == pARD->bookmark.type)
			{
				memcpy(buffer, &row, sizeof(row));
			}
			else
			{
				FillBufferWithValue(buffer,	pARD->bookmark.octet_length, pARD->bookmark.type,
									(TCHAR*)((int*)pIRD->id_records[pIRD->header.count-1].common.data_ptr[row]+1),
									*(int*)pIRD->id_records[pIRD->header.count-1].common.data_ptr[row],
									pIRD->id_records[pIRD->header.count-1].common.concise_type
								);
			}
		}

		if (pARD->bookmark.indicator_ptr)
		{/* fill buffer with bookmark length */
			SQLLEN value = (SQL_FALSE == is_null) ? GetCTypeLength((SQLSMALLINT)((SQL_DEFAULT == pARD->bookmark.type) ? SQL_C_BINARY : pARD->bookmark.type), *(int*)pIRD->id_records[pIRD->header.count-1].common.data_ptr[row]) : SQL_NULL_DATA;
			buffer = (BYTE*) pARD->bookmark.indicator_ptr +
					     ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr) +
					     index * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? sizeof(SQLINTEGER) : pARD->header.bind_type);

			memcpy(buffer, &value, sizeof(value));
		}
	}

	for (;0<=count;count--)
	{
		if (pARD->ad_records && count < pARD->header.count)
		{
			SQLINTEGER ret_size = -1;
			/* check for 'null' value */
			is_null = (FIELD_NULL == *(int*)pIRD->id_records[count].common.data_ptr[row]) ? SQL_TRUE : SQL_FALSE;

			if (pARD->ad_records[count].common.data_ptr && !is_null)
			{/* fill buffer with data */
				buffer = (BYTE*) pARD->ad_records[count].common.data_ptr +
						 ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr) +
						 index * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? GetCTypeLength(pARD->ad_records[count].common.concise_type, pARD->ad_records[count].common.octet_length) : pARD->header.bind_type);

				ret_size = FillBufferWithValue(buffer,
									pARD->ad_records[count].common.octet_length,
									pARD->ad_records[count].common.concise_type,
									(TCHAR*)((int*)pIRD->id_records[count].common.data_ptr[row]+1),
									*(int*)pIRD->id_records[count].common.data_ptr[row],
									pIRD->id_records[count].common.concise_type
								   );
			}

			if (pARD->ad_records[count].indicator_ptr)
			{/* fill buffer with data length */
				SQLLEN value = (SQL_FALSE == is_null) ? GetCTypeLength((SQLSMALLINT)((SQL_DEFAULT == pARD->ad_records[count].common.concise_type) ? GetCDefaultType(pIRD->id_records[count].common.concise_type) : pARD->ad_records[count].common.concise_type), (0 <= ret_size) ? ret_size : *(int*)pIRD->id_records[count].common.data_ptr[row]) : SQL_NULL_DATA;
				buffer = (BYTE*) pARD->ad_records[count].indicator_ptr +
						 ((SQL_BIND_BY_COLUMN == pARD->header.bind_type || (NULL == pARD->header.bind_offset_ptr)) ? 0 : *pARD->header.bind_offset_ptr) +
						 index * ((SQL_BIND_BY_COLUMN == pARD->header.bind_type) ? sizeof(SQLINTEGER) : pARD->header.bind_type);

				memcpy(buffer, &value, sizeof(value));
			}
		}

		pIRD->id_records[count].bytes_left = *(int*)pIRD->id_records[count].common.data_ptr[row];
		if (pIRD->id_records[count].bytes_left < 0)
			pIRD->id_records[count].bytes_left = -SQL_NO_DATA;
	}

	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: CheckDUQuery
 *
 * WARNING: schema(table) supposed to be SCHEMA_MAX_LENGTH(TABLE_MAX_LENGTH)+1 length
 *-----------------------------------------------------------------------------
 */
SQLRETURN
CheckDUQuery(Descriptor* pARD, Descriptor* pIRD, TCHAR* schema, TCHAR* table, TCHAR* query)
{
	SQLRETURN nRet = (query) ? SQL_SUCCESS : SQL_SUCCESS_WITH_INFO;

	if (SQL_SUCCESS == nRet)
	{/* check for ignore columns and similar */
	
	}

	if (SQL_SUCCESS_WITH_INFO == nRet)
	{
		GetDescField(pIRD, 1, SQL_DESC_SCHEMA_NAME, schema, sizeof(TCHAR)*(SCHEMA_MAX_LENGTH+1), NULL, NULL); /* schema name */
		GetDescField(pIRD, 1, SQL_DESC_TABLE_NAME,  table,  sizeof(TCHAR)*(TABLE_MAX_LENGTH+1),  NULL, NULL); /* table name */
	}

	return nRet;
}

#define ADD_COLUMN(ard,column) (ard->ad_records[column].bound && ((NULL == ard->ad_records[column].indicator_ptr) || (ard->ad_records[column].indicator_ptr && SQL_COLUMN_IGNORE != *ard->ad_records[column].indicator_ptr)))
/*-----------------------------------------------------------------------------
 * FUNCTION: GetQuery
 *-----------------------------------------------------------------------------
 */
SQLTCHAR*
GetQuery(Descriptor* pARD, Descriptor* pIRD, SQLINTEGER* columns, SQLINTEGER* first_bookmark, SQLUSMALLINT Operation)
{
	TCHAR  schema[SCHEMA_MAX_LENGTH+1];
	TCHAR  table[TABLE_MAX_LENGTH+1];
	TCHAR* query = (SQL_DELETE == Operation) ? pIRD->bookmark.delete_query : pIRD->bookmark.update_query;

	switch (CheckDUQuery(pARD,pIRD,schema,table, query))
	{
		case SQL_SUCCESS_WITH_INFO:
		{
			*first_bookmark = pIRD->header.count - pIRD->header.bookmarks;
			*columns = MIN(pARD->header.count, *first_bookmark);
			
			switch(Operation)
			{
				case SQL_DELETE:
				{/* prepare new update query */
					query = pIRD->bookmark.delete_query = GetText(_T("DELETE FROM \"\?\".\"\?\" WHERE \?"), schema, table, _T("ctid = ?"), NULL);
					break;
				}
				case SQL_UPDATE:
				{
					TCHAR       buffer[10000]; /* andyk */
					SQLSMALLINT i, col_length;
					TCHAR*      col_name = buffer;

					*col_name = _T('\"');
					/* prepare column-name's list */
					for (i=1;i<=*columns;i++)
						if ADD_COLUMN(pARD,i-1)
						{/* do not ignore this column */
							GetDescField(pIRD, i, SQL_DESC_BASE_COLUMN_NAME, ++col_name, COLUMN_MAX_LENGTH+1, &col_length, NULL);
							col_name += col_length/sizeof(TCHAR);
							_tcsncpy(col_name, _T("\"=\?,\""), STR_SIZEOF("\"=\?,\""));
							col_name += STR_SIZEOF("\"=\?,\"")-1;
						}

					col_name -= STR_SIZEOF(",\"")-1;
					*col_name = _T('\0');

					query = pIRD->bookmark.update_query = GetText(_T("UPDATE \"\?\".\"\?\" SET \? WHERE \?"), schema, table, buffer, _T("ctid = ?"), NULL);
					break;
				}
				default:
					query = NULL;
			}
		} /* NO BREAK! */
		case SQL_SUCCESS:
			return (SQLTCHAR*)query;
		default:
			return NULL;
	}
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ProcessRow
 *-----------------------------------------------------------------------------
 */
SQLRETURN
ProcessRow(Statement* pStatement, Descriptor* pARD, Descriptor* pIRD, SQLSMALLINT Operation, SQLINTEGER row, SQLSMALLINT* status)
{
	SQLRETURN nRet = SQL_SUCCESS;
	/* exit */
	return nRet;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: FindRow
 *
 * DESCRIPTION: Converts bookmark value and checks corresponding row in resultset
 *-----------------------------------------------------------------------------
 */
SQLLEN
FindRow(Descriptor* pIRD, SQLCHAR* bookmark, SQLSMALLINT bookmark_type)
{
	SQLINTEGER col = pIRD->header.count-1;
	SQLINTEGER len = strlen((char*)bookmark);
	SQLLEN     i,j;
	TCHAR*     str;

	if (SQL_C_BOOKMARK == bookmark_type)
		return *(SQLINTEGER*)bookmark;

	for(i=pIRD->header.array_size-1;0<=i;i--)
	{
		str = (TCHAR*)((int*)pIRD->id_records[col].common.data_ptr[i]+1);
		if (len == *(int*)pIRD->id_records[col].common.data_ptr[i])
		{
			for (j=len-1;0<=j;j--)
				if (str[j] != bookmark[j])
					break;

			if (0 > j)
				return i;
		}
	}

	return -1;
}

