#ifndef __CONNECTION__
#define __CONNECTION__

#include "odbc.h"
#include "error.h"
#include "common.h"

#ifdef _NEED_LOG
	void Log(const TCHAR* szText);
	
	#define BEGIN(x)    Log(x)
	#define END(x)      return x
#else
	#define BEGIN(x)
	#define END(x)      return x
#endif /* _NEED_LOG */


#ifdef _MULTITHREAD_SUPPORT
	#define ENTER_ENV(env, func_name)        BEGIN(func_name); ENTER_CRITICAL_SECTION((Environment*)env); SetError(SQL_HANDLE_ENV, env, ERR_CLEAR, NULL)
	#define ENTER_CONN(conn, func_name)      BEGIN(func_name); ENTER_CRITICAL_SECTION((Connection*)conn); SetError(SQL_HANDLE_DBC, conn, ERR_CLEAR, NULL)
	#define ENTER_STMT(stmt, func_name)      BEGIN(func_name); ENTER_CRITICAL_SECTION((Statement*)stmt);  SetError(SQL_HANDLE_STMT, stmt, ERR_CLEAR, NULL)
	#define ENTER_DESC(desc, func_name)      BEGIN(func_name); ENTER_CRITICAL_SECTION((Descriptor*)desc); SetError(SQL_HANDLE_DESC, desc, ERR_CLEAR, NULL)
	#define LEAVE_ENV(env, ret)              LEAVE_CRITICAL_SECTION((Environment*)env); END(ret)	
	#define LEAVE_CONN(conn, ret)            LEAVE_CRITICAL_SECTION((Connection*)conn); END(ret)
	#define LEAVE_STMT(stmt, ret)            LEAVE_CRITICAL_SECTION((Statement*)stmt); END(ret)
	#define LEAVE_DESC(desc, ret)            LEAVE_CRITICAL_SECTION((Descriptor*)desc); END(ret)

	#define __CRITICAL_SECTION(cs)           _CRITICAL_SECTION(cs)
	#define _INIT_CRITICAL_SECTION(handle)   INIT_CRITICAL_SECTION(handle)
	#define _DELETE_CRITICAL_SECTION(handle) DELETE_CRITICAL_SECTION(handle)
#else
	#define ENTER_ENV(env, func_name)        BEGIN(func_name)
	#define ENTER_CONN(conn, func_name)      BEGIN(func_name)
	#define ENTER_STMT(stmt, func_name)      BEGIN(func_name)
	#define ENTER_DESC(desc, func_name)      BEGIN(func_name)
	#define LEAVE_ENV(env, ret)              END(ret)	
	#define LEAVE_CONN(conn, ret)            END(ret)
	#define LEAVE_STMT(stmt, ret)            END(ret)
	#define LEAVE_DESC(desc, ret)            END(ret)
	
	#define __CRITICAL_SECTION(cs)
	#define _INIT_CRITICAL_SECTION(handle)
	#define _DELETE_CRITICAL_SECTION(handle)
#endif /* _MULTITHREAD_SUPPORT */


#define MAX_ERROR(x, y)    (x == y) ? x : ((SQL_SUCCESS == x) ? y : x)

#define SQLSMALLINT_LENGTH     15 /* number of decimal digits parameter can*/


#define UID_MAX_LENGTH            255
#define PWD_MAX_LENGTH            255
#define DATABASE_MAX_LENGTH       UID_MAX_LENGTH /* default database is user name */
#define SERVER_MAX_LENGTH         255
#define PORT_MAX_LENGTH           255
#define DESCRIPTION_MAX_LENGTH    255
#define USESSL_MAX_LENGTH           1 /* Y or N */
#define USEBUFFERING_MAX_LENGTH     1 /* Y or N */
#define CONNECTTIMEOUT_MAX_LENGTH  16

#define DSN_MAX_LENGTH            255

#define MAX_CURSOR_NAME_LENGTH     20

typedef enum
{
	CS_CONNECTED     = 1,
	CS_READY         = 2,
	CS_SENDING_QUERY = 4,

	CS_DISCONNECTED  = !CS_CONNECTED
} ConnectionState;

typedef enum
{
	TO_NOT_OPENED  = 0x00, /* initial state */

	TO_APPLICATION = 0x11, /* application request */
	TO_DRIVER      = 0x10, /* driver request */

	TO_DRIVER_THE_FIRST = 0x02 /* driver was 'THE FIRST' */
} TransactionOption;

typedef enum
{
	MSG_Unknown = -1,
	MSG_AuthenticationCleartextPassword,
	MSG_AuthenticationCryptPassword,
	MSG_AuthenticationKerberosV4,
	MSG_AuthenticationKerberosV5,
	MSG_AuthenticationMD5Password,
	MSG_AuthenticationOk,
	MSG_AuthenticationSCMCredential,
	MSG_BackendKeyData,
	MSG_Bind,
	MSG_BindComplete,
	MSG_CancelRequest,
	MSG_CloseStatement,
	MSG_ClosePortal,
	MSG_CloseComplete,
	MSG_CommandComplete,
	MSG_CopyData,
	MSG_CopyDone,
	MSG_CopyInResponse,
	MSG_CopyFail,
	MSG_CopyOutResponse,
	MSG_DataRow,
	MSG_Describe,
	MSG_EmptyQueryResponse,
	MSG_ErrorResponse,
	MSG_Execute,
	MSG_Parse,
	MSG_Flush,
	MSG_FunctionCall,
	MSG_FunctionCallResponse,
	MSG_NoData,
	MSG_NoticeResponse,
	MSG_NotificationResponse,
	MSG_ParameterDescription,
	MSG_ParameterStatus,
	MSG_ParseComplete,
	MSG_PasswordMessage,
	MSG_PortalSuspended,
	MSG_Query,
	MSG_ReadyForQuery,
	MSG_RowDescription,
	MSG_SSLRequest,
	MSG_StartupMessage,
	MSG_Sync,
	MSG_Terminate
} MessageType;

struct _message
{
	MessageType type;
	char        id;
	int         length;
	TCHAR*      message;
};

typedef struct _message Message;
/* ***** */
typedef struct
{
	const TCHAR* szKeyName;
	const TCHAR* szIdentifier;
	const TCHAR* szDefaultValue;
	unsigned int unMaxValueLength;
	unsigned int unDialogItemID; /* WIN32 configure dialog */
} DSParameter;

#define DESCRIPTION_PARAM          0

#define FIRST_CONNSTR_PARAM        1 /* number of parameters useless in Connection String */ 

#define UID_PARAM                  1
#define SERVER_PARAM               2
#define DATABASE_PARAM             3

#define NECESSARY_CONNSTR_PARAM    4 /* number of first unnecessary parameter in connection string */

#define PWD_PARAM                  4
#define PORT_PARAM                 5
#define CONNECTTIMEOUT_PARAM       6

#define DS_EDIT_PARAMETERS_NUMBER  7 /* number of parameters, configured in EDITs */

#define USESSL_PARAM               7
#define USEBUFFERING_PARAM         8

#define DS_PARAMETERS_NUMBER       9

#define DS_PARAMETERS_BUFFER_SIZE  UID_MAX_LENGTH + \
                                   PWD_MAX_LENGTH + \
                                   DATABASE_MAX_LENGTH + \
                                   SERVER_MAX_LENGTH + \
                                   PORT_MAX_LENGTH + \
                                   DESCRIPTION_MAX_LENGTH + \
																	 CONNECTTIMEOUT_MAX_LENGTH + \
                                   USESSL_MAX_LENGTH + \
																	 USEBUFFERING_MAX_LENGTH + \
                                   DS_PARAMETERS_NUMBER /* null-terminals for every parameter value */

extern const DSParameter c_stDSParameters[DS_PARAMETERS_NUMBER];
/* ***** */

#define MAX(x, y) (x > y ? x:y)
#define MIN(x, y) (x < y ? x:y)

typedef struct _column       Column;
typedef struct _environment  Environment;
typedef struct _connection   Connection;
typedef struct _statement    Statement;
typedef struct _attribute    Attribute;
typedef struct _parameter    Parameter;

/*---------------------------------------------------------
	enum QueryType
---------------------------------------------------------*/
typedef enum
{
	ST_UNKNOWN = 0x000,
	ST_SELECT  = 0x001,
	ST_INSERT  = 0x002,
	ST_DELETE  = 0x004,
	ST_UPDATE  = 0x008,
	ST_MOVE    = 0x010,
	ST_FETCH   = 0x020,
	ST_BEGIN   = 0x040,
	ST_COMMIT  = 0x080,
	ST_EMPTY   = 0x100,
	ST_SET     = 0x200
} QueryType;


/*---------------------------------------------------------
	struct _conn_attributes
---------------------------------------------------------*/
struct _conn_attributes
{
	SQLUINTEGER connection_dead; /* SQL_CD_TRUE = dead, SQL_CD_FALSE = still alive */
	SQLUINTEGER autocommit;      /* SQL_AUTOCOMMIT_OFF = manual, SQL_AUTOCOMMIT_ON = default */
	SQLUINTEGER unicode;         /* SQL_AA_TRUE = ansi, SQL_AA_FALSE = unicode */
};

/*---------------------------------------------------------
	struct _backend
---------------------------------------------------------*/
struct _backend
{
	int version_first;
	int version_second;
	int version_third;
};

/*---------------------------------------------------------
	struct EnvAttributes
---------------------------------------------------------*/
struct _env_attributes
{
	SQLUINTEGER cp_match;
	SQLUINTEGER odbc_version;
	SQLUINTEGER connection_pooling;
};

typedef struct _env_attributes EnvAttributes;
/*---------------------------------------------------------
	struct Environment
---------------------------------------------------------*/
struct _environment
{
	struct _env_attributes attributes;
	struct _diag           diag;
	struct _list           connections; /* sconnections allocated under this environment */

	__CRITICAL_SECTION(cs);
};


#define CONNECTION_OUT_BUFFER_SIZE      2048
#define CONNECTION_IN_BUFFER_SIZE       2048

typedef unsigned int ProtocolVersion;
/*---------------------------------------------------------
	struct Connection
---------------------------------------------------------*/
struct _connection
{
	ConnectionState  state; /* CS_READY, CS_DISCONNECTED */

	struct _diag            diag;
	struct _list            statements;  /* statements allocated under this connection */
	struct _list            descriptors; /* descriptors             -//-               */
	struct _conn_attributes attributes;
	struct _backend         backend;     /* backend info */

	TCHAR* dbms; /* connected dbms information */


	BOOL TransactionState; /* TRUE - transaction began,
	                        * FALSE - transaction closed or uncreated
	                        */

	/* SOCKET connection data */
	SOCKET             socket;
	struct sockaddr_in addr;

	/* Backend data */
	unsigned int processId;
	unsigned int secretKey;

	unsigned int cursor_count;

	/* buffer for the outgoing messages to backend */
	BYTE  outBuffer[CONNECTION_OUT_BUFFER_SIZE];
	BYTE* pOutBegin;
	BYTE* pOutFirst;
	BYTE* pOutLast;
	BYTE* pOutEnd;

	/* buffer for the incoming messages from backend */
	BYTE  inBuffer[CONNECTION_IN_BUFFER_SIZE];
	BYTE* pInBegin;
	BYTE* pInFirst;
	BYTE* pInLast;
	BYTE* pInEnd;

	TCHAR dsn[DSN_MAX_LENGTH+1];
	ProtocolVersion version;

	Environment* environment;
	
	/* Data Source parameters */
	TCHAR* parameters[DS_PARAMETERS_NUMBER];
	TCHAR  parameters_buffer[DS_PARAMETERS_BUFFER_SIZE];

	Connection* pNext;   /* next connection in the environment's list */

	__CRITICAL_SECTION(cs);
};

Environment* AllocEnvironment();
SQLRETURN    FreeEnvironment(Environment* pEnvironment);

Connection* AllocConnection(Environment* pEnvironment);
SQLRETURN   FreeConnection(Connection* pConnection, SQLUSMALLINT Option);

SQLRETURN Connect(Connection* pConnection);
SQLRETURN Disconnect(Connection* pConnection);

SQLRETURN ReturnError();

void ReadFromDS(TCHAR** parameters, SQLTCHAR* DSN, SQLSMALLINT DSNLength);
void WriteToDS(TCHAR** parameters, TCHAR* DSN);

SQLRETURN ParseConnectionString(Connection* pConnection, SQLTCHAR* szConnStrIn, SQLSMALLINT cbConnStrIn);
SQLRETURN GetKeyValue(Connection* pConnection, TCHAR* Key, unsigned int KeyLength, TCHAR* Value, unsigned int ValueLength);
SQLRETURN BeginTransaction(Statement* pStatement, SQLSMALLINT Option);
SQLRETURN EndTransaction(SQLSMALLINT HandleType, SQLHANDLE Handle, SQLSMALLINT CompletionType, SQLSMALLINT Option);
SQLRETURN GetConnectionInfo(Connection* pConnection, SQLUSMALLINT InfoType, SQLPOINTER InfoValue, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength);
SQLRETURN SetConnectAttr(Connection* pConnection, SQLINTEGER Attribute, SQLPOINTER Value, SQLINTEGER StringLength);

SQLRETURN sock_connect(Connection* pConnection);
SQLRETURN sock_close(Connection* pConnection);

int sock_send(Connection* pConnection);
int sock_recv(Connection* pConnection);

/* constants definitions */
extern const int c_FIELD_NULL;

extern const TCHAR c_szKeyWords[];
extern const TCHAR c_szODBC_INI[];
extern const TCHAR c_szDriverName[];
extern const TCHAR c_szBookmarkName[];
extern const TCHAR c_szDBMSVersion[12];
extern const TCHAR c_szDriverODBCVer[6];
extern const TCHAR c_szDriverVersion[17];
extern const TCHAR c_szSpecialCharacter[];

#ifdef WIN32
	extern const TCHAR c_szDriverFileName[10];
#else
	extern const TCHAR c_szDriverFileName[12];
#endif /* WIN32 */

extern const SQLUINTEGER c_nBookmarkDisplaySize;

#define SQL_STMT_NOSCAN_OFF(stmt)  (SQL_NOSCAN_OFF == stmt->attributes.no_scan)
#define AUTOCOMMIT_MODE_OFF(stmt)  (SQL_AUTOCOMMIT_OFF == stmt->connection->attributes.autocommit)
#define AUTOCOMMIT_MODE_ON(stmt)   (SQL_AUTOCOMMIT_ON  == stmt->connection->attributes.autocommit)
#define STATEMENT_NEEDS_REDECLARATION(stmt)  (0 != (SS_STATEMENT_NEED_REDECLARE & stmt->state))
#define PORTAL_NEEDS_REDECLARATION(stmt)     (0 != (SS_PORTAL_NEED_REDECLARE & stmt->state))

#define SET_PORTAL_NEEDS_REDECLARATION(stmt)    stmt->state |= SS_PORTAL_NEED_REDECLARE
#define SET_STATEMENT_NEEDS_REDECLARATION(stmt) stmt->state |= SS_STATEMENT_NEED_REDECLARE
#define SET_STATEMENT_DECLARED(stmt)            stmt->state |= SS_STATEMENT_DECLARED;stmt->state ^= SS_STATEMENT_NEED_REDECLARE

#define PORTAL_DECLARED(stmt)    (0 != (SS_PORTAL_DECLARED & stmt->state))
#define STATEMENT_DECLARED(stmt) (0 != (SS_STATEMENT_DECLARED & stmt->state))
#define STMT_USE_BUFFERING(stmt) (TRUE == stmt->use_buffering)

#define CSOC_OBJ_TABLE           0
#define CSOC_OBJ_PROC            1

#define QUOTER_FOUND             0xFFFF

#define MD5_PASSWD_LEN	35
BOOL EncryptMD5(const char *passwd, const char *salt, size_t salt_len, char *buf);

#define SCHEMA_MAX_LENGTH               63
#define TABLE_MAX_LENGTH                63
#define COLUMN_MAX_LENGTH               63
#define MAX_COLUMNS_IN_TABLE          1600


#endif /* #ifndef __CONNECTION__ */
